/*
 * @(#)ReadStringStream.java
 *
 * Copyright (C) 2001,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
 

package net.sourceforge.groboutils.util.io.v1;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Reader;




/**
 * Reads a byte array from a stream until the stream is finished.
 * You can specify a maximum size to read, and the block read size.
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    Alpha 0.9.1d (sometime in 2001)
 * @version  $Date: 2003/02/10 22:52:45 $
 */
public class ReadStringStream
{
    //----------------------------
    // Public data
    
    /**
     * Read in an unlimited number of bytes.  This can be very
     * dangerous.
     */
    public static final int READ_TO_END_OF_STREAM = Integer.MAX_VALUE;
    
    /**
     * Default block read size.
     */
    public static final int DEFAULT_BLOCK_READ_SIZE = 4096;
    
    //----------------------------
    // Private data
    
    private Reader m_is;
    private int m_maxSize;
    private int m_bufferSize;
    
    
    //----------------------------
    // constructors
    
    /**
     * Create a new stream using the default parameters and the given
     * reader.
     *
     * @param input reader to pull from.
     */
    public ReadStringStream( Reader input )
    {
        this( input, READ_TO_END_OF_STREAM, DEFAULT_BLOCK_READ_SIZE );
    }
    
    
    /**
     *
     * @param input reader to pull from.
     */
    public ReadStringStream( Reader input, int maxReadSize, int blockReadSize )
    {
        setReader( input );
        setSizes( maxReadSize, blockReadSize );
    }
    
    
    //----------------------------
    // Public methods
    
    
    /**
     * Sets the internal input stream.
     *
     * @param input reader to pull from.
     */
    public void setReader( Reader input )
    {
        if (input == null)
        {
            throw new IllegalArgumentException( "Reader is null" );
        }
        this.m_is = input;
    }
    
    
    /**
     * Sets the internal sizes.
     */
    public void setSizes( int maxReadSize, int blockReadSize )
    {
        if (blockReadSize <= 0)
        {
            blockReadSize = DEFAULT_BLOCK_READ_SIZE;
        }
        if (maxReadSize <= 0 || maxReadSize > READ_TO_END_OF_STREAM)
        {
            maxReadSize = READ_TO_END_OF_STREAM;
        }
        if (maxReadSize < blockReadSize)
        {
            blockReadSize = maxReadSize;
        }
        this.m_maxSize = maxReadSize;
        this.m_bufferSize = blockReadSize;
    }
    
    
    /**
     * Read in the byte stream, using the current settings.
     *
     * @return the read-in string
     */
    public String readStringStream()
            throws IOException
    {
        return readStringStream( this.m_is, this.m_maxSize, this.m_bufferSize );
    }
    
    
    /**
     * Read in the byte stream.  Does not close the stream after it has
     * finished reading.  Uses the default sizes.
     *
     * @param input reader to pull from.
     * @return the read-in string
     * @see #readStringStream( Reader, int, int )
     */
    public static String readStringStream( Reader input )
            throws IOException
    {
        return readStringStream( input, READ_TO_END_OF_STREAM,
            DEFAULT_BLOCK_READ_SIZE );
    }
    
    
    /**
     * Read in the stream to a String.  Does not close the stream after it has
     * finished reading.
     * <P>
     * Note that there is no variable checking, for performance reasons.
     * The user needs to verify that:
     * @param input the reader, which cannot be <tt>null</tt>.
     * @param maxReadSize the maximum number of bytes to read, which
     *      must be positive, and must be modulo 0 of <tt>blockReadSize</tt>.
     *      This is an "estimation", and may actually read in more than this
     *      many bytes if it is not modulo 0 of <tt>blockReadSize</tt>, but
     *      will always return all the bytes read.
     * @param blockReadSize the number of bytes to read in per read command,
     *      which cannot be more than <tt>maxReadSize</tt>, and cannot be
     *      less than or equal to zero.
     */
    public static String readStringStream( Reader input, int maxReadSize,
            int blockReadSize )
            throws IOException
    {
        StringWriter sw = new StringWriter();
        char buffer[] = new char[ blockReadSize ];
        int size = input.read( buffer, 0, blockReadSize );
        int totSize = size;
        while (size > 0 && totSize < maxReadSize)
        {
            sw.write( buffer, 0, size );
            size = input.read( buffer, 0, blockReadSize );
            totSize += size;
        }
        sw.close();
        return sw.toString();
    }
}

