// $Id: prpath.cc 1.5.1.1 Mon, 15 Dec 1997 14:43:04 -0800 cengiz $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

#include "config.hh"
#include <cstdlib>

extern "C" {
#if HAVE_UNISTD_H
#   include <unistd.h>
#endif
}
#include <iostream.h>
#include <iomanip.h>
#include "dbase.hh"
#include "Node.h"
#include "debug.hh"
#include "trace.hh"
#include "rusage.hh"
#include "aut-num.hh"
#include "Route.hh"
#define MAX_AS_COUNT (64*1024)
#include "misc/Stack.hh"
#include "version.hh"
#include "Argv.hh"

Rusage ru;
Route rt;

int  opt_rusage        = 0;
char *opt_prompt       = "prpath> ";
char *opt_my_as        = NULL;
int  opt_echo_as       = 0;
int  opt_max_path_length = 2;
int  path_length_limit = 12;
int  opt_interactive = 1;

void usage(char *argv[]) {
   cerr << "Usage: " << argv[0] << " to be written later" << endl;
   exit(1);
}

int start_tracing(char *dst, char *key, char *nextArg) {
   if (nextArg) {
      trace.enable(nextArg);
      return 1; // return 1 to signify nextArg is used by us
   }
   return 0; 
}

int start_debugging(char *dst, char *key, char *nextArg) {
   if (nextArg) {
      Debug(dbg.enable(atoi(nextArg)));
      return 1; // return 1 to signify nextArg is used by us
   }
   return 0;
}

void init_and_set_options (int argc, char **argv, char **envp) {
   ArgvInfo argTable[] = {
      // RAToolSet common arguments
      // key, type, src, dst, help
      {"-T", ARGV_FUNC, (char *) &start_tracing,      (char *) NULL, 
       "Start tracing the next argument"},
      {"-D", ARGV_FUNC, (char *) &start_debugging,    (char *) NULL, 
       "Start debugging the next argument"},
      {"-version", ARGV_FUNC, (char *) &version,      (char *) NULL,
       "Show version"},
      {"-h", ARGV_FUNC, (char *) &Whois::ArgvHost,    (char *) NULL,
       "Host name of the RAWhoisd server"},
      {"-p", ARGV_FUNC, (char *) &Whois::ArgvPort,    (char *) NULL,
       "Port number of the RAWhoisd server"},
      {"-s", ARGV_FUNC, (char *) &Whois::ArgvSources, (char *) NULL,
       "Order of databases"},
      {"-rusage", ARGV_CONSTANT, (char *) 1,          (char *) &opt_rusage,
       "On termination print resource usage"},
      {"-prompt", ARGV_STRING,  (char *) NULL,        (char *) &opt_prompt,
       "Prompt"},
      {"-ignore_errors", ARGV_FUNC, (char *)&Whois::IgnoreErrors, (char *)NULL,
       "Ignore IRR error and warning messages"},
      {"-report_errors", ARGV_FUNC, (char *)&Whois::ReportErrors, (char *)NULL,
       "Print IRR error and warning messages"},

      // prpath specific arguments
      {"-as",  ARGV_STRING,       (char *) NULL, (char *) &opt_my_as,
       "AS number of the aut-num object to use."},
      {"-mpl", ARGV_INT,          (char *) NULL, (char *) &opt_max_path_length,
       "Max path length"},

      {(char *) NULL, ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
   };

   for (char **p = envp; *p != NULL; p++) {
      if (strncmp(*p, "IRR_HOST=", 9) == 0)  {
	 whois.SetDefaultHost(*p + 9);
         continue;
      }
      if (strncmp(*p, "IRR_PORT=", 9) == 0)  {
	 whois.SetDefaultPort(atoi(*p + 9));
         continue;
      }
      if (strncmp(*p, "IRR_SOURCES=", 12) == 0)  {
	 whois.SetDefaultSources(*p + 12);
         continue;
      }
   }

   Whois::IgnoreErrors(NULL, NULL, NULL);

   if (ParseArgv(&argc, argv, argTable, 0) != ARGV_OK) {
      cerr << endl;
      exit(1);
   }

   // if there are remaining arguments
   // the first one is my as number
   // the second one is a destination prefix
   switch (argc) {
   case 3 :
      opt_my_as = argv[1];
   case 2:
      extern char *xx_input_string;
      xx_input_string = argv[argc - 1];
      opt_interactive = 0;
   case 1:
      break;
   default:
      cerr << "Wrong number of arguments..." << endl;
      exit(-1);
   }

   // have a prompt only if the input is coming from a tty
   if (!opt_interactive || !isatty(fileno(stdin)) || !isatty(fileno(stdout)))
      opt_prompt = NULL;
}

void Originate(Route &r, Pix as);
void Announce(Route &r, Pix as);
void Import(Route &r, Pix as, Pix laddr, Pix ras, Pix raddr);
Pix source_as;
int shortest_path_length(register Pix source, register Pix sink);
void start_prpath(Route &r);

main (int argc, char **argv, char **envp) {
   Route rt;
   Pix my_rt;
   extern int xxparse();

   init_and_set_options(argc, argv, envp);

   if (opt_my_as)
      source_as = AS_map.add_entry(opt_my_as);
   else {
      ipAddr myip;
      myip.setToMyHost();
      source_as = AS_map.add_entry(myip);
   }      

   xxparse();

   flush_symbol_tables();

   if (opt_prompt)
      cout << endl;
	
   if (opt_rusage)
      clog << ru;

}

void start_prpath(Route &rt) {
   ListNodePix *origin_ptr;

   // do a prpath from all origins of the route
   for (origin_ptr = rt.origin.head(); origin_ptr; 
	origin_ptr = rt.origin.next(origin_ptr->l)) {
      path_length_limit = shortest_path_length(origin_ptr->pix, source_as) 
	 + opt_max_path_length;
      Originate(rt, origin_ptr->pix);
   }
}

void Originate(Route &r, Pix as) {
   Announce(r, as);
}

void Announce(Route &r, Pix as) {
   ASPolicy *p;
   Filter_Action *fap;
   Filter_Action *ifap;
   InterASPolicy *ip;
   int interas_match;

   if (!as || !AS_map.define(as))
      return;

   AutNum &autnum = *(AutNum *) AS_map.contents(as).definition;

   r.aspath.append((new ListNodePix(as))->l);

   // for each peer
   for (p = autnum.peers.head(); p; p = autnum.peers.next(p->peers)) {
      // for each as-out expression
      for (fap = p->out.head(); fap; fap = p->out.next(fap->falist)) {
	 if (fap->filter->match(r)) {
	    fap->action->Execute(r);
	    interas_match = 0;

	    // for each laddr raddr pair
	    for (ip = p->interas.head(); ip; ip = p->interas.next(ip->interas))
	       // for each interas-out expression
	       for (ifap = ip->out.head(); 
		    ifap; 
		    ifap = ip->out.next(ifap->falist))
		  if (ifap->filter->match(r)) {
		     interas_match = 1;
		     ifap->action->Execute(r);
		     Import(r, p->peeras, ip->raddress, as, ip->laddress);
		  }

	    // left over route
	    if (!interas_match) { 
	       Import(r, p->peeras, NULL, as, NULL);
	    }
	 }
      }
   }

   ListNodePix *lnp = r.aspath.tail();
   r.aspath.unlink(lnp->l);
   delete lnp;
}

inline int aspath_loop_check(Route &r, Pix as) {
   for (ListNodePix *p = r.aspath.head(); p; p = r.aspath.next(p->l))
      if (p->pix == as)
	 return TRUE;

   return FALSE;
}

inline void check_source_if_not_announce(Route &r, Pix as) {
   if (as == source_as) {
      for (ListNodePix *p = r.aspath.head(); p; p = r.aspath.next(p->l))
	 cout << AS_map(p->pix) << " ";
      cout << AS_map(as) << "\n";
   } else
      Announce(r, as);
}

void Import(Route &r, Pix as, Pix laddr, Pix ras, Pix raddr) {
   ASPolicy *p;
   Filter_Action *fap;
   Filter_Action *ifap;
   InterASPolicy *ip;

   if (!as || !AS_map.define(as))
      return;

   AutNum &autnum = *(AutNum *) AS_map.contents(as).definition;
   int match;

   if (aspath_loop_check(r, as)) // route has a loop
      return;

   if (r.aspath.size() >= path_length_limit) // path length limit reached
      return;

   if (!(p = autnum.find_peer(ras))) // this peer does not exist
      return;

   match = 0;
   for (fap = p->in.head(); fap; fap = p->in.next(fap->falist))
      if (fap->filter->match(r)) {
	 match = 1;
	 fap->action->Execute(r);
	 for (ip = p->interas.head(); ip; ip = p->interas.next(ip->interas))
	    for (ifap = ip->in.head(); ifap; ifap = ip->in.next(ifap->falist))
	       if (ifap->filter->match(r)) {
		  ifap->action->Execute(r);
		  match = 2;
		  check_source_if_not_announce(r, as);
	       }
	 if (match != 2) { // left over route
	    check_source_if_not_announce(r, as);
	 }
      }

   if (!match) { // check to see if we default
      for (fap = p->dflt.head(); fap; fap = p->dflt.next(fap->falist))
	 if (fap->filter->match(r)) {
	    fap->action->Execute(r);
	    check_source_if_not_announce(r, as);
	 }
   }
}


int shortest_path_length(register Pix source, register Pix sink) {
   Pix p;
   ASPolicy *peer;
   AutNum *this_autnum, *peer_autnum;

   /* this queue will never overflow */
   BoundedQueue<Pix> bfs_q(MAX_AS_COUNT); 

   for (p = AS_map.first(); p; AS_map.next(p))
      if (AS_map.contents(p).definition)
	 (AS_map.define(p))->mark = -1;

   (AS_map.define(source))->mark = 1;

   if (source == sink)
      return 1;

   bfs_q.enq(source);
   
   while (!bfs_q.empty()) {
      this_autnum = AS_map.define(bfs_q.deq());
      for (peer = this_autnum->peers.head(); peer; 
	   peer = this_autnum->peers.next(peer->peers)) {
	 if (peer->peeras == sink) // did we reach the sink
	    return this_autnum->mark + 1;

	 peer_autnum = AS_map.define(peer->peeras);
	 if (peer_autnum->mark == -1) {
	    peer_autnum->mark = this_autnum->mark + 1;
	    bfs_q.enq(peer->peeras);
	 }
      }
   }
}

