/* $Id: List.hh 1.4 Fri, 18 Jul 1997 16:00:26 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu <cengiz@isi.edu>
*/

#ifndef DLIST_H
#define DLIST_H

#ifndef NULL
#define NULL (void *) 0
#endif

class ListNode {
public:
   ListNode *forw;
   ListNode *back;
   void     *data;

public:
   ListNode(void * _data) {
      set_empty(); 
      data = _data;
   }
   ListNode(const ListNode& b)  {
      set_empty();
      data = b.data;
   }
   
   void set_empty() {
      forw = back = this;
   }

   void un_link() {
      forw->back = back;
      back->forw = forw;
      set_empty();
   }

   void link(ListNode& before, ListNode& after) {
      forw = &after;
      back = &before;
      before.forw = this;
      after.back = this;
   }
};

// doubly linked list implemented as a circular list 
// the listhead is also a listnode, except its data pointer points to NULL
// this assumes no non-listhead listnode's data pointer can point to NULL
template <class T>
class ListHead : public ListNode {
  private:
   int       length;

  public:
   ListHead() : ListNode(NULL) {
      length = 0;
   }

   ListHead(const ListHead& b) : ListNode(NULL) { 
      // copies the list b, including the data pointed to
      // the data pointed to is copied using its copy constructor
      // and the bits about this link list is changed explicitly.
      // make sure that the data can be correctly copied using a cconst.
      // Can be safely used if T contains only one ListNode in it

      // some ugly pointer arithmetic:
      // e - e->data : offset of ListNode in the class containing it
      // t + e - e->data : pointer to ListNode corresponding to this list in t
      T *t;
      for (ListNode *e = b.forw; e->data; e = e->forw) {
	 t = new T(* (T *) e->data);
	 ((ListNode *) ((char *) t + ((char *)e - (char *)e->data)))->data = t;
	 append(* (ListNode *) ((char *) t + ((char *) e - (char *) e->data)));
      }
   }

   // Added by wlee@isi.edu
   ~ListHead(void) { clear(); }

   ListHead& operator=(const ListHead& b) {
      // This routine blindly free's memory of old elements if any
      // This routine assumes that the elements in this are not accessible
      // via other means.
      // If this assumption does not hold for your system, do not use the = op.
      clear();

      // some ugly pointer arithmetic:
      // e - e->data : offset of ListNode in the class containing it
      // t + e - e->data : pointer to ListNode corresponding to this list in t
      T *t;
      for (ListNode *e = b.forw; e->data; e = e->forw) {
	 t = new T(* (T *) e->data);
	 ((ListNode *) ((char *) t + ((char *)e - (char *)e->data)))->data = t;
	 append(* (ListNode *) ((char *) t + ((char *) e - (char *) e->data)));
      }

      return *this;
   }

   int is_empty() const {
      return (forw == (ListNode *) this);
   }

   int size() const {
      return length;
   }

   void unlink(ListNode &t) {
      t.un_link();
      length--;
   }

   void unlink_all() {
      for (ListNode *e = forw; e->data; e = forw)
	 e->un_link();
      length = 0;
   }

   void clear() {
      for (ListNode *e = forw; e->data; e = forw) {
	 e->un_link();
	 delete (T *) e->data;
      }
      length = 0;
   }

   void append(ListNode &t) {
      insert_before(*this, t);
   }

   void prepend(ListNode &t) {
      insert_after(*this, t);
   }

   void insert_before(ListNode &e, ListNode &t) {
      t.link(*e.back, e);
      length++;
   }

   void insert_after(ListNode &e, ListNode &t) {
      t.link(e, *e.forw);
      length++;
   }

/*
 * Splices two lists together, whose list heads are given by this and l.
 * The resulting head is this, and l is returned as the empty list.
 */
   void splice(ListHead& l) {
      if (!l.is_empty()) {
        l.forw->back = back;
        back->forw = l.forw;
        l.back->forw = this;
        back = l.back;
	
	length += l.length;
	l.set_empty();
      }
   }

   // traversal routines
   T *head() const {
      return next(*this);
   }

   T *tail() const {
      return prev(*this);
   }

   T *next(const ListNode &n) const {
      return (T *) n.forw->data;
   }

   T *prev(const ListNode &n) const {
      return (T *) n.back->data;
   }
};

// sorted doubly linked list
// T must have the following two methods
// T.operator <
// T.operator ==
template <class T>
class SortedListHead : public ListHead<T> {
public:
   void append(ListNode &t) {
      insert_sorted(t);
   }

   void prepend(ListNode &t) {
      insert_sorted(t);
   }

   void insert_before(ListNode &e, ListNode &t) {
      insert_sorted(t);
   }

   void insert_after(ListNode &e, ListNode &t) {
      insert_sorted(t);
   }
     
   void insert_sorted(ListNode &t) {
      ListNode *e;
      for (e = forw; 
	   e->data && (*((T *) e->data) < *((T*) t.data));
	   e = e->forw)
	 ;

      if (e->data)
	 t.link(*e->back, *e);
      else
	 t.link(*this->back, *this);
   }

   // do a sorted merge
   void splice(SortedListHead& l) {
      ListNode *t;
      ListNode *e = forw;
      while (!l.is_empty()) {
	 t = l.forw;
	 t->un_link();
	 for (; 
	      e->data && (*((T *) e->data) < *((T *) t->data));
	      e = e->forw)
	    ;
	 if (e->data)
	    t->link(*e->back, *e);
	 else
	    t->link(*this->back, *this);
      }
   }

   // do a sorted merge
   void splice(ListHead<T>& l) {
      ListNode *t;
      ListNode *e = forw;
      while (!l.is_empty()) {
	 t = l.forw;
	 t->un_link();
	 insert_sorted(*t);
      }
   }

   T *find(const ListNode &t) const {
      for (ListNode *e = forw; e->data; e = e->forw)
	 if ((*((T *) e->data) == *((T*) t.data)))
	    return (T *) e->data;
	 else if (!((*((T *) e->data) < *((T*) t.data))))
	    return NULL;

      return NULL;
   }
};

#endif /* DLIST_H */



