/*
*   Author:   Scott Bailey
*   License:  BSD
*
*   Purpose:  Functions for working with arrays of periods, 
*   effectively sets of sets
*
*/

/***************************************************************************
*                          boolean functions
***************************************************************************/


-- true if p2 contained in any period in p2
CREATE OR REPLACE FUNCTION contains(
    period[],
    period
) RETURNS boolean AS
$$
    SELECT MAX(
       CASE WHEN contains(ts, $2) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($1) ts;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- timestamp is contained in any
CREATE OR REPLACE FUNCTION contains(
    period[],
    timestampTz
) RETURNS boolean AS
$$
    SELECT MAX(
      CASE WHEN contains(ts, $2) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($1) ts;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


CREATE OR REPLACE FUNCTION contained_by(
    period,
    period[]
) RETURNS boolean AS
$$
    SELECT MAX(
       CASE WHEN contained_by($1, ts) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($2) ts;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


CREATE OR REPLACE FUNCTION contained_by(
    timestampTz,
    period[]
) RETURNS boolean AS
$$
    SELECT MAX(
      CASE WHEN contained_by($1, ts) THEN 1 ELSE 0 END
    )::boolean
    FROM unnest($2) ts;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/***************************************************************************
*                             fact functions
***************************************************************************/


-- returns the smallest period that contains all values in period array
CREATE OR REPLACE FUNCTION range(period[])
RETURNS period AS
$$
    SELECT period(MIN(first(p)), MAX(next(p)))
    FROM unnest($1) p;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


/***************************************************************************
*                        period array functions
***************************************************************************/


-- reduce a period array by removing overlap and joining adjacent sets
CREATE OR REPLACE FUNCTION reduce(period[])
RETURNS period[] AS
$$
    SELECT array_agg(t)
    FROM (
        SELECT period(start_time, MIN(end_time)) AS t
        FROM (
            SELECT first(ts) AS start_time
            FROM unnest($1) ts
            WHERE NOT contains($1, prior(ts))
        ) AS t_in
        JOIN (
            SELECT next(ts) AS end_time
            FROM unnest($1) ts
            WHERE NOT contains($1, next(ts))
        ) AS t_out ON t_in.start_time < t_out.end_time
        GROUP BY t_in.start_time
        ORDER BY t_in.start_time
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


-- returns all values in p_include that do not intersect p_exclude
CREATE OR REPLACE FUNCTION period_except(
   p_include  IN period[], 
   p_exclude  IN period[]
) RETURNS period[] AS
$$
    SELECT array_agg(p)
    FROM (
        SELECT period((t_in).start_time,
            MIN((t_out).end_time)) AS p
        FROM (
            SELECT first(ts) AS start_time
            FROM unnest($1) ts
            WHERE NOT contains($2, first(ts))
            AND NOT contains($1, prior(ts))

            UNION ALL

            SELECT next(ts)
            FROM unnest($2) ts
            WHERE contains($1, next(ts))
            AND NOT contains($2, next(ts))
        ) t_in
        JOIN (
            SELECT next(ts) AS end_time
            FROM unnest($1) ts
            WHERE NOT contains($1, next(ts))

            UNION ALL

            SELECT first(ts)
            FROM unnest($2) ts
            WHERE contains($1, first(ts))
              AND NOT contains($2, prior(ts))
        ) t_out ON t_in.start_time < t_out.end_time
        GROUP BY (t_in).start_time
        ORDER BY (t_in).start_time
    ) sub;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;


--
CREATE OR REPLACE FUNCTION period_draw(
    p1     period[],
    ctx    period,
    width  int
) RETURNS SETOF text AS
$$
    SELECT period_draw(p, $2, $3)
    FROM unnest($1) p
$$ LANGUAGE 'sql' IMMUTABLE;

/*****************************************************************
*                         Boolean Operators
*****************************************************************/

CREATE OPERATOR @>(
  PROCEDURE = contains,
  LEFTARG = period[],
  RIGHTARG = period
);

CREATE OPERATOR @>(
  PROCEDURE = contains,
  LEFTARG = period[],
  RIGHTARG = timestampTz
);

CREATE OPERATOR <@(
  PROCEDURE = contained_by,
  LEFTARG = period,
  RIGHTARG = period[]
);

CREATE OPERATOR <@(
  PROCEDURE = contained_by,
  LEFTARG = timestampTz,
  RIGHTARG = period[]
);