/* $Id: array.c,v 1.7 2006/04/02 20:18:30 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Array base type
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_namespace.h>
#include <catalog/pg_conversion.h>
#include <catalog/pg_operator.h>
#include <catalog/pg_opclass.h>
#include <catalog/namespace.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/relcache.h>
#include <utils/typcache.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/array.h>
#include <pypg/type/textual.h>
#include <pypg/type/bitwise.h>
#include <pypg/error.h>

static int *
Indexes_FromPySequence(PyObj seq, int seqlen)
{
	int *indexes;
	unsigned int i;

	indexes = palloc(seqlen * sizeof(int));
	for (i = 0; i < seqlen; ++i)
	{
		PyObj ob, nob;
		ob = PySequence_GetItem(seq, i);
		if (ob == NULL) goto error;
		nob = PyNumber_Int(ob);
		Py_DECREF(ob);
		if (nob == NULL) goto error;

		indexes[i] = (int) PyInt_AsLong(nob) + 1;
		Py_DECREF(nob);
	}

	return(indexes);
error:
	pfree(indexes);
	return(NULL);
}

static int *
Indexes_FromPyObject(PyObj ob, int *len)
{
	int *indexes;

	if (PySequence_Check(ob))
	{
		*len = PySequence_Length(ob);
		indexes = Indexes_FromPySequence(ob, *len);
	}
	else
	{
		PyObj nob;
		long l;
		nob = PyNumber_Int(ob);
		if (nob == NULL) return(NULL);
		l = PyInt_AsLong(nob);
		Py_DECREF(nob);
		*len = 1;
		indexes = palloc(sizeof(int));
		indexes[0] = (int) l + 1;
	}

	return(indexes);
}

int *
Dimensions_FromPySequence(PyObj seq, int *ndim)
{
	int *dims;

	*ndim = 1;
	dims = palloc(sizeof(int) * (*ndim));
	dims[0] = PySequence_Length(seq);

	if (PyErr_Occurred())
	{
		pfree(dims);
		*ndim = 0;
		return(NULL);
	}
	else if (dims[0] == 0)
	{
		return(dims);
	}

	seq = PySequence_GetItem(seq, 0);
	while(PyTuple_Check(seq) || PyList_Check(seq))
	{
		PyObj ns;
		int len;

		len = PySequence_Length(seq);
		++(*ndim);
		dims = repalloc(dims, sizeof(int) * (*ndim));
		dims[*ndim-1] = len;
		if (len == 0)
		{
			break;
		}

		ns = PySequence_GetItem(seq, 0);
		Py_DECREF(seq);
		seq = ns;
	}
	Py_DECREF(seq);

	return(dims);
}

static int
fill_elements
(
	PyObj elm,
	PyObj source,

	int ndims,
	int *dims,

	unsigned long *i,
	int axis,

	PyObj *datumobs,
	Datum *datums
#if PG_FEATURE_ARRAYNULLS
	, bool *nulls
#endif
)
{
	int len = PySequence_Length(source);

	if (len != dims[axis])
	{
		PyErr_Format(PyExc_TypeError, "array inbalance in axis %d", axis);
		return(-1);
	}

	if ((axis + 1) == ndims)
	{
		unsigned long ii;

		for (ii = 0; ii < len; ++ii, ++(*i))
		{
			PyObj ob;
			ob = PySequence_GetItem(source, ii);
			if (ob == NULL) return(-1);

#if PG_FEATURE_ARRAYNULLS
			if (ob == Py_None)
			{
				datumobs[*i] = ob;
				datums[*i] = 0;
				nulls[*i] = true;
			}
			else
			{
				nulls[*i] = false;

				if (elm != (PyObj) &PyPg_bool_Type)
				{
					ob = datumobs[*i] = Py_Call(elm, ob);
					if (ob == NULL) return(-1);
					datums[*i] = PyPgObject_FetchDatum(datumobs[*i]);
				}
				else
				{
					ob = datumobs[*i] = ob;
					datums[*i] = PyObject_IsTrue(ob) ? 1 : 0;
				}
			}
#else
			if (elm != (PyObj) &PyPg_bool_Type)
			{
				ob = datumobs[*i] = Py_Call(elm, ob);
				if (ob == NULL) return(-1);
				datums[*i] = PyPgObject_FetchDatum(datumobs[*i]);
			}
			else
			{
				ob = datumobs[*i] = ob;
				datums[*i] = PyObject_IsTrue(ob) ? 1 : 0;
			}
#endif
		}
	}
	else
	{
		int ii;
		for (ii = 0; ii < len; ++ii)
		{
			PyObj src;

			src = PySequence_GetItem(source, ii);
			if (fill_elements(
				elm, src, ndims, dims, i, axis+1, datumobs, datums
#if PG_FEATURE_ARRAYNULLS
				, nulls
#endif
			) < 0)
				return(-1);
		}
	}

	return(0);
}

ArrayType *
Array_FromPySequence(PyObj source, PyObj elm)
{
	TypeCacheEntry *etc = PyPgType_FetchTypeCache(elm);
	unsigned long i, nelems;
	Datum *datums;
	PyObj *datumobs;
#if PG_FEATURE_ARRAYNULLS
	bool *nulls;
#endif
	int ndims;
	int *dims, *lbs;
	ArrayType *rat = NULL;

	dims = Dimensions_FromPySequence(source, &ndims);
	if (dims == NULL) return(NULL);

	for (i = 0, nelems = 1; i < ndims; ++i)
	{
		nelems *= dims[i];
	}
	datums = palloc(sizeof(Datum) * nelems);
	datumobs = palloc0(sizeof(PyObject *) * nelems);
#if PG_FEATURE_ARRAYNULLS
	nulls = palloc0(sizeof(bool) * nelems);
#endif

	i = 0;
	if (fill_elements(
		elm, source, ndims, dims, &i, 0, datumobs, datums
#if PG_FEATURE_ARRAYNULLS
		,nulls
#endif
	) < 0)
	{
		pfree(dims);
		pfree(datums);
		if (datumobs[0] != NULL)
			for (i = 0; datumobs[i] != NULL; ++i)
				Py_DECREF(datumobs[i]);
		pfree(datumobs);
		return(NULL);
	}
	Assert(i == nelems);

	lbs = palloc(sizeof(int) * ndims);
	for (i = 0; i < ndims; ++i) lbs[i] = 1;

	PgError_TRAP(
		rat = construct_md_array(datums,
#if PG_FEATURE_ARRAYNULLS
			nulls,
#endif
			ndims, dims, lbs,
			etc->type_id, etc->typlen, etc->typbyval, etc->typalign);
	);

#if PG_FEATURE_ARRAYNULLS
	pfree(nulls);
#endif
	pfree(datums);
	for (i = 0; i < nelems; ++i)
	{
		Py_DECREF(datumobs[i]);
	}
	pfree(datumobs);

	return(rat);
}

int
array_length(PyObj self)
{
	return(ARR_DIMS(PyPgObject_FetchDatum(self))[0]);
}

PyObj
array_item(PyObj self, int item)
{
	ArrayType *at;
	bool isnull = false;
	PyObj elm;
	TypeCacheEntry *etc;
	PyObj rob = NULL;

	elm = PyPgType_FetchElementType(self->ob_type);
	Assert(elm != NULL);

	etc = PyPgType_FetchTypeCache(elm);
	Assert(etc != NULL);

	at = DatumGetArrayTypeP(PyPgObject_FetchDatum(self));
	Assert(at != NULL);

	++item;

	if (item > ARR_DIMS(at)[0])
	{
		PyErr_Format(PyExc_IndexError, "index %d out of range %d",
			item-1, ARR_DIMS(at)[0]-1);
		return(NULL);
	}

	if (ARR_NDIM(at) == 1)
	{
		Datum rd = 0;

		PgError_TRAP(
			rd = array_ref(at, 1, &item, -1,
				etc->typlen, etc->typbyval, etc->typalign, &isnull)
		);
		if (PyErr_Occurred()) return(NULL);

		if (isnull)
		{
			rob = Py_None;
			Py_INCREF(rob);
		}
		else
			rob = PyPgObject_New(elm, rd);
	}
	else
	{
		rob = Py_None;
		Py_INCREF(rob);
	}

	return(rob);
}

PyObj
array_slice(PyObj self, int from, int to)
{
	PyObj elm;
	TypeCacheEntry *etc;
	ArrayType *at, *rat = NULL;
	bool isnull = false;
	PyObj rob = NULL;

	elm = PyPgType_FetchElementType(self->ob_type);
	Assert(elm != NULL);

	etc = PyPgType_FetchTypeCache(elm);
	Assert(etc != NULL);

	at = DatumGetArrayTypeP(PyPgObject_FetchDatum(self));
	Assert(at != NULL);

	++to;
	++from;
	PgError_TRAP(
		rat = array_get_slice(at, 1, &to, &from, -1, etc->typlen,
			etc->typbyval, etc->typalign, &isnull)
	);

	if (PyErr_Occurred())
	{
		;
	}
	else if (isnull)
	{
		Assert(rat == NULL);
		rob = Py_None;
		Py_INCREF(rob);
	}
	else
	{
		rob = PyPgObject_New(self->ob_type, PointerGetDatum(rat));
	}

	return(rob);
}

static PySequenceMethods array_as_sequence = {
	array_length,	/* sq_length */
	NULL,				/* sq_concat */
	NULL,				/* sq_repeat */
	array_item,		/* sq_item */
	array_slice,	/* sq_slice */
	NULL,				/* sq_ass_item */
	NULL,				/* sq_ass_slice */
	NULL,				/* sq_contains */
	NULL,				/* sq_inplace_concat */
	NULL,				/* sq_inplace_repeat */
};

PyObj
array_subscript(PyObj self, PyObj key)
{
	PyObj rob = NULL;
	ArrayType *at;
	PyObj elm;
	TypeCacheEntry *etc;

	elm = PyPgType_FetchElementType(self->ob_type);
	Assert(elm != NULL);

	etc = PyPgType_FetchTypeCache(elm);
	Assert(etc != NULL);

	at = DatumGetArrayTypeP(PyPgObject_FetchDatum(self));
	Assert(at != NULL);

	if (PySlice_Check(key))
	{
		PySliceObject *slice = (PySliceObject *) key;
		int *from_indexes, *to_indexes;
		int from_dims, to_dims;
		ArrayType *rat = NULL;
		bool isnull = false;

		from_indexes = Indexes_FromPyObject(slice->start, &from_dims);
		if (from_indexes == NULL) return(NULL);

		to_indexes = Indexes_FromPyObject(slice->stop, &to_dims);
		if (to_indexes == NULL) return(NULL);

		if (from_dims != to_dims)
		{
			pfree(from_indexes);
			pfree(to_indexes);
			PyErr_SetString(PyExc_TypeError, "inequal slice indexes for array");
			return(NULL);
		}
		PgError_TRAP(
			rat = array_get_slice(at, to_dims, from_indexes, from_indexes, -1,
				etc->typlen, etc->typbyval, etc->typalign, &isnull)
		);
		pfree(from_indexes);
		pfree(to_indexes);

		if (PyErr_Occurred())
		{
			;
		}
		else if (isnull)
		{
			Assert(rat == NULL);
			rob = Py_None;
			Py_INCREF(rob);
		}
		else
		{
			rob = PyPgObject_New(self->ob_type, PointerGetDatum(rat));
			pfree(rat);
		}
	}
	else
	{
		int *indexes;
		int dims, i;
		bool isnull;
		Datum rd = 0;

		indexes = Indexes_FromPyObject(key, &dims);
		if (indexes == NULL) return(NULL);

		for (i = 0; i < dims; ++i)
		{
			if (indexes[i] - 1 >= ARR_DIMS(at)[i])
			{
				PyErr_Format(PyExc_IndexError, "index(%d) %d out of range %d",
					i + 1, indexes[i], ARR_DIMS(at)[i]-1);
				pfree(indexes);
				return(NULL);
			}
		}

		PgError_TRAP(
			rd = array_ref(at, dims, indexes, -1,
				etc->typlen, etc->typbyval, etc->typalign, &isnull)
		);
		pfree(indexes);

		if (PyErr_Occurred())
		{
			;
		}
		else if (isnull)
		{
			rob = Py_None;
			Py_INCREF(rob);
		}
		else
		{
			rob = PyPgObject_New(elm, rd);
		}
	}

	return(rob);
}

static PyMappingMethods array_as_mapping = {
	array_length,		/* mp_length */
	array_subscript,	/* mp_subscript */
};

static PyObj
array_get_lowerbound(PyObj self, void *closure)
{
	ArrayType *at;
	PyObj rob;
	int i, ndim, *lbs;

	at = DatumGetArrayTypeP(PyPgObject_FetchDatum(self));
	ndim = ARR_NDIM(at);
	lbs = ARR_LBOUND(at);
	rob = PyTuple_New(ndim);

	for (i = 0; i < ndim; ++i)
	{
		PyObj ob;
		ob = PyInt_FromLong(lbs[i]);
		if (ob == NULL)
		{
			Py_DECREF(rob);
			return(NULL);
		}
		PyTuple_SET_ITEM(rob, i, ob);
	}

	return(rob);
}

static PyObj
array_get_dimension(PyObj self, void *closure)
{
	ArrayType *at;
	PyObj rob;
	int i, ndim, *dims;

	at = DatumGetArrayTypeP(PyPgObject_FetchDatum(self));
	ndim = ARR_NDIM(at);
	dims = ARR_DIMS(at);
	rob = PyTuple_New(ndim);

	for (i = 0; i < ndim; ++i)
	{
		PyObj ob;
		ob = PyInt_FromLong(dims[i]);
		if (ob == NULL)
		{
			Py_DECREF(rob);
			return(NULL);
		}
		PyTuple_SET_ITEM(rob, i, ob);
	}

	return(rob);
}

static PyObj
array_get_dimensions(PyObj self, void *closure)
{
	ArrayType *at;
	PyObj rob;

	at = DatumGetArrayTypeP(PyPgObject_FetchDatum(self));
	rob = PyInt_FromLong(ARR_NDIM(at));

	return(rob);
}

static PyGetSetDef array_getset[] = {
	{"lowerbound", array_get_lowerbound, NULL,
	PyDoc_STR("The array's lower bounds")},
	{"dimension", array_get_dimension, NULL,
	PyDoc_STR("The array's dimensions")},

	{"dimensions", array_get_dimensions, NULL,
	PyDoc_STR("The number of array dimensions")},
	{NULL}
};

static PyObj
array_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj source, rob;

	if (!PyObject_IsSubclass((PyObj) subtype, (PyObj) &PyPgArray_Type))
	{
		PyErr_Format(PyExc_TypeError,
			"array.__new__ requires a Postgres.Array subtype "
			"as the first argument, given '%s'", subtype->ob_type->tp_name);
		return(NULL);
	}

	if (!PyArg_ParseTuple(args, "O", &source))
		return(NULL);

	if (subtype == source->ob_type)
	{
		Py_INCREF(source);
		return(source);
	}

	if (PySequence_Check(source)
		&& !PyString_Check(source)
		&& !PyPg_text_Check(source))
	{
		PyObj elm;
		ArrayType *ar;

		elm = PyPgType_FetchElementType(subtype);

		ar = Array_FromPySequence(source, elm);
		if (ar == NULL) return(NULL);
		rob = PyPgObject_New(subtype, PointerGetDatum(ar));
		pfree(ar);
	}
	else
	{
		rob = PyPgObject_typinput((PyObj) subtype, source);
	}

	return(rob);
}

PyDoc_STRVAR(PyPgArray_Type_Doc, "Base type for arbitrary object types");

PyPgTypeObject PyPgArray_Type = {{
	PyObject_HEAD_INIT(&PyPgType_Type)
	0,												/* ob_size */
	"Array",										/* tp_name */
	sizeof(struct PyPgObject),				/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	&array_as_sequence,						/* tp_as_sequence */
	&array_as_mapping,						/* tp_as_mapping */
	NULL,											/* tp_hash */
	NULL,											/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	PyPg_TPFLAGS_DEFAULT|
	Py_TPFLAGS_BASETYPE,						/* tp_flags */
	PyPgArray_Type_Doc,						/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	array_getset,								/* tp_getset */
	(PyTypeObject *) &PyPgObject_Type,	/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	array_new,									/* tp_new */
},
};

/*
 * vim: ts=3:sw=3:noet:
 */
