/* $Id: module.c,v 1.29 2006/04/02 22:29:18 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * The Postgres interface module implementation
 */
#include <setjmp.h>
#include <postgres.h>
#include <funcapi.h>
#include <miscadmin.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/hio.h>
#include <access/xact.h>
#include <catalog/namespace.h>
#include <catalog/pg_namespace.h>
#include <catalog/catversion.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <commands/async.h>
#include <commands/trigger.h>
#include <libpq/libpq.h>
#include <libpq/pqformat.h>
#include <mb/pg_wchar.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/elog.h>
#include <utils/syscache.h>
#include <utils/memutils.h>
#include <utils/tuplestore.h>
#include <utils/rel.h>
#include <utils/relcache.h>
#include <utils/typcache.h>
#include <storage/backendid.h>
#include <storage/large_object.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>

#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/array.h>
#include <pypg/type/record.h>
#include <pypg/type/bitwise.h>
#include <pypg/type/numeric.h>
#include <pypg/type/geometric.h>
#include <pypg/type/textual.h>
#include <pypg/type/timewise.h>
#include <pypg/type/network.h>
#include <pypg/type/system.h>

#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/function.h>
#include <pypg/query.h>
#include <pypg/call.h>
#include <pypg/call/function.h>
#include <pypg/call/pl.h>
#include <pypg/call/trigger.h>
#include <pypg/call/portal.h>
#include <pypg/error.h>
#include <pypg/encoding.h>
#include <pypg/cis.h>

PyObj
Py_RETURN_SELF(PyObj self, ...)
{
	Py_INCREF(self);
	return(self);
}

#define PYEREPORTIMP(ereport_LEVEL_)	\
	char *msg = NULL;							\
	char *hint = NULL;						\
	char *detail = NULL;						\
	char *context = NULL;					\
	int code = 0;								\
													\
	static char *kwlist[] = {				\
		"msg", "hint", "detail",			\
		"context", "code", NULL				\
	};												\
													\
	if (!PyArg_ParseTupleAndKeywords(	\
			args, kw, "s|sssi", kwlist,	\
			&msg, &hint, &detail,			\
			&context, &code))					\
		return(NULL);							\
													\
	if (errstart(ereport_LEVEL_,			\
			"Python", 0, "<wrapper>"))		\
	{												\
		errmsg("%s", msg);					\
		if (hint)								\
			errhint("%s", hint);				\
		if (detail)								\
			errdetail("%s", detail);		\
		if (context)							\
			errcontext("%s", context);		\
		if (ereport_LEVEL_ < ERROR)		\
			code = 0;							\
		errcode(code);							\
		errfinish(0);							\
	}

#define PyPgDefineEReportFunction(LEVEL) \
PyDoc_STRVAR(pypg_##LEVEL##_Doc, "ereport at the " #LEVEL " level"); \
static PyObj pypg_##LEVEL(PyObj self, PyObj args, PyObj kw) \
{ PYEREPORTIMP(LEVEL); RETURN_NONE; }

PyPgDefineEReportFunction(WARNING);
PyPgDefineEReportFunction(NOTICE);
PyPgDefineEReportFunction(INFO);
PyPgDefineEReportFunction(LOG);
PyPgDefineEReportFunction(DEBUG1);

PyDoc_STRVAR(pypg_ERROR_Doc, "ereport at the ERROR level");
static PyObj pypg_ERROR(PyObj self, PyObj args, PyObj kw)
{
	PG_TRY();
	{
		PYEREPORTIMP(ERROR);
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();

	return(NULL);
}
#undef PyPgDefineEReportFunction
#undef PYEREPORTIMP

static PyObj
pypg_read(PyObj self, PyObj amt)
{
	int len, left_len;
	void *buf;
	const char *strsrc;
	PyObj rob;

	len = PyInt_AsLong(amt);
	if (len == 0)
		return(PyString_FromString(""));

	buf = malloc(len);
	if (buf == NULL)
	{
		PyErr_SetString(PyExc_MemoryError,
			"could not allocate buffer for raw read");
		return(NULL);
	}

	strsrc = (const char *) buf;
	left_len = len;
	do
	{
		ssize_t rlen;

		do
		{
			rlen = secure_read(MyProcPort, buf, left_len);
		}
		while (errno == EINTR);

		if (rlen < 0)
		{
			PgError_TRAP(
				ereport(COMMERROR,
					(errcode_for_socket_access(),
					 errmsg("could not read data from client: %m")))
			);
			free((void *) strsrc);
			return(NULL);
		}

		if (rlen == 0)
		{
			free((void *) strsrc);
			PyErr_SetString(PyExc_IOError, "EOF detected on read");
			return(NULL);
		}

		buf += rlen;
		left_len -= rlen;
	}
	while (left_len > 0);

	rob = PyString_FromStringAndSize(strsrc, len);
	free((void *) strsrc);

	return(rob);
}

static PyObj
pypg_write(PyObj self, PyObj data)
{
	static int last_reported_send_errno = 0;

	PyObj obstr;
	void *buf;
	int len;

	obstr = PyObject_Str(data);
	if (obstr == NULL)
		return(NULL);

	len = PyString_Size(obstr);
	if (len == 0)
	{
		Py_DECREF(obstr);
		Py_INCREF(Py_None);
		return(Py_None);
	}

	buf = PyString_AS_STRING(obstr);
	do
	{
		ssize_t wlen;

		do
		{
			wlen = secure_write(MyProcPort, buf, len);
		}
		while (errno == EINTR);

		if (wlen <= 0)
		{
			if (errno != last_reported_send_errno)
			{
				last_reported_send_errno = errno;
				PgError_TRAP(
					ereport(COMMERROR,
						(errcode_for_socket_access(),
						 errmsg("could not send data to client: %m")))
				);
			}
			else
			{
				PyErr_SetString(PyExc_IOError, "EOF detected on write");
			}

			Py_DECREF(obstr);
			return(NULL);
		}

		buf += wlen;
		len -= wlen;
		last_reported_send_errno = 0;
	}
	while (len > 0);
	Py_DECREF(obstr);

	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObj
pypg_execute(PyObj self, PyObj qstr)
{
	PyObj query, rob;

	query = PyPgQuery_New(qstr);
	if (query == NULL) return(NULL);

	rob = Py_Call(query);
	Py_DECREF(query);

	return(rob);
}

static PyObj
pypg_xid(void)
{
	PyObj rob;
	Datum d = TransactionIdGetDatum(GetCurrentTransactionId());
	rob = PyPg_xid_FromTransactionId(d);
	return(rob);
}

static PyObj
pypg_sxid(void)
{
	PyObj rob;
	Datum d = TransactionIdGetDatum(GetCurrentSubTransactionId());
	rob = PyPg_xid_FromTransactionId(d);
	return(rob);
}

static PyObj
pypg_cid(void)
{
	return(PyPg_cid_FromCommandId(GetCurrentCommandId()));
}

static PyObj
pypg_uid(void)
{
	return(PyPg_oid_FromObjectId(GetUserId()));
}

static PyObj
pypg_sesuid(void)
{
	return(PyPg_oid_FromObjectId(GetSessionUserId()));
}

static PyObj
pypg_encoding(void)
{
	const char *es;
	PyObj rob;
	es = PyEncoding_FromPgEncoding(GetDatabaseEncoding());
	if (es != NULL)
		rob = PyString_InternFromString(es);
	else
	{
		rob = Py_None;
		Py_INCREF(rob);
	}
	return(rob);
}

static PyObj
pypg_encoding_code(void)
{
	PyObj rob;
	rob = PyInt_FromLong(GetDatabaseEncoding());
	return(rob);
}

static PyObj
pypg_notify(PyObj self, PyObj name)
{
	if (PyString_Check(name))
		Async_Notify(PyString_AS_STRING(name));
	else
	{
		PyErr_Format(PyExc_TypeError,
			"Postgres.Notify requires a %s object, given %s",
			PyString_Type.tp_name, name->ob_type->tp_name);
		return(NULL);
	}
	RETURN_NONE;
}

static PyObj
pypg_listen(PyObj self, PyObj name)
{
	if (PyString_Check(name))
	{
		PG_TRY();
		{
			Async_Listen(PyString_AS_STRING(name));
		}
		PYPG_CATCH_END(return(NULL));
	}
	else
	{
		PyErr_Format(PyExc_TypeError,
			"Postgres.Listen requires a %s object, given %s",
			PyString_Type.tp_name, name->ob_type->tp_name);
		return(NULL);
	}
	RETURN_NONE;
}

static PyObj
pypg_unlisten(PyObj self, PyObj name)
{
	if (PyString_Check(name))
	{
		PG_TRY();
		{
			Async_Unlisten(PyString_AS_STRING(name));
		}
		PYPG_CATCH_END(return(NULL));
	}
	else
	{
		PyErr_Format(PyExc_TypeError,
			"Postgres.Unlisten requires a %s object, given %s",
			PyString_Type.tp_name, name->ob_type->tp_name);
		return(NULL);
	}
	RETURN_NONE;
}

/* nsoid = LookupExplicitNamespace(nsns); */
/* nsoid = NamespaceCreate(nsns, GetUserId()); */

static PyObj
pypg_transact(PyObj self, PyObj callable)
{
	PyObj rob;
	BeginInternalSubTransaction(NULL);
	rob = Py_Call(callable);

	if (rob == NULL)
		RollbackAndReleaseCurrentSubTransaction();
	else
		ReleaseCurrentSubTransaction();

	return(rob);
}

static PyMethodDef PyPgModule_Methods[] = {
	{"ERROR", (PyCFunction) pypg_ERROR,
		METH_VARARGS|METH_KEYWORDS, pypg_ERROR_Doc},
	{"WARNING", (PyCFunction) pypg_WARNING,
		METH_VARARGS|METH_KEYWORDS, pypg_WARNING_Doc},
	{"NOTICE", (PyCFunction) pypg_NOTICE,
		METH_VARARGS|METH_KEYWORDS, pypg_NOTICE_Doc},
	{"INFO", (PyCFunction) pypg_INFO,
		METH_VARARGS|METH_KEYWORDS, pypg_INFO_Doc},
	{"LOG", (PyCFunction) pypg_LOG,
		METH_VARARGS|METH_KEYWORDS, pypg_LOG_Doc},
	{"DEBUG", (PyCFunction) pypg_DEBUG1,
		METH_VARARGS|METH_KEYWORDS, pypg_DEBUG1_Doc},

	{"_read", (PyCFunction) pypg_read, METH_O,
	PyDoc_STR("read raw data from the socket")},
	{"_write", (PyCFunction) pypg_write, METH_O,
	PyDoc_STR("write raw data to the socket")},

	{"Execute", (PyCFunction) pypg_execute, METH_O,
	PyDoc_STR("run bulk SQL statements")},

	{"Xid", (PyCFunction) pypg_xid, METH_NOARGS,
	PyDoc_STR("get the current TransactionId")},
	{"SXid", (PyCFunction) pypg_sxid, METH_NOARGS,
	PyDoc_STR("get the current SubTransactionId")},
	{"Cid", (PyCFunction) pypg_cid, METH_NOARGS,
	PyDoc_STR("get the current CommandId")},
	{"Uid", (PyCFunction) pypg_uid, METH_NOARGS,
	PyDoc_STR("get the current UserId")},
	{"SessionUid", (PyCFunction) pypg_sesuid, METH_NOARGS,
	PyDoc_STR("get the current SessionUserId")},

	{"Encoding", (PyCFunction) pypg_encoding, METH_NOARGS,
	PyDoc_STR("get the database's encoding as a Python encoding name")},
	{"EncodingCode", (PyCFunction) pypg_encoding_code, METH_NOARGS,
	PyDoc_STR("the database's encoding code")},

	{"Notify", (PyCFunction) pypg_notify, METH_O,
	PyDoc_STR("notify the specified channel")},
	{"Listen", (PyCFunction) pypg_listen, METH_O,
	PyDoc_STR("make the backend listen on a channel")},
	{"Unlisten", (PyCFunction) pypg_unlisten, METH_O,
	PyDoc_STR("stop the backend from listening on a channel")},

	{"Transact", (PyCFunction) pypg_transact, METH_O,
	PyDoc_STR("call an object within a new transaction")},
	{NULL}
};

/*
 * C Interface initialization
 */
struct PyPgCI_Root PyPgCI = {

/* PL */
	{
		NULL,		/* Handler */
		NULL,		/* CodeFromData */
	},

/* Error */
	NULL,
	PyExc_FromERRCODE,
	PyErr_FromErrorData,
	PyErr_SetPgError,

/* Encoding */
	PyEncoding_FromPgEncoding,

/* Abstract Types */
	&PyPgType_Type,
	&PyPgObject_Type,
	&PyPgPseudo_Type,
	&PyPgArbitrary_Type,
	&PyPgArray_Type,

/* TupleDesc */
	&PyPgTupleDesc_Type,
	NULL,
	PyPgTupleDesc_NEW,
	PyPgTupleDesc_FromRelationId,

/* HeapTuple */
	&PyPgHeapTuple_Type,
	NULL,
	PyPgHeapTuple_NEW,
	HeapTuple_FromTupleDescAndDatumNulls,

/* Function */
	&PyPgFunction_Type,
	PyPgFunction_FromOid,
	PyPgFunction_DelCached,

/* Query */
	&PyPgQuery_Type,
	PyPgQuery_NEW,

/* Call */
	&PyPgCall_Type,
	PyPgCall_Initialize,

/* FunctionCall */
	&PyPgFunctionCall_Type,
	PyPgFunctionCall_Initialize,

/* Portal */
	&PyPgPortal_Type,
	PyPgPortal_Initialize,

/* ProceduralCall */
	&PyPgProceduralCall_Type,
	PyPgProceduralCall_Initialize,

/* TriggerPull */
	&PyPgTriggerPull_Type,
	PyPgTriggerPull_Initialize,

	PyPgType_FromOid,
	PyPgType_FromPyObject,

	PyPgObject_Initialize,
	PyPgObject_FromPyPgTypeAndDatum,
	PyPgObject_FromTypeOidAndDatum,
	PyPgObject_FromPyPgTypeAndHeapTuple,

#define ACTION(NAME, ID) &PyPgTypeID(NAME),
	PyPgType_Builtins()
#undef ACTION
};

PyDoc_STRVAR(PyPgModule_doc, "Postgres interface extension module");
extern PyMODINIT_FUNC
init_prime(void)
{
	PyObj mod, ob, md;
	PythonMemoryContext = TopMemoryContext;

	PG_TRY();
	{
		PythonWorkMemoryContext = AllocSetContextCreate(PythonMemoryContext,
			"Python Work Memory",
			ALLOCSET_DEFAULT_MINSIZE,
			ALLOCSET_DEFAULT_INITSIZE,
			ALLOCSET_DEFAULT_MAXSIZE
		);
	}
	PYPG_CATCH_END();
	if (PyErr_Occurred()) return;

	md = PyImport_GetModuleDict();
	if (md == NULL) return;

	mod = Py_InitModule3("_prime", PyPgModule_Methods, PyPgModule_doc);
	if (mod == NULL) return;

	if (PyDict_SetItemString(md, "_Postgres", mod) < 0)
	{
		DECREF(mod);
		return;
	}

	/* Version information */
	{
		PyObj vi, v, major, minor, patch, state, state_level;
		major = PyInt_FromLong(PG_VERSION_MAJOR);
		minor = PyInt_FromLong(PG_VERSION_MINOR);
		patch = PyInt_FromLong(PG_VERSION_PATCH);
		state = PyString_FromString(PG_VERSION_STATE);
		state_level = PyInt_FromLong(PG_VERSION_LEVEL);

		vi = PyTuple_New(5);
		PyTuple_SET_ITEM(vi, 0, major);
		PyTuple_SET_ITEM(vi, 1, minor);
		PyTuple_SET_ITEM(vi, 2, patch);
		PyTuple_SET_ITEM(vi, 3, state);
		PyTuple_SET_ITEM(vi, 4, state_level);
		PyModule_AddObject(mod, "version_info", vi);

		v = PyString_FromString(PG_VERSION_STR);
		PyModule_AddObject(mod, "version", v);
	}

	if (MyProcPort != NULL)
	{
		if (MyProcPort->remote_host != NULL)
		{
			PyObj client_host;
			client_host = PyString_FromString(MyProcPort->remote_host);
			PyModule_AddObject(mod, "client_host", client_host);
		}

		if (MyProcPort->remote_port != NULL)
		{
			PyObj client_port;
			client_port = PyString_FromString(MyProcPort->remote_port);
			PyModule_AddObject(mod, "client_port", client_port);
		}
	}

	ob = PyCObject_FromVoidPtr(&PyPgCI, NULL);
		if (ob == NULL) return;
	if (PyModule_AddObject(mod, PyPgCI_CObjectName, ob) < 0)
	{
		Py_DECREF(ob);
		return;
	}

	/*
	 * This extension module extends it's pure python interface module,
	 * which should already be imported, so fetch the PP module's dict and
	 * fetch some attributes.
	 */
	ob = PyDict_GetItemString(md, "Postgres");
	if (ob == NULL) return;
	Postgres_PyModule = ob;

	ob = PyImport_AddModule("__main__");
	if (ob == NULL) return;
	__main__ = PyModule_GetDict(ob);

	ob = PyPgError_Initialize();
	if (ob == NULL) return;
	PyPgCI._PyExc_PostgresError = ob;
	PyPgEncoding_Initialize();
	if (PyPgFunction_Initialize() < 0) return;

	ob = PyDict_New();
	if (ob == NULL) return;
#	define TARGET ob
#	define APFUNC PyMapping_SetItemString
#		include "constants.c"
#	undef APFUNC
#	undef TARGET
	PyModule_AddObject(mod, "Const", ob);
#define READY(TYPE) (PyType_Ready(&PyPg##TYPE##_Type) < 0)
	if (READY(Function)
		|| READY(Call) || READY(FunctionCall)
		|| READY(ProceduralCall) || READY(TriggerPull)
		|| READY(TupleDesc) || READY(HeapTuple)
		|| READY(Query) || READY(Portal)
	)
		return;
#undef READY
	if (EmptyPyPgTupleDesc_Initialize() < 0) return;
	PyPgCI._EmptyPyPgTupleDesc = EmptyPyPgTupleDesc;

	if (EmptyPyPgHeapTuple_Initialize() < 0) return;
	PyPgCI._EmptyPyPgHeapTuple = EmptyPyPgHeapTuple;

	if (PyPgType_Initialize() < 0) return;

#define ADD(BASE) PyModule_AddObject(mod, #BASE, (PyObj) &PyPg##BASE##_Type)
	ADD(Object);
	ADD(Type);

	ADD(Function);
	ADD(Call);
	ADD(FunctionCall);
	ADD(ProceduralCall);
	ADD(TriggerPull);

	ADD(TupleDesc);
	ADD(HeapTuple);
	ADD(Query);
	ADD(Portal);
#undef ADD
}
/*
 * vim: ts=3:sw=3:noet:
 */
