/* $Id: function.c,v 1.22 2006/02/27 04:41:44 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres function type implementation
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <funcapi.h>
#include <access/heapam.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <mb/pg_wchar.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/typcache.h>
#include <utils/tuplestore.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <compile.h>
#include <eval.h>
#include <marshal.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/system.h>
#include <pypg/type/record.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/error.h>
#include <pypg/function.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/function.h>
#include <pypg/cis.h>

static PyObj cache;

int PyPgFunction_Initialize(void)
{
	cache = PyDict_New();
	if (cache == NULL) return (-1);
	return(0);
}

int
PyPgFunction_DelCached(Oid procid)
{
	int rv;
	PyObj lo;

	lo = PyLong_FromUnsignedLong(procid);
	if (lo == NULL) return(-1);

	rv = PyDict_DelItem(cache, lo);
	Py_DECREF(lo);
	if (rv != 0 && PyErr_ExceptionMatches(PyExc_KeyError))
	{
		PyErr_Clear();
		rv = 0;
	}

	return(rv);
}

TupleDesc
TupleDesc_From_pg_proc_arginfo(HeapTuple ht)
{
	int2 pronargs;
	Datum proargtypes_datum, proargnames_datum;
	bool pronargs_isnull, proargtypes_isnull, proargnames_isnull;

	Oid *types;
	char **names = NULL;
	int nnames;

	TupleDesc td;

	pronargs = (int2) SysCacheGetAttr(PROCOID, ht,
		Anum_pg_proc_pronargs, &pronargs_isnull);
	if (pronargs_isnull)
		ereport(ERROR,(
			errmsg("unexpected pronargs NULL-entry")
		));

	proargtypes_datum = SysCacheGetAttr(PROCOID, ht,
		Anum_pg_proc_proargtypes, &proargtypes_isnull);
	if (proargtypes_isnull == true)
		ereport(ERROR,(
			errmsg("no arguments for input parameters descriptor")
		));

#if PG_FEATURE_ARRAY_VECTORS
	types = (Oid *) ARR_DATA_PTR(DatumGetArrayTypeP(proargtypes_datum));
#else
	types = (Oid *) proargtypes_datum;
#endif

	proargnames_datum = SysCacheGetAttr(PROCOID, ht,
		Anum_pg_proc_proargnames, &proargnames_isnull);
	if (proargnames_isnull == false)
	{
		ArrayType *name_array = NULL;
		Datum *name_datums;

		/* INOUT Parameters may adjust the location of the argument name */
#if PG_FEATURE_INOUTPARAMS
		bool modes_isnull;
		Datum proargmodes_datum;
#endif

		name_array = DatumGetArrayTypeP(proargnames_datum);

		deconstruct_array(
			name_array, ARR_ELEMTYPE(name_array),
			-1, false, 'i', &name_datums, &nnames
		);
		if (nnames < pronargs)
			ereport(ERROR, (
				errmsg("too few names in proargnames")
			));

		if (nnames > 0)
		{
			names = palloc0(sizeof(char *) * pronargs);

#if PG_FEATURE_INOUTPARAMS
			proargmodes_datum = SysCacheGetAttr(PROCOID, ht,
				Anum_pg_proc_proargmodes, &modes_isnull);
			if (modes_isnull == false)
			{
				char *modes;
				int i, k;

				modes = (char *) ARR_DATA_PTR(proargmodes_datum);
				for (i = 0, k = 0; i < nnames; ++i)
				{
					switch (modes[i])
					{
						case 'b':
						case 'i':
						{
							Datum name_d;
							name_d = name_datums[i];
							names[k] = (char *) DirectFunctionCall1(textout, name_d);
							++k;
						}
						break;

						case 'o':
						break;

						default:
						{
							ereport(ERROR, (
								errmsg("invalid argument mode in pg_proc tuple"),
								errdetail(
									"Argument %d has the invalid mode \"%c\"",
									i, modes[i]
								)
							));
						}
						break;
					}
				}
				if (i < pronargs)
					ereport(ERROR, (
						errmsg("ill formed proargnames relative to modes")
					));
			}
			else /* Don't process proargnames down here if modes is valid */
#endif
			{
				int i;

				for (i = 0; i < nnames; ++i)
				{
					Datum name_d;
					name_d = name_datums[i];
					names[i] = (char *) DirectFunctionCall1(textout, name_d);
				}
			}
		}
	}

	td = TupleDesc_FromNamesAndOids(pronargs, (const char **) names, types);
	if (names != NULL)
	{
		int i;
		if (nnames > 0)
			for (i = 0; i < pronargs; ++i)
				pfree(names[i]);
		pfree(names);
	}
	return(td);
}

PyObj
PyTuple_FromTupleDescAndParameters(TupleDesc td, PyObj args, PyObj kw)
{
	PyObj rob;
	int args_len, kw_len;

	args_len = PyObject_Length(args);
	kw_len = kw != NULL ? PyObject_Length(kw) : 0;

	if (args_len == -1 || kw_len == -1)
		return(NULL);

	if ((args_len + kw_len) != td->natts)
	{
		PyErr_Format(PyExc_TypeError,
			"construction requires exactly %d arguments, given %d",
			td->natts, (args_len + kw_len));
		return(NULL);
	}

	rob = PyTuple_New(td->natts);

	if (kw_len > 0)
	{
		PyObj ob_key, kw_iter;
		kw_iter = PyObject_GetIter(kw);

		while ((ob_key = PyIter_Next(kw_iter)) != NULL)
		{
			PyObj ob_str, ob;
			char *obstr;
			int i;

			ob_str = PyObject_Str(ob_key);
			if (ob_str == NULL)
			{
				Py_DECREF(ob_key);
				Py_DECREF(kw_iter);
				Py_DECREF(rob);
				return(NULL);
			}
			obstr = PyString_AS_STRING(ob_str);

			for (i = 0; i < td->natts; ++i)
			{
				if (!strcmp(NameStr(td->attrs[i]->attname), obstr))
					break;
			}
			Py_DECREF(ob_str);

			if (i == td->natts)
			{
				Py_DECREF(rob);
				Py_DECREF(kw_iter);
				PyErr_SetObject(PyExc_KeyError, ob_key);
				Py_DECREF(ob_key);
				return(NULL);
			}

			ob = PyObject_GetItem(kw, ob_key);
			Py_DECREF(ob_key);
			PyTuple_SET_ITEM(rob, i, ob);
		}
	}

	if (args_len > 0)
	{
		int i, ai;
		
		for (i = 0, ai = 0; i < td->natts && ai < args_len; ++i)
		{
			PyObj ob;
			if (PyTuple_GET_ITEM(rob, i) != NULL)
				continue;

			ob = PySequence_GetItem(args, ai);
			if (ob == NULL)
			{
				Py_DECREF(rob);
				return(NULL);
			}
			PyTuple_SET_ITEM(rob, i, ob);
			++ai;
		}
	}

	return(rob);
}

static PyMethodDef PyPgFunction_Methods[] = {
	/*
	 * {"name", (PyCFunction) FunctionRef,
	 * METH_NOARGS|METH_O|METH_VARARGS|METH_CLASS,
	 * PyDoc_STR("docstring")},
	 */
	{NULL}
};

static PyMemberDef PyPgFunction_Members[] = {
	{"procid", T_INT, offsetof(struct PyPgFunction, fn_procid), RO,
	PyDoc_STR("pg_proc entry's Oid, zero if none")},
	{"code", T_OBJECT, offsetof(struct PyPgFunction, fn_code), RO,
	PyDoc_STR("the function's code object, None if not Python")},
	{"func_code", T_OBJECT, offsetof(struct PyPgFunction, fn_code), RO,
	PyDoc_STR("alias to code")},
	{"output", T_OBJECT, offsetof(struct PyPgFunction, fn_output), RO,
	PyDoc_STR("the function's output shaper")},
	{"input", T_OBJECT, offsetof(struct PyPgFunction, fn_input), RO,
	PyDoc_STR("the function's input shaper")},
	{NULL}
};

static void
func_dealloc(PyObj self)
{
	PyObj ob;

	ob = PyPgFunction_FetchOutput(self);
	if (ob != NULL)
	{
		PyPgFunction_FixOutput(self, NULL);
		Py_DECREF(ob);
	}

	ob = PyPgFunction_FetchInput(self);
	if (ob != NULL)
	{
		PyPgFunction_FixInput(self, NULL);
		Py_DECREF(ob);
	}

	ob = PyPgFunction_FetchCode(self);
	if (ob != NULL)
	{
		PyPgFunction_FixCode(self, NULL);
		Py_DECREF(ob);
	}

	self->ob_type->tp_free(self);
}

static PyObj
func_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj fn_input, fn_output, input, rob;
	TupleDesc td;
	HeapTuple ht = NULL;

	fn_input = PyPgFunction_FetchInput(self);
	td = PyPgTupleDesc_FetchTupleDesc(fn_input);

	input = PyTuple_FromTupleDescAndParameters(td, args, kw);
	if (input == NULL) return(NULL);

	ht = HeapTuple_FromTupleDescAndIterable(td, input);
	if (ht == NULL)
	{
		Py_DECREF(input);
		return(NULL);
	}
	fn_output = PyPgFunction_FetchOutput(self);

	if (PyPgFunction_IsSRF(self))
	{
		PyObj ht_args = PyPgHeapTuple_New(fn_input, ht);
		rob = PyPgCall_New(self, ht_args, fn_output);
	}
	else if (PyPgFunction_IsPython(self))
	{
		PyObj prerob, ht_args = PyPgHeapTuple_New(fn_input, ht);
		PyObj cargs[3] = {Py_None, ht_args};

		prerob = PyEval_EvalCodeEx((PyCodeObject *) PyPgFunction_FetchCode(self),
					__main__, NULL, cargs, 2, NULL, 0, NULL, 0, NULL);
		rob = prerob ? Py_Call(fn_output, prerob) : NULL;
	}
	else
	{
		MemoryContext former;
		FmgrInfo flinfo;
		FunctionCallInfoData fcinfo;
		int i;
		Datum datum = 0;

		flinfo.fn_addr = PyPgFunction_FetchFunction(self);
		flinfo.fn_oid = PyPgFunction_FetchProcId(self);
		flinfo.fn_nargs = td->natts;
		flinfo.fn_retset = false;
		flinfo.fn_extra = NULL;
		flinfo.fn_mcxt = PythonMemoryContext;
		flinfo.fn_expr = NULL;
		fcinfo.flinfo = &flinfo;
		fcinfo.context = NULL;
		fcinfo.resultinfo = NULL;
		fcinfo.isnull = false;
		fcinfo.nargs = td->natts;
		for (i = 0; i < td->natts; ++i)
		{
			bool isnull;
			fcinfo.arg[i] = fastgetattr(ht, i + 1, td, &isnull);
			fcinfo.argnull[i] = isnull;
		}

		former = CurrentMemoryContext;
		PgError_TRAP(datum = FunctionCallInvoke(&fcinfo));
		MemoryContextSwitchTo(former);
		if (!PyErr_Occurred())
			rob = PyPgObject_New(fn_output, datum);
		else
			rob = NULL;
	}
	if (ht != NULL) heap_freetuple(ht);

	return(rob);
}

static PyObj
func_new_from_oid(PyTypeObject *subtype, Oid fn_oid)
{
	HeapTuple ht = NULL;
	PyObj rob = NULL;
	PyObj lo;
	PGFunction fn_addr;

	lo = PyLong_FromUnsignedLong(fn_oid);
	if (lo == NULL) return(NULL);

	if (PyMapping_HasKey(cache, lo))
	{
		rob = PyObject_GetItem(cache, lo);
		Py_DECREF(lo);
		return(rob);
	}

	PG_TRY();
	{
		HeapTuple proctup;
		Form_pg_proc ps;
		FmgrInfo flinfo;
		PyObj output;
		bool isnull = true;

		proctup = ht = SearchSysCache(PROCOID, fn_oid, 0, 0, 0);
		if (ht == NULL)
		{
			ereport(ERROR,(
				errcode(ERRCODE_UNDEFINED_FUNCTION),
				errmsg("failed to fetch procedure")
			));
		}
		ps = PROCSTRUCT(ht);

		fmgr_info_cxt(fn_oid, &flinfo, PythonMemoryContext);

		fn_addr = flinfo.fn_addr;
		Assert(fn_addr != NULL);

		rob = subtype->tp_alloc(subtype, 0);
		if (rob == NULL) goto tryend;
		PyPgFunction_FixProcId(rob, fn_oid);

		/* XXX: Resolve polymorphic types? */

#if PG_FEATURE_INOUTPARAMS
		{
			TupleDesc result_desc;
			result_desc = build_function_result_tupdesc_t(ht);

			if (result_desc == NULL)
				output = PyPgType_FromOid(ps->prorettype);
			else
				output = PyPg_record_Type_FromTupleDesc(result_desc);
		}
#else
		output = PyPgType_FromOid(ps->prorettype);
#endif
		if (output == NULL) goto tryend;
		PyPgFunction_FixOutput(rob, output);

		if (fn_addr == PyPgCI.PL.Handler)
		{
			PyObj code;
			Datum probin =
				SysCacheGetAttr(PROCOID, ht, Anum_pg_proc_probin, &isnull);
			bytea *data;

			if (isnull)
			{
				PyErr_SetString(PyExc_TypeError, "NULL probin for code");
				goto tryend;
			}
			data = PG_DETOAST_DATUM(probin);

			code = PyPgCI.PL.CodeFromData(
				VARDATA(data), VARSIZE(data) - VARHDRSZ
			);
			if (PointerGetDatum(data) != probin) pfree(data);
			if (code == NULL) goto tryend;
			PyPgFunction_FixCode(rob, code);
		}
		else
		{
			Py_INCREF(Py_None);
			PyPgFunction_FixCode(rob, Py_None);
		}
		PyPgFunction_FixPGFunction(rob, fn_addr);

		PyPgFunction_FixReturnsSet(rob,
			DatumGetBool(
				SysCacheGetAttr(PROCOID, ht, Anum_pg_proc_proretset, &isnull)
			)
		);
		if (isnull)
		{
			PyErr_SetString(PyExc_TypeError,
				"unexpected NULL proretset in pg_proc tuple");
			goto tryend;
		}

		if (ps->pronargs > 0)
		{
			TupleDesc td = NULL;
			PyObj tdo;

			PgError_TRAP(td = TupleDesc_From_pg_proc_arginfo(ht));
			if (PyErr_Occurred()) goto tryend;

			tdo = PyPgTupleDesc_New(td);
			FreeTupleDesc(td);
			if (tdo == NULL) goto tryend;

			PyPgFunction_FixInput(rob, tdo);
		}
		else
		{
			PyPgFunction_FixInput(rob, EmptyPyPgTupleDesc);
			Py_INCREF(EmptyPyPgTupleDesc);
		}

tryend:
		ht = NULL;
		ReleaseSysCache(proctup);
	}
	PG_CATCH();
	{
		if (ht != NULL) ReleaseSysCache(ht);
		PyErr_SetPgError();
	}
	PG_END_TRY();

	if (PyErr_Occurred())
	{
		Py_DECREF(lo);
		Py_XDECREF(rob);

		return(NULL);
	}

	PyObject_SetItem(cache, lo, rob);
	Py_DECREF(lo);
	return(rob);
}

static PyObj
func_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	Oid fn_oid = InvalidOid;
	PyObj source = NULL, rob;

	if (!PyPgArg_ParseTypeSource(args, kw, &source))
		return(NULL);

	if (PyInt_Check(source)
	|| PyLong_Check(source)
	|| PyPg_oid_Check(source)
	|| PyPg_regproc_Check(source)
	|| PyPg_regprocedure_Check(source))
	{
		fn_oid = Oid_FromPyObject(source);
		if (fn_oid == InvalidOid)
		{
			if (!PyErr_Occurred())
			{
				PyErr_Format(PyExc_TypeError,
					"unable to fetch valid Oid from '%s' object",
					source->ob_type->tp_name);
			}
		}
	}
	else
	{
		PyObj src_str;

		src_str = PyObject_Str(source);
		PgError_TRAP(
			fn_oid = DirectFunctionCall1(
				regprocedurein, PointerGetDatum(PyString_AS_STRING(src_str))
			)
		);
		Py_DECREF(src_str);
	}
	if (PyErr_Occurred()) return(NULL);

	rob = func_new_from_oid(subtype, fn_oid);
	return(rob);
}

PyDoc_STRVAR(PyPgFunction_Type_Doc,
"Postgres function interface type");

PyTypeObject PyPgFunction_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Function",				/* tp_name */
	sizeof(struct PyPgFunction),	/* tp_basicsize */
	0,										/* tp_itemsize */
	func_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	NULL,									/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	func_call,							/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	PyPgFunction_Type_Doc,			/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgFunction_Methods,			/* tp_methods */
	PyPgFunction_Members,			/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	func_new,							/* tp_new */
};

PyObj
PyPgFunction_FromOid(Oid po)
{
	return(func_new_from_oid(&PyPgFunction_Type, po));
}
/*
 * vim: ts=3:sw=3:noet:
 */
