/***************************************************************************
 *  Copyright (C) 2009 Nippon Telegraph and Telephone Corporation
 ***************************************************************************/
/**
 * @file	stats_info.c
 * @brief   Stored-procedure for management snapshots.
 *
 */
#include "postgres.h"

#ifndef WIN32
#include "sys/utsname.h"
#endif

#include "access/heapam.h"
#include "catalog/pg_type.h"
#include "funcapi.h"
#include "miscadmin.h"
#include "utils/builtins.h"
#include "executor/spi.h"
#include "lib/stringinfo.h"

#include "stats_info.h"
#include "stats_info_sql.h"
#include "stats_info_message.h"

PG_MODULE_MAGIC;

PG_FUNCTION_INFO_V1(get_stats_no_args);
PG_FUNCTION_INFO_V1(get_stats_by_dbid);
PG_FUNCTION_INFO_V1(get_stats_with_description);
PG_FUNCTION_INFO_V1(get_stats_by_dbid_with_description);
PG_FUNCTION_INFO_V1(get_key);

#if PG_VERSION_NUM < 80400
/**
 * @fn	  static text *cstring_to_text(const char *str)
 * @brief   Convert 'char *' to 'text' 
 *
 *   Converts character string in an pointed address to text type.
 *   This routine is the same as cstring_to_text in PG8.4.
 *
 * @param[in]   str  String before converted
 * @return	  Address which pointed text-type data after converted 
 *
 */
static text *
cstring_to_text(const char *str)
{
	text   *new_text;
	int		str_len;

	str_len = strlen(str);
	new_text = (text*) palloc(VARHDRSZ + str_len);
	SET_VARSIZE(new_text, str_len + VARHDRSZ);
	memcpy(VARDATA(new_text), str, str_len);
	return new_text;
}
#endif

/**
 * @fn	  static SPIPlanPtr get_stats_prepare(
 *				  const char *qry_format, int nargs, Oid *argtypes)
 * @brief   Prepare execute-plan
 *
 *   Prepares the execute-plan. (Using SPI_prepare) 
 *
 * @param[in]   qry_format	SQL format
 * @param[in]   nargs		 Number of parameters
 * @param[in]   argtypes	  Pointer to array of Data-type OIDs of parameters
 * @return	  Pointer to execute-plan
 *
 */
static SPIPlanPtr
get_stats_prepare(const char *qry_format, int nargs, Oid *argtypes)
{
	SPIPlanPtr  plan;

	plan = SPI_prepare(qry_format, nargs, argtypes);

	if (plan == NULL)
		elog(ERROR, MSG_STATS_INFO_2004, "get_stats_prepare", SPI_result, qry_format);

	return plan;
}

/**
 * @fn	  static char *decrypt_password(int64 dbid)
 * @brief   password decryption
 *
 *   This routine decrypts the encrypted-password for connection to specified DBID.
 *
 * @param[in]   dbid  DBID
 * @return	  Encrypted password string.
 *
 */
static char *
decrypt_password(int64 dbid)
{
	int   spi_result;
	int   result_num;
	char *dec_pwd = NULL;
	SPIPlanPtr plan;
	Oid   argtypes[1];
	Datum values[1];
	char  nulls[1];

	/*
	 * Prepare execute-plan.
	 */
	argtypes[0] = INT8OID;
	plan = get_stats_prepare(SQL_DECRYPT_FORMAT, 1, argtypes);

	/*
	 * Decryption.
	 */
	values[0] = Int64GetDatum(dbid);
	nulls[0] = ' ';
	spi_result = SPI_execute_plan(plan, values, nulls, true, 1);
	result_num = SPI_processed;

	if ((spi_result == SPI_OK_SELECT) && (result_num > 0))
	{
		char *val_dec_pwd;

		val_dec_pwd = SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1);

		dec_pwd = pstrdup(val_dec_pwd);
	}
	else
	{
		elog(ERROR, MSG_STATS_INFO_2005, "get_stats", spi_result, SPI_OK_SELECT);
	}

	return dec_pwd;
}

/**
 * @fn	  static TargetHostInfo *create_target_host_info(int64 dbid)
 * @brief   create target host(DB) information to connect DB (single-DB)
 *
 *   Create the host-information for connecting to DB (specified as DBID).
 *   We take information from t_dbconn which 'DBID' is 'specified-DBID'.
 *   This routine is used for getting snapshot for single DB. 
 *
 * @param[in]   dbid  DBID
 * @return	  Host-information for connecting to single-DB
 *
 */
static TargetHostInfo *
create_target_host_info(int64 dbid)
{
	DBConnInfo *dbconn_info = NULL;	
	TargetHostInfo *host_info = NULL;
	int spi_result;	
	int result_num;
	HeapTuple tuple;
	TupleDesc tupdesc;	 /* tuple descriptor */
	SPITupleTable *spi_tuptable_local;
	SPIPlanPtr plan;
	Oid   argtypes[1];
	Datum values[1];
	char  nulls[1];

	/*
	 * Prepare execute-plan.
	 */
	argtypes[0] = INT8OID;
	plan = get_stats_prepare(SQL_SELECT_T_DBCONN_BY_DBID_FORMAT, 1, argtypes);

	/*
	 * Get connection information from t_dbconn table using by specified DBID.
	 */
	values[0] = Int64GetDatum(dbid);
	nulls[0] = ' ';
	spi_result = SPI_execute_plan(plan, values, nulls, true, 1);
	result_num = SPI_processed;
	spi_tuptable_local = SPI_tuptable;
	if ((spi_result == SPI_OK_SELECT) && (result_num == 1))
	{
		int64   host_id;	  
		StringInfo str_dbname; 
		StringInfo str_hostname; 
		int		port;
		StringInfo str_username;
		StringInfo str_enc_pwd;  /* encrypted-password */
		StringInfo str_dec_pwd;  /* decrypted-password */

		str_dbname = makeStringInfo();
		str_hostname = makeStringInfo();
		str_username = makeStringInfo();
		str_enc_pwd = makeStringInfo();
		str_dec_pwd = makeStringInfo();

		tuple = spi_tuptable_local->vals[0];
		tupdesc = spi_tuptable_local->tupdesc;

		/*
		 * Get each values.
		 */
		host_id = strtol(SPI_getvalue(tuple, tupdesc, 1), NULL, 10);
		appendStringInfoString(str_dbname, SPI_getvalue(tuple, tupdesc, 2));
		appendStringInfoString(str_hostname, SPI_getvalue(tuple, tupdesc, 3));
		port = (int)strtol(SPI_getvalue(tuple, tupdesc, 4), NULL, 10);
		appendStringInfoString(str_username, SPI_getvalue(tuple, tupdesc, 5));
		appendStringInfoString(str_enc_pwd, SPI_getvalue(tuple, tupdesc, 6));
		appendStringInfoString(str_dec_pwd, decrypt_password(dbid));

		/*
		 * Store each values.
		 */
		dbconn_info = (DBConnInfo *) palloc(sizeof(DBConnInfo));
		host_info = (TargetHostInfo *) palloc(sizeof(TargetHostInfo));

		dbconn_info->db_id = dbid;
		dbconn_info->dbname = str_dbname->data;
		dbconn_info->hostname = str_hostname->data;
		dbconn_info->port = port;
		dbconn_info->username = str_username->data;
		dbconn_info->enc_pwd = str_enc_pwd->data;
		dbconn_info->dec_pwd = str_dec_pwd->data;

		host_info->host_id = host_id;
		host_info->db_num = 1;
		host_info->db_list = dbconn_info;
	}
	else
	{
		ereport(ERROR, (errmsg(MSG_STATS_INFO_2002, SP_NAME_GET_STATS, dbid)));
	}

	return host_info;
}

/**
 * @fn	  static TargetInfo *create_target_list()
 * @brief   Create target host(DB) information to connect DBs (Multi-DB) 
 *
 *   Create the host-information for connecting to DBs (specified in t_dbconn).
 *   We take information from t_dbconn table which 'is_snapshot_target' is 'true'. 
 *   This routine is used for getting snapshot for mulit DBs. 
 *
 * @return        Host-information for connecting to Multi-DB
 *
 */
static TargetInfo *
create_target_list()
{
	TargetInfo *trgt_info = NULL;
	int spi_result;	
	int result_num;
	SPITupleTable *spi_tuptable_local;

	trgt_info = NULL;

	/*
     *  Get connection information from t_dbconn table. 
	 * 'SQL_SELECT_T_DBCONN_ON_TARGET_ALL' take all information of DBs need to get snapshot. 
	 */
	spi_result = SPI_execute(SQL_SELECT_T_DBCONN_ON_TARGET_ALL, true, 0);
	result_num = SPI_processed;
	spi_tuptable_local = SPI_tuptable;
	if ((spi_result == SPI_OK_SELECT) && (result_num >= 0))
	{
		int64 host_id;
		int64 dbid;
		int   port;
		int64 current_host_id;
		DBConnInfo *trgt_dbconn_info = NULL;
		TargetHostInfo *host_info;		/* Host information list */
		TargetHostInfo *trgt_host_info;		/* Host information store point */
		int  db_pos_in_host = 0;
		int current_dbconn_info_size;		/* size of connection information */
		int current_host_info_size;		/* size of host information */
		int host_pos;
		int dbconn_pos;	
		HeapTuple tuple;
		TupleDesc tupdesc;
		StringInfo str_dbname;
		StringInfo str_hostname;
		StringInfo str_username;
		StringInfo str_enc_pwd;			  /* encrypted password */
		StringInfo str_dec_pwd;			  /* decrypted password */

		current_dbconn_info_size = 0;
		current_host_info_size = 0;
		current_host_id = -1;
		host_info = (TargetHostInfo *) palloc(sizeof(TargetHostInfo));
		host_pos = -1;
		trgt_host_info = host_info + 0;


		/* 
		 * Get all information in this loop.
		 */ 
		for (dbconn_pos = 0; dbconn_pos < result_num; dbconn_pos++, db_pos_in_host++)
		{
			/* Initialize each variables */
			str_dbname = makeStringInfo();
			str_hostname = makeStringInfo();
			str_username = makeStringInfo();
			str_enc_pwd = makeStringInfo();
			str_dec_pwd = makeStringInfo();

			tuple = spi_tuptable_local->vals[dbconn_pos];
			tupdesc = spi_tuptable_local->tupdesc;

			/*
			 * Get host information.
			 */
			host_id = strtol(SPI_getvalue(tuple, tupdesc, 1), NULL, 10);
			dbid = strtol(SPI_getvalue(tuple, tupdesc, 2), NULL, 10);
			appendStringInfoString(str_dbname, SPI_getvalue(tuple, tupdesc, 3));
			appendStringInfoString(str_hostname, SPI_getvalue(tuple, tupdesc, 4));
			port = (int)strtol(SPI_getvalue(tuple, tupdesc, 5), NULL, 10);
			appendStringInfoString(str_username, SPI_getvalue(tuple, tupdesc, 6));
			appendStringInfoString(str_enc_pwd, SPI_getvalue(tuple, tupdesc, 7));
			appendStringInfoString(str_dec_pwd, decrypt_password(dbid));

			/*
			 * Check host_id. If 'current_host_id' is not 'getting host_id', foward the position on 
			 * host-list and get information from next host. 
			 */
			if (current_host_id != host_id)
			{
				host_pos++;			/* foward host list position */
				db_pos_in_host = 0;		/* clear db list position in new host */
				current_host_id = host_id;	/* update host id */

				if (((host_pos+1) % EXPNDSZ) == 0)
				{
					/*
					 * host_info memory size adjustment
					 */
					current_host_info_size += EXPNDSZ;
					host_info = (TargetHostInfo *)repalloc(
							host_info, sizeof(TargetHostInfo) * current_host_info_size);
				}

				/*
				 * store host information
				 */
				trgt_host_info = host_info + host_pos;

				trgt_host_info->host_id = host_id;
				trgt_host_info->db_num = 0;
				current_dbconn_info_size = 0;
				/* for repalloc */
				trgt_host_info->db_list = (DBConnInfo *) palloc(sizeof(DBConnInfo));
			}

			if ((db_pos_in_host % EXPNDSZ) == 0)
			{
				/*
				 * host_info memory size adjustment
				 */
				current_dbconn_info_size += EXPNDSZ;
				trgt_host_info->db_list = (DBConnInfo *)repalloc(
						trgt_host_info->db_list, sizeof(DBConnInfo) * current_dbconn_info_size);
			}

			trgt_dbconn_info = trgt_host_info->db_list + db_pos_in_host;

			/*
			 * store connection information
			 */
			trgt_dbconn_info->db_id	= dbid;
			trgt_dbconn_info->dbname = str_dbname->data;
			trgt_dbconn_info->hostname = str_hostname->data;
			trgt_dbconn_info->port = port;
			trgt_dbconn_info->username = str_username->data;
			trgt_dbconn_info->enc_pwd = str_enc_pwd->data;
			trgt_dbconn_info->dec_pwd = str_dec_pwd->data;

			trgt_host_info->db_num++;
		}

		trgt_info = (TargetInfo *) palloc(sizeof(TargetInfo));
		trgt_info->host_num = host_pos + 1;
		trgt_info->host_list = host_info;
	}
	else
	{
		elog(ERROR, MSG_STATS_INFO_2005, "get_stats", spi_result, SPI_OK_SELECT);
	}

	return trgt_info;
}

/**
 * @fn	  static void start_dblink_to_target_db(DBConnInfo *conn_info)
 * @brief   start connecting to DB via dblink
 *
 *   This routine is used for getting persistent connection to specified DB via dblink.
 *
 * @param[in]   conn_info
 * @return	 none
 *
 */
static void
start_dblink_to_target_db(DBConnInfo *conn_info)
{
	SPIPlanPtr plan;
	Datum values[1];
	Oid   argtypes[1] = {TEXTOID};
	char  nulls[1] = {' '};
	StringInfo   str_conn;

	plan = get_stats_prepare(SQL_DBLINK_CONNECT_FORMAT, 1, argtypes);

	str_conn = makeStringInfo();
	appendStringInfo(str_conn, DBLINK_CONN_STRING_FORMAT,
					conn_info->dbname, conn_info->hostname, conn_info->port,
					conn_info->username, conn_info->dec_pwd);
	values[0] = PointerGetDatum(cstring_to_text(str_conn->data));

	SPI_execute_plan(plan, values, nulls, true, 1);
}

/**
 * @fn	  static void end_dblink_to_target_db()
 * @brief   finish connecting to DB via dblink
 *
 *   This routine is used for shut down connecting to some DB via dblink.
 *
 * @return	  none
 *
 */
static void
end_dblink_to_target_db()
{
	SPI_execute(SQL_DBLINK_DISCONNECT_FORMAT, false, 1);
}


/**
 * @fn	  static int64 insert_into_t_snapshot_id(int64 db_id, int64 host_id, SnapshotParameters *params)
 * @brief   insert t_snapshot_id
 *
 *   Do insert into t_snapshot_id table, and get its snapshotid (using serial).
 *   t_snapshot_id is master table for management snapshots. 
 *
 * @param[in]	db_id		DBID
 * @param[in]	host_id		host ID
 * @param[in]	params		parameters for getting snapshot
 * @return	snapshot_id
 *
 */
static int64
insert_into_t_snapshot_id(int64 db_id, int64 host_id, SnapshotParameters *params)
{
	int64	snapshot_id = -1;
	int		spi_result;
	int		result_num;
	SPIPlanPtr plan;
	Datum	values[4];
	Oid		argtypes[4] = {INT8OID, INT8OID, INT2OID, TEXTOID};
	char	nulls[4] = {' ', ' ', ' ', ' '};

	plan = get_stats_prepare(SQL_INSERT_T_SNAPSHOT_ID_FORMAT, 4, argtypes);

	/*
	 * check parameter
	 */
	if (params->description == NULL)
		/* description が NULL の場合 */
		nulls[3] = 'n';

	/*
	 * Do INSERT
	 *   - get snapshotid by RETURNING
	 */
	values[0] = Int64GetDatum(db_id);
	values[1] = Int64GetDatum(host_id);
	values[2] = Int16GetDatum(params->level);
	
	/* description NULL check */
	if (params->description != NULL)
	{
		values[3] = PointerGetDatum(cstring_to_text(params->description));
	}else{
		values[3] = PointerGetDatum((text *)NULL);
	}
	spi_result = SPI_execute_plan(plan, values, nulls, false, 1);
	result_num = SPI_processed;
	if ((spi_result == SPI_OK_INSERT_RETURNING) && (result_num > 0))
		snapshot_id = strtol(
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1), NULL, 10);

	return snapshot_id;
}

/**
 * @fn	  static int insert_record_into_t_dbinfo(
 *				  int64 snapshot_id, DBConnInfo *db_conn_info)
 * @brief   getting snapshot about DB information and insert the information into t_dbinfo
 *
 *   This routine is used for getting snapshot about DB information (ex: pg_stat_database)
 *   for monitoring target DB, and insert the information into t_dbinfo.
 *   Snapshot is taken via dblink that already has persistent connection. 
 *
 *   This routine is similar to "insert_record_into_dbinfo_tables()", but differ in number of
 *   SQL parameters. But it is possible to share the code. This is a TODO.
 *   
 *
 * @param[in]   snapshot_id	snapshot_id
 * @param[in]   db_conn_info    montioring target DB connection information
 * @param[in]   sql_str_format  SQL format to execute
 * @return	results of SPI_execute()
 *
 */
static int
insert_record_into_t_dbinfo(
		int64 snapshot_id, DBConnInfo *db_conn_info)
{
	int spi_result;	
	SPIPlanPtr plan;
	Datum values[3];
	Oid   argtypes[3] = {INT8OID, INT8OID, TEXTOID};
	char  nulls[3] = {' ', ' ', ' '};
	StringInfo  sql_dblink;

	/*
	 * create SQL send to dblink
	 */
	sql_dblink = makeStringInfo();
	appendStringInfo(sql_dblink, DBLINK_SELECT_T_DBINFO_FORMAT, db_conn_info->dbname);

	/*
	 * prepare
	 */
	plan = get_stats_prepare(SQL_INSERT_T_DBINFO_FORMAT, 3, argtypes);

	/*
	 * getting snapshot and insert the information into t_dbinfo at a time
	 */
	values[0] = Int64GetDatum(snapshot_id);
	values[1] = Int64GetDatum(db_conn_info->db_id);
	values[2] = PointerGetDatum(cstring_to_text(sql_dblink->data));
	spi_result = SPI_execute_plan(plan, values, nulls, false, 0);

	return spi_result;
}

/**
 * @fn	  static int insert_record_into_dbinfo_tables(
 *				  int64 snapshot_id, DBConnInfo *db_conn_info, const char *sql_str_format)
 * @brief   getting snapshot about any acquirable information by DB and insert into own table
 *
 *   This routine is used for getting snapshot about acquirable information by DB such as
 *   table statistics info (pg_stat_all_tables), index statistics info (pg_stat_all_indexes).
 *   And insert the snapshot information into own table.
 *   Snapshot is taken via dblink that already has persistent connection.
 *   
 *
 * @param[in]   snapshot_id    snapshot_id
 * @param[in]   db_conn_info   montioring target DB connection information 
 * @param[in]   sql_str_format SQL format to execute
 * @return      results of SPI_execute()
 *
 */
static int
insert_record_into_dbinfo_tables(
		int64 snapshot_id, DBConnInfo *db_conn_info, const char *sql_str_format)
{
	int spi_result;
	SPIPlanPtr plan;
	Datum values[2];
	Oid   argtypes[2] = {INT8OID, INT8OID};
	char  nulls[2] = {' ', ' '};

	plan = get_stats_prepare(sql_str_format, 2, argtypes);

	/*
         * getting snapshot and insert the information into own table at a time
	 */
	values[0] = Int64GetDatum(snapshot_id);
	values[1] = Int64GetDatum(db_conn_info->db_id);
	spi_result = SPI_execute_plan(plan, values, nulls, false, 0);

	return spi_result;
}

/**
 * @fn	  static int64 insert_into_t_hostinfo(int64 host_id)
 * @brief   getting snapshot about host information and insert the information into t_hostinfo
 *
 *   This routine is used for getting snapshot about host information, and insert the
 *   information into t_hostinfo, and get host_snapshot_id.
 *   "host information" contains kernel-version-info, HW-info, and PG-version-info of
 *   the target monitoring DB in the DB Cluster. So, this routine is called by getting
 *   snapshot for each host (DB-Cluster) only once.
 *   
 *
 * @param[in]   host_id  host_id
 * @return	host_snapshot_id
 *
 */
static int64
insert_into_t_hostinfo(int64 host_id)
{
	int spi_result;	
	int64 host_snapshot_id = -1;
	int result_num;	

	SPIPlanPtr plan;
	Datum values[1];
	Oid   argtypes[1] = {INT8OID};
	char  nulls[1] = {' '};

	plan = get_stats_prepare(SQL_INSERT_T_HOSTINFO_FORMAT, 1, argtypes);

	/*
         * getting snapshot and insert the information into t_dbinfo at a time
	 *   - get host_snapshot_id with RETURNING.
	 */
	values[0] = Int64GetDatum(host_id);
	spi_result = SPI_execute_plan(plan, values, nulls, false, 1);
	result_num = SPI_processed;
	if ((spi_result == SPI_OK_INSERT_RETURNING) && (result_num > 0))
		host_snapshot_id = strtol(
				SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1), NULL, 10);

	return host_snapshot_id;
}

/**
 * @fn	  static int insert_record_into_hostinfo_tables(
 *				  int64 host_snapshot_id, const char *sql_str_format)
 * @brief   getting snapshot about any acquirable information by host and insert into own table
 *
 *   This routine is used for getting snapshot about acquirable information by host such as
 *   checkpoint-info, long-toransaction-info, device-usage-info.
 *   And insert the snapshot information into own table.
 *   Snapshot is taken via dblink that already has persistent connection.
 *
 * @param[in]   host_snapshot_id	host_snapshot_id
 * @param[in]   sql_str_format		SQL format to execute
 * @return	resutles of SPI_execute()
 *
 */
static int
insert_record_into_hostinfo_tables(
		int64 host_snapshot_id, const char *sql_str_format)
{
	int spi_result;
	SPIPlanPtr plan;
	Datum values[1];
	Oid   argtypes[1] = {INT8OID};
	char  nulls[1] = {' '};

	plan = get_stats_prepare(sql_str_format, 1, argtypes);

	/*
	 * getting snapshot and insert the information into own table at a time
	 */
	values[0] = Int64GetDatum(host_snapshot_id);
	spi_result = SPI_execute_plan(plan, values, nulls, false, 0);

	return spi_result;
}

/**
 * @fn	  static int get_snapshot_of_db(
 *				  DBConnInfo *db_conn_info, int64 host_id, SnapshotParameters *params)
 * @brief   getting snapshot from single DB
 *
 *   This routine is used for getting snapshot from specfied DB.
 *   This routine gets DB-basic-info and some of acquirable information by DB.
 *   Snapshot is taken via dblink that already has persistent connection.
 *
 * @param[in]   db_conn_info    montioring target DB connection information
 * @param[in]   host_id		host_id
 * @param[in]   params		parameters for snapshot
 * @return	results of snapshot
 *
 */
static int
get_snapshot_of_db(DBConnInfo *db_conn_info, int64 host_id, SnapshotParameters *params)
{
	int   rcode = 0;
	int64  snapshot_id;
	int   sql_pos;

	/*
	 * DO INSERT t_snapshot_id table
	 *   - get snapshot_id to use by this term
	 */
	snapshot_id = insert_into_t_snapshot_id(db_conn_info->db_id, host_id, params);
	if (snapshot_id < 0)
		ereport(ERROR, (errmsg(MSG_STATS_INFO_1001, SP_NAME_GET_STATS)));

	/*
	 * getting snapshot about target DB basic information and insert into t_dbinfo table
	 *   - "insert_record_intot_dbinfo" bit differ in number of params 
	 *     (vs insert_record_into_dbinfo_tables), so we do not performe in following loop. 
	 */
	rcode = insert_record_into_t_dbinfo(snapshot_id, db_conn_info);
	if (rcode != SPI_OK_INSERT)
		ereport(ERROR,
				(errmsg(MSG_STATS_INFO_1002, SP_NAME_GET_STATS,
					db_conn_info->hostname, db_conn_info->port, db_conn_info->dbname)));

	/*
	 * getting snapshot and insert into own tables.
	 *   - exection SQL are registered in SQL_LIST_FOR_INSERT_INTO_DBINFO[].
	 *   - number of exection SQL is  SQL_LIST_FOR_INSERT_INTO_DBINFO_SIZE.
	 *   - error messages for each SQL-failure are registered in MSG_LIST_FOR_INSERT_INTO_DBINFO[].
	 *   - if we add new statistical information, we will have to add necessary information to
	 *     above LISTs. 
	 */
	for (sql_pos = 0; sql_pos < SQL_LIST_FOR_INSERT_INTO_DBINFO_SIZE; sql_pos++)
	{
		rcode = insert_record_into_dbinfo_tables(
				snapshot_id, db_conn_info, SQL_LIST_FOR_INSERT_INTO_DBINFO[sql_pos]);
		if (rcode < 0)
			ereport(ERROR,
					(errmsg(MSG_LIST_FOR_INSERT_INTO_DBINFO[sql_pos], SP_NAME_GET_STATS,
						db_conn_info->hostname, db_conn_info->port, db_conn_info->dbname)));
	}

	return rcode;
}

/**
 * @fn	  static int get_snapshot_of_cluster(
 *				  DBConnInfo *db_conn_info, int64 host_id, SnapshotParameters *params)
 * @brief   getting snapshot from single host
 *
 *   This routine is used for getting some of acquirable information by host (DB-Cluster).
 *   Snapshot is taken via dblink that already has persistent connection.
 *
 * @param[in]   db_conn_info    montioring target DB connection information
 * @param[in]   host_id		host_id
 * @param[in]   params		parameters for snapshot（unused）
 * @return	results of snapshot
 *
 */
static int
get_snapshot_of_cluster(DBConnInfo *db_conn_info, int64 host_id, SnapshotParameters *params)
{
	int		rcode = 0;
	int64	host_snapshot_id;
	int		sql_pos;

	/*
	 * DO Insert into t_hostinfo
	 *   - get host_snapshot_id to use by this term
	 */
	host_snapshot_id = insert_into_t_hostinfo(host_id);
	if (host_snapshot_id < 0)
		ereport(ERROR, (errmsg(MSG_STATS_INFO_1101, SP_NAME_GET_STATS,
						db_conn_info->hostname, db_conn_info->port, db_conn_info->dbname)));
	/*
	 * getting snapshot and insert into own tables.
	 *   - exection SQL are registered in SQL_LIST_FOR_INSERT_INTO_HOSTINFO[].
	 *   - number of exection SQL is  lengthof(SQL_LIST_FOR_INSERT_INTO_HOSTINFO).
	 *   - error messages for each SQL-failure are registered in MSG_LIST_FOR_INSERT_INTO_HOSTINFO[].
	 *   - if we add new statistical information, we will have to add necessary information to
	 *     above LISTs.
	 */
	for (sql_pos = 0; sql_pos < lengthof(SQL_LIST_FOR_HOSTINFO); sql_pos++)
	{
		rcode = insert_record_into_hostinfo_tables(
				host_snapshot_id, SQL_LIST_FOR_HOSTINFO[sql_pos]);
		if (rcode < 0)
			ereport(ERROR,
					(errmsg(MSG_LIST_FOR_INSERT_INTO_HOSTINFO[sql_pos], SP_NAME_GET_STATS,
						db_conn_info->hostname, db_conn_info->port, db_conn_info->dbname)));
	}

	return rcode;
}

/**
 * @fn	  static bool get_snapshot(TargetInfo *trgt_info, SnapshotParameters *params)
 * @brief   top routine for getting snapshot
 *
 *   Get snapshot from specified target-list and insert the information into 
 *   own tables. And at last, delete the log-filtered-files.
 *   (Note: Now we get all DB's snapshot by single transaction.)  
 *   
 *
 * @param[in]   trgt_info	 target info
 * @param[in]   params		 parameters for getting snapshot
 * @return	  int		 result of getting snapshot
 *
 */
static int
get_snapshot(TargetInfo *trgt_info, SnapshotParameters *params)
{
	int rcode = 0;	
	int trgt_host_num;
	int trgt_db_num;
	TargetHostInfo *host_info;
	DBConnInfo *db_conn_info = NULL;
	int host_pod;
	int db_pos;

	trgt_host_num = trgt_info->host_num;

	/*
	 * Top loop. get snapshot for each host(DB Cluster).
	 */
	for (host_pod = 0; host_pod < trgt_host_num; host_pod++)
	{
		host_info = trgt_info->host_list + host_pod;
		trgt_db_num = host_info->db_num;

		/*
		 * sub loop. get snapshot for each DB
		 */
		for (db_pos = 0; db_pos < trgt_db_num; db_pos++)
		{
			/*
			 * create connection info
			 */
			db_conn_info = host_info->db_list + db_pos;

			/*
			 * create persistent connection to target DB 
			 */
			start_dblink_to_target_db(db_conn_info);

			/*
			 * get and store snapshot
			 */
			rcode = get_snapshot_of_db(
					db_conn_info, host_info->host_id, params);

			if (db_pos < trgt_db_num - 1)
				/*
				 * If the tagert DB is not last one of this host,
				 * we disconnect from DB. If not, we keep connection
				 * for getting info from the host. See bellow.
				 */
				end_dblink_to_target_db();
		}

		/*
		 * get snapshot about this host. we use the DB connection got above work.
		 */
		rcode = get_snapshot_of_cluster(db_conn_info, host_info->host_id, params);

		/*
		 * OK. All snapshot are taken. We finish DB connection.
		 */
		end_dblink_to_target_db();
	}

	/*
     * we delete logs on trigger.
	if (rcode > 0)
		unlink_all_statslogfiles(trgt_info);
	 */

	return rcode;
}

/**
 * @fn	  Datum get_stats_by_dbid(PG_FUNCTION_ARGS)
 * @brief   getting snapshot from specified DBID
 *
 *   This is one of entity of 'get_stats()'.
 *
 * @param[in]   PG_FUNCTION_ARGS[0]  DBID
 * @return	success：true, failed：false
 *
 */
Datum
get_stats_by_dbid(PG_FUNCTION_ARGS)
{
	int spi_result;	
	int rcode;
	int bcode = true;
	TargetInfo trgt_info;
	SnapshotParameters params;
	int64 db_id = PG_GETARG_INT64(0);

	/*
	 * (Note: Noisy?)
	 */
	ereport(LOG, (errmsg(MSG_STATS_INFO_0001, SP_NAME_GET_STATS)));

	MemSet(&trgt_info, '\0', sizeof(TargetInfo));
	MemSet(&params, '\0', sizeof(SnapshotParameters));

	/*
	 * set parameters for getting snapshot. now we use 2 params.
	 */
	params.level = DEFAULT_SNAPSHOT_LEVEL;
	params.description = NULL;

	if ((spi_result = SPI_connect()) != SPI_OK_CONNECT)
		ereport(ERROR, (errmsg(MSG_STATS_INFO_2001, SP_NAME_GET_STATS, spi_result)));

	spi_result = SPI_execute("SET search_path = statsinfo, public", false, 0);
	
	/*
	 * Create target information. 
	 */
	trgt_info.host_num = 1;
	trgt_info.host_list = create_target_host_info(db_id);

	/*
	 * Do getting snapshot
	 */
	rcode = get_snapshot(&trgt_info, &params);
	if (rcode < 0)
		bcode = false;

	SPI_finish();

	PG_RETURN_BOOL(bcode);
}

/**
 * @fn	  Datum get_stats_by_dbid_with_description(PG_FUNCTION_ARGS)
 * @brief   getting snapshot from specified DBID with description 
 *
 *   This is one of entity of 'get_stats()'.  We can set the single DBID
 *   (not multi), and add snpashot desctirption. 
 *
 * @param[in]   PG_FUNCTION_ARGS[0]  DBID
 * @param[in]   PG_FUNCTION_ARGS[1]  description for snapshot
 * @return	success：true, failed：false
 *
 */
Datum
get_stats_by_dbid_with_description(PG_FUNCTION_ARGS)
{
	int spi_result; 
	int rcode;
	int bcode = true;
	TargetInfo trgt_info;
	SnapshotParameters params;
	int64 db_id;

	/*
	 * (Note: Noisy?)
	 */
	ereport(LOG, (errmsg(MSG_STATS_INFO_0001, SP_NAME_GET_STATS)));

	MemSet(&trgt_info, '\0', sizeof(TargetInfo));
	MemSet(&params, '\0', sizeof(SnapshotParameters));

	/*
	 * get DBID.
	 */
	db_id = PG_GETARG_INT64(0);

	/*
	 * set parameters for getting snapshot. now we use 2 params.
	 */
	params.level = DEFAULT_SNAPSHOT_LEVEL;
	params.description = DatumGetCString(DirectFunctionCall1(textout, (Datum)PG_GETARG_TEXT_P(1)));

	if ((spi_result = SPI_connect()) != SPI_OK_CONNECT)
		ereport(ERROR, (errmsg(MSG_STATS_INFO_2001, SP_NAME_GET_STATS, spi_result)));

	spi_result = SPI_execute("SET search_path = statsinfo, public", false, 0);

	/*
	 * create target information.
	 */
	trgt_info.host_num = 1;
	trgt_info.host_list = create_target_host_info(db_id);

	/*
	 * Do getting snapshot
	 */
	rcode = get_snapshot(&trgt_info, &params);
	if (rcode < 0)
		bcode = false;

	SPI_finish();

	PG_RETURN_BOOL(bcode);
}


/**
 * @fn	  Datum get_stats_no_args(PG_FUNCTION_ARGS)
 * @brief   getting snapshot (No argments)
 *
 *   This is one of entity of 'get_stats()'. In this routine, we use 
 *   information of "t_dbconn table" to see which DB must be target.
 *
 * @param[in]   PG_FUNCTION_ARGS  Do not use
 * @return	  success：true, failed：false
 *
 */
Datum
get_stats_no_args(PG_FUNCTION_ARGS)
{
	int spi_result;
	int rcode;
	int bcode = true;
	TargetInfo *trgt_info;
	SnapshotParameters params;

	/*
	 * (Note: Noisy?)
	 */
	ereport(LOG, (errmsg(MSG_STATS_INFO_0001, SP_NAME_GET_STATS)));

	trgt_info = (TargetInfo *) palloc(sizeof(TargetInfo));

	MemSet(trgt_info, '\0', sizeof(TargetInfo));
	MemSet(&params, '\0', sizeof(SnapshotParameters));

	/*
	 * set parameters fo getting snapshot. now we use 2 params.
	 */
	params.level = DEFAULT_SNAPSHOT_LEVEL;
	params.description = NULL;

	if ((spi_result = SPI_connect()) != SPI_OK_CONNECT)
		/* internal error */
		ereport(ERROR, (errmsg(MSG_STATS_INFO_2001, SP_NAME_GET_STATS, spi_result)));

	spi_result = SPI_execute("SET search_path = statsinfo, public", false, 0);

	/*
	 * create target information
	 */
	trgt_info = create_target_list();

	/*
	 * Do getting snapshot
	 */
	rcode = get_snapshot(trgt_info, &params);
	if (rcode < 0)
		bcode = false;

	SPI_finish();

	PG_RETURN_BOOL(bcode);
}

/**
 * @fn	  Datum get_stats_with_description(PG_FUNCTION_ARGS)
 * @brief   getting snapshot (with description)
 *
 *   This is one of entity of 'get_stats()'. This routine performes
 *   same work as "get_stats_no_args()".
 *
 * @param[in]   PG_FUNCTION_ARGS[0]  description for snapshot
 * @return	success：true, failed：false
 *
 */
Datum
get_stats_with_description(PG_FUNCTION_ARGS)
{
	int spi_result;	
	int rcode;
	int bcode = true;
	TargetInfo *trgt_info;
	SnapshotParameters params;

	/*
	 * (Note: Noisy?)
	 */
	ereport(LOG, (errmsg(MSG_STATS_INFO_0001, SP_NAME_GET_STATS)));

	trgt_info = (TargetInfo *) palloc(sizeof(TargetInfo));

	MemSet(trgt_info, '\0', sizeof(TargetInfo));
	MemSet(&params, '\0', sizeof(SnapshotParameters));

	/*
	 * set parameters fo getting snapshot. now we use 2 params.
	 */
	params.level = DEFAULT_SNAPSHOT_LEVEL;
	params.description = DatumGetCString(DirectFunctionCall1(textout, (Datum)PG_GETARG_TEXT_P(0)));

	if ((spi_result = SPI_connect()) != SPI_OK_CONNECT)
		ereport(ERROR, (errmsg(MSG_STATS_INFO_2001, SP_NAME_GET_STATS, spi_result)));

	spi_result = SPI_execute("SET search_path = statsinfo, public", false, 0);

	/*
	 * create target information
	 */
	trgt_info = create_target_list();

	/*
	 * Do getting snapshot
	 */
	rcode = get_snapshot(trgt_info, &params);
	if (rcode < 0)
		bcode = false;

	SPI_finish();

	PG_RETURN_BOOL(bcode);
}

/**
 * @fn	  Datum get_key(PG_FUNCTION_ARGS)
 * @brief   get key-string for password encryption/decryption
 *
 *   Return the key-string for encryption/decryption of DB-connection
 *   passwords. Only super user get it.
 *   
 *
 * @return	  Key String
 *
 */
Datum
get_key(PG_FUNCTION_ARGS)
{
	/*
	 * Are you superuser ?
	 */
	if (!superuser())
		ereport(ERROR, (errmsg(MSG_STATS_INFO_2003, "get_key")));

	PG_RETURN_TEXT_P(cstring_to_text(DEC_KEY));
}
