#include "connection.h"
#include "statement.h"
#include "protocol.h"

/*-----------------------------------------------------------------------------
 * SQLBrowseConnect
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBrowseConnect_(SQLHDBC ConnectionHandle, SQLTCHAR* ConnStrIn, SQLSMALLINT ConnStrInLength,
                  SQLTCHAR* ConnStrOut, SQLSMALLINT ConnStrOutMax,
                  SQLSMALLINT* pcbConnStrOut
								 )
{
	SQLRETURN nRet; /* SQL_NEED_DATA */
	Connection* pConnection = (Connection*) ConnectionHandle;

	ENTER_CONN(ConnectionHandle, _T("SQLBrowseConnect"));

	/* parse connection string */
	nRet = ParseConnectionString(pConnection, ConnStrIn, ConnStrInLength);
	if (SQL_NEED_DATA == nRet)
	{/* we need to compile string-request */
		PrepareConnectionStringRequest(ConnStrOut, ConnStrOutMax, pcbConnStrOut, pConnection->parameters);
	}
	else
	{/* try to connect */
		PrepareConnectionString(ConnStrOut, ConnStrOutMax, pcbConnStrOut, pConnection->parameters, pConnection->dsn);
		nRet = Connect(pConnection);
	}

	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLDescribeParam
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLDescribeParam(SQLHSTMT StatementHandle, SQLUSMALLINT ParameterNumber, SQLSMALLINT* DataTypePtr,
								 SQLULEN* ParameterSizePtr, SQLSMALLINT* DecimalDigitsPtr, SQLSMALLINT* NullablePtr)
{
	SQLRETURN nRet = SQL_SUCCESS;
	ENTER_STMT(StatementHandle, _T("SQLDescribeParam"));

	if (0 < ParameterNumber)
	{
		Descriptor* pAPD;
		Descriptor* pIPD;

		pAPD = GET_DESCRIPTOR(((Statement*)StatementHandle)->apd);
		pIPD = GET_DESCRIPTOR(((Statement*)StatementHandle)->ipd);

		if (ParameterNumber <= pAPD->header.count ||
		    ParameterNumber <= pIPD->header.count
		   )
		{
			/* concise type */
			if (NULL != DataTypePtr)
				*DataTypePtr = pIPD->id_records[ParameterNumber].common.concise_type;
			
			/* nullability */
			if (NULL != NullablePtr)
				*NullablePtr = pIPD->id_records[ParameterNumber].nullable;
			
			/* display size */
			/* decimal digits */
			nRet = SQLTypeDescriptor(pIPD->id_records[ParameterNumber].common.concise_type,  /* IN: concise_type */
			                         pIPD->id_records[ParameterNumber].is_unsigned,          /* IN: unsigned value */
												       &pIPD->id_records[ParameterNumber].common.length,       /* IN: server specyfic size */
												       DecimalDigitsPtr,
												       NULL, NULL,
												       ParameterSizePtr,
												       NULL, NULL);
		}
		else
		{
			TCHAR paramNumber[SQLSMALLINT_LENGTH];

			/* no parameter binded with this number */
			_itot(++ParameterNumber, paramNumber, 10);
			SetError(SQL_HANDLE_STMT, StatementHandle, ERR_STMT_PARAM_NUMBER_NOT_BINDED, paramNumber, NULL);
			nRet = SQL_ERROR;
		}

		RET_DESCRIPTOR(pIPD);
		RET_DESCRIPTOR(pAPD);
	}
	else
	{/* DM should check this and do not let us to get here */
		;
	}

	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLSetPos
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetPos(SQLHSTMT StatementHandle, SQLSETPOSIROW RowNumber, SQLUSMALLINT Operation, SQLUSMALLINT LockType)
{
	SQLRETURN nRet = SQL_SUCCESS;
	ENTER_STMT(StatementHandle, _T("SQLSetPos"));
	nRet = SetPos((Statement*)StatementHandle, RowNumber, Operation, LockType);
	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * FUNCTION: SQLBulkOperations
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBulkOperations(SQLHSTMT StatementHandle, SQLSMALLINT Operation)
{
	SQLRETURN nRet;
	ENTER_STMT(StatementHandle, _T("SQLBulkOperations"));
	nRet = BulkOperations((Statement*)StatementHandle, Operation, SQL_FALSE);
	LEAVE_STMT(StatementHandle, nRet);
}
