/************************************************************************
 *                                                                      *
 * Geomedia BLOB CODEC                                                  *
 * Copyright (C) 2009 Claudio Rocchini                                  *
 * Istituto Geografico Militare Italiano                                *
 * web:   www.igmi.org                                                  *
 * email: ad2prod@geomil.esercito.difesa.it                             *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the  Free Software Foundation, either  version 3 of the  License, or *
 * any later version.                                                   *
 *                                                                      *
 *   This program is distributed in the hope that it will be useful,    *
 *   but WITHOUT ANY WARRANTY; without  even the implied warranty of    *
 *   MERCHANTABILITY or  FITNESS  FOR A PARTICULAR PURPOSE.  See the    *
 *   GNU General Public License for more details.                       *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program.  If not, see <http://www.gnu.org/licenses/> *
 *                                                                      *
 ************************************************************************/

 /*******************************************
  Note: the encode part is almost incomplete
  *******************************************/

#include <math.h>
#include <assert.h>

#include "blob4.h"

namespace blo
{


static inline void decode_point3d( point & pt, byte * & m )
{
	assert(m);
	pt.x = *(double *)m; m += sizeof(double);
	pt.y = *(double *)m; m += sizeof(double);
	pt.z = *(double *)m; m += sizeof(double);
}


static inline void encode_point3d( const point & pt, byte * & m )
{
	assert(m);
	*(double *)m = pt.x; m += sizeof(double);
	*(double *)m = pt.y; m += sizeof(double);
	*(double *)m = pt.z; m += sizeof(double);
}


static void decode_ring( ring & r, byte * & p )
{
	assert(p);
	size_t size = (*(size_t *)p)-1; p+= sizeof(size_t);		// Note the -1: remove last duplicated point 
	assert(size<42000000);									// Suspicious number ??
	r.resize(size);
	ring::iterator i;
	for(i=r.begin();i!=r.end();++i)
		decode_point3d( *i, p );
	p += sizeof(double)*3;									// Skip the last (duplicated) point!
}


static void encode_ring( ring & r, byte * & p )
{
	assert(p);
	size_t size = r.size()+1;							// Note the +1: duplicate first point
	(*(size_t *)p) = size; p+= sizeof(size_t);	
	ring::const_iterator i;
	for(i=r.begin();i!=r.end();++i)
		encode_point3d( *i, p );
	encode_point3d( r.front(),p );						// Duplicate first point
}


static inline void decode_arc( arc & ar, byte * & m )
{
	assert(m);
	ar.start.x  = *(double *)m; m += sizeof(double);
	ar.start.y  = *(double *)m; m += sizeof(double);
	ar.start.z  = *(double *)m; m += sizeof(double);
	ar.end  .x  = *(double *)m; m += sizeof(double);
	ar.end  .y  = *(double *)m; m += sizeof(double);
	ar.end  .z  = *(double *)m; m += sizeof(double);
	ar.normal.x = *(double *)m; m += sizeof(double);
	ar.normal.y = *(double *)m; m += sizeof(double);
	ar.normal.z = *(double *)m; m += sizeof(double);
	ar.radius   = *(double *)m; m += sizeof(double);
}


static const size_t CODE_SIZE = 16;
static const size_t POINT_CODE             = 0x0FD2FFC0;
static const size_t OPOINT_CODE            = 0x0FD2FFC8;
static const size_t TEXTPOINT_CODE         = 0x0FD2FFC9;
static const size_t LINE_CODE              = 0x0FD2FFC1;
static const size_t POLYLINE_CODE          = 0x0FD2FFC2;
static const size_t POLYGON_CODE           = 0x0FD2FFC3;
static const size_t RECTANGLE_CODE         = 0x0FD2FFC7;
static const size_t BOUNDARY_CODE          = 0x0FD2FFC5;
static const size_t COLLECTION_CODE        = 0x0FD2FFC6;
static const size_t COMPOSITEPOLYLINE_CODE = 0x0FD2FFCB;
static const size_t COMPOSITEPOLYGON_CODE  = 0x0FD2FFCC;
static const size_t ARC_CODE               = 0x0FD2FFCA;


bool BlobCache::decode_collection( byte * p )
{
	assert(p);

	size_t num_item = *(size_t *)p; p += sizeof(size_t);
	
	co.resize(num_item);
	for(size_t i=0;i<num_item;++i)
	{
		size_t item_size = *(size_t *)p; p += sizeof(size_t);
		if(!co[i].decode(p))
			return false;
		p += item_size;
	}

	return true;
}


bool BlobCache::decode( byte * buf )
{
	assert(buf);

	
	byte * p = buf + CODE_SIZE;	// Skip the code

	type = T_NONE;

	switch( *(size_t *)buf )
	{
	case POINT_CODE:
		{
			decode_point3d( pt, p );
			type = T_POINT;
		}
		break;
	case OPOINT_CODE:
		{
			decode_point3d( pt, p );
			double i = *(double *)p; p += sizeof(double);
			double j = *(double *)p;
				// k ignored, must be 0
			an = atan2(j,i);
			type = T_OPOINT;
		}
		break;
	case TEXTPOINT_CODE:
		{
			decode_point3d( pt, p );
			an   = *(double *)p; p += sizeof(double);
			p += sizeof(double)*3;	// Skip i,j,k axis
			p += sizeof(byte)*2;	// Skip unused
			p += sizeof(byte)*2;	// Skip format and alignment
			size_t size = *(size_t *)p; p+= sizeof(size_t);
			assert(size<42000);			// Suspicious number ?
			tx.resize(size);
			std::copy(p,p+size,tx.begin());
			type = T_TEXT;
		}
		break;
	case LINE_CODE:
		{
			pl.l.resize(2);
			std::vector<point>::iterator i;
			for(i=pl.l.begin();i!=pl.l.end();++i)
				decode_point3d( *i, p );
			type = T_LINE;
		}
		break;
	case POLYLINE_CODE:
		{
			size_t size = *(size_t *)p; p+= sizeof(size_t);
			assert(size<42000000);			// Suspicious number ?
			pl.l.resize(size);
			std::vector<point>::iterator i;
			for(i=pl.l.begin();i!=pl.l.end();++i)
				decode_point3d( *i, p );
			type = T_LINE;
		}
		break;
	case POLYGON_CODE:
		{
			pg.r.resize(1);
			decode_ring( pg.r.front(),p);
			type = T_AREA;
		}
		break;
	case BOUNDARY_CODE:
		{
			pg.r.resize(1);

				// Esterior

			/*size_t e_size = *(size_t *)p;*/ p += sizeof(size_t);	// Exterior size
			  size_t e_code = *(size_t *)p;   p += CODE_SIZE;		// Exterior code
			if(e_code==POLYGON_CODE)
				decode_ring( pg.r.front(),p);
			else
			{
				assert(0);		// Other values unsupported
				return false;
			}

				// Interior

			/* size_t i_size = *(size_t *)p; */ p += sizeof(size_t);	// Interior size
			   size_t i_code = *(size_t *)p;    p += CODE_SIZE;			// Interior code
			if(i_code==POLYGON_CODE)
			{
				pg.r.resize(2);
				decode_ring( pg.r[1],p);
			}
			else if(i_code==COLLECTION_CODE)
			{
				size_t count = *(size_t *)p; p += sizeof(size_t);	// Interior size
				assert(count<32768);			// Suspicious number ?
				pg.r.resize(count+1);						// +1 is the exterior
				for(size_t i=0;i<count;++i)
				{
					/* size_t l_size = *(size_t *)p; */ p += sizeof(size_t);	// Element size
					   size_t l_code = *(size_t *)p;    p += CODE_SIZE;			// Element code
					if(l_code==POLYGON_CODE)
						decode_ring( pg.r[i+1],p);
					else
					{
						assert(0);		// Other values unsupported
						return false;
					}
				}
			}
			else
			{
				assert(0);		// Other values unsupported
				return false;
			}
			
			type = T_AREA;
		}
		break;
		
	case ARC_CODE:
		decode_arc(ar,p);
		type = T_ARC;
		break;

		
	case COMPOSITEPOLYLINE_CODE:
	case COMPOSITEPOLYGON_CODE:
	case COLLECTION_CODE:
		decode_collection(p);
		type = T_COLLECTION;
		break;

	case RECTANGLE_CODE:
				// TODO: Unsupported!
		assert(0);		// Other values unsupported
		return false;

	default: return false;
	};

	return true;
}


bool BlobCache::encode( byte * buf, size_t & len )
{
	static const byte code[12]={0xBC,0x8C,0xCF,0x11,0xAB,0xDE,0x08,0x00,0x36,0x01,0xB7,0x69};
	size_t key = 0;
	assert(buf);

	byte * p = buf+CODE_SIZE;
		
	switch(type)
	{
	case T_POINT:
		encode_point3d( pt, p );
		key = POINT_CODE;
		break;
	case T_OPOINT:
		encode_point3d( pt, p );
		*(double *)p = cos(an); p += sizeof(double);
		*(double *)p = sin(an); p += sizeof(double);
		*(double *)p = 0;       p += sizeof(double);
		key = OPOINT_CODE;
		break;
	case T_LINE:
		{
			*(size_t *)p = pl.l.size(); p+= sizeof(size_t);
			std::vector< point >::const_iterator i;
			for(i=pl.l.begin();i!=pl.l.end();++i)
				encode_point3d( *i, p );
			key = POLYLINE_CODE;
		}
		break;
	case T_AREA:
		if(pg.r.empty()) return false;
		if(pg.r.size()==1)
		{
			encode_ring( pg.r.front(),p);
			key = POLYGON_CODE;
		}
		else
		{
			assert(0);
			/* TODO
			*(size_t *)p = CODE_SIZE + sizeof(size_t) + polygon.rings.front().size()*sizeof(double)*3; p += sizeof(size_t);
			*(size_t *)p = POLYGON_CODE; p += sizeof(size_t);
			memcpy(p,code,12);           p += 12;
			encode_ring( polygon.rings.front(),p);
			*(size_t *)p = CODE_SIZE +  xxxx sizeof(size_t) + polygon.rings.front().size()*sizeof(double)*3; p += sizeof(size_t);
			
			key = BOUNDARY_CODE;
			*/
		}
		break;
	case T_TEXT:
		{
			encode_point3d( pt, p );
			*(double *)p = an; p += sizeof(double);
			*(double *)p = 0;  p += sizeof(double);	// i,j,k direction
			*(double *)p = 0;  p += sizeof(double);
			*(double *)p = 1;  p += sizeof(double);
			*p = 0; ++p;		// unused space
			*p = 0; ++p;
			*p = 0; ++p;		// text format: normal
			*p = 0; ++p;		// alignment: center/center
			*(size_t *)p = tx.size(); p+= sizeof(size_t);
			std::copy(tx.begin(),tx.end(),p);
			p += tx.size();
			key = TEXTPOINT_CODE;
		}
		break;
		// TODO arc and other unsupported
	default: return false;
	}
	
		// Header
	*((size_t *)buf) = key;
	memcpy(buf+4,code,12);
	len = p - buf;

	return true;
}


}	// end namespace blo
