/*
 * XPath.c,v 1.1 2003/07/17 14:10:45 myui Exp
 *
 * What is this?
 *  -- An XPath processing system for XML Storage using RDBMS
 *
 * XPath 1.0 Public reference:
 *     http://www.w3.org/TR/xpath
 * 
 * @Author Makoto Yui
 *
 * ----------------------
 * Copyright (c) 2003 Makoto Yui <yuin@bb.din.or.jp>,
 *                       Atsuyuki Morishima <amorishima@acm.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote 
 *       products derived from this software without specific prior written
 *    permission.
 * 
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License (see COPYING.GPL), in which case the provisions 
 * of the GPL are required INSTEAD OF the above restrictions.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

//#define __DEBUG__
#define __PG__
#define GEQO_THRESHOLD 11       /* set the value in postgresql.conf, default 11 */
#define FORCE_USE_IDX_AXIS      /* should be always setted */
//#define XPATH2

#include <stdio.h>
#include <string.h>
#include <stdlib.h>         /* free() */
#include <ctype.h>          /* isdisit() */
#include <assert.h>         /* assert() */
#include <regex.h>
#include <glib.h>
#include "stack/stack.h"

#ifdef __PG__

#include "postgres.h"
#include "fmgr.h"
#include "utils/builtins.h"     /* textout() */
#include "executor/spi.h"       /* this is what you need to work with SPI */
#include "funcapi.h"            /* SRF */
#include "utils/builtins.h"     /* pg_atoi() */
#include <errno.h>              /* errno */
#include <math.h>               /* fbs() */

Datum xpath_eval(PG_FUNCTION_ARGS);
Datum xpath_evalx(PG_FUNCTION_ARGS);
Datum xpath2sql(PG_FUNCTION_ARGS);
Datum vchar2int(PG_FUNCTION_ARGS);
Datum vchar2real(PG_FUNCTION_ARGS);

#define FLOAT4_MAX         FLT_MAX
#define FLOAT4_MIN         FLT_MIN
#define GET_STR(textp) DatumGetCString(DirectFunctionCall1(textout, PointerGetDatum(textp)))

#ifndef NAN
#define NAN        (0.0/0.0)
#endif//NAN

static void CheckFloat4Val(double val);

#endif//__PG__

#define BUFSIZE 1024

#ifdef _Bool
#define bool _Bool
#else
#ifndef C_H
typedef char bool;
#endif//C_H
#endif//_Bool

#define KIND_ROOT       0
#define KIND_ELEMENT    1
#define KIND_TEXT       2
#define KIND_COMMENT    3
#define KIND_PI         4
#define KIND_ATTRIBUTE  5
#define KIND_NAMESPACE  6

/**
 *  XPath Evaluation Mode
 *
 *     bit
 *  0000 0000
 *     - order
 *    -  flagment
 *             - xml
 *
 *  example).
 *         0001 0000 is flagment OFF, ordering ON
 *         0011 0000 is flagment&ordering ON
 */

#define RET_SELECT      0x00    // 0). return select value un-ordered [default]
#define RET_ORD_SELECT  0x10    // 1). return select value ordered
#define RET_FLG         0x20    // 2). return node-flagment un-ordered
#define RET_ORD_FLG     0x30    // 3). return node-flagment ordered
#define RET_XML         0x01    // 4). return xml
#define SELECT_INTO     0xc0    // 5). 1100 0000
#define CREATE_VIEW     0x80    // 6). 1000 0000 

typedef enum
{
    AXIS_CHILD,                     // 0
    AXIS_DESCENDANT,                // 1
    AXIS_DESCENDANT_OR_SELF,        // 2
    AXIS_FOLLOWING,                 // 3
    AXIS_FOLLOWING_SIBLING,         // 4
 /* ------------------------------- */
    AXIS_ANCESTOR,                  // 5
    AXIS_ANCESTOR_OR_SELF,          // 6
    AXIS_PRECEDING,                 // 7
    AXIS_PRECEDING_SIBLING,         // 8
 /* ------------------------------- */
    AXIS_ATTRIBUTE,                 // 9
    AXIS_NAMESPACE,                 // 10
 /* ------------------------------- */
    AXIS_PARENT,                    // 11
    AXIS_SELF                       // 12
} AxisType;

typedef enum
{
    NTest_TAGNAME,              // 0
    NTest_PREFIX_TAG,           // 1
    NTest_TEXT,                 // 2
    NTest_COMMENT,              // 3
    NTest_PI,                   // 4
    NTest_PI_TRG,               // 5
    NTest_NODE,                 // 6
    NTest_ASTER,                // 7
    NTest_PREFIX_ASTER          // 8
} NodeTestType;

typedef struct __prevCondition
{
    AxisType prevAxis;
    NodeTestType prevNTest;
} prevCondition;

typedef struct __QStack
{
    Stack *ST_From;
    Stack *ST_Where;
} QStack;

typedef struct XPathData_tag
{
    int scount;                 //step counter
    int fcount;                 //from counter
    int ffirst;                 //first xml_node label
    unsigned char eval_mode;    //XPath evaluation mode
    char *docfilter;            //Document filter
    bool distinct;              //use dintinct
    bool path_q;                //Expr only with Path
    bool popflag;               //pop stack
    GString *spath;             //Path String
    QStack *qstack_val;         //Query Concreate Stack
    AxisType evaledAxis;        //current Axis
    prevCondition prevCond;     //previous Condition
} XPathData;

static char *translateXPath(char *,char);
static int translateStep(char *, GString **, XPathData *);
static int translateAxis(char *, GString **, XPathData *);
static int translateNodeTest(char *, GString **, XPathData *);
static int translatePredicate(char *, GString **, XPathData *, int);
static bool makeSQL(GString **,XPathData *,int,char);
static char *prep(XPathData *, char *);
static char *extract_docinfo(XPathData *, char *);
static char *getFormalExp(char *);
static int StrReplace( char *, const char *, const char *);
static bool StrisDigit(char *);
char *strrstr(const char *, const char *);
static void trim(char *);
static char *strxtok(char *, const char *, char **);

#ifdef __PG__

PG_FUNCTION_INFO_V1(xpath_eval);
Datum
xpath_eval(PG_FUNCTION_ARGS)
{
    FuncCallContext *funcctx;
    int            call_cntr;
    int            max_calls;
    TupleTableSlot *slot;
    AttInMetadata *attinmeta;
    TupleDesc s_tupdesc;
    SPITupleTable *fctx_tuptable;

    /* stuff done only on the first call of the function */
    if (SRF_IS_FIRSTCALL())
    {

        MemoryContext oldcontext;
        int ret;
        char *xpr;
        char *sql;
        char mode = RET_SELECT;

#ifdef __DEBUG__
        elog(NOTICE,"[xpath_eval] enter");
#endif

        /* create a function context for cross-call persistence */
        funcctx = SRF_FIRSTCALL_INIT();

        /* switch to memory context appropriate for multiple function calls */
        oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

        xpr = GET_STR(PG_GETARG_TEXT_P(0));

        if (fcinfo->nargs == 2){

            switch(PG_GETARG_INT32(1)){
                case 0:
                    // default
                    break;
                case 1:
                    mode = RET_ORD_SELECT;
                    break;
                case 2:
                    mode = RET_FLG;
                    break;
                case 3:
                    mode = RET_ORD_FLG;
                    break;
                default:
#ifdef __DEBUG__
                    elog(ERROR,"mode number is wrong\n");
#endif
                    PG_RETURN_NULL();
            }
        }

        if( (sql = translateXPath(xpr,mode)) == FALSE){

#ifdef __DEBUG__
            elog(ERROR,"wrong XPath expression\n");
#endif
            PG_RETURN_NULL();
        }


         /* Connect to SPI manager */
         if ((ret = SPI_connect()) < 0)
            elog(ERROR, "eval_sql: SPI_connect returned %d\n", ret);

        /* Retrieve the desired tuple */
        ret = SPI_exec(sql, 0);
        pfree(xpr);
        g_free(sql);

#ifdef __DEBUG__
        elog(NOTICE,"SPI_processed=%d\n",SPI_processed);
#endif

        /*
         * Only allow one qualifying tuple
         */
        if ((ret == SPI_OK_SELECT) && (SPI_processed > 0)){
            funcctx->max_calls = SPI_processed;
            s_tupdesc = SPI_tuptable->tupdesc;
        }
        else
        {
            SPI_finish();
            // RESULT SET IS NULL
            PG_RETURN_NULL();
        }

        /* allocate a slot */
        slot = TupleDescGetSlot(s_tupdesc);
        funcctx->slot = slot;

        /* Generate attribute metadata needed later */
        attinmeta = TupleDescGetAttInMetadata(s_tupdesc);
        funcctx->attinmeta = attinmeta;

        /* allocate memory for user context */
        fctx_tuptable = (SPITupleTable *) palloc(sizeof(SPITupleTable));
        fctx_tuptable = SPI_tuptable;

        /* Save spi data for use across calls */
        funcctx->user_fctx = fctx_tuptable;

        MemoryContextSwitchTo(oldcontext);
    }

    /* stuff done on every call of the function */
    funcctx = SRF_PERCALL_SETUP();

    /* return slot for our tuple */
    slot = funcctx->slot;
    attinmeta = funcctx->attinmeta;

    /* initialize per-call variables */
    call_cntr = funcctx->call_cntr;
    max_calls = funcctx->max_calls;

    /* the sql tuple */
    fctx_tuptable = (SPITupleTable *)funcctx->user_fctx;
    s_tupdesc = fctx_tuptable->tupdesc;


    if (call_cntr < max_calls)    /* do when there is more left to send */
    {
        char      **values;
        HeapTuple    res_tuple;
        HeapTuple    s_tuple;
        Datum        result;
        int i;
        int    nfields = s_tupdesc->natts;

        /* get the next sql result tuple */
        s_tuple = fctx_tuptable->vals[call_cntr];

#ifdef __DEBUG__
        elog(NOTICE,"nfields=%d\n",nfields);
#endif

        values = (char **) palloc(nfields * sizeof(char *));
        for (i = 0; i < nfields; i++){
            values[i] = SPI_getvalue(s_tuple,s_tupdesc,i+1);
#ifdef __DEBUG__
            elog(NOTICE,"call_cntr[%d]:values[%d]=%s\n",call_cntr,i,values[i]);
#endif
        }

        /* build the tuple */
        res_tuple = BuildTupleFromCStrings(attinmeta, values);

        /* make the tuple into a datum */
        result = TupleGetDatum(slot, res_tuple);

        /* Clean up (this is not actually necessary) */
        for(i=0;i<nfields;i++){
            if (values[i] != NULL)
                pfree(values[i]);
        }
        pfree(values);

        SRF_RETURN_NEXT(funcctx,result);
    }
    else    /* do when there is no more left */
    {
        pfree(fctx_tuptable);
        SPI_finish();
        SRF_RETURN_DONE(funcctx);
    }
}


PG_FUNCTION_INFO_V1(xpath_evalx);
Datum
xpath_evalx(PG_FUNCTION_ARGS)
{
    FuncCallContext *funcctx;
    int            call_cntr;
    int            max_calls;
    TupleTableSlot *slot;
    AttInMetadata *attinmeta;
    TupleDesc s_tupdesc;
    SPITupleTable *fctx_tuptable;

    /* stuff done only on the first call of the function */
    if (SRF_IS_FIRSTCALL())
    {

        MemoryContext oldcontext;
        int ret;
        char *xpr;
        char *sql;

#ifdef __DEBUG__
        elog(NOTICE,"[xpath_eval] enter");
#endif

        /* create a function context for cross-call persistence */
        funcctx = SRF_FIRSTCALL_INIT();

        /* switch to memory context appropriate for multiple function calls */
        oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);

        xpr = GET_STR(PG_GETARG_TEXT_P(0));

        if( (sql = translateXPath(xpr,RET_XML)) == FALSE){

#ifdef __DEBUG__
            elog(ERROR,"wrong XPath expression\n");
#endif
            PG_RETURN_NULL();
        }


         /* Connect to SPI manager */
         if ((ret = SPI_connect()) < 0)
            elog(ERROR, "eval_sql: SPI_connect returned %d\n", ret);

        /* Retrieve the desired tuple */
        ret = SPI_exec(sql, 0);
        pfree(xpr);
        g_free(sql);

#ifdef __DEBUG__
        elog(NOTICE,"SPI_processed=%d\n",SPI_processed);
#endif

        /*
         * Only allow one qualifying tuple
         */
        if ((ret == SPI_OK_SELECT) && (SPI_processed > 0)){
            funcctx->max_calls = SPI_processed;
            s_tupdesc = SPI_tuptable->tupdesc;
        }
        else
        {
            SPI_finish();
            // RESULT SET IS NULL
            PG_RETURN_NULL();
        }

        /* allocate a slot */
        slot = TupleDescGetSlot(s_tupdesc);
        funcctx->slot = slot;

        /* Generate attribute metadata needed later */
        attinmeta = TupleDescGetAttInMetadata(s_tupdesc);
        funcctx->attinmeta = attinmeta;

        /* allocate memory for user context */
        fctx_tuptable = (SPITupleTable *) palloc(sizeof(SPITupleTable));
        fctx_tuptable = SPI_tuptable;

        /* Save spi data for use across calls */
        funcctx->user_fctx = fctx_tuptable;

        MemoryContextSwitchTo(oldcontext);
    }

    /* stuff done on every call of the function */
    funcctx = SRF_PERCALL_SETUP();

    /* return slot for our tuple */
    slot = funcctx->slot;
    attinmeta = funcctx->attinmeta;

    /* initialize per-call variables */
    call_cntr = funcctx->call_cntr;
    max_calls = funcctx->max_calls;

    /* the sql tuple */
    fctx_tuptable = (SPITupleTable *)funcctx->user_fctx;
    s_tupdesc = fctx_tuptable->tupdesc;


    if (call_cntr < max_calls)    /* do when there is more left to send */
    {
        char      **values;
        HeapTuple    res_tuple;
        HeapTuple    s_tuple;
        Datum        result;
        int i;
        int    nfields = s_tupdesc->natts;

        /* get the next sql result tuple */
        s_tuple = fctx_tuptable->vals[call_cntr];

#ifdef __DEBUG__
        elog(NOTICE,"nfields=%d\n",nfields);
#endif

        values = (char **) palloc(nfields * sizeof(char *));
        for (i = 0; i < nfields; i++){
            values[i] = SPI_getvalue(s_tuple,s_tupdesc,i+1);
#ifdef __DEBUG__
            elog(NOTICE,"call_cntr[%d]:values[%d]=%s\n",call_cntr,i,values[i]);
#endif
        }

        /* build the tuple */
        res_tuple = BuildTupleFromCStrings(attinmeta, values);

        /* make the tuple into a datum */
        result = TupleGetDatum(slot, res_tuple);

        /* Clean up (this is not actually necessary) */
        for(i=0;i<nfields;i++){
            if (values[i] != NULL)
                pfree(values[i]);
        }
        pfree(values);

        SRF_RETURN_NEXT(funcctx,result);
    }
    else    /* do when there is no more left */
    {
        pfree(fctx_tuptable);
        SPI_finish();
        SRF_RETURN_DONE(funcctx);
    }
}

PG_FUNCTION_INFO_V1(xpath2sql);
Datum
    xpath2sql(PG_FUNCTION_ARGS)
{
    char *query;
    text *result;
    char *xpr=DatumGetCString(DirectFunctionCall1(textout, PG_GETARG_DATUM(0)));
    char mode = RET_SELECT;

    if (fcinfo->nargs == 2)
    {

        switch(PG_GETARG_INT32(1)){
            case 0:
                // default
                break;
            case 1:
                mode = RET_ORD_SELECT;
                break;
            case 2:
                mode = RET_FLG;
                break;
            case 3:
                mode = RET_ORD_FLG;
                break;
            case 4:
                mode = RET_XML;
                break;
            default:
#ifdef __DEBUG__
                elog(ERROR,"mode number is wrong\n");
#endif
                PG_RETURN_NULL();
        }
    }

    if( (query = translateXPath(xpr,mode)) == FALSE)
    {
#ifdef __DEBUG__
        elog(ERROR,"wrong XPath expression");
#endif
        PG_RETURN_NULL();
    }

    result = DatumGetTextP(DirectFunctionCall1(textin, CStringGetDatum(query)));
    g_free(query);
    PG_RETURN_TEXT_P(result);
}

/*
 *        vchar2int        - is number?
 */
PG_FUNCTION_INFO_V1(vchar2int);
Datum
vchar2int(PG_FUNCTION_ARGS)
{
    char *num =
        DatumGetCString(DirectFunctionCall1(textout, PG_GETARG_DATUM(0)));
    
    if(StrisDigit(num)==TRUE)
        PG_RETURN_INT32(atoi(num));
    else
        PG_RETURN_NULL();
}

/*
 *        vchar2real        - is real number?
 */
PG_FUNCTION_INFO_V1(vchar2real);
Datum
vchar2real(PG_FUNCTION_ARGS)
{
    double        val;
    char       *endptr;
    char *num =
        DatumGetCString(DirectFunctionCall1(textout, PG_GETARG_DATUM(0)));

    errno = 0;
    val = strtod(num, &endptr);
    if (*endptr != '\0')
    {
        /*
         * XXX we should accept "Infinity" and "-Infinity" too, but what
         * are the correct values to assign?  HUGE_VAL will provoke an
         * error from CheckFloat4Val.
         */
        if (strcasecmp(num, "NaN") == 0)
            val = NAN;
        else
            PG_RETURN_NULL();
    }
    else
    {
        if (errno == ERANGE)
            elog(ERROR, "Input '%s' is out of range for float4", num);
    }

    /*
     * if we get here, we have a legal double, still need to check to see
     * if it's a legal float
     */
    CheckFloat4Val(val);

    PG_RETURN_FLOAT4((float4) val);
}

/*
 * check to see if a float4 val is outside of
 * the FLOAT4_MIN, FLOAT4_MAX bounds.
 *
 * raise an elog warning if it is
 */
static void
CheckFloat4Val(double val)
{
    /*
     * defining unsafe floats's will make float4 and float8 ops faster at
     * the cost of safety, of course!
     */
#ifdef UNSAFE_FLOATS
    return;
#else
    if (fabs(val) > FLOAT4_MAX)
        elog(ERROR, "Bad float4 input format -- overflow");
    if (val != 0.0 && fabs(val) < FLOAT4_MIN)
        elog(ERROR, "Bad float4 input format -- underflow");
    return;
#endif   /* UNSAFE_FLOATS */
}

#endif//__PG__

/*
 *    translateXPath - analysis Location Path
 */
static char *translateXPath(char *argv, char mode)
{

    int i;
    regex_t preg;
    char *exp;
    char *p,*sp;
    char *result;
    char *buf;
    bool justroot;
    GString *res;
    QStack qs;
    char sep[] = "/";
    char *pattern = "^(/([^][/: ]+::)?[^][/ ]+(\\[[^][]+\\])*)+$";
    QStack *qs_head = &qs;
    XPathData xd = {1,0,0,RET_FLG,NULL,FALSE,TRUE,FALSE,NULL,NULL,0};
    XPathData *xd_head = &xd;
    xd_head->eval_mode = mode;

#ifdef __DEBUG__
#ifdef __PG__
    elog(NOTICE,"translateXPath..argv='%s'\n",argv);
#else
    fprintf(stderr,"translateXPath..argv='%s'\n",argv);
#endif//__PG__
#endif//__DEBUG__

    // pre-processing
    exp = prep(xd_head, argv);

    justroot = (strcmp(exp,"/")==0) ? TRUE : FALSE;
    if(!justroot)
    {
        /* Complile RE Pattern */
        i=regcomp(&preg,pattern,REG_EXTENDED|REG_NEWLINE|REG_NOSUB);
        if (i != 0)
        {
            g_free(exp);
            return FALSE;
        }
    
        /* Evolute RE */
        i = regexec(&preg,exp,(size_t) 0, NULL, 0);
        regfree(&preg);
        if(i != 0 )
        {
            g_free(exp);
            return FALSE;    /* no match */
        }
    }

#ifdef __DEBUG__
#ifndef __PG___
    puts(exp);
#endif//__PG__
#endif//__DEBUG__

    /* prepare */
    res = g_string_sized_new(300);
    xd_head->spath = g_string_sized_new(96);
    
    /* create stack for query */
    qs_head->ST_From = stack_create();
    qs_head->ST_Where = stack_create();
    xd_head->qstack_val = qs_head;

    /* if expression is just '/' */
    if(justroot)
    {
        stack_push(qs_head->ST_From, "xml_node xn0", 13);
        stack_push(qs_head->ST_Where, "icount(xn0.dewey) = 3", 22);
    }
    else
    {
        p = strxtok(exp,sep,&sp);
          while(p !=NULL)
          {
              if( translateStep(p,&res,xd_head) == FALSE)
              {
                  g_free(exp);
                  g_string_free(res, TRUE);
                  g_string_free(xd_head->spath, TRUE);
                stack_destroy(qs_head->ST_From);
                stack_destroy(qs_head->ST_Where);
                  return FALSE;
              }
#ifdef __DEBUG__
              printf("Node Step=%s\n",res->str);
#endif
            p = strxtok(NULL,sep,&sp);
        }
    }

    // return flagment ?
    if((mode&RET_FLG)==RET_FLG)
    {
        stack_push(qs_head->ST_From, "xml_node xn", 12);

        buf = g_strdup_printf("xn.docid = xn%d.docid AND "
                            "_int_descendant_or_self(xn.dewey,xn%d.dewey)",
                            xd_head->scount-1,xd_head->scount-1);
        stack_push_pointer(qs_head->ST_Where, buf);

        buf = g_strdup(" ORDER BY xn.dewey");
    }
    else
        buf = g_strdup_printf(" ORDER BY xn%d.dewey", xd_head->scount-1);

    // Stack to SQL
    makeSQL(&res,xd_head,-1,mode);

    if((mode&RET_ORD_SELECT)==RET_ORD_SELECT)
        result = g_strconcat(res->str,buf,NULL);
    else
        result = res->str;

    g_free(buf);
    g_string_free(res, FALSE);

    /* clean up */
    g_free(exp);
    g_string_free(xd_head->spath, TRUE);
    stack_destroy(qs_head->ST_From);
    stack_destroy(qs_head->ST_Where);

    return (result);

}// translateXPath

static int translateStep(char *step, GString **query, XPathData *xd)
{
    int i;
    char *pend;
    char *p,*sp;
    char *buf;
    regex_t preg;
    QStack *lqs;
    char sep[] = "[]";
    char *pattern = "[[:digit:]]+\\]";
    prevCondition *lprevCond = &(xd->prevCond);

#ifdef __DEBUG__
      puts("translateStep\n");
#endif

    if(xd->popflag)    // there is next to predicate ex)./aaa/ccc[2]/ddd
    {
        lqs    = xd->qstack_val;
        buf = stack_pop(lqs->ST_From);
        g_free(buf);        
        buf = stack_pop(lqs->ST_Where);
        g_free(buf);
        (xd->scount)--;
        xd->popflag = FALSE;
    }

    /* axis */
    if( (pend=strchr(step,':')) == NULL)
    {
        if( translateAxis("child",&*query,xd) == FALSE)
            return FALSE;
    }
    else
    {
        *pend = '\0';
        if( translateAxis(step,&*query,xd) == FALSE)
            return FALSE;
        step = pend + 2;
    }

    /* nodetest */
    if( (pend=strchr(step,'[')) == 0)
    {
        if( translateNodeTest(step,&*query,xd) == FALSE)
            return FALSE;
    }
    else{
        *pend++ = '\0';

        // Complile RE Pattern .... to be fixed to compile one time
        i=regcomp(&preg,pattern,REG_EXTENDED|REG_NEWLINE|REG_NOSUB);
        if (i != 0)
            return FALSE;
    
        // Evolute RE
        i = regexec(&preg,pend,(size_t) 0, NULL, 0);
        regfree(&preg);
        if(i==0 && xd->path_q)
        {
            switch(xd->evaledAxis)
            {
                case AXIS_CHILD:
                case AXIS_DESCENDANT:
                case AXIS_DESCENDANT_OR_SELF:
                    break;
                default:
                    xd->path_q = FALSE;
            }        
        }

        if( translateNodeTest(step,&*query,xd) == FALSE)
            return FALSE;

        /* predicate */
        p = strtok_r(pend, sep, &sp);
        for(i=1; p!=NULL; i++)
        {
#ifdef __DEBUG__
              printf("predicate argv='%s'\n",p);
#endif
            /* query setteing */
            xd->path_q = FALSE;

            if( translatePredicate(p,&*query,xd,i) == FALSE)
                return FALSE;

            if(i>1)
                xd->popflag = FALSE;

#ifdef __DEBUG__
              printf("Predicate=%s\n",(*query)->str);
#endif
            p = strtok_r(NULL, sep, &sp);
        }
    }

    xd->scount += 1;        /* step depth */
    lprevCond->prevAxis = xd->evaledAxis;

    return TRUE;
}

static int translateAxis(char *axis, GString **query, XPathData *xd)
{
    char *buf;
    char *str = NULL;
    int lcnt    = xd->scount;
    QStack *lqs    = xd->qstack_val;

#ifdef __DEBUG__
#ifdef __PG__
    elog(NOTICE,"translateAxis..argv=%s\n",axis);
#else
    fprintf(stderr,"translateAxis..argv=%s\n",axis);
#endif//__PG__
#endif//__DEBUG__

    switch (axis[0]) 
    {
        case 'a':
            if (!strcmp(axis,"ancestor"))
            {
                xd->evaledAxis = AXIS_ANCESTOR;
                xd->distinct = TRUE;
                xd->path_q = FALSE;
                // TODO index access
                str = "_int_ancestor";
                break;
            }
            if (!strcmp(axis,"ancestor-or-self"))
            {
                xd->evaledAxis = AXIS_ANCESTOR_OR_SELF;
                xd->distinct = TRUE;
                xd->path_q = FALSE;
                // TODO index access
                str = "_int_ancestor_or_self";
                break;
            }
            if (!strcmp(axis,"attribute"))
            {
                xd->evaledAxis = AXIS_ATTRIBUTE;
                if(xd->scount!=0)
                {
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;
    
                    buf = g_strdup_printf("xn%d.parent = xn%d.id AND "
                                          "xn%d.kind = '%d'",
                                          lcnt,lcnt-1,lcnt,KIND_ATTRIBUTE
                                          );
                    stack_push_pointer(lqs->ST_Where, buf);
                    
                    return TRUE;
                }
                break;
            }
             return FALSE;
        case 'c':
            if (!strcmp(axis,"child"))
            {
                xd->evaledAxis = AXIS_CHILD;
                if(xd->scount!=0)
                {
                    // use link for child
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

                    buf = g_strdup_printf("xn%d.parent = xn%d.id AND "
                                          "xn%d.kind < 5",
                                          lcnt,lcnt-1,lcnt
                                          );
                    stack_push_pointer(lqs->ST_Where, buf);
                    
                    return TRUE;
                }
                break;
            }
            return FALSE;
        case 'd':
            xd->distinct = TRUE;
            if (!strcmp(axis,"descendant"))
            {
                buf = g_strdup_printf("xml_node xn%d",lcnt);
                stack_push_pointer(lqs->ST_From, buf);
                xd->fcount++;
                
                buf = g_strdup_printf("xn%d.dewey > xn%d.dewey AND "
                                      "xn%d.dewey < nextSibling(xn%d.dewey) AND "
                                      "xn%d.kind < 5",
                                    lcnt,lcnt-1,lcnt,lcnt-1,lcnt
                                    );
                stack_push_pointer(lqs->ST_Where, buf);
                
                xd->evaledAxis = AXIS_DESCENDANT;
                return TRUE;
            }
            if (!strcmp(axis,"descendant-or-self"))
            {
                buf = g_strdup_printf("xml_node xn%d",lcnt);
                stack_push_pointer(lqs->ST_From, buf);
                xd->fcount++;
                
                buf = g_strdup_printf("xn%d.dewey >= xn%d.dewey AND "
                                      "xn%d.dewey < nextSibling(xn%d.dewey) AND "
                                      "(xn%d.kind < 5 OR xn%d.id=xn%d.id)",
                                    lcnt,lcnt-1,lcnt,lcnt-1,lcnt,lcnt,lcnt-1
                                    );
                stack_push_pointer(lqs->ST_Where, buf);
                
                xd->evaledAxis = AXIS_DESCENDANT_OR_SELF;
                return TRUE;
            }
            return FALSE;
        case 'f':
            xd->distinct = TRUE;
            if (!strcmp(axis,"following"))
            {
                xd->evaledAxis = AXIS_FOLLOWING;
                xd->path_q = FALSE;
                if(xd->scount!=0)
                {
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

#ifdef FORCE_USE_IDX_AXIS
                    // Since it is a hedge model, have to insert in between.
                    // xn3.docid = xn2.docid
                    buf = g_strdup_printf("xn%d.dewey >= nextSibling(xn%d.dewey) AND "
                                          "xn%d.dewey > subarray(xn%d.dewey,1,3)",
                                          lcnt,lcnt-1,lcnt,lcnt-1
                                          );
#else
                    buf = g_strdup_printf("_int_following(xn%d.dewey,xn%d.dewey)",
                                          lcnt,lcnt-1
                                          );
#endif //FORCE_USE_IDX_AXIS
                    stack_push_pointer(lqs->ST_Where, buf);

                    return TRUE;
                }
                break;
            }
            if (!strcmp(axis,"following-sibling"))
            {
                xd->evaledAxis = AXIS_FOLLOWING_SIBLING;
                xd->path_q = FALSE;
                if(xd->scount!=0)
                {
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

                    buf = g_strdup_printf("xn%d.parent = xn%d.parent AND "
                                          "xn%d.dewey > xn%d.dewey AND "
                                          "xn%d.kind < 5",
                                          lcnt,lcnt-1,lcnt,lcnt-1,lcnt
                                          );
                    stack_push_pointer(lqs->ST_Where, buf);

                    return TRUE;
                }
                break;
            }
            return FALSE;
        case 'n':
            if (!strcmp(axis,"namespace"))
            {
                xd->evaledAxis = AXIS_NAMESPACE;
                xd->path_q = FALSE;        // current ot supported
                if(xd->scount!=0)
                {
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;
    
                    buf = g_strdup_printf("xn%d.parent = xn%d.id AND "
                                          "xn%d.kind = '%d'",
                                          lcnt,lcnt-1,lcnt,KIND_NAMESPACE
                                          );
                    stack_push_pointer(lqs->ST_Where, buf);
                    
                    return TRUE;
                }
                break;
            }
            return FALSE;
        case 'p':
            xd->distinct = TRUE;
            if (!strcmp(axis,"parent"))
            {
                xd->evaledAxis = AXIS_PARENT;
                xd->path_q = FALSE;
                if(xd->scount!=0)
                {
                    /* use link for parent */
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

                    buf = g_strdup_printf("xn%d.id = xn%d.parent",lcnt,lcnt-1);
                    stack_push_pointer(lqs->ST_Where, buf);

                    return TRUE;
                }
                break;
            }
            if (!strcmp(axis,"preceding"))
            {
                xd->evaledAxis = AXIS_PRECEDING;
                xd->path_q = FALSE;
                if(xd->scount!=0)
                {
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

#ifdef FORCE_USE_IDX_AXIS
                    // TODO
                    buf = g_strdup_printf("xn%d.dewey < xn%d.dewey AND "
                                          "_int_preceding(xn%d.dewey,xn%d.dewey)",
                                          lcnt,lcnt-1,lcnt,lcnt-1
                                        );
#else
                    buf = g_strdup_printf("_int_preceding(xn%d.dewey,xn%d.dewey)",
                                          lcnt,lcnt-1
                                        );
#endif//FORCE_USE_IDX_AXIS
                    stack_push_pointer(lqs->ST_Where, buf);

                    return TRUE;
                }
                break;
            }
            if (!strcmp(axis,"preceding-sibling"))
            {
                xd->evaledAxis = AXIS_PRECEDING_SIBLING;
                xd->path_q = FALSE;
                if(xd->scount!=0)
                {
                    buf = g_strdup_printf("xml_node xn%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

                    buf = g_strdup_printf("xn%d.parent = xn%d.parent AND "
                                          "xn%d.dewey < xn%d.dewey AND "
                                          "xn%d.kind < 5",
                                          lcnt,lcnt-1,lcnt,lcnt-1,lcnt
                                          );
                    stack_push_pointer(lqs->ST_Where, buf);

                    return TRUE;
                }
                break;
            }
            return FALSE;
        case 's':
            if (!strcmp(axis,"self"))
            {
                xd->evaledAxis = AXIS_SELF;
                xd->path_q = FALSE;        // current...
                return TRUE;
            }
            return FALSE;
        default:
            return FALSE;
    }

    if(xd->scount == 1)
    {
        stack_push(lqs->ST_From, "xml_node xn0", 13);

        switch(xd->evaledAxis)
        {
            case AXIS_CHILD:
                 stack_push(lqs->ST_Where, "icount(xn0.dewey) = 3", 22);
                 break;
             case AXIS_DESCENDANT:
                 stack_push(lqs->ST_Where, "xn0.kind > 0 AND xn0.kind < 5", 30);
                 break;
             case AXIS_DESCENDANT_OR_SELF:
                 stack_push(lqs->ST_Where, "xn0.kind < 5", 13);
                 break;
             case AXIS_SELF:
             case AXIS_ANCESTOR_OR_SELF:
                stack_push(lqs->ST_Where, "xn0.kind = '0'", 15);
                 break;
            default:
#ifdef __PG__
                elog(NOTICE,"[XPath] syntax error");
#else
                fprintf(stderr,"[XPath] syntax error");
#endif//__PG__
                return FALSE;
        }
    }
    else
    {
        buf = g_strdup_printf("xml_node xn%d",lcnt);
        stack_push_pointer(lqs->ST_From, buf);

        buf = g_strdup_printf("%s(xn%d.dewey,xn%d.dewey)",str,lcnt,lcnt-1);
        stack_push_pointer(lqs->ST_Where, buf);
    }

    // query setteing
    xd->fcount++;

    // cleanup
    return TRUE;
}

static int translateNodeTest(char *ntest, GString **query, XPathData *xd)
{

    char *ptr,*sp;
    char *ope;
    NodeTestType lNTest;
    char *buf                 = NULL;
    bool lpq                 = FALSE;
    int lcnt                 = xd->scount;
    AxisType lAxis             = xd->evaledAxis;
    QStack *lqs                 = xd->qstack_val;
    prevCondition *lprevCond = &(xd->prevCond);
    bool hasDep = TRUE;    // includes dependant nodes ?

#ifdef __DEBUG__
#ifdef __PG__
    elog(NOTICE,"translateNodeTest..argv=%s\n",ntest);
#else
    fprintf(stderr,"translateNodeTest..argv=%s\n",ntest);
#endif//__PG__
#endif//__DEBUG__

    if(!strcmp(ntest,"text()"))
    {
        lNTest = NTest_TEXT;
        buf = g_strdup_printf("xn%d.kind = '2'",lcnt);
        stack_push_pointer(lqs->ST_Where, buf);
    }
    else if(!strcmp(ntest,"comment()"))
    {
        lNTest = NTest_COMMENT;
        buf = g_strdup_printf("xn%d.kind = '3'",lcnt);
        stack_push_pointer(lqs->ST_Where, buf);
    }
    else if(!strcmp(ntest,"node()"))
    {
        lNTest = NTest_NODE;
        
        if(xd->path_q)
        {
            switch(lAxis)
            {
                case AXIS_DESCENDANT_OR_SELF:
                    xd->path_q = TRUE;
                    lprevCond->prevNTest = lNTest;
                    return TRUE;
                case AXIS_DESCENDANT:
                    lpq = TRUE;
                    hasDep = FALSE;
                    g_string_append(xd->spath,"#/%");
                    break;
                case AXIS_ATTRIBUTE:
                    lpq = TRUE;
                    if(lprevCond->prevAxis==AXIS_DESCENDANT_OR_SELF)
                        g_string_append(xd->spath,"#%/@%");
                    else
                        g_string_append(xd->spath,"#/@%");
                    break;
                default:
                    break;
            }
        }
        else
            if(lAxis==AXIS_SELF) xd->scount -= 1;
    }
    else if(!strncmp(ntest,"processing-instruction(",23))
    {
        lNTest = NTest_PI;
        ntest += 23;

        if(*ntest == ')')
        {
            buf = g_strdup_printf("xn%d.kind = '6'",lcnt);
            stack_push_pointer(lqs->ST_Where, buf);
        }
        else
        {
            /* processing-instruction(hoge) */
            ptr = strtok_r(ntest,")",&sp);

            if( ptr == NULL)
                return FALSE;
            else
            {
                lNTest = NTest_PI_TRG;
                buf = g_strdup_printf("xml_pi%d xi",lcnt);
                stack_push_pointer(lqs->ST_From, buf);
                xd->fcount++;

                buf = g_strdup_printf("xi%d.tagid = xn%d.tagid AND "
                                    "xi%d.name = '%s'",
                                    lcnt,lcnt,lcnt,ptr
                                    );
                stack_push_pointer(lqs->ST_Where, buf);
            }
        }
    }
    else if(!strcmp(ntest,"*"))
    {
        lNTest = NTest_ASTER;
        switch(lAxis)
        {
            case AXIS_DESCENDANT_OR_SELF:
                lpq = TRUE;
            case AXIS_ATTRIBUTE:
                if(xd->path_q)
                {
                    lpq = TRUE;
                    if(lprevCond->prevAxis==AXIS_DESCENDANT_OR_SELF)
                    {
                        g_string_append(xd->spath,"#%/@%");
                    }
                    else
                        g_string_append(xd->spath,"#/@%");
                }
                else
                {
                    buf = g_strdup_printf("xn%d.kind = '4'",lcnt);
                    stack_push_pointer(lqs->ST_Where, buf);
                }
                break;
            case AXIS_NAMESPACE:
                buf = g_strdup_printf("xn%d.kind = '5'",lcnt);
                stack_push_pointer(lqs->ST_Where, buf);
                break;
            default:
                buf = g_strdup_printf("xn%d.kind = '1'",lcnt);
                stack_push_pointer(lqs->ST_Where, buf);
        }
    }
    else
    {
        ptr = strtok_r(ntest,":",&sp);
        if( ptr==ntest)
        {
            lNTest = NTest_TAGNAME;
            switch(lAxis)
            {
                case AXIS_ATTRIBUTE:
                    if(xd->path_q)
                    {
                        lpq = TRUE;
                        if(lprevCond->prevAxis==AXIS_DESCENDANT_OR_SELF)
                            g_string_append_printf(xd->spath,"#%%/@%s",ntest);
                        else
                            g_string_append_printf(xd->spath,"#/@%s",ntest);
                    }
                    else
                    {
                        buf = g_strdup_printf("xml_attribute xt%d",lcnt);
                        stack_push_pointer(lqs->ST_From, buf);
                        xd->fcount++;
    
                        buf = g_strdup_printf("xt%d.tagid = xn%d.tagid AND "
                                            "xt%d.name = '%s'",
                                            lcnt,lcnt,lcnt,ntest
                                               );
                        stack_push_pointer(lqs->ST_Where, buf);
                    }
                    break;
                case AXIS_NAMESPACE:
                    buf = g_strdup_printf("xml_namespace xt%d",lcnt);
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount++;

                    buf = g_strdup_printf("xt%d.tagid = xn%d.tagid AND "
                                        "xt%d.name = '%s'",
                                        lcnt,lcnt,lcnt,ntest
                                        );
                    stack_push_pointer(lqs->ST_Where, buf);
                    break;
                default:
                    if(xd->path_q)
                    {
                        switch(lAxis)
                        {
                            case AXIS_CHILD:
                                lpq = TRUE;
                                if(lprevCond->prevAxis==AXIS_DESCENDANT_OR_SELF)
                                    g_string_append_printf(xd->spath,"#%%/%s",ntest);
                                else
                                    g_string_append_printf(xd->spath,"#/%s",ntest);
                                break;
                            case AXIS_DESCENDANT:
                                lpq = TRUE;
                                g_string_append_printf(xd->spath,"#%%/%s",ntest);
                                break;
                            default:
                                break;
                        }
                    }
                    else
                    {
                        buf = g_strdup_printf("xml_tag xt%d",lcnt);
                        stack_push_pointer(lqs->ST_From, buf);
                        xd->fcount++;
    
                        buf = g_strdup_printf("xt%d.tagid = xn%d.tagid AND xt%d.name = '%s'",
                                            lcnt,lcnt,lcnt,ntest
                                            );
                        stack_push_pointer(lqs->ST_Where, buf);
                    }
            }
        }
        else
        {
            // prefix::nodetest
            lNTest = NTest_PREFIX_TAG;
#ifdef __PG__
            elog(ERROR,"[XPath] syntax '%s' is not yet supported",ntest);
#else
            fprintf(stderr,"[XPath] syntax '%s' is not yet supported",ntest);
#endif//__PG__
            return FALSE;
        }
    }

    if(lpq)
    {
        char *spath = xd->spath->str;

        if(strchr(spath,'%')==NULL)
            ope = "=";
        else
            ope = "LIKE";

        /* initialize stack */
        stack_init(lqs->ST_From);
        stack_init(lqs->ST_Where);

        buf = g_strdup_printf("xml_path xp%d,xml_node xn%d",lcnt,lcnt);
        stack_push_pointer(lqs->ST_From, buf);

        if(hasDep)
        {
            buf = g_strdup_printf("xp%d.pathexp %s '%s' AND "
                                "xn%d.pathid = xp%d.pathid",
                                lcnt,ope,spath,lcnt,lcnt
                                );
        }
        else
        {
            buf = g_strdup_printf("xp%d.pathexp %s '%s' AND "
                                "xn%d.pathid = xp%d.pathid AND "
                                "xn%d.kind < 5",
                                lcnt,ope,spath,lcnt,lcnt,lcnt
                                );        
        }
            
        stack_push_pointer(lqs->ST_Where, buf);

        /* query setting */
        xd->fcount = 2;
        xd->ffirst = lcnt;
    }

    /* query setting */
    xd->path_q  = lpq;
    lprevCond->prevNTest = lNTest;
    
    return TRUE;

}//translateNodeTest


static int translatePredicate(char *argv, GString **query, XPathData *xd, int pcnt)
{
    static int s_cnt_pos;
    char *p_ext;
    char *ptr;
    char *buf;
    GString *clause;
    
    // for regex
    regex_t preg;
    char *pattern;
    regmatch_t pmatch[8];
    
    // initializing
    char *predicate = strdup(argv);
    int i        = 0;
    bool isDes = FALSE;
    
    int lcnt    = xd->scount;
    QStack *lqs    = xd->qstack_val;

    char *from[]    = {
                      "position()="
                      };

    char *to[]        = {
                       ""
                      };

    char *t_from[] = {
                      "/",        /* // -> #/#/ */
                      "#/#",        /* #/#/ -> #%/ */
                      "*"
                      };

    char *t_to[]    = {
                    "#/",
                       "#%",
                       "%"
                      };

#ifdef __DEBUG__
#ifdef __PG__
    elog(NOTICE,"translatePredicate..argv=%s",argv);
#else
    fprintf(stderr,"translatePredicate..argv=%s",argv);
#endif//__PG__
#endif//__DEBUG__

    /* replace */
    for(;i<1;i++)
        StrReplace(predicate,from[i],to[i]);

    /* hoge[2] */
    if( StrisDigit(predicate))
    {
        AxisType aval = xd->evaledAxis;
        GString *subquery;

        /* some error check routine */
        if(aval>8)
        {
            int nval = atoi(predicate);

            if(aval<11)
            {
#ifdef __PG__
                elog(NOTICE,"[XPath] syntax error '%s'\n",predicate);
#else
                fprintf(stderr,"[XPath] syntax error '%s'\n",predicate);
#endif//__PG__
                return FALSE;    /* {attribute|namescape} has no order */
            }

            if(nval!=1)
            {
#ifdef __PG__
                elog(NOTICE,"[XPath] syntax error '%s'\n",predicate);
#else
                fprintf(stderr,"[XPath] syntax error '%s'\n",predicate);
#endif//__PG__
                return FALSE;    /* /foo/parent::hoge[2] */
            }
        }
        
        // case /aaa/bbb[2][3]
        if(xd->popflag)
            return FALSE;

        makeSQL(&(*query),xd,s_cnt_pos,SELECT_INTO);
        
        subquery = g_string_sized_new(200);

        g_string_append_printf(subquery,
                        "(SELECT "
                                "_xn2.target_id as id "
                            "FROM "
                                "tmp_Q%d as _xn1,"
                                "tmp_Q%d as _xn2 "
                            "WHERE ",
                        s_cnt_pos,s_cnt_pos
                        );

        if(aval<5)
        {
            // order by document order
            g_string_append_printf(subquery,
                            "_xn2.dewey >= _xn1.dewey and "
                            "_xn2.group_id = _xn1.group_id "
                            );
        }
        else if(aval<9)
        {
            // order by reverse document order
            g_string_append_printf(subquery,
                            "_xn1.dewey >= _xn2.dewey "
                            );
        }

        g_string_append_printf(subquery,
                            "GROUP BY "
                                "_xn2.target_id "
                            "HAVING "
                                "COUNT(*) = %s"
                            ") as xn%d",
                            predicate,lcnt
                        );

        buf = subquery->str;
        g_string_free(subquery, FALSE);

        // push into FROM stack
        stack_push_pointer(lqs->ST_From, buf);

        // query setteing
        xd->scount++;
        xd->fcount = 1;
        xd->popflag = TRUE;
        lcnt = xd->scount;

        // push into From stack
        buf =  g_strdup_printf("xml_node xn%d",lcnt);
        stack_push_pointer(lqs->ST_From, buf);

        // push into Where stack
        buf =  g_strdup_printf("xn%d.id = xn%d.id",lcnt,lcnt-1);
        stack_push_pointer(lqs->ST_Where, buf);

        /* setting */
        s_cnt_pos++;

        /* cleen up*/
        g_free(predicate);
        
        return TRUE;
    }

    if( !strcasecmp(predicate,"last()"))
    {
#ifdef __PG__
        elog(NOTICE,"[XPath] not yet supported\n");
#else
        fprintf(stderr,"[XPath] not yet supported\n");
#endif//__PG__
        g_free(predicate);
        return FALSE;
    }

    if( !strncasecmp(predicate,"text()=\"",8))
    {
#ifdef __PG__
                elog(NOTICE,"[XPath] not yet supported\n");
#else
                fprintf(stderr,"[XPath] not yet supported\n");
#endif//__PG__
        g_free(predicate);
        return FALSE;
    }

    if( !strncasecmp(predicate,".=\"",3))
    {
#ifdef __PG__
        elog(NOTICE,"[XPath] not yet supported\n");
#else
        fprintf(stderr,"[XPath] not yet supported\n");
#endif//__PG__
        g_free(predicate);
        return FALSE;
    }

    /* error if absolute path in predicate [/ccc] */
    ptr=predicate;
    if(*ptr=='/')
    {
        if(*(++ptr)!='/')
        {
#ifdef __PG__
            elog(NOTICE,"[XPath] Syntax Error\n");
#else
            fprintf(stderr,"[XPath] Syntax Error\n");
#endif//__PG__
            g_free(predicate);
            return FALSE;
        }
        else
            isDes = TRUE;
    }

    clause = g_string_sized_new(300);

    p_ext = g_memdup(predicate,strlen(predicate)+50);
    g_free(predicate);
    for(i=0;i<3;i++)
        StrReplace(p_ext,t_from[i],t_to[i]);

    pattern = "^([^)(!=><]+)+((!?=(~|=)?)\"([^\"]+)\")?$";
    /*             ~~~~~~~~~         ~~~~ 4    ~~~~~~~~~
                               ~~~~~~~~~~
                  /aaa/bbb   !=~             "hogehoge"
                     1        3                 5
                               ------------------------
                                       2
     */
    regcomp(&preg,pattern,REG_EXTENDED|REG_NEWLINE);
    i = regexec(&preg, p_ext, 6, pmatch, 0);
    regfree(&preg);
    if(i == 0)
    {
        bool like = FALSE;

        if(strstr(p_ext,"%")!=NULL)
            like = TRUE;

        if( pmatch[2].rm_eo==-1) 
        {
            /* case... [//aaa/bbb] */

            buf = g_strdup_printf("xml_path p%d_xp%d,xml_path p%d_xp%d,xml_node p%d_xn%d",
                                pcnt, lcnt, pcnt, lcnt+1, pcnt, lcnt+1
                                );
            stack_push_pointer(lqs->ST_From, buf);
            xd->fcount+=3;

            g_string_append_printf(clause,
                                "p%d_xp%d.pathid = xn%d.pathid AND "
                                "p%d_xp%d.pathexp %s (xp%d.pathexp || '%s%s') AND "
                                "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                "_int_flagment(p%d_xn%d.dewey,xn%d.dewey)",
                                pcnt, lcnt, lcnt,
                                pcnt, lcnt+1, like ? "~~" : "=", lcnt, isDes ? "" : "#/", p_ext,
                                pcnt, lcnt+1, pcnt, lcnt+1,
                                pcnt, lcnt+1, lcnt
                             );

            /* push into Where Stack */
            stack_push_pointer(lqs->ST_Where, clause->str);
            g_string_free(clause, FALSE);

            /* Make create view query */
            makeSQL(&(*query),xd,-1,CREATE_VIEW);
            buf =  g_strdup_printf("vw_xn%d as xn%d",lcnt,lcnt);
            stack_push_pointer(lqs->ST_From, buf);

            /* clean up */
            g_free(p_ext);
            
            return TRUE;
        }
        else
        {
            bool negative = FALSE;
            char *left    = g_strdup_printf("%.*s",pmatch[1].rm_eo-pmatch[1].rm_so,&p_ext[pmatch[1].rm_so]);
            char *ope    = g_strdup_printf("%.*s",pmatch[3].rm_eo-pmatch[3].rm_so,&p_ext[pmatch[3].rm_so]);
            char *right = g_strdup_printf("%.*s",pmatch[5].rm_eo-pmatch[5].rm_so,&p_ext[pmatch[5].rm_so]);
            bool no_att = strchr(left,'@')==NULL;

#ifdef __DEBUG__
#ifndef __PG__
            fprintf(stderr,"predicate=%s\n",p_ext);
            fprintf(stderr,"left=%s\tright=%s\n",left,right);
#endif//__PG__
#endif//__DEBUG__

            if(ope[0]=='!')
                negative = TRUE;

            if(strchr(ope,'~')==NULL)
            {
                if(strstr(ope,"==")==NULL && no_att)
                {
                    /* case... [aaa/bbb="hoge"] equal matcher */

                    buf = g_strdup_printf("xml_path p%d_xp%d,xml_node p%d_xn%d",
                                        pcnt, lcnt+1, pcnt, lcnt+1
                                        );
                    stack_push_pointer(lqs->ST_From, buf);
                    xd->fcount+=2;

                    g_string_append_printf(clause,
                                        "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                        "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                        "_int_flagment(p%d_xn%d.dewey,xn%d.dewey) AND "
                                        "stringvalue(p%d_xn%d.id) %s%s '%s'",
                                       pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                                       pcnt, lcnt+1, pcnt, lcnt+1,
                                       pcnt, lcnt+1, lcnt,
                                       pcnt, lcnt+1, negative ? "!" : "", like ? "~~" : "=", right
                                      );
                }
                else
                {
                    if(no_att)
                    {
                        /* case... [aaa/bbb=="hoge"] equal matcher */
                        // select text-node "hoge" child of ./aaa/bbb
                        
                        buf = g_strdup_printf("xml_path p%d_xp%d,xml_node p%d_xn%d,xml_node p%d_xn%d",
                                            pcnt, lcnt+1, pcnt, lcnt+1, pcnt, lcnt+2
                                            );
                        stack_push_pointer(lqs->ST_From, buf);
                        xd->fcount+=3;
    
                        g_string_append_printf(clause,
                                            "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                            "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                            "_int_descendant(p%d_xn%d.dewey,xn%d.dewey) AND "
                                            "p%d_xn%d.parent = p%d_xn%d.id AND "
                                            "p%d_xn%d.kind = '%d' AND "
                                            "p%d_xn%d.value %s%s '%s'",
                                           pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                                           pcnt, lcnt+1, pcnt, lcnt+1,
                                           pcnt, lcnt+1, lcnt,
                                           pcnt, lcnt+2, pcnt, lcnt+1,
                                           pcnt, lcnt+2, KIND_TEXT,
                                           pcnt, lcnt+2, negative ? "!" : "", like ? "~~" : "=", right
                                          );
                    }
                    else
                    {
                        /* case... [aaa/@bbb=="hoge"] equal matcher */

                        buf = g_strdup_printf("xml_path p%d_xp%d,xml_node p%d_xn%d",
                                            pcnt, lcnt+1, pcnt, lcnt+1
                                            );
                        stack_push_pointer(lqs->ST_From, buf);
                        xd->fcount+=2;
    
                        g_string_append_printf(clause,
                                            "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                            "_int_flagment(p%d_xn%d.dewey,xn%d.dewey) AND "
                                            "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                            "p%d_xn%d.value %s%s '%s'",
                                           pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                                           pcnt, lcnt+1, lcnt,
                                           pcnt, lcnt+1, pcnt, lcnt+1,
                                           pcnt, lcnt+1, negative ? "!" : "", like ? "~~" : "=", right
                                          );                        
                    }
                }
            }
            else
            {
                /* case... [aaa/bbb=~"hoge"] match with regular expression*/

                t_from[0] = "%";    // '//' -> '#%/' -> '#*/'
                t_to[0]   = "*";

                for(i=0;i<1;i++)
                    StrReplace(right,t_from[i],t_to[i]);

                buf = g_strdup_printf("xml_path p%d_xp%d,xml_node p%d_xn%d",
                                    pcnt, lcnt+1, pcnt, lcnt+1
                                    );
                stack_push_pointer(lqs->ST_From, buf);
                xd->fcount+=2;

                if(no_att)
                {
                    g_string_append_printf(clause,
                                        "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                        "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                        "_int_descendant(p%d_xn%d.dewey,xn%d.dewey) AND "
                                        "stringvalue(p%d_xn%d.id) %s~ '%s'",
                                       pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                                       pcnt, lcnt+1, pcnt, lcnt+1,
                                       pcnt, lcnt+1, lcnt,
                                       pcnt, lcnt+1, negative ? "!" : "", right
                                      );
                }
                else
                {
                    g_string_append_printf(clause,
                                        "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                        "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                        "_int_flagment(p%d_xn%d.dewey,xn%d.dewey) AND "
                                        "p%d_xn%d.value %s~ '%s'",
                                       pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                                       pcnt, lcnt+1, pcnt, lcnt+1,
                                       pcnt, lcnt+1, lcnt,
                                       pcnt, lcnt+1, negative ? "!" : "", right
                                      );                    
                }
            }

            /* push into Where Stack */
            stack_push_pointer(lqs->ST_Where, clause->str);
            g_string_free(clause, FALSE);

            /* Make create view query */
            makeSQL(&(*query),xd,-1,CREATE_VIEW);
            buf =  g_strdup_printf("vw_xn%d as xn%d",lcnt,lcnt);
            stack_push_pointer(lqs->ST_From, buf);

            /* clean up */
            g_free(p_ext);
            g_free(left);g_free(ope);g_free(right);

            return TRUE;
        }
    }

    pattern = "^([^!=><]+)+(([><]=?=?)([[:digit:]]+([.][[:digit:]]+)?))+$";
    //pattern = "^([^)(!=><]+)+(([><]=?)([[:digit:]]+([.][[:digit:]]+)?))+$";
    /*             ~~~~~~~~~     ~~~~~   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                  /aaa/bbb          >=               255.32  ~~~~~~~~~~~~~~~~~ 5
                     1            3            4            .32
                                 ----------------------------------------
                                       2
     */
    regcomp(&preg,pattern,REG_EXTENDED|REG_NEWLINE);
    i = regexec(&preg, p_ext, 6, pmatch, 0);
    regfree(&preg);
    if(i == 0 )
    {
        char *left    = g_strdup_printf("%.*s",pmatch[1].rm_eo-pmatch[1].rm_so,&p_ext[pmatch[1].rm_so]);
        char *ope    = g_strdup_printf("%.*s",pmatch[3].rm_eo-pmatch[3].rm_so,&p_ext[pmatch[3].rm_so]);
        char *right    = g_strdup_printf("%.*s",pmatch[4].rm_eo-pmatch[4].rm_so,&p_ext[pmatch[4].rm_so]);
        bool flg = (strchr(left,'@')==NULL) ? FALSE : TRUE;    // attribute,text(),comment(),pi().. 

        pattern = "^[^)(]+/(text|comment|processiong-instruction)+\\(\\)$";
        /**                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~1
         *
         */
        regcomp(&preg,pattern,REG_EXTENDED|REG_NEWLINE);
        i = regexec(&preg, left, 2, pmatch, 0);
        regfree(&preg);
        if(i==0)
            flg=TRUE;
        else
        {
            if(strstr(left,"()") != NULL)
            {
                g_free(p_ext);
                return FALSE;
            }
        }

        if(strstr(ope,"==")==NULL || flg )
        {    
            /* case... [aaa/bbb>=3], [aaa/@bbb>3] */
            buf = g_strdup_printf("xml_path p%d_xp%d,xml_node p%d_xn%d",
                                pcnt, lcnt+1, pcnt, lcnt+1
                                );
            stack_push_pointer(lqs->ST_From, buf);
            xd->fcount+=2;
    
            if(flg)
                StrReplace(ope,"==","=");
    
            g_string_append_printf(clause,
                                "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                "p%d_xn%d.pathid = p%d_xp%d.pathid AND "                                
                                "%s(p%d_xn%d.dewey,xn%d.dewey) AND "
                                "%s(%s%d_xn%d%s) %s %s",    //p.value::integer, stringvalue(p.id)::real
                               pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                               pcnt, lcnt+1, pcnt, lcnt+1,
                               flg ? "_int_flagment" : "_int_descendant", pcnt, lcnt+1, lcnt,
                               (strchr(right,'.')==NULL) ? "vchar2int" : "vchar2real",
                               flg ? "p" : "stringvalue(p", pcnt, lcnt+1, flg ? ".value" : ".id)",
                               ope, right
                              );
        }
        else
        {
            /* case... [aaa/bbb>==3] */
            StrReplace(ope,"==","=");
            
            buf = g_strdup_printf("xml_path p%d_xp%d,xml_node p%d_xn%d,xml_node p%d_xn%d",
                                pcnt, lcnt+1, pcnt, lcnt+1, pcnt, lcnt+2
                                );
            stack_push_pointer(lqs->ST_From, buf);
            xd->fcount+=3;

            g_string_append_printf(clause,
                                "p%d_xp%d.pathexp = (xp%d.pathexp || '%s%s') AND "
                                "p%d_xn%d.pathid = p%d_xp%d.pathid AND "
                                "_int_descendant(p%d_xn%d.dewey,xn%d.dewey) AND "
                                "p%d_xn%d.parent = p%d_xn%d.id AND "
                                "p%d_xn%d.kind = '%d' AND "
                                "%s(p%d_xn%d.value) %s %s",
                               pcnt, lcnt+1, lcnt, isDes ? "" : "#/", left,
                               pcnt, lcnt+1, pcnt, lcnt+1,
                               pcnt, lcnt+1, lcnt,
                               pcnt, lcnt+2, pcnt, lcnt+1,
                               pcnt, lcnt+2, KIND_TEXT,
                               (strchr(right,'.')==NULL) ? "vchar2int" : "vchar2real", pcnt, lcnt+2,
                               ope, right
                              );        
        }

        /* push into Where Stack */
        stack_push_pointer(lqs->ST_Where, clause->str);
        g_string_free(clause, FALSE);

        /* Make create view query */
        makeSQL(&(*query),xd,-1,CREATE_VIEW);
        buf =  g_strdup_printf("vw_xn%d as xn%d",lcnt,lcnt);
        stack_push_pointer(lqs->ST_From, buf);

        /* clean up */
        g_free(p_ext);
        
        return TRUE;
    }

    // clear
    g_free(p_ext);
    
    return FALSE; // not yet supported

}//translatePredicate


/*
static int translateFuncCalls(char *, GString **, XPathData *){

}//translateFuncCalls
*/

static bool makeSQL(GString **query,XPathData *xd,int tmp_q,char mode)
{
    char *tmp;
    int i;
    int fcnt;
    int j = 0;
    char *buf = NULL;
    int lcnt        = xd->scount;
    QStack *qs        = xd->qstack_val;
    Stack *from        = qs->ST_From;
    Stack *where    = qs->ST_Where;
    GString *clause = g_string_sized_new(100);

#ifdef __DEBUG__
    puts("Enter makeSQL");
#endif

    /* make 'select ... from' clause */
    if((mode&SELECT_INTO)==SELECT_INTO)
    {
        if(xd->evaledAxis == AXIS_CHILD)
        {
            g_string_append_printf(*query,"SELECT "
                                        "xn%d.parent as group_id,"
                                        "xn%d.id as target_id,"
                                        "xn%d.dewey",
                                      lcnt,lcnt,lcnt
                            );    
        }
        else
        {
            g_string_append_printf(*query,"SELECT%s"
                                        "xn%d.id as group_id,"
                                        "xn%d.id as target_id,"
                                        "xn%d.dewey",
                                      xd->distinct ? " DISTINCT " : " ",
                                      lcnt-1,lcnt,lcnt
                            );
        }
    }
    else if((mode&CREATE_VIEW)==CREATE_VIEW)
    {
        g_string_append_printf(*query, "CREATE or REPLACE VIEW vw_xn%d as " 
                                  "SELECT%s"
                                    "xn%d.docid,"
                                    "xn%d.id",
                                  lcnt,
                                  xd->distinct ? " DISTINCT " : " ",
                                  lcnt,lcnt
                        );    
    }
    else if((mode&RET_FLG)==RET_FLG)
    {
        g_string_append_printf(*query,"SELECT%s"
                                    "xn.docid,"
                                    "xn.id",
                                  xd->distinct ? " DISTINCT " : " "
                        );
    }
    else
    {
        g_string_append_printf(*query,"SELECT%s"
                                    "xn%d.docid,"
                                    "%sxn%d.id%s",
                                  xd->distinct ? " DISTINCT " : " ",
                                  lcnt-1,
                                  (mode&RET_XML)==RET_XML ? "toxml(" : "",
                                  lcnt-1,
                                  (mode&RET_XML)==RET_XML ? ") as xml" : ""
                        );
    }

    if( tmp_q < 0)
        g_string_append(*query," FROM ");
    else
        g_string_append_printf(*query," INTO TEMP tmp_Q%d FROM ",tmp_q);

    /* make 'from' clause */
    fcnt = xd->fcount;

#ifdef __DEBUG__
    printf("from count=%d\n",from->count);
    printf("fcnt=%d\n",fcnt);
#endif

    for( i=0; from->used > 0;i++)
    {
        buf = stack_pop(from);

        if(fcnt>=GEQO_THRESHOLD && strchr(buf,',')!=NULL)
        {
            tmp = buf;
            while(*tmp!='\0')
            {
                if(*tmp==',')
                    j++;
                tmp++;
            }
            
            tmp = g_memdup(buf,strlen(buf)+11*j+1);
            StrReplace(tmp,","," JOIN ");    // 1 -> 12 (+11)
            g_string_prepend(clause,tmp);
            g_free(tmp);
        }
        else
            g_string_prepend(clause,buf);
        
        g_free(buf);

        if(from->used > 0)
            g_string_prepend(clause,",");
    }
    
    if(xd->docfilter!=NULL)
        g_string_prepend(clause,"xml_document xd,");

    g_string_append(*query,clause->str);
    g_string_free(clause, TRUE);

    /* make 'where' clause */
    clause = g_string_sized_new(200);
    
    for( i=0; where->used > 0;i++)
    {
        if(i==0)
            g_string_append(*query," WHERE ");
        buf = stack_pop(where);
        g_string_prepend(clause,buf);
        g_free(buf);
        if(where->used > 0)
            g_string_prepend(clause," AND ");
    }
    
    if(xd->docfilter!=NULL)
    {
        buf = g_strdup_printf("xn%d.docid = xd.docid AND "
                                 "xd.name SIMILAR TO '%s' AND ",
                                 xd->ffirst, xd->docfilter
                             );
        g_string_prepend(clause, buf);
        g_free(buf);
        
        // refresh
        g_free(xd->docfilter);
        xd->docfilter = NULL;
    }

    if( tmp_q >= 0 || (mode&CREATE_VIEW)==CREATE_VIEW)
        g_string_append(clause,"; ");

    g_string_append(*query,clause->str);
    g_string_free(clause, TRUE);

    /* refresh */
    xd->fcount = 0;

    return TRUE;
}


/* 
 * pre-processing 
 */
static char *prep(XPathData *xd, char *argv)
{
    char *buf, *p;
    
    /* Trimming Expression */
    trim(argv);
    
    /*
     *  Extracts the specified document infomation and returns
     * the head of XPath expression 
     * 
     * document("aaa.xml")/top//name[2]
     *             ~~~~~~~~
     */
    p = extract_docinfo(xd, argv);

    /* get enough buffer for xpath expression */
    buf = g_strndup(p, BUFSIZE);

    /* return formalized expression */
    return getFormalExp(buf);
}

/*
 * extract_docinfo 
 *     - extracts the specified document infomation and returns 
 *     the head of XPath expression.
 *  
 *     document("aaa.xml")/top//name[2]
 */
static char *extract_docinfo(XPathData *xd, char *exp)
{
    char *head, *tail;

    if(strncmp(exp, "document(\"", 10) == 0)
    {
        /*
         * document("aaa.xml")
         *              ^      ^
         *             head    tail
         */
        head = exp + 10;
        if( (tail = strstr(head, "\")")) != NULL)
        {
            // copy filtering expression, like 'aaa.xml'
            xd->docfilter = g_strndup(head, tail - head);
            exp = tail + 2;
        }
    }

    return exp;
}

/*
 * getFormalExp - get xpath formal expression
 */
static char *getFormalExp(char *exp)
{
    char *p=exp;

    char *from[]    = {
                        "@",
                        "//",
                        "..",
                        "."
                      };

    char *to[]        = {
                       "attribute::",
                       "/descendant-or-self::node()/",
                       "parent::node()",
                       "self::node()"
                      };

    int i=0;

    for(;i<4;i++)
        StrReplace(exp,from[i],to[i]);

    return p;
}//getFormalExp

/*
 *    StrReplace - replace string
 *
 *  This function needs care for handling
 */
static int StrReplace( char *target, const char *from, const char *to )
{
    int f_len, t_len, rcount;
    char *limit;
    char *tmp;

    rcount = 0;
    f_len = strlen(from);
    t_len = strlen(to);
    limit = target + strlen(target) - f_len;

    while( target <= limit)
    {
        /* skip predicate statement between '[' and ']' */
        if( *target == '[')
        {
            if( (tmp=strchr(target,']')) != NULL)
                target = tmp;
              else
              {
#ifdef __PG__
                elog(ERROR,"XPath syntax error\n");
#else
                fprintf(stderr,"XPath syntax error\n");
                abort();
#endif//__PG__
              }
        }

        if( strncmp( target, from, f_len ) == 0 )
        {
              memmove( target + t_len, target + f_len, limit - target + 1 );
              strncpy( target, to, t_len );
              limit += t_len - f_len;
              rcount++;
              target += t_len;
        }
        else
              target++;
    }
    return rcount;
}// streplace

/*
 *    StrisDigit -- Srting is Digit Number ?
 */
static bool StrisDigit(char *s)
{
    while(*s){
        /* returns non zero if 0-9 */
        if(isdigit(*s)==0)
            return FALSE;
        s++;
    }

    return TRUE;
}

/**
 * strrstr.c -- find last occurence of string in another string
 */
char *strrstr(const char *str, const char *pat) 
{
    size_t len, patlen;
    const char *p;

    assert(str != NULL);
    assert(pat != NULL);

    len = strlen(str);
    patlen = strlen(pat);

    if (patlen > len)
        return NULL;
    for (p = str + (len - patlen); p > str; --p)
        if (*p == *pat && strncmp(p, pat, patlen) == 0)
            return (char *) p;
    return NULL;
}

/*
 *    trim - Remove leading, trailing, & excess embedded spaces
 */
static void trim(char *s)
{
    int len = strlen(s);

    /* trim trailing whitespace */
    while( len > 0 && isspace(s[len-1]))
        s[--len]='\0';

    /* trim leading whitespace */
    memmove(s, &s[strspn(s, " \n\r\t\v")], len);
}// trim

/*
 *    strxtok - separate into token(location step)
 */
static char *strxtok(char *str, const char *delim, char **save_ptr)
{
    char *pbegin, *pend;
    const char *ptr;

    pbegin = str ? str : *save_ptr;

    pbegin += strspn(pbegin,delim);
    if( *pbegin == '\0'){
        *save_ptr = "";
        return(NULL);
    }
    ptr = pbegin;

    for( ; *ptr; ptr++)
    {
        if( *ptr == '[')
        {
            ptr = strchr(ptr,']');
            if(ptr==NULL)
            {
#ifdef __PG__
                elog(ERROR,"XPath syntax error\n");
#else
                fprintf(stderr,"XPath syntax error\n");
                abort();
#endif//__PG__
            }
        }
        else if( *ptr == *delim)
        {
            pend = (char *)ptr;
            if(*pend != '\0')
                *pend++ = '\0';
            *save_ptr = pend;
            return(pbegin);
        }
    }
    *save_ptr = "";
    return(pbegin);
}// strxtok

#ifndef __PG__
int main(int argc,char *argv[])
{

    char *res;
    //char *str="document(\"xxx\")/aaa//ddd";
    //char *p=g_strdup(str);
    puts("start\n");
    res = translateXPath(argv[1],0);
    //res = translateXPath(p,2);
    //g_free(p);

    if(res != FALSE)
        puts(res);
    else
        puts("FALSE\n");
    
    g_free(res);
    puts("fin\n");
    return 0;
}
#endif//__PG__

