/* Module:          SQLPrimaryKeys.c
 *
 * Description:     Returns the column names that make up the primary key for a table.
 *
 * Classes:         
 *
 * API functions:   SQLPrimaryKeys
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLPrimaryKeys(
								 SQLHSTMT		hDrvStmt,
								 SQLCHAR		*szCatalogName,
								 SQLSMALLINT	nCatalogNameLength,
								 SQLCHAR		*szSchemaName,
								 SQLSMALLINT	nSchemaNameLength,
								 SQLCHAR		*szTableName,
								 SQLSMALLINT	nTableNameLength
								 )
{
static char *func = "SQLPrimaryKeys";
StatementClass *stmt = (StatementClass *) hDrvStmt;
TupleNode *row;
SQLRETURN result;
int seq = 0;
SQLHSTMT htbl_stmt;
StatementClass *tbl_stmt;
char tables_query[STD_STATEMENT_LEN];
char attname[MAX_INFO_STRING];
SDWORD attname_len;
char pktab[MAX_TABLE_LEN + 1];
Int2 result_cols;

	mylog("%s: entering...stmt=%u\n", func, stmt);

    if( ! stmt) {
		SC_log_error(func, "", NULL);
        return SQL_INVALID_HANDLE;
    }
	stmt->manual_result = TRUE;
	stmt->errormsg_created = TRUE;

    stmt->result = QR_Constructor();
    if(!stmt->result) {
        stmt->errormsg = "Couldn't allocate memory for SQLPrimaryKeys result.";
        stmt->errornumber = STMT_NO_MEMORY_ERROR;
		SC_log_error(func, "", stmt);
        return SQL_ERROR;
    }

    /* the binding structure for a statement is not set up until */
    /* a statement is actually executed, so we'll have to do this ourselves. */
	result_cols = 6;
    extend_bindings(stmt, result_cols);
	
    /* set the field names */
    QR_set_num_fields(stmt->result, result_cols);
    QR_set_field_info(stmt->result, 0, "TABLE_QUALIFIER", PG_TYPE_TEXT, MAX_INFO_STRING);
    QR_set_field_info(stmt->result, 1, "TABLE_OWNER", PG_TYPE_TEXT, MAX_INFO_STRING);
    QR_set_field_info(stmt->result, 2, "TABLE_NAME", PG_TYPE_TEXT, MAX_INFO_STRING);
    QR_set_field_info(stmt->result, 3, "COLUMN_NAME", PG_TYPE_TEXT, MAX_INFO_STRING);
    QR_set_field_info(stmt->result, 4, "KEY_SEQ", PG_TYPE_INT2, 2);
    QR_set_field_info(stmt->result, 5, "PK_NAME", PG_TYPE_TEXT, MAX_INFO_STRING);


    result = SQLAllocStmt( stmt->hdbc, &htbl_stmt);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
		stmt->errornumber = STMT_NO_MEMORY_ERROR;
		stmt->errormsg = "Couldn't allocate statement for Primary Key result.";
		SC_log_error(func, "", stmt);
        return SQL_ERROR;
    }
	tbl_stmt = (StatementClass *) htbl_stmt;

	pktab[0] = '\0';
	make_string(szTableName, nTableNameLength, pktab);
	if ( pktab[0] == '\0') {
		stmt->errormsg = "No Table specified to SQLPrimaryKeys.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
		return SQL_ERROR;
	}

#if 0
	sprintf(tables_query, "select distinct on (attnum) a2.attname, a2.attnum from pg_attribute a1, pg_attribute a2, pg_class c, pg_index i where c.relname = '%s_pkey' AND c.oid = i.indexrelid AND a1.attrelid = c.oid AND a2.attrelid = c.oid AND (i.indkey[0] = a1.attnum OR i.indkey[1] = a1.attnum OR i.indkey[2] = a1.attnum OR i.indkey[3] = a1.attnum OR i.indkey[4] = a1.attnum OR i.indkey[5] = a1.attnum OR i.indkey[6] = a1.attnum OR i.indkey[7] = a1.attnum) order by a2.attnum", pktab);
#else
	/* Simplified query to remove assumptions about
	 * number of possible index columns.
	 * Courtesy of Tom Lane - thomas 2000-03-21
	 */
	sprintf(tables_query, "select ta.attname, ia.attnum"
			" from pg_attribute ta, pg_attribute ia, pg_class c, pg_index i"
			" where c.relname = '%s_pkey'"
			" AND c.oid = i.indexrelid"
			" AND ia.attrelid = i.indexrelid"
			" AND ta.attrelid = i.indrelid"
			" AND ta.attnum = i.indkey[ia.attnum-1]"
			" order by ia.attnum", pktab);
#endif


	mylog("SQLPrimaryKeys: tables_query='%s'\n", tables_query);

    result = SQLExecDirect(htbl_stmt, tables_query, strlen(tables_query));
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
		stmt->errormsg = SC_create_errormsg(htbl_stmt);
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
        return SQL_ERROR;
    }

    result = SQLBindCol(htbl_stmt, 1, SQL_C_CHAR,
                        attname, MAX_INFO_STRING, &attname_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
		stmt->errormsg = tbl_stmt->errormsg;
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
        return SQL_ERROR;
    }

    result = SQLFetch(htbl_stmt);

    while((result == SQL_SUCCESS) || (result == SQL_SUCCESS_WITH_INFO)) {

        row = (TupleNode *)malloc(sizeof(TupleNode) + (result_cols - 1) * sizeof(TupleField));

        set_tuplefield_null(&row->tuple[0]);

		/* I have to hide the table owner from Access, otherwise it
		 * insists on referring to the table as 'owner.table'.
		 * (this is valid according to the ODBC SQL grammar, but
		 * Postgres won't support it.)
		 */
        set_tuplefield_string(&row->tuple[1], "");
        set_tuplefield_string(&row->tuple[2], pktab);
        set_tuplefield_string(&row->tuple[3], attname);
		set_tuplefield_int2(&row->tuple[4], (Int2) (++seq));
		set_tuplefield_null(&row->tuple[5]);

        QR_add_tuple(stmt->result, row);

		mylog(">> primaryKeys: pktab = '%s', attname = '%s', seq = %d\n", pktab, attname, seq);

        result = SQLFetch(htbl_stmt);
    }

    if(result != SQL_NO_DATA_FOUND) {
		stmt->errormsg = SC_create_errormsg(htbl_stmt);
		stmt->errornumber = tbl_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(htbl_stmt, SQL_DROP);
        return SQL_ERROR;
    }

	SQLFreeStmt(htbl_stmt, SQL_DROP);


	/* also, things need to think that this statement is finished so */
	/* the results can be retrieved. */
    stmt->status = STMT_FINISHED;

    /* set up the current tuple pointer for SQLFetch */
    stmt->currTuple = -1;
	stmt->rowset_start = -1;
	stmt->current_col = -1;

	mylog("SQLPrimaryKeys(): EXIT, stmt=%u\n", stmt);
    return SQL_SUCCESS;
}

