/* Module:          SQLPrepare.c
 *
 * Description:     Prepares an SQL string for execution.
 *
 * Classes:         
 *
 * API functions:   SQLPrepare
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLPrepare(
							 SQLHSTMT		hDrvStmt,
                           SQLCHAR			*szSqlStr,
                           SQLINTEGER		nSqlStrLength
						   )
{
	static char *func = "SQLPrepare";
	StatementClass *self = (StatementClass *) hDrvStmt;

	mylog( "%s: entering...\n", func);

	if ( ! self)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}
    
	/*	According to the ODBC specs it is valid to call SQLPrepare mulitple times.
		In that case, the bound SQL statement is replaced by the new one 
	*/

	switch(self->status)
	{
		case STMT_PREMATURE:
			mylog("**** SQLPrepare: STMT_PREMATURE, recycle\n");
			SC_recycle_statement(self); /* recycle the statement, but do not remove parameter bindings */
			break;

		case STMT_FINISHED:
			mylog("**** SQLPrepare: STMT_FINISHED, recycle\n");
			SC_recycle_statement(self); /* recycle the statement, but do not remove parameter bindings */
			break;

		case STMT_ALLOCATED:
			mylog("**** SQLPrepare: STMT_ALLOCATED, copy\n");
			self->status = STMT_READY;
			break;

		case STMT_READY:
			mylog("**** SQLPrepare: STMT_READY, change SQL\n");
			break;

		case STMT_EXECUTING:
			mylog("**** SQLPrepare: STMT_EXECUTING, error!\n");

			self->errornumber = STMT_SEQUENCE_ERROR;
			self->errormsg = "SQLPrepare(): The handle does not point to a statement that is ready to be executed";
			SC_log_error(func, "", self);

			return SQL_ERROR;

		default:
			self->errornumber = STMT_INTERNAL_ERROR;
			self->errormsg = "An Internal Error has occured -- Unknown statement status.";
			SC_log_error(func, "", self);
			return SQL_ERROR;
	}
	
	if (self->statement)
	{
		free(self->statement);
	}

	self->statement = make_string(szSqlStr, nSqlStrLength, NULL);

	if ( ! self->statement)
	{
		self->errornumber = STMT_NO_MEMORY_ERROR;
		self->errormsg = "No memory available to store statement";
		SC_log_error(func, "", self);
		return SQL_ERROR;
	}

	self->prepare = TRUE;
	self->statement_type = statement_type(self->statement);

	/*	Check if connection is onlyread (only selects are allowed) */
	if ( CC_is_onlyread(self->hdbc) && STMT_UPDATE(self))
	{
		self->errornumber = STMT_EXEC_ERROR;
		self->errormsg = "Connection is readonly, only select statements are allowed.";
		SC_log_error(func, "", self);
		return SQL_ERROR;
	}

	return SQL_SUCCESS;
}


