/* Module:          SQLError.c
 *
 * Description:     Returns error or status information. 
 *
 * Classes:         
 *
 * API functions:   SQLError
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

RETCODE SQL_API SQLError(
						 SQLHENV		hDrvEnv,
						 SQLHDBC		hDrvDbc,
						 SQLHSTMT		hDrvStmt,
						 SQLCHAR		*szSqlState,
						 SQLINTEGER		*pfNativeError,
						 SQLCHAR		*szErrorMsg,
						 SQLSMALLINT	nErrorMsgMax,
						 SQLSMALLINT	*pcbErrorMsg
						 )
{
	char *msg;
	int status;
    
	mylog("**** SQLError: hDrvEnv=%u, hDrvDbc=%u, hDrvStmt=%u\n", hDrvEnv, hDrvDbc, hDrvStmt);

    if (SQL_NULL_HSTMT != hDrvStmt)
	{
		/* CC: return an error of a hDrvStmt  */
        StatementClass *stmt = (StatementClass *) hDrvStmt;
        
        if (SC_get_error(stmt, &status, &msg))
		{
			mylog("SC_get_error: status = %d, msg = #%s#\n", status, msg);
            if (NULL == msg)
			{
				if (NULL != szSqlState)
				{
					strcpy(szSqlState, "00000");
				}

                if (NULL != pcbErrorMsg)
				{
					*pcbErrorMsg = 0;
				}

                if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
				{
					szErrorMsg[0] = '\0';
				}
                
                return SQL_NO_DATA_FOUND;
			}
			                
            if (NULL != pcbErrorMsg)
			{
				*pcbErrorMsg = (SQLSMALLINT)strlen(msg);
			}
			            
            if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
			{
				strncpy_null(szErrorMsg, msg, nErrorMsgMax);
			}
            
            if (NULL != pfNativeError)
			{
				*pfNativeError = status;
			}
            
            if (NULL != szSqlState)
			{
				switch (status)
				{
					/* now determine the SQLSTATE to be returned */
					case STMT_TRUNCATED:
						strcpy(szSqlState, "01004");
						/* data truncated */
						break;

					case STMT_INFO_ONLY:
						strcpy(szSqlState, "00000");
						/* just information that is returned, no error */
						break;

					case STMT_BAD_ERROR:
						strcpy(szSqlState, "08S01");
						/* communication link failure */
						break;

					case STMT_CREATE_TABLE_ERROR:
						strcpy(szSqlState, "S0001");
						/* table already exists */
						break;
						
					case STMT_STATUS_ERROR:
					case STMT_SEQUENCE_ERROR:
						strcpy(szSqlState, "S1010");
						/* Function sequence error */
						break;

					case STMT_NO_MEMORY_ERROR:
						strcpy(szSqlState, "S1001");
						/* memory allocation failure */
						break;

					case STMT_COLNUM_ERROR:
						strcpy(szSqlState, "S1002");
						/* invalid column number */
						break;

					case STMT_NO_STMTSTRING:
						strcpy(szSqlState, "S1001");
						/* having no stmtstring is also a malloc problem */
						break;

					case STMT_ERROR_TAKEN_FROM_BACKEND:
						strcpy(szSqlState, "S1000");
						/* general error */
						break;

					case STMT_INTERNAL_ERROR:
						strcpy(szSqlState, "S1000");
						/* general error */
						break;
						
					case STMT_ROW_OUT_OF_RANGE:
						strcpy(szSqlState, "S1107");
						break;

					case STMT_OPERATION_CANCELLED:
						strcpy(szSqlState, "S1008");
						break;

					case STMT_NOT_IMPLEMENTED_ERROR:
						strcpy(szSqlState, "S1C00"); /* == 'driver not capable' */
						break;

					case STMT_OPTION_OUT_OF_RANGE_ERROR:
						strcpy(szSqlState, "S1092");
						break;

					case STMT_BAD_PARAMETER_NUMBER_ERROR:
						strcpy(szSqlState, "S1093");
						break;

					case STMT_INVALID_COLUMN_NUMBER_ERROR:
						strcpy(szSqlState, "S1002");
						break;

					case STMT_RESTRICTED_DATA_TYPE_ERROR:
						strcpy(szSqlState, "07006");
						break;

					case STMT_INVALID_CURSOR_STATE_ERROR:
						strcpy(szSqlState, "24000");
						break;

					case STMT_OPTION_VALUE_CHANGED:
						strcpy(szSqlState, "01S02");
						break;

					case STMT_INVALID_CURSOR_NAME:
						strcpy(szSqlState, "34000");
						break;

					case STMT_NO_CURSOR_NAME:
						strcpy(szSqlState, "S1015");
						break;

					case STMT_INVALID_ARGUMENT_NO:
						strcpy(szSqlState, "S1009");
						/* invalid argument value */
						break;

					case STMT_INVALID_CURSOR_POSITION:
						strcpy(szSqlState, "S1109");
						break;

					case STMT_VALUE_OUT_OF_RANGE:
						strcpy(szSqlState, "22003");
						break;

					case STMT_OPERATION_INVALID:
						strcpy(szSqlState, "S1011");
						break;

					case STMT_EXEC_ERROR:
						
					default:
						strcpy(szSqlState, "S1000");
						/* also a general error */
						break;
						
					}
                }         

				mylog("       szSqlState = '%s', szError='%s'\n", szSqlState, szErrorMsg);
            
        } 
		else
		{
            if (NULL != szSqlState)
			{
				strcpy(szSqlState, "00000");
			}

            if (NULL != pcbErrorMsg)
			{
				*pcbErrorMsg = 0;
			}

            if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
			{
				szErrorMsg[0] = '\0';
			}
            
			mylog("       returning NO_DATA_FOUND\n");
            return SQL_NO_DATA_FOUND;
        }
        return SQL_SUCCESS;    
        
    }
	else if (SQL_NULL_HDBC != hDrvDbc)
	{
        ConnectionClass *conn = (ConnectionClass *) hDrvDbc;
        
		mylog("calling CC_get_error\n");
        
		if (CC_get_error(conn, &status, &msg))
		{
			mylog("CC_get_error: status = %d, msg = #%s#\n", status, msg);
            if (NULL == msg)
			{
                if (NULL != szSqlState)
				{
					strcpy(szSqlState, "00000");
				}

                if (NULL != pcbErrorMsg)
				{
					*pcbErrorMsg = 0;
				}

                if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
				{
					szErrorMsg[0] = '\0';
				}
                
                return SQL_NO_DATA_FOUND;
            }                
            
            if (NULL != pcbErrorMsg)
			{
				*pcbErrorMsg = (SQLSMALLINT)strlen(msg);
			}

            if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
			{
				strncpy_null(szErrorMsg, msg, nErrorMsgMax);
			}

            if (NULL != pfNativeError)
			{
				*pfNativeError = status;
			}
            
            if (NULL != szSqlState)
			{
				switch(status)
				{
					case STMT_OPTION_VALUE_CHANGED:
					
					case CONN_OPTION_VALUE_CHANGED:
						strcpy(szSqlState, "01S02");
						break;

					case STMT_TRUNCATED:
					case CONN_TRUNCATED:
						strcpy(szSqlState, "01004");
						/* data truncated */
						break;

					case CONN_INIREAD_ERROR:
						strcpy(szSqlState, "IM002");
						/* data source not found */
						break;

					case CONN_OPENDB_ERROR:
						strcpy(szSqlState, "08001");
						/* unable to connect to data source */
						break;

					case CONN_INVALID_AUTHENTICATION:
					case CONN_AUTH_TYPE_UNSUPPORTED:
						strcpy(szSqlState, "28000");
						break;

					case CONN_STMT_ALLOC_ERROR:
						strcpy(szSqlState, "S1001");
						/* memory allocation failure */
						break;

					case CONN_IN_USE:
						strcpy(szSqlState, "S1000");
						/* general error */
						break;

					case CONN_UNSUPPORTED_OPTION:
						strcpy(szSqlState, "IM001");
						/* driver does not support this function */

					case CONN_INVALID_ARGUMENT_NO:
						strcpy(szSqlState, "S1009");
						/* invalid argument value */
						break;

					case CONN_TRANSACT_IN_PROGRES:
						strcpy(szSqlState, "S1010");
						/* when the user tries to switch commit mode in a transaction */
						/* -> function sequence error */
						break;

					case CONN_NO_MEMORY_ERROR:
						strcpy(szSqlState, "S1001");
						break;

					case CONN_NOT_IMPLEMENTED_ERROR:
					case STMT_NOT_IMPLEMENTED_ERROR:
						strcpy(szSqlState, "S1C00");
						break;

					case CONN_VALUE_OUT_OF_RANGE:
					case STMT_VALUE_OUT_OF_RANGE:
						strcpy(szSqlState, "22003");
						break;

					default:
						strcpy(szSqlState, "S1000");
						/* general error */
						break;

				}
			}
		}
		else
		{
			mylog("CC_Get_error returned nothing.\n");

			if (NULL != szSqlState)
			{
				strcpy(szSqlState, "00000");
			}
			
			if (NULL != pcbErrorMsg)
			{
				*pcbErrorMsg = 0;
			}
			
			if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
			{
				szErrorMsg[0] = '\0';
			}

			return SQL_NO_DATA_FOUND;
		}
		return SQL_SUCCESS;
	}
	else if (SQL_NULL_HENV != hDrvEnv)
	{
		EnvironmentClass *env = (EnvironmentClass *)hDrvEnv;

        if(EN_get_error(env, &status, &msg))
		{
			mylog("EN_get_error: status = %d, msg = #%s#\n", status, msg);

            if (NULL == msg)
			{
				if (NULL != szSqlState)
				{
					strcpy(szSqlState, "00000");
				}

                if (NULL != pcbErrorMsg)
				{
					*pcbErrorMsg = 0;
				}

                if ((NULL != szErrorMsg) && (nErrorMsgMax > 0)) 
				{
					szErrorMsg[0] = '\0';
				}

				return SQL_NO_DATA_FOUND;
			}

			if (NULL != pcbErrorMsg) 
			{
				*pcbErrorMsg = (SQLSMALLINT)strlen(msg);
			}

            if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
			{
				strncpy_null(szErrorMsg, msg, nErrorMsgMax);
			}

            if (NULL != pfNativeError)
			{
				*pfNativeError = status;
			}
            
            if(szSqlState)
			{
				switch(status)
				{
					case ENV_ALLOC_ERROR:
						/* memory allocation failure */
						strcpy(szSqlState, "S1001");
						break;

					default:
						strcpy(szSqlState, "S1000");
						/* general error */
						break;

				}
			}
		}
		else
		{
			if (NULL != szSqlState)
			{
				strcpy(szSqlState, "00000");
			}
			
			if (NULL != pcbErrorMsg)
			{
				*pcbErrorMsg = 0;
			}
            
			if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
			{
				szErrorMsg[0] = '\0';
			}
            
            return SQL_NO_DATA_FOUND;
		}
		
		return SQL_SUCCESS;
	}

	if (NULL != szSqlState)
	{
		strcpy(szSqlState, "00000");
	}

    if (NULL != pcbErrorMsg)
	{
		*pcbErrorMsg = 0;
	}

    if ((NULL != szErrorMsg) && (nErrorMsgMax > 0))
	{
		szErrorMsg[0] = '\0';
	}

    
    return SQL_NO_DATA_FOUND;

}

