//
//  PgSQLResult.m
//  PostgresSQL
//
//  Created by Pascal on Thu Dec 26 2002.
//  Copyright (c) 2002 P3 Consulting. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import "PgSQLResult.h"
#import "PgSQLResultPrivate.h"
#import "PostgreSQL.h"
#import "PostgreSQLPrivate.h"
#import "pgtypes.h"

#define CLEAR_RESULT 		if (_result != nil) { PQclear(_result) ; _result = nil	;} _curRowIndex = -1 ; _rowsInResult = 0; _fieldsInResult = 0
#define	RESULT_ISOK			((PQresultStatus(_result) == PGRES_COMMAND_OK) || (PQresultStatus(_result) == PGRES_TUPLES_OK))

extern NSString *invalidRowIndex	;

@implementation PgSQLResult

+(id)newWithPgResult:(PGresult *)result forConnection:(PostgreSQL *)pgSQL
{
		return [[PgSQLResult alloc] initWithPgResult:result  forConnection:pgSQL]	; 
}

-(id)init
{
		self = [super init]	;
		_result = nil	;
		_curRowIndex = -1 ; 
		_rowsInResult = 0; 
		_fieldsInResult = 0;
		_bindingArray = [[NSMutableArray array] retain]	;			
		return self	;
}

-(id)initWithPgResult:(PGresult *)result forConnection:(PostgreSQL *)pgSQL
{
		self = [self init]	;
		_pgsql = pgSQL		;
		_result = result	;
		_rowsInResult = PQntuples(_result)	;
		_fieldsInResult = PQnfields(_result)	;
		return self	;
}

-(void)dealloc
{
		CLEAR_RESULT	;
		[_bindingArray release];
		[super dealloc]	;
}

-(PGresult *)pgResult
{
		return _result	;
}

-(int)rowsAffected
{
		return PQntuples(_result);
}

-(int)rowsAffectedByCommand
{
	int	result = 0	;

	char *s = PQcmdTuples(_result);
	
		if (s && *s)
			sscanf(s,"%d",&result)	;
			
		return result	;
}

-(BOOL)nextRow
{
		++_curRowIndex	;
		if (_curRowIndex < _rowsInResult)	{
			[self _fetchData]	;
			return YES	;
		}
#if	DEMO_MODE
		if (_curRowIndex >= MAX_RESULTS)	_curRowIndex = MAX_RESULTS	;
#else
		_curRowIndex = _rowsInResult	;
#endif
		return NO	;	
}

-(BOOL)previousRow
{
		--_curRowIndex	;
		if (_curRowIndex >= 0)	{
			[self _fetchData]	;
			return YES	;
		}
		_curRowIndex = -1	;
		return NO;
}

-(void)afterLastRow
{
#if	DEMO_MODE
		if (_curRowIndex >= MAX_RESULTS)	_curRowIndex = MAX_RESULTS	;
#else
		_curRowIndex = _rowsInResult	;
#endif
}

-(void)beforeFirstRow
{
		_curRowIndex = -1	;
}

-(void)firstRow
{
		[self goRow:0]	;
}

-(void)lastRow
{
		[self goRow:_rowsInResult-1]	;
}

-(void)goRow:(int)rowIndex
{
#if	DEMO_MODE
		if (rowIndex >= MAX_RESULTS)	rowIndex = MAX_RESULTS	;
#endif
		if (rowIndex < 0)	{
			rowIndex = -1	;
		}
		if (rowIndex >= _rowsInResult)	{
			rowIndex = _rowsInResult	;
		}
		_curRowIndex = rowIndex	;
}

-(void)goRowRelative:(int)deltaIndex
{
	int	rowIndex = _curRowIndex + deltaIndex	;
#if	DEMO_MODE
		if (rowIndex >= MAX_RESULTS)	rowIndex = MAX_RESULTS	;
#endif
		if (rowIndex < 0)	{
			rowIndex = -1	;
		}
		if (rowIndex >= _rowsInResult)	{
			rowIndex = _rowsInResult	;
		}
		_curRowIndex = rowIndex	;
}

-(BOOL)isColumnNULL:(int)fieldIndex
{
		if ((_curRowIndex < 0) || (_curRowIndex >= _rowsInResult))	{
			if ([_pgsql debugMode])
				NSLog(@"PostgreSQL FrameWork : isColumnNULL called with invalid _curRowIndex !");
			return YES	;
		}
		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if ([_pgsql debugMode])
				NSLog(@"PostgreSQL FrameWork : isColumnNULL called with invalid fieldIndex !");
			return YES	;
		} 
		return PQgetisnull(_result, _curRowIndex, fieldIndex)	;
}

-(int)resultColumnCount
{
		return [self _fetchNFields]	;
}

-(const char *)resultColumnName:(int)col
{
		return [self _fetchFieldName:col]	;
}

-(const char *)resultColumnTypeName:(int)col
{
		return [self _fetchFieldTypeName:col]	;
}

-(Oid)resultColumnType:(int)col
{
		return [self _fetchFieldType:col]	;
}

-(BOOL)resultReturned
{
		return [self rowsAffected] > 0	;
}

-(id)curRowField:(int)fieldIndex
{
		if ([_pgsql debugMode])
			NSAssert((_curRowIndex >= 0) && (_curRowIndex < _rowsInResult), invalidRowIndex)	;

		return [_pgsql _fetchFieldValue:_result atIndex:_curRowIndex forField:fieldIndex binary:PQbinaryTuples(_result)]	;
}

-(NSArray *)resultRowAsArray
{
		if ([_pgsql debugMode])
			NSAssert((_curRowIndex >= 0) && (_curRowIndex < _rowsInResult), invalidRowIndex)	;
		
	int				numFields = [self _fetchNFields]	;
	NSMutableArray	*row = [NSMutableArray arrayWithCapacity:numFields]	;
	int				binary = PQbinaryTuples(_result)	;
	int				fieldIdx	;
	
		for( fieldIdx = 0 ; fieldIdx < numFields ; fieldIdx++)	{
			[row addObject:[_pgsql _fetchFieldValue:_result atIndex:_curRowIndex forField:fieldIdx binary:binary]]	;
		}
	
		return row	;
}

-(NSDictionary *)resultAsDictionary
{
		return nil	;
}

-(const char *)resultTableName:(int)col
{
		return "NOT_SUPPORTED_BY_POSTGRESQL"	;
}

-(int)_fetchNFields
{
		return _fieldsInResult	;
}

-(const char *)_fetchFieldName:(int)fieldIndex
{
		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if ([_pgsql debugMode])
				NSLog(@"PostgreSQL FrameWork : fetchFieldName called with invalid fieldIndex !");
			return "ERROR"	;
		} 
		return PQfname(_result, fieldIndex)	;
}

-(const char *)_fetchFieldTypeName:(int)fieldIndex
{
	char		command[512], *res = "ERROR"	;
	PGresult 	*result	;
	
		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if ([_pgsql debugMode])
				NSLog(@"PostgreSQL FrameWork : fetchFieldType called with invalid fieldIndex !");
			return res	;
		} 

		sprintf(command,"SELECT typname FROM pg_type WHERE oid=%d",PQftype(_result, fieldIndex))	;
		result = PQexec([_pgsql getPGconn],command)	;
		if (PQntuples(result) > 0)	{
			res = PQgetvalue(result, 0, 0)	;
			PQclear(result)	;
		}
		else
			res = "UNKNOWN"	;
			
		return res	;
}

-(Oid)_fetchFieldType:(int)fieldIndex
{

		if ((fieldIndex < 0) || (fieldIndex >= _fieldsInResult))	{
			if ([_pgsql debugMode])
				NSLog(@"PostgreSQL FrameWork : fetchFieldType called with invalid fieldIndex !");
			return 0	;
		} 
			
		return PQftype(_result, fieldIndex)	;
}

-(void)_fetchData
{
		[_pgsql _fetchData:_result atIndex:_curRowIndex withBinding:_bindingArray]	;
}

-(void)bindDouble:(double *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Double bindingArray:_bindingArray]	;
}

-(void)bindDouble:(double *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Double bindingArray:_bindingArray]	;
}

-(void)bindFloat:(float *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Float bindingArray:_bindingArray]	;
}

-(void)bindFloat:(float *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Float bindingArray:_bindingArray]	;
}

-(void)bindShort:(short *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Short bindingArray:_bindingArray]	;
}

-(void)bindShort:(short *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Short bindingArray:_bindingArray]	;
}

-(void)bindInteger:(int *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Integer bindingArray:_bindingArray]	;
}

-(void)bindInteger:(int *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Integer bindingArray:_bindingArray]	;
}

-(void)bindLong:(long *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Long bindingArray:_bindingArray]	;
}

-(void)bindLong:(long *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Long bindingArray:_bindingArray]	;
}

-(void)bindLongLong:(long long *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_LongLong bindingArray:_bindingArray]	;
}

-(void)bindLongLong:(long long *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_LongLong bindingArray:_bindingArray]	;
}

-(void)bindString:(char *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_CString bindingArray:_bindingArray]	;
}

-(void)bindString:(char *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_CString bindingArray:_bindingArray]	;
}

-(void)bindBinaryString:(BSTRING *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_BString bindingArray:_bindingArray]	;
}

-(void)bindBinaryString:(BSTRING *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_BString bindingArray:_bindingArray]	;
}

-(void)bindText:(TEXT *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Text bindingArray:_bindingArray]	;
}

-(void)bindText:(TEXT *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Text bindingArray:_bindingArray]	;
}

-(void)bindBoolean:(BOOL *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_BOOL bindingArray:_bindingArray]	;
}

-(void)bindBoolean:(BOOL *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_BOOL bindingArray:_bindingArray]	;
}

-(void)bindChar:(char *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Char bindingArray:_bindingArray]	;
}

-(void)bindChar:(char *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Char bindingArray:_bindingArray]	;
}

-(void)bindBinary:(Oid *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Oid bindingArray:_bindingArray]	;
}

-(void)bindBinary:(Oid *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Oid bindingArray:_bindingArray]	;
}

- (void)bindPolygon:(POLYGON **)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Polygon bindingArray:_bindingArray]	;
}

- (void)bindPolygon:(POLYGON **)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Polygon bindingArray:_bindingArray]	;
}

- (void)bindLSeg:(LSEG *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_LSeg bindingArray:_bindingArray]	;
}

- (void)bindLSeg:(LSEG *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_LSeg bindingArray:_bindingArray]	;
}

- (void)bindPath:(PATH **)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Path bindingArray:_bindingArray]	;
}

- (void)bindPath:(PATH **)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Path bindingArray:_bindingArray]	;
}

- (void)bindLine:(LINE *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Line bindingArray:_bindingArray]	;
}

- (void)bindLine:(LINE *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Line bindingArray:_bindingArray]	;
}

- (void)bindBox:(BOX *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Box bindingArray:_bindingArray]	;
}

- (void)bindBox:(BOX *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Box bindingArray:_bindingArray]	;
}

- (void)bindCircle:(CIRCLE *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Circle bindingArray:_bindingArray]	;
}

- (void)bindCircle:(CIRCLE *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Circle bindingArray:_bindingArray]	;
}

- (void)bindPoint:(POINT *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Point bindingArray:_bindingArray]	;
}

- (void)bindPoint:(POINT *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Point bindingArray:_bindingArray]	;
}

- (void)bindCIdr:(unsigned char *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_Cidr bindingArray:_bindingArray]	;
}

- (void)bindCIdr:(unsigned char *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_Cidr bindingArray:_bindingArray]	;
}

- (void)bindInet:(unsigned char *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_INet bindingArray:_bindingArray]	;
}

- (void)bindInet:(unsigned char *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_INet bindingArray:_bindingArray]	;
}

- (void)bindMACaddr:(unsigned char *)var;
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_MACAddr bindingArray:_bindingArray]	;
}

- (void)bindMACaddr:(unsigned char *)var column:(int)col;
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_MACAddr bindingArray:_bindingArray]	;
}

- (void)bindBIT:(BOOLARRAY *)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_BIT bindingArray:_bindingArray]	;
}

- (void)bindBIT:(BOOLARRAY *)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_BIT bindingArray:_bindingArray]	;
}

- (void)bindNSDecimal:(NSDecimalNumber **)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_NSDecimal bindingArray:_bindingArray]	;
}

- (void)bindNSDecimal:(NSDecimalNumber **)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_NSDecimal bindingArray:_bindingArray]	;
}

- (void)bindNSNumber:(NSNumber **)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_NSNumber bindingArray:_bindingArray]	;
}

- (void)bindNSNumber:(NSNumber **)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_NSNumber bindingArray:_bindingArray]	;
}

- (void)bindNSString:(NSString **)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_NSString bindingArray:_bindingArray]	;
}

- (void)bindNSString:(NSString **)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_NSString bindingArray:_bindingArray]	;
}

- (void)bindNSData:(NSData **)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_NSData bindingArray:_bindingArray]	;
}

- (void)bindNSData:(NSData **)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_NSData bindingArray:_bindingArray]	;
}

- (void)bindNSArray:(NSArray **)var
{
		[_pgsql _bindAny:-1 address:(void *)var type:bindItem_NSArray bindingArray:_bindingArray]	;
}

- (void)bindNSArray:(NSArray **)var column:(int)col
{
		[_pgsql _bindAny:col address:(void *)var type:bindItem_NSArray bindingArray:_bindingArray]	;
}

@end



@implementation PgSQLResult(NSTableDataSource)

- (int)numberOfRowsInTableView:(NSTableView *)tableView
{
		return _rowsInResult	;
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	NSString	*colIdentifier = (NSString *)[tableColumn identifier]	;
	
		return [_pgsql _fetchFieldValue:_result atIndex:row forField:[colIdentifier intValue] binary:PQbinaryTuples(_result)]	;
}

@end
