/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "stringutils.h"
#include "converter.h"
#include "debugger.h"


	/**
 	 * Used to tokenize a string.
 	 * @param		rstrInput 			the input string that contains the strings to tokenize
 	 * @param		rstrDelimeter	the delimiter string used to tokenize the string
 	 * @param		rvstrTokens		the tokens resulting from the string/delimiter
   */			    	
	void StringUtils::tokenize(const string & rstrInput, const string & rstrDelimeter, vector<string> & rvstrTokens)
	{
		//string strMethodName = "StringUtils::tokenize";
		//Debugger::entered(strMethodName);
		//Debugger::logTrace(strMethodName, "rstrInput = '" + rstrInput + "'");		
		//Debugger::logTrace(strMethodName, "rstrDelimeter = '" + rstrDelimeter + "'");				
		int nDelimSize = rstrDelimeter.size();
		int nLastIdx = -1 * (nDelimSize);
		int nCurrentIdx = 0;
		string strToken;
		// clear the input vector
		rvstrTokens.clear();
		while (true)
		{
			nCurrentIdx = rstrInput.find(rstrDelimeter, nLastIdx + rstrDelimeter.size());
			//Debugger::logTrace(strMethodName, "Found delimeter at: " + Converter::intToString(nCurrentIdx));
			// lets see if the end has been reached
			if (nCurrentIdx == string::npos)
			{
				strToken = rstrInput.substr(nLastIdx + nDelimSize);
				rvstrTokens.push_back(strToken);
				return;
			} // end if no delimeter found
			else
			{
				strToken = rstrInput.substr(nLastIdx + nDelimSize, nCurrentIdx - nLastIdx - nDelimSize);
				rvstrTokens.push_back(strToken);				
			} // end else delimeter found
			nLastIdx = nCurrentIdx;
		} // end while loop end not found
		//Debugger::exited(strMethodName);		
	} // end tokenize

	/**
 	 * Used to replace the occurence of a substring with another substring.
 	 * @param		rstrReplaceBy		the string to be replaced by
 	 * @param		rstrToReplace		the string to get replaced
 	 * @param		rstrInput				the input string that needs to be formatted
 	 * @return	the formatted string
   */			    	
	string StringUtils::replace(const string & rstrReplaceBy, const string & rstrToReplace, const string & rstrInput)
	{
		string strOutput = rstrInput;
		int nIdx = 0;
		nIdx = strOutput.find(rstrToReplace);
		while (nIdx != string::npos)
		{
			strOutput.replace(nIdx, rstrToReplace.size(), rstrReplaceBy);
			nIdx = strOutput.find(rstrToReplace, nIdx + rstrReplaceBy.size());			
		} // end while more substrings to replace
		
		return strOutput;
	} // end replace
	
	/**
 	 * Used to prepare a string for html output. This will convert any html entity characters
 	 * into its html displayable format.
 	 * @param		rstrInput 		the input string to format
 	 * @return	the converted string into html format
   */			    	
	string StringUtils::htmlstr(const string & rstrInput)
	{
		string strOutput = rstrInput;
		// replace all html entity characters
		strOutput = replace("&amp;", "&", strOutput);		
		strOutput = replace("&lt;", "<", strOutput);
		strOutput = replace("&gt;", ">", strOutput);

		return strOutput;
	} // end htmlstr

	/**
 	 * Used to prepare a string for database string input. This will convert any characters that need to
 	 * be escaped for a database insert (i.e. quote characters).
 	 * @param		rstrInput 		the input string to format
 	 * @return	the converted string into a database string format
   */			    	
	string StringUtils::databasestr(const string & rstrInput)
	{
		string strOutput = rstrInput;
		// replace all characters that need to be escape
		strOutput = replace("\\\\", "\\", strOutput);				
		strOutput = replace("\\'", "'", strOutput);		

		return strOutput;	
	} // end databasestr

	/**
 	 * Used to remove slashes from the string.
 	 * @param		rstrInput 		the input string to format
 	 * @return	the converted string removed of slashes
   */			    	
	string StringUtils::stripSlashes(const string & rstrInput)
	{
		string strOutput = rstrInput;
		// remove all slashes
		strOutput = replace("", "\\", strOutput);				

		return strOutput;		
	} // end stripSlashes

	/**
 	 * Used to trim all the white space from the beginning and end of the string.
 	 * @param		rstrInput 		the input string to trim
 	 * @return	the trimmed string
   */			    	
	string StringUtils::trim(const string & rstrInput)
	{
		//string strMethodName = "StringUtils::trim";
		//Debugger::entered(strMethodName);
		
		//Debugger::logTrace(strMethodName, "rstrInput = '" + rstrInput + "'");		
		int nStartNonWhiteSpace = 0;
		int nEndNonWhiteSpace = rstrInput.size() - 1;
		// lets find the start of non-whitespace chars
		for (int nIdx = 0; nIdx < rstrInput.size(); nIdx++)
		{
			if (rstrInput[nIdx] == ' ' || rstrInput[nIdx] == '\t' || rstrInput[nIdx] == '\n')
			{
				nStartNonWhiteSpace++;
			} // end if white space
			else
			{
				break;
			} // end else no more white space		
		} // end for more whitespace
		
		//Debugger::logTrace(strMethodName, "nStartNonWhiteSpace = " + Converter::intToString(nStartNonWhiteSpace));

		// if all white-space then lets just return now !
		if (nStartNonWhiteSpace == rstrInput.size())
		{
			return "";
		} // end if all white-space
		
		// lets find the end of non-whitespace chars
		for (int nIdx = rstrInput.size() - 1; nIdx >= 0; nIdx--)
		{
			if (rstrInput[nIdx] == ' ' || rstrInput[nIdx] == '\t' || rstrInput[nIdx] == '\n')
			{
				nEndNonWhiteSpace--;
			} // end if white space
			else
			{
				break;
			} // end else no more white space
		} // end for more whitespace

		//Debugger::logTrace(strMethodName, "nEndNonWhiteSpace = " + Converter::intToString(nEndNonWhiteSpace));
					
		//Debugger::logTrace(strMethodName, "trimmed string = '" + rstrInput.substr(nStartNonWhiteSpace, nEndNonWhiteSpace - nStartNonWhiteSpace + 1) + "'");				
		//Debugger::exited(strMethodName);			
		return rstrInput.substr(nStartNonWhiteSpace, nEndNonWhiteSpace - nStartNonWhiteSpace + 1);
	} // end trim
		