# $Id: authors.pm,v 1.16 2004/02/04 18:09:54 mig Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::HTML::authors
# Manage registration of proposals.
######################################
# Depends on:
#
# Comas::Proposal - Handles the interaction with a person for Comas
# Comas::Person - Handles the interaction with a person for Comas
# Comas::HTML::common - Common functions for the HTML FrontEnd
# HTML::Template - Perl module to use HTML Templates
# Data::FormValidator - Validates user input based on input profile
# HTML::FromText - Marks up plain text as HTML
# Net::SMTP - Simple Mail Transfer Protocol Client

# This module is not meant to be called on its own, it should be called from
# Comas::HTML.

package Comas::HTML::authors;

use strict;
use warnings;
use Carp;

use Comas::Person;
use Comas::Proposal;
use Comas::HTML::common qw(:all);
use Comas::Conf qw(:read);

use HTML::Template;
use Data::FormValidator;
use HTML::FromText;
use Date::Calc qw(Time_to_Date);
use Net::SMTP;

use Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(authors_data authors_data_delete proposals_new
                    proposals_list proposals_edit proposals_delete
                    proposals_authors proposals_file);
our %EXPORT_TAGS = (all => [@EXPORT_OK]);

=head1 NAME
    
Comas::HTML::authors - Front End to manage authors and their proposals.

=head1 SYNOPSIS

This module is not meant to be used by itself, but as an auxiliary to
L<Comas::HTML|Comas::HTML>. Please check L<Comas::HTML|Comas::HTML>'s
documentation, in the B<Front End Tasks> section, for further details on the
usage for this functions.

=head1 SEE ALSO

L<Comas::HTML|Comas::HTML> module documentation

=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx

Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

sub authors_data {
    my ($input_profile, $template, $validator, $val_results, $person);
    my $h = shift;
    
    $template = HTML::Template->new(filename=>'attendees/authors/data.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    if (defined($h->{-params}->{validate})) {
        $person = Comas::Person->new(-db => $h->{-db},
                                     -id => $h->{-session}->{-person});
        $person->set_resume($h->{-params}->{-resume});
        return $h->attendees_menu('AUTHOR_DATA_CHANGED');
    } else {
        $person = Comas::Person->new(-db => $h->{-db},
                                     -id => $h->{-session}->{-person});
        $template->param(-resume => $person->get_resume);
        return $template->output;
    }
}

sub proposals_new {
    my ($input_profile, $template, $validator, $val_results, $proposal);
    my $h = shift;
    
    $input_profile = {required => [ qw(-title -abstract -track_id
                                       -prop_type_id) ],
                      filters => [ qw( trim ) ] };
    
    $template = HTML::Template->new(filename=>
                                    'attendees/authors/proposals_new.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    if (defined($h->{-params}->{validate})) {
        $validator = Data::FormValidator->new;
        $val_results = $validator->check ( $h->{-params}, $input_profile);
        if ($val_results->has_missing) {
            $template->param(_validator2html($val_results));
        } else {
            $proposal = Comas::Proposal->new(-db=>$h->{-db},
                                             %{$val_results->valid});
            unless(defined $proposal) {
                if (Comas::Proposal->lastError == 203) {
                    return $h->attendees_menu('PROPOSAL_TOO_LATE');
                }
            } else {
                unless($proposal->add_author($h->{-session}->{-person})) {
                    if (Comas::Proposal->lastError == 302) {
                        $proposal->delete;
                        return $h->attendees_menu('TOO_MANY_PROPOSALS_FOR_YOU');
                    }
                } else {
                    return $h->attendees_menu('PROPOSAL_NEW');
                }
            }
        }
    }

    $template->param($h->{-params});
    $template->param(-track_list => 
                     _catalog2html_opts($h->{-db},"track",
                                        $h->{-params}->{-track_id}),
                     -prop_type_list =>
                     _catalog2html_opts($h->{-db},"prop_type",
                                        $h->{-params}->{-prop_type_id},
                                        'priv_prop_types'));
    return  $template->output;
}

sub proposals_list {
    my ($input_profile, $template, $validator, $val_results, $person,
        @proposals, $prop);
    my $h = shift;
    my $message = shift;
    
    $input_profile = {required => [ qw(-resume) ], filters => [ qw( trim ) ] };
    
    $template = HTML::Template->new(filename=>
                                    'attendees/authors/proposals_list.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    $person = Comas::Person->new(-db => $h->{-db},
                                 -id => $h->{-session}->{-person});
    
    @proposals = $person->get_props;
    
    my ($prop_temp, @loop);
    foreach $prop (@proposals) {
        my %row;
        $prop_temp = Comas::Proposal->new(-db => $h->{-db},
                                          -id => $prop);
        $row{-id} = $prop_temp->{-id};
        $row{-title} = $prop_temp->get_title;
        $row{-prop_status} = _id2descr($h->{-db},
                                      'prop_status',
                                      $prop_temp->get_prop_status_id);
        push (@loop, \%row);
    }    
    $template->param(PROPOSALS => \@loop);

    if (defined $message) {
        $template->param($message => 1);
    }

    return $template->output;    
}

sub proposals_edit {
    my ($input_profile, $template, $validator, $val_results, $proposal,
        %authors, %prop_data);
    my $h = shift;
    
    $input_profile = {required => [ qw(-title -abstract -track_id
                                       -prop_type_id) ],
                      filters => [ qw( trim ) ] };
    
    # Let's chech if the proposal belong to the author.
    $proposal = Comas::Proposal->new(-db => $h->{-db},
                                     -id => $h->{-path}[4]);
    %authors = $proposal->get_authors(-result=>'hash');
    unless ($authors{$h->{-session}->{-person}}) {
        return $h->attendees_menu;
    }

    # Initial template stuff
    $template = HTML::Template->new(filename=>
                                    'attendees/authors/proposals_edit.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);
    
    %prop_data = $proposal->get_data;
    
    $template->param(-id => $prop_data{-id});
    $template->param(-prop_status => _id2descr($h->{-db},
                                               'prop_status',
                                               $prop_data{-prop_status_id}));
    $template->param(-comments => text2html($prop_data{-comments}, 
                                            lines => 1));
    
    # Cool, the form have been posted 
    if (defined($h->{-params}->{validate})) {
        $validator = Data::FormValidator->new;
        $val_results = $validator->check ( $h->{-params}, $input_profile);
        # Whops, something is bad 
        if ($val_results->has_missing) {
            $template->param($h->{-params});
            $template->param(-track_list => 
                             _catalog2html_opts($h->{-db},"track",
                                                $h->{-params}->{-track_id}),
                             -prop_type_list =>
                             _catalog2html_opts($h->{-db},"prop_type",
                                                $h->{-params}->{-prop_type_id},
                                                'priv_prop_types'));
            $template->param(_validator2html($val_results));
            return $template->output;
        } else {
            # Everything is fine, lets update the proposal
            unless($proposal->set_data(%{$val_results->valid})) {
                if (Comas::Proposal->lastError == 206) {
                    return $h->proposals_list('CANT_MODIFY_ACCEPTED_PROPOSAL');
                }
            } else {
                # Let's send the mail to the academic committee
                _send_proposal_modif_mail($proposal,
                                          $h->{-config}->{-tmpl_path} .
                                           'mail/proposal_modif.txt',
                                          %prop_data);
                # And return the proposals lists
                return $h->proposals_list('PROPOSAL_DATA_CHANGED');
            }
        }
    } else {
        $template->param(-title => $proposal->get_title);
        $template->param(-abstract => $proposal->get_abstract);
        $template->param(-track_list => 
                         _catalog2html_opts($h->{-db},"track",
                                            $proposal->get_track_id),
                         -prop_type_list =>
                         _catalog2html_opts($h->{-db},"prop_type",
                                            $proposal->get_prop_type_id,
                                            'priv_prop_types'));
        return $template->output;
    }
}

sub proposals_delete {
    my $h = shift;
    my ($proposal, %authors, $prop_path, $filename, $template);

    # Let's chech if the proposal belong to the author.
    $proposal = Comas::Proposal->new(-db => $h->{-db},
                                     -id => $h->{-path}[4]);

    %authors = $proposal->get_authors(-result=>'hash');

    unless ($authors{$h->{-session}->{-person}}) {
        return $h->attendees_menu;
    }

    if (defined($h->{-params}->{validate})) {
        # We need to know the filename before deleting the proposal, so we can
        # delete it.
        if($proposal->get_filename) {
            $prop_path = $h->{-config}->{-prop_file_path} . $proposal->{-id};
            $filename = $prop_path . '/' . $proposal->get_filename;
        }
        
        unless($proposal->delete) {
            if (Comas::Proposal->lastError == 207) {
                return $h->proposals_list('CANT_DELETE_ACCEPTED_PROPOSAL');
            }
        }
        if($filename) {
            unlink($filename);
            rmdir($prop_path);
        }
        return $h->proposals_list('PROPOSAL_DELETED');
    } else {
        $template = HTML::Template->new(filename=>
                                    'attendees/authors/proposals_delete.tmpl',
                                        path => $h->{-config}->{-tmpl_path},
                                        die_on_bad_params => 0, cache => 1);
        $template->param(-id => $proposal->{-id});
        $template->param(-title => $proposal->get_title);
        return $template->output;
    }
}

sub proposals_authors {
    my $h = shift;
    my ($template, $proposal, %authors, $author);

    # Let's chech if the proposal belong to the author.
    $proposal = Comas::Proposal->new(-db => $h->{-db},
                                     -id => $h->{-path}[4]);
    %authors = $proposal->get_authors(-result=>'hash');
    unless ($authors{$h->{-session}->{-person}}) {
        return $h->attendees_menu;
    }

    $template = HTML::Template->new(filename=>
                                    'attendees/authors/authors_edit.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);

    $template->param(-id => $h->{-path}[4]);

    if (defined $h->{-path}[5]) {
        if ($h->{-path}[5] eq 'add') {
            my $login = $h->{-params}->{-login};
            $login =~ s/^\ +//;
            my @ids = Comas::Person->search(-db=>$h->{-db}, 
                                            -login => $login);
            if ($login =~ /^\s*$/) {
                $template->param(LOGIN_EMPTY => 1);
            } elsif (scalar @ids < 1) {
                $template->param(LOGIN_DONT_EXIST => 1);
            } elsif ($authors{$ids[0]}) {
                $template->param(AUTHOR_ALREDY_ADDED => 1);
            } else {
                unless($proposal->add_author($ids[0])) {
                    if (Comas::Proposal->lastError == 302) {
                        $template->param(TOO_MANY_PROPOSALS => 1);
                    } elsif (Comas::Proposal->lastError == 303) {
                        $template->param(TOO_MANY_AUTHORS => 1);
                    } elsif (Comas::Proposal->lastError == 305) {
                        $template->param(CANT_ADD_AUTHOR_PROPOSAL_ACCEPTED =>
                                         1);
                    }
                } else {
                    $template->param(AUTHOR_ADDED => 1);
                }
            }            
        } elsif ($h->{-path}[5] eq 'move') {
            my $pos = $h->{-params}->{-pos};
            $pos =~ s/^\ +//;
            if ($pos < 0 || $pos > 99 || $pos =~ /\D/ || $pos =~ /^\s*$/) {
                $template->param(BAD_POSITION => 1);
            } else {
                $proposal->move_author($h->{-path}[6], $h->{-params}->{-pos});
                if (Comas::Proposal->lastError == 305) {
                    $template->param(CANT_MOVE_AUTHOR_PROPOSAL_ACCEPTED => 1);
                } else {
                    $template->param(AUTHOR_MOVED => 1);
                }
            }
        } elsif ($h->{-path}[5] eq 'delete') {
            $proposal->del_author($h->{-path}[6]);
            if (Comas::Proposal->lastError == 305) {
                $template->param(CANT_DELETE_AUTHOR_PROPOSAL_ACCEPTED => 1);
            } else {
                $template->param(AUTHOR_DELETED => 1);
                if ($h->{-path}[6] eq $h->{-session}->{-person}) {
                    return $h->proposals_list('DELETED_YOURSELF');
                }
            }
        }
    }


    %authors = $proposal->get_authors(-result=>'hash');
    my ($author_temp, @loop);
    foreach my $author_id (sort { $authors{$a} <=> $authors{$b} }
                           keys %authors) {
        my %row;
        $author_temp = Comas::Person->new(-db => $h->{-db},
                                          -id => $author_id);
        $row{-prop_id} = $proposal->{-id};
        $row{-author_id} = $author_temp->{-id};
        $row{-mention_order} = $authors{$author_id};
        $row{-firstname} = $author_temp->get_firstname;
        $row{-famname} = $author_temp->get_famname;
        $row{-email} = $author_temp->get_email;
        push (@loop, \%row);
    }
    $template->param(AUTHORS => \@loop);

    return $template->output;
}

sub proposals_file {
    my $h = shift;
    my ($proposal, $template, %authors);
    
    # Let's chech if the proposal belong to the author.
    $proposal = Comas::Proposal->new(-db => $h->{-db},
                                     -id => $h->{-path}[4]);
    
    %authors = $proposal->get_authors(-result=>'hash');
    unless ($authors{$h->{-session}->{-person}}) {
        return $h->attendees_menu;
    }
    
    $template = HTML::Template->new(filename=>
                                    'attendees/authors/proposals_file.tmpl',
                                    path => $h->{-config}->{-tmpl_path},
                                    die_on_bad_params => 0, cache => 1);

    $template->param(-id => $proposal->{-id});
    
    if (defined($h->{-path}[5]) && $h->{-path}[5] eq 'put'
        && ! defined($proposal->get_filename)) {
        foreach my $upload ($h->{-upload}) {
            if ($upload->size > 0) {
                unless($proposal->set_filename($upload->filename)) {
                    if (Comas::Proposal->lastError == 206) {
                        $template->param(CANT_MODIFY_ACCEPTED_PROPOSAL => 1);
                    }
                } else {
                    my ($bytesread, $buffer, $cur_dir, $cur_file);
                    my $fh = $upload->fh;
                    $cur_dir = $h->{-config}->{-prop_file_path};
                    $cur_dir .= $proposal->{-id} . '/';
                    $cur_file = '>>'. $cur_dir . $upload->filename;
                    unless(opendir(DIR,$cur_dir)) {
                        mkdir $cur_dir;
                    } else {
                        closedir DIR;
                    }
                    open (OUTFILE, $cur_file);
                    while ($bytesread=read($fh,$buffer,1024)) {
                        print OUTFILE $buffer;
                    }
                    close OUTFILE;
                    _send_proposal_file_mail($proposal,
                                             $h->{-config}->{-tmpl_path} .
                                             'mail/proposal_file.txt',
                                             $h->_get_file_info($proposal));
                    $template->param(FILE_ADDED => 1);
                }
            }
        }
    }
    
    if ($h->{-path}[5] eq 'delete' && defined($proposal->get_filename)) {
        my ($prop_path, $filename);
        $prop_path = $h->{-config}->{-prop_file_path} . $proposal->{-id};
        $filename = $prop_path . '/' . $proposal->get_filename;

        unless($proposal->set_filename(undef)) {
            if (Comas::Proposal->lastError == 206) {
                $template->param(CANT_MODIFY_ACCEPTED_PROPOSAL => 1);
            }
        } else {
            unlink($filename);
            rmdir($prop_path);
            $template->param(FILE_DELETED => 1);
        }
    }
    
    if ($proposal->get_filename) {
        my %file_info = $h->_get_file_info($proposal);
        $template->param(-FILENAME => $proposal->get_filename,
                         -SIZE => $file_info{-size},
                         -YEAR => $file_info{-year},
                         -MONTH => $file_info{-month},
                         -DAY => $file_info{-day},
                         -HOUR => $file_info{-hour},
                         -MIN => $file_info{-min});
    }
    
    return $template->output;
}

sub _send_proposal_modif_mail {
    my ($mail, %mail_subst, $smtp_host, $mail_to, $smtp, $mail_from);

    my $proposal = shift;
    my $mail_tmpl = shift;
    my %old_data = @_;
    my %new_data = $proposal->get_data;

    unless (-r $mail_tmpl and open(TMPL, '<', $mail_tmpl)) {
	warn "Could not open mail template ($mail_tmpl): $!\n";
	warn 'Mail will not be sent to the academic committee.';
	return undef;
    }
    while (<TMPL>) {
        $mail .= $_;
    }
    close(TMPL);

    $mail_subst{PROP_ID} = $old_data{-id};    

    $mail_subst{TITLE} = $old_data{-title};
    unless ($old_data{-title} eq $new_data{-title}) {
        $mail_subst{NEW_TITLE} =  $new_data{-title};
    } else {
        $mail_subst{NEW_TITLE} = 'No changes';
    }

    $mail_subst{ABSTRACT} = $old_data{-abstract};
    unless ($old_data{-abstract} eq $new_data{-abstract}) {
        $mail_subst{NEW_ABSTRACT} =  $new_data{-abstract};
    } else {
        $mail_subst{NEW_ABSTRACT} = 'No changes';
    }

    $mail_subst{PROP_TYPE} = _id2descr($proposal->{-db},
                                       'prop_type',
                                       $old_data{-prop_type_id});
    
    unless ($old_data{-prop_type_id} eq $new_data{-prop_type_id}) {
        $mail_subst{NEW_PROP_TYPE} = _id2descr($proposal->{-db},
                                               'prop_type',
                                               $new_data{-prop_type_id});
    } else {
        $mail_subst{NEW_PROP_TYPE} = 'No changes';
    }
    
    $mail_subst{TRACK} = _id2descr($proposal->{-db},
                                   'track',
                                   $old_data{-track_id});
    unless ($old_data{-track_id} eq $new_data{-track_id}) {
        $mail_subst{NEW_TRACK} = _id2descr($proposal->{-db},
                                             'track',
                                             $new_data{-track_id});
    } else {
        $mail_subst{NEW_TRACK} = 'No changes';
    }

    $smtp_host = get_conf($proposal->{-db}, 'smtp_host') || '127.0.0.1';
    
    unless($mail_to = get_conf($proposal->{-db}, 'proposal_modif_mail_to')) {
        warn "Couldn't get recipient address, please check your configuration.\n";
	warn "Mail will not be sent to the academic committee.\n";
	return undef;
    }

    unless($mail_from = get_conf($proposal->{-db}, 'proposal_modif_mail_from')) {
        warn "Couldn't get sender address, please check your configuration.\n";
	warn "Mail will not be sent to the academic committee.\n";
	return undef;
   }
    
    # Here goes my mail-processing mini-template-engine!
    for my $key (keys %mail_subst) {
        $mail =~ s/__${key}__/$mail_subst{$key}/gs;
    }
    
    unless ($smtp = Net::SMTP->new($smtp_host)) {
        warn "Couldn't contact SMTP server\n";
        warn "Mail will not be sent to the authors.\n";
        return undef;
    }

    $smtp->mail($mail_from);
    $smtp->to($mail_to);
    $smtp->data();
    $smtp->datasend($mail);
    $smtp->dataend();
    $smtp->quit();
    return 1;
}

sub _send_proposal_file_mail {
    my ($mail, %mail_subst, $smtp_host, $mail_to, $smtp, $mail_from);
    my ($proposal, $mail_tmpl, %file_info) = @_;
    my %prop_data = $proposal->get_data;
    unless (-r $mail_tmpl and open(TMPL, '<', $mail_tmpl)) {
	warn "Could not open mail template ($mail_tmpl): $!\n";
	warn 'Mail will not be sent to the academic committee.';
	return undef;
    }
    while (<TMPL>) {
        $mail .= $_;
    }
    close(TMPL);

    $mail_subst{PROP_ID} = $prop_data{-id};
    $mail_subst{TITLE} = $prop_data{-title};

    $mail_subst{FILENAME} = $proposal->get_filename;
    $mail_subst{SIZE} = $file_info{-size};
    $mail_subst{YEAR} = $file_info{-year};
    $mail_subst{MONTH} = $file_info{-month};
    $mail_subst{DAY} = $file_info{-day};
    $mail_subst{HOUR} = $file_info{-hour};
    $mail_subst{MIN} = $file_info{-min};

    $smtp_host = get_conf($proposal->{-db}, 'smtp_host') || '127.0.0.1';

    unless($mail_to = get_conf($proposal->{-db}, 'proposal_modif_mail_to')) {
        warn "Couldn't get recipient address, please check your configuration.\n";
	warn "Mail will not be sent to the academic committee.\n";
	return undef;
    }

    unless($mail_from = get_conf($proposal->{-db}, 'proposal_modif_mail_from')) {
        warn "Couldn't get sender address, please check your configuration.\n";
	warn "Mail will not be sent to the academic committee.\n";
	return undef;
   }
    
    # Here goes my mail-processing mini-template-engine!
    for my $key (keys %mail_subst) {
        $mail =~ s/__${key}__/$mail_subst{$key}/gs;
    }
    
    unless ($smtp = Net::SMTP->new($smtp_host)) {
        warn "Couldn't contact SMTP server\n";
        warn "Mail will not be sent to the academic committee.\n";
        return undef;
    }

    $smtp->mail($mail_from);
    $smtp->to($mail_to);
    $smtp->data();
    $smtp->datasend($mail);
    $smtp->dataend();
    $smtp->quit();
    return 1;
}

1;

# $Log: authors.pm,v $
# Revision 1.16  2004/02/04 18:09:54  mig
# - Restringo tipos de ponencia (pa las magistrales)
#
# Revision 1.15  2004/01/20 17:49:10  mig
# - Evitando warinings :)
#
# Revision 1.14  2004/01/11 04:56:35  mig
# - Agrego _send_proposal_file_mail y _send_proposal_modif_mail para
#   notificaciones de correo.
# - La informacin de achivos la saco de _get_file_info y no ahi mismo (menos
#   codigo :-D )
#
# Revision 1.13  2003/12/21 07:31:14  mig
# - Agrego confirmacion antes de eliminar una ponencia.
# - Simplifico la actualizacin de informacin de autores.
#
# Revision 1.12  2003/12/21 04:30:17  mig
# - Ya se normalizan los paths al cargar la configuracion cuando se crea el
#   objeto, menos codigo :)
#
# Revision 1.11  2003/12/20 04:14:51  mig
# - Agrego tags Id y Log que expanda el CVS
#
