/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// Qt includes
#include <qdragobject.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qtimer.h>
#include <qsplitter.h>
#include <qhbox.h>
#include <qvbox.h>

// KDE includes
#include <kapplication.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kstatusbar.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kurlrequesterdlg.h>
#include <kmessagebox.h>
#include <ktempfile.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>
#include <klistviewsearchline.h>
#include <kinputdialog.h>
#include <kdebug.h>

// WorKflow includes
#include "mainwindow.h"
#include "view.h"
#include "document.h"
#include "pref.h"
#include "librarymanager.h"
#include "categorylistview.h"
#include "commandlistview.h"
#include "helpwidget.h"
#include "selectworkflowdialog.h"

MainWindow::MainWindow()
    : KMainWindow(0, "MainWindow")
{
  m_splitter = new QSplitter(Qt::Horizontal, this);

  m_helpSplitter = new QSplitter(Qt::Vertical, m_splitter);

  QVBox* left = new QVBox(m_helpSplitter);
  m_searchLine = new KListViewSearchLineWidget(0, left);

  QHBox* listviews = new QHBox(left);
  m_categoryList = new WorKflow::CategoryListView(listviews);
  m_commandList = new WorKflow::CommandListView(listviews);
  m_helpWidget = new WorKflow::HelpWidget(m_helpSplitter);
  m_view = new WorKflow::View(m_splitter);

  connect(m_categoryList, SIGNAL(categorySelected(const QString&)), m_commandList, SLOT(setCategory(const QString&)));
  connect(m_commandList, SIGNAL(executed(const QString&)), m_view, SLOT(appendCommand(const QString&)));
  connect(m_commandList, SIGNAL(selected(CommandDescription*)), m_helpWidget, SLOT(setCommand(CommandDescription*)));

  setCentralWidget(m_splitter);

  setupActions();
  readSettings();

  statusBar()->show();
  setupGUI();

  // allow the view to change the statusbar
  connect(m_view, SIGNAL(signalChangeStatusbar(const QString&)),
          this,   SLOT(changeStatusbar(const QString&)));
  connect(m_view, SIGNAL(selectionChanged()), this, SLOT(changeActionState()));
  connect(m_view->document(), SIGNAL(undoChanged()), this, SLOT(changeActionState()));

  QTimer::singleShot(0, this, SLOT(init()));
}

MainWindow::~MainWindow()
{
  writeSettings();
}

void MainWindow::openOnStartup(const KURL& url, bool exec)
{
  m_startupUrl = url;
  m_justExecute = exec;
}

void MainWindow::setupActions()
{
  KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
  KStdAction::open(this, SLOT(fileOpen()), actionCollection());
  m_openRecent = KStdAction::openRecent(this, SLOT(fileOpen(const KURL&)), actionCollection());
  m_save = KStdAction::save(this, SLOT(fileSave()), actionCollection());
  KStdAction::saveAs(this, SLOT(fileSaveAs()), actionCollection());
//   KStdAction::print(this, SLOT(filePrint()), actionCollection());

  new KAction(i18n("Export..."), 0, this, SLOT(exportWorkflow()), actionCollection(), "export");
  new KAction(i18n("Import..."), 0, this, SLOT(importWorkflow()), actionCollection(), "import");

  KStdAction::quit(kapp, SLOT(quit()), actionCollection());

  m_cut = KStdAction::cut(m_view, SLOT(cut()), actionCollection());
  m_copy = KStdAction::copy(m_view, SLOT(copy()), actionCollection());
  m_paste = KStdAction::paste(m_view, SLOT(paste()), actionCollection());

  KStdAction::selectAll(m_view, SLOT(selectAll()), actionCollection());
  m_deselect = KStdAction::deselect(m_view, SLOT(clearSelection()), actionCollection());

  m_undo = KStdAction::undo(m_view->document(), SLOT(undo()), actionCollection());
  m_redo = KStdAction::redo(m_view->document(), SLOT(redo()), actionCollection());

  KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

  m_collapseAll = new KAction(i18n("Collapse All Commands"), 0,
                                m_view, SLOT(collapseAll()),
                                actionCollection(), "collapse_all");

  m_uncollapseAll = new KAction(i18n("Uncollapse All Commands"), 0,
                                m_view, SLOT(uncollapseAll()),
                                actionCollection(), "uncollapse_all");

  m_fullscreen = KStdAction::fullScreen(this, SLOT(toggleFullScreen()), actionCollection(), this);

  m_execute = new KAction(i18n("Execute"), "run", CTRL + Key_R,
                                this, SLOT(execute()),
                                actionCollection(), "execute");
}

void MainWindow::saveProperties(KConfig *config)
{
  if (!m_view->currentURL().isEmpty()) {
#if KDE_IS_VERSION(3,1,3)
    config->writePathEntry("lastURL", m_view->currentURL());
#else
    config->writeEntry("lastURL", m_view->currentURL());
#endif
  }
}

void MainWindow::readProperties(KConfig *config)
{
  QString url = config->readPathEntry("lastURL");

  if (!url.isEmpty())
      m_view->openURL(KURL(url));
}

void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(KURLDrag::canDecode(event));
}

void MainWindow::dropEvent(QDropEvent *event)
{
    // this is a very simplistic implementation of a drop event.  we
    // will only accept a dropped URL.  the Qt dnd code can do *much*
    // much more, so please read the docs there
    KURL::List urls;

    // see if we can decode a URI.. if not, just ignore it
    if (KURLDrag::decode(event, urls) && !urls.isEmpty())
    {
        // okay, we have a URI.. process it
        const KURL &url = urls.first();

        // load in the file
        fileOpen(url);
    }
}

void MainWindow::fileNew()
{
  newWindow();
}

void MainWindow::fileOpen(const KURL& url)
{
  if (!url.isValid()) {
    KMessageBox::error(this, i18n("Malformed URL\n%1").arg(url.url()));
    return;
  }

  MainWindow* wnd = newWindow();

  if (!url.isEmpty()) {
    QString tmpfile;
    if (KIO::NetAccess::download(url, tmpfile, wnd)) {
      if (wnd->m_view->document()->load(tmpfile)) {
        KIO::NetAccess::removeTempFile(tmpfile);

        wnd->m_openRecent->addURL(url);

        wnd->m_view->document()->setUrl(url);
        wnd->updateCaption();
      } else {
        KMessageBox::error(this, i18n("Could not parse workflow file"));
      }
    } else {
      KMessageBox::error(this, i18n("Could not download workflow file"));
    }
  }
}

void MainWindow::fileOpen()
{
  kdDebug() << "Open: mainappid: " << WorKflow::SelectWorkflowDialog::mainAppId() << endl;
  KURL url(WorKflow::SelectWorkflowDialog::edit(this, WorKflow::SelectWorkflowDialog::mainAppId()));
  fileOpen(url);
}

bool MainWindow::fileSave()
{
  KURL url = m_view->document()->url();
  if (url.isEmpty()) {
    return fileSaveAs();
  } else {
    KTempFile tmpFile;
    tmpFile.setAutoDelete(true);

    if (m_view->document()->save(tmpFile.name())) {
      if (!KIO::NetAccess::upload(tmpFile.name(), url, this)) {
        KMessageBox::error(this, i18n("Could not save the workflow"));
        return false;
      } else {
        changeActionState();
      }
    } else {
      KMessageBox::error(this, i18n("Could not save the workflow"));
      return false;
    }
  }
  return true;
}

bool MainWindow::fileSaveAs()
{
  bool ok;
  QString name = KInputDialog::getText(i18n("Workflow Name"), i18n("Save this workflow under the following name"), "", &ok, this);
  if (ok) {
    QString dir = WorKflow::SelectWorkflowDialog::appDir();
    if (dir.isNull()) {
      KMessageBox::error(this, i18n("Could not save the workflow. Please try <b>File->Export...</b>"), i18n("Error saving workflow"));
    } else {
      if (!name.endsWith(".workflow"))
        name += ".workflow";
      KURL url(dir + name);
      m_view->document()->setUrl(url);
      return fileSave();
    }
  }
  return false;
}

void MainWindow::filePrint()
{
    // this slot is called whenever the File->Print menu is selected,
    // the Print shortcut is pressed (usually CTRL+P) or the Print toolbar
    // button is clicked
/*    if (!m_printer) m_printer = new KPrinter;
    if (m_printer->setup(this))
    {
        // setup the printer.  with Qt, you always "print" to a
        // QPainter.. whether the output medium is a pixmap, a screen,
        // or paper
        QPainter p;
        p.begin(m_printer);

        // we let our view do the actual printing
        QPaintDeviceMetrics metrics(m_printer);
        m_view->print(&p, metrics.height(), metrics.width());

        // and send the result to the printer
        p.end();
    }*/
}

void MainWindow::optionsPreferences()
{
    // popup some sort of preference dialog, here
    WorKflowPreferences dlg;
    if (dlg.exec())
    {
        // redo your settings
    }
}

void MainWindow::changeStatusbar(const QString& text)
{
    // display the text on the statusbar
    statusBar()->message(text);
}

void MainWindow::init()
{
  WorKflow::LibraryManager::self()->loadAllLibraries();
  changeActionState();

  KListViewSearchLine* line = m_searchLine->searchLine();
  line->setListView(m_commandList);

  connect(line, SIGNAL(textChanged(const QString&)), m_categoryList, SLOT(setFilter(const QString&)));

  QValueList<int> columns;
  columns.append(0);
  columns.append(1);

  line->setSearchColumns(columns);

  updateCaption();

  if (!m_startupUrl.isEmpty()) {
    fileOpen(m_startupUrl);
    if (m_justExecute) {
      execute();
      kapp->exit(0);
    }
  }
}

void MainWindow::changeActionState()
{
  m_save->setEnabled(m_view->document()->isModified());

  m_cut->setEnabled(m_view->hasSelection());
  m_copy->setEnabled(m_view->hasSelection());
  m_deselect->setEnabled(m_view->hasSelection());

  m_undo->setEnabled(m_view->document()->undoCount());
  m_redo->setEnabled(m_view->document()->redoCount());

  updateCaption();
}

void MainWindow::execute()
{
  setDisabled(true);
  m_view->execute();
  setDisabled(false);
}

void MainWindow::readSettings()
{
  KConfig* conf = kapp->config();

  m_openRecent->loadEntries(conf);

  conf->setGroup("MainWindow");
  QValueList<int> sizes = conf->readIntListEntry("HorzSplitter");
  if (!sizes.isEmpty())
    m_splitter->setSizes(sizes);
  sizes = conf->readIntListEntry("VertSplitter");
  if (!sizes.isEmpty())
    m_helpSplitter->setSizes(sizes);
}

void MainWindow::writeSettings()
{
  KConfig* conf = kapp->config();

  m_openRecent->saveEntries(conf);

  conf->setGroup("MainWindow");
  QValueList<int> sizes = m_splitter->sizes();
  conf->writeEntry("HorzSplitter", sizes);
  sizes = m_helpSplitter->sizes();
  conf->writeEntry("VertSplitter", sizes);

  conf->sync();
}

MainWindow* MainWindow::newWindow()
{
  if (m_view->document()->url().isEmpty())
    return this;

  MainWindow* wnd = new MainWindow();
  wnd->show();
  return wnd;
}

void MainWindow::updateCaption()
{
  setCaption(m_view->document()->fileName(), m_view->document()->isModified());
}

void MainWindow::toggleFullScreen()
{
  if (m_fullscreen->isChecked())
    showFullScreen();
  else
    showNormal();
}


bool MainWindow::queryClose()
{
  if (!m_view->document()->isModified())
    return true;

  QString msg = i18n("This workflow has been modified.\n"
                     "Would you like to save it?");
  switch (KMessageBox::warningYesNoCancel(this, msg, QString::null, KStdGuiItem::save(), KStdGuiItem::discard()))
  {
  case KMessageBox::Yes:
    return fileSave();

  case KMessageBox::No:
    return true;

  case KMessageBox::Cancel:
  default:
    return false;
  }

  return true;
}

void MainWindow::exportWorkflow()
{
  KURL url = KFileDialog::getSaveURL(QString::null, i18n("*.workflow|WorKflow documents"), this, i18n("Save Workflow"));
  if (url.isEmpty())
    return;

  if (!url.isValid()) {
    KMessageBox::error(this, i18n("Malformed URL\n%1").arg(url.url()));
    return;
  }

  m_openRecent->addURL(url);
  m_view->document()->setUrl(url);
  fileSave();
}

void MainWindow::importWorkflow()
{
  // hack. TODO: make KIO-aware
  KURL url = KFileDialog::getOpenURL(QString::null, i18n("*.workflow|WorKflow documents"), this, i18n("Open Workflow"));
  if (url.isEmpty())
    return;

  fileOpen(url);
}

#include "mainwindow.moc"
