/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef WORKFLOWDOCUMENT_H
#define WORKFLOWDOCUMENT_H

// Qt includes
#include <qobject.h>

// WorKflow includes
#include "value.h"

// forward declarations
class KURL;

class QDomElement;
class QDomDocument;

namespace KIO
{
  class Job;
}

namespace WorKflow
{
  class Command;
  class Parameter;
  class ApplicationManager;
}

namespace WorKflow
{
  /**
   * @author Thomas Kadauke
   */
  class Document : public QObject
  {
    Q_OBJECT

  public:
    class Iterator
    {
    public:
      Iterator(Document* document, int row = 0);

      Command* operator*();
      Iterator& operator++();

      bool operator==(const Iterator& it);
      bool operator!=(const Iterator& it);

      inline int row() { return m_row; }

    private:
      Document* m_document;
      int m_row;
      int m_column;
    };

    Document(QObject *parent = 0, const char *name = 0);
    ~Document();

    int numRows();

    Command* commandAt(int row);

    Iterator begin();
    Iterator end();
    Iterator iterator(int row);

    bool isExecuting();

    void insertCommand(int row, Command* command);

    void removeCommand(int row);
    void removeRange(int startRow, int endRow);

    void moveCommand(int sourceRow, int destRow);

    void clear();

    int undoCount();
    int redoCount();

    Parameter* search(int startRow, const QString& text);
    Parameter* findNext();
    Parameter* findPrev();

//     bool print();

    bool load(const QString& filename);
    bool save(const QString& filename);
//     bool openFile(KIO::Job* job);

    void cut(int startRow, int endRow);
    void copy(int startRow, int endRow);
    void paste(int row);

    QString fileName();
    void setUrl(const KURL& url);
    KURL url();
    bool isModified();

    void doInsertCommand(int row, Command* command);

    void doRemoveCommand(int row);

    Value variable(const QString& name) const;
    void setVariable(const QString& name, const Value& value);

    ApplicationManager* appManager();

    void readXML(const QDomElement& e);
    void writeXML(QDomDocument& doc, QDomElement& e);

  public slots:
    void abort();
    void undo();
    void redo();

    void execute();

  signals:
    void modified();
    void filenameChanged();

    void commandAdded(Command*);
    void commandRemoved(Command*);
    void orderChanged();

    void undoChanged();

    void progress(double total, double current, const QString& command, const QString& task);

  protected:
    void reorderCommands();
    void emitUndoSignals();

  protected slots:
    void typeCheck();

  public:
    class Private;

  private:
    friend class Private;
    Private* d;
  };
}

#endif
