/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qdom.h>

// WorKflow includes
#include "commanddescription.h"
#include "commandmanager.h"
#include "parameterdescription.h"
#include "resultdescription.h"
#include "librarydescription.h"

using namespace WorKflow;

CommandDescription::CommandDescription(LibraryDescription* parent)
  : XMLDescription(), m_parent(parent), m_factory(0), m_hidden(false)
{
  m_parent->addCommand(this);
}

CommandDescription::~CommandDescription()
{
  m_parent->removeCommand(this);
  CommandManager::self()->removeCommandDescription(id());

  while (!m_parameters.isEmpty())
    delete m_parameters.first();

  while (!m_results.isEmpty())
    delete m_results.first();
}

CommandFactoryBase* CommandDescription::factory() const
{
  return m_factory;
}

void CommandDescription::setFactory(CommandFactoryBase* factory)
{
  m_factory = factory;
}

ParameterDescription* CommandDescription::parameterDescription(const QString& id) const
{
  if (m_parameterMap.contains(id))
    return m_parameterMap[id];
  return 0;
}

ResultDescription* CommandDescription::resultDescription(const QString& id) const
{
  if (m_resultMap.contains(id))
    return m_resultMap[id];
  return 0;
}

QStringList CommandDescription::acceptedDropTypes() const
{
  return m_acceptedDropTypes;
}

QString CommandDescription::iconName() const
{
  return m_iconName;
}

QStringList CommandDescription::categories() const
{
  return m_categories;
}

QStringList CommandDescription::keywords() const
{
  return m_keywords;
}

void CommandDescription::setIconName(const QString& icon)
{
  m_iconName = icon;
}

void CommandDescription::addCategory(const QString& category)
{
  m_categories << category;
}

void CommandDescription::addKeyword(const QString& keyword)
{
  m_keywords << keyword;
}

CommandDescription::ParamDescList CommandDescription::parameters() const
{
  return m_parameters;
}

CommandDescription::ResultDescList CommandDescription::results() const
{
  return m_results;
}

void CommandDescription::addParameterDescription(ParameterDescription* desc)
{
  m_parameters.append(desc);
  m_parameterMap[desc->id()] = desc;
}

void CommandDescription::addResultDescription(ResultDescription* desc)
{
  m_results.append(desc);
  m_resultMap[desc->id()] = desc;
}

QString CommandDescription::uiFile() const
{
  return m_uiFile;
}

void CommandDescription::setUiFile(const QString& file)
{
  m_uiFile = file;
}

bool CommandDescription::isHidden() const
{
  return m_hidden;
}

void CommandDescription::removeParameterDescription(ParameterDescription* desc)
{
  m_parameters.erase(m_parameters.find(desc));
  m_parameterMap.erase(desc->id());
}

void CommandDescription::removeResultDescription(ResultDescription* desc)
{
  m_results.erase(m_results.find(desc));
  m_resultMap.erase(desc->id());
}

void CommandDescription::readXML(const QDomElement& e)
{
  XMLDescription::readXML(e);

  m_hidden = (e.attribute("hidden") == "true");

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "categories")
        readCategories(e);
      else if (e.tagName() == "keywords")
        readKeywords(e);
      else if (e.tagName() == "icon")
        setIconName(e.text());
      else if (e.tagName() == "param") {
        ParameterDescription* desc = new ParameterDescription(this);
        desc->readXML(e);
      } else if (e.tagName() == "result") {
        ResultDescription* desc = new ResultDescription(this);
        desc->readXML(e);
      }
    }
    n = n.nextSibling();
  }

  CommandManager::self()->addCommandDescription(id(), this);
}

void CommandDescription::readCategories(const QDomElement& e)
{
  QDomNode i = e.firstChild();
  while (!i.isNull()) {
    QDomElement item = i.toElement();
    if (!item.isNull()) {
      if (item.tagName() == "category") {
        QString id = item.attribute("id");

        if (id.isNull()) {
          kdWarning() << "malformed category in command " << this->id() << endl;
        } else {
          m_categories.append(id);
        }
      }
    }
    i = i.nextSibling();
  }
}

void CommandDescription::readKeywords(const QDomElement& e)
{
  QDomNode i = e.firstChild();
  while (!i.isNull()) {
    QDomElement item = i.toElement();
    if (!item.isNull()) {
      if (item.tagName() == "keyword") {
        QString keyword = item.text();
        if (keyword.isNull()) {
          kdWarning() << "malformed keyword in command " << id() << endl;
        } else {
          m_keywords.append(keyword);
        }
      }
    }
    i = i.nextSibling();
  }
}

void CommandDescription::writeXML(QDomDocument& doc, QDomElement& e)
{
  QDomElement cmd = doc.createElement("command");
  XMLDescription::writeXML(doc, cmd);

  if (m_hidden)
    cmd.setAttribute("hidden", "true");

  QDomElement icon;
  icon.setTagName("icon");
  QDomText text;
  text.setData(m_iconName);
  icon.appendChild(text);
  e.appendChild(icon);

  writeCategories(doc, cmd);
  writeKeywords(doc, cmd);
  writeParameters(doc, cmd);
  writeResults(doc, cmd);

  e.appendChild(cmd);
}

void CommandDescription::writeCategories(QDomDocument& doc, QDomElement& e)
{
  if (m_categories.isEmpty())
    return;

  QDomElement categories = doc.createElement("categories");

  for (QStringList::ConstIterator i = m_categories.begin(); i != m_categories.end(); ++i) {
    QDomElement cat = doc.createElement("category");
    cat.setAttribute("id", *i);
    categories.appendChild(cat);
  }

  e.appendChild(categories);
}

void CommandDescription::writeKeywords(QDomDocument& doc, QDomElement& e)
{
  if (m_categories.isEmpty())
    return;

  QDomElement keywords = doc.createElement("keywords");

  for (QStringList::ConstIterator i = m_keywords.begin(); i != m_keywords.end(); ++i) {
    QDomElement key  = doc.createElement("keyword");
    QDomText text = doc.createTextNode(*i);
    key.appendChild(text);
    keywords.appendChild(key);
  }

  e.appendChild(keywords);
}

void CommandDescription::writeParameters(QDomDocument& doc, QDomElement& e)
{
  for (ParamDescList::ConstIterator i = m_parameters.begin(); i != m_parameters.end(); ++i) {
    (*i)->writeXML(doc, e);
  }
}

void CommandDescription::writeResults(QDomDocument& doc, QDomElement& e)
{
  for (ResultDescList::ConstIterator i = m_results.begin(); i != m_results.end(); ++i) {
    (*i)->writeXML(doc, e);
  }
}

void CommandDescription::clearCategories()
{
  m_categories.clear();
}

void CommandDescription::clearKeywords()
{
  m_keywords.clear();
}
