/*
    KMLOFax
    
    A utility to process facsimiles received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdio.h>
#include <unistd.h>

#include <qglobal.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qwhatsthis.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>

#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>

#include "kmlofax.h"
#include "configure.h"
#include "faxview.h"
#include "downloaddlg.h"
#include "preview.h"
#include "printdlg.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "global.h"




KMLOFaxApp::KMLOFaxApp() : KUniqueApplication()
{
}


KMLOFaxApp::~KMLOFaxApp()
{
}


bool KMLOFaxApp::x11EventFilter(XEvent *ev)
{
	if (KUniqueApplication::x11EventFilter(ev))
		return true;
		
	if (fw)
		fw->x11EventFilter(ev);

	return false;
}



KMLOFax::KMLOFax(QWidget *parent, const char *name) : KMainWindow(parent, name)
{
	config = kapp->config();

	initMenuBar();
	initToolBar();
	initStatusBar();
	initKeyAccel();
	
	configure = new Configure(this, "configure");
	CHECK_PTR(configure);

	senderaliases = new SenderAliases(this, "senderaliases");
	CHECK_PTR(senderaliases);

	view = new FaxView(this, "faxview");
	CHECK_PTR(view);
	setCentralWidget(view);
	
	initWhatsThis();
	
	downloaddlg = new DownloadDlg(this, "downloaddlg");
	CHECK_PTR(downloaddlg);
	preview = new Preview(0, "preview");
	CHECK_PTR(preview);
	
	readOptions();

	slotNewConfig();

	hasSelectionSlot(false);

	connect(configure, SIGNAL(newConfig()), SLOT(slotNewConfig()));

	connect(downloaddlg, SIGNAL(message(const QString &)), SLOT(slotStatusHelpMsg(const QString &)));
	connect(downloaddlg, SIGNAL(gotFile(const QString &)), SLOT(fetchFile(const QString &)));
	connect(downloaddlg, SIGNAL(done()), SLOT(fetchDone()));

	connect(view, SIGNAL(hasSelection(bool)), SLOT(hasSelectionSlot(bool)));
	connect(view, SIGNAL(command(int)), SLOT(commandCallback(int)));

	show();

	slotStatusHelpMsg(i18n("%1 File(s) in directory.").arg(view->loadDir()));
}


KMLOFax::~KMLOFax()
{
	delete preview;
}


void KMLOFax::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void KMLOFax::readOptions()
{
	config->setGroup("MainWindow Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		toolBar()->hide();
	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		statusBar()->hide();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	QSize geoSize(650, 400);
	resize(config->readSizeEntry("Geometry", &geoSize));

	view->readOptions();
}


void KMLOFax::saveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());

	view->saveOptions();
}


bool KMLOFax::queryClose()
{
	view->saveDir();

	return true;
}


void KMLOFax::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "mainfilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(SmallIcon("download"), i18n("&Download"), ID_FILE_DOWNLOAD);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("viewmag"), i18n("&View..."), ID_FILE_VIEW);
	fileMenu->insertItem(SmallIcon("fileprint"), i18n("&Print..."), ID_FILE_PRINT);
	fileMenu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_FILE_EXPORT);
	fileMenu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_FILE_MAIL);
	fileMenu->insertItem(SmallIcon("remove"), i18n("De&lete"), ID_FILE_DELETE);
	fileMenu->insertItem(i18n("Sender &Alias..."), ID_FILE_SENDER_ALIAS);
	fileMenu->insertSeparator();
	fileMenu->insertItem(i18n("P&urge Directory"), ID_FILE_PURGE_DIRECTORY);
	fileMenu->insertItem(i18n("&Rebuild Directory"), ID_FILE_REBUILD_DIRECTORY);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("exit"), i18n("&Quit"), ID_FILE_QUIT);

	settingsMenu = new QPopupMenu(0, "mainsettingsmenu");
	CHECK_PTR(settingsMenu);
	settingsMenu->setCheckable(true);
	settingsMenu->insertItem(i18n("Show &Toolbar"), ID_SETTINGS_TOOLBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, true);
	settingsMenu->insertItem(i18n("Show &Statusbar"), ID_SETTINGS_STATUSBAR);
	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, true);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Save &Options"), ID_SETTINGS_SAVEOPTIONS);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Sender &Aliases..."), ID_SETTINGS_SENDER_ALIASES);
	settingsMenu->insertItem(SmallIcon("configure"), i18n("&Configure KMLOFax..."), ID_SETTINGS_CONFIGURE);
	
	helpMenu_ = helpMenu();
	
	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu_);

	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(settingsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void KMLOFax::initToolBar()
{
	toolBar()->insertButton(BarIcon("download"), ID_FILE_DOWNLOAD, true, i18n("Download Facsimiles"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("viewmag"), ID_FILE_VIEW, true, i18n("View Facsimile"));
	toolBar()->insertButton(BarIcon("fileprint"), ID_FILE_PRINT, true, i18n("Print Facsimile"));
	toolBar()->insertButton(BarIcon("filesave"), ID_FILE_EXPORT, true, i18n("Export Facsimile"));
	toolBar()->insertButton(BarIcon("mail_send"), ID_FILE_MAIL, true, i18n("Mail Facsimile"));
	toolBar()->insertButton(BarIcon("remove"), ID_FILE_DELETE, true, i18n("Delete Facsimile"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("configure"), ID_SETTINGS_CONFIGURE, true, i18n("Configure KMLOFax"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->insertButton(BarIcon("exit"), ID_FILE_QUIT, true, i18n("Quit"));
	toolBar()->alignItemRight(ID_FILE_QUIT, true);

	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void KMLOFax::initStatusBar()
{
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);
}


void KMLOFax::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Download Facsimiles"), "DownloadFiles", CTRL+Key_D);
	
	keyAccel->insertItem(i18n("View Facsimile"), "ViewFile", CTRL+Key_V);
	keyAccel->insertItem(i18n("Export Facsimile"), "ExportFile", CTRL+Key_E);
	keyAccel->insertItem(i18n("Mail Facsimile"), "MailFile", CTRL+Key_M);
	keyAccel->insertItem(i18n("Delete Facsimile"), "DeleteFile", Key_Delete);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);

	keyAccel->readSettings();
	
	keyAccel->connectItem("DownloadFiles", this, SLOT(slotFileDownload()));
	keyAccel->connectItem("ViewFile", this, SLOT(slotFileView()));
	keyAccel->connectItem(KStdAccel::Print, this, SLOT(slotFilePrint()));
	keyAccel->connectItem("ExportFile", this, SLOT(slotFileExport()));
	keyAccel->connectItem("MailFile", this, SLOT(slotFileMail()));
	keyAccel->connectItem("DeleteFile", this, SLOT(slotFileDelete()));
	keyAccel->connectItem(KStdAccel::Quit, this, SLOT(slotFileQuit()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotSettingsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotSettingsShowStatusbar()));
	
	keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));

	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DOWNLOAD, "DownloadFiles");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_VIEW, "ViewFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_PRINT, KStdAccel::Print);
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_EXPORT, "ExportFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_MAIL, "MailFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DELETE, "DeleteFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KStdAccel::Quit);

	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(settingsMenu, ID_SETTINGS_STATUSBAR, "ShowStatusbar");
}


void KMLOFax::initWhatsThis()
{
	QWhatsThis::add(toolBar()->getButton(ID_FILE_DOWNLOAD),
		i18n("Click this button to download facsimiles from your modem's flash memory.<br><br>" \
		"You can also select the command <b>Download</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_VIEW),
		i18n("Click this button to view the facsimile that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the command <b>View</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_PRINT),
		i18n("Click this button to print the facsimile that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the command <b>Print</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_EXPORT),
		i18n("Click this button to export the facsimile " \
		"that is currently selected in the list below into a TIFF file.<br><br>" \
		"You can also select the command <b>Export</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_MAIL),
		i18n("Click this button to send the facsimile that is " \
		"currently selected in the list below as email.<br><br>" \
		"You can also select the command <b>Mail</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_DELETE),
		i18n("Click this button to delete the facsimile that " \
		"is currently selected in the list below.<br><br>" \
		"You can also select the command <b>Delete</b> from the \"File\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_SETTINGS_CONFIGURE),
		i18n("Opens the configure dialog.<br><br>" \
		"You can also select the command <b>Configure KMLOFax</b> from the \"Settings\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KMLOFax help pages displayed.<br><br>" \
		"You can also select the command <b>Contents</b> from the \"Help\" menu."));
	QWhatsThis::add(toolBar()->getButton(ID_FILE_QUIT),
		i18n("Click this button to quit KMLOFax.<br><br>" \
		"You can also select the command <b>Quit</b> from the \"Help\" menu."));
	QWhatsThis::add(view,
		i18n("The list shows the facsimiles you have formerly " \
		"received. The columns of this list have the following meaning:<br>" \
		"<dl compact>"
		"<dt>Name<dd>File name of the facsimile document." \
		"<dt>New<dd>Indicates if the file is new." \
		"<dt>Date (Time)<dd>Date and time when your modem received the facsimile." \
		"<dt>Sender<dd>Station id of the sender where the facsimile came from." \
		"<dt>Pages<dd>Number of pages in the facsimile document." \
		"<dt>Size<dd>File size in bytes of the facsimile file." \
		"<dt>File<dd>Indicates if the facsimile file still exists on your local data directory." \
		"</dl>"));
}


void KMLOFax::commandCallback(int id_)
{
	switch (id_) {
		case ID_FILE_DOWNLOAD:
			slotFileDownload();
			break;
		case ID_FILE_VIEW:
			slotFileView();
			break;
		case ID_FILE_PRINT:
			slotFilePrint();
			break;
		case ID_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_FILE_MAIL:
			slotFileMail();
			break;
		case ID_FILE_DELETE:
			slotFileDelete();
			break;
		case ID_FILE_SENDER_ALIAS:
			slotFileSenderAlias();
			break;
		case ID_FILE_PURGE_DIRECTORY:
			slotFilePurgeDir();
			break;
		case ID_FILE_REBUILD_DIRECTORY:
			slotFileRebuildDir();
			break;
		case ID_FILE_QUIT:
			slotFileQuit();
			break;
		case ID_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_SETTINGS_SAVEOPTIONS:
			saveOptions();
			break;
		case ID_SETTINGS_CONFIGURE:
			slotSettingsConfigure();
			break;
		case ID_SETTINGS_SENDER_ALIASES:
			slotSettingsSenderAliases();
			break;
		default:
			break;
	}
}

		
void KMLOFax::slotFileDownload()
{
	if (downloaddlg->startDownload()) {
		enableCommand(ID_FILE_DOWNLOAD, false);
		slotStatusHelpMsg(i18n("Looking for new facsimiles..."));
	}
}


void KMLOFax::slotFileView()
{
	if (!view->selectedFaxName().isEmpty()) {
		preview->showFax(view->selectedFaxName());
		view->setSelectedFaxNew(false);
	}
}


void KMLOFax::slotFilePrint()
{
	if (!view->selectedFaxName().isEmpty()) {
		PrintDlg printdlg(this, "printdlg");
		printdlg.printFax(view->selectedFaxName());
		if (printdlg.exec())
			view->setSelectedFaxNew(false);
	}
}


void KMLOFax::slotFileExport()
{
	if (!view->selectedFaxName().isEmpty()) {
		ExportDlg exportdlg(this, "exportdlg");
		exportdlg.exportFax(view->selectedFaxName());
		exportdlg.exec();
	}
}


void KMLOFax::slotFileMail()
{
	if (!view->selectedFaxName().isEmpty()) {
		MailDlg maildlg(this, "maildlg");
		maildlg.mailFax(view->selectedFaxName());
		maildlg.exec();
	}
}


void KMLOFax::slotFileDelete()
{
	if (!view->selectedFaxName().isEmpty()) {
		if (view->selectedFaxNew()) {
			KMessageBox::sorry(this, i18n("Cannot delete new facsimile file.\nPlease view or print facsimile first."), i18n("File Error"));
			return;
		}

		if (KMessageBox::questionYesNo(this, i18n("Really delete file\n\"%1\"?").arg(view->selectedFaxName()), i18n("Question"), i18n("&Delete"), i18n("&Cancel")) == KMessageBox::Yes) {
			QFile::remove(expandPath(view->selectedFaxName()));
			view->setSelectedFaxFile(false);
			hasSelectionSlot(true);
		}
	}
}


void KMLOFax::slotFileSenderAlias()
{
	if (!view->selectedFaxName().isEmpty()) {
		senderaliases->setAlias(view->selectedFaxSender());
	}
}


void KMLOFax::slotFilePurgeDir()
{
	if (KMessageBox::questionYesNo(this, i18n("Really remove entries with deleted files?"), i18n("Question")) == KMessageBox::Yes)
		view->purgeDir();
}


void KMLOFax::slotFileRebuildDir()
{
	if (KMessageBox::questionYesNo(this, i18n("Really discard current directory and scan for existing facsimile files?"), i18n("Question")) == KMessageBox::Yes) {
		slotStatusHelpMsg(i18n("%1 File(s) found.").arg(view->rebuildDir()));
	}
}


void KMLOFax::slotFileQuit()
{
	close();
}


void KMLOFax::slotSettingsShowToolbar()
{
	bool toolBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_TOOLBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_TOOLBAR, toolBarOn);
	if (toolBarOn)
		toolBar()->show();
	else
		toolBar()->hide();
}


void KMLOFax::slotSettingsShowStatusbar()
{
	bool statusBarOn = !settingsMenu->isItemChecked(ID_SETTINGS_STATUSBAR);

	settingsMenu->setItemChecked(ID_SETTINGS_STATUSBAR, statusBarOn);
	if (statusBarOn)
		statusBar()->show();
	else
		statusBar()->hide();
}


void KMLOFax::slotSettingsConfigure()
{
	if (configure->isVisible())
		configure->raise();
	else
		configure->show();
}


void KMLOFax::slotSettingsSenderAliases()
{
	if (senderaliases->isVisible())
		senderaliases->raise();
	else
		senderaliases->show();
}


void KMLOFax::slotStatusMsg(const QString& text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOFax::slotStatusHelpMsg(const QString& text)
{
	statusBar()->message(text, 2000);
}



void KMLOFax::slotNewConfig()
{
	bool tooltips;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	tooltips = config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS);

	configure->addToolTips(tooltips);
	senderaliases->addToolTips(tooltips);
}


void KMLOFax::hasSelectionSlot(bool on)
{
	view->gotSelection(on);
	
	enableCommand(ID_FILE_SENDER_ALIAS, on);
	
	if (on)
		on = view->selectedFaxFile();

	enableCommand(ID_FILE_VIEW, on);
	enableCommand(ID_FILE_PRINT, on);
	enableCommand(ID_FILE_EXPORT, on);
	enableCommand(ID_FILE_MAIL, on);
	enableCommand(ID_FILE_DELETE, on);
}


void KMLOFax::fetchFile(const QString& name)
{
	slotStatusHelpMsg(i18n("Received file %1.").arg(name));
	view->addFaxItem(name);	
}


void KMLOFax::fetchDone()
{
	enableCommand(ID_FILE_DOWNLOAD, true);
	view->saveDir();
}
