/*
    SPDX-FileCopyrightText: 1996 Bernd Johannes Wuebben <wuebben@kde.org>
    SPDX-FileCopyrightText: 1999 Preston Brown <pbrown@kde.org>
    SPDX-FileCopyrightText: 1999 Mario Weilguni <mweilguni@kde.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "kfontchooser.h"
#include "fonthelpers_p.h"

#include <QCheckBox>
#include <QDoubleSpinBox>
#include <QFontDatabase>
#include <QGroupBox>
#include <QGuiApplication>
#include <QLabel>
#include <QLayout>
#include <QListWidget>
#include <QLocale>
#include <QScrollBar>
#include <QSplitter>
#include <QTextEdit>

#include <algorithm>
#include <cmath>

// When message extraction needs to be avoided.
#define TR_NOX tr

static int minimumListWidth(const QListWidget *list)
{
    int w = 0;
    for (int i = 0; i < list->count(); i++) {
        int itemWidth = list->visualItemRect(list->item(i)).width();
        // ...and add a space on both sides for not too tight look.
        itemWidth += list->fontMetrics().horizontalAdvance(QLatin1Char(' ')) * 2;
        w = qMax(w, itemWidth);
    }
    if (w == 0) {
        w = 40;
    }
    w += list->frameWidth() * 2;
    w += list->verticalScrollBar()->sizeHint().width();
    return w;
}

static int minimumListHeight(const QListWidget *list, int numVisibleEntry)
{
    int w = list->count() > 0 ? list->visualItemRect(list->item(0)).height() : list->fontMetrics().lineSpacing();

    if (w < 0) {
        w = 10;
    }
    if (numVisibleEntry <= 0) {
        numVisibleEntry = 4;
    }
    return (w * numVisibleEntry + 2 * list->frameWidth());
}

static QString formatFontSize(qreal size)
{
    return QLocale::system().toString(size, 'f', (size == floor(size)) ? 0 : 1);
}

class KFontChooserPrivate
{
    Q_DECLARE_TR_FUNCTIONS(KFontChooser)

public:
    KFontChooserPrivate(KFontChooser *qq)
        : q(qq)
    {
        m_palette.setColor(QPalette::Active, QPalette::Text, Qt::black);
        m_palette.setColor(QPalette::Active, QPalette::Base, Qt::white);
    }

    void init(const KFontChooser::DisplayFlags &flags, const QStringList &fontList, int visibleListSize, Qt::CheckState *sizeIsRelativeState);
    void setFamilyBoxItems(const QStringList &fonts);
    void fillFamilyListBox(bool onlyFixedFonts = false);
    int nearestSizeRow(qreal val, bool customize);
    qreal fillSizeList(const QList<qreal> &sizes = QList<qreal>());
    qreal setupSizeListBox(const QString &family, const QString &style);

    void setupDisplay();
    QString styleIdentifier(const QFont &font);

    void slotFamilySelected(const QString &);
    void slotSizeSelected(const QString &);
    void slotStyleSelected(const QString &);
    void displaySample(const QFont &font);
    void slotSizeValue(double);

    KFontChooser *q;

    QPalette m_palette;

    QDoubleSpinBox *m_fontSizeSpinBox = nullptr;

    QTextEdit *m_sampleEdit = nullptr;

    QLabel *m_familyLabel = nullptr;
    QLabel *m_styleLabel = nullptr;
    QCheckBox *m_familyCheckbox = nullptr;
    QCheckBox *m_styleCheckbox = nullptr;
    QCheckBox *m_sizeCheckbox = nullptr;
    QLabel *m_sizeLabel = nullptr;
    QListWidget *m_familyListBox = nullptr;
    QListWidget *m_styleListBox = nullptr;
    QListWidget *m_sizeListBox = nullptr;
    QCheckBox *m_sizeIsRelativeCheckBox = nullptr;

    QCheckBox *m_onlyFixedCheckbox = nullptr;

    QFont m_selectedFont;

    QString m_selectedStyle;
    qreal m_selectedSize = -1.0;

    QString m_standardSizeAtCustom;
    int m_customSizeRow = -1;

    bool m_signalsAllowed = true;
    bool m_usingFixed = false;

    // Mappings of translated to Qt originated family and style strings.
    FontFamiliesMap m_qtFamilies;
    std::map<QString, QString> m_qtStyles;
    // Mapping of translated style strings to internal style identifiers.
    std::map<QString, QString> m_styleIDs;
};

KFontChooser::KFontChooser(QWidget *parent, const DisplayFlags &flags, const QStringList &fontList, int visibleListSize, Qt::CheckState *sizeIsRelativeState)
    : QWidget(parent)
    , d(new KFontChooserPrivate(this))
{
    d->init(flags, fontList, visibleListSize, sizeIsRelativeState);
}

KFontChooser::~KFontChooser() = default;

void KFontChooserPrivate::init(const KFontChooser::DisplayFlags &flags, const QStringList &fontList, int visibleListSize, Qt::CheckState *sizeIsRelativeState)
{
    m_usingFixed = flags & KFontChooser::FixedFontsOnly;

    // The main layout is divided horizontally into a top part with
    // the font attribute widgets (family, style, size) and a bottom
    // part with a preview of the selected font
    QVBoxLayout *mainLayout = new QVBoxLayout(q);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    auto *style = q->style();
    const int sizeListBoxGap = style->pixelMetric(QStyle::PM_LayoutVerticalSpacing) / 2;
    const int checkBoxGap = style->pixelMetric(QStyle::PM_LayoutHorizontalSpacing) / 2;

    // Build the grid of font attribute widgets for the upper part of mainLayout
    QWidget *page;
    QGridLayout *gridLayout;
    int row = flags & KFontChooser::DisplayFrame ? 1 : 0;

    if (flags & KFontChooser::DisplayFrame) {
        page = new QGroupBox(KFontChooser::tr("Requested Font", "@title:group"), q);
        mainLayout->addWidget(page);
        gridLayout = new QGridLayout(page);
    } else {
        page = new QWidget(q);
        mainLayout->addWidget(page);
        gridLayout = new QGridLayout(page);
        gridLayout->setContentsMargins(0, 0, 0, 0);
    }

    // set up the family list view and checkbox
    QHBoxLayout *familyLayout = new QHBoxLayout();
    familyLayout->addSpacing(checkBoxGap);
    if (flags & KFontChooser::ShowDifferences) {
        m_familyCheckbox = new QCheckBox(KFontChooser::tr("Font", "@option:check"), page);
        familyLayout->addWidget(m_familyCheckbox, 0, Qt::AlignLeft);
        m_familyLabel = nullptr;
    } else {
        m_familyCheckbox = nullptr;
        m_familyLabel = new QLabel(KFontChooser::tr("Font:", "@label"), page);
        familyLayout->addWidget(m_familyLabel, 1, Qt::AlignLeft);
    }
    gridLayout->addLayout(familyLayout, row, 0);

    ++row;

    m_familyListBox = new QListWidget(page);
    gridLayout->addWidget(m_familyListBox, row, 0);

    QObject::connect(m_familyListBox, &QListWidget::currentTextChanged, [this](const QString &family) {
        slotFamilySelected(family);
    });

    if (flags & KFontChooser::ShowDifferences) {
        m_familyListBox->setEnabled(false);
        QObject::connect(m_familyCheckbox, &QAbstractButton::toggled, m_familyListBox, &QWidget::setEnabled);
    }

    if (!fontList.isEmpty()) {
        setFamilyBoxItems(fontList);
    } else {
        fillFamilyListBox(flags & KFontChooser::FixedFontsOnly);
    }

    m_familyListBox->setMinimumWidth(minimumListWidth(m_familyListBox));
    m_familyListBox->setMinimumHeight(minimumListHeight(m_familyListBox, visibleListSize));

    // set up the sytle list view and checkbox
    row = flags & KFontChooser::DisplayFrame ? 1 : 0;
    QHBoxLayout *styleLayout = new QHBoxLayout();
    if (flags & KFontChooser::ShowDifferences) {
        m_styleCheckbox = new QCheckBox(KFontChooser::tr("Font style", "@option:check"), page);
        styleLayout->addWidget(m_styleCheckbox, 0, Qt::AlignLeft);
        m_styleLabel = nullptr;
    } else {
        m_styleCheckbox = nullptr;
        m_styleLabel = new QLabel(KFontChooser::tr("Font style:", "@label"), page);
        styleLayout->addWidget(m_styleLabel, 1, Qt::AlignLeft);
    }
    styleLayout->addSpacing(checkBoxGap);
    gridLayout->addLayout(styleLayout, row, 1);

    ++row;

    m_styleListBox = new QListWidget(page);
    gridLayout->addWidget(m_styleListBox, row, 1);

    // Populate usual styles, to determine minimum list width;
    // will be replaced later with correct styles.
    m_styleListBox->addItem(KFontChooser::tr("Normal", "@item font"));
    m_styleListBox->addItem(KFontChooser::tr("Italic", "@item font"));
    m_styleListBox->addItem(KFontChooser::tr("Oblique", "@item font"));
    m_styleListBox->addItem(KFontChooser::tr("Bold", "@item font"));
    m_styleListBox->addItem(KFontChooser::tr("Bold Italic", "@item font"));
    m_styleListBox->setMinimumWidth(minimumListWidth(m_styleListBox));
    m_styleListBox->setMinimumHeight(minimumListHeight(m_styleListBox, visibleListSize));

    QObject::connect(m_styleListBox, &QListWidget::currentTextChanged, [this](const QString &style) {
        slotStyleSelected(style);
    });

    if (flags & KFontChooser::ShowDifferences) {
        m_styleListBox->setEnabled(false);
        QObject::connect(m_styleCheckbox, &QAbstractButton::toggled, m_styleListBox, &QWidget::setEnabled);
    }

    // set the the size list view / font size spinbox; and the font size checkbox
    row = flags & KFontChooser::DisplayFrame ? 1 : 0;
    QHBoxLayout *sizeLayout = new QHBoxLayout();
    if (flags & KFontChooser::ShowDifferences) {
        m_sizeCheckbox = new QCheckBox(KFontChooser::tr("Size", "@option:check"), page);
        sizeLayout->addWidget(m_sizeCheckbox, 0, Qt::AlignLeft);
        m_sizeLabel = nullptr;
    } else {
        m_sizeCheckbox = nullptr;
        m_sizeLabel = new QLabel(KFontChooser::tr("Size:", "@label:listbox Font size"), page);
        sizeLayout->addWidget(m_sizeLabel, 1, Qt::AlignLeft);
    }
    sizeLayout->addSpacing(checkBoxGap);
    sizeLayout->addSpacing(checkBoxGap); // prevent label from eating border
    gridLayout->addLayout(sizeLayout, row, 2);

    ++row;

    m_sizeListBox = new QListWidget(page);
    m_fontSizeSpinBox = new QDoubleSpinBox(page);
    m_fontSizeSpinBox->setMinimum(4);
    m_fontSizeSpinBox->setMaximum(512);
    m_fontSizeSpinBox->setDecimals(1);
    m_fontSizeSpinBox->setSingleStep(1);

    if (sizeIsRelativeState) {
        QString sizeIsRelativeCBText = KFontChooser::tr("Relative", "@item font size");
        QString sizeIsRelativeCBToolTipText = KFontChooser::tr("Font size<br /><i>fixed</i> or <i>relative</i><br />to environment", "@info:tooltip");
        QString sizeIsRelativeCBWhatsThisText = KFontChooser::tr(
            "Here you can switch between fixed font size and font size "
            "to be calculated dynamically and adjusted to changing "
            "environment (e.g. widget dimensions, paper size).",
            "@info:whatsthis");
        m_sizeIsRelativeCheckBox = new QCheckBox(sizeIsRelativeCBText, page);
        m_sizeIsRelativeCheckBox->setTristate(flags & KFontChooser::ShowDifferences);
        QGridLayout *sizeLayout2 = new QGridLayout();
        sizeLayout2->setVerticalSpacing(sizeListBoxGap);
        gridLayout->addLayout(sizeLayout2, row, 2);
        sizeLayout2->setColumnStretch(1, 1); // to prevent text from eating the right border
        sizeLayout2->addWidget(m_fontSizeSpinBox, 0, 0, 1, 2);
        sizeLayout2->addWidget(m_sizeListBox, 1, 0, 1, 2);
        sizeLayout2->addWidget(m_sizeIsRelativeCheckBox, 2, 0, Qt::AlignLeft);
        m_sizeIsRelativeCheckBox->setWhatsThis(sizeIsRelativeCBWhatsThisText);
        m_sizeIsRelativeCheckBox->setToolTip(sizeIsRelativeCBToolTipText);
    } else {
        m_sizeIsRelativeCheckBox = nullptr;
        QGridLayout *sizeLayout2 = new QGridLayout();
        sizeLayout2->setVerticalSpacing(sizeListBoxGap);
        gridLayout->addLayout(sizeLayout2, row, 2);
        sizeLayout2->addWidget(m_fontSizeSpinBox, 0, 0);
        sizeLayout2->addWidget(m_sizeListBox, 1, 0);
    }

    // Populate with usual sizes, to determine minimum list width;
    // will be replaced later with correct sizes.
    fillSizeList();
    m_sizeListBox->setMinimumWidth(minimumListWidth(m_sizeListBox) + m_sizeListBox->fontMetrics().maxWidth());
    m_sizeListBox->setMinimumHeight(minimumListHeight(m_sizeListBox, visibleListSize));

    QObject::connect(m_fontSizeSpinBox, QOverload<double>::of(&QDoubleSpinBox::valueChanged), [this](const double size) {
        slotSizeValue(size);
    });

    QObject::connect(m_sizeListBox, &QListWidget::currentTextChanged, [this](const QString &size) {
        slotSizeSelected(size);
    });

    if (flags & KFontChooser::ShowDifferences) {
        m_sizeListBox->setEnabled(false);
        m_fontSizeSpinBox->setEnabled(false);
        QObject::connect(m_sizeCheckbox, &QAbstractButton::toggled, m_sizeListBox, &QWidget::setEnabled);
        QObject::connect(m_sizeCheckbox, &QAbstractButton::toggled, m_fontSizeSpinBox, &QWidget::setEnabled);
    }

    // Add the font preview into the lower part of mainLayout
    m_sampleEdit = new QTextEdit(page);
    m_sampleEdit->setAcceptRichText(false);
    QFont tmpFont(q->font().family(), 64, QFont::Black);
    m_sampleEdit->setFont(tmpFont);
    m_sampleEdit->setMinimumHeight(m_sampleEdit->fontMetrics().lineSpacing());
    // tr: A classical test phrase, with all letters of the English alphabet.
    // Replace it with a sample text in your language, such that it is
    // representative of language's writing system.
    // If you wish, you can input several lines of text separated by \n.
    q->setSampleText(KFontChooser::tr("The Quick Brown Fox Jumps Over The Lazy Dog"));
    m_sampleEdit->setTextCursor(QTextCursor(m_sampleEdit->document()));
    QString m_sampleEditWhatsThisText = KFontChooser::tr(
        "This sample text illustrates the current settings. "
        "You may edit it to test special characters.",
        "@info:whatsthis");
    m_sampleEdit->setWhatsThis(m_sampleEditWhatsThisText);

    QObject::connect(q, &KFontChooser::fontSelected, q, [this](const QFont &font) {
        displaySample(font);
    });

    mainLayout->addWidget(m_sampleEdit);

    // If the calling app sets FixedFontsOnly, respect its decision
    if (!m_usingFixed) {
        // Add a checkbox to toggle showing only monospace/fixed-width fonts
        m_onlyFixedCheckbox = new QCheckBox(KFontChooser::tr("Show only monospaced fonts", "@option:check"));
        m_onlyFixedCheckbox->setChecked(m_usingFixed);

        QObject::connect(m_onlyFixedCheckbox, &QAbstractButton::toggled, q, [this](const bool state) {
            q->setFont(m_selectedFont, state);
        });

        if (flags & KFontChooser::ShowDifferences) { // In this mode follow the state of the m_familyCheckbox
            m_onlyFixedCheckbox->setEnabled(false);
            QObject::connect(m_familyCheckbox, &QAbstractButton::toggled, m_onlyFixedCheckbox, &QWidget::setEnabled);
        }
        mainLayout->addWidget(m_onlyFixedCheckbox);
    }
    // Finished setting up the chooser layout

    // lets initialize the display if possible
    if (m_usingFixed) {
        q->setFont(QFontDatabase::systemFont(QFontDatabase::FixedFont), true);
    } else {
        q->setFont(QGuiApplication::font(), false);
    }

    // check or uncheck or gray out the "relative" checkbox
    if (sizeIsRelativeState && m_sizeIsRelativeCheckBox) {
        q->setSizeIsRelative(*sizeIsRelativeState);
    }

    // Set focus to the size list as this is the most commonly changed property
    m_sizeListBox->setFocus();
}

void KFontChooser::setColor(const QColor &col)
{
    d->m_palette.setColor(QPalette::Active, QPalette::Text, col);
    QPalette pal = d->m_sampleEdit->palette();
    pal.setColor(QPalette::Active, QPalette::Text, col);
    d->m_sampleEdit->setPalette(pal);
    QTextCursor cursor = d->m_sampleEdit->textCursor();
    d->m_sampleEdit->selectAll();
    d->m_sampleEdit->setTextColor(col);
    d->m_sampleEdit->setTextCursor(cursor);
}

QColor KFontChooser::color() const
{
    return d->m_palette.color(QPalette::Active, QPalette::Text);
}

void KFontChooser::setBackgroundColor(const QColor &col)
{
    d->m_palette.setColor(QPalette::Active, QPalette::Base, col);
    QPalette pal = d->m_sampleEdit->palette();
    pal.setColor(QPalette::Active, QPalette::Base, col);
    d->m_sampleEdit->setPalette(pal);
}

QColor KFontChooser::backgroundColor() const
{
    return d->m_palette.color(QPalette::Active, QPalette::Base);
}

void KFontChooser::setSizeIsRelative(Qt::CheckState relative)
{
    // check or uncheck or gray out the "relative" checkbox
    if (d->m_sizeIsRelativeCheckBox) {
        if (Qt::PartiallyChecked == relative) {
            d->m_sizeIsRelativeCheckBox->setCheckState(Qt::PartiallyChecked);
        } else {
            d->m_sizeIsRelativeCheckBox->setCheckState((Qt::Checked == relative) ? Qt::Checked : Qt::Unchecked);
        }
    }
}

Qt::CheckState KFontChooser::sizeIsRelative() const
{
    return d->m_sizeIsRelativeCheckBox ? d->m_sizeIsRelativeCheckBox->checkState() : Qt::PartiallyChecked;
}

QString KFontChooser::sampleText() const
{
    return d->m_sampleEdit->toPlainText();
}

void KFontChooser::setSampleText(const QString &text)
{
    d->m_sampleEdit->setPlainText(text);
}

void KFontChooser::setSampleBoxVisible(bool visible)
{
    d->m_sampleEdit->setVisible(visible);
}

QSize KFontChooser::sizeHint(void) const
{
    return minimumSizeHint();
}

void KFontChooser::enableColumn(int column, bool state)
{
    if (column & FamilyList) {
        d->m_familyListBox->setEnabled(state);
    }
    if (column & StyleList) {
        d->m_styleListBox->setEnabled(state);
    }
    if (column & SizeList) {
        d->m_sizeListBox->setEnabled(state);
        d->m_fontSizeSpinBox->setEnabled(state);
    }
}

void KFontChooser::setFont(const QFont &aFont, bool onlyFixed)
{
    d->m_selectedFont = aFont;
    d->m_selectedSize = aFont.pointSizeF();
    if (d->m_selectedSize == -1) {
        d->m_selectedSize = QFontInfo(aFont).pointSizeF();
    }

    if (onlyFixed != d->m_usingFixed) {
        d->m_usingFixed = onlyFixed;
        d->fillFamilyListBox(d->m_usingFixed);
    }
    d->setupDisplay();
}

KFontChooser::FontDiffFlags KFontChooser::fontDiffFlags() const
{
    FontDiffFlags diffFlags = NoFontDiffFlags;

    if (d->m_familyCheckbox && d->m_familyCheckbox->isChecked()) {
        diffFlags |= FontDiffFamily;
    }

    if (d->m_styleCheckbox && d->m_styleCheckbox->isChecked()) {
        diffFlags |= FontDiffStyle;
    }

    if (d->m_sizeCheckbox && d->m_sizeCheckbox->isChecked()) {
        diffFlags |= FontDiffSize;
    }

    return diffFlags;
}

QFont KFontChooser::font() const
{
    return d->m_selectedFont;
}

static bool isDefaultFontStyleName(const QString &style)
{
    /* clang-format off */
    // Ordered by commonness, i.e. "Regular" is the most common
    return style == QLatin1String("Regular")
        || style == QLatin1String("Normal")
        || style == QLatin1String("Book")
        || style == QLatin1String("Roman");
    /* clang-format on */
}

void KFontChooserPrivate::slotFamilySelected(const QString &family)
{
    if (!m_signalsAllowed) {
        return;
    }
    m_signalsAllowed = false;

    QString currentFamily;
    if (family.isEmpty()) {
        Q_ASSERT(m_familyListBox->currentItem());
        if (m_familyListBox->currentItem()) {
            currentFamily = m_qtFamilies[m_familyListBox->currentItem()->text()];
        }
    } else {
        currentFamily = m_qtFamilies[family];
    }

    // Get the list of styles available in this family.
    QFontDatabase dbase;
    QStringList styles = dbase.styles(currentFamily);
    if (styles.isEmpty()) {
        // Avoid extraction, it is in kdeqt.po
        styles.append(TR_NOX("Normal", "QFontDatabase"));
    }

    // Always prepend Regular, Normal, Book or Roman, this way if "m_selectedStyle"
    // in the code below is empty, selecting index 0 should work better
    std::sort(styles.begin(), styles.end(), [](const QString &a, const QString &b) {
        if (isDefaultFontStyleName(a)) {
            return true;
        } else if (isDefaultFontStyleName(b)) {
            return false;
        }
        return false;
    });

    // Filter style strings and add to the listbox.
    QString pureFamily;
    splitFontString(family, &pureFamily);
    QStringList filteredStyles;
    m_qtStyles.clear();
    m_styleIDs.clear();

    const QStringList origStyles = styles;
    for (const QString &style : origStyles) {
        // Sometimes the font database will report an invalid style,
        // that falls back back to another when set.
        // Remove such styles, by checking set/get round-trip.
        QFont testFont = dbase.font(currentFamily, style, 10);
        if (dbase.styleString(testFont) != style) {
            styles.removeAll(style);
            continue;
        }

        QString fstyle = tr("%1", "@item Font style").arg(style);
        if (!filteredStyles.contains(fstyle)) {
            filteredStyles.append(fstyle);
            m_qtStyles.insert({fstyle, style});
            m_styleIDs.insert({fstyle, styleIdentifier(testFont)});
        }
    }
    m_styleListBox->clear();
    m_styleListBox->addItems(filteredStyles);

    // Try to set the current style in the listbox to that previous.
    int listPos = filteredStyles.indexOf(m_selectedStyle.isEmpty() ? TR_NOX("Normal", "QFontDatabase") : m_selectedStyle);
    if (listPos < 0) {
        // Make extra effort to have Italic selected when Oblique was chosen,
        // and vice versa, as that is what the user would probably want.
        QString styleIt = tr("Italic", "@item font");
        QString styleOb = tr("Oblique", "@item font");
        for (int i = 0; i < 2; ++i) {
            int pos = m_selectedStyle.indexOf(styleIt);
            if (pos >= 0) {
                QString style = m_selectedStyle;
                style.replace(pos, styleIt.length(), styleOb);
                listPos = filteredStyles.indexOf(style);
                if (listPos >= 0) {
                    break;
                }
            }
            qSwap(styleIt, styleOb);
        }
    }
    m_styleListBox->setCurrentRow(listPos >= 0 ? listPos : 0);
    const QString currentStyle = m_qtStyles[m_styleListBox->currentItem()->text()];

    // Recompute the size listbox for this family/style.
    qreal currentSize = setupSizeListBox(currentFamily, currentStyle);
    m_fontSizeSpinBox->setValue(currentSize);

    m_selectedFont = dbase.font(currentFamily, currentStyle, static_cast<int>(currentSize));
    if (dbase.isSmoothlyScalable(currentFamily, currentStyle) && m_selectedFont.pointSize() == floor(currentSize)) {
        m_selectedFont.setPointSizeF(currentSize);
    }
    Q_EMIT q->fontSelected(m_selectedFont);

    m_signalsAllowed = true;
}

void KFontChooserPrivate::slotStyleSelected(const QString &style)
{
    if (!m_signalsAllowed) {
        return;
    }
    m_signalsAllowed = false;

    QFontDatabase dbase;
    const QString currentFamily = m_qtFamilies[m_familyListBox->currentItem()->text()];
    const QString currentStyle = !style.isEmpty() ? m_qtStyles[style] : m_qtStyles[m_styleListBox->currentItem()->text()];

    // Recompute the size listbox for this family/style.
    qreal currentSize = setupSizeListBox(currentFamily, currentStyle);
    m_fontSizeSpinBox->setValue(currentSize);

    m_selectedFont = dbase.font(currentFamily, currentStyle, static_cast<int>(currentSize));
    if (dbase.isSmoothlyScalable(currentFamily, currentStyle) && m_selectedFont.pointSize() == floor(currentSize)) {
        m_selectedFont.setPointSizeF(currentSize);
    }
    Q_EMIT q->fontSelected(m_selectedFont);

    if (!style.isEmpty()) {
        m_selectedStyle = currentStyle;
    }

    m_signalsAllowed = true;
}

void KFontChooserPrivate::slotSizeSelected(const QString &size)
{
    if (!m_signalsAllowed) {
        return;
    }

    m_signalsAllowed = false;

    qreal currentSize = 0.0;
    if (size.isEmpty()) {
        currentSize = QLocale::system().toDouble(m_sizeListBox->currentItem()->text());
    } else {
        currentSize = QLocale::system().toDouble(size);
    }

    // Reset the customized size slot in the list if not needed.
    if (m_customSizeRow >= 0 && m_selectedFont.pointSizeF() != currentSize) {
        m_sizeListBox->item(m_customSizeRow)->setText(m_standardSizeAtCustom);
        m_customSizeRow = -1;
    }

    m_fontSizeSpinBox->setValue(currentSize);
    m_selectedFont.setPointSizeF(currentSize);
    Q_EMIT q->fontSelected(m_selectedFont);

    if (!size.isEmpty()) {
        m_selectedSize = currentSize;
    }

    m_signalsAllowed = true;
}

void KFontChooserPrivate::slotSizeValue(double dval)
{
    if (!m_signalsAllowed) {
        return;
    }
    m_signalsAllowed = false;

    // We compare with qreal, so convert for platforms where qreal != double.
    qreal val = qreal(dval);

    // Reset current size slot in list if it was customized.
    if (m_customSizeRow >= 0 && m_sizeListBox->currentRow() == m_customSizeRow) {
        m_sizeListBox->item(m_customSizeRow)->setText(m_standardSizeAtCustom);
        m_customSizeRow = -1;
    }

    bool canCustomize = true;

    QFontDatabase dbase;
    const QString family = m_qtFamilies[m_familyListBox->currentItem()->text()];
    const QString style = m_qtStyles[m_styleListBox->currentItem()->text()];

    // For Qt-bad-sizes workaround: skip this block unconditionally
    if (!dbase.isSmoothlyScalable(family, style)) {
        // Bitmap font, allow only discrete sizes.
        // Determine the nearest in the direction of change.
        canCustomize = false;
        int nrows = m_sizeListBox->count();
        int row = m_sizeListBox->currentRow();
        int nrow;
        if (val - m_selectedFont.pointSizeF() > 0) {
            for (nrow = row + 1; nrow < nrows; ++nrow)
                if (QLocale::system().toDouble(m_sizeListBox->item(nrow)->text()) >= val) {
                    break;
                }
        } else {
            for (nrow = row - 1; nrow >= 0; --nrow)
                if (QLocale::system().toDouble(m_sizeListBox->item(nrow)->text()) <= val) {
                    break;
                }
        }
        // Make sure the new row is not out of bounds.
        nrow = nrow < 0 ? 0 : nrow >= nrows ? nrows - 1 : nrow;
        // Get the size from the new row and set the spinbox to that size.
        val = QLocale::system().toDouble(m_sizeListBox->item(nrow)->text());
        m_fontSizeSpinBox->setValue(val);
    }

    // Set the current size in the size listbox.
    int row = nearestSizeRow(val, canCustomize);
    m_sizeListBox->setCurrentRow(row);

    m_selectedSize = val;
    m_selectedFont.setPointSizeF(val);
    Q_EMIT q->fontSelected(m_selectedFont);

    m_signalsAllowed = true;
}

void KFontChooserPrivate::displaySample(const QFont &font)
{
    m_sampleEdit->setFont(font);
}

int KFontChooserPrivate::nearestSizeRow(qreal val, bool customize)
{
    qreal diff = 1000;
    int row = 0;
    for (int r = 0; r < m_sizeListBox->count(); ++r) {
        qreal cval = QLocale::system().toDouble(m_sizeListBox->item(r)->text());
        if (qAbs(cval - val) < diff) {
            diff = qAbs(cval - val);
            row = r;
        }
    }
    // For Qt-bad-sizes workaround: ignore value of customize, use true
    if (customize && diff > 0) {
        m_customSizeRow = row;
        m_standardSizeAtCustom = m_sizeListBox->item(row)->text();
        m_sizeListBox->item(row)->setText(formatFontSize(val));
    }
    return row;
}

qreal KFontChooserPrivate::fillSizeList(const QList<qreal> &sizes_)
{
    if (!m_sizeListBox) {
        return 0; // assertion.
    }

    QList<qreal> sizes = sizes_;
    bool canCustomize = false;
    if (sizes.isEmpty()) {
        static const int c[] = {4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 22, 24, 26, 28, 32, 48, 64, 72, 80, 96, 128, 0};
        for (int i = 0; c[i]; ++i) {
            sizes.append(c[i]);
        }
        // Since sizes were not supplied, this is a vector font,
        // and size slot customization is allowed.
        canCustomize = true;
    }

    // Insert sizes into the listbox.
    m_sizeListBox->clear();
    std::sort(sizes.begin(), sizes.end());
    for (qreal size : qAsConst(sizes)) {
        m_sizeListBox->addItem(formatFontSize(size));
    }

    // Return the nearest to selected size.
    // If the font is vector, the nearest size is always same as selected,
    // thus size slot customization is allowed.
    // If the font is bitmap, the nearest size need not be same as selected,
    // thus size slot customization is not allowed.
    m_customSizeRow = -1;
    int row = nearestSizeRow(m_selectedSize, canCustomize);
    return QLocale::system().toDouble(m_sizeListBox->item(row)->text());
}

qreal KFontChooserPrivate::setupSizeListBox(const QString &family, const QString &style)
{
    QFontDatabase dbase;
    QList<qreal> sizes;
    const bool smoothlyScalable = dbase.isSmoothlyScalable(family, style);
    if (!smoothlyScalable) {
        const QList<int> smoothSizes = dbase.smoothSizes(family, style);
        for (int size : smoothSizes) {
            sizes.append(size);
        }
    }

    // Fill the listbox (uses default list of sizes if the given is empty).
    // Collect the best fitting size to selected size, to use if not smooth.
    qreal bestFitSize = fillSizeList(sizes);

    // Set the best fit size as current in the listbox if available.
    const QList<QListWidgetItem *> selectedSizeList = m_sizeListBox->findItems(formatFontSize(bestFitSize), Qt::MatchExactly);
    if (!selectedSizeList.isEmpty()) {
        m_sizeListBox->setCurrentItem(selectedSizeList.first());
    }

    return bestFitSize;
}

void KFontChooserPrivate::setupDisplay()
{
    QFontDatabase dbase;
    qreal size = m_selectedFont.pointSizeF();
    if (size == -1) {
        size = QFontInfo(m_selectedFont).pointSizeF();
    }

    int numEntries, i;

    QString family = m_selectedFont.family().toLower();
    // Direct family match.
    numEntries = m_familyListBox->count();
    for (i = 0; i < numEntries; ++i) {
        if (family == m_qtFamilies[m_familyListBox->item(i)->text()].toLower()) {
            m_familyListBox->setCurrentRow(i);
            break;
        }
    }

    // 1st family fallback.
    if (i == numEntries) {
        const int bracketPos = family.indexOf(QLatin1Char('['));
        if (bracketPos != -1) {
            family = family.leftRef(bracketPos).trimmed().toString();
            for (i = 0; i < numEntries; ++i) {
                if (family == m_qtFamilies[m_familyListBox->item(i)->text()].toLower()) {
                    m_familyListBox->setCurrentRow(i);
                    break;
                }
            }
        }
    }

    // 2nd family fallback.
    if (i == numEntries) {
        QString fallback = family + QLatin1String(" [");
        for (i = 0; i < numEntries; ++i) {
            if (m_qtFamilies[m_familyListBox->item(i)->text()].toLower().startsWith(fallback)) {
                m_familyListBox->setCurrentRow(i);
                break;
            }
        }
    }

    // 3rd family fallback.
    if (i == numEntries) {
        for (i = 0; i < numEntries; ++i) {
            if (m_qtFamilies[m_familyListBox->item(i)->text()].toLower().startsWith(family)) {
                m_familyListBox->setCurrentRow(i);
                break;
            }
        }
    }

    // Family fallback in case nothing matched. Otherwise, diff doesn't work
    if (i == numEntries) {
        m_familyListBox->setCurrentRow(0);
    }

    // By setting the current item in the family box, the available
    // styles and sizes for that family have been collected.
    // Try now to set the current items in the style and size boxes.

    // Set current style in the listbox.
    const QString styleID = styleIdentifier(m_selectedFont);
    numEntries = m_styleListBox->count();
    for (i = 0; i < numEntries; ++i) {
        if (styleID == m_styleIDs[m_styleListBox->item(i)->text()]) {
            m_styleListBox->setCurrentRow(i);
            break;
        }
    }
    if (i == numEntries) {
        // Style not found, fallback.
        m_styleListBox->setCurrentRow(0);
    }

    // Set current size in the listbox.
    // If smoothly scalable, allow customizing one of the standard size slots,
    // otherwise just select the nearest available size.
    const QString currentFamily = m_qtFamilies[m_familyListBox->currentItem()->text()];
    const QString currentStyle = m_qtStyles[m_styleListBox->currentItem()->text()];
    const bool canCustomize = dbase.isSmoothlyScalable(currentFamily, currentStyle);
    m_sizeListBox->setCurrentRow(nearestSizeRow(size, canCustomize));

    // Set current size in the spinbox.
    m_fontSizeSpinBox->setValue(QLocale::system().toDouble(m_sizeListBox->currentItem()->text()));
}

void KFontChooser::getFontList(QStringList &list, uint fontListCriteria)
{
    QFontDatabase dbase;
    QStringList lstSys(dbase.families());

    // if we have criteria; then check fonts before adding
    if (fontListCriteria) {
        QStringList lstFonts;
        for (const QString &family : qAsConst(lstSys)) {
            if ((fontListCriteria & FixedWidthFonts) > 0 && !dbase.isFixedPitch(family)) {
                continue;
            }
            if (((fontListCriteria & (SmoothScalableFonts | ScalableFonts)) == ScalableFonts) && !dbase.isBitmapScalable(family)) {
                continue;
            }
            if ((fontListCriteria & SmoothScalableFonts) > 0 && !dbase.isSmoothlyScalable(family)) {
                continue;
            }
            lstFonts.append(family);
        }

        if ((fontListCriteria & FixedWidthFonts) > 0) {
            // Fallback.. if there are no fixed fonts found, it's probably a
            // bug in the font server or Qt.  In this case, just use 'fixed'
            if (lstFonts.isEmpty()) {
                lstFonts.append(QStringLiteral("fixed"));
            }
        }

        lstSys = lstFonts;
    }

    lstSys.sort();

    list = lstSys;
}

void KFontChooserPrivate::setFamilyBoxItems(const QStringList &fonts)
{
    m_signalsAllowed = false;

    m_familyListBox->clear();
    m_qtFamilies = translateFontNameList(fonts);

    // Generic font names
    const QStringList genericTranslatedNames{
        translateFontName(QStringLiteral("Sans Serif")),
        translateFontName(QStringLiteral("Serif")),
        translateFontName(QStringLiteral("Monospace")),
    };

    QStringList list;

    // Add generic family names to the top of the list
    for (const QString &s : genericTranslatedNames) {
        auto nIt = m_qtFamilies.find(s);
        if (nIt != m_qtFamilies.cend()) {
            list.push_back(s);
        }
    }

    for (auto it = m_qtFamilies.cbegin(); it != m_qtFamilies.cend(); ++it) {
        const QString &name = it->first;
        if (genericTranslatedNames.contains(name)) {
            continue;
        }

        list.push_back(name);
    }

    m_familyListBox->addItems(list);

    m_signalsAllowed = true;
}

void KFontChooserPrivate::fillFamilyListBox(bool onlyFixedFonts)
{
    QStringList fontList;
    KFontChooser::getFontList(fontList, onlyFixedFonts ? KFontChooser::FixedWidthFonts : 0);
    setFamilyBoxItems(fontList);
}

// Human-readable style identifiers returned by QFontDatabase::styleString()
// do not always survive round trip of QFont serialization/deserialization,
// causing wrong style in the style box to be highlighted when
// the chooser dialog is opened. This will cause the style to be changed
// when the dialog is closed and the user did not touch the style box.
// Hence, construct custom style identifiers sufficient for the purpose.
QString KFontChooserPrivate::styleIdentifier(const QFont &font)
{
    const int weight = font.weight();
    QString styleName = font.styleName();
    // If the styleName property is empty and the weight is QFont::Normal, that
    // could mean it's a "Regular"-like style with the styleName part stripped
    // so that subsequent calls to setBold(true) can work properly (i.e. selecting
    // the "Bold" style provided by the font itself) without resorting to font
    // "emboldening" which looks ugly.
    // See also KConfigGroupGui::writeEntryGui().
    if (styleName.isEmpty() && weight == QFont::Normal) {
        QFontDatabase fdb;
        const QStringList styles = fdb.styles(font.family());
        for (const QString &style : styles) {
            if (isDefaultFontStyleName(style)) {
                styleName = style;
                break;
            } else {
                // nothing more we can do
            }
        }
    }

    const QChar comma(QLatin1Char(','));
    return QString::number(weight) + comma //
        + QString::number((int)font.style()) + comma //
        + QString::number(font.stretch()) + comma //
        + styleName;
}

#include "moc_kfontchooser.cpp"
