/***************************************************************************
                          utils.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfile.h>
#include <qwidget.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kio/netaccess.h>
#include <kurl.h>
#include "utils.h"

bool Utils::minMaxFile(double* xmin, double* xmax, const double* x, int n)
{
  *xmin = 1.0e300;
  *xmax = -*xmin;
  for (int i = 0; i < n; i++)
    if (valid(x[i])) {
      *xmin = QMIN(*xmin, x[i]);
      *xmax = QMAX(*xmax, x[i]);
    }
  return (*xmax == *xmin);
}

void Utils::autoSc(double* xmi, double* xma, double* tic, int* mtic, int* ndig,
                   double xmin, double xmax, double fx, double c, bool _log)
{
  if (_log) {
    xmin = QMAX(xmin, 1.0e-300);
    xmax = QMAX(xmax, xmin);
    if (xmin == xmax) {
      xmin = QMAX(0.999 * xmin, 1.0e-300);
      xmax = 1.001 * xmin;
    }
    *ndig = -1;
    double d = c * (log10(xmax) - log10(xmin));
    if (d > 1.0) {
      *mtic = 1;
      double f = pow(10.0, (-double(Utils::int9(log10(d + 1.0e-301)))));
      d *= f;
      if (d > 7.071)
        *tic = 10.0 / f;
      else
        if (d > 3.162)
          *tic = 5.0 / f;
        else
          *tic = ((d > 1.414) ? 2.0 : 1.0) / f;
      *tic = QMIN(*tic, 100.0);
      *xmi = pow(10.0, *tic * Utils::int9(log10(xmin) / *tic + 1.0e-4));
      *xma = pow(10.0, *tic * Utils::int9(log10(xmax) / *tic + 0.9999));
      *tic = pow(10.0, *tic);
    } else {
      *mtic = 10;
      if (d < 0.7)
        *tic = (d < 0.5) ? 2.0 : 3.0;
      else
        *tic = 10.0;
    }
    double xl = log10(xmin);
    double mant = xl - int(xl + 1.0e-6 * fabs(xl));
    if (fabs(mant) < 1.0e-9)
      mant = 0.0;
    if (mant < 0.0)
      mant += 1.0;
    int e = qRound(xl - mant);
    if (*tic == 2.0)
      if (mant < log10(1.999999))
        mant = 0.0;
      else
        if (mant < log10(4.999999))
          mant = log10(2.0);
        else
          mant = log10(5.0);
    else
      if (*tic == 3.0)
        if (mant < log10(2.999999))
          mant = 0.0;
        else
          mant = log10(3.0);
      else
        mant = 0.0;
    *xmi = pow(10.0, e + mant);
    xl = log10(xmax);
    mant = xl - int(xl + 1.0e-6 * fabs(xl));
    if (fabs(mant) < 1.0e-9)
      mant = 0.0;
    if (mant < 0.0)
      mant += 1.0;
    e = qRound(xl - mant);
    if (*tic == 2.0)
      if (mant > log10(5.000001))
        mant = 1.0;
      else
        if (mant > log10(2.000001))
          mant = log10(5.0);
        else {
          if (mant)
            mant = log10(2.0);
        }
    else
      if (*tic == 3.0) {
        if (mant > log10(3.000001))
          mant = 1.0;
        else
          if (mant)
            mant = log10(3.0);
      } else
        if (mant)
          mant = 1.0;
    *xma = pow(10.0, e + mant);
  } else {
    if (xmin == xmax) {
      double dx;
      if (xmin)
        dx = 1.0e-3 * fabs(xmin);
      else
        dx = 1.0e-3 / fx;
      xmin -= dx;
      xmax += dx;
    }
    double d = c * fx * (xmax - xmin);
    double f = pow(10.0, (-double(Utils::int9(log10(d + 1.0e-301)))));
    d *= f;
    if (d > 7.071)
      *mtic = 10;
    else
      if (d > 3.162)
        *mtic = 5;
      else
        *mtic = (d > 1.414) ? 2 : 1;
    *tic = *mtic / f;
    if ((*mtic == 1) || (*mtic == 10))
      *mtic = 2;
    if (!(*ndig = -Utils::int9(log10(*tic + 1.0e-301) + 1.0e-6)))
      *ndig = -1;
    *xmi = *tic * Utils::int9(fx * xmin / *tic + 1.0e-4);
    *xma = *tic * Utils::int9(fx * xmax / *tic + 0.9999);
  }
}

void Utils::expo(double a, int* ie, double* f)
{
  if ((*ie = int9(log10(1.01 * a + 1.0e-301))) < 0)
    *ie = *ie - 1;
  *ie = -3 * (*ie / 3);
  *f = pow(10.0, *ie);
}

int Utils::int9(double a)
{
  int i = int(a);
  if (a < 0.0)
    i--;
  return i;
}

QString Utils::relPath(const KURL& uPlo, const KURL& path, bool abs)
{
  if (!abs) {
    KURL u = path;
    if (!(uPlo.isLocalFile() ^ u.isLocalFile())) {
      if (!uPlo.isLocalFile()) {
        KURL t(uPlo);
        t.setPath("/");
        if (!t.isParentOf(u))
          return path.url();
      }
      QString dPlo = uPlo.directory(false);
      u.cleanPath();
      QString dFile = u.directory(false);
      int nMax = QMIN(dPlo.length(), dFile.length()) - 1;
      int n;
      for (n = 0; n < nMax; n++)
        if (dPlo[n] != dFile[n])
          break;
      if (n > 0)
        n = dFile.findRev('/', n) + 1;
      QString s = u.path().mid(n);
      int nUp = dPlo.contains('/') - dPlo.left(n).contains('/');
      for (int i = 0; i < nUp; i++)
        s.prepend("../");
      return s;
    }
  }
  return path.isLocalFile() ? path.path() : path.url();
}

void Utils::saveSize(QWidget* w, const QString& groupName)
{
  KConfig* config = KGlobal::config();
  config->setGroup(groupName);
  config->writeEntry("Width", w->width());
  config->writeEntry("Height", w->height());
}

void Utils::saveGeometry(QWidget* w, const QString& groupName)
{
  saveSize(w, groupName);
  KConfig* config = KGlobal::config();
  config->writeEntry("x", w->x());
  config->writeEntry("y", w->y());
}

void Utils::setSize(QWidget* w, const QString& groupName)
{
  KConfig* config = KGlobal::config();
  config->setGroup(groupName);
  w->resize(config->readNumEntry("Width", w->minimumWidth()),
            config->readNumEntry("Height", w->minimumHeight()));
}

void Utils::setGeometry(QWidget* w, const QString& groupName)
{
  setSize(w, groupName);
  KConfig* config = KGlobal::config();
  config->setGroup(groupName);
  w->move(config->readNumEntry("x"), config->readNumEntry("y", 20));
}

bool Utils::boundingBox(const KURL& url, QString& filename, double* x1,
                        double* y1, double* x2, double* y2)
{
  bool success = false;
  if (url.isLocalFile())
    filename = url.path();
  else
#if KDE_VERSION >= 0x030200
    if (!KIO::NetAccess::download(url, filename, 0))
#else
    if (!KIO::NetAccess::download(url, filename))
#endif
      return success;
  QFile f(filename);
  if (f.open(IO_ReadOnly)) {
    QString s;
    while (f.readLine(s, 200) >= 0) {
      int i = s.find("%%BoundingBox:");
      if (i >= 0) {
        if (sscanf(s.latin1(), "%*s %lf %lf %lf %lf",
                   x1, y1, x2, y2) == 4)
          if ((*x2 > *x1) && (*y2 > *y1))
            success = true;
        break;
      }
    }
    f.close();
  }
  if (!success && !url.isLocalFile())
    KIO::NetAccess::removeTempFile(filename);
  return success;
}

void Utils::translate(QString& line, int idec)
{
  if (idec)
    line.replace(',', '.');
  line.replace('D', 'E');
  line.replace('d', 'e');
}
