/***************************************************************************
                          kplview.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qcursor.h>
#include <qdragobject.h>
#include <qpaintdevicemetrics.h>
#include <qtimer.h>
#include <kapplication.h>
#include <klocale.h>
#include <knotifyclient.h>
#include "frameitem.h"
#include "funitem.h"
#include "itemdlg.h"
#include "kpl.h"
#include "kpldoc.h"
#include "kplview.h"
#include "splineitem.h"
#include "utils.h"

KplView::KplView(Kpl* _parent, double f) :
 KplWidget(_parent), app(_parent), doc(_parent->getDocument()), fZoom(f),
 hPos(0.0), vPos(1.0)
{
  fxn = fyn = 1.0;
  setFrameStyle(NoFrame);
  QScrollView::viewport()->setAcceptDrops(true);
  QScrollView::viewport()->setMouseTracking(true);
  eraseWidget();
}

KplView::~KplView()
{
}

void KplView::print(KPrinter* printer) const
{
  KplWidget::print(printer, doc->items());
}

void KplView::layout()
{
  QPaintDeviceMetrics pdm(this);
  hPos = horizontalScrollBar()->isVisible() ?
         double(horizontalScrollBar()->value()) /
         horizontalScrollBar()->maxValue() : 0.0;
  vPos = verticalScrollBar()->isVisible() ?
         double(verticalScrollBar()->value()) /
         verticalScrollBar()->maxValue() : 1.0;
  resizeContents(qRound(fZoom * doc->options()->xf * pdm.logicalDpiX() / 2.54),
                 qRound(fZoom * doc->options()->yf * pdm.logicalDpiY() / 2.54));
  QTimer::singleShot(0, this, SLOT(setScrollBars()));
}

void KplView::doPaint(bool paint)
{
  layout();
  if (paint)
    QScrollView::viewport()->setCursor(waitCursor);
  KplWidget::paintWidget(doc->items(), contentsWidth(), contentsHeight(),
                         fZoom, doc->options()->xf, doc->options()->yf, paint);
  repaintContents(0, 0, pm.width(), pm.height(), false);
  if (paint)
    QScrollView::viewport()->setCursor(crossCursor);
}

void KplView::slotScrollLeft()
{
  horizontalScrollBar()->subtractLine();
}

void KplView::slotScrollRight()
{
  horizontalScrollBar()->addLine();
}

void KplView::slotScrollUp()
{
  verticalScrollBar()->subtractLine();
}

void KplView::slotScrollDown()
{
  verticalScrollBar()->addLine();
}

void KplView::slotScrollPageUp()
{
  verticalScrollBar()->subtractPage();
}

void KplView::slotScrollPageDown()
{
  verticalScrollBar()->addPage();
}

void KplView::slotScrollPageLeft()
{
  horizontalScrollBar()->subtractPage();
}

void KplView::slotScrollPageRight()
{
  horizontalScrollBar()->addPage();
}

void KplView::slotScrollHomeLeft()
{
  horizontalScrollBar()->setValue(0);
}

void KplView::slotScrollHomeDown()
{
  verticalScrollBar()->setValue(verticalScrollBar()->maxValue());
}

void KplView::slotScrollEndRight()
{
  horizontalScrollBar()->setValue(horizontalScrollBar()->maxValue());
}

void KplView::slotScrollEndUp()
{
  verticalScrollBar()->setValue(0);
}

void KplView::paintWidget()
{
  doPaint(true);
  KNotifyClient::event(
#if KDE_VERSION >= 0x030101
                       winId(),
#endif
                       "PlotCompleted", i18n("Plot completed."));
}

void KplView::setScrollBars()
{
  horizontalScrollBar()->setValue(int(hPos *
                                      horizontalScrollBar()->maxValue()));
  verticalScrollBar()->setValue(int(vPos * verticalScrollBar()->maxValue()));
}

void KplView::contentsMousePressEvent(QMouseEvent* e)
{
  if (e->button() == MidButton)
    startPos = dragPos = e->pos();
  else {
    KplItem* item = 0;
    for (item = doc->items()->last(); item; item = doc->items()->prev())
      if (item->boundingRect().contains(e->pos()))
         break;
    if (e->button() == RightButton)
      if (item)
        app->itemDlg->showPopup(doc->items()->findRef(item), QCursor::pos());
      else
        emit rightButtonPressed();
    else
      if (item) {
        if (sel.contains(item))
          sel.remove(item);
        else
          sel.append(item);
        startPos = dragPos = e->pos();
        drawBoundingRect(item->boundingRect());
        emit mouseMoved(fromcm(item->xPos(), item->yPos()), false);
      }
  }
}

void KplView::contentsMouseReleaseEvent(QMouseEvent* e)
{
  if (e->button() == LeftButton) {
    if (sel.count() && (!(e->state() & ControlButton))) {
      QPoint d = e->pos() - startPos;
      double snap = 0.05;
      double dx = snap * qRound((d.x() / (scalx * factor)) / snap);
      double dy = snap * qRound((d.y() / (scaly * factor)) / snap);
      if (dx || dy) {
        for (KplItem* it1 = sel.first(); it1; it1 = sel.next()) {
          it1->setPos(it1->xPos() + dx, it1->yPos() + dy);
          KplItem::ItemTypes typ = it1->iType();
          if ((typ == KplItem::Frame) || (typ == KplItem::Array3D) ||
              (typ == KplItem::Function3D))
            if (it1 != doc->items()->getLast())
              for (KplItem* itm = doc->items()->at(doc->items()->find(it1) + 1);
                   itm; itm = doc->items()->next()) {
                typ = itm->iType();
                if ((typ == KplItem::Frame) ||
                    (typ == KplItem::Array3D) ||
                    (typ == KplItem::Function3D))
                  break;
                if (itm->iType() >= KplItem::Legend)
                  itm->setPos(itm->xPos() + dx, itm->yPos() + dy);
              }
        }
        doc->setModified();
        doc->backupItems();
      } else {
        d = dragPos - startPos;
        for (KplItem* it1 = sel.first(); it1; it1 = sel.next()) {
          QRect r = it1->boundingRect();
          r.moveBy(d.x(), d.y());
          drawBoundingRect(r);
        }
      }
      sel.clear();
    }
  } else
    if (e->button() == MidButton) {
      QRect r = sortedRect(startPos, dragPos);
      if (r.isValid()) {
        FrameItem* it = 0;
        KplItem* item;
        for (item = doc->items()->last(); item;
             item = doc->items()->prev()) {
          if (item->iType() == KplItem::Frame &&
              (item->boundingRect().contains(startPos) ||
               item->boundingRect().contains(dragPos))) {
            it = (FrameItem*) item;
            break;
          }
        }
        if (it) {
          int mnx = minx0 + qRound(scalx * factor * it->xPos());
          int mny = miny0 + qRound(scaly * factor * it->yPos());
          double xmi = it->logx ? log10(it->xmi) : it->xmi;
          double ymi = it->logy ? log10(it->ymi) : it->ymi;
          double xsc = qRound(scalx * factor * it->w) /
                       ((it->logx ? log10(it->xma) : it->xma) - xmi);
          double ysc = qRound(scaly * factor * it->h) /
                       ((it->logy ? log10(it->yma) : it->yma) - ymi);
          double xMin = (r.left() - mnx) / xsc + xmi;
          double xMax = (r.right() - mnx) / xsc + xmi;
          if (it->logx) {
            xMin = pow(10, xMin);
            xMax = pow(10, xMax);
          }
          double yMin = (r.bottom() - mny) / ysc + ymi;
          double yMax = (r.top() - mny) / ysc + ymi;
          if (it->logy) {
            yMin = pow(10, yMin);
            yMax = pow(10, yMax);
          }
          it->autoScale(xMin, xMax, 1.0, yMin, yMax, 1.0);
          it->xOff = it->yOff = 0.0;
          for (item = doc->items()->next(); item; item = doc->items()->next()) {
            KplItem::ItemTypes typ = item->iType();
            switch (typ) {
              case KplItem::Function:
                ((FunItem*) item)->tmin = it->xmi / ((FunItem*) item)->fx;
                ((FunItem*) item)->tmax = it->xma / ((FunItem*) item)->fx;
                ((FunItem*) item)->dt = 0.0;
                break;
              case KplItem::Spline:
                ((SplineItem*) item)->xmin = it->xmi / ((SplineItem*) item)->fx;
                ((SplineItem*) item)->xmax = it->xma / ((SplineItem*) item)->fx;
                ((SplineItem*) item)->dx = 0.0;
                break;
              default:
                break;
            }
            if (typ == KplItem::Frame)
              break;
          }
          doc->setModified();
          doc->backupItems();
        } else
          drawBoundingRect(r);
      }
    }
}

void KplView::contentsMouseMoveEvent(QMouseEvent* e)
{
  if (sel.count()) {
    if (!(e->state() & ControlButton)) {
      KplItem* it1 = sel.first();
      for (KplItem* itm = it1; itm; itm = sel.next()) {
        QRect r = itm->boundingRect();
        QPoint d = dragPos - startPos;
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
        d = e->pos() - dragPos;
        r.moveBy(d.x(), d.y());
        drawBoundingRect(r);
      }
      dragPos = e->pos();
      QPoint d = e->pos() - startPos;
      emit mouseMoved(fromcm(d.x() / (scalx * factor) + it1->xPos(),
                             d.y() / (scaly * factor) + it1->yPos()), false);
    }
  } else {
    if (e->state() & MidButton) {
      drawBoundingRect(sortedRect(startPos, dragPos));
      dragPos = e->pos();
      drawBoundingRect(sortedRect(startPos, dragPos));
    }
    if (QRect(contentsX(), contentsY(), visibleWidth(),
              visibleHeight()).contains(e->pos())) {
      bool showValues = false;
      for (KplItem* item = doc->items()->last(); item;
           item = doc->items()->prev()) {
        KplItem::ItemTypes typ = item->iType();
        if ((typ == KplItem::Frame) || (typ == KplItem::Array3D) ||
            (typ == KplItem::Function3D)) {
          showValues = (item->iType() == KplItem::Frame);
          break;
        }
      }
      emit mouseMoved(e->pos(), showValues);
    } else
      emit mouseLeaved();
  }
}

void KplView::contentsDragEnterEvent(QDragEnterEvent* e)
{
  if (QUriDrag::canDecode(e))
    e->accept(QScrollView::viewport()->rect());
  else
    e->ignore(QScrollView::viewport()->rect());
}

void KplView::contentsDropEvent(QDropEvent* e)
{
  QStrList l;
  if (QUriDrag::decode(e, l)) {
    KURL::List list(QStringList::fromStrList(l));
    emit urlsDropped(&list);
    e->accept();
  } else
    e->ignore();
}

void KplView::drawBoundingRect(const QRect& r)
{
  begin(&pm);
  setRasterOp(NotROP);
  setPen(DashLine);
  drawRect(r);
  end();
  repaintContents(0, 0, pm.width(), pm.height(), false);
}

QRect KplView::sortedRect(QPoint& p1, QPoint& p2)
{
  return QRect(QMIN(p1.x(), p2.x()), QMIN(p1.y(), p2.y()),
               abs(p1.x() - p2.x()), abs(p1.y() - p2.y()));
}
