/***************************************************************************
                          arrayitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Aug 28 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <qurl.h>
#include <kapplication.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ksimpleconfig.h>
#include "arrayitem.h"
#include "kgraph.h"
#include "utils.h"
#ifndef KPL_CLASSES_ONLY
#include <qlistview.h>
#include "arraydlg.h"
#endif

ArrayItem::ArrayItem() : internal(false), ix(0), iy(0), ie(0), istart(0), n(0),
 errbars(0), nrows(0), ncols(0), idec(0), x(0)
{
}

ArrayItem::ArrayItem(const ArrayItem& a) :
 ScaledItem(a), internal(a.internal), ix(a.ix), iy(a.iy), ie(a.ie),
 istart(a.istart), n(a.n), errbars(a.errbars), nrows(a.nrows), ncols(a.ncols),
 idec(a.idec), url(a.url)
{
  x = ArrayItem::copyX(a.x, a.ncols, a.nrows);
}

ArrayItem::ArrayItem(KplNamespace::AutoStruct* aut) : ScaledItem(aut),
 internal(false), ix(aut->ixAuto), iy(aut->iyAuto), ie(aut->ieAuto), istart(0),
 n(0), errbars(aut->autoErr), nrows(0), ncols(0), idec(aut->iDec), x(0)
{
}

ArrayItem::ArrayItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
                     const KURL& uPlo) : ScaledItem(plo, aut),
 ix(0), iy(0), ie(0), istart(0), n(0), errbars(0), x(0)
{
  QString s = plo->readEntry("path", "");
  url = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  url.cleanPath();
  internal = plo->readBoolEntry("internal");
  if (internal) {
    n = nrows = plo->readNumEntry("n");
    errbars = plo->readNumEntry("errbars");
    ncols = 3;
    idec = 0;
    x = new double*[ncols];
    x[0] = new double[ncols * nrows];
    for (int j = 1; j < ncols; j++)
      x[j] = x[j - 1] + nrows;
    for (int i = 0; i < nrows; i++) {
      QString s;
      QStringList list = plo->readListEntry(s.sprintf("r%i", i), ' ');
      int cnt = list.count();
      for (int j = 0; j < ncols; j++)
        x[j][i] = (j < cnt) ? list[j].toDouble() : 0.0;
    }
  } else {
    idec = plo->readNumEntry("idec");
    if (!readFile()) {
      KMessageBox::error(0, url.url());
      return;
    }
    adjustRows(plo->readNumEntry("istart"), plo->readNumEntry("n", nrows));
    errbars = plo->readNumEntry("errbars", aut->autoErr);
  }
  adjustCols(plo->readNumEntry("ix", aut->ixAuto),
             plo->readNumEntry("iy", aut->iyAuto),
             plo->readNumEntry("ie", aut->ieAuto));
}

ArrayItem::ArrayItem(bool act, int fill, int sym, const QString& col,
                     double xn, double yn, int i_x, int i_y, int i_e,
                     int i_start, int np, int errb, const KURL& u,
                     bool _int, double relSize, int iDec) :
 ScaledItem(act, fill, sym, col, xn, yn, relSize), internal(_int), ix(i_x),
 iy(i_y), ie(i_e), istart(0), n(0), errbars(errb), idec(iDec), x(0), url(u)
{
  if (internal) {
    n = nrows = np;
    ncols = 3;
    idec = 0;
    int dim = ncols * nrows;
    x = new double*[ncols];
    x[0] = new double[dim];
    memset(x[0], 0, dim * sizeof(double));
    for (int j = 1; j < ncols; j++)
      x[j] = x[j - 1] + nrows;
  } else {
    if (!readFile()) {
      KMessageBox::error(0, url.url());
      return;
    }
    adjustRows(i_start, np);
    adjustCols();
  }
}

ArrayItem::~ArrayItem()
{
  freeX(&x);
}

ArrayItem& ArrayItem::operator=(const ArrayItem& a)
{
  if (this != &a) {
    *(ScaledItem*)this = a;
    internal = a.internal;
    ix = a.ix;
    iy = a.iy;
    ie = a.ie;
    istart = a.istart;
    n = a.n;
    errbars = a.errbars;
    nrows = a.nrows;
    ncols = a.ncols;
    idec = a.idec;
    url = a.url;
    freeX(&x);
    x = ArrayItem::copyX(a.x, a.ncols, a.nrows);
  }
  return *this;
}

double** ArrayItem::copyX(double **xv, int nc, int nr)
{
  double** x2 = 0;
  if (xv) {
    int n = nc * nr;
    x2 = new double*[nc];
    x2[0] = new double[n];
    memcpy(x2[0], xv[0], n * sizeof(double));
    for (int i = 1; i < nc; i++)
      x2[i] = x2[i - 1] + nr;
  }
  return x2;
}

void ArrayItem::freeX(double ***xv)
{
  if (*xv) {
    delete [] *xv[0];
    delete [] *xv;
    *xv = 0;
  }
}

int ArrayItem::nColumns(QTextStream& t, int idec)
{
  int nc = 0;
  while (!t.atEnd()) {
    QString line = t.readLine().stripWhiteSpace();
    if ((!line.isEmpty()) && (line[0] != '#')) {
      nc = QStringList::split(QRegExp(idec ? "[ \t]" : "[ ,\t]"),
                              line).count();
      break;
    }
  }
  return nc;
}

int ArrayItem::readLines(QTextStream& t, int nc, double ***xv, int idec)
{
  QMemArray<double> x2;
  int np = 0;
  int k = 0;
  int nr = 0;
  while (!t.atEnd()) {
    QString line = t.readLine().stripWhiteSpace();
    if ((!line.isEmpty()) && (line[0] != '#')) {
      Utils::translate(line, idec);
      QStringList list = QStringList::split(QRegExp(idec ? "[ \t]" : "[ ,\t]"),
                                            line);
      nr++;
      np += nc;
      x2.resize(np);
      int cnt = list.count();
      for (int i = 0; i < nc; i++) {
        x2[k] = (i < cnt) ? list[i].toDouble() : 0.0;
        k++;
      }
    }
  }
  *xv = new double*[nc];
  (*xv)[0] = new double[np];
  for (int i = 1; i < nc; i++)
    (*xv)[i] = (*xv)[i - 1] + nr;
  k = 0;
  for (int j = 0; j < nr; j++)
    for (int i = 0; i < nc; i++)
      (*xv)[i][j] = x2[k++];
  return nr;
}

int ArrayItem::readFile(const KURL& u, int *nc, double ***xv, int idec)
{
  freeX(xv);
  int nr = 0;
  *nc = 0;
  QString fn;
  if (u.isLocalFile())
    fn = u.path();
  else
#if KDE_VERSION >= 0x030200
    if (!KIO::NetAccess::download(u, fn, 0))
#else
    if (!KIO::NetAccess::download(u, fn))
#endif
      return nr;
  QFile f(fn);
  if (f.open(IO_ReadOnly)) {
    QTextStream t1(&f);
    *nc = nColumns(t1, idec);
    if (*nc) {
      f.reset();
      QTextStream t2(&f);
      nr = readLines(t2, *nc, xv, idec);
    }
    f.close();
  }
  if (!u.isLocalFile())
    KIO::NetAccess::removeTempFile(fn);
  return nr;
}

int ArrayItem::readFile()
{
  nrows = readFile(url, &ncols, &x, idec);
  return nrows;
}

int ArrayItem::readFile(QString* text, int *nc, double ***xv)
{
  QTextStream t1(text, IO_ReadOnly);
  int nr = 0;
  *nc = nColumns(t1);
  if (*nc) {
    QTextStream t2(text, IO_ReadOnly);
    nr = readLines(t2, *nc, xv);
  }
  return nr;
}

int ArrayItem::readFile(QString* text)
{
  nrows = readFile(text, &ncols, &x);
  return nrows;
}

void ArrayItem::adjustCols(int i_x, int i_y, int i_e)
{
  int icolmax = ncols - 1;
  ix = QMIN(i_x, icolmax);
  iy = QMIN(i_y, icolmax);
  ie = QMIN(i_e, icolmax);
}

void ArrayItem::adjustRows(int i_start, int np)
{
  istart = QMIN(i_start, nrows - 1);
  n = QMIN(np, nrows - istart);
}

KplItem::ItemTypes ArrayItem::iType() const
{
  return Array;
}

void ArrayItem::draw(KplGraph* g)
{
  if (x && active) {
    setProperties(g);
    double sav = g->relSize();
    g->setRelSize(relsiz * sav);
    g->plArray(&x[ix][istart], &x[iy][istart], fx, fy, n, fillStyle, true, x0,
               y0);
    if (errbars)
      g->plError(&x[ix][istart], &x[iy][istart], &x[ie][istart], fx, fy, n,
                 true, x0, y0);
    g->setRelSize(sav);
  }
}

#ifdef KPL_CLASSES_ONLY
void ArrayItem::setRange(int _istart, int _n, int _symb, const QString& col)
{
  adjustRows(_istart, _n);
  symb = _symb;
  color = KplGraph::rgbQt(col);
}

#else
void ArrayItem::writePlo(KSimpleConfig* plo, const KURL& uPlo, bool _abs,
                         KplDoc* m) const
{
  plo->writeEntry("Type", "ARRAYITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("internal", internal);
  plo->writeEntry("ix", ix);
  plo->writeEntry("iy", iy);
  plo->writeEntry("ie", ie);
  plo->writeEntry("errbars", errbars);
  plo->writeEntry("n", n);
  if (internal) {
    plo->writeEntry("istart", 0);
    for (int i = 0; i < n; i++) {
      QStringList list;
      for (int j = 0; j < ncols; j++)
        list.append(QString::number(x[j][i + istart]));
      QString s;
      plo->writeEntry(s.sprintf("r%i", i), list, ' ');
    }
  } else {
    plo->writeEntry("idec", idec);
    plo->writeEntry("istart", istart);
    plo->writeEntry("path", Utils::relPath(uPlo, url, _abs));
  }
}

void ArrayItem::setText(QListViewItem* it, bool* arrays, bool*) const
{
  *arrays = true;
  it->setText(1, i18n("Array"));
  KGraph g;
  it->setPixmap(2, g.pixmap(symb, color));
  QString s = (internal ? i18n("internal data") :
                          (i18n("file") + " " + url.fileName())) +
    ", " + i18n("columns") + " " + QString::number(ix) + ", " +
    QString::number(iy);
  if (errbars)
    s += ", " + QString::number(ie);
  it->setText(2, s);
}

int ArrayItem::editItem(QWidget* parent, KplDoc* m, int)
{
  ArrayDlg dlg(parent, m, this);
  return dlg.exec();
}
#endif

KplItem* ArrayItem::copy() const
{
  return new ArrayItem(*this);
}

void ArrayItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (x) {
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void ArrayItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (x) {
    Utils::minMaxFile(xmi, xma, &x[ix][istart], n);
    Utils::minMaxFile(ymi, yma, &x[iy][istart], n);
    *xmi += x0;
    *xma += x0;
    *ymi += y0;
    *yma += y0;
  }
}
