/***************************************************************************
                          psgraph.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Wed Sep 08 1999
    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfile.h>
#include <qglobal.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <kio/netaccess.h>
#include <math.h>
#include "psgraph.h"
#include "utils.h"

const double PSGraph::pts = 28.34645669;

PSGraph::PSGraph() : x0(0.0), xl(0.0), y0(0.0), yl(0.0), xmin(0.0), xmax(0.0),
 ymin(0.0), ymax(0.0), ts(0)
{
  devType = PS;
}

PSGraph::~PSGraph()
{
}

void PSGraph::psInit(QTextStream* t, double xf, double yf, bool orientation)
{
  psInit(t, xf, yf, orientation, 0, 0, qRound((orientation ? yf : xf) * pts),
         qRound((orientation ? xf : yf) * pts));
}

void PSGraph::psInit(QTextStream* t, double xf, double yf, bool orientation,
                     int llx, int lly, int urx, int ury)
{
  ts = t;
  logx = logy = false;
  kSymb = 0;
  relSiz = 1.0;
  xmin = ymin = x0 = y0 = 0.0;
  xmax = xl = xf;
  ymax = yl = yf;
  *ts << "%!PS-Adobe-3.0 EPSF-3.0\n"
    "%%BoundingBox: " << llx << " " << lly << " " << urx << " " << ury << "\n"
    "%%Creator: Kpl\n"
    "%%CreationDate: " << QDateTime::currentDateTime().toString() << "\n"
    "%%Orientation: " << (orientation ? "Landscape" : "Portrait") << "\n"
    "%%Pages: 1\n\n"
    "%User code starts at line 374\n\n"
    << "/d /def load def\n"
    "/D {bind d} bind d\n"
    "/ex {exch} D\n"
    "/ed {ex d} D\n"
    "/LT {lineto} D\n"
    "/MT {moveto} D\n"
    "/S {stroke} D\n"
    "/CP {closepath} D\n"
    "/cp {currentpoint} D\n"
    "/RL {rlineto} D\n"
    "/RM {rmoveto} D\n"
    "/NP {newpath} D\n"
    "/TR {translate} D\n"
    "/R {rotate} D\n"
    "/i {index} D\n"
    "/IE {ifelse} D\n"
    "/FF {findfont} D\n"
    "/LE {length} D\n"
    "/p {pop} D\n"
    "/sg {string} D\n"
    "/SH {show} D\n"
    "/r {repeat} D\n"
    "/L {loop} D\n"
    "/gs {gsave} D\n"
    "/gr {grestore} D\n"
    "/T {true} D\n"
    "/F {false} D\n"
    "/xmi 0 d\n"
    "/ymi 0 d\n"
    "/xma "
    << xf << " d\n"
    "/yma "
    << yf << " d\n"
    "/sx 28.346456 d\n"
    "/xs sx d\n"
    "/ys xs d\n"
    "/mix 0 d\n"
    "/miy 0 d\n"
    "/max xs xma mul d\n"
    "/may ys yma mul d\n"
    "/xc 0 d\n"
    "/yc 0 d\n"
    "/cf 0 d\n"
    "/cg 0 d\n"
    "/cd 0 d\n"
    "/di 0 d\n"
    "/sy 0 d\n"
    "/dp 0 d\n"
    "/dc 0 d\n"
    "/rs 1 d\n"
    "/nx -1 d\n"
    "/ny -1 d\n"
    "/lx F d\n"
    "/ly F d\n\n"
    << "/EF {\n"
    " FF dup LE dict begin {\n"
    "  1 i dup /FID eq ex /Encoding eq or {p p} {d} IE\n"
    " } forall /Encoding ISOLatin1Encoding d currentdict end definefont\n"
    "} D\n\n"
    << "/sc {\n"
    " /s2 ed /s1 ed s1 LE dup s2 LE add sg /s3 ed s3 ex\n"
    " s2 putinterval s3 0 s1 putinterval s3\n"
    "} D\n\n"
    << "/rr {ymi sub ys mul miy add ex xmi sub xs mul mix add ex} D\n\n"
    << "/mv {NP rr MT} D\n\n"
    << "/re {/y2 ed /x2 ed /y1 ed /x1 ed"
    " NP x1 y1 MT x2 y1 LT x2 y2 LT x1 y2 LT CP} D\n\n"
    << "/SC {dup stringwidth ex 2 div neg ex RM cp"
    " 3 2 roll SH MT} D\n\n"
    << "/SR {dup stringwidth ex neg ex RM cp 3 2 roll SH MT} D\n\n"
    << "/rl {max mix sub mul rs mul} D\n\n"
    << "/sw {rl setlinewidth} D\n\n"
    << "/sf {rl scalefont setfont} D\n\n"
    << "/mk {/sh 0.01 rl d sh mul ex sh mul ex RM {"
    "sh mul ex sh mul ex RL} r} D\n\n"
    << "/m0 {NP rr 0.01 rl 0 360 arc} D\n\n"
    "/m1 {0.004 sw mv 0 0 RL S 0.00295 sw} D\n\n"
    "/m2 {mv -1 1 1 1 1 -1 -1 -1 4 0 1 mk} D\n\n"
    "/m3 {mv 0 2 2 0 0 -2 -2 0 4 1 1 mk} D\n\n"
    "/m4 {mv -1 1.732 2 0 -1 -1.732 3 0 0.866 mk} D\n\n"
    "/m5 {mv 2 2 -2 0 2 -2 -2 0 4 1 1 mk} D\n\n"
    "/m6 {mv 0 1 0.5 0.5 1 0 0.5 -0.5 0 -1 -0.5 -0.5 -1 0 -0.5"
    " 0.5 8 1 0.5 mk} D\n\n"
    "/m7 {mv 2 -2 -1 1 1 1 -2 -2 4 1 1 mk} D\n\n"
    "/m8 {mv 2 0 -1 0 0 1 0 -2 4 0 1 mk} D\n\n"
    "/m9 {mv -1 1 -1 -1 1 1 0 0.866 4 0 -0.866 mk} D\n\n"
    "/m10 {mv 1 1 -2 0 1 -1 0 2 4 0 -1 mk} D\n\n"
    << "/Lt {\n 1 sub dup 0 lt {p 0} if 9 mod dup\n"
    " [8 7 2 1 9 8 7 4 1] ex get /p0 ed"
    " [8 0 0 0 0 1 2 4 1] ex get /p1 ed\n"
    " /f 0.003125 rl d 4 array dup 0 p0 f mul put dup 1 16 p0 sub p1 sub\n"
    " f mul p1 0 eq {put dup dup 0 get 2 ex} {0.5 mul put dup 2 p1 f mul}\n"
    " IE put dup dup 1 get 3 ex put dp setdash\n"
    "} D\n\n"
    << "/er {\n fy mul ys mul /dy ed fy mul /y ed fx mul /x ed ly\n"
    " {y 0 gt {/dy dy y div 0.4343 mul d} {/dy 0 d} IE} if\n"
    " x 0 gt lx not or dy sh ge and {\n"
    "  x lx {log} if y ly {log} if mv 0 sh RM 0 dy sh sub\n"
    "  RL 0.5 sh mul 0 RM sh neg 0 RL 0 -2 dy mul RM sh\n"
    "  0 RL -0.5 sh mul 0 RM 0 dy sh sub RL S\n"
    " } if\n"
    "} D\n\n"
    << "/format {\n /yma ed /xma ed /xmi 0 d /ymi 0 d"
    " /xs sx d /ys xs d /mix 0 d /miy 0 d /max xs\n"
    " xma mul d /may ys yma mul d /xc 0 d /yc 0 d\n"
    "} D\n\n"
    << "/window {\n /yma ed /ymi ed /xma ed /xmi ed"
    " /xma xma xmi add d /yma yma ymi add d\n"
    " /mix xs xmi mul d /miy ys ymi mul d"
    " /max xs xma mul d /may ys yma mul d\n"
    " /xc 0 d /yc 0 d 0.00295 sw"
    " /l1 FF 0.033333 sf\n"
    "} D\n\n"
    << "/Sc {\n /ly ed /lx ed /yma ed /ymi ed /xma ed /xmi ed\n"
    " lx {/xmi xmi log d /xma xma log d} if"
    " ly {/ymi ymi log d /yma yma log d} if\n"
    " /xs max mix sub xma xmi sub div d"
    " /ys may miy sub yma ymi sub div d\n"
    " /xc xmi d /yc ymi d\n"
    "} D\n\n"
    << "/sC {\n"
    " /col ed col 16#ff and 16#ff div col -8 bitshift 16#ff and 16#ff div\n"
    " col -16 bitshift 16#ff div setrgbcolor\n"
    "} D\n\n"
    << "/frame {mix miy max may re S} D\n\n"
    "/cl {mix dc add miy dc add max dc sub may dc sub re clip} D\n\n"
    << "/pn {\n"
    " /nD ed /v ed v 0 lt {/v v neg d (-)} {()} IE /v nD 0 gt\n"
    " {10 nD exp dup v mul round dup cvi /iv ed ex div}\n"
    " {v round dup cvi /iv ed} IE d /ie v 10 lt {0} {v log cvi}\n"
    " IE d /iend nD 0 gt {nD neg} {0} IE d"
    " /id 10 ie iend sub exp round cvi d {\n"
    "  iv dup id mod 10 mul /iv ed id idiv 256 sg cvs\n"
    "  sc ie 0 eq {nD 0 ge {(.) sc} if} if ie iend eq"
    " {exit} if /ie ie 1 sub d\n"
    " } L\n"
    "} D\n\n"
    << "/il {\n"
    " log /lf ed /xl ed /ma xl dup dup abs 1.0e-5 mul add cvi sub d\n"
    " ma abs 1.0e-6 lt {/ma 0 d} if ma 0 lt {/ma ma 1 add d}\n"
    " if xl ma 0.1 lt {lf add} {ma 0.4 lt {2.5 log add}\n"
    "  {ma 0.6 lt {10 3 div log add} {2 log add} IE} IE} IE\n"
    "} D\n\n"
    << "/raster {\n"
    " /yR ed /yL ed /xT ed /xB ed /yO ed /xO ed /mo ed\n"
    " /iy ed /ix ed /yt ed /xt ed mo -1 ne {mo 0 ne {\n"
    "  lx {xt 2.5 lt {/xt 2 d} {xt 5.5 lt {/xt 3 d}\n"
    "   {xt 99 lt {/xt 10 d} if} IE} IE} if\n"
    "  ly {yt 2.5 lt {/yt 2 d} {yt 5.5 lt {/yt 3 d}\n"
    "   {yt 99 lt {/yt 10 d} if} IE} IE} if\n"
    "  cg sC 0.00208 sw /dx max mix sub d /dy may miy sub d\n"
    "  mo 2 le {/dx dx 0.02 mul d /dy dx d} if\n"
    "  mo 2 le ix 1 gt and xO 0 gt and lx not and {\n"
    "   /dd xt ix div d /xm xmi xO add dd sub d {xm xmi le {exit} if\n"
    "    xB {xm ymi mv 0 dy 0.5 mul RL S} if"
    " xT {xm yma mv 0 dy -0.5 mul RL S} if\n"
    "    /xm xm dd sub d} L} if\n"
    "  /x xmi xO add d {lx\n"
    "   {/xa x d /x x xt il d\n"
    "    x xma lt {xB {x ymi mv 0 dy RL S} if\n"
    "    xT {x yma mv 0 dy neg RL S} if} if ix 1 gt {\n"
    "     /xd 10 xa ma sub exp d {/xa 10 xa exp xd add log d\n"
    "      xa x ge xa xma ge or {exit} if\n"
    "      xB {xa ymi mv 0 dy mo 2 le {0.5 mul} if RL S} if\n"
    "      xT {xa yma mv 0 dy neg mo 2 le {0.5 mul} if RL S} if\n"
    "     } L\n"
    "    } if} {mo 2 le ix 1 gt and {\n"
    "     /xm x d ix 1 sub {\n"
    "      /xm xt ix div xm add d xm xma ge {exit} if\n"
    "      xB {xm ymi mv 0 dy 0.5 mul RL S} if"
    " xT {xm yma mv 0 dy -0.5 mul RL S} if\n"
    "     } r\n"
    "    } if x xmi gt {\n"
    "     xB {x ymi mv 0 dy RL S} if\n"
    "     xT {x yma mv 0 dy neg RL S} if\n"
    "    } if /x x xt add d\n"
    "   } IE x xma ge {exit} if} L\n"
    "  mo 2 le iy 1 gt and yO 0 gt and ly not and {\n"
    "   /dd yt iy div d /ym ymi yO add dd sub d {\n"
    "    ym ymi le {exit} if yL {xmi ym mv dx 0.5 mul 0 RL S} if\n"
    "    yR {xma ym mv dx -0.5 mul 0 RL S} if /ym ym dd sub d} L} if\n"
    "   /y ymi yO add d {ly {/ya y d /y y yt il d\n"
    "    y yma lt {yL {xmi y mv dx 0 RL S} if\n"
    "    yR {xma y mv dx neg 0 RL S} if} if iy 1 gt {\n"
    "     /yd 10 ya ma sub exp d {/ya 10 ya exp yd add log d\n"
    "      ya y ge ya yma ge or {exit} if\n"
    "      yL {xmi ya mv dx mo 2 le {0.5 mul} if 0 RL S} if\n"
    "      yR {xma ya mv dx neg mo 2 le {0.5 mul} if 0 RL S} if\n"
    "     } L\n"
    "    } if} {mo 2 le iy 1 gt and {\n"
    "     /ym y d iy 1 sub {\n"
    "      /ym yt iy div ym add d ym yma ge {exit} if\n"
    "      yL {xmi ym mv dx 0.5 mul 0 RL S} if"
    " yR {xma ym mv dx -0.5 mul 0 RL S} if\n"
    "     } r} if y ymi gt {\n"
    "     yL {xmi y mv dx 0 RL S} if"
    " yR {xma y mv dx neg 0 RL S} if\n"
    "    } if /y y yt add d\n"
    "   } IE y yma ge {exit} if} L mo 2 eq mo 4 eq or {\n"
    "   xB {/yB ymi xma xmi sub rs mul xs mul ys div 0.05 mul sub d} if\n"
    "   xT {/yT yma xma xmi sub rs mul xs mul ys div 0.025 mul neg sub d} if\n"
    "   /e xma 2.99 gt xmi -2.99 lt or xt 9.9 gt and\n"
    "   lx and d /nd nx d /x xmi xO add d /xe xma xmi sub\n"
    "   1e-3 mul xma add d {\n"
    "    x xma gt {/x xma d} if e {(10#nu) x round cvi 4 sg cvs sc\n"
    "     xB {dup x yB rr 2 text} if xT {x yT rr 2 text} {p} IE}\n"
    "     {lx {x 0 lt {/nd 1 x 0.01 add cvi sub d} {/nd -1 d} IE}\n"
    "     if x lx {10 ex exp} if nd pn cf sC\n"
    "     xB {dup x yB mv SC} if xT {x yT mv SC} {p} IE}\n"
    "    IE lx {/x x xt il d} {/x x xt add d} IE x xe gt {exit} if\n"
    "   } L yL {/xL xmi xma xmi sub rs mul 0.025 mul sub d} if\n"
    "   yR {/xR xma xma xmi sub rs mul -0.025 mul sub d} if\n"
    "   /dy xma xmi sub 0.0125 mul rs mul xs mul ys div d\n"
    "   /e yma 2.99 gt ymi -2.99 lt or yt 9.9 gt and ly and d\n"
    "   /nd ny d /y ymi yO add d /ye yma ymi sub 1e-3 mul\n"
    "   yma add d {\n"
    "    y yma gt {/y yma d} if\n"
    "    e {(10#nu) y round cvi 4 sg cvs sc\n"
    "     yL {dup xL y dy sub rr 3 text} if"
    " yR {xR y dy sub rr 1 text} {p} IE}\n"
    "     {ly {y 0 lt {/nd 1 y 0.01 add cvi sub d} {/nd -1 d} IE}\n"
    "     if y ly {10 ex exp} if nd pn cf sC\n"
    "     yL {dup xL y dy sub mv SR} if yR {xR y dy sub mv SH} {p} IE\n"
    "    } IE ly {/y y yt il d} {/y y yt add d} IE y ye gt {exit} if\n"
    "   } L\n"
    "  } if\n"
    " } if cf sC 0.00417 sw /dc currentlinewidth 0.5 mul d"
    " frame cd sC 0.00295 sw} if\n"
    "} D\n\n"
    << "/text {\n"
    " /io ed gs TR di R 0 0 MT /str ed"
    " /l1 FF 0.033333 sf cf sC str (#) search {\n"
    "  p p p /dx 0 d io 1 eq {1} {0} IE 1 1 {\n"
    "   /j ed /ilev 0 d /s str d"
    " j 1 eq {dx io 2 eq {0.5 mul} if neg 0 RM} if {\n"
    "    s (#) search ex /s1 ed {p /c ed} {/c 0 sg d} IE\n"
    "    j 0 eq {s1 stringwidth p dx add /dx ed} {\n"
    "     ilev 0 eq {s1 SH} {\n"
    "      ilev 1 eq {0.015} {-0.01} IE rl dup 0 ex RM s1 SH neg 0 ex RM\n"
    "     } IE } IE c LE 2 lt {exit} {\n"
    "     c 0 get dup 96 gt {32 sub} if 1 sg dup 0 4 -1 roll put\n"
    "     (NHBOSZI) ex search {\n"
    "      LE [/l1 /l1 /l1B /l1O /Symbol /Dingbats /l1I]\n"
    "      ex get FF c 1 get dup 96 gt {32 sub} if 1 sg dup\n"
    "      0 4 -1 roll put (NUD) ex search {LE /ilev ed p} if\n"
    "      p ilev 0 eq {0.033333} {0.022222} IE sf p\n"
    "     } if p /s c 2 c LE 2 sub getinterval d\n"
    "    } IE s LE 0 eq {exit} if\n"
    "   } L /l1 FF 0.033333 sf\n"
    "  } for\n"
    " } {io 1 eq {SH} {io 2 eq {SC} {SR} IE} IE} IE gr\n"
    "} D\n\n"
    << "/arrow {\n"
    " gs cf sC sx mul ex sx mul ex TR NP 0 0 MT R -0.03333 rl dup dup\n"
    " 0.2 mul dup neg 3 1 roll LT LT CP fill sx mul dup 0.03333 rl ge {\n"
    "  0.00417 sw neg 0 MT -0.03333 rl 0 LT CP S\n"
    " } {p} IE gr} D\n\n"
    << "/arccm {\n"
    " gs sx mul ex sx mul ex TR R /h ed h div\n"
    " /a ed /wl ed /wa ed 6 array currentmatrix a 1 scale NP 0 0 h\n"
    " 0.5 mul sx mul wa wl wl 0 ge {wa add} {add wa} IE arc setmatrix S gr\n"
    " } D\n\n"
    << "/rectcm {\n"
    " gs sx mul ex sx mul ex TR R sx mul ex\n"
    " sx mul ex 0 0 re dup 0 gt {bf} {p S} IE gr\n"
    " } D\n\n"
    << "/ellcm {\n"
    " gs sx mul ex sx mul ex TR R /h ed h div\n"
    " /a ed 6 array currentmatrix a 1 scale NP 0 0 h 0.5 mul sx\n"
    " mul 0 360 arc setmatrix dup 0 gt {bf} {p S} IE gr\n"
    " } D\n\n"
    << "/scalebar {\n"
    " gs cf sC sx mul ex sx mul ex TR 0 ne {90 R} if 0.00417 sw\n"
    " /a max mix sub 0.005 mul d NP 0 a MT 0 a neg LT S NP 0 0 MT sx mul\n"
    " 0 LT cp S NP MT 0 a RM 0 a -2 mul RL S gr\n"
    " } D\n\n"
    << "/LH {{0.16975} {0.025} IE rl may add mix max add 0.5 mul ex"
    " 2 text} D\n\n"
    << "/XY {\n"
    " /iax ed /op ed /iex ed /str ed str LE iex or 0 ne {\n"
    "  iex 0 eq {str} {(10#nu) iex 4 sg cvs sc ( #nn) sc str\n"
    "   sc} IE iax 2 eq {/l 3 ly {yma 2.99 gt ymi -2.99 lt\n"
    "   or {ymi 0 lt {4} {3} IE} {yma 1.99 ymi sub 2 copy lt\n"
    "   {ex} if p cvi 1 add} IE} {ymi ny pn LE\n"
    "   yma ny pn LE 2 copy lt {ex} if p} IE add d\n"
    "   op {max} {mix} IE l 0.01875 rl mul op\n"
    "   {neg} if sub miy may add 0.5 mul /di 90 op {neg} if d\n"
    "  } {mix max add 0.5 mul op {may -0.085} {miy 0.11} IE\n"
    "  rl sub} IE 2 text /di 0 d\n"
    " } if\n"
    "} D\n\n"
    << "/LX {1 XY} D\n\n"
    << "/LY {2 XY} D\n\n"
    << "/plarry {\n"
    " gs {cl} if 3 1 roll /fy ed /fx ed sy 0 gt {\n"
    "  sy Lt 3 1 roll fy mul /ya ed fx mul /xa ed /p1 F d\n"
    "  xa 0 gt lx not or ya 0 gt ly not or and {\n"
    "   xa lx {log} if ya ly {log} if NP rr /y ed /x ed x y MT /p1 T d\n"
    "  } if 1 sub {\n"
    "   fy mul /ya ed fx mul /xa ed"
    " xa 0 gt lx not or ya 0 gt ly not or and {\n"
    "    xa lx {log} if ya ly {log} if rr dup y sub dup mul\n"
    "    2 i x sub dup mul add sqrt dp add /dp ed dup /y ed\n"
    "    /x 2 i d p1 {LT} {MT /p1 T d} IE\n"
    "   } if\n"
    "  } r S /dp dp dup 0.05 rl div floor 0.05 rl mul sub d [] dp setdash\n"
    " } {\n"
    "  /m [/m0 /m1 /m2 /m3 /m4 /m5 /m6 /m7 /m8 /m9 /m10 /m0 /m2 /m3 /m4\n"
    "  /m5 /m6] sy neg 17 mod get load d {\n"
    "   sy -17 eq {p p} {fy mul /ya ed fx mul /xa ed xa 0 gt\n"
    "   lx not or ya 0 gt ly not or and {xa lx {log} if ya ly\n"
    "   {log} if m sy neg 17 mod 10 le {S} {fill} IE} if}"
    " IE\n  } r\n"
    " } IE gr\n"
    "} D\n\n"
    << "/plerry {gs {cl} if 3 1 roll /fy ed /fx ed /sh 0.01 rl d"
    " {er} r gr} D\n\n"
    "/bf {\n"
    " /b ed b 1 eq {fill} {clip pathbbox 2 i sub /h ed\n"
    " 2 i sub /w ed TR 0.2 setlinewidth NP\n"
    " b 2 eq b 4 eq or {0 2.834 h {dup 0 ex MT w ex LT} for} if\n"
    " b 3 eq b 4 eq or {0 2.834 w {dup 0 MT h LT} for} if\n"
    " b 5 eq b 7 eq or {w h gt {0 4 w h add "
    "{dup h MT h sub 0 LT} for}\n"
    " {0 4 w h add {dup w ex MT w sub 0 ex LT} for} IE} if\n"
    " b 6 eq b 7 eq or {w h gt {0 4 w h add "
    "{dup 0 MT h sub h LT} for}\n"
    " {0 4 w h add {dup 0 ex MT w sub w ex LT} for} IE} if S} IE\n"
    "} D\n\n"
    << "%%Page: 1 1\n\n"
    "gs\n"
    "/l1 /Helvetica EF FF 0.033333 sf\n"
    "/l1B /Helvetica-Bold EF p\n"
    "/l1O /Helvetica-Oblique EF p\n"
    "/l1I /Helvetica-BoldOblique EF p\n"
    "1 setlinecap 1 setlinejoin\n\n"
    "0.00295 sw\n\n%User code starts here\n\n";
    if (orientation)
      *ts << qRound(yf * pts) << " 0 TR 90 R\n";
}

void PSGraph::format(double x, double y)
{
  *ts << x << " " << y << " format\n";
}

double PSGraph::setRelSize(double r)
{
  double old = relSiz;
  relSiz = r;
  *ts << "/rs " << r << " d 0.00295 sw\n";
  return old;
}

void PSGraph::Window(double x0a, double xla, double y0a, double yla)
{
  x0 = x0a;
  xl = xla;
  y0 = y0a;
  yl = yla;
  xmin = ymin = 0.0;
  xmax = xl;
  ymax = yl;
  *ts << x0 << " " << xl << " " << y0 << " " << yl << " window\n";
}

void PSGraph::scale(double _xl, double xr, double yb, double yt,
                    bool lgx, bool lgy)
{
  logx = lgx;
  logy = lgy;
  if (logx) {
    xmin = log10(_xl);
    xmax = log10(xr);
  } else {
    xmin = _xl;
    xmax = xr;
  }
  if (logy) {
    ymin = log10(yb);
    ymax = log10(yt);
  } else {
    ymin = yb;
    ymax = yt;
  }
  *ts << _xl << " " << xr << " " << yb << " " << yt
      << (logx ? " T" : " F") << (logy ? " T" : " F") << " Sc\n";
}

void PSGraph::setDig(int nx, int ny)
{
  *ts << "/nx " << nx << " d\n/ny " << ny << " d\n";
}

void PSGraph::setColFrame(unsigned icol)
{
  QCString s;
  *ts << s.sprintf("/cf 16#%x d\n", rgbQt1(icol));
}

void PSGraph::setColGrid(unsigned icol)
{
  QCString s;
  *ts << s.sprintf("/cg 16#%x d\n", rgbQt1(icol));
}

void PSGraph::setColData(unsigned icol)
{
  QCString s;
  *ts << s.sprintf("/cd 16#%x d\ncd sC\n",  rgbQt1(icol));
}

void PSGraph::setCol(unsigned icol1, unsigned icol2, unsigned icol3)
{
  setColFrame(icol1);
  setColGrid(icol2);
  setColData(icol3);
}

void PSGraph::frame(void)
{
  *ts << "frame\n";
}

void PSGraph::raster(double xtic, double ytic, int intx, int inty, int mode,
                     double xOff, double yOff, bool xBottom, bool xTop,
                     bool yLeft, bool yRight)
{
  *ts << xtic << " " << ytic << " " << intx << " " << inty << " "
      << mode << " " << xOff << " " << yOff << (xBottom ? " T" : " F")
      << (xTop ? " T" : " F") << (yLeft ? " T" : " F")
      << (yRight ? " T" : " F") << " raster\n";
}

void PSGraph::setDir(double ang)
{
  *ts << "/di " << ang << " d\n";
}

void PSGraph::text(const QString& str, double x, double y, int iorg)
{
  psStr(str);
  *ts << " " << x << " " << y << " " << iorg << " text\n";
}

void PSGraph::textcm(const QString& str, double x, double y, int iorg, QRect*)
{
  text(str, pts * x, pts * y, iorg);
}

void PSGraph::arrow(double x, double y, double dir, double len, QRect*)
{
  *ts << len << " " << dir << " " << x << " " << y << " arrow\n";
}

void PSGraph::arc(double x, double y, double w, double h, double a,
                  double alen, double dir, QRect*)
{
  *ts << a << " " << alen << " " << w << " " << h << " "
      << dir << " " << x << " " << y << " arccm\n";
}

void PSGraph::rectangle(double x, double y, double w, double h,
                        int fillStyle, double dir, QRect*)
{
  *ts << fillStyle << " " << w << " " << h << " "
      << dir << " " << x << " " << y << " rectcm\n";
}

void PSGraph::ellipse(double x, double y, double w, double h,
                     int fillStyle, double dir, QRect*)
{
  *ts << fillStyle << " " << w << " " << h << " "
      << dir << " " << x << " " << y << " ellcm\n";
}

void PSGraph::image(const KURL& url, double x, double y, double fx, double fy,
                    QRect*)
{
  QString fn;
  double x1, x2, y1, y2;
  if (Utils::boundingBox(url, fn, &x1, &y1, &x2, &y2)) {
    *ts << "save /showpage { } d 0 setlinecap 1 setlinewidth 0 setlinejoin\n"
           "[ ] 0 setdash\n" << x << " sx mul " << y << " sx mul TR " << fx
        << " " << fy << " scale\n" << -x1 << " " << -y1 << " TR " << x1 << " "
        << y1 << " " << x2 << " " << y2 << " re clip\n"
        << "%%BeginDocument: " << url.path() << "\n";
    ts->device()->flush();
    QFile f(fn);
    if (f.open(IO_ReadOnly)) {
      ts->device()->writeBlock(f.readAll());
      f.close();
      ts->device()->flush();
    }
    if (!url.isLocalFile())
      KIO::NetAccess::removeTempFile(fn);
    *ts << "\n%%EndDocument\nrestore\n";
  }
}

void PSGraph::scaleBar(const QString& str, double x, double y, int o,
                       double len, QRect*)
{
  *ts << len << " " << o << " " << x << " " << y << " scalebar\n";
  if (o)
    textcm(str, x + 0.01 * xl, y + 0.5 * len - 0.012 * relSiz * xl, 1, 0);
  else
    textcm(str, x + 0.5 * len, y + (2 + relSiz) * 0.005 * xl, 2, 0);
}

void PSGraph::letH(const QString& str, bool xTop)
{
  psStr(str);
  *ts << (xTop ? " T" : " F") << " LH\n";
}

void PSGraph::letX(const QString& str, int iex, bool xTop)
{
  psStr(str);
  *ts << " " << iex << (xTop ? " T" : " F") << " LX\n";
}

void PSGraph::letY(const QString& str, int iey, bool yRight)
{
  psStr(str);
  *ts << " " << iey << (yRight ? " T" : " F") << " LY\n";
}

void PSGraph::setSymbol(int is)
{
  kSymb = is;
  *ts <<"/sy " << is << " d\n";
}

void PSGraph::plArray(const double* x, const double* y,
                      double fx, double fy, int n, int fillStyle, bool clip)
{
  if (n > 0)
    if (fillStyle) {
      bool pl = false;
      QCString s;
      for (int i = 0; i < n; i++) {
        double xa = fx * x[i];
        double ya = fy * y[i];
        if (Utils::valid(xa, logx) && Utils::valid(ya, logy)) {
          xa = (((logx ? log10(xa) : xa) - xmin) / (xmax - xmin) * xl + x0)
               * pts;
          ya = (((logy ? log10(ya) : ya) - ymin) / (ymax - ymin) * yl + y0)
               * pts;
          if (pl)
            s.sprintf("%g %g LT\n", xa, ya);
          else {
            *ts << "\ngs\n";
            if (clip)
              *ts << "cl\n";
            s.sprintf("NP %g %g MT\n", xa, ya);
          }
          *ts << s;
          pl = true;
        }
      }
      if (pl)
        *ts << fillStyle << " bf\ngr\n";
    } else {
      int ie = 0;
      int ia;
      bool xNorm = (fx > 1.0e33) || (fx < 1.0e-33);
      bool yNorm = (fy > 1.0e33) || (fy < 1.0e-33);
      do {
        ia = QMIN(n - 1, ie + 127);
        *ts << "\n";
        int fail = 0;
        QCString s;
        for (int i = ia; i >= ie; i--) {
          double xa = xNorm ? (fx * x[i]) : x[i];
          double ya = yNorm ? (fy * y[i]) : y[i];
          if (Utils::valid(xa, logx) && Utils::valid(ya, logy))
            *ts << xa << " " << ya << "\n";
          else
            fail++;
        }
        int np = ia - ie - fail + 1;
        if (np)
          *ts << (xNorm ? 1.0 : fx) << " " << (yNorm ? 1.0 : fy) << " "
              << np << (clip ? " T" : " F") << " plarry\n";
        ie = ia;
        if (kSymb <= 0)
          ie++;
      } while (ia != (n - 1));
    }
}

void PSGraph::plError(const double* x, const double* y, const double* e,
                      double fx, double fy, int n, bool clip)
{
  if (n > 0) {
    int ie = 0;
    int ia;
    bool xNorm = (fx > 1.0e33) || (fx < 1.0e-33);
    bool yNorm = (fy > 1.0e33) || (fy < 1.0e-33);
    do {
      ia = QMIN(n - 1, ie + 127);
      *ts << "\n";
      int fail = 0;
      QCString s;
      for (int i = ia; i >= ie; i--) {
        double xa = xNorm ? (fx * x[i]) : x[i];
        double ya = yNorm ? (fy * y[i]) : y[i];
        double ea = yNorm ? (fy * e[i]) : e[i];
        if (Utils::valid(xa, logx) && Utils::valid(ya, logy) &&
            Utils::valid(ea, logy))
          *ts << xa << " " << ya <<  " " << ea << "\n";
        else
          fail++;
      }
      *ts << (xNorm ? 1.0 : fx) << " " << (yNorm ? 1.0 : fy) << " "
          << (ia - ie - fail + 1) << (clip ? " T" : " F") << " plerry\n";
      ie = ia + 1;
    } while (ia != (n - 1));
  }
}

void PSGraph::resetDash()
{
  *ts << "/dp 0 d\n";
}

void PSGraph::psExit(void)
{
  *ts << "\nshowpage gr\n\n%%Trailer\n%%EOF\n";
}

void PSGraph::psStr(const char* str)
{
  *ts << "(";
  QCString s;
  const unsigned char* c = (const unsigned char*) str;
  while (*c) {
    if (*c > 127)
      *ts << s.sprintf("\\%03o", *c);
    else
      switch (*c) {
        case 40:
        case 41:
        case 92:
          *ts << "\\";
        default:
          *ts << QChar(*c);
      }
    c++;
  }
  *ts << ")";
}

void PSGraph::cm2r(double xcm, double ycm, double _fxn, double _fyn,
                   double* xr, double* yr) const
{
  *xr = xmin + (xcm - x0) * _fxn * (xmax - xmin) / xl;
  if (logx)
    *xr = pow(10, *xr);
  *yr = ymin + (ycm - y0) * _fyn * (ymax - ymin) / yl;
  if (logy)
    *yr = pow(10, *yr);
}
