/***************************************************************************
                          fitdlg.cpp  -  description
                             -------------------
    begin                : Tue May 25 1999
    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qlineedit.h>
#include <qpainter.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qtable.h>
#include <qtimer.h>
#include <kapp.h>
#include <kcolorbtn.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kseparator.h>
#include <kurl.h>
#include "arrayitem.h"
#include "choosefuncdlg.h"
#include "editparameterdlg.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "funitem.h"
#include "kpldoc.h"
#include "kpldoubleedit.h"
#include "kplgraph.h"
#include "kplspinbox.h"
#include "lmfit.h"
#include "residualsdlg.h"
#include "sourcewidget.h"
#include "utils.h"

CorrDlg::CorrDlg(QWidget* _parent, int np, const int* ip, const double* corr) :
  KDialog(_parent, 0, true)
{
  setPlainCaption(i18n("Parameter correlation coefficients"));
  QTable* table = new QTable(np, np, this);
  table->setMinimumSize(100, 40);
  QString s;
  for (int iy = 0; iy < np; iy++)
    table->verticalHeader()->setLabel(iy, s.sprintf("p%i", ip[iy]));
  for (int ix = 0; ix < np; ix++) {
    table->horizontalHeader()->setLabel(ix, s.sprintf("p%i", ip[ix]));
    for (int iy = 0; iy < np; iy++) {
      double temp = corr[np * iy + ix];
      if ((temp >= -1.0004) && (temp  <= 1.0004))
        s.setNum(temp, 'f', 3);
      else
        s = "NaN";
      table->setItem(iy, ix, new QTableItem(table, QTableItem::Never, s));
    }
    table->adjustColumn(ix);
  }
  table->adjustSize();
  (new QVBoxLayout(this, 0, 0))->addWidget(table);
}

CorrDlg::~CorrDlg()
{
}

ErrModDlg::ErrModDlg(QWidget* _parent, KplDoc* model,
                     Kpl::DataErrorStruct* err0) :
 KDialogBase(Plain, i18n("Error model function"), Help | Ok | Cancel, Ok,
            _parent, 0, true, true), m(model), err(err0)
{
  errt = new Kpl::DataErrorStruct(*err);
  QFrame* frame = plainPage();
  QGridLayout* grid = new QGridLayout(frame, 4, 2, 0, spacingHint());
  grid->addWidget(new QLabel(i18n("Library"), frame), 0, 0);
  QHBoxLayout* hbox = new QHBoxLayout();
  grid->addLayout(hbox, 0, 1);
  hbox->addWidget(fileName = new QLineEdit(err->errModPath, frame));
  QPushButton* b = new QPushButton("...", frame);
  b->setFixedWidth(28);
  hbox->addWidget(b);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  grid->addWidget(new QLabel(i18n("Function"), frame), 1, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 1, 1);
  hbox->addWidget(func = new QLineEdit(err->errModName, frame));
  hbox->addWidget(b = new QPushButton("...", frame));
  b->setFixedWidth(28);
  grid->addWidget(new QLabel(i18n("Argument"), frame), 2, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 2, 1);
  bg = new QButtonGroup();
  QRadioButton* r;
  hbox->addWidget(r = new QRadioButton(i18n("x column"), frame));
  bg->insert(r);
  hbox->addWidget(r = new QRadioButton(i18n("y column"), frame));
  bg->insert(r);
  ((QRadioButton*) bg->find(err->errModArg))->setChecked(true);
  hbox->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
  setHelp("ERRORMODEL");
  connect(b, SIGNAL(clicked()), SLOT(slotFunc()));
  grid->addWidget(new QLabel(i18n("Parameter"), frame), 3, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 3, 1);
  hbox->addWidget(b = new QPushButton(i18n("Edit"), frame));
  connect(b, SIGNAL(clicked()), SLOT(slotEditParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Load"), frame));
  connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Save"), frame));
  connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
}

ErrModDlg::~ErrModDlg()
{
  delete errt;
}

void ErrModDlg::slotFile()
{
  m->getFile(fileName);
}

void ErrModDlg::slotFunc()
{
  QFileInfo fi(fileName->text());
  QString path = fi.dirPath(true);
  QFile f(path + "/" + fi.baseName() + ".def");
  if (f.open(IO_ReadOnly)) {
    ChooseFuncDlg dlg(this, &f, func);
    dlg.exec();
  } else
    KMessageBox::error(this,
                       i18n("while trying to open module definition file"));
}

void ErrModDlg::slotGetParameter()
{
  m->getPar(this, errt->pErrMod);
}

void ErrModDlg::slotEditParameter()
{
  EditParameterDlg dlg(this, m, errt->pErrMod, fileName->text(), func->text());
  dlg.exec();
}

void ErrModDlg::slotSaveParameter()
{
  m->saveFunPar(this, errt->pErrMod);
}

void ErrModDlg::slotOk()
{
  errt->errModPath = fileName->text();
  errt->errModName = func->text();
  errt->errModArg = bg->id(bg->selected());
  lt_dlhandle hErrMod = 0;
  double (*fkt)(double, const double *);
  if (FunItem::getFuncAddr(errt->errModPath, errt->errModName, &hErrMod, &fkt))
    return;
  lt_dlclose(hErrMod);
  *err = *errt;
  accept();
}

FitDlg::FitDlg(QWidget* _parent, KplDoc* model, QList<ArrayItem>* ad0,
               QList<FunItem>* fd0, int mode) :
 KDialogBase(Plain, i18n("Parameter fit"),
             Help | Ok | Apply | Cancel | User1 | User2,
             Ok, _parent, 0, true, true, i18n("&Start"), i18n("&Residuals")),
 running(false), dlgMode(mode), m(model), ad(ad0), fd(fd0), sw(0)
{
  aut = new Kpl::AutoStruct(*m->options());
  config = KGlobal::config();
  err = new Kpl::DataErrorStruct[ad->count()];
  int i;
  QString s;
  ArrayItem* a;
  for (i = 0; i < (int) ad->count(); i++) {
    err[i] = aut->err;
    s.sprintf("DataErrors%i", i);
    if (config->hasGroup(s)) {
      config->setGroup(s);
      err[i].errModPath = config->readEntry("ErrorModelPath",
                                            aut->err.errModPath);
      err[i].errModName = config->readEntry("ErrorModelName",
                                            aut->err.errModName);
      QStringList list = config->readListEntry("ErrorModelParam", ' ');
      int cnt = list.count();
      for (int j = 0; j < KPL_NPMAX; j++)
        err[i].pErrMod[j] = (j < cnt) ? list[j].toDouble() : 0.0;
      a = ad->at(i);
      if (a->ie < a->ncols)
        err[i].fitErrCol = config->readBoolEntry("FitErrorColumn",
                                                 aut->err.fitErrCol);
      else
        err[i].fitErrCol = false;
      err[i].errModArg = config->readNumEntry("ErrorModelArgument",
                                              aut->err.errModArg);
    }
  }
  FunItem* ff = fd->first();
  for (i = 0; i < KPL_NPMAX; i++) {
    m->pLast[i] = ff->py[i];
    m->pErr[i] = 0.0;
  }
  lm = new LMFit(this);
  if (dlgMode & ShowDlg) {
    Utils::setSize(this, "FitDialog");
    QHBoxLayout* hbox = 0;
    QVBoxLayout* vbox;
    QFrame* frame = plainPage();
    if (m->options()->showSource) {
      hbox = new QHBoxLayout(frame, 11, spacingHint());
      vbox = new QVBoxLayout(hbox);
    } else
      vbox = new QVBoxLayout(frame, 11, spacingHint());
    QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Parameter"), frame);
    vbox->addWidget(g);
    QVBoxLayout* vbox2 = new QVBoxLayout(g->layout(), spacingHint());
    int nGr = 1 + (KPL_NPMAX - 1) / 10;
    QGridLayout* grid = new QGridLayout(vbox2, QMIN(KPL_NPMAX, 10),
                                        5 * nGr - 1);
    for (i = 1; i < nGr; i++)
      grid->addItem(new QSpacerItem(20, 10), 0, 4 * i);
    char frm = m->options()->format;
    int prec = m->options()->prec;
    int w = 0;
    QString s0 = m->number(0.0);
    for (i = 0; i < KPL_NPMAX; i++) {
      int ix = 5 * (i / 10);
      int iy = i % 10;
      grid->addWidget(enFit[i] = new QCheckBox(s.sprintf("p%i", i), g, "f"),
                      iy, ix);
      enFit[i]->setChecked(aut->fitPar[i]);
      grid->addWidget(par[i] = new KplDoubleEdit(m->pLast[i], g, frm, prec),
                      iy, ix + 1);
      if (!i)
        w = QMAX(80, par[0]->fontMetrics().width(m->number(-1.0e-123 / 3.0)) +
                                                 3);
      par[i]->setFixedWidth(w);
      grid->addWidget(new QLabel("\261", g), iy, ix + 2);
      grid->addWidget(epar[i] = new QLabel(s0, g), iy, ix + 3);
      epar[i]->setFrameStyle(QFrame::Panel | QFrame::Sunken);
      epar[i]->setFixedWidth(w);
    }
    QHBoxLayout* hbox2 = new QHBoxLayout(vbox2, spacingHint());
    hbox2->addWidget(nonLin = new QCheckBox(i18n("Nonlinear fit"), g));
    nonLin->setChecked(aut->fitNonLin);
    widgetList.append(nonLin);
    QPushButton* b = new QPushButton(i18n("Load"), g);
    hbox2->addWidget(b);
    connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
    widgetList.append(b);
    hbox2->addWidget(b = new QPushButton(i18n("Save"), g));
    connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
    widgetList.append(b);
    hbox2->addWidget(showCorr = new QPushButton(i18n("Correlations"), g));
    showCorr->setEnabled(false);
    connect(showCorr, SIGNAL(clicked()), SLOT(slotCorr()));
    g = new QGroupBox(0, Qt::Horizontal, i18n("Data errors"), frame);
    hbox2 = new QHBoxLayout(g->layout(), spacingHint());
    hbox2->addWidget(new QLabel(i18n("Array"), g));
    a = ad->at(0);
    hbox2->addWidget(sArr = new KplSpinBox(0, ad->count() - 1, 1, g));
    sArr->setValue(0);
    hbox2->addWidget(lArr = new QLabel(g));
    lArr->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox2->addWidget(errCol = new QCheckBox(i18n("Error column"), g));
    hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
    hbox2->addWidget(errMod = new QPushButton(i18n("Model"), g));
    connect(errMod, SIGNAL(clicked()), SLOT(slotErrMod()));
    vbox->addWidget(g);
    hbox2 = new QHBoxLayout(vbox, spacingHint());
    hbox2->addWidget(new QLabel(i18n("Maximum iterations"), frame));
    hbox2->addWidget(sItmax = new KplSpinBox(1, 200, 1, frame));
    sItmax->setValue(aut->fitMaxIt);
    widgetList.append(sItmax);
    hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
    hbox2->addWidget(new QLabel(i18n("Tolerance"), frame));
    hbox2->addWidget(eTol = new KplDoubleEdit(aut->fitTol, 0.0, 1.0, frame,
                                              'g', 6));
    widgetList.append(eTol);
    vbox->addWidget(results = new QListBox(frame));
    results->setMinimumHeight(40);
    widgetList.append(actionButton(Ok));
    widgetList.append(actionButton(Apply));
    widgetList.append(actionButton(User2));
    enableButton(User2, false);
    if (m->options()->showSource) {
      hbox->addWidget(sw = new SourceWidget(frame, 0), 1);
      sw->setMinimumWidth(200);
    }
    setHelp("SEC-FIT");
    connect(nonLin, SIGNAL(toggled(bool)), SLOT(enableNonLin(bool)));
    connect(errCol, SIGNAL(toggled(bool)), SLOT(errColToggled(bool)));
    connect(sArr, SIGNAL(valueChanged(int)), SLOT(updArray(int)));
    connect(lm, SIGNAL(updateMessage(const QString &)),
            SLOT(slotMessage(const QString &)));
    enableNonLin(aut->fitNonLin);
    updArray(0);
    if (m->options()->showSource)
      QTimer::singleShot(0, sw, SLOT(slotHighlight()));
  }
}

FitDlg::~FitDlg()
{
  if (dlgMode & ShowDlg)
    Utils::saveSize(this, "FitDialog");
  delete [] err;
  delete aut;
  delete lm;
}

void FitDlg::getValues(bool ok)
{
  int i, j;
  if (dlgMode & ShowDlg) {
    aut->fitNonLin = nonLin->isChecked();
    for (i = 0; i < KPL_NPMAX; i++)
      aut->fitPar[i] = enFit[i]->isChecked();
    if (aut->fitNonLin) {
      aut->fitMaxIt = sItmax->interpretedValue();
      aut->fitTol = eTol->value();
    }
    m->setOptions(aut);
    QString s;
    for (j = 0; j < (int) fd->count(); j++) {
      s.sprintf("DataErrors%i", j);
      config->setGroup(s);
      config->writeEntry("FitErrorColumn", err[j].fitErrCol);
      config->writeEntry("ErrorModelPath", err[j].errModPath);
      config->writeEntry("ErrorModelName", err[j].errModName);
      QStrList s1;
      for (i = 0; i < KPL_NPMAX; i++)
        s1.insert(i, m->number(err[j].pErrMod[i]));
      config->writeEntry("ErrorModelParam", s1, ' ');
      config->writeEntry("ErrorModelArgument", err[j].errModArg);
    }
  }
  for (j = 0; j < (int) fd->count(); j++)
    for (i = 0; i < KPL_NPMAX; i++)
      fd->at(j)->py[i] = m->pLast[i];
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
}

void FitDlg::slotMessage(const QString& msg)
{
  results->insertItem(msg);
  results->setTopItem(results->count() - 1);
  kapp->processEvents();
}

void FitDlg::slotGetParameter()
{
  m->getPar(this, m->pLast);
  for (int i = 0; i < KPL_NPMAX; i++) {
    par[i]->setValue(m->pLast[i]);
    m->pErr[i] = 0.0;
    epar[i]->setText("0");
  }
}

void FitDlg::slotSaveParameter()
{
  KURL url;
  if (m->getWriteURL(this, url, "*.par\n*"))
    m->saveParameter(this, url, m->pLast, m->pErr);
}

void FitDlg::errColToggled(bool state)
{
  errMod->setEnabled(!state);
  err[sArr->interpretedValue()].fitErrCol = state;
}

void FitDlg::enableNonLin(bool on)
{
  eTol->setEnabled(on);
  sItmax->setEnabled(on);
}

void FitDlg::updArray(int i)
{
  ArrayItem* a = ad->at(i);
  lArr->setText(a->url.fileName() + " " + QString::number(a->ix) + " " +
                QString::number(a->iy) + " " + QString::number(a->ie));
  errCol->setChecked(err[i].fitErrCol);
  errCol->setEnabled(a->ie < a->ncols);
  if (m->options()->showSource) {
    FunItem* ff = fd->at(i);
    QFileInfo fi(ff->pathy.path());
#if KDE_VERSION_MAJOR > 2
    QRegExp rx2("\\S+__FdPCd");
    QRegExp rx3("_Z\\d+\\S+dPKd");
    int pp2 = rx2.search(ff->namey);
    int pp3 = rx3.search(ff->namey);
#else
    QRegExp rx2("[^\\s]+__FdPCd");
    QRegExp rx3("_Z\\d+[^\\s]+dPKd");
    int l2, l3;
    int pp2 = rx2.match(ff->namey, 0, &l2);
    int pp3 = rx3.match(ff->namey, 0, &l3);
#endif
    QString s = fi.dirPath(true) + "/" + fi.baseName();
    bool guessC = pp2 && pp3;
    QFile f(s + (guessC ? ".c" : ".cpp"));
    bool open = f.open(IO_ReadOnly);
    if (!open) {
      f.setName(s + (guessC ? ".cpp" : ".c"));
      open = f.open(IO_ReadOnly);
    }
    if (open) {
      sw->fill(&f);
      if (!pp2)
#if KDE_VERSION_MAJOR > 2
        sw->highlight(ff->namey.left(rx2.matchedLength() - 7));
      else
        if (!pp3)
          sw->highlight(ff->namey.mid(3, rx3.matchedLength() - 7));
#else
        sw->highlight(ff->namey.left(l2 - 7));
      else
        if (!pp3)
          sw->highlight(ff->namey.mid(3, l3 - 7));
#endif
        else
          sw->highlight(ff->namey);
    }
  }
}

void FitDlg::slotOk()
{
  getValues(true);
}

void FitDlg::slotCancel()
{
  if (running)
    lm->userBreak = true;
  else
    reject();
}

void FitDlg::slotApply()
{
  getValues(false);
}

void FitDlg::slotUser1()
{
  fit();
}

void FitDlg::fit()
{
  for (FunItem* f = fd->first(); f; f = fd->next())
    if (!f->fkty) {
      KMessageBox::error(this, i18n("no function address!"));
      return;
    }
  bool nLin;
  if (dlgMode & ShowDlg) {
    nLin = nonLin->isChecked();
    if (nLin) {
      aut->fitTol = eTol->value();
      aut->fitMaxIt = sItmax->interpretedValue();
    }
    showCorr->setEnabled(false);
    results->clear();
  } else
    nLin = aut->fitNonLin;
  np = 0;
  int i;
  for (i = 0; i < KPL_NPMAX; i++) {
    if (dlgMode & ShowDlg) {
      m->pLast[i] = par[i]->value();
      aut->fitPar[i] = enFit[i]->isChecked();
    }
    if (aut->fitPar[i])
      np++;
  }
  if (!np) {
    KMessageBox::sorry(this, i18n("No parameter to fit!"));
    return;
  }
  int n = 0;
  ArrayItem* a;
  for (a = ad->first(); a; a = ad->next())
    n += a->n;
  fvec.resize(n);
  lm->sig.resize(n);
  int j = 0;
  int k;
  for (k = 0; k < (int) ad->count(); k++) {
    a = ad->at(k);
    if (err[k].fitErrCol) {
      for (i = 0; i < a->n; i++)
        lm->sig[j++] = a->x[a->ie][a->istart + i];
    } else {
      double (*fktErrMod)(double, const double*);
      lt_dlhandle hErrMod;
      if (FunItem::getFuncAddr(err[k].errModPath, err[k].errModName, &hErrMod,
                               &fktErrMod))
        return;
      for (i = 0; i < a->n; i++)
        lm->sig[j++] = fktErrMod(a->x[err[k].errModArg ? a->iy : a->ix]
                                     [a->istart + i], err[k].pErrMod);
      lt_dlclose(hErrMod);
    }
  }
  lm->ad = ad;
  lm->fd = fd;
  lm->bFit = aut->fitPar;
  lm->p = m->pLast;
  int wadim = (6 + np + n) * np + n;
  double* wa = new double[wadim];
  memset(wa, 0, wadim * sizeof(double));
  double* guess = wa + 3 * np + n;
  double* diag = guess + np;
  double* fjac = diag + np;
  double* qtf = fjac + n * np;
  double* alpha = qtf + np;
  j = 0;
  for (i = 0; i < KPL_NPMAX; i++)
    if (aut->fitPar[i])
      guess[j++] = m->pLast[i];
  if (dlgMode & ShowDlg) {
    for (QWidget* w = widgetList.first(); w; w = widgetList.next())
      w->setEnabled(false);
    for (i = 0; i < KPL_NPMAX; i++) {
      par[i]->setEnabled(false);
      enFit[i]->setEnabled(false);
    }
  }
  kapp->processEvents();
  bool aborted = lm->userBreak = false;
  running = true;
  QString s;
  if (nLin) {
    int info = 0;
    int nfev;
    lm->lmdif(n, np, guess, fvec.data(), aut->fitTol, aut->fitTol, 0.0,
              (np + 1) * aut->fitMaxIt, 0.0, diag, 1, 100.0, 1, &info, &nfev,
              fjac, n, ip, qtf, wa, wa + np, wa + 2 * np, wa + 3 * np);
    if (info >= 8)
      info = 4;
    if (dlgMode & ShowDlg) {
      switch (info) {
        case 0:
          s = i18n("Improper input parameters");
          break;
        case 1:
          s = i18n("Relative error in chi-square is at most tolerance");
          break;
        case 2:
          s = i18n("Relative error in parameters is at most tolerance");
          break;
        case 3:
          s = i18n("Relative error in chi-square and in parameters is at most "
                   "tolerance");
          break;
        case 4:
          s = i18n("Function vector is orthogonal to the columns of "
                   "the jacobian");
          break;
        case 5:
          s = i18n("Number of iterations has reached or exceeded maximum");
          break;
        case 6:
          s = i18n("Tolerance too small. No further reduction of "
                   "chi-square possible");
          break;
        case 7:
          s = i18n("Tolerance too small. No further improvement of "
                   "parameters possible");
          break;
        default:
          s = i18n("Terminated by user");
      }
      slotMessage(s);
    }
  } else {
    aborted = lm->linFit(n, np, guess, fvec.data(), fjac, ip, qtf, wa, wa + np,
                         wa + 2 * np, wa + 3 * np);
  }
  running = false;
  if (!aborted) {
    j = 0;
    for (i = 0; i < KPL_NPMAX; i++)
      if (aut->fitPar[i])
        m->pLast[i] = guess[j++];
    double chi = LMFit::enorm(n, fvec);
    m->chisq = chi * chi;
    for (i = 0; i < np; i++) {
      for (j = 0; j < np; j++) {
        alpha[ip[i] + ip[j] * np] = 0.0;
        for (k = 0; k <= j; k++)
          if (k <= i)
            alpha[ip[i] + ip[j] * np] += fjac[i * n + k] * fjac[j * n + k];
      }
    }
    LMFit::minv(alpha, corr, np, wa, ip);
    int ij = 0;
    for (i = 0; i < np; i++) {
      diag[i] = sqrt(corr[ij]);
      ij += np + 1;
    }
    k = 0;
    for (i = 0; i < np; i++)
      for (j = 0; j < np; j++) {
        if (double temp = diag[i] * diag[j])
          corr[k] /= temp;
        k++;
      }
    int ny = n - np;
    double q = 0.0;
    if (ny) {
      q = LMFit::igamc(0.5 * ny, 0.5 * m->chisq);
      double temp = sqrt(m->chisq / ny);
      for (i = 0; i < np; i++)
        diag[i] *= temp;
    }
    k = 0;
    for (i = 0; i < KPL_NPMAX; i++) {
      if (aut->fitPar[i]) {
        ip[k] = i;
        m->pErr[i] = diag[k++];
      } else
        m->pErr[i] = 0.0;
      if (dlgMode & ShowDlg) {
        par[i]->setValue(m->pLast[i]);
        epar[i]->setText(m->number(m->pErr[i]));
      }
      if (dlgMode & Follow)
        aut->pFit[i] = m->pLast[i];
    }
    double avgErr = 0.0;
    for (i = 0; i < n; i++) {
      double temp = fvec[i] * lm->sig[i];
      avgErr += temp * temp;
    }
    avgErr = sqrt(avgErr / n);
    if (dlgMode & ShowDlg)
      slotMessage(s.sprintf(i18n("Average error = %.3g    ny = %i    "
                                 "Significance Q = %.3g"),
                            avgErr, ny, q));
    if (dlgMode & SavePar) {
      KURL url = m->URL();
      QFileInfo fi(url.path());
      url.setPath(fi.dirPath(true) + "/" + fi.baseName() + ".par");
      m->saveParameter(this, url, m->pLast, m->pErr);
    }
  }
  if (dlgMode & ShowDlg) {
    showCorr->setEnabled(true);
    for (QWidget* w = widgetList.first(); w; w = widgetList.next())
      w->setEnabled(true);
    enableNonLin(nonLin->isChecked());
    for (i = 0; i < KPL_NPMAX; i++) {
      par[i]->setEnabled(true);
      enFit[i]->setEnabled(true);
    }
  }
  delete [] wa;
}

void FitDlg::slotErrMod()
{
  ErrModDlg dlg(this, m, &err[sArr->interpretedValue()]);
  dlg.exec();
}

void FitDlg::slotUser2()
{
  QList<KplItem> items;
  items.setAutoDelete(true);
  FrameItem* fd = new FrameItem();
  fd->gridmode = KplGraph::GridWithLabels;
  fd->sx = "x";
  fd->sy = i18n("Residuals");
  items.append(fd);
  int j = 0;
  for (ArrayItem* a = ad->first(); a; a = ad->next()) {
    ArrayItem* ar = new ArrayItem(true, false, a->symb, "0", 1.0, 1.0, 0, 1, 2,
                                  0, a->n, 1, "", true);
    ar->color = a->color;
    for (int i = 0; i < a->n; i++) {
      ar->x[0][i] = a->x[a->ix][a->istart + i];
      ar->x[1][i] = fvec[j] * lm->sig[j];
      ar->x[2][i] = lm->sig[j++];
    }
    items.append(ar);
  }
  double fxt, fyt;
  double fx = 1.0;
  double fy = 1.0;
  int iext, ieyt;
  fd->iex = fd->iey = 999;
  for (KplItem* itm = items.at(1); itm; itm = items.next()) {
    iext = fd->iex;
    ieyt = fd->iey;
    itm->expoItem(&iext, &ieyt, &fxt, &fyt);
    if (iext < fd->iex) {
      fd->iex = iext;
      fx = fxt;
    }
    if (ieyt < fd->iey) {
      fd->iey = ieyt;
      fy = fyt;
    }
  }
  double xmin = 1.0e300;
  double ymin = xmin;
  double xmax = -xmin;
  double ymax = xmax;
  for (KplItem *itm = items.at(1); itm; itm = items.next()) {
    fd->xmi = xmin;
    fd->ymi = ymin;
    fd->xma = xmax;
    fd->yma = ymax;
    itm->minMax(&fd->xmi, &fd->xma, &fd->ymi, &fd->yma);
    xmin = QMIN(xmin, fd->xmi);
    xmax = QMAX(xmax, fd->xma);
    ymin = QMIN(ymin, fd->ymi);
    ymax = QMAX(ymax, fd->yma);
  }
  Utils::autoSc(&fd->xmi, &fd->xma, &fd->xtic, &fd->mticx, &fd->ndigx,
                xmin, xmax, fx, 0.2, false);
  Utils::autoSc(&fd->ymi, &fd->yma, &fd->ytic, &fd->mticy, &fd->ndigy,
                ymin, ymax, fy, 0.3, false);
  for (KplItem* itm = items.at(1); itm; itm = items.next())
    itm->normalize(fx, fy);
  ResidualsDlg dlg(this, m, &items);
  dlg.exec();
}

void FitDlg::slotCorr()
{
  CorrDlg dlg(this, np, ip, corr);
  dlg.exec();
}
