/***************************************************************************
                          knfiltermanager.cpp  -  description
                             -------------------
   
    copyright            : (C) 1999 by Christian Thurner
    email                : cthurner@freepage.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include <qlayout.h>
#include <ktabctl.h>
#include <stdlib.h>

#include "knfiltermanager.h"
#include "knfilterdialog.h"
#include "utilities.h"
#include "knglobals.h"


KNFilterManager::KNFilterManager()
{
	nextID=0;
	currFilter=0;
	menuOrder=0;
	dlg=0;
	fList=new QList<KNMessageFilter>;
	fList->setAutoDelete(true);
	menu=new QPopupMenu();
	menu->setCheckable(true);
	connect(menu, SIGNAL(activated(int)), this,
		SLOT(slotMenuActivated(int)));
	
	loadFilters();
}



KNFilterManager::~KNFilterManager()
{
	delete dlg;
	delete[] menuOrder;
}



void KNFilterManager::loadFilters()
{
  QString path;
	QDir d=QDir::home();
	d.cd(".knode/Filters");
	
	QStrList *entries=(QStrList*) d.entryList("*.fltr");
	
	for(char *var=entries->first(); var; var=entries->next()) {
		path=d.absPath()+"/";
		path+=var;
		KSimpleConfig c(path);
		KNMessageFilter *f=new KNMessageFilter();
		f->loadInfo(c);
		if(f->mId > nextID) nextID=f->mId;
		addFilter(f);
		debug("KNFilterManager: filter-info loaded \"%s\" ", path.data());
	}
 	nextID++;
 	
 	path=knDir()+"Filters/menu";
 	
 	QFile f(path);
 	
 	if(f.open(IO_ReadOnly)) {
 		int cnt;
 		f.readBlock((char*) &cnt, sizeof(int));
 		
 		menuOrder=new int[cnt+1];
 		menuOrder[0]=cnt;
 		
 		f.readBlock((char*) &menuOrder[1], cnt*sizeof(int));
 	
 		f.close();
 	}
 	
 	updateMenu();
 	
}



void KNFilterManager::saveMenuOrder()
{
	QString path=knDir();
	path+="Filters/menu";
	
	QFile f(path);
	
	if(f.open(IO_WriteOnly))
		f.writeBlock((char*) menuOrder, (menuOrder[0]+1)*sizeof(int));
	f.close();	
}



void KNFilterManager::addFilter(KNMessageFilter *f)
{
	if(f->mId==-1) f->mId=nextID++;
	fList->append(f);
}



KNMessageFilter* KNFilterManager::byID(int id)
{
	KNMessageFilter *ret=0;
	
	for(ret=fList->first(); ret; ret=fList->next())
		if(ret->mId==id) break;
		
	return ret;	

}



void KNFilterManager::editFilter(KNMessageFilter *f)
{
	if(!f->mLoaded && f->mId!=-1) f->load();
	KNFilterDialog *fdlg=new KNFilterDialog(0,0,f);
	connect(fdlg, SIGNAL(editDone(KNFilterDialog*)),
		this, SLOT(slotEditDone(KNFilterDialog*)));
	
	fdlg->show();
		
		
}



bool KNFilterManager::nameIsOK(KNMessageFilter *f)
{
	KNMessageFilter *var=fList->first();
	bool found=false;
	
	while(var && !found) {
		if(var!=f) found=(f->mName==var->mName);
		var=fList->next();
	}
	
	return (!found);
}



KNMessageFilter* KNFilterManager::setFilter(const int id)
{
	
	KNMessageFilter *bak=currFilter;
	
	currFilter=byID(id);	
	
	if(currFilter) {
		if(bak) menu->setItemChecked(bak->mId, false);
		menu->setItemChecked(currFilter->mId, true);
	}
	else currFilter=bak;
	
	return currFilter;	
	
}




void KNFilterManager:: showDialog()
{
	dlg=new KNFilterListDialog(xTop);
	
	connect(dlg, SIGNAL(newFilter()), this, SLOT(slotNew()));
	connect(dlg, SIGNAL(delFilter(KNMessageFilter*)),
		this, SLOT(slotDel(KNMessageFilter*)));
	connect(dlg, SIGNAL(editFilter(KNMessageFilter*)),
		this, SLOT(slotEdit(KNMessageFilter*)));
	connect(dlg, SIGNAL(dialogDone()), this, SLOT(slotDialogDone()));
		
	for(KNMessageFilter *f=fList->first(); f; f=fList->next())
	  dlg->addItem(f);	
	
	if(menuOrder)
		for(int i=1; i<menuOrder[0]+1; i++)
			if(menuOrder[i]!=-1) dlg->addMenuItem(byID(menuOrder[i]));
			else dlg->addMenuItem(0);
			
	dlg->show();
		
}



void KNFilterManager::updateMenu()
{
	if(!menuOrder) return;
	menu->clear();
	KNMessageFilter *f=0;
	
	for(int i=1; i<menuOrder[0]+1; i++)
		if(menuOrder[i]!=-1) {
			f=byID(menuOrder[i]);
			menu->insertItem(f->name(), f->id());
		}
		else menu->insertSeparator();
	
	if(currFilter) menu->setItemChecked(currFilter->id(), true);
	
}



void KNFilterManager::slotNew()
{
	KNMessageFilter *f=new KNMessageFilter();
	editFilter(f);
}



void KNFilterManager::slotDel(KNMessageFilter *f)
{
	if(MBox(yn, i18n("Do you really want to delete this filter?"))==1) {  	
  	QString cmd="rm "+f->path();
	
	  if(fList->removeRef(f)) {
	  	system(cmd);
	  	if(dlg) {
	  		dlg->removeItem(f);
	  		dlg->removeMenuItem(f);
	  	}
	  	if(currFilter==f) currFilter=0;
	  }
		else MBox(err, i18n("ERROR: Cannot delete the filter !!"));
	}
}




void KNFilterManager::slotEdit(KNMessageFilter *f)
{
	editFilter(f);
}



void KNFilterManager::slotEditDone(KNFilterDialog *d)
{
	KNMessageFilter *f=d->filter();
	
	if(!nameIsOK(f)) {
		QString msg;
		msg.sprintf(i18n("There is already a filter named \"%s\" !\nPlease choose a new name."), f->mName.data());
		MBox(inf, msg);
		return;
	}
	
		
	if(f->mId==-1) {
		addFilter(f);
		f->mLoaded=true;
		if(dlg) {
			dlg->addItem(f);
			if(f->isEnabled()) dlg->addMenuItem(f);
		}
	}
	else if(dlg) {
		if(f->isEnabled()) dlg->addMenuItem(f);
		else dlg->removeMenuItem(f);
		dlg->updateItem(f);
	}
		
	
	f->save();	
		
	delete d;
					
}



void KNFilterManager::slotDialogDone()
{
	int *bak=menuOrder;
	
	menuOrder=dlg->menuOrder();
	if(menuOrder) delete[] bak;
	else menuOrder=bak;
		
	
	if(currFilter)
	 if(!currFilter->isEnabled()) currFilter=0;
	
	updateMenu();
	saveMenuOrder();
	emit filterChanged(currFilter);
		

	delete dlg;
	dlg=0;
}



void KNFilterManager::slotMenuActivated(int id)
{
	KNMessageFilter *f=setFilter(id);
	
	if(f) emit filterChanged(f);
	else MBox(err, i18n("ERROR : no such filter !!"));

}






//==============================================================================


KNFilterListDialog::KNFilterListDialog(QWidget *parent=0, const char *name=0) :
	QSemiModal(parent, name, true)
{
	
	setCaption(i18n("all filters"));
	
	
	KTabCtl *tabs=new KTabCtl(this);
	QPushButton *helpBtn=new QPushButton(i18n("Help"), this);
	helpBtn->setEnabled(false);
	QPushButton *closeBtn=new QPushButton(i18n("Close"), this);
	SIZE(closeBtn); SIZE(helpBtn);
		
	
	
  QWidget *w1=new QWidget(tabs);
	
	list1=new KNListBox(w1);
	
	QButtonGroup *btns1=new QButtonGroup(w1);
	
	QPushButton *newBtn=new QPushButton(i18n("new"), btns1);
	QPushButton *delBtn=new QPushButton(i18n("delete"), btns1);
	QPushButton *editBtn=new QPushButton(i18n("edit"), btns1);
	SIZE(newBtn); SIZE(delBtn); SIZE(editBtn);
	
	QGridLayout *w1L=new QGridLayout(w1, 2,2,10);
	QVBoxLayout *btnL1=new QVBoxLayout(btns1,5);
		
	btnL1->addWidget(newBtn);
	btnL1->addWidget(delBtn);
	btnL1->addWidget(editBtn);
	
	w1L->addWidget(btns1, 0,1);
	w1L->addMultiCellWidget(list1, 0,1,0,0);
	w1L->setColStretch(0,1);
	w1L->setRowStretch(1,1);
	w1L->addColSpacing(0,200);
	
	
	
	
	
	QWidget *w2=new QWidget(tabs);
	
	list2=new KNListBox(w2);
	
	QButtonGroup *btns2=new QButtonGroup(w2);
	QPushButton *upBtn=new QPushButton(i18n("up"), btns2);
	QPushButton *downBtn=new QPushButton(i18n("down"), btns2);
	QPushButton *insSepBtn=new QPushButton(i18n("insert\nSeparator"), btns2);
	QPushButton *remSepBtn=new QPushButton(i18n("remove\nSeparator"), btns2);
  SIZE(upBtn); SIZE(downBtn); SIZE(insSepBtn); SIZE(remSepBtn);
	
	QGridLayout *w2L=new QGridLayout(w2, 2,2,10);
	QVBoxLayout *btnL2=new QVBoxLayout(btns2,5);
	
	btnL2->addWidget(upBtn);
	btnL2->addWidget(downBtn);
	btnL2->addSpacing(35);
	btnL2->addWidget(insSepBtn);
	btnL2->addWidget(remSepBtn);
	
	w2L->addWidget(btns2, 0,1);
	w2L->addMultiCellWidget(list2, 0,1,0,0);
	w2L->setColStretch(0,1);
	w2L->setRowStretch(1,1);
	w2L->addColSpacing(0,200);
		
	
	
	tabs->addTab(w1, i18n("Filters"));
	tabs->addTab(w2, i18n("Menu"));

		
	QVBoxLayout *topL=new QVBoxLayout(this, 10);
	QHBoxLayout *chBtnL=new QHBoxLayout(5);
	
	topL->addWidget(tabs, 1);
	topL->addLayout(chBtnL);
	chBtnL->addWidget(helpBtn);
	chBtnL->addStretch(1);
	chBtnL->addWidget(closeBtn);
  	
	topL->activate();
	
	
	connect(closeBtn, SIGNAL(clicked()), this, SLOT(slotCloseBtn()));
	connect(helpBtn, SIGNAL(clicked()), this, SLOT(slotHelpBtn()));
	
	connect(btns1, SIGNAL(clicked(int)), this, SLOT(slotButtons1(int)));
	connect(btns2, SIGNAL(clicked(int)), this, SLOT(slotButtons2(int)));
	
	connect(list1, SIGNAL(selected(int)), this, SLOT(slotList1Selected(int)));
	
}



KNFilterListDialog::~KNFilterListDialog()
{
}



void KNFilterListDialog::addItem(KNMessageFilter *f)
{
	QPixmap pm;
	if(f->isEnabled()) pm=ICON("fltrblue.xpm");
	else pm=ICON("fltrgrey.xpm");
	
	KNLBoxItem *it=new KNLBoxItem(f->name(),f,&pm);
	list1->inSort(it);
	
}



void KNFilterListDialog::addMenuItem(KNMessageFilter *f)
{
	if(f) {
		if(findItem(list2, f)==-1)
			list2->insertItem(new KNLBoxItem(f->name(), f,0));
	}
	
	else  list2->insertItem(new KNLBoxItem("===", 0,0));
}



void KNFilterListDialog::removeItem(KNMessageFilter *f)
{
	int i=findItem(list1, f);
  if(i!=-1) list1->removeItem(i);
}



void KNFilterListDialog::removeMenuItem(KNMessageFilter *f)
{
	int i=findItem(list2, f);
	if(i!=-1) list2->removeItem(i);		
}



void KNFilterListDialog::updateItem(KNMessageFilter *f)
{
	int i=findItem(list1, f);
	
	if(i!=-1) {
		QPixmap pm;
		if(f->isEnabled()) pm=ICON("fltrblue.xpm");	
	  else pm=ICON("fltrgrey.xpm");
	  list1->changeItem(new KNLBoxItem(f->name(),f,&pm), i);
	}

	if(f->isEnabled())	
		list2->changeItem(new KNLBoxItem(f->name(),f,0), findItem(list2, f));			
}



int KNFilterListDialog::findItem(KNListBox *l, KNMessageFilter *f)
{
	int idx=0;
	bool found=false;
	
	while(!found && idx < (int) l->count()) {
		found=(l->itemAt(idx)->data()==f);
		if(!found) idx++;
	}
	
	
	if(found) return idx;
	else return -1;	
	
}



void KNFilterListDialog::slotButtons1(int id)
{
	
	int c=list1->currentItem();
	KNMessageFilter *f=0;
	
	if(c!=-1) f=(KNMessageFilter*) list1->itemAt(c)->data();	

	
	switch(id) {
		
		case 0: emit newFilter(); 				break;
		case 1: if(f) emit delFilter(f); 	break;
		case 2: if(f) emit editFilter(f); break;
	};
	 	
}



void KNFilterListDialog::slotButtons2(int id)
{
	int c=list2->currentItem();
	KNMessageFilter *f=0;
	
	if(c!=-1) f=(KNMessageFilter*) list2->itemAt(c)->data();
	
	switch(id) {
	
		case 0:
			if(c!=0) {
				if(f) list2->insertItem(new KNLBoxItem(f->name(), f,0), c-1);
				else  list2->insertItem(new KNLBoxItem("===", 0,0), c-1);
				list2->removeItem(c+1);
				list2->setCurrentItem(c-1);
			}
		break;
		
		case 1:
			if(c!= (int) list2->count()-1) {
				if(f) list2->insertItem(new KNLBoxItem(f->name(), f,0), c+2);
				else  list2->insertItem(new KNLBoxItem("===", 0,0), c+2);
				list2->removeItem(c);
				list2->setCurrentItem(c+1);
			}
		break;
		
		case 2:
			if(c!=-1) list2->insertItem(new KNLBoxItem("===", 0,0), c);
		break;
		
		case 3:
			if(c!=-1 && f==0) list2->removeItem(c);
		break;			
	}
				 		

}



int* KNFilterListDialog::menuOrder()
{
	KNMessageFilter *f;
	int *ret=new int[list2->count()+1];
	
	ret[0]=list2->count();
	
	for(uint i=0; i<list2->count(); i++) {
		f=(KNMessageFilter*) list2->itemAt(i)->data();
		
		if(f) ret[i+1]=f->id();
		else ret[i+1]=-1;
	}

 return ret;
	
}



void KNFilterListDialog::slotList1Selected(int id)
{
	emit editFilter((KNMessageFilter*) list1->itemAt(id)->data());	
}



void KNFilterListDialog::slotCloseBtn()
{
	emit dialogDone();
}



void KNFilterListDialog::slotHelpBtn()
{
}



























































