// *** THIS FILE IS GENERATED - DO NOT EDIT ***
// See layer_chassis_generator.py for modifications

/***************************************************************************
 *
 * Copyright (c) 2015-2024 The Khronos Group Inc.
 * Copyright (c) 2015-2024 Valve Corporation
 * Copyright (c) 2015-2024 LunarG, Inc.
 * Copyright (c) 2015-2024 Google Inc.
 * Copyright (c) 2023-2024 RasterGrid Kft.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ****************************************************************************/

// NOLINTBEGIN

#include <array>
#include <cstring>
#include <mutex>

#include "chassis.h"
#include "layer_options.h"
#include "layer_chassis_dispatch.h"
#include "state_tracker/descriptor_sets.h"
#include "chassis/chassis_modification_state.h"

#include "profiling/profiling.h"

thread_local WriteLockGuard* ValidationObject::record_guard{};

small_unordered_map<void*, ValidationObject*, 2> layer_data_map;

// Global unique object identifier.
std::atomic<uint64_t> global_unique_id(1ULL);
// Map uniqueID to actual object handle. Accesses to the map itself are
// internally synchronized.
vvl::concurrent_unordered_map<uint64_t, uint64_t, 4, HashedUint64> unique_id_mapping;

// State we track in order to populate HandleData for things such as ignored pointers
static vvl::unordered_map<VkCommandBuffer, VkCommandPool> secondary_cb_map{};
static std::shared_mutex secondary_cb_map_mutex;

bool wrap_handles = true;

#define OBJECT_LAYER_DESCRIPTION "khronos_validation"

// Include layer validation object definitions
#include "thread_tracker/thread_safety_validation.h"
#include "stateless/stateless_validation.h"
#include "object_tracker/object_lifetime_validation.h"
#include "core_checks/core_validation.h"
#include "best_practices/best_practices_validation.h"
#include "gpu/core/gpuav.h"
#include "gpu/debug_printf/debug_printf.h"
#include "sync/sync_validation.h"

// This header file must be included after the above validation object class definitions
#include "chassis_dispatch_helper.h"

// Extension exposed by the validation layer
static constexpr std::array<VkExtensionProperties, 4> kInstanceExtensions = {
    VkExtensionProperties{VK_EXT_DEBUG_REPORT_EXTENSION_NAME, VK_EXT_DEBUG_REPORT_SPEC_VERSION},
    VkExtensionProperties{VK_EXT_DEBUG_UTILS_EXTENSION_NAME, VK_EXT_DEBUG_UTILS_SPEC_VERSION},
    VkExtensionProperties{VK_EXT_VALIDATION_FEATURES_EXTENSION_NAME, VK_EXT_VALIDATION_FEATURES_SPEC_VERSION},
    VkExtensionProperties{VK_EXT_LAYER_SETTINGS_EXTENSION_NAME, VK_EXT_LAYER_SETTINGS_SPEC_VERSION},
};
static constexpr std::array<VkExtensionProperties, 3> kDeviceExtensions = {
    VkExtensionProperties{VK_EXT_VALIDATION_CACHE_EXTENSION_NAME, VK_EXT_VALIDATION_CACHE_SPEC_VERSION},
    VkExtensionProperties{VK_EXT_DEBUG_MARKER_EXTENSION_NAME, VK_EXT_DEBUG_MARKER_SPEC_VERSION},
    VkExtensionProperties{VK_EXT_TOOLING_INFO_EXTENSION_NAME, VK_EXT_TOOLING_INFO_SPEC_VERSION},
};

// Layer registration code
static std::vector<ValidationObject*> CreateObjectDispatch(const CHECK_ENABLED& enables, const CHECK_DISABLED& disables) {
    std::vector<ValidationObject*> object_dispatch{};

    // Add VOs to dispatch vector. Order here will be the validation dispatch order!

    if (!disables[thread_safety]) {
        object_dispatch.emplace_back(new ThreadSafety(nullptr));
    }
    if (!disables[stateless_checks]) {
        object_dispatch.emplace_back(new StatelessValidation);
    }
    if (!disables[object_tracking]) {
        object_dispatch.emplace_back(new ObjectLifetimes);
    }
    if (!disables[core_checks]) {
        object_dispatch.emplace_back(new CoreChecks);
    }
    if (enables[best_practices]) {
        object_dispatch.emplace_back(new BestPractices);
    }
    if (enables[gpu_validation]) {
        object_dispatch.emplace_back(new gpuav::Validator);
    }
    if (enables[debug_printf_validation]) {
        object_dispatch.emplace_back(new debug_printf::Validator);
    }
    if (enables[sync_validation]) {
        object_dispatch.emplace_back(new SyncValidator);
    }
    return object_dispatch;
}

static void InitDeviceObjectDispatch(ValidationObject* instance_interceptor, ValidationObject* device_interceptor) {
    auto disables = instance_interceptor->disabled;
    auto enables = instance_interceptor->enabled;

    // Note that this DEFINES THE ORDER IN WHICH THE LAYER VALIDATION OBJECTS ARE CALLED

    if (!disables[thread_safety]) {
        device_interceptor->object_dispatch.emplace_back(
            new ThreadSafety(instance_interceptor->GetValidationObject<ThreadSafety>()));
    }
    if (!disables[stateless_checks]) {
        device_interceptor->object_dispatch.emplace_back(new StatelessValidation);
    }
    if (!disables[object_tracking]) {
        device_interceptor->object_dispatch.emplace_back(new ObjectLifetimes);
    }
    if (!disables[core_checks]) {
        device_interceptor->object_dispatch.emplace_back(new CoreChecks);
    }
    if (enables[best_practices]) {
        device_interceptor->object_dispatch.emplace_back(new BestPractices);
    }
    if (enables[gpu_validation]) {
        device_interceptor->object_dispatch.emplace_back(new gpuav::Validator);
    }
    if (enables[debug_printf_validation]) {
        device_interceptor->object_dispatch.emplace_back(new debug_printf::Validator);
    }
    if (enables[sync_validation]) {
        device_interceptor->object_dispatch.emplace_back(new SyncValidator);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceFeatures2Helper(VkPhysicalDevice physicalDevice,
                                                                VkPhysicalDeviceFeatures2* pFeatures) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceFeatures2(physicalDevice, pFeatures);
    } else {
        return DispatchGetPhysicalDeviceFeatures2KHR(physicalDevice, pFeatures);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceProperties2Helper(VkPhysicalDevice physicalDevice,
                                                                  VkPhysicalDeviceProperties2* pProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceProperties2(physicalDevice, pProperties);
    } else {
        return DispatchGetPhysicalDeviceProperties2KHR(physicalDevice, pProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceFormatProperties2Helper(VkPhysicalDevice physicalDevice, VkFormat format,
                                                                        VkFormatProperties2* pFormatProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceFormatProperties2(physicalDevice, format, pFormatProperties);
    } else {
        return DispatchGetPhysicalDeviceFormatProperties2KHR(physicalDevice, format, pFormatProperties);
    }
}

VkResult ValidationObject::DispatchGetPhysicalDeviceImageFormatProperties2Helper(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo,
    VkImageFormatProperties2* pImageFormatProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceImageFormatProperties2(physicalDevice, pImageFormatInfo, pImageFormatProperties);
    } else {
        return DispatchGetPhysicalDeviceImageFormatProperties2KHR(physicalDevice, pImageFormatInfo, pImageFormatProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceQueueFamilyProperties2Helper(
    VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties2* pQueueFamilyProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceQueueFamilyProperties2(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
    } else {
        return DispatchGetPhysicalDeviceQueueFamilyProperties2KHR(physicalDevice, pQueueFamilyPropertyCount,
                                                                  pQueueFamilyProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceMemoryProperties2Helper(
    VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties2* pMemoryProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceMemoryProperties2(physicalDevice, pMemoryProperties);
    } else {
        return DispatchGetPhysicalDeviceMemoryProperties2KHR(physicalDevice, pMemoryProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceSparseImageFormatProperties2Helper(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount,
    VkSparseImageFormatProperties2* pProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceSparseImageFormatProperties2(physicalDevice, pFormatInfo, pPropertyCount, pProperties);
    } else {
        return DispatchGetPhysicalDeviceSparseImageFormatProperties2KHR(physicalDevice, pFormatInfo, pPropertyCount, pProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceExternalSemaphorePropertiesHelper(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo,
    VkExternalSemaphoreProperties* pExternalSemaphoreProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceExternalSemaphoreProperties(physicalDevice, pExternalSemaphoreInfo,
                                                                    pExternalSemaphoreProperties);
    } else {
        return DispatchGetPhysicalDeviceExternalSemaphorePropertiesKHR(physicalDevice, pExternalSemaphoreInfo,
                                                                       pExternalSemaphoreProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceExternalFencePropertiesHelper(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo,
    VkExternalFenceProperties* pExternalFenceProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceExternalFenceProperties(physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
    } else {
        return DispatchGetPhysicalDeviceExternalFencePropertiesKHR(physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
    }
}

void ValidationObject::DispatchGetPhysicalDeviceExternalBufferPropertiesHelper(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo,
    VkExternalBufferProperties* pExternalBufferProperties) const {
    if (api_version >= VK_API_VERSION_1_1) {
        return DispatchGetPhysicalDeviceExternalBufferProperties(physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
    } else {
        return DispatchGetPhysicalDeviceExternalBufferPropertiesKHR(physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
    }
}

// Global list of sType,size identifiers
std::vector<std::pair<uint32_t, uint32_t>>& GetCustomStypeInfo() {
    static std::vector<std::pair<uint32_t, uint32_t>> custom_stype_info{};
    return custom_stype_info;
}

template <typename ValidationObjectType>
ValidationObjectType* ValidationObject::GetValidationObject() const {
    LayerObjectTypeId type_id;
    if constexpr (std::is_same_v<ValidationObjectType, ThreadSafety>) {
        type_id = LayerObjectTypeThreading;
    } else if constexpr (std::is_same_v<ValidationObjectType, StatelessValidation>) {
        type_id = LayerObjectTypeParameterValidation;
    } else if constexpr (std::is_same_v<ValidationObjectType, ObjectLifetimes>) {
        type_id = LayerObjectTypeObjectTracker;
    } else if constexpr (std::is_same_v<ValidationObjectType, CoreChecks>) {
        type_id = LayerObjectTypeCoreValidation;
    } else {
        static_assert(vvl::dependent_false_v<ValidationObjectType>, "unsupported validation object type");
    }
    return static_cast<ValidationObjectType*>(GetValidationObject(type_id));
}

template ThreadSafety* ValidationObject::GetValidationObject<ThreadSafety>() const;
template StatelessValidation* ValidationObject::GetValidationObject<StatelessValidation>() const;
template ObjectLifetimes* ValidationObject::GetValidationObject<ObjectLifetimes>() const;
template CoreChecks* ValidationObject::GetValidationObject<CoreChecks>() const;

// Takes the layer and removes it from the chassis so it will not be called anymore
void ValidationObject::ReleaseDeviceDispatchObject(LayerObjectTypeId type_id) const {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    for (auto object_it = layer_data->object_dispatch.begin(); object_it != layer_data->object_dispatch.end(); object_it++) {
        if ((*object_it)->container_type == type_id) {
            ValidationObject* object = *object_it;

            layer_data->object_dispatch.erase(object_it);

            for (auto intercept_vector_it = layer_data->intercept_vectors.begin();
                 intercept_vector_it != layer_data->intercept_vectors.end(); intercept_vector_it++) {
                for (auto intercept_object_it = intercept_vector_it->begin(); intercept_object_it != intercept_vector_it->end();
                     intercept_object_it++) {
                    if (object == *intercept_object_it) {
                        intercept_vector_it->erase(intercept_object_it);
                        break;
                    }
                }
            }

            // We can't destroy the object itself now as it might be unsafe (things are still being used)
            // If the rare case happens we need to release, we will cleanup later when we normally would have cleaned this up
            layer_data->aborted_object_dispatch.push_back(object);
            break;
        }
    }
}

namespace vulkan_layer_chassis {

static const VkLayerProperties global_layer = {
    OBJECT_LAYER_NAME,
    VK_LAYER_API_VERSION,
    1,
    "LunarG validation Layer",
};

typedef enum ApiFunctionType { kFuncTypeInst = 0, kFuncTypePdev = 1, kFuncTypeDev = 2 } ApiFunctionType;

typedef struct {
    ApiFunctionType function_type;
    void* funcptr;
} function_data;

const vvl::unordered_map<std::string, function_data>& GetNameToFuncPtrMap();

// Manually written functions

// Check enabled instance extensions against supported instance extension whitelist
static void InstanceExtensionWhitelist(ValidationObject* layer_data, const VkInstanceCreateInfo* pCreateInfo, VkInstance instance) {
    for (uint32_t i = 0; i < pCreateInfo->enabledExtensionCount; i++) {
        // Check for recognized instance extensions
        vvl::Extension extension = GetExtension(pCreateInfo->ppEnabledExtensionNames[i]);
        if (!IsInstanceExtension(extension)) {
            Location loc(vvl::Func::vkCreateInstance);
            layer_data->LogWarning(kVUIDUndefined, layer_data->instance,
                                   loc.dot(vvl::Field::pCreateInfo).dot(vvl::Field::ppEnabledExtensionNames, i),
                                   "%s is not supported by this layer.  Using this extension may adversely affect validation "
                                   "results and/or produce undefined behavior.",
                                   pCreateInfo->ppEnabledExtensionNames[i]);
        }
    }
}

// Check enabled device extensions against supported device extension whitelist
static void DeviceExtensionWhitelist(ValidationObject* layer_data, const VkDeviceCreateInfo* pCreateInfo, VkDevice device) {
    for (uint32_t i = 0; i < pCreateInfo->enabledExtensionCount; i++) {
        // Check for recognized device extensions
        vvl::Extension extension = GetExtension(pCreateInfo->ppEnabledExtensionNames[i]);
        if (!IsDeviceExtension(extension)) {
            Location loc(vvl::Func::vkCreateDevice);
            layer_data->LogWarning(kVUIDUndefined, layer_data->device,
                                   loc.dot(vvl::Field::pCreateInfo).dot(vvl::Field::ppEnabledExtensionNames, i),
                                   "%s is not supported by this layer.  Using this extension may adversely affect validation "
                                   "results and/or produce undefined behavior.",
                                   pCreateInfo->ppEnabledExtensionNames[i]);
        }
    }
}

void OutputLayerStatusInfo(ValidationObject* context) {
    std::string list_of_enables;
    std::string list_of_disables;
    for (uint32_t i = 0; i < kMaxEnableFlags; i++) {
        if (context->enabled[i]) {
            if (list_of_enables.size()) list_of_enables.append(", ");
            list_of_enables.append(GetEnableFlagNameHelper()[i]);
        }
    }
    if (list_of_enables.empty()) {
        list_of_enables.append("None");
    }
    for (uint32_t i = 0; i < kMaxDisableFlags; i++) {
        if (context->disabled[i]) {
            if (list_of_disables.size()) list_of_disables.append(", ");
            list_of_disables.append(GetDisableFlagNameHelper()[i]);
        }
    }
    if (list_of_disables.empty()) {
        list_of_disables.append("None");
    }

    Location loc(vvl::Func::vkCreateInstance);
    // Output layer status information message
    // TODO - We should just dump all settings to a file (see https://github.com/KhronosGroup/Vulkan-Utility-Libraries/issues/188)
    context->LogInfo("WARNING-CreateInstance-status-message", context->instance, loc,
                     "Khronos Validation Layer Active:\n    Current Enables: %s.\n    Current Disables: %s.\n",
                     list_of_enables.c_str(), list_of_disables.c_str());

    // Create warning message if user is running debug layers.
#ifndef NDEBUG
    context->LogPerformanceWarning("WARNING-CreateInstance-debug-warning", context->instance, loc,
                                   "Using debug builds of the validation layers *will* adversely affect performance.");
#endif
    if (!context->global_settings.fine_grained_locking) {
        context->LogPerformanceWarning(
            "WARNING-CreateInstance-locking-warning", context->instance, loc,
            "Fine-grained locking is disabled, this will adversely affect performance of multithreaded applications.");
    }
}

// Non-code-generated chassis API functions

VKAPI_ATTR PFN_vkVoidFunction VKAPI_CALL GetDeviceProcAddr(VkDevice device, const char* funcName) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    if (!ApiParentExtensionEnabled(funcName, &layer_data->device_extensions)) {
        return nullptr;
    }
    const auto& item = GetNameToFuncPtrMap().find(funcName);
    if (item != GetNameToFuncPtrMap().end()) {
        if (item->second.function_type != kFuncTypeDev) {
            Location loc(vvl::Func::vkGetDeviceProcAddr);
            // Was discussed in https://gitlab.khronos.org/vulkan/vulkan/-/merge_requests/6583
            // This has "valid" behavior to return null, but still worth warning users for this unqiue function
            layer_data->LogWarning("WARNING-vkGetDeviceProcAddr-device", device, loc.dot(vvl::Field::pName),
                                   "is trying to grab %s which is an instance level function", funcName);
            return nullptr;
        } else {
            return reinterpret_cast<PFN_vkVoidFunction>(item->second.funcptr);
        }
    }
    auto& table = layer_data->device_dispatch_table;
    if (!table.GetDeviceProcAddr) return nullptr;
    return table.GetDeviceProcAddr(device, funcName);
}

VKAPI_ATTR PFN_vkVoidFunction VKAPI_CALL GetInstanceProcAddr(VkInstance instance, const char* funcName) {
    const auto& item = GetNameToFuncPtrMap().find(funcName);
    if (item != GetNameToFuncPtrMap().end()) {
        return reinterpret_cast<PFN_vkVoidFunction>(item->second.funcptr);
    }
    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    auto& table = layer_data->instance_dispatch_table;
    if (!table.GetInstanceProcAddr) return nullptr;
    return table.GetInstanceProcAddr(instance, funcName);
}

VKAPI_ATTR PFN_vkVoidFunction VKAPI_CALL GetPhysicalDeviceProcAddr(VkInstance instance, const char* funcName) {
    const auto& item = GetNameToFuncPtrMap().find(funcName);
    if (item != GetNameToFuncPtrMap().end()) {
        if (item->second.function_type != kFuncTypePdev) {
            return nullptr;
        } else {
            return reinterpret_cast<PFN_vkVoidFunction>(item->second.funcptr);
        }
    }
    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    auto& table = layer_data->instance_dispatch_table;
    if (!table.GetPhysicalDeviceProcAddr) return nullptr;
    return table.GetPhysicalDeviceProcAddr(instance, funcName);
}

VKAPI_ATTR VkResult VKAPI_CALL EnumerateInstanceLayerProperties(uint32_t* pCount, VkLayerProperties* pProperties) {
    return util_GetLayerProperties(1, &global_layer, pCount, pProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL EnumerateDeviceLayerProperties(VkPhysicalDevice physicalDevice, uint32_t* pCount,
                                                              VkLayerProperties* pProperties) {
    return util_GetLayerProperties(1, &global_layer, pCount, pProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL EnumerateInstanceExtensionProperties(const char* pLayerName, uint32_t* pCount,
                                                                    VkExtensionProperties* pProperties) {
    if (pLayerName && !strcmp(pLayerName, global_layer.layerName)) {
        return util_GetExtensionProperties(static_cast<uint32_t>(kInstanceExtensions.size()), kInstanceExtensions.data(), pCount,
                                           pProperties);
    }

    return VK_ERROR_LAYER_NOT_PRESENT;
}

VKAPI_ATTR VkResult VKAPI_CALL EnumerateDeviceExtensionProperties(VkPhysicalDevice physicalDevice, const char* pLayerName,
                                                                  uint32_t* pCount, VkExtensionProperties* pProperties) {
    if (pLayerName && !strcmp(pLayerName, global_layer.layerName)) {
        return util_GetExtensionProperties(static_cast<uint32_t>(kDeviceExtensions.size()), kDeviceExtensions.data(), pCount,
                                           pProperties);
    }

    assert(physicalDevice);
    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    return layer_data->instance_dispatch_table.EnumerateDeviceExtensionProperties(physicalDevice, pLayerName, pCount, pProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateInstance(const VkInstanceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator,
                                              VkInstance* pInstance) {
    VVL_ZoneScoped;
    VkLayerInstanceCreateInfo* chain_info = GetChainInfo(pCreateInfo, VK_LAYER_LINK_INFO);

    assert(chain_info->u.pLayerInfo);
    PFN_vkGetInstanceProcAddr fpGetInstanceProcAddr = chain_info->u.pLayerInfo->pfnNextGetInstanceProcAddr;
    PFN_vkCreateInstance fpCreateInstance = (PFN_vkCreateInstance)fpGetInstanceProcAddr(nullptr, "vkCreateInstance");
    if (fpCreateInstance == nullptr) return VK_ERROR_INITIALIZATION_FAILED;
    chain_info->u.pLayerInfo = chain_info->u.pLayerInfo->pNext;
    uint32_t specified_version = (pCreateInfo->pApplicationInfo ? pCreateInfo->pApplicationInfo->apiVersion : VK_API_VERSION_1_0);
    APIVersion api_version = VK_MAKE_API_VERSION(VK_API_VERSION_VARIANT(specified_version), VK_API_VERSION_MAJOR(specified_version),
                                                 VK_API_VERSION_MINOR(specified_version), 0);

    auto debug_report = new DebugReport{};
    debug_report->instance_pnext_chain = vku::SafePnextCopy(pCreateInfo->pNext);
    ActivateInstanceDebugCallbacks(debug_report);

    // Set up enable and disable features flags
    CHECK_ENABLED local_enables{};
    CHECK_DISABLED local_disables{};
    GlobalSettings local_global_settings = {};
    GpuAVSettings local_gpuav_settings = {};
    SyncValSettings local_syncval_settings = {};
    ConfigAndEnvSettings config_and_env_settings_data{OBJECT_LAYER_DESCRIPTION, pCreateInfo, local_enables, local_disables,
                                                      debug_report,
                                                      // All settings for various internal layers
                                                      &local_global_settings, &local_gpuav_settings, &local_syncval_settings};
    ProcessConfigAndEnvSettings(&config_and_env_settings_data);

    // Create temporary dispatch vector for pre-calls until instance is created
    std::vector<ValidationObject*> local_object_dispatch = CreateObjectDispatch(local_enables, local_disables);

    // If handle wrapping is disabled via the ValidationFeatures extension, override build flag
    if (local_disables[handle_wrapping]) {
        wrap_handles = false;
    }

    // Initialize the validation objects
    for (auto* intercept : local_object_dispatch) {
        intercept->api_version = api_version;
        intercept->debug_report = debug_report;
    }

    // Define logic to cleanup everything in case of an error
    auto cleanup_allocations = [debug_report, &local_object_dispatch]() {
        DeactivateInstanceDebugCallbacks(debug_report);
        vku::FreePnextChain(debug_report->instance_pnext_chain);
        LayerDebugUtilsDestroyInstance(debug_report);
        for (ValidationObject* object : local_object_dispatch) {
            delete object;
        }
    };

    // Init dispatch array and call registration functions
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateInstance, VulkanTypedHandle());
    for (const ValidationObject* intercept : local_object_dispatch) {
        auto lock = intercept->ReadLock();
        skip |= intercept->PreCallValidateCreateInstance(pCreateInfo, pAllocator, pInstance, error_obj);
        if (skip) {
            cleanup_allocations();
            return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkCreateInstance);
    for (ValidationObject* intercept : local_object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordCreateInstance(pCreateInfo, pAllocator, pInstance, record_obj);
    }

    VkResult result = fpCreateInstance(pCreateInfo, pAllocator, pInstance);
    if (result != VK_SUCCESS) {
        cleanup_allocations();
        return result;
    }
    record_obj.result = result;
    auto framework = GetLayerDataPtr(GetDispatchKey(*pInstance), layer_data_map);

    framework->object_dispatch = local_object_dispatch;
    framework->container_type = LayerObjectTypeInstance;
    framework->disabled = local_disables;
    framework->enabled = local_enables;
    framework->global_settings = local_global_settings;
    framework->gpuav_settings = local_gpuav_settings;
    framework->syncval_settings = local_syncval_settings;

    framework->instance = *pInstance;
    layer_init_instance_dispatch_table(*pInstance, &framework->instance_dispatch_table, fpGetInstanceProcAddr);
    framework->debug_report = debug_report;
    framework->api_version = api_version;
    framework->instance_extensions.InitFromInstanceCreateInfo(specified_version, pCreateInfo);

    // We need to call this to properly check which device extensions have been promoted when validating query functions
    // that take as input a physical device, which can be called before a logical device has been created.
    framework->device_extensions.InitFromDeviceCreateInfo(&framework->instance_extensions, specified_version);

    OutputLayerStatusInfo(framework);

    for (auto* intercept : framework->object_dispatch) {
        intercept->instance_dispatch_table = framework->instance_dispatch_table;
        intercept->enabled = framework->enabled;
        intercept->disabled = framework->disabled;
        intercept->global_settings = framework->global_settings;
        intercept->gpuav_settings = framework->gpuav_settings;
        intercept->syncval_settings = framework->syncval_settings;
        intercept->instance = *pInstance;
    }

    for (ValidationObject* intercept : framework->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PostCallRecordCreateInstance(pCreateInfo, pAllocator, pInstance, record_obj);
    }

    InstanceExtensionWhitelist(framework, pCreateInfo, *pInstance);
    DeactivateInstanceDebugCallbacks(debug_report);
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyInstance(VkInstance instance, const VkAllocationCallbacks* pAllocator) {
    VVL_TracyCZone(tracy_zone_precall, true);
    dispatch_key key = GetDispatchKey(instance);
    auto layer_data = GetLayerDataPtr(key, layer_data_map);
    ActivateInstanceDebugCallbacks(layer_data->debug_report);
    ErrorObject error_obj(vvl::Func::vkDestroyInstance, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));

    for (const ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->ReadLock();
        intercept->PreCallValidateDestroyInstance(instance, pAllocator, error_obj);
    }

    RecordObject record_obj(vvl::Func::vkDestroyInstance);
    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordDestroyInstance(instance, pAllocator, record_obj);
    }

    // Before instance is destroyed, allow aborted objects to clean up
    for (ValidationObject* intercept : layer_data->aborted_object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordDestroyInstance(instance, pAllocator, record_obj);
    }

    VVL_TracyCZoneEnd(tracy_zone_precall);
    VVL_TracyCZone(tracy_zone_dispatch, true);
    layer_data->instance_dispatch_table.DestroyInstance(instance, pAllocator);
    VVL_TracyCZoneEnd(tracy_zone_dispatch);

    VVL_TracyCZone(tracy_zone_postcall, true);
    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PostCallRecordDestroyInstance(instance, pAllocator, record_obj);
    }

    DeactivateInstanceDebugCallbacks(layer_data->debug_report);
    vku::FreePnextChain(layer_data->debug_report->instance_pnext_chain);

    LayerDebugUtilsDestroyInstance(layer_data->debug_report);

    for (auto item = layer_data->object_dispatch.begin(); item != layer_data->object_dispatch.end(); item++) {
        delete *item;
    }
    for (auto item = layer_data->aborted_object_dispatch.begin(); item != layer_data->aborted_object_dispatch.end(); item++) {
        delete *item;
    }

    FreeLayerDataPtr(key, layer_data_map);
    VVL_TracyCZoneEnd(tracy_zone_postcall);

#if TRACY_MANUAL_LIFETIME
    tracy::ShutdownProfiler();
#endif
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDevice(VkPhysicalDevice gpu, const VkDeviceCreateInfo* pCreateInfo,
                                            const VkAllocationCallbacks* pAllocator, VkDevice* pDevice) {
    VkLayerDeviceCreateInfo* chain_info = GetChainInfo(pCreateInfo, VK_LAYER_LINK_INFO);

    auto instance_interceptor = GetLayerDataPtr(GetDispatchKey(gpu), layer_data_map);

    PFN_vkGetInstanceProcAddr fpGetInstanceProcAddr = chain_info->u.pLayerInfo->pfnNextGetInstanceProcAddr;
    PFN_vkGetDeviceProcAddr fpGetDeviceProcAddr = chain_info->u.pLayerInfo->pfnNextGetDeviceProcAddr;
    PFN_vkCreateDevice fpCreateDevice = (PFN_vkCreateDevice)fpGetInstanceProcAddr(instance_interceptor->instance, "vkCreateDevice");
    if (fpCreateDevice == nullptr) {
        return VK_ERROR_INITIALIZATION_FAILED;
    }
    chain_info->u.pLayerInfo = chain_info->u.pLayerInfo->pNext;

    // Get physical device limits for device
    VkPhysicalDeviceProperties device_properties = {};
    instance_interceptor->instance_dispatch_table.GetPhysicalDeviceProperties(gpu, &device_properties);

    // Setup the validation tables based on the application API version from the instance and the capabilities of the device driver
    auto effective_api_version = std::min(APIVersion(device_properties.apiVersion), instance_interceptor->api_version);

    DeviceExtensions device_extensions = {};
    device_extensions.InitFromDeviceCreateInfo(&instance_interceptor->instance_extensions, effective_api_version, pCreateInfo);
    for (auto item : instance_interceptor->object_dispatch) {
        item->device_extensions = device_extensions;
    }

    // Make copy to modify as some ValidationObjects will want to add extensions/features on
    vku::safe_VkDeviceCreateInfo modified_create_info(pCreateInfo);

    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDevice, VulkanTypedHandle(gpu, kVulkanObjectTypePhysicalDevice));
    for (const ValidationObject* intercept : instance_interceptor->object_dispatch) {
        auto lock = intercept->ReadLock();
        skip |= intercept->PreCallValidateCreateDevice(gpu, pCreateInfo, pAllocator, pDevice, error_obj);
        if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
    }

    RecordObject record_obj(vvl::Func::vkCreateDevice);
    for (ValidationObject* intercept : instance_interceptor->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordCreateDevice(gpu, pCreateInfo, pAllocator, pDevice, record_obj, &modified_create_info);
    }

    VkResult result = fpCreateDevice(gpu, reinterpret_cast<VkDeviceCreateInfo*>(&modified_create_info), pAllocator, pDevice);
    if (result != VK_SUCCESS) {
        return result;
    }
    record_obj.result = result;

    auto device_interceptor = GetLayerDataPtr(GetDispatchKey(*pDevice), layer_data_map);
    device_interceptor->container_type = LayerObjectTypeDevice;

    // Save local info in device object
    device_interceptor->api_version = device_interceptor->device_extensions.InitFromDeviceCreateInfo(
        &instance_interceptor->instance_extensions, effective_api_version,
        reinterpret_cast<VkDeviceCreateInfo*>(&modified_create_info));
    device_interceptor->device_extensions = device_extensions;

    layer_init_device_dispatch_table(*pDevice, &device_interceptor->device_dispatch_table, fpGetDeviceProcAddr);

    device_interceptor->device = *pDevice;
    device_interceptor->physical_device = gpu;
    device_interceptor->instance = instance_interceptor->instance;
    device_interceptor->debug_report = instance_interceptor->debug_report;

    instance_interceptor->debug_report->device_created++;

    InitDeviceObjectDispatch(instance_interceptor, device_interceptor);

    // Initialize all of the objects with the appropriate data
    for (auto* object : device_interceptor->object_dispatch) {
        object->device = device_interceptor->device;
        object->physical_device = device_interceptor->physical_device;
        object->instance = instance_interceptor->instance;
        object->debug_report = instance_interceptor->debug_report;
        object->device_dispatch_table = device_interceptor->device_dispatch_table;
        object->api_version = device_interceptor->api_version;
        object->disabled = instance_interceptor->disabled;
        object->enabled = instance_interceptor->enabled;
        object->global_settings = instance_interceptor->global_settings;
        object->gpuav_settings = instance_interceptor->gpuav_settings;
        object->syncval_settings = instance_interceptor->syncval_settings;
        object->instance_dispatch_table = instance_interceptor->instance_dispatch_table;
        object->instance_extensions = instance_interceptor->instance_extensions;
        object->device_extensions = device_interceptor->device_extensions;
    }

    for (ValidationObject* intercept : instance_interceptor->object_dispatch) {
        auto lock = intercept->WriteLock();
        // Send down modified create info as we want to mark enabled features that we sent down on behalf of the app
        intercept->PostCallRecordCreateDevice(gpu, reinterpret_cast<VkDeviceCreateInfo*>(&modified_create_info), pAllocator,
                                              pDevice, record_obj);
    }

    device_interceptor->InitObjectDispatchVectors();

    DeviceExtensionWhitelist(device_interceptor, pCreateInfo, *pDevice);

    return result;
}

// NOTE: Do _not_ skip the dispatch call when destroying a device. Whether or not there was a validation error,
//       the loader will destroy the device, and know nothing about future references to this device making it
//       impossible for the caller to use this device handle further. IOW, this is our _only_ chance to (potentially)
//       dispatch the driver's DestroyDevice function.
VKAPI_ATTR void VKAPI_CALL DestroyDevice(VkDevice device, const VkAllocationCallbacks* pAllocator) {
    dispatch_key key = GetDispatchKey(device);
    auto layer_data = GetLayerDataPtr(key, layer_data_map);
    ErrorObject error_obj(vvl::Func::vkDestroyDevice, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    for (const ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->ReadLock();
        intercept->PreCallValidateDestroyDevice(device, pAllocator, error_obj);
    }

    RecordObject record_obj(vvl::Func::vkDestroyDevice);
    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordDestroyDevice(device, pAllocator, record_obj);
    }

    // Before device is destroyed, allow aborted objects to clean up
    for (ValidationObject* intercept : layer_data->aborted_object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordDestroyDevice(device, pAllocator, record_obj);
    }

    layer_data->device_dispatch_table.DestroyDevice(device, pAllocator);

    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PostCallRecordDestroyDevice(device, pAllocator, record_obj);
    }

    auto instance_interceptor = GetLayerDataPtr(GetDispatchKey(layer_data->physical_device), layer_data_map);
    instance_interceptor->debug_report->device_created--;

    for (auto item = layer_data->object_dispatch.begin(); item != layer_data->object_dispatch.end(); item++) {
        delete *item;
    }
    for (auto item = layer_data->aborted_object_dispatch.begin(); item != layer_data->aborted_object_dispatch.end(); item++) {
        delete *item;
    }

    FreeLayerDataPtr(key, layer_data_map);
}

// Special-case APIs for which core_validation needs custom parameter lists and/or modifies parameters

VKAPI_ATTR VkResult VKAPI_CALL CreateGraphicsPipelines(VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount,
                                                       const VkGraphicsPipelineCreateInfo* pCreateInfos,
                                                       const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateGraphicsPipelines, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    PipelineStates pipeline_states[LayerObjectTypeMaxEnum];
    chassis::CreateGraphicsPipelines chassis_state(pCreateInfos);

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateGraphicsPipelines(device, pipelineCache, createInfoCount, pCreateInfos,
                                                                      pAllocator, pPipelines, error_obj,
                                                                      pipeline_states[intercept->container_type], chassis_state);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkCreateGraphicsPipelines);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateGraphicsPipelines(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                                                            pPipelines, record_obj, pipeline_states[intercept->container_type],
                                                            chassis_state);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateGraphicsPipelines(device, pipelineCache, createInfoCount, chassis_state.pCreateInfos, pAllocator,
                                                 pPipelines);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateGraphicsPipelines(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                                                             pPipelines, record_obj, pipeline_states[intercept->container_type],
                                                             chassis_state);
        }
    }
    return result;
}

// This API saves some core_validation pipeline state state on the stack for performance purposes
VKAPI_ATTR VkResult VKAPI_CALL CreateComputePipelines(VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount,
                                                      const VkComputePipelineCreateInfo* pCreateInfos,
                                                      const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateComputePipelines, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    PipelineStates pipeline_states[LayerObjectTypeMaxEnum];
    chassis::CreateComputePipelines chassis_state(pCreateInfos);

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateComputePipelines(device, pipelineCache, createInfoCount, pCreateInfos,
                                                                     pAllocator, pPipelines, error_obj,
                                                                     pipeline_states[intercept->container_type], chassis_state);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkCreateComputePipelines);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateComputePipelines(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                                                           pPipelines, record_obj, pipeline_states[intercept->container_type],
                                                           chassis_state);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateComputePipelines(device, pipelineCache, createInfoCount, chassis_state.pCreateInfos, pAllocator,
                                                pPipelines);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateComputePipelines(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                                                            pPipelines, record_obj, pipeline_states[intercept->container_type],
                                                            chassis_state);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRayTracingPipelinesNV(VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount,
                                                           const VkRayTracingPipelineCreateInfoNV* pCreateInfos,
                                                           const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateRayTracingPipelinesNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    PipelineStates pipeline_states[LayerObjectTypeMaxEnum];
    chassis::CreateRayTracingPipelinesNV chassis_state(pCreateInfos);

    for (const ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->ReadLock();
        skip |= intercept->PreCallValidateCreateRayTracingPipelinesNV(device, pipelineCache, createInfoCount, pCreateInfos,
                                                                      pAllocator, pPipelines, error_obj,
                                                                      pipeline_states[intercept->container_type], chassis_state);
        if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
    }

    RecordObject record_obj(vvl::Func::vkCreateRayTracingPipelinesNV);
    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordCreateRayTracingPipelinesNV(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                                                            pPipelines, record_obj, pipeline_states[intercept->container_type],
                                                            chassis_state);
    }

    VkResult result = DispatchCreateRayTracingPipelinesNV(device, pipelineCache, createInfoCount, chassis_state.pCreateInfos,
                                                          pAllocator, pPipelines);
    record_obj.result = result;

    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PostCallRecordCreateRayTracingPipelinesNV(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator,
                                                             pPipelines, record_obj, pipeline_states[intercept->container_type],
                                                             chassis_state);
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRayTracingPipelinesKHR(VkDevice device, VkDeferredOperationKHR deferredOperation,
                                                            VkPipelineCache pipelineCache, uint32_t createInfoCount,
                                                            const VkRayTracingPipelineCreateInfoKHR* pCreateInfos,
                                                            const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateRayTracingPipelinesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    PipelineStates pipeline_states[LayerObjectTypeMaxEnum];
    auto chassis_state = std::make_shared<chassis::CreateRayTracingPipelinesKHR>(pCreateInfos);

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateRayTracingPipelinesKHR(
                device, deferredOperation, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines, error_obj,
                pipeline_states[intercept->container_type], *chassis_state);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkCreateRayTracingPipelinesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateRayTracingPipelinesKHR(device, deferredOperation, pipelineCache, createInfoCount,
                                                                 pCreateInfos, pAllocator, pPipelines, record_obj,
                                                                 pipeline_states[intercept->container_type], *chassis_state);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateRayTracingPipelinesKHR(device, deferredOperation, pipelineCache, createInfoCount,
                                                      chassis_state->pCreateInfos, pAllocator, pPipelines);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateRayTracingPipelinesKHR(device, deferredOperation, pipelineCache, createInfoCount,
                                                                  pCreateInfos, pAllocator, pPipelines, record_obj,
                                                                  pipeline_states[intercept->container_type], chassis_state);
        }
    }
    return result;
}

// This API needs the ability to modify a down-chain parameter
VKAPI_ATTR VkResult VKAPI_CALL CreatePipelineLayout(VkDevice device, const VkPipelineLayoutCreateInfo* pCreateInfo,
                                                    const VkAllocationCallbacks* pAllocator, VkPipelineLayout* pPipelineLayout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreatePipelineLayout, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreatePipelineLayout]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreatePipelineLayout(device, pCreateInfo, pAllocator, pPipelineLayout, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    chassis::CreatePipelineLayout chassis_state{};
    chassis_state.modified_create_info = *pCreateInfo;

    RecordObject record_obj(vvl::Func::vkCreatePipelineLayout);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreatePipelineLayout(device, pCreateInfo, pAllocator, pPipelineLayout, record_obj,
                                                         chassis_state);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreatePipelineLayout(device, &chassis_state.modified_create_info, pAllocator, pPipelineLayout);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreatePipelineLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreatePipelineLayout(device, pCreateInfo, pAllocator, pPipelineLayout, record_obj);
        }
    }
    return result;
}

// This API needs some local stack data for performance reasons and also may modify a parameter
VKAPI_ATTR VkResult VKAPI_CALL CreateShaderModule(VkDevice device, const VkShaderModuleCreateInfo* pCreateInfo,
                                                  const VkAllocationCallbacks* pAllocator, VkShaderModule* pShaderModule) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateShaderModule, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateShaderModule(device, pCreateInfo, pAllocator, pShaderModule, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    chassis::CreateShaderModule chassis_state{};

    RecordObject record_obj(vvl::Func::vkCreateShaderModule);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateShaderModule(device, pCreateInfo, pAllocator, pShaderModule, record_obj, chassis_state);
        }
    }

    // Special extra check if SPIR-V itself fails runtime validation in PreCallRecord
    if (chassis_state.skip) return VK_ERROR_VALIDATION_FAILED_EXT;
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateShaderModule(device, pCreateInfo, pAllocator, pShaderModule);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateShaderModule(device, pCreateInfo, pAllocator, pShaderModule, record_obj, chassis_state);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateShadersEXT(VkDevice device, uint32_t createInfoCount,
                                                const VkShaderCreateInfoEXT* pCreateInfos, const VkAllocationCallbacks* pAllocator,
                                                VkShaderEXT* pShaders) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateShadersEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    chassis::ShaderObject chassis_state(createInfoCount, pCreateInfos);

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCreateShadersEXT(device, createInfoCount, pCreateInfos, pAllocator, pShaders, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkCreateShadersEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateShadersEXT(device, createInfoCount, pCreateInfos, pAllocator, pShaders, record_obj,
                                                     chassis_state);
        }
    }

    // Special extra check if SPIR-V itself fails runtime validation in PreCallRecord
    if (chassis_state.skip) return VK_ERROR_VALIDATION_FAILED_EXT;

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateShadersEXT(device, createInfoCount, chassis_state.pCreateInfos, pAllocator, pShaders);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateShadersEXT(device, createInfoCount, pCreateInfos, pAllocator, pShaders, record_obj,
                                                      chassis_state);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AllocateDescriptorSets(VkDevice device, const VkDescriptorSetAllocateInfo* pAllocateInfo,
                                                      VkDescriptorSet* pDescriptorSets) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAllocateDescriptorSets, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    vvl::AllocateDescriptorSetsData ads_state[LayerObjectTypeMaxEnum];

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            ads_state[intercept->container_type].Init(pAllocateInfo->descriptorSetCount);
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAllocateDescriptorSets(device, pAllocateInfo, pDescriptorSets, error_obj,
                                                                     ads_state[intercept->container_type]);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkAllocateDescriptorSets);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordAllocateDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAllocateDescriptorSets(device, pAllocateInfo, pDescriptorSets, record_obj);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAllocateDescriptorSets(device, pAllocateInfo, pDescriptorSets);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAllocateDescriptorSets(device, pAllocateInfo, pDescriptorSets, record_obj,
                                                            ads_state[intercept->container_type]);
        }
    }
    return result;
}

// This API needs the ability to modify a down-chain parameter
VKAPI_ATTR VkResult VKAPI_CALL CreateBuffer(VkDevice device, const VkBufferCreateInfo* pCreateInfo,
                                            const VkAllocationCallbacks* pAllocator, VkBuffer* pBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateBuffer, VulkanTypedHandle(device, kVulkanObjectTypeDevice));

    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateBuffer(device, pCreateInfo, pAllocator, pBuffer, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    chassis::CreateBuffer chassis_state{};
    chassis_state.modified_create_info = *pCreateInfo;

    RecordObject record_obj(vvl::Func::vkCreateBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateBuffer(device, pCreateInfo, pAllocator, pBuffer, record_obj, chassis_state);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateBuffer(device, &chassis_state.modified_create_info, pAllocator, pBuffer);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateBuffer(device, pCreateInfo, pAllocator, pBuffer, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BeginCommandBuffer(VkCommandBuffer commandBuffer, const VkCommandBufferBeginInfo* pBeginInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    chassis::HandleData handle_data;
    {
        auto lock = ReadLockGuard(secondary_cb_map_mutex);
        handle_data.command_buffer.is_secondary = (secondary_cb_map.find(commandBuffer) != secondary_cb_map.end());
    }

    ErrorObject error_obj(vvl::Func::vkBeginCommandBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer),
                          &handle_data);
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBeginCommandBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBeginCommandBuffer(commandBuffer, pBeginInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }

    RecordObject record_obj(vvl::Func::vkBeginCommandBuffer, &handle_data);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBeginCommandBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBeginCommandBuffer(commandBuffer, pBeginInfo, record_obj);
        }
    }

    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBeginCommandBuffer(commandBuffer, pBeginInfo, handle_data.command_buffer.is_secondary);
    }
    record_obj.result = result;

    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBeginCommandBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBeginCommandBuffer(commandBuffer, pBeginInfo, record_obj);
        }
    }
    return result;
}

// Handle tooling queries manually as this is a request for layer information
static const VkPhysicalDeviceToolPropertiesEXT khronos_layer_tool_props = {
    VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TOOL_PROPERTIES_EXT,
    nullptr,
    "Khronos Validation Layer",
    STRINGIFY(VK_HEADER_VERSION),
    VK_TOOL_PURPOSE_VALIDATION_BIT_EXT | VK_TOOL_PURPOSE_ADDITIONAL_FEATURES_BIT_EXT | VK_TOOL_PURPOSE_DEBUG_REPORTING_BIT_EXT |
        VK_TOOL_PURPOSE_DEBUG_MARKERS_BIT_EXT,
    "Khronos Validation Layer",
    OBJECT_LAYER_NAME};

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceToolPropertiesEXT(VkPhysicalDevice physicalDevice, uint32_t* pToolCount,
                                                                  VkPhysicalDeviceToolPropertiesEXT* pToolProperties) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceToolPropertiesEXT,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));

    auto original_pToolProperties = pToolProperties;

    if (pToolProperties != nullptr && *pToolCount > 0) {
        *pToolProperties = khronos_layer_tool_props;
        pToolProperties = ((*pToolCount > 1) ? &pToolProperties[1] : nullptr);
        (*pToolCount)--;
    }

    for (const ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->ReadLock();
        skip |=
            intercept->PreCallValidateGetPhysicalDeviceToolPropertiesEXT(physicalDevice, pToolCount, pToolProperties, error_obj);
        if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
    }

    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceToolPropertiesEXT);
    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordGetPhysicalDeviceToolPropertiesEXT(physicalDevice, pToolCount, pToolProperties, record_obj);
    }

    VkResult result = DispatchGetPhysicalDeviceToolPropertiesEXT(physicalDevice, pToolCount, pToolProperties);
    record_obj.result = result;

    if (original_pToolProperties != nullptr) {
        pToolProperties = original_pToolProperties;
    }
    assert(*pToolCount != std::numeric_limits<uint32_t>::max());
    (*pToolCount)++;

    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PostCallRecordGetPhysicalDeviceToolPropertiesEXT(physicalDevice, pToolCount, pToolProperties, record_obj);
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceToolProperties(VkPhysicalDevice physicalDevice, uint32_t* pToolCount,
                                                               VkPhysicalDeviceToolProperties* pToolProperties) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceToolProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));

    auto original_pToolProperties = pToolProperties;

    if (pToolProperties != nullptr && *pToolCount > 0) {
        *pToolProperties = khronos_layer_tool_props;
        pToolProperties = ((*pToolCount > 1) ? &pToolProperties[1] : nullptr);
        (*pToolCount)--;
    }

    for (const ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->ReadLock();
        skip |= intercept->PreCallValidateGetPhysicalDeviceToolProperties(physicalDevice, pToolCount, pToolProperties, error_obj);
        if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
    }

    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceToolProperties);
    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PreCallRecordGetPhysicalDeviceToolProperties(physicalDevice, pToolCount, pToolProperties, record_obj);
    }

    VkResult result = DispatchGetPhysicalDeviceToolProperties(physicalDevice, pToolCount, pToolProperties);
    record_obj.result = result;

    if (original_pToolProperties != nullptr) {
        pToolProperties = original_pToolProperties;
    }
    assert(*pToolCount != std::numeric_limits<uint32_t>::max());
    (*pToolCount)++;

    for (ValidationObject* intercept : layer_data->object_dispatch) {
        auto lock = intercept->WriteLock();
        intercept->PostCallRecordGetPhysicalDeviceToolProperties(physicalDevice, pToolCount, pToolProperties, record_obj);
    }
    return result;
}

// ValidationCache APIs do not dispatch

VKAPI_ATTR VkResult VKAPI_CALL CreateValidationCacheEXT(VkDevice device, const VkValidationCacheCreateInfoEXT* pCreateInfo,
                                                        const VkAllocationCallbacks* pAllocator,
                                                        VkValidationCacheEXT* pValidationCache) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    if (auto core_checks = layer_data->GetValidationObject<CoreChecks>()) {
        auto lock = core_checks->WriteLock();
        return core_checks->CoreLayerCreateValidationCacheEXT(device, pCreateInfo, pAllocator, pValidationCache);
    }
    return VK_SUCCESS;
}

VKAPI_ATTR void VKAPI_CALL DestroyValidationCacheEXT(VkDevice device, VkValidationCacheEXT validationCache,
                                                     const VkAllocationCallbacks* pAllocator) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    if (auto core_checks = layer_data->GetValidationObject<CoreChecks>()) {
        auto lock = core_checks->WriteLock();
        core_checks->CoreLayerDestroyValidationCacheEXT(device, validationCache, pAllocator);
    }
}

VKAPI_ATTR VkResult VKAPI_CALL MergeValidationCachesEXT(VkDevice device, VkValidationCacheEXT dstCache, uint32_t srcCacheCount,
                                                        const VkValidationCacheEXT* pSrcCaches) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    if (auto core_checks = layer_data->GetValidationObject<CoreChecks>()) {
        auto lock = core_checks->WriteLock();
        return core_checks->CoreLayerMergeValidationCachesEXT(device, dstCache, srcCacheCount, pSrcCaches);
    }
    return VK_SUCCESS;
}

VKAPI_ATTR VkResult VKAPI_CALL GetValidationCacheDataEXT(VkDevice device, VkValidationCacheEXT validationCache, size_t* pDataSize,
                                                         void* pData) {
    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    if (auto core_checks = layer_data->GetValidationObject<CoreChecks>()) {
        auto lock = core_checks->WriteLock();
        return core_checks->CoreLayerGetValidationCacheDataEXT(device, validationCache, pDataSize, pData);
    }
    return VK_SUCCESS;
}
VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDevices(VkInstance instance, uint32_t* pPhysicalDeviceCount,
                                                        VkPhysicalDevice* pPhysicalDevices) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkEnumeratePhysicalDevices, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateEnumeratePhysicalDevices(instance, pPhysicalDeviceCount, pPhysicalDevices, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkEnumeratePhysicalDevices);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordEnumeratePhysicalDevices(instance, pPhysicalDeviceCount, pPhysicalDevices, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchEnumeratePhysicalDevices(instance, pPhysicalDeviceCount, pPhysicalDevices);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordEnumeratePhysicalDevices(instance, pPhysicalDeviceCount, pPhysicalDevices, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFeatures(VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures* pFeatures) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFeatures,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceFeatures(physicalDevice, pFeatures, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFeatures);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFeatures(physicalDevice, pFeatures, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceFeatures(physicalDevice, pFeatures);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFeatures(physicalDevice, pFeatures, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFormatProperties(VkPhysicalDevice physicalDevice, VkFormat format,
                                                             VkFormatProperties* pFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFormatProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateGetPhysicalDeviceFormatProperties(physicalDevice, format, pFormatProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFormatProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFormatProperties(physicalDevice, format, pFormatProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceFormatProperties(physicalDevice, format, pFormatProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFormatProperties(physicalDevice, format, pFormatProperties, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceImageFormatProperties(VkPhysicalDevice physicalDevice, VkFormat format,
                                                                      VkImageType type, VkImageTiling tiling,
                                                                      VkImageUsageFlags usage, VkImageCreateFlags flags,
                                                                      VkImageFormatProperties* pImageFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceImageFormatProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceImageFormatProperties(physicalDevice, format, type, tiling, usage,
                                                                                     flags, pImageFormatProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceImageFormatProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceImageFormatProperties(physicalDevice, format, type, tiling, usage, flags,
                                                                           pImageFormatProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceImageFormatProperties(physicalDevice, format, type, tiling, usage, flags,
                                                                pImageFormatProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceImageFormatProperties(physicalDevice, format, type, tiling, usage, flags,
                                                                            pImageFormatProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceProperties(VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceProperties(physicalDevice, pProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceProperties(physicalDevice, pProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceProperties(physicalDevice, pProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceProperties(physicalDevice, pProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyProperties(VkPhysicalDevice physicalDevice,
                                                                  uint32_t* pQueueFamilyPropertyCount,
                                                                  VkQueueFamilyProperties* pQueueFamilyProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceQueueFamilyProperties(physicalDevice, pQueueFamilyPropertyCount,
                                                                                     pQueueFamilyProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceQueueFamilyProperties(physicalDevice, pQueueFamilyPropertyCount,
                                                                           pQueueFamilyProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceQueueFamilyProperties(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceQueueFamilyProperties(physicalDevice, pQueueFamilyPropertyCount,
                                                                            pQueueFamilyProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMemoryProperties(VkPhysicalDevice physicalDevice,
                                                             VkPhysicalDeviceMemoryProperties* pMemoryProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceMemoryProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceMemoryProperties(physicalDevice, pMemoryProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceMemoryProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceMemoryProperties(physicalDevice, pMemoryProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceMemoryProperties(physicalDevice, pMemoryProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceMemoryProperties(physicalDevice, pMemoryProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceQueue(VkDevice device, uint32_t queueFamilyIndex, uint32_t queueIndex, VkQueue* pQueue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceQueue, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceQueue]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceQueue(device, queueFamilyIndex, queueIndex, pQueue, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceQueue);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceQueue]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceQueue(device, queueFamilyIndex, queueIndex, pQueue, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceQueue(device, queueFamilyIndex, queueIndex, pQueue);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceQueue]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceQueue(device, queueFamilyIndex, queueIndex, pQueue, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSubmit(VkQueue queue, uint32_t submitCount, const VkSubmitInfo* pSubmits, VkFence fence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueSubmit, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateQueueSubmit]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueSubmit(queue, submitCount, pSubmits, fence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueSubmit);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueSubmit]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueSubmit(queue, submitCount, pSubmits, fence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueueSubmit(queue, submitCount, pSubmits, fence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueSubmit]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordQueueSubmit(queue, submitCount, pSubmits, fence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL QueueWaitIdle(VkQueue queue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueWaitIdle, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateQueueWaitIdle]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueWaitIdle(queue, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueWaitIdle);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueWaitIdle]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueWaitIdle(queue, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueueWaitIdle(queue);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueWaitIdle]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordQueueWaitIdle(queue, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DeviceWaitIdle(VkDevice device) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDeviceWaitIdle, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDeviceWaitIdle]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDeviceWaitIdle(device, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkDeviceWaitIdle);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDeviceWaitIdle]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDeviceWaitIdle(device, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchDeviceWaitIdle(device);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDeviceWaitIdle]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordDeviceWaitIdle(device, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AllocateMemory(VkDevice device, const VkMemoryAllocateInfo* pAllocateInfo,
                                              const VkAllocationCallbacks* pAllocator, VkDeviceMemory* pMemory) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAllocateMemory, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateAllocateMemory]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAllocateMemory(device, pAllocateInfo, pAllocator, pMemory, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAllocateMemory);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordAllocateMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAllocateMemory(device, pAllocateInfo, pAllocator, pMemory, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAllocateMemory(device, pAllocateInfo, pAllocator, pMemory);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordAllocateMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAllocateMemory(device, pAllocateInfo, pAllocator, pMemory, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL FreeMemory(VkDevice device, VkDeviceMemory memory, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkFreeMemory, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateFreeMemory]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateFreeMemory(device, memory, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkFreeMemory);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordFreeMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordFreeMemory(device, memory, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchFreeMemory(device, memory, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordFreeMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordFreeMemory(device, memory, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL MapMemory(VkDevice device, VkDeviceMemory memory, VkDeviceSize offset, VkDeviceSize size,
                                         VkMemoryMapFlags flags, void** ppData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkMapMemory, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateMapMemory]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateMapMemory(device, memory, offset, size, flags, ppData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkMapMemory);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordMapMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordMapMemory(device, memory, offset, size, flags, ppData, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchMapMemory(device, memory, offset, size, flags, ppData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordMapMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordMapMemory(device, memory, offset, size, flags, ppData, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL UnmapMemory(VkDevice device, VkDeviceMemory memory) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUnmapMemory, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateUnmapMemory]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateUnmapMemory(device, memory, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkUnmapMemory);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordUnmapMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUnmapMemory(device, memory, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchUnmapMemory(device, memory);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordUnmapMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUnmapMemory(device, memory, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL FlushMappedMemoryRanges(VkDevice device, uint32_t memoryRangeCount,
                                                       const VkMappedMemoryRange* pMemoryRanges) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkFlushMappedMemoryRanges, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateFlushMappedMemoryRanges]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateFlushMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkFlushMappedMemoryRanges);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordFlushMappedMemoryRanges]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordFlushMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchFlushMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordFlushMappedMemoryRanges]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordFlushMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL InvalidateMappedMemoryRanges(VkDevice device, uint32_t memoryRangeCount,
                                                            const VkMappedMemoryRange* pMemoryRanges) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkInvalidateMappedMemoryRanges, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateInvalidateMappedMemoryRanges]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateInvalidateMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkInvalidateMappedMemoryRanges);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordInvalidateMappedMemoryRanges]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordInvalidateMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchInvalidateMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordInvalidateMappedMemoryRanges]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordInvalidateMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetDeviceMemoryCommitment(VkDevice device, VkDeviceMemory memory,
                                                     VkDeviceSize* pCommittedMemoryInBytes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceMemoryCommitment, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceMemoryCommitment]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceMemoryCommitment(device, memory, pCommittedMemoryInBytes, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceMemoryCommitment);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceMemoryCommitment]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceMemoryCommitment(device, memory, pCommittedMemoryInBytes, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceMemoryCommitment(device, memory, pCommittedMemoryInBytes);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceMemoryCommitment]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceMemoryCommitment(device, memory, pCommittedMemoryInBytes, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL BindBufferMemory(VkDevice device, VkBuffer buffer, VkDeviceMemory memory,
                                                VkDeviceSize memoryOffset) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindBufferMemory, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBindBufferMemory]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindBufferMemory(device, buffer, memory, memoryOffset, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindBufferMemory);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindBufferMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindBufferMemory(device, buffer, memory, memoryOffset, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindBufferMemory(device, buffer, memory, memoryOffset);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindBufferMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindBufferMemory(device, buffer, memory, memoryOffset, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindImageMemory(VkDevice device, VkImage image, VkDeviceMemory memory, VkDeviceSize memoryOffset) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindImageMemory, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBindImageMemory]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindImageMemory(device, image, memory, memoryOffset, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindImageMemory);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindImageMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindImageMemory(device, image, memory, memoryOffset, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindImageMemory(device, image, memory, memoryOffset);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindImageMemory]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindImageMemory(device, image, memory, memoryOffset, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetBufferMemoryRequirements(VkDevice device, VkBuffer buffer,
                                                       VkMemoryRequirements* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferMemoryRequirements, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferMemoryRequirements]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferMemoryRequirements(device, buffer, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferMemoryRequirements);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferMemoryRequirements(device, buffer, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetBufferMemoryRequirements(device, buffer, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferMemoryRequirements(device, buffer, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetImageMemoryRequirements(VkDevice device, VkImage image, VkMemoryRequirements* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageMemoryRequirements, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageMemoryRequirements]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageMemoryRequirements(device, image, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageMemoryRequirements);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageMemoryRequirements(device, image, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageMemoryRequirements(device, image, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageMemoryRequirements(device, image, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetImageSparseMemoryRequirements(VkDevice device, VkImage image, uint32_t* pSparseMemoryRequirementCount,
                                                            VkSparseImageMemoryRequirements* pSparseMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageSparseMemoryRequirements, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageSparseMemoryRequirements]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageSparseMemoryRequirements(device, image, pSparseMemoryRequirementCount,
                                                                               pSparseMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageSparseMemoryRequirements);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageSparseMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageSparseMemoryRequirements(device, image, pSparseMemoryRequirementCount,
                                                                     pSparseMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageSparseMemoryRequirements(device, image, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageSparseMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageSparseMemoryRequirements(device, image, pSparseMemoryRequirementCount,
                                                                      pSparseMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceSparseImageFormatProperties(VkPhysicalDevice physicalDevice, VkFormat format,
                                                                        VkImageType type, VkSampleCountFlagBits samples,
                                                                        VkImageUsageFlags usage, VkImageTiling tiling,
                                                                        uint32_t* pPropertyCount,
                                                                        VkSparseImageFormatProperties* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSparseImageFormatProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSparseImageFormatProperties(
                physicalDevice, format, type, samples, usage, tiling, pPropertyCount, pProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSparseImageFormatProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSparseImageFormatProperties(physicalDevice, format, type, samples, usage,
                                                                                 tiling, pPropertyCount, pProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceSparseImageFormatProperties(physicalDevice, format, type, samples, usage, tiling, pPropertyCount,
                                                             pProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSparseImageFormatProperties(physicalDevice, format, type, samples, usage,
                                                                                  tiling, pPropertyCount, pProperties, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL QueueBindSparse(VkQueue queue, uint32_t bindInfoCount, const VkBindSparseInfo* pBindInfo,
                                               VkFence fence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueBindSparse, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateQueueBindSparse]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueBindSparse(queue, bindInfoCount, pBindInfo, fence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueBindSparse);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueBindSparse]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueBindSparse(queue, bindInfoCount, pBindInfo, fence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueueBindSparse(queue, bindInfoCount, pBindInfo, fence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueBindSparse]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordQueueBindSparse(queue, bindInfoCount, pBindInfo, fence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateFence(VkDevice device, const VkFenceCreateInfo* pCreateInfo,
                                           const VkAllocationCallbacks* pAllocator, VkFence* pFence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateFence, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateFence]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateFence(device, pCreateInfo, pAllocator, pFence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateFence);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateFence]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateFence(device, pCreateInfo, pAllocator, pFence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateFence(device, pCreateInfo, pAllocator, pFence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateFence]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateFence(device, pCreateInfo, pAllocator, pFence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyFence(VkDevice device, VkFence fence, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyFence, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyFence]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyFence(device, fence, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyFence);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyFence]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyFence(device, fence, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyFence(device, fence, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyFence]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyFence(device, fence, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL ResetFences(VkDevice device, uint32_t fenceCount, const VkFence* pFences) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetFences, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetFences]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetFences(device, fenceCount, pFences, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetFences);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetFences]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetFences(device, fenceCount, pFences, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchResetFences(device, fenceCount, pFences);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetFences]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetFences(device, fenceCount, pFences, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetFenceStatus(VkDevice device, VkFence fence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetFenceStatus, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetFenceStatus]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetFenceStatus(device, fence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetFenceStatus);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetFenceStatus]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetFenceStatus(device, fence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetFenceStatus(device, fence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetFenceStatus]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetFenceStatus(device, fence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WaitForFences(VkDevice device, uint32_t fenceCount, const VkFence* pFences, VkBool32 waitAll,
                                             uint64_t timeout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkWaitForFences, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateWaitForFences]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateWaitForFences(device, fenceCount, pFences, waitAll, timeout, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkWaitForFences);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordWaitForFences]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordWaitForFences(device, fenceCount, pFences, waitAll, timeout, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchWaitForFences(device, fenceCount, pFences, waitAll, timeout);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordWaitForFences]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordWaitForFences(device, fenceCount, pFences, waitAll, timeout, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSemaphore(VkDevice device, const VkSemaphoreCreateInfo* pCreateInfo,
                                               const VkAllocationCallbacks* pAllocator, VkSemaphore* pSemaphore) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateSemaphore, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateSemaphore]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateSemaphore(device, pCreateInfo, pAllocator, pSemaphore, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateSemaphore);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateSemaphore]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateSemaphore(device, pCreateInfo, pAllocator, pSemaphore, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateSemaphore(device, pCreateInfo, pAllocator, pSemaphore);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateSemaphore]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateSemaphore(device, pCreateInfo, pAllocator, pSemaphore, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySemaphore(VkDevice device, VkSemaphore semaphore, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroySemaphore, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroySemaphore]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroySemaphore(device, semaphore, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroySemaphore);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroySemaphore]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroySemaphore(device, semaphore, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroySemaphore(device, semaphore, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroySemaphore]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroySemaphore(device, semaphore, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateEvent(VkDevice device, const VkEventCreateInfo* pCreateInfo,
                                           const VkAllocationCallbacks* pAllocator, VkEvent* pEvent) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateEvent, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateEvent]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateEvent(device, pCreateInfo, pAllocator, pEvent, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateEvent);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateEvent(device, pCreateInfo, pAllocator, pEvent, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateEvent(device, pCreateInfo, pAllocator, pEvent);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateEvent(device, pCreateInfo, pAllocator, pEvent, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyEvent(VkDevice device, VkEvent event, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyEvent, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyEvent]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyEvent(device, event, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyEvent);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyEvent(device, event, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyEvent(device, event, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyEvent(device, event, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetEventStatus(VkDevice device, VkEvent event) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetEventStatus, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetEventStatus]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetEventStatus(device, event, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetEventStatus);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetEventStatus]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetEventStatus(device, event, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetEventStatus(device, event);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetEventStatus]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetEventStatus(device, event, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SetEvent(VkDevice device, VkEvent event) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSetEvent, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateSetEvent]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSetEvent(device, event, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkSetEvent);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSetEvent(device, event, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchSetEvent(device, event);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSetEvent(device, event, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ResetEvent(VkDevice device, VkEvent event) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetEvent, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetEvent]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetEvent(device, event, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetEvent);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetEvent(device, event, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchResetEvent(device, event);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetEvent(device, event, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateQueryPool(VkDevice device, const VkQueryPoolCreateInfo* pCreateInfo,
                                               const VkAllocationCallbacks* pAllocator, VkQueryPool* pQueryPool) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateQueryPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateQueryPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateQueryPool(device, pCreateInfo, pAllocator, pQueryPool, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateQueryPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateQueryPool(device, pCreateInfo, pAllocator, pQueryPool, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateQueryPool(device, pCreateInfo, pAllocator, pQueryPool);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateQueryPool(device, pCreateInfo, pAllocator, pQueryPool, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyQueryPool(VkDevice device, VkQueryPool queryPool, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyQueryPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyQueryPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyQueryPool(device, queryPool, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyQueryPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyQueryPool(device, queryPool, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyQueryPool(device, queryPool, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyQueryPool(device, queryPool, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetQueryPoolResults(VkDevice device, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount,
                                                   size_t dataSize, void* pData, VkDeviceSize stride, VkQueryResultFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetQueryPoolResults, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetQueryPoolResults]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetQueryPoolResults(device, queryPool, firstQuery, queryCount, dataSize, pData,
                                                                  stride, flags, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetQueryPoolResults);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetQueryPoolResults]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetQueryPoolResults(device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags,
                                                        record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetQueryPoolResults(device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetQueryPoolResults]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetQueryPoolResults(device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags,
                                                         record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyBuffer(VkDevice device, VkBuffer buffer, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyBuffer, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyBuffer(device, buffer, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyBuffer(device, buffer, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyBuffer(device, buffer, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyBuffer(device, buffer, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateBufferView(VkDevice device, const VkBufferViewCreateInfo* pCreateInfo,
                                                const VkAllocationCallbacks* pAllocator, VkBufferView* pView) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateBufferView, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateBufferView]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateBufferView(device, pCreateInfo, pAllocator, pView, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateBufferView);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateBufferView]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateBufferView(device, pCreateInfo, pAllocator, pView, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateBufferView(device, pCreateInfo, pAllocator, pView);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateBufferView]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateBufferView(device, pCreateInfo, pAllocator, pView, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyBufferView(VkDevice device, VkBufferView bufferView, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyBufferView, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyBufferView]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyBufferView(device, bufferView, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyBufferView);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyBufferView]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyBufferView(device, bufferView, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyBufferView(device, bufferView, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyBufferView]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyBufferView(device, bufferView, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateImage(VkDevice device, const VkImageCreateInfo* pCreateInfo,
                                           const VkAllocationCallbacks* pAllocator, VkImage* pImage) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateImage, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateImage(device, pCreateInfo, pAllocator, pImage, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateImage(device, pCreateInfo, pAllocator, pImage, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateImage(device, pCreateInfo, pAllocator, pImage);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateImage(device, pCreateInfo, pAllocator, pImage, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyImage(VkDevice device, VkImage image, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyImage, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyImage(device, image, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyImage(device, image, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyImage(device, image, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyImage(device, image, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetImageSubresourceLayout(VkDevice device, VkImage image, const VkImageSubresource* pSubresource,
                                                     VkSubresourceLayout* pLayout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageSubresourceLayout, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageSubresourceLayout]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageSubresourceLayout(device, image, pSubresource, pLayout, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageSubresourceLayout);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageSubresourceLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageSubresourceLayout(device, image, pSubresource, pLayout, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageSubresourceLayout(device, image, pSubresource, pLayout);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageSubresourceLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageSubresourceLayout(device, image, pSubresource, pLayout, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateImageView(VkDevice device, const VkImageViewCreateInfo* pCreateInfo,
                                               const VkAllocationCallbacks* pAllocator, VkImageView* pView) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateImageView, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateImageView]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateImageView(device, pCreateInfo, pAllocator, pView, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateImageView);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateImageView]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateImageView(device, pCreateInfo, pAllocator, pView, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateImageView(device, pCreateInfo, pAllocator, pView);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateImageView]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateImageView(device, pCreateInfo, pAllocator, pView, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyImageView(VkDevice device, VkImageView imageView, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyImageView, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyImageView]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyImageView(device, imageView, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyImageView);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyImageView]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyImageView(device, imageView, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyImageView(device, imageView, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyImageView]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyImageView(device, imageView, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL DestroyShaderModule(VkDevice device, VkShaderModule shaderModule,
                                               const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyShaderModule, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyShaderModule]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyShaderModule(device, shaderModule, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyShaderModule);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyShaderModule]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyShaderModule(device, shaderModule, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyShaderModule(device, shaderModule, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyShaderModule]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyShaderModule(device, shaderModule, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePipelineCache(VkDevice device, const VkPipelineCacheCreateInfo* pCreateInfo,
                                                   const VkAllocationCallbacks* pAllocator, VkPipelineCache* pPipelineCache) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreatePipelineCache, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreatePipelineCache]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreatePipelineCache(device, pCreateInfo, pAllocator, pPipelineCache, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreatePipelineCache);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreatePipelineCache]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreatePipelineCache(device, pCreateInfo, pAllocator, pPipelineCache, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreatePipelineCache(device, pCreateInfo, pAllocator, pPipelineCache);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreatePipelineCache]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreatePipelineCache(device, pCreateInfo, pAllocator, pPipelineCache, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPipelineCache(VkDevice device, VkPipelineCache pipelineCache,
                                                const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyPipelineCache, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyPipelineCache]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyPipelineCache(device, pipelineCache, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyPipelineCache);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyPipelineCache]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyPipelineCache(device, pipelineCache, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyPipelineCache(device, pipelineCache, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyPipelineCache]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyPipelineCache(device, pipelineCache, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineCacheData(VkDevice device, VkPipelineCache pipelineCache, size_t* pDataSize,
                                                    void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPipelineCacheData, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetPipelineCacheData]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPipelineCacheData(device, pipelineCache, pDataSize, pData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPipelineCacheData);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetPipelineCacheData]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPipelineCacheData(device, pipelineCache, pDataSize, pData, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPipelineCacheData(device, pipelineCache, pDataSize, pData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetPipelineCacheData]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPipelineCacheData(device, pipelineCache, pDataSize, pData, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL MergePipelineCaches(VkDevice device, VkPipelineCache dstCache, uint32_t srcCacheCount,
                                                   const VkPipelineCache* pSrcCaches) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkMergePipelineCaches, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateMergePipelineCaches]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateMergePipelineCaches(device, dstCache, srcCacheCount, pSrcCaches, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkMergePipelineCaches);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordMergePipelineCaches]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordMergePipelineCaches(device, dstCache, srcCacheCount, pSrcCaches, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchMergePipelineCaches(device, dstCache, srcCacheCount, pSrcCaches);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordMergePipelineCaches]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordMergePipelineCaches(device, dstCache, srcCacheCount, pSrcCaches, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPipeline(VkDevice device, VkPipeline pipeline, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyPipeline, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyPipeline]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyPipeline(device, pipeline, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyPipeline);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyPipeline]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyPipeline(device, pipeline, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyPipeline(device, pipeline, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyPipeline]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyPipeline(device, pipeline, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL DestroyPipelineLayout(VkDevice device, VkPipelineLayout pipelineLayout,
                                                 const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyPipelineLayout, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyPipelineLayout]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyPipelineLayout(device, pipelineLayout, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyPipelineLayout);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyPipelineLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyPipelineLayout(device, pipelineLayout, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyPipelineLayout(device, pipelineLayout, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyPipelineLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyPipelineLayout(device, pipelineLayout, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSampler(VkDevice device, const VkSamplerCreateInfo* pCreateInfo,
                                             const VkAllocationCallbacks* pAllocator, VkSampler* pSampler) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateSampler, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateSampler]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateSampler(device, pCreateInfo, pAllocator, pSampler, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateSampler);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateSampler]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateSampler(device, pCreateInfo, pAllocator, pSampler, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateSampler(device, pCreateInfo, pAllocator, pSampler);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateSampler]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateSampler(device, pCreateInfo, pAllocator, pSampler, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySampler(VkDevice device, VkSampler sampler, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroySampler, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroySampler]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroySampler(device, sampler, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroySampler);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroySampler]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroySampler(device, sampler, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroySampler(device, sampler, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroySampler]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroySampler(device, sampler, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorSetLayout(VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo,
                                                         const VkAllocationCallbacks* pAllocator,
                                                         VkDescriptorSetLayout* pSetLayout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDescriptorSetLayout, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateDescriptorSetLayout]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDescriptorSetLayout(device, pCreateInfo, pAllocator, pSetLayout, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDescriptorSetLayout);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateDescriptorSetLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDescriptorSetLayout(device, pCreateInfo, pAllocator, pSetLayout, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDescriptorSetLayout(device, pCreateInfo, pAllocator, pSetLayout);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateDescriptorSetLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDescriptorSetLayout(device, pCreateInfo, pAllocator, pSetLayout, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorSetLayout(VkDevice device, VkDescriptorSetLayout descriptorSetLayout,
                                                      const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDescriptorSetLayout, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyDescriptorSetLayout]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyDescriptorSetLayout(device, descriptorSetLayout, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDescriptorSetLayout);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyDescriptorSetLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDescriptorSetLayout(device, descriptorSetLayout, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDescriptorSetLayout(device, descriptorSetLayout, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyDescriptorSetLayout]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDescriptorSetLayout(device, descriptorSetLayout, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorPool(VkDevice device, const VkDescriptorPoolCreateInfo* pCreateInfo,
                                                    const VkAllocationCallbacks* pAllocator, VkDescriptorPool* pDescriptorPool) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDescriptorPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateDescriptorPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDescriptorPool(device, pCreateInfo, pAllocator, pDescriptorPool, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDescriptorPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateDescriptorPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDescriptorPool(device, pCreateInfo, pAllocator, pDescriptorPool, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDescriptorPool(device, pCreateInfo, pAllocator, pDescriptorPool);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateDescriptorPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDescriptorPool(device, pCreateInfo, pAllocator, pDescriptorPool, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorPool(VkDevice device, VkDescriptorPool descriptorPool,
                                                 const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDescriptorPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyDescriptorPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyDescriptorPool(device, descriptorPool, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDescriptorPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyDescriptorPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDescriptorPool(device, descriptorPool, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDescriptorPool(device, descriptorPool, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyDescriptorPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDescriptorPool(device, descriptorPool, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL ResetDescriptorPool(VkDevice device, VkDescriptorPool descriptorPool,
                                                   VkDescriptorPoolResetFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetDescriptorPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetDescriptorPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetDescriptorPool(device, descriptorPool, flags, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetDescriptorPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetDescriptorPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetDescriptorPool(device, descriptorPool, flags, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchResetDescriptorPool(device, descriptorPool, flags);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetDescriptorPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetDescriptorPool(device, descriptorPool, flags, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL FreeDescriptorSets(VkDevice device, VkDescriptorPool descriptorPool, uint32_t descriptorSetCount,
                                                  const VkDescriptorSet* pDescriptorSets) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkFreeDescriptorSets, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateFreeDescriptorSets]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateFreeDescriptorSets(device, descriptorPool, descriptorSetCount, pDescriptorSets,
                                                                 error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkFreeDescriptorSets);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordFreeDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordFreeDescriptorSets(device, descriptorPool, descriptorSetCount, pDescriptorSets, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchFreeDescriptorSets(device, descriptorPool, descriptorSetCount, pDescriptorSets);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordFreeDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordFreeDescriptorSets(device, descriptorPool, descriptorSetCount, pDescriptorSets, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL UpdateDescriptorSets(VkDevice device, uint32_t descriptorWriteCount,
                                                const VkWriteDescriptorSet* pDescriptorWrites, uint32_t descriptorCopyCount,
                                                const VkCopyDescriptorSet* pDescriptorCopies) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUpdateDescriptorSets, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateUpdateDescriptorSets]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateUpdateDescriptorSets(device, descriptorWriteCount, pDescriptorWrites,
                                                                   descriptorCopyCount, pDescriptorCopies, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkUpdateDescriptorSets);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordUpdateDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUpdateDescriptorSets(device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount,
                                                         pDescriptorCopies, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchUpdateDescriptorSets(device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount, pDescriptorCopies);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordUpdateDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUpdateDescriptorSets(device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount,
                                                          pDescriptorCopies, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateFramebuffer(VkDevice device, const VkFramebufferCreateInfo* pCreateInfo,
                                                 const VkAllocationCallbacks* pAllocator, VkFramebuffer* pFramebuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateFramebuffer, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateFramebuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateFramebuffer(device, pCreateInfo, pAllocator, pFramebuffer, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateFramebuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateFramebuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateFramebuffer(device, pCreateInfo, pAllocator, pFramebuffer, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateFramebuffer(device, pCreateInfo, pAllocator, pFramebuffer);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateFramebuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateFramebuffer(device, pCreateInfo, pAllocator, pFramebuffer, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyFramebuffer(VkDevice device, VkFramebuffer framebuffer, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyFramebuffer, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyFramebuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyFramebuffer(device, framebuffer, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyFramebuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyFramebuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyFramebuffer(device, framebuffer, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyFramebuffer(device, framebuffer, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyFramebuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyFramebuffer(device, framebuffer, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRenderPass(VkDevice device, const VkRenderPassCreateInfo* pCreateInfo,
                                                const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateRenderPass, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateRenderPass]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateRenderPass(device, pCreateInfo, pAllocator, pRenderPass, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateRenderPass);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateRenderPass(device, pCreateInfo, pAllocator, pRenderPass, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateRenderPass(device, pCreateInfo, pAllocator, pRenderPass);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateRenderPass(device, pCreateInfo, pAllocator, pRenderPass, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyRenderPass(VkDevice device, VkRenderPass renderPass, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyRenderPass, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyRenderPass]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyRenderPass(device, renderPass, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyRenderPass);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyRenderPass(device, renderPass, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyRenderPass(device, renderPass, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyRenderPass(device, renderPass, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetRenderAreaGranularity(VkDevice device, VkRenderPass renderPass, VkExtent2D* pGranularity) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetRenderAreaGranularity, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetRenderAreaGranularity]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetRenderAreaGranularity(device, renderPass, pGranularity, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetRenderAreaGranularity);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetRenderAreaGranularity]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetRenderAreaGranularity(device, renderPass, pGranularity, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetRenderAreaGranularity(device, renderPass, pGranularity);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetRenderAreaGranularity]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetRenderAreaGranularity(device, renderPass, pGranularity, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateCommandPool(VkDevice device, const VkCommandPoolCreateInfo* pCreateInfo,
                                                 const VkAllocationCallbacks* pAllocator, VkCommandPool* pCommandPool) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateCommandPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateCommandPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateCommandPool(device, pCreateInfo, pAllocator, pCommandPool, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateCommandPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateCommandPool(device, pCreateInfo, pAllocator, pCommandPool, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateCommandPool(device, pCreateInfo, pAllocator, pCommandPool);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateCommandPool(device, pCreateInfo, pAllocator, pCommandPool, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyCommandPool(VkDevice device, VkCommandPool commandPool, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyCommandPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyCommandPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyCommandPool(device, commandPool, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyCommandPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyCommandPool(device, commandPool, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyCommandPool(device, commandPool, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyCommandPool(device, commandPool, pAllocator, record_obj);
        }
    }

    {
        auto lock = WriteLockGuard(secondary_cb_map_mutex);
        for (auto item = secondary_cb_map.begin(); item != secondary_cb_map.end();) {
            if (item->second == commandPool) {
                item = secondary_cb_map.erase(item);
            } else {
                ++item;
            }
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL ResetCommandPool(VkDevice device, VkCommandPool commandPool, VkCommandPoolResetFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetCommandPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetCommandPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetCommandPool(device, commandPool, flags, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetCommandPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetCommandPool(device, commandPool, flags, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchResetCommandPool(device, commandPool, flags);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetCommandPool(device, commandPool, flags, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AllocateCommandBuffers(VkDevice device, const VkCommandBufferAllocateInfo* pAllocateInfo,
                                                      VkCommandBuffer* pCommandBuffers) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAllocateCommandBuffers, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateAllocateCommandBuffers]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAllocateCommandBuffers(device, pAllocateInfo, pCommandBuffers, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAllocateCommandBuffers);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordAllocateCommandBuffers]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAllocateCommandBuffers(device, pAllocateInfo, pCommandBuffers, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAllocateCommandBuffers(device, pAllocateInfo, pCommandBuffers);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordAllocateCommandBuffers]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAllocateCommandBuffers(device, pAllocateInfo, pCommandBuffers, record_obj);
        }
    }

    if ((result == VK_SUCCESS) && pAllocateInfo && (pAllocateInfo->level == VK_COMMAND_BUFFER_LEVEL_SECONDARY)) {
        auto lock = WriteLockGuard(secondary_cb_map_mutex);
        for (uint32_t cb_index = 0; cb_index < pAllocateInfo->commandBufferCount; cb_index++) {
            secondary_cb_map.emplace(pCommandBuffers[cb_index], pAllocateInfo->commandPool);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL FreeCommandBuffers(VkDevice device, VkCommandPool commandPool, uint32_t commandBufferCount,
                                              const VkCommandBuffer* pCommandBuffers) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkFreeCommandBuffers, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateFreeCommandBuffers]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateFreeCommandBuffers(device, commandPool, commandBufferCount, pCommandBuffers, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkFreeCommandBuffers);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordFreeCommandBuffers]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordFreeCommandBuffers(device, commandPool, commandBufferCount, pCommandBuffers, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchFreeCommandBuffers(device, commandPool, commandBufferCount, pCommandBuffers);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordFreeCommandBuffers]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordFreeCommandBuffers(device, commandPool, commandBufferCount, pCommandBuffers, record_obj);
        }
    }

    {
        auto lock = WriteLockGuard(secondary_cb_map_mutex);
        for (uint32_t cb_index = 0; cb_index < commandBufferCount; cb_index++) {
            secondary_cb_map.erase(pCommandBuffers[cb_index]);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL EndCommandBuffer(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkEndCommandBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateEndCommandBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateEndCommandBuffer(commandBuffer, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkEndCommandBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordEndCommandBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordEndCommandBuffer(commandBuffer, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchEndCommandBuffer(commandBuffer);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordEndCommandBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordEndCommandBuffer(commandBuffer, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ResetCommandBuffer(VkCommandBuffer commandBuffer, VkCommandBufferResetFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetCommandBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetCommandBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetCommandBuffer(commandBuffer, flags, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetCommandBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetCommandBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetCommandBuffer(commandBuffer, flags, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchResetCommandBuffer(commandBuffer, flags);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetCommandBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetCommandBuffer(commandBuffer, flags, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindPipeline(VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint,
                                           VkPipeline pipeline) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindPipeline, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindPipeline]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindPipeline(commandBuffer, pipelineBindPoint, pipeline, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindPipeline);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindPipeline]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindPipeline(commandBuffer, pipelineBindPoint, pipeline, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindPipeline(commandBuffer, pipelineBindPoint, pipeline);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindPipeline]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindPipeline(commandBuffer, pipelineBindPoint, pipeline, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewport(VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount,
                                          const VkViewport* pViewports) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetViewport, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetViewport]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetViewport(commandBuffer, firstViewport, viewportCount, pViewports, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetViewport);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetViewport]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetViewport(commandBuffer, firstViewport, viewportCount, pViewports, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetViewport(commandBuffer, firstViewport, viewportCount, pViewports);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetViewport]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetViewport(commandBuffer, firstViewport, viewportCount, pViewports, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetScissor(VkCommandBuffer commandBuffer, uint32_t firstScissor, uint32_t scissorCount,
                                         const VkRect2D* pScissors) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetScissor, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetScissor]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetScissor(commandBuffer, firstScissor, scissorCount, pScissors, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetScissor);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetScissor]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetScissor(commandBuffer, firstScissor, scissorCount, pScissors, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetScissor(commandBuffer, firstScissor, scissorCount, pScissors);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetScissor]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetScissor(commandBuffer, firstScissor, scissorCount, pScissors, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetLineWidth(VkCommandBuffer commandBuffer, float lineWidth) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetLineWidth, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetLineWidth]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetLineWidth(commandBuffer, lineWidth, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetLineWidth);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetLineWidth]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetLineWidth(commandBuffer, lineWidth, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetLineWidth(commandBuffer, lineWidth);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetLineWidth]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetLineWidth(commandBuffer, lineWidth, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBias(VkCommandBuffer commandBuffer, float depthBiasConstantFactor, float depthBiasClamp,
                                           float depthBiasSlopeFactor) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthBias, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthBias]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthBias(commandBuffer, depthBiasConstantFactor, depthBiasClamp,
                                                              depthBiasSlopeFactor, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthBias);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthBias]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthBias(commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor,
                                                    record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthBias(commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthBias]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthBias(commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor,
                                                     record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetBlendConstants(VkCommandBuffer commandBuffer, const float blendConstants[4]) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetBlendConstants, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetBlendConstants]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetBlendConstants(commandBuffer, blendConstants, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetBlendConstants);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetBlendConstants]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetBlendConstants(commandBuffer, blendConstants, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetBlendConstants(commandBuffer, blendConstants);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetBlendConstants]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetBlendConstants(commandBuffer, blendConstants, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBounds(VkCommandBuffer commandBuffer, float minDepthBounds, float maxDepthBounds) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthBounds, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthBounds]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthBounds(commandBuffer, minDepthBounds, maxDepthBounds, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthBounds);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthBounds]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthBounds(commandBuffer, minDepthBounds, maxDepthBounds, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthBounds(commandBuffer, minDepthBounds, maxDepthBounds);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthBounds]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthBounds(commandBuffer, minDepthBounds, maxDepthBounds, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilCompareMask(VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask,
                                                    uint32_t compareMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilCompareMask, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilCompareMask]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilCompareMask(commandBuffer, faceMask, compareMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilCompareMask);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilCompareMask]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilCompareMask(commandBuffer, faceMask, compareMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilCompareMask(commandBuffer, faceMask, compareMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilCompareMask]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilCompareMask(commandBuffer, faceMask, compareMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilWriteMask(VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t writeMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilWriteMask, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilWriteMask]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilWriteMask(commandBuffer, faceMask, writeMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilWriteMask);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilWriteMask]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilWriteMask(commandBuffer, faceMask, writeMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilWriteMask(commandBuffer, faceMask, writeMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilWriteMask]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilWriteMask(commandBuffer, faceMask, writeMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilReference(VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t reference) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilReference, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilReference]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilReference(commandBuffer, faceMask, reference, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilReference);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilReference]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilReference(commandBuffer, faceMask, reference, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilReference(commandBuffer, faceMask, reference);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilReference]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilReference(commandBuffer, faceMask, reference, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindDescriptorSets(VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint,
                                                 VkPipelineLayout layout, uint32_t firstSet, uint32_t descriptorSetCount,
                                                 const VkDescriptorSet* pDescriptorSets, uint32_t dynamicOffsetCount,
                                                 const uint32_t* pDynamicOffsets) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindDescriptorSets, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindDescriptorSets]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindDescriptorSets(commandBuffer, pipelineBindPoint, layout, firstSet,
                                                                    descriptorSetCount, pDescriptorSets, dynamicOffsetCount,
                                                                    pDynamicOffsets, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindDescriptorSets);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindDescriptorSets(commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount,
                                                          pDescriptorSets, dynamicOffsetCount, pDynamicOffsets, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindDescriptorSets(commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets,
                                      dynamicOffsetCount, pDynamicOffsets);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindDescriptorSets]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindDescriptorSets(commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount,
                                                           pDescriptorSets, dynamicOffsetCount, pDynamicOffsets, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindIndexBuffer(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                              VkIndexType indexType) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindIndexBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindIndexBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindIndexBuffer(commandBuffer, buffer, offset, indexType, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindIndexBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindIndexBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindIndexBuffer(commandBuffer, buffer, offset, indexType, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindIndexBuffer(commandBuffer, buffer, offset, indexType);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindIndexBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindIndexBuffer(commandBuffer, buffer, offset, indexType, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindVertexBuffers(VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount,
                                                const VkBuffer* pBuffers, const VkDeviceSize* pOffsets) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindVertexBuffers, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindVertexBuffers]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindVertexBuffers(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets,
                                                                   error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindVertexBuffers);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindVertexBuffers]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindVertexBuffers(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindVertexBuffers(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindVertexBuffers]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindVertexBuffers(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets,
                                                          record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDraw(VkCommandBuffer commandBuffer, uint32_t vertexCount, uint32_t instanceCount,
                                   uint32_t firstVertex, uint32_t firstInstance) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDraw, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDraw]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCmdDraw(commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDraw);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDraw]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDraw(commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDraw(commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDraw]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDraw(commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexed(VkCommandBuffer commandBuffer, uint32_t indexCount, uint32_t instanceCount,
                                          uint32_t firstIndex, int32_t vertexOffset, uint32_t firstInstance) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndexed, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndexed]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndexed(commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset,
                                                             firstInstance, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndexed);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndexed]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndexed(commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset,
                                                   firstInstance, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndexed(commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset, firstInstance);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndexed]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndexed(commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset,
                                                    firstInstance, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirect(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount,
                                           uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndirect, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndirect]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndirect(commandBuffer, buffer, offset, drawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndirect);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndirect]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndirect(commandBuffer, buffer, offset, drawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndirect(commandBuffer, buffer, offset, drawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndirect]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndirect(commandBuffer, buffer, offset, drawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirect(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                  uint32_t drawCount, uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndexedIndirect, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndexedIndirect]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndexedIndirect(commandBuffer, buffer, offset, drawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndexedIndirect);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndexedIndirect]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndexedIndirect(commandBuffer, buffer, offset, drawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndexedIndirect(commandBuffer, buffer, offset, drawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndexedIndirect]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndexedIndirect(commandBuffer, buffer, offset, drawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatch(VkCommandBuffer commandBuffer, uint32_t groupCountX, uint32_t groupCountY,
                                       uint32_t groupCountZ) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatch, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatch]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatch(commandBuffer, groupCountX, groupCountY, groupCountZ, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatch);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatch]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatch(commandBuffer, groupCountX, groupCountY, groupCountZ, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatch(commandBuffer, groupCountX, groupCountY, groupCountZ);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatch]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatch(commandBuffer, groupCountX, groupCountY, groupCountZ, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchIndirect(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatchIndirect, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatchIndirect]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatchIndirect(commandBuffer, buffer, offset, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatchIndirect);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatchIndirect]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatchIndirect(commandBuffer, buffer, offset, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatchIndirect(commandBuffer, buffer, offset);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatchIndirect]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatchIndirect(commandBuffer, buffer, offset, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBuffer(VkCommandBuffer commandBuffer, VkBuffer srcBuffer, VkBuffer dstBuffer,
                                         uint32_t regionCount, const VkBufferCopy* pRegions) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyBuffer(commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyBuffer(commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyBuffer(commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyBuffer(commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImage(VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout,
                                        VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount,
                                        const VkImageCopy* pRegions) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyImage, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout,
                                                           regionCount, pRegions, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount,
                                                 pRegions, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount,
                                                  pRegions, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBlitImage(VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout,
                                        VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount,
                                        const VkImageBlit* pRegions, VkFilter filter) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBlitImage, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBlitImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBlitImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout,
                                                           regionCount, pRegions, filter, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBlitImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBlitImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBlitImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount,
                                                 pRegions, filter, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBlitImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBlitImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBlitImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount,
                                                  pRegions, filter, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBufferToImage(VkCommandBuffer commandBuffer, VkBuffer srcBuffer, VkImage dstImage,
                                                VkImageLayout dstImageLayout, uint32_t regionCount,
                                                const VkBufferImageCopy* pRegions) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyBufferToImage, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyBufferToImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyBufferToImage(commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount,
                                                                   pRegions, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyBufferToImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyBufferToImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyBufferToImage(commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions,
                                                         record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyBufferToImage(commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyBufferToImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyBufferToImage(commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions,
                                                          record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImageToBuffer(VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout,
                                                VkBuffer dstBuffer, uint32_t regionCount, const VkBufferImageCopy* pRegions) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyImageToBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyImageToBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyImageToBuffer(commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount,
                                                                   pRegions, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyImageToBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyImageToBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyImageToBuffer(commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions,
                                                         record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyImageToBuffer(commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyImageToBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyImageToBuffer(commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions,
                                                          record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdUpdateBuffer(VkCommandBuffer commandBuffer, VkBuffer dstBuffer, VkDeviceSize dstOffset,
                                           VkDeviceSize dataSize, const void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdUpdateBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdUpdateBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdUpdateBuffer(commandBuffer, dstBuffer, dstOffset, dataSize, pData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdUpdateBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdUpdateBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdUpdateBuffer(commandBuffer, dstBuffer, dstOffset, dataSize, pData, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdUpdateBuffer(commandBuffer, dstBuffer, dstOffset, dataSize, pData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdUpdateBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdUpdateBuffer(commandBuffer, dstBuffer, dstOffset, dataSize, pData, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdFillBuffer(VkCommandBuffer commandBuffer, VkBuffer dstBuffer, VkDeviceSize dstOffset,
                                         VkDeviceSize size, uint32_t data) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdFillBuffer, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdFillBuffer]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdFillBuffer(commandBuffer, dstBuffer, dstOffset, size, data, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdFillBuffer);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdFillBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdFillBuffer(commandBuffer, dstBuffer, dstOffset, size, data, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdFillBuffer(commandBuffer, dstBuffer, dstOffset, size, data);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdFillBuffer]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdFillBuffer(commandBuffer, dstBuffer, dstOffset, size, data, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdClearColorImage(VkCommandBuffer commandBuffer, VkImage image, VkImageLayout imageLayout,
                                              const VkClearColorValue* pColor, uint32_t rangeCount,
                                              const VkImageSubresourceRange* pRanges) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdClearColorImage, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdClearColorImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdClearColorImage(commandBuffer, image, imageLayout, pColor, rangeCount, pRanges,
                                                                 error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdClearColorImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdClearColorImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdClearColorImage(commandBuffer, image, imageLayout, pColor, rangeCount, pRanges, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdClearColorImage(commandBuffer, image, imageLayout, pColor, rangeCount, pRanges);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdClearColorImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdClearColorImage(commandBuffer, image, imageLayout, pColor, rangeCount, pRanges, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdClearDepthStencilImage(VkCommandBuffer commandBuffer, VkImage image, VkImageLayout imageLayout,
                                                     const VkClearDepthStencilValue* pDepthStencil, uint32_t rangeCount,
                                                     const VkImageSubresourceRange* pRanges) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdClearDepthStencilImage, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdClearDepthStencilImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdClearDepthStencilImage(commandBuffer, image, imageLayout, pDepthStencil,
                                                                        rangeCount, pRanges, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdClearDepthStencilImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdClearDepthStencilImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdClearDepthStencilImage(commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges,
                                                              record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdClearDepthStencilImage(commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdClearDepthStencilImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdClearDepthStencilImage(commandBuffer, image, imageLayout, pDepthStencil, rangeCount,
                                                               pRanges, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdClearAttachments(VkCommandBuffer commandBuffer, uint32_t attachmentCount,
                                               const VkClearAttachment* pAttachments, uint32_t rectCount,
                                               const VkClearRect* pRects) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdClearAttachments, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdClearAttachments]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdClearAttachments(commandBuffer, attachmentCount, pAttachments, rectCount, pRects,
                                                                  error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdClearAttachments);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdClearAttachments]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdClearAttachments(commandBuffer, attachmentCount, pAttachments, rectCount, pRects,
                                                        record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdClearAttachments(commandBuffer, attachmentCount, pAttachments, rectCount, pRects);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdClearAttachments]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdClearAttachments(commandBuffer, attachmentCount, pAttachments, rectCount, pRects,
                                                         record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResolveImage(VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout,
                                           VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount,
                                           const VkImageResolve* pRegions) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResolveImage, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResolveImage]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResolveImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout,
                                                              regionCount, pRegions, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResolveImage);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResolveImage]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResolveImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount,
                                                    pRegions, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResolveImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResolveImage]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResolveImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount,
                                                     pRegions, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetEvent(VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags stageMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetEvent, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetEvent]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetEvent(commandBuffer, event, stageMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetEvent);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetEvent(commandBuffer, event, stageMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetEvent(commandBuffer, event, stageMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetEvent(commandBuffer, event, stageMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResetEvent(VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags stageMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResetEvent, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResetEvent]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResetEvent(commandBuffer, event, stageMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResetEvent);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResetEvent(commandBuffer, event, stageMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResetEvent(commandBuffer, event, stageMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResetEvent]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResetEvent(commandBuffer, event, stageMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdWaitEvents(VkCommandBuffer commandBuffer, uint32_t eventCount, const VkEvent* pEvents,
                                         VkPipelineStageFlags srcStageMask, VkPipelineStageFlags dstStageMask,
                                         uint32_t memoryBarrierCount, const VkMemoryBarrier* pMemoryBarriers,
                                         uint32_t bufferMemoryBarrierCount, const VkBufferMemoryBarrier* pBufferMemoryBarriers,
                                         uint32_t imageMemoryBarrierCount, const VkImageMemoryBarrier* pImageMemoryBarriers) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWaitEvents, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWaitEvents]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWaitEvents(
                commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers,
                bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWaitEvents);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWaitEvents]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWaitEvents(commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask,
                                                  memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount,
                                                  pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWaitEvents(commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers,
                              bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWaitEvents]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWaitEvents(
                commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers,
                bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPipelineBarrier(VkCommandBuffer commandBuffer, VkPipelineStageFlags srcStageMask,
                                              VkPipelineStageFlags dstStageMask, VkDependencyFlags dependencyFlags,
                                              uint32_t memoryBarrierCount, const VkMemoryBarrier* pMemoryBarriers,
                                              uint32_t bufferMemoryBarrierCount, const VkBufferMemoryBarrier* pBufferMemoryBarriers,
                                              uint32_t imageMemoryBarrierCount, const VkImageMemoryBarrier* pImageMemoryBarriers) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPipelineBarrier, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPipelineBarrier]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPipelineBarrier(
                commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers,
                bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPipelineBarrier);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPipelineBarrier]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPipelineBarrier(
                commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers,
                bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPipelineBarrier(commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers,
                                   bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPipelineBarrier]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPipelineBarrier(
                commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers,
                bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginQuery(VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query,
                                         VkQueryControlFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginQuery, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginQuery]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginQuery(commandBuffer, queryPool, query, flags, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginQuery);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginQuery]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginQuery(commandBuffer, queryPool, query, flags, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginQuery(commandBuffer, queryPool, query, flags);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginQuery]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginQuery(commandBuffer, queryPool, query, flags, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndQuery(VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndQuery, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndQuery]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndQuery(commandBuffer, queryPool, query, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndQuery);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndQuery]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndQuery(commandBuffer, queryPool, query, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndQuery(commandBuffer, queryPool, query);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndQuery]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndQuery(commandBuffer, queryPool, query, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResetQueryPool(VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t firstQuery,
                                             uint32_t queryCount) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResetQueryPool, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResetQueryPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResetQueryPool(commandBuffer, queryPool, firstQuery, queryCount, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResetQueryPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResetQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResetQueryPool(commandBuffer, queryPool, firstQuery, queryCount, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResetQueryPool(commandBuffer, queryPool, firstQuery, queryCount);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResetQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResetQueryPool(commandBuffer, queryPool, firstQuery, queryCount, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdWriteTimestamp(VkCommandBuffer commandBuffer, VkPipelineStageFlagBits pipelineStage,
                                             VkQueryPool queryPool, uint32_t query) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWriteTimestamp, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWriteTimestamp]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWriteTimestamp(commandBuffer, pipelineStage, queryPool, query, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWriteTimestamp);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWriteTimestamp]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWriteTimestamp(commandBuffer, pipelineStage, queryPool, query, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWriteTimestamp(commandBuffer, pipelineStage, queryPool, query);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWriteTimestamp]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWriteTimestamp(commandBuffer, pipelineStage, queryPool, query, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyQueryPoolResults(VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t firstQuery,
                                                   uint32_t queryCount, VkBuffer dstBuffer, VkDeviceSize dstOffset,
                                                   VkDeviceSize stride, VkQueryResultFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyQueryPoolResults, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyQueryPoolResults]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyQueryPoolResults(commandBuffer, queryPool, firstQuery, queryCount, dstBuffer,
                                                                      dstOffset, stride, flags, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyQueryPoolResults);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyQueryPoolResults]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyQueryPoolResults(commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset,
                                                            stride, flags, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyQueryPoolResults(commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset, stride, flags);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyQueryPoolResults]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyQueryPoolResults(commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset,
                                                             stride, flags, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPushConstants(VkCommandBuffer commandBuffer, VkPipelineLayout layout, VkShaderStageFlags stageFlags,
                                            uint32_t offset, uint32_t size, const void* pValues) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPushConstants, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPushConstants]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPushConstants(commandBuffer, layout, stageFlags, offset, size, pValues, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPushConstants);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPushConstants]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPushConstants(commandBuffer, layout, stageFlags, offset, size, pValues, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPushConstants(commandBuffer, layout, stageFlags, offset, size, pValues);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPushConstants]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPushConstants(commandBuffer, layout, stageFlags, offset, size, pValues, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderPass(VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin,
                                              VkSubpassContents contents) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginRenderPass, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginRenderPass]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginRenderPass(commandBuffer, pRenderPassBegin, contents, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginRenderPass);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginRenderPass(commandBuffer, pRenderPassBegin, contents, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginRenderPass(commandBuffer, pRenderPassBegin, contents);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginRenderPass(commandBuffer, pRenderPassBegin, contents, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdNextSubpass(VkCommandBuffer commandBuffer, VkSubpassContents contents) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdNextSubpass, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdNextSubpass]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdNextSubpass(commandBuffer, contents, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdNextSubpass);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdNextSubpass]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdNextSubpass(commandBuffer, contents, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdNextSubpass(commandBuffer, contents);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdNextSubpass]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdNextSubpass(commandBuffer, contents, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderPass(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndRenderPass, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndRenderPass]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndRenderPass(commandBuffer, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndRenderPass);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndRenderPass(commandBuffer, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndRenderPass(commandBuffer);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndRenderPass]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndRenderPass(commandBuffer, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdExecuteCommands(VkCommandBuffer commandBuffer, uint32_t commandBufferCount,
                                              const VkCommandBuffer* pCommandBuffers) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdExecuteCommands, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdExecuteCommands]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdExecuteCommands(commandBuffer, commandBufferCount, pCommandBuffers, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdExecuteCommands);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdExecuteCommands]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdExecuteCommands(commandBuffer, commandBufferCount, pCommandBuffers, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdExecuteCommands(commandBuffer, commandBufferCount, pCommandBuffers);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdExecuteCommands]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdExecuteCommands(commandBuffer, commandBufferCount, pCommandBuffers, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL BindBufferMemory2(VkDevice device, uint32_t bindInfoCount,
                                                 const VkBindBufferMemoryInfo* pBindInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindBufferMemory2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBindBufferMemory2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindBufferMemory2(device, bindInfoCount, pBindInfos, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindBufferMemory2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindBufferMemory2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindBufferMemory2(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindBufferMemory2(device, bindInfoCount, pBindInfos);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindBufferMemory2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindBufferMemory2(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindImageMemory2(VkDevice device, uint32_t bindInfoCount, const VkBindImageMemoryInfo* pBindInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindImageMemory2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBindImageMemory2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindImageMemory2(device, bindInfoCount, pBindInfos, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindImageMemory2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindImageMemory2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindImageMemory2(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindImageMemory2(device, bindInfoCount, pBindInfos);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindImageMemory2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindImageMemory2(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetDeviceGroupPeerMemoryFeatures(VkDevice device, uint32_t heapIndex, uint32_t localDeviceIndex,
                                                            uint32_t remoteDeviceIndex,
                                                            VkPeerMemoryFeatureFlags* pPeerMemoryFeatures) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceGroupPeerMemoryFeatures, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceGroupPeerMemoryFeatures]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceGroupPeerMemoryFeatures(device, heapIndex, localDeviceIndex,
                                                                               remoteDeviceIndex, pPeerMemoryFeatures, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceGroupPeerMemoryFeatures);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceGroupPeerMemoryFeatures]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceGroupPeerMemoryFeatures(device, heapIndex, localDeviceIndex, remoteDeviceIndex,
                                                                     pPeerMemoryFeatures, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceGroupPeerMemoryFeatures(device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceGroupPeerMemoryFeatures]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceGroupPeerMemoryFeatures(device, heapIndex, localDeviceIndex, remoteDeviceIndex,
                                                                      pPeerMemoryFeatures, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDeviceMask(VkCommandBuffer commandBuffer, uint32_t deviceMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDeviceMask, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDeviceMask]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDeviceMask(commandBuffer, deviceMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDeviceMask);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDeviceMask]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDeviceMask(commandBuffer, deviceMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDeviceMask(commandBuffer, deviceMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDeviceMask]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDeviceMask(commandBuffer, deviceMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchBase(VkCommandBuffer commandBuffer, uint32_t baseGroupX, uint32_t baseGroupY,
                                           uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatchBase, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatchBase]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatchBase(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX,
                                                              groupCountY, groupCountZ, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatchBase);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatchBase]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatchBase(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY,
                                                    groupCountZ, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatchBase(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatchBase]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatchBase(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY,
                                                     groupCountZ, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDeviceGroups(VkInstance instance, uint32_t* pPhysicalDeviceGroupCount,
                                                             VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkEnumeratePhysicalDeviceGroups, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateEnumeratePhysicalDeviceGroups(instance, pPhysicalDeviceGroupCount,
                                                                            pPhysicalDeviceGroupProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkEnumeratePhysicalDeviceGroups);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordEnumeratePhysicalDeviceGroups(instance, pPhysicalDeviceGroupCount,
                                                                  pPhysicalDeviceGroupProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchEnumeratePhysicalDeviceGroups(instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordEnumeratePhysicalDeviceGroups(instance, pPhysicalDeviceGroupCount,
                                                                   pPhysicalDeviceGroupProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetImageMemoryRequirements2(VkDevice device, const VkImageMemoryRequirementsInfo2* pInfo,
                                                       VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageMemoryRequirements2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageMemoryRequirements2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageMemoryRequirements2(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageMemoryRequirements2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageMemoryRequirements2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageMemoryRequirements2(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageMemoryRequirements2(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageMemoryRequirements2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageMemoryRequirements2(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetBufferMemoryRequirements2(VkDevice device, const VkBufferMemoryRequirementsInfo2* pInfo,
                                                        VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferMemoryRequirements2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferMemoryRequirements2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferMemoryRequirements2(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferMemoryRequirements2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferMemoryRequirements2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferMemoryRequirements2(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetBufferMemoryRequirements2(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferMemoryRequirements2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferMemoryRequirements2(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetImageSparseMemoryRequirements2(VkDevice device, const VkImageSparseMemoryRequirementsInfo2* pInfo,
                                                             uint32_t* pSparseMemoryRequirementCount,
                                                             VkSparseImageMemoryRequirements2* pSparseMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageSparseMemoryRequirements2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageSparseMemoryRequirements2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageSparseMemoryRequirements2(device, pInfo, pSparseMemoryRequirementCount,
                                                                                pSparseMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageSparseMemoryRequirements2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageSparseMemoryRequirements2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageSparseMemoryRequirements2(device, pInfo, pSparseMemoryRequirementCount,
                                                                      pSparseMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageSparseMemoryRequirements2(device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageSparseMemoryRequirements2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageSparseMemoryRequirements2(device, pInfo, pSparseMemoryRequirementCount,
                                                                       pSparseMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFeatures2(VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures2* pFeatures) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFeatures2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceFeatures2(physicalDevice, pFeatures, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFeatures2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFeatures2(physicalDevice, pFeatures, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceFeatures2(physicalDevice, pFeatures);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFeatures2(physicalDevice, pFeatures, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceProperties2(VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties2* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceProperties2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceProperties2(physicalDevice, pProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceProperties2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceProperties2(physicalDevice, pProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceProperties2(physicalDevice, pProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceProperties2(physicalDevice, pProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFormatProperties2(VkPhysicalDevice physicalDevice, VkFormat format,
                                                              VkFormatProperties2* pFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFormatProperties2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateGetPhysicalDeviceFormatProperties2(physicalDevice, format, pFormatProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFormatProperties2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFormatProperties2(physicalDevice, format, pFormatProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceFormatProperties2(physicalDevice, format, pFormatProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFormatProperties2(physicalDevice, format, pFormatProperties, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceImageFormatProperties2(VkPhysicalDevice physicalDevice,
                                                                       const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo,
                                                                       VkImageFormatProperties2* pImageFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceImageFormatProperties2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceImageFormatProperties2(physicalDevice, pImageFormatInfo,
                                                                                      pImageFormatProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceImageFormatProperties2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceImageFormatProperties2(physicalDevice, pImageFormatInfo,
                                                                            pImageFormatProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceImageFormatProperties2(physicalDevice, pImageFormatInfo, pImageFormatProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceImageFormatProperties2(physicalDevice, pImageFormatInfo,
                                                                             pImageFormatProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyProperties2(VkPhysicalDevice physicalDevice,
                                                                   uint32_t* pQueueFamilyPropertyCount,
                                                                   VkQueueFamilyProperties2* pQueueFamilyProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyProperties2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceQueueFamilyProperties2(physicalDevice, pQueueFamilyPropertyCount,
                                                                                      pQueueFamilyProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyProperties2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceQueueFamilyProperties2(physicalDevice, pQueueFamilyPropertyCount,
                                                                            pQueueFamilyProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceQueueFamilyProperties2(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceQueueFamilyProperties2(physicalDevice, pQueueFamilyPropertyCount,
                                                                             pQueueFamilyProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMemoryProperties2(VkPhysicalDevice physicalDevice,
                                                              VkPhysicalDeviceMemoryProperties2* pMemoryProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceMemoryProperties2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceMemoryProperties2(physicalDevice, pMemoryProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceMemoryProperties2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceMemoryProperties2(physicalDevice, pMemoryProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceMemoryProperties2(physicalDevice, pMemoryProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceMemoryProperties2(physicalDevice, pMemoryProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceSparseImageFormatProperties2(VkPhysicalDevice physicalDevice,
                                                                         const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo,
                                                                         uint32_t* pPropertyCount,
                                                                         VkSparseImageFormatProperties2* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSparseImageFormatProperties2,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSparseImageFormatProperties2(physicalDevice, pFormatInfo,
                                                                                            pPropertyCount, pProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSparseImageFormatProperties2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSparseImageFormatProperties2(physicalDevice, pFormatInfo, pPropertyCount,
                                                                                  pProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceSparseImageFormatProperties2(physicalDevice, pFormatInfo, pPropertyCount, pProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSparseImageFormatProperties2(physicalDevice, pFormatInfo, pPropertyCount,
                                                                                   pProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL TrimCommandPool(VkDevice device, VkCommandPool commandPool, VkCommandPoolTrimFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkTrimCommandPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateTrimCommandPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateTrimCommandPool(device, commandPool, flags, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkTrimCommandPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordTrimCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordTrimCommandPool(device, commandPool, flags, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchTrimCommandPool(device, commandPool, flags);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordTrimCommandPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordTrimCommandPool(device, commandPool, flags, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceQueue2(VkDevice device, const VkDeviceQueueInfo2* pQueueInfo, VkQueue* pQueue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceQueue2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceQueue2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceQueue2(device, pQueueInfo, pQueue, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceQueue2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceQueue2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceQueue2(device, pQueueInfo, pQueue, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceQueue2(device, pQueueInfo, pQueue);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceQueue2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceQueue2(device, pQueueInfo, pQueue, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSamplerYcbcrConversion(VkDevice device, const VkSamplerYcbcrConversionCreateInfo* pCreateInfo,
                                                            const VkAllocationCallbacks* pAllocator,
                                                            VkSamplerYcbcrConversion* pYcbcrConversion) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateSamplerYcbcrConversion, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateSamplerYcbcrConversion]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateSamplerYcbcrConversion(device, pCreateInfo, pAllocator, pYcbcrConversion,
                                                                           error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateSamplerYcbcrConversion);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateSamplerYcbcrConversion]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateSamplerYcbcrConversion(device, pCreateInfo, pAllocator, pYcbcrConversion, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateSamplerYcbcrConversion(device, pCreateInfo, pAllocator, pYcbcrConversion);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateSamplerYcbcrConversion]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateSamplerYcbcrConversion(device, pCreateInfo, pAllocator, pYcbcrConversion, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySamplerYcbcrConversion(VkDevice device, VkSamplerYcbcrConversion ycbcrConversion,
                                                         const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroySamplerYcbcrConversion, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroySamplerYcbcrConversion]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroySamplerYcbcrConversion(device, ycbcrConversion, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroySamplerYcbcrConversion);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroySamplerYcbcrConversion]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroySamplerYcbcrConversion(device, ycbcrConversion, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroySamplerYcbcrConversion(device, ycbcrConversion, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroySamplerYcbcrConversion]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroySamplerYcbcrConversion(device, ycbcrConversion, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorUpdateTemplate(VkDevice device,
                                                              const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo,
                                                              const VkAllocationCallbacks* pAllocator,
                                                              VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDescriptorUpdateTemplate, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateDescriptorUpdateTemplate]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDescriptorUpdateTemplate(device, pCreateInfo, pAllocator,
                                                                             pDescriptorUpdateTemplate, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDescriptorUpdateTemplate);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateDescriptorUpdateTemplate]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDescriptorUpdateTemplate(device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate,
                                                                   record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDescriptorUpdateTemplate(device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateDescriptorUpdateTemplate]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDescriptorUpdateTemplate(device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate,
                                                                    record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorUpdateTemplate(VkDevice device, VkDescriptorUpdateTemplate descriptorUpdateTemplate,
                                                           const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDescriptorUpdateTemplate, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyDescriptorUpdateTemplate]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateDestroyDescriptorUpdateTemplate(device, descriptorUpdateTemplate, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDescriptorUpdateTemplate);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyDescriptorUpdateTemplate]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDescriptorUpdateTemplate(device, descriptorUpdateTemplate, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDescriptorUpdateTemplate(device, descriptorUpdateTemplate, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyDescriptorUpdateTemplate]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDescriptorUpdateTemplate(device, descriptorUpdateTemplate, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL UpdateDescriptorSetWithTemplate(VkDevice device, VkDescriptorSet descriptorSet,
                                                           VkDescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUpdateDescriptorSetWithTemplate, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateUpdateDescriptorSetWithTemplate]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateUpdateDescriptorSetWithTemplate(device, descriptorSet, descriptorUpdateTemplate,
                                                                              pData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkUpdateDescriptorSetWithTemplate);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordUpdateDescriptorSetWithTemplate]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUpdateDescriptorSetWithTemplate(device, descriptorSet, descriptorUpdateTemplate, pData,
                                                                    record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchUpdateDescriptorSetWithTemplate(device, descriptorSet, descriptorUpdateTemplate, pData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordUpdateDescriptorSetWithTemplate]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUpdateDescriptorSetWithTemplate(device, descriptorSet, descriptorUpdateTemplate, pData,
                                                                     record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalBufferProperties(VkPhysicalDevice physicalDevice,
                                                                     const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo,
                                                                     VkExternalBufferProperties* pExternalBufferProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalBufferProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalBufferProperties(physicalDevice, pExternalBufferInfo,
                                                                                        pExternalBufferProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalBufferProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalBufferProperties(physicalDevice, pExternalBufferInfo,
                                                                              pExternalBufferProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceExternalBufferProperties(physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalBufferProperties(physicalDevice, pExternalBufferInfo,
                                                                               pExternalBufferProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalFenceProperties(VkPhysicalDevice physicalDevice,
                                                                    const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo,
                                                                    VkExternalFenceProperties* pExternalFenceProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalFenceProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalFenceProperties(physicalDevice, pExternalFenceInfo,
                                                                                       pExternalFenceProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalFenceProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalFenceProperties(physicalDevice, pExternalFenceInfo,
                                                                             pExternalFenceProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceExternalFenceProperties(physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalFenceProperties(physicalDevice, pExternalFenceInfo,
                                                                              pExternalFenceProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalSemaphoreProperties(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo,
    VkExternalSemaphoreProperties* pExternalSemaphoreProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalSemaphoreProperties,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalSemaphoreProperties(physicalDevice, pExternalSemaphoreInfo,
                                                                                           pExternalSemaphoreProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalSemaphoreProperties);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalSemaphoreProperties(physicalDevice, pExternalSemaphoreInfo,
                                                                                 pExternalSemaphoreProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceExternalSemaphoreProperties(physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalSemaphoreProperties(physicalDevice, pExternalSemaphoreInfo,
                                                                                  pExternalSemaphoreProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDescriptorSetLayoutSupport(VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo,
                                                         VkDescriptorSetLayoutSupport* pSupport) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDescriptorSetLayoutSupport, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDescriptorSetLayoutSupport]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDescriptorSetLayoutSupport(device, pCreateInfo, pSupport, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDescriptorSetLayoutSupport);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetDescriptorSetLayoutSupport]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDescriptorSetLayoutSupport(device, pCreateInfo, pSupport, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDescriptorSetLayoutSupport(device, pCreateInfo, pSupport);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetDescriptorSetLayoutSupport]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDescriptorSetLayoutSupport(device, pCreateInfo, pSupport, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectCount(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount,
                                                uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndirectCount, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndirectCount]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                   maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndirectCount);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndirectCount]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                         maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndirectCount]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                          maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirectCount(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                       VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount,
                                                       uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndexedIndirectCount,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndexedIndirectCount]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndexedIndirectCount(commandBuffer, buffer, offset, countBuffer,
                                                                          countBufferOffset, maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndexedIndirectCount);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndexedIndirectCount]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndexedIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndexedIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndexedIndirectCount]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndexedIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                 maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRenderPass2(VkDevice device, const VkRenderPassCreateInfo2* pCreateInfo,
                                                 const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateRenderPass2, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateRenderPass2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateRenderPass2(device, pCreateInfo, pAllocator, pRenderPass, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateRenderPass2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateRenderPass2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateRenderPass2(device, pCreateInfo, pAllocator, pRenderPass, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateRenderPass2(device, pCreateInfo, pAllocator, pRenderPass);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateRenderPass2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateRenderPass2(device, pCreateInfo, pAllocator, pRenderPass, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderPass2(VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin,
                                               const VkSubpassBeginInfo* pSubpassBeginInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginRenderPass2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginRenderPass2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginRenderPass2(commandBuffer, pRenderPassBegin, pSubpassBeginInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginRenderPass2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginRenderPass2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginRenderPass2(commandBuffer, pRenderPassBegin, pSubpassBeginInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginRenderPass2(commandBuffer, pRenderPassBegin, pSubpassBeginInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginRenderPass2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginRenderPass2(commandBuffer, pRenderPassBegin, pSubpassBeginInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdNextSubpass2(VkCommandBuffer commandBuffer, const VkSubpassBeginInfo* pSubpassBeginInfo,
                                           const VkSubpassEndInfo* pSubpassEndInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdNextSubpass2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdNextSubpass2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdNextSubpass2(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdNextSubpass2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdNextSubpass2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdNextSubpass2(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdNextSubpass2(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdNextSubpass2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdNextSubpass2(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderPass2(VkCommandBuffer commandBuffer, const VkSubpassEndInfo* pSubpassEndInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndRenderPass2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndRenderPass2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndRenderPass2(commandBuffer, pSubpassEndInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndRenderPass2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndRenderPass2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndRenderPass2(commandBuffer, pSubpassEndInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndRenderPass2(commandBuffer, pSubpassEndInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndRenderPass2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndRenderPass2(commandBuffer, pSubpassEndInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL ResetQueryPool(VkDevice device, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetQueryPool, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetQueryPool]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetQueryPool(device, queryPool, firstQuery, queryCount, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetQueryPool);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetQueryPool(device, queryPool, firstQuery, queryCount, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchResetQueryPool(device, queryPool, firstQuery, queryCount);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetQueryPool]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetQueryPool(device, queryPool, firstQuery, queryCount, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreCounterValue(VkDevice device, VkSemaphore semaphore, uint64_t* pValue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSemaphoreCounterValue, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetSemaphoreCounterValue]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSemaphoreCounterValue(device, semaphore, pValue, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSemaphoreCounterValue);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSemaphoreCounterValue]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSemaphoreCounterValue(device, semaphore, pValue, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSemaphoreCounterValue(device, semaphore, pValue);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSemaphoreCounterValue]) {
            ValidationObject::BlockingOperationGuard lock(intercept);

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetSemaphoreCounterValue(device, semaphore, pValue, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WaitSemaphores(VkDevice device, const VkSemaphoreWaitInfo* pWaitInfo, uint64_t timeout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkWaitSemaphores, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateWaitSemaphores]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateWaitSemaphores(device, pWaitInfo, timeout, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkWaitSemaphores);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordWaitSemaphores]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordWaitSemaphores(device, pWaitInfo, timeout, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchWaitSemaphores(device, pWaitInfo, timeout);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordWaitSemaphores]) {
            ValidationObject::BlockingOperationGuard lock(intercept);

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordWaitSemaphores(device, pWaitInfo, timeout, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SignalSemaphore(VkDevice device, const VkSemaphoreSignalInfo* pSignalInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSignalSemaphore, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateSignalSemaphore]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSignalSemaphore(device, pSignalInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkSignalSemaphore);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSignalSemaphore]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSignalSemaphore(device, pSignalInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchSignalSemaphore(device, pSignalInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSignalSemaphore]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSignalSemaphore(device, pSignalInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetBufferDeviceAddress(VkDevice device, const VkBufferDeviceAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferDeviceAddress, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferDeviceAddress]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferDeviceAddress(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferDeviceAddress);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferDeviceAddress]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferDeviceAddress(device, pInfo, record_obj);
        }
    }
    VkDeviceAddress result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetBufferDeviceAddress(device, pInfo);
    }
    record_obj.device_address = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferDeviceAddress]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferDeviceAddress(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetBufferOpaqueCaptureAddress(VkDevice device, const VkBufferDeviceAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferOpaqueCaptureAddress, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferOpaqueCaptureAddress]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferOpaqueCaptureAddress(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferOpaqueCaptureAddress);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferOpaqueCaptureAddress]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferOpaqueCaptureAddress(device, pInfo, record_obj);
        }
    }
    uint64_t result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetBufferOpaqueCaptureAddress(device, pInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferOpaqueCaptureAddress]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferOpaqueCaptureAddress(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetDeviceMemoryOpaqueCaptureAddress(VkDevice device,
                                                                   const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceMemoryOpaqueCaptureAddress, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceMemoryOpaqueCaptureAddress]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceMemoryOpaqueCaptureAddress(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceMemoryOpaqueCaptureAddress);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceMemoryOpaqueCaptureAddress]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceMemoryOpaqueCaptureAddress(device, pInfo, record_obj);
        }
    }
    uint64_t result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeviceMemoryOpaqueCaptureAddress(device, pInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceMemoryOpaqueCaptureAddress]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceMemoryOpaqueCaptureAddress(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePrivateDataSlot(VkDevice device, const VkPrivateDataSlotCreateInfo* pCreateInfo,
                                                     const VkAllocationCallbacks* pAllocator, VkPrivateDataSlot* pPrivateDataSlot) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreatePrivateDataSlot, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreatePrivateDataSlot]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreatePrivateDataSlot(device, pCreateInfo, pAllocator, pPrivateDataSlot, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreatePrivateDataSlot);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreatePrivateDataSlot]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreatePrivateDataSlot(device, pCreateInfo, pAllocator, pPrivateDataSlot, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreatePrivateDataSlot(device, pCreateInfo, pAllocator, pPrivateDataSlot);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreatePrivateDataSlot]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreatePrivateDataSlot(device, pCreateInfo, pAllocator, pPrivateDataSlot, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPrivateDataSlot(VkDevice device, VkPrivateDataSlot privateDataSlot,
                                                  const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyPrivateDataSlot, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyPrivateDataSlot]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyPrivateDataSlot(device, privateDataSlot, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyPrivateDataSlot);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyPrivateDataSlot]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyPrivateDataSlot(device, privateDataSlot, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyPrivateDataSlot(device, privateDataSlot, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyPrivateDataSlot]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyPrivateDataSlot(device, privateDataSlot, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL SetPrivateData(VkDevice device, VkObjectType objectType, uint64_t objectHandle,
                                              VkPrivateDataSlot privateDataSlot, uint64_t data) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSetPrivateData, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateSetPrivateData]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSetPrivateData(device, objectType, objectHandle, privateDataSlot, data, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkSetPrivateData);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSetPrivateData]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSetPrivateData(device, objectType, objectHandle, privateDataSlot, data, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchSetPrivateData(device, objectType, objectHandle, privateDataSlot, data);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSetPrivateData]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSetPrivateData(device, objectType, objectHandle, privateDataSlot, data, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPrivateData(VkDevice device, VkObjectType objectType, uint64_t objectHandle,
                                          VkPrivateDataSlot privateDataSlot, uint64_t* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPrivateData, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetPrivateData]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPrivateData(device, objectType, objectHandle, privateDataSlot, pData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPrivateData);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetPrivateData]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPrivateData(device, objectType, objectHandle, privateDataSlot, pData, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPrivateData(device, objectType, objectHandle, privateDataSlot, pData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetPrivateData]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPrivateData(device, objectType, objectHandle, privateDataSlot, pData, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetEvent2(VkCommandBuffer commandBuffer, VkEvent event, const VkDependencyInfo* pDependencyInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetEvent2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetEvent2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetEvent2(commandBuffer, event, pDependencyInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetEvent2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetEvent2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetEvent2(commandBuffer, event, pDependencyInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetEvent2(commandBuffer, event, pDependencyInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetEvent2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetEvent2(commandBuffer, event, pDependencyInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResetEvent2(VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags2 stageMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResetEvent2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResetEvent2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResetEvent2(commandBuffer, event, stageMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResetEvent2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResetEvent2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResetEvent2(commandBuffer, event, stageMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResetEvent2(commandBuffer, event, stageMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResetEvent2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResetEvent2(commandBuffer, event, stageMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdWaitEvents2(VkCommandBuffer commandBuffer, uint32_t eventCount, const VkEvent* pEvents,
                                          const VkDependencyInfo* pDependencyInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWaitEvents2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWaitEvents2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWaitEvents2(commandBuffer, eventCount, pEvents, pDependencyInfos, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWaitEvents2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWaitEvents2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWaitEvents2(commandBuffer, eventCount, pEvents, pDependencyInfos, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWaitEvents2(commandBuffer, eventCount, pEvents, pDependencyInfos);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWaitEvents2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWaitEvents2(commandBuffer, eventCount, pEvents, pDependencyInfos, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPipelineBarrier2(VkCommandBuffer commandBuffer, const VkDependencyInfo* pDependencyInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPipelineBarrier2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPipelineBarrier2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPipelineBarrier2(commandBuffer, pDependencyInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPipelineBarrier2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPipelineBarrier2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPipelineBarrier2(commandBuffer, pDependencyInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPipelineBarrier2(commandBuffer, pDependencyInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPipelineBarrier2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPipelineBarrier2(commandBuffer, pDependencyInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdWriteTimestamp2(VkCommandBuffer commandBuffer, VkPipelineStageFlags2 stage, VkQueryPool queryPool,
                                              uint32_t query) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWriteTimestamp2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWriteTimestamp2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWriteTimestamp2(commandBuffer, stage, queryPool, query, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWriteTimestamp2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWriteTimestamp2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWriteTimestamp2(commandBuffer, stage, queryPool, query, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWriteTimestamp2(commandBuffer, stage, queryPool, query);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWriteTimestamp2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWriteTimestamp2(commandBuffer, stage, queryPool, query, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSubmit2(VkQueue queue, uint32_t submitCount, const VkSubmitInfo2* pSubmits, VkFence fence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueSubmit2, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateQueueSubmit2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueSubmit2(queue, submitCount, pSubmits, fence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueSubmit2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueSubmit2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueSubmit2(queue, submitCount, pSubmits, fence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueueSubmit2(queue, submitCount, pSubmits, fence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueSubmit2]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordQueueSubmit2(queue, submitCount, pSubmits, fence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBuffer2(VkCommandBuffer commandBuffer, const VkCopyBufferInfo2* pCopyBufferInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyBuffer2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyBuffer2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyBuffer2(commandBuffer, pCopyBufferInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyBuffer2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyBuffer2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyBuffer2(commandBuffer, pCopyBufferInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyBuffer2(commandBuffer, pCopyBufferInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyBuffer2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyBuffer2(commandBuffer, pCopyBufferInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImage2(VkCommandBuffer commandBuffer, const VkCopyImageInfo2* pCopyImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyImage2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyImage2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyImage2(commandBuffer, pCopyImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyImage2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyImage2(commandBuffer, pCopyImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyImage2(commandBuffer, pCopyImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyImage2(commandBuffer, pCopyImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBufferToImage2(VkCommandBuffer commandBuffer,
                                                 const VkCopyBufferToImageInfo2* pCopyBufferToImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyBufferToImage2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyBufferToImage2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyBufferToImage2(commandBuffer, pCopyBufferToImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyBufferToImage2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyBufferToImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyBufferToImage2(commandBuffer, pCopyBufferToImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyBufferToImage2(commandBuffer, pCopyBufferToImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyBufferToImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyBufferToImage2(commandBuffer, pCopyBufferToImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImageToBuffer2(VkCommandBuffer commandBuffer,
                                                 const VkCopyImageToBufferInfo2* pCopyImageToBufferInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyImageToBuffer2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyImageToBuffer2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyImageToBuffer2(commandBuffer, pCopyImageToBufferInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyImageToBuffer2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyImageToBuffer2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyImageToBuffer2(commandBuffer, pCopyImageToBufferInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyImageToBuffer2(commandBuffer, pCopyImageToBufferInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyImageToBuffer2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyImageToBuffer2(commandBuffer, pCopyImageToBufferInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBlitImage2(VkCommandBuffer commandBuffer, const VkBlitImageInfo2* pBlitImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBlitImage2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBlitImage2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBlitImage2(commandBuffer, pBlitImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBlitImage2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBlitImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBlitImage2(commandBuffer, pBlitImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBlitImage2(commandBuffer, pBlitImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBlitImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBlitImage2(commandBuffer, pBlitImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResolveImage2(VkCommandBuffer commandBuffer, const VkResolveImageInfo2* pResolveImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResolveImage2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResolveImage2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResolveImage2(commandBuffer, pResolveImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResolveImage2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResolveImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResolveImage2(commandBuffer, pResolveImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResolveImage2(commandBuffer, pResolveImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResolveImage2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResolveImage2(commandBuffer, pResolveImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRendering(VkCommandBuffer commandBuffer, const VkRenderingInfo* pRenderingInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginRendering, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginRendering]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginRendering(commandBuffer, pRenderingInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginRendering);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginRendering]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginRendering(commandBuffer, pRenderingInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginRendering(commandBuffer, pRenderingInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginRendering]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginRendering(commandBuffer, pRenderingInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndRendering(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndRendering, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndRendering]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndRendering(commandBuffer, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndRendering);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndRendering]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndRendering(commandBuffer, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndRendering(commandBuffer);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndRendering]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndRendering(commandBuffer, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetCullMode(VkCommandBuffer commandBuffer, VkCullModeFlags cullMode) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetCullMode, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetCullMode]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetCullMode(commandBuffer, cullMode, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetCullMode);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetCullMode]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetCullMode(commandBuffer, cullMode, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetCullMode(commandBuffer, cullMode);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetCullMode]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetCullMode(commandBuffer, cullMode, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetFrontFace(VkCommandBuffer commandBuffer, VkFrontFace frontFace) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetFrontFace, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetFrontFace]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetFrontFace(commandBuffer, frontFace, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetFrontFace);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetFrontFace]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetFrontFace(commandBuffer, frontFace, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetFrontFace(commandBuffer, frontFace);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetFrontFace]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetFrontFace(commandBuffer, frontFace, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveTopology(VkCommandBuffer commandBuffer, VkPrimitiveTopology primitiveTopology) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetPrimitiveTopology, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetPrimitiveTopology]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetPrimitiveTopology(commandBuffer, primitiveTopology, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetPrimitiveTopology);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetPrimitiveTopology]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetPrimitiveTopology(commandBuffer, primitiveTopology, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetPrimitiveTopology(commandBuffer, primitiveTopology);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetPrimitiveTopology]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetPrimitiveTopology(commandBuffer, primitiveTopology, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportWithCount(VkCommandBuffer commandBuffer, uint32_t viewportCount,
                                                   const VkViewport* pViewports) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetViewportWithCount, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetViewportWithCount]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetViewportWithCount(commandBuffer, viewportCount, pViewports, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetViewportWithCount);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetViewportWithCount]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetViewportWithCount(commandBuffer, viewportCount, pViewports, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetViewportWithCount(commandBuffer, viewportCount, pViewports);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetViewportWithCount]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetViewportWithCount(commandBuffer, viewportCount, pViewports, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetScissorWithCount(VkCommandBuffer commandBuffer, uint32_t scissorCount, const VkRect2D* pScissors) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetScissorWithCount, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetScissorWithCount]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetScissorWithCount(commandBuffer, scissorCount, pScissors, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetScissorWithCount);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetScissorWithCount]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetScissorWithCount(commandBuffer, scissorCount, pScissors, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetScissorWithCount(commandBuffer, scissorCount, pScissors);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetScissorWithCount]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetScissorWithCount(commandBuffer, scissorCount, pScissors, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindVertexBuffers2(VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount,
                                                 const VkBuffer* pBuffers, const VkDeviceSize* pOffsets, const VkDeviceSize* pSizes,
                                                 const VkDeviceSize* pStrides) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindVertexBuffers2, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindVertexBuffers2]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindVertexBuffers2(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets,
                                                                    pSizes, pStrides, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindVertexBuffers2);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindVertexBuffers2]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindVertexBuffers2(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes,
                                                          pStrides, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindVertexBuffers2(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindVertexBuffers2]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindVertexBuffers2(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes,
                                                           pStrides, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthTestEnable(VkCommandBuffer commandBuffer, VkBool32 depthTestEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthTestEnable, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthTestEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthTestEnable(commandBuffer, depthTestEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthTestEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthTestEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthTestEnable(commandBuffer, depthTestEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthTestEnable(commandBuffer, depthTestEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthTestEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthTestEnable(commandBuffer, depthTestEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthWriteEnable(VkCommandBuffer commandBuffer, VkBool32 depthWriteEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthWriteEnable, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthWriteEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthWriteEnable(commandBuffer, depthWriteEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthWriteEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthWriteEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthWriteEnable(commandBuffer, depthWriteEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthWriteEnable(commandBuffer, depthWriteEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthWriteEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthWriteEnable(commandBuffer, depthWriteEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthCompareOp(VkCommandBuffer commandBuffer, VkCompareOp depthCompareOp) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthCompareOp, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthCompareOp]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthCompareOp(commandBuffer, depthCompareOp, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthCompareOp);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthCompareOp]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthCompareOp(commandBuffer, depthCompareOp, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthCompareOp(commandBuffer, depthCompareOp);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthCompareOp]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthCompareOp(commandBuffer, depthCompareOp, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBoundsTestEnable(VkCommandBuffer commandBuffer, VkBool32 depthBoundsTestEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthBoundsTestEnable,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthBoundsTestEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthBoundsTestEnable(commandBuffer, depthBoundsTestEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthBoundsTestEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthBoundsTestEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthBoundsTestEnable(commandBuffer, depthBoundsTestEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthBoundsTestEnable(commandBuffer, depthBoundsTestEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthBoundsTestEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthBoundsTestEnable(commandBuffer, depthBoundsTestEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilTestEnable(VkCommandBuffer commandBuffer, VkBool32 stencilTestEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilTestEnable, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilTestEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilTestEnable(commandBuffer, stencilTestEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilTestEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilTestEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilTestEnable(commandBuffer, stencilTestEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilTestEnable(commandBuffer, stencilTestEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilTestEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilTestEnable(commandBuffer, stencilTestEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilOp(VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, VkStencilOp failOp,
                                           VkStencilOp passOp, VkStencilOp depthFailOp, VkCompareOp compareOp) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilOp, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilOp]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilOp(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp,
                                                              error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilOp);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilOp]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilOp(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilOp(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilOp]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilOp(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetRasterizerDiscardEnable(VkCommandBuffer commandBuffer, VkBool32 rasterizerDiscardEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetRasterizerDiscardEnable,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetRasterizerDiscardEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetRasterizerDiscardEnable(commandBuffer, rasterizerDiscardEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetRasterizerDiscardEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetRasterizerDiscardEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetRasterizerDiscardEnable(commandBuffer, rasterizerDiscardEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetRasterizerDiscardEnable(commandBuffer, rasterizerDiscardEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetRasterizerDiscardEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetRasterizerDiscardEnable(commandBuffer, rasterizerDiscardEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBiasEnable(VkCommandBuffer commandBuffer, VkBool32 depthBiasEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthBiasEnable, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthBiasEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthBiasEnable(commandBuffer, depthBiasEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthBiasEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthBiasEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthBiasEnable(commandBuffer, depthBiasEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthBiasEnable(commandBuffer, depthBiasEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthBiasEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthBiasEnable(commandBuffer, depthBiasEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveRestartEnable(VkCommandBuffer commandBuffer, VkBool32 primitiveRestartEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetPrimitiveRestartEnable,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetPrimitiveRestartEnable]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetPrimitiveRestartEnable(commandBuffer, primitiveRestartEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetPrimitiveRestartEnable);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetPrimitiveRestartEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetPrimitiveRestartEnable(commandBuffer, primitiveRestartEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetPrimitiveRestartEnable(commandBuffer, primitiveRestartEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetPrimitiveRestartEnable]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetPrimitiveRestartEnable(commandBuffer, primitiveRestartEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceBufferMemoryRequirements(VkDevice device, const VkDeviceBufferMemoryRequirements* pInfo,
                                                             VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceBufferMemoryRequirements, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceBufferMemoryRequirements]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceBufferMemoryRequirements(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceBufferMemoryRequirements);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceBufferMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceBufferMemoryRequirements(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceBufferMemoryRequirements(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceBufferMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceBufferMemoryRequirements(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageMemoryRequirements(VkDevice device, const VkDeviceImageMemoryRequirements* pInfo,
                                                            VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceImageMemoryRequirements, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceImageMemoryRequirements]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceImageMemoryRequirements(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceImageMemoryRequirements);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceImageMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceImageMemoryRequirements(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceImageMemoryRequirements(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceImageMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceImageMemoryRequirements(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageSparseMemoryRequirements(VkDevice device, const VkDeviceImageMemoryRequirements* pInfo,
                                                                  uint32_t* pSparseMemoryRequirementCount,
                                                                  VkSparseImageMemoryRequirements2* pSparseMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceImageSparseMemoryRequirements, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceImageSparseMemoryRequirements]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceImageSparseMemoryRequirements(device, pInfo, pSparseMemoryRequirementCount,
                                                                                     pSparseMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceImageSparseMemoryRequirements);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceImageSparseMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceImageSparseMemoryRequirements(device, pInfo, pSparseMemoryRequirementCount,
                                                                           pSparseMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceImageSparseMemoryRequirements(device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceImageSparseMemoryRequirements]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceImageSparseMemoryRequirements(device, pInfo, pSparseMemoryRequirementCount,
                                                                            pSparseMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL DestroySurfaceKHR(VkInstance instance, VkSurfaceKHR surface, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroySurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroySurfaceKHR(instance, surface, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroySurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroySurfaceKHR(instance, surface, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroySurfaceKHR(instance, surface, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroySurfaceKHR(instance, surface, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceSupportKHR(VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex,
                                                                  VkSurfaceKHR surface, VkBool32* pSupported) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfaceSupportKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfaceSupportKHR(physicalDevice, queueFamilyIndex, surface,
                                                                                 pSupported, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfaceSupportKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfaceSupportKHR(physicalDevice, queueFamilyIndex, surface, pSupported,
                                                                       record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfaceSupportKHR(physicalDevice, queueFamilyIndex, surface, pSupported);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfaceSupportKHR(physicalDevice, queueFamilyIndex, surface, pSupported,
                                                                        record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceCapabilitiesKHR(VkPhysicalDevice physicalDevice, VkSurfaceKHR surface,
                                                                       VkSurfaceCapabilitiesKHR* pSurfaceCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfaceCapabilitiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfaceCapabilitiesKHR(physicalDevice, surface, pSurfaceCapabilities,
                                                                                      error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfaceCapabilitiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfaceCapabilitiesKHR(physicalDevice, surface, pSurfaceCapabilities,
                                                                            record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfaceCapabilitiesKHR(physicalDevice, surface, pSurfaceCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfaceCapabilitiesKHR(physicalDevice, surface, pSurfaceCapabilities,
                                                                             record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceFormatsKHR(VkPhysicalDevice physicalDevice, VkSurfaceKHR surface,
                                                                  uint32_t* pSurfaceFormatCount,
                                                                  VkSurfaceFormatKHR* pSurfaceFormats) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfaceFormatsKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfaceFormatsKHR(physicalDevice, surface, pSurfaceFormatCount,
                                                                                 pSurfaceFormats, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfaceFormatsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfaceFormatsKHR(physicalDevice, surface, pSurfaceFormatCount,
                                                                       pSurfaceFormats, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfaceFormatsKHR(physicalDevice, surface, pSurfaceFormatCount, pSurfaceFormats);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfaceFormatsKHR(physicalDevice, surface, pSurfaceFormatCount,
                                                                        pSurfaceFormats, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfacePresentModesKHR(VkPhysicalDevice physicalDevice, VkSurfaceKHR surface,
                                                                       uint32_t* pPresentModeCount,
                                                                       VkPresentModeKHR* pPresentModes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfacePresentModesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfacePresentModesKHR(physicalDevice, surface, pPresentModeCount,
                                                                                      pPresentModes, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfacePresentModesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfacePresentModesKHR(physicalDevice, surface, pPresentModeCount,
                                                                            pPresentModes, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfacePresentModesKHR(physicalDevice, surface, pPresentModeCount, pPresentModes);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfacePresentModesKHR(physicalDevice, surface, pPresentModeCount,
                                                                             pPresentModes, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSwapchainKHR(VkDevice device, const VkSwapchainCreateInfoKHR* pCreateInfo,
                                                  const VkAllocationCallbacks* pAllocator, VkSwapchainKHR* pSwapchain) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateSwapchainKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateSwapchainKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateSwapchainKHR(device, pCreateInfo, pAllocator, pSwapchain, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateSwapchainKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateSwapchainKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateSwapchainKHR(device, pCreateInfo, pAllocator, pSwapchain, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateSwapchainKHR(device, pCreateInfo, pAllocator, pSwapchain);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateSwapchainKHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordCreateSwapchainKHR(device, pCreateInfo, pAllocator, pSwapchain, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySwapchainKHR(VkDevice device, VkSwapchainKHR swapchain, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroySwapchainKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroySwapchainKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroySwapchainKHR(device, swapchain, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroySwapchainKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroySwapchainKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroySwapchainKHR(device, swapchain, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroySwapchainKHR(device, swapchain, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroySwapchainKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroySwapchainKHR(device, swapchain, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetSwapchainImagesKHR(VkDevice device, VkSwapchainKHR swapchain, uint32_t* pSwapchainImageCount,
                                                     VkImage* pSwapchainImages) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSwapchainImagesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetSwapchainImagesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSwapchainImagesKHR(device, swapchain, pSwapchainImageCount, pSwapchainImages,
                                                                    error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSwapchainImagesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSwapchainImagesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSwapchainImagesKHR(device, swapchain, pSwapchainImageCount, pSwapchainImages, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSwapchainImagesKHR(device, swapchain, pSwapchainImageCount, pSwapchainImages);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSwapchainImagesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetSwapchainImagesKHR(device, swapchain, pSwapchainImageCount, pSwapchainImages, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireNextImageKHR(VkDevice device, VkSwapchainKHR swapchain, uint64_t timeout,
                                                   VkSemaphore semaphore, VkFence fence, uint32_t* pImageIndex) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAcquireNextImageKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateAcquireNextImageKHR]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateAcquireNextImageKHR(device, swapchain, timeout, semaphore, fence, pImageIndex, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAcquireNextImageKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordAcquireNextImageKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAcquireNextImageKHR(device, swapchain, timeout, semaphore, fence, pImageIndex, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAcquireNextImageKHR(device, swapchain, timeout, semaphore, fence, pImageIndex);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordAcquireNextImageKHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordAcquireNextImageKHR(device, swapchain, timeout, semaphore, fence, pImageIndex, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL QueuePresentKHR(VkQueue queue, const VkPresentInfoKHR* pPresentInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueuePresentKHR, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateQueuePresentKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueuePresentKHR(queue, pPresentInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueuePresentKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueuePresentKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueuePresentKHR(queue, pPresentInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueuePresentKHR(queue, pPresentInfo);
    }
    VVL_TracyCFrameMark;
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueuePresentKHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordQueuePresentKHR(queue, pPresentInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL
GetDeviceGroupPresentCapabilitiesKHR(VkDevice device, VkDeviceGroupPresentCapabilitiesKHR* pDeviceGroupPresentCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceGroupPresentCapabilitiesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceGroupPresentCapabilitiesKHR]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateGetDeviceGroupPresentCapabilitiesKHR(device, pDeviceGroupPresentCapabilities, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceGroupPresentCapabilitiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceGroupPresentCapabilitiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceGroupPresentCapabilitiesKHR(device, pDeviceGroupPresentCapabilities, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeviceGroupPresentCapabilitiesKHR(device, pDeviceGroupPresentCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceGroupPresentCapabilitiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceGroupPresentCapabilitiesKHR(device, pDeviceGroupPresentCapabilities, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeviceGroupSurfacePresentModesKHR(VkDevice device, VkSurfaceKHR surface,
                                                                    VkDeviceGroupPresentModeFlagsKHR* pModes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceGroupSurfacePresentModesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceGroupSurfacePresentModesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceGroupSurfacePresentModesKHR(device, surface, pModes, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceGroupSurfacePresentModesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceGroupSurfacePresentModesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceGroupSurfacePresentModesKHR(device, surface, pModes, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeviceGroupSurfacePresentModesKHR(device, surface, pModes);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceGroupSurfacePresentModesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceGroupSurfacePresentModesKHR(device, surface, pModes, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDevicePresentRectanglesKHR(VkPhysicalDevice physicalDevice, VkSurfaceKHR surface,
                                                                     uint32_t* pRectCount, VkRect2D* pRects) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDevicePresentRectanglesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDevicePresentRectanglesKHR(physicalDevice, surface, pRectCount, pRects,
                                                                                    error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDevicePresentRectanglesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDevicePresentRectanglesKHR(physicalDevice, surface, pRectCount, pRects, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDevicePresentRectanglesKHR(physicalDevice, surface, pRectCount, pRects);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDevicePresentRectanglesKHR(physicalDevice, surface, pRectCount, pRects, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireNextImage2KHR(VkDevice device, const VkAcquireNextImageInfoKHR* pAcquireInfo,
                                                    uint32_t* pImageIndex) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAcquireNextImage2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateAcquireNextImage2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAcquireNextImage2KHR(device, pAcquireInfo, pImageIndex, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAcquireNextImage2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordAcquireNextImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAcquireNextImage2KHR(device, pAcquireInfo, pImageIndex, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAcquireNextImage2KHR(device, pAcquireInfo, pImageIndex);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordAcquireNextImage2KHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordAcquireNextImage2KHR(device, pAcquireInfo, pImageIndex, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayPropertiesKHR(VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount,
                                                                     VkDisplayPropertiesKHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceDisplayPropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceDisplayPropertiesKHR(physicalDevice, pPropertyCount, pProperties,
                                                                                    error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceDisplayPropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceDisplayPropertiesKHR(physicalDevice, pPropertyCount, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceDisplayPropertiesKHR(physicalDevice, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceDisplayPropertiesKHR(physicalDevice, pPropertyCount, pProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayPlanePropertiesKHR(VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount,
                                                                          VkDisplayPlanePropertiesKHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceDisplayPlanePropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceDisplayPlanePropertiesKHR(physicalDevice, pPropertyCount,
                                                                                         pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceDisplayPlanePropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceDisplayPlanePropertiesKHR(physicalDevice, pPropertyCount, pProperties,
                                                                               record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceDisplayPlanePropertiesKHR(physicalDevice, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceDisplayPlanePropertiesKHR(physicalDevice, pPropertyCount, pProperties,
                                                                                record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayPlaneSupportedDisplaysKHR(VkPhysicalDevice physicalDevice, uint32_t planeIndex,
                                                                   uint32_t* pDisplayCount, VkDisplayKHR* pDisplays) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDisplayPlaneSupportedDisplaysKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDisplayPlaneSupportedDisplaysKHR(physicalDevice, planeIndex, pDisplayCount,
                                                                                  pDisplays, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDisplayPlaneSupportedDisplaysKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDisplayPlaneSupportedDisplaysKHR(physicalDevice, planeIndex, pDisplayCount, pDisplays,
                                                                        record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDisplayPlaneSupportedDisplaysKHR(physicalDevice, planeIndex, pDisplayCount, pDisplays);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDisplayPlaneSupportedDisplaysKHR(physicalDevice, planeIndex, pDisplayCount, pDisplays,
                                                                         record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayModePropertiesKHR(VkPhysicalDevice physicalDevice, VkDisplayKHR display,
                                                           uint32_t* pPropertyCount, VkDisplayModePropertiesKHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDisplayModePropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDisplayModePropertiesKHR(physicalDevice, display, pPropertyCount, pProperties,
                                                                          error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDisplayModePropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDisplayModePropertiesKHR(physicalDevice, display, pPropertyCount, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDisplayModePropertiesKHR(physicalDevice, display, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDisplayModePropertiesKHR(physicalDevice, display, pPropertyCount, pProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDisplayModeKHR(VkPhysicalDevice physicalDevice, VkDisplayKHR display,
                                                    const VkDisplayModeCreateInfoKHR* pCreateInfo,
                                                    const VkAllocationCallbacks* pAllocator, VkDisplayModeKHR* pMode) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDisplayModeKHR, VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCreateDisplayModeKHR(physicalDevice, display, pCreateInfo, pAllocator, pMode, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDisplayModeKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDisplayModeKHR(physicalDevice, display, pCreateInfo, pAllocator, pMode, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDisplayModeKHR(physicalDevice, display, pCreateInfo, pAllocator, pMode);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDisplayModeKHR(physicalDevice, display, pCreateInfo, pAllocator, pMode, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayPlaneCapabilitiesKHR(VkPhysicalDevice physicalDevice, VkDisplayModeKHR mode,
                                                              uint32_t planeIndex, VkDisplayPlaneCapabilitiesKHR* pCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDisplayPlaneCapabilitiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDisplayPlaneCapabilitiesKHR(physicalDevice, mode, planeIndex, pCapabilities,
                                                                             error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDisplayPlaneCapabilitiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDisplayPlaneCapabilitiesKHR(physicalDevice, mode, planeIndex, pCapabilities, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDisplayPlaneCapabilitiesKHR(physicalDevice, mode, planeIndex, pCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDisplayPlaneCapabilitiesKHR(physicalDevice, mode, planeIndex, pCapabilities, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDisplayPlaneSurfaceKHR(VkInstance instance, const VkDisplaySurfaceCreateInfoKHR* pCreateInfo,
                                                            const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDisplayPlaneSurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDisplayPlaneSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDisplayPlaneSurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDisplayPlaneSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDisplayPlaneSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDisplayPlaneSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSharedSwapchainsKHR(VkDevice device, uint32_t swapchainCount,
                                                         const VkSwapchainCreateInfoKHR* pCreateInfos,
                                                         const VkAllocationCallbacks* pAllocator, VkSwapchainKHR* pSwapchains) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateSharedSwapchainsKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateSharedSwapchainsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateSharedSwapchainsKHR(device, swapchainCount, pCreateInfos, pAllocator,
                                                                        pSwapchains, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateSharedSwapchainsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateSharedSwapchainsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateSharedSwapchainsKHR(device, swapchainCount, pCreateInfos, pAllocator, pSwapchains,
                                                              record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateSharedSwapchainsKHR(device, swapchainCount, pCreateInfos, pAllocator, pSwapchains);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateSharedSwapchainsKHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordCreateSharedSwapchainsKHR(device, swapchainCount, pCreateInfos, pAllocator, pSwapchains,
                                                               record_obj);
        }
    }
    return result;
}

#ifdef VK_USE_PLATFORM_XLIB_KHR
VKAPI_ATTR VkResult VKAPI_CALL CreateXlibSurfaceKHR(VkInstance instance, const VkXlibSurfaceCreateInfoKHR* pCreateInfo,
                                                    const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateXlibSurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateXlibSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateXlibSurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateXlibSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateXlibSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateXlibSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceXlibPresentationSupportKHR(VkPhysicalDevice physicalDevice,
                                                                           uint32_t queueFamilyIndex, Display* dpy,
                                                                           VisualID visualID) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceXlibPresentationSupportKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceXlibPresentationSupportKHR(physicalDevice, queueFamilyIndex, dpy,
                                                                                          visualID, error_obj);
            if (skip) return VK_FALSE;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceXlibPresentationSupportKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceXlibPresentationSupportKHR(physicalDevice, queueFamilyIndex, dpy, visualID,
                                                                                record_obj);
        }
    }
    VkBool32 result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceXlibPresentationSupportKHR(physicalDevice, queueFamilyIndex, dpy, visualID);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceXlibPresentationSupportKHR(physicalDevice, queueFamilyIndex, dpy, visualID,
                                                                                 record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_XLIB_KHR
#ifdef VK_USE_PLATFORM_XCB_KHR
VKAPI_ATTR VkResult VKAPI_CALL CreateXcbSurfaceKHR(VkInstance instance, const VkXcbSurfaceCreateInfoKHR* pCreateInfo,
                                                   const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateXcbSurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateXcbSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateXcbSurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateXcbSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateXcbSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateXcbSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceXcbPresentationSupportKHR(VkPhysicalDevice physicalDevice,
                                                                          uint32_t queueFamilyIndex, xcb_connection_t* connection,
                                                                          xcb_visualid_t visual_id) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceXcbPresentationSupportKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceXcbPresentationSupportKHR(physicalDevice, queueFamilyIndex,
                                                                                         connection, visual_id, error_obj);
            if (skip) return VK_FALSE;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceXcbPresentationSupportKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceXcbPresentationSupportKHR(physicalDevice, queueFamilyIndex, connection,
                                                                               visual_id, record_obj);
        }
    }
    VkBool32 result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceXcbPresentationSupportKHR(physicalDevice, queueFamilyIndex, connection, visual_id);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceXcbPresentationSupportKHR(physicalDevice, queueFamilyIndex, connection,
                                                                                visual_id, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_XCB_KHR
#ifdef VK_USE_PLATFORM_WAYLAND_KHR
VKAPI_ATTR VkResult VKAPI_CALL CreateWaylandSurfaceKHR(VkInstance instance, const VkWaylandSurfaceCreateInfoKHR* pCreateInfo,
                                                       const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateWaylandSurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateWaylandSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateWaylandSurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateWaylandSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateWaylandSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateWaylandSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceWaylandPresentationSupportKHR(VkPhysicalDevice physicalDevice,
                                                                              uint32_t queueFamilyIndex,
                                                                              struct wl_display* display) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceWaylandPresentationSupportKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceWaylandPresentationSupportKHR(physicalDevice, queueFamilyIndex,
                                                                                             display, error_obj);
            if (skip) return VK_FALSE;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceWaylandPresentationSupportKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceWaylandPresentationSupportKHR(physicalDevice, queueFamilyIndex, display,
                                                                                   record_obj);
        }
    }
    VkBool32 result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceWaylandPresentationSupportKHR(physicalDevice, queueFamilyIndex, display);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceWaylandPresentationSupportKHR(physicalDevice, queueFamilyIndex, display,
                                                                                    record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WAYLAND_KHR
#ifdef VK_USE_PLATFORM_ANDROID_KHR
VKAPI_ATTR VkResult VKAPI_CALL CreateAndroidSurfaceKHR(VkInstance instance, const VkAndroidSurfaceCreateInfoKHR* pCreateInfo,
                                                       const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateAndroidSurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateAndroidSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateAndroidSurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateAndroidSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateAndroidSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateAndroidSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_ANDROID_KHR
#ifdef VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL CreateWin32SurfaceKHR(VkInstance instance, const VkWin32SurfaceCreateInfoKHR* pCreateInfo,
                                                     const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateWin32SurfaceKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateWin32SurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateWin32SurfaceKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateWin32SurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateWin32SurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateWin32SurfaceKHR(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceWin32PresentationSupportKHR(VkPhysicalDevice physicalDevice,
                                                                            uint32_t queueFamilyIndex) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceWin32PresentationSupportKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateGetPhysicalDeviceWin32PresentationSupportKHR(physicalDevice, queueFamilyIndex, error_obj);
            if (skip) return VK_FALSE;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceWin32PresentationSupportKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceWin32PresentationSupportKHR(physicalDevice, queueFamilyIndex, record_obj);
        }
    }
    VkBool32 result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceWin32PresentationSupportKHR(physicalDevice, queueFamilyIndex);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceWin32PresentationSupportKHR(physicalDevice, queueFamilyIndex, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceVideoCapabilitiesKHR(VkPhysicalDevice physicalDevice,
                                                                     const VkVideoProfileInfoKHR* pVideoProfile,
                                                                     VkVideoCapabilitiesKHR* pCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceVideoCapabilitiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceVideoCapabilitiesKHR(physicalDevice, pVideoProfile, pCapabilities,
                                                                                    error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceVideoCapabilitiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceVideoCapabilitiesKHR(physicalDevice, pVideoProfile, pCapabilities, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceVideoCapabilitiesKHR(physicalDevice, pVideoProfile, pCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceVideoCapabilitiesKHR(physicalDevice, pVideoProfile, pCapabilities,
                                                                           record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceVideoFormatPropertiesKHR(VkPhysicalDevice physicalDevice,
                                                                         const VkPhysicalDeviceVideoFormatInfoKHR* pVideoFormatInfo,
                                                                         uint32_t* pVideoFormatPropertyCount,
                                                                         VkVideoFormatPropertiesKHR* pVideoFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceVideoFormatPropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceVideoFormatPropertiesKHR(
                physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount, pVideoFormatProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceVideoFormatPropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceVideoFormatPropertiesKHR(
                physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount, pVideoFormatProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceVideoFormatPropertiesKHR(physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount,
                                                                   pVideoFormatProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceVideoFormatPropertiesKHR(
                physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount, pVideoFormatProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateVideoSessionKHR(VkDevice device, const VkVideoSessionCreateInfoKHR* pCreateInfo,
                                                     const VkAllocationCallbacks* pAllocator, VkVideoSessionKHR* pVideoSession) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateVideoSessionKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateVideoSessionKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateVideoSessionKHR(device, pCreateInfo, pAllocator, pVideoSession, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateVideoSessionKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateVideoSessionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateVideoSessionKHR(device, pCreateInfo, pAllocator, pVideoSession, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateVideoSessionKHR(device, pCreateInfo, pAllocator, pVideoSession);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateVideoSessionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateVideoSessionKHR(device, pCreateInfo, pAllocator, pVideoSession, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyVideoSessionKHR(VkDevice device, VkVideoSessionKHR videoSession,
                                                  const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyVideoSessionKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyVideoSessionKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyVideoSessionKHR(device, videoSession, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyVideoSessionKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyVideoSessionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyVideoSessionKHR(device, videoSession, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyVideoSessionKHR(device, videoSession, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyVideoSessionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyVideoSessionKHR(device, videoSession, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetVideoSessionMemoryRequirementsKHR(VkDevice device, VkVideoSessionKHR videoSession,
                                                                    uint32_t* pMemoryRequirementsCount,
                                                                    VkVideoSessionMemoryRequirementsKHR* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetVideoSessionMemoryRequirementsKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetVideoSessionMemoryRequirementsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetVideoSessionMemoryRequirementsKHR(device, videoSession, pMemoryRequirementsCount,
                                                                                   pMemoryRequirements, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetVideoSessionMemoryRequirementsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetVideoSessionMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetVideoSessionMemoryRequirementsKHR(device, videoSession, pMemoryRequirementsCount,
                                                                         pMemoryRequirements, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetVideoSessionMemoryRequirementsKHR(device, videoSession, pMemoryRequirementsCount, pMemoryRequirements);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetVideoSessionMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetVideoSessionMemoryRequirementsKHR(device, videoSession, pMemoryRequirementsCount,
                                                                          pMemoryRequirements, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindVideoSessionMemoryKHR(VkDevice device, VkVideoSessionKHR videoSession,
                                                         uint32_t bindSessionMemoryInfoCount,
                                                         const VkBindVideoSessionMemoryInfoKHR* pBindSessionMemoryInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindVideoSessionMemoryKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateBindVideoSessionMemoryKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindVideoSessionMemoryKHR(device, videoSession, bindSessionMemoryInfoCount,
                                                                        pBindSessionMemoryInfos, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindVideoSessionMemoryKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindVideoSessionMemoryKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindVideoSessionMemoryKHR(device, videoSession, bindSessionMemoryInfoCount,
                                                              pBindSessionMemoryInfos, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindVideoSessionMemoryKHR(device, videoSession, bindSessionMemoryInfoCount, pBindSessionMemoryInfos);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindVideoSessionMemoryKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindVideoSessionMemoryKHR(device, videoSession, bindSessionMemoryInfoCount,
                                                               pBindSessionMemoryInfos, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateVideoSessionParametersKHR(VkDevice device,
                                                               const VkVideoSessionParametersCreateInfoKHR* pCreateInfo,
                                                               const VkAllocationCallbacks* pAllocator,
                                                               VkVideoSessionParametersKHR* pVideoSessionParameters) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateVideoSessionParametersKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateVideoSessionParametersKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateVideoSessionParametersKHR(device, pCreateInfo, pAllocator,
                                                                              pVideoSessionParameters, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateVideoSessionParametersKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateVideoSessionParametersKHR(device, pCreateInfo, pAllocator, pVideoSessionParameters,
                                                                    record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateVideoSessionParametersKHR(device, pCreateInfo, pAllocator, pVideoSessionParameters);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCreateVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateVideoSessionParametersKHR(device, pCreateInfo, pAllocator, pVideoSessionParameters,
                                                                     record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL UpdateVideoSessionParametersKHR(VkDevice device, VkVideoSessionParametersKHR videoSessionParameters,
                                                               const VkVideoSessionParametersUpdateInfoKHR* pUpdateInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUpdateVideoSessionParametersKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateUpdateVideoSessionParametersKHR]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateUpdateVideoSessionParametersKHR(device, videoSessionParameters, pUpdateInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkUpdateVideoSessionParametersKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordUpdateVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUpdateVideoSessionParametersKHR(device, videoSessionParameters, pUpdateInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchUpdateVideoSessionParametersKHR(device, videoSessionParameters, pUpdateInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordUpdateVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUpdateVideoSessionParametersKHR(device, videoSessionParameters, pUpdateInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyVideoSessionParametersKHR(VkDevice device, VkVideoSessionParametersKHR videoSessionParameters,
                                                            const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyVideoSessionParametersKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyVideoSessionParametersKHR]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateDestroyVideoSessionParametersKHR(device, videoSessionParameters, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyVideoSessionParametersKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyVideoSessionParametersKHR(device, videoSessionParameters, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyVideoSessionParametersKHR(device, videoSessionParameters, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyVideoSessionParametersKHR(device, videoSessionParameters, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginVideoCodingKHR(VkCommandBuffer commandBuffer, const VkVideoBeginCodingInfoKHR* pBeginInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginVideoCodingKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginVideoCodingKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginVideoCodingKHR(commandBuffer, pBeginInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginVideoCodingKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginVideoCodingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginVideoCodingKHR(commandBuffer, pBeginInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginVideoCodingKHR(commandBuffer, pBeginInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginVideoCodingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginVideoCodingKHR(commandBuffer, pBeginInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndVideoCodingKHR(VkCommandBuffer commandBuffer, const VkVideoEndCodingInfoKHR* pEndCodingInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndVideoCodingKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndVideoCodingKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndVideoCodingKHR(commandBuffer, pEndCodingInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndVideoCodingKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndVideoCodingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndVideoCodingKHR(commandBuffer, pEndCodingInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndVideoCodingKHR(commandBuffer, pEndCodingInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndVideoCodingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndVideoCodingKHR(commandBuffer, pEndCodingInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdControlVideoCodingKHR(VkCommandBuffer commandBuffer,
                                                    const VkVideoCodingControlInfoKHR* pCodingControlInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdControlVideoCodingKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdControlVideoCodingKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdControlVideoCodingKHR(commandBuffer, pCodingControlInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdControlVideoCodingKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdControlVideoCodingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdControlVideoCodingKHR(commandBuffer, pCodingControlInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdControlVideoCodingKHR(commandBuffer, pCodingControlInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdControlVideoCodingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdControlVideoCodingKHR(commandBuffer, pCodingControlInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDecodeVideoKHR(VkCommandBuffer commandBuffer, const VkVideoDecodeInfoKHR* pDecodeInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDecodeVideoKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDecodeVideoKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDecodeVideoKHR(commandBuffer, pDecodeInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDecodeVideoKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDecodeVideoKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDecodeVideoKHR(commandBuffer, pDecodeInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDecodeVideoKHR(commandBuffer, pDecodeInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDecodeVideoKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDecodeVideoKHR(commandBuffer, pDecodeInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderingKHR(VkCommandBuffer commandBuffer, const VkRenderingInfo* pRenderingInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginRenderingKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginRenderingKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginRenderingKHR(commandBuffer, pRenderingInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginRenderingKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginRenderingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginRenderingKHR(commandBuffer, pRenderingInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginRenderingKHR(commandBuffer, pRenderingInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginRenderingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginRenderingKHR(commandBuffer, pRenderingInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderingKHR(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndRenderingKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndRenderingKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndRenderingKHR(commandBuffer, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndRenderingKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndRenderingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndRenderingKHR(commandBuffer, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndRenderingKHR(commandBuffer);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndRenderingKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndRenderingKHR(commandBuffer, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFeatures2KHR(VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures2* pFeatures) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFeatures2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceFeatures2KHR(physicalDevice, pFeatures, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFeatures2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFeatures2KHR(physicalDevice, pFeatures, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceFeatures2KHR(physicalDevice, pFeatures);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFeatures2KHR(physicalDevice, pFeatures, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceProperties2KHR(VkPhysicalDevice physicalDevice,
                                                           VkPhysicalDeviceProperties2* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceProperties2KHR(physicalDevice, pProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceProperties2KHR(physicalDevice, pProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceProperties2KHR(physicalDevice, pProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceProperties2KHR(physicalDevice, pProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFormatProperties2KHR(VkPhysicalDevice physicalDevice, VkFormat format,
                                                                 VkFormatProperties2* pFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFormatProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceFormatProperties2KHR(physicalDevice, format, pFormatProperties,
                                                                                    error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFormatProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFormatProperties2KHR(physicalDevice, format, pFormatProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceFormatProperties2KHR(physicalDevice, format, pFormatProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFormatProperties2KHR(physicalDevice, format, pFormatProperties, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceImageFormatProperties2KHR(VkPhysicalDevice physicalDevice,
                                                                          const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo,
                                                                          VkImageFormatProperties2* pImageFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceImageFormatProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceImageFormatProperties2KHR(physicalDevice, pImageFormatInfo,
                                                                                         pImageFormatProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceImageFormatProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceImageFormatProperties2KHR(physicalDevice, pImageFormatInfo,
                                                                               pImageFormatProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceImageFormatProperties2KHR(physicalDevice, pImageFormatInfo, pImageFormatProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceImageFormatProperties2KHR(physicalDevice, pImageFormatInfo,
                                                                                pImageFormatProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyProperties2KHR(VkPhysicalDevice physicalDevice,
                                                                      uint32_t* pQueueFamilyPropertyCount,
                                                                      VkQueueFamilyProperties2* pQueueFamilyProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceQueueFamilyProperties2KHR(physicalDevice, pQueueFamilyPropertyCount,
                                                                                         pQueueFamilyProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceQueueFamilyProperties2KHR(physicalDevice, pQueueFamilyPropertyCount,
                                                                               pQueueFamilyProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceQueueFamilyProperties2KHR(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceQueueFamilyProperties2KHR(physicalDevice, pQueueFamilyPropertyCount,
                                                                                pQueueFamilyProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMemoryProperties2KHR(VkPhysicalDevice physicalDevice,
                                                                 VkPhysicalDeviceMemoryProperties2* pMemoryProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceMemoryProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceMemoryProperties2KHR(physicalDevice, pMemoryProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceMemoryProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceMemoryProperties2KHR(physicalDevice, pMemoryProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceMemoryProperties2KHR(physicalDevice, pMemoryProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceMemoryProperties2KHR(physicalDevice, pMemoryProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceSparseImageFormatProperties2KHR(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount,
    VkSparseImageFormatProperties2* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSparseImageFormatProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSparseImageFormatProperties2KHR(
                physicalDevice, pFormatInfo, pPropertyCount, pProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSparseImageFormatProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSparseImageFormatProperties2KHR(physicalDevice, pFormatInfo, pPropertyCount,
                                                                                     pProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceSparseImageFormatProperties2KHR(physicalDevice, pFormatInfo, pPropertyCount, pProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSparseImageFormatProperties2KHR(physicalDevice, pFormatInfo, pPropertyCount,
                                                                                      pProperties, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceGroupPeerMemoryFeaturesKHR(VkDevice device, uint32_t heapIndex, uint32_t localDeviceIndex,
                                                               uint32_t remoteDeviceIndex,
                                                               VkPeerMemoryFeatureFlags* pPeerMemoryFeatures) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceGroupPeerMemoryFeaturesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceGroupPeerMemoryFeaturesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceGroupPeerMemoryFeaturesKHR(
                device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceGroupPeerMemoryFeaturesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceGroupPeerMemoryFeaturesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceGroupPeerMemoryFeaturesKHR(device, heapIndex, localDeviceIndex, remoteDeviceIndex,
                                                                        pPeerMemoryFeatures, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceGroupPeerMemoryFeaturesKHR(device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceGroupPeerMemoryFeaturesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceGroupPeerMemoryFeaturesKHR(device, heapIndex, localDeviceIndex, remoteDeviceIndex,
                                                                         pPeerMemoryFeatures, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDeviceMaskKHR(VkCommandBuffer commandBuffer, uint32_t deviceMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDeviceMaskKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDeviceMaskKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDeviceMaskKHR(commandBuffer, deviceMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDeviceMaskKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDeviceMaskKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDeviceMaskKHR(commandBuffer, deviceMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDeviceMaskKHR(commandBuffer, deviceMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDeviceMaskKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDeviceMaskKHR(commandBuffer, deviceMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchBaseKHR(VkCommandBuffer commandBuffer, uint32_t baseGroupX, uint32_t baseGroupY,
                                              uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY,
                                              uint32_t groupCountZ) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatchBaseKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatchBaseKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatchBaseKHR(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX,
                                                                 groupCountY, groupCountZ, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatchBaseKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatchBaseKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatchBaseKHR(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY,
                                                       groupCountZ, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatchBaseKHR(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatchBaseKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatchBaseKHR(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY,
                                                        groupCountZ, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL TrimCommandPoolKHR(VkDevice device, VkCommandPool commandPool, VkCommandPoolTrimFlags flags) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkTrimCommandPoolKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateTrimCommandPoolKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateTrimCommandPoolKHR(device, commandPool, flags, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkTrimCommandPoolKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordTrimCommandPoolKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordTrimCommandPoolKHR(device, commandPool, flags, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchTrimCommandPoolKHR(device, commandPool, flags);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordTrimCommandPoolKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordTrimCommandPoolKHR(device, commandPool, flags, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDeviceGroupsKHR(VkInstance instance, uint32_t* pPhysicalDeviceGroupCount,
                                                                VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkEnumeratePhysicalDeviceGroupsKHR, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateEnumeratePhysicalDeviceGroupsKHR(instance, pPhysicalDeviceGroupCount,
                                                                               pPhysicalDeviceGroupProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkEnumeratePhysicalDeviceGroupsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordEnumeratePhysicalDeviceGroupsKHR(instance, pPhysicalDeviceGroupCount,
                                                                     pPhysicalDeviceGroupProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchEnumeratePhysicalDeviceGroupsKHR(instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordEnumeratePhysicalDeviceGroupsKHR(instance, pPhysicalDeviceGroupCount,
                                                                      pPhysicalDeviceGroupProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalBufferPropertiesKHR(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo,
    VkExternalBufferProperties* pExternalBufferProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalBufferPropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalBufferPropertiesKHR(physicalDevice, pExternalBufferInfo,
                                                                                           pExternalBufferProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalBufferPropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalBufferPropertiesKHR(physicalDevice, pExternalBufferInfo,
                                                                                 pExternalBufferProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceExternalBufferPropertiesKHR(physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalBufferPropertiesKHR(physicalDevice, pExternalBufferInfo,
                                                                                  pExternalBufferProperties, record_obj);
        }
    }
}

#ifdef VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL GetMemoryWin32HandleKHR(VkDevice device, const VkMemoryGetWin32HandleInfoKHR* pGetWin32HandleInfo,
                                                       HANDLE* pHandle) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryWin32HandleKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryWin32HandleKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryWin32HandleKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryWin32HandleKHR(device, pGetWin32HandleInfo, pHandle);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryWin32HandlePropertiesKHR(VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType,
                                                                 HANDLE handle,
                                                                 VkMemoryWin32HandlePropertiesKHR* pMemoryWin32HandleProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryWin32HandlePropertiesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryWin32HandlePropertiesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryWin32HandlePropertiesKHR(device, handleType, handle,
                                                                                pMemoryWin32HandleProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryWin32HandlePropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryWin32HandlePropertiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryWin32HandlePropertiesKHR(device, handleType, handle, pMemoryWin32HandleProperties,
                                                                      record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryWin32HandlePropertiesKHR(device, handleType, handle, pMemoryWin32HandleProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryWin32HandlePropertiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryWin32HandlePropertiesKHR(device, handleType, handle, pMemoryWin32HandleProperties,
                                                                       record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL GetMemoryFdKHR(VkDevice device, const VkMemoryGetFdInfoKHR* pGetFdInfo, int* pFd) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryFdKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryFdKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryFdKHR(device, pGetFdInfo, pFd, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryFdKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryFdKHR(device, pGetFdInfo, pFd, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryFdKHR(device, pGetFdInfo, pFd);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryFdKHR(device, pGetFdInfo, pFd, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryFdPropertiesKHR(VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, int fd,
                                                        VkMemoryFdPropertiesKHR* pMemoryFdProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryFdPropertiesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryFdPropertiesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryFdPropertiesKHR(device, handleType, fd, pMemoryFdProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryFdPropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryFdPropertiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryFdPropertiesKHR(device, handleType, fd, pMemoryFdProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryFdPropertiesKHR(device, handleType, fd, pMemoryFdProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryFdPropertiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryFdPropertiesKHR(device, handleType, fd, pMemoryFdProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalSemaphorePropertiesKHR(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo,
    VkExternalSemaphoreProperties* pExternalSemaphoreProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalSemaphorePropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalSemaphorePropertiesKHR(
                physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalSemaphorePropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalSemaphorePropertiesKHR(physicalDevice, pExternalSemaphoreInfo,
                                                                                    pExternalSemaphoreProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceExternalSemaphorePropertiesKHR(physicalDevice, pExternalSemaphoreInfo,
                                                                pExternalSemaphoreProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalSemaphorePropertiesKHR(physicalDevice, pExternalSemaphoreInfo,
                                                                                     pExternalSemaphoreProperties, record_obj);
        }
    }
}

#ifdef VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL
ImportSemaphoreWin32HandleKHR(VkDevice device, const VkImportSemaphoreWin32HandleInfoKHR* pImportSemaphoreWin32HandleInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkImportSemaphoreWin32HandleKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateImportSemaphoreWin32HandleKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateImportSemaphoreWin32HandleKHR(device, pImportSemaphoreWin32HandleInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkImportSemaphoreWin32HandleKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordImportSemaphoreWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordImportSemaphoreWin32HandleKHR(device, pImportSemaphoreWin32HandleInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchImportSemaphoreWin32HandleKHR(device, pImportSemaphoreWin32HandleInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordImportSemaphoreWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordImportSemaphoreWin32HandleKHR(device, pImportSemaphoreWin32HandleInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreWin32HandleKHR(VkDevice device,
                                                          const VkSemaphoreGetWin32HandleInfoKHR* pGetWin32HandleInfo,
                                                          HANDLE* pHandle) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSemaphoreWin32HandleKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetSemaphoreWin32HandleKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSemaphoreWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSemaphoreWin32HandleKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSemaphoreWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSemaphoreWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSemaphoreWin32HandleKHR(device, pGetWin32HandleInfo, pHandle);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSemaphoreWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetSemaphoreWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL ImportSemaphoreFdKHR(VkDevice device, const VkImportSemaphoreFdInfoKHR* pImportSemaphoreFdInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkImportSemaphoreFdKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateImportSemaphoreFdKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateImportSemaphoreFdKHR(device, pImportSemaphoreFdInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkImportSemaphoreFdKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordImportSemaphoreFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordImportSemaphoreFdKHR(device, pImportSemaphoreFdInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchImportSemaphoreFdKHR(device, pImportSemaphoreFdInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordImportSemaphoreFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordImportSemaphoreFdKHR(device, pImportSemaphoreFdInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreFdKHR(VkDevice device, const VkSemaphoreGetFdInfoKHR* pGetFdInfo, int* pFd) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSemaphoreFdKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetSemaphoreFdKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSemaphoreFdKHR(device, pGetFdInfo, pFd, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSemaphoreFdKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSemaphoreFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSemaphoreFdKHR(device, pGetFdInfo, pFd, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSemaphoreFdKHR(device, pGetFdInfo, pFd);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSemaphoreFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetSemaphoreFdKHR(device, pGetFdInfo, pFd, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSetKHR(VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint,
                                                   VkPipelineLayout layout, uint32_t set, uint32_t descriptorWriteCount,
                                                   const VkWriteDescriptorSet* pDescriptorWrites) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPushDescriptorSetKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPushDescriptorSetKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPushDescriptorSetKHR(commandBuffer, pipelineBindPoint, layout, set,
                                                                      descriptorWriteCount, pDescriptorWrites, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPushDescriptorSetKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPushDescriptorSetKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPushDescriptorSetKHR(commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount,
                                                            pDescriptorWrites, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPushDescriptorSetKHR(commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount, pDescriptorWrites);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPushDescriptorSetKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPushDescriptorSetKHR(commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount,
                                                             pDescriptorWrites, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSetWithTemplateKHR(VkCommandBuffer commandBuffer,
                                                               VkDescriptorUpdateTemplate descriptorUpdateTemplate,
                                                               VkPipelineLayout layout, uint32_t set, const void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPushDescriptorSetWithTemplateKHR,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPushDescriptorSetWithTemplateKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPushDescriptorSetWithTemplateKHR(commandBuffer, descriptorUpdateTemplate, layout,
                                                                                  set, pData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPushDescriptorSetWithTemplateKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPushDescriptorSetWithTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPushDescriptorSetWithTemplateKHR(commandBuffer, descriptorUpdateTemplate, layout, set, pData,
                                                                        record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPushDescriptorSetWithTemplateKHR(commandBuffer, descriptorUpdateTemplate, layout, set, pData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPushDescriptorSetWithTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPushDescriptorSetWithTemplateKHR(commandBuffer, descriptorUpdateTemplate, layout, set,
                                                                         pData, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorUpdateTemplateKHR(VkDevice device,
                                                                 const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo,
                                                                 const VkAllocationCallbacks* pAllocator,
                                                                 VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDescriptorUpdateTemplateKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateDescriptorUpdateTemplateKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDescriptorUpdateTemplateKHR(device, pCreateInfo, pAllocator,
                                                                                pDescriptorUpdateTemplate, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDescriptorUpdateTemplateKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCreateDescriptorUpdateTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDescriptorUpdateTemplateKHR(device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate,
                                                                      record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDescriptorUpdateTemplateKHR(device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCreateDescriptorUpdateTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDescriptorUpdateTemplateKHR(device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate,
                                                                       record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorUpdateTemplateKHR(VkDevice device, VkDescriptorUpdateTemplate descriptorUpdateTemplate,
                                                              const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDescriptorUpdateTemplateKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyDescriptorUpdateTemplateKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyDescriptorUpdateTemplateKHR(device, descriptorUpdateTemplate, pAllocator,
                                                                                 error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDescriptorUpdateTemplateKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyDescriptorUpdateTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDescriptorUpdateTemplateKHR(device, descriptorUpdateTemplate, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDescriptorUpdateTemplateKHR(device, descriptorUpdateTemplate, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyDescriptorUpdateTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDescriptorUpdateTemplateKHR(device, descriptorUpdateTemplate, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL UpdateDescriptorSetWithTemplateKHR(VkDevice device, VkDescriptorSet descriptorSet,
                                                              VkDescriptorUpdateTemplate descriptorUpdateTemplate,
                                                              const void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUpdateDescriptorSetWithTemplateKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateUpdateDescriptorSetWithTemplateKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateUpdateDescriptorSetWithTemplateKHR(device, descriptorSet, descriptorUpdateTemplate,
                                                                                 pData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkUpdateDescriptorSetWithTemplateKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordUpdateDescriptorSetWithTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUpdateDescriptorSetWithTemplateKHR(device, descriptorSet, descriptorUpdateTemplate, pData,
                                                                       record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchUpdateDescriptorSetWithTemplateKHR(device, descriptorSet, descriptorUpdateTemplate, pData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordUpdateDescriptorSetWithTemplateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUpdateDescriptorSetWithTemplateKHR(device, descriptorSet, descriptorUpdateTemplate, pData,
                                                                        record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRenderPass2KHR(VkDevice device, const VkRenderPassCreateInfo2* pCreateInfo,
                                                    const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateRenderPass2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateRenderPass2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateRenderPass2KHR(device, pCreateInfo, pAllocator, pRenderPass, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateRenderPass2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateRenderPass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateRenderPass2KHR(device, pCreateInfo, pAllocator, pRenderPass, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateRenderPass2KHR(device, pCreateInfo, pAllocator, pRenderPass);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateRenderPass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateRenderPass2KHR(device, pCreateInfo, pAllocator, pRenderPass, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderPass2KHR(VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin,
                                                  const VkSubpassBeginInfo* pSubpassBeginInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginRenderPass2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginRenderPass2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginRenderPass2KHR(commandBuffer, pRenderPassBegin, pSubpassBeginInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginRenderPass2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginRenderPass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginRenderPass2KHR(commandBuffer, pRenderPassBegin, pSubpassBeginInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginRenderPass2KHR(commandBuffer, pRenderPassBegin, pSubpassBeginInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginRenderPass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginRenderPass2KHR(commandBuffer, pRenderPassBegin, pSubpassBeginInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdNextSubpass2KHR(VkCommandBuffer commandBuffer, const VkSubpassBeginInfo* pSubpassBeginInfo,
                                              const VkSubpassEndInfo* pSubpassEndInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdNextSubpass2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdNextSubpass2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdNextSubpass2KHR(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdNextSubpass2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdNextSubpass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdNextSubpass2KHR(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdNextSubpass2KHR(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdNextSubpass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdNextSubpass2KHR(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderPass2KHR(VkCommandBuffer commandBuffer, const VkSubpassEndInfo* pSubpassEndInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndRenderPass2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndRenderPass2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndRenderPass2KHR(commandBuffer, pSubpassEndInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndRenderPass2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndRenderPass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndRenderPass2KHR(commandBuffer, pSubpassEndInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndRenderPass2KHR(commandBuffer, pSubpassEndInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndRenderPass2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndRenderPass2KHR(commandBuffer, pSubpassEndInfo, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetSwapchainStatusKHR(VkDevice device, VkSwapchainKHR swapchain) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSwapchainStatusKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetSwapchainStatusKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSwapchainStatusKHR(device, swapchain, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSwapchainStatusKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSwapchainStatusKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSwapchainStatusKHR(device, swapchain, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSwapchainStatusKHR(device, swapchain);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSwapchainStatusKHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetSwapchainStatusKHR(device, swapchain, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalFencePropertiesKHR(VkPhysicalDevice physicalDevice,
                                                                       const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo,
                                                                       VkExternalFenceProperties* pExternalFenceProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalFencePropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalFencePropertiesKHR(physicalDevice, pExternalFenceInfo,
                                                                                          pExternalFenceProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalFencePropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalFencePropertiesKHR(physicalDevice, pExternalFenceInfo,
                                                                                pExternalFenceProperties, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceExternalFencePropertiesKHR(physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalFencePropertiesKHR(physicalDevice, pExternalFenceInfo,
                                                                                 pExternalFenceProperties, record_obj);
        }
    }
}

#ifdef VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL ImportFenceWin32HandleKHR(VkDevice device,
                                                         const VkImportFenceWin32HandleInfoKHR* pImportFenceWin32HandleInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkImportFenceWin32HandleKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateImportFenceWin32HandleKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateImportFenceWin32HandleKHR(device, pImportFenceWin32HandleInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkImportFenceWin32HandleKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordImportFenceWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordImportFenceWin32HandleKHR(device, pImportFenceWin32HandleInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchImportFenceWin32HandleKHR(device, pImportFenceWin32HandleInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordImportFenceWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordImportFenceWin32HandleKHR(device, pImportFenceWin32HandleInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetFenceWin32HandleKHR(VkDevice device, const VkFenceGetWin32HandleInfoKHR* pGetWin32HandleInfo,
                                                      HANDLE* pHandle) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetFenceWin32HandleKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetFenceWin32HandleKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetFenceWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetFenceWin32HandleKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetFenceWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetFenceWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetFenceWin32HandleKHR(device, pGetWin32HandleInfo, pHandle);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetFenceWin32HandleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetFenceWin32HandleKHR(device, pGetWin32HandleInfo, pHandle, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL ImportFenceFdKHR(VkDevice device, const VkImportFenceFdInfoKHR* pImportFenceFdInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkImportFenceFdKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateImportFenceFdKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateImportFenceFdKHR(device, pImportFenceFdInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkImportFenceFdKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordImportFenceFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordImportFenceFdKHR(device, pImportFenceFdInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchImportFenceFdKHR(device, pImportFenceFdInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordImportFenceFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordImportFenceFdKHR(device, pImportFenceFdInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetFenceFdKHR(VkDevice device, const VkFenceGetFdInfoKHR* pGetFdInfo, int* pFd) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetFenceFdKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetFenceFdKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetFenceFdKHR(device, pGetFdInfo, pFd, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetFenceFdKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetFenceFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetFenceFdKHR(device, pGetFdInfo, pFd, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetFenceFdKHR(device, pGetFdInfo, pFd);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetFenceFdKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetFenceFdKHR(device, pGetFdInfo, pFd, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
    VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, uint32_t* pCounterCount, VkPerformanceCounterKHR* pCounters,
    VkPerformanceCounterDescriptionKHR* pCounterDescriptions) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
                physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
                physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
            physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
                physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(
    VkPhysicalDevice physicalDevice, const VkQueryPoolPerformanceCreateInfoKHR* pPerformanceQueryCreateInfo, uint32_t* pNumPasses) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(
                physicalDevice, pPerformanceQueryCreateInfo, pNumPasses, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(
                physicalDevice, pPerformanceQueryCreateInfo, pNumPasses, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(physicalDevice, pPerformanceQueryCreateInfo, pNumPasses);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(
                physicalDevice, pPerformanceQueryCreateInfo, pNumPasses, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireProfilingLockKHR(VkDevice device, const VkAcquireProfilingLockInfoKHR* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAcquireProfilingLockKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateAcquireProfilingLockKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAcquireProfilingLockKHR(device, pInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAcquireProfilingLockKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordAcquireProfilingLockKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAcquireProfilingLockKHR(device, pInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAcquireProfilingLockKHR(device, pInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordAcquireProfilingLockKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAcquireProfilingLockKHR(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL ReleaseProfilingLockKHR(VkDevice device) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkReleaseProfilingLockKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateReleaseProfilingLockKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateReleaseProfilingLockKHR(device, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkReleaseProfilingLockKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordReleaseProfilingLockKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordReleaseProfilingLockKHR(device, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchReleaseProfilingLockKHR(device);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordReleaseProfilingLockKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordReleaseProfilingLockKHR(device, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceCapabilities2KHR(VkPhysicalDevice physicalDevice,
                                                                        const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo,
                                                                        VkSurfaceCapabilities2KHR* pSurfaceCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfaceCapabilities2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfaceCapabilities2KHR(physicalDevice, pSurfaceInfo,
                                                                                       pSurfaceCapabilities, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfaceCapabilities2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfaceCapabilities2KHR(physicalDevice, pSurfaceInfo, pSurfaceCapabilities,
                                                                             record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfaceCapabilities2KHR(physicalDevice, pSurfaceInfo, pSurfaceCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfaceCapabilities2KHR(physicalDevice, pSurfaceInfo, pSurfaceCapabilities,
                                                                              record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceFormats2KHR(VkPhysicalDevice physicalDevice,
                                                                   const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo,
                                                                   uint32_t* pSurfaceFormatCount,
                                                                   VkSurfaceFormat2KHR* pSurfaceFormats) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfaceFormats2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfaceFormats2KHR(physicalDevice, pSurfaceInfo, pSurfaceFormatCount,
                                                                                  pSurfaceFormats, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfaceFormats2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfaceFormats2KHR(physicalDevice, pSurfaceInfo, pSurfaceFormatCount,
                                                                        pSurfaceFormats, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfaceFormats2KHR(physicalDevice, pSurfaceInfo, pSurfaceFormatCount, pSurfaceFormats);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfaceFormats2KHR(physicalDevice, pSurfaceInfo, pSurfaceFormatCount,
                                                                         pSurfaceFormats, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayProperties2KHR(VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount,
                                                                      VkDisplayProperties2KHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceDisplayProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceDisplayProperties2KHR(physicalDevice, pPropertyCount, pProperties,
                                                                                     error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceDisplayProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceDisplayProperties2KHR(physicalDevice, pPropertyCount, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceDisplayProperties2KHR(physicalDevice, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceDisplayProperties2KHR(physicalDevice, pPropertyCount, pProperties,
                                                                            record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayPlaneProperties2KHR(VkPhysicalDevice physicalDevice,
                                                                           uint32_t* pPropertyCount,
                                                                           VkDisplayPlaneProperties2KHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceDisplayPlaneProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceDisplayPlaneProperties2KHR(physicalDevice, pPropertyCount,
                                                                                          pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceDisplayPlaneProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceDisplayPlaneProperties2KHR(physicalDevice, pPropertyCount, pProperties,
                                                                                record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceDisplayPlaneProperties2KHR(physicalDevice, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceDisplayPlaneProperties2KHR(physicalDevice, pPropertyCount, pProperties,
                                                                                 record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayModeProperties2KHR(VkPhysicalDevice physicalDevice, VkDisplayKHR display,
                                                            uint32_t* pPropertyCount, VkDisplayModeProperties2KHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDisplayModeProperties2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDisplayModeProperties2KHR(physicalDevice, display, pPropertyCount, pProperties,
                                                                           error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDisplayModeProperties2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDisplayModeProperties2KHR(physicalDevice, display, pPropertyCount, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDisplayModeProperties2KHR(physicalDevice, display, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDisplayModeProperties2KHR(physicalDevice, display, pPropertyCount, pProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayPlaneCapabilities2KHR(VkPhysicalDevice physicalDevice,
                                                               const VkDisplayPlaneInfo2KHR* pDisplayPlaneInfo,
                                                               VkDisplayPlaneCapabilities2KHR* pCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDisplayPlaneCapabilities2KHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDisplayPlaneCapabilities2KHR(physicalDevice, pDisplayPlaneInfo, pCapabilities,
                                                                              error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDisplayPlaneCapabilities2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDisplayPlaneCapabilities2KHR(physicalDevice, pDisplayPlaneInfo, pCapabilities, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDisplayPlaneCapabilities2KHR(physicalDevice, pDisplayPlaneInfo, pCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDisplayPlaneCapabilities2KHR(physicalDevice, pDisplayPlaneInfo, pCapabilities, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetImageMemoryRequirements2KHR(VkDevice device, const VkImageMemoryRequirementsInfo2* pInfo,
                                                          VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageMemoryRequirements2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageMemoryRequirements2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageMemoryRequirements2KHR(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageMemoryRequirements2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageMemoryRequirements2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageMemoryRequirements2KHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageMemoryRequirements2KHR(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageMemoryRequirements2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageMemoryRequirements2KHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetBufferMemoryRequirements2KHR(VkDevice device, const VkBufferMemoryRequirementsInfo2* pInfo,
                                                           VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferMemoryRequirements2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferMemoryRequirements2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferMemoryRequirements2KHR(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferMemoryRequirements2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferMemoryRequirements2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferMemoryRequirements2KHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetBufferMemoryRequirements2KHR(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferMemoryRequirements2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferMemoryRequirements2KHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetImageSparseMemoryRequirements2KHR(VkDevice device, const VkImageSparseMemoryRequirementsInfo2* pInfo,
                                                                uint32_t* pSparseMemoryRequirementCount,
                                                                VkSparseImageMemoryRequirements2* pSparseMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageSparseMemoryRequirements2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageSparseMemoryRequirements2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageSparseMemoryRequirements2KHR(device, pInfo, pSparseMemoryRequirementCount,
                                                                                   pSparseMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageSparseMemoryRequirements2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageSparseMemoryRequirements2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageSparseMemoryRequirements2KHR(device, pInfo, pSparseMemoryRequirementCount,
                                                                         pSparseMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageSparseMemoryRequirements2KHR(device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageSparseMemoryRequirements2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageSparseMemoryRequirements2KHR(device, pInfo, pSparseMemoryRequirementCount,
                                                                          pSparseMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSamplerYcbcrConversionKHR(VkDevice device,
                                                               const VkSamplerYcbcrConversionCreateInfo* pCreateInfo,
                                                               const VkAllocationCallbacks* pAllocator,
                                                               VkSamplerYcbcrConversion* pYcbcrConversion) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateSamplerYcbcrConversionKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateSamplerYcbcrConversionKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateSamplerYcbcrConversionKHR(device, pCreateInfo, pAllocator, pYcbcrConversion,
                                                                              error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateSamplerYcbcrConversionKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateSamplerYcbcrConversionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateSamplerYcbcrConversionKHR(device, pCreateInfo, pAllocator, pYcbcrConversion, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateSamplerYcbcrConversionKHR(device, pCreateInfo, pAllocator, pYcbcrConversion);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCreateSamplerYcbcrConversionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateSamplerYcbcrConversionKHR(device, pCreateInfo, pAllocator, pYcbcrConversion, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySamplerYcbcrConversionKHR(VkDevice device, VkSamplerYcbcrConversion ycbcrConversion,
                                                            const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroySamplerYcbcrConversionKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroySamplerYcbcrConversionKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroySamplerYcbcrConversionKHR(device, ycbcrConversion, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroySamplerYcbcrConversionKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordDestroySamplerYcbcrConversionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroySamplerYcbcrConversionKHR(device, ycbcrConversion, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroySamplerYcbcrConversionKHR(device, ycbcrConversion, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordDestroySamplerYcbcrConversionKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroySamplerYcbcrConversionKHR(device, ycbcrConversion, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL BindBufferMemory2KHR(VkDevice device, uint32_t bindInfoCount,
                                                    const VkBindBufferMemoryInfo* pBindInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindBufferMemory2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBindBufferMemory2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindBufferMemory2KHR(device, bindInfoCount, pBindInfos, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindBufferMemory2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindBufferMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindBufferMemory2KHR(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindBufferMemory2KHR(device, bindInfoCount, pBindInfos);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindBufferMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindBufferMemory2KHR(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindImageMemory2KHR(VkDevice device, uint32_t bindInfoCount,
                                                   const VkBindImageMemoryInfo* pBindInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindImageMemory2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateBindImageMemory2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindImageMemory2KHR(device, bindInfoCount, pBindInfos, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindImageMemory2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordBindImageMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindImageMemory2KHR(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindImageMemory2KHR(device, bindInfoCount, pBindInfos);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordBindImageMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindImageMemory2KHR(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL GetDescriptorSetLayoutSupportKHR(VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo,
                                                            VkDescriptorSetLayoutSupport* pSupport) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDescriptorSetLayoutSupportKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDescriptorSetLayoutSupportKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDescriptorSetLayoutSupportKHR(device, pCreateInfo, pSupport, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDescriptorSetLayoutSupportKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDescriptorSetLayoutSupportKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDescriptorSetLayoutSupportKHR(device, pCreateInfo, pSupport, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDescriptorSetLayoutSupportKHR(device, pCreateInfo, pSupport);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDescriptorSetLayoutSupportKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDescriptorSetLayoutSupportKHR(device, pCreateInfo, pSupport, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectCountKHR(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                   VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount,
                                                   uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndirectCountKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndirectCountKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                      maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndirectCountKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndirectCountKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                            maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndirectCountKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                             maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirectCountKHR(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                          VkBuffer countBuffer, VkDeviceSize countBufferOffset,
                                                          uint32_t maxDrawCount, uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndexedIndirectCountKHR,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndexedIndirectCountKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndexedIndirectCountKHR(commandBuffer, buffer, offset, countBuffer,
                                                                             countBufferOffset, maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndexedIndirectCountKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndexedIndirectCountKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndexedIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                   maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndexedIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndexedIndirectCountKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndexedIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                    maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreCounterValueKHR(VkDevice device, VkSemaphore semaphore, uint64_t* pValue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSemaphoreCounterValueKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetSemaphoreCounterValueKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSemaphoreCounterValueKHR(device, semaphore, pValue, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSemaphoreCounterValueKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSemaphoreCounterValueKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSemaphoreCounterValueKHR(device, semaphore, pValue, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSemaphoreCounterValueKHR(device, semaphore, pValue);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSemaphoreCounterValueKHR]) {
            ValidationObject::BlockingOperationGuard lock(intercept);

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetSemaphoreCounterValueKHR(device, semaphore, pValue, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WaitSemaphoresKHR(VkDevice device, const VkSemaphoreWaitInfo* pWaitInfo, uint64_t timeout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkWaitSemaphoresKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateWaitSemaphoresKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateWaitSemaphoresKHR(device, pWaitInfo, timeout, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkWaitSemaphoresKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordWaitSemaphoresKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordWaitSemaphoresKHR(device, pWaitInfo, timeout, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchWaitSemaphoresKHR(device, pWaitInfo, timeout);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordWaitSemaphoresKHR]) {
            ValidationObject::BlockingOperationGuard lock(intercept);

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordWaitSemaphoresKHR(device, pWaitInfo, timeout, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SignalSemaphoreKHR(VkDevice device, const VkSemaphoreSignalInfo* pSignalInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSignalSemaphoreKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateSignalSemaphoreKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSignalSemaphoreKHR(device, pSignalInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkSignalSemaphoreKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSignalSemaphoreKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSignalSemaphoreKHR(device, pSignalInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchSignalSemaphoreKHR(device, pSignalInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSignalSemaphoreKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSignalSemaphoreKHR(device, pSignalInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL
GetPhysicalDeviceFragmentShadingRatesKHR(VkPhysicalDevice physicalDevice, uint32_t* pFragmentShadingRateCount,
                                         VkPhysicalDeviceFragmentShadingRateKHR* pFragmentShadingRates) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceFragmentShadingRatesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceFragmentShadingRatesKHR(physicalDevice, pFragmentShadingRateCount,
                                                                                       pFragmentShadingRates, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceFragmentShadingRatesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceFragmentShadingRatesKHR(physicalDevice, pFragmentShadingRateCount,
                                                                             pFragmentShadingRates, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceFragmentShadingRatesKHR(physicalDevice, pFragmentShadingRateCount, pFragmentShadingRates);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceFragmentShadingRatesKHR(physicalDevice, pFragmentShadingRateCount,
                                                                              pFragmentShadingRates, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetFragmentShadingRateKHR(VkCommandBuffer commandBuffer, const VkExtent2D* pFragmentSize,
                                                        const VkFragmentShadingRateCombinerOpKHR combinerOps[2]) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetFragmentShadingRateKHR,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetFragmentShadingRateKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetFragmentShadingRateKHR(commandBuffer, pFragmentSize, combinerOps, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetFragmentShadingRateKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetFragmentShadingRateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetFragmentShadingRateKHR(commandBuffer, pFragmentSize, combinerOps, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetFragmentShadingRateKHR(commandBuffer, pFragmentSize, combinerOps);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetFragmentShadingRateKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetFragmentShadingRateKHR(commandBuffer, pFragmentSize, combinerOps, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetRenderingAttachmentLocationsKHR(VkCommandBuffer commandBuffer,
                                                                 const VkRenderingAttachmentLocationInfoKHR* pLocationInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetRenderingAttachmentLocationsKHR,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetRenderingAttachmentLocationsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetRenderingAttachmentLocationsKHR(commandBuffer, pLocationInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetRenderingAttachmentLocationsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetRenderingAttachmentLocationsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetRenderingAttachmentLocationsKHR(commandBuffer, pLocationInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetRenderingAttachmentLocationsKHR(commandBuffer, pLocationInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetRenderingAttachmentLocationsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetRenderingAttachmentLocationsKHR(commandBuffer, pLocationInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetRenderingInputAttachmentIndicesKHR(
    VkCommandBuffer commandBuffer, const VkRenderingInputAttachmentIndexInfoKHR* pInputAttachmentIndexInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetRenderingInputAttachmentIndicesKHR,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetRenderingInputAttachmentIndicesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetRenderingInputAttachmentIndicesKHR(commandBuffer, pInputAttachmentIndexInfo,
                                                                                       error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetRenderingInputAttachmentIndicesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetRenderingInputAttachmentIndicesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetRenderingInputAttachmentIndicesKHR(commandBuffer, pInputAttachmentIndexInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetRenderingInputAttachmentIndicesKHR(commandBuffer, pInputAttachmentIndexInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetRenderingInputAttachmentIndicesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetRenderingInputAttachmentIndicesKHR(commandBuffer, pInputAttachmentIndexInfo, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL WaitForPresentKHR(VkDevice device, VkSwapchainKHR swapchain, uint64_t presentId, uint64_t timeout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkWaitForPresentKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateWaitForPresentKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateWaitForPresentKHR(device, swapchain, presentId, timeout, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkWaitForPresentKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordWaitForPresentKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordWaitForPresentKHR(device, swapchain, presentId, timeout, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchWaitForPresentKHR(device, swapchain, presentId, timeout);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordWaitForPresentKHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordWaitForPresentKHR(device, swapchain, presentId, timeout, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetBufferDeviceAddressKHR(VkDevice device, const VkBufferDeviceAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferDeviceAddressKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferDeviceAddressKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferDeviceAddressKHR(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferDeviceAddressKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferDeviceAddressKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferDeviceAddressKHR(device, pInfo, record_obj);
        }
    }
    VkDeviceAddress result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetBufferDeviceAddressKHR(device, pInfo);
    }
    record_obj.device_address = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferDeviceAddressKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferDeviceAddressKHR(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetBufferOpaqueCaptureAddressKHR(VkDevice device, const VkBufferDeviceAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferOpaqueCaptureAddressKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferOpaqueCaptureAddressKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferOpaqueCaptureAddressKHR(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferOpaqueCaptureAddressKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferOpaqueCaptureAddressKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferOpaqueCaptureAddressKHR(device, pInfo, record_obj);
        }
    }
    uint64_t result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetBufferOpaqueCaptureAddressKHR(device, pInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferOpaqueCaptureAddressKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferOpaqueCaptureAddressKHR(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetDeviceMemoryOpaqueCaptureAddressKHR(VkDevice device,
                                                                      const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceMemoryOpaqueCaptureAddressKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceMemoryOpaqueCaptureAddressKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceMemoryOpaqueCaptureAddressKHR(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceMemoryOpaqueCaptureAddressKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceMemoryOpaqueCaptureAddressKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceMemoryOpaqueCaptureAddressKHR(device, pInfo, record_obj);
        }
    }
    uint64_t result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeviceMemoryOpaqueCaptureAddressKHR(device, pInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceMemoryOpaqueCaptureAddressKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceMemoryOpaqueCaptureAddressKHR(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDeferredOperationKHR(VkDevice device, const VkAllocationCallbacks* pAllocator,
                                                          VkDeferredOperationKHR* pDeferredOperation) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDeferredOperationKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateDeferredOperationKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDeferredOperationKHR(device, pAllocator, pDeferredOperation, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDeferredOperationKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateDeferredOperationKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDeferredOperationKHR(device, pAllocator, pDeferredOperation, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDeferredOperationKHR(device, pAllocator, pDeferredOperation);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateDeferredOperationKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDeferredOperationKHR(device, pAllocator, pDeferredOperation, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDeferredOperationKHR(VkDevice device, VkDeferredOperationKHR operation,
                                                       const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDeferredOperationKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyDeferredOperationKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyDeferredOperationKHR(device, operation, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDeferredOperationKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyDeferredOperationKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDeferredOperationKHR(device, operation, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDeferredOperationKHR(device, operation, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyDeferredOperationKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDeferredOperationKHR(device, operation, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR uint32_t VKAPI_CALL GetDeferredOperationMaxConcurrencyKHR(VkDevice device, VkDeferredOperationKHR operation) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeferredOperationMaxConcurrencyKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeferredOperationMaxConcurrencyKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeferredOperationMaxConcurrencyKHR(device, operation, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeferredOperationMaxConcurrencyKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeferredOperationMaxConcurrencyKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeferredOperationMaxConcurrencyKHR(device, operation, record_obj);
        }
    }
    uint32_t result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeferredOperationMaxConcurrencyKHR(device, operation);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeferredOperationMaxConcurrencyKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeferredOperationMaxConcurrencyKHR(device, operation, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeferredOperationResultKHR(VkDevice device, VkDeferredOperationKHR operation) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeferredOperationResultKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeferredOperationResultKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeferredOperationResultKHR(device, operation, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeferredOperationResultKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeferredOperationResultKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeferredOperationResultKHR(device, operation, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeferredOperationResultKHR(device, operation);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeferredOperationResultKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeferredOperationResultKHR(device, operation, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DeferredOperationJoinKHR(VkDevice device, VkDeferredOperationKHR operation) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDeferredOperationJoinKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDeferredOperationJoinKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDeferredOperationJoinKHR(device, operation, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkDeferredOperationJoinKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDeferredOperationJoinKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDeferredOperationJoinKHR(device, operation, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchDeferredOperationJoinKHR(device, operation);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDeferredOperationJoinKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDeferredOperationJoinKHR(device, operation, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineExecutablePropertiesKHR(VkDevice device, const VkPipelineInfoKHR* pPipelineInfo,
                                                                  uint32_t* pExecutableCount,
                                                                  VkPipelineExecutablePropertiesKHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPipelineExecutablePropertiesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetPipelineExecutablePropertiesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPipelineExecutablePropertiesKHR(device, pPipelineInfo, pExecutableCount,
                                                                                 pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPipelineExecutablePropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetPipelineExecutablePropertiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPipelineExecutablePropertiesKHR(device, pPipelineInfo, pExecutableCount, pProperties,
                                                                       record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPipelineExecutablePropertiesKHR(device, pPipelineInfo, pExecutableCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetPipelineExecutablePropertiesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPipelineExecutablePropertiesKHR(device, pPipelineInfo, pExecutableCount, pProperties,
                                                                        record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineExecutableStatisticsKHR(VkDevice device,
                                                                  const VkPipelineExecutableInfoKHR* pExecutableInfo,
                                                                  uint32_t* pStatisticCount,
                                                                  VkPipelineExecutableStatisticKHR* pStatistics) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPipelineExecutableStatisticsKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetPipelineExecutableStatisticsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPipelineExecutableStatisticsKHR(device, pExecutableInfo, pStatisticCount,
                                                                                 pStatistics, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPipelineExecutableStatisticsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetPipelineExecutableStatisticsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPipelineExecutableStatisticsKHR(device, pExecutableInfo, pStatisticCount, pStatistics,
                                                                       record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPipelineExecutableStatisticsKHR(device, pExecutableInfo, pStatisticCount, pStatistics);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetPipelineExecutableStatisticsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPipelineExecutableStatisticsKHR(device, pExecutableInfo, pStatisticCount, pStatistics,
                                                                        record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineExecutableInternalRepresentationsKHR(
    VkDevice device, const VkPipelineExecutableInfoKHR* pExecutableInfo, uint32_t* pInternalRepresentationCount,
    VkPipelineExecutableInternalRepresentationKHR* pInternalRepresentations) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPipelineExecutableInternalRepresentationsKHR,
                          VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetPipelineExecutableInternalRepresentationsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPipelineExecutableInternalRepresentationsKHR(
                device, pExecutableInfo, pInternalRepresentationCount, pInternalRepresentations, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPipelineExecutableInternalRepresentationsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetPipelineExecutableInternalRepresentationsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPipelineExecutableInternalRepresentationsKHR(
                device, pExecutableInfo, pInternalRepresentationCount, pInternalRepresentations, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPipelineExecutableInternalRepresentationsKHR(device, pExecutableInfo, pInternalRepresentationCount,
                                                                         pInternalRepresentations);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetPipelineExecutableInternalRepresentationsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPipelineExecutableInternalRepresentationsKHR(
                device, pExecutableInfo, pInternalRepresentationCount, pInternalRepresentations, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL MapMemory2KHR(VkDevice device, const VkMemoryMapInfoKHR* pMemoryMapInfo, void** ppData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkMapMemory2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateMapMemory2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateMapMemory2KHR(device, pMemoryMapInfo, ppData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkMapMemory2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordMapMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordMapMemory2KHR(device, pMemoryMapInfo, ppData, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchMapMemory2KHR(device, pMemoryMapInfo, ppData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordMapMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordMapMemory2KHR(device, pMemoryMapInfo, ppData, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL UnmapMemory2KHR(VkDevice device, const VkMemoryUnmapInfoKHR* pMemoryUnmapInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUnmapMemory2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateUnmapMemory2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateUnmapMemory2KHR(device, pMemoryUnmapInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkUnmapMemory2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordUnmapMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUnmapMemory2KHR(device, pMemoryUnmapInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchUnmapMemory2KHR(device, pMemoryUnmapInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordUnmapMemory2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUnmapMemory2KHR(device, pMemoryUnmapInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(
    VkPhysicalDevice physicalDevice, const VkPhysicalDeviceVideoEncodeQualityLevelInfoKHR* pQualityLevelInfo,
    VkVideoEncodeQualityLevelPropertiesKHR* pQualityLevelProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(
                physicalDevice, pQualityLevelInfo, pQualityLevelProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(physicalDevice, pQualityLevelInfo,
                                                                                          pQualityLevelProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(physicalDevice, pQualityLevelInfo,
                                                                               pQualityLevelProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(physicalDevice, pQualityLevelInfo,
                                                                                           pQualityLevelProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL
GetEncodedVideoSessionParametersKHR(VkDevice device, const VkVideoEncodeSessionParametersGetInfoKHR* pVideoSessionParametersInfo,
                                    VkVideoEncodeSessionParametersFeedbackInfoKHR* pFeedbackInfo, size_t* pDataSize, void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetEncodedVideoSessionParametersKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetEncodedVideoSessionParametersKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetEncodedVideoSessionParametersKHR(device, pVideoSessionParametersInfo,
                                                                                  pFeedbackInfo, pDataSize, pData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetEncodedVideoSessionParametersKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetEncodedVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetEncodedVideoSessionParametersKHR(device, pVideoSessionParametersInfo, pFeedbackInfo,
                                                                        pDataSize, pData, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetEncodedVideoSessionParametersKHR(device, pVideoSessionParametersInfo, pFeedbackInfo, pDataSize, pData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetEncodedVideoSessionParametersKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetEncodedVideoSessionParametersKHR(device, pVideoSessionParametersInfo, pFeedbackInfo,
                                                                         pDataSize, pData, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdEncodeVideoKHR(VkCommandBuffer commandBuffer, const VkVideoEncodeInfoKHR* pEncodeInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEncodeVideoKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEncodeVideoKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEncodeVideoKHR(commandBuffer, pEncodeInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEncodeVideoKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEncodeVideoKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEncodeVideoKHR(commandBuffer, pEncodeInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEncodeVideoKHR(commandBuffer, pEncodeInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEncodeVideoKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEncodeVideoKHR(commandBuffer, pEncodeInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetEvent2KHR(VkCommandBuffer commandBuffer, VkEvent event, const VkDependencyInfo* pDependencyInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetEvent2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetEvent2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetEvent2KHR(commandBuffer, event, pDependencyInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetEvent2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetEvent2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetEvent2KHR(commandBuffer, event, pDependencyInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetEvent2KHR(commandBuffer, event, pDependencyInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetEvent2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetEvent2KHR(commandBuffer, event, pDependencyInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResetEvent2KHR(VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags2 stageMask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResetEvent2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResetEvent2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResetEvent2KHR(commandBuffer, event, stageMask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResetEvent2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResetEvent2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResetEvent2KHR(commandBuffer, event, stageMask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResetEvent2KHR(commandBuffer, event, stageMask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResetEvent2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResetEvent2KHR(commandBuffer, event, stageMask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdWaitEvents2KHR(VkCommandBuffer commandBuffer, uint32_t eventCount, const VkEvent* pEvents,
                                             const VkDependencyInfo* pDependencyInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWaitEvents2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWaitEvents2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWaitEvents2KHR(commandBuffer, eventCount, pEvents, pDependencyInfos, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWaitEvents2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWaitEvents2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWaitEvents2KHR(commandBuffer, eventCount, pEvents, pDependencyInfos, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWaitEvents2KHR(commandBuffer, eventCount, pEvents, pDependencyInfos);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWaitEvents2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWaitEvents2KHR(commandBuffer, eventCount, pEvents, pDependencyInfos, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPipelineBarrier2KHR(VkCommandBuffer commandBuffer, const VkDependencyInfo* pDependencyInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPipelineBarrier2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPipelineBarrier2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPipelineBarrier2KHR(commandBuffer, pDependencyInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPipelineBarrier2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPipelineBarrier2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPipelineBarrier2KHR(commandBuffer, pDependencyInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPipelineBarrier2KHR(commandBuffer, pDependencyInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPipelineBarrier2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPipelineBarrier2KHR(commandBuffer, pDependencyInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdWriteTimestamp2KHR(VkCommandBuffer commandBuffer, VkPipelineStageFlags2 stage, VkQueryPool queryPool,
                                                 uint32_t query) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWriteTimestamp2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWriteTimestamp2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWriteTimestamp2KHR(commandBuffer, stage, queryPool, query, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWriteTimestamp2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWriteTimestamp2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWriteTimestamp2KHR(commandBuffer, stage, queryPool, query, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWriteTimestamp2KHR(commandBuffer, stage, queryPool, query);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWriteTimestamp2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWriteTimestamp2KHR(commandBuffer, stage, queryPool, query, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSubmit2KHR(VkQueue queue, uint32_t submitCount, const VkSubmitInfo2* pSubmits, VkFence fence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueSubmit2KHR, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateQueueSubmit2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueSubmit2KHR(queue, submitCount, pSubmits, fence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueSubmit2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueSubmit2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueSubmit2KHR(queue, submitCount, pSubmits, fence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueueSubmit2KHR(queue, submitCount, pSubmits, fence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueSubmit2KHR]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordQueueSubmit2KHR(queue, submitCount, pSubmits, fence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdWriteBufferMarker2AMD(VkCommandBuffer commandBuffer, VkPipelineStageFlags2 stage, VkBuffer dstBuffer,
                                                    VkDeviceSize dstOffset, uint32_t marker) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWriteBufferMarker2AMD, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWriteBufferMarker2AMD]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCmdWriteBufferMarker2AMD(commandBuffer, stage, dstBuffer, dstOffset, marker, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWriteBufferMarker2AMD);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWriteBufferMarker2AMD]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWriteBufferMarker2AMD(commandBuffer, stage, dstBuffer, dstOffset, marker, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWriteBufferMarker2AMD(commandBuffer, stage, dstBuffer, dstOffset, marker);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWriteBufferMarker2AMD]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWriteBufferMarker2AMD(commandBuffer, stage, dstBuffer, dstOffset, marker, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetQueueCheckpointData2NV(VkQueue queue, uint32_t* pCheckpointDataCount,
                                                     VkCheckpointData2NV* pCheckpointData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetQueueCheckpointData2NV, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetQueueCheckpointData2NV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetQueueCheckpointData2NV(queue, pCheckpointDataCount, pCheckpointData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetQueueCheckpointData2NV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetQueueCheckpointData2NV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetQueueCheckpointData2NV(queue, pCheckpointDataCount, pCheckpointData, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetQueueCheckpointData2NV(queue, pCheckpointDataCount, pCheckpointData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetQueueCheckpointData2NV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetQueueCheckpointData2NV(queue, pCheckpointDataCount, pCheckpointData, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBuffer2KHR(VkCommandBuffer commandBuffer, const VkCopyBufferInfo2* pCopyBufferInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyBuffer2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyBuffer2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyBuffer2KHR(commandBuffer, pCopyBufferInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyBuffer2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyBuffer2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyBuffer2KHR(commandBuffer, pCopyBufferInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyBuffer2KHR(commandBuffer, pCopyBufferInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyBuffer2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyBuffer2KHR(commandBuffer, pCopyBufferInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImage2KHR(VkCommandBuffer commandBuffer, const VkCopyImageInfo2* pCopyImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyImage2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyImage2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyImage2KHR(commandBuffer, pCopyImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyImage2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyImage2KHR(commandBuffer, pCopyImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyImage2KHR(commandBuffer, pCopyImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyImage2KHR(commandBuffer, pCopyImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBufferToImage2KHR(VkCommandBuffer commandBuffer,
                                                    const VkCopyBufferToImageInfo2* pCopyBufferToImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyBufferToImage2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyBufferToImage2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyBufferToImage2KHR(commandBuffer, pCopyBufferToImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyBufferToImage2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyBufferToImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyBufferToImage2KHR(commandBuffer, pCopyBufferToImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyBufferToImage2KHR(commandBuffer, pCopyBufferToImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyBufferToImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyBufferToImage2KHR(commandBuffer, pCopyBufferToImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImageToBuffer2KHR(VkCommandBuffer commandBuffer,
                                                    const VkCopyImageToBufferInfo2* pCopyImageToBufferInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyImageToBuffer2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyImageToBuffer2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyImageToBuffer2KHR(commandBuffer, pCopyImageToBufferInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyImageToBuffer2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyImageToBuffer2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyImageToBuffer2KHR(commandBuffer, pCopyImageToBufferInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyImageToBuffer2KHR(commandBuffer, pCopyImageToBufferInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyImageToBuffer2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyImageToBuffer2KHR(commandBuffer, pCopyImageToBufferInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBlitImage2KHR(VkCommandBuffer commandBuffer, const VkBlitImageInfo2* pBlitImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBlitImage2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBlitImage2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBlitImage2KHR(commandBuffer, pBlitImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBlitImage2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBlitImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBlitImage2KHR(commandBuffer, pBlitImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBlitImage2KHR(commandBuffer, pBlitImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBlitImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBlitImage2KHR(commandBuffer, pBlitImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdResolveImage2KHR(VkCommandBuffer commandBuffer, const VkResolveImageInfo2* pResolveImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdResolveImage2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdResolveImage2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdResolveImage2KHR(commandBuffer, pResolveImageInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdResolveImage2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdResolveImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdResolveImage2KHR(commandBuffer, pResolveImageInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdResolveImage2KHR(commandBuffer, pResolveImageInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdResolveImage2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdResolveImage2KHR(commandBuffer, pResolveImageInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdTraceRaysIndirect2KHR(VkCommandBuffer commandBuffer, VkDeviceAddress indirectDeviceAddress) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdTraceRaysIndirect2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdTraceRaysIndirect2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdTraceRaysIndirect2KHR(commandBuffer, indirectDeviceAddress, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdTraceRaysIndirect2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdTraceRaysIndirect2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdTraceRaysIndirect2KHR(commandBuffer, indirectDeviceAddress, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdTraceRaysIndirect2KHR(commandBuffer, indirectDeviceAddress);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdTraceRaysIndirect2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdTraceRaysIndirect2KHR(commandBuffer, indirectDeviceAddress, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceBufferMemoryRequirementsKHR(VkDevice device, const VkDeviceBufferMemoryRequirements* pInfo,
                                                                VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceBufferMemoryRequirementsKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceBufferMemoryRequirementsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceBufferMemoryRequirementsKHR(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceBufferMemoryRequirementsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceBufferMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceBufferMemoryRequirementsKHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceBufferMemoryRequirementsKHR(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceBufferMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceBufferMemoryRequirementsKHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageMemoryRequirementsKHR(VkDevice device, const VkDeviceImageMemoryRequirements* pInfo,
                                                               VkMemoryRequirements2* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceImageMemoryRequirementsKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceImageMemoryRequirementsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceImageMemoryRequirementsKHR(device, pInfo, pMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceImageMemoryRequirementsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceImageMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceImageMemoryRequirementsKHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceImageMemoryRequirementsKHR(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceImageMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceImageMemoryRequirementsKHR(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageSparseMemoryRequirementsKHR(VkDevice device, const VkDeviceImageMemoryRequirements* pInfo,
                                                                     uint32_t* pSparseMemoryRequirementCount,
                                                                     VkSparseImageMemoryRequirements2* pSparseMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceImageSparseMemoryRequirementsKHR,
                          VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceImageSparseMemoryRequirementsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceImageSparseMemoryRequirementsKHR(
                device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceImageSparseMemoryRequirementsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceImageSparseMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceImageSparseMemoryRequirementsKHR(device, pInfo, pSparseMemoryRequirementCount,
                                                                              pSparseMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceImageSparseMemoryRequirementsKHR(device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceImageSparseMemoryRequirementsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceImageSparseMemoryRequirementsKHR(device, pInfo, pSparseMemoryRequirementCount,
                                                                               pSparseMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindIndexBuffer2KHR(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                  VkDeviceSize size, VkIndexType indexType) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindIndexBuffer2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindIndexBuffer2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindIndexBuffer2KHR(commandBuffer, buffer, offset, size, indexType, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindIndexBuffer2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindIndexBuffer2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindIndexBuffer2KHR(commandBuffer, buffer, offset, size, indexType, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindIndexBuffer2KHR(commandBuffer, buffer, offset, size, indexType);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindIndexBuffer2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindIndexBuffer2KHR(commandBuffer, buffer, offset, size, indexType, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetRenderingAreaGranularityKHR(VkDevice device, const VkRenderingAreaInfoKHR* pRenderingAreaInfo,
                                                          VkExtent2D* pGranularity) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetRenderingAreaGranularityKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetRenderingAreaGranularityKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetRenderingAreaGranularityKHR(device, pRenderingAreaInfo, pGranularity, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetRenderingAreaGranularityKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetRenderingAreaGranularityKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetRenderingAreaGranularityKHR(device, pRenderingAreaInfo, pGranularity, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetRenderingAreaGranularityKHR(device, pRenderingAreaInfo, pGranularity);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetRenderingAreaGranularityKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetRenderingAreaGranularityKHR(device, pRenderingAreaInfo, pGranularity, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageSubresourceLayoutKHR(VkDevice device, const VkDeviceImageSubresourceInfoKHR* pInfo,
                                                              VkSubresourceLayout2KHR* pLayout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceImageSubresourceLayoutKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceImageSubresourceLayoutKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceImageSubresourceLayoutKHR(device, pInfo, pLayout, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceImageSubresourceLayoutKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceImageSubresourceLayoutKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceImageSubresourceLayoutKHR(device, pInfo, pLayout, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetDeviceImageSubresourceLayoutKHR(device, pInfo, pLayout);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceImageSubresourceLayoutKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceImageSubresourceLayoutKHR(device, pInfo, pLayout, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetImageSubresourceLayout2KHR(VkDevice device, VkImage image, const VkImageSubresource2KHR* pSubresource,
                                                         VkSubresourceLayout2KHR* pLayout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageSubresourceLayout2KHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageSubresourceLayout2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageSubresourceLayout2KHR(device, image, pSubresource, pLayout, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageSubresourceLayout2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageSubresourceLayout2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageSubresourceLayout2KHR(device, image, pSubresource, pLayout, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetImageSubresourceLayout2KHR(device, image, pSubresource, pLayout);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageSubresourceLayout2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageSubresourceLayout2KHR(device, image, pSubresource, pLayout, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePipelineBinariesKHR(VkDevice device, const VkPipelineBinaryCreateInfoKHR* pCreateInfo,
                                                         const VkAllocationCallbacks* pAllocator,
                                                         VkPipelineBinaryHandlesInfoKHR* pBinaries) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreatePipelineBinariesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreatePipelineBinariesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreatePipelineBinariesKHR(device, pCreateInfo, pAllocator, pBinaries, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreatePipelineBinariesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreatePipelineBinariesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreatePipelineBinariesKHR(device, pCreateInfo, pAllocator, pBinaries, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreatePipelineBinariesKHR(device, pCreateInfo, pAllocator, pBinaries);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreatePipelineBinariesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreatePipelineBinariesKHR(device, pCreateInfo, pAllocator, pBinaries, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPipelineBinaryKHR(VkDevice device, VkPipelineBinaryKHR pipelineBinary,
                                                    const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyPipelineBinaryKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyPipelineBinaryKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyPipelineBinaryKHR(device, pipelineBinary, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyPipelineBinaryKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyPipelineBinaryKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyPipelineBinaryKHR(device, pipelineBinary, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyPipelineBinaryKHR(device, pipelineBinary, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyPipelineBinaryKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyPipelineBinaryKHR(device, pipelineBinary, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineKeyKHR(VkDevice device, const VkPipelineCreateInfoKHR* pPipelineCreateInfo,
                                                 VkPipelineBinaryKeyKHR* pPipelineKey) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPipelineKeyKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetPipelineKeyKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPipelineKeyKHR(device, pPipelineCreateInfo, pPipelineKey, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPipelineKeyKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetPipelineKeyKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPipelineKeyKHR(device, pPipelineCreateInfo, pPipelineKey, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPipelineKeyKHR(device, pPipelineCreateInfo, pPipelineKey);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetPipelineKeyKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPipelineKeyKHR(device, pPipelineCreateInfo, pPipelineKey, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineBinaryDataKHR(VkDevice device, const VkPipelineBinaryDataInfoKHR* pInfo,
                                                        VkPipelineBinaryKeyKHR* pPipelineBinaryKey, size_t* pPipelineBinaryDataSize,
                                                        void* pPipelineBinaryData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPipelineBinaryDataKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetPipelineBinaryDataKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPipelineBinaryDataKHR(device, pInfo, pPipelineBinaryKey, pPipelineBinaryDataSize,
                                                                       pPipelineBinaryData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPipelineBinaryDataKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetPipelineBinaryDataKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPipelineBinaryDataKHR(device, pInfo, pPipelineBinaryKey, pPipelineBinaryDataSize,
                                                             pPipelineBinaryData, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPipelineBinaryDataKHR(device, pInfo, pPipelineBinaryKey, pPipelineBinaryDataSize, pPipelineBinaryData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetPipelineBinaryDataKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPipelineBinaryDataKHR(device, pInfo, pPipelineBinaryKey, pPipelineBinaryDataSize,
                                                              pPipelineBinaryData, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ReleaseCapturedPipelineDataKHR(VkDevice device, const VkReleaseCapturedPipelineDataInfoKHR* pInfo,
                                                              const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkReleaseCapturedPipelineDataKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateReleaseCapturedPipelineDataKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateReleaseCapturedPipelineDataKHR(device, pInfo, pAllocator, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkReleaseCapturedPipelineDataKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordReleaseCapturedPipelineDataKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordReleaseCapturedPipelineDataKHR(device, pInfo, pAllocator, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchReleaseCapturedPipelineDataKHR(device, pInfo, pAllocator);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordReleaseCapturedPipelineDataKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordReleaseCapturedPipelineDataKHR(device, pInfo, pAllocator, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCooperativeMatrixPropertiesKHR(VkPhysicalDevice physicalDevice,
                                                                               uint32_t* pPropertyCount,
                                                                               VkCooperativeMatrixPropertiesKHR* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceCooperativeMatrixPropertiesKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceCooperativeMatrixPropertiesKHR(physicalDevice, pPropertyCount,
                                                                                              pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceCooperativeMatrixPropertiesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceCooperativeMatrixPropertiesKHR(physicalDevice, pPropertyCount, pProperties,
                                                                                    record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceCooperativeMatrixPropertiesKHR(physicalDevice, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceCooperativeMatrixPropertiesKHR(physicalDevice, pPropertyCount, pProperties,
                                                                                     record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetLineStippleKHR(VkCommandBuffer commandBuffer, uint32_t lineStippleFactor,
                                                uint16_t lineStipplePattern) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetLineStippleKHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetLineStippleKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetLineStippleKHR(commandBuffer, lineStippleFactor, lineStipplePattern, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetLineStippleKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetLineStippleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetLineStippleKHR(commandBuffer, lineStippleFactor, lineStipplePattern, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetLineStippleKHR(commandBuffer, lineStippleFactor, lineStipplePattern);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetLineStippleKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetLineStippleKHR(commandBuffer, lineStippleFactor, lineStipplePattern, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCalibrateableTimeDomainsKHR(VkPhysicalDevice physicalDevice,
                                                                            uint32_t* pTimeDomainCount,
                                                                            VkTimeDomainKHR* pTimeDomains) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceCalibrateableTimeDomainsKHR,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceCalibrateableTimeDomainsKHR(physicalDevice, pTimeDomainCount,
                                                                                           pTimeDomains, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceCalibrateableTimeDomainsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceCalibrateableTimeDomainsKHR(physicalDevice, pTimeDomainCount, pTimeDomains,
                                                                                 record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceCalibrateableTimeDomainsKHR(physicalDevice, pTimeDomainCount, pTimeDomains);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceCalibrateableTimeDomainsKHR(physicalDevice, pTimeDomainCount, pTimeDomains,
                                                                                  record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetCalibratedTimestampsKHR(VkDevice device, uint32_t timestampCount,
                                                          const VkCalibratedTimestampInfoKHR* pTimestampInfos,
                                                          uint64_t* pTimestamps, uint64_t* pMaxDeviation) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetCalibratedTimestampsKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetCalibratedTimestampsKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetCalibratedTimestampsKHR(device, timestampCount, pTimestampInfos, pTimestamps,
                                                                         pMaxDeviation, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetCalibratedTimestampsKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetCalibratedTimestampsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetCalibratedTimestampsKHR(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation,
                                                               record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetCalibratedTimestampsKHR(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetCalibratedTimestampsKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetCalibratedTimestampsKHR(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation,
                                                                record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindDescriptorSets2KHR(VkCommandBuffer commandBuffer,
                                                     const VkBindDescriptorSetsInfoKHR* pBindDescriptorSetsInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindDescriptorSets2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindDescriptorSets2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindDescriptorSets2KHR(commandBuffer, pBindDescriptorSetsInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindDescriptorSets2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindDescriptorSets2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindDescriptorSets2KHR(commandBuffer, pBindDescriptorSetsInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindDescriptorSets2KHR(commandBuffer, pBindDescriptorSetsInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindDescriptorSets2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindDescriptorSets2KHR(commandBuffer, pBindDescriptorSetsInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPushConstants2KHR(VkCommandBuffer commandBuffer, const VkPushConstantsInfoKHR* pPushConstantsInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPushConstants2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPushConstants2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPushConstants2KHR(commandBuffer, pPushConstantsInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPushConstants2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPushConstants2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPushConstants2KHR(commandBuffer, pPushConstantsInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPushConstants2KHR(commandBuffer, pPushConstantsInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPushConstants2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPushConstants2KHR(commandBuffer, pPushConstantsInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSet2KHR(VkCommandBuffer commandBuffer,
                                                    const VkPushDescriptorSetInfoKHR* pPushDescriptorSetInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPushDescriptorSet2KHR, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPushDescriptorSet2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPushDescriptorSet2KHR(commandBuffer, pPushDescriptorSetInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPushDescriptorSet2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPushDescriptorSet2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPushDescriptorSet2KHR(commandBuffer, pPushDescriptorSetInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPushDescriptorSet2KHR(commandBuffer, pPushDescriptorSetInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPushDescriptorSet2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPushDescriptorSet2KHR(commandBuffer, pPushDescriptorSetInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSetWithTemplate2KHR(
    VkCommandBuffer commandBuffer, const VkPushDescriptorSetWithTemplateInfoKHR* pPushDescriptorSetWithTemplateInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdPushDescriptorSetWithTemplate2KHR,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdPushDescriptorSetWithTemplate2KHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdPushDescriptorSetWithTemplate2KHR(commandBuffer,
                                                                                   pPushDescriptorSetWithTemplateInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdPushDescriptorSetWithTemplate2KHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdPushDescriptorSetWithTemplate2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdPushDescriptorSetWithTemplate2KHR(commandBuffer, pPushDescriptorSetWithTemplateInfo,
                                                                         record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdPushDescriptorSetWithTemplate2KHR(commandBuffer, pPushDescriptorSetWithTemplateInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdPushDescriptorSetWithTemplate2KHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdPushDescriptorSetWithTemplate2KHR(commandBuffer, pPushDescriptorSetWithTemplateInfo,
                                                                          record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDescriptorBufferOffsets2EXT(
    VkCommandBuffer commandBuffer, const VkSetDescriptorBufferOffsetsInfoEXT* pSetDescriptorBufferOffsetsInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDescriptorBufferOffsets2EXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDescriptorBufferOffsets2EXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDescriptorBufferOffsets2EXT(commandBuffer, pSetDescriptorBufferOffsetsInfo,
                                                                                error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDescriptorBufferOffsets2EXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDescriptorBufferOffsets2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDescriptorBufferOffsets2EXT(commandBuffer, pSetDescriptorBufferOffsetsInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDescriptorBufferOffsets2EXT(commandBuffer, pSetDescriptorBufferOffsetsInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDescriptorBufferOffsets2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDescriptorBufferOffsets2EXT(commandBuffer, pSetDescriptorBufferOffsetsInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindDescriptorBufferEmbeddedSamplers2EXT(
    VkCommandBuffer commandBuffer, const VkBindDescriptorBufferEmbeddedSamplersInfoEXT* pBindDescriptorBufferEmbeddedSamplersInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindDescriptorBufferEmbeddedSamplers2EXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindDescriptorBufferEmbeddedSamplers2EXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindDescriptorBufferEmbeddedSamplers2EXT(
                commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindDescriptorBufferEmbeddedSamplers2EXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindDescriptorBufferEmbeddedSamplers2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindDescriptorBufferEmbeddedSamplers2EXT(
                commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindDescriptorBufferEmbeddedSamplers2EXT(commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindDescriptorBufferEmbeddedSamplers2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindDescriptorBufferEmbeddedSamplers2EXT(
                commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDebugReportCallbackEXT(VkInstance instance,
                                                            const VkDebugReportCallbackCreateInfoEXT* pCreateInfo,
                                                            const VkAllocationCallbacks* pAllocator,
                                                            VkDebugReportCallbackEXT* pCallback) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDebugReportCallbackEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateDebugReportCallbackEXT(instance, pCreateInfo, pAllocator, pCallback, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDebugReportCallbackEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDebugReportCallbackEXT(instance, pCreateInfo, pAllocator, pCallback, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDebugReportCallbackEXT(instance, pCreateInfo, pAllocator, pCallback);
    }
    LayerCreateReportCallback(layer_data->debug_report, false, pCreateInfo, pCallback);
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDebugReportCallbackEXT(instance, pCreateInfo, pAllocator, pCallback, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDebugReportCallbackEXT(VkInstance instance, VkDebugReportCallbackEXT callback,
                                                         const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDebugReportCallbackEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyDebugReportCallbackEXT(instance, callback, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDebugReportCallbackEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDebugReportCallbackEXT(instance, callback, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDebugReportCallbackEXT(instance, callback, pAllocator);
    }
    LayerDestroyCallback(layer_data->debug_report, callback);
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDebugReportCallbackEXT(instance, callback, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL DebugReportMessageEXT(VkInstance instance, VkDebugReportFlagsEXT flags,
                                                 VkDebugReportObjectTypeEXT objectType, uint64_t object, size_t location,
                                                 int32_t messageCode, const char* pLayerPrefix, const char* pMessage) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDebugReportMessageEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDebugReportMessageEXT(instance, flags, objectType, object, location, messageCode,
                                                                    pLayerPrefix, pMessage, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDebugReportMessageEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDebugReportMessageEXT(instance, flags, objectType, object, location, messageCode, pLayerPrefix,
                                                          pMessage, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDebugReportMessageEXT(instance, flags, objectType, object, location, messageCode, pLayerPrefix, pMessage);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDebugReportMessageEXT(instance, flags, objectType, object, location, messageCode, pLayerPrefix,
                                                           pMessage, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL DebugMarkerSetObjectTagEXT(VkDevice device, const VkDebugMarkerObjectTagInfoEXT* pTagInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDebugMarkerSetObjectTagEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDebugMarkerSetObjectTagEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDebugMarkerSetObjectTagEXT(device, pTagInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkDebugMarkerSetObjectTagEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDebugMarkerSetObjectTagEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDebugMarkerSetObjectTagEXT(device, pTagInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchDebugMarkerSetObjectTagEXT(device, pTagInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDebugMarkerSetObjectTagEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDebugMarkerSetObjectTagEXT(device, pTagInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DebugMarkerSetObjectNameEXT(VkDevice device, const VkDebugMarkerObjectNameInfoEXT* pNameInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDebugMarkerSetObjectNameEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDebugMarkerSetObjectNameEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDebugMarkerSetObjectNameEXT(device, pNameInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkDebugMarkerSetObjectNameEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDebugMarkerSetObjectNameEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDebugMarkerSetObjectNameEXT(device, pNameInfo, record_obj);
        }
    }
    layer_data->debug_report->SetMarkerObjectName(pNameInfo);
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchDebugMarkerSetObjectNameEXT(device, pNameInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDebugMarkerSetObjectNameEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDebugMarkerSetObjectNameEXT(device, pNameInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdDebugMarkerBeginEXT(VkCommandBuffer commandBuffer, const VkDebugMarkerMarkerInfoEXT* pMarkerInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDebugMarkerBeginEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDebugMarkerBeginEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDebugMarkerBeginEXT(commandBuffer, pMarkerInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDebugMarkerBeginEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDebugMarkerBeginEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDebugMarkerBeginEXT(commandBuffer, pMarkerInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDebugMarkerBeginEXT(commandBuffer, pMarkerInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDebugMarkerBeginEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDebugMarkerBeginEXT(commandBuffer, pMarkerInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDebugMarkerEndEXT(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDebugMarkerEndEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDebugMarkerEndEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDebugMarkerEndEXT(commandBuffer, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDebugMarkerEndEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDebugMarkerEndEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDebugMarkerEndEXT(commandBuffer, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDebugMarkerEndEXT(commandBuffer);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDebugMarkerEndEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDebugMarkerEndEXT(commandBuffer, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDebugMarkerInsertEXT(VkCommandBuffer commandBuffer, const VkDebugMarkerMarkerInfoEXT* pMarkerInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDebugMarkerInsertEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDebugMarkerInsertEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDebugMarkerInsertEXT(commandBuffer, pMarkerInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDebugMarkerInsertEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDebugMarkerInsertEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDebugMarkerInsertEXT(commandBuffer, pMarkerInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDebugMarkerInsertEXT(commandBuffer, pMarkerInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDebugMarkerInsertEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDebugMarkerInsertEXT(commandBuffer, pMarkerInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindTransformFeedbackBuffersEXT(VkCommandBuffer commandBuffer, uint32_t firstBinding,
                                                              uint32_t bindingCount, const VkBuffer* pBuffers,
                                                              const VkDeviceSize* pOffsets, const VkDeviceSize* pSizes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindTransformFeedbackBuffersEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindTransformFeedbackBuffersEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindTransformFeedbackBuffersEXT(commandBuffer, firstBinding, bindingCount,
                                                                                 pBuffers, pOffsets, pSizes, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindTransformFeedbackBuffersEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindTransformFeedbackBuffersEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindTransformFeedbackBuffersEXT(commandBuffer, firstBinding, bindingCount, pBuffers,
                                                                       pOffsets, pSizes, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindTransformFeedbackBuffersEXT(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindTransformFeedbackBuffersEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindTransformFeedbackBuffersEXT(commandBuffer, firstBinding, bindingCount, pBuffers,
                                                                        pOffsets, pSizes, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginTransformFeedbackEXT(VkCommandBuffer commandBuffer, uint32_t firstCounterBuffer,
                                                        uint32_t counterBufferCount, const VkBuffer* pCounterBuffers,
                                                        const VkDeviceSize* pCounterBufferOffsets) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginTransformFeedbackEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginTransformFeedbackEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount,
                                                                           pCounterBuffers, pCounterBufferOffsets, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginTransformFeedbackEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginTransformFeedbackEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount,
                                                                 pCounterBuffers, pCounterBufferOffsets, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers,
                                             pCounterBufferOffsets);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginTransformFeedbackEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount,
                                                                  pCounterBuffers, pCounterBufferOffsets, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndTransformFeedbackEXT(VkCommandBuffer commandBuffer, uint32_t firstCounterBuffer,
                                                      uint32_t counterBufferCount, const VkBuffer* pCounterBuffers,
                                                      const VkDeviceSize* pCounterBufferOffsets) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndTransformFeedbackEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndTransformFeedbackEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount,
                                                                         pCounterBuffers, pCounterBufferOffsets, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndTransformFeedbackEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndTransformFeedbackEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount,
                                                               pCounterBuffers, pCounterBufferOffsets, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers,
                                           pCounterBufferOffsets);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndTransformFeedbackEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount,
                                                                pCounterBuffers, pCounterBufferOffsets, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginQueryIndexedEXT(VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query,
                                                   VkQueryControlFlags flags, uint32_t index) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginQueryIndexedEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginQueryIndexedEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginQueryIndexedEXT(commandBuffer, queryPool, query, flags, index, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginQueryIndexedEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginQueryIndexedEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginQueryIndexedEXT(commandBuffer, queryPool, query, flags, index, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginQueryIndexedEXT(commandBuffer, queryPool, query, flags, index);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginQueryIndexedEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginQueryIndexedEXT(commandBuffer, queryPool, query, flags, index, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndQueryIndexedEXT(VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query,
                                                 uint32_t index) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndQueryIndexedEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndQueryIndexedEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndQueryIndexedEXT(commandBuffer, queryPool, query, index, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndQueryIndexedEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndQueryIndexedEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndQueryIndexedEXT(commandBuffer, queryPool, query, index, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndQueryIndexedEXT(commandBuffer, queryPool, query, index);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndQueryIndexedEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndQueryIndexedEXT(commandBuffer, queryPool, query, index, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectByteCountEXT(VkCommandBuffer commandBuffer, uint32_t instanceCount,
                                                       uint32_t firstInstance, VkBuffer counterBuffer,
                                                       VkDeviceSize counterBufferOffset, uint32_t counterOffset,
                                                       uint32_t vertexStride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndirectByteCountEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndirectByteCountEXT]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCmdDrawIndirectByteCountEXT(commandBuffer, instanceCount, firstInstance, counterBuffer,
                                                                      counterBufferOffset, counterOffset, vertexStride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndirectByteCountEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndirectByteCountEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndirectByteCountEXT(commandBuffer, instanceCount, firstInstance, counterBuffer,
                                                                counterBufferOffset, counterOffset, vertexStride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndirectByteCountEXT(commandBuffer, instanceCount, firstInstance, counterBuffer, counterBufferOffset,
                                            counterOffset, vertexStride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndirectByteCountEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndirectByteCountEXT(commandBuffer, instanceCount, firstInstance, counterBuffer,
                                                                 counterBufferOffset, counterOffset, vertexStride, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateCuModuleNVX(VkDevice device, const VkCuModuleCreateInfoNVX* pCreateInfo,
                                                 const VkAllocationCallbacks* pAllocator, VkCuModuleNVX* pModule) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateCuModuleNVX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateCuModuleNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateCuModuleNVX(device, pCreateInfo, pAllocator, pModule, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateCuModuleNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateCuModuleNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateCuModuleNVX(device, pCreateInfo, pAllocator, pModule, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateCuModuleNVX(device, pCreateInfo, pAllocator, pModule);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateCuModuleNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateCuModuleNVX(device, pCreateInfo, pAllocator, pModule, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateCuFunctionNVX(VkDevice device, const VkCuFunctionCreateInfoNVX* pCreateInfo,
                                                   const VkAllocationCallbacks* pAllocator, VkCuFunctionNVX* pFunction) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateCuFunctionNVX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCreateCuFunctionNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateCuFunctionNVX(device, pCreateInfo, pAllocator, pFunction, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateCuFunctionNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateCuFunctionNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateCuFunctionNVX(device, pCreateInfo, pAllocator, pFunction, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateCuFunctionNVX(device, pCreateInfo, pAllocator, pFunction);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateCuFunctionNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateCuFunctionNVX(device, pCreateInfo, pAllocator, pFunction, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyCuModuleNVX(VkDevice device, VkCuModuleNVX module, const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyCuModuleNVX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyCuModuleNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyCuModuleNVX(device, module, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyCuModuleNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyCuModuleNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyCuModuleNVX(device, module, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyCuModuleNVX(device, module, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyCuModuleNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyCuModuleNVX(device, module, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL DestroyCuFunctionNVX(VkDevice device, VkCuFunctionNVX function,
                                                const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyCuFunctionNVX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyCuFunctionNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyCuFunctionNVX(device, function, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyCuFunctionNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyCuFunctionNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyCuFunctionNVX(device, function, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyCuFunctionNVX(device, function, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyCuFunctionNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyCuFunctionNVX(device, function, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCuLaunchKernelNVX(VkCommandBuffer commandBuffer, const VkCuLaunchInfoNVX* pLaunchInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCuLaunchKernelNVX, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCuLaunchKernelNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCuLaunchKernelNVX(commandBuffer, pLaunchInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCuLaunchKernelNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCuLaunchKernelNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCuLaunchKernelNVX(commandBuffer, pLaunchInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCuLaunchKernelNVX(commandBuffer, pLaunchInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCuLaunchKernelNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCuLaunchKernelNVX(commandBuffer, pLaunchInfo, record_obj);
        }
    }
}

VKAPI_ATTR uint32_t VKAPI_CALL GetImageViewHandleNVX(VkDevice device, const VkImageViewHandleInfoNVX* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageViewHandleNVX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageViewHandleNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageViewHandleNVX(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageViewHandleNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageViewHandleNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageViewHandleNVX(device, pInfo, record_obj);
        }
    }
    uint32_t result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetImageViewHandleNVX(device, pInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageViewHandleNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageViewHandleNVX(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetImageViewAddressNVX(VkDevice device, VkImageView imageView,
                                                      VkImageViewAddressPropertiesNVX* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageViewAddressNVX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageViewAddressNVX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageViewAddressNVX(device, imageView, pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageViewAddressNVX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageViewAddressNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageViewAddressNVX(device, imageView, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetImageViewAddressNVX(device, imageView, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageViewAddressNVX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageViewAddressNVX(device, imageView, pProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectCountAMD(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                   VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount,
                                                   uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndirectCountAMD, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndirectCountAMD]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                      maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndirectCountAMD);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndirectCountAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                            maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndirectCountAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                             maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirectCountAMD(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                          VkBuffer countBuffer, VkDeviceSize countBufferOffset,
                                                          uint32_t maxDrawCount, uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawIndexedIndirectCountAMD,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawIndexedIndirectCountAMD]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawIndexedIndirectCountAMD(commandBuffer, buffer, offset, countBuffer,
                                                                             countBufferOffset, maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawIndexedIndirectCountAMD);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawIndexedIndirectCountAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawIndexedIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                   maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawIndexedIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawIndexedIndirectCountAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawIndexedIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                    maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetShaderInfoAMD(VkDevice device, VkPipeline pipeline, VkShaderStageFlagBits shaderStage,
                                                VkShaderInfoTypeAMD infoType, size_t* pInfoSize, void* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetShaderInfoAMD, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetShaderInfoAMD]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateGetShaderInfoAMD(device, pipeline, shaderStage, infoType, pInfoSize, pInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetShaderInfoAMD);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetShaderInfoAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetShaderInfoAMD(device, pipeline, shaderStage, infoType, pInfoSize, pInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetShaderInfoAMD(device, pipeline, shaderStage, infoType, pInfoSize, pInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetShaderInfoAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetShaderInfoAMD(device, pipeline, shaderStage, infoType, pInfoSize, pInfo, record_obj);
        }
    }
    return result;
}

#ifdef VK_USE_PLATFORM_GGP
VKAPI_ATTR VkResult VKAPI_CALL CreateStreamDescriptorSurfaceGGP(VkInstance instance,
                                                                const VkStreamDescriptorSurfaceCreateInfoGGP* pCreateInfo,
                                                                const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateStreamDescriptorSurfaceGGP, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCreateStreamDescriptorSurfaceGGP(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateStreamDescriptorSurfaceGGP);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateStreamDescriptorSurfaceGGP(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateStreamDescriptorSurfaceGGP(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateStreamDescriptorSurfaceGGP(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_GGP
VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceExternalImageFormatPropertiesNV(
    VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkImageTiling tiling, VkImageUsageFlags usage,
    VkImageCreateFlags flags, VkExternalMemoryHandleTypeFlagsNV externalHandleType,
    VkExternalImageFormatPropertiesNV* pExternalImageFormatProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceExternalImageFormatPropertiesNV,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceExternalImageFormatPropertiesNV(
                physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceExternalImageFormatPropertiesNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceExternalImageFormatPropertiesNV(
                physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceExternalImageFormatPropertiesNV(physicalDevice, format, type, tiling, usage, flags,
                                                                          externalHandleType, pExternalImageFormatProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceExternalImageFormatPropertiesNV(
                physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties, record_obj);
        }
    }
    return result;
}

#ifdef VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL GetMemoryWin32HandleNV(VkDevice device, VkDeviceMemory memory,
                                                      VkExternalMemoryHandleTypeFlagsNV handleType, HANDLE* pHandle) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryWin32HandleNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryWin32HandleNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryWin32HandleNV(device, memory, handleType, pHandle, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryWin32HandleNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryWin32HandleNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryWin32HandleNV(device, memory, handleType, pHandle, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryWin32HandleNV(device, memory, handleType, pHandle);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryWin32HandleNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryWin32HandleNV(device, memory, handleType, pHandle, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WIN32_KHR
#ifdef VK_USE_PLATFORM_VI_NN
VKAPI_ATTR VkResult VKAPI_CALL CreateViSurfaceNN(VkInstance instance, const VkViSurfaceCreateInfoNN* pCreateInfo,
                                                 const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateViSurfaceNN, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateViSurfaceNN(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateViSurfaceNN);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateViSurfaceNN(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateViSurfaceNN(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateViSurfaceNN(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_VI_NN
VKAPI_ATTR void VKAPI_CALL CmdBeginConditionalRenderingEXT(VkCommandBuffer commandBuffer,
                                                           const VkConditionalRenderingBeginInfoEXT* pConditionalRenderingBegin) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginConditionalRenderingEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginConditionalRenderingEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginConditionalRenderingEXT(commandBuffer, pConditionalRenderingBegin, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginConditionalRenderingEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginConditionalRenderingEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginConditionalRenderingEXT(commandBuffer, pConditionalRenderingBegin, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginConditionalRenderingEXT(commandBuffer, pConditionalRenderingBegin);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginConditionalRenderingEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginConditionalRenderingEXT(commandBuffer, pConditionalRenderingBegin, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndConditionalRenderingEXT(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndConditionalRenderingEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndConditionalRenderingEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndConditionalRenderingEXT(commandBuffer, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndConditionalRenderingEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndConditionalRenderingEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndConditionalRenderingEXT(commandBuffer, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndConditionalRenderingEXT(commandBuffer);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndConditionalRenderingEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndConditionalRenderingEXT(commandBuffer, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportWScalingNV(VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount,
                                                    const VkViewportWScalingNV* pViewportWScalings) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetViewportWScalingNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetViewportWScalingNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetViewportWScalingNV(commandBuffer, firstViewport, viewportCount,
                                                                       pViewportWScalings, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetViewportWScalingNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetViewportWScalingNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetViewportWScalingNV(commandBuffer, firstViewport, viewportCount, pViewportWScalings,
                                                             record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetViewportWScalingNV(commandBuffer, firstViewport, viewportCount, pViewportWScalings);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetViewportWScalingNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetViewportWScalingNV(commandBuffer, firstViewport, viewportCount, pViewportWScalings,
                                                              record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL ReleaseDisplayEXT(VkPhysicalDevice physicalDevice, VkDisplayKHR display) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkReleaseDisplayEXT, VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateReleaseDisplayEXT(physicalDevice, display, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkReleaseDisplayEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordReleaseDisplayEXT(physicalDevice, display, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchReleaseDisplayEXT(physicalDevice, display);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordReleaseDisplayEXT(physicalDevice, display, record_obj);
        }
    }
    return result;
}

#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
VKAPI_ATTR VkResult VKAPI_CALL AcquireXlibDisplayEXT(VkPhysicalDevice physicalDevice, Display* dpy, VkDisplayKHR display) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAcquireXlibDisplayEXT, VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAcquireXlibDisplayEXT(physicalDevice, dpy, display, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAcquireXlibDisplayEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAcquireXlibDisplayEXT(physicalDevice, dpy, display, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAcquireXlibDisplayEXT(physicalDevice, dpy, display);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAcquireXlibDisplayEXT(physicalDevice, dpy, display, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetRandROutputDisplayEXT(VkPhysicalDevice physicalDevice, Display* dpy, RROutput rrOutput,
                                                        VkDisplayKHR* pDisplay) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetRandROutputDisplayEXT,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetRandROutputDisplayEXT(physicalDevice, dpy, rrOutput, pDisplay, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetRandROutputDisplayEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetRandROutputDisplayEXT(physicalDevice, dpy, rrOutput, pDisplay, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetRandROutputDisplayEXT(physicalDevice, dpy, rrOutput, pDisplay);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetRandROutputDisplayEXT(physicalDevice, dpy, rrOutput, pDisplay, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_XLIB_XRANDR_EXT
VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceCapabilities2EXT(VkPhysicalDevice physicalDevice, VkSurfaceKHR surface,
                                                                        VkSurfaceCapabilities2EXT* pSurfaceCapabilities) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfaceCapabilities2EXT,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfaceCapabilities2EXT(physicalDevice, surface,
                                                                                       pSurfaceCapabilities, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfaceCapabilities2EXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfaceCapabilities2EXT(physicalDevice, surface, pSurfaceCapabilities,
                                                                             record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfaceCapabilities2EXT(physicalDevice, surface, pSurfaceCapabilities);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfaceCapabilities2EXT(physicalDevice, surface, pSurfaceCapabilities,
                                                                              record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DisplayPowerControlEXT(VkDevice device, VkDisplayKHR display,
                                                      const VkDisplayPowerInfoEXT* pDisplayPowerInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDisplayPowerControlEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateDisplayPowerControlEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDisplayPowerControlEXT(device, display, pDisplayPowerInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkDisplayPowerControlEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDisplayPowerControlEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDisplayPowerControlEXT(device, display, pDisplayPowerInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchDisplayPowerControlEXT(device, display, pDisplayPowerInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDisplayPowerControlEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDisplayPowerControlEXT(device, display, pDisplayPowerInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL RegisterDeviceEventEXT(VkDevice device, const VkDeviceEventInfoEXT* pDeviceEventInfo,
                                                      const VkAllocationCallbacks* pAllocator, VkFence* pFence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkRegisterDeviceEventEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateRegisterDeviceEventEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateRegisterDeviceEventEXT(device, pDeviceEventInfo, pAllocator, pFence, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkRegisterDeviceEventEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordRegisterDeviceEventEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordRegisterDeviceEventEXT(device, pDeviceEventInfo, pAllocator, pFence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchRegisterDeviceEventEXT(device, pDeviceEventInfo, pAllocator, pFence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordRegisterDeviceEventEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordRegisterDeviceEventEXT(device, pDeviceEventInfo, pAllocator, pFence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL RegisterDisplayEventEXT(VkDevice device, VkDisplayKHR display,
                                                       const VkDisplayEventInfoEXT* pDisplayEventInfo,
                                                       const VkAllocationCallbacks* pAllocator, VkFence* pFence) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkRegisterDisplayEventEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateRegisterDisplayEventEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateRegisterDisplayEventEXT(device, display, pDisplayEventInfo, pAllocator, pFence,
                                                                      error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkRegisterDisplayEventEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordRegisterDisplayEventEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordRegisterDisplayEventEXT(device, display, pDisplayEventInfo, pAllocator, pFence, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchRegisterDisplayEventEXT(device, display, pDisplayEventInfo, pAllocator, pFence);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordRegisterDisplayEventEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordRegisterDisplayEventEXT(device, display, pDisplayEventInfo, pAllocator, pFence, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSwapchainCounterEXT(VkDevice device, VkSwapchainKHR swapchain,
                                                      VkSurfaceCounterFlagBitsEXT counter, uint64_t* pCounterValue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetSwapchainCounterEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateGetSwapchainCounterEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetSwapchainCounterEXT(device, swapchain, counter, pCounterValue, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetSwapchainCounterEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetSwapchainCounterEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetSwapchainCounterEXT(device, swapchain, counter, pCounterValue, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetSwapchainCounterEXT(device, swapchain, counter, pCounterValue);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetSwapchainCounterEXT]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetSwapchainCounterEXT(device, swapchain, counter, pCounterValue, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetRefreshCycleDurationGOOGLE(VkDevice device, VkSwapchainKHR swapchain,
                                                             VkRefreshCycleDurationGOOGLE* pDisplayTimingProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetRefreshCycleDurationGOOGLE, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetRefreshCycleDurationGOOGLE]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetRefreshCycleDurationGOOGLE(device, swapchain, pDisplayTimingProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetRefreshCycleDurationGOOGLE);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetRefreshCycleDurationGOOGLE]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetRefreshCycleDurationGOOGLE(device, swapchain, pDisplayTimingProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetRefreshCycleDurationGOOGLE(device, swapchain, pDisplayTimingProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetRefreshCycleDurationGOOGLE]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetRefreshCycleDurationGOOGLE(device, swapchain, pDisplayTimingProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPastPresentationTimingGOOGLE(VkDevice device, VkSwapchainKHR swapchain,
                                                               uint32_t* pPresentationTimingCount,
                                                               VkPastPresentationTimingGOOGLE* pPresentationTimings) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPastPresentationTimingGOOGLE, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetPastPresentationTimingGOOGLE]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPastPresentationTimingGOOGLE(device, swapchain, pPresentationTimingCount,
                                                                              pPresentationTimings, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPastPresentationTimingGOOGLE);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetPastPresentationTimingGOOGLE]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPastPresentationTimingGOOGLE(device, swapchain, pPresentationTimingCount,
                                                                    pPresentationTimings, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPastPresentationTimingGOOGLE(device, swapchain, pPresentationTimingCount, pPresentationTimings);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetPastPresentationTimingGOOGLE]) {
            auto lock = intercept->WriteLock();

            if (result == VK_ERROR_DEVICE_LOST) {
                intercept->is_device_lost = true;
            }
            intercept->PostCallRecordGetPastPresentationTimingGOOGLE(device, swapchain, pPresentationTimingCount,
                                                                     pPresentationTimings, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetDiscardRectangleEXT(VkCommandBuffer commandBuffer, uint32_t firstDiscardRectangle,
                                                     uint32_t discardRectangleCount, const VkRect2D* pDiscardRectangles) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDiscardRectangleEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDiscardRectangleEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDiscardRectangleEXT(commandBuffer, firstDiscardRectangle, discardRectangleCount,
                                                                        pDiscardRectangles, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDiscardRectangleEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDiscardRectangleEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDiscardRectangleEXT(commandBuffer, firstDiscardRectangle, discardRectangleCount,
                                                              pDiscardRectangles, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDiscardRectangleEXT(commandBuffer, firstDiscardRectangle, discardRectangleCount, pDiscardRectangles);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDiscardRectangleEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDiscardRectangleEXT(commandBuffer, firstDiscardRectangle, discardRectangleCount,
                                                               pDiscardRectangles, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDiscardRectangleEnableEXT(VkCommandBuffer commandBuffer, VkBool32 discardRectangleEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDiscardRectangleEnableEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDiscardRectangleEnableEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDiscardRectangleEnableEXT(commandBuffer, discardRectangleEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDiscardRectangleEnableEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDiscardRectangleEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDiscardRectangleEnableEXT(commandBuffer, discardRectangleEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDiscardRectangleEnableEXT(commandBuffer, discardRectangleEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDiscardRectangleEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDiscardRectangleEnableEXT(commandBuffer, discardRectangleEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDiscardRectangleModeEXT(VkCommandBuffer commandBuffer,
                                                         VkDiscardRectangleModeEXT discardRectangleMode) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDiscardRectangleModeEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDiscardRectangleModeEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDiscardRectangleModeEXT(commandBuffer, discardRectangleMode, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDiscardRectangleModeEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDiscardRectangleModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDiscardRectangleModeEXT(commandBuffer, discardRectangleMode, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDiscardRectangleModeEXT(commandBuffer, discardRectangleMode);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDiscardRectangleModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDiscardRectangleModeEXT(commandBuffer, discardRectangleMode, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL SetHdrMetadataEXT(VkDevice device, uint32_t swapchainCount, const VkSwapchainKHR* pSwapchains,
                                             const VkHdrMetadataEXT* pMetadata) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSetHdrMetadataEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateSetHdrMetadataEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSetHdrMetadataEXT(device, swapchainCount, pSwapchains, pMetadata, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkSetHdrMetadataEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSetHdrMetadataEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSetHdrMetadataEXT(device, swapchainCount, pSwapchains, pMetadata, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchSetHdrMetadataEXT(device, swapchainCount, pSwapchains, pMetadata);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSetHdrMetadataEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSetHdrMetadataEXT(device, swapchainCount, pSwapchains, pMetadata, record_obj);
        }
    }
}

#ifdef VK_USE_PLATFORM_IOS_MVK
VKAPI_ATTR VkResult VKAPI_CALL CreateIOSSurfaceMVK(VkInstance instance, const VkIOSSurfaceCreateInfoMVK* pCreateInfo,
                                                   const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateIOSSurfaceMVK, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateIOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateIOSSurfaceMVK);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateIOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateIOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateIOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_IOS_MVK
#ifdef VK_USE_PLATFORM_MACOS_MVK
VKAPI_ATTR VkResult VKAPI_CALL CreateMacOSSurfaceMVK(VkInstance instance, const VkMacOSSurfaceCreateInfoMVK* pCreateInfo,
                                                     const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateMacOSSurfaceMVK, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateMacOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateMacOSSurfaceMVK);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateMacOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateMacOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateMacOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_MACOS_MVK
VKAPI_ATTR VkResult VKAPI_CALL SetDebugUtilsObjectNameEXT(VkDevice device, const VkDebugUtilsObjectNameInfoEXT* pNameInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSetDebugUtilsObjectNameEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateSetDebugUtilsObjectNameEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSetDebugUtilsObjectNameEXT(device, pNameInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkSetDebugUtilsObjectNameEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSetDebugUtilsObjectNameEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSetDebugUtilsObjectNameEXT(device, pNameInfo, record_obj);
        }
    }
    layer_data->debug_report->SetUtilsObjectName(pNameInfo);
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchSetDebugUtilsObjectNameEXT(device, pNameInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSetDebugUtilsObjectNameEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSetDebugUtilsObjectNameEXT(device, pNameInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SetDebugUtilsObjectTagEXT(VkDevice device, const VkDebugUtilsObjectTagInfoEXT* pTagInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSetDebugUtilsObjectTagEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateSetDebugUtilsObjectTagEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSetDebugUtilsObjectTagEXT(device, pTagInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkSetDebugUtilsObjectTagEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSetDebugUtilsObjectTagEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSetDebugUtilsObjectTagEXT(device, pTagInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchSetDebugUtilsObjectTagEXT(device, pTagInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSetDebugUtilsObjectTagEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSetDebugUtilsObjectTagEXT(device, pTagInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL QueueBeginDebugUtilsLabelEXT(VkQueue queue, const VkDebugUtilsLabelEXT* pLabelInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueBeginDebugUtilsLabelEXT, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateQueueBeginDebugUtilsLabelEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueBeginDebugUtilsLabelEXT(queue, pLabelInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueBeginDebugUtilsLabelEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueBeginDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueBeginDebugUtilsLabelEXT(queue, pLabelInfo, record_obj);
        }
    }
    layer_data->debug_report->BeginQueueDebugUtilsLabel(queue, pLabelInfo);
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchQueueBeginDebugUtilsLabelEXT(queue, pLabelInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueBeginDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordQueueBeginDebugUtilsLabelEXT(queue, pLabelInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL QueueEndDebugUtilsLabelEXT(VkQueue queue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueEndDebugUtilsLabelEXT, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateQueueEndDebugUtilsLabelEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueEndDebugUtilsLabelEXT(queue, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueEndDebugUtilsLabelEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueEndDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueEndDebugUtilsLabelEXT(queue, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchQueueEndDebugUtilsLabelEXT(queue);
    }
    layer_data->debug_report->EndQueueDebugUtilsLabel(queue);
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueEndDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordQueueEndDebugUtilsLabelEXT(queue, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL QueueInsertDebugUtilsLabelEXT(VkQueue queue, const VkDebugUtilsLabelEXT* pLabelInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueInsertDebugUtilsLabelEXT, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateQueueInsertDebugUtilsLabelEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueInsertDebugUtilsLabelEXT(queue, pLabelInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueInsertDebugUtilsLabelEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordQueueInsertDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueInsertDebugUtilsLabelEXT(queue, pLabelInfo, record_obj);
        }
    }
    layer_data->debug_report->InsertQueueDebugUtilsLabel(queue, pLabelInfo);
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchQueueInsertDebugUtilsLabelEXT(queue, pLabelInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordQueueInsertDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordQueueInsertDebugUtilsLabelEXT(queue, pLabelInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBeginDebugUtilsLabelEXT(VkCommandBuffer commandBuffer, const VkDebugUtilsLabelEXT* pLabelInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBeginDebugUtilsLabelEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBeginDebugUtilsLabelEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBeginDebugUtilsLabelEXT(commandBuffer, pLabelInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBeginDebugUtilsLabelEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBeginDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBeginDebugUtilsLabelEXT(commandBuffer, pLabelInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBeginDebugUtilsLabelEXT(commandBuffer, pLabelInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBeginDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBeginDebugUtilsLabelEXT(commandBuffer, pLabelInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdEndDebugUtilsLabelEXT(VkCommandBuffer commandBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdEndDebugUtilsLabelEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdEndDebugUtilsLabelEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdEndDebugUtilsLabelEXT(commandBuffer, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdEndDebugUtilsLabelEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdEndDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdEndDebugUtilsLabelEXT(commandBuffer, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdEndDebugUtilsLabelEXT(commandBuffer);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdEndDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdEndDebugUtilsLabelEXT(commandBuffer, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdInsertDebugUtilsLabelEXT(VkCommandBuffer commandBuffer, const VkDebugUtilsLabelEXT* pLabelInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdInsertDebugUtilsLabelEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdInsertDebugUtilsLabelEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdInsertDebugUtilsLabelEXT(commandBuffer, pLabelInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdInsertDebugUtilsLabelEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdInsertDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdInsertDebugUtilsLabelEXT(commandBuffer, pLabelInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdInsertDebugUtilsLabelEXT(commandBuffer, pLabelInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdInsertDebugUtilsLabelEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdInsertDebugUtilsLabelEXT(commandBuffer, pLabelInfo, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDebugUtilsMessengerEXT(VkInstance instance,
                                                            const VkDebugUtilsMessengerCreateInfoEXT* pCreateInfo,
                                                            const VkAllocationCallbacks* pAllocator,
                                                            VkDebugUtilsMessengerEXT* pMessenger) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateDebugUtilsMessengerEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCreateDebugUtilsMessengerEXT(instance, pCreateInfo, pAllocator, pMessenger, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateDebugUtilsMessengerEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateDebugUtilsMessengerEXT(instance, pCreateInfo, pAllocator, pMessenger, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateDebugUtilsMessengerEXT(instance, pCreateInfo, pAllocator, pMessenger);
    }
    LayerCreateMessengerCallback(layer_data->debug_report, false, pCreateInfo, pMessenger);
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateDebugUtilsMessengerEXT(instance, pCreateInfo, pAllocator, pMessenger, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDebugUtilsMessengerEXT(VkInstance instance, VkDebugUtilsMessengerEXT messenger,
                                                         const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyDebugUtilsMessengerEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyDebugUtilsMessengerEXT(instance, messenger, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyDebugUtilsMessengerEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyDebugUtilsMessengerEXT(instance, messenger, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyDebugUtilsMessengerEXT(instance, messenger, pAllocator);
    }
    LayerDestroyCallback(layer_data->debug_report, messenger);
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyDebugUtilsMessengerEXT(instance, messenger, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL SubmitDebugUtilsMessageEXT(VkInstance instance, VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity,
                                                      VkDebugUtilsMessageTypeFlagsEXT messageTypes,
                                                      const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSubmitDebugUtilsMessageEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSubmitDebugUtilsMessageEXT(instance, messageSeverity, messageTypes, pCallbackData,
                                                                         error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkSubmitDebugUtilsMessageEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSubmitDebugUtilsMessageEXT(instance, messageSeverity, messageTypes, pCallbackData, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchSubmitDebugUtilsMessageEXT(instance, messageSeverity, messageTypes, pCallbackData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSubmitDebugUtilsMessageEXT(instance, messageSeverity, messageTypes, pCallbackData, record_obj);
        }
    }
}

#ifdef VK_USE_PLATFORM_ANDROID_KHR
VKAPI_ATTR VkResult VKAPI_CALL GetAndroidHardwareBufferPropertiesANDROID(VkDevice device, const struct AHardwareBuffer* buffer,
                                                                         VkAndroidHardwareBufferPropertiesANDROID* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetAndroidHardwareBufferPropertiesANDROID,
                          VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetAndroidHardwareBufferPropertiesANDROID]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetAndroidHardwareBufferPropertiesANDROID(device, buffer, pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetAndroidHardwareBufferPropertiesANDROID);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetAndroidHardwareBufferPropertiesANDROID]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetAndroidHardwareBufferPropertiesANDROID(device, buffer, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetAndroidHardwareBufferPropertiesANDROID(device, buffer, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetAndroidHardwareBufferPropertiesANDROID]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetAndroidHardwareBufferPropertiesANDROID(device, buffer, pProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryAndroidHardwareBufferANDROID(VkDevice device,
                                                                     const VkMemoryGetAndroidHardwareBufferInfoANDROID* pInfo,
                                                                     struct AHardwareBuffer** pBuffer) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryAndroidHardwareBufferANDROID, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryAndroidHardwareBufferANDROID]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryAndroidHardwareBufferANDROID(device, pInfo, pBuffer, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryAndroidHardwareBufferANDROID);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryAndroidHardwareBufferANDROID]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryAndroidHardwareBufferANDROID(device, pInfo, pBuffer, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryAndroidHardwareBufferANDROID(device, pInfo, pBuffer);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryAndroidHardwareBufferANDROID]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryAndroidHardwareBufferANDROID(device, pInfo, pBuffer, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_ANDROID_KHR
#ifdef VK_ENABLE_BETA_EXTENSIONS
VKAPI_ATTR VkResult VKAPI_CALL CreateExecutionGraphPipelinesAMDX(VkDevice device, VkPipelineCache pipelineCache,
                                                                 uint32_t createInfoCount,
                                                                 const VkExecutionGraphPipelineCreateInfoAMDX* pCreateInfos,
                                                                 const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateExecutionGraphPipelinesAMDX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateExecutionGraphPipelinesAMDX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateExecutionGraphPipelinesAMDX(device, pipelineCache, createInfoCount,
                                                                                pCreateInfos, pAllocator, pPipelines, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateExecutionGraphPipelinesAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCreateExecutionGraphPipelinesAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateExecutionGraphPipelinesAMDX(device, pipelineCache, createInfoCount, pCreateInfos,
                                                                      pAllocator, pPipelines, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result =
            DispatchCreateExecutionGraphPipelinesAMDX(device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCreateExecutionGraphPipelinesAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateExecutionGraphPipelinesAMDX(device, pipelineCache, createInfoCount, pCreateInfos,
                                                                       pAllocator, pPipelines, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetExecutionGraphPipelineScratchSizeAMDX(VkDevice device, VkPipeline executionGraph,
                                                                        VkExecutionGraphPipelineScratchSizeAMDX* pSizeInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetExecutionGraphPipelineScratchSizeAMDX,
                          VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetExecutionGraphPipelineScratchSizeAMDX]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateGetExecutionGraphPipelineScratchSizeAMDX(device, executionGraph, pSizeInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetExecutionGraphPipelineScratchSizeAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetExecutionGraphPipelineScratchSizeAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetExecutionGraphPipelineScratchSizeAMDX(device, executionGraph, pSizeInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetExecutionGraphPipelineScratchSizeAMDX(device, executionGraph, pSizeInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetExecutionGraphPipelineScratchSizeAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetExecutionGraphPipelineScratchSizeAMDX(device, executionGraph, pSizeInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetExecutionGraphPipelineNodeIndexAMDX(VkDevice device, VkPipeline executionGraph,
                                                                      const VkPipelineShaderStageNodeCreateInfoAMDX* pNodeInfo,
                                                                      uint32_t* pNodeIndex) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetExecutionGraphPipelineNodeIndexAMDX, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetExecutionGraphPipelineNodeIndexAMDX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetExecutionGraphPipelineNodeIndexAMDX(device, executionGraph, pNodeInfo, pNodeIndex,
                                                                                     error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetExecutionGraphPipelineNodeIndexAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetExecutionGraphPipelineNodeIndexAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetExecutionGraphPipelineNodeIndexAMDX(device, executionGraph, pNodeInfo, pNodeIndex,
                                                                           record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetExecutionGraphPipelineNodeIndexAMDX(device, executionGraph, pNodeInfo, pNodeIndex);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetExecutionGraphPipelineNodeIndexAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetExecutionGraphPipelineNodeIndexAMDX(device, executionGraph, pNodeInfo, pNodeIndex,
                                                                            record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdInitializeGraphScratchMemoryAMDX(VkCommandBuffer commandBuffer, VkDeviceAddress scratch) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdInitializeGraphScratchMemoryAMDX,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdInitializeGraphScratchMemoryAMDX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdInitializeGraphScratchMemoryAMDX(commandBuffer, scratch, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdInitializeGraphScratchMemoryAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdInitializeGraphScratchMemoryAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdInitializeGraphScratchMemoryAMDX(commandBuffer, scratch, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdInitializeGraphScratchMemoryAMDX(commandBuffer, scratch);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdInitializeGraphScratchMemoryAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdInitializeGraphScratchMemoryAMDX(commandBuffer, scratch, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchGraphAMDX(VkCommandBuffer commandBuffer, VkDeviceAddress scratch,
                                                const VkDispatchGraphCountInfoAMDX* pCountInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatchGraphAMDX, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatchGraphAMDX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatchGraphAMDX(commandBuffer, scratch, pCountInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatchGraphAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatchGraphAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatchGraphAMDX(commandBuffer, scratch, pCountInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatchGraphAMDX(commandBuffer, scratch, pCountInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatchGraphAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatchGraphAMDX(commandBuffer, scratch, pCountInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchGraphIndirectAMDX(VkCommandBuffer commandBuffer, VkDeviceAddress scratch,
                                                        const VkDispatchGraphCountInfoAMDX* pCountInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatchGraphIndirectAMDX,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatchGraphIndirectAMDX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatchGraphIndirectAMDX(commandBuffer, scratch, pCountInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatchGraphIndirectAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatchGraphIndirectAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatchGraphIndirectAMDX(commandBuffer, scratch, pCountInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatchGraphIndirectAMDX(commandBuffer, scratch, pCountInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatchGraphIndirectAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatchGraphIndirectAMDX(commandBuffer, scratch, pCountInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchGraphIndirectCountAMDX(VkCommandBuffer commandBuffer, VkDeviceAddress scratch,
                                                             VkDeviceAddress countInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDispatchGraphIndirectCountAMDX,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDispatchGraphIndirectCountAMDX]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDispatchGraphIndirectCountAMDX(commandBuffer, scratch, countInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDispatchGraphIndirectCountAMDX);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDispatchGraphIndirectCountAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDispatchGraphIndirectCountAMDX(commandBuffer, scratch, countInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDispatchGraphIndirectCountAMDX(commandBuffer, scratch, countInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDispatchGraphIndirectCountAMDX]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDispatchGraphIndirectCountAMDX(commandBuffer, scratch, countInfo, record_obj);
        }
    }
}

#endif  // VK_ENABLE_BETA_EXTENSIONS
VKAPI_ATTR void VKAPI_CALL CmdSetSampleLocationsEXT(VkCommandBuffer commandBuffer,
                                                    const VkSampleLocationsInfoEXT* pSampleLocationsInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetSampleLocationsEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetSampleLocationsEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetSampleLocationsEXT(commandBuffer, pSampleLocationsInfo, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetSampleLocationsEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetSampleLocationsEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetSampleLocationsEXT(commandBuffer, pSampleLocationsInfo, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetSampleLocationsEXT(commandBuffer, pSampleLocationsInfo);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetSampleLocationsEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetSampleLocationsEXT(commandBuffer, pSampleLocationsInfo, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMultisamplePropertiesEXT(VkPhysicalDevice physicalDevice, VkSampleCountFlagBits samples,
                                                                     VkMultisamplePropertiesEXT* pMultisampleProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceMultisamplePropertiesEXT,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceMultisamplePropertiesEXT(physicalDevice, samples,
                                                                                        pMultisampleProperties, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceMultisamplePropertiesEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceMultisamplePropertiesEXT(physicalDevice, samples, pMultisampleProperties,
                                                                              record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetPhysicalDeviceMultisamplePropertiesEXT(physicalDevice, samples, pMultisampleProperties);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceMultisamplePropertiesEXT(physicalDevice, samples, pMultisampleProperties,
                                                                               record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetImageDrmFormatModifierPropertiesEXT(VkDevice device, VkImage image,
                                                                      VkImageDrmFormatModifierPropertiesEXT* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetImageDrmFormatModifierPropertiesEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetImageDrmFormatModifierPropertiesEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetImageDrmFormatModifierPropertiesEXT(device, image, pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetImageDrmFormatModifierPropertiesEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetImageDrmFormatModifierPropertiesEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetImageDrmFormatModifierPropertiesEXT(device, image, pProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetImageDrmFormatModifierPropertiesEXT(device, image, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetImageDrmFormatModifierPropertiesEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetImageDrmFormatModifierPropertiesEXT(device, image, pProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindShadingRateImageNV(VkCommandBuffer commandBuffer, VkImageView imageView,
                                                     VkImageLayout imageLayout) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindShadingRateImageNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindShadingRateImageNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindShadingRateImageNV(commandBuffer, imageView, imageLayout, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindShadingRateImageNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindShadingRateImageNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindShadingRateImageNV(commandBuffer, imageView, imageLayout, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindShadingRateImageNV(commandBuffer, imageView, imageLayout);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindShadingRateImageNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindShadingRateImageNV(commandBuffer, imageView, imageLayout, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportShadingRatePaletteNV(VkCommandBuffer commandBuffer, uint32_t firstViewport,
                                                              uint32_t viewportCount,
                                                              const VkShadingRatePaletteNV* pShadingRatePalettes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetViewportShadingRatePaletteNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetViewportShadingRatePaletteNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetViewportShadingRatePaletteNV(commandBuffer, firstViewport, viewportCount,
                                                                                 pShadingRatePalettes, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetViewportShadingRatePaletteNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetViewportShadingRatePaletteNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetViewportShadingRatePaletteNV(commandBuffer, firstViewport, viewportCount,
                                                                       pShadingRatePalettes, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetViewportShadingRatePaletteNV(commandBuffer, firstViewport, viewportCount, pShadingRatePalettes);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetViewportShadingRatePaletteNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetViewportShadingRatePaletteNV(commandBuffer, firstViewport, viewportCount,
                                                                        pShadingRatePalettes, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetCoarseSampleOrderNV(VkCommandBuffer commandBuffer, VkCoarseSampleOrderTypeNV sampleOrderType,
                                                     uint32_t customSampleOrderCount,
                                                     const VkCoarseSampleOrderCustomNV* pCustomSampleOrders) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetCoarseSampleOrderNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetCoarseSampleOrderNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetCoarseSampleOrderNV(commandBuffer, sampleOrderType, customSampleOrderCount,
                                                                        pCustomSampleOrders, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetCoarseSampleOrderNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetCoarseSampleOrderNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetCoarseSampleOrderNV(commandBuffer, sampleOrderType, customSampleOrderCount,
                                                              pCustomSampleOrders, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetCoarseSampleOrderNV(commandBuffer, sampleOrderType, customSampleOrderCount, pCustomSampleOrders);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetCoarseSampleOrderNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetCoarseSampleOrderNV(commandBuffer, sampleOrderType, customSampleOrderCount,
                                                               pCustomSampleOrders, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CreateAccelerationStructureNV(VkDevice device,
                                                             const VkAccelerationStructureCreateInfoNV* pCreateInfo,
                                                             const VkAllocationCallbacks* pAllocator,
                                                             VkAccelerationStructureNV* pAccelerationStructure) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateAccelerationStructureNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCreateAccelerationStructureNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateAccelerationStructureNV(device, pCreateInfo, pAllocator, pAccelerationStructure,
                                                                            error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateAccelerationStructureNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCreateAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateAccelerationStructureNV(device, pCreateInfo, pAllocator, pAccelerationStructure,
                                                                  record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateAccelerationStructureNV(device, pCreateInfo, pAllocator, pAccelerationStructure);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCreateAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateAccelerationStructureNV(device, pCreateInfo, pAllocator, pAccelerationStructure,
                                                                   record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyAccelerationStructureNV(VkDevice device, VkAccelerationStructureNV accelerationStructure,
                                                          const VkAllocationCallbacks* pAllocator) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkDestroyAccelerationStructureNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateDestroyAccelerationStructureNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateDestroyAccelerationStructureNV(device, accelerationStructure, pAllocator, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkDestroyAccelerationStructureNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordDestroyAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordDestroyAccelerationStructureNV(device, accelerationStructure, pAllocator, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchDestroyAccelerationStructureNV(device, accelerationStructure, pAllocator);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordDestroyAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordDestroyAccelerationStructureNV(device, accelerationStructure, pAllocator, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetAccelerationStructureMemoryRequirementsNV(
    VkDevice device, const VkAccelerationStructureMemoryRequirementsInfoNV* pInfo, VkMemoryRequirements2KHR* pMemoryRequirements) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetAccelerationStructureMemoryRequirementsNV,
                          VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetAccelerationStructureMemoryRequirementsNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetAccelerationStructureMemoryRequirementsNV(device, pInfo, pMemoryRequirements,
                                                                                           error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetAccelerationStructureMemoryRequirementsNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetAccelerationStructureMemoryRequirementsNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetAccelerationStructureMemoryRequirementsNV(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetAccelerationStructureMemoryRequirementsNV(device, pInfo, pMemoryRequirements);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetAccelerationStructureMemoryRequirementsNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetAccelerationStructureMemoryRequirementsNV(device, pInfo, pMemoryRequirements, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL BindAccelerationStructureMemoryNV(VkDevice device, uint32_t bindInfoCount,
                                                                 const VkBindAccelerationStructureMemoryInfoNV* pBindInfos) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkBindAccelerationStructureMemoryNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateBindAccelerationStructureMemoryNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateBindAccelerationStructureMemoryNV(device, bindInfoCount, pBindInfos, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkBindAccelerationStructureMemoryNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordBindAccelerationStructureMemoryNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordBindAccelerationStructureMemoryNV(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchBindAccelerationStructureMemoryNV(device, bindInfoCount, pBindInfos);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordBindAccelerationStructureMemoryNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordBindAccelerationStructureMemoryNV(device, bindInfoCount, pBindInfos, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBuildAccelerationStructureNV(VkCommandBuffer commandBuffer,
                                                           const VkAccelerationStructureInfoNV* pInfo, VkBuffer instanceData,
                                                           VkDeviceSize instanceOffset, VkBool32 update,
                                                           VkAccelerationStructureNV dst, VkAccelerationStructureNV src,
                                                           VkBuffer scratch, VkDeviceSize scratchOffset) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBuildAccelerationStructureNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBuildAccelerationStructureNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBuildAccelerationStructureNV(commandBuffer, pInfo, instanceData, instanceOffset,
                                                                              update, dst, src, scratch, scratchOffset, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBuildAccelerationStructureNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBuildAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBuildAccelerationStructureNV(commandBuffer, pInfo, instanceData, instanceOffset, update, dst,
                                                                    src, scratch, scratchOffset, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBuildAccelerationStructureNV(commandBuffer, pInfo, instanceData, instanceOffset, update, dst, src, scratch,
                                                scratchOffset);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBuildAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBuildAccelerationStructureNV(commandBuffer, pInfo, instanceData, instanceOffset, update,
                                                                     dst, src, scratch, scratchOffset, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdCopyAccelerationStructureNV(VkCommandBuffer commandBuffer, VkAccelerationStructureNV dst,
                                                          VkAccelerationStructureNV src, VkCopyAccelerationStructureModeKHR mode) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdCopyAccelerationStructureNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdCopyAccelerationStructureNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdCopyAccelerationStructureNV(commandBuffer, dst, src, mode, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdCopyAccelerationStructureNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdCopyAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdCopyAccelerationStructureNV(commandBuffer, dst, src, mode, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdCopyAccelerationStructureNV(commandBuffer, dst, src, mode);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdCopyAccelerationStructureNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdCopyAccelerationStructureNV(commandBuffer, dst, src, mode, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdTraceRaysNV(VkCommandBuffer commandBuffer, VkBuffer raygenShaderBindingTableBuffer,
                                          VkDeviceSize raygenShaderBindingOffset, VkBuffer missShaderBindingTableBuffer,
                                          VkDeviceSize missShaderBindingOffset, VkDeviceSize missShaderBindingStride,
                                          VkBuffer hitShaderBindingTableBuffer, VkDeviceSize hitShaderBindingOffset,
                                          VkDeviceSize hitShaderBindingStride, VkBuffer callableShaderBindingTableBuffer,
                                          VkDeviceSize callableShaderBindingOffset, VkDeviceSize callableShaderBindingStride,
                                          uint32_t width, uint32_t height, uint32_t depth) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdTraceRaysNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdTraceRaysNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdTraceRaysNV(
                commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset, missShaderBindingTableBuffer,
                missShaderBindingOffset, missShaderBindingStride, hitShaderBindingTableBuffer, hitShaderBindingOffset,
                hitShaderBindingStride, callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride,
                width, height, depth, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdTraceRaysNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdTraceRaysNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdTraceRaysNV(commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset,
                                                   missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride,
                                                   hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride,
                                                   callableShaderBindingTableBuffer, callableShaderBindingOffset,
                                                   callableShaderBindingStride, width, height, depth, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdTraceRaysNV(commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset,
                               missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride,
                               hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride,
                               callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride, width,
                               height, depth);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdTraceRaysNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdTraceRaysNV(commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset,
                                                    missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride,
                                                    hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride,
                                                    callableShaderBindingTableBuffer, callableShaderBindingOffset,
                                                    callableShaderBindingStride, width, height, depth, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetRayTracingShaderGroupHandlesKHR(VkDevice device, VkPipeline pipeline, uint32_t firstGroup,
                                                                  uint32_t groupCount, size_t dataSize, void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetRayTracingShaderGroupHandlesKHR, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetRayTracingShaderGroupHandlesKHR]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetRayTracingShaderGroupHandlesKHR(device, pipeline, firstGroup, groupCount, dataSize,
                                                                                 pData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetRayTracingShaderGroupHandlesKHR);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetRayTracingShaderGroupHandlesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetRayTracingShaderGroupHandlesKHR(device, pipeline, firstGroup, groupCount, dataSize, pData,
                                                                       record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetRayTracingShaderGroupHandlesKHR(device, pipeline, firstGroup, groupCount, dataSize, pData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetRayTracingShaderGroupHandlesKHR]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetRayTracingShaderGroupHandlesKHR(device, pipeline, firstGroup, groupCount, dataSize, pData,
                                                                        record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetRayTracingShaderGroupHandlesNV(VkDevice device, VkPipeline pipeline, uint32_t firstGroup,
                                                                 uint32_t groupCount, size_t dataSize, void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetRayTracingShaderGroupHandlesNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetRayTracingShaderGroupHandlesNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetRayTracingShaderGroupHandlesNV(device, pipeline, firstGroup, groupCount, dataSize,
                                                                                pData, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetRayTracingShaderGroupHandlesNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetRayTracingShaderGroupHandlesNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetRayTracingShaderGroupHandlesNV(device, pipeline, firstGroup, groupCount, dataSize, pData,
                                                                      record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetRayTracingShaderGroupHandlesNV(device, pipeline, firstGroup, groupCount, dataSize, pData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetRayTracingShaderGroupHandlesNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetRayTracingShaderGroupHandlesNV(device, pipeline, firstGroup, groupCount, dataSize, pData,
                                                                       record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetAccelerationStructureHandleNV(VkDevice device, VkAccelerationStructureNV accelerationStructure,
                                                                size_t dataSize, void* pData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetAccelerationStructureHandleNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetAccelerationStructureHandleNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetAccelerationStructureHandleNV(device, accelerationStructure, dataSize, pData,
                                                                               error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetAccelerationStructureHandleNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetAccelerationStructureHandleNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetAccelerationStructureHandleNV(device, accelerationStructure, dataSize, pData, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetAccelerationStructureHandleNV(device, accelerationStructure, dataSize, pData);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetAccelerationStructureHandleNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetAccelerationStructureHandleNV(device, accelerationStructure, dataSize, pData, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdWriteAccelerationStructuresPropertiesNV(VkCommandBuffer commandBuffer,
                                                                      uint32_t accelerationStructureCount,
                                                                      const VkAccelerationStructureNV* pAccelerationStructures,
                                                                      VkQueryType queryType, VkQueryPool queryPool,
                                                                      uint32_t firstQuery) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWriteAccelerationStructuresPropertiesNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWriteAccelerationStructuresPropertiesNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWriteAccelerationStructuresPropertiesNV(
                commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWriteAccelerationStructuresPropertiesNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWriteAccelerationStructuresPropertiesNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWriteAccelerationStructuresPropertiesNV(
                commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWriteAccelerationStructuresPropertiesNV(commandBuffer, accelerationStructureCount, pAccelerationStructures,
                                                           queryType, queryPool, firstQuery);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWriteAccelerationStructuresPropertiesNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWriteAccelerationStructuresPropertiesNV(
                commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CompileDeferredNV(VkDevice device, VkPipeline pipeline, uint32_t shader) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCompileDeferredNV, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCompileDeferredNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCompileDeferredNV(device, pipeline, shader, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCompileDeferredNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCompileDeferredNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCompileDeferredNV(device, pipeline, shader, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCompileDeferredNV(device, pipeline, shader);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCompileDeferredNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCompileDeferredNV(device, pipeline, shader, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryHostPointerPropertiesEXT(VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType,
                                                                 const void* pHostPointer,
                                                                 VkMemoryHostPointerPropertiesEXT* pMemoryHostPointerProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetMemoryHostPointerPropertiesEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetMemoryHostPointerPropertiesEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetMemoryHostPointerPropertiesEXT(device, handleType, pHostPointer,
                                                                                pMemoryHostPointerProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetMemoryHostPointerPropertiesEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetMemoryHostPointerPropertiesEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetMemoryHostPointerPropertiesEXT(device, handleType, pHostPointer,
                                                                      pMemoryHostPointerProperties, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetMemoryHostPointerPropertiesEXT(device, handleType, pHostPointer, pMemoryHostPointerProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetMemoryHostPointerPropertiesEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetMemoryHostPointerPropertiesEXT(device, handleType, pHostPointer,
                                                                       pMemoryHostPointerProperties, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdWriteBufferMarkerAMD(VkCommandBuffer commandBuffer, VkPipelineStageFlagBits pipelineStage,
                                                   VkBuffer dstBuffer, VkDeviceSize dstOffset, uint32_t marker) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdWriteBufferMarkerAMD, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdWriteBufferMarkerAMD]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdWriteBufferMarkerAMD(commandBuffer, pipelineStage, dstBuffer, dstOffset, marker,
                                                                      error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdWriteBufferMarkerAMD);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdWriteBufferMarkerAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdWriteBufferMarkerAMD(commandBuffer, pipelineStage, dstBuffer, dstOffset, marker, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdWriteBufferMarkerAMD(commandBuffer, pipelineStage, dstBuffer, dstOffset, marker);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdWriteBufferMarkerAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdWriteBufferMarkerAMD(commandBuffer, pipelineStage, dstBuffer, dstOffset, marker,
                                                             record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCalibrateableTimeDomainsEXT(VkPhysicalDevice physicalDevice,
                                                                            uint32_t* pTimeDomainCount,
                                                                            VkTimeDomainKHR* pTimeDomains) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceCalibrateableTimeDomainsEXT,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceCalibrateableTimeDomainsEXT(physicalDevice, pTimeDomainCount,
                                                                                           pTimeDomains, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceCalibrateableTimeDomainsEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceCalibrateableTimeDomainsEXT(physicalDevice, pTimeDomainCount, pTimeDomains,
                                                                                 record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceCalibrateableTimeDomainsEXT(physicalDevice, pTimeDomainCount, pTimeDomains);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceCalibrateableTimeDomainsEXT(physicalDevice, pTimeDomainCount, pTimeDomains,
                                                                                  record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetCalibratedTimestampsEXT(VkDevice device, uint32_t timestampCount,
                                                          const VkCalibratedTimestampInfoKHR* pTimestampInfos,
                                                          uint64_t* pTimestamps, uint64_t* pMaxDeviation) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetCalibratedTimestampsEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetCalibratedTimestampsEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetCalibratedTimestampsEXT(device, timestampCount, pTimestampInfos, pTimestamps,
                                                                         pMaxDeviation, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetCalibratedTimestampsEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetCalibratedTimestampsEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetCalibratedTimestampsEXT(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation,
                                                               record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetCalibratedTimestampsEXT(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetCalibratedTimestampsEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetCalibratedTimestampsEXT(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation,
                                                                record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMeshTasksNV(VkCommandBuffer commandBuffer, uint32_t taskCount, uint32_t firstTask) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawMeshTasksNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawMeshTasksNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawMeshTasksNV(commandBuffer, taskCount, firstTask, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawMeshTasksNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawMeshTasksNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawMeshTasksNV(commandBuffer, taskCount, firstTask, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawMeshTasksNV(commandBuffer, taskCount, firstTask);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawMeshTasksNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawMeshTasksNV(commandBuffer, taskCount, firstTask, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMeshTasksIndirectNV(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                      uint32_t drawCount, uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawMeshTasksIndirectNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawMeshTasksIndirectNV]) {
            auto lock = intercept->ReadLock();
            skip |=
                intercept->PreCallValidateCmdDrawMeshTasksIndirectNV(commandBuffer, buffer, offset, drawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawMeshTasksIndirectNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawMeshTasksIndirectNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawMeshTasksIndirectNV(commandBuffer, buffer, offset, drawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawMeshTasksIndirectNV(commandBuffer, buffer, offset, drawCount, stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawMeshTasksIndirectNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawMeshTasksIndirectNV(commandBuffer, buffer, offset, drawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMeshTasksIndirectCountNV(VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset,
                                                           VkBuffer countBuffer, VkDeviceSize countBufferOffset,
                                                           uint32_t maxDrawCount, uint32_t stride) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdDrawMeshTasksIndirectCountNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdDrawMeshTasksIndirectCountNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdDrawMeshTasksIndirectCountNV(commandBuffer, buffer, offset, countBuffer,
                                                                              countBufferOffset, maxDrawCount, stride, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdDrawMeshTasksIndirectCountNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdDrawMeshTasksIndirectCountNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdDrawMeshTasksIndirectCountNV(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                    maxDrawCount, stride, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdDrawMeshTasksIndirectCountNV(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount,
                                                stride);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdDrawMeshTasksIndirectCountNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdDrawMeshTasksIndirectCountNV(commandBuffer, buffer, offset, countBuffer, countBufferOffset,
                                                                     maxDrawCount, stride, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetExclusiveScissorEnableNV(VkCommandBuffer commandBuffer, uint32_t firstExclusiveScissor,
                                                          uint32_t exclusiveScissorCount,
                                                          const VkBool32* pExclusiveScissorEnables) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetExclusiveScissorEnableNV,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetExclusiveScissorEnableNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetExclusiveScissorEnableNV(
                commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissorEnables, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetExclusiveScissorEnableNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetExclusiveScissorEnableNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetExclusiveScissorEnableNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount,
                                                                   pExclusiveScissorEnables, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetExclusiveScissorEnableNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount,
                                               pExclusiveScissorEnables);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetExclusiveScissorEnableNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetExclusiveScissorEnableNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount,
                                                                    pExclusiveScissorEnables, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetExclusiveScissorNV(VkCommandBuffer commandBuffer, uint32_t firstExclusiveScissor,
                                                    uint32_t exclusiveScissorCount, const VkRect2D* pExclusiveScissors) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetExclusiveScissorNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetExclusiveScissorNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetExclusiveScissorNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount,
                                                                       pExclusiveScissors, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetExclusiveScissorNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetExclusiveScissorNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetExclusiveScissorNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount,
                                                             pExclusiveScissors, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetExclusiveScissorNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissors);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetExclusiveScissorNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetExclusiveScissorNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount,
                                                              pExclusiveScissors, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetCheckpointNV(VkCommandBuffer commandBuffer, const void* pCheckpointMarker) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetCheckpointNV, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetCheckpointNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetCheckpointNV(commandBuffer, pCheckpointMarker, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetCheckpointNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetCheckpointNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetCheckpointNV(commandBuffer, pCheckpointMarker, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetCheckpointNV(commandBuffer, pCheckpointMarker);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetCheckpointNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetCheckpointNV(commandBuffer, pCheckpointMarker, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL GetQueueCheckpointDataNV(VkQueue queue, uint32_t* pCheckpointDataCount,
                                                    VkCheckpointDataNV* pCheckpointData) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetQueueCheckpointDataNV, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetQueueCheckpointDataNV]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetQueueCheckpointDataNV(queue, pCheckpointDataCount, pCheckpointData, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetQueueCheckpointDataNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetQueueCheckpointDataNV]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetQueueCheckpointDataNV(queue, pCheckpointDataCount, pCheckpointData, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchGetQueueCheckpointDataNV(queue, pCheckpointDataCount, pCheckpointData);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetQueueCheckpointDataNV]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetQueueCheckpointDataNV(queue, pCheckpointDataCount, pCheckpointData, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL InitializePerformanceApiINTEL(VkDevice device,
                                                             const VkInitializePerformanceApiInfoINTEL* pInitializeInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkInitializePerformanceApiINTEL, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateInitializePerformanceApiINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateInitializePerformanceApiINTEL(device, pInitializeInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkInitializePerformanceApiINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordInitializePerformanceApiINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordInitializePerformanceApiINTEL(device, pInitializeInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchInitializePerformanceApiINTEL(device, pInitializeInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordInitializePerformanceApiINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordInitializePerformanceApiINTEL(device, pInitializeInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL UninitializePerformanceApiINTEL(VkDevice device) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkUninitializePerformanceApiINTEL, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateUninitializePerformanceApiINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateUninitializePerformanceApiINTEL(device, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkUninitializePerformanceApiINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordUninitializePerformanceApiINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordUninitializePerformanceApiINTEL(device, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchUninitializePerformanceApiINTEL(device);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordUninitializePerformanceApiINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordUninitializePerformanceApiINTEL(device, record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CmdSetPerformanceMarkerINTEL(VkCommandBuffer commandBuffer,
                                                            const VkPerformanceMarkerInfoINTEL* pMarkerInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetPerformanceMarkerINTEL,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetPerformanceMarkerINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetPerformanceMarkerINTEL(commandBuffer, pMarkerInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetPerformanceMarkerINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetPerformanceMarkerINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetPerformanceMarkerINTEL(commandBuffer, pMarkerInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCmdSetPerformanceMarkerINTEL(commandBuffer, pMarkerInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetPerformanceMarkerINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetPerformanceMarkerINTEL(commandBuffer, pMarkerInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CmdSetPerformanceStreamMarkerINTEL(VkCommandBuffer commandBuffer,
                                                                  const VkPerformanceStreamMarkerInfoINTEL* pMarkerInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetPerformanceStreamMarkerINTEL,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetPerformanceStreamMarkerINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetPerformanceStreamMarkerINTEL(commandBuffer, pMarkerInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetPerformanceStreamMarkerINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetPerformanceStreamMarkerINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetPerformanceStreamMarkerINTEL(commandBuffer, pMarkerInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCmdSetPerformanceStreamMarkerINTEL(commandBuffer, pMarkerInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetPerformanceStreamMarkerINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetPerformanceStreamMarkerINTEL(commandBuffer, pMarkerInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CmdSetPerformanceOverrideINTEL(VkCommandBuffer commandBuffer,
                                                              const VkPerformanceOverrideInfoINTEL* pOverrideInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetPerformanceOverrideINTEL,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetPerformanceOverrideINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetPerformanceOverrideINTEL(commandBuffer, pOverrideInfo, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetPerformanceOverrideINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetPerformanceOverrideINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetPerformanceOverrideINTEL(commandBuffer, pOverrideInfo, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCmdSetPerformanceOverrideINTEL(commandBuffer, pOverrideInfo);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetPerformanceOverrideINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetPerformanceOverrideINTEL(commandBuffer, pOverrideInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquirePerformanceConfigurationINTEL(VkDevice device,
                                                                    const VkPerformanceConfigurationAcquireInfoINTEL* pAcquireInfo,
                                                                    VkPerformanceConfigurationINTEL* pConfiguration) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAcquirePerformanceConfigurationINTEL, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateAcquirePerformanceConfigurationINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAcquirePerformanceConfigurationINTEL(device, pAcquireInfo, pConfiguration, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAcquirePerformanceConfigurationINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordAcquirePerformanceConfigurationINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAcquirePerformanceConfigurationINTEL(device, pAcquireInfo, pConfiguration, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAcquirePerformanceConfigurationINTEL(device, pAcquireInfo, pConfiguration);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordAcquirePerformanceConfigurationINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAcquirePerformanceConfigurationINTEL(device, pAcquireInfo, pConfiguration, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ReleasePerformanceConfigurationINTEL(VkDevice device,
                                                                    VkPerformanceConfigurationINTEL configuration) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkReleasePerformanceConfigurationINTEL, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateReleasePerformanceConfigurationINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateReleasePerformanceConfigurationINTEL(device, configuration, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkReleasePerformanceConfigurationINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordReleasePerformanceConfigurationINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordReleasePerformanceConfigurationINTEL(device, configuration, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchReleasePerformanceConfigurationINTEL(device, configuration);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordReleasePerformanceConfigurationINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordReleasePerformanceConfigurationINTEL(device, configuration, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSetPerformanceConfigurationINTEL(VkQueue queue, VkPerformanceConfigurationINTEL configuration) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(queue), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkQueueSetPerformanceConfigurationINTEL, VulkanTypedHandle(queue, kVulkanObjectTypeQueue));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateQueueSetPerformanceConfigurationINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateQueueSetPerformanceConfigurationINTEL(queue, configuration, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkQueueSetPerformanceConfigurationINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordQueueSetPerformanceConfigurationINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordQueueSetPerformanceConfigurationINTEL(queue, configuration, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchQueueSetPerformanceConfigurationINTEL(queue, configuration);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordQueueSetPerformanceConfigurationINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordQueueSetPerformanceConfigurationINTEL(queue, configuration, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPerformanceParameterINTEL(VkDevice device, VkPerformanceParameterTypeINTEL parameter,
                                                            VkPerformanceValueINTEL* pValue) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPerformanceParameterINTEL, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetPerformanceParameterINTEL]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPerformanceParameterINTEL(device, parameter, pValue, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPerformanceParameterINTEL);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetPerformanceParameterINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPerformanceParameterINTEL(device, parameter, pValue, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPerformanceParameterINTEL(device, parameter, pValue);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetPerformanceParameterINTEL]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPerformanceParameterINTEL(device, parameter, pValue, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL SetLocalDimmingAMD(VkDevice device, VkSwapchainKHR swapChain, VkBool32 localDimmingEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkSetLocalDimmingAMD, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateSetLocalDimmingAMD]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateSetLocalDimmingAMD(device, swapChain, localDimmingEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkSetLocalDimmingAMD);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordSetLocalDimmingAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordSetLocalDimmingAMD(device, swapChain, localDimmingEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchSetLocalDimmingAMD(device, swapChain, localDimmingEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordSetLocalDimmingAMD]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordSetLocalDimmingAMD(device, swapChain, localDimmingEnable, record_obj);
        }
    }
}

#ifdef VK_USE_PLATFORM_FUCHSIA
VKAPI_ATTR VkResult VKAPI_CALL CreateImagePipeSurfaceFUCHSIA(VkInstance instance,
                                                             const VkImagePipeSurfaceCreateInfoFUCHSIA* pCreateInfo,
                                                             const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateImagePipeSurfaceFUCHSIA, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateImagePipeSurfaceFUCHSIA(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateImagePipeSurfaceFUCHSIA);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateImagePipeSurfaceFUCHSIA(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateImagePipeSurfaceFUCHSIA(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateImagePipeSurfaceFUCHSIA(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_FUCHSIA
#ifdef VK_USE_PLATFORM_METAL_EXT
VKAPI_ATTR VkResult VKAPI_CALL CreateMetalSurfaceEXT(VkInstance instance, const VkMetalSurfaceCreateInfoEXT* pCreateInfo,
                                                     const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateMetalSurfaceEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateMetalSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateMetalSurfaceEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateMetalSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateMetalSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateMetalSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_METAL_EXT
VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetBufferDeviceAddressEXT(VkDevice device, const VkBufferDeviceAddressInfo* pInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetBufferDeviceAddressEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetBufferDeviceAddressEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetBufferDeviceAddressEXT(device, pInfo, error_obj);
            if (skip) return 0;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetBufferDeviceAddressEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordGetBufferDeviceAddressEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetBufferDeviceAddressEXT(device, pInfo, record_obj);
        }
    }
    VkDeviceAddress result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetBufferDeviceAddressEXT(device, pInfo);
    }
    record_obj.device_address = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordGetBufferDeviceAddressEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetBufferDeviceAddressEXT(device, pInfo, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCooperativeMatrixPropertiesNV(VkPhysicalDevice physicalDevice,
                                                                              uint32_t* pPropertyCount,
                                                                              VkCooperativeMatrixPropertiesNV* pProperties) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceCooperativeMatrixPropertiesNV,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceCooperativeMatrixPropertiesNV(physicalDevice, pPropertyCount,
                                                                                             pProperties, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceCooperativeMatrixPropertiesNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceCooperativeMatrixPropertiesNV(physicalDevice, pPropertyCount, pProperties,
                                                                                   record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceCooperativeMatrixPropertiesNV(physicalDevice, pPropertyCount, pProperties);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceCooperativeMatrixPropertiesNV(physicalDevice, pPropertyCount, pProperties,
                                                                                    record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(
    VkPhysicalDevice physicalDevice, uint32_t* pCombinationCount, VkFramebufferMixedSamplesCombinationNV* pCombinations) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(
                physicalDevice, pCombinationCount, pCombinations, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(
                physicalDevice, pCombinationCount, pCombinations, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(physicalDevice, pCombinationCount,
                                                                                         pCombinations);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(
                physicalDevice, pCombinationCount, pCombinations, record_obj);
        }
    }
    return result;
}

#ifdef VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfacePresentModes2EXT(VkPhysicalDevice physicalDevice,
                                                                        const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo,
                                                                        uint32_t* pPresentModeCount,
                                                                        VkPresentModeKHR* pPresentModes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(physicalDevice), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetPhysicalDeviceSurfacePresentModes2EXT,
                          VulkanTypedHandle(physicalDevice, kVulkanObjectTypePhysicalDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetPhysicalDeviceSurfacePresentModes2EXT(physicalDevice, pSurfaceInfo,
                                                                                       pPresentModeCount, pPresentModes, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetPhysicalDeviceSurfacePresentModes2EXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetPhysicalDeviceSurfacePresentModes2EXT(physicalDevice, pSurfaceInfo, pPresentModeCount,
                                                                             pPresentModes, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetPhysicalDeviceSurfacePresentModes2EXT(physicalDevice, pSurfaceInfo, pPresentModeCount, pPresentModes);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetPhysicalDeviceSurfacePresentModes2EXT(physicalDevice, pSurfaceInfo, pPresentModeCount,
                                                                              pPresentModes, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireFullScreenExclusiveModeEXT(VkDevice device, VkSwapchainKHR swapchain) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkAcquireFullScreenExclusiveModeEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateAcquireFullScreenExclusiveModeEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateAcquireFullScreenExclusiveModeEXT(device, swapchain, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkAcquireFullScreenExclusiveModeEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordAcquireFullScreenExclusiveModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordAcquireFullScreenExclusiveModeEXT(device, swapchain, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchAcquireFullScreenExclusiveModeEXT(device, swapchain);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordAcquireFullScreenExclusiveModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordAcquireFullScreenExclusiveModeEXT(device, swapchain, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ReleaseFullScreenExclusiveModeEXT(VkDevice device, VkSwapchainKHR swapchain) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkReleaseFullScreenExclusiveModeEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateReleaseFullScreenExclusiveModeEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateReleaseFullScreenExclusiveModeEXT(device, swapchain, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkReleaseFullScreenExclusiveModeEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordReleaseFullScreenExclusiveModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordReleaseFullScreenExclusiveModeEXT(device, swapchain, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchReleaseFullScreenExclusiveModeEXT(device, swapchain);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordReleaseFullScreenExclusiveModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordReleaseFullScreenExclusiveModeEXT(device, swapchain, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeviceGroupSurfacePresentModes2EXT(VkDevice device,
                                                                     const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo,
                                                                     VkDeviceGroupPresentModeFlagsKHR* pModes) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkGetDeviceGroupSurfacePresentModes2EXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateGetDeviceGroupSurfacePresentModes2EXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateGetDeviceGroupSurfacePresentModes2EXT(device, pSurfaceInfo, pModes, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkGetDeviceGroupSurfacePresentModes2EXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallRecordGetDeviceGroupSurfacePresentModes2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordGetDeviceGroupSurfacePresentModes2EXT(device, pSurfaceInfo, pModes, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchGetDeviceGroupSurfacePresentModes2EXT(device, pSurfaceInfo, pModes);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPostCallRecordGetDeviceGroupSurfacePresentModes2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordGetDeviceGroupSurfacePresentModes2EXT(device, pSurfaceInfo, pModes, record_obj);
        }
    }
    return result;
}

#endif  // VK_USE_PLATFORM_WIN32_KHR
VKAPI_ATTR VkResult VKAPI_CALL CreateHeadlessSurfaceEXT(VkInstance instance, const VkHeadlessSurfaceCreateInfoEXT* pCreateInfo,
                                                        const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(instance), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCreateHeadlessSurfaceEXT, VulkanTypedHandle(instance, kVulkanObjectTypeInstance));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCreateHeadlessSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface, error_obj);
            if (skip) return VK_ERROR_VALIDATION_FAILED_EXT;
        }
    }
    RecordObject record_obj(vvl::Func::vkCreateHeadlessSurfaceEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCreateHeadlessSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    VkResult result;
    {
        VVL_ZoneScopedN("Dispatch");
        result = DispatchCreateHeadlessSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface);
    }
    record_obj.result = result;
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->object_dispatch) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCreateHeadlessSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface, record_obj);
        }
    }
    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetLineStippleEXT(VkCommandBuffer commandBuffer, uint32_t lineStippleFactor,
                                                uint16_t lineStipplePattern) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetLineStippleEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetLineStippleEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetLineStippleEXT(commandBuffer, lineStippleFactor, lineStipplePattern, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetLineStippleEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetLineStippleEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetLineStippleEXT(commandBuffer, lineStippleFactor, lineStipplePattern, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetLineStippleEXT(commandBuffer, lineStippleFactor, lineStipplePattern);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetLineStippleEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetLineStippleEXT(commandBuffer, lineStippleFactor, lineStipplePattern, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL ResetQueryPoolEXT(VkDevice device, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkResetQueryPoolEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateResetQueryPoolEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateResetQueryPoolEXT(device, queryPool, firstQuery, queryCount, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkResetQueryPoolEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordResetQueryPoolEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordResetQueryPoolEXT(device, queryPool, firstQuery, queryCount, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchResetQueryPoolEXT(device, queryPool, firstQuery, queryCount);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordResetQueryPoolEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordResetQueryPoolEXT(device, queryPool, firstQuery, queryCount, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetCullModeEXT(VkCommandBuffer commandBuffer, VkCullModeFlags cullMode) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetCullModeEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetCullModeEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetCullModeEXT(commandBuffer, cullMode, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetCullModeEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetCullModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetCullModeEXT(commandBuffer, cullMode, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetCullModeEXT(commandBuffer, cullMode);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetCullModeEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetCullModeEXT(commandBuffer, cullMode, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetFrontFaceEXT(VkCommandBuffer commandBuffer, VkFrontFace frontFace) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetFrontFaceEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetFrontFaceEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetFrontFaceEXT(commandBuffer, frontFace, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetFrontFaceEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetFrontFaceEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetFrontFaceEXT(commandBuffer, frontFace, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetFrontFaceEXT(commandBuffer, frontFace);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetFrontFaceEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetFrontFaceEXT(commandBuffer, frontFace, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveTopologyEXT(VkCommandBuffer commandBuffer, VkPrimitiveTopology primitiveTopology) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetPrimitiveTopologyEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetPrimitiveTopologyEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetPrimitiveTopologyEXT(commandBuffer, primitiveTopology, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetPrimitiveTopologyEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetPrimitiveTopologyEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetPrimitiveTopologyEXT(commandBuffer, primitiveTopology, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetPrimitiveTopologyEXT(commandBuffer, primitiveTopology);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetPrimitiveTopologyEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetPrimitiveTopologyEXT(commandBuffer, primitiveTopology, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportWithCountEXT(VkCommandBuffer commandBuffer, uint32_t viewportCount,
                                                      const VkViewport* pViewports) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetViewportWithCountEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetViewportWithCountEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetViewportWithCountEXT(commandBuffer, viewportCount, pViewports, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetViewportWithCountEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetViewportWithCountEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetViewportWithCountEXT(commandBuffer, viewportCount, pViewports, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetViewportWithCountEXT(commandBuffer, viewportCount, pViewports);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetViewportWithCountEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetViewportWithCountEXT(commandBuffer, viewportCount, pViewports, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetScissorWithCountEXT(VkCommandBuffer commandBuffer, uint32_t scissorCount,
                                                     const VkRect2D* pScissors) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetScissorWithCountEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetScissorWithCountEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetScissorWithCountEXT(commandBuffer, scissorCount, pScissors, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetScissorWithCountEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetScissorWithCountEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetScissorWithCountEXT(commandBuffer, scissorCount, pScissors, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetScissorWithCountEXT(commandBuffer, scissorCount, pScissors);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetScissorWithCountEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetScissorWithCountEXT(commandBuffer, scissorCount, pScissors, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdBindVertexBuffers2EXT(VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount,
                                                    const VkBuffer* pBuffers, const VkDeviceSize* pOffsets,
                                                    const VkDeviceSize* pSizes, const VkDeviceSize* pStrides) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdBindVertexBuffers2EXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdBindVertexBuffers2EXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdBindVertexBuffers2EXT(commandBuffer, firstBinding, bindingCount, pBuffers,
                                                                       pOffsets, pSizes, pStrides, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdBindVertexBuffers2EXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdBindVertexBuffers2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdBindVertexBuffers2EXT(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes,
                                                             pStrides, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdBindVertexBuffers2EXT(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdBindVertexBuffers2EXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdBindVertexBuffers2EXT(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes,
                                                              pStrides, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthTestEnableEXT(VkCommandBuffer commandBuffer, VkBool32 depthTestEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthTestEnableEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthTestEnableEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthTestEnableEXT(commandBuffer, depthTestEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthTestEnableEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthTestEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthTestEnableEXT(commandBuffer, depthTestEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthTestEnableEXT(commandBuffer, depthTestEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthTestEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthTestEnableEXT(commandBuffer, depthTestEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthWriteEnableEXT(VkCommandBuffer commandBuffer, VkBool32 depthWriteEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthWriteEnableEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthWriteEnableEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthWriteEnableEXT(commandBuffer, depthWriteEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthWriteEnableEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthWriteEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthWriteEnableEXT(commandBuffer, depthWriteEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthWriteEnableEXT(commandBuffer, depthWriteEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthWriteEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthWriteEnableEXT(commandBuffer, depthWriteEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthCompareOpEXT(VkCommandBuffer commandBuffer, VkCompareOp depthCompareOp) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthCompareOpEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthCompareOpEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthCompareOpEXT(commandBuffer, depthCompareOp, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthCompareOpEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthCompareOpEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthCompareOpEXT(commandBuffer, depthCompareOp, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthCompareOpEXT(commandBuffer, depthCompareOp);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthCompareOpEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthCompareOpEXT(commandBuffer, depthCompareOp, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBoundsTestEnableEXT(VkCommandBuffer commandBuffer, VkBool32 depthBoundsTestEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetDepthBoundsTestEnableEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetDepthBoundsTestEnableEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetDepthBoundsTestEnableEXT(commandBuffer, depthBoundsTestEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetDepthBoundsTestEnableEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetDepthBoundsTestEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetDepthBoundsTestEnableEXT(commandBuffer, depthBoundsTestEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetDepthBoundsTestEnableEXT(commandBuffer, depthBoundsTestEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetDepthBoundsTestEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetDepthBoundsTestEnableEXT(commandBuffer, depthBoundsTestEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilTestEnableEXT(VkCommandBuffer commandBuffer, VkBool32 stencilTestEnable) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilTestEnableEXT,
                          VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept :
             layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilTestEnableEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilTestEnableEXT(commandBuffer, stencilTestEnable, error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilTestEnableEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilTestEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilTestEnableEXT(commandBuffer, stencilTestEnable, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilTestEnableEXT(commandBuffer, stencilTestEnable);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilTestEnableEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilTestEnableEXT(commandBuffer, stencilTestEnable, record_obj);
        }
    }
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilOpEXT(VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, VkStencilOp failOp,
                                              VkStencilOp passOp, VkStencilOp depthFailOp, VkCompareOp compareOp) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(commandBuffer), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCmdSetStencilOpEXT, VulkanTypedHandle(commandBuffer, kVulkanObjectTypeCommandBuffer));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCmdSetStencilOpEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCmdSetStencilOpEXT(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp,
                                                                 error_obj);
            if (skip) return;
        }
    }
    RecordObject record_obj(vvl::Func::vkCmdSetStencilOpEXT);
    {
        VVL_ZoneScopedN("PreCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallRecordCmdSetStencilOpEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PreCallRecordCmdSetStencilOpEXT(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp, record_obj);
        }
    }
    {
        VVL_ZoneScopedN("Dispatch");
        DispatchCmdSetStencilOpEXT(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);
    }
    {
        VVL_ZoneScopedN("PostCallRecord");
        for (ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPostCallRecordCmdSetStencilOpEXT]) {
            auto lock = intercept->WriteLock();
            intercept->PostCallRecordCmdSetStencilOpEXT(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp,
                                                        record_obj);
        }
    }
}

VKAPI_ATTR VkResult VKAPI_CALL CopyMemoryToImageEXT(VkDevice device, const VkCopyMemoryToImageInfoEXT* pCopyMemoryToImageInfo) {
    VVL_ZoneScoped;

    auto layer_data = GetLayerDataPtr(GetDispatchKey(device), layer_data_map);
    bool skip = false;
    ErrorObject error_obj(vvl::Func::vkCopyMemoryToImageEXT, VulkanTypedHandle(device, kVulkanObjectTypeDevice));
    {
        VVL_ZoneScopedN("PreCallValidate");
        for (const ValidationObject* intercept : layer_data->intercept_vectors[InterceptIdPreCallValidateCopyMemoryToImageEXT]) {
            auto lock = intercept->ReadLock();
            skip |= intercept->PreCallValidateCopyMemoryToImageEXT(device, pCopyMemoryT