/* Copyright (c) 1997-2018
   Ewgenij Gawrilow, Michael Joswig (Technische Universitaet Berlin, Germany)
   http://www.polymake.org

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version: http://www.gnu.org/licenses/gpl.txt.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
--------------------------------------------------------------------------------
*/

#ifndef POLYMAKE_PERL_WRAPPERS_H
#define POLYMAKE_PERL_WRAPPERS_H

/* The following classes are designed to be used in the files automatically generated by polymake.
   If you use them in the manually written code, you are doing it on your own risk!

   The changes in syntax and contents must be concerted with the corresponding routines
   in the package Polymake::Core::CPlusPlus
*/

namespace pm { namespace perl {

template <typename Given, typename Target>
class access {
public:
   using type = Target;
   using return_type = Target;
   using value_type = typename attrib<type>::minus_const;

   static return_type get(const Value& v)
   {
      return static_cast<Target>(static_cast<Given>(v));
   }
};

template <typename Target>
class access<Target, Target> {
public:
   using type = Target;
   using return_type = Target;
   using value_type = typename attrib<type>::minus_const;

   static return_type get(const Value& v)
   {
      return static_cast<Target>(v);
   }
};

template <>
class access<std::string, std::string> {
public:
   using type = std::string;
   using return_type = std::string;
   using value_type = std::string;

   static return_type get(const Value& v)
   {
      std::string x;
      v.retrieve(x);
      return x;
   }
};

template <>
class access<const std::string, const std::string>
   : public access<std::string, std::string> {
public:
   using type = const std::string;
};

template <typename Given>
class access<Given, std::string>
   : public access<std::string, std::string> {};

template <typename Given>
class access<Given, const std::string>
   : public access<const std::string, const std::string> {};

template <typename Target>
struct canned_may_be_missing : std::false_type {};

template <typename E, typename... SharedParams>
struct canned_may_be_missing< pm::Array<E, SharedParams...> > : std::true_type {};

// TODO: add a declaration for representative of HashMaps when CPlusPlus.pm learns to generate them for anonymous hash maps

template <typename Given, typename Target, bool try_conv>
class access_canned<Given, Target, try_conv, true> {
public:
   using type = Target;
   using return_type = typename inherit_const<Target&, Given>::type;
   using value_type = typename attrib<type>::minus_const;
   using given_value_type = typename attrib<Given>::minus_const;

   static return_type get(const Value& v)
   {
      const Value::canned_data_t canned=Value::get_canned_data(v.sv);
      const bool maybe_missing=try_conv || canned_may_be_missing<typename attrib<Given>::minus_const>::value;

      if (!maybe_missing || canned.second) {
         if (std::is_same<value_type, given_value_type>::value) {
            if (!try_conv || *canned.first == typeid(value_type))
               return *reinterpret_cast<type*>(canned.second);
         } else {
            if (!try_conv || *canned.first == typeid(given_value_type))
               return convert_input(v, canned, std::is_same<value_type, given_value_type>());
         }

         if (wrapper_type conversion=type_cache<value_type>::get_conversion_constructor(v.sv)) {
            // It is a wrapper for new<T0>(T1), it expects the prototype at stack[0] and the argument at stack[1]
            SV* mini_stack[2]={ NULL, v.sv };
            if (SV* ret=conversion(mini_stack))
               return *reinterpret_cast<value_type*>(Value::get_canned_data(ret).second);
            else
               throw exception();
         }
      }

      return parse_input(v, bool_constant<maybe_missing>());
   }

private:
   static return_type parse_input(const Value& v, std::true_type)
   {
      Value temp_can;
      value_type* value=new(temp_can.allocate_canned(type_cache<value_type>::get_descr(0))) value_type;
      v >> *value;
      const_cast<Value&>(v).sv=temp_can.get_constructed_canned();
      return *value;
   }

   static return_type convert_input(const Value& v, const Value::canned_data_t& canned, std::false_type)
   {
      Value temp_can;
      value_type* value=new(temp_can.allocate_canned(type_cache<value_type>::get_descr(0))) value_type(*reinterpret_cast<Given*>(canned.second));
      const_cast<Value&>(v).sv=temp_can.get_constructed_canned();
      return *value;
   }

   static return_type parse_input(const Value& v, std::false_type)
   {
      // should never happen
      throw std::runtime_error("internal error: wrong use of access_canned::parse_input");
   }

   static return_type convert_input(const Value& v, const Value::canned_data_t& canned, std::true_type)
   {
      // should never happen
      throw std::runtime_error("internal error: wrong use of access_canned::convert_input");
   }
};

template <typename Target, bool try_conv>
class access_canned<Target, Target, try_conv, false> {
public:
   using type = typename inherit_const<unwary_t<typename attrib<Target>::minus_const>, Target>::type;
   using return_type = Target&;
   using value_type = typename attrib<type>::minus_const;

   static return_type get(const Value& v)
   {
      return wary(access_canned<type, type, try_conv, true>::get(v));
   }
};

template <typename Given, typename Target, bool try_conv>
class access_canned<Given, Target, try_conv, false> {
public:
   using type = typename inherit_const<Target, Given>::type;
   using return_type = typename inherit_const<Wary<Target>, Given>::type&;
   using given_value_type = unwary_t<typename attrib<Given>::minus_const>;
   using given_type = typename inherit_const<given_value_type, Given>::type;
   using value_type = typename attrib<type>::minus_const;

   static return_type get(const Value& v)
   {
      return wary(access_canned<given_type, Target, try_conv, true>::get(v));
   }
};

template <typename Given, typename Target>
class access<TryCanned<Given>, Target> : public access_canned<Given, Target, true> {};

template <typename Given, typename Target>
class access<Canned<Given>, Target> : public access_canned<Given, Target, false> {};

template <typename Given>
class access<TryCanned<Given>, TryCanned<Given> > : public access_canned<Given, Given, true> {};

template <typename Given>
class access<Canned<Given>, Canned<Given> > : public access_canned<Given, Given, false> {};

template <typename TData, typename TDeleter>
class access<Canned<const std::unique_ptr<TData, TDeleter>>, Canned<const std::unique_ptr<TData, TDeleter>>>
{
public:
   using type = const std::unique_ptr<TData, TDeleter>;
   using value_type = typename attrib<type>::minus_const;
   using return_type = TData&;

   static return_type get(const Value& v)
   {
      return *access_canned<type, type, false>::get(v);
   }
};

template <typename Target>
class access< Enum<Target>, Enum<Target> > {
public:
   using type = const Target;
   using return_type = Target;
   using value_type = Target;

   static return_type get(const Value& v)
   {
      return Target(v.enum_value());
   }
};

class RegistratorQueue {
public:
   enum Kind {
      // cf. CPlusPlus.pm
      function, embedded_rule, duplicate_class_instance
   };
   RegistratorQueue(const AnyString& name, Kind kind);

protected:
   SV* queue;
};

class RegularFunctionBase
   : public RegistratorQueue {
public:
   static const Kind kind = embedded_rule;
protected:
   void register_it(const AnyString& file, int line, const char* text,
                    indirect_wrapper_type get_flags_ptr, SV* arg_types,
                    void* func_ptr, const char* func_ptr_type) const;
};

class RegularFunction
   : public RegularFunctionBase {
   RegularFunction() = delete;
public:
   template <typename Fptr>
   void add__me(Fptr* fptr, const AnyString& file, int line, const char* text) const
   {
      register_it(file, line, text,
                  &TypeListUtils<Fptr>::get_flags, TypeListUtils<Fptr>::get_type_names(),
                  (void*)fptr, typeid(type2type<Fptr>).name());
   }
};

class FunctionTemplateBase
   : public RegistratorQueue {
   FunctionTemplateBase() = delete;
public:
   static const Kind kind = function;
protected:
   void register_it(wrapper_type wrapper, const AnyString& sig, const AnyString& file, int line,
                    SV* arg_types, SV* cross_apps) const;

   void register_it(wrapper_type wrapper, const AnyString& sig, const AnyString& file, int line,
                    SV* arg_types, int=0) const
   {
      register_it(wrapper, sig, file, line, arg_types, nullptr);
   }
};

template <typename Wrapper>
class FunctionTemplate
   : public FunctionTemplateBase {
   FunctionTemplate() = delete;
public:
   template <typename TAppList>
   void register_it(const AnyString& sig, const AnyString& file, int line, TAppList cross_apps) const
   {
      FunctionTemplateBase::register_it(&Wrapper::call, sig, file, line,
                                        TypeListUtils<typename Wrapper::arg_list>::get_type_names(), cross_apps);
   }
};

template <typename T, bool enabled=!(std::is_trivially_destructible<T>::value || is_masquerade<T>::value)>
class Destroy {
   static void impl(char* p)
   {
      T* obj = reinterpret_cast<T*>(p);
      obj->~T();
   }
public:
   static destructor_type func() { return &impl; }
};

template <typename T>
class Destroy<T, false> {
public:
   static destructor_type func() { return nullptr; }
};

template <typename T, bool enabled=(std::is_same<T, typename object_traits<T>::persistent_type>::value && is_mutable<T>::value && !is_masquerade<T>::value)>
class Copy {
   static void impl(void* place, const char* p)
   {
      const T* src = reinterpret_cast<const T*>(p);
      new(place) T(*src);
   }
public:
   static copy_constructor_type func() { return &impl; }
};

template <typename T>
class Copy<T, false> {
public:
   static copy_constructor_type func() { return nullptr; }
};


template <typename T, typename enabled=void>
class Assign {
public:
   static assignment_type func() { return nullptr; }
};

template <typename T>
class Assign<T, typename std::enable_if<is_mutable<type_behind_t<T>>::value &&
                                        is_readable<type_behind_t<T>>::value &&
                                        !std::is_same<T, Value>::value>::type> {
protected:
   template <typename, typename> friend class Assign;

   static void assign(T* dst, SV* sv, value_flags flags, std::true_type)
   {
      Value src(sv, flags);
      src >> *dst;
   }

   static void assign(T* dst, SV* sv, value_flags flags, std::false_type)
   {
      type_behind_t<T> x;
      Assign<type_behind_t<T>>::assign(&x, sv, flags, std::true_type());
      *dst = std::move(x);
   }

   static void impl(char* p, SV* sv, value_flags flags)
   {
      T* dst = reinterpret_cast<T*>(p);
      assign(dst, sv, flags, std::is_same<typename object_traits<T>::proxy_for, void>());
   }
public:
   static assignment_type func() { return &impl; }
};


class Unprintable {
protected:
   static SV* impl(const char*);
public:
   static conv_to_string_type func() { return &impl; }
};

template <typename T, typename enabled=void>
class ToString : public Unprintable {};

template <typename T>
class ToString<T, typename std::enable_if<is_printable<type_behind_t<T>>::value>::type> {
protected:
   static SV* to_string(const type_behind_t<T>& src)
   {
      Value ret;
      ostream my_stream(ret);
      PlainPrinter<> printer(my_stream);
      printer << src;
      return ret.get_temp();
   }
   static SV* impl(const char* p)
   {
      const T* src = reinterpret_cast<const T*>(p);
      return to_string(*src);
   }
public:
   static conv_to_string_type func() { return &impl; }
};


template <typename T, typename enabled=void>
class Serializable {
public:
   static constexpr class_kind flag_value() { return class_kind_null; }
   static conv_to_serialized_type conv() { return nullptr; }
   static provide_type provide() { return nullptr; }
   static provide_type provide_descr() { return nullptr; }
};

template <typename T>
class Serializable<T, typename std::enable_if<has_serialized<type_behind_t<T>>::value>::type> {
protected:
   static SV* store_serialized(const type_behind_t<T>& src, SV* holder)
   {
      Value ret(value_allow_non_persistent | value_allow_store_ref | value_read_only);
      ret.put(serialize(src), 0, holder);
      return ret.get_temp();
   }

   static SV* impl(const char* p, SV* holder)
   {
      const T* src = reinterpret_cast<const T*>(p);
      return store_serialized(*src, holder);
   }
public:
   typedef pure_type_t<decltype(serialize(std::declval<const type_behind_t<T>&>()))> serialized_t;

   static constexpr class_kind flag_value()
   {
      return check_container_feature<serialized_t, sparse>::value
             ? class_is_serializable | class_is_sparse_serialized
             : class_is_serializable;
   }

   static conv_to_serialized_type conv() { return &impl; }
   static provide_type provide() { return &type_cache<serialized_t>::provide; }
   static provide_type provide_descr() { return &type_cache<serialized_t>::provide_descr; }
};

// This one is used in overloaded operators for sparse proxies of primitive types.
// It should just retrieve the data element.
template <typename TProxy>
class Serializable<TProxy, typename std::enable_if<!std::is_same<TProxy, type_behind_t<TProxy>>::value &&
                                                   std::is_arithmetic<type_behind_t<TProxy>>::value>::type>
   : public Serializable<type_behind_t<TProxy>> {
protected:
   static SV* impl(const char* p, SV*)
   {
      const TProxy* src = reinterpret_cast<const TProxy*>(p);
      Value ret;
      ret << static_cast<type_behind_t<TProxy>>(*src);
      return ret.get_temp();
   }

public:
   static conv_to_serialized_type conv() { return &impl; }
};

class ClassRegistratorBase
   : public RegistratorQueue {
   ClassRegistratorBase() = delete;
public:
   static const Kind kind = duplicate_class_instance;
protected:
   static
   SV* register_class(const AnyString& name, const AnyString& file, int line,
                      SV* someref,
                      const char* typeid_name,
                      bool is_mutable, class_kind kind,
                      SV* vtbl_sv);

   static
   SV* create_builtin_vtbl(
      const std::type_info& type,
      size_t obj_size,
      int primitive_lvalue,
      copy_constructor_type copy_constructor,
      assignment_type assignment,
      destructor_type destructor
   );

   static
   SV* create_scalar_vtbl(
      const std::type_info& type,
      size_t obj_size,
      copy_constructor_type copy_constructor,
      assignment_type assignment,
      destructor_type destructor,
      conv_to_string_type to_string,
      conv_to_serialized_type to_serialized,
      provide_type provide_serialized_type,
      provide_type provide_serialized_descr,
      conv_to_int_type to_int,
      conv_to_float_type to_float
   );

   static
   SV* create_iterator_vtbl(
      const std::type_info& type,
      size_t obj_size,
      copy_constructor_type copy_constructor,
      destructor_type destructor,
      iterator_deref_type deref,
      iterator_incr_type incr,
      conv_to_int_type at_end,
      conv_to_int_type index
   );

   static
   SV* create_opaque_vtbl(
      const std::type_info& type,
      size_t obj_size,
      copy_constructor_type copy_constructor,
      assignment_type assignment,
      destructor_type destructor,
      conv_to_string_type to_string,
      conv_to_serialized_type to_serialized,
      provide_type provide_serialized_type,
      provide_type provide_serialized_descr
   );

   static
   SV* create_container_vtbl(
      const std::type_info& type,
      size_t obj_size, int total_dimension, int own_dimension,
      copy_constructor_type copy_constructor,
      assignment_type assignment,
      destructor_type destructor,
      conv_to_string_type to_string,
      conv_to_serialized_type to_serialized,
      provide_type provide_serialized_type,
      provide_type provide_serialized_descr,
      conv_to_int_type size,
      container_resize_type resize,
      container_store_type store_at_ref,
      provide_type provide_key_type,
      provide_type provide_key_descr,
      provide_type provide_value_type,
      provide_type provide_value_descr
   );

   static
   void fill_iterator_access_vtbl(
      SV* vtbl, int i,
      size_t it_size, size_t cit_size,
      destructor_type it_destructor,
      destructor_type cit_destructor,
      container_begin_type begin,
      container_begin_type cbegin,
      container_access_type deref,
      container_access_type cderef
   );

   static
   void fill_random_access_vtbl(
      SV* vtbl,
      container_access_type random,
      container_access_type crandom
   );

   static
   SV* create_composite_vtbl(
      const std::type_info& type,
      size_t obj_size, int obj_dimension,
      copy_constructor_type copy_constructor,
      assignment_type assignment,
      destructor_type destructor,
      conv_to_string_type to_string,
      conv_to_serialized_type to_serialized,
      provide_type provide_serialized_type,
      provide_type provide_serialized_descr,
      int n_members,
      provide_type provide_member_types,
      provide_type provide_member_descrs,
      provide_type provide_member_names,
      void (*fill)(composite_access_vtbl*)
   );
};

template <typename T>
class Builtin
   : public ClassRegistratorBase {
   Builtin() = delete;
public:
   void add__me(const AnyString& name, const AnyString& file, int line) const
   {
      register_class(
         name, file, line,
         queue,
         typeid(T).name(),
         false, class_is_scalar,
         create_builtin_vtbl(
            typeid(T), sizeof(T), mlist_contains<primitive_lvalues, T>::value,
            Copy<T>::func(),
            Assign<T>::func(),
            Destroy<T>::func()
         )
      );
   }
};

template <typename T, typename Model=typename object_traits<T>::model>
class ClassRegistrator;

template <>
class ClassRegistrator<Scalar,is_opaque> {};
template <>
class ClassRegistrator<Array,is_container> {};

template <typename T>
class ClassRegistrator<T, is_scalar>
   : public ClassRegistratorBase {
protected:
   typedef typename object_traits<T>::persistent_type persistent_type;

   template <typename Target, typename=void>
   struct conv;

   template <typename Target>
   struct conv<Target,
               typename std::enable_if<std::is_constructible<Target, T>::value>::type>
   {
      static Target func(const char* p)
      {
         const T* obj = reinterpret_cast<const T*>(p);
         return static_cast<Target>(*obj);
      }
   };

   template <typename Target>
   struct conv<Target,
               typename std::enable_if<!std::is_constructible<Target, T>::value &&
                                       std::is_constructible<Target, persistent_type>::value>::type>
   {
      static Target func(const char* p)
      {
         const T* obj = reinterpret_cast<const T*>(p);
         return static_cast<Target>(static_cast<const persistent_type&>(*obj));
      }
   };

   template <typename Target>
   struct conv<Target,
               typename std::enable_if<!std::is_constructible<Target, T>::value &&
                                       !std::is_constructible<Target, persistent_type>::value>::type>
   {
      static Target func(const char* p)
      {
         throw std::runtime_error("can't convert " + legible_typename<T>() + " to " + legible_typename<Target>());
      }
   };

public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_class(
         name, file, line, someref,
         typeid(T).name(),
         is_mutable<T>::value,
         class_is_scalar | Serializable<T>::flag_value(),
         create_scalar_vtbl(
            typeid(T), sizeof(T),
            Copy<T>::func(),
            Assign<T>::func(),
            Destroy<T>::func(),
            ToString<T>::func(),
            Serializable<T>::conv(),
            Serializable<T>::provide(),
            Serializable<T>::provide_descr(),
            &conv<int>::func,
            &conv<double>::func
         )
      );
   }
};

template <typename T, bool is_iterator=check_iterator_feature<T, end_sensitive>::value>
class OpaqueClassRegistrator
   : public ClassRegistratorBase {
public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_class(
         name, file, line, someref,
         typeid(T).name(),
         is_mutable<T>::value,
         class_is_opaque | Serializable<T>::flag_value(),
         create_opaque_vtbl(
            typeid(T), sizeof(T),
            Copy<T>::func(),
            Assign<T>::func(),
            Destroy<T>::func(),
            ToString<T>::func(),
            Serializable<T>::conv(),
            Serializable<T>::provide(),
            Serializable<T>::provide_descr()
         )
      );
   }
};

template <typename TData, typename TDeleter>
class OpaqueClassRegistrator<std::unique_ptr<TData, TDeleter>, false>
   : public ClassRegistratorBase {
public:
   using Tptr = std::unique_ptr<TData, TDeleter>;

   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_class(
         name, file, line, someref,
         typeid(Tptr).name(),
         false,
         class_is_opaque | class_kind_null,
         create_opaque_vtbl(
            typeid(Tptr), sizeof(Tptr),
            nullptr,
            nullptr,
            Destroy<Tptr>::func(),
            Unprintable::func(),
            nullptr,
            nullptr,
            nullptr
         )
      );
   }
};

template <typename T>
class OpaqueClassRegistrator<T, true>
   : public ClassRegistratorBase {
protected:
   static const bool read_only=attrib<typename iterator_traits<T>::reference>::is_const;

   static SV* deref(const char* p)
   {
      const T* it = reinterpret_cast<const T*>(p);
      Value ret((read_only ? value_read_only : value_mutable) |
                value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
      ret.put_lvalue(**it, 0, nullptr, (nothing*)nullptr);
      return ret.get_temp();
   }

   static void incr(char* p)
   {
      T* it = reinterpret_cast<T*>(p);
      ++(*it);
   }

   static int at_end(const char* p)
   {
      const T* it = reinterpret_cast<const T*>(p);
      return it->at_end();
   }

   static int index_impl(const char* p)
   {
      const T* it = reinterpret_cast<const T*>(p);
      return it->index();
   }

   static conv_to_int_type index(std::false_type) { return nullptr; }
   static conv_to_int_type index(std::true_type) { return &index_impl; }
   static conv_to_int_type index() { return index(bool_constant<check_iterator_feature<T, indexed>::value>()); }

public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_class(
         name, file, line, someref,
         typeid(T).name(),
         true,
         class_is_opaque,
         create_iterator_vtbl(
            typeid(T), sizeof(T),
            Copy<T>::func(),
            Destroy<T>::func(),
            &deref,
            &incr,
            &at_end,
            index()
         )
      );
   }
};

template <typename T>
class ClassRegistrator<T, is_opaque>
   : public OpaqueClassRegistrator<T> {};

template <typename Mode, bool TEnableSparse>
struct transform_input_mode {
   typedef Mode type;
};

template <int TResizeable>
struct transform_input_mode<io_test::as_sparse<TResizeable>, false> {
   typedef io_test::as_array<TResizeable, true> type;
};

template <typename T, bool TEnable=true, bool TEnableSparse=true, bool TMutable=TEnable && !object_traits<T>::is_always_const>
struct input_mode
   : transform_input_mode<typename io_test::input_mode<T, false>::type, TEnableSparse> {};

template <typename T, bool TEnable, bool TEnableSparse>
struct input_mode<T, TEnable, TEnableSparse, false> {
   typedef std::false_type type;
};

template <typename T, int TDim=object_traits<T>::dimension>
struct container_helper;

template <typename T>
struct container_helper<T, 1> {
   typedef T type;
   static type& streamline(T& x) { return x; }
   static const type& streamline(const T& x) { return x; }
};

template <typename T>
struct container_helper<T, 2> {
   typedef Rows<T> type;
   static type& streamline(T& x) { return rows(x); }
   static const type& streamline(const T& x) { return rows(x); }
};

template <typename T, typename Category=typename container_traits<typename container_helper<T>::type>::category,
          bool _obscure=obscure_type<T>::value>
class ContainerClassRegistrator
   : public ClassRegistratorBase {
public:
   typedef container_helper<T> helper;
   typedef typename helper::type Obj;
   static constexpr bool
        is_associative=is_assoc_container<Obj>::value,
             is_sparse=check_container_feature<Obj, sparse>::value,
      is_sparse_native=check_container_feature<T, sparse>::value,
              like_set=is_among<typename object_traits<Obj>::generic_tag, is_set, is_unordered_set>::value;
   typedef typename std::conditional<is_associative, end_sensitive, void>::type iterator_feature;
   typedef typename ensure_features<Obj, iterator_feature>::iterator iterator;
   typedef typename ensure_features<Obj, iterator_feature>::const_iterator const_iterator;

   static const bool allow_non_const_access= !object_traits<T>::is_always_const &&
                                             !std::is_same<iterator, const_iterator>::value &&
                                             !attrib<typename iterator_traits<iterator>::reference>::is_const &&
                                             !object_traits<typename iterator_traits<iterator>::value_type>::is_always_const;

   typedef bool_constant<allow_non_const_access> non_const_access;

   static const int element_dim=object_traits<typename Obj::value_type>::total_dimension;

protected:
   static int size_impl(const char* p)
   {
      const T* obj = reinterpret_cast<const T*>(p);
      return helper::streamline(*obj).size();
   }

   static int dim(const char* p)
   {
      const T* obj = reinterpret_cast<const T*>(p);
      return helper::streamline(*obj).dim();
   }

   static void resize_impl(char* p, int n)
   {
      T* obj = reinterpret_cast<T*>(p);
      helper::streamline(*obj).resize(n);
   }

   static void fixed_size(char* p, int n)
   {
      T* obj = reinterpret_cast<T*>(p);
      if (get_dim(helper::streamline(*obj)) != n)
         throw std::runtime_error("size mismatch");
   }

   static void clear_by_resize(char* p, int)
   {
      T* obj = reinterpret_cast<T*>(p);
      helper::streamline(*obj).clear();
   }

   template <typename E>
   static void check_insertion(const Obj&, const E&, std::false_type) {}

   static void check_insertion(const Obj& obj, int x, std::true_type)
   {
      if (x<0 || x>=obj.dim())
         throw std::runtime_error("element out of range");
   }

   static void push_back(char* p_obj, char* p_it, int, SV* src)
   {
      T* obj = reinterpret_cast<T*>(p_obj);
      iterator* it = reinterpret_cast<iterator*>(p_it);
      typename Obj::value_type x;
      Value v(src);
      v >> x;
      check_insertion(helper::streamline(*obj), x,
                      bool_constant<check_container_feature<Obj, sparse_compatible>::value>());
      helper::streamline(*obj).insert(*it, x);
   }

   static void insert(char* p_obj, char*, int, SV* src)
   {
      T* obj = reinterpret_cast<T*>(p_obj);
      typedef typename item4insertion<typename Obj::value_type>::type item_type;
      item_type x=item_type();
      Value v(src);
      v >> x;
      check_insertion(helper::streamline(*obj), x,
                      bool_constant<check_container_feature<Obj, sparse_compatible>::value>());
      helper::streamline(*obj).insert(x);
   }

   static void store_dense(char*, char* p_it, int, SV* src)
   {
      iterator* it = reinterpret_cast<iterator*>(p_it);
      Value v(src, value_not_trusted);
      v >> non_const(**it);
      ++(*it);
   }

   static void store_sparse(char* p_obj, char* p_it, int index, SV* src)
   {
      T* obj = reinterpret_cast<T*>(p_obj);
      iterator* it = reinterpret_cast<iterator*>(p_it);
      Value v(src, value_not_trusted);
      typename Obj::value_type x;
      v >> x;
      if (!is_zero(x)) {
         if (!it->at_end() && it->index()==index) {
            **it=x; ++(*it);
         } else {
            obj->insert(*it, index, x);
         }
      } else {
         if (!it->at_end() && it->index()==index)
            obj->erase((*it)++);
      }
   }

   template <typename Iterator, bool non_const>
   struct do_it {
      typedef typename assign_const<T, !non_const>::type* ObjPtr;

      static void begin(void* it_place, char* p)
      {
         ObjPtr obj = reinterpret_cast<ObjPtr>(p);
         new(it_place) Iterator(ensure(helper::streamline(*obj), (iterator_feature*)0).begin());
      }

      static void rbegin(void* it_place, char* p)
      {
         ObjPtr obj = reinterpret_cast<ObjPtr>(p);
         new(it_place) Iterator(ensure(helper::streamline(*obj), (iterator_feature*)0).rbegin());
      }

      static void deref(char*, char* p_it, int, SV* dst, SV* container_sv)
      {
         Iterator* it = reinterpret_cast<Iterator*>(p_it);
         Value pv(dst, (non_const ? value_mutable : value_read_only) | value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
         pv.put_lvalue(**it, 0, nullptr, (nothing*)nullptr, container_sv);
         ++(*it);
      }

      static void deref_pair(char*, char* p_it, int i, SV* dst, SV* container_sv)
      {
         Iterator* it = reinterpret_cast<Iterator*>(p_it);
         if (i<=0) {
            // i==-1: FIRSTKEY;  i==0: NEXTKEY
            if (i==0) ++(*it);
            if (!it->at_end()) {
               Value pv(dst, value_read_only | value_allow_non_persistent | value_allow_store_ref);
               pv.put((*it)->first, 0, container_sv);
            }
         } else {
            // i==1: fetch value
            Value pv(dst, (non_const ? value_mutable : value_read_only) | value_allow_non_persistent | value_allow_store_ref);
            pv.put((*it)->second, 0, container_sv);
         }
      }
   };

   template <typename Iterator, bool TDim=element_dim>
   struct do_sparse {
      static void deref(char* p_obj, char* p_it, int index, SV* dst, SV* container_sv)
      {
         Iterator* it = reinterpret_cast<Iterator*>(p_it);
         if (it->at_end() || index < it->index()) {
            Value pv(dst);
            pv.put(undefined(), 0);
         } else {
            do_it<Iterator, true>::deref(p_obj, p_it, index, dst, container_sv);
         }
      }
   };

   template <typename Iterator>
   struct do_sparse<Iterator, 0> {
      static void deref(char* p_obj, char* p_it, int index, SV* dst, SV* container_sv)
      {
         T* obj = reinterpret_cast<T*>(p_obj);
         Iterator* it = reinterpret_cast<Iterator*>(p_it);
         Value pv(dst, value_expect_lval | value_allow_non_persistent);
         sparse_elem_proxy< sparse_proxy_it_base<Obj, Iterator>, typename Obj::value_type, typename Obj::reference::parameters >
            x(sparse_proxy_it_base<Obj, Iterator>(*obj, *it, index));
         if (x.exists()) ++(*it);
         pv.put(std::move(x), 0, container_sv);
      }
   };

   template <typename Iterator, bool TDim=element_dim>
   struct do_const_sparse {
      static void deref(char* p_obj, char* p_it, int index, SV* dst, SV* container_sv)
      {
         Iterator* it = reinterpret_cast<Iterator*>(p_it);
         if (it->at_end() || index < it->index()) {
            Value pv(dst);
            pv.put(undefined(), 0);
         } else {
            do_it<Iterator, false>::deref(p_obj, p_it, index, dst, container_sv);
         }
      }
   };

   template <typename Iterator>
   struct do_const_sparse<Iterator, 0> {
      static void deref(char*, char* p_it, int index, SV* dst, SV* container_sv)
      {
         Iterator* it = reinterpret_cast<Iterator*>(p_it);
         Value pv(dst, value_read_only | value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
         if (!it->at_end() && index==it->index()) {
            pv.put_lvalue(**it, 0, nullptr, (nothing*)nullptr, container_sv);
            ++(*it);
         } else {
            pv.put(zero_value<typename Obj::value_type>(), 0);
         }
      }
   };

   static conv_to_int_type size(std::false_type) { return &size_impl; }
   static conv_to_int_type size(std::true_type)  { return &dim; }

   static container_resize_type resize(io_test::as_list<Obj>)
   {
      return &clear_by_resize;
   }
   static container_resize_type resize(io_test::as_set)
   {
      return &clear_by_resize;
   }
   template <bool allow_sparse>
   static container_resize_type resize(io_test::as_array<1, allow_sparse>)
   {
      return &resize_impl;
   }
   template <bool allow_sparse>
   static container_resize_type resize(io_test::as_array<0, allow_sparse>)
   {
      return &fixed_size;
   }
   static container_resize_type resize(io_test::as_sparse<-1>) { return nullptr; }
   static container_resize_type resize(std::false_type) { return nullptr; }

   static container_store_type store_at_ref(io_test::as_list<Obj>)
   {
      return &push_back;
   }
   static container_store_type store_at_ref(io_test::as_set)
   {
      return &insert;
   }
   template <int resizeable, bool allow_sparse>
   static container_store_type store_at_ref(io_test::as_array<resizeable, allow_sparse>)
   {
      return &store_dense;
   }
   template <int resizeable>
   static container_store_type store_at_ref(io_test::as_sparse<resizeable>)
   {
      return &store_sparse;
   }
   static container_store_type store_at_ref(std::false_type) { return nullptr; }

   static destructor_type it_destructor(std::true_type) { return Destroy<iterator>::func(); }
   static destructor_type it_destructor(std::false_type) { return Destroy<const_iterator>::func(); }

   static container_begin_type begin(std::true_type)
   {
      return &do_it<iterator, true>::begin;
   }
   static container_begin_type begin(std::false_type)
   {
      return &do_it<const_iterator, false>::begin;
   }

   static container_access_type deref(std::false_type, std::false_type, std::true_type)
   {
      return &do_it<iterator, true>::deref;
   }
   static container_access_type deref(std::false_type, std::false_type, std::false_type)
   {
      return &do_it<const_iterator, false>::deref;
   }
   static container_access_type deref(std::true_type, std::false_type, std::true_type)
   {
      return &do_it<iterator, true>::deref_pair;
   }
   static container_access_type deref(std::true_type, std::false_type, std::false_type)
   {
      return &do_it<const_iterator, false>::deref_pair;
   }
   static container_access_type deref(std::false_type, std::true_type, std::true_type)
   {
      return &do_sparse<iterator>::deref;
   }
   static container_access_type deref(std::false_type, std::true_type, std::false_type)
   {
      return &do_const_sparse<const_iterator>::deref;
   }

   static provide_type provide_key_type(std::true_type)
   {
      return &type_cache<typename T::key_type>::provide;
   }
   static provide_type provide_key_type(std::false_type)
   {
      return &type_cache<typename object_traits<typename T::value_type>::persistent_type>::provide;
   }
   static provide_type provide_value_type(std::true_type)
   {
      return &type_cache<typename T::mapped_type>::provide;
   }
   static provide_type provide_value_type(std::false_type)
   {
      return &type_cache<typename object_traits<typename Obj::value_type>::persistent_type>::provide;
   }

   static provide_type provide_key_descr(std::true_type)
   {
      return &type_cache<typename T::key_type>::provide_descr;
   }
   static provide_type provide_key_descr(std::false_type)
   {
      return &type_cache<typename object_traits<typename T::value_type>::persistent_type>::provide_descr;
   }
   static provide_type provide_value_descr(std::true_type)
   {
      return &type_cache<typename T::mapped_type>::provide_descr;
   }
   static provide_type provide_value_descr(std::false_type)
   {
      return &type_cache<typename object_traits<typename Obj::value_type>::persistent_type>::provide_descr;
   }

   static SV* create_vtbl()
   {
      SV* vtbl=create_container_vtbl(
         typeid(T), sizeof(T),
         object_traits<T>::total_dimension, object_traits<T>::dimension,
         Copy<T>::func(),
         Assign<T>::func(),
         Destroy<T>::func(),
         ToString<T>::func(),
         Serializable<T>::conv(),
         Serializable<T>::provide(),
         Serializable<T>::provide_descr(),
         size(bool_constant<is_sparse>()),
         resize(typename input_mode<Obj>::type()),
         store_at_ref(typename input_mode<Obj, !is_associative, element_dim==0>::type()),
         provide_key_type(bool_constant<is_associative>()),
         provide_key_descr(bool_constant<is_associative>()),
         provide_value_type(bool_constant<is_associative>()),
         provide_value_descr(bool_constant<is_associative>())
      );
      fill_iterator_access_vtbl(
         vtbl, 0,
         sizeof(iterator), sizeof(const_iterator),
         it_destructor(non_const_access()),
         it_destructor(std::false_type()),
         begin(non_const_access()),
         begin(std::false_type()),
         deref(bool_constant<is_associative>(), bool_constant<is_sparse>(), non_const_access()),
         deref(bool_constant<is_associative>(), bool_constant<is_sparse>(), std::false_type())
      );
      return vtbl;
   }

   static SV* register_me(const AnyString& name, const AnyString& file, int line, SV* someref, SV* vtbl)
   {
      return register_class(
         name, file, line, someref,
         typeid(T).name(),
         is_mutable<T>::value,
         class_is_container | Serializable<T>::flag_value() |
            (is_sparse && !std::is_same<T, Obj>::value ? class_is_sparse_serialized : class_kind_null) |
            (is_associative   ? class_is_assoc_container :
             is_sparse_native ? class_is_sparse_container :
             like_set         ? class_is_set : class_kind_null),
         vtbl);
   }

public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_me(name, file, line, someref, create_vtbl());
   }
};

template <typename T>
class ContainerClassRegistrator<T, bidirectional_iterator_tag, false>
   : public ContainerClassRegistrator<T, forward_iterator_tag, false> {
   typedef ContainerClassRegistrator<T, forward_iterator_tag, false> super;
public:
   typedef typename super::Obj::reverse_iterator reverse_iterator;
   typedef typename super::Obj::const_reverse_iterator const_reverse_iterator;
protected:
   static container_begin_type rbegin(std::true_type)
   {
      return &super::template do_it<reverse_iterator, true>::rbegin;
   }
   static container_begin_type rbegin(std::false_type)
   {
      return &super::template do_it<const_reverse_iterator, false>::rbegin;
   }

   static container_access_type rderef(std::false_type, std::true_type)
   {
      return &super::template do_it<reverse_iterator, true>::deref;
   }
   static container_access_type rderef(std::false_type, std::false_type)
   {
      return &super::template do_it<const_reverse_iterator, false>::deref;
   }
   static container_access_type rderef(std::true_type, std::true_type)
   {
      return &super::template do_sparse<reverse_iterator>::deref;
   }
   static container_access_type rderef(std::true_type, std::false_type)
   {
      return &super::template do_const_sparse<const_reverse_iterator>::deref;
   }

   static destructor_type rit_destructor(std::true_type) { return Destroy<reverse_iterator>::func(); }
   static destructor_type rit_destructor(std::false_type) { return Destroy<const_reverse_iterator>::func(); }
public:
   static SV* create_vtbl(std::false_type)
   {
      SV* vtbl=super::create_vtbl();
      super::fill_iterator_access_vtbl(
         vtbl, 2,
         sizeof(reverse_iterator), sizeof(const_reverse_iterator),
         rit_destructor(typename super::non_const_access()),
         rit_destructor(std::false_type()),
         rbegin(typename super::non_const_access()),
         rbegin(std::false_type()),
         rderef(bool_constant<super::is_sparse>(), typename super::non_const_access()),
         rderef(bool_constant<super::is_sparse>(), std::false_type())
      );
      return vtbl;
   }

   static SV* create_vtbl(std::true_type)
   {
      return super::create_vtbl();
   }
   static SV* create_vtbl()
   {
      return create_vtbl(bool_constant<super::is_associative>());
   }
public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return super::register_me(name, file, line, someref, create_vtbl());
   }
};

template <typename T>
class ContainerClassRegistrator<T, random_access_iterator_tag, false>
   : public ContainerClassRegistrator<T, bidirectional_iterator_tag, false> {
   typedef ContainerClassRegistrator<T, bidirectional_iterator_tag, false> super;
protected:
   static void random_impl(char* p_obj, char*, int index, SV* dst, SV* container_sv)
   {
      T* obj = reinterpret_cast<T*>(p_obj);
      index=index_within_range(super::helper::streamline(*obj), index);
      Value pv(dst, value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
      pv.put_lvalue(super::helper::streamline(*obj)[index], 0, nullptr, (nothing*)nullptr, container_sv);
   }

   static void crandom(char* p_obj, char*, int index, SV* dst, SV* container_sv)
   {
      const T* obj = reinterpret_cast<const T*>(p_obj);
      index=index_within_range(super::helper::streamline(*obj), index);
      Value pv(dst, value_read_only | value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
      pv.put_lvalue(super::helper::streamline(*obj)[index], 0, nullptr, (nothing*)nullptr, container_sv);
   }

   static void random_sparse(char* p_obj, char*, int index, SV* dst, SV* container_sv)
   {
      T* obj = reinterpret_cast<T*>(p_obj);
      index=index_within_range(super::helper::streamline(*obj), index);
      Value pv(dst, value_expect_lval | value_allow_non_persistent);
      pv.put(super::helper::streamline(*obj)[index], 0, container_sv);
   }

   static container_access_type random(std::false_type, std::true_type)
   {
      return &random_impl;
   }
   static container_access_type random(std::true_type, std::true_type)
   {
      return &random_sparse;
   }
   template <typename is_sparse>
   static container_access_type random(is_sparse, std::false_type)
   {
      return &crandom;
   }
public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      SV* vtbl=super::create_vtbl();
      super::fill_random_access_vtbl(
         vtbl,
         random(bool_constant<super::is_sparse>(), typename super::non_const_access()),
         random(bool_constant<super::is_sparse>(), std::false_type())
      );
      return super::register_me(name, file, line, someref, vtbl);
   }
};

template <typename T, typename Category>
class ContainerClassRegistrator<T, Category, true>
   : public ClassRegistratorBase {
public:
   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_class(
         name, file, line, someref,
         typeid(T).name(),
         false, class_is_opaque,
         create_builtin_vtbl(
            typeid(T), sizeof(T),
            0, 0, 0, Destroy<T>::func()
         )
      );
   }
};

template <typename T>
class ClassRegistrator<T, is_container>
   : public ContainerClassRegistrator<T> {};

template <typename T, int n=0, int l=list_length<typename object_traits<T>::elements>::value>
struct CompositeClassRegistrator {
   typedef typename n_th<typename object_traits<T>::elements, n>::type member_type;
   static const bool allow_non_const_access = !attrib<member_type>::is_const &&
                                              !object_traits<typename deref<member_type>::type>::is_always_const;

   static void get_impl(char* p, SV* dst, SV* container_sv)
   {
      T* obj = reinterpret_cast<T*>(p);
      Value pv(dst, value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
      pv.put_lvalue(visit_n_th(*obj, int_constant<n>()), 0, nullptr, (nothing*)nullptr, container_sv);
   }

   static void cget(char* p, SV* dst, SV* container_sv)
   {
      const T* obj = reinterpret_cast<const T*>(p);
      Value pv(dst, value_read_only | value_expect_lval | value_allow_non_persistent | value_allow_store_ref);
      pv.put_lvalue(visit_n_th(*obj, int_constant<n>()), 0, nullptr, (nothing*)nullptr, container_sv);
   }

   static void store_impl(char* p, SV* src)
   {
      T* obj = reinterpret_cast<T*>(p);
      Value v(src, value_not_trusted);
      v >> visit_n_th(*obj, int_constant<n>());
   }

   static composite_access_type get(std::true_type)  { return &get_impl; }
   static composite_access_type get(std::false_type) { return &cget; }
   static composite_store_type store(std::true_type)  { return &store_impl; }
   static composite_store_type store(std::false_type) { return nullptr; }

   static void init(composite_access_vtbl* acct)
   {
      acct->get[0]=get(bool_constant<allow_non_const_access>());
      acct->get[1]=get(std::false_type());
      acct->store=store(bool_constant<allow_non_const_access>());
      CompositeClassRegistrator<T, n+1, l>::init(++acct);
   }

   static SV* provide_member_names() { return member_names(recognizer_bait(0), (T*)0); }
};

template <typename T, int l>
struct CompositeClassRegistrator<T,l,l> {
   static void init(composite_access_vtbl*) {}
};

template <typename T>
struct get_persistent_type {
   typedef typename object_traits<T>::persistent_type type;
};

template <typename T>
class ClassRegistrator<T, is_composite>
   : public ClassRegistratorBase {
public:
   typedef typename list_transform_unary<get_persistent_type, typename object_traits<T>::elements>::type elements;

   static SV* register_it(const AnyString& name, SV* someref, const AnyString& file=AnyString(), int line=0)
   {
      return register_class(
         name, file, line, someref,
         typeid(T).name(),
         is_mutable<T>::value,
         class_is_composite | Serializable<T>::flag_value(),
         create_composite_vtbl(
            typeid(T), sizeof(T), object_traits<T>::total_dimension,
            Copy<T>::func(),
            Assign<T>::func(),
            Destroy<T>::func(),
            ToString<T>::func(),
            Serializable<T>::conv(),
            Serializable<T>::provide(),
            Serializable<T>::provide_descr(),
            list_length<elements>::value,
            &TypeListUtils<elements>::provide_types,
            &TypeListUtils<elements>::provide_descrs,
            &CompositeClassRegistrator<T>::provide_member_names,
            &CompositeClassRegistrator<T>::init
         )
      );
   }
};

} }
namespace polymake { namespace perl_bindings {

template <typename T>
class Class
   : public pm::perl::ClassRegistrator<T> {
   Class() = delete;
public:
   void add__me(const AnyString& name, const AnyString& file, int line) const
   {
      this->register_it(name, this->queue, file, line);
   }
};

} }
namespace pm { namespace perl {

class ClassTemplate {
   ClassTemplate() = delete;
public:
   static void add__me(const AnyString& name);
};

SV* get_parameterized_type_impl(const AnyString& pkg, bool exact_match);

template <typename TypeList, bool exact_match> inline
SV* get_parameterized_type(const AnyString& pkg, bool_constant<exact_match>)
{
   Stack stack(true, 1+TypeListUtils<TypeList>::type_cnt);
   if (TypeListUtils<TypeList>::push_types(stack)) {
      return get_parameterized_type_impl(pkg, exact_match);
   } else {
      stack.cancel();
      return NULL;
   }
}

class EmbeddedRule
   : public RegistratorQueue {
private:
   EmbeddedRule() = delete;
public:
   static const Kind kind = embedded_rule;
   void add__me(const AnyString& file, int line, const AnyString& text) const;
};

template <typename Target, typename Source,
          bool enabled=can_assign_to<typename access<Source>::type, Target>::value>
struct Operator_assign_impl {
   static wrapper_type func() { return nullptr; }
};

template <typename Target, typename Source>
struct Operator_assign_impl<Target, Source, true> {
   static void call(Target& dst, const Value& src)
   {
      if (MaybeWary<Target>::value && (src.get_flags() & value_not_trusted))
         maybe_wary(dst)=src.get<Source>();
      else
         dst=src.get<Source>();
   }
   static wrapper_type func() { return reinterpret_cast<wrapper_type>(&call); }
};

template <typename Target, typename Source>
class Operator_assign
   : public FunctionTemplateBase {
   Operator_assign() = delete;
public:
   typedef cons<Target, Source> arg_list;

   template <typename TAppList>
   void add__me(const AnyString& file, int line, TAppList cross_apps) const
   {
      register_it(Operator_assign_impl<Target, Source>::func(),
                  "=ass", file, line, TypeListUtils<arg_list>::get_type_names(), cross_apps);
   }
};

template <typename Target, typename Source,
          bool enabled=std::is_constructible<Target, typename access<Source>::type>::value>
struct Operator_convert_impl {
   static wrapper_type func() { return nullptr; }
};

template <typename Target, typename Source>
struct Operator_convert_impl<Target, Source, true> {
   static Target call(const Value& src)
   {
      return Target(src.get<Source>());
   }
   static wrapper_type func() { return reinterpret_cast<wrapper_type>(&call); }
};

template <typename Target, typename Source>
class Operator_convert
   : public FunctionTemplateBase {
   Operator_convert() = delete;
public:
   typedef cons<Target, Source> arg_list;

   template <typename TAppList>
   void add__me(const AnyString& file, int line, TAppList cross_apps) const
   {
      register_it(Operator_convert_impl<Target, Source>::func(),
                  ".cnv", file, line, TypeListUtils<arg_list>::get_type_names(), cross_apps);
   }
};

Wrapper4UnaryOperator(-, neg);
Wrapper4UnaryOperator(~, com);

Wrapper4UnaryAnchOperator(-, neg);
Wrapper4UnaryAnchOperator(~, com);

Wrapper4BinaryOperator(+, add);
Wrapper4BinaryOperator(-, sub);
Wrapper4BinaryOperator(*, mul);
Wrapper4BinaryOperator(/, div);
Wrapper4BinaryOperator(%, mod);
Wrapper4BinaryOperator(&, and);
Wrapper4BinaryOperator(|, _or);
Wrapper4BinaryOperator(^, xor);
Wrapper4BinaryOperator(<<, lsh);
Wrapper4BinaryOperator(>>, rsh);

Wrapper4BinaryAnchOperator(+, add);
Wrapper4BinaryAnchOperator(-, sub);
Wrapper4BinaryAnchOperator(*, mul);
Wrapper4BinaryAnchOperator(/, div);
Wrapper4BinaryAnchOperator(%, mod);
Wrapper4BinaryAnchOperator(&, and);
Wrapper4BinaryAnchOperator(|, _or);
Wrapper4BinaryAnchOperator(^, xor);
Wrapper4BinaryAnchOperator(<<, lsh);
Wrapper4BinaryAnchOperator(>>, rsh);

Wrapper4UnaryAssignOperator(++, inc);
Wrapper4UnaryAssignOperator(--, dec);

Wrapper4BinaryAssignOperator(+=, add);
Wrapper4BinaryAssignOperator(-=, sub);
Wrapper4BinaryAssignOperator(*=, mul);
Wrapper4BinaryAssignOperator(/=, div);
Wrapper4BinaryAssignOperator(%=, mod);
Wrapper4BinaryAssignOperator(&=, and);
Wrapper4BinaryAssignOperator(|=, _or);
Wrapper4BinaryAssignOperator(^=, xor);
Wrapper4BinaryAssignOperator(<<=, lsh);
Wrapper4BinaryAssignOperator(>>=, rsh);

Wrapper4BinaryOperator(==, _eq);
Wrapper4BinaryOperator(!=, _ne);
Wrapper4BinaryOperator(<, _lt);
Wrapper4BinaryOperator(<=, _le);
Wrapper4BinaryOperator(>, _gt);
Wrapper4BinaryOperator(>=, _ge);

template <typename Arg0>
WrapperStart( Operator_Unary_boo, ".boo", Arg0 ) {
   Value arg0(stack[0]);
   WrapperReturn( !is_zero( arg0.get<Arg0>() ) );
};

template <typename Arg0>
WrapperStart( Operator_Unary_not, ".not", Arg0 ) {
   Value arg0(stack[0]);
   WrapperReturn( is_zero( arg0.get<Arg0>() ) );
};

template <typename Arg0, typename Arg1>
WrapperStart( Operator_Binary_brk, ":brk", Arg0,Arg1 ) {
   Value arg0(stack[0]), arg1(stack[1]);
   WrapperReturnLvalue( Arg0, arg0.get<Arg0>() [ arg1.get<Arg1>() ] );
};

} }

#endif // POLYMAKE_PERL_WRAPPERS_H

// Local Variables:
// mode:C++
// c-basic-offset:3
// indent-tabs-mode:nil
// End:
