"""pdbfixer: Fixes problems in PDB files

Protein Data Bank (PDB) files often have a number of problems that must be
fixed before they can be used in a molecular dynamics simulation. The details
vary depending on how the file was generated. Here are some of the most common
ones:

- If the structure was generated by X-ray crystallography, most or all of the 
- hydrogen atoms will usually be missing.
- There may also be missing heavy atoms in flexible regions that could not be
  clearly resolved from the electron density. This may include anything from a
  few atoms at the end of a sidechain to entire loops.
- Many PDB files are also missing terminal atoms that should be present at the 
  ends of chains.
- The file may include nonstandard residues that were added for crystallography
  purposes, but are not present in the naturally occurring molecule you want to
  simulate.
- The file may include more than what you want to simulate. For example, there
  may be salts, ligands, or other molecules that were added for experimental
  purposes. Or the crystallographic unit cell may contain multiple copies of a
  protein, but you only want to simulate a single copy.
- There may be multiple locations listed for some atoms.
- If you want to simulate the structure in explicit solvent, you will need to
  add a water box surrounding it.

PDBFixer can fix all of these problems for you in a fully automated way. You
simply select a file, tell it which problems to fix, and it does everything else.
"""
from __future__ import print_function
import os
import sys
from os.path import relpath, join
from setuptools import setup, find_packages
DOCLINES = __doc__.split("\n")

########################
__version__ = '1.8.1'
VERSION = __version__
ISRELEASED = False
########################
CLASSIFIERS = """\
Development Status :: 5 - Production/Stable
Intended Audience :: Science/Research
Intended Audience :: Developers
License :: OSI Approved :: MIT License
Programming Language :: Python
Programming Language :: Python :: 3
Topic :: Scientific/Engineering :: Bio-Informatics
Topic :: Scientific/Engineering :: Chemistry
Operating System :: Microsoft :: Windows
Operating System :: POSIX
Operating System :: Unix
Operating System :: MacOS
"""


def find_package_data():
    files = []
    for root, dirnames, filenames in os.walk('pdbfixer'):
        for fn in filenames:
            files.append(relpath(join(root, fn), 'pdbfixer'))
    return files

setup(
    name='pdbfixer',
    author='Peter Eastman',
    description=DOCLINES[0],
    long_description="\n".join(DOCLINES[2:]),
    version=__version__,
    license='MIT',
    url='https://github.com/openmm/pdbfixer',
    platforms=['Linux', 'Mac OS-X', 'Unix', 'Windows'],
    classifiers=CLASSIFIERS.splitlines(),
    packages=find_packages(),
    package_data={'pdbfixer': find_package_data()},
    zip_safe=False,
    install_requires=['numpy', 'openmm >= 7.1'],
    entry_points={'console_scripts': ['pdbfixer = pdbfixer.pdbfixer:main']})

