# xpp - A C++11 RAII wrapper for XCB

## Synopsis

XPP is a header only C++11
[RAII](https://en.wikipedia.org/wiki/Resource_Acquisition_Is_Initialization)
wrapper around [X protocol C-language Binding
(XCB)](http://xcb.freedesktop.org). Pointers to dynamically allocated memory,
such as events and errors are wrapped in std::shared_ptr.

Furthermore, interfaces for connection and resource types are provided to
facilitate the creation of custom classes. For convenience, a connection class
and several basic resource type classes are readily available.

XPP makes widespread use of the
[Curiously Recurring Template Pattern (CRTP)](https://en.wikibooks.org/wiki/More_C++_Idioms/Curiously_Recurring_Template_Pattern)
to avoid overhead through dynamic dispatch. Hence, most interfaces are
implicitly defined.

## Prerequisites

  * Python 3
  * GCC >= 4.8 (or Clang >= 3.3, untested)
  * libxcb

## Quick Start

```
  git clone https://github.com/jrk-/xpp
  cd xpp
  make
  make examples
  cd src/examples
  for demo in demo_*; do ./${demo}; done
```

## Documentation

### General

The bindings can be generated by calling `make` in the top level directory. If
this fails, check the [`XCBGEN`](include/proto/Makefile#L38) and
[`PROTODIR`](include/proto/Makefile#L39) variables in
[include/proto/Makefile](include/proto/Makefile). These need to point to the `xcbgen`
python package and the xml protocol description respectively.

Once the bindings are generated they can be used by including
[include/xpp.hpp](include/xpp.hpp). If an extensions is required, it needs to be
included additionally. For example, the RandR extension is available through
`proto/randr.hpp`, the Damage extension through `proto/damage.hpp`, etc.

Recent (and working) examples can be found in [src/examples](src/examples).
To compile them, call `make examples` in the `xpp` directory or just `make` in
[src/examples](src/examples).

### Requests

Requests obey this naming scheme: `xpp:: ExtensionName :: RequestName`.

##### Examples:

Core X protocol:
`MapWindow`: `xcb_map_window{,_checked}` -> `xpp::x::map_window{,_checked}`
`InternAtom`: `xcb_intern_atom{,_checked}` -> `xpp::x::intern_atom{,_unchecked}`

RandR protocol:
`SelectInput`: `xcb_randr_select_input{,_checked}` -> `xpp::randr::select_input{,_checked}`
`QueryVersion`: `xcb_randr_query_version{,_unchecked}` -> `xpp::randr::query_version{,_unchecked}`

##### Default Parameter

All `xcb_timestamp_t` parameters are alternatively available with a default
value of `XCB_TIME_CURRENT_TIME`.

##### Parameter Lists

Requests which take a list of values as parameters can be used with any STL
container by passing in Iterators. Example:

```
std::string string_example = "example string";
// std::list<char> list_example = { 'a', 'b', 'c' };
// std::map<int, char> map_example = { {0, 'a'}, {1, 'b'}, {2, 'c'} };
xpp::x::change_property_checked(c, XCB_PROP_MODE_REPLACE, window,
                                atom, XCB_ATOM_STRING, 8,
                                string_example.begin(), string_example.end());
                                // list_example.begin(), list_example.end());
                                // for associative containers the value (std::pair<..>::second_type) will be used
                                // map_example.begin(), map_example.end());
```

### Replies

XCB returns replies only when they are explicitely queried. With XPP this is not
necessary anymore, because the operators for accessing the reply are overloaded.

For example, getting the reply for the `InternAtom` request is as simple as this:

```
auto reply = xpp::x::intern_atom(connection, true, "MY_ATOM_NAME");
// do some other stuff ..
// latency hiding is still effective, because the call to
// xcb_intern_atom_reply happens but now in operator->()
xcb_atom_t atom = reply->atom;
```

#### Member Accessors

##### Simple Types

Primitive types like `xcb_window_t`, `xcb_atom_t`, etc. can be accessed either
directly through the overloaded `operator->()` or via a method which has the
same name as the member. These methods are templated with a default template
type of the native type. Any type which is default constructible from the native
type or a connection and the native type can be specified as template argument.

Examples:

```
xcb_window_t w1 = reply->member;
xcb_window_t w2 = reply.member(); // default template parameter is xcb_window_t
xpp::window w3 = reply.member<xpp::window>();
```

##### List Types

Lists (e.g. the result for `QueryTree`) are accessible through iterators. The
value type is templated, with the default being the native data type.

Example:

```
auto tree = xpp::x::query_tree(c, window);

// default template type: xcb_window_t
for (auto && child : tree.children()) {
  // child has type xcb_window_t
}

// xpp::window is constructible with a connection and xcb_window_t
// other types which are default-constructible with either the value type
// (e.g.  xcb_window_t) or a connection & the value type are possible, too
for (auto && child : tree.children<xpp::window>()) {
  // child has type xpp::window
}
```

Caveat: Some requests (in particular `GetProperty`) return an untyped array of
bytes (`void *`). To access the desired data type, a template type must be
specified. For constructible types a type trait must be implemented, like so:

```
struct my_type {
  my_type(const xcb_window_t &);
  // ..
};

namespace xpp { namespace generic {
struct traits<my_type> {
  typedef xcb_atom_t type;
};
}; }; // namespace xpp::generic
```

### Errors

XCB offers four different variants of request functions.

##### Requests without a reply:

* Error delivered through event queue: `xcb_void_cookie_t xcb_request(...)`

* Error can be checked immediately with `xcb_request_check(xcb_connection_t *, xcb_void_cookie_t)`: `xcb_void_cookie_t xcb_request_checked(...)`

##### Requests with reply:

* Error can be checked when getting the reply:
`xcb_request_reply_t * xcb_request_reply(xcb_connection_t *, xcb_request_cookie_t, xcb_generic_error_t **)`:
`xcb_request_cookie_t xcb_request(...)`

* Error delivered through event queue: `xcb_request_cookie_t xcb_request_unchecked(...)`

For more information on this, refer to [xcb-requests (3)](http://www.x.org/releases/current/doc/man/man3/xcb-requests.3.xhtml).

With xpp errors are either thrown as `std::shared_ptr<xcb_generic_error_t>` or
typed as `xpp:: extension ::error:: error_type`, e.g. `xpp::x::error::value`.

The latter are based upon `xpp::generic::error` (which inherits from
`std::runtime_error`) and come with a textual error description which is
accessible through the `what()` method.

For typed errors it is necessary to use a connection class which implements the
appropriate error dispatching. The supplied `xpp::connection` class already does
this. If no error dispatcher are available (e.g. when used with
`xcb_connection_t *`), then a simply `std::shared_ptr<xcb_generic_error_t>`
will be thrown.

### Events

Events returned by the event producing methods (`wait_for_event`,
`poll_for_event`, etc.) from `xpp::core` and `xpp::connection` are encapsulated
as `std::shared_ptr<xcb_generic_event_t>`.

For additional convenience typed events are available. An event type is based on
`xpp::generic::event`. The general structure for a typed event is

`xpp::` Extension `::event::` EventName

Examples:

```
xpp::x::event::key_press
xpp::randr::event::notify
xpp::damage::event::notify
```

Events can be converted from `std::shared_ptr<xcb_generic_event_t>` to a typed
event by either using an event dispatcher functor (e.g.
`xpp::x::event::dispatcher`) or by using the event registry described below.

##### Registry

The event registry `xpp::event::registry<Connection, Extensions ...>` can be
used to connect events and event handlers.

First, a registry object for the desired `Connection` type and `Extensions` is
necessary.

Then, arbitrary objects, which implement the `xpp::event::sink<..>` interface
need to be attached for event handling by calling the `attach()` method.
It takes two parameters. The first one specifies the priority, in case there are
more than one event handler for this event. Handlers with lower priorities are
called first. The second one is a pointer to an object which implements the
`xpp::event::sink<..>` interface.

For a detailed example, take a look at this [demo](src/examples/demo_01.cpp).

### Interfaces

Interfaces for creating custom types are available.

##### <a name="interface-connection"></a>Connection

For every extension a "connection" interface, called
`xpp:: ExtensionName ::interface<typename Derived, typename Connection>`
is available.

These encapsulate every request for a particular extension. The `Derived`
template parameter specifies the class which wants to derive from the interface.
The `Derived` class must provide a method `Connection connection();`.

Examples:

```
xpp::x::interface<typename Derived, typename Connection>
xpp::randr::interface<typename Derived, typename Connection>
xpp::damage::interface<typename Derived, typename Connection>
etc.
```

For a customizable default implementation, take a look at the `xpp::connection`
class described [here](#default-type-connection).

##### Resources

In addition, interfaces for basic resource types like `xcb_window_t`,
`xcb_atom_t`, `xcb_gcontext_t`, etc. are available.

Again, the naming scheme follows the format
`xpp:: ExtensionName :: XidType <typename Derived, typename Connection>`

Despite the `connection()` method described [here](#interface-connection),
`Derived` needs to implement a `resource()` method which returns a xid which
will be passed as parameter to the encapsulated requests.

Examples:

```
xpp::x::window<typename Derived, typename Connection>
xpp::randr::output<typename Derived, typename Connection>
xpp::render::glyphset<typename Derived, typename Connection>
etc.
```

### Default Types

##### <a name="default-type-connection"></a>Connection

`xpp::connection<Extensions ...>` provides a default
implementation of the [core connection methods](include/core.hpp), the core
X protocol and error handling facilities. In addition, it is implicitly
convertible to `xcb_connection_t *`, hence it can be used seamlessly with XCB
functions. The connection can be augmented with additional extension methods, by
specifying the desired extensions as template parameters.

Example:

`typedef xpp::connection<xpp::randr::extension, xpp::damage::extension> my_connection;`

##### Resources

For the basic resource types like `Drawable`, `Window`, `Pixmap`, `Atom`,
`Colormap`, `Cursor`, `Font`, `Fontable` and `GContext` wrapper types exist.
They are named `xpp::drawable`, `xpp::window`, etc.

Each is based upon xpp::generic::resource and provides the core X protocol
interface for the encapsulated resource type. If the resource can be acquired
from the X server (e.g. with `CreateWindow`) then a named constructor is
available (e.g. `create_window` for `xpp::window`).

Resources acquired through the named constructors are reference counted. When
their lifetime expires, the resource handle will automatically be freed on the
server. No call to destroy or free functions is necessary.
