/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2001
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/driver.hh>
#include <gecode/int.hh>
#include <gecode/minimodel.hh>

using namespace Gecode;

// Instances
extern const int* pi[];
extern const unsigned int n_pi;

/// Specifications for photo example
class PhotoSpec {
protected:
  /// Array of data
  const int* data;
public:
  /// Initialize
  PhotoSpec(int i) : data(pi[i]) {}
  /// Return number of people
  int people(void) const {
    return data[0];
  }
  /// Return number of preferences
  int preferences(void) const {
    return data[1];
  }
  /// Access preference \a p with position \a i
  int preference(int p, int i) const {
    return data[2+2*p+i];
  }
};

/**
 * \brief %Example: Placing people on a photo
 *
 * A group of people wants to take a group photo. Each person can give
 * preferences next to whom he or she wants to be placed on the
 * photo. The problem to be solved is to find a placement that
 * violates as few preferences as possible.
 *
 * \ingroup Example
 *
 */
class Photo : public IntMinimizeScript {
protected:
  /// Photo specification
  const PhotoSpec spec;
  /// Person's position on photo
  IntVarArray pos;
  /// Number of violated preferences
  IntVar violations;
  /// Random number generator for LNS
  Rnd rnd;
  /// Relaxation probability
  double p;
public:
  /// Branching to use for model
  enum {
    BRANCH_NONE,    ///< Choose variables from left to right
    BRANCH_DEGREE,  ///< Choose variable with largest degree
    BRANCH_AFC_SIZE ///< Choose variable with largest afc over size
  };
  /// Actual model
  Photo(const SizeOptions& opt) :
    IntMinimizeScript(opt),
    spec(opt.size()),
    pos(*this,spec.people(), 0, spec.people()-1),
    violations(*this,0,spec.preferences()),
    rnd(opt.seed()), p(opt.relax())
  {
    // Map preferences to violation
    BoolVarArgs viol(spec.preferences());
    for (int i=0; i<spec.preferences(); i++) {
      int pa = spec.preference(i,0);
      int pb = spec.preference(i,1);
      viol[i] = expr(*this, abs(pos[pa]-pos[pb]) > 1);
    }
    rel(*this, violations == sum(viol));

    distinct(*this, pos, opt.ipl());

    // Break some symmetries
    rel(*this, pos[0] < pos[1]);

    switch (opt.branching()) {
    case BRANCH_NONE:
      branch(*this, pos, INT_VAR_NONE(), INT_VAL_MIN());
      break;
    case BRANCH_DEGREE:
      branch(*this, pos, tiebreak(INT_VAR_DEGREE_MAX(),INT_VAR_SIZE_MIN()),
             INT_VAL_MIN());
      break;
    case BRANCH_AFC_SIZE:
      branch(*this, pos, INT_VAR_AFC_SIZE_MAX(opt.decay()), INT_VAL_MIN());
      break;
    }
  }
  /// Slave function for restarts
  bool slave(const MetaInfo& mi) {
    if ((mi.type() == MetaInfo::RESTART) &&
        (mi.restart() > 0) && (p > 0.0)) {
      const Photo& l = static_cast<const Photo&>(*mi.last());
      relax(*this, pos, l.pos, rnd, p);
      return false;
    } else {
      return true;
    }
  }
  /// Constructor for cloning \a s
  Photo(Photo& s) :
    IntMinimizeScript(s), spec(s.spec), rnd(s.rnd), p(s.p) {
    pos.update(*this, s.pos);
    violations.update(*this, s.violations);
  }
  /// Copy during cloning
  virtual Space*
  copy(void) {
    return new Photo(*this);
  }
  /// Print solution
  virtual void
  print(std::ostream& os) const {
    os << "\tpos[] = " << pos << std::endl
       << "\tviolations: " << violations << std::endl;
  }
  /// Return solution cost
  virtual IntVar cost(void) const {
    return violations;
  }
};

/** \brief Main-function
 *  \relates Photo
 */
int
main(int argc, char* argv[]) {
  SizeOptions opt("Photo");
  opt.solutions(0);
  opt.size(1);
  opt.iterations(10);
  opt.ipl(IPL_BND);
  opt.relax(0.7);
  opt.time(10 * 1000);
  opt.branching(Photo::BRANCH_DEGREE);
  opt.branching(Photo::BRANCH_NONE, "none");
  opt.branching(Photo::BRANCH_DEGREE, "degree");
  opt.branching(Photo::BRANCH_AFC_SIZE, "afc");
  opt.parse(argc,argv);
  if (opt.size() >= n_pi) {
    std::cerr << "Error: size must be between 0 and " << n_pi - 1
              << std::endl;
    return 1;
  }
  IntMinimizeScript::run<Photo,BAB,SizeOptions>(opt);
  return 0;
}

/// Small Photo example
const int small[] = {
  /// Number of people on picture
  5,
  /// Number of preferences
  8,
  /// Array of preferences
  0,2, 1,4, 2,3, 2,4, 3,0, 4,3, 4,0, 4,1
};

/// Large Photo example
const int large[] = {
  /// Number of people on picture
  9,
  /// Number of preferences
  17,
  /// Array of preferences
  0,2, 0,4, 0,7, 1,4, 1,8, 2,3, 2,4, 3,0, 3,4,
  4,5, 4,0, 5,0, 5,8, 6,2, 6,7, 7,8, 7,6
};

/*
 * The following instances have been randomly generated for
 * given number of persons \a p:
 *  - the number of preferences is chosen uniformly between 1 and p*p/4.
 *  - the preferences are chosen uniformly without duplicates.
 */

const int p_10_0[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  12,
  /// Array of preferences
  2,9, 8,0, 6,7, 6,5, 9,3, 2,3, 6,2, 2,5,
  3,6, 0,1, 1,9, 4,8
};
const int p_10_1[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  18,
  /// Array of preferences
  3,6, 9,3, 6,9, 7,9, 1,1, 0,1, 7,5, 9,2,
  6,1, 5,7, 2,2, 8,4, 5,2, 1,9, 5,8, 4,5,
  3,4, 0,9
};
const int p_10_2[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  15,
  /// Array of preferences
  5,6, 7,8, 0,6, 5,8, 8,2, 1,9, 2,1, 9,8,
  7,2, 7,7, 7,1, 2,7, 5,0, 8,1, 3,8
};
const int p_10_3[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  25,
  /// Array of preferences
  9,3, 0,3, 7,5, 7,0, 9,2, 5,3, 9,7, 4,7,
  7,4, 3,1, 3,7, 7,2, 6,8, 1,6, 8,6, 9,4,
  9,8, 5,1, 2,0, 8,3, 9,5, 3,8, 5,7, 6,4,
  2,9
};
const int p_10_4[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  4,
  /// Array of preferences
  7,3, 2,8, 1,0, 4,2
};
const int p_10_5[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  12,
  /// Array of preferences
  3,8, 3,3, 1,2, 5,4, 4,6, 8,8, 2,7, 0,8,
  3,9, 3,6, 2,2, 2,5
};
const int p_10_6[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  1,
  /// Array of preferences
  1,4
};
const int p_10_7[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  8,
  /// Array of preferences
  5,3, 7,0, 6,7, 5,4, 9,0, 9,3, 7,9, 2,6
};
const int p_10_8[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  10,
  /// Array of preferences
  5,4, 4,9, 4,3, 6,4, 2,2, 9,4, 3,4, 0,8,
  0,6, 3,9
};
const int p_10_9[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  8,
  /// Array of preferences
  0,1, 5,7, 4,7, 5,6, 2,9, 0,9, 3,8, 8,3
};
const int p_10_10[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  17,
  /// Array of preferences
  0,8, 0,7, 2,3, 9,2, 5,4, 4,4, 0,0, 3,8,
  8,9, 6,0, 8,1, 3,4, 5,0, 2,2, 9,0, 0,6,
  6,3
};
const int p_10_11[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  2,
  /// Array of preferences
  5,7, 8,8
};
const int p_10_12[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  24,
  /// Array of preferences
  6,3, 6,8, 0,8, 1,6, 9,0, 5,9, 4,3, 3,4,
  7,1, 7,2, 2,6, 6,5, 3,6, 2,8, 2,0, 5,8,
  1,5, 2,4, 2,7, 8,4, 8,1, 5,5, 4,2, 3,7
};
const int p_10_13[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  24,
  /// Array of preferences
  2,7, 9,0, 3,1, 6,9, 4,1, 1,0, 7,5, 2,2,
  8,8, 4,0, 6,8, 9,5, 5,1, 6,1, 8,4, 1,5,
  0,2, 9,8, 4,2, 2,9, 5,3, 3,6, 5,2, 9,3
};
const int p_10_14[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  4,
  /// Array of preferences
  3,2, 8,3, 2,3, 4,0
};
const int p_10_15[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  19,
  /// Array of preferences
  0,1, 9,6, 3,3, 2,7, 0,7, 4,1, 2,0, 6,6,
  8,3, 3,1, 1,1, 7,2, 3,0, 4,7, 6,4, 8,0,
  8,5, 8,2, 5,7
};
const int p_10_16[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  11,
  /// Array of preferences
  1,6, 0,7, 2,2, 4,9, 8,3, 3,6, 5,2, 0,9,
  7,4, 6,2, 8,8
};
const int p_10_17[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  17,
  /// Array of preferences
  5,8, 4,1, 5,5, 7,0, 7,3, 2,5, 4,3, 3,7,
  9,4, 0,9, 7,8, 9,9, 6,8, 2,2, 0,0, 6,1,
  7,9
};
const int p_10_18[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  5,
  /// Array of preferences
  4,9, 6,8, 1,2, 2,7, 6,6
};
const int p_10_19[] = {
  /// Number of people on picture
  10,
  /// Number of preferences
  22,
  /// Array of preferences
  9,3, 1,3, 4,8, 3,9, 5,9, 3,1, 8,5, 0,7,
  1,8, 2,6, 9,9, 8,8, 1,6, 9,4, 2,2, 6,9,
  9,8, 8,1, 3,0, 9,1, 2,1, 5,6
};

const int p_20_0[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  78,
  /// Array of preferences
   5, 2, 15, 8, 12, 4, 10,18, 19, 4,  7,10,  4,10, 14,10,
   8, 8,  7, 0,  1,12,  9,10, 16, 5, 15, 3,  6, 6,  0, 7,
   9,19,  0, 6,  2, 2,  9,17, 15, 5,  4, 7,  6, 1, 13, 2,
  14,18, 18,18, 16,17, 19, 8, 17,16, 14,13, 15, 0, 18, 7,
  19,15,  8, 1,  6, 5,  3, 8,  4, 0, 11,19,  9,11,  6,17,
  15, 2,  0,19,  6, 0, 11, 2, 10, 1,  4, 9,  4,16,  2,15,
   1,14,  5,17, 15, 9,  4, 1, 17,11,  1,13, 10, 8,  2, 7,
   4, 4,  2,16, 10, 4,  5, 1,  0, 5,  3, 4,  2,18,  9, 4,
  16, 4, 14,16, 17, 1, 17,12,  3, 3,  9, 8, 12, 9,  0,14,
  18, 5,  8, 9,  7, 1, 15,14, 19,18,  2, 0
};
const int p_20_1[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  54,
  /// Array of preferences
   4,14, 14,13, 18, 6, 11,10,  7, 2, 16, 1, 10,18,  4,19,
   0, 8,  6, 6,  6, 2,  1,14,  5,18,  1, 5, 13, 5, 18, 2,
  16,11, 19,13,  8, 4, 16,19,  6,16, 11,18, 19, 3,  7, 6,
   1,13, 18,10,  6,11,  9, 5, 19,19,  7, 9, 12,16,  8, 0,
   0,17, 11, 3, 14, 9, 12,10,  7,12, 15, 4,  3, 8, 14,10,
   8, 3,  9, 9,  4, 2, 14, 5,  0,14, 10,13,  5, 0, 10, 5,
  12,19,  1,12, 18,14,  4,11, 15, 3,  6,18
};
const int p_20_2[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  26,
  /// Array of preferences
  12, 9,  8,19, 12, 0,  4, 4,  2,10, 13, 7, 12,13,  9,15,
  10,19,  1, 2, 15, 8, 16, 7, 18, 7,  4,11,  1, 6, 13,12,
   5, 0, 14,19,  3,17, 10, 9, 11, 1,  6,15,  8, 1, 16,18,
   7, 1, 14,18
};
const int p_20_3[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  51,
  /// Array of preferences
  18,12,  2,11,  8, 5, 12,17, 14, 8,  1, 6,  9, 5,  5, 5,
  11, 3, 13,17,  9,19,  4, 0,  0, 2,  1,19,  3,12,  1, 8,
  13, 2, 14, 1, 14, 3,  0, 6, 12, 0, 16, 2,  4, 4,  8,14,
  11, 8,  9, 7, 11,10,  1,11,  4, 5,  8,11, 18, 0,  3, 9,
  14,18, 11,16, 13,11, 18,18, 18, 2,  9, 2,  8, 1, 10, 9,
   7,19,  2, 5,  3, 8,  0,11, 15, 4, 12, 9,  8,15, 10,16,
  17,16,  8,17, 12,16
};
const int p_20_4[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  23,
  /// Array of preferences
  16, 2, 19, 1, 15, 1, 16,17, 13, 1, 13, 7, 17,16, 15,19,
   2,18, 15, 7,  3, 5,  8, 9,  6,12, 16, 7,  6,15,  0, 0,
   5,15,  5, 0,  3, 8, 14, 1, 15,13, 11,19,  1,17
};
const int p_20_5[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  32,
  /// Array of preferences
   5, 7, 15, 3, 16, 6,  5,15,  1,10,  0, 1,  0, 6, 10, 1,
   3, 4,  0,12,  2, 1,  4,13, 19,15, 11, 1, 17, 5,  2, 0,
   0, 4,  9,16, 17,13,  4, 1,  9,13,  0,16, 15,19,  4, 3,
   8,12,  9,12,  0, 7,  7,15,  5, 6, 12, 1, 19, 2,  4,16
};
const int p_20_6[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  21,
  /// Array of preferences
   5, 9,  7,14, 19, 6,  1,18,  5, 8, 18, 9, 12,14, 17, 8,
  10, 2,  2, 4, 18,12,  5,19, 15,12, 12, 8,  2,13,  0,15,
   5,13, 11,16, 16,16, 15, 9, 18,17
};
const int p_20_7[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  90,
  /// Array of preferences
  18, 0, 11, 6, 12, 8,  1,14,  9,10,  0,16,  7,19,  9, 4,
   8, 7,  5, 6, 13,17, 11, 2,  1,10, 15, 9,  1, 9,  6,15,
  18,17, 14,14, 11,14,  7, 0,  2, 9, 18,13, 12, 3, 14, 4,
   2,19, 15, 3,  6,12, 15, 4,  7,12, 14, 1,  7, 4,  6, 2,
   4,19, 18,16, 10, 9,  6, 3, 17,15,  1,17, 15, 5, 18,10,
  13,12,  5, 9, 10,13,  3,15,  0,17, 10, 8,  5,13,  7,11,
  12,15,  8, 3, 16, 8,  3, 3, 19,18, 19, 1,  4,12,  0,11,
  17, 6, 14, 8, 11, 7, 19, 7,  4, 4,  3,17, 15, 7,  9, 6,
   1,18,  3,14,  2,14, 18,12,  8, 4,  3, 6,  5, 7,  0,15,
   9, 2, 17,11,  0, 9, 16, 9, 11, 0, 12, 1, 10,10,  5,14,
  18, 6,  8,12,  0,18, 10,11, 16,12, 11, 3, 11, 9,  6, 7,
  17,18, 12,17
};
const int p_20_8[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  20,
  /// Array of preferences
  15,13, 16,14, 13, 2, 18,12,  3,11,  3,18,  8, 0,  9,19,
  17,19,  1, 3, 15,10, 19, 5, 19,17,  4,13, 15,15,  9, 4,
  16, 2,  8,11,  5,12,  6,15
};
const int p_20_9[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  26,
  /// Array of preferences
  16,15, 14, 9,  8,17,  2,11,  5, 0,  9,14, 10,19,  8, 2,
   1, 9, 10, 3, 15,12, 10,18, 10,10,  3,15, 18,10,  1, 2,
   7, 1,  6, 9, 18,17, 19, 0,  8,11, 19,10,  2,12, 12,19,
  15, 6, 15,15
};
const int p_20_10[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  46,
  /// Array of preferences
   3,15, 15, 9,  8, 4,  0,12,  0, 8, 11, 2, 13, 7,  2, 7,
   7, 1, 13,12,  3,11, 14, 4,  7, 4,  1, 1, 10, 6,  4, 4,
  15, 4, 17, 3, 15,18,  9, 9, 19, 3,  9,17,  7,17,  3,14,
  12, 4, 14, 9, 16,14, 18,13,  2, 5, 17,14,  8,11, 11, 7,
  10, 5,  6,15,  7,13,  1, 8, 13,10,  5,13, 17,17,  3, 0,
  15,19,  0,16,  5,11, 12,14, 13, 5, 17, 4
};
const int p_20_11[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  45,
  /// Array of preferences
  11, 9, 16, 6,  1,15,  9, 2, 11, 7,  9,13, 17,16, 17,19,
  11,11,  2,16, 11,19,  3, 8,  7, 0,  1, 6, 15,12,  3, 2,
   4,18, 12,17, 17,17, 16, 5,  6, 7, 19,19,  3, 5, 12,18,
  19, 4, 11,17,  3, 0,  2,18,  6, 0,  2,19,  5,18,  8,11,
   9, 5, 16,18, 16,19,  2,11,  7,13,  2,12,  6,10, 19,13,
  14, 4,  9,14, 15,19,  0,15, 18, 8
};
const int p_20_12[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  40,
  /// Array of preferences
   7, 4,  7, 0, 12,13, 16, 9, 10, 5,  7,12, 13,12, 19, 7,
  18, 0, 18, 9,  4, 0,  3,12,  1, 3, 16,15, 17, 9,  7, 2,
   9,15,  9, 5, 14, 3, 16, 0,  5, 8, 19, 0, 19,10, 18,15,
   7,19, 18,19, 19,18,  4, 2,  6,11,  7, 3,  9,12, 14, 7,
  17,15, 18,14, 10,14,  3, 7,  9, 8, 19,15,  4, 8, 17, 6
};
const int p_20_13[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  71,
  /// Array of preferences
  18,13, 16,14, 18,18,  6, 6, 13,19, 19,11,  4,13, 18, 4,
  18, 1, 10, 2, 10,18, 12, 5, 17, 2, 18, 8, 10,12,  8,11,
   0,16, 15, 0,  9, 4, 19, 8, 11, 0,  9, 6,  4,19, 15, 8,
   2,14, 10, 3, 14, 4,  9,15,  2,16, 12,18, 17,13, 15,19,
   9,17, 13,16,  9, 7,  0,18, 17,18, 16, 4, 18,11, 12, 9,
   4, 7, 10, 8,  1, 6,  8, 1, 12,13, 16, 5,  3, 9,  6,12,
  11, 5, 14, 6, 15,12, 10, 1,  6, 9,  7,15,  5, 9,  2, 9,
   1, 2,  9, 1,  3, 0,  5,17, 15,15, 18, 2,  4, 5,  2,19,
   1, 5, 11, 8, 16, 7,  5,10,  7, 1,  7, 0,  5, 5
};
const int p_20_14[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  7,
  /// Array of preferences
  17, 9, 13,11, 13,10, 14,12, 16,16,  6,15,  6,17
};
const int p_20_15[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  60,
  /// Array of preferences
   9, 2, 13,18,  8,14, 14,12,  6, 5,  6, 2, 10, 6,  3, 6,
   4, 5, 16,17, 17,11,  6,12, 15,14,  7, 5,  4,12,  4, 4,
   9,16,  8, 5, 18, 5, 15,10, 16, 9,  4,14, 18,15,  2,10,
  14, 9,  5,18,  1,12, 10, 5,  1,18,  7,14, 12, 3,  7,12,
  12, 7,  2, 4,  9,17, 13, 0,  8,12,  7,11,  3, 1, 19,17,
  10, 8, 15, 3, 13, 7,  1,17,  0,13,  1, 6, 14,19,  9,15,
   9,11,  7, 3,  8, 8, 14,17, 19,11, 14, 4, 14,14, 14,13,
  16, 5,  4, 6,  7, 4,  9, 1
};
const int p_20_16[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  24,
  /// Array of preferences
  13,16, 13, 6, 18,13,  2, 1,  2, 5, 16, 3, 10,18, 13,12,
   8,18,  7,16,  2,13, 11, 6, 17,15,  0, 5,  8,19, 12,10,
   5,14, 14,16,  0,18, 12, 0,  8,13,  2,10,  1, 3,  8, 5
};
const int p_20_17[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  23,
  /// Array of preferences
   8, 7,  2,11,  4, 3, 17, 7, 15,12,  6,10,  9,10,  5, 7,
   5,16, 15, 9,  1,13, 19, 4,  8, 3, 13, 9,  4, 2,  7, 8,
   7, 9, 13, 7,  9, 8, 11, 2, 16, 1,  5, 6, 18, 8
};
const int p_20_18[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  65,
  /// Array of preferences
  12, 7,  4,16,  6, 7, 13, 6, 16, 4,  9, 0, 17, 0,  3,10,
   6,14, 19,17, 13,11,  2,17,  7, 7,  6, 0,  3, 4,  0,10,
   7, 1,  3,17,  7,10, 18,14,  6,19, 18, 1,  5, 7,  0, 4,
  15,10, 16,16,  8,16,  9,13, 19,14, 13,14, 19, 2,  1, 7,
   9, 1, 16,19,  8, 4,  4, 8,  3,16, 11,16, 14,18, 14,14,
   3, 2,  5, 4, 10, 1, 11,12, 13,13, 13, 2,  0,12, 11, 3,
  19, 7,  0, 8, 12,15, 13, 4,  6,10,  2, 4, 11, 7,  8, 6,
   5,14, 10,14, 14,12, 14,19, 11, 9,  5,11, 12, 1,  3,11,
  17,15
};
const int p_20_19[] = {
  /// Number of people on picture
  20,
  /// Number of preferences
  72,
  /// Array of preferences
   3, 1,  3,18, 19, 0, 19,16,  4,14, 11, 0, 15,12, 19, 7,
  18, 9, 14,12,  7,18, 17,12, 12, 0,  4, 2, 10,13,  2,19,
  18, 3, 10,15, 17,18,  4,13,  6,16,  8, 9, 16,17,  9, 8,
   2, 9,  5,16, 13, 0,  1, 5,  0, 2,  6, 9,  5,14,  1, 0,
  19,14, 19, 2, 12,10,  7,19, 12, 6,  0,17,  5,18,  2,18,
   4, 0,  0, 0, 10, 7, 11, 1,  2,10,  8,16,  0, 1, 18,13,
  11, 7,  0, 4,  6,12, 15,10,  4, 3,  8,13, 14,15,  6,11,
   9, 5,  7,10,  3, 7,  5, 4,  7, 2,  0, 3, 14,10,  7, 3,
  18, 7, 12, 7,  9, 0, 17, 8, 11, 8,  1,10,  8,15, 18,18
};

const int p_30_0[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  48,
  /// Array of preferences
   9,19,  3,10,  2, 8, 24,22,  6,27,  0,24, 29,18,  9, 1,
   7,14, 11,24, 17, 7, 11,18,  7,21,  2, 3,  5,13,  5, 5,
  22, 3, 16,12, 26, 8, 24, 6,  9,18, 20,12,  8,29, 12, 1,
   1,14,  8,25, 18,21, 29, 1, 18, 3, 24, 5,  2, 4, 22,13,
  23,28,  6, 9, 15,14, 17, 3, 21,13,  2,12, 25, 6, 26,19,
  29,28,  6, 5, 24, 3,  2, 9, 13,10,  3,21, 18,14, 29,15
};
const int p_30_1[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  71,
  /// Array of preferences
  18,17, 14,27, 23, 5, 21, 2, 18, 6, 21, 3, 11,14, 13,18,
  14, 5, 10, 7, 25, 7, 29,22, 16,18, 18,12, 11,11, 24,16,
   1,22, 26,28,  9, 8,  1,26, 20, 0, 23, 3,  0,23, 20,27,
  29, 0, 23,10, 28,10,  6,23, 22,26, 23,27, 28, 0, 19, 1,
  25, 9, 21,21,  6,10, 17,29, 10, 3, 23, 2, 20, 6, 24,25,
   8,18,  2, 6,  2,18,  9,19, 21,19, 21, 9, 14, 2, 18,23,
   6, 8,  0, 9, 24, 4, 11,27, 13, 7, 20,18,  1,20, 19,22,
  14,13, 11,25, 16, 4, 13,14,  0,12, 15,10, 19, 0, 18,19,
  11, 2, 17, 9, 22,22, 24,27, 27,17, 10,29, 16, 6
};
const int p_30_2[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  173,
  /// Array of preferences
  24,22, 24,13,  6, 7,  7,29, 16, 3, 16,15, 17,15, 25, 0,
   7,18, 19,11, 27,16, 21,19,  5,23, 21,16, 27,14,  7,20,
  15, 6,  8,19, 24,19, 10,25, 20,17, 29,11, 25,16, 17,24,
  16,24, 28, 9, 10,22, 22,28, 13,24,  5, 5,  9, 1, 18,28,
  29, 4,  1,18,  3,13, 15,12, 18,20, 23, 1,  3, 8,  8,20,
   6,23, 29, 1,  4, 6, 20,29,  6,18, 25,13, 22,12, 26,17,
  17,27, 28,20,  7,17, 21,28, 26,24,  2,22, 11,28,  2,16,
  20,28, 11,12,  9, 7, 15,16,  7,19,  4,21,  1,24, 13,11,
   0, 6,  0, 0,  6,22,  2, 7,  0,20,  6,29, 28,17,  7,25,
  16, 9, 25, 9, 22,25, 13, 0, 14,22, 23,27,  2, 1,  9,21,
  13,18,  6,20, 22,16,  0,12, 29,29,  3,18, 24,10,  9,18,
   7,22,  3,25, 25, 4, 26, 1, 26,25, 22,24,  4,25, 26,13,
  11,21,  4, 2, 27,19, 29,17, 10,26, 22, 7, 10, 2,  8,15,
  22,23, 14, 0, 23,17,  0, 4,  8,11, 19,22, 19, 7, 16,19,
   9,19,  5,10, 16, 8, 15,15, 26,27, 14, 1,  5,19,  1,14,
  14, 5, 20,20, 18,22,  7,12, 21,21, 17,12, 12, 9, 12,15,
   7, 2, 24, 0, 17,11,  0, 7, 28,16,  9,17, 18,15, 11, 1,
  26,26,  9, 5, 26, 7,  4,12, 17,29,  6,12, 19,27, 17,22,
   2,21, 16,10,  4, 8,  5, 2, 27,23,  0,24, 26,20, 27, 6,
  17, 0, 19,20, 11,18, 15,19,  1,20,  7,28, 15,27, 15,24,
   8, 9, 20, 0, 12,20,  6,27, 23,24, 13, 9, 10, 6, 24, 8,
  15, 4, 28, 0,  5, 7, 21, 4, 20,18
};
const int p_30_3[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  99,
  /// Array of preferences
  25, 4, 28, 3, 27,21, 10,15,  9, 1,  7,12,  3,11, 20, 4,
  24,13, 29,28, 26,16, 23,28, 10,19,  0,19, 26,20, 23, 3,
  28,11, 19,19, 15, 8,  1,16, 27,14,  9,17,  8,13,  3, 0,
  13,20, 10, 2, 23,29,  8, 6, 16, 2, 11, 1,  4,15, 25,14,
  18, 4, 14, 4, 16,26,  5, 7, 21,25, 14,13,  7,25, 14,20,
  19,24, 16,10, 10,26, 24,23, 26,19, 23,17, 13, 3,  0,29,
  12,20,  2,24, 18, 1, 11,26,  3,24,  7, 8,  6, 6, 10,21,
   7,18,  0,13,  4,12,  7,20,  3,17, 23,20,  4,27, 29, 2,
  22,23, 24,26, 10,13, 27,27,  6,25,  0,26, 15, 0, 27, 4,
  13,18,  8,21,  2,25, 22,10, 11,25, 13,23, 17,19, 28,18,
   0, 9, 19, 9, 14,24, 26,17, 29,21, 10,22, 12,14, 29, 1,
  12,25, 25,18, 24, 7, 21,18, 18, 8,  4, 8, 16,24, 23, 9,
  25,17,  8, 5, 22, 6
};
const int p_30_4[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  66,
  /// Array of preferences
  19,28,  8,15, 13,20,  7, 7,  3,12, 27, 2, 22,19, 26, 6,
  20, 9, 22,14, 29,26,  6,29, 11, 7, 23, 4,  1,14, 22,23,
  17, 2, 14,29, 19,11, 15, 2, 20,21,  9,17, 27, 9,  5,24,
   1,20, 10, 7, 21,29, 24,16, 24,24,  4,22, 25, 1, 10,27,
   7,25, 12, 0, 25,15, 27,16, 28,12, 25, 9,  0,25,  5,12,
   5,10, 21, 0,  8,28,  9,20, 20,13, 10,26,  6,11, 12,15,
   0,18,  1, 5, 22, 2, 21,13,  9,13, 16,13, 25, 5, 25,29,
  13, 6, 22, 8, 17, 1,  8,19, 13,14, 12, 1, 19, 9, 16, 4,
   5,13, 10,11
};
const int p_30_5[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  152,
  /// Array of preferences
   6, 1, 10,27,  2, 2, 29, 4, 19,20,  1,11,  9, 5, 24,17,
  27,23, 10, 9,  4, 6, 22,10, 24,19,  1,19, 22, 4, 22,13,
  25,20, 10,26, 13,13, 28, 5,  6,24, 15,12, 15, 6,  0,12,
  19,25, 11,26,  7,23, 18,19, 29,17, 28,21, 16, 6,  6,23,
  24, 6, 18, 0, 22,29, 19,24, 11,29, 11,13, 18,27, 12, 2,
  28,17, 28,16, 18,23, 10,12,  0,28, 21, 2, 24,22, 29,27,
   9, 8,  8,20,  2,20, 20,18, 26, 5,  8,10, 27, 1, 11,24,
  26,13, 11,20, 10,13, 10,20,  7, 4, 13,27, 25, 0,  8,15,
  21,28, 13, 0, 12,16, 25,19,  3, 0, 20, 0,  6, 7,  8,12,
  10,17, 18,15, 11,10,  5,19,  0, 9,  7,13, 13,20, 21, 7,
  29, 3, 28,28,  9,16,  8, 7,  9,24,  3,21, 24,20,  2,22,
  11, 9, 26,10,  6,19, 23,14, 27,12, 21,23, 25,13, 22,25,
  21, 4, 12,10,  3,14, 14,16,  9,28,  7,21, 16, 5,  4,28,
  19, 8, 20,24,  9,10, 27,11, 21,26, 26,16, 17,23, 25, 8,
   6,10, 16,15,  5,28,  1,22, 27,17,  7,10,  0,19, 14,14,
  28, 7, 23,15,  8,22, 23,25,  5, 7, 26,11, 14,12, 20,21,
  10,22,  6,27,  0,23,  5,20, 23,23,  0, 1, 29,29,  3, 5,
  18,14,  8,18,  1, 5, 15,11, 22, 3,  4,17, 21,10,  0, 0,
   9,25, 29,25, 22,11,  4, 5, 20,14, 18, 9,  3,15,  2,19
};
const int p_30_6[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  175,
  /// Array of preferences
  18,20, 17,19, 26,10, 25,21,  1,11, 17,16,  0, 3,  6,11,
  19, 5, 13,19,  0,16, 12,14,  2,11,  4, 8, 17, 0, 16,16,
  18, 7,  8,17,  3,28,  5,27,  5,26,  5, 4, 20, 8, 27,10,
  11,15,  7, 6, 16,11, 12,11, 24, 6, 15, 9, 27, 4, 15, 3,
  24,25, 21,24,  0, 0, 21,26, 13,16, 21, 0, 29,11, 23, 2,
  18,10, 15,19, 15, 1, 17,27, 10,23, 15,25, 25,26, 10,17,
  15,15, 21,21,  9, 7, 12,25, 11,22, 17,24, 19,28, 26,24,
  23,17, 10,18, 11, 9,  1,15,  5,20, 28,25, 12,15,  6,17,
   8,27, 25,22, 17,28,  4,23, 20, 5, 12, 0, 11, 1,  7,19,
  22,15, 10,25, 11,16,  5, 5, 25, 5,  3, 6,  3,21,  2,22,
   9,29, 22, 9, 12, 6, 19,11,  9,18, 16, 6, 22, 1, 17, 3,
   5,23, 21, 6,  2, 8, 13, 1, 24, 4, 15,26, 14, 7,  6,13,
  22,25,  6,14, 10, 4, 20,11,  7, 2, 26,28,  1,14, 17, 1,
  22,26, 10,15, 20,17, 25, 9,  1,27, 29,20,  2,24, 10, 2,
  14,24, 17,18, 14,16, 14,19,  3,18,  9,16, 26,22,  6,19,
  11,10, 25,18, 16,19, 23,12,  3, 0,  2,26,  2,27,  6, 2,
  19,19, 14, 5, 14, 0, 25,27,  1, 9, 19,27, 13,18, 26, 5,
   5,24, 13,20, 13, 6, 18,23,  1,25, 15,22,  4, 0, 23,22,
  18, 5,  8,14, 24,17, 10,11,  8,25,  2, 4, 25,14, 27, 6,
  24, 5, 10, 0, 14,17,  8,20, 20,24,  6,22, 11,19,  7,26,
  24, 2, 15,20, 23,18, 16,24, 26, 6, 17, 6, 20,13, 14, 8,
   8,29,  1,19, 16, 1, 19,12, 29, 3, 18,18, 16, 0
};
const int p_30_7[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  194,
  /// Array of preferences
  14,14, 15,12, 15,11, 19,17, 16,19, 14,29,  8, 3, 21, 8,
  28,16, 18,29, 10,24, 15,22, 24, 4,  3,13, 20, 5,  1, 0,
  27, 1, 20, 9,  2, 2,  1,15, 15, 3, 23,22, 16,17, 22, 8,
   6, 8,  5,17, 29,25, 18,17, 18,15, 21,29, 20, 3, 10,27,
   1, 5, 29,15, 28, 5, 15,15,  6, 1, 11,21, 18,23,  8,26,
  10, 7,  9, 0,  6,15, 29,20,  9, 6,  4,10, 18, 0, 10, 1,
  11, 1, 20,29, 20,23, 15, 9,  1,25,  9,20,  2,16,  7,18,
   1,20,  5,22,  0, 9,  9,18, 27,27,  8,29,  2,23,  9,16,
   4,29, 26, 1,  0, 2,  4,26, 12,28, 16,28,  6,11, 22,28,
  23,26, 14,28,  6,16, 11, 2,  6, 7,  3,29, 12,17, 13,26,
  17,18, 19,18, 12,18,  6,25,  7,17, 16,12, 19,25, 19, 4,
   2, 7,  9,12, 29,12, 11,19,  8,28, 17,23,  8, 5, 16, 4,
  13,20,  5,28, 14, 8, 18,11,  4,18, 22,24, 14, 7,  9, 7,
   1,26,  7,12, 29,21, 23, 3, 27,13, 15, 7,  0,17,  3, 8,
   2, 3, 21, 7, 29,24,  1,29,  7,27, 17,24,  5,19, 15, 6,
  27, 7, 25, 1,  4,15, 17,28, 17,21, 16, 0, 27, 5, 24,11,
  17, 6,  0, 8,  7,14, 21, 1, 21,25, 27, 3,  0,13, 18, 9,
  26, 9,  5,29,  5,16,  7,20, 17, 8,  7,22,  0, 3, 28, 6,
  21,13,  4,13,  2,10, 10,18,  5, 1, 29, 2, 28,11, 17,15,
   0,18, 26,22, 20,11, 17,27, 26, 5,  0,29, 12, 0, 26,29,
   2,13, 21, 0,  9,27, 13,14, 27, 0, 19,10,  2,11,  4,16,
  24, 0, 27,17, 19,27, 11,25, 14,21, 18,12, 12,14, 12,10,
  13,21,  7,25, 19,29, 21,11, 27,20,  2, 1,  9, 2, 15,21,
  11, 4,  6, 5, 18, 6,  8, 7,  7, 4, 20,13, 27,11, 19, 0,
  28, 2, 24,10
};
const int p_30_8[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  5,
  /// Array of preferences
   5,23, 11,16, 20,16, 22,11, 23, 2
};
const int p_30_9[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  110,
  /// Array of preferences
  11,24,  6, 3, 23, 9,  1,14, 20,29, 26,10,  4,25,  4,17,
   7,23,  5,21, 15,22,  5,12,  5,16, 18,21,  1,10, 12, 6,
  16,21, 26,17, 14,15, 20,27,  4, 5,  7, 5, 27, 6,  1,13,
   1,28,  4,14, 12,18,  3,15, 21,19, 26,25,  7,27, 27,20,
   4,23, 21,20, 26,12,  9,21,  6,19, 11,21,  7, 2,  6,13,
  23,10, 27,21, 17,12, 14, 3,  9, 0, 14,18, 18, 6, 17, 5,
   3,20, 28, 7, 13,21, 23,18,  5,15, 13,24,  3,11,  9, 4,
  15, 2, 12,27, 18,27,  4, 0, 22, 3,  1,22, 17,10,  6,25,
   1, 0, 23,24, 11, 9, 29,16, 17, 9, 12,24, 18,26, 11,13,
   8, 5,  6,27, 24, 3, 23,16, 13, 4, 10,12, 23,11, 26, 1,
   8, 8, 28, 1, 17, 1, 29,21, 23, 8,  4, 8, 20,28,  8,22,
   1,20,  1, 6, 14, 0, 15, 4, 13,20, 19, 1, 12,21, 15,18,
  17, 7, 18,14, 18,17,  5,14, 17,25,  6,11, 20,24,  1, 1,
  20,21, 13,22, 22, 8, 16, 4, 22,25,  9,20
};
const int p_30_10[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  221,
  /// Array of preferences
  29, 7, 28, 0,  6, 0,  2,23, 12,18,  9,11, 18, 1, 11,19,
   8, 2, 19, 3, 26, 1, 28,20, 16,14, 22, 8, 13, 4, 26,13,
  15,16, 11,25,  4, 7,  6,19, 15, 1,  2, 9, 25,26, 14,12,
  21, 5, 20, 6, 24,20,  1,28, 13,16, 27,28, 21, 2, 17,23,
   7,28, 25,10, 29, 3, 12, 4, 23,23, 23, 6,  1, 8, 16,21,
  12,25, 12,28,  2,27, 25,25,  1,20,  0,28,  8,21,  0,26,
  21, 9,  5,12, 25,29,  3,19, 10,19, 11,16, 25,19,  0, 5,
  14,19,  4,13,  9, 9, 10, 5, 17,17, 17,18, 20,15, 21,28,
   5,27, 28,16,  7,11,  9, 8,  7,26, 27,20, 24,27, 23,17,
   0, 0, 26,19, 18,13, 23,14,  1, 3, 17,22, 21, 3,  7, 1,
  24, 2,  3,26, 27, 1,  5, 1, 27, 2, 29,19, 14,17, 28, 9,
  22,15, 15, 5,  6,21, 19,14, 10,13,  5, 0,  1,16, 14,25,
   1,25,  4,24, 20,23,  5, 8, 22, 5, 18,12,  6,26, 18, 4,
  10,12,  3, 5, 14,16, 20,26, 19,28, 11,22, 27,21,  9, 0,
  24,16, 26,26, 20, 0, 19,22, 13,19, 13,25,  5,21, 20,12,
   7,23, 11,10, 25,11, 24,19, 20, 5, 29,13, 16,27,  7, 2,
   9, 4, 12,27,  9,22, 24,25, 29,15, 24, 6,  1,29,  4,20,
  11,24, 27, 3, 26,21, 23, 2, 13, 1, 10, 0, 14,24, 11, 8,
  25, 4, 24,18,  2,25,  8, 3,  4,28,  8,27, 10,28, 29,25,
   2,11,  7,25, 13, 2, 10, 7, 13,17, 28, 4, 15,19, 22, 4,
   4,22,  4,27, 16,25, 13,23,  3,17, 19,26,  2,13,  2,16,
  26,17,  4,16, 13,29, 10,21,  6,16, 15,24, 29,14, 13,13,
  21,14,  2,19, 26,10, 28, 2,  3, 1,  1,24,  5, 7,  1, 9,
  17, 7,  3,27,  1,19, 22,14,  9,10,  1, 5, 12,24, 23,24,
  19, 2,  3,21,  6,25,  8, 6, 27, 7, 16,12, 26,22, 12,16,
  26,16,  0,13, 14, 5, 15, 9, 17,27,  4,19,  7,27, 21,11,
   2,20, 13,24, 23,28,  3,10, 24,22, 20, 9, 14, 0, 23, 4,
  19, 5,  0,25, 28,21, 25,16,  7,24
};
const int p_30_11[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  177,
  /// Array of preferences
   9,11, 15, 2, 11,21, 19, 1,  6, 7,  2,15, 25, 8, 18, 7,
  26,27, 16, 5, 13,21, 28,20, 23, 6,  6, 4, 14,18,  3,26,
  23, 4,  1,13, 18,15, 21,18,  0,17, 20,15,  4, 8, 14,20,
  21,17,  8,28, 27, 8, 26, 8,  5,22, 22,22, 10,28, 14,23,
  11,27, 26,12, 17, 2,  9, 5,  4, 9, 21,23, 15,21, 26,11,
  29,21, 19,19,  2, 2, 18, 6,  5,13, 25,21, 11, 5, 24,18,
  13,25, 14,29, 18,12, 15,27, 16,24,  6,16,  8,29, 17,11,
  12, 8,  6,29, 12,18,  9,23, 27,22,  4,17,  1, 1, 28,29,
  27, 2, 27,12, 13,19,  7,20, 27,16, 14,22, 16,29, 18,21,
  20,14, 20,16, 28,17, 13,11, 14,10, 18, 0, 10,25, 25, 1,
   7,22, 23,23, 29,22, 14, 0, 14,11, 20, 0, 21,13,  5,28,
  29,20,  1,21, 19, 8,  5, 3, 21,25, 10, 0, 14,21, 28, 4,
  10, 3,  6,24,  4,27,  6,22,  4, 2,  5,27, 29,23, 25, 0,
   1,24, 14,26, 28, 3, 28,12, 26,21,  7, 0, 18,28,  2,16,
   5, 7, 18,24, 18,10,  2, 9,  4,15,  4,16,  3,10, 16,27,
  12,17, 15,25,  5,29,  2, 6,  9,10, 26, 6, 24, 7, 18, 9,
  11,22, 10, 5, 24,16, 18,25,  8,22, 15,11, 12,27, 10,19,
  29,10, 29,27, 26, 9,  4,12, 20,17,  0,15,  0,23, 23,12,
  10,21, 23,20,  4,14, 20, 9, 12,22, 27,23, 25,15, 17, 7,
   3,17, 17,14,  2,26, 29,17,  8, 6, 11,14, 15, 0, 11, 2,
   1, 6, 19,25, 19,15,  6, 3, 29,26,  8,12,  0,29,  5, 8,
   8,26, 23,14, 22, 1, 16, 2, 12,29, 13,24,  6, 1,  1, 8,
  25,12
};
const int p_30_12[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  213,
  /// Array of preferences
  20,23,  1,15,  4,15, 10,23,  9,10,  4, 9, 18,14, 10, 7,
   9,26,  5,11,  2,15, 17, 7,  8,29,  7,27,  3, 3,  4,12,
   2,18, 17,28,  7, 6, 26, 7,  0,12, 10, 0, 28,19,  3,12,
   9,27, 10,16, 23,21, 11,11, 14, 2, 13,13, 21, 1, 27,27,
  26,14, 14, 8, 23, 2, 23,23, 20, 6,  5, 7, 24,26,  3,24,
  12,19, 11,22,  1, 1,  5, 4, 26,26, 18,20,  4, 6, 14,11,
  28,18,  0,16, 15,10,  8, 3, 18,15, 15, 2, 23,16, 13, 7,
  10,13, 29,25,  1,17, 25,10, 28,12,  9, 4,  5, 3, 28,28,
  24,27, 10,20, 15,26, 13,18,  3,19, 27,13,  7,24,  6, 5,
   3, 8, 27, 4, 26,23, 13,21, 16, 9, 13,23,  4,19, 11, 3,
  27, 7,  6, 6,  6, 4,  4,18, 16, 3, 28,17, 23,19, 10,19,
  24,11, 26,16,  0, 3, 19,26, 19,24, 11,28, 24,17, 10,11,
  18, 3, 14,29, 25,21, 16, 5,  7,15,  2, 7,  1,21,  4,20,
  28, 3, 28,29, 11, 5,  9, 9,  1,29,  7,20, 26,24, 11,10,
  27, 0,  8,17, 24, 1, 23,15, 19,29, 24,16,  0, 2, 25, 2,
  19,27,  0,13, 24, 6, 18, 0, 21,21, 27,28, 10, 2, 26,19,
  10,24, 11,19,  1, 5, 25, 5,  5,22, 12,25, 18,24, 28, 6,
   2,29, 11,14, 20,25, 13,10, 16,13, 15,28,  7, 5, 29,17,
  23,14, 17, 0, 22, 9,  3,23, 20,11, 19, 1,  6,28,  0,10,
   5,29,  3,28,  3,22, 27,17,  9,17,  2,13, 17,26,  0,24,
   4,26, 17, 9, 23,27,  2, 9, 10,28, 19,21, 26, 4, 22, 6,
   5, 2, 14,24, 25, 3,  0, 0,  5,25, 17,23, 22,10,  4,27,
  13,27, 27,23, 17,15,  2,14, 19,20,  6, 0, 11,21, 29,15,
  24,29,  3,10, 18,29, 24,28, 16,24,  8,22, 20,10, 26,10,
  24, 9,  0,21, 16,20,  4,24, 16,29, 18,21, 28, 7,  8, 8,
  10,14,  4, 3,  3, 5,  6, 9, 14,25, 26,27, 21,22,  1,23,
  15,12, 22, 7, 17,17, 21, 3, 24,23
};
const int p_30_13[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  82,
  /// Array of preferences
  27, 9,  7, 4, 27, 0, 18,21, 29, 2, 14,12, 22, 2,  4,25,
  10,27, 17,20,  7,23,  9, 9, 26, 5,  5,22, 27, 1, 18, 5,
   5,15,  6, 5, 18,13,  1,12,  6,22, 11,28, 27,11, 14,16,
  23, 6,  4,28, 28,24, 19,13, 11,26, 19, 0, 27, 6, 12,29,
  20,21, 21, 8, 20,27,  9,27, 28, 8,  1, 4,  9,15,  5, 1,
  27,25,  4,10, 27,26, 13, 0,  5,10, 13,29, 19,17,  4, 0,
   5, 5, 10,21,  5,13, 11,15,  8,18, 29, 5, 27,17,  2,24,
  11, 0, 14,11, 15,28, 14, 4, 16,22, 24, 3, 24,18,  9,17,
   8, 8, 12,22,  3, 2,  1,25, 13,24, 24,14,  3, 6, 18,23,
   0,21,  6,28, 26,12,  7,19, 16, 8,  7,13, 25,16, 17,14,
  19,29, 23,18
};
const int p_30_14[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  51,
  /// Array of preferences
  18,12, 28,11, 22,25, 10,22, 24,20, 26,23, 21, 9, 19,20,
   5, 5, 15,24, 10,17, 17,17, 29,13, 22,14, 17,18, 11, 6,
  20, 1,  3,28, 23, 3, 12,14,  8,22,  7, 4, 29,20, 21, 4,
   0,14, 21,26, 29,29, 17, 9,  9,12, 28,26, 29, 0, 26, 2,
   4,29, 12,12,  7, 2,  9, 1, 21,21, 12, 4, 13, 3, 24,13,
  24,27,  3, 7, 13, 8, 22,12,  5,12, 20,11,  1,21, 18,10,
  26,24, 28,18, 27,20
};
const int p_30_15[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  130,
  /// Array of preferences
   6, 1, 13,14, 28,13, 29,27, 18,26, 28,16, 13, 3, 10, 7,
   3,21,  3,26, 11,13,  3,27,  0,22,  3,28, 25,13,  7,11,
  21,18,  4,25,  0,16,  8,17, 11, 5,  8,14, 28,24, 11,14,
  29,11, 26, 4, 12,23,  2,19, 29,14,  4,22, 21,19,  0, 0,
  10,12, 19,23, 13,24, 11, 4, 16, 8,  3, 6, 27,17,  0, 6,
  15, 4, 12, 8,  0,29, 26,17, 14,26, 10,29, 28, 5, 10,21,
   5,18, 11,21, 19, 6, 11, 0,  8,27, 13,21, 12,24,  8, 0,
  23,10,  3, 3, 22, 7,  5, 9, 24,15, 10,17, 13,19,  0,10,
  13,12,  7,15,  3,15, 20,14,  7,22, 16,17, 17,17,  7,20,
  18, 9, 10,25, 21,25,  7, 3, 12, 5, 10, 6, 15,20,  5,14,
  19,20, 23,27, 22,13, 28,26, 25,25,  0,14, 24,25,  4, 5,
  14,15, 28,28, 10,23, 17, 2,  1,14, 20,13, 14,22,  3,11,
  18, 1,  2,14,  9,18, 18,25, 24, 0, 24,24, 18, 3, 10,26,
  27,14, 27, 4, 10, 1, 17,29, 20, 6, 20, 1, 22,17, 20,24,
   2, 2,  0,28, 23,14, 24,19,  5,16, 19,19,  7, 7,  1, 9,
  13,22, 17,13, 19,17, 25, 3,  8,22, 13,10, 15,24, 24,17,
  10,20,  4,26
};
const int p_30_16[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  175,
  /// Array of preferences
  12,15, 19,23, 15,17, 18,21, 19,19, 29,29, 17,24, 15, 7,
   6,16,  0,20, 16,11, 15,22, 16,20, 19, 1, 26,16, 11,22,
  11,19, 19, 3, 25, 9, 17, 6, 25, 8, 24, 8,  7,22, 19,16,
   2,20, 11,16, 10, 1, 22,27, 27, 6,  9,22,  0,15, 12, 2,
  21,26, 27,15,  1, 8,  9,27, 26, 2, 26, 8, 23, 1, 14,23,
  27, 8, 11,27, 17,15,  5,14,  9, 7, 21,11, 16,24, 16,12,
  23, 2,  4,15,  9,19, 27,19, 23,10, 13, 0, 16,17,  9, 6,
  27,28, 26, 5,  7,29,  0,12, 16,25, 22,20, 25,24, 10, 2,
  20, 3, 28,26, 21,15,  9,16, 20, 4, 19,29, 18,22,  0,25,
  25,11, 25, 4, 22, 4, 22, 2,  1, 9, 22, 7, 23,13, 27,20,
  24,24,  4,21, 22,19, 13,20, 15,28, 10,10, 20, 5, 10, 4,
  24, 2,  2,11,  6, 4, 18,20,  8, 0, 15, 1, 18,27,  0,18,
   5,16, 28,17, 13,23, 15,14, 20,18, 18, 0, 29,14, 17, 4,
  16,28, 20, 8, 14, 5, 10, 9, 13,10, 29, 5, 15,23,  5,29,
  10,15,  8,16,  6,12,  1, 3, 18,11,  9,25, 13, 7, 12,29,
   2, 2, 24,15,  0, 8, 13,15,  8,17, 13,12, 20, 9, 23,28,
  10, 6, 11, 2, 23, 4,  4, 3, 26, 9, 16,18, 14,24, 10,21,
   0,24, 29,10, 21, 0, 10, 0, 16,14,  6,17,  8,18, 29,12,
  21, 3, 22,14, 20,26,  3,12,  7,18, 21, 6, 10,26, 25,19,
   7,13, 13, 8, 15, 4,  7,27, 28,20,  7, 2, 21,17, 17,28,
  26,17, 18,29, 14,16, 28, 1, 10,12, 10,14, 26, 7, 13, 5,
  26,26,  1,21, 20,23,  4,20, 12,25, 12,12, 14,22
};
const int p_30_17[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  167,
  /// Array of preferences
  14, 5,  4,18, 21,13, 15,26, 13,26, 19, 6, 17,17, 26, 2,
   2,25, 26, 4, 22,13, 21,10, 17, 1, 17,22, 16, 3,  9,26,
  25,22,  6,28,  2,12,  7, 6,  8,11, 18, 3, 28,23,  2,21,
  19,13,  9,13,  2,29, 21, 6, 18, 9,  8,26, 10,18, 19, 0,
  17,16,  7,24, 17,25, 28,25,  3, 5,  2, 7,  3,12, 19,14,
  27,29, 26,23, 20,19,  3,29, 21, 1, 25,13,  1,13, 18,14,
  14,11, 14, 8, 19,20, 12,20, 17,28, 25, 7, 22, 5, 11,17,
  22, 2,  8, 5, 22,21,  7,23, 19,15,  4,29,  4,11, 23,28,
   0, 0, 23,13, 19,27, 29,13, 18,18, 12, 1, 26,27,  4,19,
  18,12, 15, 0,  5, 9, 23,29, 21, 3, 15,14,  1,22, 21,12,
   0, 7, 18,10, 19, 1, 11,21,  8,20,  9,18, 17,24,  0,25,
   1, 6, 24, 2,  5, 1,  7,26, 22,28, 22, 6, 26,16, 27, 0,
  19,18,  6, 0, 16,23,  7,15,  5, 4, 25,10,  9, 5, 23,16,
  18,20, 16,14,  9, 8, 28,19, 29,19, 15,19,  5,27, 12,13,
   2, 6, 27,13,  3,17, 16, 4,  8, 0, 23,17, 28, 3,  8,13,
  29,25, 14,13, 22,14, 22, 3, 26,13, 29,22, 27, 9, 17,19,
  16, 6, 23,12,  2,19, 11,28, 27,20, 10,20, 18, 8, 12, 5,
  19, 5, 26,22,  3, 3, 17, 4, 13,23,  7,25, 13,27, 10, 8,
  11,12, 25, 2, 11,15,  5,23, 18,11, 29, 5,  0,29, 15, 4,
  16,26, 26, 8,  1,21,  8,23, 26, 6,  2,27, 23,11,  5,22,
  24,28, 21,20, 24,12, 26, 9,  9,14,  1,23, 21, 4
};
const int p_30_18[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  87,
  /// Array of preferences
  27,10,  4,18, 11, 0, 16, 0,  5,14,  6,20,  7,13, 21, 7,
  27,21, 10,25, 23, 1, 22, 8, 26,27, 20,23, 23,21, 12,23,
  25,11, 23, 8,  9,17, 18, 4, 28,26,  2, 2,  0,21, 25,10,
  29,21, 18,19,  0, 3,  3,25,  0, 7,  2, 6, 11, 1, 13,18,
   9,26, 27,18, 22, 0, 27, 6, 13, 0, 25,22, 13,22, 26, 0,
  17,12,  6,21, 22,10, 24,13,  1,13,  3,28, 25,12, 17, 3,
  21, 5, 29, 5, 29, 4,  5,24,  1,27,  2,25,  4,20, 25,16,
   6,10,  0,17,  5, 9,  6, 8, 15,18, 12, 1, 22,17, 23,14,
  21, 8, 21,27, 13,28, 24, 0,  0,13,  0,23, 24, 2, 19,12,
  20,17, 24,19,  0, 4,  0,25,  1,21, 11, 6,  0, 0, 27,25,
   8,10, 12,29, 21,26, 15,17,  2, 1,  1,29, 13,21
};
const int p_30_19[] = {
  /// Number of people on picture
  30,
  /// Number of preferences
  146,
  /// Array of preferences
  28,21,  9,28, 24,13, 23, 3, 28,13, 23,11, 10,25, 13,23,
  14,23, 21,15, 21,28,  1,23, 22,22,  3, 1, 16,28,  3,28,
   0,20,  5,10, 15,12, 27,19, 25,14, 19,25, 11,17, 10,21,
  15, 4,  6,28, 28,17,  7,21,  2,11, 14,22, 26,19, 20, 6,
  25,11, 15,10, 11, 8,  3, 4, 27,29, 19, 1, 19, 3,  4, 7,
  28, 7, 10,18, 14,20, 20,24, 10, 6, 18,21,  4,29,  2,12,
   7,11, 15, 6, 17, 4,  3,29, 23,24,  5,22,  2,22,  0,21,
   1,13,  3,23, 16, 0,  6,27, 16,29, 29,29, 26,21,  1,19,
  20,10, 10, 0, 29,10, 11,21, 26,29, 19,11, 13,11, 29, 4,
  29,27, 16, 8, 17, 0, 22,13, 24, 1, 29,20,  1,22,  7,27,
  29,12, 18,19,  1,12, 28,16,  9,15, 22,12, 19, 4,  6,13,
   8,28,  4,14, 28,10,  5,26,  1,25, 27,10, 18, 8,  2,26,
  19,14, 14,14, 21,13, 26, 1, 12,28, 13, 7, 17,22, 15, 2,
   5,11, 14,17, 14, 5, 22, 3,  6,16, 13,27, 24,26,  1, 1,
   2, 9, 13, 8, 12,13, 24, 7, 10,13,  4,22, 18,20, 24, 2,
  26,23, 12, 3,  2,24,  6,19,  7,15, 25,28,  9,16, 12,18,
  29, 3,  1, 2, 23,14,  0, 3, 26,26, 28,27, 21,20, 20, 4,
   8, 6, 24,25,  6, 7,  1, 0, 26,17, 25, 1,  8,13,  2, 6,
   9,21,  8, 8
};

const int p_40_0[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  223,
  /// Array of preferences
  18,26, 32,31, 38,31,  8,33, 36,15,  8,23, 34, 6, 33,19,
  38,11, 18, 9, 31,37, 30, 0, 33,22, 37,19, 26,30, 24,19,
  29,29, 13,18, 12,18, 35,30, 34,35,  2, 9, 31, 9, 21,19,
  24,26, 20,20, 30,14, 27,36,  6,16, 38,23,  7,21, 29,20,
  35,34, 15,39,  8,31, 10,21, 34, 5, 23,38, 34,23, 31,20,
  12, 8,  7,22,  4,37, 27,10, 31,31, 31,10,  7,15, 29,16,
  35,36,  4,30,  3,39,  3,32, 18, 2, 17,35,  9,15, 21,16,
   9,19, 10,23, 37, 1, 13,36, 13,12, 39, 9, 19, 3,  1,32,
  26, 8, 14,23, 35,13, 24,37,  5, 9, 18, 6, 13,21, 32, 2,
   2, 7, 29,36, 19, 9, 11,23, 17,15, 39, 8, 27,29, 12,35,
  12, 7,  8,10, 39,30, 29,25, 11,15,  7,28, 33,21, 24,32,
  36,35, 26,14,  0, 4, 37,28, 37,12,  9, 2,  9,17,  7,35,
   1, 1,  9,34,  8,17,  6,10, 22,34, 35,25, 32, 1, 12,38,
   1,24,  5,28, 11,10,  0,31, 36, 7, 19,10, 16, 8, 24,11,
   5, 3,  1,11, 27, 3,  5, 4, 23, 9, 29, 8, 28, 6,  1,36,
   1,27, 37,35, 10, 8,  5,10, 27, 9,  3,19,  9,31,  4,28,
  38,19, 15,36, 29, 9, 25,31, 23, 6, 13,30, 14,17,  4, 1,
   0,35, 33,29, 13,34, 28, 0,  2, 3,  8, 1, 19,28, 38,13,
   8,38, 35,29, 32,28, 22,38, 16, 2, 18,17, 32,34,  8,14,
  35,32,  8,26, 10,34, 26, 1, 10,27,  0,32, 14, 0, 31,32,
   2, 0, 14, 4, 25,21,  8,37, 27,32, 25, 6, 22,16, 30,20,
   7,29,  3,38, 25,35, 32, 9,  2,28,  5,19, 38,30, 39,16,
   1, 8, 25,14,  4, 4, 24, 3, 21,36, 14,19, 32, 4,  9,22,
   6, 5, 31,27, 35,24, 22, 0,  5,39, 31,35,  4,17, 32,14,
  24,33, 11,22, 21,25, 15,31, 24,17, 19,23, 39,33, 25,27,
  12,29, 10,20, 35,10, 36,16, 18,11, 37,32,  2,10,  3, 7,
   3, 0,  7,37, 10,29,  2,17, 22,13,  2,37, 28,27, 17,21,
  20,34,  7,33,  9,38, 36, 2, 32,39, 12,32,  5,15
};
const int p_40_1[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  229,
  /// Array of preferences
  32,21, 37,10, 23,34, 28, 9,  6, 1,  9, 0, 35,39, 34, 6,
  36,28,  3, 3, 13,22, 24,21, 36,25, 10,28, 29,22, 34,14,
  15, 5, 12,20,  5, 3, 11,15, 34,39, 13,23, 22,25,  4,33,
  13,25, 31,29, 25,20, 14, 4, 23,17, 11,33, 29,25, 20,24,
  17,29, 26,27, 35,29, 38,31,  7,37, 13,29, 21,11, 15,26,
  35,20, 18, 9, 39, 7, 25, 5, 17, 1, 14, 9, 30,31,  7,15,
  32, 9, 38,26, 23,38, 38,32,  8,35, 32,39,  0,39, 28,36,
  32, 6,  8, 3, 21, 2, 38,11,  9,37, 34,36, 27,28, 27,21,
   5,11, 26,23,  4, 2,  0,37, 34, 8, 30,21, 29,32, 33, 0,
  19,11, 11,31, 31,19,  4, 1, 18,37, 21,17, 22, 9,  3,31,
  31,15,  0, 6,  7,20, 20, 0,  5,24,  9, 9,  2,22, 26,11,
   5, 5, 16, 4, 27,32, 23,13, 38,39,  5,33, 17, 8, 27, 8,
  16,35, 11,35, 34,29, 28, 5, 21, 0, 39,29, 14,20, 11, 7,
  13,36, 20,11, 18, 7, 14,28, 18,29, 22,29,  0,26, 17,31,
  28,11, 19,22,  2,23,  6,30, 39,21, 37,26, 35, 4, 32,10,
  11,30, 34, 3,  3,35, 24,12,  8,23, 32, 4, 24,31, 12,23,
  38,15, 20,16, 37,24, 19,34,  1,38,  3,27, 20,33, 19,35,
  37,39, 34,25,  7, 5, 11,22, 38, 2,  3,30, 30,38, 24, 0,
  13,34, 21,28,  1, 4, 15,16,  2,15, 22,15, 15, 1,  9,30,
  18,16,  9, 8, 23, 2,  1, 1, 10, 2, 29, 2, 34, 0, 19, 7,
   3,17, 13,39, 31,39, 28,37, 16,18, 29, 7, 30, 5,  7,24,
   1, 3, 15, 3, 31, 3,  8,25, 14,24, 15,24,  1,34, 20,10,
  24,33,  4,21,  8,22,  9, 6, 17,21, 37, 0, 14,38,  5,30,
  10, 6, 15,13, 11, 0, 25,34, 38,30,  1,15,  9, 2, 34,20,
  13, 4,  0,34, 21,14, 13,35, 12,32, 28,39, 33,14, 20,20,
  27,23, 22,26, 13,16,  9,16, 31,24, 35,14, 12, 6, 38,28,
  36,11,  6,16, 26, 4, 22, 1, 39,28,  9,19, 28,12,  6,25,
  13, 7,  6,37, 32, 3, 27,18,  8,36,  6,23, 35,16, 10,29,
  26,12,  6,11, 26,35, 23,28, 34,22
};
const int p_40_2[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  345,
  /// Array of preferences
   3,24, 16,28, 27,36, 31,12,  0,16, 19,32, 39, 4, 38, 0,
  36,17,  2,20, 36,37, 22,32, 32,12, 28,21, 34,35, 13,31,
  16, 9, 15,32, 39,27, 34,36, 23, 3,  5,16, 11, 9,  5, 4,
  29,15, 25,13, 33,25, 35,36, 38, 4, 37,20, 10,12,  9, 0,
  26,29, 23,36,  0,14, 14,24,  3,28, 19,11,  6,14, 25, 4,
  31, 0, 36,38,  4,10,  7, 4,  4,38, 21,35, 37,16, 39, 9,
  27,24, 11, 8,  8, 8, 20,31, 24,36, 30,36, 21, 5,  3,18,
  26,24, 15,37, 12, 0, 11,25, 28,18, 25,19,  7, 3,  3,17,
   4, 7, 25,25,  7,19, 33,12, 38,14, 23, 4,  9, 8, 19,38,
  12,25, 17,39, 29,16, 16,33, 21,34,  6,11, 38,31, 26,36,
   7, 2, 32,31, 12,16, 29,25, 36, 4, 17, 1,  1,39, 15, 8,
  17,31, 37,38, 37,32, 30,30,  2, 1,  9,24,  1,20, 37,28,
  32,32,  8,15,  7,34, 31, 8,  8, 9, 11,35, 18,34, 23, 2,
  31,14, 13, 1, 28, 3, 14, 5, 25,32,  6,36, 23,33, 17,24,
  29,33,  0, 4,  0, 7, 31, 1, 28,32,  1,21, 30,25, 17,26,
  25,16,  9,15,  4,23, 31,32,  5,20, 16,21, 38,12, 32,33,
  36,14,  8,21, 15, 3, 24,19, 17,28, 24,11,  4,25,  1,29,
  21,36, 35, 1, 35,12, 23, 8,  7, 6, 22,25, 38,29, 20,15,
  22, 8, 27,11, 35,21, 13, 0, 21,31, 14,32,  6, 1, 20,29,
  39,24, 16, 8, 11,13, 38,23, 10, 8, 25, 9, 21,18,  3, 3,
  12,18, 19,19, 11,16, 19,26, 33, 8, 36,30, 22, 9,  0,34,
  21,22, 33, 0,  1,18, 39,38,  3, 9, 22,11,  3, 0, 23,26,
   3,20, 15, 0, 32, 4, 22,24, 10,30, 37,26, 17,27, 33, 2,
   0, 1, 17,17, 14,23,  3,36, 36, 2, 11,36, 21, 7, 26,33,
  14,25,  0,36,  6,30, 15, 5, 36,18,  6,26, 18,13, 34, 7,
  34, 0,  1, 5, 34,34, 20, 7,  0,18,  9,31, 39,16,  8,31,
  24,14, 36,26, 31, 4,  9,25, 27,10, 28,20, 21, 2,  1,17,
   2,16, 38,32,  8,13, 24, 1, 36,15, 16, 2,  5, 2, 21,11,
  15,10, 27,21, 27,12, 21,37, 30, 5,  8,37, 27,37,  8,12,
  36,24, 38,15,  5,36,  9, 3, 16,25, 29,37,  1, 7, 16,17,
  24,18, 26,15, 16,20, 19, 2, 32,36, 10,37, 12,26,  9,20,
   2,31,  7,23, 28,16, 18,31, 14,33, 31,15,  3,19, 37, 5,
  30,18,  9,34, 34,37, 11,34, 20,38, 12,30,  4, 1, 18, 5,
  18,29,  7, 1, 15,13, 29,35,  8,36,  2,36, 32,26,  8,23,
  30,37,  6,34,  5,19,  6,12, 14,26, 26, 7, 18,26,  7, 8,
  16,23,  9,35, 10,20, 26,16,  9,38, 13,18, 12, 9, 32, 7,
  23,39,  7,25, 32,11,  4, 4,  2,22, 10,23, 15,12, 16,24,
   2,35,  8,20, 39,11, 35,14, 25,30,  9,21, 35, 4, 39,33,
   7,17, 34, 3, 17,10,  1, 9, 29,38,  4,16, 22,14, 18,14,
  14,34, 33, 6, 11, 2, 28, 0, 37,34, 39,22, 20,32,  7,18,
  14,20, 10,14, 36, 5, 20, 0, 23, 1, 23,38, 18,18, 29,29,
  22,38,  2,25, 35,26,  1,16, 11,31, 13,36, 15,21, 17,14,
   3, 2,  2,17, 27, 8,  6,13,  0,23,  1,10, 37,17, 12,15,
   8,17
};
const int p_40_3[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  87,
  /// Array of preferences
  11,20, 16,33, 10,20, 18,27,  7,33, 15,36, 35,32, 36, 1,
   1,20, 39,39, 19,12, 38,21, 39,23, 25, 4,  8,30, 23,32,
  14,28, 10,10, 30,27, 13,36, 32,11, 22,17, 19,23, 16, 4,
  26,15, 37,16, 28,23, 36, 3, 37,35,  9, 4, 25,26, 30,21,
  17,15,  8,27,  3,20, 36,36, 23,36, 18,14, 27,29, 22,25,
  13, 8, 20,21, 23, 1,  2,15,  4,11, 25,31, 24, 5,  3,14,
  19,29, 11,25, 20,20, 29,15,  1, 4, 34,24, 35,25, 26,18,
  39,33, 29, 2, 33, 8, 34,21,  3,22, 31,34, 25, 1, 39,37,
   9,31, 13,14, 36,29, 37, 2, 34,32, 11, 2, 10,15, 34,34,
   3, 2,  6, 2, 15,12, 11,34, 16,17,  6,39, 19,34,  7,22,
  32,34,  8,17, 39,21,  5,35,  9,25, 34,25, 19,28
};
const int p_40_4[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  208,
  /// Array of preferences
  39,31, 17, 2,  6,27, 13, 5, 12,24, 12, 2, 27,29, 32,31,
  21,29, 21, 5,  3,37, 30,15,  2,31, 18,34, 38,25,  0,26,
  20,34,  8,22, 25, 4, 24,38, 34, 0,  1,14,  6, 0, 19,22,
  27,14, 20, 1,  6,25,  5,37, 18,22, 29,29, 13,25,  7,22,
  22,20, 28,14, 20,27, 37,33, 34,34, 20, 3, 21,35, 11,20,
   9,35, 31,17, 23, 3,  6,14, 26,19,  2,21, 13,30, 26,36,
  13,37, 10,26, 26,14, 18, 0,  7, 4, 24, 1,  6, 4, 38,16,
  30, 9, 10,29, 24,35, 26,24, 39, 0,  7, 9,  6,34, 30,32,
   3, 6, 11, 5,  8,36,  9,39, 14,33, 35,33, 14,18, 27, 7,
  31, 2, 16, 7, 33, 0, 39, 2,  2,18, 23,28,  8, 7, 35,36,
  20,37, 20, 6, 21,28, 27, 1, 32,29,  3,24, 27,15, 27,34,
  32,19, 23,15, 10,39, 10,32, 12,12, 23,13, 37,12, 36,10,
  11,18,  4,27, 24,15, 18,38, 24, 3,  7,28, 33,17, 19,38,
   4,30, 29,17, 37,19, 10, 3, 19,30, 26,11, 27,39, 19,18,
  31,19, 39, 1,  8, 1, 10, 9,  6, 7, 25, 9, 38,32, 14,15,
  11,25,  3,32,  0,32, 33,31, 30,23, 16,13, 18,32, 35, 3,
  35,25, 26,22, 33,18, 27,33,  3, 4, 19,39, 30,19, 18,20,
  28, 5, 33, 9, 27,16, 14,25,  2, 9,  1,24, 21,20, 38,20,
  28,22, 19,26, 32,17,  4,12, 16,28, 38, 2, 14,34, 36,23,
   7,27,  1,29, 36,39, 11,16,  1, 7, 10,16, 28,32,  6,37,
  21,26, 24,32,  3,25, 19,24,  6,16, 14,26, 38,35, 23,25,
  27, 3,  3,20,  8,37,  0,10, 38,23, 11,19, 24,21,  1,22,
   6,28,  9,16, 17, 4, 17, 3, 38,24, 12,27, 27,17, 18,24,
  39,18, 37, 6, 21,32, 36,26, 15,27, 18,15, 29,12, 34,11,
  35, 6,  0,35, 13,16,  5,35, 19, 9, 29,36, 17,38,  1,19,
   9, 0, 10,21, 18, 5, 19,36, 37,39, 19,31, 14,37,  2,28
};
const int p_40_5[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  335,
  /// Array of preferences
  33,13, 27,38, 32,29, 22,17,  8,28, 22,27, 25,14, 32, 4,
   2, 0, 11,22,  7,22,  8, 3, 18,25, 37,19,  5,16,  1,39,
  11,10,  3, 7, 31,27, 17,29, 23, 4, 11,14, 29,25,  2,37,
  31,19, 39,14, 26,12,  9, 7, 11,12, 23, 1, 35,28,  1,29,
  24, 7,  5,25,  5,17, 32,26,  1,16, 17,13, 28,34,  8,37,
   3,12, 33,30, 20, 5, 22,12, 33,12, 19,37,  7,13, 13,29,
  15,24, 34,22, 34,17, 33,28, 19,25,  5,14, 36, 4, 12,35,
   2,35, 19,24,  5,24, 36,18, 37,18, 39,31, 32,38, 28, 8,
   2,26, 29,35, 10,28, 30,29, 11, 4, 19, 7,  4,16,  5, 0,
  36, 0,  9,11, 12,28, 12, 2,  0,26, 18,37, 16,13, 15,21,
  22,31,  4,19, 17,10, 11, 6, 37,29, 19,11, 15,22, 20, 9,
  24,23, 39,17, 29,34, 12,12, 32,30,  8,16,  9,38, 26,27,
  28, 2, 38, 5, 13, 9, 17,26,  8, 8, 39,36, 38, 3, 22,30,
  19,18,  0,19, 38,10, 15,19,  1,21, 36,17, 14,29, 27, 0,
   6, 4, 21,19,  2,31, 38, 9, 22,34, 16,12, 23,39,  1,33,
  31,28, 15,26, 25,31,  3, 9, 31,33, 19,39, 28,15, 26,18,
  36, 5, 20,35,  9,35, 16,34, 22, 8, 14,39, 23,30, 31,21,
  24,31, 19,20, 33,25, 33, 9,  6,25, 14,34, 23,26, 21,29,
  23,17, 33,24, 14,31, 12,20, 37,13, 27, 6, 22, 2, 28,27,
  36, 7,  8,27, 27,39, 11,24, 27, 9, 25,15, 22,39, 30,25,
  35,34, 18,28, 12,36,  6,33, 35,29, 39,10, 38, 1, 10,13,
  17,22,  4,33,  1,14, 10, 7, 15, 5, 29,39,  9,39, 16, 2,
  25,13, 18,23, 38,23, 36,16, 35,35,  6,16, 10,21, 17,25,
  39,24, 32,14, 24,26, 20, 7, 20, 1, 26,34,  2,25, 19,32,
  30, 9, 26,39, 16,10, 20, 8, 33,11, 26, 6, 18,20, 10,31,
  20,30, 12,29, 39, 8, 17,23, 33, 0, 29,14, 17,15,  7, 5,
   9, 2, 18,24, 39,21, 24,27, 17,18, 20,10, 27,10, 36, 8,
  27, 8, 27,31, 15,12, 15, 4, 14, 6, 18, 0, 16, 8,  6,34,
   8,35,  9,17, 34,39, 38,29, 21,12,  4,32, 15,20, 11,32,
  21, 5, 32,31, 39,33, 21,31, 29,26,  4,39,  0,23, 33, 6,
  11,37, 34, 9, 13,38, 39,13,  6,13,  8,23, 11,39, 16,27,
  17,12, 17, 8, 23, 7, 23,24,  4,25,  0,35, 35,19, 24,37,
  27,21, 29,20, 22, 6, 14,38, 25, 9, 26,37, 22,22, 14, 2,
  12,13,  7,32, 34, 3,  9, 6, 32,13, 17,16, 15, 6, 19,16,
  28,33, 11,30, 37,32,  7,12,  9,36, 15,29, 33,20, 16,20,
  28, 7, 19,17,  5, 8, 20,22, 34,15, 25,16, 18,33, 37,28,
   8,29, 13,20, 16, 7,  0,22, 16,18, 29,31, 11,36,  3,27,
  39,19,  8, 9,  0,27, 35,38, 35,31,  1,20, 34,12, 32, 2,
  26, 5,  9,33, 38, 2, 21,13, 26,31, 28,29, 39, 5, 29,32,
  24,18, 23, 5, 17,20, 25, 4, 37,39, 12, 9, 30,35,  9,22,
  27, 1, 16,30,  3,34, 27,19, 28,10, 16,38, 28,20, 10,34,
   2, 9, 25,11, 19,34,  5,22, 10,39, 27,11, 12,34
};
const int p_40_6[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  28,
  /// Array of preferences
  31,35, 15, 4, 29, 4, 18, 4,  1,38,  8,38, 25,21,  9,32,
  39,26, 11,19, 23, 8, 30,30, 26,39, 25,11, 13,16, 11,18,
  29,18, 22,22,  6,24, 19, 7, 17,25, 37, 1, 17,32, 18,15,
  38,29, 25,15, 34, 1,  8, 3
};
const int p_40_7[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  389,
  /// Array of preferences
  23,36, 30,13, 23,23, 27,13,  7,26, 39, 1, 36, 0,  6,14,
   4,10, 13, 6, 28,36,  3,34, 37,26, 38,15, 37,27, 33,21,
  25,31, 26,36,  3,25, 34, 7,  4, 0, 30,20, 21,20,  3,36,
   4,28, 22, 1,  1,23,  3,27, 33, 0, 14, 9,  8,33, 37,31,
  34,35,  3,32, 11,11, 18,19,  5, 6, 24,28, 34,14,  5,13,
  21,29, 37,38,  7,14, 37, 5, 24,29, 26,39, 39,35, 35,12,
  11,15, 31,39, 30, 5,  7,17, 24,14,  1, 6,  4,29, 11,30,
   9,16, 18, 3, 24, 8, 28,27, 11,31, 10, 1, 36,31, 37,21,
  13, 4, 15,20, 21,10, 19,25, 37,17, 18, 0,  3, 9, 15,21,
  33,31, 33, 6, 28, 1, 33,34, 14, 0,  3,26, 13,37, 39,38,
  17,39,  4,39, 12,29, 23, 4, 18,32, 17,21, 17, 9, 28, 7,
   9,15, 27,21, 37,29, 32, 8,  5,21, 38,22, 22,11, 38,37,
   6, 7, 26,21, 14,23, 22,26,  5,37, 31, 1, 21,34, 27,28,
  15,29, 35,39,  1,12, 21,16, 30,36, 25,10, 22,33,  1,34,
   8, 2, 29,32, 26,12, 21, 0, 23,30,  1,30,  6,33, 31,35,
  19,11, 36,19,  3,21,  8,10, 16,23, 23,31,  1,26, 32,11,
  29,23,  2,29,  1, 4, 21,26, 25, 1, 32,12, 39,18, 37,16,
  26,35, 36, 8, 24, 7, 36,25, 34, 8, 36,15, 30,31,  1,11,
  30,24, 10,30,  7,19, 15,28, 36,33,  8,39, 21,35, 19,17,
  10,37,  7,32,  0,12,  6, 1, 38, 7, 13,18, 18,15, 38,10,
   5, 2, 11,21,  5,17, 28,37, 27,39,  1,20, 13, 1,  8,11,
  39,19, 31, 7, 35,31, 25, 2,  5,19, 26,27, 12, 7, 19,18,
  14,35, 25, 6, 14,20, 22,24,  8,23, 15,34, 37,24, 31,34,
   3,37, 16,10, 27, 2, 31,25, 19,38, 21, 9, 29,22, 17,34,
  33,26, 11,35, 27,15, 14,14,  5, 3, 39, 2, 26,30, 39,27,
  11,13, 20,30, 10, 2, 30,27, 12,32,  9,19, 33, 3,  3,14,
   6,15, 12, 6, 19,29, 14, 5, 38, 6, 33, 1,  4,34, 10,22,
  34,21, 19,21, 13,16,  7, 0, 10,13, 15,37,  6,27, 20,35,
  34,38, 23,17,  7,25,  0,10, 38,35,  4,26, 15, 2,  3,30,
  17,18,  0,35, 23,12,  0, 5, 38,14, 20,26,  7,30,  7,12,
  14,38, 18, 2, 10, 4, 12,35, 12, 8, 34,24, 32,29, 29,20,
  28,38, 30, 7, 36,35,  7,24, 20,22, 38,19, 16,36,  8,35,
   8, 3, 16,12, 31,11,  6,11, 14,33,  2,27,  7,29,  3, 2,
  28,34,  5,26, 19,36,  7, 5, 24,32, 25,17, 12,19, 28,21,
  26, 4, 29,25, 30,18, 13,17, 12, 5, 24, 5, 32,18, 18,11,
  30,34,  9,21,  6,12, 23,37, 27, 4, 10,20,  7,38,  3,11,
  39,32,  1,35, 20, 8,  6,22, 13, 8, 29,21,  1, 1,  6, 8,
   6,32, 21,30,  7, 8, 26,31, 15,31, 10,31,  3,28, 11,27,
  14,34, 17,15, 12,14,  9,31,  4, 6, 18,20,  0,23, 25,20,
  25,28, 38,11,  5,11, 19, 6, 13,39, 25,25, 39, 6, 12,25,
  16, 9,  6,38, 26,25,  7,10,  9,17,  3, 6,  3,18, 16,16,
  23,11, 32,38, 35,15, 37, 2, 25, 0,  1,25, 29,16,  0,17,
  36,30,  1, 2, 27,23, 39,37, 13, 7, 19,31, 12,34, 10,34,
  35,33, 39,30, 24,12, 26, 1, 10,19, 22, 5, 12, 9,  4,13,
  22,20, 35,19, 31,32, 14,26, 21,21, 12,30,  3, 3,  3,38,
  38,31,  8,27,  8,37,  2, 9, 15,39, 13,21,  5,39, 21, 6,
  26,18, 31,18, 30,33, 19,37, 30,37,  2,32, 33,27, 18,16,
  13, 5,  0, 8, 26,11, 38, 1, 13,33,  4, 5, 34,30, 24,15,
   5,29, 16,19, 16,28, 20,28, 15, 6
};
const int p_40_8[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  124,
  /// Array of preferences
  32, 3, 26,27, 35,26, 37,28,  7, 9, 22, 3, 39,24, 11, 8,
  16,24,  4,16, 14, 5,  3, 0, 18,29,  8,11, 37, 3,  5,33,
  21,19, 10,23, 19,31, 38,20,  3,16, 15,12, 30,24,  3,28,
  34,34,  5,26, 10,21, 23,39, 12,17, 23, 5, 31,34,  5,18,
  15,17, 18,33, 11,20, 34,23, 21,30, 13,38,  4,38,  0, 7,
  27,24, 24,30, 10,10, 12, 7,  7,18, 37, 5, 13, 2, 39,22,
  39,34, 23, 4,  5,39,  1,37,  2, 2, 27,26, 11,36,  4,24,
  20,24, 27,36,  7,29, 11, 3,  3,13,  3,20, 14,10, 31,10,
  27, 6, 11,14,  2,36, 14,25,  0,27,  3,30, 14,17, 25,30,
  33, 6, 14,15,  1,19,  1, 2, 38,30, 12, 0, 34,12, 36,18,
  32,37, 34,26, 36,30, 38,11, 39,36, 28, 5, 29,29, 19, 6,
  11,24, 32,14,  0, 2, 20,35, 29,24, 30,32, 36,15, 10,38,
  32,20, 27, 0,  6,25,  9,23, 39,18, 23, 6, 36,33, 31,21,
  10, 9, 14, 1, 30,26, 21,11, 26,11, 19,16, 32,16, 17,29,
  28,24, 37,36, 10, 7, 10, 2,  5,27, 36, 8, 13,24, 28,31,
  31, 1, 16,38,  6,18,  3,27
};
const int p_40_9[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  15,
  /// Array of preferences
  13,18, 23,21,  2,26, 29,31, 26,17, 15, 3,  1,29, 15,16,
  23,39, 13, 2, 33,10, 11,10, 11,19, 12,25,  6, 1
};
const int p_40_10[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  22,
  /// Array of preferences
  11,10,  9,24, 23,15, 11,33,  9,21,  9,35,  4,35, 21,26,
   7,12, 31,37, 35,38, 28,24, 38,10, 37,11,  2, 5, 29,23,
  39,14,  1,33, 30, 1, 23,14,  6,36, 39,18
};
const int p_40_11[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  319,
  /// Array of preferences
  11,20, 23,15, 14,10,  8,22, 23, 9, 32,31,  5,20,  9,37,
  13,31, 37,30, 33,32, 32,12,  6,35, 14,22, 28,38,  6, 1,
  17,36, 14,18, 36, 1,  6, 4, 21, 8, 32,10, 20,34, 39,31,
  35,11, 15,22, 11,30, 11,25, 20,32, 14, 7, 17,14, 11, 6,
   4,10, 13, 0, 18,35, 18, 8,  5,23,  7,22, 25,15, 32, 4,
  22,30, 20,19, 36,15, 34,19, 27,29, 24,20, 35,15, 30,15,
   2,39, 29, 6,  4,11, 12, 6,  0,14, 34, 4, 21,29, 29,15,
  34,14, 16,17, 22,17, 19,39, 18,30, 35,21, 17,20,  7, 9,
   4,30, 15,32, 27, 3, 17, 7,  5, 0, 16, 0, 23,24, 16,34,
  27,26, 31,12, 10,27,  2, 4, 28,22, 35,37, 13,12, 25, 9,
   2,25, 36,27, 10,13, 20,35, 28,29, 13,16, 29,20,  2,17,
  21,14, 30,27, 17,19,  0,16,  8,16,  5,27,  3,13, 34,26,
  13,24, 10,12,  1, 6, 26,21, 18, 3, 19, 2, 33, 2, 21,30,
  31,30,  2,28, 35,17, 37,31, 12, 4, 39,17, 16,28,  6,29,
  30,36, 14,16, 37,25, 21,16, 33, 0, 37, 4, 32, 9, 27,39,
  35,32, 31,13,  1, 5, 38,20,  8, 5, 22,38,  2,21,  9,24,
   9, 0,  3,35, 33,18, 24,18,  8,28,  4,16, 34,13, 18,20,
  10, 2,  9,32, 22,18, 23,19, 16,31, 33, 6, 27,16, 22,29,
  16,33, 38,10, 34,36, 16,13, 12,12, 25,33, 13, 1, 11,38,
  25,37,  9,36, 34, 3, 31,29, 29,31,  5,34, 32,20,  9,11,
   1,22,  0, 4,  5,26, 31, 5, 35,39,  2,12,  1, 0,  5, 2,
  14, 8, 13,13, 17,26, 19,37, 19,19, 18,12, 11,39, 32,39,
   7,19,  5,32, 10,10,  9, 2,  2,27,  4, 8, 21,34,  6,27,
   9,10, 19,10, 11,35, 11,16, 39,19, 19,11,  1,16,  8,34,
  28,14, 18,18, 20,24, 11, 8, 17, 9, 32,32, 23, 8,  5,28,
   6,37,  4, 4,  9,20, 11,14, 37,33, 13,32, 14,37, 16, 7,
   5,35,  0,25, 23,35, 37,39, 38,12, 20,39, 29, 7,  0,33,
   0,24, 25,24, 10,30, 37,36, 22,33,  4,32, 36,19,  4,22,
  33,38,  5,15, 12, 5, 12,16, 17, 4,  6,23,  5,19, 26, 8,
   0,12, 16,22,  1,37, 22, 7,  7,33, 39,25, 19,32, 17,28,
  14,24,  9,30, 23, 7, 17,38,  4,23, 38, 7, 30,33, 12,26,
  37,27, 22, 2, 37,10,  5, 3, 23,13,  1,17, 21, 4, 15,31,
   3,26, 28,37, 26,11, 33, 9, 10, 3, 21,27, 33,31, 15, 6,
  20,21, 35,35, 29,34, 29,35, 22,24, 10,20, 28,13, 33,26,
  39,32, 27, 5,  5,30, 35,38, 33,30,  6,24, 23, 4,  2,34,
  21,21, 34,11,  8,20, 14, 6, 22,13,  4,25,  0,11, 25,22,
   4,29, 34, 2,  4,28, 15,35, 21,20,  3,21, 34,31, 37,21,
  11,22, 12,20, 14,23,  1,34, 17,16, 21,18, 13,39,  6,11,
  27,19, 22,28, 26,27, 17,12, 23,12, 19,16, 24,10, 29,10,
  26, 9,  8, 6,  1, 1, 31,32, 24, 3, 24,34, 31, 0
};
const int p_40_12[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  188,
  /// Array of preferences
  38, 4, 22,35, 13,33, 26,34, 27,26, 17,35,  3,11, 13, 2,
  33, 6, 24,29, 38,24, 38, 0, 30, 1,  7,21, 20,36, 15, 9,
  10,28,  0,32, 20,17, 36,21, 18, 7, 22,23, 29,27,  5,10,
  39,31, 38,19,  1,33, 20, 9, 15,25, 14,23, 10, 1, 13, 4,
  37,28, 34, 2, 15,36, 34,14, 29, 3, 31,20,  9, 6,  9,37,
   6, 2,  6, 1, 32,35, 33,16, 15,14, 18,17, 35,21, 37,27,
   4, 9, 11,26, 36, 8, 29, 1, 34,27, 35,33, 16, 8, 35,20,
  31, 3,  3,19, 38,16,  8, 1, 25,17, 14,37,  9, 5,  7,39,
  28,38,  7,17, 14,34, 16,10, 32, 2, 34,37, 34,36, 29,21,
   7,36,  6,39, 35,16, 32,25, 37,26,  9,20,  2,33,  8, 8,
  18,24, 32,34, 30,18, 28,36, 20,33, 18,30,  3,30,  1,29,
   1,15,  5,22, 24,21, 37,11, 36, 0,  1,19,  3,28,  3, 9,
  30,38, 12, 4, 37,35, 34,39, 14,29, 38, 5, 36,33, 22, 8,
  32,10, 35,27,  8,38, 29,30,  1, 4, 24,15,  6, 7, 35, 1,
   3, 8, 17, 5, 28,33,  0, 5,  0,13,  8,28, 17,10,  6,26,
  32,23, 33,33, 36,24, 15, 7, 24,31,  1,22, 25,18, 36, 6,
  30,20, 22,37, 17, 6, 18,29, 16, 6, 32,31, 38,30, 25,27,
   5,15, 25,16,  5,34, 12,20, 13,31, 32,15, 37,25, 38,36,
  26, 0, 30,34, 27,22, 24,24,  4, 2, 16, 5, 37,38,  2, 7,
   0, 2, 23,31, 10,17, 14,13, 39,10, 19, 7, 10,33, 10,31,
  10,26, 19,16, 34,17, 10,21,  8,22, 15,16,  2,32, 38,26,
  29,25,  2,36,  1,17, 15,12, 27,39, 20,34, 37,33, 32,27,
   5, 0, 13,10, 13,32, 30,21,  1,11, 32, 5, 10,11,  7,19,
  16,37, 17,22, 24, 8, 18, 3
};
const int p_40_13[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  255,
  /// Array of preferences
  24,35, 37,10, 11, 4,  1,11, 20, 9, 25,13, 13, 6, 25,23,
  16, 3, 24,37, 25, 1, 28,17, 36,11, 28, 8, 26,26,  9,32,
  19,22,  0, 7,  3,23, 34,36, 35,39,  7,39, 15,24,  0,39,
  39, 5,  6,14, 13,25, 18,29, 26,18, 34,39, 24,23,  1,16,
  13,11, 32,26,  3, 7, 30,30, 11,32, 15,10, 36,36, 12, 3,
  24, 7, 22,27, 30,37, 38,17,  1,32, 37,24,  4, 9, 34,14,
  26, 2, 17,17,  8,22, 32,28, 16,15,  7, 7, 11,38,  2,37,
  33,18, 27, 9, 20,27, 36,32, 26,25,  1,26, 14,24,  7, 9,
   3, 1, 30, 0, 38,21,  6,24, 12,30, 15,13, 28,35, 20,24,
  36, 4,  2,21, 34,27, 21, 2,  4, 7, 18, 8, 35,20, 17, 4,
  19, 7, 36,29, 16,26, 22, 1, 26,17, 39,25, 29,28, 37,21,
  31,35, 38,16, 16,31, 19,28, 30, 2, 30,24, 25,31, 24,21,
   6,36, 34, 1, 37,22, 17,14,  7,22, 31,38, 25,25, 34,21,
  14, 6, 28,21, 16,38, 38,33, 26,12, 12,12,  5, 4, 37,30,
  23, 1,  0,16, 10,15, 35,37,  2,23, 17,29, 39,10, 39, 3,
  19,27, 25,18, 12,14, 35,34,  1,36,  2,13, 27,30, 26,28,
  34,35, 28,24,  9,18, 38, 2,  6,15, 21, 4,  5, 0, 36,10,
   2,18, 35,19, 27, 7,  2,31, 11, 3, 33,21, 26,36,  2,32,
   3, 6, 30,27, 30,14,  4,34,  6,18,  1,22, 23,20, 21,21,
  21,29, 29,39, 34,29,  7,28, 20,35,  6,12, 38,14, 12,36,
  29,23,  4, 6,  9, 9, 15,17,  6,32,  7,37, 19, 8, 26,32,
  24,19, 33,11, 36, 1,  8,23, 15,30,  3,27, 26, 5, 11,26,
  39,11,  9,17, 36,15, 24, 5, 18,13, 13,10, 11,19, 39,26,
  27, 8, 21,11, 28, 4, 29,31, 27,16, 36, 6, 14,11, 38,22,
  35,27, 18,26,  6,28, 15,15, 10,33, 25,33,  9, 3,  5,24,
  39,13, 32,25,  9,20, 35, 5,  8,38, 35, 9,  9,22, 24, 4,
  28, 5, 20,38, 13,29, 38, 4, 26,14, 10,12, 15,32, 26, 9,
  22,31,  4,22, 11,31, 10, 8, 14,30, 21,16,  4,13, 20,20,
   3,18, 13,30, 26,34, 16,25, 29,38, 21,12, 27, 5, 21,39,
  22,11, 30,26, 30, 1,  3,38,  6, 9, 17,22,  3,11, 20,17,
  18,31, 23,10, 31,10, 28,29, 11,24, 24,14, 34,17, 37,29,
  37,35, 21,33, 12,29, 36,26, 39,22, 21, 1, 29, 6
};
const int p_40_14[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  213,
  /// Array of preferences
  26,28,  1, 6, 17,13, 21, 9, 28, 5,  6,33, 33,35, 11,24,
  19,34, 13, 8, 30,34, 29,31, 34,20, 27,31,  6,24, 19,23,
  18, 8, 11,37, 23,28, 32,13, 36,24, 23, 0, 35,28,  0,17,
  19, 2, 32,35, 32,26, 31,34, 14,31,  7,33, 13,32, 26,10,
   5, 8,  0,27,  8,22, 17, 2, 25, 1, 30,13, 14,39, 29, 6,
  10,18, 21, 6, 10,12, 10,10, 20,37,  8,12, 27,17, 29,16,
   5,35, 18,10, 29,24, 21, 5,  9,18, 36,36, 24,17, 26,24,
   7,30, 18, 2, 14, 4, 38,38, 16,14, 38,33, 39, 3, 16,24,
  10, 2, 32,36, 38,26, 17,11, 18,33, 12,14, 32,24, 30,18,
  23,34, 13,13, 30,22, 22,29, 35,10, 16, 0, 37,10, 34, 3,
   9, 3, 20,10, 12, 4, 23,11, 17,32, 29,27,  6,25, 20,21,
  23,32, 32,11, 33,33, 12,36, 37,22, 20,17, 20,18, 18,36,
  38,34,  0,30,  5,31,  6, 1, 36, 4, 28,37, 26,13, 17, 1,
  22, 6, 38, 0, 27,35, 19,35,  5, 2, 20,20,  9,24,  8,16,
  35,29, 11,19, 14,13, 34,11, 15,29,  7,24, 29,38, 20,23,
  13, 3, 34,36, 24,39, 23,36, 17,26, 30,30, 13,39, 17,30,
   1,36, 14,35,  6,36, 36,10,  9, 2, 25,25, 37,30,  2,27,
  29,30, 14, 5, 29,22,  5, 1, 14,20, 18,22,  6,27, 27,19,
  15,30,  4,38, 19,14, 38, 5, 20,33, 34, 7, 14,11,  5,20,
  18, 0, 12,32, 18,13, 27, 5,  2,35, 10, 5, 26,30,  8,17,
   5, 6, 26,34,  5,27, 17,21,  8, 1,  3,12, 33,18, 25,11,
  21,15, 39,37,  3, 4, 15,25, 19,10,  0, 8,  5,34,  3,23,
  37,28,  0, 5,  4,10, 38,18,  7,12, 10,38, 16,34, 21,39,
  11,11, 35,26, 14,10,  5,15, 20, 3, 18, 9, 30,14,  6,22,
  17, 0, 35,12, 15, 8, 22, 9, 26,17,  1,15, 21, 7,  1,33,
  31,37,  1,18,  7,25, 39, 5, 27,15,  9,21, 38,15, 37,12,
  18,31, 36, 1, 28, 0, 33, 7, 12, 9
};
const int p_40_15[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  324,
  /// Array of preferences
  20,27, 26,33,  9, 9, 34,20,  3,13, 11,14, 16, 6, 16,15,
  17,23, 17,31, 14,39,  1,17,  0,19, 39, 9,  5,20, 17, 9,
  19, 1, 35,17,  5,16, 23, 2, 11,30, 39,14, 35,12, 37, 8,
  25,10, 12, 7, 22,30, 21,29,  2, 4, 13, 5, 37,20, 36, 3,
  16,33, 11, 7, 21,13, 21,36, 30,37,  6,31, 28, 5, 11, 1,
   2, 8, 35,23,  8,26, 28,39, 23,38, 21,34, 29,11,  6,25,
  14,14,  3,10,  5,27,  1,27, 39,25, 33, 9,  5,31, 35, 4,
   6,11,  8,12, 13,23,  6,17, 25, 1,  3,27, 22,11, 33,38,
  15,19, 33,35,  7, 3, 28, 3,  7,23, 20,37,  2,39, 21,25,
  24,26,  5, 7, 34, 4, 23,31, 16, 3, 30,19,  3,30, 35,13,
  29,10, 28,28, 18,20, 12,16, 10,12, 28,26, 29,28, 36,16,
   3,21, 33,27, 19,30, 32,27, 33,25, 28, 9, 12, 6, 26,29,
   6, 5,  1,13,  4,26, 18,21, 29,19, 22,35, 37,27, 19,36,
   4,37, 20,26, 16,28,  1, 0,  2,25, 19,17, 25,15,  8,27,
  20, 0, 38,30, 37, 3, 39,28,  5,30, 32,37,  5,33, 19,20,
  36,37,  0,39, 25,18,  5,18, 35,28, 30,30,  4,10, 30,14,
  38, 5, 39,31, 34,10, 12,31, 33,33, 28,18, 30, 4, 25,14,
   2,22, 11,28, 17, 0,  8,38, 20,20,  8, 5,  6,21,  4,28,
  20,39, 34,26, 18,33, 22,32, 15, 2, 20,19,  2,38, 17,27,
  27, 9,  1,28, 34,21,  2,18, 14,27,  2,20, 22, 3, 22,33,
  20,21, 27, 2, 25, 7, 17,33, 33,32, 11,27, 23,30, 12,20,
  15, 6,  0,15, 35,22,  1,12, 28, 7,  4,11, 39,15,  4,24,
   6, 4, 19,32, 19,16, 16,29,  0,10, 34,28, 28,37, 23, 4,
  11, 4,  0,14, 20,33,  2, 5, 27, 1, 32,19, 39,37,  3,31,
  29,16, 14,29, 18,14, 26, 9, 17,37,  0,20, 12, 3, 35,24,
  19, 2,  2, 6,  0, 9,  4, 7, 25,38, 24,31,  1,20,  9, 6,
  31, 0,  4, 3, 30,21,  3,16, 30,39, 15, 0,  3, 1, 20,36,
   7,35, 39,12, 17,34, 21, 1, 38,34, 29,14,  6,28, 35, 1,
   0,18, 36,21, 36, 4, 21, 0, 17,38, 36,12, 30, 9,  6,10,
  28,14, 34,11, 31,17, 15,38,  1,31, 24,20, 10, 9, 31,24,
  38, 8, 28,27, 27,12, 37,24, 23, 9,  2,30,  2,29,  1,19,
   1, 7, 11, 6, 22,17, 12,17,  2, 7, 35,36,  4,19, 10,39,
  29,21, 32,11, 34,19, 28,21, 16,30, 12, 2, 14, 6, 32,25,
   9,14,  8,36, 28, 0, 11,29,  0,38, 32, 0, 10,33, 10,31,
   9,18, 27, 7, 28, 4, 20, 2,  7,25,  0,23, 16,34, 26, 1,
   8, 6,  4,38, 35, 5, 15,11, 25,13,  0, 8, 14, 0, 34,22,
   0, 1, 33,10, 29,27,  7,31, 39,22, 13,26, 30,25,  6,32,
  36,39, 25,29, 17,17, 16,31, 39, 2,  1,16, 35, 3, 34,15,
  23, 3, 14,31, 33,18, 29, 8, 30,27, 37,11, 29,30, 29, 1,
   7, 1, 17,21, 15,21, 12,22, 39,33, 15,33, 25,34, 35,33,
  16,11,  4, 4,  8, 1, 23,27
};
const int p_40_16[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  295,
  /// Array of preferences
  13, 9, 18,16, 27,38,  9,11, 38,23, 13, 3, 12, 1,  7, 7,
   1,31, 28,36,  0,26, 31,23, 39,34, 27, 1, 30,26, 30,32,
  23,32, 26,31, 13,32,  9,23, 35,28,  1,27, 14,38, 13,15,
  25,24, 38,29, 21, 1, 10, 8, 22,11, 21,28, 23,28, 30, 3,
  11,14, 13, 2, 11,29, 11,19,  9,37, 25,34,  7, 8,  0,31,
  39,37, 34,16,  4, 8, 30,22, 19,38, 20,36, 22,39, 29,37,
  26,14,  1,23, 14,19, 33,32, 26, 3, 22,35, 27, 5, 29,32,
  31, 3, 29, 4, 35, 6, 22,21, 36, 0, 31,34, 16,19, 26,29,
  28,24, 33,15, 18,15, 10,28, 30, 4, 35,11,  9, 5, 24,17,
  28, 4, 22,34,  3,35, 19,20, 25,23, 12,29, 30,25,  4,11,
  31, 1, 20,13, 30,23, 27,24,  9,21,  7,36,  5,10, 21,13,
  14, 9, 34,39, 19,27, 15, 8, 23,19, 33, 8, 17,19, 23, 2,
  36, 1, 19, 2, 18, 6, 37,26, 16,18, 20,15, 39,17,  1,19,
  36,12,  9,35, 11, 7, 18,28, 11,32,  9,13, 33,16, 35, 7,
   5,12, 12,38, 34,21, 32, 8, 34, 4, 17,37, 35,39,  6,18,
   8,37,  5, 4, 31,11,  1,37, 17,17, 39, 6,  2,33,  6,14,
   1,26, 37,36, 20,32, 33, 0,  7,13, 39, 1, 34,37, 37, 0,
  39,24, 28,11, 38,34, 36,39, 12,15, 23,36,  6,31, 14,33,
   7,22,  2,13, 36, 4, 25,27, 37,35, 34,29,  5,21,  1,20,
  23,22,  8,10, 31, 9, 37,30, 15,39,  0,22, 12,13,  4, 3,
  22,27, 12, 4, 15,31, 11,18, 36,29,  4,27, 11,36, 26,22,
  24,35, 14,18,  7, 1, 21,22, 21, 7, 14, 6, 29, 7, 37,32,
  15,24,  7,28,  8, 8,  5,13, 38,32,  3,10,  5, 7, 24, 6,
  20,26, 10, 0, 38, 7,  2,22,  1, 7, 18,26, 34, 5, 13,19,
   5,19,  9,28,  5, 3, 24,28, 38, 5, 30, 7,  8, 1, 30,12,
  28, 3,  8,16, 39,16, 13,18, 13, 5, 13,24, 18,33, 30,11,
  38, 9, 15,30, 19,36, 39,30, 18,32,  4,14, 19,11,  5,35,
   1,24,  4, 7, 36,31,  8,28,  2,12, 14,21, 36,34, 23,37,
  21,17, 11,28, 14, 5, 15, 5,  3, 5, 11,25, 17,31, 17,26,
  39,18,  5,24, 33,31,  3,24, 28,25, 23,12, 19,14, 36,24,
  20,11, 24, 9, 18,29, 17, 1,  4,39,  1,29, 15,11, 28,22,
  34,15, 32, 6, 37,39, 38, 2, 30, 5, 16,37, 37,20, 10, 6,
  27,29, 39,31,  5,16,  1,32,  8,33, 11,20, 29,27,  4,29,
  17,32, 15,35, 17,29, 11, 6, 26,34, 12,23,  3,29, 30,31,
  10, 2,  3,25,  4,19, 32,10, 29, 9,  7,25, 24,26, 10,29,
  31,37, 15, 4,  6, 3, 29,36,  2,11,  6, 4,  8, 6, 39,32,
  27,35, 27, 4, 23,35, 32,26, 25,19,  2, 3, 34, 1
};
const int p_40_17[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  187,
  /// Array of preferences
  18,35, 25,22,  8,14, 21, 2, 16,25, 31, 1,  0,12, 31,30,
  19,19, 23,23,  2,11,  8,27, 25,39, 18, 8,  3, 2, 24,23,
   3,32, 39, 7, 38,28, 28,18, 31,32, 19,15, 24,16, 37,18,
   3, 4, 15,29, 25,34, 36,34, 24,15, 17, 8, 31,37, 32, 8,
  35, 7, 39,13, 15,23, 15,24,  8,19, 22, 9,  9,11,  5,27,
  11, 0,  8, 5,  8,33, 38, 9, 16,36, 19,17,  6,32, 27,20,
   1,16, 26,13, 28,12, 35, 1, 27,10, 20,16, 14, 6, 39,35,
  23,11,  0,10, 25,20,  3,18,  0,26, 28,14, 28, 1,  6,34,
   6,38, 26,36, 17,33, 11,21,  7, 7, 36,12,  6, 7, 28, 5,
  37, 2, 10,33, 27, 8, 19, 1,  2,39, 16,28, 29,25,  7,17,
  16, 6, 38,12, 30, 4, 19,29, 17,15, 14,10,  4, 1, 31,35,
  22, 2,  6,15, 35, 2,  1, 8, 26,14, 29,30, 34,13, 19,24,
  12,14,  3, 0, 34,17, 23,20, 35,26,  1,35, 12, 6, 27, 1,
  16, 2, 22, 3, 21,38, 23,37, 25,29, 10,24, 21, 3, 17,17,
   5, 1, 10,28,  4,26,  6,23, 24, 3,  1,36,  6,11, 19, 5,
   7,12, 13, 6, 27,12, 19,31, 12,17, 25, 8, 30, 9, 23,27,
   8,34, 35,14, 19, 2, 28,15, 12,29, 31, 9, 28, 2, 39,34,
  34,28, 20,13, 21,26, 14,36, 18, 2, 36,22, 32, 6,  9,17,
  15,35,  0, 8, 31,11, 24,38, 39,31, 38, 5, 13,36, 28,20,
  24,27, 26, 2, 30,28,  2,10, 16,19, 38,19,  1, 6, 24,21,
  31,39, 24,24, 20,33, 32,16, 33,37, 23,21,  4,16,  8,37,
  20,25, 32, 2,  5, 4,  2,28, 28,17, 38,27, 15,12, 11,16,
  19,37, 26,10, 37, 7,  9,21, 26,17, 39,38, 16,37, 17, 5,
  16, 4, 23, 2, 34, 4
};
const int p_40_18[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  98,
  /// Array of preferences
   5, 9, 37, 4,  9, 3, 16, 5, 34,29,  1,19, 30, 9, 37, 9,
   8,24, 25,20, 39, 2, 38,39, 17,19, 38,22, 31,37,  2,35,
   1,28,  2,25,  7,17, 27,38, 24,36,  4, 1, 38,23, 11,19,
   6, 8, 36,24,  0, 1, 21,26,  6,16, 34,30, 14, 6,  6,33,
   2,20,  1,33, 22,11, 13,17, 20,27, 20, 7, 35,29, 37, 8,
  29, 0,  5,10, 12, 8, 10,22, 34, 0,  3, 3,  3,31, 27,24,
  27,14, 17, 2, 37,30, 37,23,  7,23,  8,15, 37,12,  9,19,
  11,17, 20, 2, 16, 4, 25, 9, 23, 2, 32,26, 30,23, 14,16,
  10,34, 28,34,  5, 3,  3, 6, 34, 4, 23, 1, 18,10, 31,34,
  28,22, 27,15,  2,38, 20,23,  8,28, 16, 7, 10,11, 29,28,
  39,14, 14,18, 22,12, 10,20, 36,28, 35,18, 28,10, 36, 0,
  36,14,  6,32,  7,35,  9,37,  0,34, 24,18,  8,31,  2,23,
  35, 4, 11, 4
};
const int p_40_19[] = {
  /// Number of people on picture
  40,
  /// Number of preferences
  95,
  /// Array of preferences
  36,27, 20,29,  2,39, 18,29, 22,23,  1,17, 24,20, 15,11,
  13,29, 29, 6,  6,17, 25,26, 21,13,  5,15, 17,10, 19,31,
   6,12, 32,16,  3,11, 38,23, 17, 2, 17, 1,  1,28, 37,35,
  30,33,  7,19, 16, 5, 20,31, 27, 9, 30,26, 32,39, 37, 5,
  29,20,  6,37,  2,15,  8,18, 28,18, 30,36, 34,10,  8,22,
  16, 1,  2, 6, 38,38, 32,24, 35,17,  1,18, 28,39, 28, 8,
  27,15,  9,26, 14, 5, 32,30, 25,23,  4,30,  5, 8, 16,35,
  20,18, 22,22,  2,10, 27,31, 14,14, 21,20, 33, 0, 14, 1,
   1,14, 24, 9, 14,38, 35, 8, 28,38, 37,17,  3, 7,  0,34,
   1, 0,  3, 2,  0, 7, 33, 4, 30,20, 34,24,  1,33, 24,14,
   1, 1,  4,35,  3, 4, 28,37, 15,22, 12,29, 37,11, 19,17,
   6,33, 11,33,  5,31, 23, 7, 33,14, 36, 0, 17,21
};

const int p_50_0[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  174,
  /// Array of preferences
   8,31, 20,47, 46, 7, 18,39, 29,30, 27, 2, 30,45, 20,25,
   2,12, 34,33, 33,23, 48,23, 29,39, 29,31, 44,46, 33,37,
  22,41, 14,16, 31,38,  2,14, 45,29, 33,48, 27,46, 28,19,
  13,45, 19,41, 38,30, 24,10, 10,28, 21,13, 39, 5,  4,47,
  25,37, 10,19, 44,39, 48,32, 37,10, 46, 1, 35,23, 24,29,
  34, 8, 48,26, 31,26, 10,44, 16,41, 46, 9,  9,23, 41,13,
  11,21, 32, 8, 17, 4,  2, 5, 17, 8,  9, 2, 30,41, 33,47,
  21, 1, 46,29,  6,15, 46, 5, 36,31, 29, 7, 14,47, 38,34,
  25,29,  0,12, 18, 2, 36, 0, 15,29, 27,15, 17,30, 38,47,
  21,34, 31,29,  4,41, 38,26, 39,40, 31, 9, 29,26, 13,44,
  32,49,  8, 6, 15,21, 27,32, 48, 8, 22,45, 31,12, 32,48,
   9, 4,  0,47, 40,27, 24, 6,  4,37, 19,18, 48,31,  3, 6,
  48,36, 19, 6, 39,27, 31, 1, 15,20, 26,16, 16,33, 10,46,
  47,19,  5,16,  6,30, 21, 3, 29, 0, 18,37, 44,12, 31,24,
  32, 2, 48,30, 14,48, 33, 2, 31,47, 18,49, 26, 5, 25,13,
   2,11, 35,21, 27,25, 16, 4, 26, 4, 14,44, 13,46, 19, 7,
  23,38,  3,35,  3, 5, 17,25,  5,36, 30,39, 17,36, 32,22,
  21,24, 31,48,  2,34, 47,12, 30,11, 35,13, 25,34, 12,42,
  41,36, 29,17, 35,35, 10,18, 48,11, 41,23,  1,37, 36,47,
  27,16, 29,15,  3,21, 19, 0, 27,11,  1, 5,  4,16,  7, 0,
   0,28, 15,26, 47, 6,  7, 8, 18,17, 47,32, 37,23, 28,49,
  48,28,  6, 2, 32,17,  1,36,  2,43, 28, 2
};
const int p_50_1[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  13,
  /// Array of preferences
  22,13,  0,46, 24,14, 14, 5, 41,36, 35, 5, 22,34, 27,12,
   4,10,  8,47, 26, 2,  5,41, 26,24
};
const int p_50_2[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  407,
  /// Array of preferences
  39,37, 48,39, 27,15, 28,11, 20,31,  9,16,  6,40,  0,16,
  20,10, 16,39, 13,17, 14,25,  7,34, 40,28, 13,25, 15,18,
  16,22, 31, 3, 11,21, 47,14, 47,33,  3,42,  2,32,  8,33,
  10,22,  9,45,  6,11,  3,39,  2, 7, 46,44,  6, 0, 33,11,
  15, 6, 49,39, 16,32, 30,17, 19,44, 47,38, 21,15, 27,21,
  46,21, 49,43, 16, 1, 48, 7, 13,26, 12, 9, 31,39,  3,28,
   1,37, 46, 3, 45,26, 43,19,  9, 5,  2,10, 37,41, 36,23,
  20,33, 25, 0, 49,46, 40,42,  9,36, 11,16, 19, 2, 11,20,
  18,27, 18,43, 41,21, 44,31, 25,17, 14,35, 47,39, 44,30,
   5,42,  3,31, 33,40, 39,19, 19,21, 10,38, 20,28, 17,43,
   7,15, 32, 3,  1,12, 21,30,  9,19, 37,24, 46,38, 27,33,
   0,29, 37,25, 26, 9, 23,14, 22,11, 43,42, 13,41, 16,30,
  35,18, 12,24, 47, 5, 10,25, 30,46, 32,40,  2,39, 23,37,
   5,20, 21,22,  3,26, 47,45, 32,10, 33, 8, 22,14,  1,45,
  14, 9, 30, 4, 13,21, 10,18,  0,11,  7, 7, 38,16, 35,17,
  10,19, 17,34, 23,30, 31, 8, 29,35, 19,24, 12, 8, 36, 8,
   1, 1, 23,24,  7,20, 44, 2, 15,42,  0,39, 12,31, 18,15,
  20, 0,  7,18, 10, 6, 14,44,  3,12, 16,43, 24,33, 28, 7,
  13,35, 18,17,  6,39, 42,22,  1,47, 42,24,  8,39, 34,38,
  42,18, 13,36, 15, 3,  8,30, 38,45, 30,38, 34,22,  9, 8,
  46,20, 38,20, 36, 9, 45,21, 25,34, 39,48, 37,44,  6, 3,
   2,14,  6, 1, 39,15, 35,14, 13, 0, 34,15, 42,33, 17,25,
   2,31, 30,12, 19,32, 21,47, 40, 2, 24,30,  0,24, 18,10,
  39,46, 38,14,  1,13, 23,38, 37,43, 31,13, 41,48, 32,44,
  17,29,  4,15, 30,15,  1,35, 37, 4, 25,19, 24,49, 24,44,
  21, 1, 29,24, 16, 5, 22,21, 21,44,  8,23, 44,20, 34,47,
  49,38, 29,20,  8, 0, 23,20,  4,18, 46,25, 49,48, 32,26,
  36, 2,  1,23, 41,49, 21,11, 46, 2, 18, 5, 10, 0, 23,26,
   4,36, 44,48, 35,13, 35, 9,  9,21, 44,13,  2,30, 17, 8,
  35,28,  0,46, 49,34, 25, 2,  4,25, 34,34, 20,38, 48,24,
  17,27,  0,38, 31,47,  1,28, 26, 1, 45,17, 26,43,  4,46,
  16,27, 20, 8, 17,28, 22,19, 45, 0, 10,40,  6,34, 16,21,
  45,37, 34, 9, 35, 8,  0, 8, 27, 2, 13,40, 24,35, 25,39,
  16, 2, 49,23, 22,36, 44,42, 39, 3, 48,13, 19, 9, 15,21,
  36,38, 22,49, 16,23, 40,30, 43,20, 20, 5, 24,29, 26,31,
  40,48, 41,32,  7, 4, 24,38, 30,47, 20,32, 27,38, 21,43,
  23, 3, 35,39, 20, 1,  2,13, 22,20, 16,47, 27,48, 22,27,
  25,24, 23,40, 37,46, 31,30, 31, 7, 18,30, 41,22, 40,39,
  36,37, 48,33, 15,27, 27,16, 16,41, 11,12, 25,25, 15,25,
   0,48, 23,29, 21,46, 24,14, 33,34, 35, 6,  4,33, 21,28,
  48,23,  0,31, 15, 0, 20,35, 33,41,  7, 8, 48,38, 47,29,
   6, 4,  5,22, 27,30, 24,15, 40,45, 24,23, 13,45, 38,28,
  23,22, 45,46,  7,40, 17,46, 39,45, 15,46, 15, 2, 21,36,
  21,27, 43, 5, 20,39, 35,27,  7,39,  3,34,  5, 3, 33, 2,
   4,26, 33,13, 45,13, 36,28, 42,19,  8, 1, 33,48,  5,46,
  39,17, 26,47, 31,16, 24,45, 47,31, 40,13, 31,45, 24,32,
  15,14, 40, 7, 48,41, 45, 6, 46,17, 37,38, 35,48, 11,46,
  38, 1,  3, 6, 22,37,  7,37, 45, 2, 36,43, 27, 6, 33,43,
  30,23,  4,27, 44, 1,  7,22, 25,27, 26, 4, 34, 0,  1,15,
  28,41, 19,36, 19, 0, 13,12, 12,30, 10,42, 49,15, 35,16,
   3,45, 47,49, 25,42, 34,35, 35, 5, 24, 9, 43,47
};
const int p_50_3[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  416,
  /// Array of preferences
   6,26, 15,13, 33,26, 20,38, 46,36,  3, 5,  4,13, 33, 6,
  39,38, 28, 0, 24,23, 37,16, 26, 4,  5, 5, 39,45, 39,32,
  20,23, 21,37, 46,44, 46,26, 31,22, 15, 4, 19,13, 25,12,
  24,29, 15,47, 48,39,  8,14, 22, 6, 16,10, 15,48, 19, 3,
  30,21, 29,40, 34,25, 23, 9,  9,14, 23,47, 44, 3, 22,13,
  26,21, 16,27, 32, 9, 27,11, 44,42, 48, 6, 31,48, 47,25,
  29,30, 48,27, 39,47, 28,11, 14,38, 23,44, 27,32, 49,29,
  44,16, 19,34,  0, 4, 47,14,  3, 2, 24,34, 15,27, 29,45,
  32,21, 10,29, 38,27,  2, 1, 29, 9, 20,15, 37,11,  0, 2,
  22, 7, 33,25,  8,23, 13,40, 46,33,  7,13, 47,44, 19,38,
  40,40,  8,45, 13,43, 22,32,  6,31, 11, 7,  5,37, 39, 6,
  18,48, 30,34, 47,39, 45, 6, 34,12, 42,35, 34,20,  3,30,
   4,49, 45,25, 29,25, 26,28, 41,35, 20,34, 32,27, 48,44,
  16,18, 40,18, 12,38, 32,19, 39,10, 11,33, 47, 4, 47,19,
   9,26, 47,45,  5,35, 23, 5, 30,35, 47,23, 45, 5, 27,40,
  30,45, 18,38, 10,17, 29,17, 35,39, 27,19, 11,18, 46, 7,
   0,28,  7,15, 34, 4, 21,36, 25,26, 14,30, 24,24, 37,17,
  26,42, 13,13, 30,28, 44,40, 10,33, 10,45,  8, 5, 11,40,
   3,17, 35,18, 29,10,  4,47, 25,34, 47,36, 18,12, 12,44,
  19,16, 13, 0, 13,21, 15, 0, 21,46,  6, 2, 22,41, 35,31,
   8, 9, 47,35, 46,24, 20,43, 19,42, 45,44, 42, 2, 13, 2,
  13,47, 14,15, 14, 6, 10,18, 27, 7,  2,18, 11,44,  1, 9,
  26,16, 15,45, 24,45, 28,38, 16, 3, 35,46,  2, 3,  8, 1,
   2,13, 28,44, 20, 5, 48,21, 26,12, 34,43, 30,37, 15,44,
  46, 2, 44,29,  4, 3, 26,49, 27,12, 29, 8, 41, 4, 42,15,
   3,37, 45,41, 40,30, 16,23, 19,27, 32,14, 13,16,  6,11,
  47,11,  5, 6, 36,32, 36,18, 35, 8, 25,18, 16, 6, 20, 0,
  30,48, 16,17, 20,19, 41,25, 41,48, 20,17, 27, 0, 35,38,
   5,31, 14,44, 40, 3, 36, 7, 40,26, 11,12, 19,10, 48,48,
   8,20,  8,26, 29, 4, 28,36,  5,21, 28, 3, 35,36, 39,40,
  13, 6, 37,46, 21,21, 32,16, 31,18,  5,14, 33,27, 18, 0,
  49,40,  4,15, 36,35, 45,23, 12, 4, 24,48, 24,46, 25,27,
  22,47, 17,20, 19, 8,  9,37,  7,33, 14,43, 30,25, 33,21,
  12,48, 21,31, 13,20, 20,39, 17,47, 43,46, 26, 2,  3,10,
  28,35, 28,29, 14,45,  9,22, 18,26, 13,24, 24,28, 22,21,
  27,14, 34,44,  1,17,  0,45, 43, 2, 48,25, 24,16, 21,44,
  22,10, 41,30,  5, 1, 11,42, 25,40, 41,26,  6, 3, 33,49,
  30,23, 25,46, 49, 3, 11,10, 18,15, 34,16, 30,18,  1,37,
   9,33, 22, 9, 11, 8, 25,23, 41,28,  8,10, 41,13, 22,49,
  43,24, 38,39, 26,39, 12,28, 39, 3, 37,42, 35,41,  3,18,
  33,14, 34,10, 26,44, 33,22, 33,43, 27,48, 47,37,  0,38,
   7,12, 18,13,  9,36,  1,40, 12,13,  0, 6, 43,42, 49,26,
  31, 1, 23, 2, 27,23, 36,12, 28,48, 44, 6, 22,15, 21,12,
  15,16, 29,44, 19,40, 43,10, 34,23, 30,31, 16,44, 35,11,
   7,35, 35,30, 49,28, 44,25, 11,11, 28,40, 12, 8, 25,17,
  12, 0, 41,19, 42, 0,  0,20,  3,21, 31,39, 40,29, 15,31,
   2,46, 11,32, 20,10, 27,44, 38,20, 17,31, 12,31, 20,49,
  33,38, 43,41,  4,34, 22,26, 26,31, 39,13, 35,42, 49,24,
   2,31, 26, 5,  1, 6, 31,42, 42,45,  7,41, 47,16, 25,33,
  14,32, 40, 1,  5,34, 32, 6, 28,30, 46,34, 27,10, 42,29,
  38, 6, 27,33,  3,31,  1, 0, 23, 7, 31,10, 49,18, 48, 4,
  16,28, 47,21, 25,25, 12,30, 32,48,  3, 1, 48,15, 20, 4
};
const int p_50_4[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  146,
  /// Array of preferences
  22,28,  9,37, 15,18,  0,30,  2,24, 29,42, 42,42, 41,44,
  23,46, 25,46, 25,36, 36,27,  8,36, 48,15, 12,26, 42, 9,
  36, 9, 44, 7,  4,34, 41,12,  6,46, 38,25, 37, 8, 16,35,
  39,14, 28,10, 34,12, 40,23,  8,46, 13,39, 47, 3, 49,28,
  26,10, 19,18, 36,22, 45, 9, 24,42,  2,43, 10,10, 29,27,
  29,22,  2, 6, 15,42,  9,30,  2,38, 49,41, 17, 6, 15, 1,
  24, 9, 44,44, 37,35, 29,43, 43,45, 27,39, 21,35, 17,48,
  21,22, 39,35, 14,12,  9,43, 41,21, 29,23,  4,19, 17,32,
  29,31, 24,14, 33,19, 13,31, 12,34, 26,36, 42,22, 17,14,
  15,44, 22, 3,  6,28, 47,36,  8,38, 17,30, 20, 2, 27,38,
  21,29, 48, 7, 17,25, 23,13, 33,22, 32, 2, 39, 5,  7,24,
  13,23, 38,39, 44,22,  0,42, 18, 9,  4,10, 26,45, 15,10,
  34,45, 18,19, 45,43,  0, 6, 16, 8, 33,13,  6,25, 45,35,
  22,17,  2,32, 42,46, 35,20, 14,20, 13, 0, 17,19, 25,26,
   3,40, 43,20,  5,20, 47,25,  7,33, 26,29, 12,40,  6,41,
  16,24, 37,11, 21,30, 37,22, 34, 9, 37, 9, 30, 4, 17,36,
  44,31, 45,33,  3,30,  5, 3, 19,40,  0,29,  5, 7, 49, 7,
  48,23, 23,29, 43,30,  1,10, 18, 2, 42,31, 13,44,  5,24,
   3,16,  8,26
};
const int p_50_5[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  39,
  /// Array of preferences
  12,22, 10,22, 16,38, 38,16,  8,34, 22,11,  7,45, 42,48,
  15, 9, 25,23,  0,37, 12,33,  4,47,  8,36,  9,19, 35,39,
  12,24, 37,13,  2,11,  3,14, 19,27, 24,21, 48,12, 47,36,
   8,26, 28,29, 36,35, 21,21, 32,22, 20,39, 29,36, 33, 7,
  29, 2, 42,43, 28,27, 22,26, 20, 4, 43,49, 49,40
};
const int p_50_6[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  200,
  /// Array of preferences
  39,31,  3,23, 10,43,  8,17,  0,31, 21, 3,  1,32,  9,45,
  49,33, 27,26, 41,34,  4,40, 19,21, 16,15,  2,30, 23,19,
  14,18, 36, 8, 35,35, 34, 5, 37, 7, 30,11, 24,22, 10,35,
  12,49, 30,36,  3,22,  3,36,  8,18,  6,28, 41,10, 32, 8,
  49,27,  0,44, 39,39, 25,45,  3,49, 13, 0, 33,39,  9,49,
  13,19, 15,46, 12,43, 19, 5, 37, 5, 42,22, 28,49, 46,26,
  23,11, 24, 6, 27,23, 27,14, 38,38, 43,35, 16,38, 34,46,
  38,22, 43,46, 41,11, 22,16,  6,46, 13,39, 48,39, 15,13,
  36,47, 45,32, 18,43, 47,16, 18, 5, 15,37, 11,47, 32,33,
  24,42, 27,22, 14,31, 41,22, 28,23, 26,35,  4,18,  1,31,
  21,35, 22, 1, 42,39, 28,25, 17,36, 24,38, 35, 6, 46, 0,
  48, 3, 13,49, 42, 3, 13,27, 30,21, 18, 1, 35,28, 31,45,
  16, 6, 19,39, 17,12, 36,27, 48,18, 46,20,  5, 7,  4,38,
   2,49, 19,43, 38,13, 11,24, 23,25,  2,48, 14,22, 30,28,
  43, 8, 25, 5, 16,47,  2,21,  7,35, 23,15, 33,22, 29,23,
  38,43, 32,40, 46,27, 18,24, 25,17, 18,25, 16,11, 36,25,
  39,44, 23,39, 10, 3, 35, 7,  8,16,  0,23, 14,46,  7,18,
  10,15,  1,10, 29,44, 43,41, 20,33, 29, 1, 29,31, 41,42,
  24,45,  9,32, 28,16, 42,28,  6,33,  0, 3,  4,29, 29,40,
  41, 5, 44, 9, 47,12,  2,15, 14,35, 41, 3,  4,12, 22,38,
  32,31,  4,11,  4,10, 41,17, 35,46, 32,10, 42,33,  1,28,
  45,16,  5,12, 21,41, 31,11,  0, 2, 49,28, 29,28, 30,17,
  30,26, 46, 3, 43, 0,  2,14, 48,48,  7,31, 28,17,  5,45,
  26,17, 39,14, 35,22,  4,27,  1, 1,  7, 4, 27, 9, 48,40,
  20,30,  9,29, 24, 2, 12, 2,  9,22,  5,39, 38,34,  1,41
};
const int p_50_7[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  335,
  /// Array of preferences
  30,43, 13, 3, 47,37, 37,24, 36,48, 39, 9, 12,43, 25,44,
  47,19, 15,42,  0,49, 21,43,  5,49, 47,39, 49,13, 15, 4,
  31,15, 15,13, 14,20, 37,25, 46,11, 43, 5,  1,47, 35,43,
  18,12, 10,17, 36, 2, 43,44,  5,46,  7,41, 12,45,  0,20,
  34,44, 41,43, 30,32,  9,33, 23,40, 41, 6, 28,17, 21,27,
  24,35, 21,22, 13,48, 39,48, 40,32,  9,20,  1,23, 28,12,
   1,21,  6,12, 21,19, 27,40, 47,41, 38,38, 45,25,  4, 5,
  37,18, 46,35, 14,35, 45, 5, 39,39,  5,32, 45,41, 19,12,
  43,29, 24,28, 26,17, 10,32, 14, 7, 11,40, 32,17, 49,29,
   7, 2,  1,30, 45,33, 14, 0, 17,36, 31,37, 14,29, 41, 0,
  32,45,  0,42,  3, 0, 41,42, 15,39, 37, 4, 26,32, 43,40,
  14, 2, 33, 6, 11,23, 26,19, 37,40, 35,42, 36,15, 33,42,
  25, 3, 37,42, 23,11, 29,30,  9,24, 32,11, 21,33, 42,18,
   9,27, 26,45, 33,38, 19,25, 26,42, 13,20, 12,25, 47,14,
  11, 4, 15,23,  8,31, 24,44, 10,48, 48,15, 47,33,  2,48,
  22,36,  4,24, 23,17,  0,14, 41,11, 15,24, 19,31,  8,18,
   0,23,  0,18, 29,35, 49,43, 26,44,  2,23, 45,22, 26,29,
  44, 5, 47,29, 28, 1, 11,32, 38,39,  7,46, 17,28,  1,16,
   3,49, 43,12, 33,11,  9,48,  8, 0, 26, 8, 24,25,  9,11,
  49,37,  5,16,  6,39, 22,19, 27,10, 46,39,  3,28, 13,23,
  16, 7, 35,47, 10,13, 27,45, 46,19, 11,28, 38,28,  8,16,
  30,44, 42,26, 31, 4, 15,37, 43,33, 17,43,  9,28, 22, 3,
  29,12, 26,41,  8,24, 12,24,  4,47, 14,21, 23,38, 41,30,
   4,13, 31,20, 41,25, 23,41, 35,32, 30,42, 17,38, 32,21,
  10,14, 14,36, 30, 7, 31,44, 18,33,  1,22, 21, 3, 33,26,
  48,18, 16, 8,  7, 0, 24, 1, 23,15, 43, 1, 15,25, 20,45,
  19, 6, 46,15, 15,29, 11,37, 40, 4, 37,32, 39,22, 44,33,
   1,40, 28,29, 11,49, 22,14, 25,13, 28,24, 49,45,  3,13,
  31,21, 32,37,  5,27, 19,23, 23,46,  7, 5, 40,44,  5,44,
  16,11, 45,36, 10,12,  0,34, 22,26, 27,33, 26,49, 47,44,
  48, 6, 42,11, 16,45,  5,29,  4,42, 25, 2, 29,23, 26,26,
  41,49, 14, 4, 20,36, 40,30, 39,25,  9, 0, 33, 1, 43,49,
   4, 0, 49,31, 23,44, 29, 5, 15,43,  6,46, 26,18,  2,43,
  37,49, 29,29,  2,42, 44,34, 43, 8, 34,10,  4,38, 18, 1,
   8,47, 27,41, 35,20, 27,44, 34,33, 24,29, 38,10, 28,26,
  36,39, 30,24,  1,24,  9,32,  4,14, 23,32, 17,48, 49,24,
  10,29, 16,22, 29,27,  5,15, 43,26, 36,29, 31,12, 10,47,
  41,36, 14,32, 21,23, 27, 3, 45,32, 49,39, 21,41, 47,49,
  14,38, 16, 4, 26,47,  0,39, 33,49,  3,26, 31,28, 19,29,
  22,45, 38,27, 27,17, 17,37, 15,27, 25,11,  2,37, 36,47,
  45,27,  1,20, 22,27, 44, 2, 30,35,  1,43, 44,46, 14,25,
  41,26, 45,48, 17,19, 46, 8,  5, 7, 19,36, 44, 6
};
const int p_50_8[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  428,
  /// Array of preferences
  36, 2, 29,23, 17,34, 22,40, 24,11, 47,34, 43,17, 14,29,
  25,25,  9,42, 24,19, 45,20, 16,41, 40,31,  8, 2,  7, 8,
   0, 7, 37,26, 24, 1, 26, 0, 49,21, 17, 5, 25,17, 19,17,
  48,17, 14,38, 29,28, 18,16, 16,16, 15,45, 37,16, 23,48,
  29, 9, 19,43, 19,40, 47,40, 30,49, 39,18, 12,35, 28,22,
  47,12, 33,48, 19, 5, 29,30,  1,46, 32,26, 27,27, 47, 8,
  31,17, 27, 0, 36,26,  8,38, 39,43, 28,13, 14,39, 26,34,
  28,49, 43,24, 30,17, 46,43, 31, 5, 32,43, 12,42, 12,39,
  27, 7, 15,29, 13,42, 39,23, 32,30, 44, 8, 34,10, 31,10,
  36, 6, 14,35, 26,13, 24, 2,  7,45, 28,14, 36, 7, 46, 9,
  29,26, 35, 6, 18,24, 27, 3, 49,24, 11,25, 29,14, 37,36,
  41,38,  0, 1, 49,49, 15,21, 14, 2, 40,41, 46, 2,  7,11,
  41,29, 39, 3, 33, 7, 27,28, 31,37, 45,16, 17, 9, 41,23,
  25,26, 34,46,  8,17, 47, 7,  2, 3, 42, 9, 11,44, 33, 8,
   9,41, 23, 8, 31,18, 16,14, 46, 7, 43, 3, 32,41, 29,37,
  33, 6, 34,48, 44,49, 28,15, 29,41, 33,37, 38,14, 19,24,
  23,40, 41,36, 41,19, 34,30, 34,14, 31,29, 24,35, 33,10,
  28, 8, 11, 2,  6,39,  1, 6, 18,23, 10,49, 26,21,  5,31,
  10,31, 31,20, 14, 1, 38, 5, 15,26,  9,28,  1,13, 15,32,
  31,42, 18,21, 45,27, 27,29, 10,15, 35,24, 33,25,  2,18,
   0,34,  7,49, 18,47, 18, 7, 43,36, 44,25,  5,49, 14,37,
  34, 5, 20,42, 13, 4, 22,15, 39,21,  0,26,  6, 0, 15,30,
  36,41, 25,11,  4,20, 38,41, 21, 6, 29,31, 48, 5, 31,23,
  42,33, 44,44, 12, 9, 16,27, 49,45, 41,49, 37,14, 30,26,
  15,38, 42, 3, 35,11, 40, 3, 13,36, 10,45,  4,39, 30,23,
   7, 4,  0,31, 21,33, 49,14,  3,47, 43,19, 11,22, 11,32,
  25,12, 49,12, 29,21, 41, 3,  4,25, 42,46, 25,22, 47,45,
  27,23, 38,10, 33,45, 40,32,  1,15, 29,25, 13,29,  5,22,
  30,24, 11,43, 26,15,  8,21, 23,49, 20, 9,  3,38, 49, 4,
  16,40, 29,34, 45, 2, 34,27, 39,15,  5,19, 30, 0, 23,34,
  47,38, 37,18, 25,35, 49,39, 19,15, 44,30, 26,41, 44, 0,
  35, 5, 24,44, 28,10,  4,41, 27,41, 47, 4, 48,48, 48, 3,
  30,42, 46,18,  0,38,  7,12, 20,30, 27, 4,  4,13,  2,42,
  12,14, 22,21, 17,27, 37, 3, 13,41, 48,44, 45,41, 39,37,
  20, 3, 29,12, 25,24, 19,22,  8,40, 15,11, 30,40, 31,22,
   3,17,  8,23, 18, 1, 24,30, 23,14,  6,24, 13,37,  1,35,
  32,11,  7,47, 43,48, 21,18,  3,44, 25,18, 15,17,  3,40,
   8,20, 41,35, 49,28, 35,20, 18,29, 11,16, 48, 9, 39,22,
  36,49, 21, 5, 48,37, 42,23, 45,49, 37,23, 40,46,  7,17,
  22,20, 39,41, 47,32, 12, 3, 35,15, 11,47, 45, 6, 35, 0,
   6,43, 33,34, 40,20,  8,22, 15, 6,  6,12,  4,49,  4, 6,
   2,24, 32,45, 41,34, 11,38, 10,18, 35, 7, 43,11, 34,45,
   6,49, 16,23, 15, 8, 20,40,  2,23, 46,36, 22, 2, 34,32,
  14, 0, 37,31, 20, 4, 16, 2, 36,37, 44,37, 29,17, 31,47,
  18,33, 30,13, 10,30,  5,10, 25,49,  9,44, 10,26,  5, 5,
  48,28, 17,12, 44,11, 27, 2, 23,32, 20, 2, 49,48, 21,13,
  45, 5, 40, 5, 22,43,  3,24, 29,48,  0, 3, 22,32, 27,21,
  27,45, 25,32, 18,36,  1,29, 49,20, 44,23, 42,39, 38,23,
   4,37,  1,36,  4,32, 20, 7, 18,19, 23,12, 31,15,  0,33,
  18, 5, 41,43,  6,45, 46,11, 30,16, 27,30, 14,44,  2,29,
   1, 1,  0,45, 10,38, 35,17, 38,35, 24,36, 35,19, 36,42,
  40,14,  2, 4, 37,37,  9, 8, 21,44,  4,26,  2,49, 48, 4,
  39,34, 42,37,  2,15, 29, 8, 38,30, 34,38, 11,29, 47,23,
  19,47, 39,14, 47,41, 49,18
};
const int p_50_9[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  192,
  /// Array of preferences
  33,11, 13,27, 31, 8,  8,15, 32,20, 34, 4, 11,29, 13, 3,
  20,41, 16,32, 48,37, 44,38, 28,37, 29,36, 20,13,  1,29,
  49, 7,  4,14, 22,33, 33,29,  3,38,  4,42, 18,42, 28,44,
  31,49, 26,35, 34,19, 43,47, 30,36,  4,32,  4,12,  0,11,
  33,24, 15,12, 33,36,  2,22, 37,32, 43,16,  1, 5, 36,49,
  48,32, 40,13,  3,42, 12,10, 32, 7, 12,32, 44,18,  1, 6,
  25,24, 26, 0, 18,12, 24,43, 20, 2, 11,46, 38,29, 49,16,
  32,38,  0,25, 19,49, 45,47,  3,23, 46,27, 45, 1,  0,38,
  34,36, 21, 2, 12,46, 29,27,  9, 5, 11,45,  9,46, 45, 5,
  22,35, 14,17,  8,44, 35,45, 37,12, 22,11, 12, 9,  9,11,
  40, 8, 41, 4,  1,10, 26,49, 22,22,  2,44, 20,18, 47, 5,
  38, 3, 42,11, 39,34, 27,27,  9, 9, 17,30, 40,23, 47,37,
  49,11, 19,37,  9,45, 48, 3,  7, 4, 49,39, 23,23, 49,44,
  13,40,  7, 1, 20, 7, 12,30, 47,19, 47,49, 18, 5, 42, 1,
   4,33,  5,34, 43,30, 26,24, 47,14, 46,31, 24,12, 30,26,
  10,10,  9,48, 30,30, 39,49, 42, 2, 40,46,  3,19, 16, 4,
  33,20, 44,21, 41,17,  0,46, 28,15, 41,27,  8,24,  1,17,
  24,29, 45,18, 13,46, 18,17, 23,47, 36,48, 21,43, 21, 7,
  10,20, 44,28, 28, 4, 29,47, 27,35, 14,22, 19, 6, 32,40,
   0,23,  1,23, 40,12, 48,41, 25,26,  7, 2, 47, 9, 35,31,
  43, 3, 27,10, 27,39, 43,38, 22,12, 19, 8, 36,38, 49,22,
  42, 5, 36,37, 29,18, 32,41, 28,35, 45,46, 37,28, 27,36,
  22,36, 42,45, 28,24, 41,10, 37,18, 11,33, 43, 5, 29,13,
  29,28, 49,40, 39,38, 13,37, 12,42, 32,23,  0,44, 27, 9
};
const int p_50_10[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  212,
  /// Array of preferences
  16,16,  6,28, 23, 5, 23,32, 16,49, 30,27, 23,13, 46,26,
   9,36, 49,12, 42,26, 30,46, 11, 2, 23,23, 14,28, 35,19,
  32,18, 20,43,  5,34, 27, 7, 27,28, 45, 1,  1, 6, 30,30,
  40, 0, 21,23,  8, 5, 27,29,  8,18, 23,20, 29, 4, 12, 7,
  35,47,  5,26, 43, 0,  1,18, 23, 8, 44, 8,  7,11, 48,13,
   7,30, 20, 6, 31, 3, 37,40, 46,23, 13,48, 28,16,  9,20,
   6,35, 41,34, 36,21, 25,14, 12, 5,  2,32, 33,20, 31,16,
  10,35, 39,48, 28,23, 36,27, 44,31, 47,23,  2,15, 29,30,
  18,40, 25,20, 17,38, 35,16, 41,38, 31,22,  3, 0, 48,25,
   8,22, 25,31, 13, 8, 11, 3, 43,48, 12, 1, 14,43, 39,44,
  42,45, 26,16, 11,45, 14,47, 18,34,  9,44, 48,34, 15,32,
  45,24, 35, 0, 13,16, 39,36, 22,38, 43,12, 43,27, 26,13,
   7,45, 44,34, 39,46, 25,22, 31, 0, 45,10, 49,45, 12,34,
  12,44, 45,41,  1,14,  3,28, 27, 8, 27,37, 31,11, 45, 8,
  24, 3, 25,35, 39,31, 30,44, 37, 0,  7,28, 23,19, 42,42,
  20,35, 46,32,  3,15, 44,26, 26,20, 16,23, 42, 5, 11, 6,
  47,17, 33,36, 34, 3, 37,19, 14,16,  7, 8, 18,28, 17,42,
  18,38, 33,21, 21,38, 39,14, 16, 6, 46,11, 30, 4, 44,39,
  27,19, 36,19, 37,20, 29,11, 44,36,  3,34, 11,17, 33,44,
  46,29, 32,23, 45, 6, 21,25, 42,15, 26,34,  3,38,  7,26,
   8,23, 37,41, 49, 7,  9,35, 37,46, 19,33, 26,18, 39, 9,
  22,15, 24,21, 29,25, 10, 6, 30,42,  6,45, 34,33, 37,45,
   4,28, 30,16, 18, 3, 34,36, 43,36, 45,17, 24,43, 45,15,
  11,35, 38, 5,  9,48,  3,49, 34,13, 23,36, 38,35, 14,14,
  49,35, 21,17, 24,16, 15, 0,  6,42, 20,47, 42,24, 24,17,
  35,38, 22,19, 23,33,  8,33, 18,26, 28,35,  5,15, 12,33,
  38, 0, 34,46, 17,13, 19,46
};
const int p_50_11[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  389,
  /// Array of preferences
  25,48, 36, 1, 30,45, 10,17, 49,16, 38,36, 25,16, 15, 0,
  40,36, 47,43, 45, 6, 31,26, 14,32, 41,18, 10,19, 20,12,
  45,49,  3,13, 17,20, 24, 0, 47,48,  7,47,  4,44, 27,26,
  34,12, 37,39, 26, 2, 14,45, 47,15, 10,33, 21,32, 39,21,
  28,18, 43, 3, 16,19, 28,19, 19,49,  5,49, 18,42, 46,16,
  24, 7, 13,26, 21,13, 43,27,  6,32, 25,20, 30,15, 32,14,
  19,16, 36,19, 40,42, 36,28, 29,34,  4,29,  2, 9, 33, 8,
  24,49, 14, 3, 17,36, 14,49, 36,47, 28,23,  5, 1, 22, 2,
  43,13, 42,39, 25,34, 31, 7, 25,40, 13,38, 20,18, 11,34,
  23,34, 20,44,  5, 2, 25,13,  9,20, 41,27, 46,23,  7,18,
  18,46, 22,25, 36,26, 23,12, 23,21,  0,15, 47, 9,  8,33,
  24,32, 35, 5, 11, 5, 31,30,  2,15,  7,48, 18,14, 22, 7,
  41, 6, 22,31, 14,15, 26,10, 12, 6, 45,18, 17,21, 37,14,
  16,42, 44,22, 47,46, 49, 4,  1,42, 26,17, 39,41, 30,24,
  23,19, 19, 5, 15,44,  1,31, 46, 2, 17,28, 47, 1, 40,43,
  10, 0, 20, 9, 45,46, 10,10, 24,40, 25,21,  7,20,  6,49,
   4,14, 41,35, 22,20, 44,30,  6,35, 47,13, 25,35, 49,39,
   9,43,  8,31, 47,47, 12,42, 28,29,  9, 8, 16,21,  9,19,
  15,13, 16,34,  5,48, 17,39, 12,25, 22,15, 30,46, 17,27,
   2,37, 28,30, 18,18, 11,32, 16, 9,  9, 1, 36,49, 31,35,
  13, 8,  7,40,  3,22, 16, 2, 42,28, 12,47, 19,43, 22,43,
  32,17, 36,25, 38,39, 35,43, 45,31, 41,39, 16,35, 47,40,
  29,43,  0,14, 22,29, 46,14,  6,42,  5,24, 30,42, 18,33,
  18, 1,  1,40, 37,35, 14,39,  0, 7, 42,31, 17,34, 16,28,
  10,46,  3, 8, 29,29, 46, 9, 23,33,  4, 8, 19,24, 47,22,
  25,41,  9,16, 32,25, 19,10, 19,13,  4,45,  9,12, 11,26,
  46,41, 11,45,  4,23, 10,42, 19,44, 15,17, 39,30, 28,32,
   9,34,  7, 4, 49,10, 32,29,  7,13, 24,42, 49,43, 43,25,
  21, 7, 35,39, 44, 9, 41,23, 40, 1, 12, 9, 44, 6, 11, 7,
  31,19, 10,20,  7, 3, 37, 0, 19,30,  5,43, 37,17,  9,32,
  16,14,  2,35, 15,19, 29,45, 30, 9,  1,10, 33,47, 15,40,
  34,41, 12,41, 42,29, 43,43,  5,19,  2,19, 35,22, 14,35,
  42,24, 37,28,  8,15, 24,23,  9,10, 22,17, 12, 5,  1, 1,
   5,38,  2,47, 32, 6, 13,43,  7,29, 26,42, 10,40, 20,45,
  13,39, 34,34, 49,11, 31,13,  8,44, 45,13, 11, 8, 11,42,
  49,22,  7,15, 14,24, 47,42,  4,31, 20, 8, 18,35, 48, 3,
  14,29, 28,48, 33,25, 46,49, 20,24, 28,15, 45,19, 18,32,
   6,40, 39, 1,  9,27, 35,26, 24,17, 47,18, 24,39, 12,26,
  38,23, 48,34, 34,27, 12,45, 36,30, 30, 4, 10,38, 47,45,
  18,36, 34,31, 23, 3, 23,26, 24, 3, 29,39, 42,15, 15,27,
  23,15, 39,18, 40,40, 14,26, 40,33, 31,31, 17, 6,  1,26,
  26,37, 24, 1, 41,10, 25,43,  7,45, 41,31, 35, 9, 42,11,
  38,34, 45,43, 22,19, 19,12, 34,14,  2,14, 13,18,  5, 5,
  29,12, 26, 8, 26,34, 42,36, 36,39, 31, 4, 17, 1, 30, 0,
  48,16, 30,29,  0,47,  2,43, 41,45, 23,35, 24,31, 12,28,
   1,49, 26,29,  9,17, 10,45, 20,49, 43,11, 39,35, 46,33,
  26,20,  1,38, 38, 0, 27,12, 45,37, 26, 5, 25,27, 20, 6,
  38,45, 29,13,  1,13,  5,22, 39, 3, 33,32, 42,33,  6,15,
  32,13, 34, 4,  9, 9, 27, 0, 38,25
};
const int p_50_12[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  107,
  /// Array of preferences
  49,27, 11,17, 33, 0, 48,49, 25,27, 14, 8,  6,22, 11,15,
  17,23, 33,27,  4,33,  0,21, 36,28,  2,42, 42,11, 38,13,
  28,13,  9,33, 20,39, 40,38, 25, 6, 21, 3, 28,10,  2,18,
   8,13, 33,23, 49, 3,  7,22, 17, 6, 19,23, 35,33,  3,13,
  36,49, 46,42, 18,10, 38,47, 47,23,  5,10, 22,16, 45, 2,
  27, 2, 32,28, 13,40, 49,28, 15,31, 21,37, 48,46, 25,16,
  37,34, 18, 4, 34,18, 10, 5, 13, 5, 26,43, 15, 0, 12,14,
  24, 4, 10, 8,  3,47, 21,17, 42,49, 11,30, 16,28, 26,26,
   6,26, 18,23, 47,49, 12,10,  0,28, 19, 0, 16,41, 32,36,
  17, 1, 43, 6, 12,16, 24,25,  5, 1,  4,26, 46,49, 46,19,
  39,41,  7,30, 24,20, 41,19,  1,41, 40,24,  7,11, 23,12,
  49,21, 44,46, 25,44, 10,24,  1,42, 25,32, 20, 9,  3,26,
  28,26,  1,46,  4,18,  0,32, 37,33,  3,38, 30, 2, 26,28,
  23,31, 20,21, 31,49
};
const int p_50_13[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  623,
  /// Array of preferences
   7, 2, 21,21, 45,47, 13,25, 46,26, 47,15, 34, 8, 46, 8,
   4,43,  7,12, 31,17, 23,13, 46,20, 28,45,  8,17, 19,18,
  23,39, 35, 6, 45,14, 39,45,  2,30, 21,42, 21,30, 21,20,
  24,24, 22, 2, 38,24, 41,46,  3,35, 16,13, 41,19, 10,31,
  23,36, 40,13, 28,30, 48,25, 33,35, 19,12, 42,48, 14,45,
  25,18,  2,46,  6, 7, 43, 5, 20,15, 42, 6, 49,42, 49,10,
  37, 1,  3, 7, 38,43, 46, 0,  8,33,  4, 3, 47,48, 13,30,
  24,21,  6,41, 15,25, 28,27, 23,25, 38,42, 25, 6, 32, 9,
  35,43, 45, 0,  2,24,  2,41, 31,28,  4,45, 46,28,  9,40,
  16,16, 35,15, 41,28, 31,47, 23,14, 40,29, 43,37, 49,48,
  21,41, 38,45, 30,16, 43, 8, 18,11,  5,42,  9,38, 16,30,
  49,29, 40,42, 14,35, 34,16, 33,23, 15,36, 21,46, 38,25,
  21,32,  2,45, 27,33, 42,35, 21, 7, 47,14, 36,44,  1,45,
   3,25, 15, 3, 33,28, 36,24, 42,43, 13,46,  2,38, 15,35,
  20, 6,  4,19, 38,32, 28, 7, 47,28,  3,40, 15,20, 31,35,
   5,46, 14,34,  0, 5,  4,11, 43,41, 20,36, 34,47, 15, 2,
  14,20, 44,48,  9,21,  3,43, 15,33, 33,19, 22,22, 45,16,
  21,44, 41,40, 49,17, 15,27, 37,40,  7,25, 24, 5, 41,48,
  43, 6, 47,12,  7,48,  5,38,  9, 9, 22,43, 39,19, 17,19,
   5,45, 16,39, 42,36,  3, 4, 36,18, 19, 1,  9,18, 36,26,
  14, 6, 48,27,  5,23, 44, 8, 39,17, 13,11, 43,43, 30,33,
  40,22, 38, 9, 23,10, 40,33, 23, 5, 18, 6, 23,20, 27,20,
  17,20, 36, 7, 48, 3, 35, 8, 33,31, 20,28,  0,34, 23,24,
  38, 6, 27, 2, 22,37, 26,47, 23,15, 15,37, 19, 2, 31,30,
  20, 8,  5,26, 47, 0, 39, 1, 24,28, 25,19,  5,41, 12,48,
   9,42,  9,41,  8,48, 25,24, 23,45, 23,33, 14, 9, 36,39,
   9,17, 33,22, 25,42, 16,49, 34,32, 26,49,  2, 9, 42,45,
  23, 7,  7,18, 34,22, 40, 0,  9,15, 37, 6, 48,17, 19,37,
  17, 6, 21,47, 46,46, 17,12, 29,31,  6,23, 36,27, 28,47,
  27,24, 27,22, 23,29, 29, 2, 45, 8, 18,30, 49,41, 41,30,
  33, 1,  9,20, 38, 0, 20,16, 24, 9, 47, 5,  3,47, 37,18,
  37,12, 33,34, 34,34,  4,15,  3,12, 33, 5, 26,32, 44,35,
  37,23, 37,44,  7,21, 32,14, 49,32, 29,13, 14, 4, 25,14,
  30,36, 10,30, 29,30, 22,24, 20,22, 25,48, 38,16, 23,32,
  19,21, 43,17, 24,47, 46,14, 29, 9, 32,32, 48,45, 11,44,
  22, 1,  8,36, 47, 1, 47,13, 29,17, 37,27, 11,48, 20,13,
  14,25, 17,46, 28,23, 44, 4,  1, 5, 12, 0, 12, 7, 10,45,
   6,30, 13,29,  4,47, 11,31, 36,31, 23, 3, 32,23, 12, 1,
  47,25,  8,29, 32,12, 22,31, 37,22, 14,32, 10, 8, 15,40,
  36, 1, 47,10,  0, 7,  9,45, 10,16, 20,14, 17, 9, 16,46,
  39,23, 41,13, 35, 1, 43,38,  5,44,  8, 0, 42,13, 31,14,
  17,28, 43,46, 41,32, 22,29, 14,42, 31, 3,  6, 4, 13,15,
  12, 4,  4,23, 19,48, 10,25, 25,15, 42,44, 17,29, 34,17,
  12,20,  5,32,  2,34, 23, 0, 47,38,  8,34, 16,10, 35,40,
   3,49,  3,29, 24, 2, 47,41,  9,49, 27,21, 40, 7, 42,41,
  20,18,  7,33,  6,13, 38,49, 20,35, 47,21, 24,26, 16, 2,
  39,41,  6,40, 12,23, 46,22, 28, 0, 11, 7, 43,48, 47, 3,
  44,18,  5,28, 22,10, 12,10, 44,46, 43,36, 45, 2,  9, 4,
  29,28, 27,35, 31,12, 14,11, 42,25, 42,12, 19,11, 48, 6,
  43,33, 39,24, 40,37, 18,47, 15,17, 27,46, 25,20, 14,43,
  12,37, 39, 4, 26, 5, 46, 7, 23,41, 17,13, 26,20, 45,41,
  40,49,  0,10,  4, 8, 27,13, 31,32, 20,29, 20,45, 23,12,
  45,19,  5,34, 42,14,  0,14, 45,43,  2,28, 19,10, 17, 8,
  17,21, 40, 6, 44,34, 24,43, 31,49, 15, 8, 10,29, 30,20,
   8, 6,  1,13, 31,19, 44, 2, 35,29, 25,22, 29,21, 27,27,
  25,11, 38,10,  0, 0,  9,33, 32,37, 31, 2, 34,30, 18, 1,
  37,32, 11,29, 44,41, 41,11,  6,37,  0,15, 22,16, 18, 5,
   7, 6, 39,31, 27,14, 40,47, 34,31, 43,13,  9, 6, 36,47,
  27, 7, 42,17, 37,15, 22,39,  5, 1,  2, 8, 29,14, 42,18,
  27,25, 12,41, 34,21, 22,44, 32,31, 43,45, 16,23, 48,24,
  23,38, 25,37, 18,14,  7,22, 41,29, 34,45, 30, 6, 46,34,
   2,32,  4,44, 17,40, 13,39, 49,15,  9,32,  5,27, 23,11,
   3,41, 44,49, 26,35, 14,49, 40,10, 31,26, 47,45, 46, 4,
  12,34, 13, 5,  5,33, 21,13, 46,15, 30,31, 30,17, 47,20,
  40,36, 11,22, 13,33, 39,35, 17, 7, 14,12, 30, 7,  4,27,
  46,12, 30, 8, 27,11, 28, 6, 18,13, 40,21, 22,30, 32, 4,
  41,16, 38,38, 17,45,  9,26,  6,18, 34,11, 36, 5,  9,27,
  44,23, 33,17, 47,46, 12,44, 32,45, 48,31, 20,32, 18, 3,
  41, 8, 16,42, 43,30, 12,39, 26,41, 32,11, 38,34,  2,16,
  34,25, 38,41,  4,30, 33,15, 26,38, 22,33, 16,47, 32, 0,
  45,38, 20,12, 44, 5, 30,46, 23,43, 20,40, 34, 3, 28,34,
   0,13, 21,23, 35, 0, 26,48, 36,19,  1,11, 13,34, 40,27,
  41,39, 42,49, 28,49, 44,26,  0,21, 15,14, 30,47, 20,11,
  23,17, 49,40, 31,27, 24,29, 49,47, 14,27,  4,40, 47,26,
  12,24,  6,28, 10,36, 12,14, 48,38, 11, 5, 20,44, 47,34,
   5,39, 12, 3, 31, 0, 34,46,  0,43, 31,37, 16,20, 43,35,
  24,11, 45,18,  2,17, 41,20, 37, 5, 21, 3,  5,30,  0,47,
  47, 9, 26,43, 43, 0, 33,48, 34,26, 36,11, 41, 2
};
const int p_50_14[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  475,
  /// Array of preferences
  28,35, 41, 3, 19,47,  2,23, 10,34,  6,38,  2,39,  6,12,
  40, 7, 30, 6, 43, 5,  2,15, 36,13, 13,15, 26,26,  9,47,
   5,26, 49,33, 36,16,  3,42,  7, 5, 41,23, 16, 4, 13,36,
  33,49, 42,44, 31,46, 38,33, 44,38, 35, 9, 18,28, 17,14,
  10,26, 47,36, 31, 4,  3,23, 45,36,  9,40, 48,38,  3,47,
  36,25,  2,27, 39, 6, 33,36, 37,49, 23, 8, 45,46, 28,30,
  30,21, 39, 7, 41,11, 22,26, 20,27, 38,11, 35, 1, 37,40,
  49,13, 29,29, 36, 4, 33,32,  6, 7, 28,21, 35, 7, 13,12,
  34,42, 23,27, 28, 6, 37, 1, 14,12, 35,49,  3, 2, 38,46,
  18, 0,  2,26, 33,23, 46,24, 24,35,  8,24,  3,43, 12, 9,
  35,21,  5,33,  0,36, 41,34, 11,28, 21,40,  5, 0, 47,29,
  16,10, 41,41, 10, 4,  9, 0,  3,25,  3,32, 31, 8, 44,41,
   7, 4, 19,20, 45,39, 35, 6,  2,22, 48, 2, 19,38, 37,24,
  24,40, 40,16, 38,24, 16,48,  0, 2, 46,23,  1, 7, 17,12,
  10,46, 41, 9, 32,14, 25, 2,  8,27, 49,24, 38, 6, 45,19,
   9,43,  5, 4, 28,24,  2,31,  9,28, 35,25, 13,48, 42,43,
  31,34, 32,15, 20,16, 14,49, 36,38, 34,19, 27,27, 36,42,
  41,17, 46,43, 14,47,  8, 4, 24,31, 49,35, 36,48, 45,37,
  33,41, 41,33, 32,48,  7,14,  8,10,  7, 0,  3,27, 39,39,
  25,39, 24,39, 14,32,  1,39, 23,41, 43,24, 34, 4, 38,39,
  35,39, 18,20, 25,42,  4,10, 12, 5, 29,14, 11,29, 24,41,
  11,13, 36,34, 34,32,  4,23, 49,45, 18,32, 27,14, 23,47,
  46,25,  4,45, 32,45,  7,16,  1,27,  9,25,  8,36, 20,30,
  42, 2, 32,30,  9,29,  8,12, 28,36, 32,19,  4,35, 47, 6,
  30, 9,  1,42, 36,28, 42, 9, 27,35, 14,19,  9, 7, 44,21,
  38,43,  0,16, 12,47, 38,10,  9,10, 49,44, 28, 7, 43,27,
  44, 5,  0,22, 11,44, 34,30, 39,49, 48,28, 44,10,  6, 6,
  45,12, 47,25,  1,44,  6,14, 24,26, 21, 0, 38,35, 45,23,
   1,21, 25,17, 16,49, 16,42,  7, 7, 49,47, 36,17, 45,32,
   0,34, 28, 5, 22,10, 43,37, 43,14, 21,11, 42,48, 46, 9,
  37, 0, 11, 4, 40,42, 20,33,  4,30, 39,35, 22,35, 13,25,
  13,14, 23, 6, 11,25, 10,45, 43,45,  7,15, 27,44, 21,48,
   7,23, 17,45,  6,11, 32,25, 43, 8,  1,49, 33,43, 22,28,
   1,18, 27, 1, 21,19,  9,38, 22,34, 19,33, 29,27, 26,38,
  23,30, 46,45, 14, 6, 48, 7, 17,22, 31,36, 26,28, 34,26,
   0,47,  0,43,  3,14,  5,16, 14, 2, 20, 2, 43,15, 16,33,
  18,19, 13,32,  7, 1, 19,18,  7,17, 49,27, 36,47, 25, 7,
  27,22, 13,18, 19,17, 28,40, 38,12, 48,26,  0,37, 35,34,
   1,40, 34, 8, 34, 7, 39,13, 40,31, 39,14, 27, 8,  4,18,
   6, 4, 14, 4, 35,11, 29,28, 36,37, 35,33, 19,27, 22,47,
  48,25, 47,13, 25,11, 30,24, 18,40, 25,28, 48, 3,  3,21,
   4,49, 49, 6, 13, 9, 40,13,  6,37, 41,27, 41,22, 21,32,
  38,32, 16,14, 23,11, 15,11, 10,37, 42,25, 43,42,  9, 8,
  35,44, 20,14,  7, 9, 48,29,  6,48, 16,20, 45, 1, 42,15,
  48, 8, 28,11, 45,18,  8,18,  8,23,  3, 1, 25,16, 19, 1,
  16,15, 19,31, 16,32, 38,48,  4,16,  8,11, 30,38, 33,45,
  15,40, 38,38, 14, 1,  1,41, 22,16, 36,24, 14,44,  3,17,
   9,46, 14,43, 13, 2, 24,17,  3,29, 32,42,  7,45, 30,41,
   9,12,  2,25, 47,31, 21,26, 36,23, 44,29, 22, 5, 49,15,
  22, 4, 20, 9,  2,20, 36, 7,  5,11, 27,15, 18,44,  2, 2,
  33,12, 46,32, 21,10, 12, 6, 35,37, 15,13, 47,22, 37,37,
   1, 2, 41,19, 16,43,  4, 4, 17,21, 14,29, 23,18,  4, 5,
  22,24, 20,19, 24,36, 35,13,  8,21, 31, 2, 12,33, 37,46,
  16,37, 10,23, 48,44, 44,43,  4,25, 37,41,  2,47,  6,10,
  22,12, 46,11, 20, 3,  4,32, 24,32,  8,16, 20,42,  6,30,
  32, 7, 12,42, 25,44, 24,23, 25,35, 13, 5, 37, 5, 33,26,
  44,12, 40,35, 21,33, 47,21, 34,29, 37,43,  7,33,  8, 9,
  47,18, 31,21, 20,18, 27,37,  4,17,  2,44,  7,20,  2, 3,
  39,33, 30,28,  7,36, 48, 0, 17,10, 24,11, 42,35,  4, 6,
   7,38,  2,38, 17,23
};
const int p_50_15[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  155,
  /// Array of preferences
  16, 1, 43,18, 23,37, 11,48, 45,44, 10,41,  4, 0, 20, 4,
  39,24, 39,11, 46,19, 19, 5, 21,48,  8,17, 47,28, 16, 5,
  48,14,  9,45, 48,32, 38,36, 42,45, 47,17, 18,45, 32, 6,
  21,29,  8, 4, 39,30, 24,29, 36,17, 21,42, 35,42, 41,12,
  11, 6,  6,31,  5,10, 32,47, 20,48, 24,12, 35,23, 38,13,
  37,24,  7,36, 31,22, 24,34, 46, 6, 40,14, 35,27,  5, 0,
  13, 0,  9, 7, 48, 4, 24,23, 22,24, 39,22, 34, 0,  6, 0,
   7,31, 23,10,  6, 5, 10,27, 39,44, 27,46, 34, 7, 47, 3,
  45, 7, 27,22, 30,33,  0,38, 38, 1,  2,19, 12,20, 38,21,
  49,31, 45,37,  9,15, 13, 8, 27,42,  1,46,  6,23, 31,44,
  19,38, 49,26, 24,30, 35,12, 24,27,  1,39, 42,37, 44,36,
  34, 1, 25,41,  3,34, 35,29, 37,19, 28, 9, 22, 1,  8,15,
  17,20, 43,46, 45,18, 31,30, 43,49, 30,18, 25,21,  2,11,
   5,12, 42,31, 40, 5, 16,31,  6,19, 13,27,  1,35,  4,21,
  43,15,  9,24, 26,16, 12, 2, 39, 9, 49,11, 43,39, 42,39,
  14,27, 18,33, 32,49, 26,49, 13,33, 27, 4, 41,18, 41, 8,
  18,32,  2,35, 23,42, 15,22,  9,47, 23, 1,  9,42,  3,19,
  17,36,  3, 0,  0, 4, 25,44,  4,45, 45,16, 20,31, 33,14,
  38,23, 39,32,  6,20, 23, 3, 40,49, 41,45, 28,22, 29,47,
  26, 4, 40,38, 18,36
};
const int p_50_16[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  74,
  /// Array of preferences
  33,37, 13,43, 28,12, 42,24, 47,41, 47,13, 20,46, 26,46,
  37,33, 24, 0, 25,40, 42, 9,  8,47,  7,12, 19,37, 19,48,
  44,33, 46,47,  8,15, 14, 0, 34,49,  1,47,  3,43, 18, 2,
  10,47,  8,48,  5, 2, 11, 8, 12, 9, 47,40,  6,27, 17,44,
   8,10, 17, 6, 45, 9, 36,13, 25,24, 38,38,  5, 1, 13,15,
  29, 9, 17, 3, 32,13, 41,20, 19,10, 47, 3, 34,12,  2,27,
  47,17, 31,49,  3,37, 42,23, 10,17, 41,18,  0,12, 23, 9,
  45,32,  9,49,  6, 8, 20,16,  5,23, 34,14, 38,46, 41,15,
  26, 4, 30,26,  6, 5, 49,13, 33,35, 29,12, 16,39, 31,14,
   5,18, 35, 9
};
const int p_50_17[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  495,
  /// Array of preferences
  44,23, 33, 7, 25,39,  8, 7, 26,34, 19,38, 45,11, 15,47,
  12,42, 37,14, 20,42, 19,42, 48,36, 41,41, 36,43, 15,33,
  20,41, 22,30, 18, 1, 27,20, 35,45, 27, 1,  3, 1, 14,42,
  38,45, 38,35,  9,19, 15, 6, 13,22, 11,31, 35,10, 22,27,
  32,38, 40,22, 15,41, 25, 1, 47,24, 16, 0, 48, 5, 24,19,
  46,44, 11,21, 35,44, 10,48, 37,20, 39,46,  1,39,  4,23,
  47, 5, 14, 5, 35,31, 44,11, 29,44, 12,23, 39, 7, 22,29,
  46,13, 46,49, 21, 1, 46,26, 46, 7, 20,49, 22,18, 31,38,
  17, 0,  2,30,  4,34, 11,23, 46,42, 44, 7,  1,16, 29,22,
  45,46, 31, 4, 11,30, 47,30, 29, 1, 33,28, 49,43, 39, 8,
  46,18, 37,27, 43,45, 42,34, 26,45, 30, 0, 11,40, 41,32,
  17,26, 45,49, 24,22, 35,19, 21,17,  8,11, 45,12, 38,41,
  31,29, 23, 9,  1, 4, 13,40, 40,30, 21,31, 38,47, 31,27,
  35, 9,  0, 0, 21,28,  3,36, 30,15, 26,37, 34,13,  2,13,
  23,21, 24,32, 28, 7, 13,25, 49,37,  5,36, 47,16, 20, 9,
  28,27,  4,46, 30,23, 17,40, 19,21, 48,20, 27, 5, 12, 0,
  19,40, 27,36,  1,44, 30,17, 37,40, 19,33,  9,46, 44,24,
  48, 2, 13,34, 19, 3, 41,10, 46, 1, 10,39,  9,21, 32,49,
  10, 1,  7,11, 19,28, 47,45, 49, 1, 44,12, 40,15, 35,35,
  49,27, 19,22, 25,21, 40,10, 16,15, 16,42, 34,18, 16,23,
  46,41, 13,13, 27,41, 17,15, 15,25, 16,16, 22,39, 15,15,
  32,19,  3,46, 16,11,  5,25, 36, 4, 18,27, 19, 4, 24,15,
  46, 6,  8,38, 32,29, 41, 5, 15,49, 24,28, 40,43, 45,20,
  42,27, 12,18,  8,27, 49, 3, 29,39, 48,34, 27, 8, 45, 6,
  43,15,  4,26, 25,26, 33,20,  3,43, 32,28, 40,40, 45, 9,
   7,12,  9,48, 17, 7,  6,13, 41,28, 43, 4, 45,32, 49,14,
  39,14, 45,43, 38, 1, 39,27, 17,13, 18,41, 43,46,  3,29,
  35,28, 45, 2, 17, 1,  9,14, 18,11, 11,12,  0,26,  1, 5,
  23,33, 30,21, 18, 6, 28,29, 11,42, 12,33, 35,27, 45, 1,
  26,22, 37, 9, 23,14, 36, 6, 24,33, 16, 4, 26,36, 49,25,
  30,39, 11, 3, 24,12,  5, 1, 20,45,  5, 9, 24,18, 44,44,
   1, 1, 10,21, 13,42, 34,40, 38,39, 19,26, 36,34,  7,30,
  37,49, 18,12, 36,23, 41,25, 40,47, 42, 8, 25,17, 49,49,
  46,32, 25,12, 29,16,  8,21,  8, 9, 37,15, 32,18, 36,36,
  16, 7, 24,48, 29,26, 31, 1, 20,16, 13,27, 33, 3,  1, 9,
  33,34,  3,34, 43,40, 36, 0,  6,14, 29,24, 21,34, 18,19,
  20,28,  2,20, 29,43,  9,28, 27,15, 13, 9, 13,10, 25,27,
  19,37, 23,37, 14,19,  9,15, 37,31, 34,45, 41,16,  0,27,
  24,34, 24,44, 33, 0,  3, 8, 28, 3, 27,11, 36,35,  4,33,
  46,45, 48,14, 35,24, 15,12, 17,48,  5,31, 34,14, 19,45,
  23,40, 39,11, 29,40,  2,44, 43,21, 20,38, 14, 3, 23,23,
  36, 8, 49,28,  8, 0, 43,48, 28,11, 32,46, 37,38, 43,24,
  27,25, 44,29, 41,20,  0, 2,  4,47,  0,17,  3, 3, 28,10,
   7,44, 44,21, 36,12, 11,44, 27,32, 25,14, 12,48,  2,18,
  20,34, 48, 6, 48,10,  6,39, 42,19, 33,17, 38,21, 20,15,
  18,33, 31,20,  7,42, 12,36, 41, 3, 21,14, 30,26, 49,42,
   4,27, 39,48, 22,20, 20,23, 23, 2, 40,24,  8,28, 30, 6,
  33,43, 10,28, 23,10, 11, 5, 19,15, 48, 4,  1,35, 38,44,
  40, 2, 46, 3, 35,43, 21, 3, 30,10, 26,24, 30,36, 13,32,
  27,30,  3,15, 33,49, 21,46, 21, 5, 30,27, 21,30, 16,20,
  46,22, 22, 2, 25,33, 13, 6, 17, 5, 14,33, 12,35, 25,46,
  23,47, 24,37, 48,28, 38,26, 42, 6, 25,38,  2,43, 48,15,
  37,18, 36,14, 49,10,  5,16, 28,47, 16,41, 21,36, 19,46,
  13,37, 18,13, 30,29, 34,17, 23,32, 18,48, 49,34, 20,20,
   8, 2,  5,41, 12, 7,  3, 2, 24,30, 37,37, 30, 8, 48, 0,
  29,49,  1,31,  6,33, 27,27, 39,22, 35,15, 48,27, 17,36,
  25, 2, 13,26, 37,35, 44,28,  9, 6, 14,17, 23,20, 26,27,
  40,33,  2,15, 39,26,  8,13, 29,37, 22,11, 28,17, 31,49,
  10, 0,  0,10, 22,38, 17,18, 33,10, 48,42, 23,26, 27, 6,
  37, 7,  2, 7, 24,35, 24,42, 15,36,  7,45,  5,24, 14, 2,
  33,22, 43,23, 44,16, 11,33, 36,44, 32, 0, 11,28, 26,33,
  20,10, 24, 1, 23,18, 34,15, 19,10, 44, 5, 17,10
};
const int p_50_18[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  215,
  /// Array of preferences
  29,48, 32,11, 33,36, 33,48,  2,21, 16, 3,  6, 4,  2,36,
  48,36, 39,10, 19, 6,  3,44, 36,47, 39,20,  7,20, 43,12,
  20,22, 49, 8, 18,27, 18,12,  3,33, 10,23, 36,27, 26, 7,
  49,30,  7,41,  5,43, 31, 5,  1,15, 23,37, 47, 8, 33,17,
  26, 9,  8,37,  3,47, 46,30, 32,20, 16,33, 30,40,  4, 8,
   3, 4, 20,16,  8,11, 25,29, 39,40, 40,22, 31,42, 28,11,
  47,45, 24, 2, 32,23,  2,29, 16,26, 45, 6, 12,31,  7,46,
  48,45, 20,14, 17,13, 34,20, 27, 5, 15,24, 30,24,  9,29,
  15, 2, 14,22, 48,37, 28, 9,  5, 5, 25,46, 14,28, 44,40,
  15,36, 17,45,  0,43,  3,26, 20, 2,  4,12, 37,18, 37, 9,
  15,34, 24,18,  2, 8, 37,40, 11,35, 44,22, 12,43, 15,48,
  46,40,  3,19, 25, 0, 48,10, 18,34,  3,39, 20,13, 38,42,
   1,48,  4, 0, 19,49,  2,18, 40,35, 37,10, 12, 5, 43,13,
  40,38, 36,16, 22, 1, 13,35, 46, 4, 13, 1, 25,33, 40,48,
  15,27, 38,37, 14,46, 16, 5, 16,17, 33,46, 49,17, 45,45,
   2, 0, 49,42, 49, 2, 18, 2, 42,45, 42,43, 37, 4, 12, 4,
  33,21, 11,12, 38,45, 32, 6, 14,40, 25,49, 19,39,  6,16,
  30, 1, 12,38,  6,36, 26,39, 21,35,  6,34, 26, 6, 47,33,
  21,45, 13,11, 38, 3,  7,48, 32,40, 45,26, 13,22, 36,31,
   8,27, 42, 7, 31,20, 23,31, 22, 6, 29,11, 14,48, 46,34,
  46,49, 35,26,  9, 5, 48, 0, 13,34, 20,20, 39,48, 15,11,
  11,23,  5,30,  5,48, 38,12, 14, 4, 31,37, 23,49, 47, 0,
  21, 2, 38,24, 11,47, 32,48, 11,46, 11,44, 43, 2,  6,25,
   9,34,  1,16,  0,11, 38,29, 17, 4, 11,21, 16,46,  5, 8,
  14,31, 28,25, 13, 9, 34,46, 47,42, 45, 3, 25,37, 40,31,
  46, 8, 36,25, 21,30, 42, 5,  1,42, 35,20, 21,31, 36, 4,
  31,21, 34,13, 45, 5,  4,28,  6,23, 34,43, 37,37
};
const int p_50_19[] = {
  /// Number of people on picture
  50,
  /// Number of preferences
  103,
  /// Array of preferences
   7,40,  4, 4, 36,25, 20,17,  1,24,  6,21, 41,26, 41,48,
   6,18, 39,44, 11,29, 24, 2, 20,14, 33,31, 27,45, 35, 9,
   5,41, 26,34,  9,14, 40,30, 15,32, 38,31, 29,27, 32,44,
  45,30, 20,29, 31,12, 45,38, 22,20, 27,32, 48,22,  4,29,
  33,18,  6,12,  3,21, 49,38, 41,39, 21,48, 44,10, 14, 5,
  11,12, 26,39, 41, 1, 35, 0, 47, 4, 48,49, 19,36, 49,39,
  23,24, 29,16, 40,35, 44,35, 46,31, 15, 8, 43,14, 46,36,
   4,15,  3,27, 39,16,  3,13, 44,25, 23,35, 10,31, 49, 6,
  28, 9,  0,15, 26,37, 14,44, 15,23, 26,49,  0,21, 15,33,
  14, 8, 40,49, 11,43, 17,22,  9,15, 37,41,  4, 1,  3,48,
  14,17, 10, 9, 36,27, 18,46, 31,38, 27,11, 15,49, 12,37,
  41,49,  9, 9, 42,41,  4,33, 25, 9,  3,22, 24, 3, 21,21,
  17,46, 38, 1, 11, 2, 37,14,  3,32, 41,20,  6,25
};

const int p_60_0[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  156,
  /// Array of preferences
  23, 3, 53,20, 37,16, 56,51,  5,20, 22,26, 44,38, 30,56,
  22, 4, 34,28,  4,20, 30,15, 27, 3, 57,37,  0,32,  9,42,
  54,18,  2, 9, 26,32, 46, 6, 55,41, 41,38, 23,29, 15, 5,
  46,20, 13,18, 29,47, 52,44, 46,17,  0,38, 27,15, 37, 8,
  36,33, 11, 1, 13,43, 37,48, 19,12, 42,31, 22,19, 27,21,
   8,19, 37,10, 53,41, 29,19, 30,47, 44,19, 43, 9, 13, 5,
  18,10,  9,17, 26,14, 11,28, 53,55, 24,51, 29,24, 33,29,
  52,13,  8,32, 29,28, 58, 5, 46,47, 43,10, 53,29, 36,19,
  22,50,  6,46,  2,59,  6,24, 48,11, 12, 6, 37,14, 33,27,
  24,40, 34, 4, 52,33, 48, 3, 56,19, 44,16, 49,53,  3,33,
  17,44, 55,47, 52,22, 10,30, 44,45,  6,25, 47,54, 34,15,
  23,26, 52,45, 41,20, 56,36, 33,23, 41, 0, 56,32, 33, 0,
  54,59, 52, 6, 39,24, 27,32,  8,57, 35, 2, 46, 3, 48,36,
  10,59,  9, 7,  9,52, 46,42, 24,17, 35,36, 50,35, 15,57,
  56, 9, 16, 1, 10,52, 52,50, 42,58, 57,39,  0,25, 12,57,
  17,15, 13,53,  4,38, 23,46, 56,20,  2,17, 23,10, 21,58,
  33,59,  6,22, 31,43, 58,28,  0,19, 20,30, 59, 0, 43,19,
  22, 7, 53,40, 33,42, 34,55, 53,12, 10,27, 48, 5, 58,41,
  34, 7, 49,22, 17,58, 15,56,  3,57, 23,33, 41,44, 12,20,
   7, 5, 51,39, 42,22,  8,37
};
const int p_60_1[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  673,
  /// Array of preferences
  21,56,  6,37, 33,47,  9, 6, 53, 1, 45,53, 32,43, 17,23,
   4,13, 25,27, 35,58, 14,46, 57,59, 12,47, 33,40, 50,40,
   9, 0, 49,12, 16,32, 23,30, 15, 6, 49,48, 29,33, 58,54,
  42,44, 50,58, 21,39,  1,14, 45,36, 42, 5,  4,15, 31,22,
  45,38, 21,46, 22,47, 18,20, 13,33, 47,27, 30,26, 55,35,
  21,37, 40, 0, 24, 0,  3,12, 55,34, 19,47, 37, 6,  4, 8,
  33,15, 21,10, 28,43, 57,31, 51, 3, 25,55,  0,15, 20,39,
  20,26,  7, 8, 52,33, 34,43, 35, 9, 54,40, 53,58, 28,15,
   1, 4,  4,21, 33, 0, 52,27, 20,54, 51,12, 16, 3, 39,24,
  26,10, 58, 1, 19, 1, 38,36, 41,57, 30,32, 24, 3, 16,42,
   1, 7, 23,53, 38,33, 42, 9, 43,25, 38,38, 11,38, 12,13,
  13,24,  5,46, 48,14, 48,44, 58, 3, 29, 8, 42, 1, 12, 7,
  56, 9, 52,47, 11, 8, 55, 1, 30,52, 53, 6, 38,59, 24,49,
  49,19, 19, 7,  9,26, 40,44, 10,51, 16,49, 19,23, 17,40,
  53,27,  4,55, 13,11, 52,13, 33,24, 51,32, 25,15, 12,59,
  40,46, 28,35, 54,51, 29,53, 47,21, 43,47, 20,24, 44,25,
   3, 3, 18,32,  9,32, 11,13, 44,21, 36,49, 15, 9, 25, 3,
  57,21,  7,35, 40,50, 21,50, 52,28, 25,44, 11, 0, 16,12,
  29,42, 53,28, 37, 7,  2,27, 22,54, 20, 3, 31,23, 41,15,
  42, 0, 31,20,  5,20, 25,33, 18,30, 56,18, 49,49, 44,40,
  48,21,  8,57, 57,48, 27,51, 20, 1,  0,24, 19,10,  0,17,
   4,44, 58,20, 23,24, 24,52, 55,24,  1,51,  0,48, 19,52,
   1,21, 35,22,  2,14, 17,48, 38,56,  9,53, 53,56,  5,56,
   1,30, 23,58, 23, 8, 29,13,  7,27, 17,33, 42,43,  9,58,
  57,44,  0,49, 59,36, 14,31, 32,21, 50,37,  2, 7, 45,35,
  32,59, 36,34, 39,28, 51,50, 26,34, 20,35, 57, 7, 40,59,
   2, 1,  2,54, 17,46, 16,14, 24,28, 48,25, 47,29, 55,59,
  52,50, 30,33, 49,43,  4,37, 38,27, 51,28, 23,45, 18,25,
  56,28, 30,40, 13,44,  9,17, 22,12, 28,33, 20,28,  9,52,
  37,50,  4, 2, 58,41,  3,39, 32,30, 33,20,  9,28, 28,42,
  55,44, 27,22,  6,56, 29,10, 10,48,  3,18, 49,31, 22,16,
  44, 0, 24,44, 55,49, 15,49, 27,24, 17,22, 13,36, 16, 2,
   7,17,  8,51,  4, 6, 33,49, 28,11, 56,44,  7, 4,  6,15,
  41,59, 55,46,  4, 1, 55,47, 35, 1,  0, 9, 40,40, 56,12,
  29,37, 34,39, 12,33, 46,28,  1,44,  3, 2, 36,50, 12,29,
  52,12, 21,17,  0,41, 34,47, 21, 8, 22,20, 30,50, 15,29,
  12,56, 36,41, 41,28, 43,53, 56, 3, 27,52, 55, 9, 20,30,
  24,15, 12,49, 19,43, 13, 1, 29,38, 49,32,  8,39, 15,30,
  18,56, 58,49, 44,29,  6, 0, 48,49, 53, 4, 24, 9,  2, 2,
  53,42, 19,54, 50,14, 10,30, 59,51, 32,57, 24,35, 55,54,
   2,28, 38,39,  9,42, 51,48, 26,42, 20,44,  3,48,  6, 9,
  47,43, 16,10, 48,11, 44,23, 40,36, 47,56, 42,15, 23, 5,
  40,32, 43,18, 31,32, 47,22, 51,27, 19,26, 16,52, 15, 8,
  52,39, 35,39, 47,14, 27,14, 45,46, 40,31, 18, 1, 15,32,
  14,16,  0, 3, 22,58,  4,51, 25,49, 47,34,  9,33, 37,58,
   3,41,  7,31, 49,58, 30,28, 43,39, 47, 2, 41,12, 25,29,
  31, 3, 50,36, 46,27,  5,33, 12, 4,  8, 2, 56, 5, 26,38,
   6,41, 43,57, 51, 5, 43,21, 12,15, 27, 0,  7, 9, 20,47,
  33,39, 21,21, 54,13, 58, 5, 46,38, 58, 8, 38, 2,  2,43,
  55,14, 29,45, 48,59, 27,10, 26,29, 30,53, 54,19, 45,34,
   2,11, 57,45, 18,58, 54,26,  6, 3, 25, 1, 48,45, 22,52,
  17,32, 17, 1, 15, 0, 33,41, 51,59, 44,10,  4, 4, 50, 4,
  45,31, 50,49,  3,46, 28,21, 28,47, 33,32, 35,28, 22,21,
  46, 3, 58,59, 24, 6,  9,12, 31,13, 10,58, 47,50,  2,40,
  35,55, 39,44, 46,40, 42,37, 45,56, 13,14, 35,35, 54,25,
  11,44, 31,58, 40,47, 14,37, 21,54,  8,13, 52,54, 37,26,
  34,50, 15,33, 38,46, 37,24, 15,50, 41,27, 26, 0, 45,48,
  45,10, 55,39, 36,23,  7,59, 36,38, 57,27, 30,45,  0,19,
  24,45, 38, 9, 51,11, 36,24, 29,20,  9,49, 31,46,  5,16,
  46,45, 21, 9, 38,42, 47,47, 52,48, 19,32, 22,25, 48,33,
  24,58, 35,33, 13,22, 46, 6,  7,21, 52,43,  6,48,  5,35,
  19,13,  1,57,  1,19, 32, 1, 59,59, 24,48, 48,34, 42,33,
  32,41, 54, 5, 48,42, 42,10, 38,14, 58,23, 23,15, 14,40,
   0,39,  3,51, 25,22, 10,47, 18,18, 51,46, 48,23, 44,57,
  19,35, 51,22, 47,17,  7,55, 36,29,  1,47, 58,55, 29,40,
  34,28, 49, 6, 30,38, 29,17, 36,43, 10, 2, 11,26,  0,50,
  28,16, 17, 2, 19,19, 23,49, 41,53, 34,48, 49,59, 29,15,
  51,15, 32,58, 53,52, 59,38, 25,25, 27,40, 25,20, 56,35,
   3, 8, 58,42, 55,45, 46,53, 22,59, 19,18, 15,40, 58,15,
  36,37, 36,31, 42,26, 52,57, 36, 1, 51,31, 37,15, 31,51,
  42, 4, 54,33, 11,20, 14,41,  2,24, 43,54, 56,57, 34,34,
  13,35, 37,45,  2,55, 25,41, 54,18, 40,58,  4,43, 44,36,
  50,22,  9,16, 23,16, 29,47, 17,49, 36,58, 12,39, 47,35,
  38,21, 26,45,  6,58, 12,36, 26,51, 16,54,  0,21, 28, 0,
  57, 2, 14,58, 44,56, 32,16, 37, 9, 56,13,  2,18, 39,40,
   2,47,  0, 2,  8,16, 55,33, 33, 5, 32,22, 48,55, 33, 2,
  12,23, 32,42,  1,32, 30,14, 43,42, 55,38, 33,45, 53, 7,
  38,51, 23,12, 15, 1, 27,19, 56,40,  8,59,  5,41, 28,13,
   5,27, 37,13, 22, 1, 18,44, 11,28, 21,24, 30,19, 25,10,
  34,24,  8,28, 44,13, 35, 8, 23, 6, 50,48, 33,37,  6,24,
  30,17, 40,43, 26,23, 25,24, 33,33, 52,10,  7, 5, 31,26,
  29, 4, 20,17, 10,36, 18,48, 11,14, 54,16, 16,36, 38,54,
  55,12, 47,58,  3,40, 17,41, 55, 8, 41,55, 54,53, 30,47,
  43,26, 34,42, 20,48, 22,43,  6,40, 51, 8, 46,35, 12,48,
  10,28
};
const int p_60_2[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  339,
  /// Array of preferences
  29, 0, 24,25, 18,49, 41,38, 39, 0, 33, 0, 38,45, 29, 5,
  23,52, 15,44,  9,35, 49,58,  4,15, 54,52, 31,39, 22,15,
  51,13, 46,18, 39, 6, 42,52, 56,52,  9,14, 18,16,  0,56,
  14,10, 31,21, 21,30, 28, 5, 52,23, 16,20, 55, 7, 13, 3,
  43,59, 38, 4, 36,46, 55, 0, 26,49, 12,27,  8,17, 24, 9,
  11, 5, 29,31, 56,56, 38,55, 19,24, 56,38, 17,16, 40,13,
   9,25, 11, 8,  1,17, 42,21, 19,55, 25,23, 40,52, 35,33,
  42,42, 46,11, 53,53, 41, 5,  0,12,  2,55,  7,29,  4,32,
   1,29,  1, 8, 23,18, 41,47, 48,10, 31,23, 42,10, 21,15,
  28, 8,  7, 7, 11,56, 19, 2, 41,11, 15,11, 30,20, 56,42,
  22,27, 36,38, 14,16, 30,27,  2, 2, 15,31, 51,27, 11, 6,
  21, 6,  9,15,  9,32, 22,47, 38,48, 43,29, 27, 4,  9, 0,
  38,11,  2,35, 33,16, 11,20,  0,19, 31, 2,  3, 8, 53,52,
  46,21, 34,13, 25,22, 17,23, 47, 4, 20,52, 29,24, 59,12,
  54,38, 35,51, 55,54, 34,35, 40, 7, 53,22, 43,19, 36,32,
  53,29, 26,19, 49,57,  4,57, 28,49, 52,59,  7,49, 31,52,
  31,31, 11,13,  3,24,  4, 8, 58,25, 35,14, 18,56, 25,54,
   8,51, 57,50, 25,48, 27,35, 11,29, 47,23, 26,55, 26,54,
   7,44,  9,54, 12,37, 33,19, 14,24, 11, 4, 13,41, 53,32,
  21,18, 47,49, 34, 8, 58,11, 42,25, 20,28,  3,44, 37,44,
  52,48,  5,28, 48,24, 24,50, 54,16, 11,23, 56,58, 48,45,
  56,33, 22,16, 17,52, 37,59, 37,49, 13,42,  6, 9, 32,59,
   5,46, 29,20, 48,15, 12,25, 44,56, 43,57, 57,54, 25,30,
   0,39, 55,23, 50,52,  1,20, 31, 0, 53,38, 34,21, 25, 6,
  59, 1,  7,24,  5,44, 57, 0,  0, 0, 13,57, 33,58, 13, 6,
  51,24, 24, 8, 38,32, 27,18, 59,34, 19,25, 50,26,  1,40,
  30,51,  4, 2, 30,13, 52,13, 16, 0, 43,36, 56,27, 53,44,
  20,56, 10,28, 43,16,  3,51,  0,27, 37,31, 18,28, 32,42,
   0,55, 52,39, 30,12,  5,55,  8,24,  2,47,  8,48, 56,15,
  42,58, 52,57,  7,47, 23,25, 17,51, 21,27, 38,16, 18,40,
   2, 9, 20, 2, 17,53, 13,48,  9,45, 42,16, 40,42, 54, 3,
   4,58,  7,19,  0,41, 10,35, 41,20,  1,15, 31, 3, 20,40,
  21,41, 53,58, 33,31,  2,52, 14,19, 43,12, 40,12, 43,41,
  22,34, 17, 8, 42,18, 28,44, 13, 7, 24,43, 39,31, 13, 1,
  57, 9, 53,21, 36,57, 50, 9,  2, 0, 50,35, 46,16, 27,24,
  21,47, 47,32,  9,31, 34,20, 23,19, 53,46,  5,29, 53,20,
  41,45, 57,30,  4,34,  5, 1,  3,21, 45,20,  5,39, 41,15,
  13,10, 21,55, 55,28, 16,28, 15,40, 53,54, 12,58, 27,47,
  46,31, 42,34, 37,29, 39,48, 37,43,  4,46,  7,59, 48, 3,
  53,25, 37,27, 28,34,  1, 6, 30,47, 11, 2, 29,32,  9, 7,
  39,47, 37,13, 24,23, 47,15, 21,11, 50,27, 15,29, 52,29,
  39,10,  0,47, 33,26, 19,31, 49,47, 43,21, 45,56, 37,22,
  46,38, 11,26,  7,21
};
const int p_60_3[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  749,
  /// Array of preferences
  27,12, 31,19, 34, 0, 10,16,  9, 0, 11, 1, 20, 8, 28,32,
  41, 8, 26,33,  1,13, 38,16, 31,53, 23,35, 31, 0, 23,24,
  43,17, 31,27, 10,50,  8,55, 22,53, 20,24, 52, 1, 40,54,
  47,37, 50,19,  4,16, 21, 9, 29,15, 51,58, 24,58, 27, 7,
  27, 0,  7,29, 47,17,  7, 4, 30, 2, 32,26,  7,28, 44,24,
  37,59, 35,17, 26,23,  1, 0, 18,54, 10,32, 10,40,  2, 6,
   8,29,  2,18, 45,45,  0,36, 51,25, 18, 1, 46,38, 52,29,
  42,59, 26,24, 34,47,  9,11, 43,23, 17,42, 22,19, 35,46,
   8, 7, 19,47, 18, 3, 33,47, 30,37,  9,50,  5,31, 34,15,
  37,24,  6,34, 38, 3, 12,10, 40,57, 16,50, 12,16, 54,30,
   5,52, 40,24, 49,23, 53,45, 46,13, 23,11, 43,46, 53,23,
  30,19,  4,40, 34,21,  0,19, 31,11, 23,43, 22,42,  3,37,
  10,51, 54,16, 46,45, 49,26,  8,36, 11, 2, 16,22, 20, 2,
  33,48, 39,19, 31,54, 58,11, 29,22, 48,58, 28,21,  6,54,
  46, 2, 30,28, 17,49, 38,57, 37,15, 51, 8, 43,57, 20,34,
  49,41, 12,14, 28,19,  9,43, 13,49, 45,58, 24,38, 25,34,
  56, 6, 38, 1,  8,39, 19,51, 41,54, 50,46, 18,32, 28,37,
   7, 3, 39,54, 44,56, 59,43, 54,51, 22, 7, 27,34, 35,57,
  37,47,  0,55, 59,37, 20,37, 57,11, 31,45, 20, 5, 33,46,
  11,32, 36, 6, 15, 5, 48, 3,  1,33, 10, 6, 41,17, 45,53,
  14,26, 36,29, 32,22, 52, 8, 29, 6, 51,41, 21,13, 42,15,
  15,35, 28,56, 15,24, 32,58, 34,42,  1,55, 57,31, 23,41,
  59,58, 38,55, 17,43, 37,19, 15,15, 26,47, 37,26, 41,31,
  38,45, 39,57,  2, 4, 33,16, 49,32, 11,20,  7,32, 24, 9,
  38,46, 24,30, 11,44, 25, 5,  8,14, 19,29, 13,50, 41, 2,
  44,32, 25,31, 19,17, 28,26, 29,38,  4,25,  0,13, 56,21,
   9,20, 28, 9, 40,26, 46,37, 35,12, 47, 6, 21, 7, 33,35,
  22,56, 34,57, 20,45, 43,28, 53, 6, 25,15, 50,42, 29,50,
  37,30,  5, 4, 38,34, 56,10, 22,59, 34, 1, 14,15, 17,30,
   8,23, 56,14, 15, 9, 26, 1, 33,30,  3,47, 24,57, 32,52,
  52,56, 47,12, 46, 5,  8,45, 34,29,  7,13, 36,27,  6, 2,
  40,53, 59,17,  2,16, 48,46, 28,14, 24, 7,  5,16, 58,48,
  23,29, 47, 8, 23,16, 34, 6, 31, 7, 54,42, 26,36, 14,31,
  39,16, 42,20, 48,35, 36,41,  9,37,  2, 7,  8,25, 17,39,
  44,49, 48,34,  5, 2, 15,13, 14,42,  2,47, 13,36, 10,24,
  32, 6, 56,53, 50,38, 44,50, 50,55, 58,52, 20,49, 44,23,
  16,31, 43,11,  8,13, 24,26, 33,13,  6,16, 55, 8, 22,39,
  44,34, 39,27, 26,51, 33,23, 39,46,  4,21, 26,57, 25,16,
  53, 5, 54,46, 31, 5,  3,27,  9,59, 57,26,  2,10, 29,36,
  12,20, 52,35, 32,44, 31,31,  4,20, 24,14, 39,32, 57,59,
  51,18, 50,16, 43,34, 46,41, 27,10, 51, 5, 16,59, 48,42,
  11,35, 30,31, 53,39, 38,25, 29, 5, 44, 8, 48,30, 44,18,
  25,58, 23,12, 28,47, 51,28, 24,15, 20,42, 19,48, 53,20,
   6,42,  3,36, 41,18, 48, 2, 41, 9, 11,33, 46,51, 31,32,
  14,36, 17,44, 31,16,  8,42, 54,26, 27,54, 31,48, 21,30,
  20,33, 57,42,  5,29, 59, 0, 24,28, 26,26, 16,19, 35,48,
  48, 0,  9,57, 23,39, 21,46,  2,20, 36, 8, 28,17, 16,51,
  58, 3, 58,27,  6, 7,  9,38, 53,30, 42,11,  4,23, 42,29,
   2, 5, 27,22, 18, 0, 49,43, 39,20,  4,55, 48,44, 51,49,
  39,50, 26,20, 10,20, 47,28, 45, 1, 34, 5, 30,58, 11, 8,
  28,55,  6,44, 57,16, 46,59, 58,25, 35,41, 15,51, 59,39,
  50, 5, 58, 5,  6,18, 46,16, 57,57,  9,28,  3,18,  3, 6,
  26,42, 41,14,  5, 6, 46,34,  0,30, 11,43,  7,55,  2, 3,
  16,29, 14,57, 17,33, 57,54, 34,17, 21,18, 18,40, 16,49,
  37,55, 38,23,  6,47, 23, 4, 41,58, 55,17,  4,57, 23,21,
  12,51, 16,46,  0,46, 23,10, 16,34,  0,49,  5,35, 55,56,
  50,30, 42, 7, 22,40, 10,37,  9, 1, 18,34, 54,35, 54,17,
  36,17, 21,15, 34,38, 57,43, 49,28, 18,49, 24,55, 51, 3,
  29,13, 10,43,  0, 9, 27,29, 39,26, 39,44, 12,27, 22,28,
  23,20, 39,21, 12,58, 59,29, 15,45, 31,39, 59,48, 43,26,
  57, 0, 41,52, 54,54, 10,27,  9,54, 52, 0, 50,47,  2,36,
   8,19, 35,19, 43,25, 38,30, 12,54, 45, 3,  7,54, 45,39,
  41,43, 43,10, 58,32,  5,24, 11,42, 44,35, 47,59, 21,21,
   3,39, 13,13, 13,44,  6,55, 34,16,  6,36, 33,38, 27,26,
  54,53, 47,11, 45,10,  8,56, 52,43, 10,35, 59,57, 43,36,
  46,12, 18,39, 55, 0,  1,54, 15,32, 27,21, 48, 7,  9,42,
  30, 4, 52,48, 33,50, 58,20, 58,43, 34,55, 16, 1, 51,30,
  26, 9, 44,48, 55,44, 30,29, 27,28, 12, 8, 15, 6, 15,18,
  39,52, 23,22, 26,41, 40, 6, 42, 6, 32,46,  7,21, 11,17,
  13,22, 59,15, 39,53,  6,33, 19,58,  3, 0, 36,43, 45,36,
  43,38, 25,33, 24, 2, 36,48, 26,54, 26,29,  6,17, 18,38,
  27,17, 51,44,  7,44, 55,34, 44,10, 54,36,  3,49, 29,59,
  29,23, 31,41, 47,51, 42, 8, 33,18, 36, 0, 13,23, 30,34,
  21,36,  3, 2, 52, 9, 20,27,  7,34,  7, 5, 41,55, 24,41,
   9,10,  7,59, 19,41, 36,20, 33,52, 17,40, 42,26, 58,26,
  52,36, 52,10, 27,32, 16,44, 17,35, 45,34, 37,32, 16,39,
  49,13,  7, 1, 27,15, 52,46, 59,24, 33,37, 48, 4, 35,47,
  10,38, 24,22, 58,19,  5,12,  9,40, 58,39, 56,54, 59,49,
  57,46, 14,10, 18,15, 15,14,  1, 3, 48,47,  1,41, 28,42,
  25,37, 15,47, 30,45, 29,16, 37,20, 32,23,  0,47, 37,46,
  50, 6, 31,40, 23,27, 56,35, 57,18,  7,35, 33,32, 40,11,
  11,29,  6,25, 27,37, 35,55, 52,59, 58,36, 49,24,  2,23,
  55, 3, 42,43, 44,12, 26,31, 24,35, 27,11, 34, 9, 53,49,
  47,34, 15,10, 46,23,  8, 9, 43, 4, 43,56, 37,10, 13,14,
  15,53, 25,11, 52,25,  1,39, 59,44, 49,22,  4,17, 19,45,
  53,27,  0,45,  7,38, 32, 4, 50,28, 13,17, 16, 6, 42,35,
  50,44, 28,30, 40,13,  1,11, 27, 5, 24,53,  1,50, 14,45,
  54, 9, 16,33,  1,31, 50,53, 21,59, 52,49, 29,45, 50,58,
  52,57, 37,21, 40,33,  0,27, 11,57, 48,19, 43, 6,  4,45,
  49,50, 21,34, 40,59, 35,13, 13,11, 57, 6, 47, 5, 32,59,
  23,36, 56,55, 11,55, 59,38, 47, 9,  4,46,  7, 2, 48,53,
  56,23,  8,40, 51, 4,  6,45, 29,52,  3,23, 19,16, 12,17,
   7,53,  5,51, 46,57, 49,20, 44,59, 32,12, 36,26, 12,41,
  25, 6, 28, 4,  7,41, 59,59, 32,32
};
const int p_60_4[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  741,
  /// Array of preferences
  26,56, 55,21, 59,15, 43,34,  9, 1, 16,15, 29,20, 59, 0,
   6,35, 41,33, 13,40, 24,29, 12,45, 49, 9, 14,28, 43,40,
  50,49, 24,42, 40,27, 18,19, 12,18, 45, 2, 12,50, 33,24,
   6, 7, 18, 9, 56,46, 52,44, 40, 9, 13,55, 11,41, 11,34,
  55,34,  4,20, 25,48, 55,54,  3,26, 15,38, 45,43, 48,52,
  11,37, 47,22, 34,24, 34,18, 36,30, 23,29, 36, 2, 37,38,
  14,29,  0,38,  4,31, 48,19,  4,14, 37,54, 36,34, 12,21,
  37,43, 26, 3, 31,43, 24,34,  7,30, 41,32, 21,51, 13,19,
  43,51, 47,36, 54,49, 52,29, 34,52, 26, 1, 21,53, 28,58,
  46,39,  5,22, 22,17, 16,26, 33,45, 25,56, 11,38,  1,55,
   2, 5, 46,15,  0,43, 30,15, 27,47,  1,35, 20,24,  6, 2,
  10,34, 53,23, 11,18, 16,45, 17,56, 46,29, 58,51,  4, 5,
  21,56,  8,15,  2, 6, 54,28, 53,59, 47,23, 38,56, 27,19,
   2, 8,  4, 6, 40,46, 46,28,  4,53, 39,41, 46,11, 48,25,
  30,59, 49,19, 31, 7, 29,58, 53,11, 53, 5, 43,12, 53,54,
  18, 1, 10,41, 19, 9, 21, 7,  7,23, 40,47, 21,55, 35, 0,
  51,55, 21,15, 15,31,  1,43, 20,13, 22,39,  2,39, 51,37,
  51,15, 38,42, 44,38, 51,19, 15, 8, 33,28, 14,58, 38,10,
  38,14, 56,29, 57,26, 22, 8, 39, 0, 36, 1, 26,47, 41,25,
   5,58, 53,42, 46,27, 22,27, 56,57, 15,13, 26,20, 40,29,
   2,44, 22,58, 49,24,  9, 2, 46, 2,  3,11, 40,49, 15,40,
  52,38, 25,12, 39,19, 49,30, 59,18, 37,45, 26,27, 54,52,
   5,19, 19,43,  4,57, 48,43, 29,13, 10,46, 14,57, 58,23,
  43,13, 10,51,  6,19, 16,17, 11,22, 22,29, 46,20, 57,15,
  13,51, 35,28, 24,39, 59,51, 58,53, 43,17, 44,45, 22,36,
  34, 8, 50,29, 50,21, 34, 3, 58,21,  4,36, 49,43, 13,58,
  28,54, 52,43,  3,44, 49,44, 13,54, 53,36, 17,43,  3,20,
  41, 9, 14, 1, 43,22, 25,21, 27,16,  4,12, 24,10, 16,57,
  48,29, 38, 4, 11,13, 29,57, 30, 1, 40,50, 52, 9, 25,24,
   1,30, 58, 0, 52,24, 45,55, 59,48, 47, 7, 30,16, 22,31,
   9, 6, 16,35,  4,16, 50, 9, 43,33, 42,41, 35, 8, 29,14,
  30,20, 59, 3, 18,47, 52,34,  8,22, 35,20, 50, 6, 23,33,
  23,42, 59,47, 38,12,  0,48, 30,30, 28,37, 54,22, 31,36,
  47,42,  8,37, 28,13, 21,30,  8,12, 16,20, 54,17, 19, 5,
  53,26,  5,11, 40, 2, 36,19, 24, 1, 45,39, 42, 5, 45,28,
  47, 0, 31, 4, 43,21, 11,23, 12,53, 43, 1,  9,33, 12,31,
  58,40, 45,44,  3,18, 23, 9, 29,41, 56,42, 50, 0, 52,37,
  26,34, 58,30, 30, 6,  4,45, 31, 2, 23,25, 12,15, 17,52,
  20,42, 16,33, 58, 6, 38,19, 27,29, 50,50, 16,38,  3,31,
  42,39, 24,48, 16,11, 51, 1, 31, 0, 56, 9, 40, 0,  7,45,
  21,25, 22,23,  7,18, 26, 9, 34,38, 11,21, 40,25,  9,43,
  40,31, 44,10, 29,15, 47,54, 15,49, 29,40, 37,33, 24,36,
   5,12,  8,40, 12,51, 20,10, 52,54, 20,54, 58,49, 17,59,
   2,31,  2,33, 18, 2, 19,56, 54, 0, 49, 2,  8,41, 36,13,
  31,30, 24,20, 52,50,  3,10, 58, 1, 15,55, 11,43,  5,52,
   1,36, 53,18, 56, 4, 16,43, 31,39, 57, 1, 19,48, 12,23,
   6,40, 36,51, 24,59, 48,20, 12,24, 19,30, 53,31, 16,37,
   6,42,  9,36, 54,38, 37, 7,  5,44, 46,37, 58,32, 53,15,
  21,46, 32,39, 13,50, 34,27, 12, 7, 13,10, 27,25, 22,22,
  38,45, 39,27, 14,25, 32,46, 39,46, 43,45, 55,33, 33,22,
  20,43, 27,57, 33,10, 58,10, 48,24,  9,31, 14,23, 43,58,
  26, 5, 47,41, 50,53, 11, 3, 39,13, 53,55,  5,45, 39,26,
  54,14,  5, 5, 42,46, 33,48, 32,59, 36,58, 23,41, 20,39,
   5,23, 12, 5, 54,46, 32,31,  8,17, 45, 8, 46,16, 38,32,
  56,37, 50,44, 25,44, 36,49, 25,37, 32, 5, 14,18,  1,40,
  11,58, 23,48, 21,50, 18,24,  9,47, 58,27,  9, 0, 33, 3,
  54,20, 57,34, 44,37,  4,27, 45,37, 27,30, 27,55, 42,25,
  54,55, 26,30, 51, 4, 48,28, 43,10, 20, 5, 37,22, 24, 7,
  57,59, 23,46, 34,44, 34,15, 15,44, 47,39, 56,19, 10,48,
  21,41, 21,39, 38,26, 20,16, 37,50, 53,33, 53,47, 33,49,
   7,14, 10,20, 10,30, 52,32, 41,38, 31, 8,  1,49, 26,16,
  34,14, 14,39, 23,23, 50,17,  5, 7,  6,16, 15, 6, 57,39,
  46,22, 29,21, 51,52, 39,56, 47,38, 50,57, 32,30,  5, 2,
  39, 4, 14,12, 41, 8, 36,48,  9,54, 15,12, 55,27,  5,38,
  15, 0, 59,12, 19,58, 24,30, 35,31, 48,35,  0,50, 49,47,
  31,50,  6,22, 38,40, 58,14, 40,44, 58,11, 38,57, 54,27,
  17,27, 43,44, 13,32,  3,30, 47,58, 25, 7, 14,35, 30,27,
  40,56,  6,23, 23,37, 16, 5, 44,57, 41,18, 40, 5, 49,33,
  46,26, 52,33,  6,28, 24,19, 52,42, 17,53, 41,11, 51,35,
  55,36, 57,18, 22, 9,  7,51, 21,13, 53,45, 53,48, 11,10,
  12,41, 35,17, 37,29, 58,34, 55,22, 38,13, 19,49,  3,59,
   6,47,  2,48, 52,18, 35,36, 23,24, 57,36, 21, 0, 52,52,
  50,37, 42,15,  4,30, 19, 0, 27,38,  5,29, 38,43, 33, 5,
  28,31, 23,45, 41,52, 41,57,  2,59, 47,55, 19,20, 27, 8,
  44,44, 18,29, 45,59, 23,34, 30,45, 20,34, 19,13, 44,34,
  56,15, 53,56, 55,53, 48,15, 16,24, 20,51, 54,25, 25,59,
   3, 2, 52,39, 55,26, 25,31, 32,45, 56, 3, 47, 9, 14,24,
  34,58, 50,10, 55,35, 33,12,  1,31, 22, 2, 37,14, 23,16,
  59,42,  0, 9, 31, 6, 32,17,  9,53, 36, 9, 36, 8, 14,37,
  24,55, 51,58, 50,41, 40,28,  7,55, 54,34, 33,30,  1,48,
  18,16, 21,59, 23,26, 34, 5, 23, 5, 49, 7, 10,52, 29,50,
  37,51, 48,12, 21,20, 27,43, 30,11, 21,31,  7,57, 10, 6,
   3,16, 48,56, 40,15, 45,14, 10, 8, 28,29, 35,29, 49,57,
  41,56, 11,46, 12, 8,  6,53, 50,28, 44,14, 59,13, 40,34,
  32,15, 39,39, 53, 8, 22,54, 30,19,  9,27,  4, 4, 14,47,
  47,59, 34,29,  0,22,  6,30, 31,16, 10,55, 31,11, 41,19,
  12,25,  5,47,  5, 9, 19,59, 38,46, 40,19, 21,44, 28,41,
  39,36, 27,22, 24,40, 12, 3, 56,22, 24,44,  0,55, 23,47,
  34, 9, 56,45, 40,10,  1,44, 43, 7, 21, 9, 44,48, 37, 1,
   7, 7, 34,49, 48,39, 34,22, 41,46, 56,54, 15,33, 21, 3,
   2,41, 40,58,  1,33, 53, 4, 29,48, 44, 3, 31,19, 42,10,
  26,49, 47,31,  0,17,  2,27, 35, 4, 22, 4, 16,42, 38,49,
  17,51, 47,53, 27, 9,  6,11, 54,45
};
const int p_60_5[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  890,
  /// Array of preferences
  32,50, 49,51, 43,31, 11, 2, 28, 9, 50,13, 17,47, 49,39,
  57,15,  6,12, 15, 8, 54,14, 17,35, 17,18, 44, 6, 41,54,
  54,11, 45,54, 49,40, 39,28, 56,43, 32,35, 12,52, 31,30,
  16, 4, 34,31, 36,24, 19,10, 12,45, 49,34, 31,50, 25,31,
   7,33, 54, 7,  1,39,  9,54, 36,41,  0,34, 30,28, 29,29,
  55,35, 34, 8, 51, 7, 27, 6, 21,57, 11,15, 56,20, 34, 6,
  30,16, 41,38,  5,27, 41,44, 42,40, 13,41, 43,50, 39,34,
  48,47, 33,20, 24,59, 36,22, 46,19,  1,38, 55,27, 45, 0,
  10,18, 15,12, 48,10, 16,50, 10,48, 16,53, 19,57, 18,30,
  29,49,  2,32, 18,27, 58,50, 17, 2, 57,43, 11,36, 26,54,
  47,52, 57,32, 16,37, 46, 7,  7,54, 11,50,  5,30, 10,30,
  43,35, 23,47,  9,59, 45, 7, 19, 9, 16,23, 12, 9, 17,45,
  56, 7,  0,58, 25,18, 45,30, 23, 5, 40,55,  5,14, 11,11,
  25,37, 24,54, 47,20, 39, 4,  4,52, 59,31,  6,55, 38,45,
  50, 3, 30, 9, 15, 7, 13,34, 17,16, 50,24, 25,27, 15,48,
  53,15, 54, 4, 37,25,  0,45,  9,26, 58,15, 41, 5, 51,45,
   2,55, 59,18, 19,33, 40,59, 45,52,  3,36, 28,43, 41,55,
  14, 7, 44,18, 45,24, 22, 1,  1,15, 25,16, 35,55,  7,46,
  37,51,  2,21, 24, 6, 44,34, 23,55, 42, 9, 44,19,  4,57,
  53,17, 25,48,  3,10,  1, 4, 38,10,  1,19, 39,56,  4,27,
  16,35,  7,31,  9,11, 43,30, 20,16, 44,14, 45,31, 10,28,
  12,24, 38,17, 22,44,  2,26, 59,13, 41,26, 58,31, 18,35,
  48,37,  4, 8, 21,44, 15,20, 30,44, 17,53,  0, 4, 33,40,
   4, 1, 15,58, 32,59, 45,32, 53,10, 51,36, 44,45,  0,22,
  11,18, 24,51, 57,20, 19,47, 45, 3, 15,39, 39,37, 50, 6,
  46,24, 19,31, 17,13, 58,48, 34,55, 48,32, 11,26, 20,55,
   1,24, 28,58,  6,27, 11,59, 12,55,  0,48, 34,37, 37, 0,
  20, 6,  1, 3, 10,10, 48,11, 55,15,  0,25,  2, 3, 53,47,
  57,34, 36,30, 27,12, 48,23, 34,30, 36,28,  2,46, 11,21,
  47,38, 12,35,  4,14, 31,52, 39,15,  6,18, 53, 3, 19,11,
   3, 6, 19,51, 59,45, 46, 3, 39,54, 20,48, 52,16,  5,13,
   4,44, 34, 9, 41,11, 23,19,  1, 0,  4, 9, 51,16,  3,57,
  18, 7, 35,26, 36,26, 37,47, 48,51, 42,26,  3,23, 48,55,
   8,24,  5, 3, 37, 3,  3,12, 47,40,  0,37, 19,18, 38,35,
  58,46, 16,39, 53,57, 59, 3, 16,40, 21,47, 18,14, 20,29,
  16,43, 16,10, 41,15, 54,50, 36,15,  0,19, 23,28, 10,14,
  59,34, 41,27, 53,14, 18,39, 52,18, 42,29, 57,58, 55,14,
  14,25, 11,25, 42,33, 55,50, 49,18, 12, 6,  3,14, 36,56,
  43,42, 58,20, 49, 4, 17, 6,  6, 9, 30,19, 33, 7, 28,41,
  23,15, 53,22, 58, 4, 55, 3, 22,17, 28,38,  7,34, 50, 0,
  48,31, 48,13, 37,45, 50, 1, 48, 0, 33,31, 21,35, 33,30,
  15,47, 18,58, 46,57, 29,32, 38,55, 12, 2, 41,36,  1,41,
  22,26,  4, 4, 52,50, 19, 8, 12,46, 17,29, 11,17, 42,27,
  15,34, 32,49, 22,38, 39,53, 28, 3, 49,35, 16,38, 36, 6,
  47,26, 37,53, 58, 7,  3, 4, 44,22, 30,42,  4, 3, 22,32,
   3,18, 10,13, 42,19, 56,12, 32,43, 40,27, 58,24, 43,32,
  29,16, 19, 7, 58,21, 29,28, 40,21, 32,52, 36,16, 25,40,
  15,41, 46,52, 25,15, 19, 0, 37,42,  3,28,  2,22, 24, 4,
   8,34, 23,44, 46,41, 27, 3, 47, 4, 43,47,  6, 6, 53, 0,
   6, 7, 40,42, 39,18, 56, 6, 42,14, 10,45, 26,18, 12,13,
  36,45, 59, 5, 40,15,  1,49, 45,35, 16,42,  6,41, 41,20,
  43, 6, 34,17,  9,14, 51,34, 35, 2, 26, 3, 35,18, 27,35,
   3,54, 56, 2,  4,43, 30,38, 52,29,  6,49, 22,46, 30,46,
  10,41,  4,28, 48, 6, 47,27, 21,49, 54,42, 16,51, 25,33,
  41,40, 46, 2, 56,17, 36,32, 40, 0,  8,10, 55,56,  3,52,
  52, 6, 21,19,  8,12, 27, 9,  9,48, 21,21, 55, 9, 50,54,
  13,46, 50,35,  8, 2, 39,21, 10,57, 16, 6, 59,16, 40,36,
   1,23, 28,19, 11,30, 37, 1, 33, 6, 20,12, 41,35, 19, 2,
  31,41,  3, 8, 14,10, 10,23, 53,42,  7,13, 23,58, 16, 7,
   6,58, 19,41, 49,57,  1,58, 46,39, 57,39, 20,47, 13, 2,
  12,30, 40,48, 58,10, 58,54, 20, 4, 24,46, 52,25, 45,45,
  12,17, 25, 8, 26,46, 23,53, 20,58,  6,48,  2, 7, 55,23,
  35,58, 27,57,  5,18, 52,54, 42,31, 16,14,  2, 5, 43,13,
   9,15,  3,56, 10,15, 52,42,  2,27, 28,24, 23,14, 55, 4,
  14,50,  4,33, 20,33, 52, 9, 25,17, 15,26, 35,14, 49,15,
   0,50, 28,51, 45,26, 50,44,  4, 7, 18,10, 18, 3, 38,22,
  29,22, 47, 3, 24,26, 56,15, 24,27, 42,47, 44, 5,  4,47,
   2,58, 57,25, 20,43, 15,56, 37, 2, 56, 1,  7, 6, 14,43,
  13, 3, 33,25, 25,11, 34,10, 21,16, 26,26, 15,27, 24,34,
   3,19, 46,12, 35, 6,  0,35, 55,33,  5,48, 32,45, 44,27,
   0,59, 53, 2, 51,12, 50,19, 33,12, 14, 6, 53,51, 42,37,
  47, 7, 21,59,  2,43,  5,55, 29,47, 33, 9, 19,26, 20,51,
  35,53, 42,48, 31,59,  9,50, 32, 3,  8,19, 19,24, 38,29,
  14,26, 27,34, 28,57, 13, 1,  0,52, 50, 5, 46,48, 17, 5,
  29,53, 55,47, 33,42, 57,16, 21,15, 11,37,  3,30,  9,32,
  38,21,  6,14, 36,54, 49,49, 41,53, 26, 6, 36,27, 23,25,
  23,36, 16,18,  2,47, 22,42, 15,13, 37,48, 15,55,  2,51,
  30,32,  0,32, 53,37, 13,31, 56,59, 27,18,  4, 2, 17,26,
  54,59, 15,45, 57,54,  6,38,  6,45, 19,20, 43, 5, 20,53,
  37,21, 22,51, 28,37, 46,26,  5,16, 28,13, 25,54, 30,56,
  43,23, 21,41, 43,34, 33,24,  3,33, 16,48, 31,13,  4,21,
  40,47,  5,22, 50,28, 36,21, 30,48, 10,36, 26,11, 50,23,
  24,33,  0, 5, 23,34, 38,54, 33, 3, 26,28, 58,49, 42,10,
  32,24, 33,14, 30,13, 28,47, 46,34, 27,25, 23,38, 26,19,
  50, 4, 13,58, 23,37, 15,40, 45,56, 24,16, 51,46, 26,17,
   9, 9,  2, 2, 28, 1, 32, 0, 47,54, 22, 5, 35,59, 15,50,
  58, 3,  6,36, 30, 6, 47,33, 39,43,  2,56, 20,38, 53,59,
  21, 1, 59,14, 36,29, 23,18, 32,47, 10,39, 45,15,  9, 2,
  34,13, 10,58, 20,35, 33, 4, 44,51, 15,32, 16,41, 50,57,
  48,16, 17,59, 10,34, 32,28, 48,12, 36,42, 46,33, 34, 0,
  54,10, 26, 2, 31,20,  6,22, 46,58, 53,12, 10,33, 40,20,
   6,52, 31,12, 34,52,  0,46, 59,15, 24,37, 26,52, 29,25,
  29,58,  9,52, 23,56,  0, 8, 48,25,  9,36,  3,32, 40,24,
  14,36, 16,49, 18,52, 41,24, 19,58, 45,25, 54, 6, 22,40,
  11,19, 57,23, 26,21, 20,39, 19,34, 27,48, 48, 9, 39,27,
   7,23, 47,21,  5,45,  0,57, 36,34, 51,10, 12,53, 48,38,
  38,20, 31,42, 56,42, 50,32, 19, 5, 13,50, 32,46, 35,29,
  49, 6, 14,46,  9,38, 44,53,  9,56, 38,44, 26, 5, 52,12,
  21,50, 27,33, 54,54, 58,39, 14,30, 55, 0, 44,25, 45,57,
  54,13, 33, 8, 48,17, 53, 7, 52,56, 53,38, 55,58, 54,20,
  26,33,  0,49, 50,58, 31,33, 45, 4, 12,15, 12,49, 51,32,
  40,22, 59, 7, 50, 8, 29,48, 36,23, 17,28,  7,19, 44, 9,
  31, 3, 44,48, 31,44, 14, 5, 28,20, 12, 0, 17,11, 32,16,
   2,19, 49, 2, 21, 9, 22,49, 33,19, 54,58, 30,33,  4,17,
   5,38, 22, 0, 27,45, 19,43, 52,55, 28,25, 22, 6, 58,52,
  17,50, 15,21,  5, 1,  8,20, 47,34, 59,51, 37,43, 13,12,
  49,31, 21,32, 25,42, 33, 5, 47,14, 13,26, 59,33, 52,10,
  59,50, 31, 6, 59,49,  6, 5, 41,39, 28,12, 31,28, 20,17,
  11, 6, 34,29, 16,36, 24,40,  6, 2, 31,19, 27, 8, 25,55,
  58, 1, 40,26, 10,44, 47,51, 45, 8, 48,30, 34,21, 25,58,
  27,17,  7, 9, 49,52, 28,22, 36, 3,  9, 7, 17,17, 47,23,
  50,41, 58,42
};
const int p_60_6[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  470,
  /// Array of preferences
  17,45, 55,14, 31,54, 21,14,  2,10, 36, 2, 14,56, 31,14,
  24,31, 56,54, 43,17, 28,37, 28, 9, 25,45, 24,45, 56,45,
  56,56, 17,24, 28, 6, 18,55, 15,34, 55,13, 46,20,  7, 3,
  26,58, 53,16, 41,15, 45, 5, 43,28, 32,15, 28,39, 39, 9,
  14,35, 59, 4, 14,47, 33,17, 46,38, 27, 2,  4, 6, 22,14,
  41,12, 10,15, 20,25, 53,26, 54, 0, 38,48,  6,23, 16,57,
  57,18, 31,11, 34,22, 41, 2,  6, 5, 14,15, 11,25,  0,37,
  11,20, 26,17, 41,19,  9,32, 20,16, 54,14, 15,12, 40,32,
  39,13,  3,22, 36,37, 27, 8, 49,35, 32,43, 27,52, 41,49,
   8,54, 41,39, 59,43, 30,13, 22,55, 53,44, 21,46,  8, 1,
  50,33, 27,28, 36,23, 51,23, 25,29, 53,20,  4, 4, 53,52,
  35,38,  9,10, 41, 1,  9,40, 59,31, 18,37, 12,20, 43,52,
  52,54, 56,57, 49,30, 56,11, 50,13, 36,14, 22,15, 19,46,
  35,32, 45,39, 22, 7, 25,46, 15, 9, 35,46, 25,11, 49, 0,
  22,32, 54,28, 10, 0, 34,51, 49,47, 30,26, 10,18,  9, 5,
  27,50, 18, 4, 26, 2, 40,45, 21,39,  7,49, 26,22, 54, 3,
   1,32, 40,40, 44,18,  0,45, 37, 2, 49,46, 57, 8, 15,35,
   7,23,  7,30, 14,22,  6,57, 44,14, 56,55, 38,41, 33,20,
  59, 8, 31,26, 58,24, 36,52, 42,54, 46,15,  3,31, 38,18,
   7,45, 49,11, 12,16, 34,15, 54,40, 30,38, 20,13, 28,23,
   5,34,  1,53, 26,52, 36, 6, 31,47, 23,29, 36,21,  4,26,
  17,41,  2,48, 16,17, 50,18, 15,17, 11,11, 44,47, 40,11,
  34,38,  3,34, 53, 8, 37,55, 11,42, 59, 6, 19,42, 26,28,
  35,22, 19,15, 34,14,  0,55, 29,34,  9,19, 32,54, 12,44,
  12,32, 30,21,  1, 1, 14,14, 51,29, 42,17, 16,46, 51,47,
  38,26, 53,14, 57,40, 58,54, 30,56, 59,59, 35,30, 17,44,
  40,13, 44,31,  1, 2, 39,50, 10,40, 18,53, 49,40, 33,26,
  14,30, 59,45,  3,28, 26,31, 33,25, 41,14, 11,17, 57, 6,
  59,11, 12, 7, 12,45, 45,31, 44,55, 55,37, 59,57, 59, 0,
  23,28, 16,18, 44, 5,  1, 7, 39,18, 32, 5, 56,47, 32,23,
  49,26,  6,45, 24,10, 28, 7, 12,29, 18,33,  8,55, 21,44,
  39,12, 45, 8, 18,28, 31,16,  6,49, 16,37, 19,23, 10,23,
  22,33, 58,10, 55,19, 25,24, 28, 2, 46,21, 29,29, 43,39,
  26,14,  4, 7, 28,12, 59,23, 40,47, 49,23, 58,37, 27, 5,
  49,43, 13,53, 41,32, 25, 8,  3,20, 26,27, 30,44, 59,29,
  34,16,  5,43, 39,39, 29,35, 29,43, 15,40, 20,15, 11,15,
  11,35, 21,35, 59, 3, 22,56, 51,45, 50,43, 39,34, 21,51,
  43,23, 55, 3, 46,45, 58,57, 44, 6, 48, 5, 40,55,  4,55,
  43,33, 39,55, 39, 8, 48, 6, 31,40, 32, 6, 39,43,  5,19,
   5,58, 53,57, 13,25,  4,17, 17,56,  4,38, 25,17, 35, 3,
  11,33, 28,16,  4, 9,  9,37, 25,16, 26,30, 32,16, 29,13,
  55, 1, 50,45, 37,25, 33,11, 46, 1, 53,38, 39, 5,  2,43,
  54,49, 58,26, 29,31, 15,37, 47, 7, 51,44, 43,20, 40,50,
  55,28, 44,51,  2,16,  7,37, 36,15, 47, 1, 42, 2, 49,16,
  32,26, 10,51, 59,15, 46,49,  3, 7, 38,28, 33,14, 49,45,
   5,41,  8, 8, 45,32, 40, 2,  8,30, 29,23, 53,46, 38, 1,
  37,32, 25,33, 24,46, 15,32, 12,40, 19,54, 25,10, 32,17,
  29,40, 16,28, 27,40,  0, 5, 35,37, 32,19, 13,43, 11, 3,
   4,20, 57,17, 45, 4, 50,28, 25,47, 12,41, 41,16, 24,23,
  16,34, 58,12, 14,10, 28,19, 59,35, 45,19, 14,53, 49,41,
  30,52, 26,21,  6, 1,  2,39, 45,25, 14,50, 13,48,  2,18,
  13,57, 58,35, 44,48, 28,10, 44,24, 35,59,  8, 7,  6,22,
  36, 3, 42,44, 47,20, 54,19, 22,25,  7,15, 25, 1, 27,39,
   3,53, 32,34, 26,11, 35,12,  1,17, 55,46, 36,31, 10,52,
  24,56, 19,28, 47,18, 11,45, 52,29, 11,51, 46,56, 38,34,
   7,27,  4,28, 33,24, 44, 7, 24,25, 39,26,  4,32, 14,33,
  29, 1, 16,56,  4,39, 45,59, 18,14,  7,53, 30,48, 37, 7,
  19, 2, 43, 1, 33,55, 17,36, 42,26, 18,48, 57,39,  6,27,
  23, 0, 31,36, 15, 0, 30,24,  6,55, 12, 3
};
const int p_60_7[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  546,
  /// Array of preferences
  40,29, 17,35, 44,29, 18,38, 15,41, 43,26, 16,56, 29,20,
  12,32,  4,48, 23,10, 51, 6, 35,18, 17,28, 47,26, 49, 4,
  16,47, 46,44, 22, 0,  3,17, 42,16, 38,37, 37,31, 36,53,
   7,31,  4,40, 35,46, 15, 8, 27,19, 24,38,  3,33, 37,39,
  10,50, 30,13, 17, 1, 43,31, 55,12, 28,17, 36, 2, 54,37,
  38,41, 33,15, 21,32, 54, 0, 35,41, 27,37, 27,16,  5,51,
  17,54, 54,14, 15,27, 39,12, 14,11, 25,53, 10,22, 19,44,
  18,11, 32,46, 54, 6, 17,18, 54,48, 38,15,  2,32, 21,40,
  54, 7, 40,26, 22,55, 59,17, 49,41, 59,28,  3,15, 31, 8,
  42,42, 55, 7, 43, 8, 47, 9, 29,53, 43,45, 18,34, 45,49,
  51,51, 37, 5, 50,15, 26, 2,  1,37,  1,59, 22,37, 38,24,
  46,28,  3,41,  8,17, 53, 3, 19, 4, 35, 1, 38,55, 47,19,
  38,45, 50,37, 30,45, 38,53,  8,18, 21,13, 58,45, 39,34,
  29,18, 17,13, 39,44, 16,45, 38, 3, 41,36, 38,11, 51, 9,
   8,31, 30, 4, 57,36, 26,48,  1,45, 42, 9, 37,20, 23,27,
  41,22, 28, 5, 25,58, 31,24, 24, 3, 27,39, 24,49, 11,32,
  19,41, 28,26, 18,45,  6,54, 25,48, 16,13, 11, 6, 14,15,
  11,41, 13, 8, 46,45, 19,38, 56,52, 17,46, 25,30, 58,55,
   4, 1, 59,42, 39,42,  4,31,  1, 0, 10,23, 34,54, 48,30,
  17,30, 43,12, 10, 8, 14, 3, 57,40, 26, 1, 20,47, 46,35,
  52,10, 54,16,  1,50,  8,37, 34,42, 38,46,  5,21, 58, 9,
  40,34, 26,22, 36,52, 36,11,  8, 6, 59, 2, 51,46, 44,34,
  45, 3,  7,56, 53,36, 15,23, 14,49, 20,11, 50,43, 34,10,
  41,40, 58, 8, 17, 8, 21,59,  8,21, 32,34, 24,47, 48,57,
  12, 4, 44, 0, 56,53, 37, 8,  7,17, 43, 7, 52,47, 12,47,
  15,19, 25,40, 40,53, 42,15, 20, 0, 23,53,  9,32, 11,19,
  21,24, 23,34, 35,42, 56,20, 36,12, 43,57, 35,56, 30, 3,
   9,19, 12,59, 11,39, 24,28, 49,45, 57,47, 50, 3, 39,13,
  15,25, 26,47,  4,23, 17,16, 55,24, 46,39,  2,53, 22, 3,
  44, 6, 37,26,  1,24, 52,39, 10,34, 30, 7,  8,24, 45, 4,
  57,39, 46,56, 37, 3, 36,42, 58,18, 50,36, 34,11, 37, 2,
  32, 3,  4,43, 10,26, 11,25, 59, 1, 31, 5, 39,39, 57,15,
  25,52, 36, 4, 18,54, 47,40,  0, 9, 13,39,  8,34, 21,27,
  27, 2, 18,56, 55,35, 18,28, 35, 3, 24,35, 57,30, 53,57,
  42,19, 27, 3, 28,52, 47,17,  9,35, 16,12, 55,20, 28,12,
   7,13, 28,25, 32,43, 24,23, 24,30, 17,48, 47,43, 16,57,
  38,47, 24,43, 38,33,  2,54, 20,57, 57, 3, 16, 3, 36,33,
  48,51, 50,51,  4,19, 45,25, 47,34, 13,37,  4,45, 46,50,
  41, 3, 36,47, 21,16, 28,59,  8,35, 15,31, 50,25,  7, 6,
   6,53,  8,42, 59,13, 14, 8,  7, 5, 34,19, 33,24, 46,18,
  13,32,  8,45,  8, 4, 52, 7, 42,49,  4,25, 30,38, 37,48,
  57,23, 55,45, 26,54,  1,12,  9, 8, 12, 9, 46,31, 14,25,
   3,48,  7,20, 51, 8,  5, 0, 53,11, 10,28, 37,15, 46,53,
   0,45, 31,33, 48, 5, 37,33, 51,44, 10,24, 15,28, 12,22,
   7,26, 18,59, 28,54, 45, 2, 54, 9, 15,56, 59,58,  8, 8,
  53,21,  3,18,  2,26, 30,48, 30,22, 42,12, 15,36, 42,14,
  40, 4, 58,40, 20,58, 44,41,  5,46, 40,54, 25, 1, 29,24,
  27,34, 22,12, 39, 6, 16, 5, 20,50, 41,47, 51,30, 28,47,
   5, 5, 33,25, 57,34, 55, 6, 21,47, 10,17, 28, 0, 11,58,
  28, 2, 53,27, 24,17, 42,59, 54,21, 16,59, 41,15, 29,40,
  40, 9,  2,37, 15,49,  4,22,  1,58,  4,47, 40,45, 54, 1,
  38,29,  4, 2,  2,44, 30,30, 17,27, 31,11,  4,53, 14,43,
  28,19, 55,43, 43,42,  5,20,  4, 9,  7,55, 32,31, 57,58,
  35,22, 47,37,  7,25, 29,11, 45,15, 39,24, 23, 6, 39, 0,
   1,11,  3,44, 31, 0, 19, 7, 30, 9,  1,31, 24,27, 23,42,
  37,32, 33,14, 20, 5, 55,10, 11, 1, 56,41, 47,33, 33,32,
  28,29, 48,22, 37,30, 27,55, 41,39, 52,56, 36,28,  4, 4,
  43,50, 42,29, 45, 6, 32, 9, 39,30, 10,19, 44,11, 48,55,
  13,43,  9,57, 58,41, 59,19, 41,14, 27,41, 20,49, 34,58,
  39,19,  9,11, 53,23, 52,43, 17,51, 51,14, 51,13, 36,40,
  29, 1, 32,58, 47,25, 12, 6, 39,49, 11,48, 52, 5,  1,25,
  11,13, 15, 5, 30,14, 36,34, 15,20, 52,20, 41,27, 34,37,
  13,35, 58,28, 42,45, 13,34, 35, 5, 45,41,  5,15, 42, 8,
  27,50, 26,39, 28,15, 41, 2, 32,55, 31,39, 43,32, 57, 7,
  39,15, 49,22, 54,51, 58,19, 58,12, 20, 1, 40,38, 28,38,
  27,24,  6,32, 43,41, 17, 5, 12,51, 10, 5, 54,52, 47, 2,
  18,55,  6,19, 48,58, 28,45, 41,26,  0, 4, 14,54, 44,53,
  34,12, 56,39, 56,31, 32,38, 36,10, 23, 7, 15,46, 52,34,
  59,33, 53,55
};
const int p_60_8[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  243,
  /// Array of preferences
  29, 9, 19,16, 15,18, 18, 9, 20,57,  3,14, 14,10, 49,47,
  14, 8, 39, 3, 11,45, 25,59, 37,29, 31,57,  5,38, 46,16,
  34, 3, 37,41, 32,30, 40, 4, 49, 2, 14,34, 32,31,  1,49,
  17,42, 18,43,  6,14, 18, 2, 41,46,  0,50, 32,50, 24,37,
   4,37, 53,51, 10,49, 40,15, 27, 4, 48,57, 49,11, 46,29,
  51,27, 18,25, 48,33, 44,51, 17,27, 36,40, 25,19, 52, 0,
  21,30, 28,20, 50,46, 32,28, 19, 6, 43,34,  6,50, 58,41,
  12,55, 16,19, 31,31, 44,23, 57,14, 29,39, 38,58, 33,41,
  30,12, 36,17, 40,28,  6, 3, 15,45,  3, 7, 40,48,  1,50,
   7,10, 37,37,  2,29, 24, 7, 28,41, 13, 4, 45, 3, 23,17,
  52,53, 34,11, 59,57, 50,42, 10,33, 44,13, 50,52, 59, 5,
  11,44,  0,12, 43,22, 22,35, 27, 3, 30,26, 34,10, 29, 0,
  57,11, 15, 5,  0, 8, 26, 4, 57,45, 25,13, 27,19, 50,17,
   9,10, 53, 0, 16,14, 48,35,  4,30, 10,28, 50,22,  2,22,
  34, 6, 23,49, 54,42, 58, 9,  2, 4, 11,50, 30,51, 41,35,
  56,23,  6, 4, 13,21,  0,15, 59,41, 52, 9, 22,21, 44,29,
  58,20, 56,27, 27,42, 44,52, 44,11, 25,55, 16,41, 21, 2,
  36,42, 24,45, 55,37,  8,29, 46,47, 51,26, 19,46, 49,31,
  49,33, 58, 5, 51,37, 42,59,  2,15, 45,49, 21,27, 25,49,
   8,24, 39,25,  0,18, 11,18, 32,38, 51,23, 56,11, 20,24,
  24,56, 16,58, 40,18, 17,22, 17,47, 52,50, 28,26, 26,23,
  46,14, 10,37, 21,24,  4,18, 25, 6, 18,54,  1, 5, 20,50,
  42,15, 29,10, 46,49, 44, 2, 54,21, 58,18, 34,34, 42,20,
  56,29, 48,40, 12, 0, 55,12, 31, 4, 54,17, 50,16, 36,19,
  23,19, 12,12, 53,59, 49,27, 23,35, 58, 8, 31,17, 16, 4,
  35,20, 57, 7, 11, 3, 42,51, 52,45, 11,47, 41, 4,  7,47,
  14,28, 54,49, 28,59, 26,44, 16, 3, 16,27, 35,28, 38, 1,
  40,14, 18,37, 19, 2, 12,47, 35,19, 45,40, 30,23, 45,48,
   1,47, 47,49,  8,11, 51, 1, 21,54, 53,45, 47,41, 29,50,
  42,42, 18,48,  1, 1, 52,26,  4,44, 29,25, 55,14, 26,31,
  25, 2, 56,57,  5,40
};
const int p_60_9[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  286,
  /// Array of preferences
  44,22,  9,41, 13,46, 12,25, 20,27,  7, 4, 29,26,  7, 8,
  17,59, 17,22, 39,37, 44,49, 46,52, 39,26, 59, 6, 45,26,
  43, 5, 51,32, 49,51,  2,30,  5, 4, 17,48, 44,12, 20,14,
  43,53, 58, 7, 29,17, 13,56, 29, 1, 24,58, 56,27, 54,42,
  52, 2, 33,45, 33,28,  8,33, 48,27,  4,51,  9, 3,  6,51,
  32,53, 37,26, 23,27, 29,32, 48,57, 42,57, 27,46,  1, 3,
  34,25, 31,30,  0, 8, 52,15, 48, 9, 12,43,  5,24, 44,14,
  36,19,  3,10, 23, 8, 32, 4, 15,37,  2, 4, 49,39, 13,33,
  38,36, 27, 0, 21,14, 43,57,  0,22, 48,54, 10,33, 58,34,
   3,33, 21,43, 50, 8, 13,36, 22,23, 41,21, 11,41,  8,46,
   5,39, 30,28, 42,18, 23,18, 15, 0, 56,16, 45,35, 36,43,
  57,59, 23,29, 47,25, 37,47, 40,56, 55,21,  9,28, 42,16,
   1,29, 36, 6, 42,46, 48,52, 45,38, 30,43, 10,21,  9,55,
  51,23,  5,34,  7,28, 21,10, 33,19, 43,20,  4,57, 51,24,
  58, 6, 49,59, 54,41, 19,44, 51,13, 29,49, 43,16,  8,29,
  22, 6,  1,43, 31,21, 35,19, 55,59, 44, 5, 54,12, 20,36,
  49,19, 19,37, 13,35,  4, 3, 49,26, 10,43, 17,20, 58,39,
  16,37, 54,18, 23,16, 27,36, 51,37, 20,41, 44,26, 22,39,
   4,26, 55,45, 14,41, 28,25,  4,10, 27,23, 13,59,  3,36,
  45,39, 43,33, 49,33, 10,55, 56,13, 51,38, 23,26, 10, 4,
  35,11, 12, 7, 47,31, 46,12, 41,29, 34,10, 12,31, 47, 7,
  33,34, 52,11, 42,19,  9,38, 25,15, 59,29, 10,42,  5,10,
  56,44, 38,42, 43,15, 30,23,  3,19, 15,57, 39,31, 38,10,
  22,25, 52, 9, 59,39, 41,49,  8,23, 52,26, 53,19, 39,21,
  17,56,  9,53,  6, 5, 16,10, 39,25, 45,30, 39,44, 41, 1,
  35,21, 46, 9,  0,53, 51,27, 43,55,  9,30, 52,36, 40,29,
  36,15,  6,29, 13,47,  7, 5, 50, 7, 31,31,  0, 4,  2,32,
  20,53, 17,53, 13, 0,  0,14, 53,37,  0,32, 39,13, 55,11,
  47,44, 10,30,  2,33, 52,45, 35,58, 31,46, 12, 3, 47,55,
  47,23, 47, 3, 20,49, 19,41, 51,40, 50,55, 57,51, 16,50,
  18,47, 43,45, 45,37, 42,40, 21,16, 15,24, 30,12,  9,25,
   7,15, 43,32, 38,55, 20,13,  2,17, 11,33, 31,24, 25,37,
  16,56, 26, 9, 24,10, 26,26,  5, 0, 24,54,  2, 3, 58, 0,
  44,20, 56, 5, 19,18, 57,54, 10,54, 23, 9, 10,22, 22,21,
  31,53, 28,37, 51,41, 24,34, 25,29, 50,35, 46,26, 21, 1,
  42,35, 49,49, 27,45,  1,57, 39,30, 10,46
};
const int p_60_10[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  883,
  /// Array of preferences
  19, 9, 18,49, 34,50, 17, 2, 58, 4,  2,14, 23,51, 17,14,
   0,26, 49,40, 29,51, 13,22, 41, 3, 26,29, 19,45, 19,54,
  14,28, 17,33, 30,58, 47,37, 59,42, 19,10, 20,24,  9,55,
  17,34, 37,57, 17,41,  7,51, 11,42, 54,12, 15,38, 35, 2,
  48, 2, 37,29, 33, 4, 52,26, 26,55, 50,21, 11,33, 28, 1,
  49,57, 29, 5, 47,38,  7,31, 36, 1,  9, 4, 51,38, 35,27,
  20,23,  3,38, 16,35, 45,41, 10,44, 42,47, 57,20,  8,28,
  44,34,  0,50, 13,46, 11,13, 48,47,  2,32, 35,57, 20, 2,
  33,52, 39, 1, 11,27,  6, 7,  5, 5, 36,36,  3,23, 56, 8,
  16,48,  6,42, 53,37, 15,54, 33,31, 24,16, 26,46, 20,44,
   3,17, 20, 8, 10, 6, 27,21, 55, 6, 38,39, 54,53,  9, 8,
  32,47, 24, 8, 20,19,  9,40,  7,55, 52,55,  4,39, 31,14,
   6,12, 44, 2, 41,48,  6, 8, 43,16, 42,21, 36,24, 35, 5,
  53,32, 43,41, 57,33, 57,37, 33, 6, 23,25,  6,39, 49,31,
  11,36, 40,32, 34, 2, 48,59, 59,55, 14,56, 56,10, 49,39,
  49,56, 52, 8,  7,15, 18,54, 39,26, 58, 1, 46,14, 19,19,
  11, 0, 17,39, 15,52, 18,20,  8, 8,  4,14, 16, 7, 38,13,
  50,44, 24,51, 39,20, 16,37,  0,44, 16,22, 10,39, 33, 3,
  15,47, 25,19, 35,43,  1,14, 52,42, 32, 7, 16,21, 32,35,
  18, 9, 58,35,  1,42,  3,42, 29,53, 36,46, 50,34, 22,23,
  56,27, 44,20, 25, 4, 33, 7,  7, 5, 38,54, 57,41, 28,58,
  44,54, 44,42, 54,14, 50,16, 57, 7, 25,37, 38,19, 53,59,
   0,39, 20,28, 38,59,  9,53, 59,17, 25,34, 31,26, 24,52,
  21,52, 47,36, 27,39, 50,47,  8, 5, 35,21,  5, 7, 26,11,
  39,52, 32,52, 17,50, 52,24, 38,40, 41,26, 53,34, 33, 2,
  54,50, 54,43,  8,33, 42,31, 31,12, 59,39, 51, 0, 35,23,
  12,33,  6,47, 50,10,  6,18, 40,22, 30,10, 55,58, 48,36,
   3, 6, 41,40, 50,30,  8,59, 23,19, 57,32, 17,12, 17,26,
  10,22, 39, 7, 14,52,  5,19, 12, 6,  2,10, 37,44, 31, 7,
   7,50, 48,14, 41,27, 27, 1, 56, 4, 56,47, 24,34,  8,18,
  34, 7, 56, 2,  4, 4, 45, 0, 47,53, 44,19,  3,50,  1,52,
  32,53, 19,49,  9, 7, 18,51, 45,12, 48,28, 29,11, 47,27,
  51,17, 12, 3, 25,31, 25,10, 34,32, 36,17, 54, 6, 42, 6,
  38,33, 20,12,  0,15, 34,10, 32,13,  6,17, 30,18,  1,17,
   8,50, 38,50, 25, 9, 48,57, 55, 4, 52, 4, 10,23, 25,46,
  42,41, 51,22,  5,13, 44,12, 12,16, 21,59, 22,29, 31,31,
  19,52, 36,54,  4, 8, 55,46, 43,54, 41,54, 30,25, 20,43,
  36, 7, 26,50,  0,25, 51,19, 31,17, 31,19, 58,15,  2,13,
   9,44, 34,16,  2,11, 29,56, 39,29, 57,23,  1, 0, 29,33,
  50,59, 26, 7, 13, 7, 42,51, 15,18, 10, 8,  9,50, 59,38,
  47,20, 56,41, 54,55,  3,47, 26,21, 13,24,  3,33, 38,47,
  33, 0, 33, 5, 39,17, 34,27, 36,37, 58,51, 34,48, 37,31,
  53,16, 43,10, 45,30, 28, 8, 13,26, 16, 2, 32,25, 15,13,
   9,57, 57,27, 29,41, 29,52,  3,46, 31,22, 17,51,  0, 4,
   7,54, 10,55, 15, 9, 44,18, 11,44,  8,54, 37,36, 10,53,
  46, 2, 38,25,  0,31, 55, 8,  9, 0, 34,25, 27,53, 21,41,
  53,33, 41,12, 50,45, 38,22, 26,52, 24,44, 41,20,  1,25,
  35,34, 31,10, 47, 5, 13,41, 12,12, 32, 5,  7,21, 12, 1,
  55,53, 43,36, 21,13, 52,52, 43,30, 35,53,  2,44, 52,18,
  46,51, 17, 9, 40, 7,  8,49, 40,17, 16,39, 43,55, 46,19,
  53, 7, 56,29,  5,51, 37,10, 58,56, 30,22, 27,34, 22,52,
   9,19, 18,28,  8,41, 33,41, 29,34, 17, 4, 56,59, 14,12,
  38,30, 36,43, 20,51,  1,22, 49, 1, 52,11, 20, 0,  1, 9,
  42,15, 43,14, 13,51, 55,37, 21,43, 27, 0, 24,50,  1,57,
  58,22, 32,49, 18, 3, 29,30, 32,59, 10,28, 51,28, 41,15,
  50,25, 28,35, 46,44,  5, 6, 50,23, 22,41,  3,14, 29,25,
   9,10, 15,36, 57,25,  4,34, 56, 6,  4,41, 40, 5, 28,21,
   8,10, 13,34, 43, 9, 34, 1, 58,52, 19,13, 36,55, 46,38,
  22,21, 24,42, 11,21, 15,19, 11,49, 51,47, 30, 3, 36,41,
  34,30,  4,47, 43,23, 52,13, 51,57, 58,12, 56,56, 11,24,
  20,48, 35,19, 59,53,  1,37, 16,36, 25,15,  3,25, 37,53,
  42,27,  7, 2, 14, 6, 26,56, 52,31,  0,29, 39, 2, 22, 2,
  59,56, 15,28, 16,29, 27, 3, 20,57, 48,22, 12,46, 13,45,
  41,45, 19,41, 16,46, 20,16, 53, 3, 13,44, 42, 4, 23,27,
  51,12, 29, 9, 13, 2, 16, 4, 19,53, 59,36, 54,35, 30, 0,
  37,52, 43, 6, 14,54, 15,34,  4,43, 24,33, 37,18, 30,43,
  45,46, 41,35, 53,18, 52, 3, 53,40, 47,21, 44,56, 46,35,
  29,35,  4, 7, 41,43, 30,23, 24,37, 14,31, 14, 7,  9,45,
  28,14, 54,19, 23, 6, 32,27, 25, 2, 15,43, 34,22, 29,19,
  24,26, 48,21, 36, 2, 49,53, 24, 2, 42,55, 40,46, 19,51,
  13,49,  9,58, 26,18, 20,15, 18, 5, 44,46,  5,46, 39,35,
   1,41, 41,30,  0,49, 19,35, 46,12, 36,45, 36, 6,  3,29,
  29,10, 23,34,  3,40, 34,14, 26,16, 29,38, 47,33,  2, 8,
  52,17, 59,31,  2,59, 44,17,  7,27,  3, 4, 31,44, 25,35,
  37,37, 16,11, 35,33,  1, 2, 47,18, 22,18, 29,46, 27, 8,
  30,49,  7,29, 10,51,  1,58, 34,46, 11,18, 27,36, 41, 5,
  37,55, 44,41, 30,26, 55,34, 49,52, 33,14, 29,58, 32,37,
  54,30, 17,45, 19,31, 23,52, 30,57, 48, 9,  2,52, 52, 1,
  25,48, 57,14, 54,46,  5,44, 27,28, 34, 5,  6,19, 51,33,
  35,35, 32, 4, 22,25, 12,43,  3,59, 29,54, 33, 1, 38,35,
  44,40,  1,49, 57,44, 11,37, 38, 8, 46,29, 51,20,  9,33,
   5,50, 13,17, 31,55, 40,51, 55,17, 24,39, 18, 0, 54,28,
  56, 7, 27,19, 53,21,  5,22,  3, 1, 11,38, 27,11, 38,44,
  28,25, 44,22, 44, 5, 30, 1, 39,36, 22,31, 21, 9,  6,34,
  31,50, 31,42, 44,24, 21,37,  4,24, 14,23, 30,38, 57,39,
  30,28, 36,57, 56,49, 52,29, 17,30, 15,17,  9,39, 12,58,
  26,22, 46,59, 40,58, 45,24, 31,52, 17,38, 18,16, 49,34,
  33,18, 45,34,  4,58, 54,42, 18,19, 37,24, 18,42, 39,40,
  14,35, 41, 8, 13,59, 34,57, 21,45, 52, 0, 38,52, 47,56,
   0, 3, 23,56, 46,26, 40,16, 57, 6,  6,45, 51, 3, 12,15,
  16,47, 22,40, 33,27, 15,25, 22,27, 28,27,  2,57, 29,48,
  30,47, 52,36, 30,13, 21,12,  3,26, 12,51, 20,52, 45, 5,
  56,46, 32, 0, 23, 2,  2,33, 35,38,  2,35, 46,15, 20,20,
  40,57, 10,40, 34,26, 28,31,  5,47, 25, 6, 42,53, 11, 1,
  37,15,  5,35, 54,25,  9,13,  3,43, 21,15, 43,44, 15, 6,
   6,35, 22,56,  8,21, 17,17, 55,33, 40,19,  2,23, 28,40,
  49,30, 41,13, 20,33, 56,57,  7,18, 40,41, 27,31, 35,51,
  23,48, 23,32, 22,45, 53,58,  3,58, 14,29, 42,36, 58, 2,
   0,28, 47,12,  4,16, 34,44,  4,20, 55,42, 39,42, 13,29,
  39,59,  4,57,  7,12, 14,53, 14, 4, 51,30, 53,48, 36,52,
  20,34, 25,13, 41,38,  8,26, 28,59, 11,19,  1,35, 46,58,
   8,56,  9,32, 50,12, 57,43, 49,26,  1,53, 41,49, 27, 7,
  42,56, 10,35, 23,30, 40, 2,  4,23, 49,59,  7,45, 46,34,
  47,50, 36,44, 22,30, 21,27, 54, 4, 52,27, 22,14, 28,17,
  12,17, 46,20, 36,42, 43, 7, 40, 6, 42,26, 26, 1, 29,39,
  13,28, 55,43, 18,22, 31,51, 39,32, 50,46, 59, 7,  5,37,
  12,39, 56,33, 45,16, 30, 5, 22,15, 55,32, 11,14, 25,56,
  50,58, 44,45, 29, 0, 31,58,  2,56, 26,19, 53,26, 45,19,
  52,39, 50,33, 41, 1, 11,39, 19,14, 20,50, 39,16, 24,24,
  53,28,  0,59, 53,24, 29, 3, 45, 9, 16,13, 35,30, 37,20,
  11,26, 41,50, 29,18
};
const int p_60_11[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  445,
  /// Array of preferences
  40, 7,  6,25, 58, 0, 45, 2, 50, 0, 53,38,  6, 4, 23,57,
  35,41, 49,12, 42,34, 26,46,  2,35, 36,38, 54,43, 11,47,
  24, 8,  6,44, 43,29, 11,58,  3,19, 14,47, 26,55,  2,31,
  52,11, 34,43, 43,15, 40,53, 35,24, 19,58, 14,29, 44,30,
  40, 8, 30, 8,  4,24, 30,41, 47,48, 35,59, 21,16, 41,49,
   6,11, 44,20, 55, 2,  9,43, 54,24, 19,23, 35,48, 49,59,
  37,58, 53,17, 52,41, 32,37,  2,22, 24,17, 46, 2, 14,41,
  17,22, 20,20, 33,55, 50, 3,  9,25, 46,38, 24, 7,  0,40,
  13,22, 15,45, 24,55, 12,21, 22,42, 15,15,  4,47,  3,43,
  54,11,  1,51, 58,44, 37,18, 33,34, 38,21, 26,39,  1,32,
  51,23, 19,17, 45, 9, 41,32, 15,59, 52,20, 15, 2, 13,44,
  24,51, 24,20, 41,55, 25,28, 23,54,  6,23, 39,23, 19,46,
  33,16, 49, 5, 14,53, 43,53,  9,16, 56,36, 41,29, 36,40,
   5,47, 19,51, 16,22, 28,51,  3, 9, 30,49,  3,51, 34, 5,
  40, 1, 16,41,  5,48, 51,58, 40,56, 58,31, 44,35, 28,48,
  57,48, 47, 0, 51,40, 38,33, 13,25, 55,56, 11,48,  5, 9,
   8,51,  0, 5, 14,54, 27,32, 20,45, 14,26,  9, 5,  0,24,
  34,48, 22,45,  9,15, 11,14, 18,15, 11,50, 30,14, 56,24,
  24,42, 29,22, 39,39, 56,13, 29,27, 11,32,  2,15, 25, 8,
  35,54, 37,21, 15,30, 10,37,  7, 2, 47,18, 55,57, 26,40,
  54,29, 56,52, 40,12, 23, 6, 48,20, 39,21,  0,42, 52,59,
  16,51,  7,46, 38,41, 51,49,  9,58, 40,18, 50,17, 32,48,
  44,40, 43,30, 18,43, 36,41, 38,58, 40,35, 32, 8, 45,11,
  42,37, 50,28, 18,39, 56,22,  5,53, 15,40, 25,44, 42,19,
  53,35, 24,36, 29, 0, 43, 3,  0, 6, 14,33, 11,18, 58,59,
  56,18, 27,42, 26,50,  2,55, 48,51,  7,38, 54,26, 11,33,
  43,25, 33,12, 45,18, 12,36, 52,33, 44,49, 30,19, 51,11,
  44,22,  5,28,  5,10, 31,57, 49, 3, 18,40, 51,59, 30,20,
  33,35,  8,49,  9, 3,  5,51, 14,52,  5,24, 30,10, 21,17,
  17,53,  8,22, 51,25, 57, 3, 11,46, 43,18, 59,29,  0,15,
  53,47, 44,53, 33, 1, 21,33, 35,30, 39,18, 25,37,  0,18,
  22,46, 48,48, 50,48, 38,31, 14,42, 26,30,  3,47, 35, 1,
  51, 5, 21, 8, 26,54,  7,10, 50,23, 17,48, 32,20,  6, 2,
  31,58, 55,43, 54,10, 29,11,  5,33, 38,45, 16,46, 53,23,
  11,45, 26,56, 40,15, 48,18, 54,45,  5,38,  3,24,  9,26,
  20,43,  1,37, 24,59, 22,27, 12,25, 43,49,  9,17,  7, 1,
   2,45,  6,49, 56, 0, 21,26, 24,28, 30,54,  8,54, 45,25,
  17, 9, 23,47, 23,38, 14, 0, 28, 3,  6,52, 46,22, 12,49,
  27,33,  7,24, 52,56, 50,35, 13,56,  4,16, 51,48, 17,12,
  21,15, 32,31, 20,39,  4, 0, 49,41, 25,17, 57, 9, 29,38,
   4,18,  2,11, 27,48, 55,40, 48,52, 58,26,  4,52,  1,31,
  28, 1, 39,56, 12,38, 51,17,  3,11, 30, 9, 20,53, 19,26,
  48,58, 45,12, 55,41,  4,39, 27,12, 57, 0, 51,29, 38,35,
  52,17, 44,34, 15, 1, 46,14, 11, 0, 18,46,  8,30, 23,59,
  23,44,  7,25, 15,42, 28,33, 51,41,  2,47, 38,23, 39, 9,
  27,21, 45,10, 46,58, 29,56, 54,15,  3,50, 26,51, 54,37,
  47,36, 16,47, 47,31, 44,51,  2,49, 25,52, 26,11, 29,39,
  43,22,  2,12, 35, 0, 52,53, 58,52, 38, 4, 16,30, 55,10,
  21, 0, 12,23, 21,48, 15,16, 52,52, 57,13, 31,21, 43,10,
   0,48,  8,24, 32,39, 31,17, 59,54, 14, 8, 38,43, 35,39,
  19,27, 37,34, 22,50,  9,39, 17,35, 21,50,  5,25,  1,24,
  26,52,  0,13, 46,57, 14, 6, 43, 1, 25,21, 24,23, 19,40,
  44,25, 37, 7, 15, 4, 35, 2, 37,51, 31,24,  7,42, 47,21,
  25, 0, 50,49, 52,14, 49, 2, 45,45, 33,28, 54,56,  3, 7,
  47, 4, 23,18, 36,57, 17,57, 18,49, 45,48,  2, 9, 29,40,
  22,14, 28,24, 22, 5,  5,45, 58, 3
};
const int p_60_12[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  305,
  /// Array of preferences
  59,38, 52,37, 57,31, 38,34, 32,54, 45,37, 50,48, 38, 6,
  51,17, 58,45, 31, 2, 50, 2, 23,51,  3,17,  2, 7, 12,31,
  57,35, 15,20, 33, 5, 12,16, 16,40, 58,29, 42,24, 35,41,
  23,37, 25,53,  6, 5, 43,43, 48,56, 23,39, 15,55, 59, 8,
  22,42,  4,26, 51, 4, 24, 6, 46, 2,  4,56, 45, 0, 24,12,
   1,45, 38,38, 19, 4, 19, 5, 52,45, 56,18, 34,17, 44,46,
  19,37, 26,11, 42,36, 35,23, 51,50, 39,14, 22,47, 50,35,
  36,39, 12,40, 55,47, 30,58,  5,21, 31,58, 54,28, 36,10,
  19,28,  7,23,  1,22, 35,24,  7,38, 10,48, 48,50,  5,23,
  16,30, 42,14, 52,41, 23,11, 52,59, 55,41, 57,47,  9, 2,
  26,18,  4, 5, 24,40, 34,47,  9,16, 35,18,  7,16,  3,40,
  56,59, 44,52, 45,23, 45,25,  0,18, 21,57, 21,10, 25, 8,
  51, 1, 16,11,  5,46, 28,11, 40, 4, 10,41, 55,56, 29,58,
  23, 8, 29,48, 34,40, 33,46,  7,59, 56,43, 21,50, 43,48,
  31,27, 14, 0,  8,30, 21,33, 13,37, 57,19,  5, 5, 40,39,
   5,44, 31, 9,  4,29, 49,20, 32,12, 50,15, 34,49, 29,11,
  50,14, 45,33,  2,15, 48, 2, 33, 7,  9, 1, 26,39, 34,24,
  27,38,  7,18, 18,29, 39,54, 40,55, 46,50, 57,32, 24,55,
  59,10, 16,36,  0,15,  1,20, 45,51, 59,51, 49,56, 15,22,
  52,42,  9,50, 29,43, 12,10, 32,27, 57,18, 42,46, 31,21,
  54,44, 40,44, 13,13, 57,28,  9, 8, 29,57,  8, 6,  1,48,
  33, 4, 26,59, 16,46, 42,35, 41,23, 43, 9, 10,16, 14,44,
  33,27,  3,25,  7, 0, 15,34, 14,40, 13,28, 58,43,  7,27,
   6,45, 34,41, 35,56, 18,27, 11,21, 59,26, 16, 9, 32,15,
   7,34, 18,13, 32,59, 39, 1, 13,15, 28,25,  2,44, 59,16,
  43,55, 48,55, 51,32, 53, 7, 28,24, 28,29, 14,19, 15,12,
   0, 1, 48,52, 32,28, 58,37, 22, 9,  9,27, 10,15,  5,54,
   7,22, 52,20, 37,13,  9,48, 31,46, 36, 9, 39, 2,  5,57,
  36,23, 15,57, 49, 9, 35,59, 19, 9, 41,43,  9,35, 19,34,
  52,31, 58,49, 13,40, 19,13, 50,59,  2,43, 22,44, 53,36,
  21,51, 59,20, 59,47, 36,24, 52, 9, 15,15, 15, 6, 24,10,
  14,24, 27,43, 41,51, 49,13, 59,43, 22,12, 50, 5, 16, 4,
  14,54, 30,37, 56, 1, 12,54, 45,10, 38,46,  7,44, 14,34,
  42,38, 12,55,  8,10, 13,24, 58,18, 15, 9, 19,35, 20,54,
  17,27, 28,51, 11,29, 17,16, 21,52, 29,52, 10,37,  3,57,
  49, 8, 48,28, 44,22, 40,58, 39, 8, 18,16, 16,54, 58, 6,
  52,53, 35, 4, 54,31, 37,47, 56,36, 39,10, 51,38, 10,30,
  32,21,  4,44, 37,10, 23,42, 51,44, 31, 0, 50, 8, 12,14,
   6,42
};
const int p_60_13[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  482,
  /// Array of preferences
  24,37, 28, 8, 14,20,  2,22, 26,27,  0, 0,  1,31,  4,10,
  54,40, 55,45, 48,26, 47,24,  0,56, 59,53, 26, 5, 55,29,
  57,54,  2,12, 54,38, 24,31, 36,15,  7, 7, 12, 2, 44,12,
  20,43, 59,30, 19,31, 24,40, 46,46,  1, 4, 34,55, 31, 8,
   9,32, 51, 1, 19,38, 44, 6, 55, 3, 53,28, 42,25, 40,28,
   1,35, 58,15, 34,41, 54,32, 54,33, 20, 9, 27,12,  0,17,
   3,18, 19,39, 12,53, 27,20, 44, 4, 20,59, 50,58, 16,10,
  38,25, 38,49, 11,59, 32,17, 18,14, 23, 5, 28,45, 43, 6,
  40, 2, 27,44, 28,38, 47, 3, 23,39, 52,13, 52,14, 21,37,
  15,31, 50,36, 22,57, 12,25, 44,20, 43,33,  6,42, 56,20,
   3,25,  3,45, 15,43, 56,34, 58,19,  7,45, 30, 2, 24,33,
  38,10, 18,19, 18, 8, 24,46, 45,29,  7,20, 50, 0, 19,56,
  36,38, 36,39, 50,59, 43,56, 21,31, 27,52, 28,49, 51,28,
   6,37, 23,17,  4,58, 56,31, 47,34, 39,15, 44,39, 56,57,
  57,14, 44,40,  3, 8, 41,10, 17,49, 30,57, 29,58, 46,58,
  24,53,  9,22, 55,39, 36,32, 42,31, 17, 1, 29,26, 58,23,
  47,32, 12,32, 30,39, 20,47, 10, 3,  4,16, 42,59, 39,57,
  32,53, 15,38, 11, 2, 47,44, 31,46, 29,21,  2, 3,  6,31,
   0,58, 40,22, 32, 1,  4,39, 53,43,  1,20, 55,19,  8,40,
  34,28, 52,55, 56,49, 11,57, 45,28, 58,42, 56,56, 17,47,
  50,57, 53,42, 30,56, 27,54,  1,33, 50,24, 39,44,  3,39,
  25,13, 20,15, 19,37, 15, 5,  3,55, 46,28, 45,11,  6,54,
  29,24, 25,41, 36,30,  4, 3, 10,12, 36,10, 27,27, 11,48,
  50,33, 57, 9, 46, 1,  7,15, 53,35, 24,52, 32,45, 28,48,
  42,34, 22,33, 55,59, 27,23, 29,53,  8,33, 48,30, 36,34,
   3, 2, 36,20, 39, 8, 13,15, 31, 9,  3,23, 51,11, 17,27,
  16,31, 46,44, 29,23, 10, 9, 18,18, 40, 5, 45,35,  1,57,
  29,49, 29, 4,  3,15, 26, 8, 46,15, 54,49, 55,51, 15,56,
  12,18, 47,53,  0,22,  6,53, 22,24, 53,34, 58,41, 25,35,
  17,21,  2,41, 17,45, 53,17, 52,49, 37,28, 14,15, 50, 5,
   8,27, 42,32, 48,43, 49,34, 24, 1, 20,23, 41,46, 48,17,
  20,19, 55,30,  7,25, 13, 9, 17,55,  0,33, 12,40,  8,10,
  22,16, 59,35, 12, 9, 37,17, 31,10,  8,46, 52,51,  5,48,
  32,14, 40, 1, 21,51,  2,33, 41,52, 30,41, 51,59,  7,55,
  12,11, 49, 0, 39,13,  9,56, 33, 3, 59,20, 20,51, 25,59,
  12,33, 10,30,  8,29, 15,25,  2,43, 47,49, 35,50, 33,47,
  32,57, 26,52, 20,36, 20,48,  4,20,  9,31,  1,38, 22, 3,
  10,19, 41,57, 20,57, 14,33, 35, 9,  0,28, 15,45,  7, 4,
  53,57, 32,49, 31,52, 14,10,  4,59,  1,29, 24,28, 38,48,
  24,45, 19,29, 41,36,  0,42, 33,21, 44,26, 36,16,  2, 5,
  39,28, 52,54, 28,42, 24,16,  9,57, 21,27, 53,53, 34,52,
  11, 4, 32,24,  0,23, 15,33, 48, 0, 13,35,  6,20, 34,17,
  45,12, 31,21, 22,29, 44,22,  2,39, 31,25, 56, 2,  8,15,
  22,18, 10,22, 38,30, 46,24, 41,56, 11, 0,  4,28, 22, 9,
  23,11, 14,11, 54,52, 14,45, 39,37, 25,32,  5, 8, 32,51,
   7,19, 40,26, 15,23,  6,25, 18, 9, 33, 0, 44,36, 37,58,
  52,58, 35,52,  8, 6, 26,14, 49,11,  8,18,  1, 8, 38, 2,
  50,50, 30,47,  6, 1,  6,27, 42,14, 22,45, 58,59, 55, 0,
  30,11, 40,51, 42,30, 12,42,  6,24, 26,42,  7,31, 17,29,
  27,18,  3,19, 47,50, 26,20, 25,54, 22,48, 56, 5, 23,38,
  45,46,  9, 1, 17,50, 39,14, 53,49,  5,19,  9,46, 24, 7,
  18,26, 13,38, 21,26,  8,54, 57,53,  2, 9,  8,52, 23, 7,
   9,15, 11,52, 36,22,  4,56, 30,35,  8,58, 30,40, 59,31,
  52,18, 22,11, 24,36, 41, 0, 44,30, 19,41, 31,48, 47,30,
  34, 0, 31,57, 17,23, 37,59, 37, 3, 22, 0, 13,53, 41,51,
  14, 1, 17,43, 47,58,  9,45, 25, 3, 31,37, 26,37, 19,51,
  42,21, 32,27,  6,45, 41,31,  2,46,  0,13,  0,43, 41, 3,
  15,39,  9,55, 53,45, 46,20, 59, 5, 48,18, 46,10, 52, 2,
  48,37, 50,49, 39,41, 57,11, 56,51, 30,38,  1,14,  7,58,
  28, 5, 32,23,  5,17, 58,33, 38,26, 35,53,  7,46, 51,13,
  38,13, 47,25
};
const int p_60_14[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  684,
  /// Array of preferences
  35,36, 21,17,  3,13, 35,42, 25, 6,  2,47, 19,46, 34, 9,
  21,50, 50,51, 20,49, 44,17, 20, 7, 38,22, 33,41, 53,31,
  36,52, 48,24, 41, 1, 52, 7, 29,32, 33,58,  4,23, 47, 2,
  10,53, 11,57, 23,20, 22,40, 27,45, 32,41, 51, 2,  5,11,
  28, 5, 35,22, 50,46, 18,44, 47,41, 23, 1, 58,38, 19,30,
   2,42, 45,24, 46,20, 28,31, 46,35, 43, 5,  2,10, 25, 2,
  30, 6, 24, 2, 47,56, 49,31, 18,31, 43,25, 54,24, 48, 5,
  17,15, 29,58, 12,11, 54,58, 21,18, 50,34, 47,35, 48, 7,
   3,32, 42,13, 42,24, 47,50,  0,29, 22,11, 54, 7, 29,40,
  39, 8, 48,56, 32, 5, 49, 5, 15,10,  4,18,  4,10,  1, 7,
  31,17, 59,53, 22,54,  5,41, 46,38, 54,44, 20,24, 29,28,
  10, 2, 23,44, 14,33, 21,27, 16, 7, 45,54, 50,37,  5,44,
  44, 6, 23,42, 26,22, 31, 5, 26, 0, 21,33, 29,33, 36,15,
  56,34, 19,39, 14,51, 47,55, 36, 6, 29,25, 28,40, 48,10,
  11, 1, 39,51, 44, 3, 58,11, 36,27,  0,55, 17, 1, 20,45,
  18,29, 44,28, 10, 3, 33,26, 23,28,  3,48, 38, 3, 29, 0,
  26,42, 56, 6, 28,35, 44, 7, 34,57, 57,34, 11,51,  6,28,
  59,32, 37, 5, 44,56, 34,53, 22,12, 12,24, 20,50,  2, 8,
  41,29, 31,37, 34,38, 58,22, 11,54, 31,58, 28, 0, 16,45,
  18,15,  7, 1, 14,40, 53,54,  8,36, 16,49, 40,49, 12,43,
   0,49, 25,57, 56, 5, 52,40,  5,50, 50,33, 12,25,  7, 7,
  31, 0, 34,48, 13,36, 33,44,  2,35, 39,58, 32, 2,  1,22,
  41, 3,  0,56, 11, 2, 36,58, 15,48, 51, 9, 41,35,  3,53,
  44,13, 40,47, 47,33,  1,25,  8,59, 36,47, 24,13,  6,50,
  59,33, 40,57, 36, 1,  5,45, 26,43, 36,20,  1,36, 50,52,
  41,52, 40,35, 43, 2,  4,30,  2,51, 50,28, 31,57, 19, 0,
  33, 1, 56,55, 30,34, 18,59,  8,23, 40,54, 34,47, 23, 7,
  36,56, 27,55, 39,23, 40,46, 13, 2, 14,10,  7,40,  4,21,
  35, 9, 24,31, 58, 5, 46,42, 31, 2,  1,45,  2,43, 45,56,
  17,17, 14,36, 26,51, 47,51, 55,41, 35,15, 21,34, 36,35,
  58,34, 29,47,  0,54, 42,26, 53,15, 13,41, 33,47, 43,16,
   5,53, 43, 8, 12,42, 20,55, 17,40, 13,22, 43,23, 29,37,
  50,25, 28, 1, 49,25, 13,24, 17,44, 46,28, 38,46, 39,38,
   1,30, 34,33, 57,55, 42,56, 34, 6, 53,26, 42,30, 58,29,
  18,32, 56,23, 32,10, 39,59, 51,26, 53,34,  2,16,  8,11,
  38,19,  7,58, 29,50, 19,26, 45,32, 21, 2, 51,41, 45, 9,
  37,57, 54, 3,  4, 2, 16,48, 20, 1, 56,17, 58, 9, 53,56,
  56, 7, 21, 9, 44,22, 30,39, 34,37, 27,14, 40,55, 56,42,
  30,54, 43,28, 17,47, 29,44, 15, 0, 36,18, 27,48,  6,27,
  42, 9, 17,35, 54,35, 46,22, 19,14, 30,26, 10, 8, 39,39,
  36,12, 32, 1, 13, 9, 16,14,  5,26, 31,28, 59,17, 36,45,
  15,12, 35,37, 16,17, 17,24,  0,25, 55,35, 25,48, 42,46,
  45,48, 33,55,  1, 0, 25,13, 44, 4, 49,57, 12,27, 41,47,
  23,55,  5,34, 19,21, 40,26, 55, 5, 58,23, 15, 3,  2,27,
  54, 9, 45,38, 37,42, 46,19,  5,56, 33, 5, 19,40, 45,28,
  42,42, 31,56, 50,47,  4,41, 35,20,  3,35, 24, 1, 15,14,
  21,39, 33,42, 18,10, 19,31,  5,12, 52,44, 41,34, 31,48,
  44,29, 33,13, 37, 8, 59,59,  1,24,  8,43, 26,11, 43,32,
   3,22,  1,52, 45,41, 29,17, 45,22, 18,51, 43,29, 52,49,
   1,26, 12,45, 14,23,  5, 0, 39,15, 40,50, 27, 8, 36,21,
   8,44, 18,21,  9,23, 46,26, 15,15, 45,51, 27, 0, 50,16,
  41,50, 39,31,  1,59, 58,46, 42,19, 58,41,  4,50, 56,22,
  30, 7, 50, 2,  3, 8, 42, 3, 44, 1, 25,19, 59,31, 30,19,
   6, 2, 23,17, 54,26, 56,32, 17, 0, 51,13, 24, 6, 15,42,
  21, 8, 34,41, 39,32,  9,19,  2, 3, 11,24, 53,16, 14,46,
  22,32,  6,16, 56,57,  6,24,  9,26, 41,30, 25,25, 14, 9,
   1,38, 14,56, 52,24, 25,10, 32,53, 38,33, 57,32, 26,30,
  21,42, 22, 6, 25, 3, 43,13,  8,51,  1,10, 50,14, 10,28,
   1,44, 54,32, 35,10, 26,49,  9,43, 34,12, 16,19,  8,56,
   7,15,  9, 8, 50, 5,  5,37, 12,20, 35,14, 30,53, 54,53,
  22, 5, 42,29,  0, 3, 58,40,  6,13, 37,33, 48, 3, 27, 4,
   6,52, 41,20, 52,35,  5,42, 23,58, 59,11, 40,11, 48,21,
  34,19, 29,34, 48,53, 16, 9, 58,33, 43,38, 30, 9, 27,16,
  45,30,  1,14,  9,40, 49,41, 34,18, 42,22, 14,43,  4,16,
  51,10, 15,55,  5,24, 30, 4, 15,23,  6, 9,  5,47, 27,38,
  50,17, 23,47, 16, 1, 32,42,  3, 5, 20,28, 14, 3, 19,59,
  50,31, 39,42, 43,35, 42,28, 32,55, 13, 3, 24,48, 23,48,
  55,26, 22, 8, 48,28, 27,58, 50, 6, 21,45, 14,20, 37,40,
  19,38,  5,43, 31,15, 19,58, 38,13, 37,44, 21, 5,  1,11,
  53,59,  9,51, 33,52, 53,35, 23,45, 26, 6, 49,40, 26,38,
  58,17, 52,47, 44,41, 42, 6,  3,30, 38,51, 29,21, 19,42,
  58,49, 23,23, 36,41, 49,43, 48,46, 49,45, 56,48, 13,16,
  58,16, 29,20, 55,58,  1,47, 17,27, 46,13, 23,49, 54,33,
   1,35, 47, 6, 19,24, 25,59, 53, 9, 41,19, 19, 2, 57,28,
   2,34, 31,18,  6,38, 50,19,  3,54,  4,11, 54,30, 20, 5,
  15,27, 18,55, 27,50, 24,27,  4,48, 59,47,  1,19, 10,18,
  11,46, 41,55, 16,15, 28,14, 43,47, 26,35, 54,41,  3,31,
  59,46,  6,18, 21,16, 57,48, 55,19, 38,30,  3,18, 27,51,
  23, 9, 38,38, 11,50, 22,53, 11,52, 36, 5, 27,53, 15,20,
  59, 4, 42, 2, 33,12, 54,38, 50, 8, 23,40,  4,19, 37,39,
  26,23, 23,29, 48,50, 59,10, 45,33, 28,44,  6,12, 28,28,
  53, 7,  5,16, 40,15, 50,30, 10,13, 23,57, 14,49, 45,42,
  30,11, 24, 4, 28,17, 45,58, 50,11, 51,19, 57,51,  5, 2,
  41,37, 10,44,  7,24, 59,29, 44,23, 44,46, 32,46, 28,19,
  49,51, 23, 4,  3,33,  6,57,  7,22, 52,18, 17,37,  4,17,
  12,12, 52, 5,  7,16,  8,14
};
const int p_60_15[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  427,
  /// Array of preferences
  54,23,  0,31, 31,38, 17,42, 52, 3, 37,58,  5,21, 39,13,
  28,12,  1, 0, 33,20, 48,12, 44,38, 23,21, 29,38, 35, 1,
   6,51, 20,31, 22,53, 37, 6, 17,31, 32,11, 52,31,  2,36,
  41,13, 31,15, 42,11, 52,55, 22,57, 39,54, 56,26, 45,34,
  53,52, 48, 3, 10,49, 58,28, 15, 3, 40,39, 58,26, 49,17,
  38, 8, 21,20, 34,39, 20, 2, 27,16, 29,30, 57,46,  0,29,
  45, 4, 46,27, 15,33,  5,39,  4,22, 38, 9, 30,25,  1,11,
  28,49, 41,36, 44,12, 19,17, 39,29, 23,42, 51,14, 26,44,
   4,45, 10,29, 45, 0, 50,59, 37,46, 26,42, 49,36, 26,20,
   9,41, 13,35, 15,30, 44, 5, 44,56,  8,27, 55,57, 27,39,
  51, 5, 26, 2, 48,39,  6,10,  3,15, 10, 5, 34,14, 40,46,
  55,27, 10, 4, 17,55, 50,11, 43,33, 18,50,  5,23, 37,23,
  44,52, 47,44, 21,57, 13,17, 54,56, 21,36, 14, 7,  7,43,
  45,52, 59,58, 43,47, 58,13, 49,29, 48,46, 28,14, 30,30,
  39,51, 41,29, 10,37, 50, 2, 50,30, 39,38, 19,26,  8,14,
  31,39, 42,29, 30,34,  6,17, 42,31, 16,58,  2,49,  6,18,
  27, 7, 31,17, 39,26,  0,41, 21,44, 42,50, 40,20,  1,37,
   2,38, 13,20, 31,19,  5,31,  8, 5, 49,27, 31,11, 46,12,
  34,36, 38,19,  4,28, 54,58,  3,29, 17,25, 40, 1, 56, 7,
  53,45, 45, 1, 57,59, 31,34, 30, 0, 56,30, 50,40,  5,44,
  13,58, 48,14, 52,22, 15,11, 32,54, 22,52, 24,49, 28,43,
  35,53, 48,15, 33,10, 51,18, 32,37, 31, 6,  5,52, 46,28,
  58,38,  3,59, 28,38, 38,55, 31,25, 50, 3, 52,29, 57,35,
  41,23, 12,47, 13,45, 11, 0,  5,17, 46,32, 55,46, 47,52,
  53,15, 52,45, 25,22, 41,17, 30,45, 14,42, 54,51, 40,23,
  46,51, 15,51,  7,25, 48,19, 17,49,  3,56, 23,49, 44,30,
  34,53, 44,33, 35,42,  0, 2,  0,25, 16,21, 24,25, 49, 8,
  19,54, 49,30, 28, 5, 37,34,  8, 2, 50,58,  3, 4, 51, 1,
   2,29, 34,49,  6, 5,  7, 0, 39,36, 26,57, 35, 7, 40,40,
  20, 6, 27,43, 36, 1, 50,46, 20, 4, 45,58, 35,17, 42,53,
  25,36, 14, 6,  0,22, 37,12, 18, 6, 53,12, 56,53,  9,50,
  21,31, 43,28, 17,47, 19,10, 36,15, 49,14,  1,21, 53, 7,
   9,30, 12,53, 40,17,  9,43,  3, 8,  9,31, 34,40, 28,40,
  38,59, 38,32, 40,51, 16,27, 33,47, 12,38, 51,16, 24,46,
  53,22,  9,33, 27,32, 24,50, 44,39, 13,28, 47,18, 39,49,
  25,43, 14,15,  3,12, 45,53,  1,10, 28,10, 42, 9, 15,19,
  18,32, 31,23, 35,45, 12,42, 38,16, 41,47, 44,27, 22,51,
  43,53, 12,11,  8,35, 43, 4, 48,13, 36,47, 38,14, 14,32,
  42,43, 39,39, 10,22, 50,57, 34,59, 35,28, 48,53, 19,55,
  16,31, 23,29, 50,37, 16,49, 41,31, 37,18, 40, 7, 49,10,
  32,50, 46,14, 53,24, 57, 6, 21,27, 17,37,  5,35, 54,37,
  10,55,  2,47, 56,56, 12,55, 36,59,  4, 4, 22,33, 29, 3,
   3, 2, 29,53, 22, 3,  8,21, 37,56, 45,45,  8,47, 28,20,
  11, 4, 42,33, 40,12, 32, 0, 18,17, 54,30, 56, 9, 46,24,
  19, 6,  3,47, 48,22, 57, 3, 52,37, 24,59, 57,55, 10,35,
   0,36,  9,36, 42,51, 24,11, 26,41, 19,59, 31,54, 38,12,
  55,24, 19, 4,  0,18,  5,10, 51,51,  5,58, 23,32, 17,35,
  37,53, 45,31, 35,15, 30,41, 43,49, 43,26,  7,57, 54,55,
  40,34,  5,16, 52,39, 20,11, 55, 0, 35,43, 52, 1, 30,14,
   0,13, 57,25, 45,25, 22,17, 43,32, 26, 9, 48,35,  6,40,
  36, 0,  7,20,  7,51, 37,20,  0,28, 30,17, 45,44, 12,57,
  28,44,  3,50, 55,36,  2,13, 27,42,  2,45, 43,12, 14,49,
  26,26, 16,32, 57,53, 28,18,  7,49, 51,26, 20,42, 49, 4,
  15,28, 59,47, 52,11
};
const int p_60_16[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  507,
  /// Array of preferences
  20,28, 57,22,  4,21, 17,44,  2,32, 51,14, 14,32, 32,59,
   8,58, 18,42, 12,35, 44,57, 48, 8, 45, 8, 47,15, 43,24,
   5,13, 19,26,  2,19, 57,23, 21,32,  1,58, 48, 9, 37,22,
  20,30, 26,52, 44, 2, 37, 1, 42, 2, 11,51,  1,51, 46,21,
  54,48, 47,10, 53,38, 46,43, 21,28, 48,33, 12,58, 43, 9,
   0, 6, 20, 7, 13, 1,  8,29, 15,35,  3,45,  1, 7, 45,29,
  54, 7, 11,22, 40,25, 10, 3, 43,27, 50,59, 11,41, 43,16,
  28,16, 39,29, 27,58, 45,43, 58,57, 18,12, 55, 7, 45,40,
  12,36, 17,29, 57,39, 21,37, 24,10, 27,22,  7,23,  2,33,
  18,41, 24,23, 46,27, 44, 7, 51,11, 58,37, 23,19, 48,17,
  12, 5,  3,20, 53,59, 46,54, 10,20, 33,32, 54,33,  5, 8,
  32,45,  1,28, 23,43, 52,21, 33,21, 48, 4, 23,14, 38,31,
  49,11, 17,10, 47, 3, 27,30, 11,43, 12,44, 51,23, 39,34,
  20,36, 30,28, 35, 2, 46,40, 22,32, 32,53, 52,42, 42,54,
  15, 6, 55,27, 28,13, 12,38, 36,37, 42,55, 41, 4, 28,40,
  48,29, 47,23,  6,15, 29,39, 42,45, 13,39, 30,37, 17,18,
  17,23, 54,34, 34, 6, 26,15, 18,33, 38,21,  4, 7,  7,48,
  42,57, 59,36, 52,25,  3,44, 26,33, 17,20, 57,47, 43, 3,
  40,19, 26,30,  7,29, 49,20, 38,47, 50,29, 14,37, 31,49,
   3,10, 50,39, 25,31, 45,34, 36,53, 12, 6,  0,19, 44, 3,
  18,50,  6,58, 56,46, 51,20, 34, 4, 15,53, 51,48,  6,48,
  47,52, 36,23, 40,44, 19,14,  8,12, 24,39, 49, 5, 26, 0,
  30,32, 44,15, 56,22, 37,15,  9,25, 52,22, 51,32, 21,51,
  42,39, 11,10, 45,26, 56,45, 18,17, 48,37, 20,24, 57,26,
   5,53, 59,37, 54,27, 49,39, 26, 5, 19,18, 22, 9,  4,25,
  45,58, 46,46, 43,22,  3, 0, 26,11, 44,29, 22,56, 31, 8,
  25,13,  9,26, 34, 7, 42,50, 13, 0, 43, 0, 50,40, 23,39,
  30,39, 49,53, 13,28,  9,53, 30,57, 28,28, 16,23, 47,42,
  11,17, 12, 0, 14, 8, 45,25, 10,12,  9,16, 46,51, 11,14,
  26,56, 16,43,  3,51, 28, 4,  6, 0, 35,53, 28,45,  6,44,
  31,30, 44,20, 35,15, 50,44, 35, 5, 14,21, 18,45, 54, 8,
  54,17, 35,23, 38,26, 31,41, 52, 2, 56,54, 11,15, 11,29,
  18,48, 42,28, 31,17, 22,18, 23,40, 14,12, 20,21, 43,19,
  32,52, 23,53,  5,17, 42,43,  4,37, 28,12, 28,44,  6,36,
  52,19, 35,20, 39,28, 16, 9, 46,25, 36,47, 25,22,  1,11,
  15,56, 28,47, 18,29, 34,46,  2, 2,  5,48, 22,15, 23,41,
   0, 4, 23,20, 25,12,  7,14, 40,49, 45, 4,  8,22, 28, 6,
  12,52,  6,43, 52,10, 48,12,  7,25, 38,23, 44,11, 41,28,
  23,30, 56,27, 49,28, 41,38,  6,14, 46,26, 49,48, 32,17,
  20,25, 13,13,  5,19, 38,19, 12,42,  1,54, 46, 2,  1,32,
  17,41, 31,11, 12,10, 14,24, 22,55, 38, 4, 34,16, 46, 9,
  39,41, 22, 3, 18,56, 12,49, 35,34, 24,33,  5,47, 56,50,
  41,57, 50,47, 34,55, 13,21, 53, 8, 39,18,  8,16,  5,54,
  59,24, 32,49, 27,59, 20,55, 24,20,  7,19,  5,33, 36,16,
  27, 8,  2,16, 21, 4, 17,55, 29,54, 25,47, 46,55, 38,28,
  37,54, 27,53, 53,10, 46,11, 50,25, 52,46, 39,22, 14,34,
  25,48, 24, 1,  2,53,  6,16,  8,51, 34,41,  7,26,  1,34,
  28, 3, 40,41, 28,39, 58,25, 41,26, 40,22, 26, 4, 38,35,
  27,10, 33,49, 48,58, 51,41, 14,28, 14,26, 36,44, 50,58,
  33,16, 33,33, 32, 2, 30,12,  1,15, 18,46, 29,26, 54,18,
  34, 8,  1, 3, 38,29, 43,55,  4, 0, 21,17, 18,38,  2,44,
  20,11,  7,30, 44,41, 13, 6,  2,11, 56,57, 34,39, 39,33,
  10,40, 45,14, 43,10, 27,55, 50,55, 52, 9, 46,29, 57,37,
  15,59, 34,45, 10,37,  9,43, 26,14, 40,58, 21,58, 25,21,
  26,57,  2,50, 17,37,  7,11, 54,47, 29,51, 30, 0, 38,18,
  26,38, 40,39,  4, 2, 33, 4, 11, 8, 15,22, 22,59, 58,59,
  14,58, 33,39, 23,22, 45, 5,  6,30, 34,29, 27,51,  3,13,
  52, 0, 19,10, 30,20, 15,58, 29,34, 49,12, 43,15, 14,33,
   8,17, 19,48, 48,36, 49,50, 23,59, 49,41, 24, 5, 48,22,
  37,55, 24,18, 22,10, 53,34,  1,30, 19,24, 48,54, 44,31,
  41,33,  5,10, 32, 3, 58,41, 30,36, 26,18, 11,57, 24,14,
  31,26, 20,18, 58,43, 18, 6, 59, 4, 52,23, 25, 4, 23,44,
   3,46,  5,15,  2,20, 59,47,  7,17,  9, 9, 32,31, 16,19,
  59,30, 40,18, 48,28
};
const int p_60_17[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  542,
  /// Array of preferences
  48,35, 51, 6, 36,42, 51,58, 19,11, 52, 9, 30,55, 10, 7,
   7,25, 50,36, 54,58,  5, 8, 25,59, 59, 4, 51,34, 25,49,
  14,53, 56,19, 51,13, 57,30, 50,13, 18,56, 20,16, 27,48,
   7,14, 34,28, 36,38, 40, 7,  0,15, 47,28, 15, 4, 52,29,
  53,11,  2,23, 13,33, 10,49, 35, 4, 35,24, 12, 5, 11,21,
  35,31, 17,41, 42, 4, 54,29, 28,27, 50,14, 22,59, 57, 7,
   6,57, 53,27, 53,29, 18,36, 54,33, 14,54, 47,57, 43,34,
  14,11, 13,51, 34,37, 32,33,  4,18, 39,49, 55,29, 24, 2,
  36,22,  3,49, 25,30, 34,35, 54, 5, 23, 7, 22,11, 45,54,
  55,42, 25,38,  3,16,  0,58, 23,10, 22,18, 30,53, 35,22,
  23,53, 13,38, 43,35, 50, 3, 47,45,  1,10, 37,26, 26,34,
  50,32,  4,48, 17,17, 53, 5,  8,59,  7,29,  3, 9, 52,38,
  12, 3, 57,18, 59,26, 27,33, 20,59, 25,52, 33, 9, 55,58,
  56,21, 34,20, 31,54, 38,22, 25,19, 54, 3,  1,31, 44,24,
  33,35, 44, 7, 37, 2,  8, 2,  9,21, 22,34, 45,34, 36,20,
  19,43, 55,31, 24,13, 48,36, 15,49,  0,12,  6,59, 22,57,
   9,51, 55,28, 35,12, 27,42, 39,23,  2,58, 10,13, 31,12,
  23,25, 58,53, 42,25,  1,41, 13,40, 39,29, 42, 1,  7, 7,
  44,36, 12,56,  3,11, 32,13, 31,19, 38,19, 20,25, 40,32,
  10,56, 29, 7, 37,54, 59,16, 23,58, 17,29, 35,27, 25,32,
  30,27, 45,30, 31, 8, 41,38, 58,24, 47,18, 22,50, 40,26,
   3, 3,  8,43, 47,12, 37, 4, 47,29,  2,44,  9, 2, 39,58,
  56, 1, 52,44, 52, 5, 11,36, 18,42,  0,50,  3,10, 48,14,
  29,50, 43,21, 21,31,  1, 4,  9,58, 55,15, 25,43, 10,36,
   1, 5,  9, 7, 14,58, 13,28, 15,34, 32, 8, 14,46, 10,44,
  40,48, 41, 2, 55, 2,  4,39, 45, 3, 31,52, 53, 6,  2,15,
   2,36, 39, 4,  5,50,  9, 8, 39,50,  3,48,  8,33, 16,58,
  49,41, 42,13,  4,28, 46, 1, 37,24, 18,54, 20,53, 43, 6,
  45,11, 51,30, 51,23, 57,31, 42,27, 18,37, 20, 1, 57,49,
  20,33, 15,13, 51,19, 56,37, 39,55,  3,50, 20,18, 13,34,
  43,15, 57,39, 51,54, 54,23, 41,47, 22,23,  8,40, 46,41,
  44,20, 39,40, 55,26, 54, 1, 46,24, 45,10, 49, 6,  2,29,
  40,47, 30, 8, 46,15,  5,41, 23,27, 22, 6, 15,16, 28, 7,
  38,41, 12,55,  5, 5, 23,46, 10,28, 24,17, 40,54, 11,59,
  16,12, 41,49, 57,45, 18,24, 51, 9,  2,24, 36,47, 32, 3,
   5,22, 55,44, 25,23, 21,27, 16,31, 23,54, 24,50, 23,48,
  11,22, 36,32, 52,34, 14,15, 32,32, 56, 8, 24,49, 29,37,
  52,24, 19,52, 42,40, 41, 0, 36,51, 10,40, 37,35, 14,31,
  36,48, 53,35, 37,40, 57,55, 12,59, 19,20, 33,54, 33,16,
  19,21, 26,40, 18, 1, 45,38, 22,26, 15,47, 17,53,  7,34,
  55, 3, 55,36,  8,29,  9, 4, 25,27, 16,20, 15,37, 29, 9,
  24,27, 11, 4, 13, 4, 34,34, 59,32, 53, 1,  6,40, 51, 0,
  35,53,  0,30, 36,28, 16,26, 33, 2, 52,25, 32,12, 42,56,
  12, 7, 58,29,  2,31, 45,56, 40,17, 15,31, 11, 1, 20,31,
  28,25, 26, 4,  1,56, 29,28, 26, 8, 24,19, 24, 1, 16,57,
  29,38, 49,57, 11,44,  3,37, 30,13, 42,33, 12,30, 51,46,
  55,24, 17,33, 29,40, 45,44, 39,14, 48,25, 38,29, 59,42,
   5,10, 34,31, 39,22,  8,56, 18,13, 11, 5, 34, 4, 26,58,
  38,54, 45, 9, 31,40, 13,11, 35,39, 28,38, 49,42, 18, 2,
  13,43, 55, 1, 18,57, 57,42, 45,48, 16,15, 16,48,  4,29,
  19,15, 13,14, 24,11,  5,42, 44, 1, 41,17, 31,16, 58,39,
  57,48, 46,13, 13,35, 50,52, 51,12,  8,20, 17,48, 54, 9,
  18,21,  6, 5, 52,41, 23,41, 36,11, 35,28, 59,53, 16,33,
  17,50, 38,49, 12, 4, 51,43, 33, 6, 58,48, 48,34, 12,23,
  18,26, 17,15, 28,29,  5,38, 13,23, 44,15, 10,38, 46,59,
  42,42, 58,50, 48,15, 23, 4,  5,12, 10,30, 58,25, 52,49,
   0,33,  4,12, 28,44, 32,54, 26,49, 43,59, 15,29,  0,37,
  48, 6, 46,27, 33,39, 32,53, 55,32, 24,47, 24, 9, 35,49,
  51,42,  2,12,  2,41, 32, 4,  2,40, 36,44, 32,17, 30, 9,
  27,41, 19, 9, 43,30, 35,50, 15,42, 24,52, 10,15, 19,36,
  15,41, 50,47, 47,46, 38,36, 46,28, 29,44, 29,21, 25,48,
  39,37, 21,58, 47,48, 48,26, 58,18, 23,55, 52, 8, 45,58,
   2,16, 34, 6, 39, 7, 32,42, 26,42, 13,10, 57, 2, 16,49,
   9,31, 26, 6,  4,53, 27, 6, 48,10, 24,46, 35,19, 18,20,
   6, 0, 26,41, 16,59, 49,46, 38,48, 52,59,  2, 6, 46,30,
  59,34,  7,11, 58,43, 59,29, 48,23, 54,31, 24,10, 27,50,
  31,31,  5,21, 55,59, 23, 9, 43,49, 26,12, 57, 4, 35, 8,
  11,19, 45,18, 26,30, 35,51, 36,34, 20,58
};
const int p_60_18[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  264,
  /// Array of preferences
  53,34, 11,45, 39, 6, 36,26,  3,46, 52,50, 36,38, 58,50,
  32,14, 23,43, 41,23, 43,23, 42,17,  1,21, 50,12, 36,19,
   4,11, 54, 8,  9,25, 26,32, 31,58, 48,40, 37,16, 40,48,
  58,30, 14, 2, 43, 0, 13,45, 59,58,  9, 3, 52,29, 38,19,
  33,28, 47,22, 26, 2, 39,38, 27,49, 59, 1, 44,55, 55,17,
  32,55, 22,53,  1,16, 32,50, 39, 5, 54,37, 40,27, 42,52,
  52,58, 42,29, 47,15, 14,32, 52,31, 34,27, 23, 5, 12,54,
  14,55, 15,16, 10,52, 21, 9, 27, 4, 47,43, 28,15, 48,50,
  21, 6, 59,26, 57,51, 26,59, 53,21, 32,34, 28,50, 51,37,
  43,48, 47,54, 29,44, 46,29, 14,48, 51,14,  3,44,  8,23,
  17,54, 38,43, 31,14, 39,28, 52,37, 33,25,  7,26, 36,21,
  56,27, 21,39,  5, 2, 21,23, 55,41, 45,12, 30,52, 29,11,
   3,23,  8,34,  0,58, 35, 1, 58,35, 46,17, 58,31, 55,19,
  29,27, 25,53, 27,36, 49, 2, 21,27, 46,35, 51,35, 50,50,
  54, 5, 47,58, 30, 8,  2,50, 15,26, 27,16, 41,43, 22,24,
  36, 2, 59, 9, 16,22, 22,25, 30,55, 48,37, 25,10, 46,26,
  41, 4, 47,44,  4,48, 12,58,  4, 0, 57,33, 10,40, 55, 3,
   6,47,  9,29,  9,53, 40,40, 59,50, 52,11, 19,34, 43, 8,
  39,53,  8,16, 45,34, 17,14, 19,40, 40,57,  8,26, 28,16,
  21, 5, 18,17, 52,18, 37, 8, 28,13, 44, 6, 49,45, 25,23,
  58,49,  2,56,  9,43, 47,47, 59,33, 35,31, 13, 6,  2,52,
  11,40,  3,35, 15,30, 29,59, 18,12, 28,26,  1,37,  2,35,
  54,26, 25,20, 27,39,  9,19, 33,40, 15,45,  4,50, 38,40,
  22, 0, 32,35,  3,31,  2,29, 12, 5, 33,32, 58,43,  2,37,
  16,16, 57,23, 13,57, 22,31, 57,21, 15,31, 27, 2,  3,24,
  17,21, 56,53,  8,44, 46,52, 47,52,  4, 5,  5,35, 50,48,
   3,47, 11,46, 13,47, 33,18, 26, 0, 54,45, 52,13, 27,28,
  29,54, 53,36, 53, 3, 28,18, 29, 8, 50,10, 13,27,  8,49,
  42,38, 18, 1, 35,26, 30,10, 22,23, 37, 4, 28,10,  6,40,
  44,18, 14,33,  7,29, 17,46, 53,59, 36,34, 48, 7, 21,10,
  56,59, 31,38, 21,22, 48,23, 37, 9,  9,21, 48,38, 42,50,
  54, 2,  1,22, 29,30, 14,17, 20,26, 49,26, 12,50, 54,27,
  50, 5, 24,13, 11,10, 48,14, 12,44, 57,18, 30,23, 10,44
};
const int p_60_19[] = {
  /// Number of people on picture
  60,
  /// Number of preferences
  112,
  /// Array of preferences
  40,51, 39,15, 55,22, 15,46,  0,37, 43,25, 51,44,  5, 0,
  38,41, 19, 6, 45,56, 19,47, 28,15, 40,47, 54,38, 55,28,
  47,40, 26,45, 35,53, 10,28, 10,35, 12,19, 25, 3, 20,45,
  11,25, 30,23, 55, 0, 44,31, 52,34, 58,20, 45,33, 35,20,
  55,49, 12,15, 49,26, 51,52, 32,25,  4,30, 26, 4, 51,58,
  43, 1,  2,47,  3, 7, 14,43,  7,14, 14,23, 31,44, 11,11,
  13, 6, 46,36, 31,19, 37,52, 20,32, 37,19, 22,55,  2,18,
   2, 8, 44, 6, 15,33, 34, 4, 29,24,  3, 0,  7,41,  6, 5,
  28,43, 28,59, 59,55,  7, 0, 43,30, 54,36, 56,34, 56, 3,
  53, 0,  7,28, 38,54, 58,13,  7,58,  3,52, 35,52, 57, 4,
  36,58, 11,54, 37, 1, 57, 6, 10, 6, 11,21, 37,26, 41,26,
  20,30, 20,17, 23, 9, 34,40, 47,26, 57,23, 58,22, 58,10,
  16, 6, 14, 6, 36,57, 41,10, 58,42, 34,51, 38,14, 31,27,
  35,58, 56,43, 37,25, 33,19, 41,54, 11,55, 25,38, 16,14
};

const int p_70_0[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  166,
  /// Array of preferences
  30,26, 50,22, 25,36, 38,54, 51,45, 28,32, 61,28, 27,63,
  42,22, 65,51, 50, 2, 32,46, 17,20, 69, 4, 38, 6, 49, 9,
  37,24, 59,37, 14, 6, 23, 3, 63,23, 21,65, 18,26, 19,63,
  49,65,  8,61, 32,67, 14,11,  4,68,  9,63, 42,60, 18,43,
  53, 5, 38,26,  9,64, 52,32, 20,31, 44,28, 23,18, 16,47,
  66,67, 58,38, 24,47,  4,58, 12,27, 60,57, 31,30, 15,49,
  16,62, 13,42, 65,61, 58,16, 49,15, 49,63, 51,18, 40,49,
  65, 3, 44,24,  8,40, 47,55, 21,34, 25,69, 39,40, 54,51,
   8,66, 33, 7, 60,13, 42,61, 18,53,  3,12, 52,50, 55, 8,
   5,65, 66,50, 21,43,  7,14,  0,29,  7,49, 10,49, 26, 2,
  46,27, 11,21, 29,46, 64,45,  4,24,  7,28, 55,24, 28,61,
  31,27, 63,15, 18,66, 26,38, 51, 8, 56,22, 29,44, 33,38,
  28,50, 34,52,  1, 5, 53,17, 23, 4,  4,25, 51, 0, 42,34,
  29,59, 48,59, 14,12, 40,26, 35,65, 56,40,  7,36, 67,57,
  21,60, 48,43, 38, 1, 66,58, 47,45, 47, 1, 66, 7, 68,19,
  40,15, 39,58, 57,66, 38,47, 53,44, 39, 6,  6,14, 46,61,
  58,43, 22,65, 23,42, 19,45, 17,13, 43,25, 60,43,  2,11,
  29,67, 54,37, 69,46, 31,63, 67, 3, 34,29, 42,26, 65,40,
  49, 6, 53,13, 50, 1,  2,17, 24,15, 34,66, 48,33, 42,51,
  25,49, 22,54, 41,45, 11,37, 12,38, 19,17, 29, 2, 48, 9,
  14,62, 54, 9, 26,37, 48,13, 10,69, 23, 1
};
const int p_70_1[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  1145,
  /// Array of preferences
  68,47, 41,54, 42, 9, 52,41, 33,25, 45,54,  0,16, 63, 3,
  55,40, 10,23, 33,52,  2,69, 48,31,  3, 2, 18,65, 69, 8,
  10,25,  4,27, 23,16, 39,18, 60, 7, 15,40, 26,69, 43,26,
   0,67, 69,39, 15,26, 20,25, 29,32, 11,48, 10,35, 40,22,
  53,51, 31,41, 66, 1, 41,24,  4,20, 56,45, 40,46, 13,62,
  16,43, 47,38, 47,10, 47, 5, 18,18, 60,20, 22,66,  1,60,
  10, 5, 58,18,  5,37, 11,53, 38, 6, 23,48, 66,31, 51,51,
  20, 2, 11,64, 39, 0, 68,46, 69,37, 37,26,  3, 5, 34,34,
  60,41, 50, 2, 67,19,  4,29, 52,55, 13,54, 29,24, 20,52,
  54,69, 60,21, 18, 8, 65, 4, 27,36, 64,13, 54,20, 12, 0,
  37,33, 49,51, 42,19,  7, 5, 26,50, 28, 8, 37,31, 28,69,
  62,54, 68, 0, 12,62, 45,16, 29,26,  1,33, 50,25, 42,49,
  42,52,  0,32, 47,55, 31,21, 14,64, 34,27, 15,19, 55,45,
  68,23, 38,21, 13,66, 60,44, 68,24, 57, 5, 50,31, 58,39,
  39,11, 34,53, 35,41, 60,14,  9,26, 39, 8, 59, 1, 25,60,
  46,66, 16,69,  5,16, 54,23, 69,16, 21,40, 29,51,  8,17,
  20,41, 26,53,  0,37, 63, 9, 33,20, 34, 9, 46,13, 51, 8,
  32,42, 17,11, 51,24,  9, 2, 45, 8,  3,51, 39,42, 62,10,
  23,40, 44,50, 27,15,  1,45, 41,21, 62,30, 65,53, 18,22,
   9,60,  0,48, 51,45,  0,68, 32,13,  0,41, 35,12, 44,21,
  16, 5,  7,47, 29,28, 15,32, 51,61, 69,65, 40, 2, 21,42,
  37,49, 30, 6, 29,33,  9,43, 67,10, 30,69, 10,10,  6,67,
  52,25, 19,44, 33,36, 21, 1, 68,10, 40,18, 52, 3, 27,51,
  11,46,  3,34, 56,62, 48,18, 12,67, 46,43,  9,64, 36,27,
  55, 6,  9, 8, 13,58, 65,27, 63,32, 36, 5, 16,32, 47,65,
  19,17, 15,16, 42,17, 43,38, 69, 9, 29,63, 36,62, 18,21,
  57, 6, 63,59, 28,25, 67,28, 33,24, 25, 6, 53,44,  9,32,
  32,39, 35,31, 28,16, 43,14, 15,64,  1,57, 63,11, 33,64,
  48,12, 41,29, 51,41,  7,43, 37, 1, 35,67, 58,20, 40,68,
   8,27,  2,28,  8,65, 45, 2, 34,60,  3, 6, 30,48, 48,11,
  50, 1, 42,43, 40, 7, 20,22, 60, 4, 51,34, 35,34, 28, 2,
  18,33, 68,15, 67,44, 38,28, 56,43, 61,18, 35,62, 59,57,
  56,15, 32, 6, 13, 7, 35,21, 45,42, 38,53, 33,48, 45,45,
  15,21, 36,60, 24,39, 63,16, 11,29, 14,26, 19,50, 61,68,
   5,21,  6, 2, 41,28, 25,31, 15,37,  5,31, 48, 2, 52,44,
  67,34, 58,14, 26,14, 63,48, 68,59,  2,46, 10,17,  3,42,
  36,67, 38, 3,  0,15, 63, 2, 29,17, 14,20,  1,65, 44,54,
  60,43, 47,11,  9,48,  8,36, 20,37, 16,51, 22,23, 10,37,
  49,42, 65,52,  9,19, 59,33, 63,66,  5,40, 69,26, 31,10,
  31,52, 60,25, 25,35, 54,57, 21,30, 42,64, 46,61, 40,43,
  17,40,  6,65, 52,58, 27,23, 62,31, 42,51, 52,45, 53,35,
  60,32, 54, 8, 59,67, 16,28, 56,42, 53, 0, 23,53, 68,67,
  40,21, 40,57, 64, 0, 10, 1, 25,16, 21,67, 18, 0, 57,56,
  30,43, 55,41,  2,52,  1,25, 37,39, 32,10, 11,47, 14,36,
  64,26, 53,32, 40,63, 22,63,  9,24, 46,40, 17,26, 12,59,
  63,27, 47,57, 22,20, 27,38, 24,33, 33,22, 12,17, 32, 3,
  19, 6, 44,11,  2, 4, 63,34, 65,44, 56,33, 15,48, 49, 6,
   3,20, 30,68, 44,12,  1,21,  3,36, 54,21, 30,53, 67,46,
  57,58, 19,19, 30, 8, 42,42, 51,63, 20,49, 50, 0, 11, 5,
  66,44, 16,38, 19,52, 36,13, 50,48, 15,54, 33,13, 50,45,
  64,48,  7, 2, 67,20, 25,40, 21,41, 44,59, 68,35, 20, 1,
  38,33, 34,66, 46,64, 61,33, 25,67, 15,28, 67,27, 45,67,
  66,51,  2,11, 42,63, 22, 7, 61,47, 28,36, 69,35, 23,39,
  57,13, 12,42, 58,61, 28,32, 41,57, 46,38, 20,10, 49,56,
  33,62, 54,53,  0,65,  3,29,  9,56, 15, 5, 35,24, 53,33,
  18, 7, 47,60, 24,46, 18,29,  7,10,  5,24, 49,16, 43, 3,
  32, 7, 56,54, 66,24, 48,45, 39,52, 15,52, 46,30, 31,57,
   3,48,  9,40, 53,54, 23,38, 14, 9,  8,32, 51,56, 43,66,
  69,64, 64,19, 34,31,  6,27, 13,27, 19, 1,  7,11, 36,19,
  32,35, 23, 4, 59,48,  4,19, 34,20,  4,16, 24,50, 63,13,
  20, 3, 18,41, 52,65,  0,27, 37,53, 24,44, 56, 8, 67,41,
  50,51, 46,32, 58,52, 25, 3, 15,60,  1, 3, 49,28, 39,43,
  68,37, 54, 1, 47,39, 39,33, 64,61, 69, 5, 58,30, 32,14,
  25,53, 27,39, 17,16, 61,15, 25,32, 19,49, 24,11, 64,46,
   2,22, 43,64, 23,30, 24, 1, 10,15, 52,60, 31,42, 57,46,
  28,24, 21,45, 19, 3,  5,50, 61,44, 48,30, 34,50, 21,13,
  27, 5, 37, 4,  2,13, 66,57,  8, 4, 31,51, 31,50, 44,63,
  15,38, 60,49, 19, 4, 41, 7, 42,67,  5,33, 47,58, 64, 2,
  59,42,  6,11, 60,39, 26,51,  2, 8, 46,68, 65,29, 14,32,
   8,46, 31,48, 22,25,  2,55, 19,60, 36,25, 56,31, 25,11,
   5,35, 62,50,  4,66, 50,29,  6,68, 29,56, 52,66, 31,47,
  14,50, 60,45, 10,51, 44,30,  7,37, 29,58,  0,28, 33,31,
  69,34, 28, 1, 50,58, 51,31, 69,17,  5,32, 47,52, 37,66,
  14,55, 49,32, 12,46, 67,53, 15,34, 64,36, 25, 8, 13,69,
  23,49, 38,54, 60,16, 68,50,  4,21, 28,37, 24, 7, 67,51,
  23,60, 66,22,  8,42, 48,44, 55, 7,  1,23, 57, 3, 20,15,
  17,48, 31,13, 67, 0, 26,68, 60,27, 28,48, 24,19, 60,10,
  46, 6, 64,37, 47,45, 57,12, 10,63, 56,10, 20,26, 14,22,
  32,17, 21,29,  2,25,  3,14, 44,27, 27,24,  4,67, 62,55,
  68,14, 56,64, 59,60, 37,46, 17,18,  5,25, 37,65,  1,24,
  31,61, 31, 7, 25,13, 63,44, 38,13, 37, 7, 30,58, 17, 3,
  20,34, 46, 3, 43,31, 25,61, 19,47, 32,18, 35, 8, 36,55,
  52,63, 16,23,  1, 7, 26,23, 46,48,  1,61, 46,41, 24,56,
   8,53, 48,16, 44,22, 43, 9, 50,23, 36,49, 24,63, 48,24,
  45,22, 27,25, 28, 6, 65,26, 66,12, 50, 4, 54,66, 25,46,
  44,41,  1,69, 61, 4, 31,36, 58,43, 43,13,  3, 0,  5,62,
  45,44, 27,29, 22,19, 40,59, 19,12, 34,10,  3,69, 21,12,
  50,47, 32, 0, 21,65, 33, 0, 16,56, 45, 9, 62,39, 37,25,
   4,14, 61,61, 63,33, 42,44, 63,47, 14,65, 33,28, 36,42,
  30,32, 25,59, 20,30, 37,42, 53,66, 33,38, 44,36, 36,33,
  41, 3, 28,59, 44,43, 36,50,  0,61,  6,62, 48,17, 20,31,
  53,10, 55,36, 44,19, 40,17, 61,65, 55,26, 62,26,  7,19,
  61,66, 42,61, 47,67, 53,53, 45,49, 25,66, 24,57, 56,48,
  60,54,  5,42, 40,45,  6,26, 66,15, 22,26, 47, 4, 27,59,
  32,51, 65,43, 63, 4,  8,69, 31,33, 20,48, 35,27, 50,33,
  52,36, 35,22, 41,22, 22, 1, 63,58, 24,54, 24,61, 48,69,
  15,58, 17,46, 54,62, 67,17,  9,59, 14,66, 15,47, 20,69,
  48,19, 33,11,  7,27, 59,62, 65,13,  1,66, 21, 9, 24,35,
  15,41, 66,66, 64,53, 69,54, 42, 7,  7,61, 46,59, 11,43,
  51,35, 46,47,  0,10, 55,56, 31,29, 26, 1, 22,14, 52,23,
  31,60, 57,19, 55, 0, 63,43, 28,14, 53,24, 63,26, 57,64,
  32,41, 49, 8, 30,34, 63,14, 37,52, 38,50, 24, 8, 14,14,
  54,39, 50,65, 56,51, 69,40, 66, 5, 33,60, 57, 7,  4, 9,
  18,13, 34,40, 41,36,  6,10,  9,10, 56,37, 22,24, 61,20,
  14,39,  8,10, 19, 2, 28,42, 37,62, 39,46, 18,57, 32, 2,
  50,55, 19,33, 60,11, 24,14, 20,51, 33,27, 50,13, 32,59,
  12,52, 41,49,  4,46, 35, 3, 38,61, 66,48, 43, 4, 34, 4,
  29,10, 26,57, 23,31, 16,10, 39,17, 19,20, 17,28, 20,14,
   1,37,  3,27, 60,67, 41,31, 12,57, 39,23, 49,60, 35,32,
   2,61, 17,23, 18,31, 57,69, 60,68, 68,54, 35,28,  7,67,
   7,40, 57,16, 33,45, 35,63,  0, 2, 63,15, 52,61, 34,33,
  54,27, 66,25, 49,12, 59,45, 41, 4, 62,64, 66,16, 37,35,
  32,27, 32,12, 41, 1,  7,66, 36,45, 42,20,  4,62, 62,35,
  64,11, 34, 0, 46,26, 51,10,  1,13, 41,25, 29,40,  7,29,
  17,66, 60,51, 34,11, 32,32, 59,12, 69,33, 43,33, 68,66,
  12,61, 13,31, 19,15,  8, 1,  7,36, 69,11, 63,12, 20,18,
   8,40, 67,25,  0,31,  6,23,  7,52, 43,51, 19,11, 69, 0,
   3,53, 18, 1, 67,69, 21,33, 18,12, 24,45, 60,55, 20,35,
  47,29, 20,20,  6,47, 52,37, 67,15, 11,52,  9,18, 38,25,
  55,47, 60,42, 53, 1, 16, 7, 22,35, 17,42, 23,12,  0,43,
  46,60, 15,68, 40,39, 62,23, 27,48, 37,45,  7,14, 52,49,
  63,55, 16,61, 43,41, 34,15, 61, 1, 17,29, 55,46, 39,68,
  69,12, 41,65, 42,31, 16,42, 38,58, 68,20, 38,63, 54,17,
  44,10, 40,29, 13,15, 68,32, 64,22, 33,16, 17,61, 29,48,
  35,61, 68,55, 14, 3, 58, 4, 47,12,  2,27, 37,43, 53,65,
  22,27, 54,55, 11,66, 25,45, 43,24, 14,53, 46,54, 46,31,
   2,23, 48, 9, 51, 9, 52,22, 20,39, 14,42,  3,28, 58, 8,
  53,38, 60,40, 37,41, 48,65, 40,28, 55, 2,  5,12,  9,13,
   4,17, 31,49, 25,17, 45,59, 45,63, 30,42, 21,52, 45,37,
  17,20, 46,62, 45,30,  7,38, 58, 0, 39,13, 49,48,  8,25,
  37,61, 58,12, 68,60, 26,66, 12,35, 44, 1, 20,23,  0,47,
  45,24, 63,56,  0, 3, 49,13,  6, 1, 38,48, 37,34, 18,11,
  64,28, 56,16,  6,24, 25,68, 54,68, 53,55, 37,24, 13,42,
  28,38, 58,38,  8,62,  9,16, 67, 2, 26,35, 27, 9, 40,30,
  26,27,  8,33, 13,25, 43,62, 47,37,  8,31, 15,46, 18,23,
  65,37, 25,56,  4,40, 21,17, 62, 6, 16,24, 67,50, 55, 1,
   3, 4, 10,56, 29,55, 34,58, 12,60, 38,43, 53,64,  2,49,
  13,49, 60,28, 60,62, 54,63, 56,66, 25,52, 48,49, 51,20,
  69,38, 17,30, 30,38, 37, 8,  3,66, 14,12, 16, 2, 60,17,
  13,18, 11,65, 11,23, 19,51, 36,66, 16,34, 15,17, 12,45,
  65,32,  8,54, 41,43,  7,68, 41, 5, 59,22, 10,16, 23,27,
  56,57, 10,22, 10,47,  1,47, 65,24, 31,15, 14,44,  3,56,
  28,26, 59,28, 59, 3, 59,65, 31,18, 15,61, 53,49, 21,64,
   7,21
};
const int p_70_2[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  638,
  /// Array of preferences
  14,51, 62,23, 69,66,  2,66, 20, 9, 53,40, 63,44,  4,18,
  50,36, 56,63,  0,44,  6,47,  5,40, 23,22, 52,14, 53, 8,
   1, 0, 58,48, 51,45, 20,16, 12,68,  8,13, 31,16, 50,61,
  65,60, 41,17, 51, 7, 67,46, 55,36, 25,33, 48,40, 68,35,
  39,28, 26,63, 17,23, 58,23, 31,12, 40,50,  1, 1, 57,27,
  27,63, 41,57, 43, 5, 15,52, 67,33, 62,52, 52,38,  1,40,
  67,57, 28,21,  7, 4, 20,20, 38,30, 42,68, 62,48, 28,61,
  53,18, 21,21, 10,45,  0,11, 23,36, 10,68, 57, 1, 42,31,
  49,32, 30,63, 13,44, 64,28, 33,62, 31,61, 54,60, 41,28,
  34,17,  7,51, 17,51, 66, 5, 41,29,  5,38, 47,44, 39, 9,
  19, 4,  6,20, 31,24, 14,67, 31,63,  8, 8, 14,41, 52,49,
  45, 1, 13,30, 49,52, 69, 3, 34, 2, 56,23, 44,15, 27,51,
  65, 6, 24, 2, 53,39, 37,16, 17,37,  6,64, 42,54, 16,64,
  35,12, 46, 0, 15,14, 31, 3, 19,66, 12,55,  1,19,  4,23,
   3,66, 40,58, 52,18, 49,21, 54,21,  9,36,  2,46, 34,54,
   9,11, 28,42, 12,45, 65,41, 10,31, 28,49, 67,54,  9,13,
  27,48, 14,29, 35,20, 57, 7, 45,33, 11,29, 42,16, 25, 0,
  68,40, 17,43, 53,64,  3,54, 30,64, 48, 2, 54,30, 18, 3,
  66, 6, 26,65, 54,22, 56, 8, 35,23, 65,30, 19,39, 39,20,
  46,28, 45,25, 27,68, 13, 7,  4,53, 65,29, 50,64, 22,31,
  17,45, 24,22, 44,10, 38,15, 51, 9, 51,44, 68,48, 43,67,
  68,47, 14,65, 44,53, 32, 7, 57,57, 53,23, 37, 7, 11,12,
  56,27, 10,56, 65, 9, 50, 4, 34,44,  0,12, 20, 6, 42, 9,
  29,40, 28,17, 24,59,  1,56, 69, 9, 32,30, 54,33,  3,39,
  56,53,  0,63, 54,42,  8,16, 68,30, 16,53, 53,27, 10,61,
  40, 9,  1,51, 36,39,  4,58, 59,53,  5, 3, 11,47, 41,63,
   3,57, 64,53, 17,28, 39,31, 49,23, 44,25, 21,10, 33, 9,
  52, 5, 23,64, 24,68, 38,65, 57,64,  4,49, 50,41,  6,33,
  33,34, 37, 8, 31,47, 31,40, 10,10, 44,48, 41,26, 60, 9,
  17,52, 13,60, 13,19, 22,55, 12,13,  3,46, 64,55, 27,44,
  62,26, 56,56,  4,38, 28,67, 11,36, 27,40,  5,42, 21,33,
  18,28, 33,39, 32,43, 27,62, 24,21, 18,26, 67, 8, 62, 4,
  18,19, 41,11, 23,66,  6,38,  2,21, 62,30,  5,48, 26,66,
   6, 3,  8,68, 45,20, 47,56, 57, 9, 59,42, 60,54, 67,22,
  14,12,  3, 3, 47,15, 29,48, 49,39,  8,41, 68,29, 15,45,
  40,26,  1,47, 55,62, 32,66, 18,66, 17,69, 24,12, 58,53,
  57,10, 59,66, 62, 9, 38,66, 33, 1, 27,49, 35,66, 65,64,
  25,19, 67,17, 37,54, 22,34,  5,35, 54,18, 37,61, 46,62,
  54,62, 48,12, 15,59,  3,51, 21,52,  7,39,  1,49, 48,50,
  15,25, 13,17, 47,10, 51,46, 46,39, 33,52, 18,12, 19,21,
   4,13, 57,37, 62,61, 55,31, 45,34, 53,38, 63,22, 56,17,
  28,32, 36,35, 69,11, 59,39, 50,15, 16,69, 21, 0, 33,18,
  18, 7, 26,10, 13,29, 36,41, 43,46, 44, 0, 27,36, 45,16,
  52, 2, 57,26, 36,61, 45, 7, 45,46, 22,20, 64,36, 56,54,
  51,63, 15,44, 16,24, 22,54, 26,29, 28,27, 55, 2, 50,57,
  36,42, 27,10, 52,35, 53,33, 15,38,  1,29, 42, 4,  6,63,
  36,13, 31, 8,  5, 9,  5,34,  4, 6, 27, 8, 69,10, 57,55,
  29,12, 68,21, 23, 9, 49,48, 46,43,  4,68, 34,21,  0,27,
  13,50, 63,35, 24,33, 66,49, 33, 4, 65,45, 34,58, 55,30,
  21,30, 25,18, 31,17, 58, 6,  4,33,  7,28, 43,68, 57,48,
  57,39,  3,60, 38,17, 46,10, 37,63, 49,43,  5,44, 22, 8,
  10,33, 43,48, 32,23, 20, 8, 61,30, 35,30, 57,20,  1,35,
  15,49, 37,65, 37,33, 53,34, 18,25, 69,62, 57, 4, 53,52,
  35,44, 23, 2, 16,11,  7,38, 28, 9, 56, 1, 56,29, 37,24,
  42,12, 48, 6, 53,58, 20,38, 30,14, 50,13, 27,32, 62,21,
  44,59, 45,56,  9, 7, 57,66, 55, 9, 37,13,  5, 5, 28,68,
   7,24, 22, 7, 26,20, 53,69, 13,41,  1,10, 16,22, 38,31,
  36,33, 58,66, 34,65,  5,59, 12,53, 68, 2, 12,25, 51,41,
  42,11, 32,55, 53, 4, 26,28,  0,36, 66,37, 17,58, 25,34,
  28, 4, 40,53,  4,21, 26, 3, 26,33, 19,25, 19,52,  5,52,
  19,37, 67,44, 28,56, 53,14, 12, 5, 29,66, 65,46,  0,15,
   9,50, 58,43, 50,56, 60,56, 44, 8, 36,47, 45,65,  6,13,
  44, 9, 46,54, 54, 4,  1,63,  9, 6, 29,69, 53,35, 39,26,
  54,11, 34,53, 11, 5, 18, 4, 37,53, 15, 9, 45,15, 57, 6,
  60, 5, 36, 8, 47,11, 40,14, 10,42, 64, 4, 61,50, 58,10,
  65,18, 47,26, 62,55, 56,59, 39,65, 40,55,  1,65, 18,60,
  47,21, 29,39, 28,34,  4,56, 57,59, 46,48, 40,38, 49,24,
  32,42,  0,35, 19,22, 41,50, 31,51, 59,40, 49,17, 58,15,
  42,20, 45,40, 42,13, 32,20, 17,18, 14,47, 10,29, 24,27,
  52,22, 59, 7, 61,23,  5, 6, 12,69, 50,12, 28,16, 42,15,
   3,35, 45,59, 12,23, 29,33, 51, 8, 16,13, 29,60,  4,42,
   8,10, 65,55, 30,10, 29,54, 42,37, 22,63, 32,68,  6,69,
  38, 1, 61,11, 20,45, 44, 6, 18,29, 20,39, 51,29, 62,34,
  28,10, 18,45, 51,65, 48,38,  9,43,  4,22, 57,47, 34,35,
  21,40, 20,44, 55,51, 53,36, 60,23, 29, 2, 12,46, 60, 4,
  59,20,  0,16,  7,57, 27,29, 56,30, 24,17, 60,45, 48,17,
  52,57,  7,18, 29,34, 18,37, 47,39, 66,28, 66,18, 59,30,
  27,34, 39, 0, 47,52, 13,37, 28,43, 12,34, 26,26, 24,16,
  29,17, 43,63, 32,34, 46,26, 39, 3,  4,17,  8,22, 68, 0,
   9, 1, 67, 0, 18,35,  6,18, 36,27, 10,40
};
const int p_70_3[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  960,
  /// Array of preferences
  37,51, 56, 3, 40,14, 64,16, 41, 4, 64,38, 28,36, 55,15,
  17,53, 43,59, 36,43, 11, 6, 58,42, 37,60,  9,69, 63,64,
  64, 1, 60,23, 62,32, 61,19, 51,18,  7,62,  6, 4, 67,59,
  52, 2, 69,67, 16,17, 51,15, 27, 7, 22,51, 56,54, 67,64,
  47,32, 26,36, 53,32, 69,27, 67, 8, 29,21, 12,49,  1,14,
  55,55, 16,58, 52,45, 44,67, 16,65, 49,19, 13,15,  8,28,
  48, 2,  7,53,  6,40,  3,52, 58,53, 11,31, 61,22, 39,58,
   2,28, 26,34, 40,41, 24,65, 18,45, 15,44, 38,54, 17,64,
  67,65, 27,58, 31,58, 26,37,  0,29, 44, 8, 35,65, 13,52,
   5,63, 19,52, 38,44, 44,23, 16,60, 26,31, 27, 1, 35,37,
   7,43, 30,65, 25,49, 19,23, 63,45, 53,44, 51,67, 65,27,
  51,69, 63,62, 40,61, 40,66, 31,38, 13,35, 34,47, 19,45,
  46,47, 54,66, 51,46, 13, 6, 52,42, 17,69, 39,52, 35,46,
  10,19, 23,63, 20,23, 24,33, 68,18, 31, 9, 42,14,  5,34,
   3,13,  3,12,  2, 7, 49,26, 58,47, 10,10,  5,22, 68,46,
  58, 3, 13,67, 25, 2,  6,41, 48, 7,  2,37,  6,14, 60,66,
  60,12, 51,64,  1,26, 39,56, 12,64, 35,54, 27,33, 68,35,
  22, 9, 53, 0, 31,12, 19,55, 24,58, 51,13, 15,35, 10,35,
  21, 8, 27,20, 27,52, 34,26, 19,32,  8,55, 24,38, 61,11,
  41,24, 39,67, 58,69, 32,18, 49,34,  5,20,  0,66, 13, 3,
  23,49, 63, 9, 67,51,  3,43, 19,11, 65,17, 67,66, 31,50,
  21,55, 35,16, 22,32, 63,44, 20,10, 28,27, 50,44, 43,49,
  41,18, 48,64, 18,67, 38,55, 54,42,  5,15, 62,41,  9,40,
  58,46,  6,31, 38,19, 57,29, 21,36, 67,48, 68, 5, 34,20,
  37,67, 36,55, 64,34, 18, 1,  2,56, 42,27, 56,51,  5,55,
  63,38, 34,15,  5,66, 24,29, 49,14, 59,20, 52,65, 14, 9,
   4,17, 18,11, 50,56, 37,18, 54,34, 46,28, 41,50, 49, 0,
   4,36, 66,25,  6,68,  3,31, 35, 7, 17, 4, 32,53, 22, 2,
  14,60, 64, 2, 39,37,  8,64, 69,15, 43,57, 26,46, 39,61,
  69,49, 11,40, 39,21, 66,21, 40,15, 19,50, 11, 3, 45,20,
  35,50, 21,25, 57,40, 32,14, 12,15, 45,32, 13,60, 55,36,
  20,63, 33,61, 51,14, 67,12, 50,11, 64,12, 44,16,  7,10,
  12, 0, 26, 8, 68, 6, 62,13, 48,44, 60,69, 54,41, 44,30,
  30,56, 24,54,  6,49, 47,66, 57,58,  5, 3, 37,63,  7,20,
  34, 2,  0,11, 18,36, 15,16, 25,38, 23,44, 10,48, 52,54,
  38, 1, 55,47,  8,17, 51, 3,  5,45, 61,27, 50,16, 65,60,
  62,65, 35,55, 32, 0, 12,68,  5,52, 14,53, 44,42, 64,47,
   8,27, 63,69, 57,17, 56,40, 39, 9, 38,52, 30,31, 48,10,
  22,39, 12, 7, 48,34, 53,63, 26, 9,  2,41, 32,48,  1, 1,
  66, 5, 68,23, 59,32, 26, 0, 47,46, 24,68, 58,14,  3,37,
  69,10, 46,48, 42,35, 31,69, 55,56, 67,69, 14,12,  1,22,
   5,32, 20,56, 11,43, 53,62, 57, 2, 27,67,  0,64, 46,56,
  23, 9, 30,14, 54,58,  3,63,  2,25, 42,13, 30, 9, 26,49,
  64, 7, 12, 5, 41,57,  1,43, 51,45,  2,22, 14,26, 47, 0,
  41,41, 44,35,  6,24,  0,27,  3,59, 16,50, 12,12,  4,14,
  19,10, 26,64, 23,45, 57,52, 59,35, 46,55, 37,11, 46,17,
  23,15, 40,59, 54,15,  1,34, 44,63, 21,23, 14,35, 38,26,
  23,65, 43,21, 20,57, 34,60,  7,14, 51,19, 67,22, 15, 4,
  35,12, 35,47, 31,22,  9,22, 43,64, 31,62,  3,36, 27,40,
   2,68, 36,19, 25, 1, 17,30, 12,24, 15,65, 51,58, 68,66,
  43,40, 45,12, 21,32, 69, 7, 54,60, 28,67, 45,43, 10,27,
  38,31, 57,67, 39,41, 27,37, 49,33, 50,49,  6, 6, 13,14,
  34,44, 69,36, 49,56, 10,26, 28,60,  0,36, 45,56, 48,26,
   1,13, 58, 8, 38,34,  1,57, 14,49, 17,19, 42,31, 20, 0,
  30,63, 66, 0, 27,43,  5,26, 34,54, 59,13, 30,19, 61,43,
  35,45, 62,31, 13,22,  7,47, 36,36, 33,22, 17,57, 66,52,
   5,33, 62,54, 15, 6,  9,47, 42,50, 51,17, 27,24, 35,66,
  35,14, 11, 9, 43,29, 61, 0, 20,60, 43,26, 63,49, 37,54,
  64,45, 52,35, 61,10,  9,50, 28,30, 25, 5, 26,15,  6,18,
  14,25, 39,36, 55,33, 45,63, 61,32, 68,21, 16, 6,  7,65,
  23,29, 18,53, 55,44, 51,53, 16,22, 65,34, 44,41,  2,33,
  26,32, 48, 6, 42,45, 60,46, 34,11, 43,46, 52,28, 48,28,
  51,25,  3,62, 54,46, 59,66, 56,20, 11,47, 26,13, 43,67,
  64,10,  1, 2, 68, 4, 55,60,  6,42, 34,22, 13,27, 54, 2,
  38,11, 48,12, 20,64, 23,42, 41,11, 56,52, 47,69,  8,68,
  17,52, 26,17, 68,43, 28,20, 14,51,  6,13, 46,51,  7,30,
  41,10, 41,49, 50,59, 33,42, 28,55, 32,15, 26,20, 41,34,
  12,42, 15,45,  2, 4, 30,34, 21,61, 21,11, 45,68,  8,16,
  69,16, 44,52,  0,26, 66,26, 69,64, 37,38, 22,10, 18,41,
   0, 1, 33,47, 34, 5, 52,17, 61,40,  6,20, 57,10, 17, 7,
  15,28, 16,28, 32,16, 29,31, 23,18, 53, 5, 46,33, 66,51,
   1,42, 20,28, 48,65, 13,61, 69,63, 34, 4, 62,67, 68,12,
  61,25, 38,16, 43,62, 32,11, 28,46, 38,22, 56,62, 30,25,
   5,25, 68,40, 48,17, 43,69, 50,20, 50,19, 58,32, 49,66,
  30,59, 32,37, 45,37, 30,29, 28,42, 20,30, 49,12, 68,67,
  33, 3, 30,27, 25,59, 34,10, 17,68,  9,48, 42, 4, 24,39,
  13,65, 30, 4, 40,21, 23,12, 68, 7, 29,47, 36,35, 62,44,
  10,68, 49,45, 14,27,  2,11, 36,60, 46,31, 28,19, 16,18,
  13,42, 51,54, 29,34,  7,21, 60,17,  1,33, 16,54, 39,60,
  67, 1, 15,20, 27, 9, 41,32, 56,65, 49,38,  5,29, 42,23,
  50,25, 23,37, 28,53, 32,13, 22,64, 56,19, 30,43, 16,38,
  19, 6, 17,63, 62,23,  2,64, 14,14, 53,14, 68,31, 16,26,
   6,50, 51,26, 67,35, 45,51, 19,60,  2,48, 25,27, 24,30,
  55,64,  2,54, 61,33, 62,68, 10,61, 41,66, 68,54,  0,40,
  20,43, 43,36, 64, 8, 39,54, 23,34, 26,26, 64, 0, 47,57,
  58,23, 56,33, 22,18, 51,42, 23,10,  9,23, 43,53,  2,57,
   7,61, 49,69,  2,19, 36,63, 45,21, 47,18, 44,25, 38, 8,
  69,40, 37,37, 26, 2, 40,23, 37,24, 33,11, 26,30, 34,67,
  49, 8, 16,41, 47,36,  7, 7, 31,55, 16,32, 28,39, 14,46,
   8,31, 28,56, 44, 7, 64,54, 58,35, 58,21, 15,49, 60,44,
  23,38, 28, 3, 51,34, 36,65, 11,35, 46,61, 53,57, 12,60,
  47,28, 29,42, 42,44, 68,26,  3,39, 12,34, 58,25, 12, 2,
   9,57, 35,15, 57,31, 33,43, 64,60, 67,39, 32,56, 60,11,
  60,64, 13,29,  4,61, 32,65, 53,26, 46, 5, 55,24, 66, 8,
  62,62,  3,24, 60,40, 51,61,  8,49, 18,25, 69,24, 24,32,
  45,36,  2,20, 16,47, 65,59, 14,48, 18, 2, 61,42,  7,23,
   9,43,  1,50,  1,44, 56,49, 40,65,  8,33, 63,58, 42,25,
  21,49, 30,30, 41, 6, 59,36, 23,66, 23,57, 34, 6, 40, 9,
  22,26, 10,39, 47,38, 59,18, 44,27,  4, 4, 59,68, 34,35,
  37, 0, 66,39, 36,45, 52,69, 21,30, 33,38, 10,65, 38,57,
  63, 4,  3, 3, 43,60, 64,11, 37,61, 43,55, 69,61, 21,22,
   9,21, 29,65, 46, 4, 46, 8,  5,48, 60,22, 24,53, 11,57,
  65,16, 29,68, 22,40, 32, 5, 39,29, 61,47, 13,62, 33,25,
  59,37, 39,16, 29,15, 48,59, 19, 2, 61,53, 26, 5, 19,20,
   9,51, 18,15, 41,12, 17, 5, 56,17, 53,30, 24,18, 10,11,
  60,62, 60,56, 43,17, 61, 4, 26,62, 12,43, 38,56,  6,63,
  18,34, 15,62, 21,15, 52,44, 36,26, 64,43, 54,65, 47,41,
  55,37, 15,37,  3,55, 49,57,  4,44, 68,15, 37,49, 48,33,
  42,10, 15,13, 22,48, 53,10, 42, 0, 53,58, 58,20, 20,14,
  49,44, 11,37,  8,13, 64,13, 24,25, 41,59, 24,59, 49,64,
  40,31, 16,19, 42,33, 68,68, 28,29, 56,61, 62,52, 69,13,
   0,55,  8, 2, 19,59, 11,48, 26, 7,  5,35, 24,57,  3,27,
  42,46, 12, 6, 62,66, 51,55,  8,45, 45,58, 45,10, 61,58,
  61,14, 58,36, 32,41, 29,12, 31, 8, 60,26, 27,49, 49,36,
  48,35, 14,40,  9,30,  1,10,  2,34, 38,53, 37,59,  6,30,
   2,49, 25,64, 19, 3, 30, 0,  5,68, 21,62, 31,16, 65,18,
  69,17, 59,65, 37,56,  4,56, 63,46,  0,30, 27, 2, 10,57,
  27,53, 23,11, 39,40, 32,25, 42,39, 54, 9, 14,65, 57,41,
  60,29, 53,16, 22,46, 12,16, 39, 0, 23,62, 38,45, 42,40,
  45,13, 14,41, 35, 2, 10,54, 69, 5, 22, 5, 14,66, 48,62
};
const int p_70_4[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  167,
  /// Array of preferences
  69,68, 41,64, 26,43,  4,40,  1, 7, 17,53, 65,57, 68,23,
  15,41, 54,48, 22,25, 19,37,  5,52, 59,35, 67,13, 47,41,
  65,68, 60,17, 26,40, 59,18,  0,13, 47,45, 22, 2, 24,39,
  39,49,  4,26,  8,41, 40,60, 18,62, 31,30, 59,10,  0,27,
  26,28, 12,46, 51,14,  1,59, 19, 2, 62, 4, 55,19, 37,44,
  48,50, 15, 8, 34,61, 16, 3, 12,45, 44,10,  2,40, 24,33,
  39,42, 40,18, 48,45, 53,67, 64,16, 16, 6, 65,58, 14,37,
  58,39, 39,14,  6,62, 39,25, 53,51,  9,49, 48,48, 26,34,
  17,37,  0,35, 49, 2, 19,18, 19,65, 62, 8, 65,23,  0,10,
  69,24, 33, 1, 26,25, 26,31, 23,45, 16,68, 65,39, 67,61,
   3, 9, 61, 8, 23,27,  7,14, 37,34, 55,41, 59,43, 60,51,
  66, 4, 54,18, 63,29, 29,21,  1,47, 22,11, 24,21, 32,39,
  16,30, 34,17, 11,14, 19,25, 44, 8,  4,35,  2,67, 13,49,
  15,55, 52, 9, 28,14, 24, 0, 54,58, 13,37, 60,10, 53,17,
  33,41, 40,68, 65,31, 58,19, 45,48,  3,14, 22,26, 48,43,
  18,55, 50,25, 16,45, 66,19, 50,69, 20,69, 50,18, 44, 9,
  65,17, 25,44, 19,57, 68,58, 28,46, 61,18,  4,62, 16,57,
  40,19, 36,64,  1,26, 59,63, 25,63, 12,20, 57,19, 51,17,
  16,18, 54,20, 40,35, 41,59, 13,67,  5,35, 54,66, 53,64,
  62,54, 16,65, 69,27, 61,54, 66,37, 63,45,  4,57, 54,13,
  33,43, 16,50, 33,14, 18,51, 25,59, 37,66, 33,29
};
const int p_70_5[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  165,
  /// Array of preferences
  67, 1, 69,54, 48,14, 28, 8, 33,33, 63,64, 38,37, 23,68,
  38,40, 11,35, 14,20, 66,10, 31,47, 13,20, 23, 2, 27,58,
  34,53, 12,53, 34,15, 33, 4,  5,41, 40,47, 30,54, 35,67,
  66,37, 29,34,  7,50, 25,43, 29, 9, 18,63, 32, 5, 50,56,
   6,66, 25,15,  3, 6, 39,49, 69,43, 21,69, 69,57, 61,15,
  60,49, 37,12,  7,62, 56,39, 15,29, 63,20, 66,69, 32,48,
   8,58, 33,13, 27,62, 64,30, 54,53, 53, 2, 63,28,  0,11,
  55, 9, 11,52, 42,65, 67,67, 63,14, 59,67, 30,39, 12,25,
  53,49, 25,27, 62, 2, 15,52, 49,19, 33, 9, 42,54, 51,24,
   7,41, 33,27, 44,50, 41,54, 40,39, 23,29, 37,18, 11,24,
  33,49, 63,41, 16,38, 49,29, 50,28, 58,66, 68,67, 62,56,
  22, 7, 30,24, 48,15, 59, 6, 18,29, 63,66, 54,30, 57, 7,
  45, 2, 21,43, 62,49, 43,41, 66, 3, 46,38,  5, 2, 26, 8,
  24,57,  9,66, 13,43, 12,37,  4,50, 68,64, 14,33, 45,10,
   4, 9, 20, 3, 47,28, 64,36, 54,67, 15,57, 22,67, 35,32,
  51,14, 51,37,  7,58, 60,59, 50,58,  7,26, 55,38, 21,20,
  22,23, 21,36,  1, 5, 24, 3, 48,26, 61, 2, 64,10, 38,66,
  49,51, 55,52, 40,53,  1,10, 11, 7, 47,19, 16,39, 42,58,
  62,41, 68,14, 53,44, 14,46, 56,40, 48,47,  5,48, 60,23,
  23,20, 28,27, 46,33, 64, 5, 34,50, 44,10, 10,15,  9,22,
  57,44, 33,36, 33,39, 26,57, 46,46
};
const int p_70_6[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  919,
  /// Array of preferences
   4,33,  8,27, 21, 4, 17,49, 61,59, 19,30, 65, 5, 53,64,
  47,36, 10,43, 28, 1, 55,29,  7,62, 42,30, 40,68, 45,59,
  32,38, 49,65,  4,54, 62,45, 51,29, 18,38, 21,31,  0,53,
  45,47, 61,34, 33,12,  4,11, 37,13, 13, 5,  7,40, 31,32,
  38,60, 52,41, 20,40,  4,42,  5,17, 11,64, 52, 6, 45,40,
  11,11, 43,14, 17,17, 43,30, 37,28, 46,67, 36,53,  3,14,
  61,54, 24, 0,  6,21, 10, 2, 29,60, 41,55, 16, 7, 62,54,
  43,44,  0,11, 33,55, 22,43, 49,29, 44,59, 65,27, 16,60,
  64,18, 26,40, 68,21, 55, 3, 21, 6, 41,59, 23,26, 68,60,
  52,16, 64,40, 63, 9, 54,17, 34, 7, 13,54, 15,56,  4,46,
  63, 1, 48,52, 53,25, 38,27, 56,19, 41,42, 14,26, 15,38,
   6,26,  3,64,  0,61, 34,62, 19,58, 58,33, 61,13, 30, 0,
  28,34, 36,68,  8,66, 41,44, 34,58, 30, 6, 39,48, 33,10,
  50, 4, 64,58, 12,44, 57, 8, 16,40, 45,46, 21,25, 69,68,
  53,16, 66, 9,  4, 6, 12,14, 35,36, 11,21, 22,18, 17,14,
  13,29,  2,12, 63,40, 32, 6, 40,59, 27,16,  7,44, 56,12,
  51,31, 55,25, 38,31, 20,64,  1,40, 52,48, 32,63, 27,69,
  32,24,  2,27, 43,26, 45,31,  8,59, 28,67, 33,61,  9,69,
   7,50, 38, 1, 56,51, 48,27, 49,31, 11,22, 51,17, 21,29,
  44,18, 54,11, 53,19, 55,54, 51,65, 32,31, 57,37,  0, 7,
  61, 7, 28,47, 24,62,  2,47, 11,14, 68,46, 34,56, 47,52,
  61,63, 15,14, 37,58, 59,28, 39,46, 18,58,  7,49,  3,11,
  43, 9, 25,11, 47, 1, 23,25, 65,17, 12,58, 14,30, 20, 1,
  15,60, 24,16, 22, 7, 33,21, 53,54, 42,45, 62,57, 25,57,
  60,59, 28,15, 28,27, 69,49, 58,39, 49,14, 15,65, 69,34,
  42,50, 66,38, 65,38, 17,43, 33,60, 50,26, 66,12, 36,16,
  60,68, 12,20,  9,46, 31,49, 65,46, 21,20, 13,32,  4,44,
  30,53, 22,23, 68,69, 30,58,  5,12, 14,22, 39,11,  6,23,
  17,24, 37,29, 48, 9, 42,16, 17,29, 28,49, 55,42, 20,52,
  69,17, 26,10, 55,47, 54,34, 62,11, 51,45, 12,26, 35,49,
   0,18, 37,36,  3,68, 31,19, 57,11, 28,52, 10,13, 26,33,
   3,61,  6,24, 30,31, 27, 8, 13, 8, 49,67, 67,29, 27,35,
  67,44, 27, 2, 29,26, 68,50, 66,23, 61, 0, 21,21,  3,43,
  12,11, 27,13, 48,40, 51,35, 39,66, 51,57, 44,27, 11,47,
  42,10, 10,35, 33,38, 66,44, 69,32,  2,56, 16, 1, 30,67,
  53, 3, 31,27, 15,63,  6, 0, 12,52, 61,22,  8,28, 19, 9,
  38,55, 40,56, 13,33, 30,51, 63,14, 57, 5,  0, 0,  4,35,
  33,68, 41,48, 26,45, 64,53, 58,11, 10,39, 16,57,  1,44,
  17,25,  1,21, 41,28, 26,55, 15,64, 23, 0, 62,69, 16,45,
   1,19, 24,58, 18,26, 31,66, 28,13, 47,12, 34,59, 36,55,
  11, 6, 26,23, 54, 7, 40,32, 23,36, 18,46, 48,19, 31, 2,
  41,61, 43,63, 53,61, 60, 3, 60,23, 34, 9, 52,21, 35,14,
  19,37, 14,54, 51,10, 37,55, 60,16, 28,21, 54,38, 16,36,
  66, 6,  8,41, 39,33, 20, 2, 68, 5, 28,51, 49, 0, 40,54,
  31,60, 46,41,  6,15, 53,66, 36,61, 58,20, 66,50, 58,60,
  58,42, 26,30,  3,67,  5, 2, 65,64, 20,32, 48,24, 58,34,
  32,45, 50,19, 34,44, 44,25, 12,18, 43,36, 28, 8, 69,57,
  59,23, 19,33, 37,17, 41,37, 42,22, 27,57, 49,23,  7,10,
  68,49,  4,19, 19,17,  7,51,  7,42, 38,45, 30,23, 27,34,
  35,33, 69,19, 59, 1, 33,58, 20, 8, 42,12, 57,29, 19, 3,
  57,62, 49,52, 51,68, 44,32,  6,56, 24,63, 36, 6, 28,66,
   1,24, 23,31, 24, 3, 49,62,  1,55, 44,12, 41,67, 27,52,
  53,65, 28,44, 33,20, 45,25, 36,19, 12,24, 10,66, 13,23,
  13, 2, 24,43, 52, 0,  4,38, 28,50,  5,47, 37, 0, 54,24,
  67,21, 66,63, 47, 2, 17, 2, 31,47,  6,40, 66,41, 36,28,
  14,20, 50,18, 21,18, 65,48, 40,15, 59,17, 39,13, 14,58,
   9,65, 15,23,  2,49, 12,67,  9,11, 59,55, 36,65, 46,45,
   1,10, 51,52, 33,30,  7,58, 43,32, 69,35, 61,20,  1, 6,
  17,26, 49,40,  8,38, 18, 4,  4,60, 37,30, 15, 9, 56, 7,
  62,47, 63,30, 37,57, 32,58, 14,47, 53,51, 18,22, 53,21,
  26,64, 21,15, 39,43,  6,48, 58,37, 35,21, 16,52, 21,66,
  40,30, 61,15, 54,50, 32,50,  4,39, 15, 5, 50,54, 21, 9,
  14,43, 41,19, 27,11,  1,23, 21,13, 66,62, 36,29, 11,29,
   5,44, 57,43, 56,47,  6,66, 53,20,  2,57, 59, 0, 44,61,
  11,45,  8,45, 47,45, 19,21,  6,20, 53, 6, 35,17, 40, 8,
  45,48, 23,45, 45,29,  9,55, 48,66, 48,58, 20,41, 29,51,
  49,45, 30,49, 32,51, 30,18, 59,24, 65, 2, 20,62, 55, 6,
  57,63,  6,31, 41,53, 24,23, 42,24, 58,12, 56, 5, 54,55,
  24,29, 14,23, 47,61, 46,49, 46, 9, 65,55, 59,25, 32,23,
  49,43, 14,29, 45,23, 47,49, 30,54, 12,64,  2, 1, 58, 8,
  51,43,  7,46, 14, 9, 57,41, 50,63, 19,38, 62,10, 27,56,
  69,51, 50,64, 33,66, 47,38, 19,57, 44,10, 58,47, 42,37,
  65,16,  6,59, 32,55, 59,20,  3, 0, 58,22, 60, 9, 40, 0,
  58,25, 34, 1,  6,51, 39, 2, 48,18, 28,16, 38,47, 39,34,
  65,47, 41,13, 49,55, 26,17,  4, 0, 23, 9, 42,19,  1,14,
  29,58, 36,49, 36,31, 40,65,  9,68, 37,26,  8,24, 53,18,
  38,63, 11,42,  7,68,  1,30, 35,59, 38,53, 62,35, 29,31,
  44,45, 64,63, 27,63,  2,54, 41,16, 25,48,  8,48, 36,13,
   4, 4, 51,58, 45,62, 19,68, 31, 8, 10,62,  7,27, 67, 3,
  42,28, 65,69, 37,39, 56,22,  5,40,  2,32, 53,22,  7,45,
  27, 5, 18,31, 55,14,  6,63, 49,34, 38, 3, 16,24, 29,67,
  28,56, 68,10, 25,23, 50,30,  4, 7, 37,38, 62, 7, 45, 9,
   5,27, 41,30, 64,20, 68,19, 57,42,  2,41, 15,62, 53,35,
  49,21, 55,48, 45, 1,  5,55, 45,34, 47,57, 52, 9, 53, 8,
  69,29,  6,22, 46, 7, 37,62, 37,46, 16,23, 35,26, 27,32,
  25, 6, 24,10, 49,41, 58,67, 30,68, 38, 0, 26,27, 51, 9,
  69,40,  6,60, 54,28, 37,53, 44,39, 69,28, 30,13, 22, 2,
  62,44, 60,18, 20,67, 53,36, 19,44, 65,43, 26,66, 48, 2,
  45,19, 42,60, 55,69, 68,45,  1, 7, 23,32, 39,19, 23,49,
  33,57, 16,28, 38,44, 22, 1, 36,27, 48,29, 20,34,  3,21,
  57,34, 31,55, 25,31, 63,32, 40, 4, 18,23, 64,56,  4,50,
  64,23, 35,27,  8,30, 69,37, 58, 1, 39,15, 17,63, 42, 0,
  67,53, 46,26, 69,69,  1, 1, 63,49, 43,11, 59, 3, 66,37,
  13,39, 41,49, 36,18, 20, 0, 58,19, 40,50, 66,16, 56,23,
  51,66,  7,55, 49,28, 16,46, 10, 7, 16,16, 11, 1,  8,29,
  39,64, 25,18, 17,21, 21,19, 17,42, 67,57,  9,26, 49, 9,
  58,30, 67,40, 46, 3, 24, 5, 59,27,  4,62, 15,24, 39, 9,
   6,14, 19,60, 48,67, 11,54,  9,21, 24,65, 29, 2, 65,18,
  44,44, 14,51, 38,33, 60,36,  2,23, 24,54, 11,33, 36, 4,
  67,64, 30,44, 46,34, 55,56,  3,27,  3,63,  2,60, 23,17,
   7,31, 49,61, 19,24,  4,40, 52,17, 43,69, 41, 4, 57, 4,
  43,19, 56,26, 56,60, 27,60,  3,49, 68,32, 23, 4, 38,25,
  11,49, 11,34, 36,36, 42, 6, 19,27, 31,30, 48, 4, 37,67,
  67,20, 44,68, 37,45, 60, 6, 31,23, 36,48, 22,58, 31,61,
  28,53, 48,65, 26, 2, 64,21, 22,21, 66,47, 39, 5,  8,25,
  41,20,  6,62,  9,33, 54,43, 56,40, 22, 5, 62,59, 50,61,
  24, 9, 45,37, 23,58, 48,15, 18, 8, 44,55, 67,12, 33, 4,
  52,24, 48, 0, 63,46, 62,55,  6,65, 62,19, 29,25,  9,60,
   8, 8, 15,11, 18,37, 19,56, 42,42, 66,13, 38,39, 25,13,
  43,27, 44,64, 67,34, 31,11,  4,27, 28, 4, 21,57, 25,66,
  48,60, 58, 6, 67,32, 18,20, 35,41,  1,43, 32,41, 54,35,
  32,69, 59,46, 47, 3, 35,44, 54,53, 12,48, 15,10, 60,27,
  27, 3, 40,29, 46,60,  7,38, 30,15,  2,59, 39,16, 29, 0,
  10,56, 20,55, 59,19, 51,25, 47,65, 30,32, 48,54, 49,42,
  11,43, 66,19,  5, 6,  8,65, 25,36, 66,49, 18,66
};
const int p_70_7[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  539,
  /// Array of preferences
  46,59, 22, 8, 31,29, 33,17, 12,24, 24, 7, 67,69, 65, 7,
  67,53,  3,19, 54,54, 12,54, 42,23, 39, 9, 45,49, 21,19,
  50,54,  8,53, 21,26, 30,20, 11,19, 26,29, 14,21, 54,69,
  50,30, 46,50, 65,46, 47,51,  1, 9, 38, 6, 48,63, 59,67,
  38, 4, 10,68, 67,22, 68,58,  1,32, 39, 0, 23, 7, 36,14,
  38,14,  6,55, 69,18, 13,66,  8,14,  5,20, 45,23,  1,50,
  23,23, 63, 0,  8,52, 60,51, 42, 7, 55,47, 10,60, 54,57,
   4,51,  1,65, 34,63, 66,51, 42,68, 57,23, 69,27, 44,38,
  17,20, 44,50, 55,44, 18,21, 50, 5, 50,52, 17, 4, 50,40,
  56,24, 68, 1,  3,15,  5, 8, 59,31, 59,23, 34,14, 34,10,
  43,28, 28,26, 43,56, 68, 6, 49, 3, 25,18, 12, 7, 59,11,
  12, 2,  2,37, 40,31, 18,28, 48,64,  1,54, 18,63, 18,25,
  30,16, 36,43, 14, 4, 53,13, 24,69, 26,36,  4,17, 40,68,
  25,28, 60,42, 53,57, 47,52, 35,57, 40,32, 60,66,  3,43,
  12,31,  9, 5, 16,13, 25,47, 43,59,  9, 0, 22,50, 63,10,
  33,66, 23,32, 42,63,  3, 0, 54,39, 29,58,  7,35, 25,40,
  69,44, 52,55,  3,58, 37, 2, 52, 7, 46,35, 32,52, 25,21,
  54,58, 57,13, 13,52,  8,10,  0, 9,  0,29, 15,21, 51,54,
  59,46, 57, 8, 59,39, 30,55, 40,42, 36,58, 16,35,  6, 8,
  11,21, 47,55, 26,68, 52, 0, 50,29, 25,29, 60, 4, 39,13,
   9,23,  7,22, 67, 6, 65,11, 54,42, 25,24, 34,60, 34,46,
  12,67,  5,16, 52,28, 65,16, 42,41, 28,37, 48, 7, 21, 3,
   9,40, 48,20, 56,67,  4,50,  9,25, 69,52, 50,31, 10, 7,
  17,63, 38,22, 25,27, 40,65, 55,55, 36,51, 40,29, 12,11,
  19,48, 14,11, 23,48, 45,30, 37,42, 67,62, 27,49, 15,34,
  20,21, 36,30, 45,62,  0,11,  8,25, 33,62, 32,19, 45,40,
  23,50, 45,51, 34,61,  4,47, 67,21, 28,46, 46,33,  8,68,
   9,62, 33,53, 45,26,  7,26, 14,23, 51,17, 51,59, 52,53,
   8,30, 15,53, 52, 8, 47,17, 27,54,  5,47, 58,37, 52,65,
  11,30, 43,20, 38,42, 15,42, 36,36,  1,46, 69,19,  2,48,
   6,53, 24,43, 40,69, 16,21,  0,56, 15,32, 66,16, 19,50,
   3,52, 16,16,  2,49, 37, 6,  4,21, 69,68, 65,27, 31,25,
   8,32, 18,58, 59,45, 32,63, 37,59, 61,31, 60,12, 36, 4,
  13,15, 51,45, 46,67, 63, 7,  5,33, 26,35, 46,26, 57,46,
  49,65, 52,48,  9,39, 36,19, 32,16, 28,32, 34,58, 36,39,
  27,50, 10,54, 16,68, 11, 1, 55,16,  2,29,  0,64, 27, 1,
  54,64, 44,41, 56, 9, 57,69, 50,22, 41, 7, 40,28, 52,38,
  40,64,  4,52,  7,51, 37,18,  9,44,  2,11, 21,27, 37,69,
  24,11, 22,68, 46, 1, 35,42, 68,55, 49,69, 19,11, 53, 7,
   6,62, 20,68,  0, 0, 62,20, 45,28, 68,57, 20,45,  0,51,
  32,65, 11,17, 65,61, 57,68,  5,34,  5,60, 52,63, 32,40,
  35,13, 28,30,  7,41, 31,50, 45,61, 61,25,  6,14, 66,53,
  62,21, 63,16, 31,39, 31,53, 33, 8,  0,10, 14,36,  6,16,
  10,33, 14,48, 57,48, 34, 4,  4,43, 43,55, 14,27, 53,47,
  34,24, 61,20,  7,20, 21,69,  0,68, 20,57, 39,33, 43,43,
  10,49, 28,15,  2,44, 28,40, 68,51, 60,50, 65, 9, 58, 5,
  28,44, 44,59,  8,58, 61,36, 26,52, 27,62, 59,17, 26,61,
   6,48, 27, 7, 30,13, 19,17, 12, 3, 23,20, 50,42, 22,39,
  43,11, 52,17, 34,25, 50, 2, 29,10, 40,58, 23,15,  6,66,
   8,26, 57,63,  7,17, 20,18, 44,67, 59,55, 38,60, 29,46,
  25,69, 67,57, 51,28,  4, 5, 44,11, 58,59, 23,52, 12,48,
  38,66, 52,18, 66,13, 51,69, 42,56, 16,23, 15,60, 22,64,
  32,66,  7,64, 16, 0, 45,24, 25,32, 38,54,  5,31, 12,13,
  14,49, 64,64, 52,20, 53,38, 61,44, 68,28, 66,42,  7,10,
  40,37, 38,44, 59,42, 45,13, 37,37, 58,35, 62,52, 13,34,
  32,51, 53,60, 55,66, 49,14, 22,33,  1,44, 47,62, 26,14,
  14,44, 17,67, 62,69, 36,65,  7,25, 60,43, 37, 8, 68,45,
  69,38, 18,48, 14,14,  9,48, 27,34, 18, 4,  3,49, 22, 2,
  12,45,  9,50, 36,50, 69,24, 38,17, 38,56, 30,14, 40,41,
  46, 7, 58,27, 57,19, 31, 3, 22,11, 62,17, 67,44, 10,36,
  41, 5, 37,46, 13,18, 11,52, 14,63, 40, 3, 30,52, 27,17,
  45,58, 48,38, 22,18, 20, 1, 51, 1, 57,52, 18,13, 56,22,
  16, 2, 12, 8, 66, 1, 45,43, 28,35, 63,35, 16,47, 56,12,
  43,52, 25,60, 21,55, 59,20,  0,34, 47,41,  7,40, 31,59,
  49,61, 31, 1, 48,46, 33,67, 37,61,  7,56,  4,57,  5,57,
  10,28, 16,32, 59,26, 37,51, 39,68, 55,60, 45,66, 27,21,
  42,61, 50, 6, 40,21, 15,19, 45,12, 59,15, 49,56, 14,10,
  38,51, 52,47, 19,26
};
const int p_70_8[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  836,
  /// Array of preferences
   8,52, 20, 3, 65,50, 17,46, 20,35, 65,45, 43,47, 33,36,
   8,54, 64,31, 21,43, 29,28, 58,45,  9,61, 15,30, 65,64,
  62,23,  6,20, 22,14,  2,51,  5,44, 31,31, 19,16, 52,31,
  24,43, 38,48, 54, 3, 27,66, 16,36, 14,28, 34,32,  6,61,
   6,17, 29, 1, 42,29,  2,57, 61,55, 61,56, 25,23,  5,69,
  35,46,  5,54, 12,39, 57,20, 31,21, 24,12, 60, 5, 27,50,
  33,64, 43,53, 35,65, 26,10, 62,29, 29,21, 56,65, 59,15,
  29,37, 27,36, 69,59, 59,66,  5,15, 63,60, 26,18, 61,69,
  61,41, 56, 4, 49,37, 14,35, 65,58, 69, 9, 61,10, 13,24,
  42,66, 19,17, 35,27,  7,64, 40,42, 51,27, 18,48, 34,54,
  57,22,  9,40,  4,45, 51,65, 15,63, 45,19, 18,18, 43, 0,
  34,31, 65,29, 23,50, 23,69, 68,40, 51,31, 63,21, 12, 6,
  42,33, 61,14, 61,57, 69,65,  8,63, 38,10, 30,62, 54, 7,
  12,10, 30,56, 46,61, 28,52, 14, 4, 45,29, 40,13, 50,19,
  62,11, 16,29, 44,53, 47,29, 48,54, 53,21,  4,32,  6,47,
  13,38, 13,39, 11,39,  6,28, 12,32, 61,28, 51,53, 54,61,
  40,43, 65,49, 27,21, 55, 9, 11,68,  2,21,  3,29, 49,17,
  54,24, 54,20, 12,50, 53,22, 25,19, 38,58, 50,15, 32,54,
  54, 9, 31,50,  8,21, 37,68, 46,37, 27,40, 38, 8, 53, 4,
  41,55, 40,38, 35,30, 28, 9,  1,54, 45,60, 54,50, 40,21,
  58, 0, 12,26, 58,67, 58,27, 32,23, 48,34, 47,46, 63, 8,
  69,38,  8,22, 26, 2, 28,28, 32, 9, 25,44, 35,21,  5,11,
  29,56, 47,17, 39,26, 49,35, 59,52, 44,22, 69,53, 56,42,
  25,21, 69,20, 41,22, 69,27, 29,45, 33, 1, 28, 1, 25,68,
   5,62, 65,40, 21,49, 57, 2, 60,25, 60,15, 51,66, 34,43,
  47,60, 21,40, 50,66, 66,43, 55,34,  4,47, 20,32, 69, 1,
  25,26, 41,61, 37,61, 47,19,  6,13, 33,69, 26,60, 64,48,
  26, 0, 57,41, 11,46, 29,19, 17,58, 54,19, 41,15,  6, 5,
  68,41, 39,21, 44,31, 47,22, 13,32, 62,22, 20,40, 68,47,
  15,13, 60,65, 19,22, 69,47,  9,59, 57,62, 67,21,  1,38,
   1,23, 62,49, 47,40, 16, 6, 53,12, 22,12, 63,48,  1,28,
  64,27,  6,35, 12,27, 42,42, 41,54, 50,21, 40,32, 55,16,
  69,61,  8,51, 38,13, 35,22, 12,49, 39,32,  5,34, 17,22,
  30,31, 11,42, 68,37, 58, 9, 45,31,  4,43, 50, 0, 56,62,
   8,49,  3,32, 50,64, 39,39, 29,24, 43,69, 29,27,  3,57,
   9,34, 59,27, 13,45, 55,53, 47,56, 48,50, 55, 7, 68,58,
   2,24, 68,67, 49,69, 49,34, 58,31, 63,16, 14,19, 45,26,
  50,18, 36,51, 58,34, 37,32, 37,27, 36,41, 49,21, 28,62,
  67, 7, 25,43, 55,55, 29, 7, 17,61, 69,23, 10,28, 49,45,
  24, 1, 20,17, 21,26, 49,12, 61,36, 37,41, 52,53, 24,32,
   8,24, 46,49,  7,61,  6,48, 35,53, 22,26, 52,67, 68,66,
  61,66, 18,35, 52,57, 13,52, 69,21, 17,30,  3,39, 27, 5,
  51,57, 52,69, 47, 5,  9,69, 50,17, 31,28, 54,18, 15,69,
  11,14, 44,11, 45,52, 54, 8, 14, 6,  6,49,  9,48, 18,57,
  11,52, 54,43, 41, 5, 29,35, 24,65,  9,28,  7, 6, 69,13,
  58,47, 38,16, 57,10, 64,12, 37, 3,  5,33, 15,18, 38,28,
  17, 4, 37,49, 43,57, 56,19, 43,11, 13,55, 64, 9, 43, 7,
   4, 5, 66,66, 23,22, 12,46, 39,61, 40, 4, 61,60, 38,38,
  47,35, 63,29, 55,50, 23,40, 30,23, 53,59, 36,69, 12,40,
   6,52,  2,18, 63,41, 20,61, 19,30, 34,42,  7,11, 56, 7,
  17,28,  8,23, 49,20, 12,62, 44,55, 60,37, 67,13, 15,16,
  31,56, 17,69, 19, 3, 21, 1, 58,52, 50,27, 58,15, 53, 7,
  59,42, 62,20, 39,62, 56,24,  4,23, 23,44, 47,63, 60,32,
  37,21, 58,39,  4,60, 24,45, 33,55, 64,64, 31,30, 69,37,
   3, 6, 15,11, 46,26, 13,48, 56,27, 65,23, 67, 2, 58,51,
  25,27, 32,66, 47,42,  1,61, 33, 3, 18,17, 44, 3, 34,64,
  68,24, 69,26, 64,62, 27, 0,  7,41, 48,18, 19, 7, 28,50,
  53, 5, 43,65, 39,24, 66,58,  2,28, 10,32, 62,16,  0, 3,
  51,25, 58,28, 37,59, 32,43, 44,23, 46,40, 23,42, 61,37,
  68,46, 56,52, 55,46, 17, 7,  9,37, 39,41, 51,16, 17,45,
   2, 9, 39,51, 22, 5, 43,42, 26,66, 10,47,  5,68, 15,37,
   5, 1, 60,55, 19,67, 57,58, 48,16, 38,41, 40,22,  6,65,
  40,36, 58,30, 48,51, 36, 5, 49,25, 12,37, 36,46, 27,47,
  40,68, 36, 8, 30,40, 32,62, 16,10, 53,51, 57,28,  2,34,
  42,13, 52,38, 33,30, 67,42,  2,60, 21,13, 35,38, 10, 4,
  25, 6, 25,67, 53,61, 45,57, 30,25, 60, 4, 39,52, 32,14,
  67,48, 36,38, 30,44, 45,13, 43,52, 52,49, 44,34, 32,36,
  27,39,  5,26, 68,34, 54,36, 29, 0, 24,55, 62,59, 17,62,
  15,48, 50,33, 68,54,  3,35, 27,49, 32,19, 66, 4, 52,26,
  66,41, 30,51, 11,34, 67,51, 10,67,  2,49,  2,67, 23,52,
  19,39, 14,42, 52,62,  6,43, 67,66, 65,41, 53,35, 45,61,
  34, 8, 64, 0,  3,18, 52,23, 25, 5, 67,52,  2,50, 29,54,
  67,40,  0,43, 30,55,  1,29, 21,14, 15,27,  3,50, 23, 9,
  30,58,  5,66, 39,43, 66,47, 37,47, 53, 1, 52,60,  3,28,
  50,28, 44,27, 62,35, 32,12, 36,50, 12,45, 45,64, 49,57,
  33,42, 38,22, 11,28, 45, 6, 44,10, 61,53,  6,54, 19,40,
  39, 3, 44,41, 52,64, 65,27, 55, 4, 20,28, 63, 2, 18,37,
  14,26, 51,13, 22, 8, 15,52, 59,25, 69, 8, 23,53, 29, 9,
  26,30, 44,15, 34,39,  8, 8, 16,50, 20,25, 66,16, 12,25,
  16,25,  6, 6, 42, 0, 20,58, 62,31,  7,65, 40,51,  3,44,
  33,12, 43,60, 46,62, 45,67, 33,44, 38,36,  6,29,  6,19,
  17,59, 62, 5, 39,35, 46,67, 25,62, 49,54, 31,32, 67,36,
  42,43, 59,22,  1,39,  2,56, 53, 9, 16, 3, 55,15, 17,54,
   8,46, 22,16, 36,66, 27, 6,  0,40, 46,39, 21, 7, 12,54,
  52,15,  7,46, 62,51, 17,39,  0,15, 11,51, 50, 1, 27,16,
   3,37, 52, 3, 45,32, 62,10, 38,68, 23,16,  7,51, 41,41,
  36,16, 62,58, 48,59,  5,36,  5,55, 46,14, 60, 9, 49,55,
  40,37, 25,51, 25,11,  5,58, 36,15, 16,14, 24,47, 46,50,
  32,42, 16,45, 40,67, 67,68, 23, 0, 43,55, 47,23, 48,21,
  51,48, 39,64, 31, 4,  4,19, 69,29, 24,46, 59,28,  7,24,
  29,63, 15, 3, 26, 7,  0, 1, 55,32, 49,15, 47,25, 50,39,
  36,23, 39,68, 45,51, 69, 7, 29,29, 67,37, 20,56, 27,41,
  46, 7, 20,12, 57, 5, 54,11, 62,18, 45,66, 39,60, 50,37,
  61,52, 26,22, 68,18, 64,17, 49,36, 48, 6, 17,53, 36,25,
  14,66, 16, 4,  9,31, 43,68,  4, 0, 43,67,  1,57, 44, 2,
  53,25, 15,21, 28,12, 20,38, 55,61, 32,17, 41,32,  3,22,
  25,53, 20,45, 27, 3, 49,59, 54,31, 26,48, 67,30, 37,56,
  50,56, 31,20, 33,63, 41,51, 45, 3, 57,17, 21,52, 36,32,
  43,16, 50,49, 28,35,  8,45, 31,52,  7,18, 16,20, 36,26,
  11,18,  1,49, 10,27, 38,51, 25,14,  2,46,  6,10,  0,54,
  18,65, 37,23, 41,49, 39, 5, 61,58, 22,60, 23, 7, 42, 1,
   4,48, 64,39, 39,28, 59,40, 47, 3, 61, 5, 37,54,  7,34,
  42,18,  4,26, 44,18, 17, 3, 17,33,  7,20, 32,63, 54,44,
  23,27, 40,58, 68,55, 32, 8, 25,49, 65,44, 12,59, 53,62,
  42,56, 68,14, 68,62, 11,53
};
const int p_70_9[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  772,
  /// Array of preferences
  28,65, 54,16, 64,45, 28,14, 68, 5,  1,67, 60,50, 67,51,
  22, 2, 27,32, 69,21, 10,61, 26,15, 25,66, 45,23, 59,51,
  35,39, 18,11, 55, 5, 59,27, 69,33, 69, 4, 69,60, 58,20,
  61,69, 31,19, 10,27, 11,38, 48,40, 69,23, 49,35, 41,21,
  59,22, 61,67,  5,30,  6,63, 33,63, 33,40, 19,52, 45,28,
  41,35, 42,59, 22,46, 22, 1, 19,21, 35,58, 69,43, 43,19,
  15,30, 15,48, 17, 1, 60,16, 21,48, 61, 5, 38,55, 59, 5,
  18,24, 38,48, 59,40, 46,23, 61,61, 24,60, 32,26, 61,52,
  15,55, 41,46, 49,30, 45,18, 36, 6,  6,17, 61,19, 33,68,
  14,64, 64,29, 11,65, 51,26, 21,13, 40, 4,  2,23,  5,33,
  24, 4, 61,56,  4,63, 29,69, 61,38, 29,47, 48,50, 39,65,
  33,67, 49,58, 61,53, 53,14, 64,10, 69,27, 67,25, 14, 4,
  23,32, 41,32, 61,64, 15,12, 31,22, 61,48, 66,29, 69, 1,
  64, 6, 37, 0,  1,56, 43,14, 58,58,  5,62, 16,58, 34,44,
  14,15, 53,42, 53,21, 12,29, 51,50, 43, 0, 45, 3, 30,41,
  39,44,  9,47, 15,47, 22,55, 64, 8, 33,48, 68,22, 44,59,
  47, 4, 17,29, 20,58, 30, 9, 40,34, 13, 5, 10,34, 65,42,
  37,13, 17,65, 47,43, 47,20,  3,35, 15,14, 65,69, 51,65,
   4,27, 16, 6,  0,66, 41,27, 32,19, 57,17, 11,43, 22,62,
   9,27, 68,45, 31,51, 10, 5, 52,22, 32,10, 46,20, 37,43,
  22,26, 41,31, 27,67, 54,42, 63,51, 51,44, 16,49, 57,52,
  17,20, 58, 6, 27,60, 35,20, 17,53, 67,43, 26,10, 15, 6,
  46, 7, 44,42, 40,42, 36, 2, 45,67, 63, 6, 63, 4, 46,41,
  51,12, 40,30, 26,55, 34, 1, 18, 5, 32,38, 44,27, 46,13,
  39,14, 55,56, 59,46,  8,68, 42,32, 55,52, 63,62,  4, 8,
  65,23, 23,19, 37,37, 58,42,  9,32, 42, 6, 22,15, 52,19,
  33, 8,  6,50, 52,67, 67,33, 40,28, 33,55, 41, 5, 58, 1,
  53,63, 50,65, 60,29, 49,32,  9,65,  2,56, 56,59, 23,53,
  42,23,  6,26,  3, 8, 59,31, 38,69,  8,35, 19,40, 69,53,
  40,62, 42,67, 21,30, 22,27, 27,35, 10,66, 21,52, 36,34,
  53,39,  7,36, 62,48, 42,25, 64,30, 14,58, 24,57, 62,67,
  26,54, 56,16, 37,53, 27, 5, 49,49, 62,27, 54, 9, 57,41,
  54, 0,  8,37, 67,68, 49,55, 34,62, 13,21, 56, 9,  7,43,
  19, 8, 68,31, 41,10, 20,12, 55,23, 59,13, 38, 1,  1,15,
  25,26,  4,28, 13,17, 17,68, 26, 5, 34,22, 11,50,  5,16,
  44,32, 40,56, 69, 9, 38,57, 56,52, 40,12, 41,69, 34,47,
  64,48, 36,31, 23,59, 46,57, 11,27,  1,28, 25,29, 35, 8,
   6,14, 60,40, 40,18, 54,57, 62, 8, 19, 0, 29,25, 23,34,
  10,63,  5, 9, 27,11, 36,46, 48,55, 39,39, 28,22, 36,54,
   5,10, 12,20, 23,14,  8,57, 50, 2, 42,14,  4,59, 15,41,
  14,50, 58,22, 10,65, 18,37,  4,29, 52,60, 34,43, 59, 6,
  41,40,  4,56, 22,37, 44,35, 41,51, 13,59, 23,44,  6,28,
   9,66, 64, 7,  7,59, 48,44,  7, 7, 59,58, 66,38, 47,35,
  26, 0, 68,10,  1,23,  1,59, 51,59, 24,51, 37,38, 16,34,
  56,37,  3, 0, 39,22, 26, 9, 23,63, 15,39, 27,10, 68,21,
  51,54, 27,42, 17,22, 49,44,  5,64, 56, 4, 27,59, 48,35,
  17,47,  0,51, 61,40, 48, 7, 34,66, 25,37, 49,59, 42, 9,
  26,12,  7,38, 65, 0, 44,12,  0,58, 44,21, 62, 5, 20,41,
  67,65, 64,38, 18,61,  2,69, 27,44, 17,33, 28, 6, 64,50,
  21,27, 49,52, 53,38, 65,24, 56,12, 69, 5, 10,50, 31,40,
  49, 1, 67,53, 61,47,  5, 7, 20, 5, 27,23, 19,63,  3,60,
   9, 9,  9,34, 61,23, 66,28, 33,47, 22,12, 41,34,  9,67,
  33,54, 66,16,  9,28, 16,46, 33,13, 38,65,  7,58, 53,15,
   5, 1, 30,28,  5,34, 35,25, 52,18, 27, 8, 35,59, 12,60,
  49,54, 20,63, 22,51, 16,51, 53,61,  6,19, 61,29, 46,16,
   8,30, 25,19, 62,40, 54,62, 13, 6, 32,51, 11, 6,  7,37,
   9,43,  5,44, 53,16, 62,61, 51, 4, 35,47, 47,22, 67,57,
   2,32, 62,33, 25,42, 11,23, 42,15, 31,53, 65,17, 64,14,
   1,41, 16,40, 69,37, 50,32, 48, 1, 64,28, 50, 4, 18,45,
  32,67, 11,34,  1, 8, 30, 8, 30,58, 62,37, 31,17, 44, 7,
  66,23, 15,62, 35,42, 17,10, 35,21, 43,35, 59,30, 52,55,
  47,52, 44,39, 25,58, 31,64, 14,32, 40, 7, 28,59, 60,35,
   0,69, 21, 8, 47,62, 48,15, 45,21, 10, 9, 41,50, 46,42,
  48,12, 36,52, 59, 1, 38, 0, 40, 8, 22,59, 27,63, 28,63,
  38,25,  2, 4, 62,41, 41,42, 40,15, 10,45, 20,16,  2,38,
  44,54, 64,49, 44,13, 41,68, 53,64, 10,56, 60,28, 53,57,
  50,23, 65,11, 58,18, 29,39, 18,55, 39, 8, 65,36, 53,44,
   3,44, 33,24, 35, 5, 27,50,  6,57,  5,22, 63,53, 31,11,
  32,32,  7,32, 32,33, 43,20, 37, 6,  4,36, 48, 4, 52,33,
   6,42, 24,52, 50,40, 53,47, 59, 8, 14,66, 44,60, 64,20,
  46, 3, 63,34, 67,26,  5,53, 23,28, 37,59, 56,29, 43,18,
  46,11, 38,51, 23,10, 25,17, 18,63,  2,57, 23,69, 15,38,
   1, 4, 15,64, 16,44,  2,27, 24, 6, 38,60, 51,40, 51,11,
   5,47, 42, 2, 17,19, 31,60, 62,64, 62,38, 59,60, 16,39,
  17,60, 55,48,  8,22, 17,38,  8,23, 21,24, 65,37, 18,17,
  58,45, 42,43, 65, 8, 59,14,  7,16, 65,53,  7, 4, 50,29,
  11,20, 39,15, 39,29,  5,41, 66, 0, 57,39, 35,52, 18,12,
  16,69, 51,48, 18,27,  6,41, 64,37, 41,20, 25,20, 50,39,
  58,16, 22,52,  1, 9, 48, 0, 11,11,  4,26, 25, 3, 50, 3,
  23, 9,  6,60, 69,26, 64,21, 38, 9, 34, 7,  6,47, 43,63,
  19,66, 23,54,  8,31, 64,33, 57, 5, 26,23, 51,38, 58,47,
  68,53, 42,56, 56,56, 36, 5,  9,15, 20,30, 67,60, 22,32,
  60,67, 17, 7,  1,44, 32,30,  9,18, 51,47, 10,55, 65,52,
  48,10, 57, 1, 17,67, 39,64, 28,69, 68, 8, 52,20, 19,29,
  32, 7, 10,59, 66, 3, 46,33, 32,36, 20,27, 43,15, 68,28,
  43,13, 54,15, 49,15, 65,46, 25,12,  1,54, 32,12, 68,63,
  57,48, 24,49, 31,25, 58,65, 56, 0, 21,62, 50,14, 55,12,
  49,60, 49,61,  3,31, 33, 4,  0,68, 65,65, 16, 1, 28, 5,
  34,63,  7,13, 50,11, 37,41, 47,11, 62,53, 52,37, 18,68,
  18, 7, 54, 1, 52, 2, 36, 9,  4,22, 41,22, 31,54, 55,11,
  36,50, 40, 5, 11,35, 49,12, 39,27, 55,54, 34,20,  7,31,
   2,18, 19, 7, 51, 2, 13,56, 51,69, 29,23, 63,48, 38,40,
  38,39, 14,14, 66,27, 20, 4, 64, 2, 49,62, 56,32, 26,45,
   8,14, 15,17,  6,15, 14,27, 38,27, 16,67, 63, 0, 21,61,
  33,21, 39,23, 13,42, 60, 3, 68,36, 12,46, 60,52, 59,44,
  12,25, 66,48, 47,65, 55,40, 42,58,  8,52, 38,37, 39,51,
  65, 9, 23,26, 42,11, 18,30
};
const int p_70_10[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  790,
  /// Array of preferences
  41,67,  1, 3, 46,48, 32,59, 23,30, 34,26, 39,29, 20,63,
  35,64, 58, 7, 60,41, 26,24, 61,24, 69,21, 15, 9, 45,14,
   3,64, 64, 4, 52,45, 28,12, 61,48, 51,40, 44,51, 51,29,
  28, 9, 20, 3, 47,19, 28,14, 53,58, 52,34, 15,47, 12,48,
  28,28, 61,39,  5,27, 42,48,  2,23, 35, 3, 56,20,  6,45,
  67,69, 68,60, 61,49, 57, 2, 24,31, 38,26, 22, 3, 14,65,
   2,53, 60,50, 59,24, 24,12, 41,18, 19, 9, 27,15, 49,12,
   3,37,  3,36, 41,24, 59,47, 51,30, 40, 7, 38,43, 64,27,
  30,51, 38,12, 31,25,  6,61, 63, 8, 64,13, 18,11, 49,14,
  27,12, 36,38, 58,13, 50,53, 36,51, 21,50, 51,31, 48,57,
  24,68, 40,50,  8,34, 13,41, 58,15, 58,27, 45,10, 11,37,
  24,55, 66,26, 58, 8, 37,32, 66,59,  4,14, 18,15, 11,68,
  33,12,  5,34, 28,33, 56, 0, 26,65, 66,58, 29,12,  1,39,
  20,27, 35, 9, 11,33, 40,61, 15,57, 67,59, 47,40, 54,46,
   9,21, 35,27, 33, 1, 20,16, 56,64, 39,30, 57,62, 60,42,
  15,48, 24,45, 23,49, 13,37, 59,35, 61, 3, 45, 8,  6,58,
  65, 4,  0,28, 34,53, 27,17, 21,67, 60,47, 34,29, 40, 3,
  60, 6, 25, 3, 47,65, 35,47, 11,35, 49,46, 44,36, 50, 3,
  43,32,  0, 5, 53,15, 65,59, 67, 1, 34,51, 21, 8,  2, 7,
  49, 2, 44, 3, 63,57, 26,39, 33,34, 27,19, 21,27, 23,66,
  19,62, 54,65, 63,35, 39, 3, 15,41, 22,63,  1, 8,  9,12,
  61, 1, 57, 7, 68,68, 43,59, 65, 2, 10,20, 64,30, 35,53,
  22,41, 55,56, 11,50, 67,27, 19,26, 18,36, 66,39, 25,41,
  34,56, 25,46, 15,12, 44,58, 55,34, 29,63, 23, 1,  0,42,
  17,59, 13, 3, 22,17, 28,41, 55,67,  3,20, 11,45, 19,24,
  67,66,  3,38,  7, 1, 17, 9, 43,58, 60,69, 51,49, 32,45,
  58,14, 54,21,  0,45, 67,48, 49, 4, 29, 1, 24,40, 22,56,
  24,63, 37,63, 64,31, 48, 9, 15, 7, 15,69, 59,40, 17,56,
  50,34,  2,15,  5,51, 26, 5, 47,22, 61,52, 13,42, 63,48,
  34,34, 34,46, 59, 7, 60, 9, 62,34, 54,36, 32, 6,  8,60,
  16, 2, 25,57, 68,28, 61, 8, 57, 5, 67,55, 10, 4, 23,67,
  43,49, 16,45, 65,21,  3, 9, 39,46, 25, 4,  8,11, 16,34,
   4, 4, 23,54, 44,60, 53,33, 11,25, 56,50,  1,36, 48,11,
  43,28, 23,61, 25,66, 25,69, 56,19,  2,63, 61,50, 43,13,
  60,56, 38,44, 54,18, 37,36, 50, 1, 31,59,  3,48,  5,44,
   3,46, 21,31, 25,59, 53,31, 58,67, 49,45, 30,60, 28,63,
  35,51, 21,40, 62,66, 33,30, 65,22, 24,27,  7,51, 61,22,
  54,61, 16, 5, 51,22, 60,27, 52,10,  7, 3, 31,68,  8,18,
  65,51, 54,62, 52,29, 58,12, 52,13, 47,69, 51,43, 42,35,
  22,19, 22,36, 68, 1, 21,35, 55, 5, 41,69,  2,36, 58,35,
  46,22, 16,50, 54,68, 10,63, 53,23, 31,60,  1,52, 63,41,
  66,23, 16,29, 33,51, 68,64, 14, 5, 56,60, 53, 9,  4,64,
  16,61, 62, 0, 64,16, 21,36, 68,45, 15, 5, 63,25, 62,18,
  36, 0, 33,14, 33,31, 69, 1,  2, 4,  7,22, 51, 2, 31,26,
   8,37, 58,19, 69,58, 32,65, 61,33, 24,35, 40,54, 60, 2,
  54,56, 51,23, 50,40, 48,48, 66,37,  3,31, 23, 9, 69, 4,
  40,51, 15,30, 36,29, 57,40, 27,25, 44,41,  2,21,  2,42,
  29,25, 12,28, 38,15, 11,28, 53,50, 66,63, 41, 6, 25,39,
  51, 6, 45,37, 46,56, 48,25, 15,64,  9,45, 15,18, 17,13,
  30,18, 13,34, 42,57, 40,34, 24,16, 37,29, 49,49,  3,44,
  57,24,  3,60, 21,49, 60,54, 67, 3, 38,24, 58,18,  0,65,
  13,48, 15, 3, 29, 4, 12,29, 42,67, 62,59, 63,62,  4,66,
  12, 8, 48,42, 68,31, 31, 5, 15,54,  5, 9, 67,64, 29, 6,
   8,64, 66,65, 25,53, 11,29, 59, 2, 13,67, 16,57, 26,47,
   9,44, 39,37, 45,56, 32,40, 21,60, 46,13, 16,14, 38,67,
  66,49, 33,45, 50,20, 19,68, 29,42, 36,24, 28, 6, 42, 2,
  22,59, 12,30, 38,46, 39,34, 34,48, 48,24,  2,47, 30,30,
  20,57, 47,13, 34,13, 15,66, 43,50, 69,48, 24, 1, 21,13,
  46,39, 29,47, 41,38, 14,45, 24, 8, 41,35, 36,18, 58,68,
  31,57, 35, 4, 26, 4, 24,65, 17, 3,  8,12, 48,45, 10,44,
  49,52, 10, 6, 36,52, 34,19, 57,29, 10,38, 65,66, 34,30,
  32,64, 30, 4, 28,64,  6,32, 45,57, 19,41,  5,19,  6,65,
  39,69, 50,29, 31,45, 27,40, 19, 6, 67, 4, 27, 6, 60,36,
  44,21, 31,31, 26,18, 17, 0, 40,38,  7,36, 44,44,  0,27,
  57,33, 21,34, 60,63, 37,20, 20,42, 48,37, 54,13, 55,13,
  34,54, 63,55, 49,64, 55,51, 11, 5, 11,21, 65,49, 12, 1,
  58,44, 57,65, 34,16, 19,47, 39,61, 40,57, 34,49, 68,22,
  68,24,  7,26, 54,54, 40,66,  1,45, 16,20, 27,65, 29,49,
  47,48, 54, 3, 40,26, 27,50, 60,51, 10,50,  2,62, 65,10,
  50,69, 44,13, 67,16, 56,28, 30,14, 44,30, 19,11,  3,62,
  51,44, 29,50, 31,52, 34, 0, 15,61, 64,41, 48,17, 40,15,
  46,19, 56, 3, 54, 4, 64, 1, 18, 7,  0,69, 18,68,  7,38,
   7,53, 44, 9, 15,45, 49,55, 58,69, 26,46, 26,44, 60,40,
  34, 7, 51,53, 39, 2,  5,25, 19,61, 18,31, 29,38, 36,23,
  17,45, 32, 9, 66,45, 19,29, 49,48, 45,19, 19,64,  6, 9,
  40,32, 63, 1,  9,52, 38,35, 46,15, 13,47, 68,12, 55, 8,
  59,11, 44,61, 13,20, 52,56, 14, 7, 57,11, 45,13, 27,67,
   7,45, 49, 8, 64,68, 47,18, 16,44, 30,61, 34,66, 13, 1,
  23,62, 37,19, 46,61, 19,53, 60,58, 30,57, 23, 4, 20,11,
   3,56, 12,23, 42,24, 51,33, 61,56,  8,36,  5, 4, 51,14,
  11,46, 36,47, 34,14, 19, 0, 58, 4, 65,13, 62,20, 32,56,
  69,62, 59,58, 65, 6, 12,10, 27,37, 21,52,  1,61, 68, 6,
  60,39, 26,33, 22,58, 57,56,  8,50, 49, 1, 45, 1, 24,10,
  30,16, 38,39, 27,56,  9,51, 32,22,  6,69, 54,40, 40,59,
  34,28, 29,33, 11, 1, 62,10, 54,55, 17,53, 33,66, 64,50,
  44,39, 13,14, 63,10, 23, 8, 34,33, 45,58, 23,32,  3,67,
   7,69, 62,23, 46,26, 61,53, 45,32, 11,10, 62, 6, 52,49,
  55,41, 45,33, 65,48, 19,34,  4,54, 38,11, 13, 6, 54,28,
  51,16, 64,15, 56,39, 52,14, 47,59, 31,24, 16, 7,  9,34,
  66,55, 25,21, 38,29,  6,30, 42, 6, 28,61, 37,40,  9,31,
  45,38, 30,56, 35,67, 57,34, 30,53,  9,26,  0,38, 33,56,
  46, 7, 41,53, 54,38, 52,33, 30,20,  1, 1, 37,30, 30,26,
  28,35,  2,38,  1,13, 65,45, 24,19, 54,26, 32,39, 67,17,
  34,23, 53, 5, 49,57, 61,14, 43,25,  0,11, 34, 3, 63,32,
  69,11, 25,15, 13,17,  2, 9, 58,28, 43, 9, 35,25, 10,37,
  44,12, 34,15, 14,39, 55,36,  8,39, 13,29, 16,53,  4,49,
  63,21, 38,52, 67,25,  2,44,  0, 0, 65,58, 29,35, 46,36,
  10,31, 13,68, 66,18, 53,63, 52,41, 56,67, 46,30, 68,52,
  39,56,  2,35, 66,61, 13,25, 40, 8, 30,69
};
const int p_70_11[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  597,
  /// Array of preferences
  23,12, 17,45,  5,56, 36,35, 68,60, 17,47, 49,46, 54,16,
  41, 7, 27, 8, 14, 3, 11,10,  7,68, 31,44, 31,27, 44,66,
  69,37, 23,36, 27, 6, 12,62,  3,67, 60,46, 65,56, 58,43,
  50,39, 46,43, 50,63,  4,48, 20,61, 56,21, 50,41,  2,52,
   1,10, 63,53, 37,30, 40,58, 64,69,  6,27, 37,44, 66,11,
  25, 5, 36, 0, 10,14, 18,27,  7, 7, 32,65, 56,61, 61,32,
  52,36, 24,15, 15, 4, 46,49, 67,28, 55,17, 12,46, 27,51,
  59,26, 40,48, 55,18, 36,61, 51, 8, 51,58, 42,10, 30,64,
  55,66, 17,27, 31, 3, 44,14, 34, 7, 40,65, 49,48,  5,54,
  40,28, 65, 7, 10,26, 27,35,  5,60,  1,49, 46, 6, 53,64,
  25,60, 65,39, 41,66, 60,10, 22, 4, 54,56, 15,12, 12,56,
  66,54, 51, 6, 16,66,  8, 4,  8,32,  6,11, 29,60, 69,57,
  52,50, 51,51, 59,47, 11,51,  9, 8, 15,45, 10,30, 37,20,
  26,48, 26,49, 13,23,  4, 7, 34,58, 23,30, 49,15, 22,43,
  46,55, 18,61,  2,32,  1,25, 32,40, 28,15,  4,60, 56,23,
  19, 5, 48, 1, 42,22, 31,28,  9,23, 32,61, 19,16,  2,23,
  45,51,  0,26, 28,56, 59, 4, 14,17, 41,64, 61, 3, 20,63,
  59,49, 52,62, 35,13, 43,26, 33,49, 54,31, 46,17, 19,55,
  44,62, 51,36, 58,30, 68,22, 12,55, 18,11, 25,22, 68, 6,
  62, 5, 28,61, 58,53, 35,36, 14, 7, 14,46, 38,24, 21,58,
  15,30, 42,66, 69,32, 28,27, 22,69, 11,40, 55,38, 61,31,
  43,55, 28,25, 28, 8, 25,41,  7,15, 12,39, 35,22, 19,45,
  20,56, 62,55, 55,43,  2,19,  5,55, 67,64, 58,64, 45,47,
  37,27, 36,53,  0,67, 32,30, 25,61, 65, 2,  3,39, 65,24,
  42,37, 15,62, 45, 9, 56,14, 16,49, 43, 8, 40, 1, 31, 0,
  11,65, 45,64, 11,54, 23,49, 10,34,  9,47, 41,23, 60,49,
  69,40, 48, 8, 26,60, 52, 1, 69,60,  8,42, 63,38, 29,17,
  58,42, 49,68,  3,22, 19, 4, 66,24, 29,67, 16,34, 49,17,
  61,42,  1, 4, 29,69, 16, 3, 55, 2, 34,31, 59,66, 12,35,
  20,17, 69,53,  9,51, 37, 2, 29,35, 21,66, 12,40, 32,34,
  62,66, 14,15,  4,38, 49,18, 57,48, 47, 1, 35,65,  2,29,
  55,16, 40,32, 31, 7, 46,44, 18,37,  5,37, 45,48, 43,29,
  39,11, 47,41, 56,35, 59,12, 66, 8, 55,69, 63,52,  2,28,
   2,49, 17,44, 27,13, 20, 4, 44,26,  8,66, 37,62, 37,43,
  38,47, 24,49,  9, 6, 23,65, 59,46, 66,50, 35, 7,  8,56,
   5, 7, 53,18, 64,39, 48,37, 29,44, 12, 3, 16,65, 17,48,
  13,22, 32,62, 68,34, 62,69, 28,33,  3,42, 33,34, 45,65,
  40,62, 69,17,  3,15, 64,19, 63,67, 17,40, 37, 4, 46,59,
  45,50, 60,43, 21,68,  0,54,  0, 2,  1, 5, 48,13, 33,19,
  26, 0, 27,27, 64,59, 27,47, 27,52, 56,57, 30,31, 36,59,
  43,52,  9,20, 38, 6, 37,67, 21, 9,  1,63, 24,42, 65,15,
  61,64, 31,65, 56,17, 44,22, 22,22, 12,16,  2, 9, 51,53,
   1,56, 23,15,  7,65, 52,63, 53,32, 63,50, 10,44, 69,11,
  19, 2, 26,56, 44,34, 14,43, 43,14, 12,69, 36, 9, 54,42,
   5,23, 31,56,  4,42, 43,53, 52,33, 36,69,  3,34, 44,36,
   1,22, 64, 8,  1,40, 34,44, 25, 8, 55,57, 41,26, 64,27,
  60, 5,  9,24, 65,60, 59,60,  2,39, 13,15, 56,39, 41, 3,
   6,25, 22,11, 17,33, 15,52, 36, 6, 68,19, 46,30, 25,25,
   7,51, 18,64, 53,34, 58,25, 18,28, 62,56, 11,57, 10,67,
  28,31,  8,62, 24,65, 43,21, 11,31, 69,33,  1,16, 59, 7,
  47,48, 16,50, 50,44, 61,61, 68,58, 68,18, 61,49,  2,60,
  14,52, 33,39, 11,21, 36,64, 57,46, 65,67, 19,64, 42,47,
  20, 3, 39,18, 50,34, 61, 7, 68, 3, 11,52, 22, 1, 33, 5,
  14,39, 19,12,  6,48, 15,19, 69, 1, 39,55, 29, 6, 41,43,
  30, 4, 39,29, 39,32, 44,41, 11,68, 46, 7, 41,53, 60,32,
  63,28, 28,40, 32,51, 55,67, 20,37, 60,47, 14,50, 47,40,
  66,49, 44,58,  8,41, 10,32, 51,66,  1,55, 36,32, 28, 1,
  25,30, 22,42, 13,32,  4,14, 40,22, 11,53, 45,27, 20,18,
  44,55, 44,25, 32,66, 23,24, 16, 9, 37, 3, 31,69, 40,53,
  18, 7, 53, 8, 11,60, 65,21, 60,56, 31,33, 65,51, 68,28,
  54,15, 45,13, 32, 0, 55,63, 38,42, 47,22, 58,38, 68,51,
  45,69, 33,26, 61,36, 63,16, 65,18, 38,31, 19,28, 36,41,
  38,36, 20,69, 34,33, 28,22, 10,50, 24,18, 13,35,  4,57,
  31,57,  6,17,  6,62, 59,32,  5,27, 18,12, 47,18,  5,14,
  18,25, 43, 9, 54,19,  8, 0, 43,36, 52,35, 13,34, 39,38,
  12,19, 16,14, 51,67, 14,62, 59,54, 63,40, 31,36, 27,61,
  18,66, 57,56, 32,38, 40,37, 55,51, 58,34, 60,37, 40,18,
  33,42, 63,58, 54,49, 28,41, 38,57,  1,51, 61,23,  6,35,
  61,25, 60,40,  0, 7, 56,10, 64,67,  2, 7, 26,67, 66,36,
   9,63, 52, 0, 31, 4,  6,40, 67,50, 31,64, 53,15, 26,15,
  58,41, 57,12,  0,50, 39,41,  8,69, 34,61, 10, 2, 16, 6,
  40,41,  7,12, 35,46, 66,16, 67,43, 34,49, 10,66, 12, 2,
  44, 3,  8, 1, 16,41, 12,22, 60,60, 58,22, 47,46, 57,54,
  20,42, 61,58, 18,38, 40,42, 64,30, 29,36, 61, 2, 53,56,
  30,37, 27,36, 25,27, 16,33, 26,61
};
const int p_70_12[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  388,
  /// Array of preferences
   5, 5, 65,43, 67,64, 19,50, 20,50, 52,39, 53, 2, 27,16,
  49,44, 46,37, 50, 1, 27,40, 69,35, 32,21, 64,50, 10,31,
  13,66, 53,69,  8,18, 16,14,  7,14, 37,11, 68,35, 32,39,
   4,59, 18,59, 59,68, 25,37, 49,18, 63, 4, 41,46, 43,20,
  44,56, 47,62, 38,44, 40,16, 15,33, 56,64, 17, 3, 64, 0,
  60,24, 53,53, 22,44, 69,18, 28,10, 69,46,  6,11,  7,65,
  26,18, 47, 4, 19,34, 25,52, 55,16, 30,60,  7,38, 19,40,
  10,41, 54, 7, 58,38,  5, 9, 27, 1, 12,10,  5,62, 50,16,
  29,69, 23,34, 51,66, 21,10, 55,57,  5,24, 18,58, 39,25,
  16,17, 24, 7, 14,11, 50,69,  7,55, 34,21, 66,44, 51,23,
  19,30, 30,23, 47,39,  2,57, 52,12,  0,67, 60,10, 46, 6,
  13,35, 23,46, 44,38, 39, 7, 22,54,  9, 0, 11,11, 56,12,
  56, 3, 23,50, 10, 2, 31,30, 47,29, 38,55, 67,54, 43,24,
  52,64,  2, 0, 58, 8, 63,35, 32,44, 20,58, 50,39, 58,29,
   5, 7, 64,26, 47,67,  8,49, 32,54, 59,49, 47,23,  6,68,
  49,37,  7,22, 51,60, 49,25, 66,34, 26,22,  4,46,  0,26,
  54,40, 15,25, 27,34, 33,59, 15,46, 22,67, 38,21, 15,55,
  64,28,  5,65, 12,55, 36,29, 68,14, 54,46,  8,16, 17, 9,
  44,46,  1,65, 63,19, 55,69,  6,30, 24, 4, 43,58, 12,26,
  32,65,  6,10, 68, 4, 62,23, 55,65, 27,44, 58, 0, 32, 1,
  34,41, 58,65, 10,13, 18,29,  5,30, 22,39, 52,31, 22,34,
  53, 0, 17,25, 45,49,  4,65,  0,36, 50, 9, 29,53, 67,42,
  23,26,  7,46, 41,58, 19, 6, 14,35, 43,42, 10, 8, 34,19,
  49,43,  8,63, 26,64, 12,45, 64,18, 36,66, 44,16,  5,29,
  34, 4, 50,32, 64,51, 55,15, 63,23, 26,17,  5,27,  4,42,
  38,49, 40, 4, 52,54, 68,33, 38,53, 45, 1, 41,33, 18,67,
  36,14, 13,37, 63,55, 22,11, 30,53, 31, 6, 22, 6, 10,46,
  32,36,  2,37,  8,59, 40,41, 58,60, 22,22, 10,14,  6,46,
   9,26, 51, 5, 19,38, 62,56, 39,51, 44, 4, 54,43, 13,50,
  16,50, 16,51, 46,25, 64,38, 65,17,  7,45, 13, 3, 32, 3,
   9,50, 35,66, 52,52, 56,50, 50, 5, 68,17, 25,41, 31,53,
  65,62, 60,59,  1,52, 41,28, 14,67, 29,32, 49,16, 68,11,
  47,32, 57,10, 46, 0, 58,10, 49, 7, 65,41, 39,64,  0,10,
  17,64, 24,18, 65,52, 27, 3, 69,50, 11,38,  2,29, 11,14,
   5, 2, 22,18, 49,58, 51,37,  1,18, 62,48,  8,47,  6,25,
   0,57, 60,56, 59, 6, 44,37, 58,39, 44, 1, 66,63, 39,61,
  17,41, 51,16, 35,12, 37,37, 45,64, 15,59, 61, 3, 60,49,
  19, 1,  2,39, 38,46, 16,46, 49, 1, 18,51, 48, 5, 41, 9,
  22,57,  2,54, 41,56, 21,67,  7,33, 65,34, 59, 1, 52,17,
  60,65, 11,32, 17,24,  7,20,  7,62, 61, 5, 57,34, 63,46,
   2,44, 57, 8, 28,35, 46,63, 33,44, 16,66, 32,24, 25,25,
  66,11, 67,29, 41,64, 29, 0, 24,67,  0, 2, 67,46, 14,34,
  40,19, 57,57, 18,19,  5, 0, 26,19, 27,21, 20,22, 24,16,
  34,59, 17,26, 38, 7, 48,22, 15,48, 33,15, 19, 4,  4,10,
  15,47, 28,11, 33,35,  4,22, 57, 4, 34, 8,  9,55, 65, 5,
  59, 8, 60,14, 65,28, 29,36, 37,35, 59,30, 51, 8, 50,61,
   8,31, 16, 0, 51,49,  5,13,  0,13,  6,62, 22,23, 39,41,
  28,20,  8,67,  7,16, 59,42, 64,53, 45,54, 11,46, 55,27,
  69,32, 53,41, 47,43, 48, 9
};
const int p_70_13[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  703,
  /// Array of preferences
  66,33, 57,11, 33, 1,  2,33, 36,29, 12,65, 25, 3, 59,65,
  35,27, 41,47, 49,48, 26,19,  0,33, 51,65, 18, 5, 44,19,
  67,65,  7,26, 66,21,  7,35, 19,31, 16,38, 41,49, 52,39,
   5,28, 26,67, 40,64, 18,49, 15, 8,  7,61, 26, 5, 14,31,
   1, 4, 51,24, 11,32,  0,50,  5,23, 28,69, 54,53, 33,55,
  13,67, 39,32,  7,14, 41,36, 36,49,  5,68, 24,10, 17,69,
  15,34, 63,56, 33,63, 38,66, 15,16,  8,64, 31,27, 31,50,
  47,42,  3,26, 38,50, 45,24, 27, 5, 54,35, 18,38, 34,43,
  57,25, 28, 7, 42, 9, 13,66, 55,51, 64,32, 51,64, 51,16,
  20,50, 40,27, 57, 2, 43,67, 39,30, 55,46, 35,59, 11,18,
  61,51, 30,49, 52, 8, 25,43, 55, 7, 32,15, 32,23,  8, 1,
  33,65, 42,20, 48,58, 36,67, 20,61, 69,38, 60,26, 27,18,
  38,32, 38, 5, 22, 8,  0, 5, 47,28, 46,41, 55,10, 32,69,
  39,13, 41,64, 24,14, 39,16,  5,60, 41,59, 13,62, 44,15,
  21,51, 27,33, 61,30, 12,55, 20,26, 56,24, 58,27, 49, 1,
  44,28, 41,48, 62,62, 21,55, 17,16, 25, 4, 66,13, 49, 7,
  29,54, 33,53, 48,30, 63,61, 68,59, 30,64, 28, 5, 15,14,
  54,32, 53, 7,  6,15, 46,59, 57,14, 27, 3, 63,68, 54,15,
  46,37,  5,17, 15,54, 17, 7, 23, 7, 63,36, 25,11, 25,12,
  41,54,  4,35, 21,44, 39,69, 14,68, 37,40, 35,57, 31,43,
  35, 4, 62, 2, 65, 0, 25,51, 68,66, 10,21, 42,32, 68,58,
   4,18, 26,36, 56, 3, 48,68, 10,66, 40,23, 45,43,  2, 6,
  39,63, 26,45, 15,56, 56,28, 23,37, 20, 5, 63,32, 54,37,
  69,43,  8,26,  8,69, 41,62, 41,39,  1,47, 64,31, 67,67,
  10, 7, 19,59, 20,31, 65,51, 35,37, 49,13,  9,23, 59,10,
  50, 4,  4,28, 53, 9, 31,16, 19, 0, 31,46, 29, 9, 60,10,
   2,44,  7,15, 52,27, 34, 1, 51,44,  5,56, 23,39, 35, 2,
  41,57, 15,31, 44,26, 47,27, 25,58, 29,46, 25,40, 24,17,
  40,61, 62,32, 54,45, 62,36,  8,19, 22,49, 14,33, 12,22,
   0, 3, 23,19,  0,52, 33,59, 43,58, 19,63,  7,38, 42, 1,
  40,54, 49, 2, 34,36, 41,56,  3,39, 33,21,  6,43, 45,12,
  67, 0, 37,54, 63, 4, 16,52, 16,49, 57,65, 63,23,  9,25,
  11,64, 64, 5,  2,52, 44,53, 48,61, 34,66,  6, 0, 31,15,
  35,13, 43,29, 49,41, 66,51, 66,50, 18,61, 63,60, 69,40,
  14,28, 17,11, 27,14, 46,52, 25, 1, 29,40, 45,16, 41,29,
  27, 6, 38,30, 25,26, 25,14, 45,22, 39, 4, 46,45, 46,29,
  59, 7, 39,65,  8, 0, 37,22, 31,29, 65, 3, 34,68, 29,17,
   6,30, 23,42, 36,39,  2,25, 27,32, 50,62, 66,59,  1, 5,
  38, 9, 17,41, 47,16, 48, 6, 67,18, 31,48, 18, 2, 62,63,
  26,26, 24,39, 52,37, 22, 2, 41, 3, 44,31,  4,17, 24,65,
  59, 0, 54,42, 40, 3, 42, 5, 54,52, 39,12, 32,54, 39, 1,
  14,52, 59,35, 22, 5,  9,36, 27,46, 20,45, 46,38, 46, 6,
  37,49, 50,65, 52,21, 53,54, 36,42, 58,13, 26,55, 43, 4,
  38, 7, 41,43, 63,18, 58,44, 20,11, 13,36,  6,35, 61,55,
  37,62, 31, 7,  3,37, 36,20, 21,12, 36, 2, 53,11,  9,54,
  11,29, 36,46, 17, 5, 48, 3, 25,52, 30,60, 36,50, 11,10,
  39, 9, 64,37,  0,42, 40, 1, 55,56, 18,12,  3,41, 50,33,
  34,65, 27, 7, 49,67, 30,44,  2, 1, 31,23, 30, 7, 53,42,
  17,34, 14,47, 64,15, 16,55, 10,47, 19,27, 25,60, 24,66,
  58,20, 66,55,  9,20, 39,45,  2,55, 43,36,  6,28, 12,57,
  30,40, 68,33, 16,12, 66, 0, 15,43,  5,50, 26,50, 64,19,
  24,32, 18,45, 61,47, 45,18, 69,36, 20, 8,  2,43, 13,55,
  19,69, 52,13, 38,33,  1,52, 32,52, 69,10, 55,24, 24, 0,
  59,69, 57, 5,  5,25, 21,34, 44,32, 28,12,  5,14, 42,34,
  35,20, 16,15, 32,63, 55,28, 61,52,  9,15, 25,21, 18,34,
   9,12, 38,56, 59, 1, 24,26,  6,29, 38,51, 17,30, 16, 4,
  46,56, 54,68, 50,20,  3,23,  8, 3, 45,20, 43,38, 53,10,
  20, 7, 66, 3, 55,35, 50,11, 29,12, 21,20, 49,66,  5,59,
   8,58, 36, 5, 19,19, 66,61, 15,33,  6,24, 18,20, 35,36,
  63,33, 59,16, 10,40,  6,36, 65,60, 13, 6, 67,40, 35,48,
  25,17, 46,30, 41,23,  0,68, 54,11, 23, 0, 41,14,  0,60,
  41,58, 20,25, 43,68, 37, 0, 34, 5, 26,43, 38,21, 47,30,
  44,40, 39,55, 44,42, 35, 0, 37,58, 13, 7, 22,54, 35,66,
  31,37, 64,54, 10,20, 41, 6, 35,17, 66,54, 29, 4, 54, 1,
  66,57, 54, 7, 26,25, 61, 5, 25,57, 56,13, 39,66, 13,30,
   9,26, 15,26, 16, 2, 21, 9, 29,53, 21, 5, 28,13,  5,21,
  14,69, 30,41, 14,22, 24,47, 46,65, 69, 3, 68,61, 26,64,
  45,10,  6, 7, 17,22, 48,41, 20,56, 12,54, 26,52, 14,62,
  22,43, 45,46, 18,53, 23, 3, 16,17, 38,29, 57,24, 41,16,
  22,62, 10,26, 35,33, 29, 2, 59,31, 65,42, 67, 6, 54,60,
  46,14, 61, 4, 19,43, 18,65, 51,12, 40,48, 29,18,  8,32,
   6,60, 45,38, 36,33, 48, 9, 38,17,  3,62, 51,23, 42,27,
  32,64, 63,11, 69,26, 34,20, 26,68, 14,36, 44,30, 41,37,
   8,61, 31, 2,  6,17, 56, 6,  0,45, 64,62, 45,35,  7,60,
   5,69, 22,45, 60,14, 31,35, 69, 0, 29,10, 12,35, 61,53,
   8,16, 13,42, 51,68, 59,28, 12,56,  9,58, 56,31, 54,41,
   1, 8,  7,12,  2,65, 48,43, 16,61, 30,48, 28,46,  6,57,
  22,25, 37,68, 30,19, 68, 0, 54,61, 41,67, 37,20, 38,57,
  26,15,  4,40,  1,25, 49, 9, 61, 7,  3,24,  7,23,  2,38,
  27,39, 49,20, 49, 0, 68,21, 34,50, 26,46, 51, 1, 60,29,
  43,57, 45,31, 18,21, 53,30, 61,15, 69,19, 54,48, 58,65,
  63,63, 62,26, 52,50, 10,43, 41,17, 59,39, 56,60, 50,45,
  63,39, 38,46, 41,46, 33,19, 32,35, 17,40, 62,68, 23,17,
  56,34, 26,57, 28,38, 21,10, 39,20, 46,57, 36,17,  9,13,
  48,11, 40,37, 34,62, 13,58, 13, 4, 52,47, 23,16,  6,65,
   5,13, 65,44, 65, 5, 54, 9, 53,57, 43, 8, 58,41, 48,36,
  22,56, 62,10, 54,62, 38,48, 42,60, 61,65, 35,69,  7,42,
  53,31, 56,14, 21,11, 41,65, 19,48, 26, 8, 28,51
};
const int p_70_14[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  1199,
  /// Array of preferences
  69,43,  8,39,  7, 7, 48,56, 30,15, 61, 8, 10,62, 25,40,
  60,63, 16,48, 21,57, 38,29, 59,62, 13,47, 36,38, 62,56,
  37,67,  7, 6,  5,58, 36,36, 28,42, 29,30, 11,13,  7,54,
  56,13,  2,16, 69, 5, 38, 3, 56,65, 22,15,  4,37, 44,29,
  63,22, 11,18, 57,17, 25,14, 24, 9, 69,25, 35,22, 27, 5,
  56,42, 59,38, 35,67, 24,47, 18,34, 41,27, 42,59, 37,62,
  58,51, 50, 4, 23,60, 12,57, 32,24, 16,44, 52,38,  4,35,
  32,54,  4,50, 29, 8, 13,17, 19,21, 11,36, 35,36, 15,64,
  53,10, 42,33, 46,43, 24,24, 14,28, 38,10, 69,39, 46,68,
  50,59, 27,29, 58,64,  1,15, 65,45, 32,13,  2,69, 14,24,
  39,25, 50,24, 52, 7, 56,41,  7,19, 57,66, 16, 3, 27,24,
  20,51, 15,16, 13,57, 55,58, 66,15, 13, 6, 67,10, 14,46,
  57,26, 54,28, 33,42, 64,37,  1,69, 56,19, 37,45, 45,45,
  48,65, 53,58,  3,12, 59,14, 67,24, 32,63, 12,10, 47,20,
   2,47, 69,44, 31,68, 24,12, 16, 2, 10,22, 13,38, 64,36,
  31,17,  6,35, 29, 4, 34,10, 50,57, 52, 0, 63,57, 61,17,
  32, 1, 10,42, 45,35, 48,46, 14,47, 18,40, 34, 0, 66,20,
  55,49, 38,27,  2,50,  6,55, 24,68, 30, 1,  1,62, 15,49,
  28,19, 22,14, 40,47, 66,40, 66,49, 29,16,  0,15, 64,31,
   5,65, 47,16, 17,22, 17,33, 59,15, 48,43, 24,52, 11,14,
  26,10, 42,44, 54,67, 43,39,  6, 6, 55,64,  8,66, 12,17,
  47,58, 22,28, 31,31, 50,12, 52, 1, 22,34, 30,58,  2,27,
  59,54, 47,44, 69,61, 66, 8, 56,55, 22,67, 37, 7, 52,51,
  68,33, 12, 0, 30,41, 69, 4, 33,56, 25,53, 22,42, 37,24,
  43,41, 53,67, 15,25, 68, 0, 20,11, 41,13, 34,55, 60,28,
  39,64, 52,34, 53,68, 45, 3,  5,67, 15,20, 20,24, 38,24,
  50,28, 12,35, 24,55, 15,66, 47, 7, 67,49, 13,48, 12,16,
  68,56, 65,59,  7,56, 62,52, 20,23, 56,31, 69, 1, 12,45,
  21,38, 69,51, 58,62, 15, 7, 36, 1, 34,48, 24,14, 42,68,
  61,66, 45, 8, 59,39, 30, 4, 20,66, 25,55, 10,40,  1,42,
  54,66, 57, 5, 44,58, 10,37, 23,53,  1, 7, 43, 0, 18,24,
  62,10, 67,57, 60, 8, 29,62, 36,56,  2,36, 53,40,  0,35,
  61, 6, 69,66, 20,39, 49,64, 56,17, 46,10, 17,19, 60,69,
  43,56, 12,42, 42, 4,  9,15, 46,12, 14,16, 59,48, 57,27,
  36,29, 69,10, 37,29, 25,25, 16,34, 54,58, 34,42, 23,41,
  37,37,  8,21,  8,68, 32,14, 16,31, 24,13, 15,24, 25,23,
  46,67, 22,18, 63, 4, 13,69, 25, 5,  2,52, 29,18, 58,65,
  31,33, 36,43, 57, 3, 25,37, 11,68, 44,28, 30,35, 31,16,
  43,66, 60,31,  7,66, 38,43, 35, 8, 46,50,  1,50, 13,23,
   6, 8, 28,58, 61,49, 43,26, 63,33, 18, 7, 28,38, 44,52,
  24,26, 31,15, 36,46, 28,64, 43,11, 26,27, 50,15, 53, 3,
  44,56, 26, 3, 65,61,  9,42, 53,53, 22, 0, 25,41,  6, 2,
  25,64,  7, 4, 39, 2, 15,42,  2, 2,  9,54,  2,51, 52,19,
  34, 6, 30,21, 28,21, 44,46, 34,68, 30,61, 68, 4, 16,38,
  36,32, 68,21, 29,55, 16,41, 49, 0, 11,19, 38,60, 44, 5,
  63,28, 42,16, 14,67, 66, 3, 34, 8, 32,36, 43,55, 50,43,
  32,22, 38,32, 30,53, 42,49, 39,30, 41,19, 14,59,  2,38,
   1,41, 47,22, 25, 6, 12,44, 35,38, 52,10, 57,55,  9,56,
  49, 9, 64,42, 38,11, 24, 7, 68,26, 16, 0,  5,27, 52,55,
  12,34, 25,31, 32,65,  0,12, 47,43, 35,12, 16,20, 55, 5,
  11,30, 62, 4, 43, 6, 45,19,  2,30, 21,49,  9,65, 25,59,
  42,36, 64,29, 18,18, 53,25, 61,38, 62, 0, 26,54, 69,45,
   8,58, 27,65,  5, 3, 69,35,  3, 5, 53,18, 51,21, 62,36,
  53,30, 48,23,  1,28, 43,40, 24,17, 23,59, 60,19, 49,20,
  33,54, 26, 7, 20,37, 20, 4, 57,67, 43,22, 13,27, 40,26,
  35,57,  8,22, 43,43, 56,23, 27,55, 43,57, 27,62, 62, 1,
  36, 0, 13,37, 22,48,  6,47, 20,36, 57, 6, 17,50, 29,26,
  38, 9, 30,17, 49,55,  2,29, 35,42, 50,60,  0,41, 65,54,
  41, 6, 16,65, 42,17, 35, 4, 65, 4, 47,51, 38,58, 52,21,
  19,69, 46, 9, 38,16, 63,39, 60, 0, 41,34, 39, 6,  6,17,
  68,31,  7,58, 47,35, 49, 1, 63,21, 11,65, 48,49, 28,23,
  48,57, 26,14, 50,42,  5, 9, 32,10,  8,42, 60, 1, 33,40,
  29,66, 54,48,  5,25,  6,22, 62,57, 34,12, 33,47, 15,11,
  64,19, 39,31, 48,31, 62,55, 16,19, 22,41, 50,58, 43,58,
  63,31, 18,52, 61,22, 29,58, 66,28, 27,35, 51,32, 61,48,
  34,17, 69,54,  9,31, 53,22, 55,17, 57,37, 39,59, 61,57,
  28, 7, 46,14, 17,60,  7,40,  8,67,  5,12, 61,41, 14,11,
  20,50, 15,29, 37, 5, 36,52, 20,16, 20,20, 45,23,  4,64,
  40,17, 18,49, 31,37, 59,24, 37,43, 50,65, 57,64, 32,32,
  39,50, 46, 4,  2,44, 13,25, 49,30, 55, 9, 11, 9, 45,39,
   5,37, 44,62, 37,48, 44, 0, 35,53, 58,38, 21,44, 66,63,
   1,49, 41,57, 68,24, 31,28, 47,46, 62, 7, 13,68,  9, 8,
  32,27, 54, 6, 47,34, 53,24,  0,28, 32,18, 35,65, 54,33,
  23,11, 20,63, 64,16, 50,45, 51, 2, 40,57, 43,24,  8,44,
  63,55, 10,36, 49, 5, 64,55, 50, 1, 54,16,  4,18, 49,29,
  12,31, 22,12, 23,31, 12,43, 36, 4, 60,68, 42,56,  6, 9,
  42,29,  2,63, 24,46, 66,25, 52,36, 12,69, 25,24, 47,67,
   5, 7, 30, 6, 34,15, 35,69, 18,33, 35,27, 20,42, 33,27,
  64,18, 23, 0, 22,11, 11,41, 27,15, 42,40, 43,60, 11,10,
  12,63, 57,63,  0,53,  7,37, 35,17, 51,48, 16,40, 45,36,
  14, 0, 63,37,  2, 1, 53,32, 48,58, 69,14, 68,11,  3,65,
  64,41, 54,55, 52,64, 23,58,  0, 8, 16,11, 51,27, 68, 5,
  63,54, 13,53, 41,64,  7,48, 20, 0, 10,23, 10,63, 69,27,
  33,58, 50,54, 25,51, 49,57, 59,23, 29, 9, 13,31, 20,15,
   7,23, 26,64, 59,34, 61,34, 31, 1, 49,11, 54,57, 14,41,
  35,46, 47,29,  2,48, 68,49, 37,28,  2,21, 48,10,  5,64,
  52,56, 21,52, 39,53, 57,65, 38,45, 56,15, 21,48, 57,44,
  41, 2, 48,25, 36,57, 43,33, 56,14, 17,32, 65,25, 53,51,
  44,21, 18,51, 49,58, 11,62, 61, 0, 61,28, 57,25,  2,41,
  35,54, 61,24, 14,65, 55, 4, 41,29,  8, 0, 16, 5, 67, 2,
  39,54, 12, 2, 20,61, 34,13, 38,59, 45,48, 29,47, 59,20,
  56,26, 49,28, 65,69, 59, 0, 29,14,  9,30,  3,14, 55,53,
  59, 8, 25,10, 23,16, 36,23, 25,15, 17,28,  4,14, 62,40,
  48,18, 49,52, 54,60, 66,48, 58, 3, 31,56, 53,50, 58,32,
  57,38, 29,35, 56,59, 19,41, 20,27, 28,35, 53,60, 42,32,
   5,45, 30, 2, 14,25, 20,64, 45,11, 49,51, 63, 3, 20,44,
  18,29, 19,67,  8,53, 49,21, 22,36, 60, 7, 31,69,  0,11,
  61,44, 49,67, 48,15, 50,20, 15,39, 11,54, 67,23, 57, 9,
   4,61,  5,54, 11,17, 40,27, 31,60, 58,47, 68,25, 67,60,
  52, 2, 19,40, 64,11, 21,17, 60,15, 11, 0, 68,45, 20, 6,
  53,33, 64,56, 31,36,  6,49, 38,23, 25,18, 33, 4, 63,44,
  64,69, 18,55, 63,66, 17,54, 16, 8, 48,20, 12,54, 57,31,
  34,56, 43,53, 62,34, 37,13, 39, 7, 24,27, 38,61, 16,35,
  16,16, 46,49, 67, 5, 28, 5, 38,69, 19, 2, 25,69, 47, 5,
  42,30, 65,29, 61,54,  5,56,  4,51, 50,63, 31,32, 47,57,
  24,37, 66,44, 52,65, 43,50, 47,59, 46,16, 35,15, 36,44,
  23,48, 57,39, 26,31, 69,47, 61,11, 68,28, 66,59, 63,15,
  10,16, 47,31, 56,33, 14,18, 41,30, 68,15, 62,49, 52,69,
  43,18, 30,28, 67,43, 13,18, 69,36,  8,20, 51,29,  8,30,
  10,56, 13,44, 67,21, 61,21, 27,14, 45,38, 65,50, 39,16,
  39,56, 15, 1, 64,20, 55,40, 42,22, 21,33, 56, 8, 58,57,
  10,43, 27,56, 68,67, 61,62, 28,37, 50, 3,  0, 7, 51, 4,
  24,41, 21,51, 60,24, 56,56, 17,63, 64,63, 23, 7, 45,56,
  20, 2, 44,11, 31,11, 58,14, 50,34, 60,14, 65,13,  4,40,
  22,26, 33,28, 53,59, 60,65, 69,18, 46,17, 54,22, 10, 9,
  14,52, 34,40,  8,18, 68,54, 39,66, 37,57, 50,29, 21,55,
  28,26, 66,14, 52,46, 22,33,  5,42, 65,26, 56,34, 64,49,
  13,46,  6,69, 26,20, 24,28, 56, 4, 67,61, 61,33,  3,15,
  31,49, 43,36, 21,11,  7,14, 54,62, 29,69, 30, 0, 50,62,
  37, 4, 56,47, 34,45, 32,68, 48,13, 30,24, 12, 7, 15,28,
  34,14, 20,58, 36,16, 28,16, 22,50, 17, 0, 37,56, 33,29,
  55,39,  5, 2, 61,10, 13,65,  9,20, 31,40, 34,61, 42,26,
  34, 5,  4,57, 12,46, 18,50, 48,33, 64,15,  7,69, 58, 9,
   4,39, 20,25, 17,48, 19,63,  0,13, 64,24,  6,54,  2,55,
  23, 4,  0,32, 62,66, 26,32, 67,50, 31, 6, 22,17, 41,43,
  40,67, 29,54, 23,24,  0,17,  0,47, 20,26, 27,52, 63,11,
   5, 4, 49,59,  9,37, 48,32, 65,55,  1,26, 57,24, 13,20,
  36,19, 46, 6,  8,26,  1,13,  7, 1, 68,64, 63,48,  5,52,
  41,50, 47,32, 59,17,  5, 1, 47,23, 43,13, 33,22, 52,28,
  33,32, 27,34, 60,33, 31,54, 23,52, 16,29,  0,14, 61,35,
  57,69, 32,15, 25,43, 59,21, 62, 5, 14,26, 40,37, 23,14,
   7,21,  3, 2, 30,69,  9,34,  3,34, 53, 6, 56,67, 44,68,
  53,64,  5,14, 22,43, 29,20, 59,47, 20,18, 36,13, 28,17,
   8,57, 16, 4,  8, 8, 20, 8,  1,31, 40,43, 35, 6,  8,16,
  66, 2, 37,38, 69, 8,  7,30,  4,58,  7,25, 65,32, 48,66,
  64,28,  0,58, 65,12, 12,19, 34,51, 47, 0, 40,13, 60, 9,
  68,36, 18,41,  0,49,  3, 6, 26,24, 49,27, 58,21, 50,25,
   3, 1, 24,21, 33,57, 67,62, 17,12,  6,30, 34,26, 44,40,
   7,64, 47,39,  6,37, 50,36, 64, 5, 14,14, 30,10, 41,25,
  11,52, 11,55,  8,12, 44,33, 59,42, 65, 0,  4, 6, 63,47,
  58,56, 60,51, 38,62, 31,41,  2,43, 24,65, 12,29, 49,48,
  36,55,  2,46, 25,35, 34,64, 64, 4, 18,25, 45,67, 30,43,
   3, 9, 22,66,  5,35,  7,61, 25,17, 57,19, 40,35, 39, 0,
  17,30, 35,11, 20,34, 13,63,  8,61, 32,21, 68,69, 59,56,
  19, 1, 67,31, 63,30, 23, 3, 42,67,  8,25, 31,53,  1,40,
  51,37,  6,27, 56,10, 27,12, 65,41, 61,40, 58,18,  7, 9,
  35,14, 10,46, 10,31, 67,41, 67,20, 66,43, 60,55,  6,64,
   7,53, 39,11, 31, 7, 35, 5, 31,20, 35,26, 20,54, 39,40,
  12,38, 68,13, 57,61, 55,25, 37,53,  0,64, 44, 1, 26,60,
   9,22, 20,35, 42,28, 30,56, 62,42, 11,33, 52,40
};
const int p_70_15[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  17,
  /// Array of preferences
  26, 2, 40,13, 42, 9, 34,53, 66,16, 25,37, 46, 7, 49,48,
  45,13, 21,12, 35,44, 34,35, 39,27, 32, 5, 69,23, 14,29,
  66,28
};
const int p_70_16[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  789,
  /// Array of preferences
  42,57, 38, 5, 57,29, 42,44, 22,37, 67,41, 44, 1, 53,43,
  32,56, 50,13, 12,26, 53,15, 13,16, 33,57, 22,17, 58,20,
  46,26, 58,28, 24,18, 55,20, 53, 5, 43,46,  9, 4, 31,44,
   9,17, 12,65, 35,51, 37,18, 13,28, 37,14, 17,17, 13,20,
  31,61, 52,22, 66,69, 38,13, 60,30, 40,28, 30,50, 22,44,
   9,50, 69,66, 46,30, 51,29, 13,62, 41,44, 53,48, 53,40,
   3,60,  5,27, 62,37, 11,36, 10,68, 37,40, 41,17, 11,10,
  16,31,  5,45, 40,37, 20,20, 62,32,  3,16, 56, 5, 56,69,
   9,42, 39,42, 16,47, 29,17, 17,54, 35,36,  8,66, 16,32,
  41,47, 28,44, 12,42, 11,37, 33,55, 46,40,  9,66, 26,34,
  12,60, 49,21, 16,26, 20,24, 31,65, 32,68,  3, 8, 59,53,
  31,41, 14,16,  3,30, 30,60, 56,59, 31,54, 25,52, 24,21,
  33,41, 26,68, 33,40, 18,17, 49,32, 10,32, 55,51, 37,44,
   6,28,  2,59, 43, 9, 66,49, 40,38, 55, 6, 51, 4, 54,52,
   0,30, 16,30,  1, 7, 59,45, 24,40, 39,16, 23,21, 57,56,
  55,64, 34,63, 35,32,  8, 4, 59,36, 14,53, 27,18, 66,17,
  67,29, 44,69, 20,28, 63,28, 36,47, 25,29, 36,37, 12,61,
  61, 7,  0,26,  1,51, 40,24, 44,30, 11,68, 47,24, 49,17,
  29,40, 19,69, 55, 5, 43,60, 41, 3, 37,27, 54,66, 63,62,
  26,14,  1,61, 41,68, 50,37, 48,46, 19,30, 18,48,  5,52,
   9,11,  8,46, 10, 5, 14,64, 41,32, 28,68, 27,49, 48,15,
  40,32, 34,11, 58,46, 64,14, 10,51, 50,18, 21,61, 67,39,
  63,10, 45,49, 61,50, 17,45, 54, 1, 56,12, 36,60, 56, 4,
  46,28, 37,29,  8,50, 60,34,  7,32,  4,69, 32,46, 50,60,
  31,55, 21, 8, 28,21, 21,45, 19,11, 55,52,  8,69, 16, 0,
  23, 2,  9,39, 51,52, 31,62, 69,58, 10,12, 67,68, 28, 1,
  48,24, 52,30, 27, 7, 54,36, 17,26,  7,36, 68,45, 25,14,
  25,65, 43,48,  9,38, 27,19, 36,63, 68,25, 10,39, 24,43,
  51,22, 27,31, 39,63, 29,42, 52,60, 21,53, 16,28, 22,21,
  37,61, 12,14, 45,25, 31,14,  2,55, 64, 8,  2,69, 19,39,
  10,36, 21,20, 34,15,  8,68, 60,20, 38,69,  1,15,  7, 9,
  58,63, 42, 0, 58,43, 58,42, 39,52, 47,47, 11,33,  8,61,
  15,50, 63,16, 15,20, 21,13, 17,31, 51,65, 36,23, 43,69,
  63,50, 24,47, 26,26, 54,46, 60,49, 36,68, 33,19, 52,16,
  46,62, 12,69,  0,47, 68,66, 67,20, 12,13, 19,56, 54,27,
  63,19, 33,45, 16,59, 41,14, 66, 4, 59,18, 25,46, 46,64,
  19,64, 43,32, 43,36, 11,12, 26,67, 35,68, 13,52, 27,28,
  69,67, 63, 0, 34,10,  5,25,  3,54, 53,28, 63,63,  3,44,
   7,15, 47,59, 30,19,  5,13, 61,58,  4,21, 61,17, 27,22,
  39,34,  2,62, 20,17,  2, 4, 29,44, 13,37, 43,63,  9, 2,
  33,26, 19,40, 11,42, 35, 4, 62,57, 28, 3, 51,61, 10,34,
  16,16,  3,25, 39, 9, 53,64, 22,10, 22,40, 48,20, 24,64,
  19,37, 48,43, 65, 9, 44,27, 38,50,  1,58, 39,51, 58,68,
  10,69, 66,37, 31,64, 19,17, 48, 8, 27,59, 15,58, 65, 8,
  42,63, 10, 9, 24,10, 63,51, 46, 9, 54,17, 62,66, 63,69,
  65,50,  5, 5,  4,13, 21,23,  5,69, 43, 3, 63,52, 42,51,
  46,42, 48,40, 20,60, 26,60, 57, 0, 39, 4, 21,42, 14, 1,
  30,68, 32, 4, 69, 0, 16,22, 32,17, 30,48, 56,24, 30,35,
  64,40, 56,54, 29,19, 61,46, 33,25, 64,57, 29,37,  1,36,
   8,10, 28,51, 31,27, 50,50, 44,13, 55,26, 28,26, 45,13,
  20,32,  2,64, 39,46, 26,59, 59,11, 64,16, 52,66, 64,53,
  29,60, 12,66, 55,37, 40, 5, 16,56, 28,60, 25,11, 38, 3,
  44,44, 32,40, 21,55,  8,22, 14,18, 62,40, 21,51, 22,60,
  34,56, 65,45, 44,58, 14,56, 32, 5, 35,58, 17,67, 11,51,
   0,14, 43,25, 33,53, 13, 9,  7,61, 15,29,  1,46, 63,53,
  31,47, 16,39, 62,43, 31,28, 28,11, 65,31, 60,48,  7, 2,
  33,68, 15,24,  2,40, 66, 7, 40,21, 18,29,  1,68, 58, 3,
  65,49, 42,26, 13,40, 16,66, 65,12, 62,36, 66,44, 57,14,
   6,44,  7, 0, 66,35, 17,65, 55,66, 56, 6, 43,30, 65,34,
  45,36, 62,47,  9,69, 49,15, 40,53,  4,51, 27,24, 35,67,
  20,69,  7,27, 20,31, 18,28, 32, 8, 38,36, 18, 8,  6,26,
  43,66, 44,45, 15,33, 14,38, 39,54, 59,35, 53,55,  6,25,
  64, 4,  2,61, 23,43, 54,23, 56,42, 47,17, 41, 4, 52,27,
  61,60, 16, 8, 60, 0, 23,44, 28,25, 65,24,  0,35, 38,42,
  20,45, 31, 4, 23, 4, 12, 9,  9,67, 64, 3, 19,52,  9,22,
  18, 7,  0,49, 65,40, 61, 2, 48, 4, 11,27,  7,57, 32, 7,
  58, 1, 41,35, 64,45, 33, 5, 31,58, 24,28, 54,18, 69,57,
  40,22,  3,64, 68,53, 61,62, 21,43,  7,56, 35,66, 47,62,
  35,29, 56,64, 65, 4, 17, 9,  1,53, 22,32, 33, 0, 58, 5,
  50,69, 55, 4, 35,61, 20,53, 24,17, 23,42, 18,39, 68,33,
  23,68,  1,66, 36,13, 23,58, 62,17,  1,45,  8,48, 18, 5,
  47,65, 30, 8, 29,36,  9,46, 49,34, 31,13, 26,11,  6,57,
  19,29, 13,41, 13,39, 65,19, 55, 0,  4,53, 24,65, 51,53,
  26,41, 36,12,  2,34, 19,63, 41, 6,  4,44, 19,10, 44,29,
  13, 0, 47, 5, 21,41, 10,50, 53,51, 39,41, 13, 4, 39,15,
  40,20, 29,41,  2,19, 12,24, 46,68,  4,16, 60,31, 16,44,
  65,35, 12,39, 44,51, 34,58, 66,10, 14,42, 42,28, 65,69,
  37,30, 17,41, 29,22, 41, 0, 53,39, 36,46, 61,51, 48,16,
  47,31,  1,28, 60,11, 53,58, 17,62, 24,34, 34,27, 51,57,
   7, 8, 10,56, 18,27, 10, 4, 19,47, 24,31, 45,33, 26,56,
  42,25, 46,52, 68,19,  4,48,  9,28, 15,21, 57,67, 28,33,
   6,24, 52,44, 60, 8, 15,16, 30,58, 45,14, 39,37, 32, 1,
  43,52, 21,54, 12,38, 41,18, 19, 4, 44,59, 35,52, 41,40,
  51,49, 13,68, 16,63, 49,51, 42, 7, 58,58,  5,10,  5,47,
  44,65, 34,23, 66,52, 10,14, 36,66, 68,29, 66,24, 52,49,
  46,22, 64,64,  2, 9, 56, 1, 36,52, 40,54, 52,57, 58,64,
  63, 1, 10,31, 39,43, 52,45, 11,50, 20,11, 65,17, 23,11,
  15,49, 36,28, 50,28,  6,64, 38,66, 45,58, 50,45, 59,41,
  11,41, 20,46, 59,47, 47,16, 43,58, 51, 6,  6,30, 27,11,
  55,68, 15,27, 62,31, 13,18, 44,34, 26,10,  7,55,  9,21,
   2,58, 38,59,  8, 1, 24,61, 27,15, 57,17, 41,15, 36,35,
  36,48, 24,25, 49,44, 54,21, 23,20, 30,11,  5,66, 47,33,
   3,35, 57,37, 54,45, 35,57, 56, 3, 45,59, 35,40, 55, 2,
   1, 9, 19, 1, 20,52, 38,21, 27, 9, 17,36, 30,28,  4,22,
  34,24, 40,52, 26,25,  0, 1, 25,54, 62,63, 65,64, 20,65,
  27,38, 57,49, 62, 0, 11,64, 46, 0, 53, 4, 13, 7, 32,27,
  12,29, 60,19, 14,66, 43,64, 16,54, 63,15, 57,22, 63,40,
  32,67, 40,59, 36,22, 39,40, 37,62, 37,58,  5,33, 11, 9,
  21, 9, 10,59, 43,41, 56,44, 11,17
};
const int p_70_17[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  933,
  /// Array of preferences
  52,30, 67, 1, 26,42, 64,30,  5,62, 67,10, 33, 5, 59,25,
  33,62, 13,13, 69, 9,  0, 4, 17,16, 63,18, 17,18, 29, 4,
   0,62, 44,51, 62,23, 48,13, 56,25, 61,62, 27, 0,  6,32,
  44,38, 20,56, 67,25, 19,56, 13,23,  7,47, 29,36,  8, 8,
  65,12, 22,33, 36,62, 49,30, 12,52, 39,43, 38,11, 24, 6,
  10,24, 54,38, 27,50, 15,19, 34,25, 44,68, 64,18, 66,49,
  20,42, 34,37, 67,18,  4,53, 48,21, 31,23, 42,43, 11,26,
  16,22, 56,44, 19,22, 55,49, 60,49,  3,48, 27,42, 13, 7,
  61,25, 35,29, 43,21, 29,50, 18,62, 12,53,  6,52, 12,49,
  20,61, 22,18, 15, 1, 42,50, 13,21, 56,18, 13,41, 37,10,
  25,69, 25,41, 62,38,  5,14,  3, 4, 30,47, 49,68, 56,34,
  14,66, 54,27, 13, 3, 61,27, 26,24, 55,12, 48,52, 60,64,
   2,67, 31,68, 42,65, 55,68, 67,66, 37,39, 23,31, 56,51,
   6,46, 46, 3, 60,35, 19,69, 52,26, 64,39,  2,60, 54,15,
  19,27, 66,57, 34, 5, 36,30, 48,42, 39,50, 39,68, 46,64,
  32,14,  7, 8, 14,24,  3,66, 45,12, 10,43,  1,38,  7,12,
  33,69, 13,33, 33, 1, 57,16, 30,21, 58,31, 33, 4, 48,55,
  54,34, 40,60, 56,24, 66, 3, 62,17, 14, 8,  1,16, 51,56,
  57,34, 53,64,  1,23, 15,58, 40,32, 49, 5,  8,38, 41,21,
  51,65, 62,13, 36,44,  6,50, 20, 8, 60,21, 66,35, 11,52,
  25, 2, 40,55, 40,17, 66,36, 24, 3, 33,19, 53, 8, 13,34,
   5,55, 21,50, 69, 7, 60, 0, 67,17, 62,42, 37,66, 24,23,
  50,68, 66,56, 60,23, 24, 7, 26,28,  7,65, 12,26, 62,35,
  34,54, 38,28, 50,50, 67,55, 16,23, 58,39, 16, 6, 62,11,
  33,10,  4,59, 54,32, 25,42, 38,22, 19,33, 65,58, 51,64,
  69,68, 50,51, 33,56, 50,58, 45,24, 42,12, 61, 1, 61,21,
  29, 3, 64,60, 45,22, 56,37, 10,52, 23,60, 52,54, 10,63,
  36,53,  7,29, 53,21, 28,10, 65,21, 57,56,  9,31, 35, 7,
  39,38, 50,12, 26,57, 17, 9, 35,63, 14,48, 53,39, 55,14,
  11,14, 21,34,  2,41, 56,64,  5,17, 31,19, 48,19,  0,16,
  26,47, 16,34, 21,20, 12,40, 54, 9, 59, 1, 40, 1, 25, 7,
  32,65,  3,29, 64,57, 10,37, 69,64, 44, 0, 46,26,  2, 4,
  29,53,  2,18, 59,17, 11,38, 37,12,  0,67, 17,35, 14, 9,
  34,55, 27,15, 49,25, 53,20, 30,61, 29,41, 65,63, 54,43,
  15,30, 28, 5, 17, 1, 61, 7, 21,12, 52,39,  7,10,  3,11,
  63,33, 37,35, 17,54, 63,66,  5,51, 33,61, 22,39, 68,32,
  21,41, 39,35, 48,34, 43,69, 67, 4,  3, 3,  5,59,  9,28,
  45, 8,  6,34,  4,15, 66,48, 53,31, 21,16, 64,54, 53,67,
  18,12, 22,23, 11,28, 13,65, 43, 0, 30,39, 62,19, 59,49,
  44,66,  1,59, 20,24, 48,10, 25,14,  5,27,  2,16, 38,50,
  29,34, 30,62, 40,27, 65,17, 52, 7, 66,32, 29,64,  4,64,
   3,69, 23, 1, 44,64, 13,36,  1,66, 51,24,  9,54, 49,38,
  17,46, 22,51, 25,49,  7,57, 40,23,  2,49, 29,68,  3,37,
  32,11, 58,44, 66,33, 15,18,  8, 0, 12,69, 50,27, 26,62,
  40, 3,  2, 3, 46,29, 46,43, 50,64, 35,68, 34, 3, 55, 9,
  62,58, 19,31, 50,42, 16,15, 42,37, 40,30, 51,51, 16,44,
  42,17, 11,47,  8, 5, 23,26,  2,37, 67,28, 22,31, 15,66,
  28,64, 57,39, 47,24,  8,18, 24,68,  0,11, 36,46, 37,50,
  33, 6, 49, 0,  6,62, 29,42, 14,67, 25,27,  8,55, 32,44,
  26,38, 60,41, 67,23, 25, 9, 31, 9, 24,26, 34, 6, 38,69,
  31,64, 63,49, 32,43, 38,35,  0, 1, 13,20,  5, 0, 32,48,
  19,20, 59,45, 21,35, 69,67, 39,13, 57,68, 54,35,  4,11,
  47, 4, 16,54, 47,42, 20,64,  4,22, 47,65, 40,41, 40,14,
  35, 4, 48,45, 56,55, 63,58, 68,37, 41,68, 15,57, 32,63,
   6, 7, 41,50, 20,51, 33, 9,  0,14, 58,36,  0,59, 43,58,
  28,45,  7,28, 42,28, 18,29, 44,21, 57,53, 10,14, 59,21,
   9, 3, 22,46,  4, 4, 35,11, 18,55, 16, 5, 23,45, 53,25,
  49,58, 28,51, 29,63, 36,50, 55,52,  9,59,  1,24, 64,14,
  67,46, 56,30, 32,25, 11, 7, 25,33, 14,27,  1,50, 20,22,
   5,26, 38,21, 55,33, 69,40, 58,64, 40,52, 47,18, 39,25,
  55,19, 49,31, 33,30, 50, 2, 10,58, 58,37, 33,52, 62,14,
  41,40, 35,16, 50,33, 43,11, 50,14, 35,38, 35, 3, 62, 2,
   6,22, 69,37, 17,56, 64,23, 53, 0, 67,62, 68,23, 31,48,
  42,62,  8,37, 55,26, 16,29,  1,28, 50,28, 12,37,  0,28,
  62,26, 47,35, 57,31, 50,66, 41,49, 55,43, 24,60, 20,41,
  13,32, 47,63, 23,25, 45,63, 17,17, 23,28, 45,50, 35,21,
  30,38, 47,38, 10,30,  7,43, 41,22, 59,19, 42,64, 52,67,
  12,25, 33,36, 27,56, 40,24, 14,10, 37,27, 35,47, 63,21,
  23,27, 51,49, 46, 9, 15,28, 41, 6,  2,64, 19,59, 57,62,
  45,37, 24,69, 31,67, 35,55, 67,65, 61, 5,  7,59, 31,51,
  51,44, 67,39, 18,50, 34,60, 37,37, 22,30, 15, 3, 29, 5,
  25,66, 62,34, 20,53, 54,33, 23,38, 53, 1, 17,57, 25,63,
  62,43, 68,11, 54, 1, 39, 1, 58,15, 58,47,  9,46, 62,32,
   3,44, 56,54,  6,25, 32,46, 21,26, 44,42, 30,37, 58,41,
  27,51,  8,26, 61,55, 55,21, 50,13, 31,54, 51,58, 39,39,
  32,34, 39, 2, 52,31, 26,10, 68,40, 18, 3, 44, 1, 49,62,
  34,29, 40,57, 61,14, 22,29,  4, 0, 61,51, 42,55, 25,46,
  10,28, 69,16, 29, 8, 56,10, 42,33, 47,64, 56,14, 62,50,
  55,63, 45,28, 63,59, 66,39, 65,55, 55,62, 44,57,  3,17,
  56, 0, 40,26, 36,60, 12,39, 46,16, 68,18, 46,21, 37,24,
  12,21, 16,62, 52,57, 59, 0, 61,56, 66,52, 65,25, 64,59,
  34,57, 53,60, 37,57, 56,38, 15,41, 31,20, 30,18, 52,55,
  59,60,  6,13, 20,36, 10,44,  8,12, 65,66, 32, 1, 36,55,
  54,16, 25, 8, 27, 3, 29,58, 33,40, 28,15, 11,49, 27,16,
  63,36, 27,34, 44,28, 39,22, 11,68, 15,29, 21,61, 49,32,
  32,10, 27,13, 26,19, 68,52, 67,59,  3, 2, 32,57, 36,28,
  49, 3, 17,47,  1, 6, 68,51,  5,60,  0,63, 60,29, 30,56,
  14,37, 19,54, 27, 2, 67, 5, 40,29, 23,22, 52,34, 36,17,
  58,35, 25,48, 53,12, 55,27, 28,48, 15,63, 16,28, 17,41,
   5,23, 44,22,  2,45, 21,64, 44,39, 12,62, 60,13, 42,18,
   4,46, 37,44,  3,18, 29,20, 10, 4, 33,12,  3,34, 21,60,
  65, 0, 12,42, 57,10, 68,16, 23,19, 34,35, 21,66, 17,48,
   3,43, 36,56,  2,30, 14,26, 61,19, 21,48, 48,37, 21,22,
   0, 9, 31,24,  7,20, 65,49, 25,59, 43,49, 23,23, 47,19,
   8,59,  8,63, 10,64, 43, 9, 55,29, 24,16, 63,12, 64, 1,
   0, 8,  9,55,  6,65, 45,10,  7,17, 51,11,  7,56, 67, 2,
  21,33,  2,59, 67,31,  0,31, 37,52, 43,66, 19,68, 41,26,
  47,48, 61,28, 38,12, 14, 0, 35,15, 15,49, 61,68, 28, 8,
  30,27, 32,22, 18,21, 23, 3, 38,39,  4,49, 61,32, 15,52,
  46,42, 17, 5,  1,33,  8,49, 56,19, 37, 6, 17,59, 36,39,
  46,39, 37,64, 51,35, 14,57, 51,43, 57, 3, 44,62, 36,42,
  19,30, 17,13, 30,32, 68,22, 56,28, 34,65, 46,33, 19, 3,
  66,21, 28,14,  6,10, 51,28,  3,26, 58,57, 11, 9, 49,49,
  44, 6, 50,46, 43,68,  3,61, 26, 0, 41,20, 49,53, 19,66,
  46, 5, 20,62, 38, 8, 35, 2, 32, 7, 22,64, 53,53, 68,14,
  66,14, 14,38, 60,52, 45,34,  3,45, 22, 2, 44,37, 59,44,
  52,28, 41,36, 68,42, 12, 7,  9, 4, 43,35, 34,31, 21,49,
  35,25, 31,32, 57,65, 12,35, 32,21, 67,11,  8,31, 26, 8,
  32,27, 54,14, 24, 2,  7,37, 31,53, 35,60, 26, 4, 58,65,
  62, 4, 38,45, 53,11, 49, 6, 33,14, 64,20, 43,10, 67,58,
  65,27, 63,14, 16,66, 69,44, 16,51,  9,12, 60,16, 34,51,
  43,32, 64,42, 20,37, 66, 1, 18,23, 24,39, 62,59, 50,30,
  47,20, 14,58, 34,28, 20,15, 61,40, 58,54, 27,59,  6,55,
   4,27, 45,51, 30,42, 63,42, 35,36, 15,31,  0,18, 13,19,
  40,51, 34,14, 13,31, 57,66, 33,11,  6,58, 62,37, 49,65,
  37,42,  9,19, 16, 3,  3,54, 42,38, 61, 9,  9, 5, 56,49,
  63, 0, 67,13, 12, 1, 21,63, 67,19
};
const int p_70_18[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  23,
  /// Array of preferences
  66,36, 67,50, 31,52, 21,45, 48,68, 65,32, 52,34, 60,55,
  49,55,  5,23, 65,27, 32,17, 25,54, 34,61, 11,29, 55,15,
   0,11, 67,20,  5,64, 55,16, 28,12, 67,29, 13, 2
};
const int p_70_19[] = {
  /// Number of people on picture
  70,
  /// Number of preferences
  273,
  /// Array of preferences
  49,38, 42,49, 40,54, 49,51, 53,31, 42,61, 58,64, 41, 5,
  59,46, 13,22, 67,40, 25,18, 21,17,  7, 2, 30,21,  6,48,
  13,62, 38,68, 65,39, 49,24, 43,41, 39, 7, 69,50, 37,55,
  16,61, 25,12, 26,48, 25, 9, 29, 5, 41,22, 32,55,  1,67,
  29,51, 61,63, 38,55, 64,22, 16, 8, 42,46,  2,53, 65,26,
  68, 8, 32,33,  2,12, 45,56, 63,65,  2, 6, 47,59, 31,63,
  42,65, 39,57, 59,14, 64,66, 29, 2, 45,30, 55,61, 39,66,
  63,44, 47,47, 58,49, 29,18, 65,20, 58,18, 67,35, 45,20,
  21,24, 61,16, 61,30, 27,27, 41,42, 44,28, 53,69, 60,10,
   6,13, 42,22, 58,15, 34,13,  3,20, 46,12,  2, 0, 49,48,
  50,13, 55,40, 39,59, 52,58, 16,48, 56,55, 60,47, 47,60,
   6,44, 12,54, 36,65, 46,51, 63, 4, 66,41, 12,37,  7,33,
  66,61, 19,59, 60, 5, 12,64, 33,43, 68, 3, 24,65, 58, 0,
  21,38, 30,19,  2,16, 16,21, 24,25, 44, 1, 52,49, 14, 1,
   7,32, 34,35,  3, 7, 22,35, 66,63, 27,69, 30, 5, 48,41,
  66, 1, 23,54, 46,47, 45,29, 15,17, 31,61, 14,68, 45,31,
  50,65, 65,24, 63,13, 64,58, 27, 9, 42,44, 62,42, 20,38,
  35,37, 27,35, 52,69, 54,36, 65,21, 51,29, 36,18, 41,40,
  11,41, 67,63, 10,17, 24,57, 31,44, 61,48, 49,16, 62,56,
  60,59, 46,39, 45,22, 43,26, 26,20, 16,44,  2,32,  2,41,
  24,50, 44,43, 13,30, 40,47, 43,18,  0,57,  9,35, 31,53,
  53,27, 46,25, 37,22,  7,69, 59,65, 22,11, 68,48, 69,44,
  30,51, 22,25, 55,37, 67,26, 58, 3, 20,42, 40,31, 25,14,
  50,44,  4,50, 67,28, 31,34, 53,19, 15,59, 27,45, 60,66,
   1, 2, 27,59, 20,47, 11,66, 25,43, 36,68, 69,49, 58,12,
   8,33, 54,46, 21, 8, 38,62, 51,57, 30,67,  1,23, 23,18,
  40,17, 38,66, 28,26, 15,49, 48,59, 55,21, 65,56, 30,56,
  10,12, 47,67, 39,67, 58,46, 56,21, 36,41, 14,31,  8, 3,
  12,51, 36, 6, 19,54, 12,33, 43,46, 64,67, 55,31, 12,16,
  41,64, 65,48, 59,26, 36,45, 17,65, 28,51, 38,30, 65,53,
  34,53,  3,11, 48,28, 39,40, 20,49, 33,25, 68, 0,  9,62,
  64,45, 50, 8, 30,10, 65,12, 36,53, 68,38, 40,12, 18,14,
  61,27, 25,27, 21,21, 25,34, 59, 5, 47,20, 12,40,  8,40,
  23,37, 63,45, 27,15, 61,59, 45,59,  5,15, 25,13, 44,42,
  52,48
};

const int p_80_0[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1317,
  /// Array of preferences
  69,49, 76,73, 31,58,  1,41, 78,61, 76, 8, 16,36, 60,31,
  48, 0, 75,38, 30,66, 46,34, 26,56, 73,43, 16,12, 36,66,
  59,43, 48,42, 28,60, 45,34, 49,39, 65,21, 27,50, 18,53,
  42,21, 12,23, 55,33, 10,35, 71,69, 53,67, 28, 5,  3,37,
  21,62, 61, 5, 35,40, 76, 1, 18,27, 73,79, 33,51,  4,28,
  44,23, 44,61,  2,32, 34,53, 27,14,  5,42, 32,67, 32,21,
  27, 4, 18,22, 53,30, 49,53, 41,40, 49,49, 52,20, 60,41,
  39,70, 55, 3, 52,70, 33,34, 43,37, 39,14, 61,18, 10,15,
  47,72, 22,75, 74,24, 43,32, 26,24, 24,55, 57,43, 43,79,
  43,19, 10,33, 43,58, 31, 7, 78,75, 52,69,  0,73,  1,17,
  47,51, 26, 4, 31,73, 12,56, 16,21, 18,54, 10,30, 66,69,
   1, 6, 61,49, 62,77, 44,42, 52,49, 75,77, 41,79, 16,79,
  38,21, 43,34, 37,24, 54,67, 54,64, 43,41, 67,18, 41,51,
  33, 7, 69,45, 44,12, 74,36, 30,23, 57,34, 73,12, 17,75,
  15,40, 19,18, 50,22, 20,79, 71,41, 11,36, 40,56,  6,57,
  56,43, 45,28, 63,57, 56,21,  4,20, 14,58,  1,16, 73,70,
  30,63, 71,24,  5,73, 77,78, 19,62, 77,49, 20,56, 42,79,
  47,40, 17,54, 27, 7, 10,39, 33,56, 18,39, 62,63, 60,19,
  23, 1, 79,61, 70,47, 54,66, 39,57,  5,57,  9,12, 39,44,
  67,28, 19,37,  3,59, 22,70, 63,14, 71,40, 55,15, 21,74,
  16,63, 61,12, 45,36, 27,76, 59,38, 12, 7, 71,62, 66,26,
   7,33, 19,30, 78,48, 47,10, 27,67, 51,65, 15,62, 34,21,
  78,74, 35,23, 59,69, 11,50, 70,75, 53,52, 32, 8, 50, 2,
  14,19,  7,78, 77,44, 38,33,  8, 7, 68,25, 61,58, 72,52,
  18,47,  8,72, 66,71, 57, 4,  8,42, 37,17, 11,40,  7, 7,
  36,67, 48, 4, 55,76, 14, 6, 68,72, 79,56, 63,11, 59,52,
  13,42, 41,24,  6, 7, 14,77, 62,36, 56,30, 31,56, 50,52,
  39,59, 75,54, 73, 1, 65,73, 30,68, 19, 1, 77,75,  5, 1,
  11,78, 44, 0, 15, 8,  7,29, 14,71, 42,72, 44,19, 40,20,
  66,25, 19,24, 28,28, 50,53,  1,12, 63,54, 33,31, 69,67,
   1,71, 79,28, 52,24, 77,32, 37,72, 78,69, 61,64, 21,46,
  50,79, 44, 1, 75,33, 10, 6, 57,17, 56,47, 70,26,  2,76,
  73, 6,  9, 7, 29,60, 41, 8, 34,56,  7,52, 24,48, 33,73,
  30,30, 38,43, 37,30, 65,17, 21,37, 25,67, 67,55, 50,58,
  61,76, 73, 4, 76,70, 79,11,  9,22, 35, 7,  4,45, 44,76,
  31,24, 57,49, 15,58, 65,71, 77,18, 17, 7, 30,70, 28, 6,
  66,43, 43,48, 23, 8, 13, 5, 21,47, 48,79,  1,59, 55,18,
  72,12, 78,59, 52,40, 66, 8,  9,79, 16,56, 79,68,  9,51,
  50,35, 66,50, 59,58,  6,45, 38,70, 29,61, 51,59, 71,16,
  49,36, 45,11, 43,15, 44,15, 42,30, 28,13, 15,25,  1,25,
  71,54, 28,20, 33,71, 68, 3,  2,26, 15,26, 15,75, 69, 1,
  33,36, 79,53, 63,45, 78,14, 30,48, 57,54, 78,77, 39,55,
  68,56, 40,52, 75,36, 70,71, 29,78,  8, 1, 78,36, 25,74,
  37, 3, 23,51, 41,36, 36, 8, 75,29, 13,30,  4,25, 78,73,
  38,65, 16, 9, 78,13, 28,78, 23,69, 11,39,  0,20, 10,61,
  22,56, 17,59, 59,48, 77, 0, 64,53, 64,55,  7,17, 13, 6,
  56,20, 34,16,  4,31, 29,21, 17,29, 64,73, 38,49, 66,37,
  16,35, 12,58, 20,30,  4,42, 56, 6, 49,42, 40, 5, 25,70,
   9,48, 16,40, 64,44, 43,30, 29,79, 47,19, 77,22, 37,22,
  42,29, 15, 5, 54,71, 73,30, 29,57, 23,52, 11,67, 37,71,
   7,26, 24,58, 36,34, 68,43, 77,62,  6,10, 67,31, 71, 9,
  61,57, 76,31, 32,58, 36,69, 76,43,  5,30,  7,10, 35,29,
  28,50, 13,46, 64,68,  6,55, 33,42, 77,17, 58,45, 57,71,
  71,23,  7,61, 23,78, 24,52,  2,68, 43,68, 21,48, 78,64,
  51,22, 62, 0, 56,29, 70,17, 77,58, 42,12, 22,46, 61,72,
  16,14, 20,73, 41,18, 18,37, 53,29, 12,65, 57,65, 40,79,
  45,32, 28,70, 35,11, 74,41, 41,38, 50, 3, 71,59, 10,49,
  60,78, 27,73,  1, 0, 17,55,  0,41, 51,68, 72,14, 14,24,
  34,69, 11,38, 20,14, 22,40, 17,69, 63, 0, 32,65, 12, 5,
  26,34, 27, 3, 76,40, 78, 6, 19,71, 48,58, 49,63, 69,50,
  74,33, 57, 0,  8,69, 30, 3, 63,24, 33,12, 19,26,  2,52,
  25,66, 43, 0, 32, 7, 53,42, 52,78, 73,21, 64,25, 72,32,
  14,30, 25,53, 78,55, 37,48,  0,72, 69,32, 56,66, 40,10,
   0,28, 79,67, 30, 8, 24, 1, 22,24, 73,49, 66,58, 42,28,
   1,40, 11,47, 66,23, 47,33,  3,56, 75,48, 37,67, 36,75,
   8,24,  4,18, 68,10,  3,60, 45,69, 60,47, 75, 1,  4,35,
   5,76, 57,56, 55,31, 19,35, 42,44, 39,28, 51,62, 25, 8,
  60,37, 35,64, 13,14, 31,41, 27,21, 63,53, 67,78, 35,18,
   3, 5, 30,20, 36,47, 44,46, 17,18, 10,68,  0,45, 36,59,
  12, 6, 42,61, 23, 4, 76,34, 10,55, 11,71,  9,26, 17,10,
  17,51, 23,58, 75,58, 12,57, 35,34, 57,77, 31,12, 64,27,
  56, 1, 51,50, 52,33, 48,59, 50,23, 21,32, 51,45, 70,43,
  41,10, 59,63, 50,61,  4,68,  9,47, 58,39, 50,25, 31,22,
  17,76, 39, 0,  3,47,  4,59, 17, 1, 64,48, 46,56, 65,34,
  54, 2, 71, 8, 31,44,  8,70, 73,66,  6,32, 63,55,  3,31,
  54,51, 45, 0, 36,53, 19,22, 46,35, 78,17, 36,13,  8,71,
  77, 2,  5,64, 21,65, 13,19, 55, 0, 62,21, 20,46, 19, 2,
  67,39, 56,53,  6,43, 27,29, 61,50, 73,24, 53, 3, 34,40,
   9,60, 49,29, 31,16, 61,22,  7,58, 38,22, 47,59, 13, 9,
  33, 1, 76,77, 74,57, 49,47, 37,74, 15,61, 59,21, 14,11,
  74,74, 61,35, 58,17, 72,35, 76,12, 15,57, 56,13,  4, 2,
  78,12, 56,24, 18,44, 50,48,  5,32, 66,76, 60,50, 33,52,
   4, 5, 75,53, 41,59, 37, 1, 63,34, 62,43, 24,47, 48,52,
  29,49, 54,62, 23,33, 51,48, 52,51, 40, 8, 51,71, 45,21,
  44,60, 20,60, 62,19, 20,32, 45,56, 20,41, 55,53,  3,49,
  78,24, 16,17, 50,32,  8,14, 68,12, 74,60, 33,64, 28,53,
   2,36, 62,13, 20,19, 59,74, 13,44, 78,47, 28,45, 71,60,
  57,35, 32,29,  0,39, 34,30, 47,28, 10,12, 14,68, 41,73,
  58,70, 23,64, 13,39, 29,52, 75,71, 42,50,  7, 5, 17,48,
  57,40, 68,74, 58, 1, 51, 3,  9,10, 61,43, 70, 2, 68,60,
  27, 5, 41,25, 27,10, 73,63, 40,53,  6,41, 16,67, 25,25,
  77,63,  2,29, 51,37, 64,65, 62,79, 79,47, 65,39, 29,63,
  42,76, 49,19, 13,66, 66,28, 26,63, 34,34, 52,61, 24, 8,
  59,70,  0,25,  3,41, 32, 1, 17,22, 68,79, 48, 9, 24,40,
  72,37, 47,34, 72,47, 68,69,  7,56, 17,53, 29, 7, 30, 4,
  72,79, 22,18, 42,24, 74,79, 12,55, 19,55, 33,60, 65, 1,
  29,65, 22,60,  7, 6, 22,13, 65,18, 42,36, 26,77,  0,57,
  77,71,  0,30, 32,30, 22,54, 64,76, 35,15, 58,14, 39,74,
  69,62, 24,46, 69,48, 28,17, 47,67, 30,53, 34, 8, 48,54,
  29,59, 44,66, 16,29, 32, 4, 12,70, 32,18, 53,13, 75,37,
  67,53, 61,68, 75,63, 30,41, 14,70, 47,52,  0,60, 25,20,
   9, 0, 41,65,  0,37, 27,49, 11,21, 50,31, 63,36, 12, 4,
  35,68, 65,33, 52,55,  6,13, 38,14, 13, 2,  5,60, 36, 3,
   7, 0, 71,61, 69,34, 39,53, 78,72, 12,51, 50,43, 65,70,
  45,35, 38,55, 42,18, 64,46, 69,29,  0, 0, 63,52, 53, 6,
  77,59, 51,27, 20,54,  9,57, 64,47, 26,33, 35,44,  2,22,
  45, 7, 36,25, 57,27, 31,65, 51,57,  9,13, 57,36, 48, 6,
  16,51, 78,18, 33, 9, 38,23,  9,70, 16,44, 65,60, 18,28,
  13,12, 61,44, 35,79, 64,70, 19,75, 36,48, 27, 9,  3,12,
  57,32, 17,50, 79,62,  8,30, 74,16, 32, 3, 14, 9, 40,72,
  43,46, 26,51, 29,34, 72, 6, 11,13, 12,19, 34,20, 35,60,
  45,38, 73,15, 54,65, 36,79, 42,63, 40,62, 17,71, 24,19,
  22, 8, 33,37, 11,53, 72,26, 56,48,  4,12,  2,35, 46,37,
  68,14,  1, 2, 47,37, 37,31, 26,69, 56,15, 61,16, 39,10,
   4,30, 53, 2, 64,79, 57, 7, 47,66, 28,41, 67,44, 23, 0,
  31,78, 11,77, 46,79,  2,46, 28,65, 44, 2, 66,53, 75,64,
  13,52, 41,48, 70,46, 33, 8, 27,52, 43, 3, 59, 8, 47,64,
  75,43, 19,50, 63,66, 68,57, 49, 1, 46,75, 72,51, 14,66,
  78,30, 22,64, 41,50, 25,48, 54,61, 73,50, 59,20,  6,40,
  11,31, 48,20, 42, 5, 13,61, 19,52, 25,55, 58,34,  5, 8,
  43,33, 68,36, 75,44, 76,53, 72,38, 53, 1, 39, 7, 26,37,
  27,68, 43,35, 52,35, 21,21, 29,56, 41,27, 16,55, 47,25,
   5,15, 19,46, 22,68,  8,51,  8,27, 48,34, 52,65, 73,38,
   0,67, 72,60, 46,78, 29,23, 17,39, 65,28, 13,22, 10,47,
  39,60, 76,52, 10,19, 24,63,  8,53, 39,76, 70, 5,  5, 9,
  56,18, 67,21, 64,56, 31, 2,  1,34, 59,72, 60,21, 73, 7,
  17, 0,  5,25,  3,50, 69, 0, 32,68, 52,25, 49,31, 15,68,
  74, 0, 53,46, 18, 1, 23, 2, 23,75, 24,45, 20,45,  9,71,
  46,61, 50,37, 37,65, 41,23, 55,57, 44,70,  6,31, 53,50,
  59,55, 48,50, 45,46,  1,53, 63,62, 41,22, 48,74, 20,47,
  23,68, 50,51, 55,72,  7,51, 22,52, 57,78, 66,64, 56,55,
  38,75, 11,79, 52,11, 59,64, 33,35, 47,21, 71,27, 74,39,
  48,35, 13,56, 23,77, 33,43, 39,62, 56,44, 44,41,  0,18,
  15,56, 32,25, 53,33, 43,22, 62,12, 31,52, 43,16, 41,60,
  66,32, 34,72,  9,19, 65,44,  1,36, 17,44, 12,34,  4,49,
  47,46, 69,18, 72,48, 42,42, 50,70,  0,70, 62,39, 39, 4,
  68,37, 66,55, 60,46, 59,44, 28,73, 60, 1, 12, 1, 36,35,
  76,14, 20,37, 12,46, 13,49,  7,71, 74,72,  1,10,  7,11,
  36, 6,  9,75, 55,48, 14,39, 34,62, 79,31, 23,50, 27,77,
   1,35, 74,38, 56,71, 43,42,  7,43, 66,59, 29,71, 41, 4,
  56,33, 13,63,  4,26, 76,78, 40, 7, 71,68,  0,11, 26,45,
  33,32, 70,31, 37,55, 19,20, 28,55, 43,40,  2,44, 65,24,
  10,22, 30,58, 24,32, 24,14, 37,18, 29,40, 72,78, 51,33,
   3,10, 14,35, 13,17, 33,23, 27,57,  3,64, 60,40, 57,21,
  24, 3, 32,43, 36,40, 45,66, 74,34, 25,12,  8,29, 52,14,
  49,64, 47, 2, 72,30,  6,67, 24,67, 25,71, 34, 0, 52,77,
  73,47, 31,33, 26,72, 78, 3, 39,77, 34,67, 57,10, 15,36,
  38,24, 15,60, 78, 9, 52,71, 54,11, 77,64, 15,51, 10,58,
  79,63, 54, 4, 39,30, 64,15, 79, 4, 64,43, 77,35, 47, 6,
   6,59, 23,48, 50,66, 79,46, 50, 1, 15, 9,  5,53, 41,43,
  11,60, 42,22,  1,49,  4,75, 10,10, 29,27, 25,10, 11,57,
  73,48, 45,12, 36,73, 74,77, 55,25,  5,71, 35,20, 39,15,
  62,64,  1,47,  8,54, 12,29, 78,79, 77,38,  1,24, 68,30,
  56,26, 48,16, 62, 7, 78,44, 28,40, 51,38, 42,17, 21,75,
  52,79, 52,42, 34, 5, 74,35, 28,19,  9, 5,  5,12, 40, 4,
  77,46, 67,77, 61,31, 38, 9, 53,38, 33,47,  0,13, 53, 5,
  45,59, 56, 2, 46,25, 46, 8, 31,30, 27,19, 23,11, 54,20,
  17,40, 68,40, 11,34,  8,34, 72,42, 39,34, 22,20, 32,16,
  14,63,  9, 9, 72,11, 12,17, 73,33, 20, 8, 29,45, 14,33,
  30,15, 73, 3, 32,44, 21,63, 23,43, 59,37, 52,43, 22,29,
  54,43, 58,75, 79,27, 77,70, 61,40, 46, 4, 55,63, 72,56,
  48,31, 58,61, 21, 6, 46,48, 18, 5, 60,57, 73,75, 50,49,
  56,72, 34,54, 46, 5, 63,76, 48,33, 58,35, 43,67, 47,39,
  32,70, 71,10, 29,12, 43,76, 25, 1, 67,79,  0,22, 27,79,
  30,31,  5,78, 41,54, 40,73, 41,30
};
const int p_80_1[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  509,
  /// Array of preferences
   9,59, 40,61, 63,11, 41,67, 58,38, 41,52,  0,72, 14,50,
  33,26, 31,46,  8, 2,  4,16, 22,59, 44,66, 12,72, 26,40,
  75,26, 56,67, 52,62, 25,43, 28,48, 26, 3, 78,37, 72,25,
  67, 7, 62,27, 42,49, 17, 6, 59,50,  2, 6, 49,34,  9,65,
  64, 3, 52,55, 59,58, 50,55, 42,25, 23,49, 28,38, 57,59,
  16,50, 53,61, 76,44, 73,66, 51,37, 20,47, 38,11, 39,76,
  54,74, 67,53, 25,35,  8,31,  7,30, 24,46, 36,55, 23,55,
  37,30, 76,28, 71,23, 53,10, 21, 1, 60,72, 58,48, 78,72,
  30,50,  1,25, 31,38, 75,49, 23, 3, 58, 0, 19,27, 68,57,
  53,36, 77,13, 50,39, 55,28,  9,32, 66,18, 35,46, 11,69,
   7,23, 47, 6, 23,23, 66,79, 33, 4, 28, 0, 22,48, 57,47,
  78,31, 46,42, 55,74, 33,78, 18,57, 66,64, 47,61, 11,72,
  58, 4, 75,59, 79,70, 69,46, 37,58,  5,27, 58,66, 52,59,
  40,75, 21,39, 10,63, 52,68, 27,16, 10,22, 53,42, 44,20,
  69,12, 32,62, 13,17,  5,65, 47,32, 54,56, 41,34, 76,31,
  77,64,  0,34,  0, 1, 41,69, 59,49, 37,59, 14,13, 19,39,
  59,61, 25,42, 64,74, 64,68, 34,42, 37,34, 76,40, 46,10,
  64,42, 71,40, 33,76, 45,39, 63,14, 10,12, 56,20, 70, 1,
  29,79, 63,16, 69,26, 62,55, 75,41,  7,59, 36,58,  2,34,
  74,31, 14,52, 10,15, 59,73, 25,27, 10,65, 13,72, 70, 4,
  74,22, 70,57,  5,71, 52,41, 24,36, 60,76, 38,57, 41, 6,
  35, 6,  5, 8, 72,15,  7,76, 57,69, 48,63, 36,76, 38,70,
  67,61, 66,47, 56,61, 61,62,  7,42, 62,49, 57,24, 48, 9,
  65,69, 64,43, 78,69, 24,50, 26, 7, 68,65, 66, 4, 67,65,
  50,44, 16,53,  9,49, 26,44,  9,19,  5,66, 35,45, 33, 2,
  18,72,  2,65, 46,37, 79,72, 42,74, 32,72, 30,30, 79,28,
  56,10, 76,17, 56,11,  3,29, 32,36, 47,66, 14,41, 69,13,
  41,29, 53,78, 71,62, 38,48, 45,35, 44,19, 19,46, 49,35,
  73,29, 36,65, 44,31,  7,32, 18,38, 77,59, 26,77, 49,27,
  39,69, 15,18, 24,68, 19,65,  8, 9, 49, 2, 34,14,  5, 2,
  58,43, 28,65, 56,73, 67,32, 22,40, 77, 0, 34,32, 21,31,
   4,25, 50,33, 39,31, 70,78, 49,57,  6,71, 53,56, 18,10,
  42,10, 12,14, 59,43, 64,59, 37,74, 24,74, 13,64, 24,60,
   1,67, 28, 8, 76,74, 35,52, 28,33,  8,59, 50, 4,  6,39,
  72,42, 66,25, 18,21, 11,64, 41,21, 51,63, 62, 2,  7, 6,
  34,71, 73,76, 54,43, 57,10, 79,45, 13,44, 34,59, 35,42,
  32,55, 44,43, 28,49,  8,40, 24,17, 40,57, 54,23, 46,11,
  70,38, 25,12, 68,14, 48,50, 72, 6, 10,39, 62, 3, 10,46,
  45,32, 60,74, 45,47, 63,54, 31, 7, 56,55, 68,63, 78,26,
  29,15, 77,18, 49,13, 66,58, 68,42, 17,26, 19,34, 44,47,
  11,73, 63,58, 28,69, 74,69, 36,75, 19,18, 34,45,  8,58,
  51,74, 16,25, 20,16, 46,58, 50,57, 50,53, 47,30, 14,60,
  39,60, 12,42,  9,38, 11,53, 67,33, 76,16, 54,39, 58,45,
  54, 1, 71,39,  2,26, 41,68, 37,76,  8,76, 48,58, 63,31,
  12, 9, 24,24, 68,16, 23,78, 22, 8,  7,65, 34,24, 53,35,
  58,70, 71,12,  9,14,  4,22, 59,55, 22,10, 68,60, 59,67,
  45, 1, 18,18, 20, 4, 37,19, 45,26, 41,75,  8,54, 12,54,
  56,35, 51,65,  3,18, 30,74, 12,73,  8,14, 64,79, 33,21,
  55,72, 29,38, 41,23, 58,11,  0,79,  3, 1, 30,69, 35,62,
  52,47, 60,14, 50,56, 55,42, 53,51, 38, 3, 26, 1, 56,27,
  20, 1, 21,54, 36,35, 54,32, 65,13, 23,14, 11,58, 30, 2,
  34,21, 44,37, 73,71, 38,59, 71, 5, 38,64, 75,18, 24,45,
  55,56, 74,29, 45,56, 35,76,  8, 6, 14, 5, 63,10, 18,26,
  10,44, 15,27, 10, 6,  4,35, 61,19, 32,77, 31,65, 50,52,
  26,25, 59,23, 52,48, 12,74, 48,12, 56,18, 55,25, 42,42,
  10,26, 73,17, 17,75, 68, 9, 33,54, 25,14, 28,45, 40,73,
  45,68, 36,16, 25, 1, 13,16, 71,30, 28,76, 41,26, 71,17,
  32,66,  1,39, 32,56, 34, 6, 59,21, 50,76, 49, 1, 55,47,
  65,15, 32,41, 44,25, 43,69,  8,23, 73, 8, 27,72, 26,12,
  18,47, 76,52, 37,16, 17,68, 10,25, 41, 1, 57,26, 20,35,
  18,61, 59,13, 16,41, 75,25, 62,33, 38,66, 15,75,  2,73,
   5,76, 18,36,  9,77, 56,49, 67,66, 36,11, 48,43, 79,37,
   9,58, 20,57, 18,16, 47,26, 22,32, 66,26, 45,24, 19,26,
  71, 7, 32,76, 60,58, 20,69, 18,14
};
const int p_80_2[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  749,
  /// Array of preferences
  37, 4,  9, 0, 21,79, 22,48, 35, 7, 36,55, 20,38, 29,33,
  74,35, 78,58,  6,27,  0,60, 69,19, 72,46, 42,46,  6, 5,
  43, 0,  7,15, 27, 3, 45, 2, 67,33, 72, 4,  8,13,  4,75,
  20,40,  5,24, 31,35, 66,37, 39, 7, 16,30, 79,11, 52,48,
   4,23, 44,43, 46,34, 39,53, 28,49, 32,18,  7,61, 76,10,
  27,36, 57,10, 46,67,  8,24, 44,39, 46,26, 62,73, 74,66,
  67, 3, 69,41, 58,23, 58,28, 40,66, 78, 6, 71,17, 73,67,
  60, 9, 16, 2, 79,68, 38, 8, 15,57, 38, 9, 78,12, 32, 0,
  63,26,  3,72, 56,59, 17,25,  9,73, 26,13, 29,63, 76, 0,
  42,40, 24,12, 65,46, 62,55, 50,56, 66,47,  6,51,  7,35,
  68,34,  6,11, 20,22, 27,25, 54,28, 55,22, 10,23, 35,69,
  34,33, 76, 7, 23, 9, 16,20, 62,28, 51, 4, 12,32, 12,18,
  60, 2, 75,62, 75,36, 41,43, 69,64, 77,46, 67,10, 22,78,
  38,62, 25,52,  9,53,  5,12, 31,71,  0,62, 72,51, 42,61,
   3,53, 32,39, 40,65, 77,17, 10,51, 34,43,  1,53, 24,40,
  26,69,  6,28, 41,16, 71,61, 26, 1, 61,47, 43, 4, 44,66,
  53,11,  1,45, 16, 9, 18,13, 32, 4, 57,50,  8,37, 49,76,
  25,51, 51,34, 65,50,  0,10, 37,34, 19,16, 74,79, 59,56,
   8,57, 20,29, 31, 9, 43,20,  9,13, 12, 7, 65, 2,  7, 9,
  21,31, 42,21,  8,33, 72,24,  7,47, 70,69, 58,47, 18, 0,
  27,15, 72,33, 16,33, 61,13, 39,47,  7,49, 41,54,  1,33,
   5,41,  4,20, 25,74, 42,19, 25,62, 53,65, 11,51, 73,35,
  35,57, 38, 1, 10,75, 61,26, 26,22, 42,41, 38,52, 53, 9,
  54, 8, 19,43, 35,10, 16, 5,  8,11, 66,13, 28,76,  2,30,
   4,24, 41,66, 24,70, 79,16, 54,19, 36,13, 36,19, 11,49,
  61,51, 66,72, 31,16, 31,59, 61, 6, 48,36, 31,29, 45,36,
  28,36, 53,33, 17,54, 22,65, 15,43, 46,44,  0,64,  4,73,
   9, 4, 52,31,  5,52, 22,73, 32,15, 68,76, 23,46, 12, 4,
   3,43, 49,17, 21,27,  0,11, 63,57, 26,35, 25,28, 21,58,
  58,56, 20,52, 38, 7, 51,32,  8,22, 40,75, 50,36,  2,35,
  53, 2, 74,38, 61,76, 17,72, 20,57,  4,54, 45,75, 64,14,
  62,66, 72,75, 71,37, 72,56, 37,21, 53,67, 27, 5, 48,58,
  72,17,  9,70, 59,77, 48,72, 65,72, 43,47, 25,34, 13,69,
  52,34, 10,50, 74,76,  5,63, 48,39, 37,74,  7,26, 66,27,
  45,46, 56,79, 27, 4, 11,44, 65,16, 30,18, 10,27, 46,12,
   6,69, 56,66, 42,34, 14,35, 33,52, 37,49,  3,23, 62,19,
  37, 0, 49,13,  5,27, 55,78, 22,54, 42,17, 24,72, 53, 3,
  49,40, 20,12, 52,75,  9,32, 21,77, 37,42, 27,46, 10,73,
  22,66, 71,24, 13,39,  2,20, 41, 4, 15,12, 26,79, 45,52,
  74,31, 13,56, 40,46, 43,63, 56,12, 12,20, 77,51,  4, 4,
  14,46,  9,36, 36,73, 27,79,  8,27,  9,76, 37,23, 63,62,
  52,61, 45,74, 47,66,  0,57, 75, 6, 56,65, 74,43, 57,75,
  53,61,  4,52, 17,10, 30,62, 21,22, 55,19, 78,23, 75, 1,
  21,11, 45,24, 48,20, 13,12, 67,23, 23,58, 78,48, 47,51,
  63,61, 49, 7, 68,13, 59,72,  1,52, 76,12,  7, 3, 11,40,
  66, 8,  7,20, 55,68, 43,50,  1, 9, 12,46, 27,16, 14,37,
  73,42, 69,62, 73,20, 33,42, 37,79, 48,76, 50,23, 19,66,
  48,38, 59,68, 56,34,  2,73, 24,49, 61,27,  6,67, 38, 5,
  24,18, 78,25, 24,23, 50,22, 52,68, 50,63, 32,37, 33,23,
  13,49, 22, 2, 17,51, 42,33, 41,28, 19,14, 50,57, 17,38,
  75,26, 79,54, 61, 1, 67,72,  8, 0, 24,39, 79,35,  3, 7,
  28, 2, 54,44, 52,14, 65,75, 46,42, 48, 1,  4,30, 34, 2,
  16,38,  1,77, 50,27, 70,32, 22,35, 20,59,  4,44, 78,44,
  19,18,  3,51, 10,64, 47,19, 45,38, 47,34, 45,50, 56,14,
  39,50, 65,23, 63,30, 19,15, 45,27,  0,78, 72,58, 37,27,
   2,58, 51,43, 38,65, 47,50,  1,76, 60,25, 37,77, 34,36,
  46,31, 32, 6, 22, 8,  2, 2, 59, 1, 51, 0, 21,74,  0,73,
  47, 9, 27,34, 48,27,  3,76, 67,36, 65,14, 28,73, 40,52,
  26,57, 23,19, 15,61, 44,56, 33,77,  3,79, 70,70, 16,14,
  75,20, 25,47, 72,15, 73,21,  3,28,  2, 5, 54,57, 61,28,
  33,46, 39,66, 72,13, 61,37,  0,51, 11,17, 76,30, 27,55,
  26,70, 46, 6, 74,49, 64,46, 77,64, 59, 7, 43,46, 72,10,
  39,55, 44,49, 29,56, 42,56, 58,49, 29, 2, 15, 3, 62,67,
  19,27,  8,79, 51,19, 48, 0, 30,22, 25,20, 23,57, 67,73,
  77,29, 29, 5, 79,49, 43, 3, 23,31, 42,29, 77, 2, 74,42,
  44,21, 64,72, 28, 7, 39,25, 46,22,  6,32, 32,17, 77, 4,
  12, 2, 32,30, 70,35, 71,14, 17, 2, 25, 2, 27,32, 30,24,
  71,60, 51,23, 74, 8, 51,14, 23, 1, 71,57, 16,13, 35, 6,
  50,40,  1,18, 66,25, 49,41, 22, 4, 73,57, 30,49, 23,38,
  70, 9, 35,60, 21,33, 72,26, 50,64, 44,35, 18,14, 77,56,
   9,49, 39,67, 39,65, 24,15, 31,64, 40,24, 35, 1, 68,29,
   8,50, 40,38,  1,38, 68,31,  9,68, 50,68, 16,63, 46,50,
   5,42, 32,44, 29,75, 22,33, 21,45, 29, 3, 53,31,  9,74,
  66,53, 43,15, 57,29, 63,59, 17,32,  8, 7, 59, 0,  8,25,
  52,40, 18,21, 18,26, 42, 6, 11,71,  8, 6, 44,33, 10,13,
  56,52, 56, 2, 59,55, 31,54, 76,32, 70,36, 37,28, 62,71,
   8,12,  3,68, 13,20,  7,17, 11,57, 79,37, 74,61, 22,60,
  61,40,  1,66, 46,57, 12,68, 72,72, 78,11, 40,55, 39,34,
  47,52, 52,39,  9, 9, 50,32, 69,10, 47,48, 66,45, 63,67,
  43,48, 51,66, 34,64, 64,45, 78,47, 32,49, 39,49, 45,60,
   5,79, 64, 8, 27,63, 10,63, 60, 4, 73,17, 14,44, 74,24,
  30,52, 60,37, 42,32, 18,31, 67,56,  0, 3, 55,35, 37,18,
  19,60, 60,56, 37,32, 78, 7, 65,68, 34,13, 12,49, 35,45,
  22,23, 42,72, 45,11, 43,72, 60,17, 78,69, 46,25, 41,68,
  55,71, 22,50, 73,64, 56, 4, 48,50, 43,13, 12,14, 13,64,
  18,11, 26, 7, 56,28,  0,21, 26,40, 34,20, 35,54, 44,52,
  24,34, 23,51, 49,37, 18, 8, 11, 8,  0,14, 32,79, 21,72,
  79,60, 74,19, 29,44,  3,48, 20,50, 50,20, 16, 4, 60,35,
  49,31, 72,29, 45, 4, 77,31, 32,41, 47,68, 14,27, 63,12,
   2,49, 35,74,  6,15,  9,46, 17,50, 23,13, 48,10,  2,16,
  35,71, 20,23,  3, 0, 21,47, 35,23, 63,13, 55,20, 20,54,
  46, 3, 56,30, 34,66, 77,49, 73,52, 55,52, 35, 5, 58,27,
  34,62, 29, 6, 48,32, 58,58,  4,77, 60,19, 29,55, 56,32,
   4,67, 44,29,  7,31, 15,19, 55, 5
};
const int p_80_3[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1246,
  /// Array of preferences
  27,67, 45,22, 58, 3, 54,19,  2,49, 76, 5, 69,24, 12,58,
  77, 7, 64,74, 79,72, 36,59, 65,19, 75, 0, 22, 1,  7,34,
  67, 4, 32, 4, 55,52, 47,26,  2,77, 64,66, 66, 8, 56, 0,
   0, 7, 29,28, 46,15, 70,75, 32, 0,  3,40, 24,76, 15,50,
  33,70, 65, 6, 73,49, 44,19, 71,47, 68,25,  3,62, 57,77,
  47, 9, 31,13, 77,71, 37,44,  7,47, 77, 1,  8,76, 77,48,
  66,26, 51,20, 57,54, 79,63, 40,41, 47,32, 74,37, 77,58,
  70,67, 67,12, 10,44, 23,34, 21,10, 34,21, 57,46, 17,57,
  51,26,  3,60, 29,19, 16,20, 10,67, 71,44, 39,59, 26,23,
  76,58, 41,71, 40,11, 23, 8, 18,74, 12,67,  9,72, 12,35,
  34,43,  2,26, 16, 4, 23,35, 63,21,  0,32, 76,12, 75,19,
  62, 9, 74,29, 72, 2, 79,22, 75,56, 54,63, 51, 5, 36,12,
  50,40, 41,22, 55,64, 42,66, 12,69, 29, 3, 23,22, 16,55,
  70,18, 14,49, 77,69, 64,23, 69,23, 20,51, 30,67, 76,19,
  56,10, 29, 9,  9,55, 68,61, 46,76, 51,62,  5,15, 33,49,
  55,19,  5,13, 37,42, 54,32, 64,63, 49,60, 76,32, 41,36,
  40,25,  5,44, 11,12, 63,66, 24,39,  2,27, 18,54, 48, 3,
  47,57, 61,24, 29,22, 42, 1, 78,44, 39,48, 30,53, 72,18,
  16, 7, 79,35, 30,72, 69,12,  7,73, 15, 3, 58,32, 15,13,
  23,56, 76,49, 25,45,  0,28, 47,17, 31,39, 50,30, 50,14,
  64,71, 25,23, 50,76,  9,46,  5, 0, 23, 5, 43, 6,  9,20,
  50, 1,  9,17, 21,74, 73,70, 43,33, 19,44, 36,45, 19,37,
  31,55, 50,52, 32,66, 20, 6,  4,17, 10,41,  3,27, 26,30,
  12,74,  8, 0, 55,41, 36,50, 25,59, 77,15, 45,16, 27,15,
  36,26, 47, 3, 15,77,  6,58, 56,28, 17,21, 68,69, 78,42,
  21,40, 59,71, 64,20, 34,57, 78,39, 41,25, 73,39, 78,29,
  13, 0,  9,13, 79,46, 36,24, 71,28, 34,60, 59,43, 76,60,
  37,45, 67,44, 59,53, 31, 5,  8,12, 71,62, 70,12, 23,16,
  17,47, 14,29, 46,19, 15,41, 40,79, 29,34, 74, 4, 44,64,
  70,72, 41,77, 15, 6,  2,19, 45,74, 69,15, 33,25,  4,72,
  58,37, 79,26, 34, 1,  9,11, 10,70,  3,52, 44,66, 57,37,
  25,16, 42,14, 37,46, 61, 9, 53,18, 34,79, 27,61, 33,79,
  27,43, 17,16, 62,69, 57,64, 12,45, 40, 3, 75,77, 16,47,
  47,30, 58,33, 70,63, 50,23, 31,76,  9,66, 32,36, 61,40,
  51,53, 33,28,  7, 7, 59,72, 71,15, 70,54, 12,42, 18,12,
  25,15,  8,62, 30,27, 61,10, 69,47, 72,74,  4,30, 73,28,
  61,17, 14,11, 20,70, 51,18,  2,23, 35,25, 50,19, 79, 2,
  66,72, 62,67, 42,65, 38,64,  4,50, 22,67, 18,52, 69,13,
  32,44, 20,52, 60,16, 14,61,  8,31, 20,25, 53,21,  9,47,
  77,33, 21,36, 70,33, 21,73, 16,32, 67,50, 31,47, 45,66,
  71,54, 44,58,  3,67, 23,38, 47,18, 49,36, 12,27, 34,26,
   8, 3,  1,46, 18,31, 56,72, 24,61, 52,68, 20,74, 79,31,
  27,72,  7,18, 15, 7, 28,78, 23,66, 15,47, 79,44, 78,11,
  28,18, 55,38, 68,48,  8,66, 38,69, 41,45,  8,39, 26, 7,
  64,78, 39,76, 41,50, 22, 6, 18,37, 64, 2, 38,11, 72,43,
  62,65, 69,40, 49,79, 45, 8, 10,15, 29,58, 67,43, 21,21,
  25,47, 37,43,  1,64, 43,28, 43,32, 79,10, 22,48, 42,16,
  20,58, 41,26,  5,79, 69,52, 30,73, 64,51, 63, 8, 67,61,
  41,70, 70,24, 17,73,  1, 5, 13,32, 57,39, 47,59, 22,57,
   0,57, 77,38,  2,41, 49,72, 19,27, 35,76, 65,22, 56,61,
  20,17, 11,62, 37,35, 29,42, 23,73, 16,49, 54,58, 50, 8,
  27,39, 24,19, 32,76,  9,70, 62,47, 48,34, 32,32, 36, 3,
  77,40, 42,42, 50,67, 40,39, 10,16, 20,15, 42,70, 42,53,
  67, 2, 28,61, 41, 2, 71, 7, 69,28,  1,69, 30,26,  8,57,
  39,67, 19,23, 52,78, 71,37, 22,46, 19,32, 16,68, 43,40,
   0, 2, 39,44, 10,25, 30,74, 52,30,  1,54,  7,61, 71,14,
  22,22, 12,24, 15,58, 41,74, 63,44, 75,42,  0,34, 75, 8,
   9,75, 70,20, 31, 4, 21, 5, 28,55,  9,62, 45,68, 37,41,
  49,74, 17,49,  0,13, 67,41, 54,68, 74,71,  1,57, 44,61,
   5,40, 48,45, 54,61, 56,16, 79,52, 44,77, 23,19, 57,63,
  45,28, 33, 7, 50, 7, 13,34, 21,30,  2,70, 38,75, 78,14,
  46,70, 10, 6, 18,47, 60,51, 39,66,  0,15, 66, 6, 19,25,
  47, 1, 62,11, 21,11, 31,24, 41,10,  7,42, 44,39, 71,67,
  19,51,  4,71,  4,27, 30,13, 74,69, 12, 1, 60,25, 37,48,
   6,20, 62,59, 26,61, 43,66, 54,51, 65,67, 35,44, 56,42,
  15,36, 66,56, 59,30,  4, 7, 12,26, 41,51, 24,50, 64, 1,
  11,56, 32,40,  6,54, 14,33, 39,71, 28,74, 18,27, 49,52,
  62,74, 60,74, 66,23,  2,68, 28,14,  0,77, 35,61, 29,55,
  72,20, 70,17, 44,33, 42, 4, 55,58, 48,30, 76,65, 62,76,
  44,12,  4,44, 50, 3, 39,75, 25,60, 73,48, 73,46, 26,45,
  77,44, 75,61, 61,23, 67,77, 50,46, 58,64, 34, 6, 19,58,
  14,35, 53,16, 64,15, 54,34, 53,40, 37,56, 64,44, 25,70,
  67,27, 12, 3, 13,63, 12,39, 38,48, 22,23, 79, 0, 54,38,
  71,74, 38,50, 43,13, 55,75, 79,53, 49,58, 21,60, 40,69,
  46,78,  9,79, 51,47,  5,54, 55,50, 28,29, 24, 4, 79,29,
  45, 7, 19,26, 30,17, 32,26, 42,69, 18,13, 22,40,  2,21,
  15,10, 65,18,  6,48,  5,37, 18,42, 58,12, 10,64, 53,27,
  14,73, 30,39, 41, 1, 73,27, 77,75, 11, 2, 23,79, 48,72,
  42,10, 44,31, 69,50, 38,22, 42,47, 61,13, 51,46, 77,18,
  29, 0, 67,39, 56, 3, 35, 4, 22,72, 54, 9, 53,19, 34,39,
  68,70, 64,17, 42,48,  5,14,  5,30,  9,52, 14,23, 51,60,
  34,69, 30, 9, 46,42, 29,67, 25,76, 46,52, 67, 5, 10,66,
  58, 2, 52,72, 78,46, 16,79, 63,62, 78,59, 10,73, 66,32,
  66,22, 70,68, 26,70, 11,27, 65,79, 11,63, 57,24, 68,27,
  31, 3, 24,11,  7,13, 32, 3,  6,33,  6,50, 62,35, 47,27,
  37,69, 78,76, 51,43, 51,25, 54,25, 20, 1, 22,70, 30,35,
   6,46, 66,76,  8,54,  1,52, 29, 5, 50,27, 14,56, 69,69,
  65,78, 28,35, 25,13,  6,37, 76,24, 47,68, 50,21, 70,43,
  18,38, 43,34,  3,46, 14,72,  7,12, 50,79, 13,50,  7,49,
  33,50, 29,31, 69,62, 40,33,  1,12, 51,74, 28,56, 44,79,
  65,48, 58,36, 73,58, 79,78, 48,44, 53,49, 45,78, 20,31,
  31,57, 15,51, 18, 4, 29,37, 13,19, 65,50, 29,38, 20, 7,
  33,57, 20,73, 60, 0, 45,57, 47,67, 61,71, 14, 4, 19,22,
  25,37, 78,71, 29,63, 39,37,  1,44, 69,29, 39,54,  6,19,
  56,76, 47,44, 69,68, 73,17, 27, 4,  5,34, 57,36, 54,27,
  67,28, 59,65, 15,40, 17,59, 15,64, 52, 2, 10,49, 17,78,
  14,32, 16, 8, 42,25, 76,13, 69,76, 26,38,  9, 5, 79,69,
   5, 8, 29,78,  3,41, 35,21, 43,39, 78,20, 64,33, 60,55,
  50,58, 27,75,  9, 8,  6,18, 15,39, 34, 7, 21,51, 72,79,
   0,17, 52, 9, 28,60, 22,78, 49,73, 26,10, 36,21, 26,75,
  67,79, 74,63, 18,55, 30,11, 62,48, 13,28, 66,68, 18,28,
   7,55, 30,14, 24, 7, 11,72, 12,51, 52,27, 27,45, 53,32,
  37, 1, 49,38, 14,40, 28,43, 20,18,  2,28, 45,11, 51,79,
  56,79, 43,76, 34,40, 35,30, 72,72,  0, 3, 66,59, 75,60,
  31,58,  6,65, 61,69, 31, 6, 48,64,  8,74, 15,79, 38,26,
  23,25,  2,33, 58,34, 57,43, 27, 2, 66,41, 37,28, 33,30,
  78,45, 71,23,  6,63, 40,54,  8,56, 16,23, 58,25, 20,38,
  53, 0, 56, 5,  5,53,  9,35, 25,21,  1,11,  2, 5, 59,76,
  25,71,  3,14, 29,41, 52,24,  3,36, 45,55, 36,69, 50,72,
  15,75, 74,77, 45,24,  3,58, 55,27, 34, 2, 43,72, 39, 0,
  47,35, 70,36, 57,65, 70,23, 77,66, 47,55, 10,77, 62,51,
  66,75,  9,65, 19,45, 24,34, 66,18, 45, 9, 16,35, 35,64,
  70,41, 65,62,  3, 0, 24,46, 35,12, 54, 6,  4,32, 37,18,
  66,61, 27,50, 73,79, 46, 1, 28,15, 33,33, 72, 3, 19,50,
  72,75, 54, 7, 75,12, 10,57, 64,67, 73,67, 33,42,  3,17,
  35,11, 62,26, 53,57, 50,64, 43,61, 45, 2, 52,45, 65,37,
  28,39, 66,50, 16,33, 37,11, 10,38, 28,38,  4,20, 74,43,
  20,37,  1,75, 42,59, 40,66, 11,13, 25,77, 31,26, 21,55,
   6,59, 59,69,  2,31, 68,45, 49,22, 65,74, 72,67, 16,74,
  48,10, 35,46, 14,21, 43,12, 44,36, 39,30, 60,57, 64,48,
  72,57,  7,33, 75,65, 29, 1, 42, 9, 61,66, 50,66, 14,47,
  19, 4, 74,32, 52,21, 25,63, 46, 7,  0,61, 32,42,  0,73,
  65,10,  9,16, 35,51, 77,68, 13, 9, 32,59, 13,47,  4, 3,
  47,42, 60,45, 73,31, 24,38, 63,78, 69,49, 22,28, 33,29,
  28, 6, 48,23,  1,39, 23,67, 13,71, 52,50,  0,26, 63,29,
  70,46, 35,66, 57,12,  0,46,  4,51, 14,64, 66,79, 14,39,
  65,60, 44,69, 36,23, 58,61, 59,67,  5,71, 56,15, 70,71,
  41,58, 14,36, 76,57, 60,68, 73,14,  0,72, 29, 8, 31, 0,
  37,23, 41,40, 47,23, 20,56, 79,28, 22,60, 28,28, 11,49,
  45,69, 18,50, 21,26,  8,32, 21,22, 60,10, 15,46, 33,66,
   2,75, 71,35, 73,73, 33,36, 21, 6, 46,59,  8,24, 65, 1,
   6,66, 10,32, 18,79, 68,10, 34,38, 79,64, 32,78,  9, 1,
  59,42, 17,70, 27,64, 38,57, 18,44,  4,75, 17,37, 21,78,
  49,34, 63,58,  4,31, 10,40, 64,49, 75,11, 17,15, 13,45,
  58,11, 66,47, 11,69, 13,20, 58,26, 66,44, 39,68, 29,23,
  29,53, 32,18, 73, 4, 22,54, 20,50, 17,31, 71,63, 38,12,
  77,16, 57,79, 43,41, 32,17, 48,61, 24,79, 17,32,  0,76,
  39,40, 41,65, 15,33, 42,71, 59,35, 52, 7, 40,65, 20,47,
   3,42, 14,13, 29,43,  7,50, 68,34, 63, 4, 55,54,  8,38,
  27,25, 74,57, 73,50, 50,47, 26,69, 47,77, 53,51, 70,60,
  40,19, 30,48, 24,37, 46,20, 20,49,  6,36, 53, 5, 44,76,
   5, 5, 77,20, 44,71, 46,31, 33,43, 49,67, 60,69, 12,66,
  28,21, 79, 8,  3,30, 74,42,  2,67, 29, 7, 79,15, 24,15,
  63,37,  8,50, 75,63, 66, 9, 40,61, 23,14, 44,22, 14,77,
  35,63, 78,21, 10,26, 14,22, 48, 8, 48,39, 68,14, 64, 0,
  19,55, 11,74,  1,10, 38,53, 64,50, 15,27, 35,43, 67,35,
  24,43, 66,71, 50,25,  9,41, 24,52, 39,72, 31,17, 76,20,
  61, 8, 38,38, 27,46,  2,48, 21,25, 70,50, 37,65, 16,41,
   0,20, 38, 2, 34,35, 68,19, 35,72, 46,79, 61,20, 69,26,
  62,42, 46,55, 15, 9, 39, 9, 65,30, 40,72,  9, 4, 24,57,
  15,24, 28,49, 48,33, 60,44, 48,69, 21,58, 30,77, 67,26,
   5,27, 70,65, 44,73, 20,77, 14,12, 54,12, 63,55, 68, 4,
  75,70,  9, 7, 74,26, 51,61, 62,14, 67,70, 30,20, 56, 9,
  36, 4, 70,52, 48,70, 64,73, 59,47, 25,72, 45,32, 62, 4,
  19,16, 76,52,  2, 7, 44,37, 50,10, 42,15, 48,54, 29,75,
  57,25, 64,12, 58,29, 50,29, 61,53, 65,33
};
const int p_80_4[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1190,
  /// Array of preferences
  23,67, 48,56, 31,25, 25,70, 59, 4,  8,20, 11,11,  1,45,
  51,33, 48,67, 17,40, 62,39, 14,59, 65,62, 43,48, 72,19,
  64,76,  0,79, 79,15, 53,35, 31, 2, 26,72, 34,16, 49,36,
  67,15, 58,53,  3,29, 13,69, 44,59, 34,28, 24,39, 78,26,
  28, 8, 28,17, 56,31, 66,26,  9,62, 76,34, 59,48,  1,26,
  22,42, 56,40, 20,70,  2,59, 49,16, 58,51, 55,66, 26, 5,
  27,20, 29,54, 30,70, 48,35, 50,13, 23,28, 62,49, 19,59,
  32,48, 46, 5,  2,79,  9, 9, 56,17,  4,60, 54,37, 35,41,
   8,57, 24,78, 21,38, 26,66, 26,64, 54,59, 38,62, 52,58,
  24,40, 36,37, 42,74, 43,45, 24,31, 36,57, 15,24, 15,68,
  38,69,  3,21, 12,22, 62,42, 77,35,  4,70,  3,40, 37,33,
  50,51, 21,76, 12,60, 65,66, 11,50, 53, 2, 68,46, 25,41,
  42,52, 20, 5, 69,47, 53,49, 23,23, 44, 6, 31,43, 39,12,
  39,30, 22,25, 24, 8, 40,61, 24,12, 40,72, 50,55, 65, 4,
  61,72, 27,69, 29,35, 15,78, 72,24, 58,61, 36,66, 57,63,
  55,44, 53,66, 68,13, 66,59, 18,37, 76,36, 74,37, 77,21,
  72,78, 76,61, 23,71, 75, 7,  3,35, 49,70, 77,62, 22, 1,
  72,44,  7,26, 15,51, 49,30, 67,66, 18,36, 31,71, 14,12,
  41, 7, 55,67, 22,37, 30,47,  3,58, 18,72,  3,56,  7,59,
  58,21, 55,43,  4,62,  3,55,  5,48, 79, 6, 44,33,  6,71,
   6,65, 48,33, 59,47, 72,65, 14,52,  4,48, 24,60, 61,33,
  34,58, 68,44, 39,28, 35,45, 32,78, 59,12, 31,13, 11,55,
   8,75, 34,48, 67,31, 20,33, 35,51, 46,42, 63,49, 69,52,
  16,15, 79,35, 33,68, 27,68, 29, 1, 35,19, 41,26, 27,73,
   3,28, 19,23, 44, 3, 63,35, 68,33, 66,65, 47,41, 66,39,
  17,64, 33,77, 55,32, 76,15,  0,68, 21,27, 56,48, 53,47,
  66, 7, 70,39, 29,27, 20,69, 24,66,  2,62, 62,69, 29,79,
   0,27, 13,41, 38,61, 65,58, 64,72, 65, 8, 51,55, 58,27,
  30,51, 33,22,  3,13, 79,64,  9, 5, 27,70, 59,71, 28,63,
   8,38, 59,49, 63, 2, 53,74, 65,63,  7,16, 22,39, 76, 4,
   9,43, 48,61, 37, 3, 61,13, 21, 7, 23,10, 25,54, 58,76,
  66,35, 10,43, 71, 8,  9,72, 42,34, 12,79, 44,66, 76,44,
  78,77, 62,64, 63,16, 50,31, 36,65, 27,22, 35,58, 13, 1,
  50,58, 26, 0, 54, 8, 17,11, 34,20, 18,43, 26,46, 21,19,
  64,77,  2,40, 34, 6, 22,66, 10,26,  5,69, 73,63, 59,29,
  36,76, 64,44, 41,21, 77,27, 78,75, 52,73, 46,62, 25,60,
  38,39, 36,23, 42,26, 57, 2, 42, 6, 32,33, 70,49, 20,62,
   5,50,  2,24, 51,63, 13,35, 71,11, 73, 9,  2,10, 35,42,
  43,33, 22,69,  9,29, 10,48, 46,15, 50,65,  6,25, 74,70,
  42,71, 37,45, 35,39, 21,44, 30,50, 11,45, 18,60,  3,44,
  57,52, 23,78, 19,24, 61,68, 51,19, 21,54, 20,72, 65,44,
  39,43, 27,28, 11,38, 27,51, 12,73,  4, 6, 53,75, 47, 9,
  38,70, 65,68, 18, 5, 73,34, 44,76, 53,53, 34,64, 36,41,
  56,51, 48, 8, 22,56, 34, 7, 24,74, 62, 3, 60,28, 21, 5,
  21, 2, 13, 0, 46,59, 27,34, 61,36, 45,71, 47,36,  5, 1,
  13,70, 77,78, 31,66, 49,73, 69,14, 17,36, 54, 0, 24,18,
  52,26, 69,66, 54,43, 36,35, 61,54, 66,71, 77,24,  4,28,
  26,56, 78, 2, 44,32, 29,43, 30,25, 42,25, 38,52, 77,41,
  72,14, 15,63, 11, 6,  8,50, 71,31, 66,49,  4,18,  7,52,
   3, 3, 38, 4, 41,11, 28,67, 13,48, 28, 7, 70,10, 38,74,
  32,60,  8,34, 52,11, 55, 0, 66, 9, 47, 4, 51,62, 43,68,
  78,43, 63,44, 72,50, 39,73, 42,19, 34, 1,  6,67, 60,51,
  28,56, 41,55, 19,72, 69,38, 48,78, 65,23, 26,61, 39,60,
  65,20, 30,33, 64,35, 14,40, 24,65, 54,67,  5,22, 35,33,
  12,10,  5,54, 67,53, 37, 1, 58,31, 69,69, 22,51, 32, 6,
  20,30, 21,41, 34,46, 66,13, 18,71,  8,45, 53,57, 66,32,
  27,53, 14,38, 25,22, 45,67, 51,52, 64,61, 28,75, 15,74,
  15, 8, 78,37,  1,15, 60,11, 43,67, 31,75, 64,33, 70,12,
  47,10,  3,39, 63,63, 38,17, 34, 9, 56,43,  6,48, 62,51,
  13,36, 70,55, 25,25, 54,26, 58,33, 27,72, 57, 5,  2,51,
  70,53, 19,37, 32,68,  2,69, 26,67, 60,62, 37, 5,  1,40,
  76,56, 79, 4, 66,21, 38,30,  5,27, 34, 5, 47,75, 41,47,
  28,27, 47,48, 51,17, 55,61, 19,10, 17,56, 17,58, 37,26,
  47,46, 19,38, 39,39, 74,71,  8,64, 40, 4, 74,52, 74,56,
  19, 2, 52, 7,  3,67, 25, 9, 75,35, 52,54, 27, 2, 43,15,
   4,72, 45,36, 37,44,  7,58, 68,61, 47,70, 50,77,  3,73,
  48,64, 22,50, 70, 7, 34,67, 74,44, 40,74,  8,14,  1,10,
  42, 4, 56,68, 22,44, 18,45, 21,67, 19,29,  6,15,  5,19,
   7,42, 23,73,  8,13, 62,35, 13,45, 74,33, 36,67, 60,26,
  51,22, 31,72,  7,70,  4,73, 60,35,  0,57,  7, 3, 14, 7,
  47,57, 70,64, 29,37, 56,18, 14,54, 78,72, 34,13, 77,46,
  22,71, 42,23, 32,47, 49,42, 73,31, 67, 0, 77,31, 73,10,
  26,68, 29,18, 71,35, 54,65, 69,35,  4,50, 39,23, 36,48,
  18,69, 29,13, 28,55, 52,36, 41, 6, 75,44, 51,24, 76,11,
  49,55, 53, 5, 59,79, 12, 7, 65,16, 33,44, 36,25, 23,48,
  55,27, 30,23, 39, 3, 35,18, 70, 2, 25, 0, 23, 8, 54,20,
  10,71, 67,29,  0, 3, 46,48, 68,30, 44,40, 46,56, 73,27,
  35,70, 28,52,  1,58, 72,58,  9,30, 67, 6,  3,32,  0,78,
   8, 1, 13,29, 37,57,  5,24, 62,56, 68,12, 15,20, 15,40,
   5,28,  7,64,  5,49, 50,61, 11,64, 19,18, 50, 4, 39, 9,
  46,60,  0,23, 67,34,  0,53, 63, 3, 52,64, 18, 7, 79,58,
  38,54, 70, 6, 57, 1,  8,26, 72, 1, 14,16, 67,63, 65,14,
  71,60, 50, 6, 13,50, 35,16, 67, 4, 50,48, 11,63, 66, 0,
  12,72, 31,14, 61,62, 45,79, 18,11,  8,59, 21,22, 77,59,
  14,57, 14,17, 36,27,  5,13, 24,36, 63,19, 44, 7, 74,12,
  57,78, 47,76, 32, 8, 48, 6, 75, 1, 76,20, 47,40, 43,76,
   7, 1, 57,39, 24,50, 30,59, 60,38, 56,33,  0,51, 30,26,
  75,48, 26,60, 12,48, 74,41, 25,62, 42,77,  6,23, 27,78,
  45,58, 78,45, 59,52,  1,55,  7,25, 69,41, 12,51, 59,77,
  79,46, 41,53, 20,18,  4,46, 51,67, 47,30,  6,44, 62,36,
  62,13, 36,22, 17,72, 73,45,  1,49, 78,52, 71,18, 77,34,
  68,37,  1,34, 52,21, 35,10, 10,29, 28,16, 69,34, 40,17,
  27, 3, 28,22,  5,77, 70,71, 13,54,  7,65, 16,52,  0,36,
  57,32, 67,14, 52,44, 18,57, 27,18, 43,25, 35,48, 52,27,
  78,20, 24,14, 54,41, 49,60, 50,74, 31,42, 46,31, 34,24,
  16,24, 32,21, 19,73, 71,78, 40,32, 41, 5, 72,67, 20,53,
  63,32, 74,45, 15,23, 56,73,  3, 0, 18,23, 58,34, 24,33,
  35,37, 45,66,  2, 8, 19,55, 63, 8, 29, 8,  1,31, 79, 7,
  76,76, 49,56, 17,34, 64,71, 55,46, 79,67,  6,49, 47,33,
  79,39, 17,69, 48, 7,  8, 4, 47,65, 50,54, 31,21, 27,75,
  59,51, 23,59, 79,31, 49,31,  5,41, 49,61, 20,77, 58,77,
  52,48, 26,49, 23, 7, 13,46, 53,50, 50,52, 19,34, 35,68,
   9, 8, 77,32, 55,51, 46,25,  0, 8, 66,72, 69,48, 43, 1,
  22,72, 14,53, 29,64, 58,58, 78,25, 11,35, 34,57, 72, 7,
  30,61, 35,71, 19, 3, 68,43, 57,12, 35,22,  6, 6, 23,19,
  61,47, 52, 8, 16, 3, 36,46, 41, 3, 25,30, 12,36, 56,49,
  64,78, 77, 5, 24,69,  9,61,  1,21, 44,19, 20,45, 55,54,
  12,77, 24, 3,  0,60, 51, 7, 61,50, 12, 5, 40,71, 47,60,
  31,53, 61,65, 65, 5, 60,16, 29,65, 77,37, 32,11, 79,77,
  52,39, 19,49, 73,59, 13,57, 69,54, 45,53, 57,26, 65,76,
   4,10, 16,48, 14,44, 22,31, 50,38,  4,20, 40,28, 13,52,
  77, 9, 26,75, 34,27, 11,30, 53,41, 39, 1,  6,17, 37,34,
  62, 6, 44,41, 22, 2, 27,10, 49,22, 54,72,  2,45, 12,28,
  25, 1, 20,57, 72,28, 11,71, 42,53, 25,26,  8,61, 44, 8,
  22,62, 74,50, 74,57, 54,55, 63,55, 26,33, 69,76, 15,17,
  44,69, 64,32, 31,76,  9,48,  8,33, 11,75, 54,47, 37, 6,
  13,56, 68,65, 12,50, 12,58, 18,56, 23,74, 59,75, 46,49,
  10,20, 33,49,  2,22,  8,12, 55,37, 52,22, 66,29, 56,76,
  21,16,  9, 1, 78,79, 16,21, 67,71, 12,42, 32, 3, 30,69,
  50, 2, 67,46,  8, 2, 29,71, 38,43,  1, 5, 28,25, 33,34,
  73,23, 74,27, 29, 4, 59,33, 47,22, 20,58, 64,14, 69, 5,
  25,19, 32,39,  1, 7,  8,76,  6, 4, 62,66, 31,45, 23,35,
  15,61, 42,72, 52,42, 62,31, 43,50, 65,12, 28,14,  5,79,
  56,23, 65,51, 30,58, 10,63, 56,26,  5,67,  6, 7, 32,29,
  18,29, 29,68, 67,67, 19,75, 53,70,  9,73, 22,14, 69,31,
  56,77, 49,23, 56, 0, 77,57, 67,45,  8,73, 27,40, 37,53,
  73,69, 37,52, 63,15, 65, 7, 22,20, 54,19, 53,73, 78,29,
  42,66, 15,43, 78,42, 26,43, 55,63, 71,69, 59,36, 14,19,
   9,36, 28,73, 48,41, 20,36, 50,28,  1,69, 22,60, 43,23,
  55,19, 79,38, 51,59, 63,45, 65,27, 73, 1, 49, 2, 69,40,
  74, 8, 40,67,  5,45, 27,59,  7,39, 70,54, 78,68, 77,30,
  15,72, 35,25, 45,51, 29,60,  0,30, 40,13, 58,25, 38,50,
  12,74, 58,24, 67, 5, 58,43,  8,30, 36,53,  4,29, 32, 4,
  44,39, 62,18, 15,57, 76,33,  9,10, 52,34, 17,28, 26,58,
  66,11, 71,17, 47,39, 60,67, 15,44, 62,21,  0,46, 47,54,
  32,57, 69,75,  9,19, 10,56, 44,16, 15, 5, 38,19, 63,33,
  38,60, 28,71, 22,79, 66,51, 54,42, 26,65, 37,48, 11,42,
  37, 2, 51,68, 76,52, 55,59, 67, 2, 53,37, 11,18,  1,77,
  25,40, 73,73,  5,57, 63,78, 55, 4, 59, 3, 24,35,  0, 0,
  44,54, 65,64, 68,40,  4,15, 13,47, 69,16, 63,50, 31,56,
  52,18, 39,38, 55,49, 28, 5, 62,68,  5,20, 32,49, 43,32,
  13,65, 15,60, 21,68, 47,11, 25,69, 45, 7, 29,41, 56,53,
  10,60, 12, 4, 60,76, 24,20, 50, 3, 34,31, 23, 1,  1,12,
  77,47, 12,67,  7,68, 24,75, 42,79, 12, 8, 20,17, 62,11,
  76,53, 42,69, 63,74, 48,37, 13,10, 30,75, 26,54, 41,25,
  23,68, 76,67, 31,37, 21,71, 14,28, 73,28,  3,51, 45,74,
  28,33, 73,35, 23,62, 16,36, 62,54, 54,61, 18,15, 70,17,
  45, 4,  3,57, 69,65,  2,31, 22,43, 53,39, 77,60, 14,66,
  68,71, 14,61, 26,21, 33,21, 62,52, 18,76
};
const int p_80_5[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1401,
  /// Array of preferences
  59,57, 13,55, 27,31, 21,48, 78,40, 51,49, 23,39, 20, 8,
  68, 2, 64,41,  5,72, 29,63, 52,33, 30,30, 71,10, 79, 5,
  10, 9,  9, 9, 19,47, 79,25, 48, 8, 48,27, 69,79,  5, 5,
  41,39, 18,16, 55,26, 63, 8, 24,75,  6,50, 51,69, 19,58,
  56,45, 57,49, 79,20, 45,16, 37,25, 24,28, 58,29,  5,53,
  29,42, 58,68, 15, 6, 70,77, 64,43,  4, 5, 54,52, 63,44,
  75, 5,  0,60, 73,25, 20,43,  7,47, 30,44, 78, 2, 56,57,
   4,25, 52,70, 38,20,  8,47, 78,62,  4,78, 22,37, 67,51,
  13,72,  2, 0, 54, 5, 52,19, 20,24, 26,52, 12, 1, 41,68,
  78,74, 41,21,  3, 3, 45,10, 48,28, 45,49,  6,13, 10,49,
  72,49, 15, 8, 69, 1, 34,45, 17,28, 36,37, 26,78, 68,43,
  75,19, 53,47, 62,60, 56,39, 14,19, 10,23, 39,72, 11,69,
   6,73,  0,76,  9,25, 28,53, 29,22, 79,79,  3,30, 60,56,
  23,37, 53, 3, 43,62, 38,33, 69,55, 56,67, 67,57, 79,54,
  61,55, 36,71, 25,68, 23, 7, 29,35, 12,31, 52,43, 74,71,
   8,42, 52,55, 50,58, 41,72, 35,12, 53,31, 52,42, 31,29,
  61,31,  4,27, 15,29,  8,52,  0,70, 43,68, 11,64, 78,24,
  18, 2, 50,23, 65,79, 14,71, 46,74,  3,26, 21,37, 64,21,
  13, 0, 32,17, 20,54, 34,30,  0,31, 65,33, 32,26, 23,18,
   6,22, 55,57, 28,19,  9,41, 17,38, 61,22, 48,53, 47,19,
  17,78, 58,73, 50,55,  4,26, 47,56, 45,77, 57,25, 51,65,
  71, 7, 77,70, 72,10, 69,72,  0,14, 53,52, 63,15, 45,53,
  42, 1, 42, 3, 59,52, 12,47,  2,63, 16,16, 54,75,  3, 7,
  59, 0, 16,57, 46,59, 13, 3, 63,74, 31,42, 58,76, 70,52,
  37, 6, 34,54, 51,17, 39,37, 55,61, 28,35,  3,34, 21,58,
  23,12, 59,12,  9,32, 48,72, 11,21, 55, 8, 67,42, 68,51,
  36,38, 31,12, 48,30, 56,22, 47,70, 11,71, 35,46, 75,16,
  41,23, 72,71, 13,16, 44,38, 74,44, 75,43, 42,26, 48,12,
  47,51, 25,70, 50,26,  8,28, 57,12, 68,37, 34,47, 65,48,
  26,27,  7,46, 40,71, 77,16, 76,49, 67,71, 71,78, 36,16,
  29,56, 23,36, 63, 3,  0,61, 12,62, 42,36, 12, 6, 36,52,
  42,57,  0,45, 45,14,  6,27, 40,12, 54,20, 79,44, 62,39,
  21, 5, 48, 5, 18,11, 22,59, 77,58, 52,49, 64,68, 36,76,
  67,59, 32,67, 44,59, 36,65, 42,16, 77,30, 35,70, 25, 8,
  15,19, 65,57, 21,26, 65,44, 71,45,  3,27, 13,62, 36,70,
  13,54, 28,42, 61,37, 76,67, 34,46, 73,24, 50, 7, 19,31,
  26,54, 70,71, 36, 6, 41,19, 44,52, 38,38,  8,14, 51,12,
  23,45, 62,22, 39, 9,  4, 6, 42,55,  8,40, 31,25, 60,63,
  34,72, 21,64, 61, 5, 24,65, 50,14, 23,40, 33,77, 65,30,
  43,25,  2,74, 18,77,  2,45, 17,37, 33,44,  2,25, 55,75,
  29, 5, 51,43, 48,21,  1,26, 49,54, 75,18, 53,64, 25,65,
  18,42, 19, 3,  3,40, 66,15,  2,17, 19,53,  1,21, 56,59,
  34,77, 73,18, 53,45, 68,70, 63, 9, 27, 8, 56,60, 70,34,
  12,67, 30, 6, 44,14, 66,62, 28,60, 40,44,  7,26, 53, 1,
  38,46, 40,24, 31,74, 79,55, 60,16, 34,21, 16,79, 34,14,
  45,51, 72,61,  3,71, 66,52, 69,52, 17,26, 36,24, 36,28,
  65,34,  0,68, 66,39, 70,39, 38,23, 55,45, 10,79, 78,79,
  74,67, 24,37, 42,75, 36,41, 57,61, 69, 0, 63,11, 12,48,
   8, 1, 68,63,  4,50, 57,75, 21,12,  5,39, 69,53, 69,61,
  31,47, 13,32, 16,53, 15,69, 65,22, 61,60, 37,67, 33, 0,
  54,62,  3,79, 19,51, 69,24, 78,69, 64,49, 50,69, 12,40,
  65,40, 48,39, 72,68, 33, 8, 36,35,  5,23, 22,35, 67, 0,
  38,76, 45,30, 64,19, 53,16, 47,77, 18,73, 31,59, 59,21,
  34, 7, 26,49, 68, 1, 38,10, 17, 8, 57,21, 24,29, 20,10,
  72, 0, 67,60, 30,10, 65,56, 65,66,  9,73,  2,16, 17,40,
  41, 3, 69,34, 60,65, 14,21, 24,13, 38,14, 63, 2, 14,13,
  69,17, 69,59, 75,31, 45,17, 10,46, 43,55, 45,39, 76,25,
   0,15, 69,23,  8,29, 22,16, 73,59,  7,21, 32,77, 57,37,
  26,77, 63,77,  1,52, 63,62, 77,31, 52,29, 49,72, 17,25,
  50,36,  4,52, 15,62, 53,28, 50, 6,  9,14, 29,24, 32,15,
   1,64, 67,54, 51,41, 32,12, 76,38, 52, 5, 45,54, 58, 6,
  30,49, 60,71, 71,47, 66,73, 56,36, 12,53, 21,52, 48,33,
  22,27, 45,52, 35,39, 30,74, 67,72,  4,58, 14,50, 39,70,
  66,17, 76,30, 56, 4, 57,74, 50,42,  3, 2, 57,76, 59,30,
  72,57, 10,34, 54,12, 54,30, 70,53, 35, 6, 12,38, 71,25,
  50,12, 62,79, 29,49, 71,14, 63, 7, 58,78, 63,16, 43,37,
  73, 1, 23,77, 11,51, 39,30, 24,39,  7,39, 41,58, 37,55,
  74,24, 24,24, 76,18, 66,64, 79,14,  0,57, 32,19, 45,25,
  21,11,  1,37, 45,56, 65,63, 52,13, 59, 5, 75,56, 48, 3,
   8,37, 39,17, 41,38, 78,54, 33,43, 37,66,  4,22,  2,58,
  65,26,  7, 5,  0,44, 36,56, 51,77, 72,29, 11,29, 29,55,
  73, 5, 17,65, 74,54, 62,72, 26,48, 60,26, 49,44, 72,54,
  22, 5, 15, 4, 35,56, 16,11,  4,29, 28, 4,  1,69, 60,22,
  13,29, 22,19, 18,22,  2,11,  4,12, 24,46, 41,10, 44,61,
  46,11, 69,22, 37,60, 46, 8, 45,47, 31,67,  9,33, 59,15,
  31,32, 46,45, 46,50, 68,22, 59,45, 45, 7,  7,33, 77,66,
   8,76, 67, 5, 59,66, 34,43, 38, 5, 65,16, 78,57, 57,47,
  65,62, 14,63, 34,17, 16, 6, 46,51, 72,26, 67, 6,  2,62,
   4,33, 73,43, 49,45, 34,59, 16,10, 79, 3, 23,34, 28,46,
  59,59, 65,72, 28,58, 68,41, 42,51, 74,28, 49,48, 47,25,
  19,50, 79,23, 21,27, 55,76, 51,14, 71,18,  3, 4, 17,59,
  25,71, 17, 7, 20,69, 43,17, 23,74, 59,16, 48,17, 47,34,
  75,73, 70,65, 58,66, 26,69,  9,78,  5,35, 67,24, 51,19,
  58,18, 30,78, 67,20, 39,52, 44,30, 37,22, 59, 4, 67, 1,
  20,78, 20,21, 73, 0, 68, 6, 60,13, 63,19, 12,58,  3,44,
  23,61, 70,68, 42,77, 70,47, 39,71, 54, 1, 50,79, 26,57,
  39,27, 66, 5, 26,31, 78,11, 39,63, 78,26, 23,68, 35,48,
  16,61, 37, 9, 55,41, 79,30, 77,39, 45,23,  9,28, 19, 5,
   1,22, 39,21, 28, 1, 79,36, 65,32, 14,40, 50,32, 20,12,
  18,60,  7, 9, 44,45, 78,22, 29,54, 20,49, 72,59, 40,74,
  41,60, 48,29, 24,58, 12,12, 57,19, 60, 2, 77,76, 18,20,
  42,54, 57, 1, 21,71, 64,59, 67,52, 28,15, 28,32, 15,17,
  12,73, 57,29,  9,21, 57,63, 40,23, 36,74, 28,62, 35,20,
  54,28, 41,66, 26,56, 49, 7, 75,41, 55,37, 11,54, 47, 8,
  28,50,  8,21, 37,10, 46,25, 69,77, 43,63,  5, 7, 38,32,
  30,39,  3,11, 34, 0, 30,16, 64,34,  6,18, 78,63, 55,42,
  63,40, 77,46, 44,31, 48,18, 48,23,  8,51, 42,41, 44,51,
   6,71, 13,58, 68,19,  4,21, 58,11, 12,75, 40,11, 14,54,
   6,75, 45,67, 72,58, 48,69, 65,46, 15,50,  2,67, 23,75,
  75,63, 65,12, 40, 5, 73,69, 36,68, 54,10, 43, 5, 25,67,
  57,35, 16,27, 40,79, 73,48, 43,67, 64,65, 51,72, 35,11,
  14,61, 43,18, 28,26, 11, 3, 68,15, 75,69, 38,29, 58,59,
  38,59, 67,16, 56,44,  7,71, 25,31, 14,24, 54,46, 30,75,
  47,69, 57,71, 15,55, 34,62, 64,78, 15,14, 15,61, 66,63,
   7,50, 69,36, 77,61, 10,64, 19,36, 32,29, 56,30, 77,27,
  65,20, 62,37, 73,31, 76,48, 29,37, 23,50, 46,46, 42, 0,
  56, 3,  6,60, 21,44, 66,45, 35,36, 30,59,  6,57, 79,28,
  22, 1, 26, 3, 78,12, 25,64, 62,73, 16,74, 31, 9, 12,16,
  33,12, 41,25, 47, 6, 19,33,  4,30, 47,76, 28,67, 24, 8,
  36,32, 26,20, 47,59, 21,38, 35,78,  3,46, 31,23,  4,32,
  58,10, 15,43, 53,73,  0, 1,  3,22, 72,55, 49,50, 48,54,
   8,45, 70, 2, 70,35, 19,65, 72,42, 42,40, 21,43, 73,52,
  32,30, 79,58, 41,34, 71,71, 60,44, 79,57, 18,70, 77,13,
  31,33, 66,58, 39,42, 73,66, 74,74, 67,30, 46,49, 26,40,
  33,34,  7,66, 22,20, 38,47, 55,72, 61,35, 16,31, 30,76,
  29,21, 39,39, 41,32,  1,66, 59,60, 59,71, 27,76, 74,43,
   6,47, 61,65, 31,17, 41,43, 26,51, 11,44, 16,56,  9,79,
  22,14, 48,65, 30, 5, 35,74, 60,39, 61,10, 16,58, 24,57,
   4,56, 32,60, 35,26, 69,73, 21,50, 20,25, 20,46, 66,41,
  74,64, 64,24,  2,77, 78,34, 29,12, 19,18, 59,65, 25,30,
  38,70, 33,69, 63, 1, 53,60,  2,19, 17,34, 35,64, 78,28,
   5,51, 59,33, 22,39, 42, 4, 53,34, 79,64, 17,55, 33,24,
  56,55, 62,75, 65, 7, 26,50, 72, 7, 32,74, 59,29, 78,23,
  41, 6, 56, 6, 60,70, 62, 0, 66, 0, 78,51, 42,68,  7,10,
  70,37, 39,15, 19,19, 65,52, 53,55, 71,12, 43,12, 38,58,
  17,12, 55,20, 49,10, 42,14, 77,25, 53,53, 14, 7,  4,65,
  64,25, 27,17, 12, 5, 34,58, 69,76,  0,33, 58,34, 13,57,
  66,21, 31,75, 37, 5, 33,42, 40, 6, 20,44, 69,44, 31,60,
  74,77, 28,17, 19,63, 55,10, 78, 8, 29,71, 21,61, 42,50,
  75,29, 46, 2, 74,12, 46,18, 37,37, 21,63, 58,38, 64,12,
  69,28,  3,60, 68,16, 26,74, 28,38, 36, 4, 51,21, 12,65,
  41,76,  0,20, 50,54, 73,12, 57, 3, 24, 4, 78,17, 74,65,
  57, 8, 12,43, 77, 2, 76,16, 67, 4, 43,32, 15,23, 43,71,
  78,53, 78,18, 10,74, 64,70, 40,67, 10,53, 68,47, 36,64,
   1,48, 19,54, 27,77, 51,74, 71,49, 45, 1, 43,46, 79,70,
  41,75,  5,57, 55,65, 11,47, 74,31, 30,27, 52,31, 64,16,
  59,39, 76,63, 54,78, 44,70, 18,48,  2,54, 55,59, 62,58,
  47, 2, 19,32, 51,10, 21,60,  6,35,  8,74, 40,54, 34,56,
  19,69, 38,51, 69,43, 61,15, 19,66, 19,16, 63,52,  9,47,
  17,31, 45,27, 64,26, 49,59, 53,40, 12,30, 65,15, 63,45,
  74,50, 12,79, 20,62,  0,66, 74,25,  6, 5, 56,10, 52,69,
  36,59, 51,56,  0,69, 39,40, 18,40,  3,47,  2,40, 14,78,
  46,22, 69,68, 73,10, 17,24, 73,15, 79, 0, 51,22, 77,75,
  20,36, 25,54, 31,71, 10,61, 44,16, 29,59, 26,44, 30,25,
  15,33, 58, 7, 35,42, 71,52, 47,58, 79,47, 23,13, 26,19,
   1,45, 17, 2, 50,38, 18, 5, 73, 2,  2,73, 34,75, 77,26,
  74,45, 10,18, 47,36, 65,77, 29,13, 17,16, 14,39, 22,51,
   7,62, 14,18, 79, 1, 77, 0, 60,79, 22,18, 22,73, 12,28,
  47,37, 30,61,  7,43,  7,31, 18,72, 46,65, 35,45, 41, 5,
   2,52, 34,73,  1,27, 30,63, 43, 0, 43, 2, 35,77, 13,59,
  67,25, 65,43, 33,19, 12,35,  1, 6, 59,31, 43,31, 46,19,
   1,16, 38,17, 37,20,  7,20, 26,10, 39, 3, 55,13, 77,37,
  38,74, 65,19, 48,36, 18, 4, 28,33, 76,31, 64,15, 20,53,
  11,72, 42,11, 72,16,  1,53, 39,53, 68,44, 19,60,  1,78,
  73,53, 40,40, 36,66, 32, 8, 71,60, 58,77, 66,34, 28,29,
  15,51, 47,33, 41,22, 15,13, 54,31,  8,36,  4,79, 40,59,
   8,39,  8,55, 55,69,  0, 5, 72,37, 55,19,  3, 6, 32,72,
   6,44, 75,66, 77, 3, 37, 3, 31,77, 23,33, 14,41, 53,14,
  60,58, 21,29, 25,66, 72,34, 74,60, 62, 7, 19,23,  6,21,
  22,29, 66,61, 50,34, 14,26,  8,17, 45,60, 53,22, 26,25,
  60,55, 32,70, 50,68, 40,50,  8, 7, 22,22, 54,17, 57,70,
   2,31, 46,79, 77,55, 29,29, 32,59, 54,37, 28,40, 17,67,
  46,36, 65,28, 49,35, 24,72, 42,31, 50,22,  1,31, 33,10,
  14, 5, 18,24, 41,16, 24,23, 53,24, 74, 7, 12,22, 37,47,
  79,35, 66,69, 37,16,  1,62, 61,17, 18,18, 76,41, 64,23,
  76,10, 75,32, 14, 1, 66,71, 49,19, 39,14, 23,41, 78,43,
  10,30, 28,70, 20, 3, 70,27,  2, 5, 16, 3, 61,23, 59,64,
  32,54, 49,33, 55,17, 14,34, 33,53, 24, 6, 63,63, 38,60,
  23, 3, 24,69, 29,46, 35,66, 79,12, 15,59,  1,10, 19,76,
  43,15, 74,72, 52,44, 46,47, 78,56,  6,79,  6,59, 40,41,
  73,78, 67,18, 42,29, 23,38,  7, 2,  3,56, 71,70, 69,70,
  58,42, 11,75, 20,47, 20,67, 63,42, 37,76, 47,48, 24,64,
   4,47, 10,62, 27,79, 77,41, 29,38, 47,20, 78,21, 11,53,
  37,45, 30, 1, 38,50, 63,46, 67,10, 75,45, 58, 8, 55,67,
  54,34, 28,13, 44,75, 42,21,  2,79, 11,22, 26,59, 35,21,
   7,49
};
const int p_80_6[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1453,
  /// Array of preferences
   5, 2, 31,77, 10,28, 76,66, 36, 8, 24,24, 14,12,  5,48,
  34,19, 62, 6, 67,78, 62,62, 56,59,  1,79, 79,73, 33,15,
  73,53, 67,11,  9,37, 36,62,  3,75,  9,12, 44,72, 51,49,
  25,57, 71,16, 45,72,  9,39, 73,34, 29,45, 54,10,  7,43,
  32, 8, 49,79, 64, 0, 47,35, 23,25, 27,74, 70,19, 60,29,
  42, 2, 32,59, 66, 3, 66,54, 40,21, 43,22, 70,74, 60,21,
  49,78, 16,64,  7,30, 43,51, 18,47, 67,71, 69,25, 21,34,
  63, 4,  0,59,  0,40, 79,29, 57, 5, 55,16, 28,33, 34,41,
  50,54, 40,46, 16,60, 32, 2,  1,53, 74, 8, 40,25, 57,30,
  76,63,  9,58,  9,63, 41,76, 48,31, 75,58, 64,48, 35,63,
  19,25, 14,42, 59,47,  8, 3, 65, 2, 67,55, 62,44, 69,54,
  31,47,  7,68, 74,12,  6, 6, 59,43, 65, 1, 53,52, 56,11,
  71,50, 11,72, 63,78, 45,18, 15, 9,  2,78, 64,44, 18,74,
  72,17,  3,39, 77, 0,  5,22, 58,79, 76,24, 30,76, 52,14,
  43, 2,  4,51,  3,43, 41,48, 78,58, 51,20, 36,48, 49,16,
  47,66, 54,29, 38,41, 37,40, 73,45, 15,24, 53,16, 40,63,
  31,38, 10,53,  4,12,  3,77, 40,61, 39,73, 13,36, 67,65,
   9, 0, 50,65, 51,14, 17,63, 79,37,  1,60, 23, 2, 52,77,
  72, 8,  2,21, 73,18, 17,54, 46,57, 71,66, 69,70, 20,39,
  46,37, 51,11, 68,12, 69,49, 13,75, 76,61, 72,10, 70,78,
  22,65, 10,33, 30, 6, 31,65, 67,35, 45,45, 57,16, 21, 8,
  38, 4, 79,21, 16,73, 12,72, 63,75, 48,24, 11,55, 40,48,
  24,30, 46, 1, 35,67, 23, 5, 52,69, 65,27, 58,23, 35,31,
  17,52, 28,36, 49,12, 26,28, 68,73, 31,30, 30,22, 19,14,
  50,50, 54,51, 61,61, 74,37, 73,64, 60,76, 47, 5, 10,72,
  37,29,  5,37, 35,53, 67,29, 24,61, 23,35, 46,71, 54,12,
  25,15, 20,18, 28,14, 10,22, 20,78, 29,26,  0,22, 46,62,
  13,70, 44, 1, 57,11, 23,67, 64,53,  0, 7, 59,30, 37,57,
  64,66, 64, 3, 35,20, 37,56, 25,69, 30,40, 22,17, 43,26,
  65,49, 14,64, 32,45,  6,76, 28,76, 14,75, 31, 3, 79,56,
  31,63, 10,60, 28,29,  2,65, 66,28, 38,17, 21,64, 17,55,
  52, 5, 41,10, 74,75, 10,12, 41,38, 19,35,  8, 6, 50,35,
  65,79, 56, 6, 47,50, 27,17, 34,26,  7, 7, 74,32, 72,50,
  75,53, 46,65, 43,57,  3,42, 26, 3, 50,17, 36, 7, 61,54,
  62,48, 44,16, 30,24, 10, 4, 30,27, 18,50, 44,74, 20,75,
  78,22, 25, 8, 37,50, 40,75, 26,79, 13,68, 79,51, 38,69,
  54,21, 57,51, 61,36, 24,40, 20,70, 35,45, 77,29, 72,42,
  39,17, 73,44, 42, 0, 78,26, 78, 2, 71,54, 67,56, 21,59,
  65,72,  5,28, 35,79, 40,45, 78,45, 32,42, 24,76, 31, 9,
  20,59,  4,17,  0,33, 63,13,  2,68, 53,43, 37,11,  3, 7,
  70,31,  2,48, 75, 4,  1,65, 14,36, 57,37, 60,27, 30,25,
  31,71, 20,57,  2,31, 55,59, 64,75, 40, 3, 61,77,  4,66,
  12,48, 18,65, 79, 4, 61,28, 50,70, 20, 9, 49, 2,  3, 6,
  20,23,  3,13, 65,52, 61,26, 27,15, 70,65, 18,54, 30,59,
  24,66,  2,27, 72, 2, 58,19, 31,42, 44,27, 14,72, 15,55,
  39,44, 27,19, 11,47, 75,54, 51, 3, 53,54, 64,16,  4,19,
  79,38, 20,32, 54, 0, 70,14, 74, 0, 35, 1, 53, 9, 19, 6,
  37,73, 52,62,  1,27, 28,60, 42,67, 17,47,  8,58, 36,32,
   5,32, 22,30, 40,31, 79,16, 34,15, 24,10, 70,42, 55,42,
  68, 4, 11, 6, 46,23, 41,63, 60,41,  1, 8,  5,29, 66, 6,
  46,58, 14,46,  6,54, 54,34, 36,38,  8,67, 23,64, 14,17,
  79,58,  7, 3, 45,35, 21, 9, 58,49, 40,65, 12,79, 65,61,
  33,56, 71,17, 62,63, 26,32, 60,58, 40,78, 57,59,  5,50,
  56,54, 66,14, 73,75,  6,60, 56, 5, 73,52, 73,36, 36,17,
  18,38, 33,38, 36,73, 38,20, 55,29,  2,42, 71,14, 74,53,
   4,69, 65,23, 12,37, 41,37, 29,37, 30, 5, 20,66, 69,24,
  48,43,  6,26,  2,45, 75,17, 10,79, 14,62, 54,63, 79,28,
  75,29, 38,67, 31,26, 25,74, 41, 0, 16,62,  9,27, 59,73,
  71,30, 46,63, 33, 2, 51,31, 25,62, 18,18, 44,57, 37,48,
  28,58,  0, 6, 20,76, 37, 2, 17,62, 78,44, 20,10, 55,20,
  69,33,  9,20, 23,34,  9,29, 70, 9, 19,53,  1,19, 41,39,
  76,75, 19,62, 41,16, 24,54, 10,35,  3,36, 45,60, 57,21,
  41,61, 34,39, 37,75, 19,11,  7,39,  1,37, 50, 9, 41,11,
  79, 8, 71,39, 58,74, 25,14, 12,68,  4,27, 22,70, 52,44,
  68,37, 45,24,  3,48, 34,55,  6,56, 71,59, 59,36,  6,48,
  34,32, 77,31, 18,14, 42,30, 22, 8, 12, 1, 12, 2, 33,29,
  52,17, 71,19, 21,50, 39,68,  9,21,  1,18, 68,77, 25,65,
  74,52, 47,59, 50,31, 79,44,  5,47, 42,39, 38,54, 64,26,
  61,59, 32, 1, 21,29, 21,24, 51,15, 79,65, 27,53, 16,16,
  13,53, 26,36, 21,66, 33, 0,  8,12, 27,32, 45,71, 31,13,
  39,57,  8,64, 41, 4, 14,79, 33,72, 65,45, 60, 1,  1,63,
  55,64, 41,58, 45, 4, 37,71, 69,51, 27,12, 36, 3, 23,17,
  68,45, 19,55, 65,68,  5,53, 62,68, 58,59, 22,55, 50,63,
  51,19, 14,58, 69,10, 48,16, 44,59, 70,13, 32,40, 21,11,
  22,35, 72, 6, 43,68, 54,38, 44,71, 65,19, 72, 1, 49,35,
   1,23, 14,60,  2,58, 16,27, 72,11, 52,42, 15, 0, 56,15,
  58,67, 46,70, 26,52, 13,32, 55,70, 39,72, 70,76, 21, 7,
  51,36, 77,27,  0,38,  2,43, 56,35, 76,65, 21,28, 10,50,
  34,50, 77, 3, 27,31,  5,24, 21, 1, 52,25, 13,44, 32,44,
  40,51,  3,40, 50, 4, 51,65, 61,68, 34,17, 50,46, 34,16,
  41,23,  7,13, 58,34, 69,64, 45,63, 27,41, 45,17, 18,17,
  30,67, 54,55, 22,62, 47,64, 47,14, 55,22, 27,52, 46,13,
  53, 2, 65,50, 75,46, 35, 7, 21,72, 12,57, 56,60,  4,70,
  70,24, 42,73, 50,34, 14,24, 44,32, 15,15,  9,41, 64,56,
  37,45, 57,45, 34,47, 22,58, 61,31, 49,11, 66,34, 79,10,
  30,15, 26,16, 28,24, 34,27, 27,37,  2,67, 30,38, 58,51,
   4,64, 36,30, 16,38,  6,64, 76,30, 50,44,  4,35, 52,20,
  54, 5, 68,22,  7,63, 52,23, 24,34, 24,78, 53,38, 47,44,
  60,24, 31,37, 62,54, 45,52, 65,55, 40,52, 16,44, 47, 9,
  13,30, 63,32,  2,55, 21,38, 35,32, 42,44,  8,60, 35,35,
  11,67, 53,24, 33,66, 70,70, 15,23, 30,78, 64,23,  1,72,
  36,24,  4,46, 11, 7, 67,37, 37,33, 71,10, 62,10, 36,78,
   5,36, 73,69, 62,41, 66,26, 75, 0, 18,60, 52,36, 67, 6,
   4,10, 61,78, 32,57, 62,29, 12,60, 24,55, 23,76, 36,22,
  28,47, 44,44, 43,74, 10,49,  0, 5,  9,10, 26,25, 60, 6,
  38,37, 59,58, 20,40, 16, 3, 27,51, 13,16, 70,58, 64,62,
  38,36, 54,54, 69,61, 28,10, 47,34, 71,15, 23,30, 29,40,
   8,71, 73,65,  4,39, 76,11, 32,33, 14,57, 26,74,  0,41,
  78,27, 77,75, 60,46,  4,63, 52,54, 24,79, 18,66,  0,28,
  11,13,  2,54,  7,59, 77,44, 39,58,  0,23, 38, 6, 32,23,
  59,31, 12,23, 77,39,  6,37, 49, 3,  4,37, 42,18, 17,21,
  77,49, 33,79, 19, 2, 12, 5, 72,24,  0,31, 20,44, 78,35,
  70,43, 29,41, 45,32, 62,66, 58,63, 39,10,  9,51, 24,14,
  31,20, 35,75, 16,18, 67,41, 32,37, 33,68, 40, 8,  9, 2,
  71,37, 70,36, 38,58, 54,40, 21,76, 67,57, 76,46, 48,39,
  18,24, 36, 1, 14, 7,  5,66, 26, 1, 15,40, 68,46, 24,47,
  77,50, 45,31,  8, 0, 11,59, 35,14,  0,55,  4, 9, 24,63,
  37,61, 76,39, 54,19, 56,40, 70,26, 16,32, 34,22, 46,41,
  57, 2, 38,47, 19,42, 52,29, 48, 2, 37, 1,  3,46, 35,40,
  22,64, 58,39, 17, 7, 60,59, 39,63,  5, 6,  2,72, 31,41,
  40,68, 20,16, 47,74, 39,30, 20, 5, 13,63, 43,42,  5,52,
  57,68, 73, 3, 54,37, 61,53, 40,64, 74,77, 68, 0, 41,54,
   1, 0, 71,31,  7,21, 28,12, 61,50, 42,52, 24,43, 64,31,
  66,36, 64,50, 14,19, 33,58, 42, 7, 21,77, 41,20, 76,49,
  43,25, 51,73, 10,64, 15,25,  6,65, 63, 1, 70,18, 50,14,
  40,19, 51,71, 15,41, 62,20, 72,20, 27,45, 61, 1, 46,17,
  18,16,  7,52, 76, 0, 16,54, 10,66, 26,55, 40,26, 78,66,
   6, 9, 23,31,  0,44, 38,75, 25,58, 60,62,  9,32, 15,34,
  38,49, 60,73,  6,22, 12, 9, 30,79,  8,50, 47,76, 47,18,
  54,59,  2,76, 43,12,  1,33,  5,16, 68,31,  5,14, 66, 1,
  55,60, 41,43, 74,24,  7,50, 69,67, 18,70, 11,78, 65,70,
  13,20, 14,66, 12,38,  4,52, 30,28, 10,68, 57, 6, 32, 0,
  32,55, 39, 9, 24,31,  5,49, 40,27, 15,63,  8,41, 63,58,
  29,38, 45,48, 42,74,  1,78, 22,74, 11, 0, 53,68, 46,49,
  13,29, 54,62, 66,47, 47,54, 25,34, 72,38, 78,18, 68,41,
  62, 3, 71,73, 78, 6, 43,61, 33,24, 79,20, 55,46, 44,40,
  20,65, 45,43, 17,72, 48,59, 50,16, 46,74, 34,38, 31,69,
  40,14, 25,16, 31,16, 51,68,  9,11, 62, 8, 30,21, 56,64,
  51,47, 61,25,  9,47, 37,36, 33,77, 74,65, 24,57, 78,19,
  12,74, 79,41, 43,29, 34,64, 14, 9,  6,50, 75, 6, 70,37,
   2,70, 38, 0, 49, 4, 75,40, 45,66, 36,11, 27,44, 29,31,
  49,26, 70, 5, 55,47, 10,14, 17,44,  0,35, 40,57, 47,53,
  13,38,  8,38, 63,25, 23,14, 64,22, 26, 4, 34,33,  7,23,
  72,30, 30,74, 19,18, 73,31, 19, 7, 24,41, 14, 5, 14,76,
  17,66, 24,52, 25,33, 49,29, 58,18, 74,29, 56, 7, 36, 6,
  34,65, 76,14, 61, 0, 42, 3, 21,33, 19, 9, 65,41,  4,76,
   3,30, 56,22, 28,25, 37,53,  7,61, 29,57, 72,63, 77,41,
  26,30,  3,68, 41,57, 55, 2, 51,46, 55,32,  5,23, 57,36,
  13,60, 74,36, 61,51, 73,48, 75,36,  9, 6, 53,49, 34, 2,
  44,24, 69,77, 38,38, 59,20, 71,77, 23,13, 50,48,  3,21,
  58,11,  6,49, 79,39, 56,12, 60,45, 19,47, 63, 5, 23,32,
   5,21, 72,41, 64,64,  2,17,  3,51,  2,39, 12,55, 15,54,
  18,32,  7,28, 28, 0, 50, 1, 74,19, 40,50,  1,34, 71,74,
  63,60, 65,34, 54,67,  4,21, 57,70, 17,65, 64,78,  8, 9,
  78,73, 46,14, 20, 2, 74, 4,  9,35, 70,64, 17,51,  0,62,
  12,12, 50,15, 38,50, 45,64, 44,48, 25,56, 48,70, 63,61,
  54,77,  4,13, 54,32, 69,21, 13,25, 59,74, 11, 3, 50,29,
  73, 2, 70,23, 32,48, 23, 0,  2,38, 10,56, 29,43, 61,75,
  73,19, 57, 0, 10,41, 49, 6, 19,68, 24,62, 47,45, 68, 9,
  73,61,  9,49, 32,13, 42,32, 33,14,  7,35, 16,47, 79,13,
  57,34, 75,67, 13,41,  6,31, 44,60, 41,64, 52,18, 60,72,
  79,17, 29,79, 33,54, 24, 5, 48,21,  3,41, 42,63, 73,21,
   7,72, 75,28, 53, 5, 47,77, 77,74, 38,61, 30,12, 33,67,
  20,46, 62,46, 75,20, 40,59, 59,45, 54,26, 28,20, 48,60,
  61,17,  8,42, 75,43,  2,49, 11,31, 25,19, 71,28,  8,14,
  24,38,  1,51, 18, 5, 33,60,  2, 1, 29,49, 76,73, 45,29,
  55,77, 11, 9, 61,65, 77,30,  0,30,  8,55, 42, 4,  7, 6,
  33,21, 48,65, 65,56, 40,74, 12,76, 77,68, 69,17, 58,58,
  60,57, 28,49, 67,28, 29,28, 41,62, 73,29, 25, 3,  0,42,
  75,18, 60,37, 13,76, 53,63, 15,13, 59,25, 16,25, 13,43,
  51,62, 52,38, 23,40, 12, 0, 35,41,  1,59, 15,33, 70,27,
  35,43, 47,43, 49,72,  3, 4, 54,20, 28,34, 58,69, 69,39,
  33,11, 45, 5, 40,47, 25,36, 56,17, 15,68,  7,47, 12,58,
  58,24, 14, 8, 71,53, 74, 3, 62,11, 38,63, 61,52, 39,26,
   6,70, 26,38, 28,22, 57,66, 46,27, 72,34, 55,52, 35,77,
  59,56, 59,42, 23, 8, 68, 5, 15, 4,  1,30, 17, 6, 33, 9,
  56,77, 17,33, 42,64, 74,43, 70,46, 62,61,  2,32, 36,61,
  50,73, 10,21, 60,71, 71,12, 69, 8, 75,55,  1,44, 35, 0,
  10,16, 14,56, 38,52, 78,69, 56,31, 10,62,  8,61, 46,56,
   9,30, 31,46, 63, 9, 11,35, 40,62, 64,39,  6,36, 25,64,
  31, 0, 17,57, 10,47, 75,75, 16,72, 18, 9, 55,44, 38, 1,
  55, 5, 41,22, 41,77, 67,68, 21,27, 38,46, 33,51, 36,74,
  62,60, 67,24, 36, 0,  8,70, 12,63, 43,32,  2,10, 34,23,
  78, 4,  8, 4,  7, 5, 41, 1, 63,28, 51,24, 71,61, 24,44,
   2,56,  6,27, 68,67, 53, 6, 21,44, 28,42, 11,30,  3,67,
  31,66,  7,20, 70,17, 64,12, 55,27,  9,13, 64,27, 29,12,
  54,66, 28,69, 79, 6,  3,73, 74,35, 53,10, 72,48, 18,31,
  21,53, 56,30, 49,59, 17,70, 65, 3, 60,69, 73,79, 51,51,
  30,17, 71,43,  7,11, 43,23, 40, 2, 65,67,  7,42, 22,63,
  40,15, 52,40, 73,76, 50,28, 44,17, 59,64,  4, 3, 56,28,
   5, 4, 54, 7,  9,18, 11,15, 58,12,  8,27, 68,33, 29,66,
  74,68, 10,65, 30,26, 71,33, 41,60
};
const int p_80_7[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1254,
  /// Array of preferences
  20,30,  0,16,  1, 1, 11,23, 75,79, 57,79, 57,13, 48, 1,
  60, 1, 44,67, 32,27, 55,31, 26,25, 10, 3, 50, 4, 78,60,
  41,54, 73,27, 13,46, 26,71, 70,10, 24,67, 34, 0, 48,70,
  41, 4, 39,70,  0,56, 32,62, 68,26, 69,59,  5,12, 33,60,
   7,29, 62,42, 23,59, 23,52, 59,56, 31,49,  5,59, 48,79,
  65,24, 30,48, 48,49, 23,58, 77,56, 78,72, 21,59, 27,54,
  30,18, 47,34, 27,15, 33,41, 34,35,  6,63, 40, 7, 42,22,
  65, 3, 43,76, 61, 5, 60,75, 29,30, 76,13, 56,46, 55, 1,
  25,39, 15,25, 48,53, 54,15, 55,56, 17,23, 18,64, 69,28,
  26,36, 54,68, 75, 3, 68,77, 59,16, 76,39, 26,68, 54,35,
  63, 0, 71,39, 61,29, 11,22, 57,22, 41,41,  5,49, 14,57,
  31,67, 73,68, 12, 0, 20,18, 55, 8, 44,35, 44,22, 75,34,
  16,63, 44,13, 23,42, 72,79, 10,21, 48,38, 76,59, 43,23,
  28,55, 78,59, 28,43, 46,13,  2,48, 70,25,  8,42, 12,73,
   7,40, 30, 5, 19,59, 43,71, 12,20, 16,19, 62,66, 10, 7,
   9,79, 52,79, 11,15, 16, 2,  4,67, 27, 5, 62,33, 65,74,
  20,57, 79,33, 77,36, 42,79, 29,48, 33,63, 60,33, 33,24,
  52,14, 13,77,  2,25, 13,78, 43,18, 31,21, 65,62, 36,74,
  79,62, 53,59, 73, 3, 50,33, 71,67,  7, 9, 63,14, 54,41,
   1,47, 43,12, 16,46, 60,17, 47,22, 62,31, 73,15, 75,65,
   5,23, 43,30, 48,58, 48,50, 56,51, 58, 5, 11, 7, 75, 1,
  31,33, 30,56, 29,58, 46,27, 51,20, 65, 7, 62, 4, 64,27,
  51,71, 44,29,  8,19, 20, 4, 63,17,  0, 8, 18,12, 41,47,
  71,22, 54, 2, 32,44, 66,36, 67,32, 53, 5, 75,58,  5, 8,
  50,70, 39,78, 24,53,  3,77,  2,51, 62,45,  0,54, 13,67,
  24,20, 78,26, 79,38, 24,61, 10,57, 58,43, 22,36,  9,38,
  75,78, 43,44, 22,30, 47,27, 78,25,  2,44, 51,16, 33,37,
   8,75, 78, 1, 72,31, 20, 7, 79,55, 34,18, 53,53, 77,73,
  44,30, 51, 5,  3,50, 11,37, 50,71, 52,60, 27,37, 42, 2,
  65,55, 16,27, 33,48, 36,28, 20,66, 56, 0,  7,67, 23,37,
  10,67, 11,67, 76,50, 54,24, 19,69, 32,45, 38, 2, 11,58,
  44,47, 78,66, 71,17,  5,11, 68,51,  1,17, 57,74,  7,31,
  22, 1, 41,10,  2,22, 46, 1,  3,44, 64,13,  4,13, 29,11,
  12,26, 31,71, 72,41, 12,18, 32,10, 58,38, 20,34, 65,68,
  64,39, 44,77, 30,24, 31,56, 66,45, 78,79, 64, 6,  9,35,
  49,68, 30,62, 59,63, 52,30, 36,39, 76, 0, 71,52, 20,74,
   7,10, 11,27,  3,31, 10,18, 43,49, 72,43, 58,79, 23, 3,
  16,24,  9,17, 40,16, 69,68, 55,13, 16,66, 43,60, 43,47,
  23,15, 59, 8, 45,26, 27,79, 47,57,  4,69, 51,32, 37,66,
  55,43, 46,36, 40,26, 67,21, 35,61, 71,33, 22,75,  9, 2,
  54,10, 39, 9, 36, 0, 58, 3, 14,14, 41,65, 58,56,  2,56,
   9,50, 35,20,  2, 1, 73,18, 21,78, 19,60, 66, 5, 43,63,
  55,32, 56,72, 78,24, 55,12, 15,78,  1,59,  3,58, 59,59,
  76,17, 46, 0,  8,27, 51,44, 11,38, 78, 8, 58,39,  9, 1,
   7,19,  2,64, 22,76, 66,19, 75, 4, 58,55, 47,28, 75,68,
  21,69, 30, 1, 39,49, 74,75, 22,55, 12,48, 48,15, 75,73,
  70,78, 44,50, 10,22, 17,32,  2,42,  7, 1, 33,56, 56,75,
   7,36, 50,48, 52,42, 15,38, 33,55, 12,61, 42,41, 58,52,
   0,66, 74,56, 69,76, 53,55, 46,28, 26,23, 37,52, 67,72,
  37,25, 25,55, 21, 0, 22, 7,  7,70, 64,35,  8,50, 60,51,
  43,51, 78, 4, 68,55,  6,70, 60,35, 35,72, 23,32, 42,24,
  30,37, 79,75, 51,60, 33,62, 45,22, 20,72, 69,38, 70,51,
  35,67, 29,63, 18,40, 17,45, 12,22, 58,58, 33,58, 76,44,
  47,37, 30,36, 17,31, 45,45,  1,78, 30,14, 46,52, 35,70,
  38, 4, 11,24, 55,63, 36,65, 75,52, 32,24, 44,72, 71,26,
  56, 1, 21,75, 33,65,  8,59, 27,62,  5,73, 74, 9, 17,51,
   7,22, 76,30, 61,67, 18,36, 21,38, 13,35, 10,45, 19,62,
   8,45,  7,75, 21,31, 59,33, 69,42, 62,58, 56,68, 58,19,
  66,57, 17, 8, 28, 6, 39,79, 45,21, 68,38, 54,33, 19,48,
  55,51, 12,60, 36,21, 76,41, 49,26, 33,38, 31,36, 56,35,
  21,52, 51,63, 76,11, 71,47, 70,43, 14,62, 37,19, 20,20,
  54,36, 60,76, 69,71, 68,16, 48,74, 48,73, 11,62, 10, 9,
  65,58, 79,57, 32,54, 54,50,  3,76,  7, 3, 37,27, 45,61,
  61,54, 14,59, 54,20, 47,70, 74,74, 66,77, 73,52, 52,68,
  33,45, 78,69, 37, 4, 12,70, 50,22, 55,70, 17,25,  9,70,
  64,20, 65,51,  7,13,  0,61, 74, 6, 50,15, 36,54, 37,69,
  65,38, 53,75, 73,47, 72,44, 68,66,  2,53,  7,74, 53,19,
  14,35, 25,70, 19,70,  9,31, 44,51, 52,25,  0,50, 51,30,
  52,33, 15,40, 27,49, 38,56, 14,75, 21,79, 30,42, 12,24,
  21,71, 76,72, 51,34, 12,16, 53,62,  6,50, 66,33, 43,58,
  76,76,  5, 0, 59,67, 70,70,  3,54, 79,69, 20,76,  2,30,
  76,48, 30,20, 73,48, 54, 6, 49,74, 58,49, 79,76, 32,69,
  76, 8, 58,23, 72, 0, 57,55, 21,39, 27,76, 49,44, 71,48,
  70,32, 71,45, 45, 7, 41,60,  0,13, 54,73, 53,48,  3, 6,
  51,64, 49, 8, 75,41, 24,40,  5,37, 11,75, 36,75, 46,40,
  36,47, 56,44,  8,48, 71,16, 19,14, 44,59, 23,11, 62,22,
  65,46, 39,33, 43, 2, 25,24, 17,60, 66,63, 29,46, 70,72,
  72,32, 41,29, 23,53, 77,13, 32,34, 50,16,  2,71,  7,48,
   1,42, 40, 3, 79,48, 76,43, 53,20, 15,51,  8,79, 75,43,
  37,16, 56,11, 76,73,  2,31, 52, 5, 67,35, 52,66, 27,65,
  25, 1,  1,36, 40,35, 47,66, 58,73, 39,18, 49,42, 60, 3,
  12,36, 11,34, 25,11, 22,56, 51,13, 56,65, 60,19, 30,54,
  77,53, 36,56, 21,54, 41,69, 43, 7,  4, 2,  5,55, 52,48,
   3,47, 51, 9, 30,59,  8,56, 68, 4, 48,11, 11,35, 53,58,
  73,28,  3,46, 42, 3, 76,61, 21, 2, 79,46, 24,55, 68,14,
   8,30, 11,18, 63,73, 62,14, 60,29, 18,28, 23,46, 30,35,
  55,18, 31,30, 55,19, 75,69, 20,22, 63,79, 41,14,  3,13,
  35,66, 57,75, 61, 9,  5,38, 40, 8, 32,64, 36,24, 30,27,
  29,42, 14,21, 34,70, 18,20, 65,52, 35,77, 20,60, 38, 0,
  63, 3, 35,29, 13,19, 39,72, 13,30, 10, 2, 31,27,  3,36,
  38,10, 35,79, 66, 4, 55,79, 60,10, 39,19,  1,45, 56,32,
  66,74, 67,29, 79, 9, 43,37, 12,37,  1,18, 27,59, 49, 1,
  14,13, 64,59, 54,71, 47,56, 35,24, 75,63,  0,36, 56,74,
  67,67,  0, 5, 54,64, 50,61, 42,31, 61,56, 26,47, 10,60,
   1,14, 74,10, 20,37, 39,67, 62, 8, 43,64, 34,15, 37,71,
  72,48, 10,56, 39,68, 42,75, 61, 4, 58,41, 73,34, 61,32,
  25, 9, 47,51, 33,26, 44,73, 18,53, 40,55, 34,79, 57,10,
  48, 3, 30,57, 10,13, 74,13, 38,45, 64,57, 12,45, 27,66,
  11,71, 32,66, 59,13,  2,18, 34,13, 32,49, 44,55, 36,30,
  32,59, 77,24, 60,72, 21,15, 11, 5, 15,67, 59,29, 62,36,
  16,67, 41,15,  8,51, 55,41, 61,53, 63,18, 73, 2, 58,48,
  76,69, 42,37, 23,76, 71,70, 54,56, 19,41, 11,61, 73,65,
   0,35,  8,29, 43,55, 20,16,  4,49, 28,24, 42,13, 47,60,
  46,16, 69, 9, 23,71,  2,14, 75,12, 45,33, 74,14, 58,13,
   1, 5, 36,77,  7,54, 78,36, 26,77, 60,25, 19,16, 17,20,
  24,15, 69,75, 31, 2, 55, 9, 61, 0, 49,29, 40,50, 22,17,
  14,23, 49, 4, 11,69, 68,57, 59,77, 68, 5,  7,33, 46,26,
  50,11, 51,48, 32,46, 21,33,  3,62,  0,75, 20,24, 62,78,
  14,43, 60,63, 77,68, 34,44, 60,50, 58,67, 42,67,  4,77,
  38,77, 29,40, 70,37, 68,69,  4, 5, 48,22, 60, 0, 50, 3,
  36,41, 21,11, 10,50, 49,30, 62, 9, 52,54, 27,10, 60, 2,
  30,28, 64,67, 16,69, 45, 5, 19,73, 55,42, 49,41, 52,55,
  21,27, 79,79, 43,31, 44,45,  5,45, 48,33, 49,67, 17,65,
  42,44, 75,19, 37,22, 54,77,  2,60, 25, 8, 30,13, 22,51,
  70,76, 75,15, 72,36, 37,49, 42,29, 56,40,  4,34, 12,27,
  11,73, 39,24, 13, 2, 53,76,  5,44, 67, 4, 44, 3, 26,69,
  64,49, 69,20, 27,29, 67,66, 60,53, 69,78, 10,66, 19,75,
  38,41, 21,37, 64,50, 77,67, 64,54, 13,10, 76,42, 65,75,
  51,54, 74,26, 55,10, 17,11, 40,20, 65,32, 53, 3, 10,49,
  25,30, 74,60,  5,15,  1,66, 22,40, 49,66, 17,17, 63,77,
  62,64, 35,48, 76,34, 24,68, 28,72, 12,75,  8,12, 73,77,
  47,48, 51,14, 59,31, 60,16, 69,10,  6,39, 16,37, 49,40,
  32,12, 34,68, 64,45, 70,12,  1,76, 53,32, 46,41, 32,17,
  71, 2, 53,16, 30,47, 69, 5, 67,11, 51,36, 34,65, 79,28,
  35,50, 24,37, 77,69, 68,37, 29,67, 64,47, 71,37, 36,59,
  13,34, 65,41,  9,30, 59,41, 42,74, 32,25,  9,40, 63,75,
  52,28, 37,65,  0,20,  0,27, 57, 4, 34,76, 19,66, 63,35,
  14,73, 25,67, 58,10,  6,15, 78,39, 29,34, 24,64, 61,48,
  39,16, 46,49, 47,35, 72, 2, 40,42, 73, 0, 18,38, 43,73,
  79,44, 27, 2, 13,47, 10,63, 36,44, 77,63, 65,15, 73,61,
  46, 9,  2,12, 56,37, 21,26, 48,34, 32,32, 75, 9, 40,72,
  79,45, 60,31, 54,37,  3, 5, 13, 0, 77,60, 18,15, 19,21,
   5,78, 59,19, 76,32,  4,78,  5,29, 44,33, 74,18, 25,27,
  71,12, 12,50, 37,43, 68, 8, 41,58, 63,44, 25,15, 67,70,
  26,21, 68,19, 32,71, 48,18, 52,35, 76,31, 61,75, 78,33,
  50,26, 10,78, 28,29,  5,26, 34,33, 31,44, 44,71,  0,38,
  23,25, 39,66, 60,18, 28,61, 27,41, 68,50, 60,32, 65, 8,
  72,19,  9,57, 65,59, 48,16, 63,13, 44,39, 79,78, 30,25,
   3,41, 70, 1, 12,13, 69,65, 59,36, 11, 3, 69,53, 16,36,
  44,54, 48,32, 49,33, 29,60, 37,12,  5,70, 36,19, 15,30,
   8,60, 47, 0,  4,38, 76,66, 26,74, 47,32,  3,29, 64,56,
   1,49, 77,76, 50,55,  2,39, 72,78, 63, 8, 11,19, 23,34,
  57,36, 49,53,  9,71, 77,15, 40,43, 67,14, 64,58, 25, 0,
  12,40, 32, 3, 26,72, 42,54, 74,62, 30, 3, 36,31,  8,21,
  27,25, 60,62, 71, 1, 36,12, 46, 3, 13,74,  6,68, 77, 5,
  40, 6, 74,33, 54,57, 28,27, 71,72, 71,29, 65,22,  0,30,
  60,30, 10,59, 63,33,  0,11, 55,48, 55,26, 24,56, 11,47,
  29,24,  8,71, 46, 6, 23,41, 68,56, 42,16, 36,51, 31,48,
  46,60, 20,45, 27,16, 55,69,  1,29, 17,48,  9, 6, 61,62,
  53, 1, 63, 7, 62,46, 36,38, 74,35,  7,44, 29, 6, 41, 0,
  12, 7, 45,58, 30,75, 40,19, 15,74, 10,79, 26,79, 20, 5,
  33, 3, 62,53, 42,56, 69,14, 61, 2, 25,32, 41,50, 53,72,
  45,52,  7,50, 31,20, 38,63, 33,51, 15,11, 65,31, 20,61,
  28,11,  4,57, 32,33,  3, 3, 59,21, 12, 3, 11,56, 64,72,
   0, 3, 12,56, 31,39,  7, 4,  4, 4, 43,32, 11,14, 63,16,
  71,71, 28, 1, 77, 8, 42,73, 60,69, 25,60, 36, 5, 55,54,
  13, 9, 55,15, 17, 1, 31, 4, 58,59, 52,31
};
const int p_80_8[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  690,
  /// Array of preferences
  44,36, 41,71,  5,10, 26,54,  3,29, 29, 8, 19,44, 57,49,
  74,69, 29,25,  5,76, 16,19, 54,79, 12,75, 67,76, 11,62,
  48,43, 64,37, 44,63, 28,22, 59,42, 45,37, 49, 5, 45,75,
  25,32, 70,22, 40, 6, 36,39, 24,57,  8, 1, 15,73, 47, 9,
  67,78, 38, 1, 17,59, 71,42, 66,15, 23,64, 61,50,  8,57,
  10,53,  4, 1, 27,12, 23,33,  3, 5, 25,69, 17,27, 62,46,
  17,31, 11,51,  2,33, 23,22, 33,59, 54,40, 49,62, 15,43,
  43,11, 67,53, 46, 1, 68, 7, 35,68, 33,63, 57,76,  9,25,
   4,21, 47,77, 12,57, 41,30, 52,60, 56,28,  3,18, 25, 7,
  74, 2, 79,73, 22, 0, 54,72, 50,46, 49, 1, 64, 3, 77,23,
  51,71, 55,28, 26,60, 22,27, 45,36, 42,68, 16,73, 66,61,
  24,75, 10,41, 41,66, 30,12, 70, 3, 46,72, 67,75, 72,68,
  41, 0, 61,25, 11,15, 60,36, 69,53,  0, 4,  0,29, 18,44,
  19,10, 26,11, 65,16, 54,78, 49,41,  6,34,  1,19, 16,40,
  49,14, 68,10, 72,51, 30, 1, 18,62, 25,60, 61,35,  9,45,
  54,44, 16,77, 29,45, 79,28,  3,51, 15,20, 13, 2, 37,35,
  53,66,  4,37, 50,41, 66,45, 72,28, 53, 9, 15,36, 11,58,
  53,72, 28, 8, 63,71, 35,40, 36,44,  1,16,  4,66, 77, 0,
  35, 0, 55,71, 30,25, 58,69, 54,74, 21,11, 29, 0, 65,39,
  39,10, 57,11, 42, 3,  7,34, 20,35, 65,17, 40,47, 71,53,
   3,79, 65,40, 65, 9, 16,50,  7,67, 54,19,  5, 0, 62,62,
  20, 7, 46,40,  0, 1, 59,66, 78,30, 44,56,  5,29, 41,42,
  30,41, 26,77, 54, 3, 34,50, 28,15, 53,27, 10,16, 51,44,
   8, 4, 18, 5, 27,45, 63,73,  1, 6, 56,10, 49,77, 13,74,
  47,52, 77,48, 22,49, 18,12, 56,59, 50,11, 18,37, 36,32,
  70,37, 20,75, 79,64, 41, 2, 54,75, 12, 8, 36,13, 44,11,
  57,74, 11,61, 79,24, 43,22, 70,43, 58,61, 43,21, 50,52,
  48,20, 14,67, 31,46, 14,20, 38,77, 38,70,  6,12, 10,75,
  28,41, 35, 3, 59,48, 10,68,  7,77,  9,27, 65,64, 12,40,
  64,50, 21,73,  6,69, 19,35, 48,11, 73,38, 11,21, 22,59,
   8,28, 42,22, 72,11, 20,31, 60, 0, 62, 4,  2,46, 59,43,
  70,78, 47,38, 47,62, 60,42, 16,39, 46,22, 34,72, 11,39,
  58,11, 52,63, 69,56, 39,78, 52,16, 16, 8, 70,38, 71,18,
  12, 0, 39,56,  3, 0, 44, 0, 14,43, 22,70, 22,64,  3,72,
   3,28, 77, 8, 54,65,  1,77, 68,35, 40,17, 11, 9, 19,53,
  37,36, 47,44, 30,48, 63,16, 65,47, 44,24, 45,74, 16,33,
  17,39, 44,65, 42,24, 35,13, 52, 2, 59,21, 31,76,  6,40,
  72,16, 38, 9, 44, 7,  6,51, 42,78, 58,73, 16,46, 49,12,
  10,63, 34,15, 27,50, 26,34, 43,40, 30,74, 50,75,  9,29,
  36, 2, 55,10, 45,73, 13, 3, 64,18,  1, 9, 77,10, 52, 5,
  79,48, 31,71, 74, 1, 56,39, 52,52,  1,56,  5, 8, 35, 6,
  47,54, 64, 8, 55,62, 19,63, 16,56, 15, 5, 74,27, 42,13,
  48, 0, 14,55,  6,26, 15,18, 27,46, 32,66, 73,61, 76,49,
  67,68, 36, 8, 78,13,  2,55, 21,16, 40,10, 41, 3, 29,10,
  57,22, 68, 1, 21,57, 42,55, 72,35, 78,60, 67,67, 58,51,
  13,42, 15,23, 22,23, 14,12, 19,47, 40,16, 79,16,  9,57,
  22,31, 36,74, 15,12, 35,23,  1,43, 35, 2, 34,31, 48, 7,
  26, 3, 65, 8, 26,38, 38,67, 29,17, 60,25, 69,57, 63,64,
  16,12, 76, 8,  3, 3,  6,32, 61,65, 24,11, 61,51, 70,64,
  22,75, 55,31, 31,31, 31,40, 35,34, 15,58, 19,17, 32,25,
  49,43, 34,68, 17,55, 32,72, 33,40, 60, 5, 25,47,  3,10,
  66,78, 44,73,  8,26,  7,30, 30,38,  6, 7, 58,57, 37,26,
  45,78,  6,79, 42,16, 18,30, 51,11, 26,15, 68,19, 74,20,
  48,52, 25,51, 34,59, 48,36, 62,45, 62,71,  2,44, 18,79,
  50,77, 39,70, 22,46,  0, 2, 73,73, 21,32, 23,79,  5,34,
  19,40,  5,32, 63,21, 35, 1, 70,56, 31,68, 40, 8, 26,40,
   8,13, 74,50, 75,61,  1,69, 11,70, 19,74, 78,58, 35,15,
  72,79, 29,74, 73,45, 55,38, 71,24, 43,25, 43,44, 31, 4,
  21,75, 68,72, 30,32,  6,65, 63,74, 78,10,  7,57, 71,16,
  62,28, 10, 7, 72,34, 28,21, 11,52, 41,15, 50,17, 69,47,
  43, 4, 35,72, 65,20, 64,26, 78,51,  4,24, 67,41, 37,10,
  51, 5, 61,24, 41,56, 41,39, 58,10, 26,75, 25,57, 68,57,
   7, 7, 68,36, 25,22, 49,22, 48,14,  7,26, 10,23, 60,68,
  52,54, 65,22, 57,35, 27, 5, 37,48, 63,31, 45,27, 33,13,
  63,54,  1,32, 15,51,  3,45,  2,19, 62,73, 66,36, 14,62,
  53,65, 48,15, 23,26,  7,43, 60,50, 44,43, 57, 6,  8,21,
  19,68, 56,67, 20,63, 44,69, 13,16,  4,75, 19, 8, 66,69,
  42,56, 33,44,  9,23, 17, 4, 38,64, 41,12,  6,20, 73,18,
  18,53,  9,11, 57,13, 24,73, 13,46, 26,24, 69, 1, 62,17,
  76, 9, 34,13,  6,24, 52,24, 19,30, 31,62, 12,52, 75,22,
  75, 9, 29, 4,  8, 5, 32,51, 28,38, 31,26,  1, 4, 64,48,
   4,45, 54,12, 57,23, 22,10, 29,49, 31,78, 23,70, 57,14,
  43,59, 17,57, 13,50, 14,53, 17, 5, 47,23, 26, 7, 72,17,
  76, 0, 36,62, 77,47, 28,33, 11, 6, 16,16, 45,77,  8, 6,
  49,49, 66,11, 16,13, 35,69, 12, 4,  2,13, 76,55, 32,79,
  28,20, 37,40, 67,70, 61, 1, 73,77, 44,51, 33,35, 64, 1,
  53,51,  8,11, 42,48, 28,52, 58,66, 43,14, 36,47, 32,74,
  42,14,  2,41, 44,18, 63,43, 23,73,  1,48, 61,11, 32,32,
  46,52, 44,31, 33,60, 73,52, 78,61, 34,38, 47,43, 50,69,
  43,32, 13,72, 67,47, 44,27, 28,34, 25,53, 75,26, 59,71,
  43,54, 46,14, 42,51, 62,75, 39,53, 37,38, 39,69,  8,65,
  25,23, 31,12, 35,59,  5,65, 32,54, 62, 6, 64,74,  5,75,
  61,26, 62,29, 65,59, 74,65, 59,73, 30,15, 10,78, 12,16,
  26,62, 16,70, 74,21,  7,48, 22,21, 13,36, 26, 9, 35, 5,
  54, 1, 67,18, 19,27, 56, 1, 28,37, 49, 8, 70,60, 49,65,
  28, 1,  0,35, 11, 7, 53,61, 52,31, 39,50, 55,45,  3,70,
  59,68, 69,43
};
const int p_80_9[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  469,
  /// Array of preferences
  23,63, 41,57, 78,11, 39,62, 36,15, 24,40, 74, 3, 34,61,
  79, 3, 18,35, 74,63, 20, 9, 15,16, 50,74, 30,22, 42,43,
  34,67, 43,79, 30,39, 35,65, 53,10, 56,48, 65,66, 11,30,
  60, 7, 54,17, 18, 1, 24,16, 49,13, 74,75, 56, 2, 32,50,
  70,25, 72,32, 53,31, 79,53, 58,62, 58,39, 14,66, 71,73,
  25, 1,  5,18, 29,70, 55,31,  4,79,  3,63, 24,49, 78, 0,
  47,57, 75,48, 20,71, 76,24, 24,15, 53, 9, 28,36, 69,13,
  40, 9, 45,27, 78, 8, 49, 0, 72,74, 18,30,  0,20,  6,32,
  60,72, 28,77, 24, 2, 37,37, 70,66, 11,33, 60, 8, 43,47,
   5, 1, 47,60, 12,24, 22, 9, 17,12, 54,42, 75,34, 40,30,
  57,70, 67,66, 36,77, 69,41, 27,76, 60, 6,  8,66, 49,28,
  72,43, 51,21, 20,27, 55,35, 79,34, 21,60,  3, 2, 51,12,
  30,63, 36, 7, 19,30, 25,24, 34,31, 70, 8,  9,41, 59,29,
  57,79,  7,19,  0,52,  6,30, 25,10,  6,72, 39,56, 29,63,
   8,16, 58,69, 40,14, 25,60, 61,46, 45,66, 37,71, 11,36,
  61,74, 56,66, 70,55, 26,10,  2,28,  1,32,  8,68, 28,72,
  52,15, 73, 8, 23,44, 66,40, 35,73, 45,40, 68,24, 11,73,
  29,79, 47,13, 21,67, 62, 9, 74,65,  0,41, 42, 8, 70,62,
  66, 9, 23,25, 31,51,  9,58, 31,72,  3,12, 20,20, 56,22,
  75,61, 55, 6, 52,75, 59, 1,  3,50, 48,31, 14,61, 79,58,
  19,53, 14,19, 15,37, 38,15, 48,66, 62,74,  5,14, 51,53,
  69,39, 49,38, 30,44, 23,79, 16,22, 67,15, 69,66, 63,72,
  21,75, 24,26, 59,59, 71,46, 48,10,  0,27, 75,77, 51,41,
   1,61, 42,48,  2,76, 13,48, 24,33, 56,77, 57,16, 13,46,
   4,43, 28,27, 56,75, 28,63,  5,47, 33,39, 52,79, 25,22,
  62,29, 54,22, 62,17, 30,78,  3, 7, 60,42, 41,71, 39,25,
  22, 8, 56,51, 54,37, 37,14, 70, 9, 26,65, 70, 7, 59,63,
  73,29, 41,34, 76,50, 25, 9,  2,38,  8,30,  9,72, 14, 0,
  21,20, 46,55, 39,53, 74,32, 39,40, 44,61, 37,76, 45,74,
   2,51, 16,19, 46,73, 71, 8, 41,55, 70,17, 47,18, 42,25,
  21,10, 52,22, 40,45, 25, 5, 29,61, 21,66, 37,44, 41,42,
  77,48, 53,15,  4,72, 56,12, 67,32, 15,13,  1,56, 70, 0,
  70,22, 33,24, 59,42, 62,27, 74,68, 23,20, 33,57,  0,32,
  26,44, 43, 3, 42,77, 45,53, 56,38, 77,40, 45,68,  9,50,
  43,27, 26,51, 52,51, 74,78, 34,72, 30,58, 22,65, 32, 2,
  11,60,  5,54, 66,19, 11,77, 61,15, 56, 4,  7,11, 26, 4,
  63,37, 42,61, 57,15, 77,30, 14,24, 16,63, 46,29, 66,64,
  69,55, 17,49, 75,41, 11, 1,  3,23, 17,59,  0, 9, 36,69,
  31,67, 63,26, 21,43, 45,67,  0,43, 44,18, 56,45, 37, 0,
  26,68, 49,55, 47, 4, 17,65, 59,32, 68,44, 13,65,  7,73,
  53,13, 63,50, 41, 8, 42, 7, 32,24, 49,33, 62, 1, 70,44,
  75,72, 63,32, 21,22, 39,72, 66,73, 22, 1, 42,40, 52, 3,
  78,52, 47,32, 45, 3, 66, 4, 52,28, 62,23, 51,11, 32,75,
  43,48,  6,51, 12,20, 38,71, 22,75, 22,33, 49, 4,  7,37,
  22,70, 65,44, 11,78, 32,21, 62,26, 78,62, 17,56,  1,40,
  52, 4, 35,25, 20,73, 25,37, 15,68,  4,70, 61,71, 50,67,
  56,52, 47,53, 65, 0, 51,38, 36,47, 60, 1, 77,23, 35,16,
   6,37, 71,36, 18,14, 11,56, 26,73,  5,35, 50,18,  0,61,
  49,58, 12,79, 49,36, 35,19, 26,21, 78, 9, 49,46, 56,78,
  11,24, 61,64, 15,63, 59,48, 73,23, 34, 6, 66,55,  5,22,
  49,51, 65,45, 67, 0, 17,60, 65,47, 12, 5, 39,15, 20,29,
  77,70, 58,13, 55,49, 33,77, 45,18, 43,19,  7,65,  2,14,
  64,47, 64,11, 69,19, 47, 6, 16,69, 65,30, 78,34, 27,37,
  63,10, 60,53, 50,68,  0,60, 19, 2, 58,68, 76,36, 15,75,
  66,18, 64,35,  3,14, 13,69, 16, 2, 55,78, 21,78,  6,44,
   9, 6,  1,58, 20,23, 31,39, 48,77, 34,54, 10,75,  5,50,
  13,52, 34,38,  1,60, 56,15, 11, 4,  9, 3, 72,47, 20,54,
  79,57, 75,68, 73,22, 41,78, 29,14, 51,76, 76,79, 35, 7,
  20,26, 66,36, 10,65, 55,76, 43,41
};
const int p_80_10[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1474,
  /// Array of preferences
  32,46,  8,29, 75, 3,  6,10, 20,37, 32,18, 24,45, 65,41,
  35,52, 23,13, 11,62,  1,58,  7,27, 71,61, 18,41, 59,76,
   6,18, 58,46, 12,45, 62,29, 67,37, 37,66, 28, 7, 72,74,
  32,62,  9,35,  4, 2, 33,11, 31,47, 13,67, 29,27, 19,71,
  64,58, 22,54,  7,55,  0,61, 11,40, 54, 5, 76,11, 57,31,
  33,64, 68,73, 20,54, 25, 6,  8,66, 23,51, 12,62, 56,53,
  12,36, 21,32, 69,31, 35,37, 10, 3, 70,63,  4,38, 29,13,
  20,42,  4,46, 15,67, 17,29, 70,52, 23,64, 73,71, 26,42,
  64,79, 65,27, 24,66, 29,63,  9,79, 54, 7, 36,15,  7,26,
  15,65, 59,34,  4,39,  3,59, 71,49,  4,59, 33,27, 27,23,
  42,68, 48, 6, 44,17, 56,65, 32,47, 11,14, 76,67, 23, 3,
  66,41, 26,32, 58,37, 28,22, 69,79, 55,70, 19,56,  1,38,
  67,34, 38,21, 68,63, 60,20, 64,18, 54,63, 51,52, 34,28,
  60,43, 42,59, 29,65, 45,71, 26,48, 11,60, 18,30, 54, 1,
  19,16, 30,31, 55,79, 15,10, 38,73, 19,48, 46,70,  9, 5,
  55, 1, 27,27, 37, 6, 77,58, 55, 5,  9, 0, 46,35, 43,18,
  68,23, 23,46,  7,10, 69,18, 62,16, 53,14, 34,60, 61,35,
  15,15,  3,54, 24,51, 43, 3, 26, 2, 17,35, 25,43, 29,24,
   5,59, 11,23,  1,21, 36,21, 12,51, 60, 0, 55,74, 13,62,
  49,51, 76,72, 78,33, 55,25, 37,22,  2,33, 75,68, 25,50,
  40,55, 13,16,  8,58, 51,21, 52,37, 50,79, 63, 8, 71,34,
  79,41, 77,23, 36,55, 44,22, 59, 6, 52,48, 64,30, 71, 4,
  18,23, 27,41,  0,23, 37,42, 48, 0,  4,21, 68,24, 40,34,
  66,65, 20,63, 48,37, 71,40, 43,16,  1,20, 27, 9, 70, 8,
  29,14, 52,71, 12,11, 71,79, 37,31, 76,60, 28,72, 65,35,
  46,15, 39,73, 46,68,  1,23,  2, 4, 60,63, 48,24, 12,14,
  47, 8, 48,32, 79,76, 28,26, 54,48, 32,15, 22,47, 65, 0,
  38,18, 54,73, 39,57, 55,40, 21,55,  4, 9,  8,52, 67, 9,
  21,66, 18,26, 26,58, 74, 9,  7,25, 61,13, 64,11, 44,15,
  73,21, 11,37, 62,66,  5,21, 40,69, 77,29, 72,36, 70,60,
  32, 3, 36,41, 44,52, 27,51, 27,14, 18,66, 76,39, 43,75,
  69, 2, 23, 4, 25,55, 38,49, 41,28, 18, 8,  6, 8, 79,53,
   9,25, 28,42, 64,54, 50,19,  4,16, 27,72,  4,70, 23,22,
  14,61,  6,57, 72,69,  6,50, 31,70, 53, 5,  8, 3, 52,52,
   9,68, 77,17, 23, 2,  5,78, 71,29, 45,77, 79,36,  4,19,
  32,25, 79,54, 62,68,  0,35, 37,32, 49,42, 79,21, 59,26,
  21,23, 13,35, 79,58,  3, 3, 23,48, 68,51, 78,39, 29,53,
  55,13, 38, 8, 50, 5, 67,51, 15,52, 68,39, 22,61, 63,37,
  44,38, 55, 3, 42,47, 47,70, 10, 4, 57,64, 74,28, 27,30,
  72,35, 71,27, 20,56, 41,16,  3,25, 19,46,  9,34, 16,22,
  37,14,  4,22, 63,75, 27,64,  2,42, 40,77, 19,69, 74,43,
  26,69, 33, 8, 35,41, 28,73, 56, 5,  1,10,  8, 6, 41,29,
  72,31, 69, 8, 69,30, 67,75, 49,59, 35,73, 29,58, 39,18,
  63, 2, 59,19, 49,36, 52,76, 32,71,  8,57, 55,34, 46,16,
  41,65, 31, 3,  0, 2, 21,26, 39,29, 15,50, 25,75, 10,67,
  26,56, 49,62, 34,73, 51,76,  1,64, 69,21, 38, 3, 77,16,
  17,68, 39, 6, 18,65, 57, 0, 42, 3, 17,16, 39,32, 28,78,
  59,54, 49,28, 23,10, 62,33, 68, 9, 70,25, 41, 3, 66,13,
  43, 9,  9,56, 79,52, 28,56, 71,21, 32,74, 10,25, 18,68,
   8,79, 70, 5, 61,63, 34,37, 36,37, 55,24, 34,40, 37,54,
  45,29, 68,34,  8,63, 55, 8, 43,35, 44, 7,  4,35, 74,78,
  40,66, 31,28, 43,29, 71,72, 42,69, 37, 8, 73,19, 14,40,
   1,50, 63, 9, 34,10, 17,30, 15,37, 28,62,  2,14, 49, 0,
   7,75, 53,42, 26,67, 39, 2, 18,21,  1,28, 73,22, 73,67,
  31,25, 74,79, 78,53, 62,13, 29, 9, 16, 7,  5,70, 75, 8,
  38,48, 45,11, 79, 4, 52,11, 44,14, 16,13, 51,28,  5,52,
  50,68,  8,31,  9,16, 16,41,  2,64,  3,38, 74, 8,  1,65,
  72,34, 58,54, 28, 9,  9,53, 17,11, 47,41, 48,20, 50,26,
  15, 3, 50, 4, 33,45, 41,48, 27,37, 51,67, 73,17, 38, 4,
  27,56, 34, 9, 71,47, 26, 0, 73,43,  9,36, 45,43, 41,20,
  75,34, 44,26, 72,50, 25,72, 65,57, 79,15,  7,71,  9,78,
  28,54, 54, 3, 73, 2,  2,70,  1,59, 54,45, 52,41, 58,28,
  23,57, 24,76,  6,35, 23,15, 57,74, 67,12, 73,53, 42,74,
  36,76, 70,12, 26, 9, 49,45, 78,29, 74,15, 22,77, 53,11,
  65,28, 43,34, 12, 3, 33,12, 46,21, 35,56, 70,44, 33,50,
  10,63, 65,42, 23,37, 26,34, 18,42, 42,78, 27,12, 60,73,
  64,49, 71,35, 54,64, 34,56, 37,13, 37,60,  9,10, 78,61,
  60,53, 28,64, 42, 8, 46,29, 59,64,  5,67, 27,32, 30,52,
  16,26, 50,22, 15,75, 62,59, 39,34, 51,29, 24,59, 28,45,
   7,32, 47,34,  3,15, 60,28, 72,48, 69,64, 77,10, 69,33,
   4,34, 52,50, 22,16, 33, 1, 64, 9, 31, 8,  8,70, 51,17,
  54,77, 34,31, 78,41, 39,19, 35,58, 49,35,  0,28, 40,75,
  35,39, 60,57, 52,77, 29,57, 56,35, 57,40,  4,60, 67,73,
   2,44, 72,37, 74,63, 77,66, 46,60, 50,57,  4, 3, 79,28,
  14,33, 69,13, 26,31, 37,29, 69, 4, 74,62, 65,44,  1,40,
  44,70, 36,75, 14,32, 72,27, 53,78, 32,52, 64,32, 58,73,
  75,62, 72,21, 50,78, 73,63, 62, 4, 40,11, 24,28, 36,51,
  11,31, 39,41, 42,53, 40,50,  7,20, 76,61, 37,50, 40,79,
  31,44,  9,42, 10,38, 22,49, 67,70, 54, 9, 70,13, 25,36,
   3,34, 37,11, 74,57, 30,41, 64,69, 28,25,  1,78, 39,64,
   7,53, 41,74, 44,42, 64,70, 39,67, 71,67, 77, 5, 71,15,
  12,13, 25,61, 12,35, 77,67, 36,19, 23,59, 11,70,  8,67,
  42,13, 62,58, 57,16, 29, 8, 70, 0,  9,60, 44,10, 68,27,
   2,26, 32,53,  8,39, 68,16,  6,56, 58,70, 43,72, 26,18,
  51,49, 41,31, 37, 5, 52,79, 52,67, 62,31, 43,21, 50, 7,
  74,73, 48,21, 39, 9,  1,61, 40,62, 70,22, 47,61, 27,24,
  67,55, 65,24, 42,52, 11,32, 70,28, 15, 2, 21,57, 14,45,
  12,27, 63,38, 68,33, 54,41, 73,26, 61,10,  8,22, 32,75,
  33,31, 57,43, 34,25, 58,38,  9,47, 12,40, 17,20, 21,56,
  57,13, 74,74, 52,39, 77,79, 40,46,  8,69, 14,37, 60,58,
  73,28, 50,64, 68,21, 63,69, 38,24,  1, 5, 27,49, 70, 7,
   6,37, 34,17,  9,67, 56,54, 71,60, 62,50, 68,52, 21,67,
  24, 1,  0,46, 10,34, 42,55, 49,31, 20,18,  2,60, 10,29,
  37,44, 72,78, 45,70, 38,40, 12,32, 25,67,  6,69, 28,10,
  53,12, 68,50,  7,56,  0, 9, 75, 1, 56,28, 46,52, 59,17,
  50,51, 78, 5, 38,63, 34,52, 38,74, 66,74, 45,57, 23,66,
  36,65,  7,62, 53,69, 29,66, 27,54,  7,33,  0,62, 37,59,
  33, 5, 78,67, 49,43, 36,32, 29,33, 42,64,  0,20, 39,31,
  33, 9, 24,55, 64,21, 58,51, 20,61, 47,37, 78,11,  2,27,
  71,22, 25, 8, 38,15, 44,24, 56,24, 29,76, 31,48,  6,31,
  58,34,  2,65, 33,61,  2,31, 63,10, 53,59, 60,66, 24,34,
  18,59,  7, 8, 68, 1, 13,42, 14,16, 33,10, 31, 9,  0,19,
  47,55,  5,54, 25,27, 59,67, 17,56, 46, 7,  6,34, 72,73,
  60,56, 21,15, 51,59,  2,22, 46,39, 35,44, 29,43, 26,57,
  42,22, 52,78,  0,43, 46, 9, 15,28, 75,13, 73,58, 12,38,
  22,28,  5,46, 74,40, 25,32, 25,64, 46,79, 52,56, 72,22,
  46,30, 71,53, 10,52, 64,72, 16, 4,  5,20, 23, 5, 23,74,
  48,51, 13,78, 19,67, 49,46, 46,69, 23,65,  9,76,  7, 9,
  17,63, 41,57, 24,41, 73,56, 30,42, 16,47, 78,16, 79,27,
  64,71, 52,30,  0,27,  0,15, 67,76, 50,24,  2,32, 77,45,
  55,63, 51,50, 39, 3, 31, 2, 25,47, 52, 4, 20,17, 15,72,
  11, 6, 39, 5, 20,25,  6,11, 69,57,  7,65, 14,59, 51,61,
  49,70,  9,48, 62, 1, 52,22, 69,36, 67,52, 56, 6,  7,67,
  63,78, 39,20, 16,70, 50,36, 25,70, 37,70, 74,54, 56,27,
   4,47, 33, 3, 47,52, 21,36, 79,47, 51,19, 60,39, 47, 2,
  44,75, 65,65,  6,61, 13,18, 15,68, 39,38, 53,46, 79,66,
  58, 1, 11, 1, 10, 6, 63,21, 58,20, 40,25, 62,69, 57,55,
  36,44, 11,73, 38,26, 79,44, 32,42,  0,55, 40,71, 68,68,
  24,72, 50,58, 74,75, 34,75, 50,49, 49,24, 62,14, 56,71,
  10,64, 58,64, 36,34, 78,18, 13,20, 59,42, 62,28, 47,11,
  72, 1, 51,39, 56,13, 11, 8, 14,22, 56, 9, 68,53, 10,36,
  42,54, 66,79, 59,50, 16,60, 60,25,  4,55, 14,68, 41,58,
  16,43, 17, 4, 36,69, 21, 0, 61,37, 74,49, 68, 2, 16,59,
  45, 4, 33,43, 55,27, 71,44, 48,33,  6,24, 47,54, 39,45,
  63, 6,  1,47, 74,21, 25,51, 70, 3, 31, 5, 60,62, 25, 9,
  65,40, 75,11, 68,11, 72,51, 74,39,  7,45, 42, 0, 69,52,
   1, 3, 75,26, 27,19, 18,49, 35,43,  2,50, 55,68, 16,27,
  63,11, 68,49, 50, 9, 10,65, 27, 7, 27,79, 32,45, 35, 0,
  38,35, 19,75,  1,44, 34,65,  8,17, 47, 1, 76,70, 51,57,
  73,54, 41,17, 47,51, 20,39, 76,18,  5,42, 77,55, 72,76,
  10,54, 78,27,  9,46, 59,69, 33,68,  4,66,  0,52, 74,31,
  62,18, 52, 9, 49,63, 28,33, 73,62, 32, 8, 29,34, 77, 2,
   1,76,  0,79, 48,27, 41, 4, 48,79,  5,11, 59, 5, 48,22,
  38, 5, 25,15, 76,15, 70,76, 65,13, 18,28, 19,13, 10, 5,
   6,77, 60,27, 45,74, 11,42, 65,53, 64,24, 37,65, 36,72,
   6,25, 62,76,  4,72, 65, 3, 44,51, 44,20, 15,79, 59,24,
  57,75,  5,68,  9,14,  4,57, 32,26, 51,74, 45,21, 19,27,
  75,40, 71,52, 14,23, 54, 8, 54,29, 45,72, 21,73, 73,70,
  11, 5, 47,71, 70,68, 73,30, 59, 3, 67,60, 49, 1, 72,77,
   3,56, 77,70, 45, 7, 55,62, 56,10, 23,47, 45,69, 74,19,
  72,10, 15,53, 37,45, 17, 6, 28,30, 38,60, 27, 0, 60,34,
   5,33, 26,11,  5,72, 28, 4, 54,65, 51,62, 26,61, 62,70,
  24,50, 48,42, 54,30, 18,25,  0, 3, 57,11, 55,47, 53,24,
  26,20, 65,79, 63,27, 61,24, 68,71, 53,16, 53,61,  5, 9,
  52,49, 22,67, 31,20, 46,12,  2,17, 52,73, 19,18, 78,24,
  31,63, 47,14, 48,67, 17,39, 55,33, 18,45, 25,54, 53,28,
  43,33, 64, 8, 78,37, 25,19, 73,27, 51, 1, 15,70, 39,22,
  10,12, 56,17, 45,56, 18,36, 60,61, 30,21, 29,60, 29,47,
  10,43, 34,23, 61,14, 40,47, 19,72,  9,21, 64,13,  0,49,
  54,38, 55, 6, 42,24, 24,32, 55,20, 69,49, 27,43, 46,62,
  22,25, 33,78, 14, 3, 46, 3, 42,38, 73,38, 31,60, 61,61,
  72,53, 14, 0, 68,45, 32,60, 70,58,  7, 0, 45,40, 54,66,
   2,72, 25, 0, 17,34, 49,20, 26,17,  8,51, 27,60, 11,10,
  49,60, 17,64,  7,59, 10,33, 75,65, 54,59, 20,53, 69,24,
  57,76, 39,72,  3,20, 72,62,  8,32, 26,53, 32,33, 56,67,
  51,13, 21,46, 46,26, 79,68, 60,54,  7, 3, 46,43, 67,10,
  57,12, 66,73,  1,14, 76,42, 41,53,  6,71, 40,58, 79,16,
  46,10,  9,62, 67,64, 77,24, 76,10, 69, 0, 17,28, 71,42,
  66, 0, 72,28, 12,17, 43,46, 49,41, 18,77, 14,10, 47, 5,
  55, 2, 24,63, 25,35, 45,65, 75, 9, 25,23, 53,22, 32,73,
   0,76, 15,51,  3,36, 39,70,  2,67, 63,62, 61,19, 62,54,
  12,39, 39,27,  6,30, 55,78, 59,51, 74,48, 23,41, 41,41,
  40,16, 72,41, 54,33, 18,43, 67,16, 31,11,  0,58, 62,11,
  13,11, 46, 5, 51,54, 78,70, 38,54,  3,47, 42,29, 76,30,
  43,13, 41,66, 11,38, 67,23, 19,35, 19,33, 31,29, 43,48,
   1,36,  3,65, 56,32, 28,59, 22,51,  4,48, 76, 3,  7,54,
  46,45, 52, 6, 22,21, 78,10, 37,12,  9,33, 40,13, 32,19,
  43, 1,  4,40, 47,72, 16,30,  0,24, 60,21,  2, 7, 21,79,
  77,50,  5,27, 48,36, 52,21, 76,44, 79,51, 64, 5, 77,59,
  34,34, 34,13, 63,63, 19, 5, 19, 0, 65,50, 54,53, 79,23,
  15,45,  3,16,  3,72, 47, 4, 48, 5, 24,29, 60,23, 48,48,
  47,50, 69,56, 13, 2, 25,21, 25,74, 43,55, 65, 8, 53,70,
  46,64, 15,43,  7,39, 28,27,  3,23, 57,41, 47,13, 79,10,
  60,14, 14,67, 70, 6, 75,10, 35,30, 49,71, 72,57, 66,51,
  30,64, 33,29,  4,44, 60,45, 35,71, 69,47,  3,78, 23,72,
  75,22, 68,32, 68,74, 78, 7, 79,56,  4, 5, 33,17, 62,20,
   5,18, 53,52, 36,67, 62,60,  4,37,  4,42, 29,22, 52,47,
  25,34, 39,16, 38,79, 74,13,  6,67, 59, 1,  2,29, 35,18,
  16,78, 42,72, 40,37, 67,15, 72,46,  9,57, 38, 7, 69,76,
   7,76, 23,27, 28,68, 39,78, 11,61, 42,48, 19,53, 41, 7,
  49,77, 21,33, 20,22, 14,50, 57,20, 65,26, 33,19, 53,66,
  70,64, 29,21, 56,49, 41,30, 65,38, 32,50, 51,36, 79, 5,
  13,23, 50,10, 22, 3, 23,44, 72,71, 78,79, 12,65, 22,42,
  51,47, 28,57, 78,73,  5,30, 25,62, 29,62, 16,16, 29,54,
  63,15, 74,32
};
const int p_80_11[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1238,
  /// Array of preferences
  64,22, 54,37, 67, 4, 72,64,  4,56, 27,59, 65, 3, 37,77,
   0,16, 77,47, 11,20,  6,19, 73,45,  2,24, 61, 8, 21,18,
  63,77, 49,64,  0,18, 48,29, 53,39, 60,22, 23,49, 55,44,
  49,56, 61,30,  9,50,  5,40, 65,70, 56,36, 41, 9, 35,30,
   6,17, 42,63, 68,77, 17,47, 51,17, 20,20, 39,21,  8,38,
  48,45,  1,69, 25,36, 33,33, 71, 3, 69,26, 60,38, 14,50,
  59,67, 13,33, 14,58, 37,24, 62,53, 52,34, 46,34, 59,13,
  62,15, 20,37, 29,10, 63,16, 72,74, 37, 2, 12,65, 17,55,
  29,71, 35,18, 33,74, 12,69, 39,11, 67,10, 63,45, 18, 0,
  74,51, 55,36, 44, 2, 31,14, 53,15, 31,58, 39, 6, 68,49,
  51,64, 71,43,  4,70, 44, 3, 69,45, 38,43, 22,60, 73,75,
  76,22, 13,13, 36,36, 22,68, 40,11, 65,50, 71,33, 30,19,
  35,29, 48,24, 51,51, 21,51,  6,74, 32,53, 34,77, 56,50,
   4,34, 13,55, 39,14, 11,66, 19,60, 77,38, 37,15, 17,52,
  23,14, 28,56, 77,58,  1,74, 19,25, 55,72, 78,64, 16,14,
   0,45, 23,24, 50,78, 13,66, 53, 6, 47,11,  0,48, 13, 0,
  79,66, 38,59, 74,11, 57,29, 48,19, 11,24,  5,39, 40,55,
   3,24, 47, 4, 32,29, 61, 6, 23,22, 35,23, 75,50, 32,30,
  74,79, 54,50, 42,77,  5, 8, 12,70, 24,74, 28,33,  8,46,
  21,14, 15,52, 26,76, 11,49, 10,19, 54,65, 52,10, 75,38,
  58,26, 21, 2,  2,23, 54,38, 79,67, 75, 0, 47,16, 45,16,
  44,79, 75,22, 61, 9, 67, 0, 40,66, 32,46, 21,30,  3,78,
  14,23, 15,28, 10,49, 10,42, 26,17, 46,23, 74,39, 14,77,
  37, 0,  2, 4, 79,34, 11,62, 76,66, 60, 1, 29,40, 31,57,
  14,46, 62,13,  0,26, 31,69,  9,24, 16,22, 26, 9, 59,51,
  27,70, 51,50, 26,31, 23,55, 27,16, 30,52, 25,68,  3,62,
   2,55, 77,46, 14,24, 45,24, 55,22, 19,29, 24, 2, 51, 0,
  18,52, 22,40, 10,48, 14, 8, 33,28, 58,16, 15,59, 31,64,
  16,62,  9, 8, 62,18, 60,39, 26,63, 42,43, 33,27, 40,74,
  24,49, 46,36, 76,77, 23,28, 35,45, 36,18, 29,74, 32,19,
  24,47,  7,12,  5,62, 42,41,  6, 8, 56,16, 33,42, 22,56,
  36,65, 53,23, 74,55, 27,52, 11,79, 51,33, 56,44, 71, 0,
  30,50, 56,60, 53,46, 68,29, 72,49, 44,54, 61,60, 72,37,
  31,61, 64,20, 48,33, 78,49, 64,38, 63,26, 77,26, 41,35,
   0,55, 67,39, 43,60, 70,25, 69,64, 79, 7, 30,41, 12, 6,
  36, 8, 53,37, 48, 1, 20, 6,  4,33, 57, 7, 36,62,  6,14,
  20,26, 15,63, 25,48, 32,67, 36,21, 66,66, 11, 8, 46,26,
  60, 3, 77,10, 13,64, 57,72, 36,26, 20,18, 59,41, 35,51,
  47,10, 15,78, 66,48, 26,56, 27,35,  6,13, 56,29, 59,69,
  22,25, 44,75,  6,55,  9,13,  0,21,  5,59,  8,35,  9,47,
  46,11,  1,17, 59,29, 20,69, 60,14, 14,59, 14,10, 51, 8,
  73,51, 76,25, 11,41, 35,65, 78,14, 15,51, 76,45, 39,68,
  42,36, 64,63, 73,63, 44,70, 27,37, 51,49, 67,27, 72,46,
  33,70, 27,12,  7,56,  4,79, 13,41, 49,30, 79, 6, 74, 5,
  46, 0, 45,78, 31,15, 70,36, 64,56, 27,13, 33,21,  8,29,
   3,75,  5,16, 18,59, 52,38, 36, 5, 33,46, 25,26, 72, 2,
  13,74, 56,79,  0,73,  6,20,  6,77,  1,67, 76,43, 30,44,
  28,44, 74,54, 50,73, 63,11, 61,43, 44,22, 18,46, 34,74,
  36,16, 23,47, 24,72, 49,34, 72,51, 13,73, 65,52, 41,44,
  39,42, 43,46, 49,61,  6,60, 45,63, 25,41,  2,44, 25,66,
  26,75, 22,27, 55,75, 74, 9,  5,68,  2,35, 38,19, 47,56,
  52,29, 67,68, 29, 0,  3,14, 46,18, 52,75, 35,20, 10, 6,
   2,64, 78,19, 20,10, 10,78, 18,49, 26,48, 27,10, 48, 5,
  71,68, 16,18, 74, 1, 44,62, 26,52, 71,10, 44,45, 13,68,
  62,69, 74,22, 50,74, 38,31, 79,63, 11,47, 34,39, 25,62,
  28,58, 79,44, 41,31, 60,52, 79,71, 73,55, 54,75, 47,12,
  14,41, 19,22, 32, 9, 60,49, 21,61, 53,49, 48, 4, 16,19,
   4,64, 38,51, 76, 3,  0,34, 33,68, 63,30, 57,35, 17,56,
  59,16, 16,32, 62, 8, 18,66, 60,65,  5,74, 15,38, 36,30,
  44,67, 27,60, 10,58, 62,16, 19,49, 58,55, 79,13, 15,13,
  46,41, 72,62, 76,16, 63,14, 38,20, 64,62, 72, 7, 66,63,
  50,22, 54,25, 23,45, 48,50, 42,35, 21, 5, 29,59, 37,43,
  51,66, 31,45, 52,51, 40,49, 43,29, 40,67, 49, 2,  9, 0,
  18,38, 32,79,  5,11, 40,79, 24,62, 16, 0, 20, 9, 48,13,
  13,59,  8, 3, 20,63, 40,35, 70,58, 76,59, 70,50, 54,31,
  66,37, 23, 0,  9,74, 23, 5, 78,44, 74,30, 62,62, 57, 5,
  77,30, 74,40, 69,50, 63,73, 67,71, 29,17, 66, 0, 43,67,
  22,69, 36,76, 73,61, 35,52, 48,41, 43,73,  2,33, 61,56,
  16,12,  8, 9, 61,40, 77,34, 30,79, 66,59, 25,78, 14,73,
  35, 4, 50,59,  0, 3,  0,15, 37, 6,  3,27, 72,79, 59,40,
  45,77,  1,79, 23,29, 38,23, 26,20, 32, 6, 39, 2,  5,75,
  56,61, 12,47, 33,24,  4,61, 74,43, 44,57, 29, 8, 32,31,
   0,69, 33, 1, 58,67, 56,63, 42,37, 36,69, 58,58,  9,42,
  50,28, 46, 9, 14,64, 12,66, 26, 4, 12,18, 41,67,  7,39,
  36,53,  0,22, 44,17,  9,52, 36,50, 31,17, 35,69, 72,34,
  26,65, 75,47, 16,13, 39,35, 43,33, 41,76, 70,62, 63, 2,
  43,27, 64, 1, 43,28, 32,18, 70,52, 62,33, 58,63, 67, 9,
  14,28, 27,42, 51,74, 26,35, 10,51, 49,13, 29,38, 54, 7,
  32, 0, 62,66, 43,53,  9,20, 63,35, 13,70, 51, 2, 71,35,
  51,69, 51,56, 64,55, 17,23, 54,41, 28, 4, 33,76, 60,78,
  58,32, 61,73, 37,49, 46, 7, 60,63, 47,44,  5,24, 75,71,
   6,42, 37,72, 50,55, 16, 5,  9,16, 43,62, 56,45, 73,49,
   2,14, 34,34, 49,23, 59,78, 52,22, 71,42, 20, 0, 71,70,
  42,75, 74,37, 45,79, 71,50, 10, 8, 74,44, 72,66,  7,71,
  76,63,  5, 1, 44, 4, 31,18, 24,42, 69, 8, 13,63, 58,10,
   5,53, 75, 8, 28,70, 42,67, 30,51, 38,12, 55,64, 65,56,
  77, 7, 54,77,  4, 1, 62,47, 50,41, 21, 7, 63,54, 35,31,
  56,14, 59, 5,  2, 6,  6,25, 77,35, 50,32, 29,67,  3,63,
   2,42, 70,70, 37,27, 18,45, 48,59, 30, 1, 10,38, 54, 1,
  63,31, 25, 9, 23,57, 33, 7, 39,54, 70,12, 42,27, 58,20,
  45, 2, 18,26, 67,24, 73,52, 21,10, 34,68, 60,33, 23, 6,
   5,28, 30,55, 66,75, 77,69,  2,67, 20,71, 31,62, 70,48,
  39,45, 61,12, 29,20,  0,62, 17,24, 41,75, 28,31, 57, 2,
   2,31, 36,37, 42,14, 20,65, 22,39, 72,32, 11,57, 41, 0,
  32,12, 63,34, 74,29, 78, 3, 74,68,  9,51, 52,76,  6,40,
  14,45, 71,14, 45,51, 27,71,  2,49, 51,41, 65,21, 27,74,
  56,64, 52,45, 71,55,  5,60, 61,20, 39,23, 45,21, 12,36,
  53,29, 45, 7, 78,52, 68,76, 33,34, 74,69, 74,36,  7,28,
  45,10, 72,24, 43,42, 28,74, 77, 4, 75,12, 47,17, 31,33,
  39,18, 67,78, 52,24, 78, 8, 59,17, 69,69, 10,33, 28, 3,
  50,77, 49,46, 29,79,  4,37, 47,59, 11,37,  5,58, 29,43,
   8,28, 14,20, 52, 1, 35,63,  8,63, 53,56, 22,64, 43,10,
  28, 2, 29,65, 75,78, 32,59, 43,18, 76,74, 23,54, 57,15,
  63,65, 79,52, 67,26, 45,19, 20, 5, 68,32, 20,25, 63,58,
  29,25, 61,51,  7,31, 52,68, 73, 6, 12,72, 30,20, 54,42,
  58,24, 34,23, 29, 2, 34,59, 43,11, 70, 6, 27,31, 19, 5,
  63,36,  4,67, 74, 3, 68,68, 23,77, 78,21, 26,43, 10,35,
  77,52, 50,46, 53,55, 32,54, 33,50, 40,32,  1,28, 44,61,
  54,32, 56,19, 39,27,  8, 6, 27,17, 75,72, 37,54, 17,20,
  15,45, 64,75, 31,38, 42,23, 54,36, 28,10, 71,59,  8,67,
  30,33, 13,22, 65,59, 60,76, 21,59,  8,70, 12,37, 57,71,
  30,73,  1,46,  8,60, 35,50, 51,11, 14, 7, 50, 7,  5, 5,
  51,71, 20,23, 34,75,  5,30, 74,16,  8, 0,  6,45, 53,73,
  73,36, 46,39, 54,56, 72,38, 23,17, 40, 9,  2,54,  0,49,
  64,72, 33,36, 44,42, 55,34, 19,72,  9,27,  9,22, 57,23,
  45,59, 28,23, 53,43, 77,62, 76,69, 10,64, 75,62, 12,26,
  79,22, 29,69, 58,56, 64,43, 30,32, 74,26,  1,53, 12,63,
  32,66, 59,53, 69,10, 63,37,  4, 3, 79,54, 10,39, 47,37,
  79, 5, 17,68, 26,51, 14,25, 59,71,  8,19, 65,24, 18,69,
  34, 7, 33,58,  0,39, 10,62, 17,69, 69, 7, 41,57,  5, 2,
   1,37, 18,76,  4,73, 36, 0, 13,27, 51,77, 60,66, 43,20,
  30, 4, 13,76, 21,31, 42,73, 75,75, 69,57, 31, 1, 44,66,
  49,29, 79,16, 61,79, 77, 1, 12,12, 27, 2, 76, 1, 15,30,
  42,51, 25,52, 22,42, 35, 2, 76,31, 11,27, 52,53, 75,45,
  75,60, 24,33,  9,46,  4,68, 53,27, 50,53, 28,41, 38, 3,
  11, 7, 42,78, 15,20, 76,62, 50,23, 28,78, 36,46, 66,36,
   0,28, 22, 7, 10,50, 46,33, 54,64, 52,14, 21,37, 54,13,
  75,67, 35,48, 44,64, 56,71, 72,20, 55,66, 77,66, 28,57,
  24,46, 21,29, 42,66, 73, 5, 45, 0, 72,11, 57,70, 70,57,
  75,37, 21,28, 46,71, 39,22, 66,57, 33, 8,  0,19, 22,67,
   1,24,  9,38, 40,46, 32,77, 20,58, 31,21, 20,61, 30,70,
  70,47, 54,10, 73,27,  0, 1, 79,75, 27,56, 47,31, 14,14,
  19,15, 14, 9,  5,29,  0,70, 31,54, 13,18, 51,58, 63,41,
  72,72, 31,63, 17,19, 40,42, 47,40, 59,65, 40,41, 61,19,
  43,22,  2, 3, 22, 1, 29,31,  0,51,  4,42, 39,62, 71,66,
  20,68, 30,26, 48,16, 69,68, 34,35, 72,54, 37,57, 73,74,
  28,22,  5,44, 27,23, 62,42, 43,56, 44,55,  3,16, 36, 6,
  78, 1, 58,39, 60,19, 77,43,  1,12, 49, 3, 17,43,  8,56,
  16,20, 24,69, 20,74,  3,22, 79,60, 69,41, 79,15, 48,55,
  37, 3, 39,71, 51, 7, 68,55, 34,13,  9,67, 48,20, 23,36,
  48,64, 23,58, 41,10, 55,38, 10, 7, 38,60, 59,36, 25, 2,
  44,69, 27,50, 20,45, 52,66, 10,60, 64,49, 19,24, 24,36,
  20,21, 10,63, 22,24,  8,69, 26,49, 25,19, 63,75, 73,16,
  30,14, 53,30, 40,56, 26,36,  6,76, 73,20, 71,19, 36,39,
  34,65, 50,45, 57,56, 74,45, 49,21, 76, 6, 57,57, 50,40,
  17,54, 66,43, 56,35, 73,33,  0,61, 65,53, 63, 0, 51,20,
  75,11,  4,29, 10,28, 27,41, 51,52, 41, 3, 58,62, 49,52,
  53,26, 27,19, 53,12, 66,11, 66,16,  6, 6, 15,65, 65,77,
  10, 5,  4, 6, 76,29, 75,16, 28, 9, 46,14, 54,44, 53,70,
  67,22, 59,38, 75,32,  3,46, 21,77, 13,52, 16,42, 29,54,
  16,27, 70,73, 52,20, 73,30, 30,76, 38,63,  5,56, 51, 9,
  65,47, 74,49, 67,52, 66,50, 62,32, 42,10,  4,66,  8,37,
  38,28, 42,32, 44,14, 34,37, 59,58, 66,21, 28,50, 30,47,
  10,74, 49,76, 24,57, 10,17, 36,22, 74, 0, 16,33, 57,44,
  59,24,  0, 8, 15,67, 45,62, 50,44, 35,68
};
const int p_80_12[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  262,
  /// Array of preferences
  55,15,  8,14, 79,53, 23,31, 26,37, 22,62, 77, 0, 57, 7,
  78,14, 70,30, 32,63, 73, 1, 47,69, 20, 2, 35,56, 50,54,
  48, 8, 41,41, 65,73,  4,56, 31,52, 27,53, 45,34, 22,75,
  77,46,  2,55, 13,61,  9,51, 68,40, 11,28, 19,39, 42,16,
   6,23, 11,19, 18,10, 42,27, 76, 8, 49,21, 12,50, 70,64,
  51,74, 26, 2, 74,50, 18,13, 28,52, 52,44, 27,79, 57,55,
  53,48,  5,64, 70,34, 26,16, 60,58, 52,50, 68,51, 10,49,
  43,36, 58,51, 26, 1, 60,11,  4,72,  9,50, 29,46, 52,51,
  20,21, 49,19, 69, 0, 41,57, 35,19, 69,74, 54,31, 62,49,
  69,76, 21,40, 36,33, 76,38, 30,67, 38,23, 51, 7,  8,54,
  66,44, 61,49, 72,48, 58,44,  7,26, 22, 8, 77,62, 41,48,
  34,75, 46, 9, 20,49, 40,35, 57,61, 67,61, 33,17, 29,60,
  21,35, 75,61, 55,32, 10,18, 23,53, 60,65, 66,48, 38,33,
   5, 5, 48,60, 59,17, 57,45,  0,34, 64,24, 30,65,  0,42,
  43,23, 18,50, 53,50, 47,68, 23,19, 47, 4, 57, 5, 68,16,
  36,74, 28, 8, 72,20, 33,36, 37,54,  4,59, 43,52, 11, 1,
  66,30, 30, 1, 41,23, 33,23, 54,26, 43,64, 14,18, 50,29,
  53,66, 46,75, 35,11, 71, 8, 55,29, 21,71, 76,31, 23,79,
  74,78, 51,38, 14,51, 29,45,  2,37, 12, 4, 40, 1, 46,69,
  60,23, 58,33, 76,64, 69,51,  2,24, 14,69, 27,10, 47, 7,
  13,34,  0,16, 51,62, 77,38, 60, 7, 76,24, 32,67, 22,60,
  72, 5,  9,48, 31,28, 11,10, 13,64,  3,77, 59,68, 39,77,
   1,60,  7,43, 49,39, 36,44, 30,70, 77,31, 17,49,  7,46,
  44,42, 25,13, 60,63, 19,72, 11,31, 63,36, 70,15, 42,41,
   3,47, 23,37,  7, 1,  9,15, 75,25,  8, 6, 45,61, 54,22,
   2,43, 44,63, 17,77, 24,36, 56,75, 71, 4, 76,61, 43,70,
  45,70, 39, 2, 77,17, 39, 1, 10,47, 67,16, 77, 6, 70,69,
   9,75, 40,22,  0,73, 25,37, 63,22, 29,36, 52,70, 36,72,
  16,69, 15,31, 21,39, 76, 5, 56,64, 18,38,  9,30, 29,63,
   9, 4, 74,76, 75, 3, 28,54, 71,11, 67,56, 29,30, 16,78,
  40,32, 12,47, 43, 1, 57,64, 28,13, 30,79, 77,49, 48,75,
  52,77, 12,23, 78,34,  7,68, 21,58, 54,36, 58,25, 62, 5,
  41,75, 26,76, 43,63, 17, 0, 14,26, 44,49
};
const int p_80_13[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  363,
  /// Array of preferences
  49,34, 17,15, 11,20, 17,10, 47,31, 63,56, 60, 3, 36,39,
   5,76, 76,21, 73,21, 58,10, 74, 3,  7,16, 22,46, 65,11,
  28,29, 44,31, 26,35,  0,60, 65, 4,  4,79, 46,77, 10,60,
  48,61,  5,22, 19,72, 46,56, 59,30, 54, 8,  7,46, 34,34,
  22,31, 45,11, 50,10, 17,74, 55,56, 39,51, 67,76, 39,75,
  77,38, 42,73, 34,40, 74, 4, 36,68, 54,63, 48,59, 18,27,
  64,26, 76,49, 71,66, 20, 1,  5,71,  1,58, 77,59, 20,21,
  25,16, 27,48, 71, 5, 78,30, 54,58, 68,77, 67,35, 67,31,
  43, 4, 36,75, 67,63,  0,31, 65,62, 33,70, 16,72, 17,32,
  23,74, 78,16, 51, 7, 71,56, 28, 3, 15,63, 68,22, 68,24,
  26,45, 25,33, 67,53, 72,23, 53,28, 74,47, 19,38, 50,75,
  19,71, 66,39, 79,73, 25, 7, 24,10,  8, 4, 14,24, 69,19,
  21,62, 47,18, 25,43, 57,56, 31, 8,  4,60, 25,71, 43,76,
  11,73,  6, 1,  7,66, 54,46, 13,63, 25,31, 18, 5, 49,76,
   7,54, 40,76, 39,50, 49,38, 73,62, 68,49, 51,51, 74,29,
  13,17, 74,50, 26,16, 58,65, 36,64, 64,41, 47,58, 68,72,
  29, 2, 72,66, 68, 6, 79, 9, 19,39, 33,29, 16,34, 58,73,
   8,37, 68, 4, 79,44,  5,54, 45,17, 30,30, 33,56, 48,40,
  27,23, 39,17, 75,10, 64,56, 57,39, 25,70, 26, 4, 40,23,
  40,25, 78,76, 30,25, 69,28, 47,38, 56,33, 67,55,  0, 4,
  20,20, 55,51, 21,29, 68,39, 73,30, 16,75, 76,44, 59,11,
  55, 9, 75,27, 43, 2, 53,40, 41,62,  1,78, 64, 7, 68,11,
   7,36, 10, 3, 31,59, 15,56, 11,24,  1,27,  8,72, 72,10,
  70, 3, 49,43, 74,22, 26,69, 70,61, 67,56, 51,60, 40,22,
  21,53, 40,38, 16,16, 21,72, 47,61, 33,25, 58,23, 15, 2,
  64,60, 36,10, 35,39, 17, 1, 36,44, 15,40, 35,46, 50,30,
  42,77, 52,77, 17,29, 13,28, 21,42, 68,26, 35,14, 49, 4,
  47,41, 27,47,  7,32, 36,63, 49,28, 32,76, 20,34, 36, 1,
  48,55, 54, 3, 34,76, 63,53, 57, 9, 30,48, 63,34, 43,69,
  39,78,  5,20, 57,50, 30,71, 64,76,  2,21,  6,47, 75,69,
  59,67, 64,21,  1, 8, 48,31, 55,65, 29,63,  7,41, 44,14,
  54,34, 23,24, 65,10, 13,55, 51,13, 66,33, 34,23,  7,76,
  14, 4, 38,49, 52,65, 20,52, 71,41, 25, 0, 46,61, 69,68,
  68, 5, 56,23,  8,42,  8,50,  5,27, 31,20, 32,52, 25,64,
  75,44, 38, 9, 47,22, 77,23,  2,41, 60,78, 67,64, 29,75,
  64,62, 40,28, 39,63,  2,73, 47, 2, 33, 0, 22,18, 59,31,
  56,49,  5,29, 18,55, 79,52, 11, 3, 39,57, 56,54, 56,18,
   7,69, 59,22, 70,20, 36,35, 58,31, 50,34, 77,24, 48,48,
  37,24, 37,59, 47, 7, 61,13, 71,37,  0,12,  8,10, 35,35,
  24,32, 56, 1,  8,15, 68,15, 76, 5, 30,22, 35,22,  1, 4,
   5,49, 73,56, 18,69, 78,70, 66,21,  4,52, 40,11, 23,36,
  69,40, 35,18, 57,74, 73,41, 77, 5, 68,50, 70,41, 24, 4,
  49, 3, 46,19, 50,59, 27,18, 17,17, 37,33, 26,49, 37,76,
  54,28, 50,26,  9,12,  7,72, 12,24,  0,47, 37,55,  3,61,
   6,63,  0,21, 28,16, 34, 1, 17,53, 61,35, 22, 1, 39,35,
  51,69, 15,37,  5,12
};
const int p_80_14[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1576,
  /// Array of preferences
  77, 6, 31,55, 12, 6,  8,72, 27, 7, 29, 5, 75,17, 33,19,
  15, 1, 72,33, 67,19, 73,32, 68,75, 11,38, 61,12, 16,59,
  46,63, 71,77, 52,67, 47, 0, 63,55, 27,67, 74,55, 60,47,
  20,57, 35,59, 21,33, 63,46, 44,73, 75,66, 60,24, 72,36,
  77,10, 56, 0, 39,60, 14,23, 42,62, 12,22, 61,77, 37, 5,
  74,44,  5,11, 77,48, 61,57, 48,64, 26, 0, 24,57, 70,22,
  77,13, 52,41, 33, 6, 79,21, 79,75, 51,13, 31,32, 22,73,
  52,47, 71,48, 17,71, 33,45, 24,63, 10,49, 37,30, 56,17,
  28,40,  1,77, 77,70, 70,39, 23,11, 65,27, 54,36,  5,65,
  36, 5, 14,54, 68,66, 10,52, 49,33, 40,50, 20,12, 25,30,
  55,42, 39,16, 21,26, 52,10, 42,29, 52,78, 33,27, 35,60,
  27, 1, 25,31, 39,30, 36,36, 67,79, 31, 2, 70,67, 66,15,
  25,66, 70,45, 52, 4, 64,77,  0,36, 57,55, 17,65, 38,31,
  55,78,  2,30, 31,21, 32,14, 51,60, 26, 1, 50,45, 20,21,
  35,51, 45,28, 78,53, 73,39, 26,24, 77,56, 59,68, 26,34,
  22,53, 50,43, 73,36, 68,40, 52,34, 22, 3,  8,20, 35,42,
  65,11, 66,42, 69,60, 72,54, 46,47,  2,74, 13,40, 22,47,
  59,51, 44,37, 61, 8, 32,12, 34,66, 16,15, 10,74, 53,74,
  51,19, 48,37, 40,37,  2,31, 63, 6, 34,41, 38,45, 75,12,
  18,32, 20,65, 19,67, 38, 1, 72,21, 42,71, 35, 7, 27,72,
  15,43,  8, 9, 73,35, 21,19,  8,36,  9,73, 42,14, 14, 5,
   5,45, 75,23, 16,31, 28,69, 73,58, 10,62, 36,14, 74,19,
  18,47, 53,15,  1,21, 24,78, 22,50, 75,78, 68,58, 31,16,
  15,29, 15,19, 60, 7, 74,10, 51,68, 21,78,  1,57, 45,55,
   1,23, 47,79, 56,48, 20,14, 58,29, 27,79, 41,27, 33,51,
   3,64, 19,73, 75,55, 77,53, 37,24, 65,18, 58,26, 37, 9,
  41,64, 52,49, 76,42, 25,27, 57, 5, 32,17, 42,70, 15,42,
   7,42, 46,50, 60, 2, 69,35, 13,14, 24,62, 11,60, 14,32,
  72,46, 51, 7, 72,35, 41,17,  7, 0, 53,20,  2,76, 74,56,
  14,10, 66, 7,  7, 3, 66,69,  1,74, 77,11, 71,34, 58,14,
  50,44, 25,53, 17,74, 73,18, 45,26, 14,72, 75,76, 43,34,
  12,67, 23,71, 10,64, 30,64, 55,46, 68,76,  6,24, 79,71,
  73,59,  2, 7, 63,15, 53, 7, 18,25, 67,77, 68,65, 23,10,
  37,16, 79,24, 45,10, 75,74, 72,30, 13, 3,  7,47, 51,54,
  74,26, 36,60, 79,67,  4,26, 18,18, 40,27, 50,33, 42, 2,
   2,15, 14,30, 70,47, 50, 8, 57,64, 15,72, 66,10, 44,12,
  34,48, 37,78, 60,25, 68,32, 14,49, 35,71, 34, 2, 27,61,
  59,57, 38,62, 16,49, 28,24,  3,70,  9,13,  2,79, 77,12,
  16,50,  0,14, 61, 4,  2,61, 47,33, 45, 1, 45,72, 25,12,
  47,32,  9,57, 76,52, 65,70, 34,69, 47,73, 69,72, 16,13,
  67,49, 19,58, 23,64, 67,43, 39,79, 12,48, 71,54, 67,64,
  38, 4, 45,24, 38,28, 26,30, 45,70, 64,52, 75,27, 12,24,
   2,68, 39,44, 63,49, 68,14, 11,78, 41,48, 18, 6, 39,34,
  56,65,  3,34, 54,75, 51, 0, 40,28, 61,50, 28,49, 56,50,
  59,32, 27,34, 74,32, 70,23, 76,22, 11,34, 62, 4, 62,19,
   6, 6, 19,42, 30,47, 45,11,  9,22, 42,22,  8,59,  9, 0,
  62,56, 45,18, 61, 2, 76, 5, 15,25, 55,65, 11,36,  9,72,
  55,19, 70,36, 76,65, 12,62, 76,47, 57,43, 37,17, 73,30,
  41,37, 11, 5,  5, 6, 35,27, 44,50, 33,28,  2,51, 29,68,
  53,35, 52, 9, 26,12, 46,51, 54,31, 59,22, 43,53, 41,19,
  72, 4, 21,17, 73, 4, 31,33, 36,35, 39,68, 13,66, 18,41,
  15,75, 41,53, 32,39, 46,59, 35,56, 17,44, 13,72, 11,75,
  63,56, 50,30, 46, 6, 70,40, 62,27, 53,54, 16, 4, 67,58,
  41,18, 62,78, 57,53, 77,57, 35,13, 57,75,  0,48, 35, 5,
  50,59, 24, 4,  7,40, 24,52, 37,37, 15,39, 26,29, 14, 8,
  64,43, 42,79, 62,17,  8,75, 77,62,  4,38, 47,40, 35,40,
  23,17, 78,59, 10,42, 44,57, 53,70, 73,79, 78,45, 51,42,
  29,69, 19,79, 20,39, 44,60, 31,44, 19,27, 22, 5, 73,71,
  38,78, 68,25, 64,30, 41,11, 34,63, 64,51, 35,28, 51,77,
  65, 3, 79,44, 51,62, 69,18, 65,29, 11,11, 44,18, 79,47,
  56,25,  6,60, 31,11, 11,12, 34,25, 39,15, 63,24, 53,55,
   9,76, 21,11, 10,47, 46,67, 51,18,  0,40, 27,43, 48,34,
  56,76, 23,37, 12,69,  6,25, 30,52, 21,71, 31,28, 19,18,
  12,79, 14,43,  5,35, 26,21, 49, 7, 21, 7,  1,79, 37,57,
  69,64, 70,61, 25,52, 73,40, 23,15, 46,45, 16,40, 70,75,
  44,55, 12,61, 62,65, 77,46, 10,77, 43,38, 78,44, 50,74,
  68,38, 37,74, 38,58, 59,48, 30,50, 78,68, 13,13, 16, 3,
  37,12,  6,14, 68,15, 73,38, 49,51, 40, 6, 51,50, 46,37,
  66, 9, 66,51, 55,10, 66,78,  0, 5, 13, 9, 10, 2, 51,35,
   7,53, 24,30, 17,73, 31,26, 11,17, 12,29, 15, 6, 13,69,
  29,54, 37,40,  9,30, 31,19, 73,70, 33,18, 75,39, 28, 3,
   7,46, 13,50, 51,74, 15,22, 26,65, 58, 0, 16,11,  7,62,
  33,56,  7,79, 54,22, 41,56,  3,49, 42,52, 53,58, 46,65,
  71,67, 17,61, 71,73, 46, 4, 27, 5, 44,10, 64,14, 11,13,
  41, 7,  1,29, 58, 4, 49,11, 29,51,  9,18, 59,35, 77,14,
  42, 4, 37,66, 47,36, 76,28, 60,30, 65, 7, 28,57, 25,39,
  49,52, 59,50, 64,22, 73,31, 75,29, 14, 7, 78,11, 63,64,
  73,78, 46, 8, 64,69, 41,73,  9,55, 39,14, 16,18, 27,10,
   3,68, 63,70, 42,64, 64,44, 48,68, 15, 9, 23, 8, 24,36,
  75,15, 33,57, 59,73, 28,61, 48, 9, 45,60, 20,37, 72,57,
  74,14, 41,49,  0, 1, 10, 5, 30,44, 22,60, 58,69, 33,49,
  33,60, 70,16, 60,37, 15,21,  9,78, 10, 7, 72, 3, 70,48,
  23,12, 37,48, 18,61, 79,73, 51,10, 37,55, 49,21, 43,42,
  13,26, 34,34, 33,78, 70,32, 60,38, 44,62, 32,24, 65,73,
  49,41, 18,78,  4,48, 22, 4, 16,24, 77,74, 33,65,  9,79,
  59,54, 54,48,  6,57, 66,46, 61, 6, 60,64, 25,36, 34,75,
  42,63, 31,23, 64,48, 26,52, 25,48,  9,25, 35,19, 44,72,
  35,70, 20, 4, 58,61,  4,21, 62,67, 69,10, 22,62, 14,60,
  78,57, 52,74, 33, 9, 46,11, 31,50, 63,38, 41,52, 70,56,
  38, 6, 22, 2, 45,23, 18,12, 40,35, 75,21, 68,60,  7,68,
  38,29, 44, 8, 24,55, 57,52, 54,38, 15,11,  9,36, 21,22,
  71,45, 21,30, 48,75,  3,75, 62,73, 60,19, 71,35, 76,55,
  54,66, 42,56, 10,17, 16,58, 12,46, 32,63, 60,27, 33,41,
  67,37, 45, 0, 48, 6, 54,79, 71,66, 56,67, 72,25, 57,20,
  45,77, 17,22, 16,52, 32,66, 36,62, 13, 1, 61,46, 25,18,
  27,75, 15,73, 43,50, 28,59, 22,37,  6,23, 66,74, 52,50,
  37,70, 32,26, 65,13, 77,18, 77,31,  2, 2, 36, 3, 41, 8,
  35,15, 40,12, 42,27,  0,79, 43,20, 55,54, 35,21, 59,56,
  28,55, 59,60, 76,38,  7,56, 74,43, 58,51, 31,72, 50,66,
  17,72, 37,68, 14,46, 77,17, 54,24, 43,12,  0,64, 50,28,
  31,42, 27,42, 58,42, 57, 3, 74,15, 50,22, 61,52, 19,75,
  61,21, 72,76, 40,72, 63,57, 12,58, 71,56, 45,21, 68,33,
  22,20, 18,22, 25, 9, 25, 3, 64,31, 38,74, 76,26, 13,47,
  43,25, 34,35, 40, 1, 15,33, 71, 5, 42,69, 57,38, 22,74,
  23, 4,  0,49, 79,26, 70, 1, 48,50, 77, 5, 58,67, 57,17,
  29,56, 73,47, 30,39, 27,68, 71,23, 31,68, 46,39, 46,44,
  10, 6, 28,70, 36,32, 17,64, 69,26, 35, 4, 45,51, 36,31,
  44,67, 14, 6,  0, 4, 47,65, 56,64, 61,60,  6,17, 52,35,
   6,62, 67, 6, 21,36, 11, 7, 62,13,  3,79, 51,33,  5,54,
  19,68, 62,38, 69,14, 35,44, 52,63, 32,19, 11,26, 34,38,
  15,59, 48,44, 10,75,  2,58,  1,73, 32,73, 13,55, 27,47,
  59,38, 18,58, 40,34,  8,25, 21,23, 53,16, 70,70,  9,43,
  22,31, 73,61, 46,18, 28,12, 73,67, 17,58, 62,39, 23,31,
  49,40,  3,11, 11,70, 76,37, 56,31, 45,50, 30,49, 24,42,
  70,78, 78,13,  4,52, 50,31, 17,32,  0,12, 72,14, 61, 0,
  40,64,  2,53, 11,47, 13,27, 79,39, 24,70, 20,51, 78,79,
  54,42, 40,22, 59,62, 14,66, 66,32, 32,11, 35,76, 48,16,
  63,79, 32,31,  9,45, 27,39, 61,68, 54,40, 28,10,  8,26,
  24,46, 15,30, 13,37, 39,66, 12,64, 31,10, 18,45, 41,32,
  50, 1, 43,37, 44, 2, 48,70, 12,45, 74,17, 65,75,  0,54,
  44,40,  8,53, 21,50, 62,59, 57,39, 78,58, 75,48, 31,40,
  53,33, 47,25, 32,72, 63,17, 35,61, 11, 6, 25,16, 63,44,
   6,73, 62,34,  5,70, 51,30, 39,25, 28,38, 74,12, 75,58,
  24, 5,  5,43, 66,62, 68,22, 25,68, 70,29, 73,10, 13, 6,
  41,66, 70,34, 36,12, 58,40, 55, 6,  5,50, 71,14, 50,51,
  71,19, 21,62, 70,33, 28,31, 53,65,  2,39,  1,58, 55,24,
  64,12, 62,21, 20,41, 53,59, 79,49, 25,78, 72,79, 65,43,
  79,19, 26,10, 43, 7, 70,15, 50,12,  7,10, 60,79, 39,23,
  29,49, 21, 3,  1,54, 57,27, 12,11, 78,43, 16,70, 57,31,
  19,45, 54,74, 19, 4, 62,30,  2,70,  6,29, 63,61, 27,29,
  66,73, 76,34, 47,57, 77,77, 13,77, 10,60, 47,46, 63,16,
  66,20, 52,13, 27, 6, 40,67, 41,78, 20,79, 17,34, 59,34,
  11,25, 53,29, 39,33, 42,55, 27,26, 71,21, 49,68, 32,59,
   9, 5, 42,77, 56,52, 40,58, 62,32, 77,25, 26,20, 60,39,
  42,46,  2,69, 37, 4, 52,68, 66,17, 24,79, 10,33, 66, 5,
  39,59,  2,18, 70, 2, 22,41, 38,25,  4,46, 34, 6, 50,36,
  37,22, 21,54, 32, 4, 72,55, 79,37, 40,41, 11,61, 44, 9,
  77,79, 22,42, 69,62, 58,63, 39, 3, 62,60, 62,24, 20,53,
  15, 4, 60,43, 47,61, 12,59, 15,46,  5,49, 20,76, 36,16,
  40, 0, 53,52, 70,13, 11,63, 75,28, 78, 7,  6,10,  0,30,
   0,37, 63,52, 57,59, 13,74, 57,60, 44,11, 18,54, 31,51,
  45,65, 61,69, 69, 6, 37, 6, 13,76, 19,48, 10,22, 17, 8,
  20,52, 30,65, 60,29, 50,42, 71,42, 44,13, 65,66, 65,17,
  29,10, 18,62, 20,10, 51,71, 73,17, 17, 6, 74, 2, 46,31,
   3,63, 49,15, 46,58, 25,33, 45,40,  7,44,  7,72, 31,34,
  54,52, 36,15, 19,25, 48, 0, 55,71, 22,72, 19,49, 70,58,
  22,68, 29,11, 79, 1,  1,11, 26,75, 46,46, 43,29,  7,27,
  23,20, 27,28, 24,28, 33,13, 16,61, 75,37, 44,21, 46,74,
  68,68, 57, 9,  8, 7,  1,13, 15,50, 56,35, 37,79, 78,37,
   2,59, 77,78, 44,36, 74,54, 19,21, 66,11, 64,25,  6,52,
  65,71, 68,69, 53, 1, 33, 0,  7,54, 59,59, 21,35, 12,51,
  51,57, 42,11, 61,28, 65,60, 79, 4, 34, 7, 58,73, 15, 5,
  49, 5, 28, 4, 55,64, 11,52, 52, 0, 13,42, 28,15, 12,25,
  28,14, 72,34, 32,32, 60, 1, 66,76, 46,55, 66,23, 12,36,
  39,19, 36, 6, 79,41, 36,63, 38,64,  3,44, 42,49, 29,36,
  68,55, 53,73, 14,75, 44,49, 35,64,  7,45, 60,58, 22,75,
  40,61, 48,52, 47,16, 61,70, 11,23, 23,54, 31,30, 59,26,
  59,74,  7,76, 32,51, 68,53, 29, 9, 64,35, 40,75,  6,46,
  20,49,  0,78, 20,19, 33,14, 27,19, 29,70, 38,70, 34,44,
   1,53, 25, 4,  1,46, 51,63,  5,10, 64,71, 26,23, 39,36,
   5,71, 62,11, 53,50, 51,45, 10,14, 73,45, 59,67,  2,48,
  53,67, 41, 4,  2,40, 58,54, 30,27, 60,69, 13,44,  8,77,
  14,22,  9,65, 75, 9, 69,22, 70,21, 65,24, 64,37, 21,67,
  46,52, 54,77,  4,61, 46,72, 77,52, 39,75,  3,22, 71,40,
  20,32, 61,64, 46,57, 33,42, 60,62,  3,29, 66,47, 63,40,
  32,75, 44, 3, 10,28, 13,75, 21,42, 65,55, 56,56, 26,74,
  50,17,  5,26, 37, 0, 43,61, 10,12, 32,54, 21,77, 75, 8,
  46,19, 12,71, 36,69, 63,67, 66,65, 68,10, 49,12,  0,71,
  71,32,  0,17, 37,45,  5,34, 46,40, 22,18, 43,67, 23,52,
   9, 3, 45,45, 63,53, 33,37, 16,68, 66,22,  0,21, 78,36,
  15,55, 79,34, 10,72, 77,45, 64,55, 79,31, 75,54, 65,72,
  12,21, 72,47, 28,63, 30,56, 35, 3, 68,36, 29,64, 28,23,
  47,31, 34,70, 50,53, 31,27, 45,69, 77,29, 23, 0, 36,50,
  62,44, 35,58, 62,72, 40,56, 52, 3, 21,69, 78,12, 42,28,
  78,73, 69,76, 35,45, 54, 3, 11,19, 43,26, 27, 3, 16,10,
  53,14, 34, 0, 68,63, 26,46, 36,56, 49,38, 27,66, 28,79,
  17,25, 66,60, 45,53, 33,68, 62,23, 30,36,  0,76, 52,52,
  41,28, 46,24, 43,73,  9, 4, 12,76, 64,61, 59,41, 51,78,
   9,69, 18,71,  9,77, 67,42, 52,20, 69,51,  7,78, 78, 6,
  57,42, 29,22, 47,60, 71,52, 32,40, 77,69, 73,44, 36, 9,
  72, 6, 28,48,  5,14, 58,55, 32,36, 16,14, 72,63, 27,57,
  49,31, 51, 2, 55,57, 68,71, 42,34, 10,36, 78,41, 73,65,
  61,24,  9,52, 51,49, 23,61,  1,59,  9,48, 52,55, 56,10,
  79,64,  2, 5, 60,75,  2,55, 59,55, 64,38, 45,35, 58,34,
  39,63, 49,63, 41,79, 15,77, 42,66, 29,23, 45,14, 38, 0,
  42,36, 67, 8, 43, 0, 74,23, 19,52, 70,12,  9,40,  5,67,
  58,37, 44,20, 75,16, 43,57, 14, 1,  2,65, 64,67, 76,36,
  75, 7,  2,67, 28, 7, 17,56, 44,35, 35,14, 52,23,  5,24,
  73,43, 73, 1, 30,35, 42,51, 76, 2, 37,39, 47,10, 36,53,
   3, 8, 12,49, 55,58, 22,34, 55,61, 75,62, 27,35, 14,47,
  19,60, 46, 0, 27,27, 37,44, 46,38, 39, 0, 79,30, 21,49,
  39, 1, 23,58, 62,36, 28,44,  0,13, 52,16, 64,21, 11,68,
  39, 9,  5,29, 69,53, 74,49, 33,10,  5,55, 49,49, 14,69,
  48,39,  6,56, 62, 6, 50,50, 22,16, 62,76,  7,28, 24,49,
  25,64, 54,19, 78,38, 68,23, 64,74, 35,75, 41,72, 41,16,
  39,24, 60,71, 59,66,  8,35, 58,71, 60,59, 15,15, 14,12,
  42,24, 15,74, 49, 8, 76,23, 75,68, 62,28,  3, 6, 39,74
};
const int p_80_15[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  891,
  /// Array of preferences
   0,43,  4,60, 12,29, 75,18,  3,28, 44,32, 65,26, 37,16,
  65,36, 36,68, 16,11, 59,24, 20,58, 48, 8, 49,22, 68,27,
  13,68, 13,17, 48,18, 23,55,  9,51, 33,14, 70,55, 27,39,
  14,41, 76,47, 17,28, 22,17, 73,31, 72, 0,  9,41, 41,45,
  42,20, 41,36, 45,78, 47,18, 20, 7,  5,67, 10, 3, 12, 8,
   0,75, 34,52, 51,10, 74, 0,  8,16, 11,62, 67,35, 28,58,
  54,25, 49,30, 71,62, 47,22, 15,12, 40, 9, 71,41, 44,63,
  67, 9, 37,56, 32,71, 47,64, 39,53,  5,28, 55,47, 79, 7,
  77,11, 38,57, 51,49, 79,76,  9,72, 21,75, 20, 6, 69, 9,
  67,55, 65,30, 16,15, 29,58, 27,11,  3,57, 33,35, 19,14,
  25,31, 51,65, 76,70, 22,38, 50,12, 69,63, 44,68, 79,54,
  36,57,  9,19, 49,43, 11,45, 49,65, 19,63, 79,47, 38,40,
  14,75, 22,61, 34, 5, 19,16,  2,73,  0,74, 40,10, 49,76,
  67,77, 47,40, 70,57, 10, 4, 49,55, 52,78, 77,65, 78,28,
  23,35, 23,54, 23,26, 70, 9, 72,44, 32,79, 60,30, 77,39,
   6,38, 15,55, 46,50,  1,75, 24,24, 48,17, 48,46, 39,37,
  12,36, 64, 2, 12,64, 45,61, 21,55, 14,14, 39,49, 19,18,
  72,67, 69,38, 76,17, 27,28,  7,78, 28,49, 45,65, 69,32,
  56, 4, 78, 6, 68,50, 73,73, 41,37, 48,74, 37,47, 52,29,
  59,26, 35,55, 19,48, 32,31, 18,72, 33,61,  0, 0,  1,72,
  58,63,  5,13,  1,12,  5,41, 71,58, 65,74, 68,42, 25,54,
  37,59, 51,42, 55,38, 28,23, 75,13, 22,47,  8, 4, 68, 5,
  34,46, 28,59, 50,10, 37,20, 41,47, 76,42, 56,19, 19,56,
  38,23, 72,76, 36, 8, 20,15, 64,41, 45,12, 25,32, 10,48,
  48,65,  8, 2, 59,30, 52,61, 25,30, 18,59, 74,40, 17, 5,
  73,30, 19,37,  1,13, 59, 5,  3,71, 16, 0, 75,45, 54,73,
  43,76, 30,23,  4,23,  8, 9, 42,43, 32,17, 54,15, 59,72,
  47,63, 13,64, 42,28, 68,73, 17,48, 78,65,  3,56, 23,61,
  54,42, 43,45, 47,25,  1,60, 12,61, 17,41, 62, 9, 26,30,
  39,45, 54,48, 55,19, 16,44, 39,36, 48,66, 50, 3, 46,78,
  43,40, 14,65, 33,17, 52,68,  3,37, 75,33, 25,74, 59,19,
  53,22, 42,58, 18,17, 61,52, 60,54, 30,36, 72,51,  1,59,
  13,60, 58, 5, 49,63, 63,57,  0,50, 65,20, 47,60, 18,74,
   1,33,  4,68, 34,56, 77,20,  2,72, 10,25, 47,59, 31,76,
  45,77, 46,38, 11,65, 45,15,  2,34, 17,18, 39,68, 60,21,
  23, 1, 52,15, 71,38, 62, 2,  7,30, 60,17, 40,37, 54,58,
  71,22, 59,43, 58,23, 78,68, 37, 5, 51,71, 22,67, 31,16,
  76,10,  4,58, 75,69, 29,27, 70, 5, 68,65, 40,45, 21,37,
  67,62, 73,43, 22,79, 38,41, 52,41, 74,44, 52,71,  9, 1,
  39,14, 68,76, 21,50, 51,34, 59,34, 35,78, 28,72, 32,26,
  39, 5, 13,15, 49,52, 13,25, 18,31,  3,68, 34,71,  1,71,
  21, 0, 31,44, 64,78, 60,55, 57,36, 56,69, 76,46, 32,43,
  58,73,  3,50, 15,26, 77,70, 39,35, 79,43,  0,28, 59,76,
  34,48, 45,48,  4,56, 38,15, 37,26, 17,17, 53,65, 34, 8,
  10,52, 28,41, 66,32, 48,58, 37,33, 63,36, 78, 3, 10,11,
  66,64, 73,52, 30,35, 21,49, 59,51, 49,78, 22,57, 31,21,
   9,50, 40,24,  2,71,  5,74, 19,22, 58,58, 40,48, 50,58,
  29,21, 26,39, 37,31, 58,15, 13,51, 60,73, 10,24, 48,71,
  58, 3, 57,41, 64, 6, 25, 5, 59, 0, 35,21, 71,26, 55,39,
   4,19, 20,24, 19,38, 66,76, 55, 4, 57, 6, 22,41, 53,37,
  17,43, 65,38, 52, 9, 58,49, 70,53, 60,39,  2,67, 58,40,
  60,22,  9,46, 17, 4, 29,50,  0,44, 20,64, 29,19, 27,13,
  16,52, 53, 0,  0,69, 34,41, 36,23, 10,76, 14,44,  9,66,
  79,62, 20,21, 41,46, 52,47,  4,55, 66, 5, 74,75, 26,19,
  59,77, 46,35, 56,27,  3,32, 72,40, 75, 0, 75,59, 27,48,
  41,51, 76, 7, 47,51,  4, 0, 72,63, 35, 5, 45,26, 57,77,
  21,61, 69,57, 49,37, 16,72, 54,32, 20,23, 51, 3, 45,17,
  25,77, 76,23, 15,17, 28, 2, 16,31, 75,40, 58,38, 36, 6,
   1,69, 29,34, 46,19, 55,26, 45, 3,  0,39, 44,36,  4,51,
  37,13,  0,76, 11,46, 11,72, 13,13,  0,64, 39,70, 41,40,
  37,49, 43,71, 15,51, 41,13, 55, 0, 62,16, 26, 2, 15,53,
  34,37, 20,12,  1, 8, 71,11, 23,16,  6,79, 19,19, 51, 0,
  48,10, 25,68, 68,26, 38,70, 13,34, 12,10, 20,68, 43, 4,
  72,13, 60,51,  8,24, 32,37,  9,30, 75,41, 26,16, 56, 0,
  31,73, 47,50,  5,66, 54,52, 69,24, 69,34, 20,74, 76,44,
  72,75, 61,24, 20,50, 35, 1, 73,10, 78,38, 71,57, 18,18,
  48,69, 45,69, 57, 9, 35,74,  2,64, 77,19,  4,54, 17,26,
  49,66,  1, 3, 29, 3, 75,27, 28,71, 77,16, 22,39, 66,46,
  32,69, 28,10, 22,52, 60,12, 23,14, 73, 3, 47, 5, 76,75,
  70,66, 59,64, 65,57,  0,30, 11,37, 43,13, 37,58, 39,76,
  26,40, 14, 3, 33,58,  3,31, 67,33, 48,51, 22,50, 63,32,
   5,15, 18,76, 62,50, 50,46, 58,44, 62,60, 51,22, 10,35,
   8,31, 36, 4, 21,36, 42, 7, 43,47, 30,50, 46,68, 23,64,
   2,54, 24,27,  9,40,  0,38, 50,54,  3,70, 33,22, 45, 2,
  51,32, 44,25, 63,77, 46,61,  3,59, 78,21, 77,41, 57,69,
  68,51, 22,54, 37,14, 64,27, 13,20, 67,42, 71,43, 66,26,
  38,66, 15,48, 19,10,  2,37, 71,18, 54,53, 32,44, 42,53,
  22,68, 20,51,  9,32, 59,14, 43,77, 55, 9, 40,57, 47,27,
   3, 6, 58,54, 17,47, 31,75, 23, 7,  7,48, 43,68, 18,27,
  33,63, 19,46, 26,17, 45,28, 65,52, 64,25, 46, 9, 50,33,
  64,20, 67,28, 31,72, 19,17, 79,42, 47,73, 58,60, 61,27,
   4,22, 64, 8, 57, 7, 41,53, 26,20, 59,78, 54,26, 27,12,
  65,66, 12, 2, 62,74, 65,69, 30,15, 58, 4,  6,22, 22,66,
   7,37, 59,23, 24,38,  8,37, 49, 7, 76,30, 61, 0, 23,44,
  53,79,  0,29, 66,41, 25,13, 68, 2, 60,65, 22,30, 68,41,
  31, 7,  1,65,  9,10, 29,74, 36, 7, 72,24, 78,43, 45,62,
  65,53, 26, 3, 46,17, 56,38, 42,47, 37,19, 59,42, 55,21,
  17,70,  7, 1, 70,25, 31,77, 40,26,  8,73, 62,78,  5,25,
  62, 7,  7,60, 18,15, 35,42, 36,29, 75,56, 37, 9, 49,21,
  75,57, 11,69, 18, 8, 55,69, 32,78, 45,70, 40,20, 19,79,
   4,71, 11,35, 54,43, 30,58, 33,65, 35,56, 77, 1, 27,72,
   8,21, 73,33, 10,13, 44,11, 74, 9, 31,27, 11,52, 76,76,
   7,29,  3,79, 29,13,  7,67, 27,21, 36,39, 54,34, 18,37,
  16,41, 45, 8, 48,72, 41,28, 73,54, 77,46, 61,33, 71,49,
  70,17, 59,66, 49,77, 52,72, 19,42, 55,61, 26,34,  4,64,
  39,52, 61,73, 46, 2, 72,62, 73,48, 35,18,  3,67, 77,17,
  39,46, 67,71, 75,30, 56,18, 18, 5, 28,53, 32,56,  0,34,
  36,56, 57,53, 27, 4, 46,27, 56,59, 28,46, 71,64, 34,24,
  31,53, 13, 3, 61,19, 56,16, 10,28, 50,71, 17,16, 55,11,
  31,58, 63, 1,  1,74, 51,63, 66,61, 42,37, 73,49, 27,78,
  44, 6, 78,58, 13,50, 67,58, 62, 6, 24,13, 58, 8, 21,71,
  44,70,  8,71, 31,42,  1,22, 46,28,  1,25, 25,40, 11,41,
   5,76, 28,25, 20,37, 30,44, 65,33, 70,68,  6,71, 61,62,
  13,39,  7,36, 67, 1, 47,34, 66,21, 64,69, 23,34, 60,56,
   4,79, 59,31, 51,79, 63,45, 30,74, 51,69, 58,69, 77,18,
  42,16, 36,15, 70, 2, 16,71, 63,79, 35,28, 64,57, 22,40,
  37,41, 75,79, 22, 3, 10,61, 21,18, 27,33, 77,57, 67,17,
  69,27,  0,78, 38,28, 53,28, 28, 0, 13,11, 78,55, 33,55,
  11,17, 78,34, 54,19, 26,18, 15,56, 78,37, 23,56, 47, 1,
  15,11, 47, 9, 43,17, 78,46, 28,75, 25,15, 72,39, 61,10,
  66,40, 18,16, 23,30, 71,30, 76,51, 58,57, 36, 5, 61,16,
  19,34, 41,21, 65,40
};
const int p_80_16[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1228,
  /// Array of preferences
  58,35, 29,47, 68,69,  0,75, 42,67, 21,18, 51,52,  8,56,
  41,71, 72,23, 70,75, 48,11, 22,31, 74,70, 29,35, 55,12,
  53,34, 65,49, 71,66, 39,41, 40,72, 44,73,  5, 8, 45,55,
  78, 2, 60,54, 63,34, 49,13, 29,16, 25,36,  9, 8, 32,27,
  68,48, 27,71, 38,13, 66,67, 10,30, 78,21, 17,53, 12,13,
  63,73, 29,42,  2,58, 33, 9,  3,45, 74,60,  4,32, 15, 4,
  70, 8, 27,46, 23,71,  8,74, 44,53, 73,59, 58,61, 78, 3,
  18, 4, 25,69, 23,62, 13,67, 70,14, 24,49, 79,21, 73,13,
  66,77, 11,13, 63,64,  1,35, 77,43, 17,54,  5,76,  3,70,
  61,73, 79,58, 29,26, 72,72, 66,16, 52,19, 36,10, 14,31,
   7,37,  7,15, 19,74,  3,33, 64,60, 75,74, 37,70, 51,42,
  78,68, 70, 5, 40,23, 17,72,  8,48, 49,52, 40,50,  3,68,
   4,17, 31,41, 23,16, 45,14,  2,39, 47,68, 64,66, 42,23,
  66,53, 11,27, 64,43, 33,64, 42, 3, 48,12, 59,65, 43, 5,
  53,32,  0,50, 47,54, 23,68,  4,28, 30,67, 32,23, 55,53,
  58,54, 34,73,  3,43,  8,60, 72,52, 36,66, 25,75, 19,63,
  45,19, 62,34, 15,33, 64, 7, 24,37, 25, 9, 15,59,  1,15,
  19, 6, 18,34, 51,23,  5,70, 33,13, 11, 6, 27,14, 74,25,
  64, 9, 72,28, 79, 1, 31,14,  4, 3, 41,16, 78,23, 68,55,
  55, 1,  7, 9, 13,46, 42,32, 41,21, 75, 4,  8,47, 24, 2,
  34,32, 60,37, 22,45, 13,36, 73, 6, 37,25, 52, 9, 29, 7,
  23,46, 45,65, 60,16, 75, 8, 75,14, 39,16,  8, 2, 47,62,
  54,45, 48,56, 54,31, 32,72, 76,54, 23,19, 39,31,  3, 1,
  52,22, 49,71, 33,47, 30, 6, 63, 8, 77,55, 49,56, 36, 7,
  36,65, 51,47, 44,72, 54,28, 48,18, 16,21, 23,13, 50,50,
  45,51, 21,12,  9,38,  5,57, 24, 4, 38,50, 10, 8, 59, 5,
   3,37, 75,53, 37,44, 23,23,  9,56, 21,21, 18,78, 53,13,
   1,64, 69,40, 41,54, 76,62, 76,26, 57,63, 33,20, 71,61,
  51,36, 72, 8, 23, 7, 39,44, 78, 1, 76,22,  6,67, 50,75,
  69, 0, 48,43, 75,10, 74, 3, 38,25, 59, 9, 68, 0, 54,47,
  31,76, 26, 6,  2,14, 15,39, 17,14, 28,53, 38,47,  6,28,
  74,40, 31,51,  6,42,  1,38, 74,37, 56,60, 54,64, 77,27,
  68,26, 22,68, 44,54, 47, 8, 67,68, 31,36, 62,29, 40,43,
  19,25, 45, 7, 53, 6, 69, 8, 49,34, 76,38, 35,25, 13, 1,
  35,52, 30,21,  8,75, 13,28, 16,35, 67, 7, 17,61, 75, 0,
  42,35, 45,72,  9,63, 77,10, 39,13, 16,68, 75,54, 40,75,
  58,13,  6,18, 65,57, 47,15, 45,30, 55,70, 15,31, 33,28,
  69,12, 27,62, 22,54, 22,66, 41,39, 74,19, 20, 3, 57,27,
  27, 6, 70,64, 64,35,  3,76, 11,30,  1,30, 48,41, 66,58,
  14,54, 12,79, 13,63, 26,19, 76,63, 59,29, 39,57, 64,34,
  11,65, 78,79, 49,40, 51,64, 26,57, 46,44, 65,30, 58, 2,
  56,22, 14, 1, 33,24, 50,39, 31, 8, 10,38, 49,78, 10,62,
  13,73, 70,31, 70, 3, 20,48, 51,33, 65, 4, 48, 1,  2,13,
  22,24, 16, 1, 11,21, 35, 8, 32,76, 45,47, 12, 3, 44,50,
  22,20, 20,56, 60, 3,  8,33, 34,38, 65,19, 38, 1, 20,43,
   6,62, 48,26, 52,34, 35,24, 26,71, 77,23, 49,10, 35,76,
  11, 2,  6, 1, 71,14, 69, 4, 34, 3, 33, 1, 62,56, 12,23,
  54,14, 17,37,  3,22, 37,78, 15,50, 48,57, 47, 1, 69,53,
  57,19, 46,43, 71,49, 36,48, 31,75, 55,40, 52,72, 64,23,
  43,11,  2,62, 79, 7, 61,19,  0,47, 61,14, 56,61, 47,57,
  75,22, 67,43, 47,63, 42,17, 76, 0, 20,72, 68,28, 12,61,
  57,73, 44,22, 47,31, 77,21,  5,45, 35,48, 34,72, 55, 2,
  25,20, 55,24, 28,22, 40, 4, 34, 6, 34,64, 48,15, 18,30,
  54,58, 17,15, 24,46, 32,11, 31,53, 65,65, 61,59, 13, 2,
  67, 1, 30, 7, 41,50,  7,76, 22,44, 54,32, 66,68, 42,18,
  27,49, 21,24, 65,61, 20, 8, 10,33, 62,64, 13,32, 18,17,
  26, 8, 41,62, 25, 6, 51,63, 12,58, 77,30, 44,20, 69,63,
  73,23, 43,46, 76,49, 35,21,  1,40, 21,33, 17,71, 40,16,
  53,27, 11,76,  3,73, 28,49, 57,41, 45,40,  4,61, 74,38,
  52,60, 41,20, 42,55, 43,50, 53,56, 33,46, 41,25, 14,30,
   4,31, 62, 9, 40,41, 65,75,  2,36,  7,26, 57,68, 27,78,
  14, 9,  9,61, 60,73, 10,69, 54,20, 70,78, 50,42, 17,68,
  63,17, 55,74, 46, 6, 67,60, 68,22, 43,30, 50,13, 58,60,
  64,74, 48,21, 22,55,  6,34, 38,45, 21,23, 10, 6, 54,76,
  35,17, 27, 5, 42,47, 59,31, 11,16, 79,75, 40,54, 53,58,
  36,21, 43,71, 32,64,  8,54, 15, 0, 59, 8, 23,65, 14,19,
  39,22, 27,30, 31,58, 56,11, 39,61, 61,64, 55,31, 13,77,
  53,29, 30,37, 61,74, 51,27, 44,36, 73, 9, 49,57, 48,16,
  59,11, 40,26, 10,27, 74,18, 34, 9, 69,41,  7,57, 30,61,
  45,42, 71, 4, 14,29,  0, 3, 43,76, 75,73, 32,36, 79,16,
  74,23, 28,20, 33,78, 31,44,  2,66, 65, 5, 53, 5, 17,65,
  47,46, 71,75, 67,14, 61,75, 20,78,  9,24, 56,21, 13,43,
  33,50, 40,61, 67,44, 16,31,  3,77, 60,57, 59,39, 75, 9,
  19,52, 17,57, 33,23, 16,62, 20,64, 13,29, 30, 5,  6,40,
  37,75, 35,36, 77,37, 63,54, 23,32,  6,50, 35,40, 23,36,
  57,13, 73,62, 71,19, 27,28, 69,35, 33,10, 67,21, 44,79,
  78,40, 40,25, 21,15, 21,70, 78,36, 45,68, 74,32, 75,43,
  39,26, 16,37, 43,36, 55,68, 75,31, 67,17,  6,57, 44,28,
  73,22, 45,61,  3,49,  0,64, 12,70, 46,45, 62, 7, 69,32,
  28,17, 66,56, 78,28,  3,15, 43,67, 55,25, 41,63, 16,51,
  41,23, 49,79, 32, 7, 29,31, 44,15, 11,60,  6,73, 27,38,
  32,13, 18,38, 23,18, 70,22, 67,13, 19, 9, 59,57, 73,46,
  77,71, 25,16,  5,78, 16, 7, 79,49, 18,76, 58,52, 33,62,
  27,17, 11,14,  4,60, 30,65, 12,33, 24,34, 68,20, 42,27,
  21,54, 27,52, 64,44, 12, 0, 35,38, 70,72, 17,44, 57,17,
   0,63, 68,75,  9,42, 32,37, 24,44, 56,14,  5,74,  8,30,
  12,22,  0,32,  9,55, 18,29, 60,74,  2,38, 15,53, 12,20,
  44, 3, 50,69, 61,31, 58,34, 68,79,  1,24, 40,67, 64,75,
  23,72, 34,33,  3,53, 46,65,  8,26, 25,59,  7,56, 44,57,
  34,58, 72,69,  1,29, 53,57, 75,77, 51,12, 26,72,  6,55,
  56,54, 63,46, 18,64,  5,25, 18,13, 36,78, 25, 5, 58,28,
  56,49, 65,74, 27, 1, 79,72, 54,44,  7,78,  7,39, 37,45,
  59,17, 38,48, 56,44, 44,27, 27,21, 56,27, 50, 8, 68,33,
  55,17,  7,19, 42,46, 71,25, 10,67, 36,60,  1,19, 14, 6,
  51, 0, 29, 9, 63,18, 57,40, 49,60, 44,78, 33,66, 36,46,
   7, 3, 51,41, 35,34, 66,71, 48,64, 42,34, 79,22, 10,48,
  27,25, 38, 8, 39,20, 41,76, 42,21, 28,44, 79,65, 12,31,
   6,17, 75, 7, 65,38, 53,64,  7,35, 36,62, 44, 1, 78,15,
  55,58, 25,52, 71,59, 46,17, 59,34, 55, 5, 32,68, 36,45,
  34,66, 52,13, 70, 2, 17,31, 21, 1, 65,20, 75,12, 16,42,
   7,43, 68,25, 72,68, 73,10, 54,19, 15,68, 64,50, 53,53,
   7,45, 19,38, 62,60, 59,73, 79, 9, 26,53,  9,10, 70,55,
  43,35, 60,43, 35,26,  2,23, 45,13, 27, 3, 38,71, 47,74,
  61, 8, 77,69, 38,41, 27,36, 39, 9, 53,74, 43,78, 25, 4,
  42,71, 29, 0, 64,65, 62,18, 66, 0, 11,46,  0,48, 71,76,
  31,72,  9,62, 55,27, 24,17, 52,74, 49,46,  5,68, 42,73,
  35,57, 36,37, 57, 3, 13,10, 75,75, 33,51, 11,23, 34,57,
  75,27, 59,28, 69,10, 54,63, 22,56, 78,30, 22,74, 14,52,
  28,51, 13,19, 70,28, 65, 7, 40,51, 24,11, 72,55, 47,78,
  45,46, 24,52, 34,69, 54,36, 46,16, 12,46, 51,25, 63, 0,
  67,15, 41,26, 61,54, 54,29, 42,79, 36,35,  7,25, 66,47,
  19, 7, 26,45,  5,27, 48, 2, 22,73, 18,28, 73,45, 17, 6,
   9,31, 17,47, 77,11, 57, 4, 32,45, 49,23, 67,18,  1,48,
  54,68, 53,38, 31,29, 13,64, 67,54, 69,17,  6,22,  0,23,
  22, 5, 12,68, 10,68, 71,67, 53,52, 23,67, 51,18, 77,63,
  41,11, 79,28, 31,55, 10,19,  1,42, 29, 2, 58,30, 49, 9,
  73,16, 79,62, 79,19, 41,47, 44,67, 11,31, 71,48, 58,37,
  41,57, 27,15, 18,52, 63,45, 75, 3, 62, 3, 21, 2,  4,25,
  53,61, 60,61, 52,21, 50,52, 43, 6, 37,54, 36,25, 70,50,
  38,67, 15,17, 69,76, 70,66, 70, 9,  8,16, 27,23, 51,70,
  29, 8, 55,79, 53,31, 40,68, 11,75, 33,75, 65,41, 19,75,
  46, 5, 17,24, 12,51, 41,29, 20,28, 22,35, 32, 9, 71,43,
  58, 9,  3,57, 72,24, 62,36, 68,56,  6,29, 36,70, 28,35,
  15,14, 69,66,  9,36, 47,45, 63,69, 64, 6, 66,41, 64, 5,
  19,31,  2, 2, 45,74, 77,18, 20,26, 11,64, 19,66, 29,58,
  15,57, 16,32, 17,36, 78,29, 62,11, 73,49, 67,16, 14,51,
  54,46, 48,42, 71, 5,  8,79, 50, 5, 71, 2, 68,68, 13,57,
  67, 4,  4,11, 19,27, 28,70,  2, 4, 66,32,  2,16,  5,17,
  30,46, 75,15, 57,79, 76, 5, 63,55, 65,10, 71,50, 20,51,
  47,14, 79,38, 11,18, 21,58, 52,12,  5,38,  2,51,  5,43,
  76,44,  0,68, 41,17, 16,59, 21,59, 19,47, 44,31, 38,35,
  46,61, 16,17, 64,48, 75,38, 57,76, 75,41,  9,33, 16, 9,
  34,75, 37, 0, 58,51,  6,32, 32, 8, 66,39, 58,63, 39,59,
  10,76, 57,54, 58,69, 36, 5, 48,19, 13,11, 68,11, 20,59,
  42,54, 19,34, 70,77,  2,43, 26,10, 14,39, 78,37,  3,72,
  34,45, 34, 5,  4,52, 32,50, 74,39, 37,66, 14,46, 54,74,
  25, 2, 14,45, 60, 0, 42,65, 18,31, 61,41, 27,58, 42,37,
   8,63, 38,56, 68,14, 69,16, 13,12, 14,11, 35,66,  9,59,
  38,52, 64,12, 16, 3, 38,44, 47,18, 28,19, 36,31, 22,46,
  70,35, 38,73, 12,71, 29,66, 56,47,  1,10, 63,22, 38,57,
   6,64, 11,48, 49,21, 20,23, 22,43, 27,31, 73, 8, 28,68,
  54, 4, 14,14, 20,62, 72,27, 75,21,  8,57,  3,41, 39, 3,
  65,50,  8,70, 62, 6, 58,15, 40, 6, 55, 7,  5,26,  5, 1,
  70,38, 71,78, 76,58,  1,79, 62, 8, 47,75, 54,62, 22,26,
  39,69, 42, 6, 28, 6, 49,22, 33,70, 23,63,  6,56, 69,56,
  62,77, 57,10, 29,61, 70,74, 24,24,  5,59, 48,17, 36,34,
   9,21,  2,63, 73, 5, 60,30, 79,55, 15,66, 65,53, 46,59,
  62,23, 20,61, 13,22, 30,78, 77,70, 47, 9, 20,60,  1,22,
  28,26, 57,22, 28,46, 14,13, 30,41, 79, 0, 14,72, 38,20,
  16,18,  8,77, 77, 7, 45,12, 18,39, 58,45, 65,18, 41,55,
  10,16, 78,67, 17,64, 45,17, 79,34, 38,51,  6,75, 71,15,
  22,22, 62,54, 63,57, 53,54, 55,28, 76,79, 25,12, 24,18,
  37,28, 68,73, 48,51, 19, 2,  2,42, 45, 0, 57,66, 20, 9,
  72,38, 41, 1, 33,43, 52,68
};
const int p_80_17[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1064,
  /// Array of preferences
  41,15, 65,48, 24,34, 39,54, 31,16, 12,20,  3,50, 58,75,
  44,53, 32,57, 67,79,  0,18, 26,20, 70,17, 48,25, 26,11,
  18,28, 66, 8, 73,61, 51,11, 31,32,  6,41, 53,56, 21, 7,
  66,48, 31,59, 26,21,  9,75, 52,69, 77,33, 54,27, 47,11,
  69,78, 53, 3,  7, 7, 51,63, 72,50, 69, 8, 53,18, 34,60,
  15,65, 60,52, 44,70,  7,40, 47,25, 35,36, 74,10, 43,19,
  32,14, 47,31,  7,11, 18,42, 66,29, 21,26,  9,30, 15,54,
  28,38, 70,32, 49,23, 19,12,  3,73, 55, 0, 68,32, 37,36,
  59,15, 41,49, 78,63, 74,13, 67, 1, 63,26, 78,25, 71,63,
   1, 9, 63,12, 26,60, 71,74, 46,13, 76,32, 56,56, 74,56,
  60,44, 72,76, 36, 5, 15,51, 25,34, 64,47, 64,31, 40,12,
   4,20, 46,22, 12,10, 42,33,  2,42, 16, 0, 43,18, 27,64,
  38,79, 23,56, 12,43, 11,18, 62,47, 49,30, 60,66, 75,32,
   9,65, 76,18, 71,53, 78,73, 22, 4, 36, 3, 37,64, 60, 2,
  66,21, 12, 1, 65,26,  8,42, 77,43, 32,50, 19, 6, 25,61,
  78,29,  7,33, 70,15, 32,60,  0,43,  4,76, 70,69, 50,51,
  15, 3, 46,31,  9, 6, 36,76, 56,52, 64,20, 28,35, 19,51,
  47,75, 78, 3, 75,60, 77,68, 42,46, 21,49, 74, 2, 61,39,
   8,19,  8,49, 39,29, 40, 5, 42,73, 69,12, 57,23, 12,34,
  58,76, 18,14, 73,69, 40,16, 32,46, 15,15,  3,64, 12,15,
  29,59, 41,25, 74,43, 21,73, 47,74, 49,49, 66,69, 39,39,
  32,70, 39,61, 42, 3, 77,65, 38,71,  0,31, 51,31, 71,76,
  64,77, 17,29, 25,25,  7, 9, 67,17, 28,67, 16, 2,  3,66,
  28,22, 42, 8, 70,40, 25,62, 63, 6, 14,43, 52, 0, 13,24,
  67,43, 19, 4, 41,40,  5, 5,  3,69, 41,34, 54,40, 28,48,
  62,13, 56,67, 51, 7, 61,47, 55,57, 56,29, 23, 1, 67,45,
  63,49, 35, 2, 38,73, 12,55, 53,59, 71,43, 72,78, 52,28,
  47,76, 17, 0, 16,18, 53,42, 39,55, 61,63,  7,34, 33,29,
  44, 8, 45,25,  5,21, 68,41,  6,70, 20,19, 56,39, 42,72,
  35,76,  0,74, 37,72,  8, 7, 11,27,  7,13, 74,69, 66,77,
   8, 8, 56,22, 75, 8, 40,25, 13,75, 77,17, 27,55, 22,43,
  35,15, 16, 1,  6,10, 73,31, 78,45, 76,42, 70,43, 24,35,
   2,59, 76,51, 36,49, 37,17, 55,15, 62,48, 10,56, 69,35,
  49,11,  1,49, 65, 0,  4, 7, 61,79, 19,72, 18,22, 68,17,
  69,29, 48,51, 67,53, 69,40, 20, 8, 44,11,  0,25, 57,29,
  39,31, 35,73, 23,38, 54, 1, 49,35, 17,25, 59,35, 71, 9,
  49,22, 14, 9, 29,32, 57,47, 66, 5, 11,51, 39,11, 69,26,
  14,16, 26,23, 78,47, 42, 0, 47,73, 51,37, 72,58, 34,43,
  48,43, 47,65,  4,10, 66,11, 21,28, 77,75, 24,17, 49,66,
  41,74, 58,21, 20,69, 10,31, 37,74, 57,64, 17, 3, 39,21,
  54,78, 55,44, 16,68, 30, 1,  8,68, 22,76, 21,62, 47,56,
  75,10, 35,45, 63, 3,  7,19,  6,32, 63,45, 46,15, 25,51,
  36,34, 52,60, 69,34, 47,72, 10,33,  6,45, 72,63, 59,46,
  36,52, 73, 7,  5,48, 47,42,  0,28, 36,26, 45,14, 49,12,
   1,55, 62,58,  8,39, 25, 1, 23,64, 50,58, 54,24, 40,60,
  64,46, 34, 8, 37,56, 40,63, 21,17, 29,66, 47,50, 15,44,
  45,64, 33,26, 44,76,  1,13,  9, 1,  4,49,  9,63, 45,24,
   3,28, 37,67, 47, 5, 70,22, 25,16, 13,54, 43,47, 51,34,
  23, 5, 36,11, 36,58, 23,67, 37,23, 18,41, 72,49, 53,62,
  51,25, 57,14, 27,59, 13,52, 22,18, 47,64, 13,45, 65,15,
  73,58, 77,18, 52,67, 75,63, 31,23, 59, 4, 36,51, 38,78,
  46,46, 48,55,  4,35, 47, 9,  7,57, 56, 7, 73,47, 68,77,
  57,71,  6,51, 57,40, 67,41, 42,61, 75,61, 47,13, 22,15,
  42,38, 40,76, 58,42, 32,15, 18,27, 57,43, 65,16, 62,74,
  79,55, 36,22, 65,57, 68,67, 17,24, 41, 4, 37, 2, 68,48,
  55,20, 13,64, 26,25, 17,12, 78,21, 18, 4, 78,20, 71,42,
  16,44, 55,10, 19,65, 40, 8, 30, 3, 62,20, 47,49,  2,66,
  21, 4, 51,45, 78,65, 28,73, 62,72, 42, 1, 44,22, 20,49,
  51, 6, 18,55, 35,49, 67,48, 16,34, 18, 1, 29,11, 27, 8,
  51,65, 62,43, 36,27, 11,37, 19,24, 29,52, 76,70, 25,73,
  34,59,  0,24, 53,43, 52,74, 79, 0, 43,68, 72,28,  2,27,
  44,65, 66,76, 56,50, 28,15, 43,21,  0, 8, 12,32, 62,16,
   8,56, 63,69, 17,52, 29,70,  1,50, 48,61, 76, 3, 28,44,
  21,69, 24,56, 54,31, 17,76, 76,73, 43,17, 66,68, 39,46,
  12, 2,  2,12, 22,35, 23,11, 23,21, 56,41, 49,70, 77,53,
  59,63, 46,12, 58,49, 72, 8, 76, 0,  5,76, 23,48, 49,63,
  31,18, 47,79, 77, 9, 45,52, 13, 6,  1,78, 41,48, 37,12,
   0,71, 76,63, 56,36, 60,37, 62,49, 66,18, 64, 6, 33,59,
   4, 2,  6,59, 34,40, 75,27, 63,21, 15,47,  6,68, 34,63,
  29,57,  7,64, 19,27, 46,50,  8,13, 43,58, 64,69, 64,75,
  12,62, 65,13, 33,35, 78,60, 36,56, 76,53, 68,10, 20,36,
  48,65, 72,53, 51,57, 68,71, 54,16, 45,21, 73,30, 77,56,
  18,26, 61,72, 13,17, 43,38, 25,15, 78,34, 75, 1,  5,67,
   0,51, 36,59, 66,22, 23, 2, 43,25,  7,65, 21,15, 66,62,
  28, 2, 38, 7, 35,20,  6,31, 23,20,  7,50, 67,63,  1,25,
  58,70, 55,64, 37,50, 50,10, 71,52, 13,41, 72,43, 10,66,
  48,63, 72,36, 51,13, 18,76, 58,38, 16,55, 61,11, 78,48,
   9,43, 26,27, 49, 9, 22,61, 27,21,  2,44, 29,76, 32,13,
  17,74, 20,68, 52,27, 75,19, 34,70, 18,10, 19,64, 10,78,
  19, 7,  6,13, 35,57, 74,55, 46,56, 28, 8, 26, 3, 30,40,
   1,37, 50,74, 66,34, 46,54, 29,68, 70,37, 60,23, 16, 3,
  19,59, 14,73, 55,73, 17,45, 71,12, 43,66, 17,56, 24,42,
  17,19, 60,26, 72,55, 32,61, 75,49,  9,18, 69,75, 31,40,
   6,61, 41,57, 66,24, 22,78, 15,77, 29,13, 71,56, 71,41,
  35,12, 23,31, 33,39,  3,29, 58,78,  5,31,  0, 2, 49,42,
  71,45, 38,38, 18,67, 11,35, 31,14, 33,31, 35,50, 19,39,
  77,10, 34,79, 21,33, 56,68, 33,11, 48,40, 64,60, 58,44,
  39,13, 44,54, 67,46, 57,58, 17,11, 12,70, 74,73, 57,70,
  50,47, 43,31, 60,55,  9,21, 31,75, 54,70, 22,72, 12,76,
  42,32, 40,37, 78,36, 16,11,  2, 8, 70,29, 26,31, 21,16,
  55,63, 43, 4, 22,63, 53,19, 39,71, 34,26, 52,55, 71,22,
  44,42, 39, 1, 63,72,  3,61, 79,56, 46,48, 41,73, 28,50,
  41,41, 41,51, 63,54,  4, 0, 10,45,  9,50, 48,32, 53,41,
  62,27, 26,22, 72, 2, 18, 3, 60,36, 16,10, 34, 6, 31, 8,
  62,15, 15,72, 64,64, 57,38,  0,58, 51,48, 58,40, 10,60,
  57,79, 69,21, 55,54, 53,78, 13,46, 46, 0, 28,10, 22,70,
   5,62, 69, 4,  5,37, 57,66, 14,55,  9, 5,  6,78, 64,25,
  54,75, 32,79, 65,10, 64,71, 14,19, 45, 0, 59,61, 29,29,
  33,33, 76, 5, 76,29,  3,10, 54,67, 11,69, 65,73, 18,54,
  35, 3, 58,30, 71, 3, 44,26, 32,56, 65, 1, 74,34, 11,22,
  51,10, 76,19, 49,54, 41,22, 42,27, 77,69, 22,77,  3, 1,
  31,53, 38,14, 56,61, 37,34, 66, 6, 18,69, 46,24,  9,16,
  58,32, 34, 5, 24,67, 60,69, 13,60, 18,78, 37,79, 71, 6,
   8,50, 59,66, 59,54, 15, 1, 52,16, 64,37, 64, 4,  1,61,
  40,53, 45,50, 79,53, 24,26, 37,69, 78,55,  0,42, 13,11,
  24,28, 50,57, 26,61,  7,12, 34,10, 37,10, 63,37,  6,16,
  56,74, 74,41, 28,31, 34,34, 71,36, 21,45, 18,68,  6,57,
  47,22, 28,17, 71, 8, 30, 7, 14,34, 36,61, 16,74, 31, 3,
  67,30, 34,72, 73,68, 41,71, 25,77, 19,15, 44,28,  4,13,
  10, 6, 47,17, 14,31, 30,69, 67,61, 54,34, 60,42, 37,22,
  29,23, 47,61, 67,23, 21,37,  4,52, 62,67, 59,64, 26,12,
  27,23, 67,56, 68,44, 23,50, 79,34, 10,50, 77,34, 28,23,
   4,62, 65,30, 39,33,  0, 7, 29,19, 50,53, 58,26, 32,53,
   1, 0, 53, 8, 79,33, 45,57, 67,14, 66,64,  8, 6, 78,41,
  51,59, 10,59, 61,67, 64,32, 15, 7, 55,22, 26,15, 18,31,
  30,31, 73,17, 45,43, 76,50, 61, 6,  2,30, 35,19, 13,67,
  40,68, 22,22, 29,79, 46,36, 22,44, 12,25, 27,77, 78,77,
  40,56, 77,61, 25, 9, 40,13,  6,27, 76,56,  7, 5,  2,10,
  34,23, 57, 3, 30,39, 34,21, 16,30, 54,66, 31,20, 37,59,
  33,64, 62,60, 64,50, 54,46,  4,45, 36,32, 75,50, 63,31,
  12,47,  3, 6, 74,22, 40, 7, 18,46, 27,72,  7,76, 20, 3,
  74,35, 55,31, 28,58, 38,51, 52,78, 55, 4, 11,19, 75,59,
  66,67, 33, 8, 29,35,  3,45, 50,52, 48, 0, 64,26, 17,34,
  32,59, 67,21, 13, 1, 20,45, 73,18, 44,48, 66,45, 43,27,
  75,39, 16,78, 13,59, 42,62,  2,22, 38,32, 56,12, 51,71,
  35,71, 22,32, 63,66, 25,48, 79,72, 60,17, 61,76, 36,16,
   8,73, 77,66, 16,16, 15,78, 61, 7, 15,36, 52,24, 63,59,
  76,36,  7,74, 57,48, 35,77, 35,30, 28,72,  4,17, 33,57,
  38,53, 75,78, 50,71, 67,37, 27,46,  0,46, 26,18, 65,37,
  46,35, 57,31,  9,22, 76,78, 68,38, 26, 4, 25,26, 73, 6,
  46,38, 53,51, 54,41, 36,28, 57,56, 74,32,  7,67, 60,48,
  32,72, 48, 6, 52,79, 11,54,  3,13, 42,42, 12,69, 54,76,
  44,60, 26,69, 27,58, 17,38, 47,78, 68,61, 14,44, 67,65
};
const int p_80_18[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  421,
  /// Array of preferences
  71, 3, 69, 6, 22,22, 56,26, 45,73, 53,38, 22,50,  1,30,
  69,24, 31,60, 77,59, 70,42, 12,14,  7,77, 14,10, 78, 4,
  79,66, 29,16, 41,14, 13,62, 42,34, 48,43, 77,69, 63,50,
  57,57, 50,10, 16,20, 25, 5, 71,64, 71,40, 40,33,  7,65,
  64,34, 22, 3, 33,50, 53,56, 32,77, 26,79, 10,53,  6,27,
  15, 4, 38,74, 38,52, 34,10,  8, 0,  2,19, 44,51, 68,67,
   7,12, 52,36, 10, 1, 73,26, 17,71, 73,16, 37,68, 10,61,
  47,50, 23,70, 58,30, 71,58, 52,45, 34, 8, 52,63, 59,31,
  18,25, 57,55, 69,61,  1,62, 31,45, 44, 9, 60, 5,  6,61,
  38,13, 24,22, 32,20, 39,38,  3,46,  4,11, 23,72,  1,74,
   6,77, 41,15, 28,55, 19, 6, 17,38,  3,21, 56,49,  3,42,
  70,66, 72,78,  1,71, 27,25, 48,75, 31,76, 68,20, 10,28,
  74,27, 72,66,  6,65, 11,62, 40,28, 79, 4,  2,31, 68,18,
  57,16, 19,19, 64,14, 56,35, 73,54, 53,52, 15,40, 31,35,
  57,78,  8, 8,  3,40, 50,28,  5, 4, 45,15, 38,27, 74,78,
  12,42, 15,34, 50,72, 70,64,  2,14, 70,41, 63, 7, 28,16,
  35,12,  1,16, 66,62, 10, 0, 15,70,  4,55, 33,23, 72,64,
   8,29, 34, 4, 67,35, 67,38, 74, 1, 46,36, 66,37, 42,24,
  70,75,  2, 8, 77,10, 59,59, 60,14,  8,67, 72,39, 25,49,
  68,69,  4,60, 58,73, 21,60, 27,49, 25,75, 21,15, 26, 5,
  27,17, 26,60, 26,17,  3,45, 40,17, 78, 6,  6,47, 50,16,
  40,46, 42, 0, 30,31, 45, 1, 28,48, 35,74, 63,66, 59, 6,
  76,13, 24,39, 53, 7, 29,27, 74,17, 53,79, 63, 1, 18,49,
   8,74,  3,36, 64,67, 34,57, 53,73, 43,53, 65,32, 74,57,
  27, 2, 28,73, 23,73, 21,16, 77, 0, 65,71, 65,26, 54,60,
  67,62, 19, 7, 31, 1, 51,72, 74,14, 19,47, 68,23, 61, 5,
  47,10, 76,12, 65,72, 36,18, 72,59, 66,26, 30,47, 13,44,
  38,59, 53, 9, 48,29, 34,53, 34,58, 64,35, 63,68, 68,10,
  48,18, 38,20, 57,32, 58,16, 31,37, 70,52, 71,10, 79,16,
  65,27, 79,58, 46,22,  2,13, 15, 2, 63,27,  2,61, 54,62,
  17,21, 63,52, 17,31, 70,31, 51, 7, 15,49, 11,15, 27,12,
  13, 9, 58,29,  0,36, 34,38, 51,77, 51,44, 78,73, 55,41,
  53,43, 77,22,  6,29, 71,69, 22,30, 68,38, 36,58, 68,45,
   9,27, 70, 7, 58, 5, 65,33,  2,38, 64,78, 69,27,  1,58,
  53,55, 57,64, 65,49, 35,65,  9,44, 18,61, 60,26, 68,19,
   1,59, 71,75,  2,16, 57,66, 45,77, 62,52, 55,68, 45,30,
   7,70,  0,61, 70,22, 34,60, 43,74, 35,30, 41,42, 43,75,
  59,11, 64,45, 15,55, 64,41,  5,41, 32,10, 50,25, 47,23,
  44,18, 75,28, 20,50, 45,74, 52,29,  1,25, 33,61,  7,20,
  53,67, 18,13, 78,78, 24,14, 16, 2, 29, 0, 61,49, 71,53,
  10, 5, 28,79, 20, 5, 57,21, 18,79, 76,27, 44,48, 46,16,
  15,71, 19,20, 14,32, 65,47, 30,40, 25,78, 15,33,  9,66,
  42,26, 26,34, 43,33, 27,76, 11,25, 24,77, 37,79, 27,21,
  17,76,  3, 1,  9,22, 52,79, 47, 6, 64,23, 16,72, 63,55,
  34, 5, 26,20, 33,40, 36, 9, 59,54, 17, 1, 20,78, 27,57,
  47,35, 42,14,  7,45, 26,44, 55,10, 22,69,  9,67, 78,79,
  68,35, 51, 3, 18,37, 63,34, 50,17,  3,30, 37,43, 45,14,
  18,40, 76,30, 63,33, 60,40, 16,69,  6,60, 10,55, 62,67,
  77,17, 32, 8,  4,47, 62, 4, 73,10, 61,33, 40,70, 48,55,
  75,46, 25,27, 34,73, 32,28, 13,43, 28,28, 76,37, 78,56,
   4,31, 49,26, 36,71, 29,68, 38, 1, 56,18, 16, 9, 52,33,
   9,40, 38,53, 11, 8, 54,22,  4,50,  0,51, 50,46, 68,31,
  11,33,  2,51, 49,60, 11, 9, 67,12
};
const int p_80_19[] = {
  /// Number of people on picture
  80,
  /// Number of preferences
  1298,
  /// Array of preferences
   7, 8, 11,74, 23, 5, 33,72, 15,38, 68,28, 59, 4, 42,28,
  26,44, 34,26,  7, 4, 11,30,  7,34, 29,29, 60,38, 57,14,
  73,22, 37,70, 40,32, 11,53, 27,37, 29,24, 63,15,  8,59,
  69,31,  8,14, 14,25, 73,32, 17,77, 59,31, 22,73, 25,57,
  20,67, 75,60, 50,40, 27,64, 32, 8, 74,12, 26,18, 52,73,
   0,22, 38,44, 11,29, 15,47, 39,21, 74,21, 10,37, 34, 9,
  71,61, 69,43, 26,69, 77, 9, 25,37,  7,12, 68,42, 36,52,
   0,17, 79,50, 55,56, 75,29, 17,32, 67,40, 25,16, 49,17,
  62,28, 44,67, 19,16, 33,38, 38, 0,  1,75, 51, 0, 67, 8,
  74,79, 73,62,  5,57,  7,52, 61,71, 71, 6, 78,27, 18, 6,
  61,41, 56,52, 51,48, 38,32, 65,55, 56, 9,  0,23, 77, 4,
  42,76, 11,34,  6, 3, 47,76, 57, 4, 41,21, 58,42,  2,54,
   5,69, 27,75, 39,39, 24,74, 47, 3, 52,62, 11,39, 57, 9,
  25,46,  9,43, 45,30, 58,29,  9,12, 76, 3, 56,14, 34,65,
  27,12, 74,59, 11,71, 64,76, 10,26, 59,78, 42,57, 35,49,
  47, 8, 70, 3, 79,73, 44,11,  5, 8, 65,43, 67,41,  2, 0,
  20,77, 35,50, 22,42,  5,29, 14,26, 38,72, 12,63, 53,20,
  60,26, 59,47, 22,37, 47,70, 47,50, 27, 2, 73,45,  7,63,
  75,43, 32,12, 51,64,  2,72, 67, 5,  5,36, 46,43, 66,19,
  17,73, 53,35, 20,39, 21,74, 58,19, 67,14, 16,60, 50, 7,
  43,48, 39,13, 57,15, 73,12, 34,48, 45,67, 79,69, 46,70,
  45,17, 42,32, 33,27,  1,48, 41, 4, 47,10, 12,16, 40, 9,
  32,67,  6,76, 12,29, 28,33, 65, 2, 29, 7, 68,44, 47, 0,
  56, 4, 33,19,  2,34, 39,42, 18, 2, 35,37, 17,53, 76,48,
  42, 7, 75, 8,  5,54, 40,37, 57,65, 60, 0, 63,61,  1,54,
  29,20, 16,42, 36,65, 57,38, 49,20, 22,70, 39,69, 67,70,
  52, 7, 13, 4, 19,53,  8, 8, 19,68, 45,74, 51,21, 36,59,
   4,68, 53,12,  3,75, 36,36, 75,79, 72, 4, 48,20,  4, 2,
  11,17, 75,13, 50,65, 45,66,  4,71, 35,26, 18,33, 14,17,
  60,71, 20,51,  8,32,  9,27, 31,44, 32,51, 27,65, 17,60,
  18, 1, 37,47, 76,25, 48,27, 56,65, 77,51,  1,40, 19, 1,
   7,29, 17,45, 54, 4, 12,10, 71,15, 72,51, 28,46, 16,64,
  64,63, 63,47, 10,32, 59,48, 18,37, 28,54, 20,20, 22,23,
  18,48, 55,42, 34, 4, 63,41, 69, 9, 11,70, 36,40, 66,40,
  71,39,  3, 0, 45,21,  2,16,  1,47, 10,33, 19,20, 11,42,
  31,60, 13,49, 44,17, 25,64, 18,60,  7,45, 55,79, 69,26,
  28,57, 18,65, 27, 1,  3,59, 39,33, 51,30, 45,56, 42,21,
  76, 0, 20,12, 47,14, 65,60, 34, 0, 71,54, 52,42, 35, 5,
   9,44, 26,17, 24,15, 19,43, 44,76, 33,51, 31,61, 29,31,
  51,53, 32,38, 57,50,  6,62, 60,73, 79,38, 47,28,  9, 9,
  67,55, 55,65, 77,70, 63,29, 75,28, 26,22,  2,14, 72,70,
  53,53, 79,13, 76,69, 72,30, 51,47, 75,74, 74,63, 49,15,
  41,61, 47,17, 54, 7, 22,78, 47,15,  4, 4, 25,20, 39,64,
  77,60, 20,70,  2,10, 69,68, 27, 6, 13,25, 10,20,  7, 9,
  51,22, 51,23,  4,64, 72,41, 66,66, 38,77, 20,26, 61,16,
  28,45, 68,24, 35,28,  8,28, 42,53, 36, 1,  6,66, 46,76,
  14,41, 56, 1,  9,46, 16,77, 72,54, 58, 4, 14,30, 69,71,
  62,58,  8, 1, 72,36, 23,58, 64, 7, 18,42, 33,57, 70, 0,
  28,31, 27,69, 77,52, 58,53, 62,33, 34,22, 11,73, 49,42,
  38,64, 57,10, 37,13, 64,58, 71,18, 27,77, 22,54, 28, 8,
  11,40,  7,27, 71, 0, 41,31, 31,40, 60,48, 40,62, 51, 5,
  55,18,  9,41,  4,78, 62,19, 15,39, 73,13, 57, 8, 65,35,
  35,12, 22,22, 42,33, 22,60, 77,12, 63,10, 18,46,  6,58,
  49,75, 79,36, 49,38, 54,67, 54,69, 13,24, 30,26, 24,27,
  71,47,  1,65, 46, 0, 69,18,  9,37, 76, 1, 51,12, 39,52,
  39,53, 79,59, 32,70, 58,35, 34,30,  8,60, 29, 0, 11, 5,
  69,44, 43,65, 77,53, 17,43, 64,54, 23,59, 49,47, 49,66,
   0,24, 16,39, 75,75, 28,76, 10,74, 77,24, 29,62, 41, 2,
  72,72, 15,31, 18,27,  4,58, 73,70, 13,75, 45,55, 23,54,
  31,43,  3,78, 29,43, 47,64, 25,65,  8, 5, 71,11,  9,53,
  16,48, 73,21, 24, 4,  9,73, 66,37, 78,37, 71,55, 20,25,
   0,13, 11,58, 79,52, 71,72, 24,67, 61,78, 19,41, 61,20,
  30,58, 24,61, 59,49, 44, 7, 14,27, 46,60, 70,48,  0,75,
  46, 5, 37, 2, 47,74, 57,68, 12,15,  5,15,  8,42,  6,70,
  73,36, 69,51, 59,10, 14,11, 42,19,  0,35, 36,33, 32,57,
   6,39, 41,66,  0,74, 18,47, 39,72, 37,21, 74,22, 23,49,
  37,12, 34,74,  4,14, 76,58, 11,36, 72,67, 30,68, 35,15,
  53,58, 49,13,  4,11, 59,74, 37,42, 68,63, 47,42, 66,13,
  63,27,  1,24, 36,70, 26,45, 56,55, 16,55, 26,57, 35,55,
  16,12, 12, 4, 74,45,  5, 0, 60,66, 54,59, 50,16, 57,73,
   7,78,  5, 2, 13,67, 30,32, 25,15, 53,15, 42,51, 59,70,
  23,44, 66,47,  0,63, 23, 2, 31,66, 42,71,  0,39, 60,56,
  54,78, 40,44, 18,10,  5,48, 57,31, 24,29, 38,67,  6,24,
  15,14, 63,34, 18,41,  5,67, 27,72, 11,51, 55,60, 18,11,
  16,47,  0,70, 43,46, 67,49, 21,62, 57,28, 37,32, 65,31,
  49,14, 61,56, 16,36, 13,34, 21,51, 25, 6, 61,28, 28,28,
   2,30, 56,68, 49, 6, 27,38, 66,57,  8,74, 46,23, 12,49,
  18,17, 45,61,  6, 1,  4,37, 35, 7, 41,39,  7,25, 75,36,
  31,14, 40,74, 65,68, 41,41, 69,45, 64,75, 50,13, 74,65,
  26,73, 50,25, 40,75, 44,21, 14,65, 55,24, 78,77,  9,18,
   2,53, 69,24, 59,79, 42,36, 22,33, 63,67, 14,45, 48,32,
  27,24, 45,25, 39,27, 12,57, 52,69,  4,59, 30,11, 41,23,
  68,33, 19,61, 39,65, 78,45, 34, 6, 57,63, 21,34, 23,75,
  34,72, 58,77,  5,23, 28, 1, 13,13, 77,40, 10,75,  2,36,
  58,43, 59,15, 37,27, 43, 7, 29,58, 32,18, 42,65, 71,51,
  69,27, 46,33, 18,79, 33, 9, 17,30, 66,29, 53,46, 39,43,
  37,23, 63,72, 50,17, 60,27, 53,65, 11,31, 18,54,  3,62,
  69,57, 32,33, 34,54, 47,62, 73,46, 34,29, 75, 2, 54,27,
  63, 8, 41,50,  8,67, 66,44, 40,34, 68,57, 76,79,  3, 6,
  33,10, 53,23, 46,41, 31,20, 49,56,  3,72, 44,46, 11,33,
  12,18, 14,15, 18,70, 70,69, 47,73, 31,42, 54,58, 58,26,
  73,38,  8,25, 53,27, 65,25, 15,34, 12,55,  0,36, 12,38,
  76,38, 22, 7, 62,43, 55,75, 28,64, 69, 1, 19,72, 32,42,
   5,65, 24,49, 66,55, 67,34, 54,23, 12,17, 74,23, 64,18,
  44,44, 70,64, 31,64, 56,10, 71,68, 16,69, 20,76, 39,11,
  39,59, 10,70,  1,58, 49,57, 46, 8, 12,22, 62,20, 57,39,
  36,39,  7,62, 29, 1, 24,11, 75,41, 65,72, 53,55, 44,49,
  73,57, 67,68, 26, 2, 13, 1,  8,75, 36,60, 47,36, 43,76,
   1,32, 18,76, 62, 1, 42,62, 45,64, 42,25, 53,71, 53,61,
  44, 5,  1,13, 41,68, 29,41, 75,71, 40, 5, 34,37, 53,21,
  36,13, 42,43, 61,64, 30,63,  1,37, 60,33, 70,60, 38,62,
  59,14, 19,78, 32,47, 29,48, 50,24, 13,74,  9,52, 68, 0,
  46,44, 66,77, 70,65, 33,39, 43,19,  4,51, 48,68, 54,20,
  17, 6, 37,65, 51, 4, 68,18, 41, 6, 61,57, 45, 9, 50,71,
  65,50, 43,43, 54,24, 31,56, 35,30, 77,27, 71,63, 22,38,
  31, 0, 60,17, 31,22, 49,71, 69,46, 20, 7, 69,40, 38,79,
  71,58, 17,41, 68, 4, 17, 5, 40,52, 34,38, 43,18, 52,21,
  14,18, 26,35,  6,60, 77,62, 12,13, 34,77,  3, 4, 23,10,
  73, 3, 54,57, 58,71, 45,48, 31, 3, 72,63,  0,52,  5,50,
  57,53, 29,65,  0, 9, 33,50, 58, 3, 14,16, 48,39,  9,21,
  57,16, 28,32, 28,52, 26,51, 67,47, 22,52,  1,39, 21,30,
  48,61,  6,30, 41,48,  9, 2, 15,54, 19, 7, 48,66, 59,72,
  50,57, 74,62, 40,78, 37,67, 64,32, 76,59, 40,40, 64,41,
  11,26, 54,34, 32,41, 48,11, 47,43, 35,18, 27,70, 11,24,
  70,68, 31,41, 75,65, 31,16, 59,38,  8,79, 54, 3, 23,29,
   9,19,  2,68, 63,18, 76,57, 31,58, 32,48, 76,14,  5,19,
  21,78, 52,78, 47,34, 28,39,  7,51,  1,30, 45,71, 40,63,
  46,22, 32,23, 22,27, 12,66,  3,37,  6,32,  4,12, 59,44,
   4, 9, 38,54, 54,17, 15,76, 35,22, 73,69, 62,23,  2, 7,
  39,16, 62,21, 29,76, 15,60, 76, 5, 45,16, 71,36, 49,48,
  56,79, 57,19, 43,33, 13,44, 51, 7, 59,63, 15,41, 24,42,
  16,26, 68, 6, 13,20, 56,15, 69,41,  2,42, 25,17, 31,31,
  27,40, 77,78, 50,61, 42,77,  1,60,  7, 2, 68,71, 39,36,
  42,63, 21,19, 29, 2, 38,13, 40,26, 14,64, 57,76, 24,35,
  79, 0,  0,64, 55,71, 79,49, 60,20, 25,59, 78,66,  2,44,
  49,24, 47,18, 31,27, 24,46, 71,73, 49,54, 71,71, 55, 4,
  28,62,  5,40, 64,21, 19,13, 43,27,  0,58, 69, 5, 74,78,
  26,37, 56,34, 22, 6, 40,67, 53,24, 50,64, 60,21, 32,15,
  36,77,  0, 8, 14,37,  3,67, 69,66, 78,69, 38,31, 52,60,
  44,45, 62,65, 40,66, 78,61, 66,71, 42,47, 70,34,  1,56,
  12,31, 13,47, 44,53, 77, 1, 78, 8,  0,34, 23,77, 24,55,
  34,36, 16,57,  3,13, 79,16, 72,74, 52,41,  6, 6, 40,17,
  25,29, 48,48,  0,65, 69,11, 19,67, 71,59, 53,48, 34,28,
  30,74, 24,79, 11, 6, 66, 2, 66,28, 55,17, 36,10, 40,69,
  56,56, 54,41,  3,47, 50,12, 49,19, 65,62, 20,48, 13,63,
  42,41, 61,52, 48, 3, 67,74, 31,24, 63, 2, 20, 1, 32,19,
  68,35, 30,34, 32,26, 17,22, 71,32, 60,36, 19,52, 63,46,
  46,21, 67,56, 27,27, 62, 3, 72,19, 47, 1,  8,30, 46,18,
  52,40, 28,78, 56,66, 51,13, 59,77, 13,41, 17,66, 52,34,
  21,49, 70,14, 26,28, 79,15, 33,77, 49, 8, 53,19, 65,30,
  11,45, 51,50, 14,59, 31, 8,  2,19, 38,10, 22,49, 39,35,
  13,70, 65,51, 59,25, 16,20, 57,55, 61,74, 30,59, 64,43,
  23,68, 77,64, 45,52, 33,26, 23,64, 64,46,  9,30, 41,36,
  68, 7, 54,56, 76,65, 48,63, 79,35, 77,75, 66,60, 26,63,
  38,69,  7,72, 40, 3, 79,21, 69,10, 24,34,  5,12, 24,43,
  11,19, 11,41, 44, 0, 32,75, 71, 8, 33, 5, 37,20, 31,34,
  29,11, 53,40, 73,34,  9,15, 28, 9,  6,18,  9,22, 60,14,
  69,72, 12,53, 30,36, 40,79, 67,23, 32,20, 56,73, 26,48,
  43, 0, 62,25, 40,47, 68,48,  0,78, 74, 3, 38,66, 55, 5,
  18,72, 64,68, 52,50,  2, 6, 58,49, 74,61, 28,42, 35,62,
  62,31, 68,21, 68,76, 11,50, 79,12, 47,45, 65,38, 17,46,
   2,33, 50, 5,  1,68,  6,20, 39,51, 79,68, 35,31, 72,34,
  27,17, 34,42, 21,25, 59,62,  4, 8, 23,69, 62,79, 15,74,
  24,57, 36,16, 43, 1, 15,58, 14, 7, 49, 4, 17, 7, 73,66,
  51,26,  9,59, 32,31, 78,26, 65,34, 65, 1, 79,37, 38,70,
  16,21, 78,49, 72,71, 32,22, 28,67, 18,36, 33,24, 75,45,
  75, 1, 10,48, 77,55, 29,72, 65,52, 53,62, 10,12, 73,35,
  48,56, 61,11, 14,53, 23,57, 43,15, 70,78, 62,71,  7,57,
  41,37, 33,29, 66, 7, 64,55, 19,30, 15,45,  1,64, 73,67,
  48,46, 23,43, 72,18, 21,45, 78,70, 78,72, 56,59, 74,24,
  79,63, 79,62, 74, 2, 72, 3, 59,58, 42,64, 70,70,  0, 4,
  73,23, 17,29, 35,35,  2,35, 76,56, 38,51,  5,42, 48,12,
  72,29,  3,66
};

const int p_90_0[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  611,
  /// Array of preferences
   0, 1, 13,19, 68,12, 22,14,  0,13, 12,70, 15,58, 62,76,
  20,43, 36,45, 45,12, 81,46, 32,61, 67,41, 89,59, 40, 5,
  43,32,  1,65,  0,11, 47,61, 27,59, 27,33, 32,51, 70, 8,
  47, 4, 21,88,  4,36, 71,11, 14,50, 50,11,  9,27, 83,58,
  53,76, 73, 4, 67,85,  5,52, 79,34, 14,60, 54, 2,  4,67,
  65,62, 86, 2, 26,70, 24, 2, 60,66, 55,60, 50,52, 31, 4,
  27,76, 26,83, 42,14, 39,31, 34,80, 42,12, 40,51, 15,35,
  76, 3, 56,41,  2, 1,  9,19, 85,39, 52, 3, 24,54, 26, 2,
  28,52, 29,63, 74,57, 14, 8, 77,75, 89,48, 38,84, 55,32,
   6,20, 26,41, 82,15, 69,57, 86,70, 19,13, 23,41, 44,75,
  34,83, 81,24, 75,37, 84,53, 15,82,  4,81, 45,69, 20,55,
  82,45, 74,63, 88,22,  6, 6, 50,38, 80, 7, 59,70, 18,48,
  63,16, 44,24, 29,33, 48,76, 77, 7, 33,80, 80,53, 33,27,
  53,85, 27,87, 64,82,  8,75, 47,31, 79,13, 42,76, 23,46,
  87,35, 11,17, 34,88, 18,20, 63,32,  3,43, 45,46, 49,89,
  63,64, 28,41, 75,10, 33,70,  6,46, 14,38,  1,61, 38,25,
   0,71,  9,69, 81,76, 40,35, 89,15, 81,88, 69,77, 65,58,
  46,64,  2,71, 26,58,  1, 1, 45,43, 87,78,  1,59, 24,68,
  16,29, 20,45, 32,75, 60,61, 70,43, 18,85,  9,59, 54,40,
  85,82,  0,32, 31,53, 85,87, 14,52, 21,64, 81,14, 30,68,
  54,12, 40,83, 67,26, 22,34,  2, 2,  9,65, 10,14, 75,49,
  27,36, 65,41, 44,40, 74,22, 80,10, 26,49, 73,18, 24,80,
  26,51,  1,17, 17,66, 45,83, 32,42, 68,48, 60,51, 19,27,
  22,63, 39,54, 15,43, 23, 2, 87,85, 61,63, 54,48, 76,75,
  27,12, 34,15, 86,30, 14,77, 66,40, 72,88, 65,63, 16,89,
  51, 6, 46,13, 35, 8, 73,13, 28,78, 74,13, 50,72, 15,75,
   6,28, 60,56, 69, 6, 40, 0, 61,19, 81,67, 61, 6, 70,44,
  77,57, 20,86, 74, 3,  6,68,  1,67, 81, 8, 16,34, 83,87,
  58,19, 26,69, 16,53, 78,15, 80,72, 23,67, 20,20, 67,73,
  44,71, 17,78, 72,22, 48,18, 17,61, 74,29, 51,40, 65,52,
  70,21, 10,35, 11,49, 38,59, 47, 8, 55,73, 23, 5, 29,16,
  80, 4, 71,17, 84,83, 51,82,  4,71, 47,52, 53, 1, 72, 4,
  12,59, 66,65, 35,36, 49,72, 79,62, 14,53, 83,73, 20,25,
  81,87, 39,22, 52,12, 31,52, 38,61, 84,47, 67,70, 85,19,
  50, 0, 47,79, 38,56, 36,86, 78,80, 27,28, 41,67, 79,25,
  87,76, 79,27, 85,26, 89,26, 67,11, 55,87,  9,32, 71,61,
  44,43, 62,14, 24,39, 81,36, 36,24, 21,12, 88,43, 26,79,
  49,37,  9,16, 81,35, 29,55, 19,33, 85, 8, 49,86, 52,63,
   3,21, 86,38, 30, 3, 67,49, 11,38, 69, 2, 17,62, 61, 5,
  47,38, 70,42, 80,43, 65,74, 55,67, 56,37, 62, 3, 66,85,
  81,64, 31,55, 79,26,  4,22, 61,44, 87,13,  9,36, 70,29,
   2, 6, 80,37, 47,19, 46,30, 71,20, 61,39, 70, 1, 49,52,
  46,51, 62,47, 71, 6, 55,71, 51,17, 65,20,  0,33, 16, 7,
  70,53, 47,60, 65,23, 18,69, 19,22, 55,89, 27,85, 27,29,
  54,39, 75,23, 11,79,  9,60,  6,67, 23,58, 32,11, 73,20,
  58,43, 34,30, 10,27,  3,61, 63,77, 27,42, 49,59, 88,16,
  68,65, 64,21, 11, 7, 59,60, 19,62, 17, 8, 64,49, 21, 9,
  43,21, 66,28, 33,22, 65,44, 61,16, 82,21, 51,69, 36,23,
  30,19, 17,47, 65,22, 27,38,  3,66, 72,83, 40,62, 71,54,
  53, 2, 30,80, 80,49, 13,41, 82,55, 68,49, 74, 0, 11,34,
  21,48, 76,52, 35,19, 67, 0,  3, 9, 46, 6, 67, 8, 44,78,
  40,38, 75,78, 86,85, 45,88, 43,41, 81,17, 80,27, 38, 0,
   8,66, 19,42, 62,32, 65,68, 11,45, 66,83, 29,82, 64, 9,
  10,50, 57,83, 38,42,  6,58,  0,50,  7,42, 76,19, 45,39,
  86,72, 67,79, 76,12, 66,86, 58,78, 63,29, 74,39, 32,41,
  59,19, 86,12, 52, 5, 48,40, 23, 0, 26,31, 27,64, 75,15,
  39,35, 40,50, 12,25, 29,17, 29, 0, 51,27, 86,76, 27,21,
  59,62,  6,86, 36,73,  1,11, 72,54, 73,77, 12,32, 84,65,
  43,14, 19,39, 48,65, 80,82,  3,44, 31,21, 25,56, 50,37,
  22,56, 43,53, 69,16, 16,27, 68,85, 72, 9, 51,33, 49,51,
  17, 2, 15, 0, 32,53, 34,86, 49,63, 23,23, 59,15, 54,44,
  35, 3, 47,25, 74, 2, 18, 4, 71,40, 83,89, 88,67, 33,76,
  64,13, 67,52, 22,30, 76,49, 64, 7, 28,17, 23,10, 81, 3,
  32,12,  7,22, 70,56, 21,15,  7,65, 82,80, 52,69, 10,18,
  34,42, 85,62, 11,26, 29,47, 11,51, 54, 4, 62,31,  9,81,
  41,10, 65,39, 35,31, 20,85, 36, 3, 68,20, 14,26, 60,21,
   8,10, 43,12, 54,85, 29,70, 37,64, 13,79, 23,34, 32,89,
  53,61, 13,43, 21,58, 58,41, 13,66, 73,65, 21,32, 30,52,
  60,37, 22, 6, 41,62, 19,31, 44,72,  6,49, 41,48, 49,66,
  45,53, 79,21, 34,29, 46,40, 85,63, 61,23, 75,30, 53,56,
  50,19, 47,40, 36,53, 13,48, 61,83, 43,18, 75,79, 43,73,
  19,83,  1,85, 25,72, 84,11, 18,35,  0,34, 45,80, 72,84,
  44,57, 83, 1,  3,54, 56, 1,  1,13,  9,63, 34,18, 19, 7,
  33,66,  9, 2, 16,47, 59,71, 57,68, 58,11, 46,66, 62,73,
  19,21, 20, 7, 39,38, 21,59,  0,48, 40,45, 50,17, 55,11,
  11, 2, 56,57, 35,41, 37,78, 78,33, 58,31, 64,53, 54,69,
  34,47, 30,62, 45,47
};
const int p_90_1[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1981,
  /// Array of preferences
  47,58, 87,30, 14,12, 44,86, 84,44, 75,80, 54,66, 48,38,
  68,39, 56,80, 51,21, 15,29, 66,72, 43,87, 12,18, 41,71,
  18,45, 69,35, 58,63, 26,39, 26,58, 56,88,  4,40, 69,11,
  77,13, 36,36, 63,69, 31,48, 10, 2, 85,68, 52,48, 56,29,
  71, 4, 50,10, 35,75, 10,41, 61,36,  4,64, 31,30, 13,29,
   0,76,  7,58, 32,86, 40,64, 62, 7, 33, 9, 59,12, 54,36,
  42,74, 22,22, 76,44,  5,52, 80,74, 56, 5, 20,46, 50,63,
  13,52, 15,87, 82,52, 34, 7, 18,28, 87,63, 31,63, 48,41,
  37,16, 82,13, 41,61, 51,42, 31,24,  4,28, 62,25, 17,76,
  37,67, 13, 1, 74, 1, 51, 5, 39,29,  3,49, 18,77,  2,32,
  38,59, 80,89, 81,86, 19,60, 82,29, 83,25, 62,21, 33,32,
  23,88, 32,35, 69,31, 34,63, 68,81, 48,18, 35,31, 37, 2,
  46,12, 35,62, 82,88, 61,31, 36,51, 73,43,  5,58, 16,53,
  16,71, 28,62,  4, 0, 22,84,  7,68, 52,52, 87,68, 71,17,
  44,77,  0,13, 73,57, 63,67, 60, 5, 72,16, 70,64, 17,55,
  45,81,  2,18,  7,48, 12,54,  9,29, 45, 5, 25,11, 47,64,
  47,66, 54,83, 43,17, 79,82, 89,84, 66,43, 41,73, 42,40,
  44,87, 57,88, 40,54, 39,31, 46,54, 28,12, 47,50, 21, 2,
  62,20, 35,18, 70,62, 59,16, 36,48, 68,87, 83,30, 68,23,
  57,54, 85,26,  5,76, 19,16, 20, 6, 31,78, 49,54, 53,38,
  15,12, 16, 4, 78,39, 14, 9, 29,75, 12,11, 79,65, 71,82,
  86, 4, 79,83, 69,45, 61, 4, 88, 9, 21, 8, 53,77, 31,46,
  22,32, 11,60, 16,35, 40,62,  6,17,  3,53, 32,13, 89,47,
  71,51, 25, 4, 16,45,  3,88, 53,27, 57,63, 76,67, 72,41,
  74,67, 37, 6, 29,70,  6,16, 70, 2, 83,60, 73, 3, 16,69,
  39,46,  8,19, 35,25, 62,71, 56,73,  8,36,  0,36, 62,59,
  21,38, 21,58, 43,58, 21,88, 44, 0, 42,64, 79,34, 62,38,
  38,51, 26,11, 25,64, 37,70, 19,74,  1,27, 62,13, 27,67,
  77,68, 27, 9, 72, 7, 38,86, 64,59, 48,73,  2,51, 86,35,
  66,24,  2, 5, 77,41, 13,36, 12,22, 46,88, 44,12, 32, 7,
  14,19, 83,67, 58,89, 37,66, 81,28, 19,18, 85, 2, 85,57,
  65,15, 53,44, 58,23, 26,54, 30,49, 41,14, 31, 8, 38,68,
  61,65, 11,32, 85,21, 79,59, 65,46, 25,68, 78,83, 18,36,
  71, 8, 21,86, 43, 8,  1,73,  4,42, 70,39, 73,19, 17,36,
  64,26, 51,13, 88,84,  4,81,  5,38, 33,23, 75,78, 66, 6,
  61,12, 57,70, 36, 1, 22,50,  9,59, 34, 0, 62,12,  6, 2,
  43,51, 61,10, 40,12, 67,16, 40,40, 45,50, 74,22, 60,39,
  82,35, 70,45, 67,37,  8,28, 44,81,  1,81, 69,39, 64,66,
  44,53, 81,43, 18,29, 33,76, 33,12,  3,87, 20,48, 33, 5,
  85,88, 51,48, 39,76, 74,88, 39,37, 70,42, 50,52, 57, 0,
  82, 5, 36,79, 57,49,  1,35, 29,57, 32,73, 32,58,  5,12,
  28,28, 15, 6, 14,43, 47,34, 70,57, 23,66, 36, 2,  9, 2,
  56,71, 35,84, 58,64, 41,67, 81,65, 76,73,  0,24,  1, 5,
  23,89, 71, 9, 63,85, 43,12, 32, 4, 41,55, 73,58, 30,72,
  88,25, 51,61, 28,75,  5,77, 88, 3, 79,87, 67,82,  4,72,
  43,27, 62,87, 88,27, 37, 4, 84,82, 27,33, 87,52, 74,58,
  85,60, 39,26, 21, 6, 71,63, 45,69, 77,87, 37,54, 79, 1,
   3,16, 87,19, 24,38, 44,83, 45,89, 54,38, 23,17, 61,39,
  19,11, 61,47, 47,30, 74, 6,  1,40, 17,65, 41,16, 72,85,
  56,51, 81,23, 35,88, 82,50, 85, 8, 59,29, 49,25, 68,25,
  82,46,  9,87, 64,53, 66,57, 52,54,  0,73, 20,61, 55,55,
  66,78, 78,86, 27,78,  7,20, 39,72, 37,53, 76,86, 33,13,
  14,10, 59,77, 89, 3, 86,59, 28,61, 28,65, 43,36, 33,61,
   4,45, 13,84, 75,66, 75,59, 10,82, 77,61, 50,19, 69,79,
  54,27, 89,69, 20,85, 52,79, 78,30, 69,12, 75,87, 88,61,
  24,63, 80,29, 43,74, 44,84, 89,15, 27,79, 83,10, 39,74,
  58,67, 83,54, 34,81, 53,24, 72,42, 79,40, 65,21, 12,52,
  52,70, 70,61, 10,39, 19,53, 40,21, 11,23, 89,25, 46,33,
  46,89, 37,85, 15,33, 35,79, 31,10, 25,54, 10,78, 55,85,
  41,54, 35,49, 78, 9, 16,74, 58,71, 73,60,  8,84,  9,76,
  59,56, 31,26, 57,16, 28, 9,  8,27, 72,33, 80,86, 53,73,
  24,85, 79,66, 88,47, 88,60, 46,60, 28,53, 54,48, 10,62,
  72,10, 57,12, 14,58, 42,37, 53,16, 81,14,  2,53, 87,14,
  66,56, 35,15, 84,37,  1,83, 66, 0, 51,28, 67,43, 36,76,
  18,84,  3,85, 56,27, 63,74,  7,72, 53,17, 33,65, 60,89,
  66,70, 60,63, 87, 9, 43,54, 31,47, 75,67, 34,47, 26,44,
  19,37, 81,82, 35,47, 75,89, 10,40, 29,46, 76,78, 43,31,
  30,36, 34,67, 71,77, 28,17, 78,16, 34,17, 78,73, 42,48,
  41,63, 74,69, 59,24, 81,53,  1,48, 52,25, 39, 0, 81, 6,
  50,41, 51,73, 77,64, 29,31, 16,65, 38,10, 39,45, 71,18,
  36,82, 30, 5,  4,68, 45, 9, 25,18, 51,10, 89,23,  8,81,
   9, 5, 14,72, 10,86, 34,40, 16,22, 72,67, 86,38, 17,23,
  65,52, 24,21, 12,44, 31,16, 23,28, 70, 4, 61,59, 30,15,
  33,20, 25,37,  1, 7, 61,21, 29,44,  5,63,  0,35,  3,71,
  43,50, 33,40, 35, 8, 22,17, 21,37,  8,18, 55,68, 44,36,
   0,16,  5,50, 59,36, 29,33, 67,18, 75,50, 54, 0, 77,38,
  52,24, 83,38, 28,22,  3,13, 46,77, 74,37, 41,35, 55,46,
  22,29, 22,28,  6,37, 53,11, 14, 5, 76,83, 85,54, 26,89,
  28,50,  7,85,  4,84, 76,63,  3,63, 69,24, 49,35, 84,47,
  46,46,  6,41, 46,75, 41,58, 18,13,  8,32, 45,29, 28,52,
  33,18, 33,75, 87, 6, 24,56, 35,17, 62,29, 64,79, 84,70,
  55,24, 68,10, 25,69, 52,47, 22,66,  9,26,  6,45, 15,24,
  78,42, 31,50, 13, 7, 68,27, 66,47, 52,57, 55,61, 31,74,
  80, 5, 87, 2, 14,13, 33,78, 65,51, 80,40, 78,43, 17,13,
  51,49, 69,42, 85,67,  2,88, 57,66, 14,77, 45,41, 62,74,
  72, 4, 20, 2, 18, 0, 27,18, 19,44, 24, 3,  8,49, 75,75,
  86,30, 55,51, 48,24, 39,83, 49,56, 27,53, 79,43, 86,74,
  87,34, 83,56, 78,27, 85,80, 86,58, 78,28, 23,84, 12,33,
  87,24, 36,83, 68, 3, 74,53,  7,70, 53,61, 70,38, 68,13,
   4,62, 15,63, 22,56, 81,10, 89,80, 17,42, 82,40, 61,75,
  69,64,  8,34,  6,10, 83,64, 35,86, 55,42, 69,81, 88,33,
  73,49, 67,77, 82,23, 40,11, 80,17, 55,69, 14,61, 56, 4,
  88,22, 80,72, 36,81, 40,78, 27,39, 85,38, 88,18, 45,18,
  82,28, 59,25, 49, 5, 60,17, 17, 6, 11,10, 18,70, 13,65,
  10,48, 15,11, 13,37, 68,47,  9,64, 57,15, 45,12, 47,88,
  51,22, 86,77, 47,81, 83,36, 20,63, 73,41, 29,76, 37,28,
  36,85, 43,39, 68,22, 24,17, 25,12, 39,28, 82,18,  4,11,
  74,18, 17,25,  9, 6, 54,26, 12,13,  2,65, 60,23, 59,58,
  88,37, 49, 0, 82,17, 69,48, 27,82, 73,23, 54, 4,  0,28,
  72,38, 69,10, 18,16, 23,23, 37,10, 58,43, 87,87, 63,29,
  78,80, 60,34, 63,51, 65,63, 39,49,  9,50, 69,49, 72,22,
  48,36, 14,38, 21,67, 20,52, 37,50, 36,15,  0,44, 15,26,
  85,22, 27,37, 67,59, 72,74, 47,60, 67, 9, 39,78, 25,26,
  45,75, 40,52, 81,83, 51,47, 22,15, 25,40, 22,57, 69, 4,
  55,84, 74,24,  4,47, 60,53, 20,72, 79,36, 41,36, 16,21,
   7,55, 59,45, 57,68, 89,76, 27,64, 11,78, 21,81,  4,69,
  36,63, 12,27, 57,50, 14,53,  6,65,  8,61, 77,69, 72,87,
  82,89, 40,67,  6,64,  8,62,  9,84, 87,70, 80,11, 72,11,
  87,73, 39,88, 86,65, 45,17, 12,77, 64,31, 57,32, 53,29,
  74,17, 84,84,  0,66, 42,58, 42,88, 47,26, 62,44, 40,81,
  29,56, 50,80, 25,22, 68,46, 54,76, 81,48, 84,58, 45,79,
  24,47, 63,63, 86,57,  2,28,  4,23, 24,45, 87,67, 21,13,
  19,14, 23,65, 47,55, 67,83, 62, 5, 10,13, 36,11,  4,76,
  49,76, 45,16, 60,72, 35, 1,  1,31, 49,13, 23,29, 41,59,
  28,55, 84,28, 79,39,  0, 5, 61,69, 39,44, 50,88, 53,70,
  50,25, 16,72, 87,50, 70,32, 10,11, 54,46, 65,30, 22,63,
  45,68, 61,30, 26,76, 23, 1, 27,56,  1,55, 64,51, 68,68,
  40,60, 21,28, 61,89, 14,83, 25,65,  7,31, 46,21, 28, 0,
  31, 5, 42,72, 29,28, 59,15, 21,31, 57,44, 19,55, 20,59,
  60,21, 59,14, 79,26, 33,52, 73,84, 46,70, 26,70, 63,37,
  14,87, 86,88, 87,49,  8,35, 56,62, 56,38, 41, 9, 26,72,
  50,79, 58,55,  8,45, 33, 8, 77,58, 52,85, 29,88, 59, 8,
  70,89, 33,30, 63,26,  6, 7,  5,59, 57,78, 71,41, 40,34,
  55,49, 54,85, 47,87, 20,14, 42,22, 63,11, 46,80, 73,79,
  15,20, 28,19, 82,51, 89,67, 63,66,  9,68,  9, 0, 85,16,
  80,56, 28, 4, 44,27, 40,44, 76,59, 23,70,  3,17, 55,39,
  62,48, 63,13, 34,35, 59,87, 39,82, 25,21, 14,81, 79, 9,
  45,14, 46,65, 30,63, 68,65, 26,84, 12,65, 71,16,  7,13,
  28,15, 26,43, 14,29, 13,61, 16,87, 64,69, 72,80,  3, 1,
  14,88, 26,19, 28,29, 76,76, 12,83, 23,51, 64,76, 85,14,
  56,31, 57,20, 18,61, 74,72, 54,23, 55,82, 41, 5, 50,20,
  26,27, 71, 1,  0,58,  0,63, 23,62, 39,33, 68,40, 37,61,
  46,45,  9,36, 78,53, 68,28, 17,78, 22,71, 78,14, 50,74,
   3,75, 58,60, 50,44, 83,70, 74, 9, 25,74, 43,33, 36,44,
  23,25, 51,66, 44,26, 40,71, 75,26, 77,15, 33,84, 80,38,
  39,19, 18,44,  9,34, 77, 7, 82,66, 35,64, 63,18, 78,22,
  87, 1, 67,66, 54,30, 22,82, 28,16, 63,48, 30,75, 55, 1,
  29,67, 77,21, 24,80, 36, 8, 15,61, 61,67, 26,55, 40,48,
  24,33, 40,88, 84,48, 80,71,  1,68, 49,18,  5,31, 10,60,
  22,53, 35, 7, 76,65, 83,72, 83,13, 30, 0, 86,87, 10,35,
  23, 4, 27,87, 80,12, 49,59, 47,14, 35,72, 12,24, 30,24,
  86, 0, 13,76,  8,66, 79,21, 41,75, 20,18, 35, 3, 28,60,
  11,66, 82,44, 20,17, 11,71, 12,29, 55, 2,  1,67, 54,12,
  52,12, 35,68, 24,68, 80,51, 12,37, 47,51, 16,42,  4,56,
  87,78,  0,32, 38,87, 63,30, 77,42, 35,46, 21,10, 41,44,
   6,69, 58,42, 87,28, 11, 5, 48,67, 69,46,  0,57,  6, 3,
  58,17, 38,53, 32,18, 79,71, 46,64, 12,28,  2,72, 56, 2,
  74,36, 46,78, 58,18, 27,16, 65,88, 51,24, 21,56, 42,50,
  74,47, 81,69, 56,22, 70,25, 58,65, 43,52, 36, 5, 15,10,
  22,19,  0,67,  8,52, 46,11, 39,18, 42,77, 34,37, 25,28,
  54,18, 68,63, 64,48, 84,87, 13,22, 30,48, 78, 3, 20,77,
  60,11, 35,24, 82, 3, 78,40, 71,84, 64,44, 19,65, 66,45,
  67,11,  7,16, 77,20, 75, 5, 44,40, 15,16, 25,77, 24,22,
  59,52, 11,28, 11,85, 76,19, 15,67, 79,47, 68,36,  9,78,
   6,36, 71,76,  7,35, 61,82, 70,54, 40, 4, 51,75, 82,24,
  66,31, 13,30,  4,70, 15,64, 33,36, 56,25, 67,57, 22,34,
  33,64, 19,78, 11,62, 41,56, 86,62, 18, 3, 58,88, 31, 3,
  44,62,  4,13,  3,40, 19,33, 36,72, 78, 4, 58,38, 59,84,
  49,41, 18,55, 56,77, 29,83, 64,80, 51,74, 19,36, 21, 0,
  84, 0, 14,14, 21,87, 62,75, 84,63, 16,83, 46,16, 49,27,
  59,89, 64,61, 84,71, 54,52, 68,86, 73,51, 77,79, 17,51,
  76,10, 89,78, 78,19, 11,74, 14,37, 78,12, 79,81, 68,77,
  67,22,  7, 4, 53,64, 59,42, 42,24, 82,41, 11,79, 35,30,
  22,58, 81,50, 85,55, 16,30, 23,79, 36,24, 57,71, 10,55,
  74,27, 71,68, 72,79, 32,56, 43,37, 68,51, 49,87, 21,61,
   1,17, 19,71,  2,57, 20,76, 84,75, 65,23, 81,85, 68,37,
  73,68, 12,78, 35,14,  4,41, 22,72,  0,77, 58,78, 48,57,
  36,43, 37,68, 87,22,  3, 0, 40,63, 50,84, 15,51, 69,33,
   9, 4, 86,46,  4,59, 46,14, 46,56,  5,43, 76,85,  8,20,
  13,21, 11,17, 54,50,  2,58, 88,13, 38,15, 29,38, 12,30,
  89,37, 21,80, 43,89, 18,52, 32,69, 39,36, 87,74,  3,48,
  47,47, 23,54, 55,52, 54,44, 22, 4, 40,37, 15, 2, 33,27,
  21,50, 13,67, 47,28, 41,10, 61,53, 13,82, 74,26, 72,73,
  17,59, 64, 1, 77,14, 41,17, 79,57, 72,28, 65,80, 22,78,
  70,78, 83, 8, 71,29, 15,49, 81,42, 11,39, 83, 5, 38, 3,
  31,77, 26,30, 17,38, 51,76, 20, 4, 54,88, 12,69, 87,36,
   9,82, 74,20, 71,47, 85,43, 77, 0, 55,11, 71,85, 86,84,
  76,36, 65,36, 55,25, 37,78, 17,30, 42, 6,  3,81, 89,52,
  13,15, 13,24, 10, 1, 74,83, 36,20, 23, 2,  1,77, 55,62,
  37,34, 63,71, 71,87, 51,41, 86,67, 24,23, 50,35,  5,26,
  24,50, 75,71, 10,44, 80,45,  5,87, 45,11,  5,41, 11,77,
  59,28, 69,86,  6,75, 48,42, 42,49, 78, 2, 81,77, 67, 3,
  75,14, 30,40, 63,80, 29,32, 63,77, 70,18, 83,23, 57,65,
  10,67, 37,19, 13, 4, 88,38, 27,31, 62,47, 23,37,  7, 2,
   3,65, 24,64, 59, 4, 47,62,  7,62, 11,56, 11,24, 56,13,
  27,89, 58,70, 63,47, 49,11,  7,41,  4,34, 78,24, 20,42,
  44,24, 66,61, 14,79, 86,24, 55,27,  9,31, 62,65, 78,21,
  88,49, 27,44,  0,56, 88,77, 61,38, 23,32, 87,45, 15,15,
  78, 5, 57,61, 15,45, 71,12, 62,89, 19,12, 11,25, 39,60,
  72,36, 42,71, 61,42, 50,87, 36,70, 25,13, 69, 8, 54,86,
  44,22, 84,78,  0, 6, 37,36, 48,11, 38,45, 10, 9, 23,18,
  50,47,  6,60, 25,45, 61,25, 51, 0, 47, 4, 87,80, 38,79,
  44, 9, 37, 8, 31,33, 85,48, 57,62, 69,68, 66,87, 36,50,
  43, 4, 83,81, 57,82, 89,48, 17,77, 55,60, 34,56, 86,17,
  51, 9, 66,16, 78,25,  0,23, 14,50, 27,81, 41,62, 21,54,
  10,81, 83,89, 14,68, 69,60, 48,34, 67,38, 49,73, 24,75,
  27,21, 53, 3, 42,21, 72,20, 88,57, 21,33, 29, 1, 70,35,
  18,25, 81,22, 60,12, 37,18, 58,52, 50,45, 45,15, 73,62,
  66,82, 84,19, 12, 7, 19,43, 49,39, 85,65, 79,33, 76,84,
   4, 4, 73, 7, 58,22, 85,40, 44,15, 21,46, 80,22, 23, 8,
  21,49, 22,23, 72,53, 61,57, 88,66, 74,55, 71,35, 12,67,
  59,11, 52,39, 69,56, 41,24, 25,73, 28,78, 19,81, 86,16,
  14,62, 50,11, 80,61, 13,77, 71,89, 41,72, 37,51, 63,43,
  29,55, 34, 9, 67,49, 11,51, 38,73, 81,20, 30,89, 35,39,
  11,70, 80,65, 51,85, 21,77,  5,17, 76,48,  0,52, 13,44,
  16,58, 40,17, 24, 1, 28,27, 16,47, 33,71, 63,35, 36,58,
  66,75, 22, 2, 56, 3, 43,22, 31, 6, 53,30, 71,19,  6,27,
  74,41, 22,76, 44,47, 16,39, 67,21,  0,45, 87,72, 81,32,
  66,86, 50,49, 16,24, 41,15, 81,47, 76,34,  6,67, 69,37,
  77,34, 31,75, 33,87, 48,46, 45,85,  3,45,  8,76, 61,20,
  28,33, 58,68, 22,60,  5,85, 68,56, 11,89, 62,80,  0,65,
  89,70, 58, 8, 76,12, 85,42, 67,12, 75,32,  2,23, 34,32,
  59,71, 70,73, 46,58, 52, 1, 28,13, 30,54, 49, 6, 73,50,
   3,28, 63,60, 66,54, 87,86,  8,58, 73,47,  3,19,  8,56,
  48,64,  3,69,  8,29, 40,39, 56,28, 78,48, 49,86, 84,61,
  44,31, 42, 0, 69, 3, 45,47, 62,10, 39,64,  2,70, 52,62,
  32,51, 75,68, 40, 9, 87,60, 77,73,  7,29, 70,66, 44,88,
  68,11, 56,40, 38, 5, 13,58, 10,84, 83,51, 81,18, 31,36,
  50,17, 76, 4, 56,12, 11, 8, 60,69, 54,87, 18,58,  3,23,
  44,19, 40,84, 26,35, 19, 2, 43,16, 34,83, 69, 2, 78,69,
  49,74, 61,78, 34,66,  1,41, 63,55, 55,65, 59,50, 85,71,
  39,56, 75,41, 71,10, 29,41, 60,24, 84,72, 38, 4, 80,34,
  63,31, 37,30, 32,37, 42,84, 53,21, 82,16, 40, 0, 30,52,
  52,32, 17,67, 21,51, 59,78, 24,20, 38,58, 68,35, 67, 7,
  68,34, 49,84, 83,12, 47,89, 88,21, 86,71, 16,13, 40, 7,
  24, 6, 55,40, 10,32,  1,30, 68,42,  2,19, 65, 6, 30,20,
  15,31, 67,28, 18,67, 22,87, 62,58, 41,11, 75,53, 24, 7,
   6,87, 10,53, 73,29, 68,61, 32,57, 48,80, 72,69, 81,84,
  10,16, 62,85, 78,32, 76,72, 39, 7, 70,55, 30,77, 40,43,
  65,54, 61,37, 36,54, 31,41, 60,43, 60,30,  8,63, 14,16,
  78,65,  4, 3, 59,65,  4,57,  5,75, 71,30, 74,44, 13,83,
  71,54, 41,78, 38,83, 84,31, 44,60, 17,74, 44,70, 68,88,
  73,65,  6,42, 35,65, 85,41, 50,50, 17,63,  7,59, 76,47,
  61,22, 39, 5, 82,55, 50,58, 61, 1, 77,66, 89,38, 80,33,
  10,51, 32,63, 56,81, 39,14, 70,84, 11,57, 72,13, 67,78,
  10,66, 41,37, 17,80, 38, 9, 56,83, 44,52, 75,48, 58,56,
  11,44, 77,62, 10,42, 43,49, 70,27, 19,35, 51,36, 40,61,
  50,43, 53,76, 11,63, 42,73, 42,35, 44,50, 58,82, 36,88,
  71,28, 47,46, 88,63, 57,60, 89,34, 25,38,  5,46, 31,58,
  84,32, 19,26, 50,27, 55,32, 25,55, 51,67, 38,13, 79, 6,
  43,72, 61,46, 74,10, 33,14, 25,30, 47,76,  4,26, 73,20,
  54,31, 79,27, 16,77, 80,21, 51,80,  3,74, 64,65,  8,16,
  81,64, 82, 1,  3, 3, 30,14, 62,49, 32,50, 87,58, 66,18,
  15,42, 60,19, 69,38, 43,63, 29, 8, 15,76, 22,86, 73,71,
  22,40,  0,10, 47,49, 36,22, 84,52, 37,32, 14,34, 88,53,
  78, 8, 58,37, 17,52, 47,35, 73,83, 61,16, 20,65, 52,59,
  46,85, 65,29,  6, 0, 26,74, 45,82, 40,70, 89,64,  2,83,
  56,67, 65,85,  7,75,  8,72,  6,63
};
const int p_90_2[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  728,
  /// Array of preferences
  72,82, 68,23, 85,70, 61,88, 89,57, 55, 3,  9,80, 72,56,
  68,27, 67,69, 74,39, 47,20, 81,51, 71,16, 83,31, 54,81,
   4,15, 68,15, 36,15, 66,51, 57,89, 21,46, 81,28, 25,13,
  19,38, 89,63, 34,76, 28, 3, 34,12, 43,16, 46,51, 12,18,
  50,49,  2,69, 56, 0,  6,72, 66,45,  6,35, 69,59, 46,63,
  80,74, 36,19, 79, 7,  4,89, 77,59, 68,78, 41,81, 24,10,
   2,27,  9,44, 71,85,  2,41, 30,70, 40,75, 12,35, 34,72,
  83,24, 30,30,  5,69, 30,76, 76,39, 13,62, 83, 8, 71,79,
  73,31, 21,19, 77,51, 28,70, 48,55, 26,21, 76,10, 23,31,
  69,80, 18, 1, 60,81, 37,41, 10,80, 73,14, 78,44, 11,44,
  62,76, 89,18,  8,25, 13,11, 22,59, 63,21, 23,54, 44,31,
  34,28,  3,17, 34,71, 62, 5, 89, 7, 69,13, 11,73,  8,21,
  83,43, 12,89, 86,13, 81,58, 84,39, 16,58, 79,21, 76,43,
  53,22, 51,58, 74,26, 75,22,  1, 8, 58,86, 24,29, 69,53,
  81,21, 49,60, 38,85, 54,62, 30, 9, 23,28, 30,13, 39,24,
  82,32, 23,46, 22,55, 81,49, 32,84, 29,31, 62,46, 13,10,
   4,53, 17,30, 48,76, 89,38, 56,24, 57, 7, 70,45, 83,68,
  58,69,  0,85, 87,27, 47, 1, 61,39, 60,25, 24,25, 79,30,
  81,75, 71, 5, 42,17, 28,88, 53, 7, 72,21, 55,79, 76,18,
  62,84, 54,29, 51,17, 61,60, 26,64,  1,70, 12,56, 67, 2,
  71,70, 38,54, 43,35, 13,50, 76,31, 59,20, 56, 3, 55,11,
   6,69, 27,36, 69,41, 41,39, 51,67, 58,47, 48,29, 71,68,
  35,88, 71,45, 45, 6, 11,63, 16,22,  9,62, 55,67, 75,36,
   6,71, 46,32, 55,28, 11,10, 63,29, 74,61, 37,27, 59,60,
  56,40, 82,48, 89,23, 75, 6, 10,37, 62, 3, 56,34, 26,60,
  50,50, 12,58, 79,71, 71,51, 83,67, 61, 7, 74,57, 30,80,
   2,61, 42,41,  2,87, 40,49, 15,89, 24,36, 26,20, 51,22,
  31,19, 88,15,  8,87, 25,68, 59, 3, 65,68, 80, 5, 38,42,
  34,50, 34,59, 84,77, 53,49, 40,83, 34,48, 19,21, 52,75,
  26,49, 61,54, 25,26, 81,63,  3, 6, 29,48, 50,36, 10,50,
  47,68,  9,86, 10,65, 16,29, 80,14, 86,24, 62,28,  3,54,
  75,88, 52, 3, 41,26, 74,37, 67,74, 62,16, 57,84, 38,25,
  57,13, 63,79, 49,84, 21,87,  8,64, 15,24, 48,23, 17,17,
  49, 4, 89,16, 54,44,  3,33, 64,42, 25, 1, 74,22, 38, 1,
   0,49, 69,17, 50,67, 22, 9, 22, 1, 28,56, 11,45, 19,65,
  64, 0, 86,26, 53,52,  9,54, 12,13, 42,29, 35,80, 42,59,
  64,58,  4,85, 49,71, 66, 0, 87, 5, 57,12, 48,51, 54,33,
  81, 4, 36, 8, 89,60, 28,68,  8,11, 81,77, 16,52, 18,21,
  84,29, 71, 7, 58,88, 61,20, 22,45,  9,43, 52,45, 57,40,
  27,73, 67,47, 18,14, 64,21, 87,76, 73,20,  4,49, 42,68,
  12,84, 29,10, 43,29, 68,41, 26,34, 77,36, 34,47, 33,43,
  35, 0, 51, 4, 20,81, 79,69, 39, 5, 56,72, 80,22,  7,33,
  75,10,  8,38, 64, 3, 60,78, 53,17, 60, 2, 14,72, 46,23,
  41,41, 88,47, 13,53, 34,53, 25,60, 30,69, 51,76,  0,75,
  73,54, 74,33, 14,85, 80,45, 20,33, 56,29, 53,31, 42,60,
  42,51, 59,37, 50,56, 87,19, 10,26, 15,51, 85,63, 14,78,
   5,39, 82,59, 79,47, 68,44, 27,25,  8,63, 89,13, 59,64,
   4,21, 77,30, 65,27,  8,70, 89,45, 72,52,  7, 3, 54,47,
  67,43, 56,26, 52, 0, 49,47, 86,83, 71,40,  0,62, 68,82,
   7,58, 24,83,  7,63,  1,73, 82,84,  0,59, 54,74, 41,43,
  47,74, 73,48, 36,75, 69,29, 26,10, 11,34,  4,20, 89,61,
  25,81, 58,76, 65,66, 22,50, 26,12, 82,16, 26,26, 49,36,
  54,84, 34,40, 25,17, 14,37, 66,56, 67,18, 41,12, 14,42,
  11,12, 17,41, 20,73, 10,64, 13, 7, 86,88, 48,82,  1,71,
  59,31, 70,34, 46,71, 40,81, 33, 3, 28,11, 16, 9, 39,78,
   9,82, 67,39, 59,21, 84,43, 83,54, 32,65, 25,39, 83,55,
  62,34, 44,44, 12, 5, 79,43, 51,21, 58,42, 41,27, 35,56,
  30,88, 38,60, 49,88, 68,55, 41,54, 25,42, 61,72, 64,75,
   6, 3, 72,25, 75,25, 83,53, 62,38, 24,85, 21,29, 21,72,
  48,18,  1,42, 59,24,  8,61,  7,22, 64,40, 43,50,  8,16,
  19,67, 54,66, 65,51, 66,81, 30,65, 52,60, 26,41, 79,44,
  24,17, 38,52, 27,12, 14,45, 58,22, 38,11,  0,77, 23,32,
  15,11, 89,88, 73,61, 63,58, 88,17, 42,47, 25,47, 52,22,
  24,14, 86,14, 83,10, 48,89, 44,19, 88,69, 81, 3, 28,81,
  86,80, 62,52, 82,75, 55,54,  6, 4, 25,79, 18, 0, 37, 7,
  61,37, 35, 1, 70,24, 60,36, 59, 6, 70,69, 53,85, 16,63,
  46,86, 19,68,  5,77, 34,77, 16,24, 41,19, 70,29, 86,54,
  28, 2,  3,57, 22,39, 40, 3, 84,65, 45,64, 50,72, 19,55,
  49,65, 57,22, 62,47, 15,44, 11,14,  8,10, 50,82, 69, 4,
  52,52, 88,75, 37,30, 15,31, 54,70, 21,28, 61,85, 15, 6,
  26,42, 34,10, 77,37, 44,77, 65,17, 49,52, 50,53,  7, 2,
   7,87, 42,77, 89, 2, 56,39, 33,36, 19,39, 51,29, 67,32,
  80,65, 80,83, 37,35, 67,40, 17,62, 17, 2, 83,63, 59,74,
  22,68, 55,57, 24,50, 69,65, 33,33, 38,46, 62,21, 66,19,
  49,38,  0, 9, 19,32, 22, 8, 52,57, 22,85, 86,74, 76,59,
  29,46, 80,55, 85,49, 39,16,  0,37, 60,71, 80,21, 43,78,
  44,65, 71,48,  4,63, 41,72, 31,34, 74,14, 24,78,  2,32,
  67,89, 10,67, 17,81, 14, 2, 71,58, 68, 2, 87,24, 36,55,
   8,22, 71,46,  9,45,  4,82, 71, 2, 42,36, 85,66, 52,62,
  19,20, 78,45, 25,61, 37,72, 89,56, 52,37,  7,83, 86,71,
  44,37, 55,58, 15,45, 68,22,  3,55, 23,12, 63,12, 53,26,
  33,11, 62,68, 52,81, 74,17, 68,28,  5,76, 80,89, 53,50,
  74,41,  2,58, 81,64, 84,83, 64,34, 59,48, 27,27, 44, 6,
  27,77,  8,77, 87,86,  7,39, 47,35, 11,25, 27,67, 71,52,
  22,88,  2, 5, 16,18, 80,54, 23,30, 50,47,  4,28, 68, 5,
  52,29, 80,40, 56,10,  2, 3, 12,34, 87,38, 85,88,  3,39,
  22,23, 30,28, 65,31, 58, 4, 83,51, 40,56, 64,73, 68,11,
  66,58,  7,77, 53,19, 49,23, 49,61, 40, 9, 49, 9, 11,60,
  62,63, 83,39, 75, 3, 85,84, 38,79, 10,63, 46,41,  5, 5,
  22,79, 51,44, 77, 4, 19,33, 66,25, 84,53, 11,81, 27,68,
  62,35, 14,64, 65,33, 17,28, 45,56, 46,57, 38,32, 39,17
};
const int p_90_3[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  963,
  /// Array of preferences
  62,71, 82,63, 28,31, 73,81,  5,86, 54,62, 33,86,  2, 5,
  42,59, 75, 0, 86,24, 43,62, 17,71, 79,51, 83,39, 65,24,
  37,80,  8,30, 63,84, 59,72, 13, 7, 54,73, 20,28, 73,72,
  20,62,  5,35, 18,87, 80,51, 21,25, 72,57, 61,49, 73,12,
  61,60, 79,52, 56,50, 39,63, 61,56, 82,57, 80,19, 79,64,
  32,51, 19,76, 23, 9, 56,21, 49,66,  3,59, 64,60, 81,63,
  43,80, 10, 9, 71,20, 78,15, 72,48, 60,24, 63,45, 34,43,
  17,12, 25,40, 40,18, 15, 4, 56,43,  4,80, 66,22, 37,39,
  45,28,  6, 0, 29,46, 37,36, 71,57, 77,78, 56,18, 77,60,
  12,32, 77,30, 15, 1, 27,67,  5,77, 69,22, 21,74, 41,49,
  39,51, 74,65, 50, 0, 29, 2, 77,33, 19,12, 39,57, 65,56,
  63,34, 52,47, 80,23, 20,80, 12,48, 34, 2, 71,54, 22,80,
  12,16, 52,50, 79,54, 14, 3, 52, 0, 89,80, 80,67, 13,17,
  36, 0, 64, 5,  0,64, 39,28, 85,79, 45,26, 11,60, 43,59,
  71,77, 57,65,  8,50, 21,27, 82,27, 13,37,  4,39, 73,68,
  79, 6, 80,57, 67,71, 42,73, 88,35, 32,78,  7,32, 73, 4,
  25,75, 67,75, 71,12, 73,39, 29,89, 69,57, 19,18,  8,23,
   9,57, 70,20, 68,51, 89,65, 82,15,  2,42, 57,51, 37,64,
  34,49, 27,65, 77,69, 88,61, 87,14, 20,77,  1,60, 87,63,
  39,13, 78,64, 44,23,  7,25, 62, 9, 19, 2, 27,56, 19,42,
   8,41, 10,62, 74,33, 88,20, 14,72, 25,39, 44,89, 32,38,
   5,11,  5, 5, 78,51, 87,11, 88, 4, 36,72,  0,39, 23,23,
  43,46, 23,84, 67,30, 18, 4,  5,82, 45,37, 75,63, 29, 4,
  61, 8, 62,35, 15,13, 89,73, 55,47, 32,63, 10,38, 37,15,
   1,83, 27,28, 67,16, 58,36, 16,15, 48,47,  7,28,  2, 2,
  26, 0, 49,24, 15,50, 13,26, 18,35, 53,55, 83,54, 74,74,
  68,42, 66,86, 20,45, 77,36, 10,35, 56,69, 21,39, 84,59,
  83,83, 44,20, 20,18, 74,17, 14,89, 88,74, 25,58, 17,61,
  64,66, 54,40, 19,65, 75,67, 62,40, 48,30, 13,57, 34,39,
  51,46, 35,42, 39,59, 46, 0, 88,45, 18,69, 46,14,  6,35,
  60,61, 85,14, 30,36, 88,72,  1,54,  8,58, 75,18, 42,36,
  36, 3, 33, 4, 80,60, 64,52, 75,24, 27, 8, 46,30, 55,20,
  34,84,  5,88, 16,77, 22,35, 31,18, 16,24, 33,67, 11,10,
  16,68, 64,39, 48, 1, 19,88, 84,43, 43,53,  7,88, 52, 5,
  86,49,  5,28, 57,80, 28,43, 25,34, 31,46, 46,80, 87,70,
  15,16, 63,49, 78,14, 32,17, 85,75, 63,52, 57,88, 11, 9,
  36,10, 58,77, 29,50, 13,39, 43,58, 30,64, 40,62, 10, 4,
  57,77, 33,32, 38,68, 36,36, 18,57, 81,37, 82,28, 55,82,
   6,33, 21,48, 71, 2, 69,29, 10,81,  7,49, 38, 6, 40,67,
  56,13, 12,88, 71,44, 11,83, 13,49, 64,29, 35,18, 65,89,
   2,15, 79, 8, 15,31, 30,19, 76,22, 22,62, 13,10, 71,51,
  15,77, 35, 4, 41,87, 41,65, 47,46, 28,70, 88,21,  2,26,
  23,37, 24,87,  6,74, 21,85,  6,43, 42,22, 27,44, 89, 9,
  38,73, 31,12, 63,48, 17,35, 27,73, 39,11, 27,30, 80,14,
  84,42, 63,21, 69,44, 89,57, 61,78, 50,49, 37,79, 32, 6,
   4,26, 19,86, 73,21, 35,69, 61,14, 35,29, 42,48, 80, 3,
  38,40, 38,50, 60,59, 66,23, 14,67, 60,13, 36,58, 46, 3,
  42, 7, 46,54, 47,26,  2,29, 48,60, 42,54, 56,49, 14,37,
  43, 9, 35,45, 48,87, 56,19, 71,19, 30,52, 14,65, 74,63,
   2,73, 65,38, 75,46, 65,71,  6,66, 84,28, 43,87, 42,21,
  77,79, 61,30, 76,85,  4,57, 22,84,  3,61, 67,47, 84,88,
  39,32, 72,88, 79,34, 32,66, 80,48, 73,37, 30,65, 20,55,
  88,79, 49,31, 14,26, 40,44, 30,80, 78,36, 83,43, 26, 2,
  32,53, 61,38, 70,59, 21,68, 53,26, 22,61, 63,42, 39,87,
  44,49, 67,86, 84,16, 68,59, 50,68, 27,75, 29,33, 87, 7,
  13,41, 86,59, 29,62, 22,78, 53,84, 23,34,  3,70, 76,25,
  68,84, 25,89, 29,66, 76,45,  7,11, 31,47, 54, 5,  3, 6,
  81,87, 12,58, 43,31, 74,64, 65,70, 39, 3, 74,16, 68,74,
  51,55, 19,29, 66,47, 74,23, 63,63, 14,59, 86,39, 81,51,
  87,37, 57,89, 22,28, 80,52, 45,35, 16,21, 13,66, 74, 0,
   3,28, 13,14, 41,28,  5,76, 34,59, 65,27, 47, 4, 50,52,
  60,77, 41,70,  9,60,  9, 1,  7,10, 83,24, 78,87, 86,17,
   0,16, 70, 8, 54,19,  7,22, 67,48, 26,36, 77,86, 31,14,
  79,19, 30,85, 72,85, 63,16, 78,66, 16,30, 38,47,  5,60,
  41,42, 47,41, 25,87, 17, 6,  9, 8, 64,57, 86,22, 80,42,
  75,28,  1,36, 36,19, 54,81, 81,75, 49,68, 68,28,  0,63,
  77,82, 64,71, 15,35,  9,12, 55, 5, 41, 7, 36,81, 87,42,
  22, 6, 11,19, 30,27, 80,49, 37,34, 71,81, 85,59, 76,66,
  20,58, 37,67, 61, 9, 54,13, 12,72,  5,68, 25,78, 42,66,
  44,13, 66,35, 66, 8, 25,63, 60,72, 76,60, 38,69, 50,38,
  18,37, 64, 9, 12,57, 50,25, 47,28,  9,30, 62,69, 53,43,
  50,18, 66,61, 78, 5, 59, 3, 72,39, 24,27, 86,54, 69,75,
  82,80, 44,77, 61,36,  3,69, 46,84, 59,75, 62,44,  8,71,
  78,61, 57,52, 52,44, 15,86, 24, 6, 44,40, 54,15, 73,36,
  28,12, 54,75, 55,80, 48,61, 30,40, 12,50, 26,27, 22,24,
   1,65, 88,48, 73,53, 89,74, 18,42, 13,22, 31,75, 17,31,
  32,23, 52,62, 45,31, 66,72, 18,26, 18,75, 46,29, 67,76,
  19,60, 68, 0,  6,76, 64,42, 44,79, 53,75, 15,36, 54,48,
  78,63, 82,40, 44,56, 87,45, 88,76, 55,28, 20,89, 89, 7,
  39,75, 75, 5, 11,69, 54,31, 63,32, 24,15, 21,63, 86,66,
  82,24, 33,35, 48, 5, 61,45, 42,31, 12,44, 58,29, 60,65,
  27, 0, 44,54, 76,84, 84,63, 71,33, 34,24, 53,60,  0,35,
  49,35, 18,33, 61,32, 80,71, 84,30, 24,69,  0,13, 40,29,
  53,16, 51,20,  5, 6, 75,83, 29,49, 45,54, 85,66, 50,87,
  19,45, 42,47, 47,14,  1,48, 61,43, 65,49, 58,55,  4,47,
  38, 9, 13,42, 64,64, 51,31, 30,18, 65,66, 81,36, 64,41,
  19,81, 84,20, 79,17, 83,40, 80,87, 41, 0, 28,48, 25,16,
  49,73, 63,10, 73,26, 66,55, 10,27, 47,74,  1,22, 79,11,
  75,52, 35,61, 73,67, 14,74, 63,82, 47,72, 88, 8, 80, 7,
  42,20,  9,40, 19,52, 23,60,  5,36, 59,53, 51,27,  8,45,
  33, 9, 30,61, 77,50, 51,36, 17,78, 15,46, 45,73, 58,25,
  82,76, 11,49, 33, 2,  8,77, 27,18,  3,38, 86,64, 49,67,
  20,75, 16,23, 30,33, 35,66, 86,14,  1,43, 26,13, 66,57,
  22,14, 18,52,  9,14,  6,26, 14,47, 88,12, 19,82, 36,45,
  48,27, 66,42, 36,47, 32,75, 68,75, 77,39, 66,54, 74,46,
  29,42, 40, 4, 37,84,  0,42, 34,36, 58,33, 75,13, 42,34,
  70,64, 26,69,  4, 8, 12,40, 16,62, 50,77, 64,58, 82,23,
  21,12,  4,59, 14,54, 17, 2,  2,33, 70,88, 30,57, 44,83,
  45,24, 21,10, 85,23,  3,84, 29,72, 56, 3, 66,80, 53, 0,
   3,18, 84,61, 56,84, 18,39, 20, 8, 68,62, 24,10, 25,33,
  21,26, 18,62, 44,11, 88,81, 37,37, 38,29,  2, 9, 42,35,
  62,32, 68,52, 31,11, 61,48, 55,77, 59,63, 29,16, 62, 7,
  87,85, 21,60, 47, 5, 10,11, 69,59, 77, 8, 35,53,  6, 3,
  57, 7, 48,57, 10,51, 33,46, 82,37, 31,67,  0, 9, 83,72,
  55,60, 44, 3, 85,74,  8,57, 27,38, 55,61, 46,47, 27,13,
  84,34, 16,48, 57,59, 43,15, 44,42, 14, 2, 14,35, 59,27,
  57,23, 30,84, 18,61, 75,20, 71,11, 13,54, 18,13,  4,29,
  79,23, 62,16, 43,76, 55,23, 42,38, 83,16, 72,58, 13,19,
  74,29, 16, 4, 65,17, 78,57,  9,43, 52,84, 33,29, 35,76,
  69,39, 57,67, 19,47,  4,46, 54, 1, 85,31, 72,79, 65,68,
   3,19, 32,54, 48,25, 42,78, 67,62, 82, 1, 78,20, 14,68,
  38,64, 31,34, 66,16,  9,41, 18, 6, 69,55, 42,61, 10,33,
  39,70, 44,38, 16,36, 20,25, 57,32, 35,37, 36, 2,  0,22,
  48,33, 87,73, 18, 5, 40,82, 45,45, 55,78, 10,21, 63,23,
  45,30,  0,23, 27,80, 80,30,  7,19, 68,83, 81,28,  3,50,
  42,10, 65,81, 48,28, 30,11,  1,32,  0,49, 40,22, 76,56,
  39,65, 28,26, 31,29, 10,67, 79,10,  4,33, 55,49, 13, 4,
   8,61,  9,35, 54,34, 49,26, 52,85, 18,29, 73,89, 74,32,
  50,12, 39,31, 11,59,  8,42, 52,78, 40,32, 58,45, 67,28,
  78,75, 56,36, 48, 7, 67,73, 21,80, 37,33, 51,71, 60, 6,
  42, 2, 86,53, 16,46
};
const int p_90_4[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1549,
  /// Array of preferences
  55, 7, 67,82, 18,59,  8,44, 26,17, 14,20,  5,81, 71,71,
   3,85, 83,32, 27,68, 63,70, 74,87, 12,10,  1,11, 10, 1,
  48,12, 79,40, 43,41,  5,38, 61,75, 87,43, 81,79, 28,11,
  77,38, 12,45, 27,25, 50,24, 67,27, 26,89, 58,52, 69,81,
  50,79, 70,89, 50, 4,  7,65, 25, 9, 31,15, 23,32, 79,84,
  56, 8,  3,81, 48,17, 25, 5, 58, 6,  0,18, 35,60, 78,36,
  33,56, 23,67,  5,37, 60,80, 72,64, 74,69, 61,68,  8, 3,
  76,71, 23,64, 76,78, 52,74, 77, 0, 25,44, 10,42, 29,12,
  40,18, 80,47, 69,17, 37, 9,  1, 8, 78,67,  5,43,  2,23,
  77,11, 21,64, 82,26, 53,64, 28,36, 46,54, 42,88, 54,47,
  18,66, 50,70,  6,72, 71,67, 68,83, 56,72, 15,70,  6,89,
   2,54, 52,76, 41,81, 68, 6, 80,15, 84,65, 43, 5, 35,24,
  23,40, 45,39, 67,54, 71,11, 76,55, 43, 8, 51,35, 87,17,
  16,79, 72,57, 29,86, 28,71, 58,82, 71,55, 70,13, 54, 0,
  13,61, 72,80, 60,41, 88, 9, 24, 2, 50,30, 30,36, 50,81,
  67,24, 52, 1, 52, 2, 65,15,  6,30, 84,88, 45, 7, 56,47,
  16,15, 71,40, 60,71,  4,78, 67,69,  9,45, 79,88, 16,62,
  34,25, 80,31, 42,30, 56, 2, 68,89, 24,68, 17, 9, 63,38,
  18, 6, 52,81, 21,15, 80,49, 34,52, 40, 5,  9,59, 14,38,
   8,54, 58,68, 76,63, 29,74, 13,32, 30,27, 83, 5, 42,63,
   3,32, 81,67, 64, 8,  0,20, 88,34, 71,82, 11,65, 49, 8,
  75,85, 39,84, 78,40, 61,89,  2,53, 56,52, 10, 6, 49, 0,
  51,51, 36,22, 73,51, 86,69, 58,85, 58,14, 76,66, 58,60,
  32,45,  9,39, 84,10, 48,75, 87,84, 32,24, 39,86, 66,43,
  13,27, 29,41, 79,62, 51,86, 16,55, 70,82,  4,13, 69,56,
  13, 2, 34,83, 55,22, 81,45, 69,11, 44,21, 58,45, 72,85,
  17,55, 77,74, 40,24, 34, 2, 61,44, 20,65, 25, 1, 78,43,
  57,83, 80,60, 62,86, 17,54, 11,64, 28,41, 65, 8,  8,48,
  89,12, 31,36, 63,41,  6, 1, 27,73,  1,50, 46,33, 49,89,
  78,77, 88, 5, 62,31, 27, 4, 26, 4,  3,23, 74,68, 20,50,
  22,89, 45,57, 33,63, 38,70, 70, 6, 44,33, 86,12, 81,14,
  61,50, 47,48, 10,61, 51, 9, 74,41, 71,87, 40,42, 16, 9,
  49,49, 87,49, 45,74, 65, 1,  3,18, 47,32,  5, 5, 67,31,
  74,70, 48,72, 26,48, 89,30, 87,68, 86,49, 63,73, 13,40,
   0,69, 25, 0, 60,63,  0,68, 21,61, 58,67, 35,84, 89,76,
   1,84, 71, 1,  4,58, 33,88, 50,53, 33,86, 45, 3, 32,19,
  87,15, 45, 5, 86,89, 28,37, 35,75, 69,77, 39,63, 87,27,
   7,53, 12, 0, 86,46, 14,35, 67,72, 88,46, 59,89, 10, 5,
   0,81, 36,49, 46,22, 68, 8,  9,68, 12,14, 77,55, 46,69,
   7,42, 35,17, 81,82,  3,87, 24,48, 52,44, 62,48, 71,33,
  22,17, 52,50, 86,19, 58,35, 46,66, 65,86, 30, 3, 59,23,
  82,67, 68,53, 69,63, 85,86, 21,52, 15,69, 79,76, 54,42,
  44,46, 14,58, 32,60, 73, 5, 49,32, 38,58,  9,21, 69, 9,
  69,23, 72,42, 18,54, 25,53, 83, 3, 10,79, 20,79, 12,84,
  77,71, 89,73, 64, 3, 81,86,  2,78, 86,40, 66,79, 79,14,
  41,53, 66,44, 66, 4, 65,18,  0,80,  9,46, 68,46, 45,19,
   4,34,  7,81, 37,45, 14,36, 34,39, 62,10, 42,47, 86,39,
  40,79, 65,13, 10,50, 46, 5, 83,45, 15,41, 56,35,  1,85,
  31, 8, 87,78, 20,49, 11,23, 61, 6, 18,15, 43,68,  8,41,
  25,70,  3,65, 76,34, 82,30,  7, 3, 22,62, 64,73, 61,72,
  81,27, 42,25, 40,15, 53,22, 77,64, 10,85, 58,19, 69,46,
  61, 8, 89,37, 59,30, 80, 8, 75,67, 50,85, 85,42, 26,12,
  78,57, 75,87,  4,82, 55,57, 11,81, 23,42, 61,23, 53,16,
  13, 0, 38,24, 10,66, 56,20, 71,45, 69,41, 60,39, 58,40,
  43,27, 33,34, 25,43, 68,12, 81,44, 26,65, 32,71, 16,13,
   5,61, 81,41, 79,59, 10,20, 43,20, 60,29, 69,33, 20,67,
  49,20, 77,22, 16,70, 86,30, 88,44, 12, 2, 35,28, 35,85,
  62,38, 37,89, 27,22, 80,69, 26,20, 19,86, 37,83, 89,47,
  37,24, 24,63, 63,84, 84,61, 15,42, 13,86, 55,71, 26,61,
   0,21, 82,66, 13,89, 87,26, 40,63, 13,82, 34,17, 29,69,
  65,70, 55,33, 31,41, 56,33, 43,18, 20,59, 14,89, 14,76,
  59,13,  9,30, 34,20, 65, 9, 43,82, 15,31, 71,23, 33,24,
  32,61, 72,53, 65,66, 21,46, 31,32, 55,17, 55,37,  0,14,
  37,18, 44,47, 71,62, 73,80,  6,88,  2,14, 53, 8, 87,53,
  10, 7, 76,61, 14,26, 57,35, 64,32, 70,50, 52,43, 72,30,
  64,74, 64,15, 21,14,  8,61, 87,19, 47,27, 62,21, 46,35,
  43,45, 19,87,  8,29, 32,55, 24,34, 72,29, 70,78, 15,34,
  63, 9, 14,19, 46,64, 84,19,  2,87, 79,60, 77,83,  3,15,
  87,70, 26,33, 78,41, 70,80, 32,65,  3,27, 46,42, 71,69,
  42,11, 74,21,  9,52, 16,75, 64,75, 14,70, 72,38, 63,16,
   1,31, 14,68, 64,31, 10,68, 29,81, 19,72,  1,43, 14,24,
  32,73, 51,11, 47,73, 62,51, 53,53, 36,16, 68,59, 41,61,
  36,61, 75, 7, 51,33, 40,57, 37,25, 43,51,  4,36, 59, 0,
  62,77, 29,61, 85, 8,  3,61, 10,25, 11,89, 34,57, 77,26,
  64,85, 13,54, 58,86, 13,66,  6,40, 71,49, 42,72, 71,61,
  51,26, 34,70, 48,65, 40,74, 57,63, 83,51, 30,63, 56,29,
  56,84, 46, 7, 76,83, 46, 6, 18, 0, 81,26, 51, 7, 55,32,
  58,70, 52,10, 75,38, 27, 3, 58, 2,  2,10, 80,26, 12,42,
  27,83, 74,66, 65,19, 27,31, 30,18, 47,21, 67, 8,  5,64,
  54,22, 48,59, 12,29, 68,60, 45, 2, 50,32, 42,65, 78,15,
  29,76, 71,12, 35,22, 15,17, 78,48, 11,54, 49,64,  5,24,
  36,84, 65,54, 51,28, 36,17, 82,23, 22, 7, 70,53, 47,80,
  65,30, 74,62, 68,45, 87,10, 40,68, 46, 4, 74,32, 77,81,
  54,63, 28, 6, 68,54, 80,17, 78,74, 66,27, 52,14, 20,72,
  33,20, 21,45, 55,23, 11,57, 78,17, 43, 6, 89,34, 70,61,
   4,69, 67,56, 56,50,  4,30, 38,20, 21,19, 38,89, 70,23,
  51,39, 16,17, 28,43, 82,41, 36,60, 79,44, 44,25, 82, 1,
  68, 0, 22, 9, 43,43, 77,48, 14,50, 63,18, 11,47, 25,23,
  54,14, 35,14, 55,39, 74,89, 19,26, 81,70, 79,66, 69,52,
  62,40, 81,16, 73,55, 40, 6, 44,58, 64,38, 27,12, 22,19,
  27, 6, 86,51, 77,63, 54,13, 34,72, 84,13, 64,27, 63,25,
  25,33, 66,30, 28,55,  4,73, 43, 7, 87,13, 52,29, 22,48,
  34,43, 32,46, 27, 2, 14,30, 12,43, 75,35,  2,45, 37,11,
  47,36, 23,34, 82,33, 89,87, 28,50, 70,60, 89, 7, 48,28,
   8,67, 27,75, 83,61, 36,20, 72,32, 17,74, 83,40, 11,86,
  56,82, 32,80,  5,87, 39,79, 58,83, 49,21, 26,57, 43,53,
  77,24, 57,43, 73,57, 38,65, 72,68, 83,72, 25,57, 88,70,
   8,18, 34, 1, 20, 3, 33,85, 68,16, 17,52,  6, 2, 65, 6,
  11,50, 52,24, 55,45, 59, 7, 63,32, 79,20, 87,40, 36,42,
  73,74, 48,33, 23,89, 34,14, 25,66,  1, 4, 59,74, 57, 5,
  88,25, 22,29, 71,51, 49,69, 78, 8, 67,23, 87,82, 84,17,
   7,85, 46,76, 49,48, 27, 5, 46,31, 52,22, 82,47, 89,11,
  17,39, 68,50, 58, 4, 12,81, 14,55, 31, 2, 44,71, 46,43,
  57,28, 80,84, 68,82, 60,21,  0,83,  6,33, 25,52, 27,54,
  49,56, 34, 5, 87,60, 12,41, 30,85, 69,21, 65,40, 37,86,
  61,59, 36,80, 89,20, 88,56, 19,65, 65,14, 69,76, 89,43,
  85,30, 70,19, 32,53, 47,47, 33,40, 76,76, 31,63, 84,78,
   8,89, 60,27, 28,17, 13,71, 73,65, 29,27, 86,13,  3,72,
  86,43, 47,30, 25,54, 74,45, 14,15,  8, 9, 64,45, 89,26,
  15, 9, 67,50, 78,37, 38,12, 32,38, 46,37, 15,13, 43,63,
   7, 6,  4,59, 32,88, 40, 8, 50,28, 26,28, 80,85, 11,67,
  41,86,  7, 4, 48,77, 89,27, 13, 4, 14,67, 81,40, 73,60,
  16,51, 66,41, 23,12, 69,79, 16,14, 27,40, 30,76, 15,39,
  84,36, 61,29, 17,18, 33, 5, 40,62, 84,66, 28,77, 83,57,
  36,64, 67,60, 34,29, 18,24, 19,12, 33,42, 33,30, 54,26,
  77,42, 50,46, 47,86, 62,55, 38, 0, 82,18, 62,64, 21,40,
  44,43, 38,44, 59,29,  2,19, 59,26,  7,74, 50,14, 55,73,
  41, 8, 46,45, 30,40,  6,47, 20, 9,  9,28,  5,65, 17,69,
  42,45, 41,67,  4,83, 15,32, 23,13, 12,64, 78, 3, 45,21,
   6,27, 75,45, 16,36, 57,64, 38,57, 62,13, 63,66, 76, 1,
   4,27, 64,89, 68,56, 40, 1, 18,25, 62,50, 58,53, 34,40,
  59,38,  1, 3, 13,46, 60,73, 72,24, 74, 5, 78,24, 83, 7,
  20,87, 66,51, 14,61, 10, 2, 82,32, 84,37, 35,19, 33,80,
  46,34, 55,83, 50,23, 41,49, 81,87, 28,66, 65,39, 61,52,
  88,26, 72,14, 64,76, 28,49, 31,43, 86,57, 25,17, 73,27,
  49, 3,  3,89,  7,83, 37,47, 25,59, 19,42, 88, 6, 70,12,
  18,39, 71,21, 79,68, 62,29, 53,73, 58,47,  2,84, 12,68,
  41, 3, 87,88, 65,22, 47, 0, 44,35,  1,18, 33,49,  3,82,
  12,49, 82,58,  2,81, 30,67,  5,83, 85,62, 32,47, 53, 3,
  36,27, 63,17, 40,45, 11,78, 36,78, 67,48, 70,75, 86,33,
  63,86, 20,47, 55,43, 50,86, 61,34, 60,43, 44,23, 22,86,
  66,12, 86,45, 27,86, 71,80, 39,77, 59,42, 65,59, 69,44,
  46,86, 62, 0, 27,57, 78,23, 51,19,  0,35,  2, 0,  1,53,
  51,76, 87,61, 74,33, 65,44, 42,17, 62,87,  3,37, 22,57,
  80, 4, 46,49, 71,64, 19,48, 53,46, 17,81, 68,67, 57,44,
  18,60, 40,33, 81,61,  9,60, 85,53, 28,51, 40,78, 36, 9,
  86,59,  0, 1,  7, 8, 71,56, 67,11, 83,26, 58,72, 79,37,
  82, 5, 11,29, 34,74, 51,73, 85,79,  7,59, 60,82, 52,19,
  51,72, 38,60, 45,16, 54, 2, 45,17, 59,58, 84,20, 39,27,
  30, 4, 81,25, 72,58, 20,30, 53,82, 20,55, 79,15, 65,27,
  88,59, 49,76,  8,20, 15,21, 52,36, 24,59, 66,42, 33,60,
  84,16, 76,73, 59,88, 32,54, 75,39, 83,17, 80,64, 55,49,
  49,46, 32,42, 43,44, 55,41, 54,20, 60,65, 23,25, 37,43,
  70,46, 74,55, 75,50, 63,26, 77, 2, 55,24, 81,84,  9,86,
  88,36, 67,26, 54,77, 26,35, 59,84, 63,24, 38,46, 19,89,
  63,62, 70, 1, 67,71, 41,64, 55,84, 38,59, 82,36, 10,49,
  60,11, 55,60, 51,48,  0,25, 12,89, 85,84,  0,36, 56,68,
  86,53,  8,69, 16,82,  3,70, 21,18, 70,55, 12, 3, 16,46,
   7,35, 86,58,  4,32, 35,11, 27,17, 30,84, 34, 9, 19,16,
  23,14, 52,57, 22,61, 50,48, 81,55, 52,70, 77, 5, 60,74,
  86,26, 14,64, 50,50, 60,47, 39, 3, 22,34, 49,59, 10,60,
  72,39, 74,29, 58,77, 38,33, 28,24, 83, 6, 40,52, 10,74,
  24,86, 12,17, 45,47, 44,83, 37,27, 77,88, 83,27, 87,72,
  55,19, 45,31, 16,80, 61,48, 17,89, 47,45, 32,15, 58, 8,
   4,25, 78,63, 52,34, 41,45, 10,31,  0,52, 23,84, 35,32,
  49,19, 66,31, 89,28, 25,55, 80,80, 78,64, 24,32,  1,48,
  32,14, 33,15, 61, 0, 28,14, 72,55, 14,81, 60,77, 60, 1,
  51,62, 10,72,  7,45, 49,87, 54,25, 16,11, 87, 2,  4,63,
  49, 4,  5,51, 57,17, 26,42, 55, 3, 34,23, 36, 3, 42,52,
   8,10, 14, 9, 64,28,  9, 3, 60,60, 69,85, 67,89, 63,88,
  16,63, 86, 4,  1, 1, 86,42,  0,56, 45,13, 18,41, 63,52,
  17,33,  0, 7,  7,62, 49,80, 57,65, 87,37, 12,33, 17,12,
  84,28, 17,40, 53,68, 17,15, 34,82, 74,53, 12,80, 79, 7,
  27,11, 75,15, 47, 4, 30,54, 85,40, 64,77, 23,50, 20,51,
  21,67, 36,29, 79,69,  0,57, 28,30, 34,68, 71,88, 82,59,
  58,17, 21,38, 81,15, 38, 3, 32,56, 14,69, 45,83, 52,56,
  72,59, 14,53, 54, 8, 44, 1, 89,59, 18,44,  0,58, 88,14,
  73,38, 85,83, 15,55, 23,71,  9,84, 11,21,  3,14, 53,62,
  80,41, 33,41, 11,77,  9,33, 67,79, 55,12,  7,76, 42,44,
  50,18, 13,70,  3,77, 73,34,  5,62, 63,79, 46,11, 87, 6,
  38,51, 65,56, 15,16, 65,24, 53,15, 50,80,  3,78, 61,46,
  14,63, 65,83, 64,35, 60,55, 25, 4, 79,24, 36,21, 43,39,
  11,14, 44,52, 39,33, 53,44, 79,11, 39,31, 78,16, 22,56,
  41,74, 69,12, 54, 7, 66,13, 44,80, 40,43, 85,24, 56, 1,
  75,18, 41,58, 22,11, 52,35, 79,56,  5,18, 76,72, 48,87,
  89,24, 80,56, 48,22,  8,88, 89,67, 45,40, 26,75, 59,36,
  23,11, 81,28, 32,12, 88,41, 25,45, 16,64,  9,34,  4,28,
  84,72, 59,19, 23,22, 73,29, 51,31, 54,66, 87,25, 55,66,
  28,20, 79,85,  5,82, 60,58, 13,76, 80,48, 25,25, 75,47,
  18,53, 60, 7, 80,79, 21,74, 21,30,  1,10, 16,31, 29,87,
   2,86, 11,39, 10,69,  4,39, 45,49, 57,62, 86, 9, 17,77,
  40,67, 42, 6, 79,63, 23, 5,  4,40, 54,62, 64, 4, 16,60,
  39,22, 89,25, 30,11, 70,30, 70,86, 34,10, 26,76, 61,36,
  83,77,  9,36,  5,80, 23, 9,  9,13, 58,64, 84,70, 39,64,
  36,85, 47,46, 75,70, 78,73, 37,49, 48,61, 73,59, 79,13,
  77,79, 17,85, 28,84, 55,55, 77,65, 72,35, 18,50,  1,19,
  39,24, 88,19, 11,87, 67,47, 85,45, 22,33, 46,39, 82,73,
  88,71, 47,35, 41,19, 10,34, 12,39, 85,67, 53,70, 80,86,
  33,22, 85,13, 67,46, 33,44, 28, 1, 38,16, 79,61, 83,87,
  12, 1, 82,38,  8,46, 89,48,  9,73, 31,52, 66,52, 56,51,
   4,77, 77,39, 50,33,  2,37, 43,40, 70,34, 44,74, 15,35,
  38,83, 53,57, 46,88, 50,58, 16,48, 34,47, 87, 1, 22,85,
  33,84, 82,65, 28,54, 27,81,  7,13, 78,29,  9,78, 73,37,
  76,13, 40,48, 61, 5, 80,10,  8,26
};
const int p_90_5[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1063,
  /// Array of preferences
   6,73,  4, 6, 55,69, 53,10,  0,36, 66,59, 52, 8,  1,22,
  21,37, 33,48, 81, 7, 67,54, 36,48, 63,73, 57,58, 89,53,
  69,52, 43,22, 35,32,  5,35, 24,56, 51,21, 84,56,  1,12,
  51,74, 68,22, 28,13,  8,17, 72,61, 21,12, 52,56, 36,79,
  77,65, 52, 4, 79,61, 12,17, 83,70, 24,78, 25,60,  3,28,
  11,83, 76,29, 88,52, 46,46, 86,68, 47,78, 35,46, 72,70,
  85,53, 34, 3, 24,72, 86,59, 31,61,  8,70, 26,38, 35,76,
  56,63, 26,37, 24,48, 43,37, 62,10, 27,86, 15,61, 19,48,
  22,53, 30,21, 20,21, 87,53, 30,74, 35,69, 84,73, 68,89,
  81, 8, 11,88, 53,41, 61,57, 63, 2, 22,83, 18,57, 72,20,
  81,57, 85,46, 74,53, 60,44, 45,82, 75,44,  6,18, 80,48,
  21,66, 11, 5, 35,13, 64,43, 21,24, 60,84,  0,61, 28,73,
   9,31, 54,63, 42,80, 36,58,  4,17, 79,14,  4,13, 24,82,
  71,81, 29,63, 43, 9,  5,36, 20, 8,  4,53, 69,72, 14,50,
  58,78, 42,49, 37,66, 30, 1, 39,12, 42,52, 60,56, 78,17,
  23,34, 39,62, 68, 2, 37,44, 89, 5, 88,84, 10,78, 71,44,
  47,88, 42,33, 43,20, 79,60, 56,40, 86, 5, 45, 0, 13,69,
  76, 0, 58,10, 14, 3, 73,64, 17,82, 21,58,  8,74, 22,78,
  86,27, 53,38, 36,60, 62,74, 77,51, 45,59, 40,22, 18,47,
  40,19, 76,12, 51,84, 57,70, 78,48, 84,30, 36, 4, 79,34,
  66,56, 27,51, 88,65, 10,84, 12,70, 37,31, 48,27, 36,16,
  60,60,  3,72, 38,31, 70,66, 30,82,  4,42, 37,67, 46, 4,
  29,38, 65,68, 55,50, 59,79, 56,76, 61, 4, 52,28, 72,14,
  24,23, 42, 3, 81, 4, 58,18, 18, 5, 61,74, 74,57, 50,87,
  46,84, 68,18,  0,15, 30,62, 57,47, 60,78, 20,23, 43,66,
  56,83, 64, 9,  7, 8, 65,46, 13,32, 61,48,  8, 1, 64, 4,
  53, 5, 59,17, 34,86, 11,77, 20,56, 70,29, 51,68, 26,39,
  63,35, 34,23, 33,61, 41,23, 19,31, 42, 8, 24,68, 78,52,
  41,27, 31, 2, 64,64,  1,81, 54,54, 20,77, 61,39, 20,65,
  49,82, 28,79, 70,39, 55,31,  7,85, 53,25, 26,18, 75,89,
  31, 4, 52,85, 89,31, 41,70,  9, 7, 78, 2, 82,64,  4,71,
  82, 6, 26,25, 27,44, 82,44, 40,26, 58,27, 16,28, 18,70,
  48,68,  2,88, 36,25, 35,71, 88,15, 12,67, 39,47,  8,46,
  49, 8,  1,45,  9,49,  0,34, 71,40, 72, 8, 72,10, 82,86,
  83,35, 29,73, 68,14, 15,33,  2,41, 54,12, 26, 9, 27, 9,
   1,60,  6,80, 58,68, 37,47, 61,15, 39,71, 24,47, 24,53,
  62,20, 54,34,  2,68, 58, 2, 31,47, 35,27, 61,68, 29,78,
  17,31, 61, 6, 17,62, 49,36, 36,54, 60,18, 76,14,  1,65,
   5,72, 44,65, 56,26, 74,50, 44,10, 44,78,  2,25, 59,56,
  44,45, 81,83, 44, 5, 57,34, 13,61, 22,61, 34,62, 45,51,
  13,21,  1,40, 61,56, 22,11, 77,80, 33,82, 51, 7, 28,66,
  77,18, 76,50, 77, 6,  6, 6, 44, 2, 88,49, 29,79, 30,47,
  34,51, 58,42,  2, 5, 11,57, 86,41,  1,32, 62,11, 80,69,
  41,22,  2,57, 15,27, 22,70, 45,32, 77,64, 56,23, 34,25,
  13,76, 42,76, 61,26, 76,64, 46, 9, 84,20, 57,64, 68,74,
  21,35,  4, 5, 42,70, 64,30, 49,62, 57,88, 82,12, 86,51,
  29,21, 48,81, 54,68,  6,39,  4,35, 61,34, 56,62,  6,60,
  57,49,  0,25, 42,61, 45,74, 38,56, 13,34, 32, 1, 57,40,
  37,73, 42,38, 25,30, 41,79,  5,73, 43,79, 75,45, 31,20,
  10,52, 78,10, 24, 7, 37,74, 66,54, 16,65, 83,30, 60,33,
   5,75,  5,54, 31,62, 78,25, 14,62, 66,16, 79,26, 61,89,
  14,54, 78,86, 31,65,  4,46, 18,34,  9,80, 73,66, 44,57,
  60,51, 69, 4,  7,12, 15,17, 30,71, 60,77, 21,22, 77,62,
  47, 9, 73,59, 43,12, 51,30, 21,69, 19, 3,  7,27, 75,84,
  18,42, 18,86, 85, 1, 85,54, 75,63, 26,46, 82,67, 63,47,
   0,83, 57,50, 72,26, 27,68, 39,53,  6,50,  4,55,  4,33,
  41,10, 50, 0, 72,60,  3,83, 67,56, 76,32,  2,58, 32,62,
  24,31, 15,35, 69,10, 49,33, 66,70, 65,79, 33,75, 27,87,
   9,89,  4,88, 77,60, 50, 6, 46,41, 48,43, 57, 6, 77,74,
  79,31, 26,34, 80,66,  8,26, 15,51, 36,82, 20,15, 36,14,
   4,74, 41,42, 77,61, 81,37, 24,29, 60,28, 18,52, 18,35,
  41,57, 74,20,  0,86, 30,45, 80,81, 70, 7, 12,52, 27,58,
  24, 8, 54,77, 54,55, 82,62, 64,25, 53,50, 37,36, 35,81,
  25,24,  9,64, 81,48, 28,33, 13,67, 64,68, 50,82, 78,21,
   8,85, 35,74, 79,12, 66,68, 87,80, 17,35, 86,72, 24,46,
  36,88, 41,81, 87,32, 12,81, 57,16,  5,82, 50, 5, 32,38,
  30,49, 55,32, 70,35, 22,43,  0,77, 35,42, 67,61, 11,86,
  14,57, 32,69, 41,19, 80,88, 45,62, 46,77, 74,89, 26,28,
  19,59, 58,21, 77,13, 54,33, 33,73,  4,58, 53,45, 31, 5,
   0,73, 86,29, 31,58, 80,74, 61,87, 40,59, 29,56, 77,21,
  85,61, 79,42, 72, 0, 11,24, 39,50, 33,45, 48,36, 58, 6,
  37, 2, 69,20,  5,13,  6,49, 51, 5, 58,79, 14,87, 35,58,
  74,33, 16,29, 40,58, 26,85, 73,20, 82,40, 24,87, 64,10,
  58,25, 17,29, 34,42, 47,33, 30,34, 55,27, 74,68, 15,79,
  68,73, 77,40, 27,67, 71,85, 87,63, 82,51,  2,26, 37,28,
  42,17, 47,74, 71,54, 18,64, 13,52, 34,18, 19,88, 16,38,
  31,49, 59,65, 40,75, 10,89, 37,88, 43,58, 23,85, 34,28,
  58, 3, 73, 2, 68,10, 52,50, 65,75,  6,87, 74,30, 79,40,
  22,72, 16,16, 67,72, 80,65,  0,40, 37,58,  5,32, 77,27,
   8,18, 23,89,  4,30, 36,12,  2,52, 65,40, 77,72, 22,15,
  13, 2, 86,47, 75,18, 82,47, 63,79, 68,57, 68,64, 54,84,
  82,52, 67, 9, 38,20,  1, 7, 58,29, 85,33, 62,17, 88,74,
  48,49, 41,87, 55,10, 11,81, 68,25, 23,28, 72,69, 83,16,
  85, 2, 88,50, 68,49, 51,32, 44,21, 12,20, 25, 9, 27,55,
   0, 8, 65,72, 88, 9, 30,16, 33, 7, 70,51,  4,23, 30, 9,
  21,73, 63,22, 10,25, 22,62, 17,19, 30,58, 76,37, 30,24,
  24,22, 44,76, 76,40, 29,22, 13,53, 50,57, 46,69, 43,31,
  26,61, 53,22,  7,15, 70,55, 66,64, 88,27, 43,41, 25,36,
  19,15, 36,38, 14,81, 48,35, 48,66, 67, 2, 49,21, 67,46,
   6,58, 73,43, 44,39, 56,15, 75,41, 50,12, 43,21, 84,10,
  77,86, 40,40, 63,16, 16,17, 19,18, 11,42, 31,14, 60,10,
  25,81, 29,89,  7,19, 50, 3,  3,84, 78,76, 86,65,  1,51,
  21,34, 23,88,  5, 8, 52,58, 75,51, 14,11, 73,19, 32,12,
  51,59, 85,76, 82,55, 49,60, 14,67,  4,60, 61, 7, 87,19,
  36,89, 41,88, 62,71, 66,36,  5,26, 25,75, 64,77, 56,86,
  23, 8, 79,10, 89,54, 52,36, 32,87, 27,62, 12,22, 57,62,
  10,49, 81,18, 61,88,  3,53, 45, 4, 83,59, 38,43,  8,32,
  53,61,  3,10, 59,14, 44,53, 16,25,  9,75, 89,30, 13,60,
  55,11,  4, 2,  2,86,  7,78, 74,87, 31,60, 51,76, 30,29,
  39,84,  0,53, 86,69, 59,87, 38,44, 21,63,  1,37,  3,85,
   2,18, 19,13, 10,42, 43,59, 68,47, 55,54,  9, 9, 48,87,
  61,20, 48,34, 85,40,  3,35, 80,22, 70,34, 48,53, 23, 0,
   6,46, 70,47, 20,66, 13,80, 56,68, 88,71, 82,36, 14,10,
  16,69, 22,52, 54,59, 72,84, 88,86, 57, 5, 72,49, 44,61,
  46,85, 20,84, 40,23, 20,50, 32,46, 15,49, 83,27, 35, 4,
   7,29, 79,67, 83,47, 62,38, 39, 5, 85,24, 12,85,  8,22,
  84, 0, 38,36, 74,38, 41,76, 70,85, 59,55, 71,30, 19,22,
  66,82, 27, 0,  0,64,  8,77, 43,89, 77, 0, 44,42, 39,21,
  18,49, 45,38, 50,30, 16,60, 42,35,  5,80, 39, 4, 69,66,
  80,36, 85,84,  2,42, 49,84, 59,88, 84,21, 59,24, 38,40,
  57,53, 30,15, 20,29, 60, 3, 88,11,  1,80, 87,20, 15,36,
  33,23, 23,77, 17,56,  6,37, 11, 9, 32,54,  5,45, 70,14,
  24,41, 33,10, 24,84, 69,48, 68, 9, 14,58,  1,64, 73,11,
  16,89, 87,16, 86,62, 60,49, 63, 8, 13,35,  9, 3,  4,64,
  28,69, 51,75, 77,37, 55, 1,  0,80, 66,61, 12,61,  4,39,
  40,73, 85,42, 73,26, 47,48, 33,60, 28,38, 52, 7, 23,68,
  32,29, 43,54, 66,75, 76,56,  8,76, 67, 6, 69,37,  6,71,
  24,64, 77, 1, 40,85, 84,89, 44,47, 62,81, 68,84, 50,27,
  38,51, 81,33, 44,20, 60,66, 67, 5, 31,44, 66,45, 35,43,
  84,59, 46,65, 67,47, 78,84, 54,35, 48, 3, 51,18, 18, 3,
  16, 8, 21,87, 49,85, 20,25, 33,50, 34,33, 19,37, 65,56,
   8,84, 73,16, 62,37,  1,49, 11,85,  5,20, 48,12, 40, 7,
  60,63,  0,49, 89,57, 80,19, 29,44, 11, 6, 75,43, 17,86,
  21, 7, 21,33, 81,23, 73,82, 27,47,  6,32, 34,71, 21,47,
  27,10, 46,48, 59,59, 60,61, 49,18, 61, 9, 80,37, 21,11,
   1,62,  9,76, 25,72, 66,25,  4,68,  2,10, 19, 8, 29,10,
  88,63, 49,87,  2,44, 29,32, 34,58, 89,70, 37, 7, 81,69,
  20,81, 74,60, 64,45, 38,24, 60,54, 11,15, 15,56, 15,44,
  36,75, 83,48, 72,46, 71,33, 59,57, 39,74, 82,31, 82,81,
  23,67, 16,21, 10,51,  9,34, 14,68, 40,68, 30,75, 14,46,
  64, 3, 66, 9, 53,13, 62,53, 64,32, 58,46, 20,36, 60,53,
  62,57, 42,20, 27, 2, 88,81, 25,33, 52,15, 85, 7,  8,89,
  66, 0, 16,73, 78,58, 11,60, 72,47, 65, 0, 66,14
};
const int p_90_6[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1613,
  /// Array of preferences
  87,77,  7,33, 45,18, 20,76, 51,71,  1,81,  7,35, 14,83,
  59,65, 81,24, 83,57, 32,58, 25,23, 27,59, 67,34, 38,62,
  21,20, 55,26, 68,10,  7,29, 30,15, 40,56, 49,61, 64,18,
  60,88, 88, 2, 72,44, 42,20, 23,52, 68, 8, 87,56, 73,33,
  55,33, 78,89, 80,70, 65,31, 52,84, 87,24, 70,18, 75,65,
  66,11, 38,19, 76,50, 80,39, 33,47, 30,46, 53,48, 29,33,
  36, 1, 65,86, 60,44,  2, 9, 86,87, 17,61, 54,48, 74, 7,
   0,62, 88,47, 13,77, 22,21, 16,81, 31,78, 46,80, 41,71,
  38,72, 25,25, 23,72, 22, 2, 42,13, 11,65, 36,30, 21,50,
  14,49, 27,63, 10,31, 81,62, 56,34, 24, 5, 81,45, 48,78,
  75, 9, 51,38,  5,52,  2,88, 78,10, 85,79,  5,71, 18,18,
  29, 1, 51,41, 82,64, 75,33,  3,81, 82,33, 62,41,  7,47,
  28,74, 63,78, 70,87, 25,49, 86,13, 46,40, 53,62, 27,82,
  59,45, 23,67, 56,29, 39,25,  3, 1, 50,71, 47,13, 23,41,
  60,10, 49,78,  1,25, 51,21, 13,51, 36,37, 28,25, 71,37,
  49, 5, 11, 2,  7,43, 18,15, 70,40, 45, 2,  5,45, 61, 2,
  55,17, 70,37, 32,83, 24,20, 56,67,  0, 1, 52,28, 29,12,
  58,87, 48,23, 34,48, 41,25, 45,73, 25,78, 34,47, 74,26,
  81,51, 51,55, 23, 7, 81,67, 35,12, 70, 7, 75,63, 17,54,
  45,65, 84,86, 86,73, 21,10, 10,47, 76,75, 73,52, 84,36,
  25,39, 42,26, 70,44, 34,53, 66,82, 75,79,  4,86, 72,49,
  19,22, 74,85, 69,77, 24,74, 37,84, 67,39,  0,28, 32,72,
  79, 1, 88,63, 63,65, 52,49, 60,23, 19, 4, 15, 8, 15,84,
  58,74, 60,32, 42,85, 16,24, 42,74, 62,53, 49, 9, 43,75,
  61,67, 61,37,  3,33, 56,89, 89,73, 53,65, 12,76, 73,36,
  79,74, 46, 7, 49,89, 60,39, 71,44, 19,51, 21,82, 39,72,
  89,67,  6,57, 66,68, 54,56, 77,16, 24,46, 50,51, 82, 7,
  66,38, 89,38, 11,67, 75,72, 71,79, 81,25, 55,45,  9,14,
  44,79, 14,24, 77,37, 89,29, 53,63, 77,30, 64, 4, 81,46,
  49,57, 72,14, 27, 7, 84,56, 43, 9, 64,21,  5,42, 64,85,
   6,69, 22,82, 62,67, 12,89, 14,61, 74,65, 44,18, 17,66,
  81,81, 79,16, 60,51, 85,89,  3,23, 33,88, 54,74, 24,27,
  46,89, 83,51, 26,61, 59,72, 79, 7,  0,78, 36,16, 50,84,
  70,52, 44,89, 55,60, 10,69, 23,35, 69,17,  0,51, 84,47,
   3, 2,  8,69, 43,43, 74, 9, 23,20, 45,71, 13, 1, 45,74,
  43,53, 58,23, 48,51, 44,37, 65,35, 87,34, 16, 5, 21, 8,
  88,72, 75,29, 16,65, 22,38, 89, 6, 22,17, 73,66, 35,14,
  60,71, 73,73, 27, 9, 31,23, 49,43, 18,35, 25,38,  9,66,
  81,58, 42,40, 37,12, 29,61,  8,56, 20,54,  5,72, 27,28,
   0, 2, 16,71,  9, 2, 43, 1, 64,44, 71,27, 70, 5, 44,20,
  25,84, 69,20, 14,58, 76,65, 33,84, 35,15, 73, 1, 58,70,
   6,65,  9,23, 60,56, 40,76, 39,18, 39,23, 44,38, 70,71,
   0,20, 26,49, 74,59, 46,16,  2,77, 74,25, 22,75, 39,82,
  28,72, 30,49, 48, 4, 22,74, 36,29, 22,83, 43,62, 86,51,
  74, 3, 15,71, 85,65, 68,38, 41,76, 71, 9, 67,64, 41,17,
  40,35, 42,47, 25,31, 55,62, 12,14,  9,69, 67,56, 38,43,
  43,18, 22,39, 26,55, 29,55, 89, 4, 61,68, 76,78, 73,63,
  27,62, 82,47, 20, 0, 24,77, 48,13, 44,87, 34,42, 63,87,
  78,28, 63,69, 10,21,  1,75, 64,54, 68,54, 67, 2, 11,40,
   0,83, 46,57, 26, 7, 78,71, 28,14, 54,36, 29,47, 78, 4,
  69,33,  9,22, 15, 0,  9,88, 55,85, 35,60, 23,39, 37, 6,
  33,51, 10, 5, 75, 4, 77,34, 46,24, 15,13, 84, 0, 70,72,
  69,68, 59,64,  8,46, 74,20, 47,89, 37,27, 88,13, 41,68,
  43,47, 14,15, 14,47, 79,67, 54,84, 32,46,  1,76, 44,54,
  85,51, 39,73, 81,41, 88,89, 18,81,  1,82, 26,12, 45,29,
  50,76, 89,78, 19, 6,  2,48, 44,80, 18,19, 42,24, 89,41,
  10,78, 62,77, 64,57, 80,72, 36,66, 36,48, 21,16, 83,46,
  48,41, 61,47, 16,34, 46,51, 12,43, 30,24, 39,56, 56,25,
  32,28,  4,76, 80,32, 65,72, 48,55, 37,83, 48,57, 14,26,
  88,45, 43,16, 37,57, 20,24, 23,71, 59,40, 51,54, 20,13,
  51,36, 46, 1, 16,83, 32,89, 26,38,  4,40, 45, 1, 88, 8,
  30,12, 60,73, 42,33, 75,86, 37, 0,  8,64, 10,51, 36,56,
  38,11, 67,46, 75,46, 50,31,  9,65, 30,28, 32,38, 30,63,
  43,59, 85,88, 77,39, 51,40, 30,13, 27,57, 37,80, 63,34,
  79, 8, 81,69, 43,61, 44,58, 34,66, 80,27, 78,88,  0,56,
  23,14, 76,25, 11,25, 52,25, 55,87, 57,14, 44, 7, 60,52,
  40, 9, 40,39, 69,85, 89,36, 62,57,  2,10,  6,47, 33,59,
  85,62, 42,15, 15,21, 16,16, 16,58, 17,24,  8,55, 11,48,
  31,67, 41,70, 45,48, 43,10, 81, 5, 24,54, 55,52, 80,51,
  22,31, 35,77, 22,26, 45,82, 83,33, 16,89, 66,75, 26,67,
  31,64, 88,52,  1,80, 89,70, 31,70, 81, 3, 42, 4, 43,36,
  42,81, 49,13, 69,72,  6,42, 13,60, 35,11, 20,21, 22,67,
  77,27, 82,82, 52,66, 88,68, 82,20, 28,15, 38, 3, 47,82,
  18,52,  9,76, 51,25,  3,22, 77,44, 20,14, 69,70, 73, 7,
  51,43, 39,67, 27, 3, 29,67, 20,71, 27,69,  0,77, 76,61,
  34,23, 67,18, 51,52, 77,67, 37,29, 66,13, 85,40, 36,85,
  24,42, 72,55, 24, 2, 80,54,  9,39, 57,55, 74,53, 45,89,
  12,66, 66,47, 88,38, 32,23, 58,46, 53,43,  5,54, 16,13,
   9, 5, 29,82, 84,41, 87,57, 33,21, 34,21, 32,19, 77,17,
  12,26, 63,73, 14,14, 39,65, 63,72, 53,60, 30,53, 87, 9,
  70,31, 42,61, 16,25, 83,79, 41,50, 37,64, 18,80, 60,21,
  26,19, 13,61, 40,87, 12,13, 20,73, 55,53, 31,55, 87,81,
  89,80, 76,55, 88,11, 46,53, 78,34, 41,54, 72,36, 57,37,
  45, 3, 88,21, 40,62, 71,52, 63,16, 34,30, 50,70, 18,32,
   6,67, 38,83, 50,58, 73,69, 56,48,  1,40, 83,50, 86,83,
   5,57, 34,10, 51,31, 21,44, 84,42, 46,10, 69,13, 41,46,
  35,65, 43,29, 86,79,  3,74, 68,88,  1,62, 38,69, 16,15,
  18,59,  7,12, 77, 8, 71,15, 32,41, 80,65, 52, 6, 76,49,
  82,52,  3,72, 15,26, 83,14, 71,71,  2,37, 83,80, 66,42,
  35,13, 16,27, 60,40, 87,89, 78,70, 61,78, 54,44, 42,79,
  12,86, 78,40,  8,54, 43,69,  8,53,  2,47, 51,70, 36,46,
  37,78, 51,76, 48,75, 29,83, 52,44, 51,49, 17,67, 31,47,
  31,31, 83,74, 68,77, 52,61, 55,12, 62,39, 27,47, 28, 3,
  15,74, 88,82, 59,48, 18,67, 17,35, 41,65, 84,49,  1,85,
  68,78, 54,55, 56,83, 50,85, 17,29, 38,80, 88,70, 83,67,
  53,40, 77,66, 70,54, 41,89, 10, 4, 17,62, 35, 5, 50, 2,
  16,36, 74,46,  6,78, 82,41, 28,54, 89,68, 64,13, 19,79,
  53,89,  5,50, 70,33, 26, 5, 20,19, 87,11, 60,59, 37,50,
  33,62, 35,76, 10,84, 35, 3, 84,37, 25,71,  7,23, 87,62,
  75,28, 50,13,  6,49, 67, 1, 74,61, 76,51, 62,11, 33,61,
  41,21, 88,65, 28,51, 80,82, 60,63, 31,39, 21,88, 87,72,
  21,26, 42,87, 78,13, 72,17, 84,82, 49,26, 89,50, 64,24,
  23,11, 59, 0, 62,17, 52,41, 75, 0, 66,72,  5,21, 77,60,
  55,35, 44, 6, 74,80,  6, 7, 45, 7, 70,60, 44,28,  9,54,
  79,25, 55, 7, 68,12, 55,57, 17,32, 61,15, 26,76, 32,74,
  46,44, 60,76, 28,64, 87,44, 24,39, 75,30, 86,33,  2,12,
  69,49, 12,46,  2,34, 77,31, 15,25, 43,57, 86,64,  1,23,
  71,13, 83,37, 80,31, 89,88, 56,71, 49, 0, 55,68, 78,37,
  77, 3, 19,21, 88,54, 32,75, 33,65, 82,53, 34, 9, 70, 8,
  86,34, 84, 6, 77,32, 78,81, 65,64, 78,39, 69,12, 27,55,
  29,11, 81, 8, 58,86, 73,84, 66,73, 11,38, 74,36, 59,77,
  72,21, 32,86, 61,12, 88,66, 68,46, 36,67, 40,68, 75,49,
  41,31, 34,75, 23,88, 14,33, 35,59, 25,20, 79,48, 41,26,
  72,10, 52,62, 42,89, 54,62, 39,78, 35,71,  3,21, 62,60,
  37,41, 55,67, 17,75, 41,47,  5,79, 86,77,  6,45,  0,23,
  19,31, 26,30, 65,33, 17,51,  0,81, 40,40,  0,27, 78,26,
  55,44, 38, 5, 55,77, 41,33, 69,39, 87, 7, 63,18, 61,84,
  74,39,  9,37, 27,14,  1,88, 50,53, 64, 3, 20,44, 76, 1,
  19,89, 54,43, 85,21, 85,73, 45,38, 64,47, 72,29, 48,60,
  27,32, 56, 9, 83,17, 44, 4, 51,37, 27,25,  7,11, 42,23,
  40,89, 39,11, 89,33, 61,21, 12,65,  7,68, 34,59, 78,64,
  31,19, 59,52, 35,85, 57,51,  7,44, 89,26, 17,74, 30,87,
  18,47, 67,68, 44,45, 22,61, 83, 5, 68,47,  9,49, 55,15,
  28,35, 29,73, 48,24,  8, 6, 26,72, 79, 9, 45,76, 51, 9,
  34,34, 44,48, 69,22,  0,11, 61,74,  4,28, 64,61, 80,24,
  86,68, 81,76, 19,45, 78,14, 41,48, 59,28, 42,32, 58,24,
  89, 5, 63,82, 40,14, 65,10, 11,46, 70,32, 56,27,  2,85,
  86, 1, 76,66, 42,69, 81,14, 40, 5, 87,27, 65,45, 84,64,
  19,57, 12,37, 55,89, 26,63, 75,59,  1,87, 30, 6, 39,32,
   5,61, 31,82,  7, 1, 70,41, 34,35, 54,39, 39,58, 34,56,
  44, 2, 64,77, 58, 8, 37, 4, 12,68,  5,63, 48,63, 78,19,
  82,24, 65,77, 39,29, 38,50, 81,20, 30, 3, 77, 5, 42, 0,
  62,36, 34,26, 28,45, 56,77, 56,58, 73,45, 18,10, 24,14,
   4,88, 34,12, 44,19, 81, 2, 52,86, 88,39, 53,78,  9,18,
   2,26, 19,42, 44,69, 73,83, 25,17, 48,64, 17,39, 33,87,
   1,41, 55,48, 82,42, 39,24, 28,52, 72, 5, 75,25, 25,18,
  45,36, 59,79, 50,20, 47,53, 62, 8, 89,48, 87,60, 54,19,
  10,54, 67, 9, 60,26,  0,47,  1,56, 65,24,  7,59, 78,58,
  23,75, 69,82, 11, 8, 57, 8, 85,84,  3,34, 58,27, 51,86,
  53,73, 38, 0, 83,44, 61,18, 47,55, 18,11,  9,36, 87,82,
  25,53, 54,38, 56,56,  8,38, 16, 8, 56,39, 49,32, 13,75,
   4, 4, 59,76, 33,83, 36,19,  6,28, 73,24, 10,39,  1,60,
  58,80, 59,89, 66,45, 21, 4,  5,14, 15,80, 26,88, 26,33,
  40,69, 35,36, 45,33, 45,61, 59, 6,  8, 8, 20,35, 68,18,
  24,76, 60,35, 26,59, 30,72, 70,26, 78,51, 20,17, 55, 3,
  10,22,  4,68, 86,19, 30,65, 63, 3, 58, 5, 26,45, 81,15,
  77,38, 24,69, 26,60, 85,80, 49,58, 66,44, 22,48, 19,58,
   8,19, 46,68, 84,66,  7,41, 68,66, 84,25, 19,82, 49,50,
  49, 1, 18,82, 47,34, 36,23, 56,22, 85,71, 83,82, 17,37,
  19, 7, 14,20, 33,35, 42,38, 70,29, 58,78, 53,16, 12,18,
  78,80, 32,56, 54,72, 81,16, 48,10, 67, 5, 11,57,  6,52,
  55,54,  0, 0, 34,41, 82,27, 88,15, 58,82, 71,20, 58,35,
  12,17, 78,73, 80,60, 50, 7, 67,28, 68,80, 42,22, 40,22,
  51,72, 62,56, 47,31, 19,76, 22,49, 39,71, 67, 6, 83,28,
  72,64, 38,70, 86,55, 63,23, 77,75, 84,40, 67,36, 60,36,
  26,37, 88,75, 53,13, 62,46, 61,27, 37,56, 58,51,  2,40,
  17,12, 64,49, 71,16, 47,36, 59,80, 51,73, 10,34, 78,30,
  82,55, 34,64, 18,17, 60,28, 81,71, 10,45, 13,88, 50,18,
  11,53, 63,49, 86,45, 60, 3, 22,66, 80,14, 67,11, 70,25,
  11, 7, 24,60, 72,81, 14,88, 56,75,  6,31, 75,42, 65,67,
  12,20, 14,27, 21,24, 88,58, 73,72, 33,13, 71,63, 54,29,
  32,64, 13,47, 45,56, 69,26, 72,40, 30,89, 25,88, 28,55,
  56,76, 58, 0, 40,58, 25,46, 70, 4, 16,17, 65,80, 82,83,
  26,39, 12,74, 66,63, 67,65, 15,89, 17,13, 68,48, 67,87,
  52,77, 67,78, 73,39, 68,26, 58, 2,  1,59, 81,10, 29,41,
  56,26, 10,20, 72,65, 20,11, 67,38, 40,30,  7,36, 36,68,
  21,60, 70,38, 49,77, 88,48,  8,36, 43,64, 69,69, 65,82,
  49,14, 69,40, 68, 6, 62,88,  1,31, 59,86, 73,14, 61,24,
  72,11, 56,28, 35,56, 81, 9, 60,37, 46, 0, 89,82, 23,42,
  34,73, 48,33,  6,73, 54, 0, 52,36, 76,64, 28,27, 46,59,
  88,22, 70,16, 12,53, 42, 7,  4, 0, 46, 2, 56,17, 34,18,
  78, 9, 46,48, 39,88,  0,21,  9,61, 37,32, 56,87, 20,69,
  38,63, 52,57, 23,44, 59,24, 75,69, 68,55, 74, 2,  3, 8,
  10, 2, 28,34, 18,50, 23,60, 76,67, 16,11, 32,20, 79,79,
  27,89, 67,53, 74,84, 59,27, 53,58, 51,23, 73,38,  0,70,
  11,11,  3,60, 19,28, 46,78, 77,73, 79,75,  3,26, 16,47,
   4,62, 46,35, 12,80, 31,27, 16,84, 65,68, 37,49, 46,54,
  65,75,  6,59, 24,59, 89,42, 55,27, 89,65, 18,58, 12,60,
  15,35, 61, 1, 44,64, 27,44, 21,17, 60,33, 62,80, 64,82,
  53,32, 10,10, 26,31, 84,54, 76,46, 13,30, 57,53, 54, 1,
  18,56, 31, 5, 84,88, 70,49, 68,84, 52,37, 69,30, 37,48,
  69,29, 75,45, 69,41, 27,24, 59,44, 89,79, 30,19,  0,36,
  74,72, 23, 9, 65,84, 49,44, 83,69, 44,11, 25,64, 73,76,
  28,81, 86,32, 23,62, 49,31, 54,73, 17, 9, 14,67, 41,63,
  68,14, 47,74, 17,71,  1,67, 10,27,  9,21, 61,80, 40,77,
  20,49, 75,82, 31,22, 66,41, 49,36, 38,35, 70,84, 80,66,
  21,55, 79,68, 43,44, 68,81,  7, 5, 41,22, 18,49, 73,58,
  14,13, 55,13, 75,58, 30, 9, 25,33, 73,88, 38,14, 58,54,
  25,51, 10,23, 60,53, 35,45, 14,52, 48, 3, 89,85, 31,76,
  65,17, 85,35, 30,35, 36,57, 78,53, 68,50, 60,60, 55,74,
  26, 8, 54,85, 59,59, 21,14, 34,81, 61,62, 55,18, 76,58,
  47,30, 64,69, 34, 3, 52,17, 11,13, 59,13, 43,76, 80,48,
  89,34, 14,45, 62, 5,  3,58, 49,46, 42,12, 47,52, 59,69,
  77,10,  8,40,  6,76, 59,23, 80, 3,  5,49, 70,80, 38, 7,
  30,26, 16,77, 58,55, 64, 9, 16,85, 56,18,  1,50, 72,20,
  43,72, 76,33, 43,33,  6,37,  1,89, 40,38, 44,42, 28,75,
  33,74, 60,65, 72, 2, 19,62, 40,26, 86,66, 10,17, 78,57,
  22,27, 25,62, 32,24, 33, 5, 56,10, 44,67, 14,84, 57,62,
  69,76, 76,73, 26,86,  2,74, 78,35, 60,77, 89,18, 57,44,
  38,15, 32,67, 16,26, 53,81,  6,62, 86,41, 54,14, 16,61,
  32,42, 81,79, 33,11, 69,67, 47,75,  8,50, 26, 0, 12, 2,
  88,43,  8,72, 73,48, 50,24,  6,11
};
const int p_90_7[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1154,
  /// Array of preferences
   3,69, 75,19, 28,83, 31,46, 41,73,  1,71, 15,54, 58,38,
  32,26, 76,80, 89, 1, 33,21, 31, 0, 74,88, 80,32,  9,33,
  50,50, 57,89, 71,37, 32,73, 58,35, 74,75, 49,75,  4,81,
  72,41, 38,33, 47,36,  0, 5, 82,81, 41,39,  0,27, 25,20,
  71,45, 57,62, 70,50, 32,12, 86,24, 67,26, 84,56, 71, 2,
  65,49, 39,87, 52,58, 64,59, 44,73, 76, 2, 77,81, 41,11,
  13,60, 87,80, 52, 7, 57,87, 83,80, 38,46, 68, 6,  5,60,
  78, 3, 87,12, 14,62, 61,30, 24,67, 14,43,  7,21, 29,14,
  36,13, 14,58, 35,61, 36,87,  0,39, 22,16, 43,65, 87,35,
  81,51, 63,10, 22,77, 86,58, 59,74,  7,42, 81,13, 45, 1,
   5,81,  4,87, 80,30, 68,40,  6,25, 61,48, 85,69, 58,36,
  78,61, 74, 3,  8,85, 68,68, 55,12, 36,30, 67,87, 85,26,
  63,47, 87,54, 16,87, 59,37,  3,11, 48,34, 49, 8, 35,23,
  61,63, 74,12, 82,62, 16,81, 33,67, 18,82, 37,60, 15,58,
  55,41, 30,80,  9,37, 76,73, 41,57, 61,35, 50,89, 28,87,
  37,89, 13,52, 36,57, 21,46, 89,78, 45,53, 82,41, 15,37,
   0,89, 22,85, 21,68, 84,31, 18,80, 30,64, 55,77, 18,28,
  62,60, 17,18, 12,82, 35,56, 69,54, 53,58, 88,31,  3,24,
  48,75, 10,76, 53,79, 68,15, 66,78, 44,29, 29,51,  2,74,
  34, 8,  7,22,  7,65, 48,38, 48, 1, 40,87, 71,20, 16,28,
  25,50,  6,28,  0,48, 82,23, 21,50, 42,51, 66,57, 85,29,
  23,67, 88,48, 55,30, 21,70,  5,58, 87,14, 61, 0, 46,72,
   6,10, 79,25, 40, 1, 36,33, 75,62, 58,15, 43,53, 89,63,
  68,45,  0,54, 83, 3, 38,13,  8,86, 16,17, 19, 8, 79,85,
  81,76, 31,38, 22,42, 17,34, 70, 7, 76,77, 69,61, 44, 6,
  56,85, 86, 0,  6,72, 48,53, 38,16, 21,49, 26,58, 73,80,
  37,25, 68,44, 31,40, 65,62, 72,10, 16,39, 35,58, 39,34,
  25,46,  7,89, 53,25, 86,52, 21,85, 73,73, 26, 3, 83,73,
  13,11, 13,33, 48,36, 53,28, 68, 5, 80,54, 51,29, 42,40,
  21, 9, 39,13, 79, 5, 77,10, 52,44, 83,27, 34,60, 62,51,
  80,75, 55,72, 29,26, 45,10, 24,19, 16,59,  0,65, 77,84,
  17,12, 77, 2, 31,10, 70,14, 38,42, 43,73, 48,54, 10,39,
  56,53, 10,15,  3, 9,  4,85, 84,10, 72,47,  9,21, 68,62,
   5,43, 38,39, 88, 5, 54,48, 60,28, 35,86, 59,51, 47,53,
  53,16,  1,73, 22,46, 62, 2, 86,11,  7,24, 29,77, 52,50,
  53,76, 60,56, 47,68, 38,25, 55,40, 52,29, 23,65, 29,43,
  30, 7, 30,51, 56,67, 18, 4, 88,19, 39,56,  0,64, 67,28,
  55,46, 59,32, 87,46, 36, 6, 14,83, 47, 2, 34,62,  5,82,
  11,81, 11,83, 66,16, 10, 1, 36,81, 44, 5, 44,66, 28,10,
  52,47, 45,48, 82,88, 74,38, 18,14, 44, 3, 81, 5, 11,80,
  19,30, 84,50, 32,60, 30,31, 17,71, 15,65, 53, 7, 23, 0,
  63,73, 38,62, 80,14,  3,28, 31,31, 35,22, 20,66, 50, 9,
  29,28, 48,19, 26, 9, 86, 4, 20,45, 85,34,  7,44, 26, 8,
  88,29, 43,68, 70,84, 38,60, 13,67, 45, 2, 14, 8, 27,47,
   2,52, 78,59, 22,37,  3,70,  4,84, 47,34, 60, 8, 52, 0,
  54,19, 18,62,  1,11, 70,25, 42,34, 60,12, 28,23,  3,13,
   8,50, 46,36, 29,35, 31,12, 77,15, 21,12, 63,51, 64,70,
  31,78, 55,36,  8,70, 54,68, 10,82, 24,52, 69,69, 45,72,
  36, 3, 64,44,  5,44, 11,65, 69,49, 52, 9, 11,63, 49, 0,
   3,72, 59,72, 21, 6, 77,78, 85,54, 77,52, 84,69, 52,35,
  31,61, 17,16, 82,47, 32,15, 75,50, 84,16, 76, 5, 29,58,
  51,85, 70,12, 59,21, 62,86, 39,19, 20,16, 59,56, 50,52,
  39,15, 79,72, 63,17, 17,89, 75,55, 61,71, 32,89, 85, 8,
  25,16, 20,20, 71,65, 72,70, 75,68, 30,44, 55,23, 84,26,
  81,84, 53,45, 88,88, 63,41, 66,76, 79,55, 35,71, 81,20,
  43,28, 61,69, 84,83, 88,69,  6,59, 20,52,  4, 9, 63,65,
  25, 2, 10,73, 44,43, 39, 0, 40,28, 32,19,  1,68, 47,24,
  58,39, 19,54, 74,85, 16,61, 69,63, 53,67, 12,35, 25,82,
  38,69, 22,44, 21,89, 40,73, 50,23, 52,84,  6,37,  6, 9,
  55,73, 49,68, 52,32, 82,43,  8,22, 88,89, 57,48, 89,68,
  63,79,  7, 6, 69,64, 37,40, 17,87, 77,47, 75,59,  5,67,
  89,51, 40,66, 69, 2, 36,44, 43, 0, 40,58, 29, 6, 22,59,
  25,77, 26,41, 11,41,  1,57, 41, 7, 56,39, 34, 3, 34,19,
  14,26, 76, 1, 67,10, 24,57, 46,38, 27,78, 42,81, 70,83,
  27,72, 40,80, 46,41, 78,69, 60, 7, 57, 4, 20, 1, 23,59,
  11, 7, 56,19, 76,56, 88,75, 48,86, 71,60, 20,44, 21,26,
  67,72, 37,85, 83,71, 81,56, 55,14, 89,72, 35,38, 89,19,
  88,83, 59,68, 84,17, 72,21, 47,89, 29,79,  3,36, 74, 2,
  55,89, 19,62, 24,12, 71,21, 57, 3, 24,32, 34,83, 39,14,
  82,60, 78,58, 83,78,  4,50, 81,55, 69, 5, 12,32, 61,36,
  12,86, 28,17, 39,54, 29,74, 81,87,  2,12, 29, 8,  1,69,
  34,73, 84,25,  0,77, 22,73,  0,46, 69,89, 58,82, 66,55,
   9,89, 30,37, 75,52, 43,17, 47,79, 89,30, 77,35, 14,34,
  25,62, 24,18, 59,63, 54,88, 18,63, 41,12, 21,81,  8, 8,
  36,25, 41,43,  1,40, 14,38, 53,87, 77,20, 41,22, 37, 7,
  27, 1, 39,86,  8,37, 69,56, 48,68, 58,23, 23,74, 36, 5,
  68,33, 67,73, 28,37, 35,65, 81,58, 46,70,  9, 4, 18,37,
  34,84, 32, 5, 65,51, 21,69,  6,63, 48,43, 69,40, 61,20,
  61,32, 11,13, 17,67, 64,40, 38,58, 68,61, 18, 0, 11,38,
  87,29, 13,85, 60,54,  0,18, 15,62, 40,12,  4,17,  5,19,
   0, 6, 31,74, 63,11, 75,66, 18,33, 42,67, 29, 3, 43,71,
  19,42, 17,79, 21,18, 86,49, 82,28, 40,45, 80,13, 61, 2,
  32,35, 84,36, 47,28, 28,74, 81,18, 77,22,  3, 6, 70,41,
  64,84, 13,27, 45,11, 47,87, 55,80, 33, 7, 67,15, 79,18,
  70,64, 41,45,  5,71, 31,50, 83, 6, 55,19, 51,81, 33,12,
  36,62, 61,80, 36,59,  2, 8, 23, 8, 58,72, 55,55, 42,43,
  35,31, 77,42, 17,68, 28,84, 48,83, 30,16,  7,38, 78,80,
  15,56, 88,86, 43,59, 13,37, 33,76,  0,70, 46,84, 52,39,
  57,67, 16,10, 25,57, 87,74, 29,27, 49,64, 23,38, 22,60,
  68,89, 77,45, 87, 3, 71,50, 64,79, 68,14, 45,58, 64, 3,
  43,10, 47,58, 31,73, 35,27, 30,35, 57,23, 47,72, 63,48,
  54,73,  1,34, 33,74, 41,44, 22,41, 72,34, 78,13, 88,55,
  28,27, 18, 5,  3,19, 55,66, 60,21,  9,35,  6,12, 51,62,
  36,29,  0,14, 77,37, 70,53, 13, 2, 54,29,  0, 4, 49,72,
  19,34, 81,89, 54,75, 77,53, 63,89, 16,12, 54,55,  9,62,
   3,18, 69,29, 16,22, 46,30, 50, 4, 20,74, 42,60, 47,71,
  40,48, 59,28, 81,25,  8,80, 30,43, 77,23, 77,44, 67,18,
  29, 2, 27,39, 16,23, 33,15, 11,62, 67,48, 86,68, 69,71,
  75,57, 86,80, 64,34, 85,19, 67,40, 19,39, 66,24, 19,46,
  43, 8, 81,23, 71,66, 73, 4, 80,26, 73,34, 49,19, 14,80,
  12,69, 49,46, 40,41, 66,71, 46,83, 28,72,  6,29, 70,79,
  29,63, 13,56, 12,68, 12,14,  9,63, 26,16, 46,39, 48,52,
  18,11, 74,15, 77,80, 85,82,  1,22, 74,58, 51,49, 73,15,
  65,42, 23,78, 69,82, 12,84, 72,73, 59,67, 80,19, 49,57,
  16,37, 86,12, 53,13, 31,43, 70,37, 39,63, 43,33, 61,57,
  50,22, 52,21, 64,88, 41,50, 62,45, 33,33, 38,78, 29,42,
  78,65, 57,73, 77, 1, 26,66, 28,61,  3,76, 36,16,  7,58,
  87,51,  1,49, 51,77, 59,36, 28, 5, 54, 2, 78,46, 27,77,
  53,43, 24,28, 29,38,  0, 8, 68,84, 78,47, 61,88, 69,83,
  13,83, 83,88, 61,13, 68,73, 55,63, 62,79, 30,63, 13,45,
  11,28, 45,82, 67,64, 75,58, 42,85, 50,78, 63,70, 65, 5,
  14,47, 55,79, 11,86,  0,82, 39,57, 29, 9, 31,42, 87,81,
   2,88, 88,16,  1,25, 76,35, 78,52, 21,51, 50,10, 23,18,
  86,57, 64,61, 78, 1,  6,31, 69,66,  4,37, 64,51, 39,65,
  88, 1, 60,18, 28,86, 29,69, 34,80, 17, 7, 49,26, 36,89,
  66,37,  9, 8, 13,81, 64,82, 28,44, 21,52, 15,57, 32,74,
  87,15, 44,15, 29, 1, 23,13,  4,66, 68,30, 68,80, 24,13,
  30,23, 83,33, 54,13, 62,64,  2,60, 42,76, 66, 8, 64,72,
  22,69, 66,33, 46,76, 79,54, 84,39, 51, 5, 63,86, 70,34,
  39,36, 33,13, 35,14, 87,75, 28,32, 47,42, 11,73,  0,73,
  62,22,  9,50, 15,83, 18,12, 17,65, 75,25, 59,41,  4,12,
   6,20, 19,13, 22,51, 28,57, 21,79, 13, 9, 35,73, 32, 0,
  86,82,  6,60,  5, 5, 26,85, 78,19, 26,71, 84,57, 27,88,
  26,10, 15,43,  9,88, 60,29,  9,16, 35,57, 17,61,  9,68,
  46,47, 58,27, 20, 6, 52,61, 85,89, 37,59, 49, 4, 33,69,
  83,57, 33, 5, 26,21,  3,80, 74,62, 56,32, 87,76, 56,64,
  31,26, 52, 8, 15,17, 37,27, 77,85, 83,53, 28,40, 32,30,
  68,28, 76,71, 45,59,  1,37,  7,84, 25,72, 58,51, 56,37,
  11,79, 37,16, 74,37, 53,84, 70,86, 11,18, 14,25, 11,74,
  13,66, 42,86,  7, 4, 41,70, 81,38, 87,40, 29,21, 30,52,
  69, 6, 36,23, 44,39, 14,52, 88,79,  6,79, 54,42, 50,65,
  11,47, 43,52, 51,63, 12,67, 37,14,  6,68, 27,80, 60,89,
  22,30, 64,46, 77,76, 56, 7, 38,53, 12,44, 29, 7, 19,12,
  46,80,  0,38, 20,82, 44,13,  4,70, 82,85, 77, 0, 28,31,
  76,34,  3,29, 74,41,  4,36, 51, 1, 87,31, 30,33,  7,63,
   6,58, 21, 2, 22,18,  3,43,  7,16, 10,19, 72,81, 60,70,
  89,32, 59,83, 62,81, 21, 1, 76,60, 48,16,  1,20, 49,25,
  49,10, 74,39, 11, 3, 23,23, 74,23, 44, 4, 47, 8, 38,35,
  39,22, 29,25, 87, 7, 56,47, 58,31, 70, 9, 75,76, 50,34,
  44,82, 32,71, 26,73, 11,19, 44,63, 67,35, 15,24, 36,56,
  76,18, 11,40,  7, 7, 72,89, 83, 1, 29,30, 29,81, 69,17,
  12,51,  8,42, 54,31, 88,56, 80,41, 31, 2, 61,67, 14,28,
  34,16, 87,89, 30,65, 72,49, 59,20,  0,43, 54,49, 78,24,
  53,82, 60,88, 75,33,  3,54, 28,18, 28, 4, 78, 8, 45,67,
  82,56, 41,74
};
const int p_90_8[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1439,
  /// Array of preferences
  87,64,  8,70, 13,16, 26,33, 42, 4, 85,46, 24,50,  1,87,
   3,32, 14,17, 84,23, 34,30, 80,55, 45,25, 39,67, 61,40,
  69,38, 12, 0, 40,30, 54,59,  6, 2, 66,73, 65,20, 19,85,
  48, 6, 10,42, 44,51, 31,64, 18,28,  4,55, 29,26, 67,52,
  50,21, 18,14, 24,46, 28,55, 17,87, 59,43, 53,60, 49,52,
  47,40, 88,63, 23,44, 44,13, 89,67, 51,13, 49,46, 15,83,
  68, 3, 41,86, 66,86, 22,51, 37,85, 58,35, 47,52, 51, 0,
  10,48, 84,59, 63,49, 74,84, 69,46, 74,80, 45,31, 60,56,
  82,34, 78,22, 27,23, 66, 0, 43,52, 19, 3, 36,28, 59,19,
  77,36, 51,24,  7,28, 58,68, 71,81,  4,64, 63,35, 38,79,
  20,42, 40,35, 59,66, 71, 5, 74,82, 24,10, 59,86, 19, 4,
  78,10, 32,35, 53,21, 39,87,  3,49, 13,19, 19,42,  1,69,
  56,29, 62,33, 50, 8, 80,48, 67,31, 78, 6, 32,23, 56,47,
  74,59,  2, 3, 47,46, 47, 5, 53,74, 81,33,  7,58, 41,61,
  59,69,  9,50,  4,89, 19,22,  9,74, 84, 0, 51, 6, 30, 9,
  19,36, 76,27, 20,21, 64,48, 85,52,  4, 6, 72,44, 57,41,
  66,74, 55, 9, 70,84, 61,86, 87,72, 20,23, 88,66, 70, 7,
  29,12, 77,88, 16,17, 56,50, 20,89, 43,11, 88,41, 74, 4,
  39,88, 83,62, 67,29, 84,63, 68,22, 45,65, 13,89, 86,51,
  46,80, 13,56, 27,44, 89,27, 47,55, 17,86, 70,82, 85, 2,
  67,27, 14,82, 21,48, 69,59, 34,14,  5,21, 77,62, 73,78,
   8,54, 78,28, 42,27,  7,25, 68,24, 25,61, 36,18,  6,38,
   1,66, 42,50, 66,89, 60,84, 51, 9, 82, 4, 79,38, 43,36,
  33,67, 13,64, 65,69, 48,77,  1,17,  2,13, 73,88, 15,24,
  58,25, 34,27, 68,77, 31, 9, 57,26,  3,17, 23, 8, 79,25,
  36,50, 52,71, 56,24, 11,20, 44,28, 71,67, 39,16, 88,65,
  24,47, 62, 4, 55,67, 54,84, 43, 0, 27,27, 58,52, 81,68,
  23,38, 77,40, 37,68, 65,47, 83,16, 61,11, 56,65, 10, 8,
   3,82, 35,52, 30,87, 11,24,  1,72, 41, 8, 67,51, 25, 9,
  17,54, 83, 1, 27, 8, 46,38, 39, 6, 10,52, 70,52, 62,13,
  38,45, 31,36, 28,30, 78,21, 78,15, 69,57, 28,34, 55,63,
  11, 9, 37,16, 34,52, 21,51, 22,11, 20,58,  0,73, 23,84,
  24,32, 32,39, 54,31, 14,24, 16,43, 11,32, 11,57, 14,16,
  77,68, 14,67, 24,40, 49,49, 52,18, 67,23, 65,21,  4,73,
  24,51, 80,24, 48,38, 64,38, 32,82, 78,11, 41,23, 60,80,
  11,72, 49,83, 68,33, 31,57, 56,18, 52,82, 46,72,  3, 2,
  44,14, 22,22, 87,49, 44,39, 44,86, 23, 1, 51,39,  9,15,
  56,52, 12,33, 15,33,  1,63, 53, 2, 50,62, 87,13, 47,78,
  25,78, 10,66, 80,31, 74,36, 21,39, 61,36, 52,36, 41,63,
  48, 2, 11,52, 59,18, 70,26,  9,28, 49,45, 71,30, 64,72,
  14,18, 51,29, 46,36,  0,58, 45,53, 45, 4, 29,59, 40,34,
   2,80, 69,10, 47,19,  9,64, 85,31, 18,66, 82,66, 53, 5,
  30,15, 85,16, 69,22, 49, 9, 31,65, 43,38,  7,23,  5, 8,
  58,66, 37,87, 11,61, 10,29, 12,80, 41,36, 39, 2, 39,21,
  71,55, 42,49, 69,82, 18,80, 41, 3, 27,37,  6, 9, 48,69,
  21,34, 61,20, 18,24, 86,29, 49,63, 25,64, 68,17, 57,40,
  54,23, 71,72, 76,76, 49,36,  0,43, 41,77, 53,42, 36,66,
  46,88, 31,73, 55,57, 57,30, 35,73, 14,20,  7,46, 27,57,
  20,51, 22,32,  2,51, 52,35, 64,50, 12,73, 75,37, 20,38,
   6,65, 19,31, 17,23, 80, 5, 74,89, 45,80, 86,59, 54,67,
  14,79, 37,39, 53,56, 83,69, 24,54, 29,49, 72,83, 88,23,
  51,16, 68, 0, 26,54, 19,78, 18,25, 66,60, 71,29, 36,72,
  54,43, 43,18, 22,23, 69,73, 27,52, 23,13, 86, 8,  1,70,
  73,52, 24,25, 14,71, 50,82, 10,36, 50,13, 70,56, 55,62,
  37, 1, 45,64, 85,56, 83,53, 82,31, 86,68, 32,40, 71,11,
  32,62, 60,31, 86,45, 57, 2, 16, 8, 44,21, 35,48, 29,77,
  84,31, 69,34, 85,47, 70,12, 18,11,  2,17,  3,43, 79,64,
  70,70, 79,27,  4,26, 10,21, 85,17, 48,30, 22,40, 86,27,
  57,10, 28,76, 71,23, 11,69, 39,62, 59,87, 85,36, 72,80,
   5,72, 25,70, 52,62, 78,77, 27,64, 47,59, 26,47, 45,57,
   7, 1, 70,23, 74, 2, 88,33, 63, 9, 30,20, 55,18, 25,71,
   2,58, 63,48, 66,68, 67,32,  9,58, 42, 6, 15,36,  5,84,
  49,81, 67,48, 54,56,  3,22, 33,28, 43,82, 12,32, 20,69,
  25,11, 12,36, 89,80, 67,86, 22,12, 57,46, 46, 9,  1,78,
  50,32, 44,58, 33,56, 42,16, 83, 9, 26,37, 31,14, 46,48,
  65,89, 37,18, 22,86, 28,11, 86,83, 35,22, 39,68, 39,76,
  12,65, 41,82, 59,89, 68, 2, 87,55, 44,59, 17, 5, 14,31,
  42,38, 48,85, 46,52, 16,45, 14, 2, 36,31, 28,31, 53,85,
  72,52,  7,60, 65,46, 36,24, 11,13, 73,12, 66,10, 17,16,
  26,61, 53,58, 86,66, 36,55, 73,37, 50,24, 69,52, 75,29,
   7,77,  7,15, 80,51, 68,20, 51,71, 49,57, 66,41, 57, 9,
   9,63, 73,11,  4,85, 62,75, 61,55, 56,12, 47,76, 40,55,
  10,70, 12,55, 44,56, 44, 2, 70,10, 58,28, 18,31, 31,40,
  83,80, 58,64, 81,88, 32,65, 21,80, 49,56, 21,85, 23,50,
  23,55, 33,70, 30,37, 50,76, 79,28,  2, 4,  3,53,  8,89,
  67,72, 48,89,  5,55, 57,20, 24,18, 49,24, 32,52, 59,29,
  20,56, 29,14, 18,62, 65, 1, 81,73, 20,87, 34,35, 39,50,
  18,47, 27,10, 57,67, 44,69, 80, 3, 16,78, 57,48, 65,35,
   1,76, 78,84, 35,37, 16,76, 61,84, 32,53, 65,78,  3, 0,
  77,47, 81,66, 50,35,  3,46, 75,25, 44,20, 25,88, 27, 9,
  84,38, 21,17, 70, 3, 66,15, 21,16, 88,15, 16,36, 15,73,
  58, 7, 32,10, 37,61, 58,82, 12,68, 65,40, 39,13, 59, 7,
  32,57, 46,70, 51,40, 85,72, 56,74,  5,42, 70,63, 20,44,
  22,63, 42,18, 53,47, 29, 3, 63,34, 60,60, 15,51, 83,47,
  21,36, 82,89, 89,85, 68,79, 49,68,  8,45, 57,82,  6,69,
  80,33, 32,67, 86,82, 34,53, 39,52, 16,79, 72,46, 59,52,
  66,40,  6,32, 19,75, 27,13, 37,14, 24,74, 53,37, 39,59,
  76,32, 88, 9, 50,85, 19,15, 74,73, 58,20, 63,74, 35, 2,
  81,20, 32,81, 84,75, 74,67,  1,44, 84,15, 86,79, 41,70,
  53,86,  5,50,  7,87, 75, 7, 81,43,  8,84, 66,31, 85,34,
  10,15, 57,12, 66,82, 23,28, 17,65, 79,17, 28,54, 39,75,
  62,39, 19,54, 14,27, 54,51, 62,60, 41,40, 48,29, 30, 8,
  84,37, 65,67, 10,78, 28,60, 58,12, 86,56, 37,67, 31,53,
   9,46,  2,31, 66,64, 52, 3, 22,75, 80,38, 15,85, 22,46,
  78,41, 54, 0, 12,86, 61,67, 45,54, 82,69, 26,88, 54, 5,
   5,86, 87, 2, 19,45, 46,64, 64,89, 31,85, 34,12, 71, 1,
  69,86, 12,75, 37,72, 23,41, 67,36, 33,39, 73,66, 42,70,
  52,21, 53,80, 19, 8, 89, 9, 73,55, 13,12, 36,69,  6,21,
  50,31, 51,20, 20,46, 60,74,  1,61, 35,10, 21,86, 31,72,
  70,35, 62,29, 20,32, 26,24, 67,66, 52,84, 71,83,  9,41,
  70,81, 38,40, 17,73, 86,77, 40,73,  6,29, 61,32, 36,75,
  77,42, 28,85, 88,86, 30,22, 79,51,  7,37, 49,19, 70,46,
  64,49, 78,72, 17, 7, 81,46, 71,49, 31,13, 58,73, 27,89,
  51, 7, 75, 0,  8,37,  1,29, 31,31, 35,64, 29,56, 82,23,
  87,50, 43,28,  0,23, 35,18, 15,56, 38,57, 51,52, 44,77,
   5,44, 13,24, 47,41,  3,57, 47,43,  7,79, 21,45,  3,85,
  81,31, 36,45, 28,43, 86,34, 52,19, 67,85, 22,15, 24,73,
  49,54, 42,56, 32,69, 28,89, 39,86, 34,34, 64,47, 69,29,
   7,74, 88,22, 27,38, 83,56, 71,48, 53,32, 19,52, 50,19,
  13,14, 11,51, 43,80, 60,79, 52,80, 26,31,  0,29, 43,16,
   7,27, 12,31, 27, 2, 76,16, 48,46, 46, 4, 30,61,  3,33,
  56,44, 62,40, 28,48, 64,29, 59,57, 76,45, 68,78, 16,13,
  75,47, 49,58, 82,39, 15,21, 62,28, 86,33, 34,76, 42,10,
  21,19,  3,59, 30, 1,  9,20, 80,77, 42,67, 17,29, 35,45,
  13,82, 64,37, 29,45, 80, 4, 41,64, 56,87, 21, 2, 63,61,
  49,12, 15,23, 40,13, 59,26, 71,70,  5,24, 87,78, 19,23,
  20,75, 76,28, 76,38, 65, 3, 45,43, 82,47,  3,36, 87,74,
  82,78, 57,37, 10,88, 39,26, 14,86, 45,39, 27,79, 11, 5,
  28,46, 49,14, 70,73, 38,82, 41,81, 69,15, 29,10, 60,21,
   8,42, 32,32, 34, 6, 16,44, 13,65, 74, 3, 28,75,  9,68,
   0,27, 57,78, 13,30, 44,34, 71,80, 12,37, 83,46, 66,65,
  63,14, 29, 4, 13,80, 53,28, 58,21, 56,48, 22,25, 16,32,
  36,20, 10,34, 71,47,  3,86, 37,26, 22,81, 53, 7, 69,62,
  81,19, 32,47, 50,33, 48, 9,  3,89, 16,68, 23,73, 77,19,
  61,38, 51,82, 87,18, 72,81, 25,65, 88,73, 71,65, 37,33,
  71,68,  6,46, 29,34, 57,83, 11,67, 72,88,  9,23,  0,65,
  80,57, 22,52,  6,74,  7,20, 24,83, 86,85, 41,89, 53, 0,
  75,31, 16,84, 13,69, 55,35, 45,51, 52,28, 63, 8, 25,75,
   4, 2, 11,56,  0,40, 87,34, 44,36, 71, 7, 34,25, 87,81,
  84,22, 79,74, 10,83, 47,61, 34, 1,  3,64, 38,54, 36,64,
  20,24, 56,61, 61,57, 40,47, 54,49, 39,55, 27,33, 73,18,
   6,81, 50,11, 59, 3, 36,52, 42, 0, 10,85, 28,70, 50,55,
  45,77, 68,31, 25,22, 73,54,  6, 0, 50,41, 70,32, 45,22,
  85,50, 57,11, 61,66, 46,29,  7,88,  3,48, 45,47, 48,14,
  14, 3, 54,32, 80,79, 11, 6, 33,59, 41,76, 89,73, 85,35,
  19,35, 29, 9, 76,33, 42,35,  1,37, 50,54, 82,42,  3,87,
  66,25,  4,34, 86,81, 64,45, 89,26,  1,84, 58,41, 11,48,
  12,51, 32,64, 65,71, 85, 9, 17,41, 29, 1, 29,55, 43,24,
  30,48, 60,33, 72,51, 36,43,  5, 6, 40,75, 89,65, 76,85,
  22,41, 72,16, 80,16, 12,72, 31,18, 59,33, 89,10, 83,77,
  24,62, 66,69, 53, 6, 69, 4, 34,83, 47,82, 18,12,  2,29,
  65,74, 29,13, 22,61, 66,20, 54,61, 36,33, 80,52, 33,57,
  87,71, 28,39, 51,19, 29,24, 37,13,  4,68, 65,13,  6,58,
  34,66, 85, 8, 87,53, 42,15, 74,40, 33,22, 86,52, 46,32,
  75,62, 85,59, 63,56, 20,49,  0,66, 70,72, 47,10,  9,51,
  24,37, 69,41, 23,87,  4,74, 73,60, 42,88, 87,63, 40,83,
  70,64, 18,64, 10,68, 48,74, 38, 2, 89,83, 38,18, 65,19,
  12, 3, 50, 3, 15,27, 56,72, 84,82, 64,33, 63,58, 48, 5,
  45,60, 20,27, 23,72, 81,24,  2,56, 30,45, 66,67, 48,20,
   9,32, 38,89, 62,14, 22,64, 41,52, 21,53, 55, 2, 40,78,
  27,86, 47, 6, 33,18, 11,73, 77, 0, 54,66, 16,18, 89,43,
  18,74, 27,53, 62,12,  1,47, 49,10, 21,20, 11,30, 75,74,
  88, 3, 72, 6, 43,23, 48, 1, 53,16, 72,14, 44, 9,  7,84,
  37,30, 65,54, 76,75, 58, 9, 83,12, 68,23, 33,55, 45,63,
  87,26,  8,82, 15,15, 27,69, 55,43, 40,81, 76,62, 40,42,
  25,60, 28, 3, 26, 5, 27,35, 74,72, 51,43, 64,59, 37,36,
  22,78,  9,69, 34,23, 33,48, 32,12,  9,11, 83,32, 10,30,
  42,53, 50,50, 15,60, 43,75, 48,62, 85,62,  3,88,  3,19,
  89,87, 44,22, 28,28, 73,85, 54,36, 38, 7, 56,51, 21,59,
  58,26,  6,73,  2,73, 72, 9, 35,44, 58,32, 80,62,  7,63,
  75,68, 48,19, 73,23, 26,76, 89,30,  4,71, 65,70, 77,20,
  11,41, 88,71, 54,19, 76,24, 35,87, 38,31, 80,29, 14,14,
   6,67, 76,66, 38,26, 82,83, 80,61, 39,39, 79,36, 69,71,
  68,18, 63,59, 64,36, 73,53, 54,10, 24,31, 23,75, 69,79,
  68,59, 15,22, 39,73, 29,42, 47,25, 11,65, 15,88, 66,87,
  46,54, 83,64,  1,19, 78, 1, 21,62, 29,46, 21,28,  4,19,
   2,14, 13,36, 51,74, 52,65, 63,86, 48,60, 39,46, 11,33,
  13,76, 86,60, 50,47, 24,16, 44, 8, 52,60, 68,21, 27,87,
  55,53, 46,85, 26,72,  3,50, 33, 4, 75,64, 75, 3, 64,21,
  47,16,  7,42, 83,30, 28,18, 77,58, 40,36, 59,11, 61,58,
  56,25, 51, 4, 71,44,  3,26, 36,58, 23,69, 33,16, 25,37,
  17,71,  7,72, 10,45, 41,49, 72,70,  7,53, 44,43, 10,20,
  35,75, 84,30, 80,20, 16,47, 86,40, 71,20, 26,16, 50,70,
  44,37, 34,13, 63, 5,  1, 8, 69,72, 38,74, 55, 5, 60,68,
  84,29, 12,54, 56,53, 35,31, 38, 9, 79,44, 38,64, 11,46,
  53,30, 37,17, 84,53, 47,65, 20, 3, 86,30,  3,21, 16,40,
  58,15, 34,79, 42,21, 53, 1, 62,43, 69, 2, 40,57, 67,61,
  52,47, 77,65, 89,16, 52,15, 48,61,  2,26, 77,53, 34,58,
  72,53, 38,72, 61,22, 64,12, 80, 9, 62,22, 47,11,  6, 3,
  50,87, 58,69, 76,63, 62,49, 21,33, 11,89, 54,20
};
const int p_90_9[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1138,
  /// Array of preferences
  59,15, 71,30, 51,66, 77,63, 10,11, 13,15, 88,28,  1,47,
  56,19, 18,25, 88,54, 14,50, 11,13, 51,76, 57, 7, 75,46,
  89,13, 58,47, 49,60, 66,71, 20,73, 74,35, 70,82, 11,46,
  69,68, 72,82, 83,30, 82,19, 50,59, 88,87, 48,41, 35,65,
   2,65, 83,89, 14,48, 12,53, 39,29, 38,78, 49,41, 56,55,
  38,66, 55,68,  4, 5, 36,11, 28,54, 44,37, 62,76, 85,80,
  64,81, 42,30,  1,73, 41,17, 21,61, 28,70, 22, 7,  9,71,
  88,23, 69, 8, 82,38, 27,19, 20,38, 41,33, 38,26, 54,73,
  28,20,  6,33, 58,51, 21,68, 11,14,  1,76, 49,82, 76,89,
  28,65,  3,60, 86,49, 10,41, 44,40, 35,21, 81, 8, 58,79,
  11,57, 85,25, 43,11, 55,43, 50,75, 78,37, 19,45, 35,10,
  70,68, 28,77, 49,88,  6,45, 15,46, 73,76,  2,31, 38,10,
  82,81, 42,89,  7,22, 51,38, 65,88,  5,43, 89,37, 80,28,
  24,84, 30,43, 17,52,  4,83,  8,49, 69,64, 82,14, 65,45,
  53,71, 21,19, 30,72, 47,73, 43,68, 19,33, 22,25, 73,35,
  62,14, 26,22, 28,32, 33,86, 85,61, 28,83, 66,11,  8,61,
  70,67, 61,64, 12,23, 32,30, 50,64, 67,35, 16,50,  9,38,
  44,32,  3,29,  9,22,  4,16, 25,40, 56, 0, 56,76, 17,15,
  17,19, 29,84, 24,60, 17,35, 41,23, 29,88, 82,67, 36,86,
  61,52, 42,86, 77, 6, 59,59, 77,56, 72,57,  8,67,  6,25,
  82,85, 40,77, 78, 3, 33,73, 46,42, 24,83, 89,57, 60,59,
  70,61, 84,31, 74,46, 83,19,  6,78, 42, 8, 81, 2,  5,17,
  61,14, 83,34, 29,19,  0,15, 46,13, 75,16, 37, 9, 52,25,
  84,66, 35,86, 60,66, 55,12, 42,71, 38,30, 42,29, 81,62,
  48,44, 37,45, 39,87, 16,87, 80,15,  3,76, 80,18, 89,38,
  51,82, 18,17, 15,64, 64,72, 58,70,  9,12, 35,28, 12,66,
  81,43, 37,34, 53,62, 21,64,  0,22, 43,63, 59,39, 34,12,
  80,47, 71, 5, 73,30, 82,18, 26,66, 61,78,  9,15,  5,53,
  15,10, 61,17, 50,13, 25,83, 48,19, 44,42, 76,82, 77,42,
  38,35, 50,85, 26,89,  6,89,  0,30, 13,13, 37,51,  3,59,
  63,23, 26,62, 61,74, 31,18, 13, 0,  7,13, 78,20, 34,20,
  40,32,  0,77, 34,61, 55,67, 17,25, 43,47, 35,22, 69,75,
  14,30,  3,12, 40, 7, 64,86,  6,55, 88,43, 60,44, 19,39,
  77,16, 84,48, 32,57, 82, 8, 78,49, 20,42, 63,53, 47,17,
   0,71, 78,53, 10,89, 30, 8, 81,80, 33,29, 73,55,  8,65,
  11,48, 20,50, 88, 4, 41, 4, 78,51, 74,81, 54,75, 67,14,
  39,35, 84,76, 41,37, 61,51, 16, 0, 50,41, 52,23, 15,65,
  44,78, 54,67, 62,56,  0,18,  3,68, 71,58, 68, 2, 19,27,
   0,79, 50,72, 73,58, 74,61,  2,66, 37,32, 30,37, 36,62,
  80,61, 80,55,  2,69, 52,30, 60,60,  2, 5, 89,44,  5,89,
  43,19, 34,13, 38,22, 35,66, 25,44, 52,67, 55,51, 48,67,
  56,26, 42, 9, 27,41, 33,67, 63,47, 28,17, 62, 3, 73,56,
  67,89, 36,75, 37,83, 56, 4, 69,37, 35,31, 73,27, 81,26,
  30, 9, 13,27, 57,35, 17,68, 28,47, 87,41, 39,52, 60,13,
  10, 0, 60,77, 79,84, 66, 0, 77,80, 64,26, 58,67,  8,42,
  24,33, 84,42, 80,37, 35,18, 11, 3, 63,22, 63,71, 15,81,
  68,43, 48,40,  5,14, 66, 3, 59,80,  8, 7, 28,79, 81,85,
  33,68, 20,18, 30,61, 78,12, 30,51, 11,29, 66,49, 22,66,
  48,46, 32, 8, 10,52, 81,58,  0, 7, 47,46, 63,52, 23, 2,
  36,36, 29,15,  3,40, 80,41, 87,62, 50,24, 42,49,  9,13,
  31,32, 55,37,  2, 0, 68,12, 89,27, 64,46, 52,85, 81,77,
  39,11, 55,83, 47,81, 45,40,  3,85, 12,57,  7, 5, 55,73,
  56,45, 53,86, 20,83, 62,37, 34,38, 71,69, 30,39,  9,20,
  21, 2, 86, 1, 69,60, 42,64, 33,32, 36,21,  8,89, 45,21,
  28,85, 50, 9, 53,14, 27,42, 66,20, 46,26, 23,58, 50,23,
  79,17, 61,47, 71,10, 62,67, 32,19,  3,67, 16,25,  6,54,
  50,14, 33,53, 33, 8, 54,55, 72,17,  4,32, 18,33, 65,40,
  84,71, 26,65, 57,65, 80, 6, 19,31, 68,84, 81,32, 41,30,
  41,81, 21,10, 82,69, 67,30, 23,60, 88,14,  6,26, 31,28,
  43, 1, 24,41, 19,80, 51,28, 82,61, 82,37, 39, 8, 18,19,
  26,64, 11, 4, 49,31, 67,21, 54,42, 26,39, 13,26, 53,76,
  62,77, 42, 1, 72,60, 39,26, 38,87, 78,55, 60,76,  9,54,
  74,71,  1,33,  8,81, 41,19, 13,34, 12,59, 17,47, 89,59,
  87,65, 28,21, 43,55, 85,31, 16,33, 61,56, 88,75, 22,75,
  20,24, 81,70, 29,18, 55,56, 83,61, 48,14, 47,21, 71,41,
   8,60,  3,39,  9, 3, 74, 3, 74,84,  7,16, 55, 6, 70, 1,
  43, 2, 40,26, 44,67, 66, 4, 15,86, 89,69, 81, 4, 82,28,
  40,78, 78,75,  1, 3, 59,47,  1,43, 67,37, 60,74, 15, 9,
  29,36, 62, 0, 16,22, 80,39, 78,46, 82, 3, 54,60,  4,15,
  72,40, 15,25, 26,50, 37,56, 12,46,  7,27,  5,88, 75,24,
  38,44,  5,20, 76,35, 46,87, 65,51, 75, 6, 45, 6, 89,86,
  75,66, 71,52, 44,68, 35,19, 59,71, 58,69,  6,42, 59,28,
  66,51, 81,74, 14,10, 29,79, 82, 7, 14, 5,  4,89, 45,33,
  34,55,  7,17, 76,76,  4,28, 72,31, 86,44, 55,32, 38,88,
  19,87, 11,52, 38,76, 35,72,  8,63, 45, 2, 80,17,  7,33,
  21,82, 14,68, 77,21, 75,63, 24,61, 10,63, 35, 6, 25,82,
  35,35, 51,50, 80,51, 23,75, 72,53, 65,48,  0,10, 13,43,
  42,85, 24,42, 37,87, 44,46, 43, 9, 79, 7, 54,28, 33,69,
   9,66, 42,40, 17,53,  2,37, 50,49, 62,53, 33, 9, 86,20,
  84,38, 15,45, 18,41, 56,23, 12,43, 74,32, 45,87, 70,53,
  88,88, 83,32, 83, 5, 58,68, 68,13, 44,77, 52,52, 64,56,
   8,62, 14, 2, 62,69,  1,35, 17,73, 35, 0, 55,45, 16,18,
  31,60,  5,12, 28,46, 41,45, 46,71,  0,83, 52,60, 52,41,
  69,66, 22,41,  1,84, 80, 4, 79,42,  5,70, 30,18, 34,60,
  62, 2, 84,35, 54,71, 76,23, 70,34, 31,50, 57,51, 67,33,
  83,42,  4, 9, 32,67,  6,70,  9,23, 22,29, 48,35, 42,25,
   8,13, 72,10, 76,27,  9,81, 86,83, 40,72, 45,32, 49, 2,
  19,13, 40,42, 30,29, 85,19, 83,36, 26,16, 67,78, 67,15,
  69,31, 40,37, 25,39, 27,81, 39, 7, 63,26, 71, 1, 56,62,
  67,27,  9,33, 24, 5, 89, 3, 31,88, 32,59, 36,19, 54,57,
  29,38, 83,54, 47,85, 75,79,  3,23, 87,31, 46,53, 81,61,
  49,14, 10,75, 71, 3, 77,76, 20,44, 65,32, 39,65, 52,74,
  29,55, 64,45, 85,58, 18,71, 46,74, 36,30, 11,21, 20,85,
  24,71, 83,83, 83,58, 76,51, 38, 7, 12,54, 27,16, 18,84,
  43,64, 42,55, 56,29, 61,61, 57, 5, 17, 4, 23,36, 19,44,
  64,12, 16,19, 84,41, 79,20, 51,63, 80,79, 10,48, 21,72,
  16,56, 61,22, 26,81, 11,51, 84, 4, 40,74, 23, 9, 39,31,
  47,52, 61,76, 57,30, 22, 3,  5, 4, 10,69, 61,39, 56,72,
  81,83, 80,52, 73,74, 40,83, 81,36, 76,37, 38,71, 73,88,
  85,53,  0,14, 73,81, 44,51, 26,75,  7,28, 28, 4, 18,82,
  81,49, 16,57, 60,27, 13, 9, 33,14, 15,38, 33,25, 25,86,
   5,11,  6, 8, 80,10, 15,17, 74,16, 69,34, 40, 1, 81,86,
  63, 5, 80,87, 52,21, 84,44, 13, 5, 28,38, 25,19, 74,72,
  57,80, 33,80, 65,33, 54,34, 76,30, 53,32, 46,70,  2,68,
  82,45,  5,60, 27,33, 54,43, 23,86, 66,52, 82, 6, 59,78,
  21,42, 62,59, 83,66, 79,68, 74,40, 24, 4, 33,13, 87,13,
  51,21, 50,46, 75,80, 48,74, 73,54, 34,72, 89,60, 15,84,
  75,12, 30, 5, 53,60, 38,41, 33,35, 65,30, 22,88, 24,10,
  89, 1,  0,46, 48, 0, 53,34, 87,54, 47,27, 74,75, 56, 6,
  41,26,  6,66,  7,42, 84,68, 49,65, 19,69, 81,63,  6,69,
  15,67, 25,75, 39,82, 31,13, 66,57, 66,47, 81,12, 76, 9,
  82,42, 31,53, 52,48, 60,14, 50,34, 36,56, 11,63, 57,43,
  89,10,  1,65, 61,31, 69,84, 89,20, 30,13, 25,53,  8,75,
  26,44, 86,61, 64, 6, 31,68, 49,71,  7,57, 49,74, 34,28,
  11,53,  6,43, 48, 9, 40,21, 11,69, 77,69, 83,64, 40,58,
  89,85, 36,69, 50,35, 66,35, 88,19, 82, 5, 42, 0, 81,48,
  26,25, 10,30, 83,67, 40,16, 35, 4, 37,28, 85,57, 42,72,
  52,11, 55, 2, 80,75, 52, 8, 59, 4, 29,81, 12,85, 56,86,
  87, 8, 87,14, 42,65, 21,88, 69,28, 16,72,  1,19, 55,65,
  72,76, 48,42, 38,49,  7, 1, 35,73, 89,30, 80,54, 72,35,
   1,51, 79,48, 10,82, 60,38, 89,83, 73,41, 74,67, 27,38,
  82,65, 43,24, 68,66, 48,22, 83,81, 68,41, 30,16, 84,40,
  85,67, 82,55,  0, 9, 40,39, 13,59, 68,17, 42,78, 32,16,
  62,45, 73,87, 87,66,  1, 8, 85, 6, 87,15,  8,19, 42,52,
  32,84, 41,88, 61,37, 83,68, 10, 2, 27,61,  7,24,  4,33,
   1,24, 35,33, 48,45, 87,26, 88, 8, 74, 4, 53,25, 34,51,
  73,34, 62,75, 73,22, 53,27, 18,18, 55, 5, 88,30,  1,64,
   2,15, 24,57, 85,73, 41,15, 47,53, 18,76, 32, 4,  6,71,
  20,86, 49,50, 38,11, 20,10, 16,31, 49, 5, 34,25, 55,33,
  71,87, 59,41, 79,58,  5, 7, 53, 5, 26,53, 82,13, 56,52,
  89,39, 27,83, 80,43,  1,79, 89, 4, 87,32, 11,27, 78,57,
  58,31, 11,76, 43, 6, 25,76, 65,19, 72,21, 10,84, 89,43,
  16, 4, 55,70, 28,10, 52,29, 52,76, 15,82,  6,56,  3,58,
  20,72, 45,35, 26,40, 67,79,  0,43, 76,65, 62,13, 32,78,
  80,84, 61,66,  3, 5, 73,69,  2,48, 14,32, 68,50, 42,80,
  17,63, 36,26, 25,54, 35,89, 76,60, 22,24, 89,29, 25,14,
  60,17, 53,81, 71,84,  8,14, 76,80, 68,73, 24,34, 13,75,
  23, 6, 48,71, 55,19, 85,10,  5,24, 66, 1, 22,37, 55,64,
  44, 2, 48,30, 73,86, 61,72, 75,22, 58,82, 58,54, 21, 5,
  77,30, 79,72,  8,39, 71,19, 28,19, 75,11, 57,12, 30,46,
   2,18, 78,62, 73,85, 83,85, 52,45, 38,68, 78, 7, 44,60,
  55,62, 39,86, 41, 2, 44,35, 52,75, 24,45, 49,18, 20,89,
  22,45, 54,54
};
const int p_90_10[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1752,
  /// Array of preferences
  36,87, 69,77, 19,30, 28,19, 40, 3, 39,81, 49,55, 88,47,
  40,56, 65, 3, 15,67, 72,73, 58,68, 28,51, 75,56, 11,30,
   3,50, 35, 9, 68, 1,  5, 2, 86,24, 68,59, 22,69, 51,19,
  81,76, 15, 5, 37,40, 45,45, 39,42, 22,72, 84,33, 52,15,
  87,59,  9,73, 47,60, 39,86, 41,72, 51,35, 13,21, 73,11,
  82,43, 37,78, 72,55, 13,14, 46,55, 44,76, 16,28, 56,29,
  12, 3, 56, 2, 52,78, 13,18, 73,85, 29,56, 58,55, 68,80,
  71,51,  3,84, 88,68,  2,37, 64, 9, 79,68,  4,83, 36,79,
   2,83, 61,59, 87,52, 77,69, 68,64,  3, 8, 29,52,  5,59,
  76, 5, 14,41, 11,19,  2,82,  5,57, 81,75, 61,67, 73,35,
  43,54, 46,49, 21,56, 12,65, 15,56,  0, 8, 17,27, 66,49,
  44,42,  6,57, 21,48, 75, 3, 16,67, 49,14, 20,88, 61,55,
  30,88,  4,77, 55,69, 64,53, 66, 8, 71,43,  9,87, 33,39,
  64,66, 60,74, 61,71, 32,20,  5,20, 28,30, 25,89, 44,68,
   1,24, 47,53, 37,63, 29,53, 35,71, 74,56, 86,44, 43,30,
  33,59, 54,82,  9,88, 69,22, 22,25, 76,59, 77, 4, 18, 6,
  15,51,  0,80, 79,24, 20,87, 31,68, 28,26, 64,76, 66,66,
  71,76, 31, 4,  8,45, 47,29, 45,81, 10,59, 45, 2,  6,36,
  55,35, 30,71, 48,46, 19, 3, 45,70, 48,72,  0,34, 76,78,
  33,20, 57,73, 44,37, 52,46, 59,46, 75,44, 67,68, 83,54,
  64,25, 24,32, 38,61, 80,72, 12,57, 82,55, 56,41, 79,40,
  64,22, 67,20, 57,10, 78,69, 37,30, 19,12, 51,43, 31,63,
  73,86, 43,66, 22,41, 85,41,  2,44, 12,86, 23,35, 49,72,
  10,43, 83,41, 18,89, 20,13, 20,17, 31,47, 15,62, 55,12,
  37,69, 24,76, 89, 1, 75,40, 43,45, 58,18, 20,58, 18,60,
  70,67, 77,18, 47,23, 27,69, 89,24, 52,23, 13,33, 78,77,
  74,18, 49,54, 82,19,  7,40, 87,30, 24,67, 52,18, 44,25,
   5,37, 33,15, 70,22, 62,46, 65,42,  2,64, 31,11, 35,24,
  66,53, 44,62, 47, 4, 61,80, 48,70, 15,11, 46,44,  5,83,
  75,81, 61,28, 80,31,  2, 6, 25,64, 31,22, 44,80, 20,23,
  24,38, 22,73, 63,51, 60,68, 68,77, 69,13, 42,30, 36,43,
  82,18, 54,59, 13,79, 67,15, 56,88, 36,34, 78,21, 46,70,
  75,42, 72, 7, 64,59, 84, 1, 12,33, 14,87, 16,59, 87,74,
  66,87,  7,65, 38,87, 37,48, 75, 1, 77,30, 58,57, 55,38,
  72,61, 75,84, 60, 5, 20,32, 44,86, 71,88, 75,31,  9,38,
  54,39, 11, 4, 17, 3, 86, 7,  8, 6, 28,25, 52,13, 52,58,
  21,20, 81,65, 27,20, 89,71, 40, 6,  2,45, 50,30, 79,67,
   8,89, 11,36, 78,66, 85,87, 16,88,  5,89, 31, 5, 29,23,
  39,45, 41,55, 22, 5, 70,24, 63,85, 28,18,  6,70, 31,41,
  22,32, 77, 5, 85, 5, 56,11, 73,58, 18,86, 27,68,  6,71,
  42,63, 72,56, 42,33,  6,37, 75,45, 65,25, 60,51, 60, 1,
   9,35, 18,17, 38,31, 24,57, 28,47, 65,41, 85,13, 41,12,
  83,16, 22,37, 14, 9, 84,76, 15, 4, 64,21, 18,73, 32,78,
  27,32, 14,51, 35,11, 25,12, 59,30, 16,36, 57,12, 48,63,
  69,36, 67,58, 52,73,  0,62, 13,68, 21,77, 16,66, 80,21,
  26,27, 23,73, 80, 3, 35,36, 71,47, 39,76, 78,75, 26,40,
  72,26,  9, 2, 71,34, 77,66, 80,54, 80, 1, 21,27, 36,44,
  13,39, 46,27,  9,53, 10, 7, 33,19, 64,14, 83,11, 45,61,
  83,82, 32, 0, 27,87, 23,67, 47,50, 46,43, 54,42, 13,45,
  11,64,  8, 5, 72, 9,  8, 4,  6,80, 76,18, 39,29, 81, 1,
  59,89, 33,79, 86, 4, 33,54, 72,19, 62,79, 46,29, 67,57,
  41,35, 20, 2,  9,83,  6,51, 55,84, 57,44, 58,66, 15, 9,
  46,13, 70,29, 16,84, 43,14,  5,79, 64,89, 69,44, 23,52,
  62,43,  5,32, 14,85, 40,71, 13,44, 57,81, 11,81, 78,38,
  87,84, 66, 9,  8,21, 67,83,  7,43, 47, 9, 63,71,  4,84,
  29,41, 76,21, 88,37, 39,32, 16,45, 52,80, 53,16, 47,89,
   2, 5, 71,84, 63,24, 83, 0, 19,24, 65,51, 55,29, 61,43,
  87,16, 37,87, 27,80,  6,38, 34,81, 48,75, 57,21, 25,17,
   5,44, 45,22, 20,30, 88,10, 58,49, 12,80, 36, 7, 30,53,
  48,24, 42,70, 80, 6, 56,72, 27,14, 66,40, 37,11, 77,57,
  41,43, 61,68,  0,61, 39,10, 62,86, 63,62, 17,38, 14,73,
  39,28, 18,64, 41,27,  4,35, 16, 8, 80,45, 34,34, 48,21,
  63,11, 22,59, 20,72, 19,26, 16,60, 25, 1, 80,30, 53,39,
  57,37, 41,65, 50,44, 56,79, 88,67, 61,37, 25,81, 80,35,
  33,35, 51,26, 65,39, 23,69, 72,44, 84, 6, 42, 4, 62,26,
  61,18, 62,76,  1,40, 61,50, 22,43,  4,28, 63, 6, 77,31,
  64,15, 17,54, 50, 5, 78,68, 78, 7,  1,70, 16,75, 20,46,
  47,19, 69,21, 81,83,  6,15, 65,44,  9,81, 23,21, 17,24,
  48,83, 54,23, 31, 1, 68,57, 16,86, 87,19, 75,60, 53, 3,
  35,79, 40,54, 18,36, 50,38,  0,29, 63,78, 48,23, 18,31,
  70,30, 11,20, 46,85, 88, 1, 47,72, 23, 3, 86,65, 75, 9,
  55,30, 89,53, 41,71, 26,13,  1,76, 43,25, 11, 9, 46,22,
  19,56, 57,55, 35,70, 46,78, 54,47, 63,59, 86,18,  7,72,
  16,50, 73,29,  3,25, 69,71, 74,13, 36,53, 29,69,  8,76,
  76,77, 38, 4, 65, 9, 55, 3, 78,74, 87,77,  7,42, 82,80,
  57,30, 51,75, 15,57, 35,52,  0,23, 71,38, 71,79, 34,21,
  23, 1, 62,65,  9,45, 27,35, 66,39, 23,26, 83,33,  7,66,
  83,88, 71,15, 28,11, 84,30, 66,29,  0,44, 16, 9, 34,33,
  40, 5, 79,38, 51,80, 48,53, 44,26,  4,50, 62,42, 40,65,
  36,81, 36,11, 33,80, 47,18, 40,55, 47,44, 30,12, 78, 5,
   7,64, 30,11, 51, 6, 47,77, 50,10, 36, 1,  4,89,  8, 3,
  74,59,  6, 8, 12,23, 67,62, 81,85, 32,63, 32,53, 61,40,
  77,17, 76,19, 20,50, 29,33, 74,88, 23,47, 69,15, 65,54,
  10,21,  3,85, 53,67,  8,23, 48, 2,  3,52, 58,69, 69,45,
   0,20,  8,40, 41,15, 12,81, 34,51, 45,24, 47,81,  7,33,
  25, 7, 14, 5, 20,29, 13,73, 49,33, 73,27, 72,65, 42,67,
  46,36, 55,80, 79, 3, 31,80, 32,65, 38,14,  1,53, 28,38,
  72, 8,  4,15, 25,50, 77, 8, 26,56, 18,30, 23,83,  5, 3,
  75,29, 45,46, 44,33, 58, 7,  6,40, 14,30, 41,51, 73,19,
  47,12,  1,22, 49,39, 73,33, 27,65, 79,36, 47,75, 30,63,
  58,67, 30,62, 38,74, 43,50, 75,54, 11, 0, 17, 4,  1, 4,
  38,59, 38,41, 62,17, 39,56,  6, 7,  6, 0, 59,37, 39,22,
  74,55, 67,14, 72, 5, 83,69, 65,12, 62,75, 37,76, 33,55,
  75,85,  4,76, 52,48, 87,27,  3,58, 42,40, 28,14, 13,17,
  50,72,  8,49,  3, 5, 71,31, 75,32, 66,32, 45,17, 82,15,
  88, 7, 36,10, 47, 2,  3,72, 49,28, 51,83, 80,43, 82,30,
  70,37, 87,10, 22,44, 75,25, 20,31, 55,71, 61, 8, 50,14,
  25,73, 83, 3, 63, 2, 76,50, 87,62, 47,63, 10,73,  2,15,
  38,43, 24,50, 38,60, 80,33, 43,82, 22,82,  0, 5, 24,22,
  12,20, 44,88,  6,32,  1, 5, 72,16, 54,50, 22,17,  0,48,
  27,23, 85, 3, 17,79, 19,57, 48,14, 78,83, 28,54,  0,74,
  81,69, 62,41, 83,45, 19,25, 10, 9,  9,66, 75,69, 75,23,
   5,69, 51,86, 16,65, 29, 4, 62,70, 49,56, 42,52, 52,62,
  23,82, 39,49, 86,76, 71, 7, 78,85, 70,79, 59,54, 80,63,
  56,83, 29,61, 13,26, 23,14, 33,85, 48, 4,  3,65, 41, 9,
  75,82, 43,21, 89,63, 81,73, 77,15, 80,71, 26,51, 45,52,
  33,40, 40,21, 79,28, 79,10, 86,51, 74,57, 10,89, 81,51,
  70,71, 88,64, 76,17, 80,46, 67,44, 31,82, 37,34, 47,39,
  86,50,  2,73, 63,26, 75,14, 88,48, 15,10, 54,27, 21,66,
  13,56, 81, 0, 64,33, 61,69, 43, 2, 29,11, 54,86, 34,74,
  87,43,  7,60, 52,54, 37,88, 12,42, 20,66, 31,38, 60,33,
  34, 6, 25, 9,  4,63, 58,47,  3,57, 31,32, 40,59, 83,26,
  27, 9, 61,78, 51,69, 65,21, 72,82, 28,21, 86,12, 44,69,
   8,51,  5,49, 74,19, 27,74, 85,78,  6, 6, 29,88, 82, 2,
  46,86, 81,67, 68,72, 85,25, 50, 4, 83,34, 48,10, 25, 6,
   0,76, 14,67,  2,78, 44,59,  5,24, 11,40, 16,61, 13,53,
   0,46, 76,69, 66,80, 72,80, 49,45, 77,78, 22,78, 52, 0,
  11,38, 35,16, 83,38, 27,78,  4,37, 44,71, 70,16, 11,85,
  13,70, 53,43, 29,66, 66, 6, 51,51, 34,50, 68,71, 49,64,
  79,17,  6,20, 30,19, 61,87, 60,13, 13,65,  1,82,  7,74,
  57,89, 10, 0, 60,72, 70,27, 39, 5, 40,80, 21,18, 51,62,
  86,48, 26,63, 36,73, 88,27, 71,70, 32,69, 18,69, 35,57,
  41,54, 59,60, 73,59, 87,26, 83,74, 89, 0, 18, 2, 41,68,
  85,16, 24,34, 78,32, 54,37, 33,37, 79,70, 80,25, 52,69,
  17,73, 18,25, 57,54, 60,79, 51,71, 74,15, 63,79,  2,34,
  26,37, 71, 9, 51, 0, 63,39, 28,44, 13,69, 32,33, 60,64,
  81, 2, 71,23, 74,61, 42,26, 25,27, 54,55, 23,89, 37, 6,
  38,71, 51,45, 11,55, 77,65,  6,29,  7,59, 24,26, 44,48,
  66,76, 43,76, 35,85, 25,53, 27,15, 31, 9, 79, 4, 35,84,
  58,77, 21,39, 71,69,  3,21, 59,77, 21,86, 56,37, 38,25,
  26,14, 87,47, 29, 8, 32, 3, 83,47, 65,32, 64,74, 10,45,
  49,20,  1,26, 88,50, 59, 7, 61, 0, 54,11, 78,36, 46,88,
   1,85, 70,36, 58,89, 86,79, 61,51, 68,27, 53,32, 81,55,
   2,58, 61,30, 40,72, 56,33, 30,20, 37,56, 50,54, 82,74,
  77,52, 11,24, 38,64, 26,75,  3,53, 30,37, 76,14, 60,43,
   8,70, 87,21, 72,77, 32,51, 50,73, 85,50, 62,40, 43,17,
  25,25, 50,86, 37, 2, 54,61, 53, 9, 82,70, 11,72, 34,11,
  33,68, 69,61,  5,22, 86,17,  9,51, 12,70, 31,31, 58,27,
  59,34, 16,81, 26,74, 57,36, 16,70, 71, 6, 74,69, 43,51,
  55,63, 75,50, 30,74, 73,67, 50,59, 61,52, 27,57, 15,31,
  19,70, 79, 6, 80,67, 48,49, 71,14, 10, 6,  7,46, 70,57,
  77,85, 81,35, 84,24, 76,27, 80,39, 78,64, 17,85, 12,54,
  54,16, 79,73, 67, 9, 80, 4, 36, 4, 83,15, 26,15, 78,80,
  74, 5, 25,31, 31,24, 83,25, 25,26,  9,61, 54,48, 48,16,
  30, 3, 29,60, 13,42,  6, 4,  4,11, 30,32, 54,64, 84,20,
  35, 8, 45,78, 43,20, 72,83, 11,37, 23,24, 27, 7, 86,31,
  77,35, 62,25, 26,43, 61,77, 42,15, 48, 3,  2,72, 40, 9,
   8, 0, 70,86, 16,71, 82,81, 16,63, 59,85, 10,31, 19, 0,
  30,42, 21,55,  5,88, 30, 9,  0, 7, 44,34, 28, 9,  8,72,
  33,57,  3, 4, 45,28, 50,37, 72,32, 19,61, 14,83,  0,19,
  35,30, 13,20, 60,15, 34,16, 15,61, 41,49, 57,62, 24,59,
  86,20, 34,75, 87,64, 66, 2, 26,71, 81,45,  2,61,  6,33,
  31,67, 73,56, 61,21, 62,16, 50,82, 85,46, 54, 2, 37,10,
  87,63, 66,64, 77, 2,  3,32, 60, 2, 58,85, 19,87, 50,69,
  31,23, 10,33, 88,60, 19,47, 39,53, 44,27, 20, 7, 22,33,
  89,38, 85,18, 33,26, 83,27, 81,89, 78,52,  7,87, 64, 7,
   4, 0, 40,44, 72,75, 86,29, 36,68, 54,54, 14,55,  6,35,
   7,58, 82, 3, 44,11, 29,71, 11, 8, 74,32, 77,73,  2,10,
  56,47, 35,77, 11,70, 71,85, 80,76, 39,78, 77,59, 76,57,
  26,79, 49, 7, 26,85, 34,35, 13,84,  3,39, 14,77,  0,33,
  37,58, 44,79, 28,62, 31,21, 24,54, 29,87, 30,47, 42,75,
  80,79, 71,20, 51, 8, 59,12, 65,59, 50,61, 11,27, 49,80,
  54,76, 52,89, 45,55, 49, 1, 37,38, 36,28, 19,43, 70, 8,
   8,34, 42, 7, 64,63, 89,56, 74,62, 77,14, 54, 9, 14,44,
  57,64, 10,85,  7,26, 69, 1, 21,59,  4,55, 49,58, 28, 3,
  10, 1, 26,52, 55, 9, 74,54, 16,37, 52,38,  9,24, 40,31,
  21,12, 31,19,  6,39, 48, 6, 77,89, 72,11, 39, 9, 20,52,
  88,34, 63,21, 23,60, 33,41, 17,14, 27,75, 71, 2,  8,10,
  51,77,  0,52,  5,52, 31,65, 18,39, 85,76, 10,29, 82,26,
  70, 3, 53,42, 39,44,  3,89, 80,84, 67,53, 49,18, 88,44,
  65,19, 41, 2, 83,35, 68,14, 85,49, 40,85, 68,54, 33,67,
  53,28, 25,76, 81,88, 65,82, 18,84, 38, 1, 38, 5, 42, 0,
  85,11, 85,86, 28,83, 39, 3, 20,36,  4,36, 42,71, 46,23,
  71,37, 14,78, 45,33,  1,64, 16,83, 45,56, 62,61, 50,88,
   4,29, 47,80, 28,31, 54,80, 24,88,  3,59,  0,55,  7,27,
  49,75, 82,16, 24,31, 29,43, 15,24, 67,66, 63,36, 44,73,
  31,33,  7, 9, 26,88, 53,65, 70,33, 73,88, 67,22, 12,39,
  37,32, 16,89, 51,38, 72,64, 21, 9, 59,74, 36,59, 53,85,
  48,20, 26,19, 78,10, 79,20,  3,43, 49,71, 69,68, 81,10,
  13,38, 56, 6, 49,16, 23,16, 39,50,  0,63, 25,32, 11,29,
  72,43, 22,35, 71,48,  4,74, 46,83, 46,57, 89,65, 64,57,
  10,18, 75,19, 33,66, 57,49, 39,26, 15,42, 48,55, 16,31,
  28,33, 74,49, 55, 8, 62,60, 38,73, 17,67, 11,88, 23,84,
  19, 6, 79,46, 44,50, 41,70, 63,86, 30,86,  9, 7, 51,25,
  73,40, 72,87, 39,33, 60,55, 57,15,  1,79, 60,65,  9,77,
   0,83, 22,50, 32,62, 59,50, 31,60, 14,54, 86,14,  8,17,
  86,71, 30,68, 76,31, 41,44, 25,61, 32,60, 35,23, 61,63,
  81,36, 65,13, 71,28, 58,60, 20,68, 34,55, 85,42, 69,88,
  55,68, 18,11, 84,75, 72,10, 32,82,  1,38,  7,30, 68, 9,
  86, 5, 67,31,  8,22, 72,20, 32,76, 62,64, 22,77, 61, 2,
  56,42, 18,42,  6,14, 22,12, 53,86,  5, 4, 40,29, 49,40,
  86,53, 17,23, 56,27, 10,70, 50,35, 71,78, 71,60, 52, 2,
  49,50, 48,11, 30,28,  6,60, 43,71, 87,20, 28, 8, 86,87,
  30,67, 58,84, 12,58, 55,75, 69,11, 45,80, 36,83,  8,43,
  70,34, 17,52, 89,13,  3,37, 54,79, 31,43, 47,83, 78,86,
  66,52,  5,29, 42,22, 27,84, 67, 6,  4,79, 75, 0, 77,71,
  80,74,  2,77,  5,45, 88,16, 21,11, 30,56, 69,42, 18,12,
  10,34, 40,33, 48, 1, 45,83, 83, 2, 75,48, 25,43, 70,64,
  11,80, 21,88, 52,66,  5,43, 29, 0, 56,89, 36,67, 15, 3,
  58,79, 76,32,  1,35, 10,22, 53,82,  4,71, 24,64, 82,46,
  66,16, 72,14, 88,86, 16,73,  8, 8, 55,28, 58,88, 74,41,
  58,48, 53,62, 65,23, 40,18, 51,85, 52,33, 75,61,  2,11,
  18,21, 81,61, 59,10, 41,76, 48,42, 33,22, 75,10, 84,11,
  28,22, 65,66, 67,26, 25,48, 63,38, 39,43, 43,61, 21,37,
   0, 0, 69,14, 53,51, 27,76, 89,18, 40, 0,  7,89,  8,25,
  70,10, 67,21, 24,56, 47,36, 36,72, 19,23, 74,28, 67,29,
  18,32, 63,17, 78,49, 19,28, 89,12, 55,20, 57,61, 78,81,
  56,10,  9,43, 64,70, 24,20, 54, 8, 21,65, 24,61, 34,57,
  86,83, 88,51,  2,36, 18,34,  4, 1, 81,62, 10,11, 61,84,
  10,51, 66,10, 75,65, 69,49, 82,53, 20,81, 64,79, 84,39,
  25,18, 47,51, 67,47, 79,29, 63,28, 28,80, 29, 1, 43,33,
  88,79, 23,58, 42,19, 52,39, 28,12, 58, 8, 65,33,  3,83,
  12,25,  4,87, 38,23, 72,88, 55,48, 58,30,  1,54,  8,54,
  39, 2, 72,76, 84,67, 36,66, 21,41, 16,62, 87,50, 59,49,
  83,43, 34,32, 88,63,  8,50, 15,14,  7,32, 57,68, 72,35,
  83,42, 65,84, 84,17, 27,26, 18,79, 72,22, 74,35, 51,88,
   6,85, 83,65, 16,74, 81,56, 27,28, 62,71, 45,18,  4,68,
  29,34,  7,39, 30,70, 14, 7,  6,75, 64,72, 49,22, 18,78,
  83,71, 86,52, 42,44, 26,82, 51,70, 19,64, 14,11, 53,15
};
const int p_90_11[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1524,
  /// Array of preferences
  62,18, 67,85, 89,69, 41,14,  0,54, 67,84,  9, 8, 15,89,
  78,73, 48,41, 25,26,  7,49, 54,89,  3,72, 57,13, 29,82,
  68,77,  9,49, 24,22,  9,69,  9,51, 12, 3, 38,12, 89,42,
   4,77, 58,89, 44,16,  5,77, 88,78, 32, 8,  7,52, 18,81,
  76,76, 72,74, 82,42, 57,69,  5,42,  9,61, 84,64,  3,21,
  12,77, 79,76, 70,79, 50,63, 68,46, 47,24,  2, 1, 25,86,
  84, 5, 43,14, 17,79, 28,45, 26, 9, 47,29, 58,63, 60,89,
  70,16, 27,44, 39,29,  5,75, 66,48, 35, 5, 21,86, 62,80,
  36,30, 88,89,  3,48, 50,54,  7,45,  5,12, 11,43, 26,23,
  89,22, 78,84, 85,52,  4,17, 12,38, 18, 4, 32,63, 64,15,
  43,68, 16, 4, 85,46,  8,61, 38, 5, 29,50, 54,27, 49,51,
  26,71, 57,83, 22,37, 46,79, 11,44,  5,38, 46,44,  3,28,
  30,34, 80,32, 30,25, 27,29, 88,59, 71,50, 49,61, 76,59,
   5,40,  4,45, 84, 8, 86,88, 27,19, 64,69, 31,85, 41,27,
  22,29, 34,84, 28,11, 25,41,  2,40, 40,37, 15,28, 75,89,
  10,42, 55,83, 29, 4, 12,18, 89,77, 17,21, 38,59, 37,56,
  17,13, 58,38, 41, 2, 40,60, 43,62, 53,66, 17,61, 73,35,
  62,76, 79,44, 73,11, 74,15, 59,72, 69,80, 67,47, 32,48,
  55,58, 20,73, 69,66, 66, 4, 66,18, 59,89, 75,75, 47,74,
  10, 4, 22,82, 43, 2, 56,26, 45,71, 15,81, 57,71, 23,51,
  59,49, 47,14, 28,13, 74,89, 77,48, 74,82, 62, 7, 68,72,
  67,64, 78, 2, 52,88, 41,74, 34,62, 71,36, 89,33, 37,70,
  35,70, 11,64, 14,65, 86,21, 21,29, 45,68, 63,36, 19,67,
  28, 4, 35,25,  4,52, 52,50, 71,19, 36,13, 28,75,  7,14,
  85,27, 38,74,  7,74, 42,78, 36,12, 40,64, 26,78, 70, 4,
  81,69, 78,69, 58,14,  8,37, 36,83, 19,45, 22,22, 81,82,
  16, 6, 65,49, 28,32, 80,62, 58,47, 80,76,  6,56, 52,56,
  58,45, 26,18, 13,38,  2,82, 53,40, 56,33, 82,31, 79,45,
  23,86, 83,15, 79, 5, 49,56,  6,88, 23,35, 18,42,  9,81,
  85, 9, 77,63, 65,75, 58,65, 66,69, 84,27, 55, 5, 22,62,
  20,60, 44,61,  0,77, 22,40, 40,49, 40,28, 40,85, 78,62,
  80,16, 42, 2, 24,61, 77,72,  6,70,  4,38, 22,79, 23,84,
  42,44, 21,39, 17,64, 33, 2, 30,45, 23,78, 39,77, 74,25,
  42, 7,  4,85, 67,16,  1,32, 62,60, 50,50, 19,68, 51,43,
  54,78, 31,22, 70, 1, 29,44, 27,71, 18,66, 84,16, 22,17,
  79, 3, 57,44, 62,71, 79,60, 62,27, 40,75, 89,46, 78, 6,
  56,36, 51,62,  3,70, 60,76, 11,76, 56,20, 29,68, 67,13,
  12,81, 41,85, 79,63, 52,19, 53,55, 68,20, 27,78, 32,41,
  30,54,  7,57,  3, 6, 37,41, 68,51,  2,75, 81, 3, 54,65,
  50,43, 65,34, 83,20, 64,59, 50,52, 67, 8, 19,33, 66,27,
  77,18, 36,31, 38,58,  5,69, 31,33, 47,45, 79,86,  7,51,
  37,87, 60,34, 29,51, 87,42,  9,63, 29, 1, 76,84, 86,69,
  43,33, 18,50, 27,66, 50,11, 46,30, 53,54, 59,73, 63,28,
  86,62, 68,33, 78, 5, 19,71, 67,25, 11, 3, 67,56, 27,50,
   4,42, 30,32, 70,62, 11,58, 28,55, 22,59, 83,81,  7,47,
  42,87, 23,55, 19,37,  8,23, 70,41, 27,43, 39,49, 72,48,
  89,57, 56,77, 21, 3, 34,82, 54,43, 26,41, 43,34, 83,57,
  14, 6, 73,70, 50, 2, 47,67, 39,57,  5, 5, 81,70, 11,66,
  17,30, 89,74, 47,19, 64,32,  1,15, 67,18,  6,13, 87,14,
  23,76, 48,58, 67,80, 29,60, 55,31, 28,68, 66,43, 89,88,
  27, 3, 55,22, 79,12,  4,18, 54, 2, 67,70, 11,37, 10,51,
  51,40, 12,33, 59,47, 32,19, 71,52, 36, 0, 59,12, 30,15,
  83,29, 66,59, 14,20, 44,58, 46,40, 12,56, 75,71, 45,61,
   0,80, 52, 1, 45,64, 78,28, 88,33, 42,47, 22,45, 42,18,
   4,63, 31,18, 78,72, 78, 1, 59,29, 67, 1, 40,89,  8,66,
  59,15,  0,85,  6,44,  0,69, 63,12, 63,16,  8, 2, 22,71,
  31,56, 22,15, 67,73, 54, 1, 38,50, 21,37, 86,67, 74,55,
  82,73,  9,57, 32,12, 47,79, 78, 4, 84,21, 22,33, 49,74,
  35,29, 10,49, 21,82, 54, 9, 58,61, 79,69, 88,26, 87,19,
  79,72, 88, 2, 28, 1, 77,71, 43,88, 52,58, 49,82,  9,83,
  18,87, 57,74,  6,81, 13,48, 83, 0,  6,67, 15,65, 29,40,
  20,77, 75,77, 84,33, 30,67, 44,21, 31,87, 24,43, 37,35,
   1,13, 52,78, 80,79, 80,66, 69,29, 43, 5, 77,30, 25,18,
  57,34, 20,71, 24, 1, 29,83, 11,85, 71,32, 39, 1, 78,35,
  66,34,  8, 9,  0,83, 23,80,  5,39, 67,44, 32,54, 21,12,
  17,15, 83,50, 65,26, 81,41, 21,74, 37,86, 35,47, 61,86,
  33,60, 42,16, 71,60, 13, 3, 67,35,  2,30, 26,22, 72,38,
  78,77, 43,19,  8,86, 77,89, 86,79, 47, 0, 78,65, 12, 8,
  22,30, 77,65,  5,85, 39,56, 26,81, 45,24, 87,75, 72,22,
  16,88, 29,23, 42,72, 36,41, 81,66, 18, 0,  9, 0, 11,52,
  89,65, 74,65, 41, 7,  8,72, 47,65, 20,83, 89,83, 18,86,
  72,81, 18, 3, 33,51, 86,15,  4,86, 11,79, 41,67,  8,31,
  39,72, 30,22, 61,89,  8,74,  5,52, 40,57, 71,61, 53,50,
  63,37, 32,52, 64,89, 80,34, 65, 4, 23,46, 17,71, 88, 6,
  50,89, 26,80, 28,66, 87,62, 53,88,  0,68, 63,54, 36,82,
  76,12, 37,69, 41, 8, 46,62, 16,26, 66,74, 57,55, 52,25,
   7,20, 53,35, 54,48, 60,64, 46,13, 20, 6, 32,79, 45,49,
  25,38, 37, 0, 83, 9, 19,89, 51,18, 19,81, 13,86, 72,17,
  35,57, 40,86, 74, 3, 63,33, 10,50, 33,76,  9, 5, 88,23,
  10,82, 48,73, 64,68, 81,19, 57,63, 80, 2, 63,83, 51, 4,
  38,81, 50,84, 36,66, 60,86, 46,57, 59,14, 69,19, 65,67,
  44, 1, 62,41,  2,53, 27,20, 55, 1, 52,48, 85,38, 64,81,
  22,16, 78,52, 43, 6, 14,85, 17,56, 61,53, 88,10, 38,24,
  79,27, 58,83, 41, 0, 65,31, 75,11, 49, 4,  7, 6, 38,65,
  40,82, 36,15, 68,45, 73,63, 67, 5, 84,75,  6,42, 15,30,
  71,78,  2,81, 20,29, 18,76, 66,41,  8, 8, 12,57, 35, 1,
  55, 8,  6,26, 35,32,  7,24, 11,20, 38,51, 19, 8, 61,61,
  48,68, 41,52, 76,45, 85,32, 16,85,  0,45, 35, 3,  6,18,
   5,78, 70,58, 67,54, 82,87, 33,75, 66,72, 84,51, 33,82,
  89,24, 25,30, 64,84, 13, 5, 16,33, 49, 1, 70,64, 32,28,
  11, 6, 55,78, 14,32, 22,19, 73,85, 25,88, 53,28, 79, 1,
  61,65, 70,74, 44,65, 26, 4,  4,14, 50,60, 16,83, 23,20,
  42,74, 25,76, 18,85, 49,88, 76,79, 72,68, 21,64, 70, 8,
  14,53, 55,19, 52,36, 36,88, 37,28, 89,15, 82,12, 36,34,
   5, 9, 87,44, 71,73, 53,82, 67,62, 88,40, 17,60, 74, 6,
  39,65,  5,22, 50,32, 60,13, 61,47, 56,24, 81,46, 45,25,
  87,54, 50,68, 28,65, 27,73, 85,24, 47,57, 55,27, 85,85,
  42,26, 14,87, 82,84,  8,67, 38,73, 45,85, 51,64, 54,28,
  48,16, 24,11, 76,22, 63,45, 26,36, 19,83, 67,23, 15,44,
  63,43, 19,16, 43,56, 17, 7, 43,51, 43,36, 74, 2, 75,69,
   1,83, 84,87, 50,88, 26, 8, 45,34, 57,25, 13, 8,  8, 0,
  13,27, 70,83, 25,36, 43,24,  4,11, 29,21,  7,33, 74, 1,
  70, 9, 87,84, 71,64, 25,78, 56,51, 44,67, 70, 0, 15,18,
  15,14, 15,39, 84,81, 39,34,  5,48, 27,23, 36, 6, 12,22,
  71,41,  7,80, 33, 5, 48, 6, 17,12, 52,79, 36,23, 28,30,
  23,19, 35,86, 37,16, 47,48, 29,27, 85,67,  7,83, 47,68,
  84,47, 43,74, 31, 8, 39,35, 14,10, 84,53, 30, 2, 62,70,
  24,68, 24,72, 79, 8, 50,71, 70,55, 69,49, 49,59, 51,23,
  20,59, 73,31,  5,32, 85,48, 46,54, 44,19, 81,87, 79,42,
  60,66,  0,33, 54,35, 46, 2, 84,59, 58,82, 85,73, 60,23,
  43,86,  5,87, 48,72, 18,77, 47,47, 23,15, 74, 7, 88,88,
  43, 9, 28,82, 37, 5, 15,26, 29,29, 74,64, 31,49, 50,75,
  62,83, 27,83, 15,32, 15,48,  1,66, 21,25,  3,77, 82,25,
  20,51, 26,37, 40,80, 35,21, 42,50,  7,56, 84,39,  7,40,
  21,72, 80,88, 80,87,  9,27, 14,30, 28,71, 60,65, 32,83,
  24,57, 53,29, 67,79, 43,81, 83,77, 42,84, 45,19, 20,22,
  21,30,  6,40, 89,20,  0,59, 51,31, 21,41, 58,13,  3,39,
   5,66, 22,26, 22,83,  7,65, 23,66,  8,53, 80,74, 11,55,
  46,16, 21,51, 48,52, 23,43,  7,59, 59,34, 25,22, 85,56,
  51,38, 44,85, 12,15, 48,38, 69,17, 76,46, 89,58, 21,32,
  46,70, 62, 9, 52,13, 60,24, 28,20, 49, 2, 65,71, 60,17,
  64,88, 11,11, 27,45, 74,87, 52,37, 87,81, 50,36, 88,61,
  84,68, 35,84, 68,63, 46,65, 28,86, 28,42, 47,49, 81,67,
  86,38, 77,87, 62,35, 30,30, 55,44, 11, 8,  4,78, 23,85,
  32,72, 69,64, 21,36, 59,86, 48,14, 12,19, 16,60, 67,86,
  61,25, 46,32, 38, 1, 58,23, 33,78, 89,41, 55,68, 56, 8,
  88,71, 43,54, 61,46, 21, 5, 43, 4,  4,55, 79,70, 14,56,
  42,54, 66,55, 83,26, 51,69,  6, 1, 60,54,  3,89, 60,44,
  81,79, 86,24, 59,65, 24,50, 56,52,  6,52, 47,66, 40,21,
  18,36, 54, 3, 17,57, 43,37, 58,50, 12,13, 67,45, 52,77,
  51,58, 63,42, 70,81, 42,12,  3,49, 25,44, 81,81, 46,86,
   6,16, 34, 8, 69,65, 36,54, 32,80, 23,14, 71,34, 51,47,
  31,37, 39,15, 10,74, 72,65, 87,83, 62,36, 38,42, 19,26,
  20,12, 39,12, 11,10, 75, 1, 61,48, 85,15, 13,22, 37,44,
  21,58, 71, 0,  2,54, 47,40, 74,11, 82,52, 60,74, 16, 5,
  74,59, 35, 9, 13,44, 57,56, 74,31, 21,66, 50, 6, 22, 8,
  12,89, 12,58, 78,89, 68,43, 32,33, 41, 3,  4, 0,  7,76,
  66,42, 76,64, 58, 6, 82, 2, 20,14, 13,69, 11,69, 86,84,
  85,58, 69,60, 78,11, 58,77,  8,78, 24,13,  4,33, 73,13,
  67,65, 87,50,  1, 7, 62,42, 77,42, 17,59, 27,36, 45,41,
  78,58, 72,24, 80,67, 13,50, 39,55, 59,61, 52,28,  2,31,
   4,81, 54, 5, 20,84, 66,70, 14,54, 24,65, 73,28, 60, 5,
  60,39, 57,60, 40,76, 70,45, 80,18, 24,23, 46,45,  9,24,
  28,63, 40,20, 55,17, 44,11, 73,27, 25,47, 62,64,  6,43,
  55,25, 49,45, 69, 7, 53,41, 45,50, 31,23, 73, 4, 45,72,
  56,13,  6, 6, 28, 6, 33,81, 23,30, 22,21, 68,67, 14,74,
  65,25, 88,12, 73,89, 70,18, 33,31, 34,22, 15,72, 69,33,
  77,49, 87,53, 37,62, 13,28, 76,78, 89,26, 50,45, 34,33,
   3,78, 81,54,  8,15, 17,44, 70,84, 38,63, 49,21, 77,47,
  41,21, 76,18,  7,73, 20, 2, 37,33, 59,41, 53,69, 57,14,
  45,33, 16,67, 61,81, 23,23, 65, 0, 39,36,  0,72, 13,43,
  75,78,  0,29, 69,75, 15, 6, 73,75, 26,43,  5,28, 75,72,
  81,73, 85,45, 37,80, 33,85, 39, 8, 26,74, 36,85, 27,52,
  89, 9, 60,26,  2,43, 84,31,  4,62, 14,22, 48,60, 63,55,
  73,54, 82,37, 88, 7, 13,16, 78,55,  1,17, 26,34, 34,41,
   3,73, 37, 8, 77,75, 10,22, 38,22, 56,39, 41,89, 52,11,
  53,22, 25,75, 75,62, 47,61,  6,20, 26,49, 22,53, 39,51,
  38,38, 56, 2, 65,79, 48,30, 23,58, 56,32, 17,29, 33,17,
  51,57, 81,64, 89,32, 82,85, 57,57, 26,31, 47,58, 85,40,
  18,78, 69,28, 46,80, 70,50,  9,22, 58,81, 62,78, 71,48,
  12,25, 58,68,  2,70, 40,72, 74,46,  1,25, 47,50, 26,77,
  68,57, 61,79, 36,20, 54,70, 30,64, 66,49, 56,82, 61,32,
  13,35, 21, 9, 53,71, 77,28, 18,39, 55,86, 11,54, 49,17,
  73,22, 46,59, 89,56,  8, 4,  8, 5, 19,38, 82,36, 48,79,
  75,79, 32,89, 30,17, 76,23, 56,84, 79,37, 53,15, 89,59,
  15,58, 53, 8, 81,36,  5,59, 67,10, 17,80, 27,21, 35,58,
  74,16,  4,61, 18,49, 48,23, 34,17, 10,57, 71,29, 25,24,
  11,48, 36,53, 12,31, 51,60, 71,84, 72,15, 20,28,  4,84,
  59, 7,  3,82, 75,40, 69,25, 49,12, 89,29, 41,40, 25,58,
  12,85, 45,82, 21,11, 71,57, 54,42, 20,70, 52,26, 46,87,
  83,16, 43,32,  9, 3, 30,23, 86,53, 74,52, 32, 1, 34,18,
  67,66,  3,18, 44,52, 35,60,  6,85, 80,17, 43,10, 70,56,
  88,83, 53, 4, 16,71, 36,89, 80,50, 17,75, 23,16, 61,11,
  72,25, 24,10,  9,17, 19,61, 79,56, 11,80, 27,84, 85,86,
  43,47, 82,56, 81,65,  0,67, 50,20, 63,38, 11,75,  1,52,
   1,67, 39,58, 11,81, 57,67, 34,21, 47,32, 57,87, 21, 0,
  43,12, 33,23, 47,33, 19,77, 60, 9, 56,64, 54,86, 68,28,
  70,37, 29,46, 68,54, 73,58, 55,21, 74,43, 81,59, 27,30,
  29, 9, 79,58,  4,22, 89,14, 83,68, 76,58, 70,46, 35,13,
  69,37, 35,63, 77,23, 58,56, 60,52, 54,71, 26,16, 62,49,
  28,22, 12,20, 76,13, 65,84, 75,18, 43,70, 58,29, 18,69,
  76,21, 56,34, 87,18, 11,16, 33,58, 27,81, 51,28,  0,16,
  40,38, 81,44,  2, 2, 65,47,  2,60, 18,71, 30,13,  4,39,
  10,25, 69,51,  1,79, 25, 7, 47, 8, 55,41, 85,11, 41,59,
  50, 3, 27,58, 21,14, 10,76, 62,34, 86,43, 50,29, 12, 9,
  75, 0,  6,57, 82,70, 34,27, 12,84, 86,22, 16,87, 70,24,
  30,84, 49,60, 28,48,  6, 3, 28,14, 21,47, 62,66, 78,66,
   9,64, 73,83, 24,44, 67,34,  7,42, 62,39, 37,10, 65,22,
  83,45,  2, 7, 79,23, 57,45, 35, 0, 32,66, 43,57,  9,48,
  59,31, 47,11, 44,32, 17,52, 33,42,  4,58, 35,23, 74,26,
  57,54, 58,10, 15,47,  4,12
};
const int p_90_12[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1292,
  /// Array of preferences
  39,14,  3,37, 16,88, 84,56, 13,33,  2,59, 46,31,  0, 0,
  54,69, 72,38, 38,20, 62,58, 40, 8, 40, 7, 47,74, 85,52,
  13,25, 23,42, 36,42, 72, 5, 16,56, 34,28, 83,27, 60,12,
  32,36, 19,12, 72,39, 18,50, 44,58, 10,58, 12,40, 59,18,
  36,88, 77,18, 64,84, 14,86, 33,89, 37,71, 13,15, 51,28,
  45, 0, 59,40, 61, 7, 57,40, 17,79, 40,72, 89,88, 14,49,
  78,17,  2,75, 78,55, 69,19, 51,25, 78,61, 10,28, 12,39,
  61,13, 69, 1, 40,68, 84,13, 10, 4, 77,61, 10, 2, 70,30,
  55,18, 39, 5,  3, 5, 76,74, 15, 9, 29,51,  6,82, 55,61,
   4,57, 67,40, 75,54, 24,13, 20,55, 19,11, 76, 4, 66,46,
  18, 5, 75,15, 14, 1, 89,58, 49,49, 34, 7, 25,13, 81,24,
   3, 9,  0,77, 66, 5, 48, 4, 15,45, 40,74, 14,61, 29,60,
  23,45, 74,51, 18,36, 56, 2, 63,53, 61, 3, 62,63, 68,66,
  41, 5, 75,57, 75,68,  0,58, 25,66, 69,43, 16,86,  4,75,
   6, 1, 86,38, 21,50, 57,31, 82,60, 30,46, 32,39,  9,53,
  89,46, 63, 0, 44,66, 12,80,  7,47, 21, 5, 55,86, 27,44,
  79,87, 19,65, 74,16, 73,69, 38,88, 41, 0, 45,84, 46,77,
  69,16,  0,50,  8,48,  3,88, 69,57, 18, 0, 18,58, 41,72,
  44, 8, 32,26, 57, 7, 52,66, 72,55, 55,50, 74, 0, 27,45,
  19,36, 23,34, 61,33,  7,18, 36,15, 58,12, 83, 2, 45,35,
   7,70, 86,15,  3,15,  9,52, 82, 8, 29,44, 22,14, 56,70,
  62,54, 27,56, 50,47, 30,11, 69,27, 22,75, 52,86, 33,33,
  58,86, 74, 7, 55,78, 57,55, 62,25, 43,62, 23,58, 72,27,
  58,66,  3,53, 73,27, 19,20, 53,67, 54,89, 44,11, 16,89,
  52,68, 32,33, 60,68, 56,59, 86,47, 67,10, 62,50, 78,10,
  76,26, 76,11, 18,85, 84,88, 65,24, 46,75, 87,54, 79,21,
  77,32, 68,58, 80, 5, 30,20, 37,86, 26,13, 54,55, 46,54,
  74,37, 77,59, 52,48, 79, 2,  6,80, 72,63, 67,71, 16,40,
  70,19, 56,22, 26,62, 15, 4, 13,73, 51,37, 33,23, 25, 8,
  34,19, 20,19, 29,53, 45,38, 12,12, 23,39, 14,13, 65,42,
  20,15, 55,62, 48, 0, 52,59,  4,88, 61,62, 83,42, 49,37,
  25, 5, 61,12, 79,15, 10,63,  5, 4, 70, 8, 47,61, 48,41,
  76,67, 89,30,  4, 5, 31,15, 39,63, 16,50, 42,68, 37,87,
  35,75, 41,79, 62,78, 28,47, 65,45, 20,56, 51,36,  1,36,
  75,80, 18,66, 34, 0, 45,79, 30,85, 22,82, 31,12, 21,41,
  46,59, 34,75, 21,85,  5,86, 50,34, 29,23, 78,85, 23, 9,
  45, 1, 40,33, 52,87, 86,53, 22,19,  7,43, 31,58, 35,57,
   3,34, 36,45, 47,43, 59,87,  1,55, 17,16, 60,23, 45,63,
  21,87,  9,67, 32,51, 27,19, 49,76, 65, 3,  1,61, 47,55,
  62,51, 40,81, 55,69, 18,10, 80,21,  0,26, 80,46, 17,59,
  51,86, 56,76, 36,44, 73,17, 45,17, 79,52, 51,66, 87,43,
  35,17, 87,86, 59,49, 19,34, 28, 4, 47,16, 30, 1, 55,70,
  32,18, 13, 8, 22, 6, 72, 8, 78,32,  2,43, 37,63, 81,38,
  66,42,  2,30, 41, 8, 20,17, 56,17, 48,38, 74,88, 63,82,
  79,55, 40,17,  4,76, 16,31, 57,73, 28,76, 78,25, 62,86,
  66,37, 66,59, 47, 4, 61, 8, 56,27,  4,22,  7,76, 83,44,
  42,38, 66,89, 16,52, 68,72, 50,87,  2,61, 30,86,  8,61,
  57,44, 52,37,  8,52, 63, 1, 56,24, 61,80, 76, 0, 14, 5,
  11,10, 15,18, 88,35, 66, 8, 28,42, 22,72, 82,46, 54,42,
  48, 1, 38,57, 75,21, 61,14, 12,64, 14,78, 54,86,  7,88,
  35,18, 18,49, 63,43, 72,74, 61,67, 22,24, 32,49, 77,83,
  13,42, 15,73, 41,43, 36, 4, 61,69, 65,51, 10,35, 19,24,
  47,81, 39,80, 76,49,  8,53, 80,56, 14,81, 21,21, 89,36,
  80,63, 32,60, 19,27, 85,79, 42,50, 48,31, 47,63, 34,13,
  52,43, 58,48,  7,48, 22,85, 25,59, 48, 7, 68, 5,  2,85,
  73,88, 28,29, 88,58, 23,47, 16,48, 85, 5, 76,31, 89,32,
  67,70, 57,57,  3,27, 42, 2, 30,30, 82,70, 77,21, 75,53,
  71,33, 60,86, 69,56, 14,21, 66, 9, 14,50, 36,26, 20,70,
   0,20, 47,67,  4,53, 42, 6, 24,86, 16,49, 51,48,  6,48,
  57,56, 34,51,  1,50, 47,64, 87,53, 78,69, 53,66, 85,57,
  35,48, 48,80, 88,30, 31,44, 31,83, 86,69, 37,73, 81,85,
  10,70, 32,14, 78,41, 54,81, 67,46,  8,66, 34,82, 48,89,
  82,81, 86,84, 22,89, 82,71, 44,27, 33,78, 40,82,  1,30,
   3,65, 70, 5, 54, 8, 88,55, 14, 2, 65,84, 73,16, 44,87,
  64,52, 47,18, 88, 5,  8,55,  5,46, 52,69, 62,88, 53,43,
  24,54, 27, 9, 29,36, 69,30, 50,80, 78,86, 72,14,  5,71,
  12,81, 49,66, 78,18, 80,76, 50,54,  4,12, 64,48, 77,53,
  77,27, 13,32, 27, 3, 35,86, 16,63, 27, 7, 33, 0, 67,41,
  59,84, 84,30, 15,71, 12,68, 82,44, 84,66, 12,38, 66,49,
   4,41, 28,17, 29,45, 19,47, 22,18,  2,82, 59,76, 83,28,
  66,52, 48,66, 54,44, 59,14, 84,68, 41,38, 46,73,  8,81,
  86,41, 85,77, 73,71, 48,76, 57,65,  9,27, 38,79, 41, 3,
  81,66, 50,42, 89,24, 50,11, 69,69, 57,11, 70,41, 35,46,
  72,66, 55, 3, 13,76, 75,51, 80,35,  2,79, 83,13, 20,35,
  52,64, 87,31, 65,26, 31,71, 67,44, 27,42, 39,55,  0,66,
  14,30, 26,45, 30,52, 12,49, 10,49, 51,80, 17,83, 41,83,
  60,76, 34,45, 37,84, 76, 6, 16,87, 62,59, 29,52, 81,30,
  54,12, 48,36, 37,89, 66,36, 66,40, 32,59, 44,73, 42,12,
  25, 4, 84,62, 43,30, 64,29, 86,21, 61,34, 67,29, 69,11,
  31,62, 85,20, 35,36, 21,61, 55,65, 55,12, 57,15, 12,66,
  77,66, 24,71, 82, 7, 84,20, 33,61, 33,65, 71,14, 83,31,
   4,20, 79,69, 72,48, 73,80, 12,17, 54,83, 29, 7, 87,21,
  56, 8, 37,16, 46,69, 52,80, 83,63, 81,20, 24,58,  4,35,
  50,78,  6,33, 74,70, 41,31, 28,74, 71,30, 65, 1, 44,48,
  70,77,  6,63, 11,84, 88,77, 45,89,  6,74, 55,17, 66,62,
  29,37, 20,89, 29,25, 53,50, 61,48, 88,31, 66,76, 64,21,
  27,71, 84,55, 76, 5, 60, 0, 74,85, 37,85, 47,30, 64, 6,
  86,72, 13,77, 67,54, 78,44, 49, 0, 33,70, 64,86, 21,67,
  48,60, 67,59, 23,69, 65,19, 13,24, 42,49, 39,15, 37,19,
  12,46, 22,47, 74,69, 45,53, 40,66, 12,20,  2,34, 20,21,
  39, 1, 74,76, 71,85, 32,69, 74, 5, 37,64, 54,35, 58,83,
  15,84, 78,51,  4,65, 30,78, 22,66, 33,60, 32,40, 35,84,
  75,42,  3,81,  8,44, 54,45, 41,74, 73,18, 46,19, 34, 1,
  31,32,  4,73,  0,64, 34,79, 29, 6, 11,74, 19,87, 21, 0,
  23,67, 11,20, 30,60, 61,36, 78,42, 41,35, 42,31, 86, 7,
  82,82, 84,46, 79,49, 12,30, 52,73, 85,73, 70,68, 59,58,
  68,61, 51,44, 56,51, 56,72, 45,45, 76,85, 59, 9, 54,37,
  36,28, 76,86, 67,14, 76,63, 13,50, 61,55, 66,47, 47,80,
  77,29, 67,52, 47,59, 87,85, 43,85, 80,66, 11,51, 25,18,
  86,58, 22,11, 53,76, 14,54, 57,67, 26,82,  3,87, 41,55,
  11,69,  2,12, 71,67,  5,80, 89,17, 70,62, 66,85, 87, 9,
  22, 5, 87,50, 57,16, 11,65, 20,43, 84,73, 86,25, 30,42,
   0,13, 15,37, 64,51, 27,29, 35,26, 77,34, 26,81, 72,22,
  27,39,  4,86, 30,21, 58, 5, 56,54, 41, 6,  8,29,  7,14,
  67,73, 59,12, 21,45,  1,25, 16,32, 39,36, 32,38, 45,28,
  51, 7, 26,86, 25,72, 66,20, 89,67, 51, 9, 14,64, 38,48,
   4,23, 28,10, 78,78, 76,36, 76,17, 80,22, 72,15, 44, 6,
  64,83, 20,80, 59,13, 13, 5, 48,40, 46,71, 15,41, 39, 0,
  64, 0, 72,85, 27,63, 37,78, 25,15, 43,60, 75,47, 84,89,
  15,28, 89,54, 34,89, 10,55, 70,11,  8,56, 42,64, 11,62,
  84,24, 54,41, 86,78, 53,71, 31, 1, 82,55, 34, 3, 25,85,
  33,43, 65, 0, 78,11, 34,33, 58,13,  0,33, 18,29, 16,84,
   0,84, 73,46, 83,80, 61,89, 10,42, 85,66, 51,51, 16,13,
  23,81, 51,82, 81,32, 53, 2, 86,77, 46,42, 19,77, 88,32,
  23,22, 11,56, 50,63, 12,35, 73,67, 47,84, 15, 7, 37,31,
  39,85, 17,81, 73,61, 81,29, 13,22, 57,42, 11,61, 76,41,
  20,81, 78, 2, 32,85, 20, 3, 74,14, 14,56, 23,30, 52,45,
  85,14, 79,84, 67,36, 74,36, 17,28, 66,11, 31,88, 14,43,
  32,15, 19,80, 84, 7,  8,89, 66,41, 12,65, 22,55, 35, 3,
  10, 6, 26,25, 16,33, 63,36, 83,30, 68,48, 42,36, 42, 5,
  88,33, 20,48, 16,71,  7, 9, 38,38, 45,54, 56,61, 13,16,
  48,23, 39,28, 45,24, 73,64,  6,58,  1,45, 15,29, 49,20,
  71,44,  1,75, 51,27, 50,71, 35,56, 64, 5, 34,35, 25,22,
  29,85, 73,37, 23,63, 65, 2,  1,52, 29,63,  4, 6, 53,56,
  36,10, 18,88, 32,80, 22,34, 31,79, 45,23, 23,13, 51,14,
  81,41, 17,32, 39, 9, 82,33,  2,76, 76,53,  3,28,  2,11,
  57,45, 75,41, 78, 0, 17, 5, 61,16, 84,72, 65,86, 27,78,
  37,77, 38,29, 74,24, 15,22,  2,16, 50,30, 57,88, 33,40,
  44,68, 55, 7,  1,33, 56,66, 52,10, 28,34, 40,86, 13,11,
  65,16, 58,75, 31,55, 17,29, 43,73, 33,16, 77, 8, 84,60,
  49,56, 26, 2, 15,53, 26,22, 85,32, 10,57, 46,39, 68,17,
  18,76, 15,79,  9,31, 65,77, 71,79,  7,30,  5,20, 84, 9,
  58,70, 82,20, 80, 4, 46,83,  5,19, 11,68, 15,59, 66,60,
  19,64,  2,65, 74,81, 31,24, 47,42, 81,14, 43,72, 68,34,
   4,31, 43,77,  1,69, 67,53, 37,13, 60,10, 27,88, 85,47,
  23,38, 54, 4, 37, 6,  3, 6, 66,15, 14,51, 75, 8, 67,87,
  25,44, 16,81, 43,32, 73,51, 51, 0, 11,64, 60,36, 54,59,
   5,89, 87,79, 87,29, 45,80, 16,75, 56,25, 12,62, 46,45,
  54,28, 16,69, 39,89, 19,70, 50,56, 53,14, 87,74, 27,31,
  46,44, 16,53, 18,20, 30, 7, 38, 0, 70,18, 10,29, 85,65,
  39,48, 47,34, 42,39, 12,22, 22,78, 77,15, 33,15, 75,87,
  40,60, 10,34, 11,67, 48,69, 27,64, 39, 7, 67,47, 11,81,
  68,75,  1,51, 89,65, 19,78, 55, 4, 20,51, 36,36, 45,16,
  59,71, 35,74, 14,55, 64,26, 23,29, 19,38, 66, 2, 42, 7,
  62,76, 21,70, 77,26, 69,45, 45,74, 56,83,  9,24,  8,34,
  15,78, 76,40, 25,33, 53,46, 71,48, 35,21, 37,25, 46, 8,
  39,23, 72,28, 49, 9, 25,47, 22,64, 48,35,  6,85, 48,57,
   2,51, 61,68, 10,73, 80,71, 27,23, 82,30, 67,56, 26,77,
  25,83, 70,56, 60,85, 51,50, 13,43, 62, 3, 55,88, 78,16,
  63,46,  4, 3, 77,25, 80,31, 31,42, 76,23, 29, 0, 75,88,
  17,75, 78, 8, 60,65, 60,66,  0,15,  7, 5, 23,78, 17,37,
  54,34, 59,53, 65,64,  9,43, 19,63, 10,79, 38,11, 25,27,
  26,68, 20,85, 76,82,  4,78,  5,34, 25,60, 60,11, 83,84,
  62,13, 41,40, 75,12, 17,82, 64,76, 23,75, 41,28, 27,18,
  35,76, 40,69, 46,18,  1,71, 48,12, 75,13, 22,53, 36,25,
  85,29, 55,85, 22,87, 82,13, 27,59, 52,46, 34,81, 31,65,
   4,36, 69,68, 44,80, 46,15, 51,10, 63,69, 42,13, 64,25,
  37,29, 74,29, 55,45,  2,70, 72, 3, 88,50, 36,21, 31,28,
  54,88, 26, 0, 66,24, 40,38, 87,49, 43,64, 28,48, 53,57,
  37,60, 36,77, 32,53, 71,21, 85,40, 70,40, 49,25, 21,40,
   9,14,  3,59, 73,19, 17,13, 89,79, 61,81, 42,72, 87,46,
   1,60, 73,28, 63,66, 77, 3
};
const int p_90_13[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  97,
  /// Array of preferences
  54,64,  4,53, 64,44, 24,49, 54,84, 51,33, 75,60, 39, 7,
  65,38, 59,52, 38,19,  4,15, 19,36, 71,70, 62,76, 51,78,
  88, 6, 44,19, 76,46, 70,27, 54,35, 42,50, 21, 8,  9,49,
   4,48, 85,62,  3,63, 14,33, 35, 0, 87,67,  9,55, 28,22,
  51,74, 86,46, 89,25,  8, 7, 30,42,  5, 6, 22,77, 66,66,
  30,46, 85,58, 82,22, 28,47, 89,59,  6,37, 47,37, 70,60,
  59,70, 14,51,  6,35, 13,10, 58,13, 56,59, 15,22, 85,55,
  56,66, 89,71, 36, 9, 58, 9, 60, 6, 71,44,  8, 4, 26,35,
   1,68, 88,82, 79,15, 28,48, 11,74, 33,42, 28,75, 73,23,
  44,67, 30,66, 44, 9, 72, 1, 79,79, 81,58, 49,15, 27,29,
  88,74, 61, 4, 36,16, 34,28, 47,57, 41,89, 83,49, 45, 2,
  76,48, 53,77, 28,44, 65,14, 52,10, 22,87, 10,58,  8,44,
  30, 8
};
const int p_90_14[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  912,
  /// Array of preferences
  35,27, 49,82, 67,18, 66,48, 16,17, 59,48, 44,61, 69,89,
  77,13, 81, 3, 82,47, 69,54, 14,73, 70,10, 55, 0, 26,67,
  52,16, 73,53,  1,60, 89,45, 42,46, 26,28, 61, 4, 78,21,
   1,65, 20,66,  5,69, 43,28,  4,57, 75,38, 18,69, 74,10,
  15,27, 86,33,  0,45, 11,19, 32,89, 18,39, 19,86, 19,71,
  51,77, 82,15,  6,72, 80,54, 37,59, 13,47, 44,36, 31,80,
  54,17, 88,31, 63,37, 40,68,  7,77, 25, 8, 51,85, 40, 7,
  15,71, 33, 3, 65,48, 65,53, 72,35, 35,41, 68,80, 72,30,
   2,51, 41,59, 19, 9, 36,79, 33,63, 44,12, 15,29, 25,64,
  67,37, 37,79, 80,29, 20,77, 45,89, 66,21, 83, 1, 64,31,
  11,61, 21,55, 21,43, 11,36, 50,21, 26,61,  6,83, 83,81,
  40,65, 67,70, 64,10, 57,77, 10,46,  1,73, 34, 8, 89,54,
  32, 6, 20,16, 64,54,  1,44,  3,54, 52,34, 87,72, 15,51,
  29,49,  0, 2, 75, 3, 30,50, 18, 6, 54,42, 67,56, 67,43,
  49,31, 86,67, 32,27, 62, 7,  4,67, 79,24, 47,68, 71,26,
  47,20, 77, 5, 36,61, 28,42, 65,74, 62,17, 58,50, 22,72,
  82,43, 56,30, 88, 7, 22,56, 32,68, 70,78, 20,50, 84,14,
  82,62, 31, 5,  8,54, 47, 1,  1,68, 76, 1, 57,80, 48,14,
  49, 1, 53,69, 43,58, 52,56, 80,21, 74,29,  1,45, 43,31,
  42,82, 61,71,  1,63, 61,47, 49,27, 71,41, 75, 6, 73,24,
   6,36, 75,23, 31,86,  8,21, 84,84, 11,65, 77,56, 61,35,
  33, 6, 73,70, 35,60, 85,15, 13,37, 78,77, 79,16, 65,58,
  34,86, 58,25, 23,87, 85,39, 28,34, 83,67, 38,10,  7,10,
  53, 0,  2, 8, 59,17, 37, 3, 42,84, 80,43, 46,82, 54,52,
  67,44, 36,23, 67, 0, 74,57, 40,30, 10,62, 25,16, 33,88,
  71, 4, 58,37, 35,86, 68,76, 35,61, 30,48, 23,28, 47,64,
   8,68, 50,29,  7,38, 72,69, 14,14, 46, 1, 63,40, 66,30,
  70,13, 72,37,  2,21,  3,51, 46,31, 54,67, 68,39, 56,67,
  60,81, 22,63, 33,82,  0, 5, 60,26, 38,56, 55,66, 30,72,
  52,52, 84,34, 73,13, 19,80, 31,29, 13,53, 39,52, 34,35,
  82,72,  8,13, 81,83, 12,76, 26,12, 60,86, 68, 7, 26,74,
  79,65, 52,76, 42,88, 47,62, 17,48, 88, 0, 24,37, 43,80,
  66,24, 23,47, 15,73, 79,74, 13,28, 63,89, 46,39,  1, 9,
  34,39, 45,29, 66,16,  6,21,  0,57, 73,22, 58,74, 30,81,
  45,24, 87,13, 58,34, 28,78, 32, 0, 41,69, 57,67, 15,33,
  80,84, 35,55, 63, 9, 13,77, 21,11, 22,19, 85,75, 50,28,
  82,81,  6,16, 51,23, 30,49, 30, 5, 16,89, 50,55, 57,30,
  48,19, 63,41, 68,45,  9,32, 56,52, 76,52, 64,23,  7,31,
  42,25,  9,66, 86,37, 76,47, 56,55, 52,42, 10, 1, 88,23,
  48,15, 36,12, 29,13, 78, 0, 87,16, 44,41, 70,43, 53,66,
  15,49, 20, 5,  0,71, 27,81, 74,80, 11,85,  1,57, 88,68,
   5,66, 60,63, 78,85, 65, 4, 21,48, 40,82, 46,19,  6,84,
  86,46, 56,54,  0,80, 74, 6, 28,44, 60,61, 76,61, 30,85,
  37, 5, 12,59,  2, 0,  5,59, 11,23, 27,27,  5,89, 82,42,
  32,18, 11,73, 88,45,  8,14, 51,83, 54,65,  3,77, 37, 0,
  72, 6, 47,80, 74,46, 62,89, 21,13, 57,76,  7, 1, 37,47,
  58,33, 10,56, 24,24, 25,39, 47, 6, 38,34, 68,81, 48,46,
  17,37, 74,12,  8,71, 17,18, 70,23, 42,30,  8,12, 11,84,
  46,61, 12,37, 62, 4, 63,49, 13,18, 23, 3, 73,67, 62,35,
  81,18, 23,38, 33,29, 75,42,  5,36, 64,29, 88,35, 59,63,
  11, 4, 73,86, 35,65, 18,14, 82,80, 82, 9, 20,14, 84,64,
  88,14, 38,79,  6, 3, 36,69, 17, 6, 13,56, 23,51,  8,42,
  19,87, 35,70, 80,18, 80,77, 11,79, 13,59,  5,50, 21,10,
  64,39, 33,49,  5,67, 40,17, 63,85, 89,26, 59,38, 25,78,
  50,71, 58,14, 60,71, 61,29,  2, 1, 42,27, 31,34, 19,13,
  31,39, 86,80, 87,15, 15, 7, 41, 6, 17,69, 48, 7, 72,76,
  27,79, 29,40, 76,75, 87,67, 58,58, 55,74,  3,64, 31,16,
  63,68, 12,28, 13, 4, 62,32, 81,48,  5,75, 65,22, 26,77,
  23,26, 16,42, 55,52, 21, 6, 24,41, 68,17, 85,56, 85,71,
  72,88, 33,33, 74,39, 10,17, 60,67, 17,61, 16,52, 85,86,
  83,82, 12,49, 56,70, 68,15, 55,17, 79,28, 48,81, 84,83,
  72,62, 28,24, 22,24, 76,23, 80,72, 17,75, 41,68,  5,17,
  16, 8, 28,65, 46,76, 32,64,  9,83, 53,67, 81,82, 54,59,
  25,23, 35,36, 27,77, 46,38, 15,50, 39,39,  6,49, 75,54,
  39,68,  6,54, 57, 3, 81,43, 56,50, 22,86, 33,85, 83,47,
  29,29, 69, 8, 11,25, 85,82,  4, 6,  2,57, 51,39, 15,80,
  42,79, 50,72, 53, 2, 77,32, 42,38, 76,26, 57, 5, 78,84,
  89,52, 89,35, 50, 3, 23,77, 19,67, 78,36, 45,49, 79,39,
  22, 1, 22,48, 76,38, 25,41,  0,53, 31,88, 39,33, 50,76,
  51,38, 35,34,  7,60, 47,75, 70,67, 72, 2, 20,87, 70,70,
  32,58, 89,12, 33,30, 16,87, 84,76, 25,66, 59,50, 61,34,
  85,18, 85,85, 42,11, 13,30, 88,80, 30,35,  8,63,  5,62,
  36,27, 11,15, 85,67, 47,23, 60,50, 37,66, 10,70, 32,72,
  21, 1, 10,76, 10,51, 22,18, 58,13, 82,71, 25, 2, 71,14,
  74,77, 19,83,  2,77, 69,88, 63,11, 58, 1, 82,69, 79,54,
  88,56, 53,51, 80,87, 52, 8, 25,26, 44,80, 61,38, 68,56,
  15,47, 15,45, 27,46, 20,75, 45,51, 68,75, 64,25, 47,19,
  68,79, 74,41, 15, 8, 10,63, 32,31, 44,89, 69,48, 68,61,
  25,61, 49,17, 55,13,  3,56, 27,63, 22, 2, 86,65, 57,74,
  86,79, 53,40, 79,36,  8,64, 62, 0, 24,15, 63,55, 26, 6,
  26, 1, 34,41, 22,45, 57,89, 19,78, 12,23, 46,71, 25,74,
  71,82, 15,42, 20,48, 49,50, 77,89, 63, 2,  9,76, 81,13,
  15,56, 24,12,  2,28, 34,71, 60,37, 60,69, 32,88, 51,19,
  47, 0, 28,37, 25,89, 32, 5, 36,89, 61,84, 40,31, 73,64,
  50,63, 69,63, 64, 6, 40,78, 83, 2, 20,41, 65,21, 49,41,
  49,79, 26,44, 77,81, 61,88, 35,78, 16,34, 50,12, 13,32,
  88,62, 28,41,  3,33, 48,73, 59,75, 24,66, 63,70, 50,53,
  75,59, 48,78, 56,78, 47,79, 49,20,  5, 6, 68,16, 44,18,
  44,54,  0,18, 15,55, 33,59, 64,56, 89,17, 26,50, 69,34,
  43,59, 41, 7,  7,17, 34,31, 38,87, 28,63, 17,66, 82,66,
  54,50, 55,64, 10,69, 61,14, 44,66, 13,86, 34, 3, 72,52,
  36,38, 24,35, 48,31,  4,14, 77,66, 75,32,  8,33, 65, 8,
  70,46, 58,27, 17, 1,  1,86, 61,73, 48, 2, 64,61, 80,23,
  89, 0, 40, 8, 20,60, 66,49, 41,82, 21,71, 66,58, 48,40,
  30,41, 23,80, 34,65, 77,29,  3,22, 19,24, 78,46, 79,42,
  37,26, 75,25, 30,17, 85,73, 27,62, 26,29, 60,52, 12,52,
  14,70,  2,65, 79, 8, 44,26, 83,25, 41,30, 79,62, 19,43,
  28,20, 54,20, 58,42, 63,61, 64,32, 89, 3, 18,82, 56,60,
  28,56, 53,77,  9,10, 43,73, 28,27, 67,58, 49,61, 16,30,
  72, 4, 63,72, 15, 5,  8,56, 67,50, 13,61, 20,52, 89,67,
  45,15, 63,87, 64,18, 14,43, 20,29, 48,47, 72,33, 16,23,
  40,39, 47,32, 78, 8, 45,20, 77, 1, 35,33, 17,39, 53,75,
  12,83, 48, 9, 46,25,  7, 5, 51,52, 67,17, 20,28, 12,55,
  78,24, 58,75, 68,55, 81,65, 42,36, 24,31, 54,11, 64,20,
   3, 3, 86,14, 51,50, 25,58,  4,66, 10,74, 21, 7, 30,58,
  70,59, 29,74,  9, 3,  6, 6, 16,73, 56, 6, 15,48, 65, 3,
   2,14, 31,14, 25,70, 70,71, 79,27,  6,12, 46,46, 52, 4,
  10,10, 14,16, 41,76, 31,52, 14,53, 84,81, 46,14, 63,28,
   0,60, 24,75,  5,79, 52,51,  5,28, 14,86, 18,32, 45,65,
  86,21, 53,58, 46,12,  0,74, 76,21, 12,27, 13,44, 85,35,
  21,42, 66,62,  6, 0, 53,61,  7,71, 42,35, 30,89, 80, 0,
   3, 4, 82,11, 76,30, 39,41, 47,11, 87,10, 34,18, 65,65,
  73,54, 80,46, 68,43, 54,48, 37,52, 69,52, 54,30, 21,44,
  47,15, 54,22, 82,75, 20,83, 62,70, 48, 5, 36, 2, 50,50
};
const int p_90_15[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1744,
  /// Array of preferences
  34,69, 59,60,  4, 2, 65,63, 16,11, 55,78, 64,49, 13,74,
  68,47, 46,69, 33,80, 22,24, 71,88, 27,13, 33,73, 54,10,
  10,63, 40,80, 24,32,  0,80, 47,28, 88,33,  4, 5, 73,72,
  66,53, 47,26, 19, 5, 62, 5, 18,84, 14,23, 46,25, 16,46,
  58,27, 61, 6, 84,40, 76,88, 17,21, 48,53, 24,75, 89,65,
  14,61, 68,26, 49,11, 59,73, 63,78, 24,37, 60,26, 43,42,
  23,87, 69,19, 53,27, 20,47, 69, 4, 61,46, 12,33, 69,23,
  42,21, 55, 1, 52,72, 40,36, 24,56, 44,37, 66,38, 77,74,
  56,35, 77,87, 65,82, 85,68, 36, 3, 84,76, 30,26, 39,78,
   3,57, 43,49, 30,65, 77,55,  9,16, 82,62, 58,30, 32,38,
  84,81, 73,69, 38,61, 58, 9, 42,29, 39,68, 11,62, 28,26,
  42,45, 17,79,  5,23, 24, 7, 11,47, 80,52, 71,32, 55,11,
  86,61, 36,42, 72,46, 49,38, 40,44, 11,81, 81,53, 79,38,
  74,19, 48,58,  9,25,  2, 6, 66,45, 23,11, 19,25, 28,69,
  42,32, 15,72, 17, 2, 66,86, 58,32, 85,32, 66,10, 49,23,
  36,29, 81, 3,  1,52, 49, 3, 28,11, 66,28, 85,70, 53,84,
  47,85, 59,69, 17,73, 74, 1, 29,44, 84,89, 52,67, 57,32,
   1,79,  5,32, 29,22, 55,75,  5,66, 89,79, 26, 5, 62,29,
  53,74, 62, 7, 48,19, 55,34, 83,19, 79,43, 87,54,  3,53,
  71,37, 20,41, 69,79, 67,43,  5,53, 51,49, 25,66, 17,33,
  38,19, 66,34, 37,73, 45,49, 77,46, 77,17, 36,48,  0,46,
  70,47,  2,46, 65,52, 82,87, 50,70, 77,61, 33,66, 12,64,
  67,33, 33,65, 24,67, 66, 7, 69,60, 47,56, 83, 3, 60,57,
  21,70,  9,13, 80,34, 53,43, 86,37, 89,41, 45,35,  5,31,
   1,13, 84,63, 34,48, 22,52, 23, 6, 24,20, 49,75, 51,39,
  70,52, 81,32, 72,29, 17,51, 55,50, 14, 3, 15,88, 48,17,
  12,12, 17,18, 42,88,  2,14, 13,76, 82,69, 35,49, 60,36,
  72,89, 48,20, 15,46, 64,22,  9,41,  7,29, 81,26, 19, 4,
  47,43, 58,68, 25,85, 29,66, 22,19, 56,34, 12,74, 24,50,
  60,79,  5,25, 74,48, 86,55,  4,54, 75,41, 52,85, 84,62,
  71,73, 65,86, 72,45, 86,58, 32,68, 10,18, 39,17, 37,12,
   1, 2, 25,54,  9,62, 81,15,  8,71, 48,82, 10,87, 68,37,
  56,67, 50,89,  3,78, 75,29, 26,16, 17,84, 67,68,  3,42,
  60,25, 44,82, 89,32, 58,43, 59,14,  2, 9, 34,72, 89,52,
  68, 4, 15,24, 57, 9,  9, 8, 74,16, 22,73, 14,20, 18,26,
  86,39,  4,23, 80,27, 40,37, 54,33, 51,69, 33,35, 64,11,
  53,23, 28,84, 76,45, 15, 9, 82,23, 10,88, 20,22, 63,72,
  73,57, 55,47, 39,24,  7,64, 44,23,  7,51, 28,48, 65,47,
   0,39, 48,36, 51,81, 11,85,  1,73, 84,23, 58,86, 48, 3,
  63, 7, 52,32, 32, 1, 47,73, 57,26, 58,16, 54,65,  9,20,
  89,37, 76,38, 23,56, 32,63, 77,64, 74,14, 33,57, 60,40,
  13,65, 81,42, 29,57, 63, 5, 74,81, 34,36,  5,36, 10,30,
  17,61, 15,10,  8,74, 82, 4, 74,27, 86, 3, 78,31, 43,23,
  78,43, 13,17, 52, 5, 33,43, 80,20, 88,60, 16,81,  3,11,
  48,34, 59, 9, 63,12, 54,45, 12, 2, 72, 7, 36, 0, 12,27,
  55,26, 15,17,  5,73, 80, 0,  6,59, 38,37, 62,42, 41,83,
  58,72, 15,87, 43,15, 32,32, 75, 1, 11,23, 76,65, 15,15,
  28,61, 19,73, 40,47, 32,66, 14,60, 14,21, 45,81, 60,34,
  68,17, 73,89, 58,46, 63, 1, 73,86, 52,68, 10,49, 26,68,
  43,55,  2,10, 47,51, 28, 6, 39,87, 24,14, 56,53, 70,40,
  67,86, 43,22, 84,27, 20, 4, 27, 8, 31,42, 73,74, 38,10,
  87,35, 62,54, 73,79, 20,37, 38, 5, 37,61, 14,80, 39,65,
  16,73, 27,37, 70,77,  6,86, 30,56, 21,84, 71,39,  7,13,
  16, 7, 67,57, 55,71, 36,57,  2,50, 21,30, 19,74,  6,15,
  24,30, 81,38, 16,63, 68,44, 61,65, 76,77, 88,22, 55,41,
  20,65, 45,45,  0,42, 58, 8, 10,67, 74,22,  1,69, 87,74,
  50,10, 13,83, 68,10, 70, 1, 34,22, 13,63, 61,67, 29,46,
   3,36, 70,62, 34,11, 45,86, 39,41,  2,40, 54,69, 62,81,
  66,51, 36,59, 69,15, 23,39,  2,83, 27,32, 73,27,  6,23,
  23,16, 87,48,  6,53, 17,56, 42,77, 19,30, 45, 3, 22,13,
  69, 3, 20, 5, 63,71, 82,64, 63,77, 14, 5, 74,68, 63,30,
  65,20, 57,28, 37, 7, 78,52, 76,60, 29,35, 53,47, 52,89,
  23,29,  4,38, 62,43, 31, 5, 53,58, 24,23, 43,80, 26, 4,
   4,51, 28,64, 43,81, 21,60, 67,60, 86,69, 78, 0, 83,47,
  49,57,  1,87, 33,84,  5,35,  7,75, 67,49, 57,34, 11,11,
  38,82, 23,83, 63, 3, 31,23, 50,54, 82,65,  0,30, 51, 8,
  69,46, 82,63, 70, 0, 57,13, 46, 6, 17,78, 29,26, 39,77,
  83,62, 44,71, 71,14, 67,53, 16,75,  9,15, 67,58, 41, 8,
  18,86, 37,50, 79,13,  7,39, 36,10, 68,49, 49,29,  8,15,
  76,22, 70,72, 89,87, 11,16, 21,65, 50,72, 65,28, 82,59,
  81,43, 32,60, 72,44, 86,49, 78, 9, 79,88, 63,49, 44,85,
  51,89, 69,89, 67,61, 26, 2, 69,34, 81,62, 89,50, 51, 3,
  50,86, 25,73, 32,45, 82,50, 34,12, 61,45, 62,47, 67,16,
  68,73, 19,28,  0, 1, 81,87, 11,49, 14,54, 37,42,  6,34,
  88,30, 38, 1, 18, 3, 83,13,  9,63, 85,20, 56,50, 69,87,
  62,41, 52,86, 86,89, 13,37, 18,60, 28,60, 22, 0, 49,21,
  69,55, 71,18, 88,78, 56,49,  0,29, 19,36, 85,29, 37,30,
  88,14, 71,67, 14,44,  5, 0, 72,62, 47,17, 51,51, 19, 0,
   3,22, 39, 5, 68,31,  4,78, 73,29, 15, 3, 11,17, 32,24,
  83,39, 45, 0, 41, 6, 25,20, 34,61, 27,66, 68,83,  2,13,
  14,43, 72,38, 87,83, 36,74, 26, 9, 54,17, 39,44, 61,43,
  78,60, 75,46, 81,47, 78,89, 85,52, 11,74, 26,45,  8,56,
  26,27,  4,27, 13,80, 19, 8, 39,28, 80,61, 86,17, 28,77,
   3,31, 72,68, 45,77, 18,64, 43,31, 54,13, 84, 5, 51,68,
  58,36, 56, 4, 81,33, 73,19, 25,29, 52, 1, 29,83, 67,14,
  42,12, 79,19, 43,33, 86,78, 52,13, 11,89, 50,36, 33,30,
  69,41, 16, 5, 27,64, 41,68, 56,79, 43,79,  2, 4, 35,72,
  16,44, 69,59,  3,79,  3,73, 83,14, 50,84, 40,66, 38,69,
  19, 3, 12,14,  5,24, 34,67, 22,72, 71,52, 13,44,  6,26,
  60,74, 52,31, 51,33, 52,66, 49,25,  6, 1, 65,25, 87,44,
  36,51, 30,22, 23,43, 69,75, 44,48, 23,84, 34,39, 68,16,
  88,52, 43,83, 41,52,  3,49, 35,61,  4,60,  0,50, 10,33,
  32,85, 16,10, 78,88, 59,78, 44,62, 46,70, 32,75,  5,27,
  33,71, 74,75, 46,72, 77,82, 23,37, 86,73, 82,10, 25,70,
  80,71, 85,84, 70,51, 30,42, 31,52, 74,44, 36,19,  0,12,
  66,37, 28,85, 22,58, 35,57,  7,60, 80,55, 71,84, 14,18,
  87,26, 76,23, 30, 1,  2,43, 84, 7, 55,45, 78,13, 76,10,
  48,88, 45,55, 69,50,  1,15, 52,69, 51,76, 33, 1, 21, 9,
  22,17, 20,52, 37,77,  7, 5,  4, 3, 65,16, 65,38, 70,65,
  50, 7, 58,35,  2,25, 38,81, 83,29, 64, 9, 67,41, 30, 4,
  71,63, 30,40, 88,25, 32,65, 27,46, 18,43, 32,81,  5,60,
  38,43, 56,81, 18,30,  5,15, 26,71, 79,77,  4,55,  2,59,
  48,21, 21, 1, 58,38, 20,10, 70,41, 77,52, 50,76, 25, 7,
  13,50, 85,83, 45,19, 60,85, 87,43, 84, 4, 31,68, 32, 9,
   6,74, 43,84, 73,67, 16,69, 57,17, 87,30, 55,43, 16,65,
  15,68, 79,14, 52, 0, 54,57, 19,80,  1,86, 86,22, 30,15,
   1,39, 17,85, 21,56, 33,51, 43,10, 27,31, 62,75, 71, 6,
  15,26, 41,57, 28,53, 84,13, 32,64, 70,24, 70,18, 61,56,
  51,67, 17,89, 86,23, 88,56, 15,21, 62,72,  6,64, 84,53,
  64,16,  7, 0, 37,18, 81,64, 67,17, 65,83, 84,77, 59,48,
  83,84, 11, 3, 21,55, 37,23, 30,48, 72,54, 44,53, 30,74,
  16,40, 52, 6, 86,74, 59,11, 49, 5, 30,46, 65, 0,  0,89,
  26,21, 87,87, 60,38, 51,88, 68,18, 78,78, 88,39, 50,46,
  38,68, 72,43, 15,11, 59,42, 32,39,  8,72, 80,66, 84,47,
  19,37, 73,78, 47,46, 89,80, 19,52, 69,47, 62,23,  7,10,
  66,22, 39, 4, 18,73, 12,43, 84,78, 32,18, 46,88, 46,56,
  74,82, 46,73, 18,87, 60,24, 23,71, 48,38, 29,29,  5,42,
  84,28, 39,21, 65,77, 50,63, 33,78, 87,77,  5,55, 53,13,
  48,71, 81,35,  8,41, 10,29, 19,15, 36,61, 49,66, 41,49,
  46,18, 41,63, 37,57, 10,62, 68, 0, 73,80, 83,77, 73,41,
  81,68, 73,31, 48, 7, 84,74, 58, 0, 84,14, 33,23, 56, 8,
  82,46, 37,89, 53,15, 77,78, 52,57, 12,22, 50,85,  8, 2,
  41,64,  6, 4, 27,70,  0, 7, 22,79, 17,13, 67,87, 44,31,
  41,30, 26,22, 31,12, 45,30, 38,77,  1,72, 69,76, 25,47,
  42,59, 74,17,  0,77, 41,10, 48, 5,  3,46,  5,34, 65,32,
  70,21, 44, 5, 26,32, 83,27,  2,72, 38,13, 37,86, 12,13,
  80,13, 39,19, 64,86, 43,11, 82,44, 43,74, 67,22, 39,61,
   4,75,  8,66, 25,51, 85,23, 29,59, 43, 8, 23,19, 86,20,
  67,83, 26,64, 57,30, 70,50, 77,38, 11, 2, 46,68,  9, 2,
  52,33, 74,24, 69,43, 31,36, 81, 9, 14,15, 34,76, 33,34,
  77,39, 10,23,  6, 3, 86, 4, 59,25, 80,10, 77,88, 87,68,
  65,36, 77,54, 69,20, 35,69, 23, 4,  5,16, 77,84, 61,86,
  68,59, 41,53, 26,39, 40,23, 42,10, 52,88, 36,35, 59,62,
  34, 9, 40,69,  4,48, 41,85, 64,80, 87, 8,  7,12, 45,79,
  72,36,  7,17, 77,45, 64,67, 51,65, 81,18, 33,19, 17,64,
  57,86, 48,86,  7, 1, 12,57, 48,57, 26, 3,  7,52, 53,83,
  23,73, 58,19, 77,28, 73,44, 19,61, 52,35, 69, 6, 69,17,
  69,44, 49,12, 34,88, 23,74,  5,49, 28,34, 12,50, 19,10,
  27,63, 60,87, 37,63, 20,33, 85,24, 12,66, 21,80, 77,49,
  17,40,  3,80,  3,50, 88,38, 68,52, 41,24, 18,39, 33,18,
  72,81, 64,34, 80,57, 10,56, 70,12, 62,80, 55,30, 44,41,
  27,30,  7,69,  8,34, 11, 5,  1, 9, 70,87, 53,71, 81,84,
   3,62, 27,47, 19, 9, 27,36, 82,76,  9, 3, 30,72, 79, 3,
  71,55,  6,89, 28, 7,  2,66, 46,40, 46,24,  2,71,  8,40,
  48,70, 89,15, 69,86,  1,56, 35,16, 85,15,  6,50, 80,37,
  16,41, 57,59,  4,21, 22,43, 19,59, 79,69,  4,59, 88,17,
  42,18, 54,48, 16,54, 30,62, 66,40, 51,15, 71,25, 81,83,
  83,37, 66, 6, 74,36, 56,54, 65,64, 73,56,  1,47, 63,23,
  22,81, 60,12, 52,78, 38,67, 42, 2, 54,62, 37,15, 31,21,
  33, 7, 41,14, 86,31, 16,24, 39,39, 72,37, 71,69, 24,82,
  75,38, 76,15, 50,39, 34,83, 40,40, 82,37, 11,32, 64,48,
  51,25, 23,24, 48,11, 48,52, 58,33, 21,66, 87,25, 64,38,
  38,59, 19, 2, 35,41, 12,81, 53, 1, 11,48, 77,65, 47,39,
  85,18, 89,38, 47,12, 43,61, 45, 7,  4,82,  8, 4, 69,73,
  27,44, 87,88, 35,60,  9,69, 87,60, 45,89, 54, 5, 16,18,
  56,52,  1,25, 12,84, 74,84, 61, 9, 31,61,  8,47, 85, 3,
  11,53, 29,73, 52,76, 64,40, 62, 6, 13,15, 30,77, 30,85,
  31,50, 18,21, 44,38, 75,69, 13,41,  1,18, 63,63, 74,70,
  39,66, 25,16, 54,74, 32,71, 44,27, 65,78, 60,22, 13,10,
  41,65, 28,74, 17,70,  8,63, 88,29,  1,33, 18,68, 42,26,
  73,71,  5,67, 81,25, 65,58, 41,27, 29,34, 28,50, 23,61,
  51,38, 54,18, 83,52, 68,68, 13,59, 60,19, 52,14,  9,87,
  11,31, 14,69, 26,87, 21,25, 46,43, 69,32, 84,25, 26,40,
  30,60, 37,19, 78, 5, 16,23, 79,44, 11,21, 63,68, 75, 2,
   0,59,  5,76, 18,66,  1,22, 29,43, 10,14, 89, 1, 43,41,
  42,54, 89,28, 31,48, 42,60, 54,22, 78,51, 23,17, 23,46,
  77,33, 71,31, 84,34, 52,15, 50,49, 34,87,  4,72, 10,58,
  80,73, 36, 5, 37,72,  9,17,  2,56, 20,24, 89,43, 49, 8,
  40,38,  6, 2, 59,18, 61,51,  8,11, 78,21, 58,77, 67,38,
  51,52, 51,23, 79,20, 66,59, 63,39, 70,44, 52,18, 22,25,
   7,36, 14,28, 52,23, 31,70, 38,15, 27, 7, 13, 6,  5,62,
  66,17, 57, 8, 83,58, 22,48, 59,74, 46,16, 16,36, 79,39,
  31, 6, 88, 6,  4,62, 35,15, 89,45, 58,24,  7,26, 17,86,
  78,35, 17,65, 83,33, 72,51, 82,29, 45,83, 74,25, 25,35,
  24,68, 74,80, 21,50, 78,16, 30,33, 11,14, 63, 9, 23,81,
  63,13,  8,65, 87,34, 27,15,  4, 0, 73,36, 57,15,  6,22,
  64,36, 31,88, 46,71, 54,20, 11,58, 14,45,  0,11, 85,10,
  30,81, 53,20, 37,51, 54,29, 84,39, 77,16, 82,85, 60,30,
  14,53, 22,85, 29,33, 62,27, 78,46, 23,79, 32,46,  9,72,
  10,82, 28,66, 16,17, 26, 7, 14,74, 20,77, 12,82, 74,56,
  65, 2, 36,54, 50,62, 52,46, 27,82, 63,54, 37,60, 53,73,
  76,85, 78, 2, 46,41, 12, 3, 49,88, 21,79, 82,19, 63,86,
  63,53, 31,75, 31,27, 60,10, 82, 9, 10,64, 29,10, 57,20,
  81,48, 78,48, 18,13, 26,75, 24,33,  4,86,  0,33, 41, 4,
  42,51, 24,15, 55,23, 33, 0,  5,47, 38,29, 63,34, 40,29,
  76,17, 67,45, 82,22, 33,38,  9,36, 23,67, 57,57, 67,29,
  80,59, 48,50, 35,68, 53,77, 76,46, 52, 2, 16,83, 88,55,
  63,87, 30,55, 23,30, 20,38, 56,85, 72,52, 26,48, 16,20,
  11,70, 71,41, 23,65, 51,60, 63,59, 77,70, 60,48, 82,52,
  44,69, 27,79, 31,47, 35,28, 10,81, 74,78, 28,76,  8,31,
  70,70, 76,36, 65,71, 32,41,  9,42, 13,43, 15,47, 13,34,
   7,88, 62,48, 38,88, 46,78, 16,50, 20, 7, 21,47, 78,27,
  28,39, 86,19,  5,85, 75,32,  5,48, 20,68, 78,65, 86,24,
  73, 5, 86,18, 51,70, 45,50, 76,63, 77, 1, 76,68, 22,60,
  78,34, 15,77, 25,81, 26,44, 26,33, 36,69, 67, 1,  3,48,
  61, 0, 11,82,  9,71, 61,26, 75,87,  6,42, 27,56, 55,40,
  12,69, 47,35,  3,32, 46, 9, 67,18,  6,61, 69, 8, 57,83,
  86,52, 34,45, 61,52, 53,72, 30,11, 80,29, 87,16, 78,79,
  81,41, 11,35, 47,30, 44,43, 39,29, 61,61, 41,56, 38, 8,
  66,50,  6,60, 36,52, 24,86,  5,88, 62,86, 59,72, 11,65,
  26,20,  0,63, 12,38, 64,54, 19,45, 64,35, 67,44, 65,49,
  73, 2, 30,19, 83,82, 26,29, 69,49, 70,54, 29,82, 49,87,
  50,23, 19,55, 17,68,  4,58, 72,71, 54,54, 48,77,  9,84,
  88,18, 38,31, 37, 6, 65, 9,  5,61, 85,77, 69,37, 27,84,
  46, 1, 56,19, 55,88, 58,87,  6,24, 75,83, 29,55, 60,21,
  45,51, 79, 5, 37,47, 28,22,  5,17, 39,40, 49,85, 41,62,
  80,89, 69, 7,  7,62, 36,27, 14,52, 60,46, 42,11, 24,11,
  26,24, 57,24,  1,16, 64,23, 33,69, 54,50, 12,89, 54,60,
  71,62, 63,20, 40, 4, 83,20, 55,72, 19,51, 66,43, 76,56,
   0,66,  4,25, 70,49, 80, 1, 17, 4, 57,25,  3,23, 72,67,
  74,40, 10,60, 68,46, 87,27, 20,15, 53,22,  1,54, 33,50,
  10,28, 17, 5, 32,26, 84,58, 80,74, 59,41, 17, 0, 36, 4,
   7,54, 82,33, 88,71, 46,85, 37,67, 23,41, 89,42, 84,71,
  25,24, 49,83, 13,57, 30,54, 65,73, 75,79, 23,89, 87,23,
  72,55,  8,82,  1,36, 56,20,  2,75, 43,25, 39,64, 29,38,
  48,31, 82,41, 54,59, 12,73, 30, 0, 20,69, 64,83, 24,29,
  22,10, 75,76, 41,25, 76,59, 38, 4,  8, 7, 59, 5, 64,64,
  74,64, 88,31, 17,32, 24,73, 40,20, 89,11, 64,12,  9,60
};
const int p_90_16[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  162,
  /// Array of preferences
  57,61, 32,49, 89,25,  7,39, 18,25, 26,86, 35,67, 53,59,
  17,46, 85,56, 63,74, 44,56, 82,52, 69,22, 35,31,  3,41,
  60,36, 24, 4, 78,58, 76,50, 25,81, 13,54, 17, 0, 75,80,
  35,46, 75,44, 48, 2, 10,72, 27,17, 59,73, 76,66,  4,84,
  18,67, 45,24, 30,13, 65,27, 83,16, 30,52, 66,20, 66,14,
  36,65, 55,71, 75,76, 82,34, 26,59, 49,31,  0, 3, 61,42,
  16,70,  2,39, 20,23, 60,57, 39,16, 36,33,  6, 7, 18,42,
  43,68, 47,83, 50,13, 28,33, 17,78, 50,30, 28,16, 55, 8,
  23,56, 75,32, 82,72, 25,68, 44,25,  0,30, 72,22, 55,34,
  61,80, 12,70, 24,46,  9,31, 11,63, 83,40, 24, 8, 59, 6,
  38,49, 84,47, 18,55, 72,69, 44,33, 79,42,  1,73, 69,77,
  89,57,  0,73, 23, 1, 50,38, 46,46, 25,22, 20,65, 60,64,
  37,10,  3,71, 81,75,  2,77, 11,40,  0,59, 36,68, 67,42,
  85,17, 36,32, 60, 8, 89,58, 32,33, 32, 3, 39,85,  4,41,
  79,17, 80,24, 14,13,  3,58,  5,11, 17,67, 72,62, 48,53,
  20,48, 63,56, 23,60, 55,66, 68,72, 74,36, 52,67, 10,41,
  57,22, 77, 8, 37,17, 56,14, 50,61, 34,88, 11,55, 47, 9,
  65,81,  6,69, 61, 0, 70,48, 42,31, 34,61, 26,76, 32,37,
  22,75, 11,22, 40,21, 61,24, 37,11, 75, 6, 68,21, 27,22,
   5,84, 62,83, 71,84, 72,12, 60,67, 82, 6,  8,45, 23,59,
  32,50, 19,38
};
const int p_90_17[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  1584,
  /// Array of preferences
  72,44, 77,37, 71,64, 55,48, 84,79, 45,52, 83,89, 76,16,
  82,55, 69,22, 68,27, 37,50,  6,79, 25,24, 33,54, 21,10,
  26,81, 34,55,  3,78, 54,24,  4,79, 56, 0, 11,87, 43,43,
  15,14, 39,83,  4,47, 41,76, 83,44, 84,33, 16,65, 54,84,
  87,24, 51,68, 70,35, 19,39, 15,21, 88,77, 36,87,  2,40,
  25,37, 23,67, 19,63, 35,17, 31, 6, 69,59, 38,50, 78,65,
  23,68, 63,40, 60,56, 79,29,  6, 6, 65,71,  4,34, 73,35,
  22,69, 75,12, 78,73, 89,22, 46,41, 53,35, 39,31, 17,87,
  35, 0,  8,23, 83, 8, 35,52, 26,87, 40,83, 64,13, 36, 0,
  79,70, 78,82, 21,87, 27,26, 41,45, 77,50, 17,15, 32,23,
  44,61, 10,42, 22,18, 81,15, 59,71, 27,25, 60,54, 13,42,
  82,60, 25,75, 51,27,  8,38, 38,39, 41,62, 46,37, 19, 9,
  48,34, 32,31, 44,27, 65,33, 76,18, 37,58, 45,87, 53,51,
  63,86, 88,35, 30,29, 34,63, 78,64, 14,53, 83,61, 77,53,
  67,30, 44,86,  4,70,  8,18, 54,19, 86, 1, 18,36, 36,57,
  73,57, 78,72, 59,65, 36,85, 31,83, 63,79, 76,86, 19,42,
  69,78,  4,44, 61,81, 55,30, 80,18, 57,17, 29,50, 70,18,
  49, 3, 24,49, 17,66, 42, 9, 83,57, 58,38, 15, 6, 55,40,
  86,28,  2,39, 27,59, 19,46, 56,52, 39,10, 79,20, 65,86,
  43,85, 84,67, 16,18, 83,64, 85,41, 28,84, 75,13, 58,87,
  15,24, 77,54, 29,44, 45,25, 69,10, 21,25, 17,88, 71,48,
  11,65, 33,42,  0,66, 28,21, 49,71, 82,11, 35,83, 51,57,
  22,26, 80,89,  7,78, 18,56, 81,63, 58,45, 80,62, 28,49,
  70,73, 84,27, 42,79, 86,39, 81,25, 32,36,  2,23, 41,39,
   2,84, 85,74, 31,24, 10,25, 61,46, 27,56, 41,58, 68,73,
  59, 0, 57,63, 57,27, 16,33, 86,40, 61,44, 15,46, 82,53,
  41,77, 70,66, 82,66, 52,38, 51,89, 51,74, 73, 6, 62,75,
  87,36, 40,76, 71,22, 26,20, 55,84, 82,58, 55,25, 20, 7,
  76, 9, 46,47, 55,18, 70,65, 65,79, 88,32, 16,29, 47,50,
   7,52, 60,88,  7,68, 13,71, 52,47, 42,24, 13,72,  5,63,
  76,58, 27,88,  3, 2, 25, 2, 31,54,  6,69, 19,76,  6,84,
  73,72, 72,21, 11,42,  3,89, 70,45, 50,43, 58,62, 73,40,
  42, 6, 78,50, 49,29, 29,11, 20,27, 71,51,  0, 5, 37,10,
  41,38, 54,70, 67,84, 79, 0,  0,84, 65,38, 55,82, 50,24,
  40,88, 89,38, 26,72,  3,43, 71,43, 40,41, 11,57, 88,57,
  55,70, 70,11, 10,79, 59,23,  4,12, 26,33, 27,78, 25,53,
  29,23, 16,79, 72,67, 62,50, 46,73, 57,71, 74,66, 66,60,
  13,68, 61,82, 79,13, 10,56,  3,10,  2,16, 43,44, 84,45,
  47,66, 22,25,  8,15,  3,73, 45,21, 72,42, 26,74, 63, 0,
  89,77, 65,51, 40,17, 67,28, 47,33, 49,51, 32,64, 60,69,
  82,44, 16,52, 83,32,  4,52, 72,58, 86,85, 24,66,  1,80,
  69,77, 55,74, 27,28, 12,34, 45,48, 24,67, 54,60, 87,72,
  28,35, 70,32, 72, 3, 40,67, 46,69, 89,32, 55,69, 58,36,
  40,45, 62,62, 74,86, 21,30, 64,89, 69,11, 25,42, 78,23,
  78,52,  2,88,  9,11, 52,80, 81,36, 32, 6, 88,17, 13,12,
   1,17, 26,19, 55,39,  5,11, 82,73, 53,47, 33,83,  7,21,
  41,10, 81,56, 17,48, 43,17, 24,42, 67,26,  2,18, 38,85,
  31,78, 44,88, 80,10,  3,72, 24,78, 53,63, 20, 5, 83,13,
  19,33, 13,49, 39,20, 37,72, 19, 1, 34,41, 35,85, 30,69,
  76,88,  2,17, 63,43, 29,82, 85,56, 49,80, 38,30, 40,48,
   0, 6, 51,23, 23,15, 62,89,  2,71,  0,69, 31,35, 88,33,
  32, 8, 84, 5, 31,59, 54,69, 17,40, 46,87, 41, 0, 52,34,
  41,32, 81,45, 45,44, 87, 9, 82,15,  2,32, 84,41,  3,79,
  38,27, 39,57, 58,54,  1, 9, 47,42, 37,35,  2,85, 71,37,
  47, 7,  0,57, 72,22, 38,22, 17,33,  3,86, 56,20, 82,87,
   6,71, 24,43, 38,76, 46,38, 26,60, 36,30, 58,67, 46,77,
  76,70, 12,27, 34,26,  0,74, 64,80, 28,20,  2,21, 56,64,
  29,25, 88,39,  7, 2,  3,50, 80,72, 19,55, 67, 7, 55,13,
  46, 3, 69,41, 36,24, 17,38, 54,81, 27,20, 41,51, 36,56,
  70,52,  8,86, 63,15, 88,83,  4,25, 80,53, 37, 6, 36, 7,
  88, 1, 25,63, 79,10, 54,71, 45,54,  7,84, 29, 2, 36, 5,
  89,76, 29,35, 49,17, 74,59, 16,12, 12,15, 64, 0, 16,16,
  63,34, 24,23, 60,25, 12, 9, 74,45, 53, 2, 61,15, 45, 4,
  15,82, 39,58, 65, 0, 62,11, 59,87, 18,55, 36,60, 16,81,
  13, 5, 85,80, 39,84, 78,12, 33, 6, 53, 3, 85,23,  7,34,
  37,74, 43,68, 76,57, 69,28,  5,48, 43,87, 27,65, 38,63,
  39,28, 56,66, 48,52, 52,25, 68,50, 62,36, 42,47, 13,64,
  88,78, 45,20, 62,10, 83,51, 68,58, 24,57, 76,45, 53,52,
  70,44,  9, 8, 78,67, 87,38, 25,86, 44,65, 89, 7,  8,79,
  60,50, 13,13, 50, 7,  9,42, 59,78, 58,49, 50,68, 71,66,
  59,44,  6,75, 33,79, 70,20, 26, 5, 88,41,  6,13, 59,42,
   1,57, 38,82, 56,10, 24, 2, 10,41, 69,50, 51,14, 12,47,
  67,42, 80,23,  8,49, 26,24, 68,61,  6,28, 26,44,  9,68,
  10,76,  3, 7,  6,88, 16,32, 84, 9, 88,13, 28,85, 16,68,
  53,48,  8,54,  1,45, 36,38, 33, 5, 48,68, 37,78, 15,60,
  63,63, 64,22, 33,11, 52,67, 28,45, 53,66, 71, 7, 44,46,
  34, 9, 23,34, 38,84, 25,49, 55,37, 57,60, 75, 9, 37,45,
  58,40, 58,88, 52,48,  1,16, 30,35, 69,81, 85, 5, 72,23,
  21,27, 13,16, 48,79, 49,20, 71,65, 44,78, 81,59, 27,36,
  25,62, 78,26, 10,69, 64,32,  1,69, 81, 5, 27, 6, 12,79,
  11,62, 16,83, 52,75, 62,44, 64,72, 10,63, 19, 8, 43,34,
  42,16, 43,65, 43,12, 27,38, 39,24, 83,33, 58,13, 48,38,
  76,82, 27, 7, 64, 3, 23,24, 42, 3, 87,65, 56,14, 75,85,
  29,70, 42,22, 29,51, 18,34, 71,52, 61,68,  5,67, 66,11,
   8,25, 75,68, 78,87, 19, 7,  7, 6, 15,17, 37,84, 37,52,
  10,17,  1,30, 10,60, 81,81, 44,85, 32,20, 26,37, 75,17,
   3,65, 50,78, 20,46, 75,52, 69,48, 75,88, 83,81, 64,27,
  19,51, 19,20, 16,22, 72,43, 21,60, 61,36, 37,51, 76,77,
  27,45, 72,40, 53,81, 32,46, 35,88, 11,20,  1,35, 26,27,
  65,30, 61,52, 80,24,  9,65, 87,10, 10, 2, 48,35, 26,15,
  62,79, 85,66, 13,76, 60,82, 22,56,  1,70, 88,11, 65,50,
  89,15, 38,69,  6,38,  5,20, 25,74, 39, 8, 21,71, 17, 3,
  36,26, 13, 9, 17,89, 59,13, 41,41, 61,60, 74,47, 69,39,
  46,48, 77, 7, 50,55, 78,62, 34,38, 35,31,  0,63, 65, 4,
   5,61, 24,51,  5,16, 84,71, 56,60, 26, 0, 68,10, 38,61,
  13,74,  8, 0, 11,69, 16,39, 72,59,  2,63,  3,51,  7, 8,
  61,11, 66,10, 29, 8, 73,73, 41,49, 43,47, 82,19,  4, 5,
  28,13, 11,31, 83,67, 59,74, 71,39, 48,80, 75,72, 65,46,
  84,55, 33,74, 43,66, 42,33, 75,83, 50,80, 55,27, 29,37,
  46,66, 41,67, 29,65, 59,12, 74,36, 68,63,  8,53, 73,84,
  51,37,  7,22, 63,38, 60,33, 23,73, 32,48, 74,75, 21,17,
  29,86, 66,79, 80,84, 31,58, 51,29, 86, 8,  7,51, 80,50,
  27,70, 75,80, 84,72, 21,62, 62,56, 43,11, 50, 4, 11,33,
  17,83, 88,14, 53,68, 31,18, 39,67, 77,87,  6,45, 49, 8,
  73,86, 51,38,  9,49, 22,66,  4,23, 13,25, 31,80, 75,19,
  83,79, 50,83, 72,48, 72,52, 33,29, 57,12, 51,71,  1,64,
  41,61, 36,68, 30,67, 11,35,  9,26, 13,10, 70,78, 66,34,
  25,52, 75,66, 72, 9, 54, 2, 30, 6, 54,10, 46,65, 26,49,
  17,67, 83,12, 64, 9,  6,24, 20,56, 86,31, 43,54, 68,67,
  12, 7, 24,44, 63,75, 73,16, 45,51, 20,10,  6,17, 36,83,
  38,60, 15, 0, 19,78, 80,63, 21,21, 57,67, 77,57, 48,51,
  58,47, 74,53, 12,22, 72,15, 82,24, 24,69, 48,85, 56,25,
  51,55, 71,56, 16,61, 21,46, 87,50, 69,17, 38,32, 19,64,
  38, 7, 48,59, 63,59, 75,18, 27,31, 84,34, 15,44, 15,15,
   1,82, 17,16, 79,35, 39,87, 25,57,  2,29, 21,55, 10, 7,
  33,20, 73,77, 79,24, 65,29, 71,88, 37,64,  1,72, 78,28,
  14,25, 16,84, 24,70, 50,66, 78,51, 77,85, 20, 4, 87,69,
  55,20, 59,72, 42,42, 43,61, 41,47,  0,87, 81,55,  5, 0,
  89, 6, 27,62, 73,18, 33,32, 56,68, 69,83, 18,20, 45,80,
  57,36, 80,38, 42,85, 38,78, 60,21, 68,78, 73,10, 79,31,
  40,39, 30,89, 37,67, 59,40, 55,67, 12,81, 76,23, 20,22,
  22, 6, 27,39, 34,76,  4,31, 22,46, 66, 0, 43,69, 18,53,
   7,49, 28,36,  6, 2, 27,17, 74,83, 27,50,  6,76, 68,52,
   5,59, 13,51, 72,34, 37,27, 35,24, 74,80, 74,68, 31, 7,
  89,11, 15,13, 50, 3, 20, 9,  0,35, 24,74, 71,29, 42,61,
  17,18, 85,85, 28,44, 65,45, 22,39, 75,24,  1,62,  7,33,
  28,87,  5,57, 46,32, 28,64, 49,62, 29,56, 46,45, 12,36,
  84, 7, 47,35,  9,40, 56,50, 42,60, 14,44, 85, 2, 24,55,
  50,15, 60,72, 16,47, 68,60,  7, 1, 44, 9, 83,36, 22,60,
  25,27, 82,27, 33, 7,  7,79, 26,88, 61,43, 15, 3, 33,30,
  28,72, 44,47, 32,61, 61,21, 85,68, 45,83,  5,53, 64,64,
  35,16, 82,74, 18,61, 24, 6, 24,85, 78,14, 17,80, 54, 9,
  47,18, 78,11, 75, 0,  2,52, 47,53, 49,32,  2, 2, 19, 4,
   3,31, 53,21, 51,59,  9,85, 88,24, 63,19, 70,33, 36,18,
  82,23, 27,81, 20,65, 83,26, 17,71, 76,83, 78,69, 63,35,
  69,18, 13,83,  8,73, 65,56, 70,37, 45,55, 33,58, 51,85,
  79,81, 26,12, 83, 1, 43,32,  7,18, 11,39, 82,17, 25,70,
  29,59, 54, 8, 37,28,  8,81,  8,87, 31,21, 78,30, 46, 7,
  55,43, 18,22, 83,77, 65,10, 66,39, 80, 4, 84,21, 19,53,
  55,45,  1,87,  7,70, 47,62, 54,42, 32,87, 35,61, 18,31,
   6,37, 81,23, 53,50, 76,87, 18,14, 18,82, 38,67, 63,21,
  74,70, 45, 0,  2,58, 29,17, 28,46, 89,30, 13,37,  9,83,
  14,54, 63,31, 20,67, 40,78, 78,43, 51,83, 17, 8, 86,34,
  42,51, 59,36, 52,78,  4,86,  8, 4,  3,70,  8,17, 31,72,
  38,45, 12,35,  8, 8, 75,61, 70, 4, 26,22, 48, 1, 36,22,
  10,55, 82,79, 74,88,  7,55, 81,60, 87,56, 43,39, 53,54,
  68,82, 72,89,  4,17, 69,42, 50,52, 82,29, 36,52, 89,75,
  52,76, 14, 2, 77, 0, 41,19, 40,75, 33, 4, 51, 1,  5,32,
  65, 8, 39,32, 12,73, 25,76, 42,78,  3,38, 72,14, 56,79,
  60,68, 68, 0, 60,62, 20, 0, 72, 1, 18, 5, 76, 2, 69,32,
  27,14, 32,22, 38, 8, 74,39,  8,32, 36,41,  4,88, 71,30,
  37,49,  7,77, 29,38, 50,47, 58,51, 76,63, 35,89,  7,83,
  35,19, 56,83, 67,62, 11,70, 79,64, 13,53, 81,75,  1,52,
  87,71, 89,31, 47,88, 75, 3,  6,49, 28,30, 18,48, 32,45,
  18,27, 25,64, 44,76, 81,19, 11,89, 10, 5, 34, 8, 81,37,
  79,14, 87, 6, 44, 2, 50,86, 76,39, 89,58, 51,21, 21,28,
  86,77, 45,81, 53,80, 27,15, 22,73,  8,58, 47,61, 11,66,
  40,60, 22,82, 85,58, 22,51, 10,58, 87,68, 67,29, 74,56,
   2,12, 25,22, 33,57, 14,23, 17,44, 80,83, 12,74, 64,12,
  42,80,  2,51, 56,15, 73,71, 73,28, 73,64, 46,80, 68,45,
  36,69, 20, 8,  8,48, 68,31, 63,37,  5,75, 67,75, 46,84,
  84,86, 78,34,  6,34, 49, 6, 15,20, 11,75, 69,40, 66,71,
  39,38, 84,43, 43,74, 81,12, 38,11, 66,86,  1,42, 83,84,
  82, 0, 77,12, 76,12, 54, 1, 17,29, 29,54, 20,40, 18,30,
  53,85, 60,28, 35,11, 30,12, 32,80, 46,70, 82,69, 69,63,
  73,38, 50,28, 55,50, 14,40, 30,84, 52,61, 31,65, 47,79,
  45,45, 34,62, 54,17, 89,37, 31,61,  3,49, 29,88, 42,88,
  31,40, 24,84,  2, 8, 39,22, 83,24, 78,33, 89,21, 37,61,
  66,48, 19,43, 37,20, 13,14, 15,85,  1,85, 62,84, 37,79,
  47,67, 68,87, 50,64, 68,43,  3,39, 66,62, 13,75,  3,66,
  53,36, 35,78, 11,23, 58,27, 77,39, 49,63, 27,30, 56,70,
  72, 2, 77,48, 73, 0, 10,52, 36,67, 41,64, 54, 5,  5,60,
  84,81, 31,68, 54,68, 66,66, 12,12, 75,57, 23,82, 12, 3,
  34, 5, 34,36, 27,74, 11,37, 25,14,  7,45, 68,18, 33,89,
  49,25, 65,70, 89,19, 71, 4, 60,75, 80,11, 31,60, 23,38,
  38,89, 86, 2, 66,65, 72,10, 19,23, 55,58, 19,52, 40,54,
  74,72, 51,13, 37,55,  1,21, 19,26, 46,52, 10,51, 57, 1,
  17,77, 11, 8, 83, 9, 22,84, 43,78,  4,37, 84,66, 87,15,
  33, 8, 31,53, 47,13, 50,13,  2,89,  6,15, 70,21, 22,37,
  68,64, 41,33, 22,62, 28, 7, 29,89, 45,60, 30,57, 68,14,
  73,59, 80, 0, 89,73, 17,64, 72,27, 30,30, 68,20, 82,38,
  84, 1, 26,78, 71,69, 71,49, 57,79, 67,10, 56,73, 85,73,
  20,37, 44,30, 58,19, 31, 1, 60,53, 85,87, 13,38, 59,56,
   3,37, 54,25, 41,36, 42, 0, 73,20, 44,70, 41,59, 54,53,
   0,16, 53,31, 33, 3, 79,79,  8,61, 71,72, 82,51, 73,80,
  36,21,  4,89, 23,37, 32,40,  4,32, 59, 2, 16,64, 31,15,
  36, 1, 65,24, 87,18, 45,38, 84,12, 25, 5, 36,29, 65,47,
  27,19, 63,67, 30,32, 33,85, 64,77,  1,89, 45, 7, 33,67,
  14,35, 29,39, 88,12, 67,78, 76,54, 34,15, 10, 1, 78,58,
  79,40,  6,64, 22, 4, 32,68, 20,84, 24,28,  0,83, 75,58,
  17,42, 17,55, 11,53, 20,59,  8,33, 78,70, 44,40,  4,73,
   7,50, 16,55,  9,67, 51,86, 27,11, 20,85, 23,76, 34,14,
  54,73, 74,52, 37,85, 24, 9, 14,56,  9,29, 27,85, 38, 2,
  22,22, 35,37, 24,39, 43,37, 37, 3, 75,69, 74,62, 74, 4,
  43,20, 42,49,  9,89, 23,13, 39,39, 15,48, 59,38, 11,41,
  68,17, 74, 3, 16,51, 48,20, 43,62, 17,75, 40,12, 29,21,
  68,71, 86,73,  5,73, 33,47, 70,56, 64,63, 21,76, 87,22,
  16,88, 88,37, 43,55, 79,77, 49,21, 44,13, 29,42, 57,69
};
const int p_90_18[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  989,
  /// Array of preferences
  21,63, 37,16, 45,86, 67, 9, 55,88, 60,32, 73,83, 63,22,
   7,48, 77,51,  4,26, 79,35, 59,53, 51,62, 56,71, 76,59,
  77,58,  4,33, 25, 1,  3,53, 45,33,  1,58, 22,88, 64, 1,
  30,73, 31,36, 56,20, 33,88, 53,62, 58,11, 57,57, 58,21,
  56,55, 37,75,  2, 5,  9, 4, 35,20, 58,60, 71,14, 19,29,
  22, 8, 77,20,  9,60, 67,18, 27,43, 64,41, 61,80, 68,16,
  11,33, 87,25, 28,46, 79,82, 43, 7,  0,23, 67,19, 39,60,
  22,61, 48,60, 18,49, 54, 4, 59,12, 60,22, 81,15, 45,55,
  58,84, 48,26,  0,14, 45,89, 32,83, 21,40, 80,65, 70,42,
  82,24, 58,26, 35,58, 69,87, 41,44, 84, 9,  3,24, 14,67,
   8,10, 75,49, 62,81, 21,29, 79,31, 27, 4, 11,10, 18,63,
  11,46, 58,43, 46,62, 84,24, 59,63, 14,37,  1,52, 28,89,
  44,47, 40,37, 37,38, 31,54,  5,59, 42,59, 62,64, 13,44,
  13,81,  6,68,  9,74, 58,81, 86,27,  5,14,  1,78, 23,27,
  19,61, 26,38, 50, 1, 72,53, 24,85, 13,56, 51,58, 80,24,
  41,54, 69,40,  0,69, 23, 9, 39, 6, 39,38, 35,62, 43,48,
  48,36, 35,38, 29,20, 59,26, 13, 8, 42,26, 77, 3, 16,13,
  60,50, 72,19,  2,23, 82,54, 25,18, 47, 3, 13,64, 70,20,
  31,51, 85,34, 40,68, 44,75, 48,76, 24,18, 60,39, 60,69,
   0,49, 69,79, 39,55, 32,76, 25,87, 64,29, 73,14, 85,33,
  63,40, 66,19, 45,38, 61,22, 51, 0, 62,66, 83,69, 34,33,
  54,12, 14,19,  2,43, 77,76, 76,84, 89,84, 76,13, 66,67,
  36,17, 61,76, 46,21, 47, 6, 48,75, 79,46, 15,44, 45,85,
  77,40, 13,71, 38, 1, 41,62, 68,82, 24,16,  6,38, 75, 6,
  71,53,  3,19, 89,58, 65,69, 66, 7, 43,76, 25,78, 45,60,
  10,15, 59,58, 41,60, 70,58, 79, 8, 67,72, 43,88, 43,57,
  85,58, 16,61, 68,25, 47,27, 39,86, 54,59, 27,89, 51,18,
   9,73, 31,46, 37,81, 76,40, 40,74, 62,82,  0,47, 53,18,
   1, 5, 54,43, 27,75, 75,14, 89,45, 17,10, 14,25, 72,81,
  87,32, 60,51, 45,59,  7,22, 59,14, 76,32, 65,89, 69,38,
  80, 3, 27,84, 13,89, 64,40, 14,76, 21,68, 71,12, 14,84,
  42,87, 29,15, 40,84, 31,18, 76,70, 67,76, 47,53, 33,66,
  70,59, 64, 9, 67,49,  5,76, 44,34, 86,38, 57,22,  5,89,
  81,58, 42,75, 35,87, 47,30, 67,44,  1,66, 73,29,  9,89,
  46,42, 61,31, 26,14, 43,85,  1,65, 23,10, 64,34, 82,64,
  41,81, 50,14, 10,42, 74,83,  3,61, 53,31, 71,75, 22,76,
   3, 8, 19,20, 77,72, 38,41,  5,19, 77,36, 67,10, 12,86,
  25,38, 68,27, 46,82, 11, 9,  8,19, 26,51,  2,62, 21,13,
  55,45, 81,79,  3,13, 33,38, 68,29, 11,79, 78,44, 46,36,
   5,23,  8,72, 22,78, 36,75, 37,27, 87,76,  5,42, 61,63,
  56,68, 39,35,  1,62, 39,18, 15,46, 30, 4, 39, 5, 77,65,
  48,33, 39,13, 69,55, 33,14, 87,45, 70,16, 28,32, 69, 7,
  23,26, 32,56, 56,21, 17,43, 11,60, 77,39, 26,61, 60,88,
  61,45, 23,56, 74, 9, 22,19, 12,47, 50,29, 65,73, 43,29,
   0,81, 13,79, 73, 3,  3,56, 41,82, 47,64, 35,31, 25,69,
  62,11, 18, 7, 24,81, 66,82, 26,78,  6,62, 80,78, 40,11,
  84,18, 34,68, 32,30, 67,48, 77,48, 57,33, 11, 7,  1,24,
  11,80,  7,40, 79,39, 23,42, 38,49, 46, 5, 53,86, 32,71,
  46,11, 22,66, 28,30, 77,80, 23,67, 21,38, 59,24, 69,41,
  20,82, 71,47, 32,15, 46,32, 48,79, 82,20, 18,59, 77,16,
  29,66,  9,69, 57,37, 75,26, 69,51, 50,46,  4, 9,  9,64,
  59,23, 49,33, 22,38, 73,23, 45, 2, 62,29, 15,41, 32,13,
  43,44, 62,75, 33,41, 25,67, 63,33, 59,22, 27,76, 16,72,
  33,50, 58,36, 30, 7, 24,31, 79,70, 53,81, 86,63,  4,28,
  39,85, 42,42, 61,25, 86,32,  1,33, 41,17, 13,49, 34,17,
  77, 0, 84, 8, 25,53, 20,23, 11,35, 38,40, 14,46, 29,30,
  46,46, 81, 5, 39,20, 88,50, 45,68, 36,63, 81,47,  4,31,
  56,82, 73,26, 40,47, 33, 7, 88,25, 69,37, 78,50, 46,55,
  45,53, 87,59, 38,80,  5,74, 88,61, 39,11, 25,11, 37,62,
  75,34, 19,39, 81,82,  5,79, 36,20, 71,70, 21,23, 58,83,
  54,33, 58,20, 69,46, 14,44, 47,35,  0,71, 42,69, 53,79,
  40,57, 21,44, 84,17, 61,29, 44,83, 54,22,  1,30, 13,87,
  25, 9, 22,21, 19,15, 10,25, 64, 4, 66,80, 34,50, 50,33,
  13,50,  0, 8, 67,62, 30,18,  2,21, 71, 1,  0,51, 76,87,
   2,65, 79,59, 56,47, 80, 2, 60,81, 75,44, 46,53, 50,24,
  47,78, 21,26, 61,14, 82,47, 17,88, 86,77, 38,15, 69,29,
  40,34, 89,15, 37,10, 70,60, 28,54, 25,30, 89,23,  8,21,
  23,68, 17,59, 18,14, 83,89, 15,62, 27,44, 16,88, 70, 7,
  10,74,  9,81, 55,42, 17,37, 81,33, 73, 1, 42,65, 10,71,
  74,54, 36,42, 63,35,  4,81,  9,39, 23,86, 12,85, 42,49,
  68, 0, 40,46, 23,71, 39,65, 43,81, 25,76, 48,49, 34, 6,
  17,56, 56,23, 82,50, 78,41, 81,59, 23,55,  6, 2, 56,37,
  31, 6, 71,23, 36,70,  9, 2, 18, 6, 63,36, 57, 9, 46,52,
  62,24, 49,23, 52,43,  7,66, 19,63,  2,60, 76,65, 35,42,
  47,46, 55,11, 76,28,  3, 6, 48,35, 36,87, 53,12, 81,17,
  54,60, 37,53, 59, 3, 13, 6, 49,24,  2,57, 17,14, 20,12,
  57,34, 27,65, 70,15, 12, 2, 74,73, 88,82, 78,76, 86,74,
  63,54, 70,70, 62,37,  7,87, 35,32, 29,78, 47,26, 61,59,
  38, 8, 32,57, 28,69, 24,84, 30,58, 79,12, 41,73, 26,18,
   4,43,  7,64, 78,61, 20,52, 47,68, 75,13, 71,54, 31,63,
  42,66, 62,68,  8, 3, 87,62, 41,66,  9,56, 46,68, 41,26,
  40,55,  0,79,  9,82,  9,67, 11,69, 35,13, 56,60, 57, 5,
  51,10, 67,63, 83,22, 43,67, 33,53, 31,84, 74,17, 22, 5,
   2,28,  7,39, 26, 7, 57,35, 25,56, 16,27, 53,10, 14,13,
  17, 3, 48, 0, 18,42, 41,25, 39,44, 10,26, 10,10, 39, 2,
  12, 0, 84,38, 89,40, 50,55, 61,12, 42,61, 82,77,  7,15,
  27,54, 61,60, 32,22, 69,54, 75, 5, 85,36, 29,10, 87,52,
  59,65, 39,27,  2,73, 51,44,  6,26, 39,21,  1,26, 30,61,
  58, 5, 22,53, 51,63, 43,70, 34,62, 21, 6, 47,79, 43, 2,
   9,79, 56,15, 78,39,  8,15, 64,42, 78,20, 57,76, 83,50,
  32,58,  4,34, 14, 8, 26,39, 60,42, 78,66, 58,89, 73,30,
  23,29, 62,39,  3,43, 41,72, 28,45, 31,81,  7,13, 86, 4,
  16,55, 81, 2,  6,11, 41,10,  5, 3, 34,49, 47, 9, 41,80,
  10,28, 43,77, 49,16, 27,16, 11,27,  6,43,  2,32, 12,22,
  27,40, 47, 4, 28,18, 24,33, 83,40, 59,61, 84,28, 81,56,
  88,86, 50,58, 12,75, 85,26, 79,24, 58,28, 13, 3, 80,81,
  70,85, 57,28, 59, 8,  2,45, 74,11, 20,62, 33,54, 28,55,
  50,84, 27,71, 62,34,  7,78, 25, 8, 75,81, 82,73,  7,61,
  16,42, 85,39, 79,34, 48,56,  7,54, 18,25, 34, 1, 78,56,
   2,24, 40,51, 53,25, 20,47, 16,82, 20,40, 88,75, 58,85,
  69,25, 63,11,  6,57, 64,76,  9,54, 62,73,  5,41, 20,13,
  11,37, 29,13, 39, 8, 10,17, 81,55, 61,19, 15,50, 37,14,
  57,14, 85,70, 71,27, 44,60, 20,80,  2, 3, 46,28, 26,50,
  49,89, 39,63, 84,46, 56,30, 56,85, 31,67, 66,73, 32,55,
  37,57, 23,72, 53,75,  5,78, 74,42, 69,48, 75, 1,  0,78,
  29,83, 32,66, 46,56, 37,21,  5,34, 21,76, 12,40, 39,36,
  76,62, 12,57, 21,30, 14, 1, 69,62, 49,70, 48,70, 73, 2,
  83,70, 59,56, 31, 3, 41,31, 56,77,  8,49, 33,20, 20,67,
  46,27, 82,26, 41,77, 60, 8, 67,47, 67,11, 61,15, 24,37,
  48,80, 40,38, 75,19,  2,88, 73,36, 61,21, 25,41, 24,14,
  33,44, 59, 0, 37,68, 41,68, 82,55, 29,23, 87,12, 59,15,
  50,52, 39,72, 30,74, 54,32, 76,50, 54,36, 88,31, 79,36,
  68,59, 41, 7,  9,27, 28,37, 16,66, 13,53, 50, 6, 28,64,
  85,41, 34,84, 42,25, 46,26, 13,15, 11,61, 52,29, 89, 4,
  82,29, 49,26, 51, 4,  4,89, 27,79, 65,72, 59,66, 80, 4,
  60,74, 45,76, 26,12, 48,21, 19, 3, 20,16, 40,30, 36,29,
  38,21, 65,83,  0,22, 29,52,  9,19, 27,46, 85, 1, 28,31,
  83, 8, 61,70, 39,49, 62, 3, 36, 5,  8,67, 49,36, 39,62,
  53,15, 76,10, 15,71, 86,86, 34,12, 88,29, 17,80, 69,13,
  13,86, 89,34, 69,81, 72,27, 32, 9,  4,20, 40,31,  7,24,
  47,38, 23, 2, 65,65, 54,31, 51,38, 32,41, 65,16, 36,52,
  52,87, 50,57, 15,31, 58,38, 38, 9, 65,86, 32,84, 13,28,
  18,86, 60,10, 50,70, 24, 8, 39,61, 82, 8, 46, 3, 18,41,
  82,87, 75, 2, 78,14, 45,52, 42,85
};
const int p_90_19[] = {
  /// Number of people on picture
  90,
  /// Number of preferences
  381,
  /// Array of preferences
  31,10, 70,31, 73,49,  2,12, 87,63, 61,85, 47,88, 70,73,
  44,39, 26,65, 87,46,  1,17, 89,71, 33,71, 76, 2, 20,74,
  15,83, 72,78, 29,60,  4,15, 74,13, 49,31, 73, 6, 82,11,
  11,79, 15,15, 15,86, 57,41, 41,15, 68,73,  2,52, 61,60,
  56,71, 60,62, 69,87, 12,60, 72,10, 46,68, 17,88, 70,64,
  30,47,  3,71, 19,45, 10,15, 79,39,  9,49, 67,43,  4,59,
  71,51, 62,37, 79,34, 58,48, 43,11, 30,83, 18,77, 62,10,
  73,44, 37,10, 28,59, 17,76, 25,45, 41,55, 67,88, 13,56,
  25,60, 34,33, 25,35, 61,42, 47, 8, 78, 1, 13,18, 76,82,
  60,21, 45,18, 35,64, 67,61, 83,20, 72,42, 25,44, 25,24,
  77,15, 20,31,  0,28, 73,36, 77,41, 58,80, 16,69, 84,68,
  50,12, 85,87, 66,15, 72,57, 29,70, 78,80, 65,67, 11,54,
  38,55, 26,72, 12,44, 30, 2, 45, 9, 71,14,  0,35, 39,41,
  57,85, 50,80, 27,29, 64,24, 50,25, 58,27,  2,64, 78,57,
  62,15,  6, 3,  7,21, 19,55, 57,17, 25,32, 22,38, 49,42,
  22,66, 55,28, 71,11, 69,60, 30, 9, 88,64, 50,81, 78,66,
  87,57, 70,40, 30,53, 58,84, 75, 1,  6, 7, 38,66, 21,39,
  23,67, 64,57, 44,28, 54,20, 86,21, 15,79, 89,61,  0,23,
  69,18, 60,80,  9,53, 68,55, 64,49, 22, 2, 70,60, 46,77,
  24,82, 72,17, 75, 4, 20,80, 84,61, 38,77, 68,87, 65,77,
  41,76, 44,26, 59,13, 46, 3, 53,50,  4,44, 15,64, 15,24,
   5,46, 84,60, 43,39, 46,72, 35,66, 42, 9, 17,38, 14,33,
  87,34, 64,66, 81,36, 65,82,  1,48,  1,53, 26,73, 61,11,
  59,43,  4,22,  6,53, 12, 7, 70,17, 68, 7,  3, 7, 79,84,
   9,21, 36,32, 38,20, 52,12,  7,16, 13,12,  9,69, 14,13,
  27, 0, 18,39, 43,46, 52,34, 56,41, 45,17, 85,70, 68,22,
  57,82, 40, 8, 53, 4, 14,21, 64,72, 22,29, 32,83, 25,63,
  33,89, 47,11, 29,57, 70,85, 47,65,  3,37, 83,41, 78,34,
  45,33, 35,59, 37,32, 79,57, 58,19, 70,75,  1,34, 63,48,
  71,80, 83,60, 39,89, 68,75, 61,47, 18,44, 87,47, 62,74,
  51,83,  0,32, 49,74, 52,73,  3,69,  9,59, 43,78, 76, 4,
  88,67, 86,66, 28,17, 18,59, 89,52, 21,30, 85,46, 18,16,
  86,80, 13,83, 64,22, 16,48, 13,54, 64,88, 16,49, 33,48,
  79,59, 72,58, 36,89, 41,61, 28,71, 17,84, 10,67, 17,61,
  65, 1, 17,58, 74,50, 30,34, 11,80,  6,22, 47,19, 78,74,
  80,70, 74,38, 68,10, 34,53, 21,60, 59,21,  9,47, 13,61,
  63,68,  5,61, 62,62, 56,24, 71,53, 41, 4, 88,30, 49,41,
   0,19, 52,61, 64, 3, 19,32, 12,85, 62, 5, 13,13, 65,52,
  33,15, 21,86, 34,40, 83,15, 68, 5, 53,75, 17,41, 39,70,
  77,19, 32,81, 16,81, 85,81, 42,60, 42,32, 89,78, 53,69,
  68,11, 21, 4, 46,19, 13,59, 63,30, 56, 7, 56,79,  1,66,
   6,84, 78, 3, 88,73, 43,14, 36,76, 48,87, 72,66, 19,67,
   4,71, 43,29,  6,16, 62,55, 36,84, 57,15,  9,16, 67,83,
  33,82, 74,64, 34,21, 33,76, 46,25, 60,87, 51,75, 25,38,
  33,23, 64,10, 46,53, 12,31, 40,81, 34,10, 25, 3, 84,20,
  19,75, 39,16,  2,17, 18,86, 68,63, 85,11, 45,35, 42,15,
  73,78,  6,79, 10,21, 21,32, 85,69, 30,56, 68, 0, 37, 6,
  54,84, 19,52, 30,27, 51,70, 85,47
};

const int p_100_0[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  319,
  /// Array of preferences
  50,41, 84, 8, 22,70, 14,10, 15,84, 35,69, 61,17, 16,29,
  28,40, 11,77, 52,63, 66,95, 15,54,  5,66, 39,85, 93,39,
  18,71, 20,87, 60,99, 77,77, 35,23,  3,36, 76, 5, 20,12,
  75,91, 83,71, 48,18, 52,91, 90,95, 19,87, 80,54, 62,56,
  30,30, 77, 6, 96, 6, 20,37, 62,78, 90,76,  7,69, 40,92,
  89,61, 69, 8, 74,70,  7, 7, 33,31, 19,79, 21,66,  9,64,
  71,57, 93,46, 68,12,  2,94, 11,29, 28,83, 10,39, 70,69,
  16,63, 60, 9, 61,36, 37,53, 92,41,  6,87, 35,26, 29,59,
  51,58, 12,65,  6,85, 13,78, 89,55, 72,42, 56,80,  4,25,
  21,32, 57,52, 40, 6, 15,57, 72,14, 32,21, 41, 7, 77,83,
  83,29, 48,35, 94,71, 12,53, 41,84, 37,31, 22,57, 73,50,
  25,72, 48,29, 82,14, 99,24, 11,83, 89,26, 23, 8, 60,44,
  40,46, 13, 7, 73, 9, 45,26, 65,88, 70,29, 74,43, 72,10,
  25,20, 28,51,  5,29, 39,36, 42,48, 68,64, 25, 5, 50,79,
  71,84, 17,18, 54, 4, 71,22, 30,69, 42, 2, 21,74,  7,80,
  60,40, 19,41, 16,62, 56,83, 12,41, 68,74, 96,73, 52,25,
  59,52, 45,98, 15,95, 79,14,  1,83, 65,68, 73,15, 43, 7,
  82,92, 56,72, 63,17, 61,81, 18,93, 40,55, 68,40, 21,50,
  18,14, 53,25, 25,26, 38,24, 26, 6, 73,72, 61,19, 55,15,
  37,11, 67, 9, 59,68, 90,63, 56,64,  6,18, 60,18, 39,27,
  53,61, 37,37, 41,77, 24,12, 32,12,  5, 1, 62,71, 25,44,
  34, 9, 45,97, 34,39,  0,10, 62,76, 87,11, 66,36, 30,11,
  57,82, 87,29, 13,31, 46,50, 56, 2, 11,90, 22,23,  9,59,
  57, 4, 75,29, 98,11, 32,65, 30,91, 46,31, 43,71, 15,64,
  51,80, 76, 1, 53,33,  6, 2, 91,58, 51,96, 82,33, 41,72,
  70,70, 58,45, 37,21, 11,49, 18,43, 27,40, 28,76, 18, 6,
  19,73, 87,64,  5,50, 44,79, 90,20, 56,91, 96,93,  4,45,
  77,74, 43,55,  8,40, 91,70, 92,72, 52,69, 40,74, 32,80,
  54,38, 69,79, 12,58, 13,32,  6,68,  3,28, 62, 9, 77,57,
  15, 2, 47,15, 29,63, 96,81, 83,22, 34,80, 16,32, 92,61,
  96, 9,  9,34, 46,56, 44,82, 67,47, 21,98, 15,28, 66,18,
  63,19, 77,96, 47,64, 86,69, 46,99, 65,30, 23,52, 28,12,
  82,86, 72,39, 25,63, 28,56, 50,87, 56,51, 18,56, 61, 7,
  15,11, 45,84, 22,65,  3,52, 49,41,  6,32, 61,54, 38,17,
  94,91, 55,60,  9,54, 59,90, 79,42, 34,38, 91,24, 69,17,
  85,94, 51,11, 23,65, 92,60,  1,62, 84,86, 46,74, 19,36,
  49,50, 33,71, 12,85, 31,85,  5,79, 57, 9, 43,27, 85,65,
  10,36, 18, 1, 50,96,  9,38, 44,70, 95,76, 47,77, 60,54,
  43,23, 28,28, 10,98, 22,76, 95,12, 67,23,  6,26, 23,56,
  50,67, 25,83, 13, 4, 44,69, 83,64, 49,67, 35,31
};
const int p_100_1[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  282,
  /// Array of preferences
  33,20, 17,76, 17,85, 96,65, 28,57, 10,83, 42,82,  6,73,
  87,60, 79,89, 78,62, 41,32, 35,81, 17,53, 47,31, 83,59,
  86, 9, 98,51,  9,66, 97,25, 78,56, 51,86, 66,53, 76,56,
  82,21, 25, 4, 40,31, 31,87, 73,16, 61,58, 97,49, 56,18,
  38,83, 17,59, 28,59, 69,30, 74,89, 30,49, 45, 8, 68,15,
  16, 0, 67,24, 78,71, 28,96, 15,20, 52, 7, 20,11, 80,93,
  33,67, 83, 4, 67,17, 56,36,  8,82, 60,12, 70,57, 90,50,
  23,19, 33,65, 27,32, 82,76, 99,90, 31,16, 95, 5, 56,80,
  68,49, 86,16,  0,66, 90,83, 27,98, 73, 8, 99,93, 69,58,
  40,40, 82,34, 58,18, 91,39,  9,45,  8,86,  3,68, 95,57,
  44,71, 25,38, 69,35, 84,65, 34,19, 47,51, 69, 8, 64,17,
  22,41, 87,48,  5,65, 62,80,  7,70,  7,56, 53,18, 87,24,
  84,61, 90,71, 55,59, 81,40, 39,19, 77,13, 27,62, 18,35,
  38, 1, 49,97, 47,30,  6,48, 71,70, 27,68, 81,33,  3,40,
  58,31, 21,96, 91,74, 36,16, 98,32, 53,44, 32,10,  0, 3,
  40,42, 71,29, 61,54, 47, 8, 79,32, 36,94, 69,21, 21,52,
  30,76, 17,70, 32,61, 96,11, 23,10, 23,48, 98,45, 14,87,
  97,48, 39, 8,  5,72,  5,18,  8,95, 44,39, 72,87,  5,15,
  13,93, 97,18, 74,73, 93, 1, 41,50, 32, 9, 44,48, 42,65,
  70,69, 82,57, 42,56, 53,79, 50,14, 66,68,  1,77,  0,70,
  63,10,  0,10, 71,84, 55,48, 23,18, 28,58, 44,30,  3,66,
  95,17, 56,15,  9,33, 31,55, 89, 3, 23,86, 18,38,  5,27,
  48,36, 36,61, 98,95, 74, 2, 26,47, 14,42, 55,44, 63,95,
  39,58, 87,10, 87,44,  4,89, 81, 1, 52,81, 78,82, 13,31,
  62,84, 60, 4, 61,75, 15,87, 42,90, 88,83,  6,32, 69,23,
   4,64, 91,68, 38,90, 22,28, 77,15, 57, 7, 20,99, 87,57,
  23,45, 29, 1, 93,99, 72,14, 24,50, 36,57, 74,64, 65,71,
  56,60, 82,55, 28,13, 19,55, 32,89, 80,18, 57,33, 72,35,
   7,49, 21,23, 47,69, 60,49,  2,66, 18,59, 91,48, 45,70,
  69,42, 65,36,  1,86, 58,60,  5,92, 94,76, 87,45,  9,99,
  43,38, 58,61,  2,72,  4, 9, 74,78, 73,47, 26,11, 51,34,
  93,18, 14,34, 27,71, 72,20, 33,39, 63,48, 45,98, 87,58,
  26,18, 29, 2, 97,81, 66,13, 80,86, 59, 7,  6, 6, 70,62,
  92,89, 93,97, 98,60, 23,61, 90,86, 38,84, 86,99, 85, 3,
  15,89, 80,66, 28,73, 21,31, 74,74, 12,95, 43,73, 76,12,
  90,80,  0,74
};
const int p_100_2[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  2367,
  /// Array of preferences
  34,40, 54,24,  9,10, 27,67, 86,64,  6,48, 12,60, 94,58,
  58,90, 84,42, 30,18, 50, 7, 70, 3, 35,91, 36,45, 62,80,
   6,77, 17,50,  3,83, 20,37, 71,77,  7,69, 34,79, 27,53,
   1, 6, 32,92, 40,61, 14,71, 78,59, 28,32, 36,67, 25,21,
   3,95, 94,78, 46,68, 59,98, 75,34, 68,55, 38,48,  3,28,
  68, 1, 58,22, 57,33, 59,56, 87,59, 47, 7, 25,32, 10,98,
  12,87, 69, 1, 19,94, 60,53, 76,74, 31,92, 67,58, 36,85,
  30, 6,  4,37, 88,23, 65, 9, 58,31, 34,29, 95,79, 79,25,
  56,64, 42, 7, 28,46, 99,10, 57,52, 24,39, 29,71, 48, 6,
   6,34, 50,78, 59,29, 12, 6, 37,60, 85,62, 44,51, 97,50,
  71,79, 66,41,  6,75, 77,26,  5,54, 41,25, 14,21, 14, 8,
  46,23, 13,31, 46,30, 17, 7, 12,36, 47,24, 30,49, 25,42,
  79,98, 18, 9, 85,12, 12,69, 87,92, 45,44, 73,48, 13,40,
  60,89, 42,91, 45,32,  6,89, 57,10, 43,34, 57,57, 37,23,
  70,88,  5,71, 35,49, 80,25, 82, 2, 77,86,  5,14, 77,76,
  10,63, 25,68, 52,56, 42,76, 96,86, 57,34, 53,64, 84,50,
  99,32,  7, 5, 27,28,  8,95, 42,57, 12,75, 82,96, 76,25,
  65,82, 17,21, 26,49, 49,93, 48,46, 70,55, 78,67, 89,35,
  36, 4, 49,48, 78, 9, 91,13, 82,69, 88,16, 82,46, 56,34,
  35,21, 32,47, 32,66, 96,89,  6,18, 26,33, 77,14,  0,81,
  15,94, 83,79, 59,53,  6,36, 19,89, 35,47, 68,65,  6,87,
  43,29, 73,43, 62,61, 70,12, 92,95, 45,65, 87,12, 96,23,
  52,66, 77,46, 66,34, 24,95, 23,49, 16,39, 99,16,  7,90,
  65,57, 80,90, 91,56, 32, 0, 84,59, 37,93, 17,41,  4,77,
  94,41, 16,56, 36,37, 83,14, 38,77, 28,20, 56,30, 77, 1,
  84,72, 81,14, 88,41, 97,99, 28,64, 24,16, 41,53, 15,97,
  86,91,  6,67, 24,98, 55,33, 89,20, 88,86, 31,61, 94, 3,
  15,47, 19, 1, 55,67, 99,69, 72,31, 56,27,  9,69, 17,19,
  37,10, 49,46, 64,99, 30,24, 21, 2, 32,27,  0,24, 20,50,
  48,72, 67,43, 14,41, 47, 6, 59,80, 27, 3, 10,31, 56,54,
   1,96, 63,20, 65,28, 12,59, 76,30, 63,57, 60,88, 44,47,
  28,86, 79, 7, 90,30, 91,89, 37,43, 84,15, 26,50, 65,27,
  33,94, 23,78, 37,63, 59,83, 47, 5, 16,94, 64, 5, 34,20,
  37,48, 60,49, 83,34, 84,87, 85,72, 95,98, 35, 9, 35,23,
  56,46, 77, 4, 82,61, 24,21, 60,35, 81,43, 12,11, 48, 5,
  49,63, 22,92, 89,34, 17,51, 30,86, 69,65, 20,68, 92,99,
  29,23, 46,22, 89,50, 93,97, 62,19,  5,16, 39,33, 19,12,
  99,48, 92,49, 78,61, 96,80, 89,77, 39,97, 21,27,  6,12,
  92, 7, 71,68, 39,87, 98,20, 34,77, 48,47, 13,44, 20,43,
  46, 0, 25, 9, 81,22, 71,37, 54,48, 89,11, 68,64, 10,43,
  40,96, 39,19, 92, 2, 19, 4,  7,17, 15,74,  3,51, 28,94,
  75,95, 13,67, 82,55, 66,45, 60,57, 85,92, 12,38, 52,23,
  24,31, 29,63, 65,88, 32,79, 41,91, 34,39, 63,80, 79,34,
   2,92, 72,43, 42,66, 72, 7, 23,42, 60,40, 88,85, 42, 1,
  15,44,  1,27, 81,78, 79,59,  4,86, 40,69, 56,99, 50,48,
  71,65, 53, 7, 68, 8, 25,43, 41,75, 18,19,  1,95, 94,50,
  10,37, 98,63, 84,36, 14,60, 19,48, 91,53, 21,82,  9,75,
  80,57, 98,45,  2,65,  6,27, 36,51, 12,20, 56,40, 83,97,
   7,54, 36,71, 95,35, 10, 3, 39,39, 52,15, 14,66, 18,42,
  66,64, 75,50, 18,97, 33, 9, 49,44, 23,86, 58,10, 75,16,
  55,63, 86,42, 50,88, 82,64, 78,41, 20,48, 10,92, 71,10,
  81,80, 32,46,  7,78, 76,16, 50,52, 11,77, 10,96, 66,24,
  11,42, 29,87, 28,84, 77, 5, 53,13, 58,98,  5,25, 54,47,
  62,64, 21,93, 44,16, 58,30, 47,12, 21,83, 89,82, 18,49,
  24,45, 87,25, 99,96, 87,96, 29,67, 95,22, 86,30, 66,16,
  94,54, 33,48, 67,82, 37,15, 89,78, 63,95, 54, 4, 15, 8,
  66, 0, 12,79, 89,69, 29,20, 12,93,  9,46, 96,69, 91,23,
  47,91, 58, 4, 85,93, 18,84, 67,91, 61,16, 97,61, 96, 5,
  56,10, 53,65, 20,79, 28,17,  3,36, 42,25, 32,67, 13,30,
  28,21, 82,87, 42,45, 21, 0,  9,67, 31, 7, 72,64,  8,62,
  40,23, 59, 6, 20,69, 68,90, 31, 0, 11,33, 74,11, 16,34,
  27,84, 42,22, 47,44, 74,51,  1,42, 51,99, 93,73, 44,83,
  38,71, 96,34, 47,33, 38,69, 30,41, 66,78, 34,95, 38,34,
  22,81, 93,79, 85, 3, 67,71, 82,85, 77,10, 45, 3, 97,76,
  66,72, 24,15, 79,30,  3,13, 55,32, 31,46, 66,44, 31,79,
  54,86, 74,89,  0,65, 28, 8, 34,50, 89,24, 98, 1, 66,63,
  75, 5, 52,53, 28,65, 64, 9, 81,62, 81,28, 25,65, 17,56,
   4,72, 76,57, 36,98, 29,50,  5,88, 65,77, 18,16, 27,75,
  70,80, 65, 3, 81,90, 19,55,  4,24,  3,47, 98,77, 58,40,
  68,81, 76,12, 58,28, 81,91, 55,28, 63, 4, 21,72, 57,72,
  87,87,  3,10, 84,34, 77,41, 33,96, 62, 5, 40,94, 82,94,
  14, 4, 17,37, 79,27, 45,63, 73,71, 48,98, 75,49, 44,53,
   0,54, 77,16, 21,98, 43,30, 28,50, 52,45, 48,16, 32,83,
  58,81, 63,30, 48,42, 78, 8, 76,94, 44,27, 59,15,  0,70,
  48,76, 65,61, 49,10, 71,11, 23,68, 57,95, 30,89, 30,19,
   5,44, 25,72,  8,78, 10,13, 31,50, 48,54, 22,58, 64,73,
  62, 7, 75,15, 21,66, 80, 1, 70, 8,  1,25, 99,34, 91,24,
  22,59, 33,76, 45,22,  7,32, 56,93, 25,11, 23,82, 57, 9,
  30,39, 87,26, 12,26, 33,60, 29,96, 92,22, 68,48, 69,89,
  26,84, 14,82, 20,81, 44,72,  8,15, 99,71, 72,46, 62,39,
   2,20, 15,67, 64,30, 76,67, 85,37, 21,78, 13,37, 91,63,
  92,11, 73,37, 83,88, 51,43, 46,88, 14,48, 62, 0, 22,37,
  24,37, 44,26, 90,68, 74,67, 99,35, 40, 5, 84,86,  7,68,
  28,87, 17,70, 75,64, 69, 0, 39, 7, 70,25,  9,51, 58,38,
  27,92, 54,83, 22,79, 66,47, 12,46, 19,16, 12,42, 84,58,
  70,65, 35, 8, 71, 6, 92,21, 72,68,  1,93,  4, 1, 19,21,
   3, 2, 27,99, 63,53, 77,58, 42,50, 83,15, 64,78, 21,36,
  36,78, 36,63, 43,49, 39,50, 81,15,  6,37, 83,60, 30,12,
  44, 4, 91,58, 33,54, 33,32, 88,11, 65,47, 78,22, 92, 8,
  19, 8, 21,62, 71,12, 18,27, 69,11, 23,38, 56,56, 91,54,
  99,90, 89,39, 10,18, 14,72, 92,40,  2,98, 72,48, 17,46,
  71,25,  2,83, 89,45, 65,34, 43,63, 30,78, 78, 4, 55,85,
  82,63, 65,98, 24,10, 87, 3,  5,46, 32,69, 97,60, 85,15,
  72,59, 86,73, 55,17, 87,76, 21, 9, 32,59, 39,51, 14,93,
  27,96, 31,40,  3,87, 15,82, 41,96,  1,43, 59,85, 22,82,
  38,70, 58,95, 95,19, 41,52, 89,28, 61,23, 45,29, 90,84,
  59,96, 76,62, 90,79, 56,87, 72,75, 18,47, 42,15, 42,37,
  72,55,  9,18, 66,60, 45, 6, 87,10, 61, 9, 14,98, 93,28,
  26,57, 45,70, 89,62, 16,67, 18,93, 94,91, 16, 1, 15,46,
  96,32,  0,43, 33,30, 41,92, 80,89, 37,85, 38,65,  0,99,
  88,28,  5,92, 34, 5,  4,95, 89,19, 32, 6, 95,75, 99,84,
  72,66, 25,52, 31,64, 72,95, 18,67, 34, 3, 20,84, 46,14,
  23,28, 67,77,  1,51, 51,40, 10,59, 39, 6, 44,55, 73,84,
  95,76, 16,29, 85,80, 87,45, 20,55, 30,77, 39,61, 11,47,
  99,14, 22,55, 67,84, 32,55, 80,91, 44,66, 40,52, 71,16,
  98,34, 95,58, 91,90, 95,55, 93,29, 45,21, 85,56, 62,38,
  25,22, 75, 0, 54,70,  8,36,  5,23,  9,17, 42,31, 26,95,
  51, 1, 58,65, 92,50, 14,15, 23,33, 99,49,  2,91,  3,62,
  15,68, 11,97, 83,43, 18,52, 25, 4, 56,21, 95,62, 31,42,
  51,27, 64,38, 29, 4, 96,36, 30,94, 36, 0, 87,43, 59, 0,
  90,82, 69,90, 77,17, 22,27, 24,48, 95,68, 78,52, 30,93,
  83,12,  6,14, 73,38, 75,77, 62,71, 66,81, 63,38, 36,20,
  35,32, 84,39, 65, 6, 71,86, 70,48, 76,64, 90,61, 96,52,
  19,45, 60, 4, 46,76, 93,33, 42, 2, 29,41,  0,76, 77,24,
  59,31, 50, 5, 32,64, 73,72, 42,62, 44,33, 75,79, 41,54,
  81,57, 91,65,  2,72, 89,33,  7, 4, 39,86,  9,97, 70,81,
  97,33, 13,94, 47,17, 35,13,  9,89, 55, 6, 32,89, 24,11,
   6,94, 74,35, 30,52, 57,45, 86,85, 43,71, 60,43, 66,32,
  90,39, 56,15, 68,11, 88,71, 74,50, 57,70, 81,44, 11,57,
  50,56, 92,71, 39,35, 33,70, 53,76, 84, 3, 73,12,  8,69,
  78,89, 99, 6, 79,32, 14,24, 23,99, 34, 1, 10,99, 17,43,
  97,41,  3,22, 56,24, 42,23, 32,82, 88,61, 94,66, 71,56,
   4,14, 31,12, 66,59, 56,73, 47,18, 31,69, 92,47, 64,51,
  72,20, 76,95, 96,18, 10,84, 11,27, 46,31, 26, 4, 41,43,
  58,97, 27,39, 93,84, 25,29, 58,18,  4,50, 72,85, 26,58,
  77,69,  7, 3, 57,67, 95,97, 79,65, 67,98, 86,36, 25,14,
   2, 6, 52, 7, 11, 4, 66,96,  8,96, 28,23, 40,16, 64,64,
  15,41, 49,57,  5,85, 10,41, 67,26, 50,38, 23,59,  2,17,
   7,36, 69,82, 73,60, 12,67, 78,51, 59,63, 44,37, 22,30,
   1,40, 98,86, 85,98, 38,56, 13,87, 38,25, 18,95, 65,37,
  45,98, 55,39, 49,13, 70,18, 80,61, 55,18, 91,18, 29,78,
  65,91, 67,36, 32,17, 42,94, 75,32, 54,41, 52, 5, 21,64,
  72,39, 63, 0, 41,70, 83,44,  6,79, 63,87, 40,26, 54,54,
  80,99, 17, 5, 20,86, 63,36, 44, 1, 47,65, 33,83, 68,63,
  61,67, 90, 5, 15,62, 74,48, 45,11, 39,24, 23,20, 46,25,
  66,55, 20,78,  2,76, 70,38, 14, 6, 99,50, 37, 3, 89,52,
  40,86,  1,46, 29,70, 36,83, 28, 1, 15,13,  3,54,  3,21,
  63, 9, 64,20, 33, 0, 21,65, 65, 5, 25,95,  5,42, 53,12,
  85,65, 20,94, 94,23, 30,16, 70,42, 53,27,  5, 7,  4,79,
  31,93,  2,64, 20,21, 38,95, 32,98, 53,19, 71, 7, 99,36,
  97,23, 48,86, 81,61, 97,63, 95,80, 67,81, 94,39, 26,30,
  80, 6, 99, 0, 67, 9, 63,60, 18,85, 70,60, 96,79,  4,31,
  25,88, 85, 8, 47,81, 94,46, 74, 9, 43,42, 37,38, 51,32,
   6,78, 90,50, 35,64, 92,27, 20,96, 13, 5, 44,67, 71,38,
  55,29, 75,89, 37,82, 55,44, 57,42,  0,45, 84,70, 40,74,
  37,98, 36,81, 78,88,  2,56, 88, 9, 35,24, 75,54, 23,72,
  30, 1, 83,82, 94,75, 49,28, 45,97, 62,91, 73,68, 78,85,
  62,60, 80,81, 50,35,  4,76, 71,66, 83, 4, 30,10, 55,99,
  23,56, 35,33, 32,40, 80,96, 18,56, 68,80, 82,73, 73,64,
  32,11, 26,35, 90,23, 63,45,  8,72, 71,73, 24, 4, 43,41,
  57,27, 15,91, 12,25, 83,25, 38,99,  3,30, 25,39,  3, 3,
  64,77, 83,51, 85,53, 17,79, 24,65, 58, 0, 78, 2, 69,58,
  12,32, 65,74, 54, 8, 78,19, 28,22, 70,97, 83,92, 87,64,
  87,99,  2, 5, 21,71, 71,51, 77,52, 95,50, 16,57, 32,39,
  55, 3, 80,32, 53,73, 24,35, 56,76, 98,55, 93,48, 69,21,
  60,90, 17,82, 81,56, 47,56, 33,15, 97,83, 23,19, 99,12,
  50,83, 28, 7, 63,11, 18,15, 31,66, 78, 6, 74,70, 46,18,
  32,71, 72,57, 62,21, 99,87, 25,78, 69,59, 14,20, 65,44,
  68,35, 23,74, 24,43, 13,93, 54,99, 78,78, 29,97, 58,94,
  47,74, 20,12, 63,90, 99,23, 96,41,  6,33, 45,53, 76,89,
  97,62,  8,94, 27,76, 87,66, 63,19, 71,89, 82,41, 97,84,
  76,93,  0,53, 29,80, 36,86, 95,27, 86,40, 43,79, 76,58,
  21,50, 99, 3, 27,45, 27,16, 45,77, 66, 1, 82,89, 21,74,
  22,89, 82,77, 84,37, 57,18, 88, 5, 96,22, 83,41, 81,36,
  47,64, 88, 8, 64,19, 44, 7, 32,18, 70,49, 16,60, 23,62,
  37,19, 10,56, 98,22, 40,63, 93,39, 98,51, 67,17, 64,83,
  74,73,  5,21, 44,28, 26,34, 34,27, 75,84, 96,67, 25,60,
  60,73,  7,38, 26,25, 71,31, 85,19, 25,31, 82,10, 25,73,
   9, 8, 39,14, 61,99, 86,23, 64,17, 28,90, 54,51,  3,57,
  18,55, 36,75, 65,83, 67,37, 86,71, 35,72, 73,30, 10,78,
  53,14, 13,13, 27,80,  9,78, 99,29, 13,35, 67,28, 80,42,
  89,93, 41,61, 23,54, 25,76,  4,19, 31,94,  2,95, 42,51,
  72,32, 44,38, 20,46, 16,23, 26,32, 68,92, 26,13, 55,37,
  14,68, 36,36, 69, 2, 83,77, 45,31, 64,79,  9,65, 33,65,
  40,40, 44,11, 34,28, 46,91,  4, 9, 29,29, 58,23, 58,14,
   3,64, 79,60, 85,89, 73,57, 56, 2, 29,76, 11,50, 20, 2,
   9,80, 75,67, 12, 0, 36,64, 50,80, 93, 7, 48,18, 61,31,
  52,18, 17,59, 85,41, 16,80, 46,87, 42,95, 28,97, 37, 8,
  71,39, 53,87, 98,50, 78,10, 75,75, 64,46,  5,90, 34,88,
  30,35, 16,63, 49,16, 87,36, 35,88, 64,29, 84,33, 85,17,
  72,62, 81,19, 64,35,  2,67,  7,65, 83,84,  1,17, 81,42,
  33,57, 99,94, 18,21, 98,61, 19,87, 62,34, 18,86, 53, 3,
  18,69, 92, 5, 28,63, 93,43, 31,91, 78,97, 47,61, 27,86,
  54,23, 99,55,  7,60, 29, 8, 73,41, 60,29, 55,74, 33,21,
  99,31, 84,94, 70, 5, 85,76, 74,44, 21,47, 77,61, 19,71,
  32,20, 99,79, 30,45, 33,19, 64,76,  8,13, 51,20, 97,56,
  66,48, 41,24,  6, 3, 36,60, 14,44, 29,32, 79,96, 70,83,
  73,35, 60,61, 84,13, 55,98, 13,12, 44,76, 14,42, 80,98,
  25,62, 92,61, 33,36, 55,12, 69,27, 43,16, 71, 5,  9,33,
  54,55, 14,36, 93,23, 93, 4, 35,95, 93,93, 63, 1,  8,84,
  57,26, 68,84, 72,53, 48,17, 28,93, 31,19, 13,39, 36,89,
  24,52,  2,96, 68, 2,  5,61, 85,77, 24,93, 30,30, 20,51,
  17,66, 48,96, 60,86, 55,78, 61,20, 32,37, 32,30, 69,20,
  31,35, 31,36, 29,45, 43,25, 20,93, 62,63, 48,66, 96,82,
  95,83, 70,41, 50,93, 34,26, 25,40, 29,16, 74,78, 14,51,
  55,54, 44,65, 70,90, 27,41, 44,93, 50,34, 85,90,  8,76,
  86,68, 92, 0, 48,67, 79,28, 41,17,  5,50, 72,18, 65,30,
  26,85, 28,34,  2,47, 64,75, 41,35, 76,26, 61,48, 28,24,
  74,69, 31,77, 36, 9, 18,24, 48,15, 41,74, 47,79, 99,76,
  12,35, 39,42, 30,85, 73, 3,  3,89, 49,90, 80,35, 55,34,
  60,15, 74,31, 91,91, 79,10,  0,21, 78,77, 55,64, 37, 6,
   3,11, 86,66, 79,62, 23,53, 25,16, 40, 9, 88,53, 19,46,
  46,38, 10,51, 60,12, 55,42, 40,44, 85,39, 48,85, 24,62,
  77,49, 95,28, 77,42, 60,85,  2,88, 43,90, 41,46, 72,28,
   4,67, 73,34, 93,24, 58,60, 86,43, 93, 6, 74,63, 48,61,
  62,24, 19,32, 58,56, 56,69, 79,84, 75,46,  4,40, 45,30,
  66,27, 53, 5, 93,98, 79,92, 98,38,  6, 7, 85,20, 91, 0,
  57,51,  9, 4, 63,48, 25,20, 50,14, 58,89, 68,17, 26,71,
   1,26,  5,96, 70,59, 22,75, 24,36, 62, 8, 12,52, 29,90,
  57,93, 60,94, 28,72, 18,82, 88,81,  5,63,  7,19, 96,10,
  64,68, 73,45,  6, 0, 82,79, 12, 3, 26,26, 59,86, 33, 5,
  21,38, 68,20, 90,17, 37,99, 37,18, 91,73, 69,77, 17,68,
  90,90, 74,34,  5,87,  4, 0, 92,36, 83,28, 35,74, 69,26,
  50,39, 55,80, 80,38, 31,39, 92,48, 17,54, 52,27, 85,66,
  34,80, 18,79, 51,95, 32,60, 38,72, 84,64,  9, 9, 66,74,
  91,37, 82,78, 17, 4, 55,49, 14,31, 57,69, 13,41, 13,52,
   2,78, 73,23, 86,76, 93,15, 94,13, 96,49, 71,30, 50,17,
  89,36, 68,72, 38, 8, 30,87, 43,95, 23,85, 63,27, 66,71,
  66,29, 75,69, 71,19, 40,30, 86,60, 93,86, 88,26,  4,29,
  12,92, 96,48,  3,69, 68,94, 62,52, 41, 0, 77,96, 64,25,
  56, 1, 51,14, 91,49, 98,41, 18,68, 31,99, 44,97, 11,74,
  25,61, 40,90, 42,55, 93,88, 91,51, 89,88, 39,78, 39,60,
   6,92, 24,25, 70,21, 47,16, 92,73, 43,44, 97,82, 11,28,
  71,14, 18, 3, 44,50, 43,88, 60, 1, 89,46, 18,32,  3,31,
  93,89, 78,37, 21,63, 64,89, 26, 5, 73,36,  5,30, 47,87,
  87,13, 33,31, 32,42, 86,95, 28,69, 96,20, 22,10, 51,82,
  17,44, 51,64, 21,48, 41, 3, 78,11, 66,73,  8, 4, 30,58,
  81,37, 72,45, 20,45, 34,92,  7, 0, 56,23, 23,97, 72,72,
  80,50, 13,54, 97,73, 23,76, 97,13, 17,42, 63,92,  3,45,
  68,39,  6,52, 85,38, 83,26, 69,67, 86,49, 55,73, 18,30,
   5,64, 39,79, 30, 9, 34,78, 63,74, 23,73, 95,74, 78,30,
  82, 6, 62,15, 84,21, 15,27, 44,49, 15,52,  1,89, 31,20,
  62,53, 68,37, 32,85, 32, 5, 11,73,  2, 2, 51,58, 84,75,
  65, 1, 29,66, 87,73, 54,18, 37,91, 48,10, 17,60, 67,29,
  24,75, 97, 8, 52,59, 54,87, 74,38, 43, 2, 83,11, 76,59,
  61,35, 32,65, 33,42,  3,26, 32,22, 46,61,  3,73, 12,13,
  35, 7, 28,13, 33,90, 53,45, 76,34, 87,94, 31,83, 19, 2,
  58, 7, 63,63, 75, 3, 48,25, 64,16, 51,44, 24,19, 79, 2,
  80,14, 66,69, 58, 1, 75, 9, 95,42, 52,25, 36,38, 98,48,
  21,41,  2,11, 66,58, 11,30, 84,98,  7,98, 69,71, 72,12,
  17,92, 20, 9, 20,22, 91,29, 10,97, 40,53, 71,70, 51,68,
   1,85, 79,45, 97,44, 19,30,  0,56, 98,12,  7,96, 20,82,
  58,11, 34,45, 28,75, 21,68, 28,91, 97,59, 75,82, 73,11,
  68,75, 63,16, 72,70, 35,94, 15,17, 42,69, 49,67, 79,58,
  35,83, 12,53, 74,84, 56,67, 55,59, 63,14, 68,62, 25,30,
  32,63, 10,57, 18,20, 96,96, 94,35, 61,84, 66,76, 76,15,
  39,34, 79,79, 72,78,  4,90, 48,99, 93, 3, 39, 8, 75, 7,
  99,19, 10,44, 97,18, 56,88, 67,72, 39,80, 50,76, 62,86,
  39,56, 76, 0,  5,40, 26,38, 60,96,  2,49, 18,75, 26,69,
  60,36, 84,76, 64,36, 43,97, 64,13, 59,27, 45,57, 89,13,
  67, 6, 86,46, 42,39, 86,28,  9,85, 62,47, 47,63, 53,91,
  30,28, 12,80, 14,29, 36,11, 68,56, 24,29, 90, 1, 63,58,
  12,17, 23, 9,  0,95, 87,14, 52,96, 86,52, 39,96,  0,85,
  19,58, 45,80, 41,57, 86,37, 86,93, 34,33, 84,96, 95,26,
  19,59, 69,97,  8,66, 52,92, 12,91, 34,48, 37,16, 28,45,
  83,16, 98, 9, 53,77, 57,79, 38,23, 39,93, 54,31, 69,56,
   3,37, 65,21, 72,13, 97,42, 41,48, 15,19, 72, 3, 40,47,
  49,79, 28, 5, 37,69, 51,66, 48,94, 59,11, 82,19, 38,43,
  36,21, 30, 0, 89, 1, 36,95, 97,20, 81,10, 82,27, 51,28,
  87,54, 58,35, 21,37, 35,96, 86, 2, 53,51, 22,73, 67,96,
  71,35, 95,92,  4,45, 17,24, 16,95, 59,41, 70,73, 46,13,
   4, 3, 25,53, 85,55, 68,88, 82,23, 59,22, 82,91, 19,68,
  56,53,  3,86,  8,40, 57,65, 62,14, 54,66, 50,28, 44,99,
  92,25, 29, 1,  7,55,  0,57, 34,69, 95,44, 91,43,  4,60,
  64,69, 55, 7, 97,52, 83,66, 19,14, 18,59,  3,77, 15,76,
  53,46, 50,98, 58,16, 60,79, 35, 1, 75,52, 13,62, 37,75,
  36,97, 11,70,  1,24, 85,25, 64,24, 51,96, 21,22, 61,58,
  41,45, 59,21, 65,35, 25,63,  7,95, 23,70,  6,15, 53,81,
  59, 3, 26,10, 75,30, 56,92, 67,93,  4,59, 72,37, 66,87,
  25,91, 84,12, 43,94, 54,13, 33,18, 65,79, 70, 4, 48,13,
  26,47, 23, 1, 95,54, 59,46, 90,51, 22,78, 75,60, 16, 6,
  97,31, 97,94, 90,47, 87,53, 62,75,  2,84, 13,42, 55,77,
   5,20, 83,89, 36,35, 96, 4, 67, 4, 38,24, 67,12,  9,95,
  71,97, 84,28, 69,29, 57, 8, 56,44, 16,77, 33,62, 77,67,
  10,64, 58,91, 44,90, 79,52, 43,92, 12,95, 34,52, 40,37,
   8,60, 24,85, 20,99, 66,66, 88,58, 67,63, 74,60, 57,77,
  42,71, 28,95, 40,76, 74,32,  0, 1, 48, 9, 70,44, 40,88,
   5,58, 45,73, 93, 5, 33,41,  5,10, 72,50, 14,34, 91,77,
  26, 9, 57,90, 48,27, 91,98, 45,83, 28,10, 70,91, 19,79,
  97, 4, 87,71, 68,31, 49,69, 88, 1, 47,86,  3,71, 85,14,
  76,80, 68,67, 84,26, 38,54, 67,10, 56,45, 82,52, 76,61,
  19,78, 43,37, 81,16, 43,55,  6,99, 17,95, 29,74, 47,49,
  90, 4, 47,89, 98,52,  3,90, 80,31, 37,92, 27, 8, 54,56,
   4,68, 28,16, 28,85, 76,90, 36,18, 27,44, 18,76, 78,75,
  53,47, 89,67,  6,80, 85,81, 34,67, 66,84,  1,57, 10,66,
  52,52, 67,39, 98,64, 83, 5, 49, 3, 11,64, 72,79, 41,33,
  19,65, 23,66, 70,23, 74,65, 96,87, 89,56, 31,51, 48,91,
  85,11,  8,79,  6,83, 52,20, 59,95,  9,15, 89,38, 84,18,
   2,34, 99, 5, 51,23, 48,40, 69,85, 35,37, 42,88, 95,36,
  86,11, 89,95, 37,52, 39,52, 64,98, 97,72, 44,58, 75,80,
  29,38, 36,73, 17,45, 37,35, 98,49, 69,83, 22,19, 61,34,
  26,24, 85,68, 95,46, 47,62, 44, 9,  9,83, 34,56, 65, 0,
  97,70, 16,79, 82,71, 90,20, 47,48, 18, 6, 15,26, 26,59,
  51,63, 70,67, 29,35, 18,40, 81,70, 67, 3, 93,58, 47,60,
  25,69, 14,39, 68,29, 75,62, 71,52, 67,69,  4,65, 82, 5,
  91,85, 38,42, 34,31, 96,43, 87,93, 88, 4, 64,54
};
const int p_100_3[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  1599,
  /// Array of preferences
  26,12, 58,75, 72,14, 68,16, 33,39, 25,41, 44,95, 61,64,
  84,82, 81, 6, 24,74, 97,87, 56,81, 81,90,  7,61, 45,60,
  43,55, 51, 8, 53,84, 40,16, 18,87, 30, 2, 26,41, 71,79,
   7,79, 35,58, 90,37, 50,96, 92,50, 14,14,  4,47, 77,73,
  54,11, 38,88, 74, 1, 82,97, 76,63, 81,85, 42,13, 44,55,
  22,56, 26,79, 15,45,  4,94, 35,61, 37,25, 19,76,  8,56,
   7,99, 94, 7, 62,34, 51,32, 87, 3, 47,31,  4,50, 61,20,
   3,69, 29,57, 27,42, 22,46, 96,61,  0,36, 55,98, 36, 7,
   3,49, 80,53,  7,66, 93,24, 60,11, 36,96, 11,68,  2,76,
  72,11, 59,79, 91,41, 15,75, 86,38, 26,59, 64,21, 32,18,
  49, 4, 18,86, 38,94, 92,94, 49, 1,  2,99, 86,22, 51,37,
  35,83, 57,73, 83,14, 24,29, 60,56, 83,85, 27,29, 51,14,
  41,42, 14,24, 49,41, 41,89, 39, 5, 85,98, 61,19, 30,19,
  84,75, 14,22, 78,95, 43,97, 55,40, 62,83, 35,28, 98,40,
  77,76, 49,74, 60, 0, 59, 1, 47,93, 31,62, 67,37, 13,80,
  42,99, 47,55, 42,75, 68,22, 18,30, 38,18, 60,46, 95,14,
  81,54, 17,20, 69,93, 95,11, 76,33, 12,68, 37,38, 76,71,
  95,63,  3,67, 95,82, 48,85, 93,85, 98, 8, 90,29, 78,83,
  14,80, 53,50, 53,16, 82, 0, 59,44, 16,16, 14,19, 24, 0,
   7,40, 20, 3,  1,46,  2,81, 62,64, 39,23, 12,41, 51,63,
  27,95, 93,13, 60,67, 94,38,  0,94,  8,38, 24,62,  1, 3,
  35,38, 29,49, 83,22, 99,36, 47,33, 29,56, 14,46, 22,45,
  61,66, 21,33, 62,18, 11,69, 27,69, 63,12, 21,26, 31,70,
  78,16, 57,18, 56,54, 54,92, 19,26, 13,93, 45, 1, 30,52,
  35,72, 96,22, 44,21, 42, 5, 46,53, 93,47, 49,67, 39,31,
  78,68, 51,82, 82,26, 71,75, 70,20, 50,98, 78,66,  0,62,
  60,45, 37,27, 55,26, 14,41, 66,84, 15,20, 44,65, 22,98,
  46,48, 67,36, 36,11, 62, 5, 41,85, 77,90, 31,26, 84,91,
  81,50, 56,44, 27,27, 79,13,  0,35, 44,11, 61,51, 51, 5,
  79,29, 82,85, 66,82, 88,53, 11,39, 13, 7, 77,69, 97,62,
  96,67, 47,90, 79,15, 48, 2, 44, 2, 59,66, 59,84, 63,28,
  91,29, 98,99, 97,27, 11,59, 72, 9, 98,97, 78,17, 79,27,
  86, 6, 40,51, 48,42, 60,47, 66,96, 24,44, 92,80, 42,33,
  10,27, 39,30, 29,34, 30,84, 59,61, 31,36, 71,97, 97,92,
   2,15, 60,84, 22,41, 34,20, 23,56, 42,93, 85,29, 44,60,
  84,98, 14,42, 98, 3, 53,74, 96,95, 56,87,  4,88, 36,97,
   0,10, 95,48, 17,27, 62,94,  3,62, 20,54, 62,22, 40,99,
  71,67, 42,56, 24,95, 51,31, 84,24, 13,71, 82,39, 90,50,
  92,95, 87,41, 40,21, 83,19, 35,56, 55,15, 90,76, 89,22,
  14, 7, 54,20, 15,70, 53, 0, 80, 6, 58,30, 62,97, 70, 4,
  39,75, 78,38, 54,71, 62,41, 97,16, 62,99, 88,41, 59, 7,
  15,99, 52,17,  3,48, 56,73,  5,73, 47,26, 98,55,  4,56,
  26,36, 38,25, 77,98, 79, 8, 93,58, 84, 1, 63,30, 43,72,
  12,61, 59,72,  5,67, 50,36,  8,76, 61,99, 95,44, 91,49,
  24,61, 93, 9, 75,57, 55,42, 67,23, 18,82, 92, 1, 26,34,
  54,84, 92,65, 97, 0,  4,78, 83,45, 16, 5, 41,33,  8,86,
  33, 9, 21,25, 20,77, 42,12, 68,71, 88,94, 91,34, 70,65,
  50,18, 51,41, 91,96, 54, 5, 31,57, 38,12, 66,85, 33,64,
  20,63, 46,33,  9,66, 15,30, 38, 0, 64,82, 69,65, 79,78,
  44,89, 92, 7,  2,12, 94,45, 54, 2, 87, 0, 38,68,  1,91,
  85,15, 50, 5,  7,73, 67,54, 32,41,  3, 6, 66,74, 18,65,
   0,14, 53,48, 40,82, 51,89, 68,99, 12, 1, 34,75, 49,31,
  38,16,  0,33, 38, 1, 19,79, 98,75, 80,73, 49,12, 16, 3,
  10,98,  5,65, 24,59, 71,87, 34,50, 30,76, 47,45, 17,52,
  73,23, 56, 0, 15,65, 87,14, 50,86, 91,81, 90,85, 24,18,
  26, 9, 49,53, 92,68, 40,83, 34,73, 31,87, 65, 1, 32,75,
  85,78, 72,97, 35,54, 32,79, 64,69,  7,28, 44,48, 89,91,
  39, 1, 85,83,  8, 8, 81,86, 47,91, 90,79,  7,94, 77,63,
  69,15, 64,50, 90,40, 41,32, 32,26, 69, 5, 50,41, 38,53,
  87,66, 24,26, 35,35, 72,58, 21,97, 27,61, 61,15, 76,29,
  20,86, 22, 1, 44,34, 19,92, 84,54, 25,93, 45,75,  0,24,
   1,94, 69,57,  5,51, 93,48,  8,26, 32,30,  9,11, 70,81,
  64,38, 20,53, 78,63, 95,55, 61,28, 52,44, 65,73, 68,65,
  52,14, 59,49, 59,23, 23,44, 41,40, 96,48, 48,93, 28,92,
  80,63, 62,26, 78,49, 12,40,  7,96,  9,45, 42,74, 78,85,
  25, 5, 50,21,  0, 9, 38,63, 56,10, 22,39, 14,53, 44,32,
  58,29, 86,51, 39,51, 75,58, 69,84, 70,32, 27, 7, 86,78,
  19,93,  9,71, 42,73, 90,60, 36,14, 82,51, 15, 2, 72,18,
  22,52, 73,19, 15,53, 59,48, 36,63,  3, 7, 45,96, 54,22,
   4,36, 36,28, 99,91, 57,35, 80,79, 84,44, 58,45, 96,26,
  51,92, 75,89, 11,67, 79, 9, 28,87, 15,21, 88,86, 65,94,
  88,19,  1,29, 38,54, 70,25, 82,59, 18,43, 84,56, 21,29,
  28,58,  5,77, 90,34, 64,87, 99,25, 19,37, 35,33, 15, 0,
  50,24, 41,92, 98,22, 16,98, 13,42, 86,31, 27,44, 24,42,
  92,51, 76,27, 20,40, 25,53,  5,57, 70,78, 15,95, 70,52,
  89,19, 25,98, 53,79, 19,74, 26,16, 12,82, 27,46, 82,22,
   1,59, 36,64, 57, 0, 84,52, 89,14, 48,40, 46,89, 62,58,
   2,98, 99,63, 24,71, 83,78, 26,80,  0,43, 86,24, 24,67,
   4,57, 96,16, 83,25, 56,34, 60,14, 59, 2, 79,93, 44,17,
  92,28, 83,41, 70,84, 83,24, 90,35, 91,72, 58,56, 91,93,
  70,19, 56,52, 26,55, 15,62, 79,24, 78,36, 41,45, 77,50,
  34,25, 19,32, 53,17, 33,67, 27,30, 94,35, 50,12,  1,96,
  98,11, 54,14, 87,25, 15,66, 13,52, 85,91, 47,46, 87,58,
  14, 0, 80,62, 47,53, 74,19, 46,85, 92,56, 13,54, 64,88,
  63,78, 58,15, 66,43, 36,26, 29,73, 40,34, 61,58, 72,22,
  33,42, 36,59, 16,47, 77,81, 59,67, 74,76, 75,35, 41,87,
   9,75, 25,15, 25,57, 53,31, 24,89, 97,26, 85,60, 86,79,
  92,44, 40,88, 70,18, 93,54, 97,72, 65,58,  0,63, 61,81,
  62, 1, 61,22, 45, 0, 45,56, 41, 6, 50,39, 96,69, 77, 7,
  20,74,  4,27, 55,62, 64,16, 76,69, 68, 4, 37,49, 90,74,
   6,41, 13,60, 48,13,  9,88, 95,74, 51,66, 18,17, 98,86,
  32,58, 42,29, 52,47, 71,55, 40,53, 17, 4, 39,77, 32,15,
  22,12, 36, 2, 91,97, 16, 0, 24,97, 18,61, 67,41, 23,67,
  88,72, 86,94, 74,16, 55,81, 17, 5, 31,89, 94,74, 52, 3,
  58, 9,  6,64, 38,61, 52,37, 98,26,  1, 6, 82, 6, 72,21,
  59,32, 73,96, 55,36, 32,85, 72,41, 56,70, 88, 3, 22,26,
  18,66, 91,20,  8,46, 20, 8, 51,81,  4,23, 46,84, 47,41,
  49,91, 51,59,  2,31, 91,54, 43,90, 75,85,  7,45, 16,31,
  92,71, 91, 2, 38,85, 28,65, 23, 4,  5,54, 75,23, 64,63,
  71,98, 25,33, 50,22, 16,24, 21,78, 66,41,  1,41, 64,33,
  68,41, 16,19,  5,79, 20,94, 86,30, 59,25, 68,54, 81,10,
  88,17, 42, 3, 27,48,  7,25, 77,42, 53,51, 20,22, 71,70,
  80,91, 69,38, 81,55, 67,58, 21,75, 18,80, 73,71, 54,60,
  39,71, 35,77, 45,77, 14,68, 60,96,  3,70,  8,81, 90,14,
  86,52,  5,34, 17,25, 57,96, 49,93, 67, 9, 11,42, 44,58,
  21,50, 93, 0,  4,86,  9,53,  4,80, 62,40, 45,32, 34,66,
  80,31, 94,40, 41,35, 82,70, 45,95,  2,84, 37,29, 33,35,
  50,93, 16,28, 43,49, 56,71, 81,28, 57,97, 84,79,  6, 0,
  88,59, 68,95, 28, 3, 64,91, 23,92, 63,62, 41,54, 43,77,
  54,88, 41,27, 54,93, 44,41, 49,61, 25,54, 82,45, 24,36,
  80,75, 83,52, 34,51, 81,69, 75,34, 85,63, 50, 1, 19,90,
  47,15, 78,60, 71,93,  7,86, 95, 6, 14,82, 57, 2, 92,34,
  98,93, 58,99, 39,60, 96,11, 37,45, 53,20, 77,36, 49,56,
  80,48, 15,39, 49, 7, 68,39, 80,44, 94,83, 56,19, 53,77,
  43,16, 25, 2, 21,69, 91,76, 53,34, 70,10, 19,77, 37,64,
  38,93, 68,74, 19,55, 70,86, 29,94, 84,76,  2,83, 69,42,
  28,47, 56,94, 58,78, 41,84,  7,55, 20,55, 27,62, 59, 4,
  53,81, 51,34,  6, 8, 48,60, 87,91, 93,52, 23,24,  2,67,
  23,34, 14,47, 69,33, 30,97, 97, 7, 29,93, 69,10, 37,47,
  86,42, 98,48,  7, 9, 65,49, 24,78, 10,17, 28,48, 15,14,
  13,68, 73,38, 53,73, 47,74, 59,89, 77,80, 53,14, 94,78,
  26,22, 88,67, 18,97, 64,61, 61, 7, 90,80, 88,88, 27,84,
  88, 5, 87,74, 31,55, 94,58, 54,65, 34, 4, 90,65, 31,25,
  24,25, 11,72, 49,84, 40,22, 17,41, 95,36, 89,38, 79,67,
  53,15, 91,60, 30,58, 40,11,  7,50,  6,25, 92, 3, 66,99,
  17,69, 25,52, 94,61, 10,66, 96, 0, 11,35, 69,53, 82,54,
   0,30, 36,19, 14,69, 60,87, 54,97, 64,60,  5,89, 73,56,
  58,81,  7,88, 15,49, 11,62, 35,78, 44,88, 77,95, 69,30,
  14,56, 71,27, 52,67, 94,60, 76,42, 68,11, 33,55, 91,42,
  23,22, 10, 6, 35,80, 76,58, 48,52, 31,90, 97, 3, 88,21,
  62,45,  8,41, 49,18, 94,71,  4,22, 14,31, 53,10, 77, 0,
  34,39, 37,30, 42,76, 33,52, 74,56, 45,47, 53,62, 61,46,
  22,96, 68,89, 90,96, 53,82, 95, 2, 69,68, 83, 7, 16, 1,
  32,10, 52,90, 63,44, 40,12, 17,86, 79,44, 96,50, 63, 5,
  81,12, 58,61, 92, 4, 33,59, 84,53, 20,43,  7,82, 64,22,
  12,90, 24,90, 35,18, 17,91, 58,14, 28,31, 28,60, 48,74,
  78,24, 50,19, 89,31, 40,25, 85,43, 80,41, 79,68, 69,39,
  61,96, 28,57, 64,18, 33,99, 13,78, 69,95, 77,28, 76,53,
  98,90, 35,47, 72, 3, 80,90, 76,77, 62,46, 73,92, 85,79,
  61, 5, 12,15,  2, 0, 58,97, 86, 2, 82,24, 47,61,  9,56,
  98,13, 23,11, 56,33, 40,74, 61,34,  9,18, 48,21, 21,55,
  53,72, 59,83, 20,72, 70,92, 58,22, 50,62, 48,22, 98,39,
  80,47, 81,57, 69,52, 78,41, 32,90, 62,76, 58,73, 73,32,
  86,99, 79,28, 42,26, 20,89,  6,94, 82, 3, 94,81, 72,37,
  10,81, 53,54,  1,77, 92,36,  5,83, 21,57, 57,59,  2,72,
  20,79, 79,96, 86,86,  0,70,  4,46, 45,58, 81,75, 60,15,
  71,83, 85,69, 77,74, 67, 2, 90,89, 42,51, 27,76, 95,39,
  24,83, 65,93, 75,86, 43,69, 60,40,  1, 8, 46, 4,  7,92,
  49,62, 90,25, 53, 6, 52,51, 62,28, 73,87, 82,27,  5,36,
  86,82, 99,23, 37,80, 98,78, 88,33, 89,47, 83,56, 27,41,
  95,85, 38,26, 91,23, 97,25, 63,76, 73,36, 23,54, 83,95,
  15,32, 49, 9, 96,76, 50,54, 34,47, 17, 1, 51,42, 67,65,
  17, 8, 65,66, 66,91, 19,50, 87,54,  6,91, 78,54, 71,69,
  49,97, 58,69, 99,52, 72,84, 50,75, 27,92, 57,29, 29,96,
  78,84, 64,39, 60,81, 98,60, 35,85, 94,14, 90, 1, 92,58,
   2,61, 43,75, 99,87,  5,19, 51, 2, 17,29, 41,37, 91,48,
  12,64, 99,24, 21, 8, 76,73, 44, 8,  6,73, 58,43, 98,59,
  89,12, 69,17, 27,32, 77,32, 74,12, 52,27, 67,42, 57, 4,
  34,96, 47,27,  9,62, 97,35, 39, 0, 10,21, 30,80, 68,49,
  20, 1, 65,83,  4,73, 18,13, 53,65, 66,23, 26,19, 12,49,
  54,52, 53,39, 21,22, 26,57, 38,96,  0,47, 35,45, 90,87,
  98,24,  8,54, 54, 3, 57,45, 93,20, 31,35, 71,15,  2,32,
  51,26, 13,11, 93,29, 40,64,  8,49, 96, 9, 92,53, 49,88,
  97,50, 51,79, 68,20, 98, 4, 58,67, 25,47, 44,77, 67,98,
  20,50,  1,13, 66,76, 15,16, 77,27, 21,84, 17,19, 15, 3,
  84,46, 31,60, 17,39, 21,32, 87, 6, 21,10, 11,44, 81,45,
  40,87, 22,10, 61,24, 28,21, 58,48, 22,14, 57,71, 81,23,
  61, 9, 65,69, 60,60, 84,42, 26,64, 76,95, 25,85, 77,88,
  41,65, 59,46, 25, 1, 65,29,  7,43, 47,51, 24,33, 51,28,
  52,23, 98,47, 48, 5, 64,28,  2, 6, 90,71, 61,78, 99,44,
  49,99, 82,91, 88,98, 86,76, 78,80, 51,98, 33,82, 85,11,
  98,80, 65, 4, 53,66, 88,42, 49,37, 59,64, 80,57, 13,56,
  17,92, 34, 5, 57,32, 26,66, 52,94, 31,40, 87,75, 90, 6,
  37,11, 42,16, 39,78,  3,85, 44, 7, 80, 9,  2,63, 43, 8,
  21,70, 56,72, 46,31, 31,96, 94,92, 44,22, 59,81, 66,70,
  55,74, 80,54, 97,69,  5,24, 34,36, 74,83, 54,44, 67,96,
   6,98, 26,14, 69,71, 29,29,  6,79,  4,74, 18,49, 35,17,
  93, 7, 30,39, 46,13, 41,61, 96,18, 10,45, 76,43, 52,25,
  87,57,  4, 9, 33, 7, 16,12, 18,58, 13,12, 44,73, 73,52,
  86,48, 67,97, 14,59,  6,34, 47,37, 98,65, 86,91, 73,99,
  39,83, 22,67, 82,74,  5,94, 87,93,  7,59, 65,11, 47,77,
  79,16, 68,81, 68,36, 88,84,  3,92, 24,31, 39,91, 68,47,
  29,25, 61,39, 94,32, 59,94, 33,85, 38,76, 58,59, 81,81,
  42,17, 36,92, 81,78, 89,50, 76,32, 47,78, 96,85,  9,67,
  31,74, 61,55, 61,88, 23,50, 92,48, 19,61, 45, 4,  1,12,
  27,14, 99,90, 61,73, 10,47, 55,25, 27,24, 62, 4, 75,70,
  57,20, 91,16, 90,28, 95,23, 67,27, 63,26, 85,22, 10,46,
  38,19, 16,44, 84,22, 38,91, 88,99, 91, 0, 47, 8, 39,73,
  37,61, 81,65, 65,84, 59,56, 93,68, 64, 4, 28,15, 24,28,
  49,32, 29,14, 63,14, 84,57, 91,68, 54,72, 94,64,  6,37,
  21,17, 12,62, 36,87, 11,74, 47,67, 29,54,  7,51, 84,69,
  95,24, 41,59, 41,66, 94,33, 73, 5, 61,82, 34,87, 43,14,
  85,70, 53,30, 40,65, 23,63, 33, 2, 37,89, 43,57, 18,92,
  16,67, 49,30, 16,34,  5,71, 64,86, 58, 6,  0,18, 94,10,
  71,91, 82,17, 45,34, 67,40, 71,77, 62,38, 54,98, 39,34,
  67,90, 84,16, 68,40, 96, 1, 16,88, 28,71, 19,39, 79,35,
  11,66, 37,17, 22, 5, 76,75, 19,65, 99,49, 51,17, 14,62,
  91,73, 10,31, 89,83, 66,63, 72,12, 53,70,  7,16, 58,46,
  28,94, 46,49, 53,44, 84,30,  6,85, 82,34, 46,99, 46,26,
  68,37, 66,46, 48,53, 13,22, 10,74,  4,34, 93,28
};
const int p_100_4[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  1227,
  /// Array of preferences
  23,76, 35,95, 32,68, 90,63, 75,77, 70,42, 70,94,  8,70,
  25,28, 99,61, 90,62, 41,15, 77,70, 65,65, 86,76, 59,29,
  56,82, 78,90, 93,30, 10,89, 85,67, 51,41, 68,76, 40,97,
  47,52, 86,82, 90,90, 19, 5,  3,63, 27,11, 45,53, 81,58,
  33, 4, 95, 4, 65,52, 70,77, 91,20, 21,44, 87,71, 97,90,
  61,85, 98,60,  5,85, 97,51, 57,91, 16,31, 59,98,  0,97,
  18,47, 15,59,  6,20, 88,78, 71,98, 42,86, 88,62, 44,92,
  55,73, 19,62,  2,62, 97,22, 81,59, 57,10, 25,52, 69,70,
  15,94, 42,62, 50,97, 72,16, 81,45, 56,73, 53,53, 34,91,
  34,32, 65,36, 76,31, 15,29, 85,90,  1,43,  6,22, 34,44,
  89, 5,  5,39, 26,34,  7,44, 27,18, 11, 6, 36,75, 68,62,
  69,50,  0,38, 64,46, 99,64, 95,74, 79,81, 12,29, 68,45,
  81,57, 78,43, 56,22, 57,92,  4,37, 95,10, 86,25,  1,98,
  20,61, 24,70, 98, 2, 72, 2, 11, 2, 56, 3, 95,56, 78, 5,
  13,83, 66,45, 96,81, 98,11,  3,60, 20, 9, 10,33, 96,86,
  96,65, 38,22, 51,95, 78,25, 44,51, 16,11, 53,19, 77,43,
  31,13, 62,48, 33,63, 44,50, 98, 9, 35, 6, 50, 7, 47, 0,
  74,46, 18,93, 89,21, 20,75,  8, 8, 85, 5, 45,71, 33,21,
  12,71, 74,21, 60,11,  0,67,  0,20, 24,89, 84,35, 82,89,
  47,55, 90,73, 93,18, 12,53, 85, 0, 91,13, 40,51, 27,41,
  97,16, 86,98, 41,38, 57,50, 44,39, 78,52,  2,80, 97,35,
  17,78,  9,70, 71,67, 35,84, 46,66, 73,72, 64,51, 81,20,
  27,83, 61,30, 41,11, 46,39,  5,21, 62, 6, 97,48, 54,56,
  27,40, 35,25, 32,20, 84,13, 15,26, 57,16, 68,44,  4,84,
  30, 5, 75,63,  1,42, 21,45, 14,74, 66,65, 44,88, 95, 3,
  80,63, 58,23, 79,58, 30,18, 98,14, 46,78, 28, 1, 42,80,
  83,81,  7,51,  2,91, 27,28,  4,43, 20,47, 78,15, 22, 8,
  85,37, 61,76, 43,15, 87,54, 78,33, 60,94, 72,11, 30,33,
  54,41, 61,92, 67,53,  0,42, 99,33, 59,57, 14,85, 64,86,
  41,70, 82,65, 49,71, 33, 3, 70,71, 32,37, 64,19, 29,63,
  99,75, 65, 1, 38,52, 21,52, 62,55, 46,92, 34,22, 66,16,
  34,51, 92,10, 62, 9, 97,97, 77,82, 95,94,  2,16, 70,98,
  65,83, 97,53, 27,27, 18,45, 31,72, 32,72, 32,53, 35,31,
  63,74, 36,40, 16,65, 62,43, 28,74, 31, 7, 23,35, 28,62,
  62,96, 76,34, 21,27, 97,69, 75,60, 76,98,  7,69, 35,53,
  21, 4, 30, 4, 59,39, 18,30,  9,35, 59,11, 85, 4, 35,78,
  74,95, 87,29, 28,40, 23,69, 11,85, 46,46, 54,54, 11, 4,
  70,54, 30, 8,  0,17, 91,50, 19,49, 34,13, 30,75,  2, 9,
  62,51, 30,37, 62,68, 16,98, 83,20, 43,64, 55,17, 21, 7,
  22,37, 35,83, 16,55, 82,77, 49,36, 33,13, 64,40, 16,20,
  45,56, 50, 1, 41,67, 50,50, 13,47, 44,53, 48,16, 29,20,
   9,97, 38,58, 40,14, 77, 0, 57, 9,  8,66, 83,71, 20, 2,
  19,76,  2,78, 51,22, 23,64, 28,23, 19,20, 98,35,  4,69,
  36,89, 24,43, 66,14, 86,83, 20,24, 67,41, 17,70,  9,64,
  31,77, 36,64, 94,70, 11, 5, 54,28, 51,77, 27,85, 28, 2,
  73,60, 58,77, 95,14, 30,45, 92,97, 18,24, 94,13, 76,73,
  34,45, 82,43, 65, 4, 54,35, 22,24, 49,46, 32, 5, 21,18,
  21,32, 33,89, 42,19, 19,31, 52,31,  4,92, 23,40, 30,94,
  14,87, 81,64, 82,59, 48,90, 24,42, 95,73,  5,43, 17,67,
  26,97, 25,11, 29,37, 50,96, 42,31, 41, 4, 81,41, 33,88,
  37,13, 49,59, 35,87, 42,33, 51,64, 66,56, 64,53, 59,71,
  78,46, 35,26, 43,80, 44,60,  5, 4, 17,60, 20,27, 74,52,
  53,93, 62,73, 58,37, 53,35, 56,21, 82,28, 57,94, 59,33,
  28,64, 24,39,  6,32, 12, 4, 72,64, 86, 4, 25,36, 75,35,
  87,90, 84,63, 71,38, 12,21, 60, 6, 90,57, 67,48, 84,12,
  71,14, 16,40, 13,16, 97,93, 89,57, 30,61, 89,82, 74,63,
  28,31, 28,41, 24,21, 11,89, 66,92, 57,39, 14,73, 42,13,
  69,52, 63,42, 19,89,  9,26, 27,82, 20,55, 43,65, 82,96,
   4,30, 87,73, 67,61, 39,94, 13,14, 11,24, 62,42, 30,59,
  63, 7, 29,23, 34,23, 56,45, 77,49, 77,89, 84, 0, 27,20,
  51,62, 42,82, 30,92, 71,46, 88,91, 88,14, 75,80,  9,12,
  63,72,  2,35, 83,26, 60,51, 83,24, 62,15, 96,36, 21,11,
  64,99, 53, 1, 64,75, 89,62, 55,18, 33,90, 66,63, 67,70,
  48,33, 26,25, 67,72, 78,54, 34,72, 99, 5, 42,90, 72,95,
  40,92, 79,42, 70,43, 25,88,  8,16,  7,56, 25, 8, 48,13,
  63, 1, 30,22, 52,71, 99,17, 41,80, 32,42, 42,45, 23,78,
  73,93, 21,66, 84,55, 74,16, 19,30, 30,49, 80,25, 66,60,
   1, 3, 34,67, 90,12, 91,62, 96,62, 33,49, 11,58, 12,20,
  60,68, 79,88,  2,94, 52,27, 99,50, 72, 5,  0,63, 26,75,
  19,74, 20,32, 24,45,  1,48, 83,92, 82,97, 31,93, 35,45,
  53,63,  8, 3, 98,65,  0,24, 80, 8, 49,35, 31,21, 64,93,
  62,12, 10,21, 22,48, 33,95, 88,36, 68,47, 75,89, 22,23,
   9,17,  4,63, 43, 1, 19,46, 42,34, 94,15, 60,58, 43,55,
  89,84, 27,46, 32,58, 78,55, 14,91, 33,41, 43,10, 43,33,
  26,67, 85,77, 10,81, 71,69, 62,94, 85,38, 85,58, 56,39,
  67,52,  5,36, 54,45,  1,12, 79,45, 80,94, 79,93, 37,98,
  55,13, 75,95, 68,73, 76, 5,  6,85,  3,33, 69,89, 91,63,
  58,28, 29,22, 24,57, 27,47, 97,49, 60,59, 99,44,  3,52,
  50,69, 31,97, 30,55, 87,50, 23,94, 32,49, 74,18, 69,34,
  22,77, 14,97, 86,60, 68,11, 88,75, 35,99, 81,29,  4,91,
  85,66, 27,77, 73,28, 49,18, 79,44, 77,10, 14,24, 33,14,
  50,28, 90,37, 67,24, 49, 1, 65,18, 43,17, 67,97, 72, 0,
  89,72, 34,92, 12,44, 45,67, 23,11, 62,67, 24,32, 30, 9,
  41,30, 84,46, 69,84, 60,79, 48,48, 79,71, 56,80,  1,14,
  66,49, 91,81, 42,11, 22,15, 32,52, 69,23, 62,86, 18,69,
  58,24, 82,57, 41,24, 76,36, 12,73, 22,13, 24,71, 61,61,
  92,68, 39,42, 32,77,  5,28, 53,96, 90,14, 14,34, 51,19,
  23,73, 51,39, 89, 2, 51, 0, 14,44, 78,66, 69,33, 28,84,
  22, 7, 90,56, 10,29, 41,14, 64,76, 55,19, 41, 8, 49,87,
  80,83, 72,10, 15,65,  1,59, 73,55, 28,61, 15,92, 51,66,
   4,64, 10,78, 51,82, 68,58,  7,24, 55,85, 59, 1, 80,90,
  40,42,  8,22, 37,38, 21,81, 95, 8, 96,45, 60,61,  8,94,
  61,24,  5,72, 76,42, 36,97, 50,17,  1,84, 15,50, 74,25,
  34,49, 39,85,  9,93, 85,61, 66,12, 20,46, 74, 1, 70,58,
  85,62, 36,72, 18,54, 99,79, 50,24, 20,33, 48,80, 20,84,
  23, 7, 14,76, 23, 6, 79,59, 52,10,  9,83, 33,64, 17,84,
  56, 9, 87,61, 44, 6, 27,13, 42,78, 21,53, 28,18, 92,71,
  54,12, 21,60,  8,39, 82,76, 19,27, 90,29, 27,60, 96,87,
  48,56, 64,42, 17,92, 61,54,  8,23, 11,83, 34,57, 22,82,
  58,87, 86,77, 63,77, 14,22, 83, 8, 17, 5, 16,47, 22,58,
  68,60, 94,97, 93,81, 81,15, 33,91, 25,56, 21,88, 65,66,
  44,41, 89,43, 94,56, 93,13, 34,33, 68,23, 85,36, 91,79,
  41,56, 10,49, 53,20, 61,64, 94,91, 48,40, 24,56, 32,56,
  59,48, 71,84, 60,20,  6,90, 47,72, 84,16,  2, 2, 39, 0,
  54,66, 72,24, 12,56, 84,62, 30,85, 66,42, 12,17, 42,60,
  86,13, 58,64, 85,31, 13,81, 59,84, 88,27, 38,91, 82,62,
  47,49, 95,61, 36,21, 67,79, 28,30, 25,58, 64,74, 62,47,
  15,58,  7,23, 46,82, 37,50, 58,96,  3,79, 71,96, 34,98,
  26,10, 74,38, 80,12, 78,75, 31, 3, 52,77, 43,89, 81,17,
  78,50, 37, 6, 22,17, 56,32,  5,89, 34,48, 87,16, 97,58,
  73,24, 37,41, 67, 8, 58,11, 37,12, 63,66, 61,95, 69,98,
  58,82, 78,62, 31,79, 14,50, 60,96, 26,69, 73,21, 94,78,
   8,80, 74,28, 18,72, 74,88,  8,15, 75,64, 35,96, 31,83,
  49,84, 30,42, 33,20, 53,73, 73, 4, 39,67,  5,93, 74, 8,
  41,27,  4,18, 97,45, 70,57, 63,48,  2,32, 99, 1, 24,29,
  13,67, 13,93, 46,85, 84,14, 95,78, 50,42, 72, 7, 45,43,
  13, 3, 77, 9, 81,92, 85, 3, 66,99, 42,47, 78,23, 26,40,
  80, 9, 41,57, 75,43, 32,83, 40, 3, 56,13,  4,81, 93,63,
  31,96, 29,99, 95, 2, 40,76, 34,97, 26,36, 59,86, 72,27,
  77, 5, 60, 4, 67,85, 64,30, 18, 9, 38,45,  2,89, 99,76,
  41,32,  2,85, 29,43, 87,60, 19,73, 88,52, 89,26, 23,30,
  48,87, 93,45, 24,88, 21,39, 85,91, 86, 6, 91,36, 93,20,
  35, 0, 90,77, 22,11, 98,30, 66,23, 56,83, 67, 4, 38,13,
  16,56, 50,99, 80, 1, 97,14, 14,81, 16,30, 52,57, 50,12,
  56,37, 91,98, 65,75, 90,20, 15,47, 19,42, 81, 9, 49,34,
  70,55, 20,69, 67,42, 67,23, 35,27, 28,65, 79,27, 18,19,
  28,77, 28,87, 89,99, 34,28, 53,25, 79,30, 80,39,  3, 5,
  11,86, 37,61, 87,19, 13,13, 31, 5, 86,67,  9,74, 17,46,
  53,22, 56,78, 13,19, 15,86, 31,34, 63,84,  9,50, 74,62,
  23,61, 53,42, 47,87, 93,25, 38,47, 25,83, 61,56, 14,90,
  99,99, 71,30, 45,55,  8,41, 51,91, 25,39, 16,89,  2,60,
  27, 3, 44, 7, 22, 3, 16,61, 77,29, 76,57, 44,14,  5,40,
  47,40, 43,22,  8,21, 56,63, 79,10,  9,92,  3,93, 45,33,
  77,92, 65,47, 78,83, 42, 4, 66,98, 67, 2, 25,10, 68,27,
  20,95,  1,11, 89,48, 70,44, 36,98, 38,20, 69,73,  1,91,
  40,28, 10,44, 82,46,  0,35, 79,72, 76,28, 91,15, 63,96,
  73,18, 78,48, 89,70, 24,54, 95, 9,  7,18, 98,75, 62,80,
  25,67,  3,86, 85,82, 28,33,  6, 2, 45,50, 46,53, 11,90,
  86,63, 52,42, 77,94, 22, 2, 44,97, 30,52, 55,24, 17,36,
  73,36, 94,40, 18,35, 75,70, 29,78, 98,81, 18,48, 12,87,
  68,71, 52,81, 96, 0,  8,76, 82,72, 77,98, 76,66, 63,54,
  19,50, 14,68, 37,24, 92,92, 91,97, 92,44, 41,99, 84,27,
   9,67, 78,68, 11,80, 56, 4, 46,30, 94,19, 28,22, 43,23,
  71,33, 78,64, 40,54,  1,23, 68,10, 14,92, 39,19, 47,77,
  96,56, 82,78, 41,34, 18,52, 15,53, 57,82, 19,81, 39,39,
   2,14, 68,82, 40,77, 13,39,  2,33, 14,30, 32, 9, 80,13,
  88,77, 46,27, 23,60, 63,55, 81,46, 95,17, 27,17, 83,40,
  10,27, 46,97, 65,49, 98,87, 39,96, 89,35, 88,93, 42,51,
  27,43, 15,38, 56,93, 83,98, 74,23, 72,71, 39,80, 78,44,
  13,30, 78,73, 35,55, 80,11, 66,58, 15,43, 56,42, 63,92,
  92,50, 49,97, 60,87, 50,51, 97,74, 85,39, 25,38, 54,16,
  73,19, 42, 1, 94,61, 10,46, 70,22, 89,33, 27,64, 88,66,
  98,15, 12,57, 10,53, 51,94,  8,52, 82,42, 49,76, 15,60,
  41,82, 78,72, 23,67
};
const int p_100_5[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  626,
  /// Array of preferences
  17,15, 64,19, 48, 0,  4,63, 87,70, 41,89, 67,17, 55,63,
  65,83, 30,50,  2,23, 94,13, 41,34,  6,56, 87, 0, 70, 4,
  36,40, 49,61, 92,83, 89,73, 75,59, 64,55, 68,98, 10,97,
   7,74, 10,42, 15,86, 89,15, 39, 2,  6,89, 89,53, 89,52,
   8,87, 48,56, 61,70,  4, 6, 19,69, 19,29,  0,73, 52,59,
   5,51, 58,98, 63,16, 11,43,  8,21, 84,85, 84,66, 62, 5,
   0,54, 32,44, 38,33, 23,67, 51,38, 71,33, 69,62, 90,79,
  89,21,  9,50, 34,56, 37,40, 62,93, 73,11, 36,72, 94,19,
  43,52, 33,66, 91, 7, 25,44, 40,39, 75,70, 15,96, 76,51,
  21,49, 37,49, 31, 5, 14,60,  4,77, 48,14, 22,37,  4,68,
  87,62, 53,79, 31,19, 63,88, 65,64, 46, 2, 45,34, 50, 7,
  92,78, 53,60, 71,93, 26, 7, 72,34, 93,70, 58,11, 39,34,
  89,84, 17, 0, 17,51, 73, 2, 74,50, 38, 3, 45,12, 16,11,
  78,10, 93,51, 39,48,  3,11, 44,21, 29,68, 15,42, 78,80,
  14,50,  3,89, 54,18, 35,75, 86,18,  5,42, 96,85, 68,88,
  87,19, 17,41, 83, 3, 29,82, 86,57,  2,77, 29,56, 48,57,
  65,28, 51,42, 23,14, 20, 4, 54,22, 20,55, 54, 2, 35,51,
  26,19,  8, 0, 39,54,  1,45, 46,64, 92,60, 34,35,  7,37,
  86,61, 66,97, 16,27, 85,51, 79,69, 88, 3, 33,95,  7,70,
  98,45, 12,46, 20,65,  5,97, 78,72, 63,17, 35,14, 45,69,
   7,98, 66,60, 91,30, 75,81, 79,78, 12, 8, 48,20, 62,13,
   4,54, 59,12, 30,64, 11,84,  2,71,  0,61, 91,62, 46,37,
  24,33, 92,20,  9,42, 51,83, 97,38,  1,53,  0,63, 97,45,
  58,32, 71,90, 97,85, 85,94, 64,20, 25,55, 27,58, 56, 0,
  69,14, 45,63, 77,50, 80,56, 35,38, 62,15, 68,78, 71,83,
  79,20, 15,61,  7,90, 57,26, 63,64, 73,59, 39,69, 72,27,
  78,67, 84,74, 82,54, 51,10, 91,60, 53,14, 61,36, 35,11,
  52,50, 78,66, 12,10, 93,98, 64,17, 47,20, 85,47, 15,56,
  34,64, 15,38, 19,57, 38,78,  6, 4, 41,29, 61,19, 31,75,
  72,82, 60, 3, 56,87, 86,60, 54,65, 79,37,  5, 9, 31,55,
  35,98, 63,19, 63,78, 48,71, 12,40, 63,35, 19,86, 56,61,
  37,54,  1, 8, 81,48, 66,57, 36,43, 30,70, 90,59, 25,95,
  18,44, 49,14, 75, 6, 89,80, 49,65,  3,36, 36,12, 10,61,
  85,30, 52,15,  0,42, 86,23,  4,52, 67,86, 91,46, 31,25,
  70,16, 25,54, 95,75, 31,72, 61, 0, 19,65, 78,76, 69,63,
  29,45, 53,81, 84,90,  9,27, 89,36, 28,10, 83,86, 26,21,
  67,88, 74,27, 62,74, 60,49, 40,58, 89,25, 57,43, 21,94,
  83,99, 20,80,  7,33, 50,93, 92,62, 35,58, 79,41, 80,33,
  42,59, 15,97, 11,37, 27,16, 71,98, 50,68, 41, 0,  9,35,
  71, 7, 81,54, 90, 9, 63,72, 15,23, 11,62, 65,18, 72,96,
  67,55, 26,98, 76,65, 39, 9, 19,72, 90,70, 77,79, 50,90,
  23,17, 20,16, 56,78, 15,92,  2,12, 29, 9, 20,29, 78,36,
  88,26, 47,43, 72,21, 24,84, 41,61, 70,84, 54,36, 25,69,
  20,99, 46, 4,  0,29, 42,65,  6,50,  5, 7, 66,43, 70,42,
  56, 3, 23,78, 36,38, 98,73, 83,77, 49,41, 59,96, 60,79,
  18,71, 49,34, 68, 7, 89,98, 31, 4, 98,94, 62,36,  3,92,
  10,22, 28,94, 40,74, 65,32, 82,62, 72,38, 97,12, 94,30,
  51,67, 23,34, 39, 1, 52,16, 37,16, 24,92, 66,51, 15,18,
  24,73, 68,53, 58,82,  7, 6, 75,36, 99,36, 12,81,  5,16,
  29,17, 13,22,  3,20, 66,20, 31,78, 56,92, 65,44, 32, 0,
  66,67, 56,66, 29,89, 14,18,  5,64, 63,26,  5,20, 21, 0,
  42,49, 57,68, 37,22,  1,71, 52,95, 77,46, 75,57, 90,31,
  25,47,  4,34,  2,54, 23,95, 17,79, 80,79, 19,94, 63,76,
  26, 3, 22,66, 84,18, 84,50, 57,17, 16,46, 18,63, 78,74,
  43,27, 86,29, 30,60, 18,59, 83,28, 37,44, 79,75, 20,42,
  17,32, 70,49, 44,70, 33,90, 30,99, 13,78, 46,47, 25,42,
  25,78, 96,68, 17,86, 44, 1, 41,63, 34,27, 18,60, 69,13,
  37,43, 77,49, 98, 4, 35, 3, 85,26, 20,56, 75,53, 18,62,
  23, 8, 68,12, 16,57, 95,37, 97,80, 16,75, 11,13, 62,53,
  83,42, 68,96, 23,28,  7,58, 44,46,  0,52, 53,89, 85,59,
  27,13, 95,34, 40,26, 75,95, 32,96,  9,31, 10,58,  7, 3,
  20,96, 29,62, 36,77, 28,34, 83,80, 55, 7, 92,80, 51,33,
  49,92, 18, 9, 88,90, 38,12, 12,61, 89,55, 90,83, 70,25,
  73,51, 31,91, 59,94, 16,58, 35,13, 66,90, 22,50, 93,17,
  14,72, 67,56, 17, 6, 33,56, 41,92, 13,50, 11,31, 39,70,
  70,26, 66,55, 90,43, 19,21, 25,63, 97,23, 91,96, 20,76,
  69,91, 18,80, 90,16, 30,10, 21,18, 61,47, 13,82, 58,12,
  77,94, 59,17, 30,71, 70,62, 52,45, 23,72, 75,87, 12,51,
  59,43, 21,68,  3,63, 85,11, 31,99,  5,69, 85,31, 42,66,
  86,87, 96,51, 44,44,  2,73,  2,60, 85,96, 76,52, 11, 8,
  92,90, 50,86, 58,33, 26, 1, 40,97, 89,66, 22,92, 75,64,
  97,13, 56,23,  1,76, 41,53, 87,38, 73,74, 97, 7, 10,75,
   9,43, 80, 3, 11,10, 63,27, 59,60, 64,11, 17,40, 37,68,
  38,11, 44,14, 78,45, 38,57, 24,49, 98, 9, 63,52, 81,27,
  10,62, 73, 3, 11, 6, 68, 4, 65,37, 42,30, 37,67, 33,40,
  74,54, 89,70, 93,65, 70,65, 68,75, 99,74,  9,67, 25,75,
   0,12, 71,80, 58,34, 63,96,  5,60, 92,26, 46,10, 79,55,
  31,18, 40,23, 77,55, 22,59,  5,99, 24,62, 83,21, 42,29,
  86,31, 67,96
};
const int p_100_6[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  2051,
  /// Array of preferences
  15,65, 79,35, 89,65, 75,69, 95, 6, 97,98, 47,18, 43, 3,
   2,60, 52,86, 92,62, 75,97, 10,60, 75,24, 39,15, 78,75,
  45,12, 69,91, 16,42, 28,60,  7,38, 47,16,  2,23,  7,25,
  33,51, 63,47, 79,91, 53,17, 54,85, 85,56, 42,98, 42,85,
  12,96, 20,95, 53,46, 18,85, 26,98, 28,30, 10,90, 12,67,
  66,76, 67,66, 69,74, 18,40, 19,59, 67,43, 37,68, 81,67,
  91,95, 75,19, 46,39, 43,88, 47,58, 37,49,  4,37, 85,17,
  72,68, 63,13, 21,79, 15,98, 14,17, 69, 9, 37,50, 36,46,
  31, 2, 48,46, 57,76, 25,78,  1, 3, 29,42, 72, 3, 71,19,
  98,50, 66,56, 91,53,  2,53, 92,48, 11,50, 14, 7,  8,25,
  48,16, 56,65,  3,31, 60,89, 91,22, 60,46, 10,95,  3,99,
  99,54, 58,26, 53,95, 43, 0, 19,98, 75,11, 70,58, 33,48,
   8,33,  7,92, 64,53, 42,52, 63,95,  7,58, 48,94, 10,24,
  62,79, 25,67, 89,33,  0,62, 55,36, 54, 1, 85,98, 79, 5,
  53,15, 23,64, 49, 9, 55,57, 12, 4, 71,95, 53,67, 99,85,
  76,72, 16, 9, 55,22, 44,26, 86,73, 90,42, 44,84, 15,11,
  48,38, 66,26, 53,63, 62,14, 81, 0, 35,74, 46,72, 96,59,
  82, 6, 15,13,  1,19, 59,41, 23,69, 65, 0,  7,83, 71,72,
  70,40, 99, 4, 12,25, 77,27, 66,21, 89,49, 26,59, 53, 7,
  71,22, 80,73, 55,84, 27,41,  8,72, 93,90, 23,24, 11,86,
   4,25, 40, 9, 10,35, 78,61, 79,31, 43,75, 42,43, 29,62,
  31,59, 22,92, 50, 5, 17,51, 78,76, 93,28, 13,20, 14,31,
  87,73, 85,77, 26,28, 75,81, 73,72, 84,20, 97,29, 74,88,
  89,14, 78,49,  6,42, 30,90,  6,70, 36,95, 10,14, 95,17,
  71,48, 73,27, 42,25, 64,50,  1,89, 40,56, 97,93, 24,14,
  23,86, 38,92, 53,80, 90,75,  1,17, 76,12, 26, 3, 16,17,
  39,17, 36,60, 10,43, 36,11, 71,65, 11,21, 41,44, 71,98,
  57,66, 41,27, 68,49, 24,37, 82,89, 63,63, 48,39,  3,25,
  94,34, 43,12, 89, 3, 45,85, 80,33, 58,67, 86,82, 50,50,
  90,71, 48,95, 52,89, 24,24, 92,32, 41,69, 21,74, 40,71,
  27,80, 98,27, 36,98, 38,57, 17,41, 32,94,  0,92, 58,46,
  29,54, 27,85, 47,59, 83,16, 55,67, 34,11, 72,36, 89,57,
  19,40, 23,18, 82,90, 58,55, 60,13, 87, 7, 49,62,  7,63,
  67, 7, 10,82,  1,46, 26, 5, 69,99, 83,30, 95,66, 85,54,
  15,67,  5,62, 88,27, 41,68, 87,13, 83,63, 66,55, 61,78,
  64,15,  6,78, 55,55,  6,56, 94,83, 23,10, 17,65, 77, 8,
   5,93, 49,29, 30,88, 21,73, 36,55, 21,54, 24,19, 52,97,
  65,60,  5,40, 88,96, 79,45, 23,66, 28,67, 27,73, 54,91,
  40,12, 55,66, 97,67, 86,83, 79,97, 12,90,  6,72, 40,81,
  31,70, 87,55, 48,19, 60,26, 35,93, 25,40, 73,80, 20, 8,
  67,39, 80,19, 74, 6, 58,52, 30,52, 35,84, 73,39, 98, 3,
  16,69,  9, 9, 11,76, 93,24, 22,44, 11,49, 92, 2, 44,82,
  40,62, 17,91, 31,75, 24,56, 18,24, 69,14, 27, 8, 94,85,
  83,84, 51,28, 65,50, 45,88, 46,89, 41,34, 85,19, 71,91,
  85,36, 23, 0, 14,90, 85,42, 98, 1, 30,81, 97,11, 79,11,
  59,47, 66,88, 67,79, 14,55, 70,68, 73,19, 67,25, 36,20,
  14,70, 35,85, 69,51, 37, 5, 59,51, 51,31, 32,52, 78, 7,
  46,88, 25,34, 96,40, 93,15, 15,50, 68,10, 19,49, 86,34,
  77,88, 25,32, 99,12, 65, 5, 57,43, 68,84, 48,34, 16,10,
  86,62, 68,38, 20,41, 55, 5, 78, 5,  9,11, 38, 5, 86,32,
  98,39, 14,41, 88,95, 54, 8, 70,67,  0,82, 25, 6, 11,77,
  83,14, 65,22, 32,79, 78,80, 42, 0, 94, 6, 12,22, 18,45,
  17,67, 96,31, 89,93, 67,51, 54,30, 55,27, 85,29,  4,99,
  58,56, 84,34,  6,57, 74,58, 91,88, 94,68, 26,60, 73,12,
  95,68, 63,99, 28, 7,  5,10, 92,77, 91,66, 17,63, 41, 2,
  10,48, 23,16, 33,46, 21,61, 70,95, 58,48, 61,12, 86,58,
  10,84, 89,71, 62,15, 97,20,  3,54, 82,86, 20,23, 27,11,
  23,60, 75,34, 41,26, 84,84, 94, 2, 33,76, 69,71, 20,37,
  69,62, 36,34, 92,80, 37,15, 34,32, 34,27, 53,76, 46,50,
  19,38, 20,66, 23,46, 54,48,  1,59, 95,73, 16,23, 17,18,
  38,82, 84,30, 35,65, 69, 8, 25,72, 99,48, 36,96, 24,45,
  96, 6, 41,97, 95,67, 19,96, 63,21, 73,53, 53,57,  6, 1,
  61,34, 11,78, 89,88, 80,48, 33,45,  0,79, 18,58, 75,75,
   1,23, 32,13, 52,62, 13,35,  2,57, 25,76, 36,52, 34,24,
  42,86, 56,60, 47,35, 57,51, 51,64, 50,51, 41,82, 31,40,
   4,19, 44,94, 60,42, 39,10,  0,58, 39,78, 96, 7,  9,53,
   2,22, 29,53,  2,39, 82,71, 70,63, 67,60, 45,45,  0, 4,
  73,82, 70,57, 52,94, 65, 8, 45,28, 19,35,  1,32, 87,56,
  70,33, 24,67, 63,77, 74,45, 98,56, 36,24, 22,37, 99,44,
  77,50, 79,13, 25,54, 22, 8, 22, 9, 65,54, 97, 3, 57,78,
   4,75, 25, 8, 69,17, 82,62, 79,77,  3,61, 46,84, 96,37,
  25,24, 75,20, 87,48, 92,24, 72, 7, 28,45, 83,31, 69,64,
  40,38, 19,11, 43,18, 27,96, 43,54, 10, 3, 69,57, 73,77,
  92,98, 15, 9, 27,88, 57,96,  8,73, 32,78, 57,35, 42,63,
  73,89, 62,92, 49,17, 50,22, 55,12, 28,59, 34,21, 21,20,
  80,47, 48,18, 99,81, 40,83, 76,66, 71,33, 30,98, 84,79,
  82,64, 73,69, 57,29, 24, 3, 46,59, 79,65, 95,51, 52,39,
  86,56, 85,53, 51,51, 81,64, 25,18, 92, 8, 90,55, 39,42,
  40,64,  8,44, 22,21, 80,88,  9,56, 82,61, 51,27, 92,22,
   7,71, 33, 0, 13,65,  3,55, 22,50, 70,35, 78,26, 43,29,
  56,21, 83,25,  6,83, 68,63, 16,98, 71,89,  0, 2,  6,50,
  87,93, 76,99, 89,87, 40,32, 55,61, 99,33, 54,56, 64,46,
  88,38, 40,93, 67,69, 96,82, 32,35, 78,43, 71,99, 72,12,
  45,64, 32,55, 26,52, 46,52, 72,85, 85,60, 90,27, 11,83,
  36,13, 46,25, 11,14, 35,23, 75,23, 26,65,  9,34, 60,48,
  66,29, 91,39, 89,64,  8,74, 32, 9, 91,47, 50,95, 49,83,
  57,79, 52,53, 37, 1, 70,66, 92,87, 21,71, 27,35, 36,44,
   3,90, 66, 8, 33,99, 81,77, 86,85, 55,88, 66,25, 75,15,
  20,59, 74,93, 59,83, 22,76, 89,83, 49,79,  7,44, 84,75,
  81,38, 28,47, 45, 7, 93,77, 30,85,  6, 9, 11,72, 64,35,
  61,39, 47,77, 51,95, 80,93,  9, 2, 41,41, 29,82, 79,47,
  67,28, 45,83, 19,70, 57,60, 45,43, 32,33, 95,23, 56,89,
   4,36, 63,85, 31, 6, 82,59, 59,80, 35,52, 10,74, 59,36,
  55,31, 97,59, 76,49, 29,51, 52,13, 57,28, 70,94, 54,28,
  41, 7, 48, 4, 61,40,  0,85,  3,51, 40,72, 46, 4, 18,80,
  32,67, 37,26, 99,29, 84,42,  3, 7, 40,46, 50,49, 32,64,
   2,43, 86,44, 76,34, 22,53, 48,33, 43,67, 64,63, 67,99,
  80,71,  0, 3, 22,96, 59,44, 57,52, 55,10, 78,64, 76,93,
  53,38, 70,18, 50,39, 93,87, 45,47, 40,88, 73,68, 33,79,
  86,84, 47,53,  9,94, 11,41,  0,16, 97,64, 75,36, 37,35,
  49,78, 36,47, 85,55, 30, 6, 41,77, 77,53,  0,70, 26,27,
  75,21, 39,89, 49,14, 50,94, 81,76, 14,86, 19,44, 56,53,
  29,39, 59,70, 52,52, 84,88, 46,63, 57,27, 24,60, 43,22,
  16, 5, 32,10, 64,28,  2,89, 24,55, 84,40, 11,42, 33,27,
  55,70, 54,25, 88,16, 51,65,  0,21, 33,81,  5,34, 57, 5,
  37,58, 30,70, 47, 1, 48,21, 21,72, 64,21, 54,63,  3, 3,
  66,60, 54,88,  2,47, 22,89, 93,33, 71,46, 46,54, 34,20,
  76,31, 58,37, 61, 0, 82,73, 92,38,  2,74, 50,44, 86,96,
  14,18, 86,51, 21, 2,  3,52, 12,44, 63,59, 33,97, 99,21,
  65,64, 41,56, 58, 5, 65,24, 70,30, 83,55,  0,44, 60, 3,
  83,85,  2, 4, 88,59, 90,88, 82, 8, 24,63, 96,93, 51,25,
  43,25, 16,37, 52, 3, 81,34, 90,34, 66, 7, 55,33, 80,77,
   5,14, 96,98, 47,86, 17, 2, 56, 2, 37,97, 12, 1, 22,52,
  87,86, 23,61, 60,55,  4, 0, 52, 5, 19,12, 25,37, 72,37,
  57,19, 50,48, 28,53, 20,57, 83, 7, 64,73, 39,54, 83,24,
  10,80, 81,21, 35, 3,  0,94, 41,33, 86,18, 72,39, 82,38,
   7,57,  7,93, 52,50, 94,19, 96,34, 41,65, 71,77, 77,57,
  17,64, 64,58, 58,97, 83,72, 11,57, 88,85, 78,17, 83,28,
  88,64, 35,35, 99,83, 36,12,  1,99, 17,69, 68,66,  4, 2,
   8,64, 87,49, 88,88, 72,30, 65, 6,  5,19, 95,54, 71,36,
  40,74, 14,49, 87,92, 23,93, 69,93, 21,18, 30,58, 69,88,
  20, 6, 28,29, 64,69,  2,58, 56,79, 89,56, 65,34, 90,87,
  33,35, 65,67,  6, 5, 42,19, 74,92, 18,57, 85,61, 52,99,
  21,55, 18,49, 53,82, 86,48, 36,50, 64,86, 59,55,  5,63,
  71,20, 18,56, 29,96,  9,85, 57,72, 15,92, 11,89, 29,67,
  70,62, 77,22,  0, 0, 79,37,  7,80, 60,87, 48, 1, 37,88,
   8,36,  7,40, 10, 5, 36,74, 43,84, 63,62, 55,21, 44,25,
  49,28, 81,17, 23,27, 76,80, 95,20, 13,89, 20,79, 41, 1,
  75,28, 76,78, 72,34, 58,44, 15,80, 30,95, 19,21,  2, 2,
  11,88, 58,94, 91, 8, 63, 3, 22,31,  8,81, 90,65, 58,58,
  32, 1, 80, 6, 24,10, 33,26, 87,67, 92,47, 75,35, 38,15,
  94, 1, 81,48, 79,55, 39,50,  1,30, 26,79, 94,54,  3,13,
  78,27, 82,63, 83,91, 23,11, 49,84, 36,18, 97,72, 46,57,
  36,87, 96,94, 58,84, 47,19, 31,43, 21,89, 71,47, 24,71,
  18, 6, 19,15, 12,23, 93,49, 99,96, 59,22, 34,44,  5,92,
  38,25, 99,57, 51,47,  9,55, 25,51, 39,77, 19,36, 69,16,
   7,20, 25, 0, 13,40, 35,31, 37,52, 25,69, 54, 3, 39,40,
  53,35, 48,27, 47,87, 81,12, 29,10, 25,98, 46,83,  4, 1,
  71,92, 64,34, 13,94,  4,32, 66,38, 92,23, 94,55, 28,73,
  82,55, 50,30, 72,99, 71,97, 78,52, 67,89, 21,57, 98,98,
  10,54, 23,70, 87,88, 68, 2, 39,66, 26,41, 43,34, 76,96,
  81,13, 47,42, 15,59, 97, 5, 84,41, 72, 2,  9,99, 35,53,
  23,37, 67,74, 94,60, 46, 8, 56,77, 88,36, 12, 5, 70,90,
  31,18,  8, 2, 84,14, 59,34, 22, 2, 96,13,  5,29, 98,80,
  25,91, 21,41, 95,94, 19,10,  3,18, 99,40,  3,72, 63,27,
  17,81, 27,77, 75,48, 66,61, 60,45, 16, 3, 88,28, 82,14,
  84,39, 89,69, 59,19, 67,36, 60,73, 87,43, 93,99,  2, 8,
   9,19, 59,68, 47,27, 50,35, 18, 8, 70,89, 29,92, 26,99,
  50,13, 99,89, 63,58, 53, 5, 85,16, 44,20,  3,34, 99,64,
  19,93, 85,47, 62,28, 56,74, 11,59, 48,60, 66,92, 90,48,
  43,83, 38,36, 51,44,  5,31, 90,58, 64,94, 32,83, 94,14,
  72, 8, 86,92, 21,42,  6,87, 23,75, 85,21, 81,80, 94,86,
  12,58, 88,89, 75,63, 61,73, 31,71, 28,31, 88,84, 38,24,
  29,59, 62,76,  4,20,  5,68, 14,52, 18,77, 44, 5, 35,56,
   3,21, 75,32,  8,29, 18,98, 39,74, 33,83, 48, 7, 73,18,
  11,43, 81,82, 53,69, 63,45, 50,29, 56,99, 50,97, 78,93,
  88,86, 34,37, 23,65, 95,33, 75,66, 89,12, 28,70, 60,64,
  80,57, 13,71, 30,87, 78,36,  1,90, 51,55, 93,97, 38,39,
  63, 5, 88,53, 41,48, 64,11, 53,64, 69,24, 24,92, 19, 4,
  82,75, 24,75, 82,94,  5, 0, 71,88, 99,30,  6,61, 23,50,
  96,20, 46,96, 71, 3, 32,15,  9, 4, 47,67, 57,68, 77,83,
  97, 8, 51,11, 80,62, 22,26, 30,43, 36,53, 21,40, 99, 9,
  98,70, 27,42, 41,54,  4,43, 11,98,  5,12, 83,76, 25,36,
  62,18, 53,56, 17,53, 71,81, 10,76, 50, 0, 51,15, 53,59,
  36,27,  0,30, 35,68,  3,14,  7,70,  1,82, 54,54, 16,27,
  31,42, 99,38, 97,68, 36, 9, 90,44,  8,10, 84,99, 92,45,
  82,79, 57,88, 87,62, 56,78, 92,69,  0,63, 62,38, 95,83,
  18, 0, 41,87, 67,49, 84,54, 38,26, 50,23, 68,94, 97, 1,
  56,82, 19,55, 39,16, 11, 0, 82,60, 66,62, 19,28, 97,60,
  46, 3, 16,73, 38,16, 98,91, 23,33, 71,54, 24,12, 13,34,
  46,58, 90,12, 65,46,  2,80, 16,12, 68,78, 43,16,  1,75,
  41,11, 54,70, 89,77, 40,47,  3, 5, 19,69, 69,50, 79,57,
  68,13,  0,53, 80,97, 39,52, 80,39, 17,50, 47,30, 28,91,
  87,58, 29,48, 18,35, 17,23, 41,81, 31,27, 80,91, 62,45,
  39,95, 65,14, 27,66, 62, 7, 11, 2, 44,78, 26,96, 74,51,
  40,59, 37,32, 32,31, 59,65, 49, 1, 18,26, 33,41, 74,59,
  18,62, 77, 2, 40,91, 89, 9, 69,43, 83,50,  9,44, 87,81,
  65,86, 27,92, 69,23, 83,51, 10,21, 23,26, 49,38, 77,82,
  73,88, 47,89, 81,44, 81,57, 80,36,  7,19, 45,92, 25, 7,
  35,26,  3,20, 99,84, 19,68, 80,86, 67,41, 55,65, 52,90,
  51,90, 45,22, 10,96, 31,62, 17,21, 97,13, 98,53, 64,84,
  91,11, 45, 6, 26,40, 75,55, 61,21,  2,61, 20,78, 60,32,
  45,39, 49,80, 56,29, 16,21,  9,82, 12,36, 96,88, 60,31,
  49,31, 11, 1, 58,13, 42,40, 13,58, 96,95, 42,46, 80,63,
  89,35,  2,94,  3,82, 98,23, 33,50,  3,95, 23,91, 86,50,
  80,14,  3,11,  8,39, 66,27, 95,32, 15,83, 27,99, 95,90,
  12,66, 14,82, 32,82, 15,89, 26,18, 87,63, 80,32, 91,54,
  45,26, 37,78, 99,23, 22,32, 65,66, 83, 5, 91, 7, 37,14,
  40,51, 34,84, 60,36, 30,55,  3,46, 78, 9, 10,11, 36, 1,
  29,97, 66,67,  6,88, 75,58, 73,79, 63,71, 39,88, 97,47,
  91,45, 19,34, 34,82, 13,59, 55,28, 37,11, 22,27, 88,10,
  16, 8, 65,84, 54,97, 84, 1, 76,50, 24,30, 83,10, 94,90,
  40, 4, 54, 0, 23,38, 58,93, 43,76,  0,47, 73,74, 80,74,
   7,30, 93,54, 91,20, 30,94, 55,83, 22,84, 80,24, 72,60,
  80, 9,  4, 9, 56,28, 47,71, 27,30, 21,62, 97,46, 74,73,
  66,30, 75,37, 92,49, 89,50, 63,43, 49,89,  8,89, 47,17,
  56, 9, 59, 3, 72,28, 58,47, 61, 1, 90, 8, 61,50, 26, 1,
  84,10, 70,91, 43,36, 65,27, 18, 1, 59,13, 70,38, 99, 2,
   9,88, 79,17, 66,49, 30,19, 86,41, 44,69, 80,84, 68,53,
  45,98, 97,82, 49,97, 35,89, 24,78, 90,78, 23,20, 90,51,
   5,27, 62, 1, 39,31, 13,37,  9,91, 65,49,  9,71, 61,16,
  79,42, 63, 2, 10,22, 26,89, 38,64, 19,57, 72,78, 88,42,
  52,95, 42,47, 36,61, 49,24, 50,25, 89,41, 90, 6, 89,28,
  65,31, 47,51, 10,42, 99,53, 10,88, 40,30, 98,36, 92,92,
  45,53, 18,81, 84,58, 86,42, 86,87, 23,30, 98,33, 30,24,
  30,29, 15,58, 54,53, 77,60, 70,50, 78,38, 62,20, 60,83,
  73,60, 87,65, 98,71, 84,61, 55, 9, 78,45, 53,88, 78,62,
   3,38, 54,34, 51,67, 99,88, 30,40,  8,22, 26,72, 38,91,
  57,62, 18,20, 93,16, 18,18, 80,37, 66,20, 17, 3, 50, 6,
  35,59, 20,20, 31,55, 59, 0, 51,71, 22,65, 94,15,  8,56,
  91,33, 31,13, 66,46, 64,39, 57,33, 98,64, 33,24, 25,68,
  97,89, 18, 3, 61,76, 61,80, 78,97, 32,49, 36,83, 87,76,
  80,52,  5,61, 66,12,  4,89, 47, 4, 45,72, 49,71, 36,43,
  13,30, 86,28, 39,36, 48,44, 91,13, 76,83, 25,64, 62,91,
  62,64, 91,25, 87,40, 69, 2, 70, 1, 79,95, 84,91, 83,23,
  68, 3, 74,77, 10,49, 10,61, 53,10, 87,99, 98,93, 41,73,
  96,21, 85,80, 96,74, 79,53, 46,61, 79,94, 95,56, 33,68,
  91,92, 63,35, 52,79, 38,52, 51,81, 64,44, 83,37, 50,72,
  83, 2, 35,34, 91, 1, 20,73, 16,88, 47,12, 29,95,  4,22,
  31,99, 24,83, 76,14, 14,99, 31,56,  4,51, 61,64, 19,24,
   6,73, 28,54, 87,45,  3,49, 24, 2, 38,21, 44, 0, 19, 3,
  82,24, 44, 9, 63,25,  0,39, 65,47, 54,68, 85,49,  7, 6,
  48,17, 98,57, 57,40, 67,59, 94,29, 73,83, 55,82, 75,52,
  57,50, 39,27, 82,42, 57,25,  9,28,  0,56, 84,65, 18,61,
  94,63, 34,26, 29,34, 41, 8, 60,66, 77, 7, 43,78, 19,97,
  69,69, 65,80, 59,59, 80,53, 46,40, 55,24, 93,31, 83,36,
  33, 7,  5,80,  7,94, 61,87, 32, 7, 50,78, 69,34, 41,25,
   8,90, 86,15,  8,14, 57, 2,  0,20, 99,46, 68,22, 17, 0,
  15,72, 12,42, 37,17, 32,36, 11,45, 77,26, 17,76, 96,38,
  77, 1, 28,46, 65,30, 69, 7, 61,79,  6,97, 73,41, 32, 2,
  60,12, 80,34, 89,16, 45,79, 90,73, 85, 5, 43,50, 25,27,
  90,36, 77,67, 98, 4, 95,81, 20,64, 61,86, 36,39, 85,83,
  91, 5,  7,85, 61,35, 69,76, 59,86,  3,66, 15,31, 11,28,
  52,20, 60,37, 29,52, 26, 7, 36, 6, 46,35, 65,38, 11,13,
  36,91, 36,14,  8,46, 91,34, 26,66, 45,21, 13,99, 53,11,
  46,19,  7,23, 59,52, 10,45,  2,65, 53,99, 86,74, 90,70,
  88,32, 32,11, 87,15, 91,35, 32,63, 25,55, 81,26, 33,49,
  39,12, 46,55,  2,41, 14, 9, 62,32,  4,78, 31,19, 45,13,
   5,35, 67,64,  4,85, 13,95,  4,70, 91, 9, 17,82,  2,96,
  80,89, 14,62, 58, 9, 21,67, 76, 7, 39,62, 20,14, 25,58,
  76,76, 89,38, 73,28, 72,73, 55,14, 97,22, 92,56, 91,29,
  15,39, 55,39, 59,17, 21,96, 83,47, 41,94, 81,50,  4,88,
  74,48, 91,69, 79,89, 27,97, 27,82, 84,15, 67,15, 42,62,
  68,60, 92,34, 78,60, 26,87, 17,71, 69,29, 63,72, 82, 2,
  72,49,  6,10, 68,59, 58,36, 39,96, 89,82, 99,65, 47,22,
  74,95, 33,96, 11,33, 66,14, 45,62, 86,17, 49,39,  1,77,
  40,33, 84,83, 48,65, 89,91, 46, 7, 77,25, 47,26, 40,17,
  35,60, 93,68, 67,61, 20,19, 49,42, 47,85, 79,10, 98,48,
  60,54, 61,58, 56,94, 69,75, 44,44, 84,56, 24,73, 88,69,
  43,63, 22,60, 24,80, 41,37, 46, 9, 79,76, 94, 9, 87,21,
  74,91, 20,25, 32,93,  5,51,  4,54, 61,28, 18,34, 24,99,
  94,18, 95,29, 56,13, 17, 1, 92,33, 15,74, 10,85, 29,75,
  83,83, 70,47, 71,40, 80,45, 91,87, 46,92, 59,40, 52, 4,
  74,36, 87,87, 26,10,  4,74, 80,92, 38,99, 92,71, 50,37,
  29,72, 78,72, 91,23, 13,84, 74, 5, 74,26, 23,90, 99,73,
  51,41, 52,44, 32,47, 21,22, 25, 3, 68,54, 13,83, 57,44,
  31,20, 13,16, 65, 9
};
const int p_100_7[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  2170,
  /// Array of preferences
  27,18, 33,70, 62, 7, 50,64, 31,52,  2,89, 70,14, 30,65,
  28,14, 21,76, 60,93, 69,88, 87,89, 98,74, 91,27, 15, 6,
  13, 6, 49,90, 68,46, 52,60, 25, 3, 32,34, 82,73, 75,14,
   1,40, 79,55, 72,26, 86,71,  0,97, 97,78, 67,31, 13,68,
  57,72,  2,11, 36,22, 51,95, 53,89,  6,55,  7,14, 37, 9,
  66,20, 50,13, 91, 8, 65,53, 62,75, 57,15, 88,57, 19,28,
  20,65, 65,12,  5,82, 43,62, 78,26, 97,89, 79,87, 36,67,
   2,96, 35,78, 31,74, 29, 7, 35,95, 65,94,  2,59, 81, 9,
  33,85, 86,34, 97,60, 98,25, 36,45, 60,20, 67,10, 24,79,
  62,68, 68,23, 69,98, 33,88, 18,27, 39,21, 54,51, 81,70,
  69,24, 91,13, 85,31, 78, 1, 50,23,  4,64,  2,45, 57,43,
  67, 9, 82,44, 24,92, 22,32, 23, 3, 95,62, 18,54, 36,14,
  93,52, 83,76, 31,94, 20, 3,  0,81, 80,46, 35, 8, 67,61,
  49,96, 74,24, 63,15, 42,75, 64,78, 40,13, 37,57, 98,36,
  75,27, 72,74, 43,55,  2,28, 75,24,  4,65, 73,76, 80,84,
  35,30, 60,40, 85,37, 37,22,  0,90, 18,62, 94,37, 58,70,
  99,33, 23, 4, 45,82, 65,16, 59,39, 47,32, 34,94, 31,19,
  27,59, 53,76, 23, 8, 20,93,  6,24,  8,97, 24,22, 35,97,
  72,18,  5,31, 25, 8, 40,49, 76,76, 20,86, 98,98, 75,23,
  84, 7, 22,52, 32,24, 39,34, 98,37, 98,88, 80,38, 91,65,
  83,70, 85,91, 55,82, 98,52, 67, 5, 27, 8, 49,82, 69,96,
   9,59,  6,67, 19,87,  8, 7, 90,36, 51,82, 60,86, 60,97,
  61,98, 71,17, 40,68, 30,70,  8,16, 31,25, 64,15,  2,36,
  15,36, 26,55, 66,13, 35,77, 55,56,  0,31, 31,95, 65,85,
  14,18, 81, 2, 37,26, 38,10, 99,32, 56,73, 86, 2, 82,49,
  37,60,  1,81, 29,58, 11, 2, 41,16, 80,11, 23,14, 97,45,
  47,68,  8,68, 22, 6, 14, 9, 81,23,  7,25, 45,83,  0,24,
  12,93, 66, 0, 60,63, 33,41, 65,78, 83,21, 38,65, 71,31,
  64,63, 26,60, 66,28, 26, 0,  8,76, 26,49, 94,53, 48,10,
   1,14,  2,85, 45,21, 19,88, 36, 7, 70,29, 65,62,  7,49,
  27,92, 66,50, 95,97, 75,61, 29,18, 32,14, 59,75, 38, 9,
  68,80, 42, 0, 17,86, 40,35, 49,20, 77,92, 88,80, 49,83,
  66,26, 37,91, 65,26,  9,23, 20,69, 82,70,  4, 9, 96,37,
  91, 3, 45,79, 62,90, 86,19, 65,32, 75,17, 27,31, 82,29,
  27, 6, 53,67, 35,43, 73,25, 66,80, 82,14, 14,83, 15,32,
  50,92, 75,92, 21,10, 81,10, 16,10, 86,79, 34,10, 21,65,
  47,61, 71, 8, 80,87,  7,45, 72,93, 46,72,  4,91,  5,50,
  11, 9, 73,28, 30,10, 62,39, 55, 9, 13,67, 64,20, 89,86,
  67,46, 94, 4, 97,33, 45,74, 23,55, 64,72, 73,97, 42,98,
  89,58,  9,88, 27,23, 58, 2, 49,79, 10,41, 17, 3, 46,98,
  89,11, 56,15, 50,10, 53,58, 82,43, 53,64, 59,34, 27,29,
  74,25, 84,55, 92,44, 90,32, 48,80, 78,19, 90, 9, 71,22,
  18,61,  3,60, 41,62, 10,71, 40, 6, 97,52, 99,72, 92,28,
  78,39, 20,23, 77,75, 53,50, 60,55, 65, 3, 71,96,  1,45,
  11,98, 30,22, 53,80, 36, 4,  7,20, 64,68, 43,49,  2,72,
  42,66, 18,45, 57, 5, 33,94, 22,89, 33,12, 99,69, 41,68,
   0,86, 99,42,  2, 2, 71, 4, 21,91, 33,73, 98,24, 96,12,
  20,75, 16,39, 21,92, 83,78, 80,97, 35,61, 68,13, 66, 9,
  14,58, 21,44, 72,32, 36, 2, 14,64, 96,79, 38, 0, 10,55,
  50, 6, 45,35, 60,48, 40,11, 56,90,  1,59,  4,85, 10,53,
  88,74, 84,46, 17,15, 96,29, 67,80, 68,79, 32,96, 20,16,
  27,69,  4,58, 25,62, 93,70, 45, 5, 23,12, 51,88, 83,97,
  17,31, 55,24, 50, 7, 99,58, 36, 6, 53, 5, 70,61, 46, 1,
  48,51, 17,19, 79,35, 28,18, 20,36, 73,42, 22,28, 25,44,
  89,44, 70,96, 77,31, 73,38, 40,82, 93,90, 28,52, 83,67,
  36,90, 52, 4, 76, 4, 43,84, 61,92,  6,77, 70,17, 34,40,
  67,48, 89, 0, 87,12, 94,20, 91,57, 69,48, 67,69, 52,34,
  70,38, 39,78, 44,66, 23,57,  1,70, 48, 5, 38,46, 66,15,
  47,53, 62,43, 32,27, 27,35, 91,96, 16,43, 45,10, 15,51,
  27,52, 62,17, 14,84, 56,43,  8,80, 93,93, 80,76, 22,16,
  66,71, 18,47, 91, 1, 16,25,  6, 8, 53,18, 60,51, 38,98,
  65, 9, 25,42, 74,60, 71,79, 37, 1, 61,30, 87,30,  9,14,
  12, 1,  0,73, 22,36, 27,66, 20,98, 55,12, 16,66,  7,63,
  57,98, 77,36, 70,37, 64,48, 26,53, 29, 6, 20,53, 92,93,
  92,87, 82,61, 95,82, 68,86, 18,11, 63,19, 18,93, 25,43,
  72,30, 43,11,  2,99, 40,32, 43, 0, 32,80, 89,89, 49,48,
  49,61,  3,26, 25,87, 40,66, 66,48, 29, 4, 45,69, 66,57,
   4,80, 73, 8, 84, 9, 36,12, 47,99, 52,47, 41,25, 55,33,
  84,18, 69,60, 67,16, 39,84, 93,64, 15,10, 10,98, 53,27,
  22,77, 81, 0, 65,71, 88,33, 87,29, 69,26, 68,18, 27,50,
  80,81, 34,48, 37,40, 15,23, 63,16, 92,19, 81,78,  5,46,
  28,44, 68,57, 82,32, 42,70, 92,49, 71,27, 36,33, 41,96,
   6, 2, 41,52, 19, 3, 92,20, 43, 5, 66,91, 40,37, 59,68,
  26, 5,  3,24, 55,14, 34,27, 92,50, 93,28, 90,58, 29,95,
  29,70, 47,97, 20,70, 51, 7, 27,75, 41,94,  7,17, 60,89,
  51,63, 96, 7, 21,74, 82,40, 24,42,  4,52, 80,37, 57,26,
  69,47, 53,20, 40,34, 64,76, 15,54, 64,54, 49,94, 65,20,
  18,43, 85,28,  4,61, 52,80, 13,11, 80,41, 59,10, 58,73,
  25,45, 37, 5, 82,87, 34,78, 39,92,  4,47, 60,80, 29,50,
  94,62, 91,17, 48,41, 12,59, 64,55, 24,41, 29,82, 63,34,
  31, 1, 62,14, 23,69, 34,63, 20,41, 45,62, 12,14, 56,68,
  18,91, 98,39, 84,81, 87,79, 26,78, 60,49, 44,47, 80, 4,
  25,64, 70,33, 53,38, 19,79, 31,44,  3,46, 41,85,  2,15,
  69,64, 37,70, 76,85, 36,54, 27,10, 70,41, 39,90, 58,16,
  59, 2, 91,31, 41,46, 29,72, 48,73,  8,81, 18,82, 65,77,
  84,91, 63,18, 52, 1, 49,63, 51,99, 37,47, 90,54, 58,79,
  20,97, 55,19, 12,26, 41,21, 48,14, 31, 9, 77,14, 30,88,
  32,85, 94,35, 62,74, 81,53, 95, 0, 52,12, 10,76, 11,59,
  67,73, 96,15, 55,23, 99,28, 93,33, 73,72, 38,85, 26,86,
  50,36, 68,42, 87,73, 34,41, 73, 1, 72,98, 34,22,  2,52,
   5,54, 34,72, 79, 8, 91,61,  5,51, 32,35, 12,21, 59,29,
  21,78, 19,55, 36,24, 95,68, 91,37, 54,49,  0,96, 95,42,
  57,78, 55,89, 49,22, 91,33, 93,10, 91,44, 77,19, 45,77,
  30,40, 87, 9, 51,57, 58,58, 97,64, 72,44, 43, 2, 93,71,
  26,96, 74,72, 92,34, 55, 1, 67,78, 10,93, 40,39, 58,37,
  98,59, 60,43, 90,15, 74,76, 38,17, 87,62, 42,58, 24,49,
  26,64, 79,59, 73,49, 29,16,  9,91, 80,82, 85,56, 12,70,
  15,16, 32, 6, 34,79, 84,28, 93,63, 53, 9, 54,62, 98,46,
  71, 9, 31,78, 54,93, 52,83, 94,16, 99,85, 55,64, 55, 2,
  21,96, 52,33, 60,87, 61,76, 67,95, 70,56, 84, 3, 34, 0,
  26,62, 78,31, 33,64, 52,81,  3,80, 54,94, 85,41, 76,51,
  79,37, 49,77, 72,71, 92,17, 51,17, 26,65, 57,42, 74,40,
  62,45, 34,57, 55,52, 42,11, 58,48, 15,88, 87,49, 78,12,
  99,88, 24,76, 63,53, 30,19, 60,32, 23,88, 30,41, 73,59,
   0,54, 10,67, 29,86, 63, 4, 84,41, 97,61, 54,99,  7,97,
  74,54, 81,64, 17,71, 41,37, 44, 9,  8,21, 26, 4, 10,56,
  71,37, 68,55, 29,92, 73,18, 59,62, 57,40, 18,33, 33,38,
  75,88, 54,59,  0,67, 96, 5, 96,22, 94,74, 33,31,  3,79,
  76, 6,  3,65, 89,68, 62,55, 91,77, 37,46, 82,71, 40,51,
  75,28, 83,85, 32,23, 16,61,  4,96, 86,55,  8,13, 35,70,
  99,94, 42,72, 39,98, 22,41,  5,81, 27, 5, 60,52, 40,24,
  20,72, 77,43, 33,80, 24,74, 83,55, 71,15,  7,70, 69,11,
  71,83,  0,91, 47,23, 19,38, 24,16, 78,82, 99,36, 71,90,
  17,14, 70,95, 94,18, 39,64, 79,36,  2,34, 72, 7, 90,61,
  26,20, 96,25,  6,58, 18,73, 37,53, 99,66,  8, 9, 96,49,
  52,93, 49,64, 97,18, 31, 8,  9,22, 13,38, 96,33, 48,55,
  78,80, 88,50, 37,49, 29,77, 54, 1, 52,17, 64, 1, 24,84,
  84,37, 23,94, 42,32,  8,59, 36,95, 78, 4, 19,99, 89,51,
  78,54, 55,32, 25,83, 46,48, 81, 1, 36, 9, 52, 6, 22,63,
  20,80, 48,90, 69,19, 76, 9, 12,51, 21,58, 13,61, 57,23,
  79,42, 63,46, 14,30, 81,48, 26,84, 24, 0, 77,22, 12,15,
  98,40,  4,35, 70,83, 79,31, 93,36, 52,86, 33,66, 71,53,
  28,67, 32,54, 75,89, 97,58, 16,73, 69,80, 23,30,  7,40,
  66,16, 69,32, 11,72, 13,78, 54, 4,  1,65,  2, 8, 45,72,
  47,57, 23,75, 92,62,  4,71,  6,15, 31,80,  2,57, 20, 6,
  71,41, 57, 6, 89,60, 53,96, 14,26, 97,95, 32,16, 93, 1,
  28,65, 81, 7,  8,55,  8, 4, 67,33, 12,90, 98,75, 43,74,
  47,98, 92,95, 80,91, 65,45, 78,61, 39, 1, 45,42, 36,50,
  11,67, 26,99, 14,53,  9, 4, 29,47, 22,12, 92,81, 26,10,
   4,17, 36,98, 65,69, 39,48,  8,22, 47,55, 47,54, 28,84,
  23,82, 22, 1, 26,28, 25,29, 56,16, 90,28, 90,35, 91,20,
  49,92, 24, 5, 36,34,  1,77, 95,32, 79,75, 99, 3, 24,66,
  27,28, 22,53, 31,46, 37,77, 18,14, 35,81, 29,21, 81,32,
   7,23, 78,29, 35,85, 29,84, 61, 9, 78,42, 30,86, 12, 9,
  73, 3, 63,77, 36,39,  6,32, 31,14, 26,94, 93,74, 35,79,
  55,84, 80,36,  8,53, 12,89, 82,59, 81,77, 27,96, 33,32,
  85, 9, 26,33, 33,92,  1,80, 25,24, 14,90, 60,60,  4,86,
  39, 3, 99,51, 38,66, 20,67, 80,80, 95, 8, 33,98, 46,83,
   4,66, 84,58, 55,36, 39,38, 41,11, 64,42, 78,59,  6,99,
  67,36, 63,62,  6,43,  8,74, 37,80, 74,66, 51,11, 11,48,
  91,18, 34,95, 39,31, 70,67, 28,66, 34,59,  3,95, 29,64,
  14,69, 37,19, 34,60, 20,17, 96,56, 74, 8, 36,57, 91,70,
  10,65, 82,86, 62,20, 88, 9, 84,10, 22,17, 18,10,  5,53,
  96,61, 72,41, 28, 2, 49,47, 49,31, 16,51, 13,12, 36,27,
  20,43, 41, 3, 38, 1, 14,28,  7,16, 95,81, 66,38, 86,80,
  42,59, 42,24, 45,84, 83,29, 77, 8,  9,27, 50,55,  2,92,
  23,79, 60,71, 54,28, 61,85, 55,65, 28,23, 42,20, 89,75,
   6,29, 11, 0, 48,18, 66,78, 55,96, 11,45, 15,57, 80,99,
  55,77, 37,18, 29,75, 69,20, 79,14, 83,98,  4,37, 51,59,
  59,66, 94, 2, 20, 1, 15,89, 65,88, 45,50, 93,77, 28,35,
  37,37, 31,61,  7,62, 33,79, 44,89, 72,61, 44,28, 18,89,
  50,27, 56,76, 75, 7, 68,96, 41,40, 79,46, 41,34, 84,73,
  49,71, 99,64, 80,25, 48,11, 32, 1, 47,90, 91, 9, 35, 1,
  56,12, 89,35, 98,12, 65,31, 42,89, 67,40, 70,25, 43,67,
   9,10, 93,97, 77,99, 60,56, 18,60, 89,84, 62,16, 95,54,
  45,26, 19,31, 30,43, 67,63, 57, 4,  5,43, 13,30, 14,33,
   6,13, 21,45, 19,32, 85, 0,  6,76, 16,82,  9,86, 44,50,
  62,53,  9, 7, 98,89, 85,75, 51,87,  4,12, 94,76, 27,83,
  80,70, 43,29, 36,36, 82,65, 80, 5, 84,47,  7,18, 35,16,
  95,28,  0,70, 62,32, 45,47,  5,61, 43,98, 26,54, 35,46,
  98,41, 69,12, 46,71, 29,45, 51,50,  9,45, 43,24, 56,50,
  56,40, 33,52,  0,66, 73,89, 89,61,  3,99, 75,98, 87,74,
  39,70, 36,58, 36,32, 76,61, 47,39, 15,19, 45,65, 38,11,
  41,91, 98,96, 22,85, 49,76, 95,34, 64,30, 62,61, 25,41,
  71,99, 33,15, 39,47, 56,29, 57,37, 70,18, 80,39,  7,29,
  19,27, 33, 9, 61, 3, 41,35, 91,35, 33,29, 12,95, 44,44,
  83,14, 62,89, 62,18, 77,50, 90,82, 24,39, 78,72, 40,96,
  38,49, 87,65, 47,13, 62,30,  0,82, 80,19, 76,97, 19,63,
  91,72, 80,22, 87,57, 57,55, 73, 9, 22,99, 45,71, 73,46,
  71,45,  4, 4, 65,59, 97,44, 38,93,  1,21, 79,79, 73,26,
  30,35, 33,18, 39,29,  2,30, 16,14, 62,93, 25,88, 91,62,
  92,12, 55,47, 68, 7,  0,28,  3,91, 55,97, 64,46, 98,54,
  36,73, 19,94, 28, 8, 19,91, 93,66, 33,50, 81,42, 11,22,
   6,84, 76,63, 60,58,  1,79,  4,51, 59,55, 67,18,  5,29,
  20,40, 66,75, 76,54, 41,24, 33,67, 93,25, 56,71, 70,59,
  66,51,  3,67, 96, 2, 89, 8, 79,52, 61,65,  8,28, 81,49,
  12,40, 99, 5, 78,91, 56,48, 64,24, 41,55, 46, 0, 10,91,
  81,88, 44,26, 87,35, 16,79, 64,34, 11,57, 56,34, 90, 1,
  91,83, 61,48, 67,11, 82,60,  6,80, 55,62,  9,69, 75,87,
  73,36, 59,15, 53,98, 60,24,  0,12, 71,62,  7,78, 27,48,
  59,93, 60,33, 64, 3, 41,64, 54,21, 39,97, 91, 2, 48,40,
  71,11, 20, 4, 72,29, 49,42, 82,64, 98, 4, 46,31, 42,51,
  63, 3, 31,10, 55,10,  6,56, 73,95, 78,40, 28,74, 84,51,
  37,71, 54,14, 67,52, 51,24, 94,42, 17,47, 63,82, 65,37,
  39,10, 61,18, 47,21,  1,38, 71,91, 62,46, 67,47, 70, 2,
  96,31, 72,55, 16,38,  1,58, 48, 8, 79, 0, 98,79, 83,39,
  24,15, 60,47, 42,67, 20,37, 95,36,  5,49, 37,24, 23,72,
  91,55, 77,89, 59,99, 61,17, 17,13, 48,13, 20,88, 44,59,
  55,20, 46, 8, 36,80, 25,14, 48,32, 32,56, 66,73, 24, 2,
  24,90, 63,87, 59,61, 73,86, 63,61, 87,39, 59,42, 66,68,
  40, 0, 85,29, 16,71,  2,22, 20,91, 72,16, 42,50, 33,35,
  66, 8, 35,90, 23,22, 36,91, 20,77, 65,21, 19,45, 62,86,
  24,57, 18,87, 38,37, 71,39, 22, 4,  3,34, 91, 5, 43,94,
  76,46, 31,37, 25,70, 50, 8, 35,65, 66,61, 47,25, 95,13,
  28,97, 18,39, 44,20, 58,80, 66,30, 80,50, 23,28, 40,41,
  15, 4, 38,45, 71, 6, 19, 4, 44,58, 30,28, 16, 8, 92,26,
  93,58, 54,33, 38,41, 78,87, 46,67, 24,58, 84,92, 42,65,
  62,44, 85,84, 64,90, 92,82, 72, 0,  3, 4, 39,36, 11,63,
   5,12,  2,24, 82,89, 73,61, 84,13, 11,87, 10,50, 34, 6,
  42,74, 58,12, 47,88, 29,87, 85,27, 34,19, 69, 0, 79,58,
  28,40, 58,85, 61,57, 50,66, 67,39, 36,17,  8,48, 73,85,
  70, 6, 62,42, 76,36, 48,86,  1,89, 29,59, 63,57, 57,97,
  91,87, 45,53, 26,46, 85,46, 55,69, 30,14, 43,68, 46,63,
  88,30, 28,61, 52,25, 11,30, 76,39, 64,73, 27,32, 96,82,
  93,94, 37,76, 11,49, 42, 8,  5,42, 98,43,  8,99, 33,24,
  91,19, 61,37, 24,27, 38,39, 52,74, 46,33, 86,31, 58, 5,
   8,20, 29,98, 72,31, 95,21, 98,93, 36,66, 68,77, 47,85,
   6,10, 35,23, 40, 3, 11,65, 42,73, 37,14, 22,46,  8,92,
  20,21, 73,44, 44,83, 84,42, 58,61, 13,36, 15,24, 10,84,
  75,58, 43,88, 23,38, 15,75, 27,91, 18,12, 81,85, 71,93,
  22, 9,  6,14, 54,77, 44,43, 17,32, 15,34, 74,89, 67,44,
  78,34, 75,32, 71,61, 11,29,  2,29, 60,69, 94,10, 62,15,
  25,99, 94,19,  3,42, 47,69, 49,85, 47,60, 57,63, 77,26,
  47,14,  3,57, 93,83, 97,35,  5,35, 76, 3, 41,74, 50,45,
  12, 8, 80,66, 44, 8, 27,49, 71,68,  7,15, 84,17, 32,63,
  76,91, 33,56, 74,51, 94,38, 61,73, 78,98, 74,71, 71,47,
  71,19,  2,67, 93,18, 97,81, 74,38, 86,33, 76, 1, 63,52,
  67,34, 69,90, 58,36, 14,92, 58,21, 71,36, 68,56, 87,93,
  93,62, 28,90, 65,42, 62,21, 63,80, 72,66, 43,46, 16,11,
  24,55, 45,44, 22,13, 74,58, 85,58, 25,37, 92, 8, 91,98,
  29,11,  6,41, 86,36, 94,75, 33,53, 17,91, 48,49, 65,75,
  57,68, 35,14, 68,12, 80, 7, 13,77, 81,67, 51,37, 51,94,
  40,17,  4,46, 82,41, 70,79, 60,18, 54,25, 67,66, 76,26,
  33,90, 23,29,  9,15, 42,83, 93,80, 93,45, 90,12, 88,21,
  25,21, 39,55, 76, 5, 70,80, 54,80, 72,17, 44,55, 14,49,
  73,47, 13,37, 66,53, 12,35, 71,66, 83,34, 28,79, 17,64,
   4,25, 10,37, 77,30, 52,31, 69,67, 43,90, 74, 0, 23, 5,
  95,66, 31,66, 78,11, 67,85, 70,43, 14,97,  1,98, 13,56,
   0,41,  1,60, 48,21, 24,28, 50,17, 38,35, 54,13, 64,58,
  42,56,  7,55, 54,17, 59,60,  5, 9, 43, 7, 55,34, 86,51,
  31,58, 17, 7,  0,98, 72,51, 11,77, 75,99, 33,30, 62,34,
  56,89, 92, 0, 94,13, 36,72, 64,33, 92,22, 29,35, 90,16,
   8,43, 27,58, 80,85, 84,72,  3,30, 11,18,  9,74, 42,34,
  50,34, 75,67, 23,53, 51,53, 13,86, 16,53, 95,15, 19,42,
  76,33, 79,66, 68,65,  5,57, 57,35, 64, 7, 83,87,  7,87,
  71,30, 26,31, 61,38, 27,94, 14,42, 89,66, 14,29, 16,65,
  97,74, 68,29, 50,99, 43,10, 79,34,  7,84, 96,27, 19,12,
  37,43, 89, 5, 55,57, 66,34, 94,33, 10,61, 16,50,  1,24,
  27,70, 50,89, 95,49, 93,51, 24,23, 35,53, 97,90, 56,45,
  75,75, 77,45, 76,31, 38,34, 32,42, 38,13, 49,13,  5,67,
  19,71, 97,59, 97, 8, 24,82, 98,87, 12, 4, 29,83, 78,66,
  70, 7, 85,30,  1,71, 32,53, 59,59, 65,86, 37,75, 66, 3,
  99,70, 84,49, 70,19, 27,80, 66,23, 36,44, 69,71, 28,32,
  77,71, 45,16, 57,47, 10,17, 90, 4, 45,86, 71,65, 50,86,
  35,60, 42, 1, 35,40, 67,99, 20, 0, 44,30, 22,50,  4,45,
  51,75, 97,29, 45,63, 37,85, 80,78, 16,42, 65,30, 61,64,
  94,57, 88,60, 86,70,  5,52, 42,61, 29,96, 32,82, 47,74,
   7,13, 76,94, 79,12, 18, 8, 19,80,  7,67, 17,25, 23,34,
  92,70, 73,93, 57,89, 84,54, 62,37, 28,38, 39, 0, 70,13,
  74,73, 86,30, 29,13, 87,50, 61,56, 24,43, 10,81, 53,86,
  49,69, 75,47, 79,60, 44,77, 57,86, 68,92, 86,58,  7,43,
  24,67,  0,37, 70,54,  5,78, 96,77, 10,43, 69,73, 61,97,
  52,36, 52,38, 70,58,  6,11, 17,89, 22, 7, 24,95, 48,88,
  18,79, 26,92, 53, 1, 28,82, 59,58, 81,80, 10,82, 39,74,
  37,12, 51, 5, 65, 8, 94,40, 89,40, 58,90,  6,66, 28,34,
   1,43, 52,11, 18, 6, 81, 6, 68,50, 83, 7, 10, 1, 78,45,
  14,55, 85,51, 85,17, 25,25, 29,19,  4,89, 20,24, 62,12,
  63,39, 21,72, 82,23, 16,32, 70,68, 93,99, 88,20, 75,15,
   7,83, 55,38, 29,66, 99,75, 99,96, 99, 7, 13,18,  9,34,
  29,38,  1,47, 82,94,  3,68, 98,26, 57,75, 95,22, 14,60,
  28,75, 32,31, 64,45, 79,53, 10,52, 35,57,  7,34, 28,71,
  71, 0,  3,15, 21,50, 52,44, 57,62, 54,55, 16, 0, 54,97,
  74,93, 54,71, 38,18, 76,30, 98,56, 74,45,  7, 1, 55,22,
  46,17, 67,55, 99,91, 42, 2, 13,26, 41,31,  0,35, 70, 0,
  52,89, 34,90, 48,98, 46,21, 13,21, 82,50, 44,71, 35,59,
  14,45, 16,75,  6, 4, 69, 8, 55,25, 70,66, 20,61, 75,40,
  43,80, 81, 3, 26,29, 79,99, 31,68, 79,29, 84,70,  6,25,
  97,66, 97, 1, 82,63, 30,77, 76,28, 61,28, 94,84, 24,29,
  15,39, 64,21, 83,62, 23,89, 81,46, 13,83, 95,37,  4, 2,
  38, 2, 14,57, 47,20, 88,85, 51,55, 77,62, 57,69, 31,47,
  16,64, 37,86,  7,86, 85,95, 60,94, 88,81, 67,37,  3,29,
  43,41, 36,86
};
const int p_100_8[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  946,
  /// Array of preferences
   2,81, 71,16, 56,13,  5,84, 69,64, 78,91, 53,83, 67,80,
   1,24,  7,96,  8,82, 41,74, 74,50,  4,75, 84,84, 95,90,
  75,37, 56,67, 77,30, 75,34, 10,82, 68,97, 44,51, 14,22,
  93,45, 22,63, 27, 4, 38,92, 37,61, 14,18, 53,13, 87,39,
  67,33, 89,98, 64,85, 24,50, 18,18, 43,20,  6,67, 95,83,
  94,41, 64,93, 54,73, 33,30, 20,97, 33,41,  9, 7, 77,83,
  21,40, 14,87, 25,14, 75,82, 26,19,  7,36, 45,16, 56,42,
  55,54, 50,86, 11,76, 31,56, 48,75, 64,59, 21,70, 22,76,
  31,82, 38,80, 38,14, 61,71,  2,16, 63,19,  9,35, 35,36,
   2,15, 34,95, 69,97, 80,61, 89,65, 71,95,  9,15, 91,53,
  88,38, 63,20, 20, 9, 74,66, 55,61, 49,55, 78,72, 44,14,
  93,22, 10,34, 84,69,  2,49, 92,92, 52,91, 26,40, 65,74,
   1,41, 72,90, 54,56, 60,32, 38,52, 65,78, 30,88, 15,56,
  84,26,  2, 6, 89, 1, 31,96, 89,26, 99,20, 38,36, 80,70,
  27,46, 89,43, 87,58, 28,31, 88,88, 27,33, 26,38, 33,58,
  13,65, 87,72, 28,55, 50,27, 85,76, 59,81, 61,48, 50,83,
   9, 1,  7,71, 26,18, 64,82, 88,11, 68, 2, 65,34, 39,92,
  77, 4, 28,10, 30,31,  7,73, 10,45,  6,87, 76,85, 29,85,
  12,63, 32,63, 23,13, 25,37, 17, 9, 24,14,  6, 5, 37,23,
  25,85, 84,33, 21,59, 97,34, 36, 9, 13,42, 50, 8,  8,25,
   1,61,  1,85, 43,60, 65,81, 14,33, 16,29, 47,39, 47,28,
  19, 5, 27,80, 98,55, 24, 2,  9,70, 38,89, 40,29, 51,40,
  29,47, 91,36, 66,71, 82,86, 84,77, 76,16, 76,44, 16,13,
  36,24, 35,57, 59,24, 36,14, 13,41, 92,54, 14,57, 82,10,
  19,37, 30,85, 80, 8, 63,61, 27,39, 55, 9, 98,14, 97,91,
  20,95, 11,40, 26,35, 16,36, 98,95, 85,35, 88,99, 53,82,
  51, 0, 31,53,  8,50, 66,69, 57,53,  8,65, 50,34, 87,47,
  56,71, 31,87, 70,81, 47, 1,  2,91, 66,32, 84,86, 72,80,
  56, 4, 41,50,  5,27, 26,73, 85,88, 31, 0, 23,36, 86,12,
  43,27,  3,70, 77,50, 12,41, 32,72, 25,34, 24,19, 35,90,
  86,47, 27,36, 77, 6, 75, 2, 68,53, 70, 4, 16,89, 80, 4,
  56,62, 35,98, 23,40, 41,19, 35, 7,  3,64, 83, 1,  1,23,
  72,98, 47,40, 73,48, 30, 6, 16,53, 70,50, 89,92, 14,15,
  11, 7, 37,39, 83,63,  5,33, 60,20, 65,62, 15,28, 49, 6,
  53,50, 42,77, 54,18, 91,79, 41,43, 21,94, 69,33, 10,68,
  11,36, 11,64, 91,56, 81, 5, 82,50, 16,55, 61,18, 85, 7,
  53,40, 70,19, 41,73, 96,98, 89,68, 10,16, 67,83, 55,89,
  26,42, 11,17, 70,49, 72,70, 98,68, 71,77, 23,65, 64,73,
  16,17,  1,28,  2,57, 69,69, 59,59, 56,22, 13, 9, 34,31,
  30,57, 19,45,  9, 4, 86,92, 40, 9,  4,29, 19,88, 32, 4,
  20,76, 57,79, 35,52, 51,59, 94,47, 67,99, 48,37, 82,70,
  73,98, 52,64, 98, 9, 91,52, 80,92, 19,12,  2, 8, 64,83,
  48,96, 14,48, 98,45, 44,60, 92,16, 96,51, 32,37, 11,34,
  57,92, 76,66,  0,16, 85,37, 41,96, 68,18, 29,17, 95,51,
  51,10, 12, 6, 84,56, 93,57,  0,22, 16, 2, 41,35, 80,34,
  86,72, 36,37, 24,13, 82,64, 64,52,  5,40, 40,16, 35,92,
  31,60, 77,56,  0,48, 75, 8, 22, 9, 60,52, 47,45, 64,62,
  23,92, 87, 5, 44,56, 36,27, 79, 3, 84,59, 98,20, 64,28,
  52, 7, 19,93, 49, 5,  7,92, 69, 7, 82,35, 62,28,  3, 8,
  99,99, 83,28, 25,47, 15,63, 85,26, 83,22, 64,99, 51,79,
   7, 4,  4,16, 57,90, 41,44, 99,23, 94, 1, 21, 1, 52,38,
  86,42, 95,78, 79,66, 50,60, 65,13, 71, 3,  7,15,  3,33,
  63,94, 18,93, 52,73, 83,64, 63,50, 49,56, 83,78, 20,65,
  73,15, 22,48, 91,37, 92,12, 18, 5, 70,58, 93,74, 79,98,
  46,39, 33,81, 11,71, 75,72, 76,93, 38,83, 25,54, 20, 7,
  50, 4, 39,57, 55,68, 71,66, 94,31, 27,27,  9,96, 81, 6,
   4,60, 39, 3, 24,81, 59,55, 78,84, 83,54, 10,44, 35,43,
  73,51, 18,40, 11,98, 26,36, 50,63, 84,35, 55,82, 20,47,
  54,29, 26,13, 66,35, 95,99, 37,29, 47,94, 54,97, 76, 6,
  79,41, 20,71, 57,13,  0, 0, 72,46, 54,25,  6,91, 84,61,
  35,12, 78, 5, 83,62,  4,37, 58,82, 14,75, 64,42, 58,67,
  92,73, 25,35, 83,30, 21,26, 77,75, 78,80, 71,17, 75,93,
   3,56, 28,32, 22,73, 51,80,  5,65, 23,12, 16,93, 63,13,
  68,72, 67, 8, 93,24, 70, 2,  2,34, 90,37, 18,19, 72, 2,
  50,93, 19,44, 63,29, 38,22, 73,19, 18,58, 16,18, 26,31,
  76,57, 99,77, 38,24, 56,68, 18,27, 86,22, 88,26, 90,77,
  82,94, 44,37, 12,20, 19,16, 41,57,  4,73,  1,42, 98,58,
  77,37, 70,89, 62,18,  2,43, 71,56,  7,29, 47,19, 70,16,
  97,60, 50,33, 72, 1, 59,68, 59,90, 26,87,  5,16, 92, 1,
  17,56, 19,22, 75,38, 42,31, 73,50, 35, 8, 64,49,  9,45,
  18,67, 15, 6, 85,75, 46,28, 72,42, 95,39, 36, 3, 24,35,
  62,32, 53,57,  1,94, 93, 5, 26,44, 70,56, 45,69, 24,94,
   1,63, 20, 5,  2, 3, 84,20, 64,38, 32,45, 78,63, 85,53,
   5,36, 27,32, 37,46,  7,41, 17,89, 83,88, 76,82, 74,46,
  62,33, 46,21, 43,31, 37,80, 45,13, 94,59, 73,63,  0,86,
  37,97, 86,85, 34,55, 88,35, 61,74, 81,87, 45,76, 64,43,
  77,22, 49,51,  2,61, 92,32,  4,92, 42, 5,  4,51, 43,46,
   6,49, 10,10, 79,44, 92,11, 48,70, 53,33,  6,11, 10,23,
  27,29, 47,38, 64,21, 79,86, 41,47, 86,87, 52,36, 39,89,
  14,94, 68,27, 78,10, 64,33, 38,81, 30,34, 84, 3, 44,58,
  85,16, 84,97, 92, 7, 34,85, 33,52, 78,29, 37, 4, 37,27,
  74,35, 16,78, 79,90, 18,71, 55,29, 87,15, 97,51, 91,75,
  93,40, 19,86, 98,17, 49,75, 51,83, 31,78, 98,22, 49,30,
  42,67, 69,49, 89,94,  6,22, 78,27, 98,60, 61,44, 19,31,
  81, 9, 97,92, 98,74, 49,77, 53,19, 57,54, 83,72, 74,80,
   0,26, 66,42, 43,97, 24,20, 16,10, 49,67, 53,67, 73,60,
   1,12, 78,24,  1,30, 12,87, 63,72, 57,48, 72,76, 83,50,
  35,94, 52,99, 44,22, 46,97, 64,77, 12,91, 61,82, 20,52,
  19,33,  3,15, 97,39, 90,25, 91,12,  8, 7, 62,47, 87,50,
  95,70, 38,48, 23,43, 79,37,  8,53, 85,45, 26,34, 38, 0,
   7,34,  0,14, 64,61, 21,57,  7,80,  0,32, 87,91, 61,66,
   4, 7, 42,64,  8,18, 24,38, 64,88, 91,48,  7,13, 98, 5,
  51, 8, 93,52, 71,25, 51,13, 41,64, 95,23, 96,92, 55,18,
   7,72, 71, 6, 34,54, 97,17, 98,44, 95,28, 78,68,  7,18,
  70,96, 89,18, 83,40, 51, 4, 48,98, 48,42,  4,44,  7,74,
   3,53, 38,55, 96,22, 85,92, 64,92, 90,66, 70,43, 64,22,
  49,53, 12,75, 50,36,  1,26, 71,97, 19,97, 17,24, 65,52,
  36, 5, 33,53, 45,57, 52,85, 70,70, 14,53, 34,99,  2,83,
  85,12, 26,24, 36, 4, 65,58, 23,55, 28,22, 40,79, 44,16,
  46,52, 40,59, 72,43, 23,69, 33,88, 58,92, 14,55, 97, 6,
  84,65, 65,65, 54,31, 62,58, 84,57, 80,56, 96,42, 82,55,
  46,98,  6,60, 47,10, 57,29, 51,69, 44, 8, 69,76, 75,94,
   7,38, 45,73, 38,13,  3, 6, 43,12,  1,79, 60,98, 63,77,
  68,46, 47,33, 14,66, 98,10, 40,99, 68,82,  5,71, 25,57,
  65,14, 59,38, 42,49, 23,98, 72,89, 93,42, 98, 2, 70,34,
  42,92, 38,43, 95,56, 28,54, 85,99, 36,30, 18,41, 78,61,
  85,84, 96,52, 53,68, 65,10, 31,17, 88,22, 24,73, 92,48,
   5,77, 83,98, 13,27, 37,49, 25,92, 86,28,  6,35, 69,73,
  74,85, 70,20, 73,76, 18,72, 30,45, 10,84,  6,17, 30, 0,
  18,24, 72,56, 23,56, 57,98, 46, 0, 55,20, 65,73, 31,25,
  32,47, 33,86, 91,20, 90,71, 65,22, 99,25, 82,21, 90, 4,
  92,14, 37,83, 54,71, 66, 9, 62, 6, 97, 0, 40,20, 81,69,
  72,22, 88,33,  2,39, 50,70, 93,82, 51,14, 86,97, 32,74,
  93,29, 34,11, 71,51, 30,97, 97,14, 38,75, 95,53, 60,62,
  53,32, 85,81, 94,37, 49,58, 50,65, 99,93, 52,56, 37,70,
  15,42, 23,57, 57,71,  2,78, 13, 3, 71,59, 22, 3, 78,53,
  70, 0, 48,36, 52, 4,  9,98, 19,30, 59,75, 96,72, 33,31,
  60,33, 89,67,  8,68, 43,15,  1,16, 73,62, 82,47, 23,93,
  49,94, 99, 2
};
const int p_100_9[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  394,
  /// Array of preferences
  32,39, 61,75, 44,22, 94,45,  0,58, 19, 1, 85,58, 35,84,
  52, 7, 31,61, 30,57, 45, 8, 51,50, 66,52, 74,53, 64,74,
   8,81, 78,37,  4,47, 89,90, 70,64, 20,43, 55,58, 19,68,
   6,52, 55,99, 74,80, 55, 4, 97,50, 78,22, 28, 7, 82,37,
  10,40, 96,85, 75,80, 85,61, 43,94, 29,92, 12,24, 95,43,
   3,19, 37,26, 23,39, 83,45, 68,26, 88,42, 22,51,  2,78,
  98,12, 11,68, 75,36, 28,62, 97,27, 81,24, 88,85, 99,60,
  19,98, 93,77, 48,25, 75,77, 32,15, 83,49, 99,13, 57,46,
  61,77, 63,23, 29,86, 60,15, 47,52, 53,51, 66,38, 91,17,
  88,46, 11,17, 61,49, 67, 1, 45,69, 37,65, 86,62, 65,31,
  10,42, 35,82, 93,31, 85,32, 78,54, 73,54, 82,60,  8,62,
  99,41, 63, 6, 41,56, 24,17, 58,28, 31,14, 35,64, 25,61,
  10,86, 64,23, 98,93, 27,12, 61,40, 14,81, 24,39, 58,21,
  34,80, 30,34, 23,44, 67,50, 42,30, 77,79, 16,79,  6, 3,
  67,25, 72,40, 35, 7, 59,25, 91, 5, 93,22, 27,45, 91,28,
   4,38, 56,45, 91,50, 32,13, 32,60, 40,58, 72,86, 62,59,
  13,65, 96,77, 69,16, 35,62, 79, 9, 73,49, 62,61, 36,11,
   0,92, 59,30, 86,76, 67,72, 68,32, 47,72, 96,21,  4,52,
  52,30, 66,62, 45,61, 84,58, 94,86, 19,50, 76,79, 22,99,
  18, 4, 41,88, 48,52, 81,21, 56,47, 28,61, 82,35, 10,69,
  63,65, 93,47, 99,55, 77,39, 50,50, 97,63,  0,77, 79,39,
  96,49, 99,70, 65, 2, 48,93, 22,61, 48,80,  2,22, 52,18,
  78,51, 30,62, 20,83, 46,21,  8,76, 11, 2, 27,14, 81,65,
  97,94, 40,64, 87,42, 73,76, 54,52, 22,73, 73,90, 81,60,
  97,76, 76,69, 50,86,  4,19, 30,30, 89,78, 46,95, 81,80,
  11, 1, 33,93, 62,88, 15,72,  7,62, 31,28, 68,52, 82,18,
  87,91, 33,24, 81,52, 29,45, 53,20, 54,55,  1,44, 22,76,
  33,34, 19,15, 24,71, 30,33, 41,72, 44,41, 18,39, 29,64,
  19,74, 94,33, 76,47, 50,54, 10,56, 28,21, 74,78, 17,45,
  27,83, 27,55,  6,76, 65,56, 28,17, 48,86, 66,28, 54,22,
  16,11, 53,24, 45,80, 71,62, 93,56, 82,26, 67,46, 64,51,
  60, 5, 23,57, 30,39, 18,20, 49,54, 85,51, 61,98, 99,79,
  17,19,  6,74, 92, 1, 74,24, 79,15, 43, 6, 99,78, 72,79,
  30, 5, 17,80, 80,30, 59,68, 55,92, 18,54, 33,45,  3,93,
  87,26, 95,73, 33,49, 44,15, 49,35, 59,91, 97,57, 60, 7,
   9,75, 86,86, 31,64, 30,90, 77,51, 85, 6, 43,86, 94,58,
  14,78, 83,42, 39,87, 70,22, 79,21, 22,55, 99,15, 36,42,
  95,31, 91,13, 43, 5, 82,77,  7,88, 47,45, 23,90, 59,45,
  21,96, 88,86, 82,66, 64,97, 46,12, 29,97, 29,48, 14,32,
  61,26, 90,67, 63,44, 46,64,  9,66,  8,95, 39,42, 23,79,
  45,41, 58,81, 64,58,  8,47, 27,20, 80,78, 24,70, 64,26,
  16,51, 27, 1, 18,83, 23, 2, 60,87, 89,44, 40, 0, 18,95,
  68,59, 22,64, 21,47, 34, 7, 77,38, 44,73, 13,79, 54, 6,
  83,86, 61,86, 84,71, 35,61,  9,27, 55,31, 25,85, 98,94,
  56,62, 51,91, 44,95, 12,66, 73,19, 97,98, 89,36,  8,40,
  50,31, 42,36,  6,33, 17, 4, 66,53, 21,86, 60,41, 94,65,
  57,69, 33,48, 53,32, 55,41, 84,19, 12,37,  4, 9, 87,76,
  88,78, 99,69, 61,52, 94,66, 75,48, 47, 9, 47,75, 60,63,
  96,83, 15,74, 54,35, 13,89, 45,23, 81,33, 15,35, 62,68,
  36,84, 30,92
};
const int p_100_10[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  2275,
  /// Array of preferences
  57,75, 26,75, 75,22, 34,20, 24,37, 28,26, 98,66, 31,30,
  23,15, 52,17, 51,34, 54,85, 59,43, 38, 0, 21, 3,  2,94,
  12,82, 89,22, 65,85, 51,90, 55,96, 35,23, 39,96, 98,64,
  50,14, 25,48, 53,15, 80,54, 91,18, 29,44, 20,75, 34,59,
  84,78, 27,84, 87,41, 19,33, 47,97, 22,70, 25,66, 58,67,
  41,86, 47,89, 55,73, 67,13, 12,36, 83,27, 55,91, 25,20,
  98,22, 54,36, 96,51, 71,57, 60,15, 10,73,  5,34, 24, 0,
  23,24, 96,90, 73,37, 93,72, 22,83, 95,62,  6, 5, 91,87,
  22,58, 47, 5, 37,23, 34,29, 91,55, 99,25, 58,91, 96,61,
  71,90, 21,50, 34,99, 96,56,  4,91, 50,17, 51,26, 38,35,
  51,54, 31,97, 95,30, 22,33, 79,94, 12,70,  8,59, 89,27,
  34,12, 49,11, 13,92, 71,80, 52,73, 28,85, 93,89, 96, 2,
  45,46,  4,10, 36,20, 34,26, 44,17, 81,23, 60,33, 63,72,
  33,31,  1,61, 72,14, 81,35, 87,39, 33,32, 40,14, 95,54,
  74,74, 74,50, 45,95, 58,62, 57,91, 38,20, 85,30, 68, 0,
  50,74, 90,32,  3,21, 62,58, 73,45, 92, 9, 32,35, 64,19,
  26,99, 58,75, 44,99, 55,95, 91,56, 31,60, 83,54, 68,44,
  69, 1,  7,98, 26,95, 37,87, 34,61, 58, 7, 87,81,  9,10,
  70,77,  7,45, 14,42,  7,23,  7,50, 22,89, 58,34, 77,26,
  53,64, 91,17, 60,66, 11,80,  5,17, 60,90, 11,11, 16,42,
  51,32, 25,87, 64,93, 55,62, 41,83, 40,92, 27,15, 28,14,
  34,37,  8,19, 31, 9, 84,75, 27,87, 44,56, 35, 7, 67,66,
  87,12, 80, 2, 28,16, 93,87, 38,95, 38,69, 35,20, 37,18,
  35,77, 60,30, 36,22,  0,56, 60,14, 60,45, 39,98, 16,25,
  88, 4, 59, 3,  4,13, 41,36, 69,83, 67,40, 49,43, 42,75,
  55,63,  0,36, 61,56, 85,27, 98, 1, 37,82, 74,12, 71,97,
  96,48, 64,54, 37,99, 10,80, 65,97, 52,33, 57, 3, 15,39,
  11,94, 87,17, 30,67,  1,14, 58,27, 16,74, 63,79, 26,73,
  25,44, 25,49, 55,48, 69,33, 47,87, 58,71, 34,55, 46,86,
  92,26, 29, 1, 74,31, 25,76, 11,26,  9, 1, 10,52, 10, 2,
  70,79, 74,34,  3,14, 45,78, 25,91, 48,18, 94,14, 65,90,
  71,82, 37,47, 80,74, 46,36, 38,80, 10,64, 49,93, 19, 3,
  18,89, 62,37, 49,96, 37,32, 89,16, 96,34, 57,10, 94,52,
  89,31, 51,57,  1,35, 68,25, 39,43, 29,27, 20, 4, 55,76,
  36,97, 61,33, 34,94, 59,13, 92,52, 45,13, 54,59, 58,31,
  50,18, 30,73, 18,50, 97,60, 83,41, 91,47, 29,63, 41,69,
  13,61, 35,28, 89,92, 33,38, 72,86, 38,11, 54, 7, 73, 0,
  25,23,  8,14, 27,60, 83, 6, 78,17, 32,84,  8,12, 82,86,
  77,62, 43,95,  5,27, 84,40, 51,22, 19,52, 42,97, 79,38,
  75,65, 19,45, 28,50, 79,39, 86,46, 42,31, 47,96, 57,67,
   2,41, 37, 4, 96,97, 72,54, 98,88, 48,73, 58,48, 37,85,
  76,72, 49,75, 72,42, 12,46, 17,75, 29,94, 78,93, 64, 1,
   5,44, 43,29, 31,84, 12,73, 98,96, 63,61, 11,72, 46, 0,
  61,12, 33, 2, 92,53, 23,10, 11,15, 63,71, 61,43, 82,80,
  11,49, 51,60, 70,74, 78,75, 12,53, 74,68, 99,27, 88,64,
  12, 7, 94,22, 26,27, 59,84, 73,21,  8,20, 22,35, 79,89,
  11,61, 95,44, 14,31, 17,78, 19,73, 58,69,  1,71, 75,83,
   9,44, 42,43, 89,68, 40,25, 88,22, 48,82, 85,51, 40,58,
  81, 4, 30,82, 94,30, 81, 2, 23,78, 57, 4, 25,21, 73,92,
   0,23, 58,39, 52,66, 95,98, 13,15, 86,23, 14,61, 48,12,
  90,53, 64,36, 93,27, 23,93, 55,11,  4,46, 30,66, 49,91,
  62,33, 88,63, 19,23, 18,83, 91, 2, 49,12, 31,42, 22,77,
  24,60, 76,85, 76,89, 43,88, 70,38, 80,93, 68,36, 75, 2,
  84,89, 19,74, 86,15, 47,44, 23,65, 80,98, 87,57, 23,52,
  41,26, 75,16, 30,20, 60,37, 70,75, 86,35, 75,44, 93,51,
  97,41, 63,47, 37, 3, 47,52,  5, 2, 44, 7, 92,55, 70,55,
  10,81, 73,61,  4,85, 98,45, 35,64, 29,66, 26,80, 56,44,
  22,80, 49,21, 62,55, 81,20, 20,12, 58,87, 31,56, 28,30,
  92,83, 42,66, 11,55, 57,43, 56,53, 83,55, 46,40, 34,56,
  53,22, 85,10, 58, 8, 19,27, 72,70, 72,91, 79,83, 85,73,
  63,36, 49,63, 14,92, 90,21, 15,98, 91,78,  5,45, 16,61,
   5, 5, 76,24, 50,84, 65,55, 65,52,  2,11,  8,58, 17,22,
  10,29, 94,35, 88,37, 93,30, 50,33, 38, 7,  7,21, 32, 0,
  18,70, 84,90, 85,13, 63,11, 86,48, 82,34, 70,95, 83,93,
   0,12, 58,83, 45,41, 63,43, 29,86, 49,42, 37,95, 64,51,
  16,36, 87,85, 65,44, 66,55, 78,29, 26,87,  3,80, 21,85,
  33,83, 48,21, 49,65, 31,17, 85, 1, 72, 5, 28,11, 83,70,
  15,82, 96,18, 44,89, 24,79, 27,23, 59,78,  3,32, 12,37,
  38,92,  8,43, 72, 0, 43,60, 15,78, 28,74, 36, 8, 36,87,
  79,26, 70,15, 29,16, 92,82, 28,78, 64,43, 29, 2,  4,11,
  15,12, 87,66, 39, 9, 24,71, 38, 8, 60,54, 46,43, 69,13,
  10,93, 34,66, 28,82, 26,50, 63, 5,  3,83, 63,37, 87,25,
  50,22, 42,69, 42,29, 58,74, 62,23, 10, 5, 93,54, 62,80,
  56,36,  3,67,  5,84, 50,73, 92,76, 70, 7, 61,53, 53,60,
  85,88, 19,37, 48,72, 76,39, 78,15, 50, 5, 36,52, 22,49,
  40,99, 42,74, 36,30, 24,89, 20,21, 68,98, 42,17, 11,60,
  96,63, 68,88, 43, 0, 56,30, 65,96,  6,66, 12,31, 54,35,
  94,11, 44,10, 20,45, 76, 3, 66,46, 89, 9, 73,65,  1,12,
  27,67, 69,69, 28,10, 48,69, 82,79, 72,61, 36, 9, 64,70,
  32,60, 80,28, 26,49, 25, 4, 60, 3, 20,49, 74,76,  8,33,
  42,63, 71,88, 26,81, 47,95, 40, 8, 26,22, 85,89, 38, 1,
  20,39, 52,28, 68,17, 51,41, 86,91, 95,91, 15,67, 26,51,
  81,92, 12,47, 91,34, 88,36, 89,40, 76,31, 55,36, 64,83,
  18,87, 19,88, 73,29, 82,19, 97,29, 39,63, 73,36, 18,40,
  51,39, 99,41, 34,43, 32,19, 55,81, 19,83,  1, 0, 35,31,
  57,42, 88,80, 38,79, 61,47, 55,12, 67,95, 39,49, 35,86,
  10,39, 73, 4, 33,14, 73,32, 84,86, 68,42, 75,30,  2,84,
  30,55,  1,59, 86,66, 76,67, 17,16, 70, 9, 91,35, 10,56,
  28,40, 95,74, 41,97, 96,42, 80,11, 94,56, 43, 3, 93,23,
  97,28, 18,65, 64, 6, 86,61,  6,71, 84, 0, 36,50, 68,99,
  35,70, 33,97, 31,24, 27,74, 21,64,  8,62, 22,32, 94,55,
  94,71,  1,47,  0, 4, 15,74, 80,73,  7,41, 23,90, 70,57,
  11,21, 27, 5,  9,42, 36,56, 66,13, 74,47, 96,22, 26, 8,
  12,94, 85,33, 78,45,  8,26, 46,75,  2,72, 22,71, 46,73,
  23,46, 33,35, 75,66, 50, 4, 59,18, 71,72, 45, 4, 62,29,
  48,26, 14,36, 71,98, 15,80, 74,29, 43,62, 42,45, 92,14,
  11,93, 22,68,  4,50, 86, 8,  0, 7, 20,19,  8,31, 97,22,
  37,94, 35,43,  3,76, 77,52,  0,75, 95,47, 90,27, 43,34,
  31, 5,  7,85, 73,12,  8,55, 69,40, 78,50, 33,76, 35,55,
  55,54, 72,48, 40,32, 67,82, 54,49, 40,71, 46,13, 33,79,
  44,46, 88,93, 27,17, 61,72, 41,92, 65,70, 87,94, 16,43,
  25,39, 51,89, 10,14, 40,88, 84,55, 15,93, 75,61, 62,77,
   9,99, 71, 5, 85,69, 52,54, 40,63, 56,70, 81,54, 69,81,
  50,66, 73,72, 34,24, 96,68, 48,44, 23, 4, 42,34, 57,19,
  59,11, 54,83, 18, 3, 12,81, 32,45, 61,29, 96,81, 91,52,
  20,34, 31,39, 96,54, 20,50, 65,80, 10,58, 32,59, 29, 9,
   4,47, 93,29, 93,24, 79,67, 22,90, 31,98, 80,12, 77,69,
  70,34, 34,91,  7,80, 69,78, 59,33, 85,58, 85,43, 41,27,
  36,47, 95,40, 19,72, 10,78, 64,28, 93,19, 95,18, 78,76,
  46,61,  8,84, 48,67, 55,53, 62,47, 36,69, 75,58, 36,58,
  45, 6, 62,93, 52,36, 81,62, 20,71, 58,70, 43,44, 47,36,
  80,85, 20,25,  5,91, 61,96, 56,23, 54,79, 33,69,  5, 0,
  85,79,  7,92, 51,67, 24, 1, 95, 7, 16,62, 64,42, 10,70,
  61,70, 30,70, 37,75, 19,12, 98,53, 44,16, 90,29, 19,57,
  91,95, 46,72,  8,94, 57,44, 86,10, 90,11, 81,37, 22,39,
  18,69,  2,77, 44,92, 53,55, 98,10, 81,30,  6,62, 16,21,
  70,89,  7,70, 87, 9,  3,86, 51,33, 82,92, 54, 4, 17,63,
  19,54, 58,35, 58,30,  6,57, 68,64, 20,89,  7, 1, 92,59,
  94,81, 50,16, 50,67, 48,14, 91,13, 55,89,  0,21, 48,95,
  86,73,  1,29, 49,46,  2,13, 88,44, 88,67, 24,65, 25,54,
  56,20, 15,70, 46,23, 22,28,  5,87, 36,65, 65,35, 27,12,
  34,57, 26,38, 68,20, 26,92, 89,85, 87,47, 28,90, 37,54,
  35,91, 94,65, 46,68,  9,43, 75,59, 76, 4, 66,22,  2,57,
  92,95, 57,46,  6,15, 67,91, 97,68, 72,18, 22,61, 32,32,
  59,80, 31,12, 48,81, 32,26, 36,45, 15,36, 94,64, 70,90,
   9,33,  4,38, 99, 8,  4,28, 41,29, 62,25,  4,96, 43,51,
  89,71, 95,61, 30,64, 48,13, 69,63, 99, 7, 82,44, 76,41,
  61,75, 96,69, 79,33, 58,32, 34, 6, 84,68, 83,29, 23,81,
   3,43, 84,30, 55,90, 65,50, 96, 6, 39,86, 58,64, 40,47,
  32, 8, 24,13, 47,49, 39,42, 97,38, 63,62, 87,44, 90,16,
  92,94, 26,34, 37,22, 59,44, 47,46, 14,12, 84,11, 64,64,
  73,46, 25, 0, 75,18, 43,70, 24,16, 80,77, 28,37, 96,31,
  22,43, 81,90,  3, 0, 86,63, 41,21, 88,13, 92,12, 50, 9,
  46,90, 38,39, 25,59, 80,47, 40,68, 72,20, 79,81, 94, 1,
   0,77, 69,57, 69, 0, 38,60, 94,17, 15,32, 72,97, 51,27,
  74,62, 64,16, 39,75, 29,47, 79,48, 18,34, 73,43, 65,68,
  78,61, 98,23, 79,27, 19,98,  1,25, 41,67,  4,97, 28,72,
  41,31, 51,23, 44,94, 88,29, 30,68, 45,57, 79,80, 82,57,
  93,36, 76,60, 98,65, 90,36, 80, 3, 99,32, 63,60, 76,47,
  31,61, 38,96,  0,33, 17, 4, 97, 9, 41,90, 99,59, 75,94,
  46,38, 97,63, 51,92, 74,69, 58,51, 61,77, 81,65, 73,15,
  18,72, 99,87, 34,23, 74,96, 52,57, 57,69, 60,46,  8,63,
  79,96, 64,91, 62,16, 86,70, 26,29, 81,55, 31,45, 45,21,
  93,56, 81,93, 35,22, 24,81, 80,80, 46, 5, 46,53, 10,72,
  24,57, 70,65, 65,64, 23,41, 99,10,  1,32, 73,95, 20,68,
  20,33, 33,48, 16,49, 24,94, 89,91, 50,87,  8,38, 24, 6,
  92,71, 19,17, 57, 9, 99,79, 29,35, 68,91, 77,91,  9,62,
  75,11, 50, 0, 95,31, 69,29, 35,32, 74,82, 69, 3, 63,50,
  11,12, 39,26, 26,25,  6,97, 95,55,  8,54, 34,95, 52,46,
  72,13, 79,66, 18,64, 58,18, 27, 2, 93,75, 44,44, 64, 5,
  12, 6, 95,87, 25,29, 59,62, 45,82, 33,75, 92,87, 56,28,
  66,88, 83,57, 19,75, 30,71, 57, 7, 53,34, 72,88, 50,86,
  35,36, 48,90, 94,74, 68,57,  9,65, 64,76,  6, 3, 48,70,
  52,53, 20,77, 10,74,  1,87, 29,82, 51,30, 59,92, 91,59,
  10, 4, 87,43,  0,76, 64,65, 57,74,  1,67, 99,60, 99,21,
  63,32, 22,81, 26,47, 27,22, 12,79, 67,52, 59,23, 92, 2,
  61,95, 43,63, 96,45, 38,62, 63, 9, 43,78, 96,65,  0,66,
  17,82, 14,93, 11,29, 90,18, 23,20, 29, 6, 57,60, 26,70,
  78,25,  3,63, 88,55, 73,33, 88,15, 74,59, 21,15, 46,71,
  43,23, 25,68, 91,64, 45,25, 74,45, 24,87, 71,20,  7,36,
   4,64, 92, 7, 13,29, 49, 5, 93,57, 37, 8, 48,50, 24,74,
  98, 9, 86,62, 87,52, 88,68, 39,17, 22,12, 96,25, 80, 8,
  66, 7, 85,68, 30,31, 97,80, 94,49, 21,47, 45,18, 71,32,
  86,67, 19,32, 44,33,  2,45, 60,82, 17,98, 25,73, 26,64,
  23,48, 25,58, 83,17, 49,67, 90,33, 46,15, 63,17,  1,89,
  12,16, 64,38, 56,98, 31,63, 79,90, 14,57, 44,12, 13, 5,
  84,97, 21,58, 23,68, 50,23, 31,29, 19,78, 51, 7,  8, 9,
  38,24,  1,83, 12,52, 69,22, 43,36, 30,72, 94, 9, 64,45,
  34,35, 72,92, 67,92, 61,10, 56,74, 95,46, 65,24, 42,92,
  33,53, 62,82, 56,82,  5,36, 75,39, 95,48, 88,94, 42,72,
  33,42, 14,32,  8,53, 27,42, 41,61, 72,39, 44,29, 93,38,
  40,95, 34, 7, 27,95,  3,93, 60,18, 23,91, 67,68, 11,96,
  52,55,  4,49, 72,90, 30,89, 52,32, 28,95, 82,26, 65,59,
  23,37, 75,72, 88,84, 77,17, 12,30, 83,86, 53,81, 46,32,
  46,49, 90,60, 46, 7, 44, 2, 38,40, 17,87, 78,19, 30,13,
  41,34, 25, 8, 28, 2, 55,37, 62,19, 34, 2,  4,89, 41, 0,
  71,35, 46,65, 81,40, 15,61, 54,67, 85,38, 96,55, 32,41,
   0,94, 90,99, 72,51, 25,53, 99,17, 23,85, 15,94, 85,95,
  61,55, 52, 6, 94, 2, 17,79, 41,20, 88,28, 28,24, 38,57,
  89,86, 99,68, 83, 3, 74,44, 66,21, 15,92,  7,93, 99,56,
  11,10, 65,48, 96,10, 80,33, 23,95, 31,44, 93,69, 75,28,
   7,29,  8,65, 65,94, 60, 6, 21,82, 79,45, 93,74, 77,81,
  28,64, 48,17,  5,66, 59,82, 83,50, 27,98, 45,83, 29,89,
  36,79, 26,19, 54,56, 55,64, 71,41, 96,27,  1,68, 59,40,
  95,97, 10,87, 49,39, 66,90,  8,41, 39,33, 60,19,  1,64,
  86,12, 53,93, 67,34, 36,16, 83,45, 88,98, 77,76, 40,65,
  65,53, 12,83,  0,44, 32,68, 55,38, 69,92, 90,19, 56,66,
  18,79, 80,50, 78,33, 27, 3, 85,12,  5,14, 76,10, 99,35,
  33,11, 64,74, 80,65, 32,96, 36,81, 75,76, 86,76, 17, 2,
  37,62, 71,76,  2,86, 65, 0, 54,72, 75,48, 28,84, 58,98,
  14,82, 89,20, 34,42, 42,25,  3,24, 33,45, 28, 1, 87,54,
  67,18, 66,95, 21,75, 86,60, 12, 5, 24,69,  0,78, 22,37,
  32,91,  4,66, 15,52, 63,77, 91,30, 10,41, 55,24, 48,58,
  91, 8,  5,98, 61, 0, 58,65, 55,88, 20,14, 99,52, 52,89,
  53, 0, 17,51, 84, 1, 59, 5, 49,54, 54,97,  6,37, 98,49,
  22,30, 37,63, 54,98, 31,25, 38,93, 25, 2, 97,33, 36, 4,
  24, 2, 53,45, 89,82, 97,31, 45,66, 85,77, 10,79, 31, 1,
  44,49, 36,53, 33,68, 59,14, 84,93,  9,18, 17,55, 72,28,
  21,31, 23,63, 44,57, 12,85, 97,97, 11, 6, 41,76, 81, 9,
  44,51, 78,31, 35,94, 94,80, 30,86, 56,12, 85,39, 23, 5,
  89,51, 80,92, 71,62, 92,23, 54,11, 85,16, 84,59, 23, 3,
  65,32, 86,37, 77,24, 73, 9,  7,55, 41,93, 50,11, 65,45,
  16, 0, 22,51, 53,72, 75,55, 23,29, 30,58, 45,65, 66,48,
  97,72,  9, 4, 35,27, 31,67,  6,24, 93, 8, 26,85, 25,12,
  30,34, 19,36, 99,30, 55, 5, 14,26, 27,27, 46,16, 83,67,
  12,41, 81,25, 37,88, 48,42, 34, 9, 97,46, 92,98, 42,56,
  53,71, 33,61, 42,81, 59,58, 56,40, 41, 5, 90,56, 51,74,
  86,20, 98,59, 56,88, 60,21, 65,75, 74, 3, 60,35, 53,48,
  58,44, 47,81,  4,83, 66, 9, 35,52, 42,61, 57,11, 59,91,
   6,84, 97,74, 70,44, 92,17, 27,56, 93, 9, 18,59, 83,68,
  76,33, 54,95,  0,42, 84,60, 62,50, 38, 2,  0,45, 30,12,
  82,93, 55,45, 32,37, 43,77, 56,17,  8,49,  5,28, 59,12,
  71,99, 35,12, 36,41, 14,55, 32, 5, 22, 4, 68,82, 97,69,
  84, 2, 45,74, 57,83, 53,32, 53, 7, 86,43, 41,96, 15,50,
  16,20, 36,63, 33,37, 31,23, 98,39, 28,79, 27,45, 92,25,
  63, 2, 66,64, 74,20, 83,43, 26,12, 30,27, 22,48, 54,18,
  79,54, 48,23, 26,46, 96,32, 44,15,  4,62, 45,27, 55,74,
  88,34, 77,22,  2,10, 42, 5, 38,52, 72,25, 87,78,  9,48,
  73,59, 16,60, 42,22, 70,26, 95,76, 86,98, 81,27,  1,39,
  14,69,  8, 8, 29,54, 52,86, 10,71, 48,38,  8,17, 58,86,
  38,78, 58,21, 89,81, 32,43, 10, 1, 18,11, 84,88, 76,42,
  39,85, 58,53, 32,44, 47,42, 13,72, 85,63, 43,15, 63,49,
  62,87, 11,51, 58,52, 74,63, 58,24, 86,74, 86,29,  0,25,
  86,31, 98,50, 51,50, 35,69, 68,24,  0,59, 98,11, 21,73,
  95, 2, 88,23, 61,23, 74,30, 11, 2, 99,99, 19,41,  9,35,
  77, 6,  6,26, 42,84, 24,15, 71,92, 29,60,  7,83, 50,29,
  37,10, 75,86, 21,32, 46,97, 46,46, 28,68, 91,24, 64,71,
  66,24, 16,53, 90,50,  1,37, 97,98, 53,28, 91,19,  7,46,
  52,15, 62,18,  6,45, 17,50, 34,30, 73,13, 39,90, 56,64,
  75,63, 20, 0, 72,77,  5,24, 67,74, 42,95, 30,83, 41,59,
  30,85, 26,76, 38,86, 16, 8, 81,87, 45,15, 20,91, 50,42,
  59,96, 55,40, 14,91, 42,82, 14,80, 28,29,  0,10, 50,49,
  55, 6, 58,12, 47,63, 96,59, 72,21, 99,43,  6, 7, 87,70,
  18, 0, 63,59, 55,32, 77,67, 72,67, 94,59, 72,55, 27,86,
  69, 2,  8,83, 36,12, 90,71, 44,80, 14,68, 42,10, 61,83,
  81,96, 88, 1, 86,41, 61,85, 97,44, 81,84, 70,69, 52,25,
  56,41,  5,32, 72, 6, 75, 4, 39,46, 37, 6, 44,54,  1, 4,
  49,77, 43,18, 35,14, 67,46, 67,54, 75,23, 25,94, 30,24,
   3, 7, 32,88, 64,87, 14,90, 17,81, 86,34, 95,60, 93,95,
  61,20, 66,45, 43,87, 56,95, 54,65, 37,93, 64,25,  4,70,
  25,17, 64,14, 20,47, 37,50, 63,86, 42,62, 95,84, 72,56,
  18,35,  9,22, 53,10, 59,77, 35, 9, 45,43, 77, 1, 32,22,
  10,33, 52,75, 40,97, 66,26, 86,65, 76,25, 13,87, 63,81,
  77,65, 45,16, 29,81,  7,15, 90,68, 51,15, 76,18, 83,87,
  30,91, 59,38, 38, 4, 41,25,  4,30, 30,28, 23,72, 97,96,
  62,12, 16,37, 36,77, 35,58, 52,79, 16,48, 87,18, 91,83,
  89,75, 15,26, 83,37, 82,51, 95,24, 72,29, 34,69, 84,62,
  47, 3, 85,98, 34,68,  2,35, 98,44, 96,86, 66,63, 85,20,
  33,63, 78,40, 24,30, 47,37, 21,81, 58,96, 35,57, 71,14,
  59,39, 12,86, 97, 0, 91,14, 81,58, 68,47, 50,10, 10,99,
  13,36,  4,75, 16, 6,  5,52, 84,74, 37,13, 61, 8, 12,61,
  95,88, 31,99, 54,62, 15,86,  5, 6, 94,50, 13, 2, 19,26,
  80,56, 15,43, 91,46, 87,72, 62, 8, 93,20, 29,77, 37,16,
   1,45, 71,69, 38,49, 87,42, 88,21, 18,33,  9, 2,  9,19,
  54,42, 77,87, 49,56, 82,74, 80,40, 28,49, 41,18, 79,79,
  49, 2, 78,67,  2,28, 80, 4, 49,14, 47,60, 10,97, 41,70,
  73,27, 51,77, 48,55, 24,73, 40,40, 21,94, 66,49, 55,84,
  56, 7, 93,39, 20,95, 70,28, 18,78,  6,78, 61,15, 76,94,
  56,58, 25,84, 82,42, 50,19, 46,20, 83,31,  4,31, 46,14,
  82,87,  0,41, 56,27,  8,95, 38,97, 90, 3, 54,86, 70,42,
  88,78, 18,42, 54,63, 24,56, 81,10, 13,97, 40,35, 61,35,
  53,94, 90,44,  1,10, 13,40, 85, 2, 18,43, 44,47, 36,74,
  78, 9, 33, 8, 16,78, 59,10, 91,22,  6,10, 88,73,  3,37,
  61,37, 58, 5, 49,15, 75, 0, 96,40, 58,61, 32,67, 27,13,
  21,19, 73,88, 31,31, 41,79, 46,54, 98,48, 25,86, 77,99,
  85,85, 90,84, 33,15, 69,28, 59,55, 82,75,  3,13, 35,46,
  26,59, 84,64, 72,74,  7,14, 55,60, 24,33, 29,32, 41,47,
  27,52, 61,69, 33,66, 79,13, 66, 8, 79,37, 90,83, 77,70,
  91,43, 92,42, 35,76, 36,96, 46,74, 95, 1, 87,11, 98,32,
  13,54, 11,62, 74,48, 78,69, 80, 7, 12,80, 97,99, 25,15,
  48,31, 47,69, 51,35, 66,96, 73,89, 36,78, 88,75, 72,46,
  72,37, 24,21, 90,90, 47,70, 60, 0, 46,48,  4,23, 79,49,
  91,51, 61, 9, 91,37, 42,67, 62,41, 41,54, 18,39, 77,63,
  35,59, 21,67, 74, 5, 50,81, 27,37,  9,89,  5,29, 14,27,
  64,92, 60,75, 58,68, 52,26, 70,64, 65,33, 92,62, 67,35,
  76,81, 52,14, 56,90, 22,95, 76,51, 31,82, 19,69, 65,86,
  43,99, 98,68, 59,25, 91,88, 36,70, 51,71, 84,99, 95,83,
  18,76, 42,83, 18,75, 56,32, 44,40, 12,57,  8, 5, 46,39,
  85,45, 88,38, 94,62, 77,42, 47,80, 47,25, 34,92, 96, 9,
  99,69,  3,59, 92,19, 80,83, 16, 4, 56,78, 37,40, 94,26,
  19,30, 34,28, 46,44, 26,10, 17,24, 82,82, 52,71, 81,49,
  86,95, 16,98, 89,42, 98,73, 18,61, 53,63, 20,59, 41,77,
  98,14, 28,88, 53,36, 19,81, 90,93, 27,79, 97, 6, 74,57,
  38,18, 62,13, 32, 4, 21,92, 52,97, 47,16, 78, 8, 56,52,
   4,67,  0,38, 93,92
};
const int p_100_11[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  712,
  /// Array of preferences
  46,74, 78, 5, 64,40, 50,11, 98,69, 70, 7, 74,13, 64, 8,
  47, 3, 18,84, 23,59, 63,34, 88,61,  8,57,  8,69, 21,60,
  55,27, 52,32, 38,33, 42,25, 58,69, 98,20, 28,87, 52, 6,
  85,80, 81,55, 61,93, 18,29, 48,68, 61,47, 76,75, 67,51,
  20,66, 63,37, 87,19,  3,78, 61,90, 42,35, 24,35, 87,58,
  55,90, 56,56, 27,83, 41,41, 78,31, 70,91, 22,91, 21,90,
  23,85, 15,92, 63,59, 42,69, 11,83, 12,61, 95,26, 14,72,
  47,87, 68,89, 82,23, 47,14, 23,93, 80,93, 34,16, 91,54,
  81,17, 63,69, 64,78, 21,45, 63,38, 20,24, 53,94, 73,45,
  28,85, 22,84, 41,11, 86,20, 16,43, 37,84, 41,74, 67,41,
  50,92, 37,59,  3,48,  3,28, 89,47, 32,30, 20,28, 34,48,
  58,25, 10,40, 54,90, 66,32, 59,53, 87,25,  1,71, 66,19,
  37,19, 91,92, 40,69, 84,77, 57,50, 46,73, 55,71, 23,43,
  18,39, 44,25, 68,88, 18,48, 77,86, 77,64, 18,40,  4,55,
  41,30, 71,29, 65, 1, 10,14, 80,89, 11,43, 88,40, 24,25,
   0,17, 31,33, 36,72, 57,90, 19,95, 17,31, 26,60,  2,62,
  18,19, 32,33, 96,64, 62,42, 97,12, 17,56,  9,36,  9,32,
  30,17, 64,32, 31,14, 80,52, 17,71, 70,34, 76,61, 59,81,
  70,78, 86,81, 18,76,  4,89, 67,50, 85,41, 17,42,  0,15,
  50,32, 31,83, 75,34, 62,85, 68,15, 44,35, 19,66, 12,39,
  60,11, 81,50, 11,86, 64,26,  5,52, 61,72, 89,99, 76,72,
   2,26, 21, 7, 58,75,  5,96, 81,69, 59, 2, 57,76, 82,45,
  57,58, 48, 3, 26,65, 99, 4,  2,50, 50,46, 19,29, 14,89,
  43,56, 10,56,  6,24, 22, 4, 15,19, 60,92, 42,13, 36, 3,
  87,71, 61,16, 55,60, 26,92, 33,39, 50,64, 95,66,  9,54,
   7,66, 71,44,  5,36, 86,56, 35,82, 92,98,  1,56, 31, 4,
   4,64, 15,55, 54, 2, 98,76, 21,84, 11,56, 52,43, 27,10,
  30,82, 11,69, 58,34, 84,44,  4,56, 68,68, 88,37, 90,72,
  21,13, 40,11, 56, 7,  9,82, 69,63, 74,39, 26, 6, 36,23,
  73,23, 66,20, 88,31, 56,66,  3,38, 92,38, 53,67, 77,70,
  73,40, 75,92, 54,23, 68,49, 30,97, 96,55, 90,59, 43,22,
  52, 9, 65,93, 87,46, 82,77, 73,74, 13,59,  8,89,  8,92,
  71,64, 92,86,  2,72, 97,65, 46,20, 35,87, 74,36, 43,86,
  82,44,  6,64, 53,21, 26,55, 85,45, 25,37, 37,11, 31,44,
  23,87, 53,62, 28,77, 75,37, 41,64, 38,70, 76, 5, 54,57,
  90,39, 57,48, 99,21, 98, 8, 85,22, 75,54, 19,30, 49,21,
  99,16, 42,52, 55,28, 53,77,  9, 2, 99,89, 65,12, 89,73,
  30,96, 37,95, 61,76, 85,28, 76,88, 27, 6,  5,62,  8,25,
   5,48, 98,48, 20,84, 21,99, 39,38, 78,51,  2,73, 14,23,
  29,72, 67,83, 42,29, 59,97, 67,88, 28, 6, 80,92, 22,72,
  39,72,  0,34, 87,67, 13,20, 55,22, 43,19, 37,17, 66,24,
  15,36, 19,52, 67, 2, 27,23, 14,52, 59, 0, 26,23, 92,32,
  79,74, 33, 1, 59, 4, 79,32, 44,27, 27,29, 84,36, 63,94,
  46,40, 95,61, 92,28, 68, 0, 95,19,  5,27,  5,72, 75,28,
   7, 9, 31,68, 23,96, 92,99, 54,27, 50,21, 46,29, 77,93,
  22,82, 38,98, 19,65, 64, 0, 80,10, 92,37, 23,57,  0,47,
  75,23, 20,30, 26,14, 67,53, 46,19, 73,88, 56,30, 11,92,
  25,33, 31,54, 81,64, 51,38, 56,41, 25,85, 75,83, 72,71,
   2,78, 14,85, 86,27, 45,51, 13,40,  3,71, 39,60, 62,90,
   5,24, 47,70,  9,38, 61,77, 40,41, 81,51, 10,23, 28,29,
  20,75, 16,26, 42,33, 24,44, 10,45, 21,50, 14,83, 48,55,
   9,52, 44,71, 69,74, 65,34, 90,18,  0,29, 58,40, 15,61,
  65, 7, 30,79, 55,19, 92,63, 82,51, 75,63, 37,72, 92,70,
  70,46, 55,21, 74,56, 74,81, 39,74, 45,16, 64,13, 80,65,
  23,71, 80,80, 88,28,  7,13, 89,39, 84, 9, 90,56, 77,49,
  73,43, 77,36, 65,86, 36,68, 28,65, 17,63, 79,30,  5,43,
  17,46, 95,98,  3,61, 72,81, 30,48, 56,92, 85,30, 98,89,
  81,39, 83,99, 89,97, 15,73,  9,95, 11,15, 15,48, 28,22,
  22,93, 53,87, 19,12, 20,23,  4,99, 99,37,  3,23, 97,68,
   7,58, 46,94, 45,94, 90,25,  3,26, 55,11, 90, 0, 93,24,
  98,40, 92,59, 19,62, 76,64, 71,49, 24,84, 23,64, 71, 6,
  74,62, 64,38, 85,59, 49,19, 24,26, 72,30,  6,90,  6, 5,
  70,36, 71,96, 83,40, 15,69, 55, 5, 73,86, 67,44,  6,34,
   7,47,  5,49, 91,73, 52,50, 28,39, 33,62, 58, 7, 30,98,
  38,21, 63,55, 43,84, 32,11, 45,41, 22,96, 26,82, 16,87,
  86, 2, 89,51, 75,45, 36,70, 76,67,  6,43, 29,32, 62,56,
  22,24, 52,58, 90,51, 10,57, 70,44, 37,23, 35,77, 84,30,
  44,45, 33,26,  4,40, 97,90, 33,19,  8, 9, 83,32, 32,96,
   6,35, 76,40, 80,83, 96,92, 86,49, 82,63, 81,77, 71,61,
  10,50, 42,71, 62, 4, 89,83, 31,71, 54, 1, 59,11, 72,24,
  55,41, 86,90, 44,83, 88,95,  1,40, 22,46, 76,56, 26,30,
  60,79, 73,78, 60,86, 62,81, 32,52, 44,96, 39,59, 30,45,
  10,29, 33,25, 57,59, 23,61, 96,20, 73,53, 50,98, 53,96,
  32,73,  5,14, 79,46, 35,18, 41,17, 59,17, 84,67, 39, 8,
  55,17, 42,46, 46,93, 91,98, 56,68, 89,76, 91,64, 65, 9,
  46,24,  4,83, 74,46,  8,58, 41, 6, 84,46, 10,86, 18,80,
  82,30, 50,59, 50,54, 68,66, 27,49, 10, 4, 13,22, 71,85,
  28,30, 86, 8, 30,34, 33,88, 56,73, 44,23, 99,35, 24,86,
  19,87, 93,87, 15,83, 66,30, 67,48, 30,13, 19, 0,  1,35,
  94, 6, 86,71,  5,70, 47,24,  8,10, 94,76, 88,19, 96,43,
  85,70, 52,26, 83,71, 37,31, 79,33, 21, 4, 56, 4, 81, 3,
  84,97, 79,24, 99,86, 69,73, 66,44, 29, 6, 18,65, 65,42,
  38,20, 22,65, 39,40, 66,50, 31,47, 15,20, 73,61, 71,32,
  14,27, 35,35, 97, 2, 37,88, 61,31, 16,73, 22,48, 48,59,
  45,31, 40,30, 31,66, 93,43, 93,81, 95,23, 82,41, 68,20,
  41,53, 52,55, 33,76, 78,56, 50,38, 86,83,  3,35, 54,91,
  29,29, 99,81, 24,23, 44,62, 38,83, 88,56, 53,18, 89, 1,
  10,64, 69,57, 73,28, 20,77,  0,11, 43,64, 82,61, 42,62,
  99, 9, 56,26,  6, 1, 31,45,  7,69,  8,60, 30, 5, 63,95
};
const int p_100_12[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  1091,
  /// Array of preferences
  29,54,  5,74, 15,96, 61,56, 12,48, 80,64, 79,13, 24,24,
  35,77, 77,71, 73,36, 86,64, 59,51, 94,72,  9,23, 30,69,
  79,29, 72,79, 57,10, 30,98, 73,30, 47, 5, 48,89, 66,70,
  15,49, 95,56, 11,36, 78,49, 40, 5, 49,82,  5,55, 34,44,
  69,48, 80,98, 21,21, 33,55, 10,16, 65,94, 18,83, 76,14,
  79,80, 16,88, 98,89, 97,12, 36,49, 34,36, 51,80, 79, 7,
  33, 8,  5,24, 93,53, 51,28, 88,12, 27,49,  1,27, 94,89,
  43, 9, 73, 4, 63,11, 83,36, 76,64, 11, 9, 44, 4, 52,25,
  66,10, 39,84,  4, 6, 32, 2, 54,14, 24,26, 85, 7, 22,89,
  77, 9,  7,18, 14, 8,  3,70,  6,73, 43,34, 71,34, 63, 4,
   5,51, 45,66, 48,35, 38, 7, 98,74, 74,83, 45,64, 24,30,
  29,72, 20,32, 58,53, 54,59,  6,27,  9,36, 44,87, 84,70,
  44,29, 20,72, 94,22, 24, 3, 34,39, 87,29, 85,78,  2,54,
  30,61, 37,48, 71,30,  9,81, 89,97, 64,91, 45,70, 67, 2,
  27,52, 22,46, 16, 4, 42,55, 99, 4, 17,96, 94,91, 98,36,
  65,54, 60,72, 80,25, 50,25, 91,86, 59,57, 53,78, 67,53,
  14,92, 58,44, 72,63, 91,99, 22,73, 50,70, 45, 0, 75,24,
  76,77, 13,22, 36,59, 59,22, 61,17, 32,73,  6,23, 98,59,
  24,28, 63,23, 30,81, 50,19, 15,27, 10,37, 91,42, 29,14,
  35,50, 80,21,  6,72, 71,33, 10,34, 35,47, 14,52, 34,63,
  30,48,  5,73, 72,73, 39,46, 41,99, 71,72, 44,34, 43,67,
   7,47,  1,13, 38,59, 73,37, 62,19, 71, 5, 68, 2, 40,28,
  15,83, 17,61, 97,18,  7,84, 88,36, 61,84,  8,23, 29,35,
  65,42, 49, 6, 35,71, 16,29, 26, 0,  5,47, 31,95, 95,58,
  71, 0,  5,57, 50, 7, 21, 6, 47,85, 85,31,  0, 0, 18,63,
  79,14, 54,99, 47,52, 21,19, 32,20, 53,11, 52,70, 42,28,
  69,41, 56,93, 87,73, 83,96, 91,93, 11,16, 61,20, 60,85,
   3,98, 35,60, 99,33, 25,93, 67,35,  1,65, 51,78, 95,74,
   3,63, 20,23, 40,29, 34, 8, 25,52, 20,67, 87,19, 26,28,
  99,65, 69,95, 58,91, 43,39, 13,70, 54,84, 61,18, 49,45,
  98,57, 99,46,  8, 7, 88,51, 18,13, 59,82, 51,72, 90,77,
  78,46, 94,12, 16,25, 52,92,  5,65, 86,87, 16,89, 48,61,
  79,95,  1,53, 83,33, 25,41, 80,56, 35,39, 64,29, 86,57,
  54,15, 52,54, 59,46, 53,93, 95,11, 36,81, 79,98, 61,45,
  65,17, 22, 7, 75,22, 66,14, 48,80,  0,26, 82,32, 36,94,
  33,53, 39,16, 78,59, 93,21, 39,15, 49,60, 42,62, 86,90,
  75,83, 50,52, 91,70,  4,26, 57, 7, 23,75, 95,81, 84,68,
  37,10, 18,33, 58, 7, 60,41, 25,34, 70,54, 81,55, 26,55,
  10, 7, 81,57, 24,42, 88,43, 33,58, 87,33, 71,92, 65,90,
   7,29, 70,89, 96,56, 49,96, 24, 4, 36,50, 29,21, 90,25,
  32,62, 77,88, 28,80, 59,24, 75,26, 13,61, 77,40, 35,20,
   9,79, 70,91, 77, 1, 90,88, 45,28,  8,91, 36, 6, 41,57,
   9,65, 76,62, 45,97, 70,17, 59,39, 62,52, 14,54, 65,28,
  46,49, 66,84, 84, 5, 67,71, 12,92, 67,73, 73,79, 59,12,
  29,39, 15,71, 84,67, 48, 7, 87,77, 63,43, 68,43,  0,80,
  23,57,  3,25, 92,92, 73,89, 40, 2, 34, 7,  7,66, 12,46,
  51,95, 31,32, 65,15, 25,38, 55,48, 47,24, 76,28, 74,50,
  84,47, 71,15, 80,27, 99,19, 98,55,  5,96, 50,61, 52, 8,
  41,25, 72,59, 69,24, 33,63, 79,36, 16, 3,  6,88, 35,64,
  44,98, 12,82, 45,22, 22,83, 12,19, 99,26, 97,11, 55,29,
  91,32, 93,29, 89,77, 51,60, 52,94, 74,42, 23,10, 62,96,
  23,60, 23,96, 87, 4,  8,95, 11,11, 98, 0, 40,59, 42,40,
  32, 3,  6,62, 55,18, 84,94, 85,58, 97,28,  1,74, 28,59,
   5,49, 47, 9, 77,96, 22,27, 43,68, 30,73, 83,62, 88,60,
  66,76, 62,77, 13,23, 87,10, 90,42, 91,10, 42,18, 31,40,
   8,63, 36,76, 27, 3, 30,27, 90,33, 66,57,  6,68,  2,81,
  66,33, 71,61, 43,55,  4,75, 17,84, 83,90, 66,35, 91,61,
  83,34, 88,39, 49,20, 55,98, 35,72, 71,80, 13,88, 52,71,
   6,20, 22,78, 18,93, 64,54, 78,35, 62,66, 43,18, 61,55,
  67,79, 86,83, 54, 5, 81,51, 88,90, 42, 0, 97,33,  3,50,
  53,47, 91,22, 99, 3, 13, 8, 25, 6, 75,31, 57,77, 27,40,
   0,94, 47,98, 90,23, 76,13, 62,21, 70, 5,  5,67,  9,50,
  31,96, 80,33, 16,94, 65,74, 24,47, 30,23, 57,97, 97,35,
  51,86, 73,31, 49,27, 50,80, 38,32, 90,27, 39,78, 29,16,
  63,24, 92,41, 96,25, 55,39, 28,94, 45,17, 68,93, 34,58,
   6,12, 60,36, 46,80, 46,44, 35,45, 79,73, 20,52, 39,11,
  40,12,  0,54, 26,95, 58,45,  6, 4, 56,61, 57,19,  6, 8,
  86,45, 96,22, 79,54, 93,34, 79,70,  6,85, 27,88, 32, 7,
  70,29, 19,55, 11,39, 32,97, 58,82, 16,52, 14,98, 94, 8,
  20, 2, 11, 7,  5,26, 38,90, 15,44, 83, 7, 73,11, 71,45,
  70, 9, 65,16, 85,37, 44,38, 38, 4, 68,61,  8,72, 83,18,
  63,12, 42,94, 12,47, 69, 7, 40,78, 98,35, 74,12, 60,96,
   9,77, 42,13, 63,97, 49,22, 43,69, 21,76, 25,49, 46,26,
  20,66, 14,45, 21,84, 77,86,  5,66,  6,98, 13,64,  0,51,
  12,54, 48,37, 89,17, 80,19, 90,64, 37,19, 25,19, 61,50,
   3,22, 91,45, 55,72, 55,14, 19,17, 75,57, 19,71, 73,72,
  62,39, 14,68, 52,18, 85,65, 23, 4, 43,13, 42,78, 93,10,
  25,57, 93,47, 75,19, 10,28,  7,76, 60,95, 53,21, 70,37,
  19,27, 60,77,  1,75,  3,69,  6, 7, 61,21, 20,68, 65,64,
  82,96, 87,89, 91, 9, 32,55, 32, 9, 25,45, 76,89, 61,48,
  99,74, 48,18, 58,39,  7,19, 16,67, 18,24, 80,87, 11, 2,
  15,14, 16,58, 43,79, 99,91, 28,35, 78,53, 74,25,  2,37,
  74,44, 45,63, 96,84,  4,17, 75, 8, 28,48, 56,65,  7,56,
  21,56, 90,68, 48,41, 62,12, 10, 2, 35,61, 10,69, 77,93,
  41,82,  6,16, 96,87,  6, 0, 42,45, 49,65, 31,39, 87,30,
  76, 3, 63, 1, 13,32, 51,22, 32,92, 30,42,  9, 9, 48,20,
  18,20, 26,29, 99,98, 31,67, 31,89, 31,47, 89, 2, 37,28,
  96,16, 19,44, 44,21, 26,50, 33,36, 75,92, 69,39, 13,75,
  74,16, 98,95, 47, 3, 69,36, 20,63, 46,45,  9,78, 84,35,
  95,15, 35,12, 90,56, 20,56, 48,96, 32,86,  4,10, 22, 4,
  25,77, 65,67, 92,81, 85,48, 12,85, 31,91, 63,93, 90, 0,
  77,70, 69,74, 92,22, 40,72, 46,48, 76,50, 83,76, 96, 2,
  74, 0, 81,26, 34,11, 39,62, 80,79, 25,72, 32,34, 60, 3,
   3,97, 27,78, 14,97, 17,17, 14,20, 10,45, 73,65, 72,87,
  56,26, 26,73, 95,93, 54,67, 67,96, 53,37, 94,23, 90,22,
  57,88, 95,82, 34,29, 45,26, 85,97, 40,38, 95,24, 16,73,
  51, 7, 82,49, 83,39, 44,15, 83,42, 61,80, 82,43, 52,55,
  89,40, 74,13, 24,74,  5, 1, 42,27, 22, 6, 22,14, 98,46,
  76,17,  5,37, 48,63, 69,25, 82,27, 93,42, 38,99,  6,40,
  48,92,  8,25, 45, 1,  5,81,  9,76, 96,82, 81,76, 30,43,
   9,75, 39, 7,  1, 1, 79, 4, 78,87, 55,81, 13,16, 89,98,
  77,74, 23,36, 32,10, 76,60, 61,82, 30,65, 41,81, 88,32,
  13,97, 99,51, 88,93, 17,90, 62,93, 77,60, 11,27, 89, 9,
   0,22, 87,50, 55,88, 23, 6, 81,80, 17, 6, 21,79, 58,29,
  90,61, 94,92,  3,73, 81, 7, 41, 9, 68, 8, 97,53, 71,27,
  86,66, 43,29, 14,37, 47,29, 88,22, 17,63, 72, 9, 93,91,
  45,49, 96,70, 26,72, 11,70, 29,96, 54,78, 90,85, 20,42,
  17,98, 22,30, 56,97, 37,14, 24,43, 50,77, 30,11, 93,87,
  18,39, 17,15, 16,78, 86,46, 48,82,  0, 8, 99,99, 94,65,
  86,31, 43, 8,  7,61, 56,92, 61,35, 78,52, 32,46, 47,82,
  65,88, 25,96, 70,93, 79,53, 63,15, 60,51, 97,45, 11,87,
  95, 7, 68,76, 19, 5, 96, 6, 99,40, 57,43, 51,97,  2,64,
  30,33, 34,86, 67,84, 12, 8, 62,73, 36,32, 73,59, 39,60,
  83,98, 85,94, 98,43, 82, 7, 46,31, 58, 8, 51,46, 97,38,
  68,85, 85, 8, 98,27, 20,12, 25,30, 49,81, 34,82, 70,34,
  16, 7, 91,58, 81,46,  8,53, 50, 4, 40,97, 45,85, 97,90,
  34,12, 36,17, 31,69, 58, 2,  1,92, 25,99, 48,30, 11,68,
  52,48, 54,94, 86,55, 34,25, 44,58, 79,20,  4,86, 27,87,
   1,50, 74,40, 56,47, 69, 0, 74,74, 25,74, 66,18, 64,72,
  35,99, 56,89, 99,84, 72,19, 15,23, 26, 9, 64,48, 25,10,
  73,95,  7,68, 37,72, 98,97, 49,91, 32,12, 93,62, 17,46,
  92,33, 80,58, 88,25, 80,55, 53,82, 90,93, 37,44, 52,36,
  90,47, 62,85, 77, 6,  7,31, 93,59, 23,35, 23,22, 54,96,
  20,90, 78,85, 50,13, 81,32, 85,30, 28,58, 52,33,  0,17,
  14,33, 61,92, 70,55, 94,71, 21,17, 27, 2, 87, 3, 28,13,
  76,80, 87,22,  7,17,  9,80, 44,10, 50,99, 12,90, 95,29,
  70,64, 64,35, 45, 8, 77,72, 50,45, 59,19, 40,43, 91,69,
  30,58, 41,21, 14,91,  2, 1, 60,67, 32,77, 68,42, 15,61,
  24,31,  2,52, 55,82, 17,45, 62,41, 72,47, 48,90, 18,42,
  71,53, 21,53, 74,21, 98,17, 52,42, 21,30, 45,82, 37,69,
  18,59, 29,27, 85,70, 12,66, 60,27, 31, 3, 40,69, 17,51,
  15, 5, 16,66, 87,62, 39,22, 33,84, 39,95, 29, 9, 57,36,
  19,42, 80,93, 24,39, 75,62, 98,42, 42,34, 34, 9,  9,39,
  37,91,  1,55, 96,31,  5,90, 18,65, 78,97, 24,88, 45,90,
  31,22, 99,61, 53,32, 20,82, 14,59,  1,35,  8,86, 14,49,
  65, 2,  2,36, 91, 6, 49,41, 45,52, 88,54,  6,54, 68,14,
  34,31, 67,91, 27,75
};
const int p_100_13[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  1040,
  /// Array of preferences
  48,57, 49,93, 93,24, 24,98,  4,30, 28,56, 29, 1, 88,78,
  63,56, 72,95, 98,72, 44,78, 97,29, 41,77,  9,90, 40,70,
  39,58, 36,32, 81,98, 44,32, 26,90, 71,85, 55,43, 64,57,
  52,42, 38,87, 42,26, 68,80, 43,14,  6,64, 16,60, 98,60,
   2,68, 11, 8, 47,70, 61,67, 96,24, 81,71, 33,69, 72, 5,
  66,44, 52,35, 79,36, 49,55, 90,16, 26,17, 35,31, 47,24,
  56,75, 78,55,  3,80, 88,91, 99,86, 37,57, 30,20, 47,41,
  35,53, 59,70, 75,29, 90,90, 75,13,  9,20, 49,60, 16,61,
  74,21, 15,33, 88,55, 51,90, 74,63, 30,62, 79,13, 48, 9,
  93,23, 63,58,  9,61, 19,17, 90,78, 76,28,  4,45, 91,30,
  91,56, 99,36, 25,59, 69,40, 25, 1, 14,48, 85,39, 46,95,
  78,78, 72,69, 39, 2, 74,24, 35,99, 23,31, 76,29, 30,66,
  79,16, 65,36,  9,42, 38,46, 45,63, 81,34, 97, 6, 75,26,
  62,13, 13, 6, 58, 1,  3, 1, 86,77, 34,34, 19,82, 97,89,
  62,90, 16,62, 67,38, 62,89, 92,61, 26,87,  3, 9, 92,99,
  11,93, 95,91, 38,70,  4,79,  8,78, 67,73, 27,30,  0,80,
  92,57, 53,38, 46,22, 51,64, 84,66, 76,22, 86,23, 22,77,
  25,96, 67,90, 29,80, 39,65,  9,10,  1,18, 47,35, 78,24,
  53,98, 70, 7, 81,10, 37,22, 43,42, 35,40, 78,27, 31,24,
  54,31, 56,99, 62, 0, 68,22, 89,72, 41,55, 66,37, 13, 7,
  58,10, 43,20, 46,61, 70, 5, 60,99, 24, 4, 65,14, 44, 6,
  51,27,  1,56, 35,52,  8,31, 10,87, 36,11,  5,93, 14,82,
  50,65, 39,71, 34,72, 52,92, 67,94, 93,71,  3,69, 64,26,
  84,89,  7,31, 47,85, 51,86, 70,76, 42,31, 10, 7, 40,42,
  82,16, 96,77, 78,94, 63,79, 13,53, 15,21, 11,15, 50,35,
  13,98, 11,87, 20,95, 62,84, 87,49, 31,27, 75,88, 23, 4,
  56,58, 45,61, 70,43, 83,28, 37,86, 61,59, 79, 9, 62,53,
  66,47, 84,71, 46,73, 57,99, 54,22, 13,20, 14,11, 81,44,
  48,49, 84,73, 21, 1, 74,61, 34,90, 67,52,  5,24, 77,60,
  99, 5, 17,83, 71,67, 68,47,  8,69, 97,58, 32,29, 39,63,
  12,29, 47,59, 76,94, 70,78, 81,79, 32,52, 83,60, 40, 5,
  54,28, 88,70, 63,32, 92,58, 30,23, 16,48, 32,49, 78,20,
  50,97, 43,56, 56,24,  2,83, 88,76, 31,12, 30,58, 26,51,
  38,17, 93,25, 72,10, 14,85, 64,23, 75,61, 50,63, 75,67,
  31,42,  4,83, 53,86, 49,72, 59,88, 31,83, 81, 6, 88,83,
   9,12, 35,45, 77,70, 64,44, 28,83, 87,48, 95, 7,  5, 8,
  63,11, 68,77, 73,33, 40,27, 48,37, 83, 1, 66,25, 17,63,
  90, 7, 26,79, 30,28, 61,43, 55,79, 89,24, 85,38, 41,73,
  14,35, 15,10, 35,96, 33,40, 30,70, 39,62, 93,29, 12,57,
  56,18, 81,92, 60,76, 21,63, 96, 4, 20,14, 77,65, 83,95,
  50,16, 25,16, 80,73, 40,18, 70,31,  5,55, 61,99,  0,47,
  88,13,  0,87, 82,55,  5,83, 51,89, 86,34, 95, 4, 48,96,
  60,42,  3,34, 47,56, 17,62, 19,31, 85,63, 10,69, 36,33,
  26,47, 73,28, 94,81,  5,22, 30,41, 72,38, 72,81, 76,47,
  31,38,  8,95, 50, 9, 91,98, 50,81, 14,22, 41,29, 27,52,
  81,70, 18,64, 43,28, 59, 4, 57,45, 28,19, 55,91, 78,15,
  84,14, 25, 6, 16,93, 14,63, 19, 2, 19,73, 89,94, 93,12,
  47,64,  3,65, 26,36,  0,89, 25,30, 42,91, 10,60,  3,36,
   5,66, 78,21,  8,37, 43,79, 47,84, 78,89, 42,51, 66,52,
  48,13,  7,53,  6,48, 52,34, 22,29, 32,66, 53,95, 96,67,
  20,70, 27,86, 47,49, 17,86, 46,47, 10,22, 44,96, 24,82,
  28,68, 41,78, 50,79, 96,20, 67,43, 29,97, 61,77, 53,22,
  83,62, 21,20, 99,20, 47,89, 52,81, 92,49, 21,19,  3,57,
  98,16, 11,64, 42,30, 55,92,  9,38, 26,39,  4,19, 58,31,
   7,74, 37,20, 55,89, 19,52, 74,80, 99,51, 25,49, 51,91,
  77,96, 82, 6, 78,47, 93,95, 76,54,  9,58, 57,66, 97,59,
  48,86, 53,50, 89,18,  9,16,  3,28, 52,44, 43,63, 56,38,
   1,41, 45,77, 60,79, 65,74, 84,46, 92,83, 32,74, 87,70,
  95,92, 52,70, 69,94, 19,42, 44,59, 44,28, 89,65, 65, 9,
  29,54, 66,54, 56,82, 81,16, 43,45, 71,77, 15,19, 39,23,
  16,14, 81,14, 66,57, 15, 3, 45,15, 14,32, 56,22, 87,91,
  46,77, 23,34, 74, 9, 10,12, 20,46, 66,90, 14,21, 49,14,
  93,74, 95,33, 30,10, 50,32, 10,84, 66,60, 46,97, 24,97,
  59,73, 29,20, 58,39, 48,16, 89,21, 56,44, 37,84, 25,35,
  91,11,  1,58, 76,33, 71,97, 76,74, 67,30, 34,50, 40,47,
  25,99,  6,78, 34,38,  4,56, 44,43, 77,44, 68,56, 28,23,
  45,84, 71,28, 71, 3, 56,67, 40,68, 90,11, 25,83,  4,63,
  62, 1,  6,81, 64,81, 32, 5, 61,72,  1,30, 82,19,  7,48,
  97,22, 77,79, 45,12, 80,49, 88,21, 42,80, 51,81, 20,73,
  50,91, 92,26, 97,51, 69, 7, 60,29,  5, 2,  9,44, 75,25,
  32,33,  6,55, 78, 0, 38,54, 71,68, 53,67, 13,34,  5,49,
  52,16, 86, 4, 45,29, 72,89, 87,66, 24,64, 74,37, 26,49,
  90,39, 52,86, 30,75, 59,81, 23,15, 16,15, 17,48, 62,96,
  74,72, 46,56, 24,35, 58,90, 62,73, 24,96,  7, 0, 57,68,
  52,32, 81,52,  3,55, 92,43, 13,82, 69,11, 78,12, 65,59,
  15,84, 14,78, 12,91, 30,89, 97,91, 91,33, 64, 4, 23,19,
  83,27, 25,20, 38,77, 63,80, 29,52, 32,64,  1,71, 30,98,
   0,91, 35,26, 33,28, 48,11, 96,62, 81,88, 88,90,  7, 3,
  74,15, 43,99, 99,24, 96,35, 11,80, 37,69, 38,22, 28,77,
  69,19,  8,22, 55,21, 56,52, 46,88, 69,31, 37,12, 55,82,
  83, 9, 21,69, 64,56, 29,55, 32,42, 58,53, 23,40, 31,61,
  71,93, 45,10,  8,86, 39,19,  5,70, 84,55, 21,52, 44,31,
  79,94, 21,86, 80,16,  5,88, 87,41, 30,56,  5,37, 41,69,
  18,70, 33,55, 46,36, 73,14, 36,31,  8,94, 74,71, 80,62,
  32,12, 80,27, 85,21, 46,38, 65,46, 75,41, 68,18, 25,34,
  79,97, 99,58, 35,42, 15, 8, 20,13, 66,83, 55,88, 64,84,
  90,37, 84,22, 21,26, 51,23, 76,99, 63,42,  1, 3,  4,66,
  50,23, 68,36,  5,85, 93,44, 32,30, 93,68, 56,41, 18,43,
  35,73, 52,66,  7,18, 50,84, 25,73, 49,31, 76, 8, 11,73,
  56,13, 25,23, 46,75, 52,93, 59,40, 42,35, 32,41, 98,10,
  40,46, 29,94, 35, 8,  7,20, 73,78, 52, 7, 18, 0, 87,86,
  65,62, 12,99,  8,62,  2,24, 57,86, 59,52, 11,40, 49, 9,
  57,72, 50,92, 74,43, 69,82, 30,91, 44,66,  0,33,  0,48,
  17,77, 97,50, 38,62, 53,68, 92,90, 30,46, 13,26, 85,12,
  83,40, 67,35, 83, 8, 85,50, 64,78, 55,41, 62,74, 71,19,
  82,65, 20,41, 72,61, 21,29,  0,25, 37,56, 67,40,  5,87,
  64,41, 73,24, 95,36, 39,98, 85,72, 39,67, 98,39, 28,79,
  29,23, 75, 4, 83,83, 21,32, 20,78,  2,88, 32,63, 80,59,
  59,28,  3,76, 81,97, 12,31, 14,27, 50,25, 60,50, 80,53,
  56,79,  2,90, 71,89, 47,99, 29,25, 68,90, 48,91, 30,53,
  59, 6, 68,88, 18,92, 15,70, 98,18, 24,63, 21,53, 74,16,
  70,69, 40,15, 75,80, 22,86, 69, 2, 61,71, 42, 6, 65,55,
  21,11, 53,41, 11, 5,  3,64, 87,15, 88, 0, 72,82, 42,96,
  37, 6, 63,45, 77, 9, 94, 2, 28,55, 66,70, 88,19, 36,58,
  33,23, 92,12, 36,24, 19,24, 82, 1, 76,79, 87,72, 33,97,
  41, 1, 40, 0, 38,48, 32,88, 48,66, 26,67,  7,41, 14,57,
  18,10,  4, 7, 56,94, 72,62, 59,95, 81,41, 57,51, 30,76,
  26, 1, 75,66, 97,95, 22,43, 29,44, 29,45, 75,85, 33,67,
   1,22,  1,73, 67,32,  6,73, 21,13, 90,35, 54,96, 84,67,
  19,85, 63,24, 59, 0, 10,89, 84,45, 22,31, 42,83, 79, 1,
  32,98, 23,36, 49,88, 62,59, 14,60, 25,31, 34,63, 91,31,
  94,86, 31,94, 80,18, 91,92, 78,85, 27,41, 54,20, 78,53,
  73,31, 28,33, 97,18, 72,12,  2,93, 97,69,  3,43, 35,17,
  63,93, 65, 1, 39,33,  6,10, 43,24,  0,41, 78,43, 14,92,
   9,27, 80,78,  3, 3, 45,76, 91,54, 91,13, 23,49, 56, 4,
   9,48, 65,96, 18,72, 35,20, 77, 6,  5,11, 40,80,  4, 4,
   1,34, 43,87, 44,71, 45,79, 21, 6, 54,27, 91,87, 58,33,
  34,64, 12,92, 84,31, 12,13, 87,51, 99, 3, 10, 9, 89,57,
  37,98,  7,65, 49,28, 21,81, 72,52, 48,76, 84,36,  4,97,
  47,22, 46,80, 99,16, 91,39, 10, 0, 72,43,  6,54, 21,99,
  72,74, 73,69, 61,34, 47,28, 37,76, 18, 8, 17, 7, 99,93,
  20,81, 96,18, 80,29, 31,20, 33,52, 41,68, 56,78, 74,91,
  12,28, 73,21,  9,63, 69,28, 88,32,  1,83, 90,94, 36,89,
  29,62,  3,73, 31,79, 90,26, 56,85, 30, 4, 36,63, 52,48,
  34,75, 46,48, 18,61, 41,83, 24,12, 62,33, 24,92, 10,92,
  83,85, 73,38,  9,29, 26,97, 27,66, 35,51, 67,98, 81,86,
   2,58, 94, 7, 12,90, 92,40, 45,67, 45,17, 46,62, 45,38,
  24,47,  5,61, 30,61, 49,40, 40,64, 44, 7, 51,61, 95,48,
  98,70, 26,48, 94,71, 95, 3, 10,13, 53, 4, 98,42, 40,86,
   4,36, 54,71, 69,48, 51,45, 48,84, 16,80, 39,57, 60,97
};
const int p_100_14[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  1391,
  /// Array of preferences
  65,43, 72,99, 50,24, 68,45, 59,49, 94,31, 52,88,  6,41,
   5,82, 88,30,  3,94, 17,29, 17,73, 54,47, 58,33, 99,48,
  72,53, 45, 0, 17,18, 44,42, 82,15, 25,25, 48,99, 85,13,
  86,86, 47,15,  9,93, 80,36, 71,57, 52,64,  3,69, 97,14,
   8,99, 64,57, 67,11, 58,78, 61,37, 43, 3,  7,17, 45,45,
  53,66, 30,50, 23, 6, 29, 4, 12,96,  0,30, 65,27, 61,11,
  57,60, 65,16, 53,81, 81,15, 27,69, 62,79, 25,85, 86, 6,
  10,54, 53, 1, 54,69, 21, 1,  6,49, 35,48, 66, 8, 12,81,
  11,59, 72,49,  9,70, 40, 2, 43,79, 73,90, 58,45, 66, 3,
   4,94,  5,77, 25,60, 41,33,  0,23, 62,76, 15,85, 47,52,
  54,55, 73,15, 73,39, 14,35, 44,13, 68,16,  5,76, 19,94,
  13, 4, 86,33, 39,40, 15,29, 58,20,  0,74, 68,50, 66,93,
  22,97, 70,53, 73,11, 34,44, 24,11, 15,86, 87,45, 56, 3,
  30,69, 32,83, 11,88, 42, 1, 99,64, 48,13, 22,54,  9,91,
  78, 0, 64,13, 48,88, 27, 5,  2,31, 77,34, 37,34, 89,45,
  37,39, 80,75, 40,70, 86,61, 38,55, 54,41,  3,96, 19,54,
   8,27, 23,74, 36,53, 23,52, 53,82, 52,82, 68,30, 56, 2,
  94,42,  0,93, 50,18,  0,17, 50, 9, 31,92,  1,66, 64,92,
  29,39, 63,50, 97, 0, 83,11, 98,34, 32,11, 34, 9, 10,17,
  25,97, 32,15,  6,74, 13, 0, 96,13, 11, 8, 76,66, 95,16,
  71,68, 93,75,  0,41, 87,77, 13, 9, 84, 8, 90,89, 58,92,
  41,11, 80,62, 55,27, 39, 8,  6,42, 66,36, 45,78, 64,24,
  46,36, 85,39, 94,99, 62, 8,  5,66, 77,38, 64,23, 59,79,
  57,34, 64, 3, 49,98, 79,96, 46,64, 78,77, 50,38, 66,97,
   7,77, 96,33, 77,57, 32,92, 87,92, 34,83, 17,52, 51,41,
  78,98, 79,30, 52,55, 17,10, 29,63, 12,16, 88, 4, 90,86,
  94,54, 12,44, 57,61, 35,17, 53,93, 51, 3, 99,13, 28,80,
  46,17, 87,17, 61,23, 60,51, 35,50, 44,71, 48,33, 20,90,
   9,23, 57,72, 55,28, 23,59, 44,10, 57,10, 95,76, 62,95,
  64,41, 79,15, 16,79, 32,21, 81,39, 24,75, 40, 4, 17,62,
  82,65, 98,91, 72,16, 11,16, 16,16, 33,87, 55,98, 42,55,
  34, 0, 21,95,  5,18, 17,96, 72,81, 71,74, 28,28, 62,28,
  31,59, 90,51, 71,94, 31,55, 79,33, 16,84, 49,44, 38,38,
  81,24, 98,39, 45,18, 65,76, 60, 3, 21, 8, 80,38, 37,77,
  57,35,  6, 3, 32,34, 80,78, 68,91, 86,41, 71,95, 56,67,
  14,77, 97,52, 92,27, 96,38, 39,67,  0, 1, 15, 1, 58,25,
  75,18, 84,40, 27,90, 68,23, 52,69, 78,43, 11,29, 65, 3,
  10, 3, 30,38, 64,26,  2,89, 40,80, 12,91, 45,95,  7,88,
   5,28, 86,31, 35,13, 53,25,  9,34,  5,97, 26,30, 71,34,
  39,82, 77,95, 38, 8, 86,79, 29,33, 26,63, 31,87, 10, 5,
  80,65,  5,50, 31,19, 31,93, 89,47, 34,92, 31,84, 86,48,
  63,83, 61,70,  7,98, 53,86, 55,50, 30,21, 89,24, 75,37,
  73,55, 73,98, 97,96, 21,59, 93,98, 41,59, 89,50, 10, 4,
  86,23, 52,86, 80, 4, 80,96, 30,31, 32,31, 94,60, 55,24,
  82,82,  5,29, 72, 3, 60,77, 32,63, 40, 9, 44,66, 37,22,
  71,14, 42,29, 31, 8, 47,65, 28,81, 34,57, 82,30, 80,61,
  13,56, 14,34, 85, 5, 19,89, 92, 0, 48,21, 94,51,  3,31,
  29,64, 15,27, 37,36, 97,98, 50,36, 22,82, 43,35, 66,26,
  79,19, 52,76, 16,67, 24,46, 16,47, 66,82, 25,93, 42,53,
  51,17,  6,44, 27,62, 63,25, 35,55, 82,10, 89,10, 31,21,
  37, 6, 23, 8, 27,92, 42,15, 38,81, 58,91, 48,97, 68,63,
  53,90, 50,68, 25,51, 95,88, 72,68, 16,49, 70,90, 17,46,
   8,33, 35, 6, 35,81, 83,60, 90,41, 29,26, 46, 7, 63,89,
  58,30, 43,80, 39,16, 39,80,  4,52, 58,31,  4,14, 60,44,
  49,51, 92,80, 34,59, 11,65, 77,63, 41,48, 20,25, 22,94,
  34,32, 32,19, 98,20, 91,56, 63,62, 73, 8, 14,15, 43,65,
  85,45,  7,58, 12,25, 25,76, 31,70, 30,57, 57,44, 43,59,
  53,76, 79, 6, 64,66, 65,60, 88,16,  6,51,  2,85, 64,75,
  31,94, 26,69, 22,21, 21,34, 55,81, 16,43, 61,24, 61,57,
  83,74, 10,82, 68,25, 81,34, 58,80, 92,35, 38,91, 19,60,
  17,11, 20,21, 19,43, 10,29, 44,94, 11,44, 34,18, 85,99,
  81,27, 68,95, 36,46, 49,78, 87, 6, 67,42, 24,26, 37,14,
  22,69, 70,88,  4, 1, 13,25, 23,70, 78,18, 11,53, 78,84,
  23,31, 85,36,  7,29,  0, 6, 59,10, 83,39, 10,91, 53,89,
  83,69, 23,11, 71,93, 20,29,  8, 7, 69, 6,  2,27, 20,98,
   0,36, 52,87, 15,92, 35,28,  3,82, 75,72, 28,34,  2,86,
  43,27, 80,64, 87,88,  9,52, 89,19, 74,45, 41,86, 90,33,
  25,63, 71,49, 37,66, 39,18, 85,58, 19,47, 24,71, 81,54,
   6,17, 22,60, 24,45, 78,21, 75,63, 28,39, 37,52, 38,73,
  42,37, 45,39, 76, 7, 84,49, 45,74, 45,61, 80,55, 69,65,
  30,66, 19,33, 30,82, 35,14, 81,62, 36,93, 68,51,  3,25,
  17,65,  5,13, 16,20, 61,48, 41,39, 30,72, 90,20,  2,11,
  96,55, 34,21, 37,40, 17,21, 59,47, 71,99, 32,12, 81,43,
  49,74, 60,31, 88, 3, 89,65, 26,11, 79,42, 93,66,  6,61,
  24,83, 52,38, 95,81, 43,49,  7, 6, 83,59, 47,63, 54,86,
  19,29, 78,82, 35,76, 39, 9, 83,84, 28,41, 80,48, 62,78,
  68,99, 40, 1, 10,33,  1,56, 17,54, 80,98, 50,74, 98,38,
  72,85, 81, 6, 66,32, 88, 0, 92,20, 26,37, 74,68, 87, 4,
  36,54, 10,37, 49,56, 67,69, 80,90,  8, 0, 56,69, 88,31,
  50,53, 51,85, 67,30, 45,65,  9,29, 65,12, 94,33, 95,24,
  23,69, 65,68, 65,13, 13,48, 37,70, 15,13, 24,28, 29,32,
  54,12, 74,67,  9,13, 45,62, 26,36,  1,15, 10,16, 32,24,
  24,22, 67, 4, 80,21, 57,27, 54,79,  0,65, 53,40,  3,65,
  21,60, 14,62,  8,86,  5,40, 90,81, 91,73, 36,83, 78,53,
  51,91, 70,58, 81,76,  0,69, 79,88, 86,34, 79,63, 80, 3,
  73,50, 68,15, 77,16, 82,49,  4,50, 83,65, 18,66, 47,94,
  24,91, 45,90, 82,96, 88,45, 63,27, 26, 5, 39,97, 36,70,
  10,30, 17,69, 25,91, 20,95, 11,98, 33,81, 87,13, 60,75,
  80,51, 53,32, 18,53, 95,52, 36,85,  7,83, 91,54, 79, 3,
  42, 3, 37,15, 38,88, 37,17, 43,99, 76,64, 11, 7, 42,68,
  25,67, 21,24, 53, 3, 82,24, 91,16, 50,88,  7,42, 79,97,
  71,32,  4, 8, 91,91, 69,83, 22,39, 77,37,  0,58,  5,16,
   7,57, 22,23, 93,12, 67,76, 59, 8, 80,88,  6,28, 62,70,
  67,95, 39,64, 80,43, 72,20, 25,99, 75,51, 59,95, 52,18,
  45, 7, 94,30, 33,22, 59,89, 46,62, 88,60, 16,71, 90,78,
  90,58, 80,10, 20,96, 76,38, 76,73, 29, 9, 88,77, 75,45,
  71,65, 42,14, 67, 9, 17,67, 43,76, 67,52, 70,38,  6,18,
  52,80, 98,63, 72,27,  1,28, 92,59, 52,47, 92,99, 67,58,
  45, 2, 46,38, 93,99, 29,87, 84,98, 29,22, 96,20, 63,75,
  95,18, 72,24, 77, 8, 11,34, 99,18, 57,75, 71,89, 81,70,
  78,59, 14,50, 37,10, 55,35, 30,20, 33,83, 88,93, 55,63,
  58, 4, 37,64, 63,99, 56,82, 89, 6, 67,93, 16,87, 19,34,
  62,92, 52,24, 66,91, 97, 2, 93,33, 17,60, 19, 8, 18,50,
   7,95, 51,92, 48,84, 81,51, 96,24, 11,33, 84,88, 34,86,
  84,75, 27,49, 56,98, 58,41, 30,63, 90,40, 11,61, 95,56,
   4,48, 98,61,  4, 4, 15,39,  6,88, 32, 6, 46,81, 97,28,
  26,26, 32,57, 38,36, 94,50, 98,35, 56,30,  1, 2, 42,33,
   1,49, 47,88, 54,57, 22,31, 93,13, 93, 7, 28,54,  8,79,
  98,32, 58,99, 61,20,  7,78, 77,18, 23,25, 28,37, 72,52,
  45,73, 29, 3, 66,18, 88,79, 25,43, 74,21, 21,15, 60,35,
  65,50, 48, 0, 34,52, 85,24, 63,96, 40,16, 85,34,  6,48,
  97,17, 13,15, 82,48, 43,40, 86,97, 90,29, 20,89, 63,73,
  20, 6, 20,34,  4,46, 54,70, 23,93, 35,71,  3,18, 77, 5,
  28,78, 47,32,  9,89, 39,72,  2,82, 43,98, 93,80, 50, 7,
  96,40, 24,58, 19,28,  7,25, 55,77, 99,39, 80,40, 96, 5,
  70,54, 81,82, 73, 7, 82,21,  8,98,  4,54, 54,93, 23,81,
  92,15, 74,25, 91, 6, 57,64, 60,91, 93,95, 29,81, 45,34,
  39,70,  2, 1, 78,74, 46,59, 14,11, 78,67, 75,96, 78,96,
   8,60, 50,67, 57, 4, 43,43, 73,22, 91,10, 86,87, 45,75,
  88,14, 36,62, 45,16, 87,33, 98,14, 51,72,  2,80,  2,62,
  63,33, 11,30, 22,86, 41,15, 82, 8, 89,51, 64,31, 19,20,
  88,73, 96,81, 10,65, 78,33,  5,96, 39,23, 90,43, 56,40,
  41,60, 68, 3, 50,26, 71,36, 76, 8, 13,22, 72,76, 21,25,
  61, 4, 46,32, 17,85,  4,79, 19,90, 78,15, 10,70, 87,57,
  74,13, 49,96, 23,50, 92,82, 12,77, 70,27, 31,75, 33,64,
  98,12, 59,99, 37, 1, 14,44, 28,58, 61,33,  5,72, 44,20,
  92,89, 86,80, 83,91, 69,24, 92,74, 24,77, 74,44, 38,58,
  61,86, 96,49, 43,25, 68,79,  9,94, 10,27, 71,87, 14,31,
  30,99, 87,78, 45,13, 41,32, 83,26, 84,85,  1,39, 26, 6,
  39,81, 28,95, 96,85, 89,97, 85,22, 32,47, 41,34,  1,84,
  71,43, 10,48, 19,57, 74,60, 97,95, 54,23, 18,20, 92,95,
  23,23, 61,62, 61, 6, 32,62, 11,37, 98,76, 37,94, 60,32,
  61,55, 27,45, 44,63, 62,81,  0,55, 95,62, 34,40, 15,46,
  11,73,  9,76,  1,59, 58,68, 60,23, 67,38, 28,55, 91,24,
  61,50, 65,91,  4,62,  5,39, 95,87, 37,55, 59,31, 98,36,
  66,33, 83, 3, 85,61, 82,75, 96,16, 14, 3, 82,66, 97, 7,
  34,28, 31,58, 61,31, 13,69, 51,52, 59,66, 49,21, 27, 9,
  51, 7, 79,40,  2,73, 53,75, 78,32, 37,81, 30,42, 51,42,
  95,89, 39,53, 28,79, 30,18, 31,28, 27,35, 66,99, 16,30,
  27, 3,  3,20, 73,92, 79,51, 31,88, 57,25, 85,35,  1,27,
  54,22, 35,72, 63,12, 94,82,  5, 5, 36,48, 18,73, 75,10,
  88,58, 44,59, 34,69, 88,23, 70,64, 99,52, 28,18, 73, 4,
   9,90, 70,26, 52,73, 82,55, 44,31, 45,94, 84,62, 78,63,
  13,43, 73,20, 84,36, 62,42, 88,22, 94,27, 12,34,  0,99,
  95,48, 96,95, 86,32,  6,14, 29, 8,  8,95, 83,23, 89,30,
  82,90, 28,16, 20,24,  9,67, 66,38, 60,10, 43,21,  3,79,
  48,39, 48,80, 18,71, 17,93, 33,77, 93, 8, 11,68, 89,34,
  41,25, 64,44, 78, 1, 51,67, 69, 4, 10, 0, 10,43, 11,96,
  24,53, 20,61, 48,66, 14,10, 91, 3, 15, 3, 54,13, 68,83,
  63,67, 28,32, 46,54, 54,89, 15,48,  4,63, 53,48, 59,29,
  48,71,  6,34, 53,99, 47,92, 57,88, 78,60, 24,68, 48,60,
  64,16, 22,11, 59,82, 55,23, 68,73, 90,27, 45, 5, 89, 0,
  57,28,  8,65, 86,39, 65,95, 87,70, 17,53, 29,71, 43,14,
  98,54, 53,47, 84,32, 63,49, 12,36, 54,99, 80,26, 90,92,
  65, 6, 48,29, 58, 0, 63,16, 42,57, 41, 5, 28,73, 75,85,
   2,16, 34, 5,  0,87, 35,11, 18,93, 95,61, 67,15, 13,10,
  78, 4, 96,39, 10,55,  3,87, 37,46, 31,14, 88,76, 26,22,
  43,23, 81,44, 42,47,  7,52, 87, 9, 91,94, 86,40, 12,80,
  58,74, 81, 5, 95,41, 72,36, 42,18, 57,12,  7,16,  5,56,
  94, 5, 59,27, 90,36, 38,54, 44,98,  7,30, 79,71,  5,88,
  58,67, 23,19, 10,83,  9,39, 44,51, 10,78, 16,85, 76,68,
  53,61, 17,72,  0, 5, 23,32, 92,45, 37, 3, 18,75, 74, 4,
  94,90, 10,92, 11,62, 39,44,  5,73, 99,30, 28,25, 16,93,
  28,89, 25,86, 74,63,  2,52, 27,50,  8, 8, 15,71, 41,96,
  43,89, 31,85, 70,28, 11,28, 57,57, 46,39, 77,33, 89,87,
  34,98,  2,33, 60,81, 52,11, 98,86, 98,69,  7,43,  8, 2,
  86, 2, 88,26, 68,61, 70,39, 98,80, 36,65, 11,15, 11,71,
  95,31, 67, 6, 59,55, 17,12, 54,91, 45,63, 95,43, 10,39,
  69, 2, 15,89, 84,54, 86,75, 89,37, 26,48, 66,88, 11,90,
  55,70, 65, 7, 42,58, 50,33, 32, 9, 66,55, 44,55, 27,78,
  78,47, 13,75, 91,80, 71,83, 53,28, 10,25, 51,33, 37,16,
  70,49, 13,14, 58,97, 66,78, 21,70,  3,67, 52,16, 91,17,
  71,17, 59,20, 64,96,  0,54, 74,43, 43,19, 66,47, 29,70,
  71,96,  1,44, 18,36, 81,61, 88, 1, 79,31, 45,85
};
const int p_100_15[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  383,
  /// Array of preferences
  99,46,  2,61, 97,28, 18,71, 71,99, 82,86, 58,38,  8,86,
  96,48, 30,31, 65,29,  1,53, 33,10, 73,81, 73,56, 25,68,
  50, 6, 64,31, 82,34, 91,18, 66,19, 71,43,  6,13,  7,69,
  70,70, 30, 7, 59, 5, 63,33, 50,48, 88,86, 85,76, 85, 4,
  50,20, 82, 1, 75,95, 37,88, 22,20, 35,66,  4, 1, 74,36,
  27, 2, 91, 6, 12,91, 68,85, 21, 9, 32,60,  9,32, 93,79,
   4,30, 90, 4, 96, 7, 57,34, 95, 2, 13,58, 93,11, 66,46,
   3,32, 42,25,  7,43, 22,91, 80,26, 37,81, 56,33,  6,26,
  37,41,  3,48,  5,54, 54,62, 88,65, 71,45,  3,69, 13,74,
  61,45,  5,74, 58,28, 62,30,  5, 4, 82,90, 33,41, 58, 2,
  32,86, 91,87, 22,97, 62,64,  9,27,  9,81, 19,13, 91,62,
  17,54, 34,75, 31,77, 18,41, 30,99, 21,74, 53,40, 85,66,
  94,24, 59,38, 40,12, 38, 3, 56,69, 21,79,  5,98, 92,28,
  33,61, 89, 0, 12,96, 73,73, 55,71, 33,15, 49,51, 69,24,
  69,63, 76,40, 71,37, 97,64, 41,63, 60,86, 44,45, 99,10,
  61,20, 31, 1, 58, 0, 72,19, 88,11,  6,98, 48,22, 84,79,
  56,98, 22, 2, 61, 1, 25,43, 21,24, 85,90,  1,64, 95,61,
  97,22, 88, 1, 93,80, 17, 6, 67,53,  6,39, 13,13, 57,84,
  73,90, 72,10, 33,43, 43,90, 74,48, 75,72, 96,14, 36,16,
   4,55,  3,22, 27,62, 11,84, 75,93,  7,72,  1,35, 22,42,
  90,52, 81, 0,  8, 6, 27,76, 59,99, 98,81, 72,69, 57,55,
  85,10, 51,11, 47,45, 43,14,  6,70, 15,25, 94, 5,  8, 8,
  64,53,  6, 0, 15,54, 34,87, 75, 8, 24,44, 56,87, 35,88,
  92,90, 57,73, 96,75, 48,88, 55,89, 40,86, 43,20, 39,46,
  86,10, 82,59, 72,94, 29,61, 73,71, 81,56, 37,16, 30,96,
  70,80,  1,57, 46,21, 80,49,  0,20, 55,18, 16,43,  0,99,
  87,43,  6,68,  6,61, 99, 1, 92,49,  2, 4, 54,90, 57,36,
   3,85, 29,51, 65,25, 20,46, 18,93, 28,73, 28,83, 82,75,
  11,17, 10,14, 55,90, 14,20, 32,22, 92,50, 15,47, 10,33,
   0,60, 33,73, 26,14, 18,20, 83,22, 51,65, 30,40, 22,73,
  25,88, 61,19, 74,94, 37, 2, 66,18,  2,63, 61,89, 10,88,
  16,37, 38,85, 98,40, 37,49, 68,14,  6,94, 10, 5,  3,63,
  29,34, 22,89, 80,50, 31, 8, 89,72,  9,71, 16,33, 13,38,
  27,55, 88, 0,  8,49, 82,15, 33, 5, 78,31,  3,15, 11,15,
  39,97, 76,81, 85,82, 26,30, 41,15, 22,48, 65, 8, 95,87,
  55,59, 25,84, 85,92, 82,64, 89,55, 79, 4, 68,69,  7,79,
  70, 8, 41,27, 92,92, 67,72, 46,81, 87,61, 31,91, 74, 9,
  48, 9, 57, 0, 57,90, 21,47, 12,61, 96,16, 77,85, 83,24,
  12,99, 45,64, 82,71, 86,52, 45,30, 33,34, 11,90, 44,68,
  43,62, 52,66, 60,16, 88,25, 77, 6, 31,41, 36,11, 93,33,
  78,53, 66,75, 90,95, 15,24, 42,93, 84,87, 36,67, 77,83,
  47, 5, 19,25, 67,91, 48,60, 51,23, 95,27, 65,20, 46, 4,
  74,97, 30,38, 33,74, 89,17, 30,75, 33,56, 57,80, 91,85,
  70,57, 27,18, 56,20, 30,28, 81,83, 45,39, 94,91, 31, 9,
  91,58, 44,16, 45,76, 78,70, 42,71, 57,52, 75,19, 93,61,
   0,84, 78,19, 50, 8, 98,19, 78,15, 86,35, 33,65,  8,66,
  88,17, 91,63, 55,29, 14,26, 18,49, 51, 7, 17,61, 60,89,
  95,46, 62,61, 18,86, 10,40, 92,57, 86,11, 32,21
};
const int p_100_16[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  950,
  /// Array of preferences
  87,36, 61,56, 26,10, 68, 9, 12,55, 55,13, 83,24, 66,85,
  44,49, 56,91, 94,29, 93,30, 56,59,  5,82, 86,78, 48,52,
  64,35, 78,83, 93,25, 49,23, 53,21, 96,55, 30,56, 72,83,
  38,42, 99,16, 77,28,  6,94, 22,54, 55,57, 41, 2, 55,63,
  55, 1, 54, 4, 56,61, 25,81, 14,90, 87,19, 41,36, 99,79,
  90,95, 72,23, 34,78, 30,22, 69,86, 29,29, 23,64, 25,40,
  68,26, 54, 6, 31,21, 31,46, 48,50, 83,61, 14,94, 21,22,
  69,46, 51,85, 66,61, 27,76, 52,25,  4,60, 79,54, 32,75,
  34,63, 23,11, 41,71, 28,81, 96,61, 90,20,  8,16, 91,19,
   4,17, 18,16, 54,38,  7, 3, 21,76, 74,27, 35,21, 77,66,
  33,88,  9,52, 91,40, 40,57,  9,66, 41,33, 26,46, 42,56,
  49,49, 40,76,  8, 2, 16,71, 80,40, 88,38, 46,75, 51,21,
  40,77, 85,19, 45,99, 37,17, 74, 3, 55,29, 19,37, 87,17,
  29,59, 83,37, 79,35, 13,65, 58,40,  3,26,  6,42, 38,94,
  82,22, 53,55, 51,46, 18,43, 41,32, 95,58, 76,22, 56,92,
  79,59, 26, 0, 21,16, 42,21, 43,44, 48,95, 32,23, 61,67,
  33, 7, 59,19, 87,11, 38,31, 97,91, 25,13, 52,22,  5,29,
  28, 4, 17,53, 68,73, 55,47, 83,80, 79,81,  5,39, 51,30,
  17,45, 35,26, 18,33, 50,98, 86,82, 25,25,  4,74, 22,14,
   1,69, 77,82, 80,39, 44,97, 73, 4, 94,38, 39, 3, 59,66,
  14,60, 72,58, 29,93,  4,25,  7,33, 11,92, 99,92, 50,69,
  22,53, 59,15, 34,96, 51,44, 18,99, 13,56, 38,96, 33,22,
  78,65, 58,99, 35,48, 19,17, 60,70, 71,37, 67,49, 75, 3,
  89,69,  2,71,  5,19, 87, 3, 66,33, 30,49, 62,12, 46, 5,
  18,44, 81, 7, 47,86, 54, 3, 23,59,  6,12, 86,37, 58,64,
  47,49, 85,87, 90,33, 61,82,  1,97, 28,95,  6,10, 24,84,
  48,78, 91, 2, 44,68, 75,12, 29,99, 30,68,  1,57, 14,50,
   7,32, 10, 0, 37,93, 46,97, 49, 3, 52,61, 66,71, 45,35,
  78,84, 83,41, 84,73, 77,48, 38,90, 80,38, 86,10, 33, 2,
  21,40, 98,55, 94,24, 52,30, 16,50, 17,77, 22,40,  3,86,
  24,71, 52,49, 47, 3, 62,34, 48,86,  5,90, 98,98, 76, 0,
  55, 0, 41,35, 35,91, 41,92, 39,87, 41,12, 71,11, 99,23,
  13,88, 53, 5, 76,14, 38, 0, 10,83, 77,35, 49,16,  2,46,
  93,15, 39,54, 75,27, 34,26, 56,33, 98,67, 93, 9, 98,16,
  33,24, 99,31, 35,89, 58,78, 34,60, 95,84, 78,75, 30,10,
  13, 3, 78,61, 36,70, 68,54, 61,53, 31,53, 23, 5, 40,59,
  60,49, 23,74, 15,25, 96,37, 18,20, 20,82, 20,66, 35,20,
  82,16, 71,33, 45, 5, 20,59, 12,63, 76,58, 79,88, 83,51,
  45,40, 58,55, 91,88, 56,68, 85,81, 44,95, 37,98, 21, 3,
  26,85,  8,61, 86,30, 31,63, 91,72, 82,38, 16,83, 35,57,
  91,47, 69,51, 26,66, 79,36, 91,41, 89,99, 40,53, 71,66,
  11, 0, 96,39,  9,26, 22,92,  8,60, 55,42, 79,44, 82,34,
  93,45,  1,89, 92,10,  5, 1, 85,95, 35,86, 81,13, 84,75,
  10,60, 88,60, 29,42, 29,87,  9,69, 73,13, 68,79, 30,77,
  32, 4, 37,27, 40,23, 66,84, 84,88, 94,94, 18,80, 73,94,
  50,64, 20,80, 99,84, 99,30, 83,35,  8,56, 24,24, 84, 3,
   9,44, 30,67, 28,46, 67,89, 30,74, 63,58, 28,18, 10,40,
  57,50, 81,40, 88,62, 11,27, 62,85, 82,99, 72,82, 41,48,
   4,15, 58,17, 16,12, 69,41, 29,37, 23,71, 92,46, 35,49,
  16,15, 83,53, 17,64, 47,67, 35,53, 88,64, 42,43,  9,25,
  72,90, 69,77, 92,43,  1,78, 63,36, 79,78, 60,42, 19,85,
  10,25, 60, 9, 51,97, 73,51, 26,95, 90,43, 72, 4, 77,99,
   5,87, 59,82,  0, 5, 57,80, 87,88, 11,13, 26,84, 54,58,
  93,46, 82,73, 24,11, 82, 9, 99,54, 61,28, 84,98, 76,59,
   2,87, 70,68, 57,83, 12,18, 14,63, 51,45, 81,89, 87,51,
  86,48, 25,21, 87,91, 91,59,  6,36,  8,96, 27,25, 16,14,
  96,77, 55,39, 58,62, 95, 2, 29, 3, 89,41, 48,97, 69,84,
  66,55, 70,85, 64,80, 30,71, 29,74, 21,43, 18,57, 91,23,
  38,59, 42,27, 85,58, 27,78, 96,65, 16,22,  7,91, 69, 5,
  13,87, 86,94, 45,11, 63, 8, 18,58, 89, 5, 34,93, 72, 9,
   2,90, 44,29, 86,42,  8,17,  1, 3, 17,88, 33,17, 65,43,
  85,26, 23,20, 72,34, 65,86, 32,34, 49,81,  3,46,  9,87,
  17,89, 38,22, 32,50, 75,28, 40,48, 47,78, 24,69, 23, 6,
  73,11, 66,92, 12,73, 81,55, 13, 0, 36,34, 64, 6, 41,11,
  71, 4, 70,41, 65,95,  2,59, 48, 7, 88,21, 70,23, 23,48,
  44,15, 53,20, 58,48, 12,28, 77, 0, 97,51, 18,63, 90,72,
  34,39,  3,24, 23,91, 98, 5, 40,97, 38,15, 91,28, 48,13,
  54,90, 96,28, 57, 4, 47, 1, 70,20,  3,57, 62,98, 50,51,
  47,45, 83, 6, 34,55, 96,36, 81,86, 73,58, 23,36, 41,57,
  53,44, 27,84, 88,87, 71,86, 18,40, 47,26, 22,49, 38,98,
  17,92, 47,53, 12,22, 72,86, 16,59, 83,10, 48, 5, 58,91,
   8,38, 26,52, 94,82, 96, 2, 60, 4, 49,10, 57,41, 26,33,
  46,80, 48,40, 83,95, 84,97, 46,70, 76,11, 20,46, 65,50,
  50,82, 30,72, 71,96, 16,54, 10,39, 24,77, 81,49,  1,37,
  43,65, 28,74, 54,87, 97,62, 33,62, 45,95, 42,74, 75,42,
  84,69, 42,28, 42,30,  5,61, 48,24, 34,25, 17, 8, 55,70,
   6,78, 89,85, 42,47,  6,85,  6,21, 78,51, 68,57, 38,95,
  63,67, 10,48, 91,12,  0,71, 52,84,  5,75, 17,31, 43,96,
  45,64, 81,18, 55,34, 53,74, 85, 0, 49,32, 17,57, 13, 5,
  74,98, 70,31, 55,76, 27,37, 70,72, 98,51,  4,90, 10,97,
  62,57, 47, 6, 45,79,  7,49, 62,44, 55,95,  4,80, 63,65,
  24, 0, 51, 5, 85,32, 13,47, 40,33, 67,86, 95,54, 99,58,
  61,32, 22,22, 55,20, 42,34, 67, 3, 97, 1, 29,70, 12,13,
  82,84, 96,72, 34,11,  2,77, 39,65, 55, 7, 64,15, 25,43,
  16,92, 19,72, 75, 5, 57,13, 65,32,  9, 9, 79,66, 76,10,
  28,35, 98,90, 65, 9, 36, 5, 53,18, 46,88, 77,61, 77,75,
  81, 5, 13,67, 52,83, 80,10,  9,45, 40,60, 27,18, 90,17,
  39, 6, 60,71,  8,47, 21,18, 85,45, 96,98, 19,19, 56,46,
  69,18, 82,65, 19, 4,  8,15,  7, 5,  5,12, 73,19, 96,46,
  82,82, 75,55, 25,37, 67,50, 15,14, 62,30, 62,89, 61,57,
   4, 7, 64,88, 35,30, 76,20, 46,81, 64,61, 80,66, 70,43,
  80,56, 98,56,  1,34, 30,54, 33, 0, 64,42, 58,65, 68,80,
  10,56, 74,72, 45, 1, 91,81, 39,76, 68,15, 91,93, 93,56,
  90,29, 10,33, 56,53, 19,52, 69,26, 86,57, 80,34, 49,28,
  49,39, 12,57, 25,12, 23,90, 34,70, 32,20, 54,50, 76,65,
  58,25, 75,19,  6,76, 64,66, 18,26, 19,73, 53,62, 19, 1,
  29,40, 66,60, 40,89, 66,25, 79,29, 99, 8, 93,94, 90,10,
   3, 8, 35,83, 83,55, 85,93, 13,85, 63,24, 54,29,  5,34,
  50,72, 45,81, 31,72,  3,84, 11, 6, 80,62,  7,34,  4,41,
  40,18, 55,67, 77,20, 75,73, 61,38, 86, 7, 18,64, 89,73,
  94,91, 51,48, 12,12, 96,47,  5,24, 94, 3, 83,34,  7,35,
  31,78, 86,74, 66,56, 30,60, 43,34, 89, 1, 91,74, 89,90,
  91,78,  1,43, 76, 9, 42,93, 11,77, 63, 4, 56,16, 43,13,
   8,18, 52,60, 66,21, 71,15, 10, 6, 80,31, 73,22, 94,15,
  32,89, 89,35, 55,91, 15,66, 92,29, 10,80, 28, 6, 58,50,
  99,15, 78,97, 22,57, 92,65, 51,64, 84,58, 74,60,  1,74,
  50,29,  6,31, 69,42, 66, 2, 42,20, 21,13, 28, 5,  0,61,
  31,99, 73,12, 95,29, 68,35, 76,95, 35,67, 80,96, 49,26,
  91,17,  7, 4, 22,95, 70,32, 64,74, 66,31, 86,35, 22,41,
  81,62, 31,11, 12,39, 18, 2, 28,14, 71,78, 96, 8, 75,96,
  98,80,  9, 4,  0,12, 81,39,  9,23, 33,16, 19,99, 74, 4,
  70,55, 25,82, 10,66, 32,93, 60,26, 90,16, 41,13, 72,75,
  61,27, 15,85, 79,62, 94,52, 83,20, 13, 9, 37,51, 24,45,
  11,25, 79,43, 43,92, 98,59, 11,14, 28,25, 37,66, 69,27,
  41,25, 19,62, 98,92, 49,71, 46,86, 15,49, 32,47, 74,56,
  47,63, 28,57, 61,46, 37,58, 10,54, 22,29, 45,50, 44,67,
  77,57, 90,26, 67,42, 34,28, 90,55, 99,44, 74,59, 75,67,
  95,96, 62,51, 96,71, 56,36, 52,21, 25,58, 52,36, 18,71,
  34, 7, 97,13, 61,87, 43,75, 47,47, 17,65
};
const int p_100_17[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  1777,
  /// Array of preferences
  55,23, 79,98, 44,57, 39,34, 73,54, 94,33, 22,59, 53,56,
  81,42,  3,65, 46,92, 64,49, 94,56, 20,13, 45, 1,  7,12,
   3,27, 21,91, 43,83, 18,93, 61,92, 76,63, 33,56, 36,89,
  60,71, 67, 0, 84,81, 38,62, 94,24, 71,48, 45,97, 49,54,
  52,77, 52, 2, 94,78, 30,80, 16,52, 61,57, 47,95, 63,96,
  52,74, 60,73, 20,86, 41,47, 16,46, 27,58, 24,64, 94,34,
  73,19, 76, 2, 14, 3, 78,91, 18,82, 47,21, 40,29, 26,59,
  92,83, 84,97, 93,75, 72, 7, 84,69, 75,59, 65,61, 82,97,
  79,34, 41,62, 37,56, 20,12, 20,75, 53,17, 36,22, 94,70,
  40,28,  3,80, 34,59, 23,92, 71,96, 42,54, 84,44, 21,79,
  43,36,  8,74, 23,51, 87,82, 89,36,  2,40, 25,87, 55,85,
  13,85, 28,18, 14, 0, 33,50, 38,86, 85,95,  3,89, 75,31,
  37,29, 12,77, 51,39, 47,79, 48,11, 79,56, 18,62,  6,10,
   2,32, 51,62, 73,51, 53,46, 34,13, 77,92,  9, 1, 81, 2,
  34,18,  3, 7, 46,38, 43,71,  2,22, 73,96, 11,73, 49,81,
  60,86, 98,91, 88,40, 66,67, 86,13, 86,68, 43,52, 21,28,
  32,79, 62,71, 99,45, 49,99, 92,37, 98,57, 97,47,  4,81,
  54,72, 22,42, 42,22, 12, 6, 16, 9, 27,20, 56,97, 97,40,
  21,50, 55,95, 65,46, 16, 7, 43,31, 54,81, 93, 9, 35,62,
  55,84, 32,57, 29,89, 87,36, 67,45, 28,39, 91,91, 54,27,
  93,46, 84,61, 41,49, 89, 9, 98,42, 16,87, 55,32, 56,67,
  20,41, 14,53, 51,99, 50,48, 27,18, 39,60, 43, 0, 54,16,
   1,57, 10,69, 66,34, 87,99, 84,41, 59,52, 70,67, 34,31,
  46,12, 78,23, 98,11, 64, 0, 13,40, 92,77, 83,17, 51,44,
  45,77, 24,63, 44,41, 37, 2, 34,60, 64,50, 40,76, 82,48,
  72, 1, 68,60,  4,34, 28,41, 24,59, 33,97, 48,27, 95,60,
  66, 3, 21,85, 24,97, 95,81, 16,20, 53,24, 55,63, 21,88,
   4,88, 84,76, 85,99, 16,97, 93,73, 76,82, 74,39, 72, 3,
  39,80, 31,84, 48,70, 99,71,  3,29, 54,99, 41,53, 15,42,
  25,43, 44,13, 87,24, 68,69, 10,81, 14,51, 44,88, 93,41,
   6,93, 85,90, 62,88, 16,93, 69,23, 34,21, 76,70, 67,69,
  10,31, 18,55, 58,52, 17,76, 24,14, 31,49,  9,35, 22, 4,
  97,94, 70,23, 16,69, 77,52, 96,54, 42,94, 45,30, 25,57,
  96, 1, 65,39, 64,19, 22,47, 97,70, 36,25,  0,97, 10,64,
  50, 4, 92,88,  8,24, 20, 1, 90,77, 26, 9,  8, 1, 42,59,
  48, 1, 59,83, 42,20, 58,76, 71,14,  2,82, 59,11, 73,75,
  57,61, 28,17, 41,63, 94,11, 65,30,  2,88,  4,64, 75, 6,
  21,95, 29,48, 30,99, 75,69, 10,18, 31,32, 12,50, 37,61,
  92,27, 27,25, 97,71, 61,29, 46,49, 20,38, 24,46,  2,56,
  53,69, 78,29, 88,62, 45,64, 98,53, 55,74, 31, 6, 52,34,
  71,97,  1,25, 84,42, 34,23, 83,81, 36,63, 45,25, 28,45,
  65,35, 93,14, 31,43, 28,74,  4,73, 23,48, 38,58, 29,77,
   2, 8, 59,70, 35,97, 31,10, 57,73, 99,58, 27,40, 72,67,
  83,87, 80,32,  1,19, 42,29, 35,46, 85,46, 98,61, 63,28,
  79, 3, 36,68, 49,66, 63, 4, 32,99, 54,95, 34,77, 24,54,
  32,14, 19,28, 90,32, 87,41, 34,66, 93,98, 77,51, 98,63,
  81,82,  8,65, 35,12, 18,89, 31,46, 47,17, 27,19, 46,33,
  13,82, 54,65, 79,40, 25,20, 96,38, 45,12, 39,25, 18,97,
  19,45, 13,91,  9,60, 76,47, 28,15, 73,16, 49,50, 95,76,
  41,99, 39,62, 16,99, 80,17, 23,75, 45,43, 31,68, 94,71,
  89,34,  6,47, 73, 9, 92,47, 42,47, 33,96, 32, 3, 60,27,
   3, 2, 43, 9, 46,57,  0,92, 41,76, 64,35, 68,63, 62,52,
  48,72, 26,82, 68,41, 29,96, 50,90, 52,86, 88,90,  1,28,
  72,24, 34,54, 30,51, 27,34, 29,60, 24,81, 37,26, 23, 7,
  79,18, 56,73, 25,42, 34,95, 91,65, 59,29, 32, 4, 58, 7,
  39,99, 47,65, 10,92, 38,65, 46,13, 16,22, 92,57, 18, 6,
  52,35, 98,31, 83,36, 30, 8, 47,82, 10, 4, 70,37, 64,89,
  56,34, 24,51, 41,51, 45,48, 69,47, 46,75, 68,88, 91,42,
  89,43,  2,24, 36, 6, 30, 0, 77,16,  9,97, 32,81, 22,66,
  39,43,  0,33, 97, 2, 59,40, 52, 7, 53,52, 90,68, 16,72,
  93,23, 90,69, 76,16, 38,51, 50,75, 80,48,  9,25, 57,67,
  70, 9, 44,19, 21,37, 75,19, 66,59, 80,53, 38,25, 23,93,
  53,88, 94,35, 97,78, 14,85, 41,69, 75,42,  0,45, 41,89,
  48, 5, 75,73,  0,18, 27,81, 59,19, 37,63, 93,43, 84,75,
  60,90,  7,26, 29,56, 89,77, 97,72, 62,53, 72,15, 45,40,
  74,70, 66, 7, 80,80, 76, 6, 37,89, 16,84, 96, 2, 30,62,
  88,76, 85,89, 78,72, 61,48, 96,92, 50,41, 20,18, 50,59,
   3,46, 42,82,  5, 4, 94,91, 52,94, 51,25, 17,61,  4,76,
  66,85, 33,77,  9,44, 80,59, 84,52,  7,29, 55,45, 61,70,
   0,54, 71,36, 25,19, 92, 5, 94,26, 23,77, 29,43, 98,74,
  95,25, 50,70, 50,23, 35,16, 59,30, 96,19, 85,66,  2,31,
  32,64, 15,37, 95,72, 11,93, 46,91, 36,44, 68,75, 72,34,
  95,47, 35,17, 17,84, 66,98, 37,28, 46,46, 99,87, 93,97,
  28,12,  4,23, 91,35, 44, 5, 17,45, 82,59,  6,34, 22,26,
  71,25, 80,90, 46,97, 59,94, 10, 7, 36,55, 59,36, 58,58,
  42,58,  2,25, 24,74, 81,95, 88,93, 55,71, 18,53, 72,57,
  87,29, 55,60, 82,39, 48,54, 76, 3, 23,83, 42,55, 77,66,
   5,33, 16, 4, 90,12, 16,61,  8,76,  4,66, 35,28, 62,38,
   3,96, 89,89, 34,40,  0, 1, 68,76, 67,61, 63,19,  7,56,
  33,55, 82, 5, 85,21, 27,73, 87,96, 47,11, 38,15,  1,91,
  19,98, 27,95, 51,93, 85,67, 39,26, 76,78, 64,40, 84,96,
  56,28,  9,99, 20,47, 37,79, 47,45, 20,68, 36,40, 69,63,
  32,70, 80, 5, 83,22, 42,84, 96,48, 23,90, 54,56, 14,62,
  33, 4, 61,30, 90,61, 44,46, 45,29, 80,61, 73,62, 18,56,
  78,97, 97,97, 91,20, 99,70, 57,19,  4,53, 29,31, 24,40,
  87,62, 27,85, 90,95, 33,37, 69,69, 86,66, 22,25, 29,18,
   7,49, 45,95, 66,54, 50,11, 20,16, 43,63, 10,77, 25,78,
  50, 5, 39,87, 64,83, 36,24, 15,22, 12,17, 91,27, 70, 0,
  84,68, 66,42, 87,79, 10,39, 75,70, 19, 6, 55,70, 51,49,
  20,50, 28,31,  2,48, 82,99, 39,17,  9,52, 51,70, 14,56,
  70,93,  0,63, 85,42, 68, 4, 70,42, 12,18,  6,79, 90,14,
  87,66, 19,71, 36,67, 18, 1, 96,96, 59,35, 57,75, 47,61,
   0,87, 30,64, 41,28, 80,54, 83,28, 20,87, 22,23, 44,42,
  21,15, 28,19, 87,80, 18,22, 31,58, 41,64, 92, 8, 37,75,
  80,95, 66,68, 64,98, 67,28, 87,11, 70,31, 60,49,  0,14,
  63,51, 66,84, 66,36, 88,26, 84,58, 35,82, 91,46, 92,33,
  53,96, 94,92, 54,96, 42,16, 52,45, 69,51,  2,11, 75, 2,
  33,63, 77,53,  5,55, 95,38, 81,88, 15,89,  2,66, 62,25,
  27, 8, 16,90, 11,25, 72,30, 11,54, 53,32, 13,84, 85,65,
  42,81, 72,42,  2,39, 42,89, 55,25, 93,30, 46,64, 19,39,
  83,73, 81,52, 21,61, 58,25,  1,80, 38,78, 53,53, 98,10,
  73,59, 53,70, 14,42, 20,22, 81,54,  5,70, 43,33, 43,96,
  56, 5, 88, 8, 88,71, 99,41, 95,66, 35,19, 46, 7, 14,47,
  17,69, 82,68, 45,91, 92,36, 89,28, 67, 2, 93,39, 64,71,
  33, 6, 20,83, 40,72,  7,53, 86,38, 23,12, 65,83, 56,80,
  18,26, 65,87, 54,57, 93,48, 97,48, 21,71, 45,92, 37,55,
  50,64, 62,30, 84,54, 78,87, 67,97, 20,71, 32,80, 49,57,
  52, 6, 51,56, 78,30, 34,44, 49,42, 78,89, 93,18, 68,25,
  75,29, 49,24, 22,18,  0,21, 34, 7,  8, 0, 54,38, 12, 9,
  45,76, 59,15, 81,31, 54,12, 49,78, 30,34, 66,14, 68,89,
  73,67, 26,89, 32,35, 28,97, 19,52, 79,57, 96,27, 83,94,
  86,59, 78,41, 10, 5, 99,54, 25,50, 35,99, 37,21, 59,84,
  52,64, 34,86, 94,54, 15,87, 46,28, 26,53, 29,52, 10,55,
   6,72, 71,87, 58,82, 29,86, 91,80, 36,32, 73,35, 56,38,
  69,57, 15,91, 25,99,  9,10, 54, 4, 87, 2, 36, 4, 33,34,
  23, 4, 11,40, 53,40, 78,39, 46, 9, 70,77, 23,56, 89,17,
  47,91, 24,73, 77,15, 49,91, 88,91, 17,66, 12,46, 21,70,
  83, 8, 25,14, 75,98, 92,11, 43,15, 91,28, 11,80, 58,28,
   9,75, 89,14, 78,77, 37,18, 63,55,  9,86, 73,81, 33,88,
  65, 2, 16,27, 35, 6, 89,85, 78,46, 32,17, 50, 8,  6,60,
  35,92, 62,13, 89,44, 56, 7, 37,23, 17,98, 20,10,  6, 3,
  97,58, 98,27, 78,85, 30,50, 75,55, 62,86, 69, 2,  4,36,
  98,46, 58,87, 43,55, 66,86, 20,72,  4,11,  6,59,  0,61,
  41, 2,  3,17, 98,89,  6,20, 17,38, 24,20, 61,73, 99,65,
  63,80, 96,37, 37,72,  2,60, 75,76,  2,36, 50,19, 98,37,
  98,41, 57,16, 78,47, 30,33, 34,74, 67,13, 64,37, 34,75,
  91,32, 72,73, 94,10, 52,43, 22,68,  9,23, 61, 9, 44,93,
  11,10, 54,33, 91,95, 58,86, 47,72, 18,15,  1, 8, 70,75,
  98,20, 81,30, 87,98, 47,37, 51,68, 97, 4, 44,72, 39,16,
  37,47, 79,70, 83,69, 97,35, 24,45, 89,56, 49, 3, 24,89,
  50,45, 94, 6, 62,74, 22,32,  8,83, 35,80, 33,58, 18,88,
  72,45, 73,28,  2,67, 69,31,  2,57, 21,42, 69,62, 69,84,
  63, 3, 74,90, 41,65, 84,38,  3,62, 48,37, 84,60, 80,51,
  35,26, 85,56, 70,33, 71,89, 68,98,  1,83, 46,60, 86,58,
  19,96, 51,50, 72,80, 70,25, 43,45, 21,13, 78,12, 23,72,
  48,51, 60,29,  7,61, 57,22, 36,84, 18,59, 33, 8, 94,31,
  67,94, 53,58, 90,66,  3,74, 84,50, 70,94,  8,62, 39,19,
  33,23,  0,38, 51, 3, 70,74,  5,68, 73,29, 25, 0, 28,82,
  49,70, 68,71, 85,48, 73,55, 64,43, 52,36, 42,91, 95, 7,
  50,98,  6,36, 49,18, 73,34, 73,24, 97,74, 39,18, 50,80,
  42,48, 66,70,  1,10, 52,16,  6, 2, 12,90, 74,13, 37,54,
  81, 5, 12,73, 68,27, 15, 1,  8,68, 14,94, 82,37, 24,92,
  13,11, 42,66, 27,55, 77,61, 40,86, 54,28,  2,98, 73,56,
  30,96, 86,30, 27,35, 20,43, 16,47, 59,71, 33,26, 62,26,
  38,76, 90,30, 72,56, 51,59, 12,84, 77,40, 25,92, 88,31,
  52,99,  2,55, 13,81, 67,63, 21,58, 17,35, 42,15,  8,42,
  29,87, 85,11,  1, 0, 21,60, 53,59, 55,57, 76,87, 72,92,
  61,19, 61,66, 38,72, 30,60, 71,98, 96,41, 32,66, 43,18,
  70,70, 55,16, 77,82, 56,79, 60,77, 40, 8, 22,46, 19,26,
  61,47, 55,92, 24,69, 97,85, 72,82, 73,42,  8,71, 48,78,
  78,70, 64, 5, 27,32, 23,71, 88,21, 17,34, 57,39, 12,11,
  93,63, 54,92, 45,31, 77,35, 11,36, 72,43, 84,35, 60,74,
  86,99,  2,17,  7,93, 44, 8, 69,98, 55,26, 29,93, 40,80,
  30,91, 52,51, 75,90, 65,25, 35,69, 33,32, 26,56, 36,62,
  25,86, 83,64, 65, 1, 46,59, 33,90, 95,15, 68,72, 92,60,
  90,38, 32,12, 13,19, 63,94,  3,15, 72,31, 83,60, 37,32,
  94,68, 24,11,  4,30, 65,70, 81,48, 71,23, 95, 3, 69,38,
  46,52, 33,64, 34,34, 46,37, 70,60, 27,74, 88,95, 99,91,
  90,43, 91,69, 52,83, 42,42, 70,68, 10, 3, 56,25, 85,62,
  81,44, 61, 2, 65,72, 17,81, 50,30, 86,16, 98,64, 46,85,
  26,46, 86,21, 37,78, 55,33, 20,76, 10,49, 61,79, 91,11,
  48, 2, 46,83, 78,34, 36,64, 52,71, 36,60, 23,70, 98,66,
  26,73,  7,36, 80,99, 40,77,  1,44, 57,44, 39,64, 98,98,
  44,90, 81,53, 15,88, 81,83, 84,10, 78,16, 88,75, 60,12,
  50,37, 77,45, 90,71, 78,32, 90, 9, 36,70, 47,27, 84,48,
  76,46, 95,50, 78,92, 16,57, 67,38, 22,73, 40,60, 90,13,
  34,16, 89,95, 57,80, 99,77, 45,33, 99,62, 69, 7, 72,41,
   2,92, 67, 1, 70,52, 49, 1, 55,76, 72,98, 42,61,  4, 2,
  32,74, 67,92, 81,55, 22,64, 99,13, 40,78,  6,82, 24,47,
  67,87, 35,25, 28,66, 99,36, 29,16, 31, 9, 73,58, 13,46,
   5,82, 50,29, 63,16, 26,39, 78,94,  8,35, 26,78,  8,64,
  99,10, 33,21, 13,21, 46,14, 11,19, 92, 2, 79,84, 30,69,
  48,82, 32,61, 67, 5, 12,12,  7,76, 42,33, 52,50, 66,62,
   4,63, 61,86, 74, 5, 72,66,  0,37, 98,68,  1,20, 22,56,
  86,60, 54,34, 90, 1, 41,82, 40,15, 47,77, 27,15, 60,43,
  15,98, 87,55, 19,80, 33,41, 71,19, 13,14, 16, 1, 91,15,
  39,79, 52,78, 52,11, 96,16, 34,45, 85,30, 43,48, 39,22,
  77,49,  9,51, 93,69, 55,82, 82,98, 91,47,  9,74,  6,62,
   5,13, 83,97, 90,36, 42,73, 30,18,  4, 0,  8,80,  9, 7,
  66,16, 49,80, 12,19, 84,70, 35,10, 95,58, 79,29, 93,36,
  58,17, 28,70, 21,86, 67,54, 89,31, 51,40, 62,43, 24,82,
  61,50, 43,10, 45,67, 88,14,  4,47, 27,71, 14,72, 46,11,
  11,87, 80, 7, 40, 2,  8,21, 30,68, 30,53, 54,36, 14,28,
  70,18, 66,96, 59,77, 96,14, 94,63, 12, 1, 86,14, 41,88,
  91,50, 22,11, 52,17, 24,33, 69,24, 90,33,  3,18, 27,98,
  29,74,  1,52, 31,65, 84,34,  8,41,  3, 6, 48,13, 12,65,
  87,72, 89,87, 50,22, 64,55, 63, 6, 32,87, 46,22, 96,94,
  31,64, 67, 7, 14,60, 46,17,  8,54, 69,65, 89,93, 57,46,
  60, 4, 72, 8,  5,30, 50,81, 36,49, 79,96, 27,94,  5,75,
  41,57, 37,16, 76,50, 89,59, 75,27,  8,70, 62,36,  3,57,
  45, 5,  5,79, 40,59,  5,48, 75,35, 88,56, 22,65, 94,66,
  74,54, 17,30, 27,99, 52, 1, 13,95, 23,99,  5,95, 57,45,
  24,44,  4,29, 90,21, 53,94, 54,13, 55,18, 42,95, 95,36,
   9,80, 34, 8, 52, 4, 22,13,  0,41, 73, 5, 85, 0, 12,28,
  22,72, 63,41, 53,98, 94,55, 79,85, 28,67, 41,40, 54,52,
  38,30,  5,16, 35,41, 11, 3, 71,40, 90,90, 35,20, 29,70,
  73,82, 96, 4, 55,20, 96,72, 47,86, 99,96, 32,13, 21,67,
  92,99, 66, 8, 89,58, 15,14, 93,17, 63,50, 52,29, 25,61,
   3,98, 13,10, 89,72, 35,83, 27,37, 63,29,  8, 3, 61,58,
  23,86, 88,11,  5,40, 45,45, 70,69, 39,75,  3,58, 68,19,
  53,77,  1,66, 95,53, 18, 2, 57,38, 37,71, 41,22, 96,29,
  99,53, 83,21,  2,10, 29, 1, 86,18, 54,46, 87,38,  8,59,
  88,70, 57,87,  2,33, 10,88,  2, 5, 79,60, 17,60, 75,25,
  72,46, 26,28, 39,46, 15,45, 13,32, 19, 0, 50,66, 39,27,
  69, 5, 36,34, 83,45, 33,74, 34,65, 84,15, 79,46, 48,95,
  15,96, 52,47, 71,53, 22, 5, 22,30, 88,29, 86,63, 70,81,
  27,68, 89,35, 62,84,  3,81, 99,38, 50,91, 61,41, 26,11,
  47, 9, 16,96, 10,82, 40,56, 97,32, 52,42, 40,20, 89,19,
  18,41, 36,18, 33,80, 99,74, 76,84,  1,64, 93,89, 74,99,
  19,90, 20,52, 73,18, 80,91, 78,19, 22,40, 95,55,  6,91,
  93,13, 30,46, 38,55, 28, 5, 53,19, 49,73, 34,80, 41,13,
  35, 8, 39,67, 76,39,  9,82, 35,11, 64,69,  0,59, 42, 4,
  79,71,  7,40, 55,98, 61,78, 85,49, 66,28, 43,70, 43,74,
  99,17, 89,86, 45,68, 11, 1, 46,88, 56,15, 49,37,  5,80,
  84,30, 61,97, 85,51, 50,76, 79,22, 50,58, 43,66,  8,12,
   8,22, 23,22, 67,14,  9,73, 95,16, 94,16, 48,17, 89,83,
  96,67,  8,67, 99,35, 95,89, 25, 8, 54,60, 50,57, 46,82,
  50,84, 65,78, 40,27, 73,71, 61,96, 52,24, 71,68, 17,78,
  87,27, 78,50, 28,57, 32,77, 10,71, 21,43, 13,50,  3,51,
  63,78, 53,22, 66, 4, 14,14, 27,46, 41,54, 74,12, 89,71,
  34,63, 42,77, 35,48, 78,60, 95,41, 86,15, 28,77, 48,41,
  80,43, 76,57, 17,18, 51,95, 87,34, 98,95, 81,24,  6,80,
  20,34
};
const int p_100_18[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  713,
  /// Array of preferences
  67,78,  3,66, 92,55,  4,78, 49,15, 12,67, 54,93, 17,95,
  70,82, 92,26, 70,35, 55, 1,  1,37, 34,37, 74,97,  5,94,
  89,17, 29,33,  8, 1, 35,81, 77,56, 54,24, 13,20, 70,40,
  87,22, 95,71, 28,63, 60,17, 56,89, 52, 5, 27,22, 79,87,
  65,24, 26,66, 58,99, 30,16, 85,46, 91,92, 89,54, 96,92,
  16,58, 61,10, 81,99,  7,38, 67,62,  5,57, 21,45, 26,32,
  70,30, 13,12, 94,31, 68,43, 42,11, 68,56, 18,51,  5,81,
  78,69, 72,46, 45,45, 40,20,  0,33, 77,97, 94,95, 69, 6,
   9,89,  0,62, 63,26, 83,29, 88,47, 33,43, 84,12, 12,35,
  25,28, 36, 1, 31,48, 24, 8, 49,20, 95,51, 36,40, 42,97,
  94,50, 46, 5, 92,83, 12,32, 49,60, 62,98, 24,78, 66,88,
  81, 3, 47,37, 10,31, 77,48,  9,39, 42,29, 33,70, 65,32,
  18,73, 93,84, 80,66,  8,38, 73,90, 26, 7, 84,61,  1,82,
   0,31,  6,86, 60,18, 65,70, 65, 5, 79,84,  2,35, 97,93,
   1, 9, 40,75, 54,57, 53, 7, 61,90, 43,21,  8,17, 89, 2,
  93,87, 41,95,  5,32, 53,50, 33,65, 74,51, 45,31, 91, 3,
  52,48, 94,11, 28,97, 57,25, 69,80, 97,83, 14,70, 83,13,
  54,60,  5,63, 86,28, 61,61, 30,32, 54,10, 23,97, 96,39,
  45,85, 47,78, 72,41,  6,22, 79,19, 63,13, 29,71, 82,36,
  50,10, 16,99,  1,89, 82,74, 23,98, 80,99, 10,12, 23,13,
  11,93, 10,49, 38, 0, 64,14, 91,88, 49,99, 96,32, 59,18,
  15,26, 87,60, 17,42, 99,12, 48,41, 84,74, 16,54, 88,26,
  16,31, 76,14, 87,73, 73,69,  7,30, 66,82, 68,90, 77,70,
  49,42, 84, 9, 96,54, 40,25, 50,58, 19, 7, 74, 5, 60,35,
  99,47,  5, 1, 37,64, 63,34, 88,95, 81,32, 50,48, 87,76,
  46,71, 32, 0, 41,23, 80,43, 96,24, 80,82, 55,20, 87,51,
  27,52, 19, 5, 26,10, 83,44, 53,43, 81,52, 62,53, 84,15,
  60,93, 37,51, 27,46, 20,10, 89,25, 71,20, 35,68, 62, 1,
   0,15, 15,18, 58,23, 25,43, 48,77, 73,18, 35, 6,  0,52,
  25,24, 75,88, 70,13, 14,74, 57,49, 65,83, 17,92, 76,35,
  41,76, 32,82, 83, 8, 28,37, 80,85, 77,95, 22,29, 52,81,
   1,61, 61,44,  2,78, 25, 9, 88,62, 96,99, 67,35, 21,95,
  73, 4, 84,34, 21,35, 26,55, 98,85, 95,34,  4,61, 71,22,
   8,30, 39,49, 10,75, 44,68, 86,81, 98, 4, 97,75, 25, 6,
  28,48, 14,93, 91, 5, 34,95, 77, 6, 48,17, 62,15, 49,49,
  20,69, 42,72, 98,77, 99,88, 89,63,  7,94, 52, 8, 28,49,
   8,28, 27,72, 38,78, 34,94, 15,59, 42,21, 99,54, 59,65,
  29,48, 31,42,  1,78, 17, 5, 36, 4, 63,36, 69, 5, 92,17,
  77,73, 34, 5, 51,71, 76,29, 83,85, 67,32, 36,84, 79,77,
  65,43, 99,49, 90,18, 81,65, 74,38, 19,41, 36,56,  6,55,
  78,41, 91,43, 48,84, 88,22, 31,95, 73,81, 17,77, 96,70,
  99, 3, 31, 1, 51,88, 77,52, 12,51,  0,92, 81,97,  4,47,
  16,76, 42,20, 24,54, 30,29,  7,32, 90,25, 93,92, 49,35,
  19,43, 10,22, 24,58, 49,65,  6,54,  9,29, 35, 9, 39,91,
  78,64, 28,47, 11,88, 47,67, 86,59, 13,83, 60,12,  6,77,
  78,18, 62,79, 39, 2, 56,98, 86,27, 51,28, 17,98, 24, 4,
  35,66, 29,62, 25,77, 81,36, 48,80, 57,47, 11,38, 44,74,
  10,78, 60,73, 76,68, 84,92, 76,48,  2, 6,  6,80,  9,99,
  26,47, 70,45, 49,64, 42,67, 45,32, 77,20, 44,60, 86,86,
  83,24, 12,30, 91,61, 38, 9, 11,80,  1,35, 71,23, 17,26,
  80,28, 83,46, 61,31, 23,89, 31,18, 96,73, 46,86, 84,81,
  14,51, 11,48, 91,30, 12,88, 12,92, 10,68, 66,71, 38,60,
  50, 2,  1,13, 34,46, 13,19, 24,93, 14,47,  6,92, 20,60,
  81,59, 45,37, 98, 9, 36,57, 82,35, 62,38, 95,95,  6,25,
   6,59, 42,59, 70,19, 38,57, 69,73, 87,70, 18,18, 45,29,
   4,83, 56,77, 24,32, 22,71, 79,82, 24,33, 62,17, 18,72,
  98,31, 42,78, 42,75, 96,85, 90,63, 41,47, 11,61, 80,83,
  56,69, 17,57, 55,25, 21,70, 26, 3, 23,12, 31,88, 70,46,
  39,84, 90,20, 41,14, 20,55, 82,83, 48,68, 49,43, 43,57,
  99,24, 94,92, 84,54, 59,35, 73,65, 76,67, 29,27,  7,34,
  51,22, 79,76, 18,77, 69,40, 70,56, 70,29, 21,14,  6,14,
   0,76, 38,70, 29,12, 71,72, 30,76, 20,29, 66,32,  8,31,
  88,88,  3,69, 40,39, 18,84,  8,65, 25,84, 45,58, 39,87,
  14,39, 34,86, 53,74, 57,44, 75, 9, 40,31, 83,65,  6,13,
  40,36, 77,83, 69,63, 53,94, 99,66, 77,38, 57,10, 31,91,
  57,40, 67, 8, 40,99, 87,16, 93,63, 34, 2, 88,41, 84, 6,
  65,67, 65,85, 10,73, 54,99, 33,20, 58,56, 33,51, 38,92,
  47,91, 69,93, 98,32, 31, 5, 21, 5, 37,75, 27,27, 83,75,
  71,85, 67,99, 42,71, 69,87, 57,61, 16,67, 26,61, 31,97,
  59,90, 78,79,  9,70, 64,67, 71,70, 60,33, 13, 9, 40,18,
  22,84, 43, 9, 26,39, 56,50, 64,10, 64,57, 19,80, 22,74,
  35,94, 44,83, 71,79, 34,74, 13,57, 98,13, 32,89, 56,53,
  34, 7,  3,30, 58,64, 94,93, 24,42, 86,56, 50,99, 59,57,
  39,54, 31,22, 73,89, 73,68, 59,82, 94,41, 65,40,  2,28,
  84,79, 74, 9, 82,48, 97,45,  9,55,  1, 8,  6,90, 36, 0,
  19,35, 25,98, 29, 1, 80,65, 46,73, 75,27, 22,68, 33,55,
  81,12, 33,45, 94, 8, 62,89, 17,35, 39,40, 50,33, 62,83,
  13,31, 72,88, 53,34, 35,98,  8,71, 95,60,  0,19, 62,46,
  39,33, 74, 6, 16,27, 39,12, 27,37, 91,37, 44,67, 62,20,
  69, 2,  7,80, 65,31, 47,56, 76,44, 77,74, 64,51, 77, 9,
  85,35, 77,32, 64,25,  6,65, 88,21, 71, 7, 55,65, 14,31,
  67,34, 43,92, 47,19, 94,94, 86,78, 88,75, 97, 5, 14,53,
  38,97, 41,37, 58,96, 18,69, 16,98, 24,88, 79,99, 39,62,
  55,66, 43,13, 27,78, 44,52, 35,85, 93,42, 95,35, 52, 2,
  63,87, 44,58, 22,49, 36,89, 31,28, 19,22, 74,91, 89,42,
  90,84, 12, 8, 27,40, 34,26, 75,91, 71,27, 81,98, 32,61,
  49,57, 80,20, 38,17, 35, 1, 82,66, 17,76, 48,37, 44,28,
  24,29, 13,41, 50,46, 73,77, 64, 3, 65,91,  8,84, 92, 3,
   3,53, 80,98, 11, 5, 36,78, 52,23, 40,19, 93,10, 82,95,
  89,84
};
const int p_100_19[] = {
  /// Number of people on picture
  100,
  /// Number of preferences
  723,
  /// Array of preferences
  50,57, 46,64, 39,24,  7,70, 86, 0, 66,63, 76,86, 52,35,
  71,35, 89,84, 46,19, 47, 2, 71,94, 18,80, 60,94, 99,61,
  38,37, 97, 7, 32,27, 38,67, 48,64, 90,14, 95,68, 94,50,
   9,96,  7,89, 33,26, 76,70, 32,89, 80,12, 13, 6, 72,21,
  15,92, 54,52, 80, 6,  4,66, 64,23,  7,32, 22,48, 69,45,
  75,92, 20,11,  8,69, 78,67, 25,72, 85,45, 20,48, 15,62,
  76,39, 53,82, 60, 1, 70,45, 59,83, 33,57, 74,78, 90,33,
  88,57, 75,60, 70,19, 63,65, 94,40, 52,82, 98,93, 61, 1,
  87,92, 14,53, 88,65,  3,77, 14, 7, 84, 7, 84,42, 59,24,
  30, 3,  6,13, 36,93, 49,46, 51,85, 22, 6, 86,48, 50,17,
  29, 9, 25,94, 34,91, 40,36, 63, 5, 85,84, 18,15, 16,12,
  70,46, 32,76, 44,58, 11,80, 12,71, 85,10, 38, 0, 35,91,
  28, 5, 35,92, 42,82, 12,73, 11,21, 77,12, 60,35, 30,76,
  40,43, 32,94, 27,24, 75,28, 73,74, 64,36, 93,68, 70,73,
  54,97, 35,27, 81,59, 24,47, 86,12, 90,23, 63,77, 96,74,
  48,93, 99, 1, 16,30, 27,51,  4,58, 72,58, 56,98, 13,75,
  46,92, 76,50, 84,89, 68,68, 37,42, 22,15, 40, 9, 26,32,
  37,67, 93,61, 11,77, 90,17, 49,84, 10,76, 54,14,  4,24,
   6,35, 42,30, 63,13, 99,22, 90,91, 62,13, 76,53, 14,16,
  12,87, 40,11, 96,68, 21,29,  7,74, 75,70, 75,53, 54,86,
  41,96,  2,62,  3,97, 14,26, 73,47, 29,65, 18, 0, 40,48,
  38,43, 59,97, 24,43, 50,66, 75, 7, 63,96,  9,71, 53,28,
  26,70, 83,58, 51,22, 71,19,  3, 1, 83,92, 42,22, 30,66,
  92,30, 80,52, 31,67, 18,46, 27, 5,  0,44, 96,48, 59,71,
  72,18, 18,41, 21,82, 24,29, 68,96, 13,33, 71, 5,  2,33,
  14,76, 87,52, 55, 7, 41,55, 99,41, 78,19, 71,59, 22,39,
  93,17, 14,85, 43,27, 12,84, 26,18, 96,84,  2,21, 23,16,
  58,10, 13, 5,  5,79, 83,33, 56,25, 16,38, 11, 6,  9,42,
  42,16, 36,29, 55,47, 17,87, 79,70, 19,90, 21,13, 47,15,
  66,56, 35,59, 80,70, 93,95, 92,51, 17,57, 56,50, 23,99,
  56,86, 82,47, 28,65, 96,21, 89,75, 62,40, 58, 7, 73,89,
   7,46, 89,86, 74,45, 27,21, 32,59, 19, 5, 60,17, 82,60,
  11,25, 73,41, 60,38, 37,58,  6,84, 66,17, 50,59,  2, 4,
  70,26,  4,34, 36, 3, 31,32, 36,51, 63,63, 61,92,  1,66,
  84,83, 14,86,  4,14, 92,91, 49, 5,  4,47, 65,56, 71,82,
  99, 0, 97,19, 19,88, 24,38, 69,52, 97,96, 34,45, 13,72,
  82,93, 69,63, 83,80, 73,66, 93,82, 87,47, 88,29,  2,12,
  31,16, 98,12,  9,47, 78,29, 67,96, 49, 3, 79,53, 58,90,
  11,72, 41,60, 58,51,  2,32, 99,65, 97,65,  1, 3, 85,64,
  84,71, 80,34, 79, 8, 62,11, 98,20, 79,88, 98,91, 43, 5,
  43,23, 24, 4, 70,20, 65,10, 68,12, 41,23, 81,82, 76,97,
  15,33, 40,42, 66,46, 19,39, 80,41, 69,96, 48,52, 36,54,
  68,77,  8,80, 61,59, 37,95, 76,75, 47,46, 78,42, 25,64,
  71,23, 12, 2, 56,70, 65,89, 70,34, 20,10, 13,35, 47, 3,
  47,53, 66,94, 15,18, 98,64, 90,84, 42, 9, 50, 9, 72,44,
  83,34,  1,90, 17,14, 74,72,  5,83, 65,96,  3,73,  2,97,
  67,47, 47,28, 46,65, 39,69, 43,67, 19,78, 68,95, 64,25,
  73,75, 34,70,  0,64, 79,57, 12,44, 65, 3, 60,34, 21, 5,
  85,66, 80,35, 84,55, 80,76, 14,81, 38,62, 94,58, 59,16,
  18, 7, 88,45, 30, 2,  5,85, 52,53, 72,46, 99,63, 48,84,
  40,34, 67,32, 93, 2, 88,47, 76,28, 40,22, 30,42, 40,65,
  30,58,  3,56, 80,92, 67,68, 42,76,  9,65, 45,38, 35,84,
  19,23, 49,29, 46,14, 83,74, 43,10,  5,33, 62,84, 42,51,
  43, 9, 13,25, 48,76,  4,89, 60,98, 67, 8, 79,61, 83,17,
  23,90,  5,29, 57, 9, 54,72, 71,78, 10,81, 10,36, 73,79,
  42,94, 54,59, 71, 2, 81,83, 26, 3,  7,84, 58,32, 42,53,
  13,59, 46,68, 33,93, 34,68, 77,61,  0,46, 51,73,  7,15,
  90,50, 33, 7, 29,93, 14,49,  0,12, 90, 8, 25,71, 99,23,
  27,73, 16,42, 41,22, 16,37, 70,93, 83,44, 83,83,  6,67,
  12, 5, 92,50, 65, 8, 18,64, 57,79, 43,91, 71,60, 28,17,
  35,18, 40,84, 38,59, 60,24, 85,29, 23,61, 27,44, 87,89,
  26,16, 31,13, 67,88,  9,15, 66, 7, 47,90, 52, 5,  7,99,
  53,88, 15,93, 99,82, 80,97, 34,31, 16,84,  2,24, 80,11,
  94,80, 79,25, 93,99, 39,86, 11, 3,  2,51,  0,98, 57,27,
  22,76,  6,71, 24,10, 84,98,  9,28, 77,46, 53,38,  1, 5,
  29,59, 75,85, 82,63, 68,56,  4,94, 38,22, 64,48,  7, 3,
  40, 0, 55,32, 73,64, 28,77, 23,54, 95,99, 97,13, 71,53,
  91,81, 53,10, 47,97, 99,39, 84,15, 13,79, 90,76,  2,64,
  53,22, 18,43, 48,58, 64,89, 49,41, 80,80, 18,16, 19,10,
   9, 5, 70,13, 89,59, 86, 5, 65,71, 73,16, 58, 3, 36,49,
  27,26, 75,74, 23,85, 80,87,  4,41, 55,81, 12,24, 14,95,
  98,72, 44,93, 99,50, 34, 0, 20,20, 87, 3, 40,79, 87,80,
  51,68, 95,55, 86,85, 71, 3, 59,18,  9,21, 82,48, 33,13,
  52,52, 73,44, 84,22, 17,26, 49,72, 54,13, 95,12, 10,73,
  78,11, 49,39, 99,54, 83,14, 14,79, 31,50, 85,97,  5,78,
  59,70, 21,99,  3,60, 89,40, 74,25, 24,85, 75, 6, 55,93,
   4,12, 12,56, 11,16, 35,24,  3, 5, 26,60, 96,55,  4,23,
  80,63, 64,34, 65,59, 84,67, 86,51, 15,21, 65,13,  4,36,
  11,74, 76,95, 98,38, 49,66, 25,97, 42,26, 55,48, 69,87,
  60,88,  0,51, 76,91, 86,46, 86,57, 11,14, 58,52, 51,80,
  28,92, 69,62, 55,20, 33,21, 51,18, 89, 9, 62,15, 64,20,
  20,61, 45,55,  6,29, 97,73, 57,29, 91,27, 94,18, 98, 3,
  38,50,  3,95, 57,52, 97,90, 67,52, 40,32, 81,70, 79,42,
   1,58, 91,18, 47,78, 14,93, 71,64, 84,86, 59,56, 39,85,
  66,53,  2, 3, 18,26, 22,75, 87,23, 31,68, 23,94, 80,75,
  84,45, 97,40, 63, 1, 95,19, 32,39, 85, 1, 73,26, 77,96,
  14, 5, 30,43,  3,80, 18,68, 34,74, 36,38, 28,71, 16,53,
  19,83, 81,25, 65,36, 82,83, 98,73, 98, 1, 34,75, 35,79,
  52,87, 49,12, 53, 1, 50,95, 36, 6, 25,74, 73,93, 49,74,
  94,24, 51,82, 38,84, 99,51, 43,41, 48,46, 45,30,  9,53,
  98,87, 28,75, 69,27
};

const int* pi[] = {
  &small[0], &large[0],
  &p_10_0[0],   &p_10_1[0],   &p_10_2[0],   &p_10_3[0],   &p_10_4[0],
  &p_10_5[0],   &p_10_6[0],   &p_10_7[0],   &p_10_8[0],   &p_10_9[0],
  &p_10_10[0],  &p_10_11[0],  &p_10_12[0],  &p_10_13[0],  &p_10_14[0],
  &p_10_15[0],  &p_10_16[0],  &p_10_17[0],  &p_10_18[0],  &p_10_19[0],
  &p_20_0[0],   &p_20_1[0],   &p_20_2[0],   &p_20_3[0],   &p_20_4[0],
  &p_20_5[0],   &p_20_6[0],   &p_20_7[0],   &p_20_8[0],   &p_20_9[0],
  &p_20_10[0],  &p_20_11[0],  &p_20_12[0],  &p_20_13[0],  &p_20_14[0],
  &p_20_15[0],  &p_20_16[0],  &p_20_17[0],  &p_20_18[0],  &p_20_19[0],
  &p_30_0[0],   &p_30_1[0],   &p_30_2[0],   &p_30_3[0],   &p_30_4[0],
  &p_30_5[0],   &p_30_6[0],   &p_30_7[0],   &p_30_8[0],   &p_30_9[0],
  &p_30_10[0],  &p_30_11[0],  &p_30_12[0],  &p_30_13[0],  &p_30_14[0],
  &p_30_15[0],  &p_30_16[0],  &p_30_17[0],  &p_30_18[0],  &p_30_19[0],
  &p_40_0[0],   &p_40_1[0],   &p_40_2[0],   &p_40_3[0],   &p_40_4[0],
  &p_40_5[0],   &p_40_6[0],   &p_40_7[0],   &p_40_8[0],   &p_40_9[0],
  &p_40_10[0],  &p_40_11[0],  &p_40_12[0],  &p_40_13[0],  &p_40_14[0],
  &p_40_15[0],  &p_40_16[0],  &p_40_17[0],  &p_40_18[0],  &p_40_19[0],
  &p_50_0[0],   &p_50_1[0],   &p_50_2[0],   &p_50_3[0],   &p_50_4[0],
  &p_50_5[0],   &p_50_6[0],   &p_50_7[0],   &p_50_8[0],   &p_50_9[0],
  &p_50_10[0],  &p_50_11[0],  &p_50_12[0],  &p_50_13[0],  &p_50_14[0],
  &p_50_15[0],  &p_50_16[0],  &p_50_17[0],  &p_50_18[0],  &p_50_19[0],
  &p_60_0[0],   &p_60_1[0],   &p_60_2[0],   &p_60_3[0],   &p_60_4[0],
  &p_60_5[0],   &p_60_6[0],   &p_60_7[0],   &p_60_8[0],   &p_60_9[0],
  &p_60_10[0],  &p_60_11[0],  &p_60_12[0],  &p_60_13[0],  &p_60_14[0],
  &p_60_15[0],  &p_60_16[0],  &p_60_17[0],  &p_60_18[0],  &p_60_19[0],
  &p_70_0[0],   &p_70_1[0],   &p_70_2[0],   &p_70_3[0],   &p_70_4[0],
  &p_70_5[0],   &p_70_6[0],   &p_70_7[0],   &p_70_8[0],   &p_70_9[0],
  &p_70_10[0],  &p_70_11[0],  &p_70_12[0],  &p_70_13[0],  &p_70_14[0],
  &p_70_15[0],  &p_70_16[0],  &p_70_17[0],  &p_70_18[0],  &p_70_19[0],
  &p_80_0[0],   &p_80_1[0],   &p_80_2[0],   &p_80_3[0],   &p_80_4[0],
  &p_80_5[0],   &p_80_6[0],   &p_80_7[0],   &p_80_8[0],   &p_80_9[0],
  &p_80_10[0],  &p_80_11[0],  &p_80_12[0],  &p_80_13[0],  &p_80_14[0],
  &p_80_15[0],  &p_80_16[0],  &p_80_17[0],  &p_80_18[0],  &p_80_19[0],
  &p_90_0[0],   &p_90_1[0],   &p_90_2[0],   &p_90_3[0],   &p_90_4[0],
  &p_90_5[0],   &p_90_6[0],   &p_90_7[0],   &p_90_8[0],   &p_90_9[0],
  &p_90_10[0],  &p_90_11[0],  &p_90_12[0],  &p_90_13[0],  &p_90_14[0],
  &p_90_15[0],  &p_90_16[0],  &p_90_17[0],  &p_90_18[0],  &p_90_19[0],
  &p_100_0[0],  &p_100_1[0],  &p_100_2[0],  &p_100_3[0],  &p_100_4[0],
  &p_100_5[0],  &p_100_6[0],  &p_100_7[0],  &p_100_8[0],  &p_100_9[0],
  &p_100_10[0], &p_100_11[0], &p_100_12[0], &p_100_13[0], &p_100_14[0],
  &p_100_15[0], &p_100_16[0], &p_100_17[0], &p_100_18[0], &p_100_19[0]
};

const unsigned int n_pi = sizeof(pi) / sizeof(int*);

// STATISTICS: example-any
