#                              -*- Mode: Perl -*- 
# Dependency_List.pm --- 
# Author           : Manoj Srivastava ( srivasta@tiamat.datasync.com ) 
# Created On       : Wed Jan 22 09:53:33 1997
# Created On Node  : tiamat.datasync.com
# Last Modified By : Manoj Srivastava
# Last Modified On : Sat May 15 05:24:57 1999
# Last Machine Used: glaurung.green-gryphon.com
# Update Count     : 257
# Status           : Unknown, Use with caution!
# HISTORY          : 
# Description      : 
# 
# 


use strict;
require 5.001;

package Debian::Package::Dependency_List;
use Debian::Package::Dependency;
use Carp;

=head1 NAME

  Debian::Package::Dependency_List - Handle dependencies for packages.

=cut

=head1 SYNOPSIS

      use Debian::Package::Dependency_List;
      
      # initialize the dependency list object using the value of the
      # Pre-Depends, Dependes, Recommends, or Suggests field values.
      $dep_lst = 
  	Debian::Package::Dependency_List->new('string' => "Value");
    
      # Print it back out again
      $dep_lst->print();
    
      # Exercise new and print
      Debian::Package::Dependency_List->test_relations();

=cut


=head1 DESCRIPTION


This module implements a Debian::Package::Dependency_List object. This
object may then be used to perform dependency checks, or to decide on
the correct order to build/install the packages.

=cut

=head2 new

This is the constructor for the package.  It takes a named argument,
I<string>, whose value is the value of the Pre-Depends, Dependes,
Recommends, Suggests, Conflicts, or Replaces field values for the
package (it is expected that one Debian::Package::Dependency_List
object shall be created for each of those fields, if they exist). It
also takes a named argument I<Package>, which is the name of the
package under consideration. The final named argument, I<Type>, 
holds the name of the field.

=cut

sub new {
  my $this = shift;
  my %params = @_;
  my $class = ref($this) || $this;
  my $self = {};
  
  croak("Illegal arguments (Missing string)") unless defined $params{'string'};
  croak("Illegal arguments (Missing Package)") unless defined $params{'Package'};
  croak("Illegal arguments (Missing Type)") unless defined $params{'Type'};
  
  bless $self, $class;
  $self->initialize('string' => $params{'string'},
		    'Package' => $params{'Package'},
		    'Type' => $params{'Type'});
  croak ("Could not initialize dependency List $params{'string'}.")
    unless $self;
  $self->{' _Debug'} = 0;
  return $self;
}

=head2 

Internally, new uses the method B<initialize>, which parses the string
and creates a new Dependency object for each element. And calls add to
add each element to the list. Like new, It takes named arguments,
I<string>, I<Package>, and I<Type>. 

=cut

  my $No_Dups = 0;

sub initialize {
  my $self = shift;
  my %params = @_;
  my @packages = ();
  my $i;
  
  croak("Illegal arguments") unless defined $params{'string'};
  croak("Illegal arguments") unless defined $params{'Package'};
  croak("Illegal arguments") unless defined $params{'Type'};
  if ($params{'string'} =~ /,/o) {
    @packages = split /,/, $params{'string'};
  }
  else {
    push (@packages, $params{'string'});
  }
  for (@packages) {
    $self->add('string'  => $_,
	       'Package' => $params{'Package'},
	       'Type'    => $params{'Type'});
    
  }
  $self->{' _Type'} = $params{'Type'};
}

=head2 add

This method adds a new dependency to the list. It also handles
alternates as a simple linked list branching off from the associative
array which is the Dependency_list object. Like new, It takes named
arguments, I<string>, I<Package>, and I<Type>, except in this case the
string represents a fragment of the dependency line corresponding to
each element (it represents the text in between commas)

=cut


sub add {
  my $self = shift;
  my %params = @_;
  my $i;
  
  croak("Illegal arguments") unless defined $params{'string'};
  croak("Illegal arguments") unless defined $params{'Package'};
  croak("Illegal arguments") unless defined $params{'Type'};
  $i = $params{'string'};
  

  if ($i =~ /\|/) {
    my @alternates;
    my $var;
    my $j;
    
    @alternates = split /\|/, $i;
    #print STDERR "DEBUG:(dependency_list.pm):154: recording alternates $i\n";
    
    # the main alternative
    $j = shift(@alternates);
    #print STDERR "DEBUG:(dependency_list.pm):159: recording $j\n" ;
    $var = Debian::Package::Dependency->new('string' => $j,
					    'Type' => $params{'Type'});
    
    if ($No_Dups && defined $self->{$var->{'Name'}}) {
      carp("Duplicate dependency:(1) $params{'Package'}:$var");
      print " $params{'Package'} \n";
      $self->{$var->{'Name'}}->print();
      print "\n";
      $var->print();
      print "\n";
    }
    
    $self->{$var->{'Name'}} = $var;
    
    foreach $j (@alternates ) {
      my $alt;
      
      #print STDERR "DEBUG:(dependency_list.pm):159: recording $j\n" ;
      $alt = Debian::Package::Dependency->new('string' => $j,
					      'Type' => $params{'Type'});
      if ($No_Dups && defined $self->{$alt}) {
	carp("Duplicate dependency:(2) $params{'Package'}:$alt");
	print " $params{'Package'} \n";
	$self->{$alt}->print();
	print "\n";
	$var->print() if defined $self->{$alt};
	print "\n" if defined $self->{$alt};
      }
      
      $var->{'Alt'} = $alt;
      $var = $alt;
    }
  }
  else {
    my $var;
    $var = Debian::Package::Dependency->new('string' => $i,
					    'Type' => $params{'Type'});
    if ($No_Dups && defined $self->{$var->{'Name'}}) {
      carp("Duplicate dependency:(3) $params{'Package'}:($var->{'Name'})");
      print " $params{'Package'} \n";
      $self->{$var->{'Name'}}->print();
      print "\n";
      $var->print();
      print "\n";
    }
    $self->{$var->{'Name'}} = $var;
  }
  
  
}

=head2 delete

This routine take a required named argument Name and deletes a package
with that name from the list if it exists. It also take care of
erasing the package from the provides list, if indeed this package
provided a virtual package.

=cut

=head2 delete

This routine take a required named argument Name and deletes a dependency
with that name from the list if it exists. 

=cut

sub delete {
  my $self = shift;
  my %params = @_;
  
  croak("Need Name") unless defined $params{'Name'};
  
  return unless defined $self->{$params{'Name'}};
  
  delete $self->{$params{'Name'}};
}

=head2 print

This method takes the internal representation of a dependency list
(complete with alternatives) and produces a string, which should be
the string used to initialize the object.

=cut


sub print {
  my $self = shift;
  my $key;
  my @keys = @_ ? @_ : sort keys %$self;
  my $docomma = 0;
  
  foreach $key (@keys) {
    my $var;
    next if $key =~ m/^ _/o;	# ignore internal variables
    
    print ", " if $docomma++;
    $self->{$key}->print();
    $var = $self->{$key};
    while (defined $var->{'Alt'}) {
      print " | ";
      $var->{'Alt'}->print();
      $var = $var->{'Alt'};
    }
  }
  print "\n";
}


=head2 match

This routine takes a dependency object as an argument, and returns
'Yes' if that dependency object is identical to  a member of the
dependency list.

=cut

sub match {
  my $self = shift;
  my $dependency = shift;
  my $ret = 'No';
  my @keys = sort keys %$self;
  my $key;
 
  croak("Required parameter Package absent") unless $dependency;
  
  foreach $key (@keys) {
    my $dep;
    next if $key =~ m/^ _/o;	# ignore internal variables
    
    $dep = $self->{$key};
    $ret = $dep->match($dependency);
    last if $ret eq 'Yes';
  }
  return $ret;
}



=head2 depend 

This method performs dependency checks for all elements of the
dependency list. It takes required named argument I<Package>, which is
a pointer to the package to which this dependency list belongs.

It also takes optional named argument I<Conflicts> (true if this is a
conflicts list).

It cycles through all dependencies, calling the depend method of each
dependency object, handling alternates as it goes.

=cut


sub depend {
  my $self = shift;
  my $params = shift;		# I know, this is unusual
  my @keys = sort keys %$self;
  my $key;
  my @badreports = ();
  my $resolved = 0;
  my $mytype = $self->{' _Type'};
  my $report = '';
  
  croak("Required parameter 'Package' absent") unless
    $params->{'Package'}->{'Package'}; 
  croak("Required parameter New package list absent") 
    unless $params->{'New'};
  
  my $results = $params->{'Package'}->{' _Results'};
  
  foreach $key (@keys) {
    my $dep;
    my $ret = 'Unknown';
    next if $key =~ m/^ _/o;	# ignore internal variables

    $dep = $self->{$key};
    $ret = $dep->depend($params);
    print STDERR "DEBUG:(dependency_list.pm):333\n\t " . 
      $params->{'Package'}->{'Package'} . 
      "($dep->{'Name'}):= $mytype returned:($ret):\n" 
	if $self->{' _Debug'};

    # well, see if there was no resolution
    $resolved =  ($ret ne 'Unknown');

    if (defined $dep->{'Alt'} &&  !$resolved) {
      # we have alternates and things are not yet resolved (maybe
      # unknown?) 
      $report = "$dep->{'Name'}";
      $report .= "($dep->{'Rel'} $dep->{'Ver'})" if $dep->{'Rel'};
      $report .= "\n";    
      print STDERR "\tDEBUG:(dependency_list.pm):350 report:$report"
	if $self->{' _Debug'};

      push (@badreports, $report);
      while (defined $dep->{'Alt'} &&  !$resolved ) {  
	print STDERR "\tDEBUG:(dependency_list.pm): 356: alternates\n"
	   if $self->{' _Debug'};
	$ret = $dep->{'Alt'}->depend($params);
	$resolved = 
	    $results->check('Type' => $mytype, 'Category' => 'Failed') 
	  + $results->check('Type' => $mytype, 'Category' => 'Found') 
	  + $results->check('Type' => $mytype, 'Category' => 'Conflict');

	$dep = $dep->{'Alt'};
	$report = "$dep->{'Name'}";
	$report .= "($dep->{'Rel'} $dep->{'Ver'})" if $dep->{'Rel'};
	$report .= "\n"; 
	push (@badreports, $report);
	print STDERR "\tDEBUG:(dependency_list.pm):368 report:$report"
	  if $self->{' _Debug'};

      }
    }
    # Remove all the bad warnings we have created while examining the
    # alternates 
    if ($resolved && $#badreports) {
      print STDERR "DEBUG:(dependency_list.pm): 377: " . 
      $params->{'Package'}->{'Package'} . 
      " resolved\n\n"
	if $self->{' _Debug'};
      my $category = "Unknown";
      if ($params->{'Warn'}) {
	$category = "Warn";
      }
      for (@badreports) {
	print STDERR "DEBUG:(dependency_list.pm): 390 ",
	"removing [$mytype $category] report ($_)\n" if $self->{' _Debug'};
	$params->{'Package'}->{' _Results'}->remove('Type' => $mytype,
						    'Category' => "$category",
						    'Report' => $_);
      }
    }
  }

  if ($self->{' _Debug'}) {
    print STDERR "\nDEBUG: (dependency_list.pm):395 " ,
    $params->{'Package'}->{'Package'} , "($mytype)",
    " Failures=", 
    $results->check('Type' => $mytype, 'Category' => 'Failed'),
    " Conflicts=", 
    $results->check('Type' => $mytype, 'Category' => 'Conflict'),
    " Unknown=", 
    $results->check('Type' => $mytype, 'Category' => 'Unknown'),
    "\n\n";
  }

  return $results->check('Type' => $mytype, 'Category' => 'Failed') 
       + $results->check('Type' => $mytype, 'Category' => 'Conflict') 
       + $results->check('Type' => $mytype, 'Category' => 'Unknown');
}

=head2 order 

This method takes a named argument, New, which is a pointer to the new
packages list, and a named parameter Package, which is a pointer to
the package this dependency list belongs to (the results are appended
to the package object).

=cut

sub order {
  my $self = shift;
  my $params = shift;		# I know, this is unusual
  my @keys = sort keys %$self;
  my $key;

  
  croak("Required parameter Package absent") unless
    $params->{'Package'}->{'Package'}; 
  croak("Required parameter New package list absent") 
    unless $params->{'New'};
  
  print STDERR "\tDEBUG Dep_List: 434: Type=$self->{' _Type'}\n"
    if $self->{' _Debug'};
  
  
  my $results = $params->{'Package'}->{' _Results'};
  my $errors = 
    $results->check('Type' => $self->{' _Type'},
		    'Category' => 'Failed')     + 
    $results->check('Type' => $self->{' _Type'},
		    'Category' => 'Conflict')   + 
    $results->check('Type' => $self->{' _Type'},
		    'Category' => 'Unknown')    ;
  print STDERR "\tDEBUG Dep_List: 446: errors = $errors\n"
    if $self->{' _Debug'};
    		
  if (($errors) > 0) {
    print STDERR 
      "  DEBUG: DLIST:451 Not including $params->{'Package'}->{'Package'} $errors Errors\n"
	if $self->{' _Debug'};
    return $errors;
  }
  print STDERR 
      "   DEBUG: DLIST:453 Including $params->{'Package'}->{'Package'} $errors Errors\n"
	if $self->{' _Debug'};
  foreach $key (@keys) {
    my $dep;
    my $ret;
    next if $key =~ m/^ _/o;	# ignore internal variables
    
    $dep = $self->{$key};
    $ret = $dep->order($params);    
  }
}

=head2 test

A class  method has been provided to test the Dependency_list class,
and the undelying class B<Debian::Package::Dependency>. The following
test set of dependency strings are used

=over 4

=item "mfbin"

=item "amsfonts, latex"

=item "libc5 (>= 5.4.7-7)"

=item "libc5 (>= 5.4.7-7), libgdbm1, netstd"

=item "libc5 (>= 5.4.7-7), xlib6 (>= 3.2-0), netstd"

=back

=cut


my @test_relationships = 
  (
   "mfbin",
   "amsfonts, latex",
   "libc5 (>= 5.4.7-7)",
   "libc5 (>= 5.4.7-7), libgdbm1, netstd",
   "libc5 (>= 5.4.7-7), xlib6 (>= 3.2-0), netstd"
  );

sub test {
  my $dependency;
  my $i;

  my @test_relationships = 
    (
     "mfbin",
     "amsfonts, latex",
     "libc5 (>= 5.4.7-7)",
     "libc5 (>= 5.4.7-7), libgdbm1, netstd",
     "libc5 (>= 5.4.7-7), xlib6 (>= 3.2-0), netstd"
    );

  
  foreach $i (@test_relationships) {
    $dependency = 
      Debian::Package::Dependency_List->new('string' => $i,
					    'Type' => 'Depends',
					   'Package' => 'Dummy');
    $dependency->print();
  }
}



=head1 NOTES

=cut

=head1 CAVEATS

This is very inchoate, at the moment, and needs testing.

=cut

=head1 BUGS

None Known so far.

=cut

=head1 AUTHOR

Manoj Srivastava <srivasta@debian.org>

=cut


1;

