/*-
# MOTIF/X-BASED TRIANGLES
#
#  xtriangles.c
#
###
#
#  Copyright (c) 1993 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/01 Xt/Motif
  Version 4: 94/06/07 Xt
  Version 3: 93/04/01 Motif
  Version 2: 92/01/06 XView
  Version 1: 91/09/05 SunView
*/

#include "file.h"
#ifdef WINVER
#include "TrianglesP.h"
#include "wtriangles.h"
#define TITLE "wtriangles"

static TrianglesRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->triangles.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Triangles.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/triangles.xpm"
#endif
#include "icons/triangles.xbm"
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Triangles Version 7.2.4\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-tile {color}] [-{border|bd} {color}] [-delay msecs]\n"
"[-[no]sound] [-bumpSound {filename}]\n"
"[-{font|fn} {fontname}] [-size {int}] [-base {int}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"A sliding tile puzzle.  A triangular version of the "
"15 puzzle which was originally made around 1879 (the 14-15\n"
"puzzle of Sam Loyd was made soon after).   T. R. Dawson "
"in Original Puzzles has a similar puzzle with 4 diamond-\n"
"and 6 triangular-shaped pieces.  This puzzle is easier to "
"solve than 15 puzzle.\n"
};

static const char featuresHelp[] = {
"Click \"mouse-left\" button to move a tile.  Clicks on "
"spaces, or clicks on tiles that are not in line with\n"
"two spaces will not move tiles.  Notice, one can move "
"more than  one tile at a time  by clicking on\n"
"the tile farthest from the spaces but still in line with the "
"spaces.\n"
"\n"
"Click \"mouse-right\" button, or press \"C\" or \"c\" "
"keys, to clear the puzzle.\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"Z\" or \"z\" keys to randomize the puzzle.  New records "
"can only be set from a cleared state.\n"
"\n"
"\"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"I\" or \"i\" keys to increase the number of tiles.\n"
"\n"
"Press \"O\" or \"o\" keys to decrease the number of tiles.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Triangles as:\n"
"7   9  Upper Left, Upper Right\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"1   3  Lower Left, Lower Right\n"
};

static const char referencesHelp[] = {
"L. E. Horden, Sliding Piece Puzzles  (Recreations in "
"Mathematics Series), Oxford University Press  1986,\n"
"pp 1, 148, 149.\n"
"Jerry Slocum & Jack Botermans, Puzzles Old & New (How to Make "
"and Solve Them),  University of Washington\n"
"Press, Seattle 1987, p 126, 127.\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "triangles.scores"
#endif

#define MAXTILES 16
#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord trianglesRecord[MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
#ifdef ANIMATE
static Widget speed;
#endif
static Widget moves, record, message, tile;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, clearDialog;
static Arg arg[3];
#else
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
static Boolean randomized = False;
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap icon = None;
static Widget topLevel, triangles;
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int i;

	(void) printf("TRIANGLES  HALL OF FAME\n\n");
	(void) printf(" TILES USER            MOVES\n");
	for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
		if (trianglesRecord[i].score > 0)
			(void) printf("%6d %-12s%9d\n",
				i + 1,
				trianglesRecord[i].name,
				trianglesRecord[i].score);
	}
}
#endif

static void
InitRecords(void)
{
	int i;

	for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
		trianglesRecord[i].score = NEVER;
		(void) strncpy(trianglesRecord[i].name, NOACCESS, USERNAMELEN);
	}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, n;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
		(void) fscanf(fp, "%d %s\n", &n, userName);
		if (n <= trianglesRecord[i].score ||
				trianglesRecord[i].score <= NEVER) {
			trianglesRecord[i].score = n;
			(void) strncpy(trianglesRecord[i].name, userName,
				USERNAMELEN);
		}
	}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (i = 0; i < MAXTILES - MINTILES + 1; i++)
			(void) fprintf(fp, "%d %s\n",
				trianglesRecord[i].score,
				trianglesRecord[i].name);
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(int size)
{
	int i = size - MINTILES;

	if (size > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	} else if (trianglesRecord[i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			trianglesRecord[i].score, trianglesRecord[i].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int size
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	(void) sprintf(titleDsp, "%s: %d@ (%d/%s) - %s",
		progDsp, size, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(triangles), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, int size)
{
	int i = size - MINTILES;

	if (size <= MAXTILES && (counter < trianglesRecord[i].score ||
			trianglesRecord[i].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		trianglesRecord[i].score = counter;
		(void) strncpy(trianglesRecord[i].name, userNameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(size);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
TrianglesWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int size;
	char *userName, *scoreFile;

#ifdef WINVER
	InitializePuzzle(w, brush);

	size = w->triangles.size;
	userName = w->triangles.userName;
	scoreFile = w->triangles.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNsize, &size,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	if (size > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, size, NULL);
	XmScaleSetValue(tile, size);
#ifdef ANIMATE
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(size);
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, size
#endif
		);
}

#ifdef WINVER
void
setPuzzle(TrianglesWidget w, int reason)
#else
static void
CallbackPuzzle(Widget w, caddr_t clientData,
		trianglesCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int size;
	Boolean cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	size = w->triangles.size;
	cheat = w->triangles.cheat;
#else
	XtVaGetValues(w,
		XtNsize, &size,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case PUZZLE_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case PUZZLE_CLEAR_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(clearDialog);
#else
			XtVaSetValues(triangles, XtNmenu, MENU_CLEAR, NULL);
#endif
			break;
#endif
		case PUZZLE_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case PUZZLE_RESTORE:
		case PUZZLE_RESET:
			movesDsp = 0;
			randomized = False;
			break;
		case PUZZLE_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case PUZZLE_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "Spaces can not slide",
				MESSAGELEN);
#endif
			break;
		case PUZZLE_IGNORE:
			(void) strncpy(messageDsp, "Clear to start",
				MESSAGELEN);
			break;
		case PUZZLE_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case PUZZLE_SOLVED:
			if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", userNameDsp);
			else if (!randomized && HandleSolved(movesDsp, size))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", userNameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case PUZZLE_RANDOMIZE:
			movesDsp = 0;
			randomized = True;
			SET_STARTED(w, False);
			break;
		case PUZZLE_DEC:
			movesDsp = 0;
			size--;
			PrintRecord(size);
#ifdef WINVER
			w->triangles.size = size;
#else
			XtVaSetValues(w, XtNsize, size, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tile, size);
			if (size >= MAXTILES)
				XtVaSetValues(tile, XmNmaximum, size, NULL);
#endif
#endif
			break;
		case PUZZLE_INC:
			movesDsp = 0;
			size++;
			PrintRecord(size);
#ifdef WINVER
			w->triangles.size = size;
#else
			XtVaSetValues(w, XtNsize, size, NULL);
#ifdef HAVE_MOTIF
			if (size > MAXTILES)
				XtVaSetValues(tile, XmNmaximum, size, NULL);
			XmScaleSetValue(tile, size);
#endif
#endif
			break;
		case PUZZLE_COMPUTED:
			SET_STARTED(w, False);
			break;
		case PUZZLE_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
		case PUZZLE_REDO:
			movesDsp++;
			SET_STARTED(w, True);
			break;
#if defined(HAVE_MOTIF) && defined(ANIMATE)
		case PUZZLE_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(triangles,
					XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtVaSetValues(w,
						XtNdelay, delay, NULL);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, size
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.triangles.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			EnterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.triangles.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			LeavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyPuzzle(&widget, brush);
			break;
		case WM_SIZE:
			ResizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ExposePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ClearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			SelectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ReleasePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetPuzzle(&widget);
					ResizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WritePuzzle(&widget);
					break;
				case IDM_EXIT:
					DestroyPuzzle(&widget, brush);
					break;
				case IDM_HIDE:
					HidePuzzle(&widget);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					UndoPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_REDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					RedoPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					ClearPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					RandomizePuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					SolvePuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SPEED:
					SpeedPuzzle(&widget);
					break;
				case IDM_SLOW:
					SlowPuzzle(&widget);
					break;
				case IDM_SOUND:
					SoundPuzzle(&widget);
					break;
				case IDM_TR:
				case IDM_RIGHT:
				case IDM_BR:
				case IDM_BL:
				case IDM_LEFT:
				case IDM_TL:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					(void) MovePuzzle(&widget, (int) LOWORD(wParam) - IDM_TR);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_DEC:
					DecrementPuzzle(&widget);
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_INC:
					IncrementPuzzle(&widget);
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*triangles.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*triangles.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*triangles.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*triangles.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*triangles.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*triangles.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*triangles.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-install", (char *) "*triangles.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*triangles.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*triangles.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*triangles.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*triangles.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*triangles.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*triangles.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*triangles.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*triangles.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*triangles.font", XrmoptionSepArg, NULL},
	{(char *) "-size", (char *) "*triangles.size", XrmoptionSepArg, NULL},
	{(char *) "-userName", (char *) "*triangles.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*triangles.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*triangles.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*triangles.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackPuzzleClear(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(triangles, XtNmenu, MENU_CLEAR, NULL);
	}
}

static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int size = cbs->value, old;

	XtVaGetValues(triangles,
		XtNsize, &old, NULL);
	if (old == size)
		return;
	XtVaSetValues(triangles,
		XtNsize, size, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(size);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

#ifdef ANIMATE
static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(triangles,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(triangles, XtNdelay, delay, NULL);
	}
}
#endif

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == MENU_EXIT)
		exit(0);
	XtVaSetValues(triangles, XtNmenu, val, NULL); /* GWQ */
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	XtVaSetValues(triangles, XtNmenu, val + MENU_EXIT + 1, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, tilesRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, solveString;
	XmString incrementString, decrementString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Triangles",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) triangles_xpm, &icon, NULL,
			&xpmAttributes);
	}
	if (icon == (Pixmap) NULL)
#endif
		icon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) triangles_bits,
			triangles_width, triangles_height);
#ifdef HAVE_MOTIF
	XtVaSetValues(topLevel,
		XtNiconPixmap, icon,
		XmNkeyboardFocusPolicy, XmPOINTER, NULL); /* not XmEXPLICIT */
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	randomizeString = XmStringCreateSimple((char *) "RandomiZe");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'Z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	clearDialog = createQuery(topLevel,
		(char *) "Are you sure you want to start over?",
		(char *) "Clear Query",
		(XtCallbackProc) CallbackPuzzleClear);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Clear", 6, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size", 5,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
#ifdef ANIMATE
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
#endif
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Triangles! (use mouse or keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	triangles = XtCreateManagedWidget("triangles",
		trianglesWidgetClass, mainPanel, NULL, 0);
#else
	XtVaSetValues(topLevel,
		XtNiconPixmap, icon,
		XtNinput, True, NULL);
	triangles = XtCreateManagedWidget("triangles",
		trianglesWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(triangles, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
	Initialize(triangles);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(triangles), (unsigned int) AnyButton, AnyModifier,
		XtWindow(triangles), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(triangles),
		XCreateFontCursor(XtDisplay(triangles), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
