/* ========================================================================== */
/* === UMF_solve ============================================================ */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* UMFPACK Version 3.2 (Jan. 1, 2002), Copyright (c) 2002 by Timothy A.       */
/* Davis, University of Florida, davis@cise.ufl.edu.  All Rights Reserved.    */
/* See README, umfpack.h, or type "umfpack_details" in Matlab for License.    */
/* -------------------------------------------------------------------------- */

/*
    Not user-callable.  Solves a linear system using the numerical factorization
    computed by UMFPACK_numeric.  No workspace is dynamically allocated.  Counts
    flops, but does not include ABS ( ) and MAX ( ) in flop count.

    Returns UMFPACK_OK if successful, UMFPACK_ERROR_argument_missing if
    required arguments are missing, UMFPACK_ERROR_invalid_system if the sys
    string is not valid.
*/

#include "umf_internal.h"
#include "umf_lsolve.h"
#include "umf_usolve.h"
#include "umf_ltsolve.h"
#include "umf_utsolve.h"

PRIVATE Int iterative_step
(
    double omega [3],
    Int step,
    const double B [ ],
    double X [ ],
    const double W [ ],
    const double Y [ ],
    const double Z [ ],
    double S [ ],
    Int n,
    double Info [UMFPACK_INFO]
) ;

/* ========================================================================== */
/* === UMF_solve ============================================================ */
/* ========================================================================== */

GLOBAL Int UMF_solve
(
    const char sys [ ],
    const Int Ap [ ],
    const Int Ai [ ],
    const double Ax [ ],
    double X [ ],
    const double B [ ],
    NumericType *Numeric,
    Int irstep,
    double Info [UMFPACK_INFO],
    Int Pattern [ ],
    double W [ ],
    double Y [ ],
    double Z [ ],
    double S [ ]
)
{
    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    Int *Rperm, *Cperm, i, n, p, step, j, nz ;
    double omega [3], axx, wi, xj, yi, zi, xi ;

    /* ---------------------------------------------------------------------- */
    /* initializations */
    /* ---------------------------------------------------------------------- */

    n = Numeric->n ;
    nz = 0 ;
    omega [0] = 0 ;
    omega [1] = 0 ;
    omega [2] = 0 ;
    Rperm = Numeric->Rperm ;
    Cperm = Numeric->Cperm ;
    Info [UMFPACK_SOLVE_FLOPS] = 0 ;

#ifndef NDEBUG
    UMF_dump_lu (Numeric) ;
#endif

    /* ---------------------------------------------------------------------- */
    /* determine which system to solve */
    /* ---------------------------------------------------------------------- */

    if (STRING_MATCH (sys, "Ax=b"))
    {

	/* ------------------------------------------------------------------ */
	/* solve Ax=b with optional iterative refinement */
	/* ------------------------------------------------------------------ */

	if (irstep > 0)
	{

	    /* -------------------------------------------------------------- */
	    /* using iterative refinement:  check workspace and compute Y */
	    /* -------------------------------------------------------------- */

	    if (!Ap || !Ai || !Ax || !Y || !Z || !S)
	    {
		return (UMFPACK_ERROR_argument_missing) ;
	    }

	    nz = Ap [n] ;
	    Info [UMFPACK_NZ] = nz ;

	    /* A is stored by column */
	    /* Y (i) = ||A_i||, 1-norm of row i of A */
	    for (i = 0 ; i < n ; i++)
	    {
		Y [i] = 0 ;
	    }
	    Info [UMFPACK_SOLVE_FLOPS] += nz ;
	    for (j = 0 ; j < n ; j++)
	    {
		for (p = Ap [j] ; p < Ap [j+1] ; p++)
		{
		    Y [Ai [p]] += ABS (Ax [p]) ;
		}
	    }
	}

	for (step = 0 ; step <= irstep ; step++)
	{

	    /* -------------------------------------------------------------- */
	    /* Solve Ax=b (step 0): */
	    /*  x = Q  (U \ (L \ (Pb))) */
	    /* and then perform iterative refinement (step > 0): */
	    /*  x = x + Q (U \ (L \ (P (b-Ax)))) */
	    /* -------------------------------------------------------------- */

	    if (step == 0)
	    {
		for (i = 0 ; i < n ; i++)
		{
		    W [i] = B [Rperm [i]] ;
		}
	    }
	    else
	    {
		for (i = 0 ; i < n ; i++)
		{
		    Z [i] = B [i] ;
		}
		Info [UMFPACK_SOLVE_FLOPS] += 2*nz ;
		for (i = 0 ; i < n ; i++)
		{
		    xi = X [i] ;
		    for (p = Ap [i] ; p < Ap [i+1] ; p++)
		    {
			Z [Ai [p]] -= Ax [p] * xi ;
		    }
		}
		for (i = 0 ; i < n ; i++)
		{
		    W [i] = Z [Rperm [i]] ;
		}
	    }
	    Info [UMFPACK_SOLVE_FLOPS] += UMF_lsolve (Numeric, W, Pattern) ;
	    Info [UMFPACK_SOLVE_FLOPS] += UMF_usolve (Numeric, W, Pattern) ;
	    if (step == 0)
	    {
		for (i = 0 ; i < n ; i++)
		{
		    X [Cperm [i]] = W [i] ;
		}
	    }
	    else
	    {
		Info [UMFPACK_SOLVE_FLOPS] += n ;
		for (i = 0 ; i < n ; i++)
		{
		    X [Cperm [i]] += W [i] ;
		}
	    }

	    /* -------------------------------------------------------------- */
	    /* sparse backward error estimate */
	    /* -------------------------------------------------------------- */

	    if (irstep > 0)
	    {

		/* ---------------------------------------------------------- */
		/* A is stored by column */
		/* W (i) = (b-Ax)_i, residual */
		/* Z (i) = (|A||x|)_i */
		/* ---------------------------------------------------------- */

		for (i = 0 ; i < n ; i++)
		{
		    W [i] = B [i] ;
		    Z [i] = 0 ;
		}
		Info [UMFPACK_SOLVE_FLOPS] += 3*nz ;
		for (j = 0 ; j < n ; j++)
		{
		    xj = X [j] ;
		    for (p = Ap [j] ; p < Ap [j+1] ; p++)
		    {
			i = Ai [p] ;
			axx = Ax [p] * xj ;
			W [i] -= axx ;
			Z [i] += ABS (axx) ;
		    }
		}

		if (iterative_step (omega, step, B, X, W, Y, Z, S, n, Info))
		{
		    /* iterative refinement is done */
		    return (UMFPACK_OK) ;
		}

	    }

	}

    }
    else if (STRING_MATCH (sys, "A'x=b"))
    {

	/* ------------------------------------------------------------------ */
	/* solve A'x=b with optional iterative refinement */
	/* ------------------------------------------------------------------ */

	if (irstep > 0)
	{
	
	    /* -------------------------------------------------------------- */
	    /* using iterative refinement:  check workspace and compute Y */
	    /* -------------------------------------------------------------- */

	    if (!Ap || !Ai || !Ax || !Y || !Z || !S)
	    {
		return (UMFPACK_ERROR_argument_missing) ;
	    }

	    nz = Ap [n] ;
	    Info [UMFPACK_NZ] = nz ;

	    /* A' is stored by row */
	    /* Y (i) = ||A'_i||, 1-norm of row i of A' */
	    Info [UMFPACK_SOLVE_FLOPS] += nz ;
	    for (i = 0 ; i < n ; i++)
	    {
		yi = 0 ;
		for (p = Ap [i] ; p < Ap [i+1] ; p++)
		{
		    yi += ABS (Ax [p]) ;
		}
		Y [i] = yi ;
	    }
	}

	for (step = 0 ; step <= irstep ; step++)
	{

	    /* -------------------------------------------------------------- */
	    /* Solve A'x=b (step 0): */
	    /*	x = P' (L' \ (U' \ (Q'b))) */
	    /* and then perform iterative refinement (step > 0): */
	    /*	x = x + P' (L' \ (U' \ (Q' (b-A'x)))) */
	    /* -------------------------------------------------------------- */

	    if (step == 0)
	    {
		for (i = 0 ; i < n ; i++)
		{
		    W [i] = B [Cperm [i]] ;
		}
	    }
	    else
	    {
		for (i = 0 ; i < n ; i++)
		{
		    Z [i] = B [i] ;
		}
		Info [UMFPACK_SOLVE_FLOPS] += 2*nz ;
		for (i = 0 ; i < n ; i++)
		{
		    zi = Z [i] ;
		    for (p = Ap [i] ; p < Ap [i+1] ; p++)
		    {
			zi -= Ax [p] * X [Ai [p]] ;
		    }
		    Z [i] = zi ;
		}
		for (i = 0 ; i < n ; i++)
		{
		    W [i] = Z [Cperm [i]] ;
		}
	    }
	    Info [UMFPACK_SOLVE_FLOPS] += UMF_utsolve (Numeric, W, Pattern) ;
	    Info [UMFPACK_SOLVE_FLOPS] += UMF_ltsolve (Numeric, W, Pattern) ;
	    if (step == 0)
	    {
		for (i = 0 ; i < n ; i++)
		{
		    X [Rperm [i]] = W [i] ;
		}
	    }
	    else
	    {
		Info [UMFPACK_SOLVE_FLOPS] += n ;
		for (i = 0 ; i < n ; i++)
		{
		    X [Rperm [i]] += W [i] ;
		}
	    }

	    /* -------------------------------------------------------------- */
	    /* sparse backward error estimate */
	    /* -------------------------------------------------------------- */

	    if (irstep > 0)
	    {

		/* ---------------------------------------------------------- */
		/* A' is stored by row */
		/* W (i) = (b-A'x)_i, residual */
		/* Z (i) = (|A'||x|)_i */
		/* ---------------------------------------------------------- */

		Info [UMFPACK_SOLVE_FLOPS] += 3*nz ;
		for (i = 0 ; i < n ; i++)
		{
		    wi = B [i] ;
		    zi = 0 ;
		    for (p = Ap [i] ; p < Ap [i+1] ; p++)
		    {
			axx = Ax [p] * X [Ai [p]] ;
			wi -= axx ;
			zi += ABS (axx) ;
		    }
		    W [i] = wi ;
		    Z [i] = zi ;
		}

		if (iterative_step (omega, step, B, X, W, Y, Z, S, n, Info))
		{
		    /* iterative refinement is done */
		    return (UMFPACK_OK) ;
		}

	    }

	}

    }
    else if (STRING_MATCH (sys, "P'Lx=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve P'Lx=b:  x = L \ Pb */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    X [i] = B [Rperm [i]] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_lsolve (Numeric, X, Pattern) ;

    }
    else if (STRING_MATCH (sys, "Lx=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve Lx=b:  x = L \ b */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    X [i] = B [i] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_lsolve (Numeric, X, Pattern) ;

    }
    else if (STRING_MATCH (sys, "L'Px=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve L'Px=b:  x = P' (L' \ b) */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    W [i] = B [i] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_ltsolve (Numeric, W, Pattern) ;
	for (i = 0 ; i < n ; i++)
	{
	    X [Rperm [i]] = W [i] ;
	}

    }
    else if (STRING_MATCH (sys, "L'x=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve L'x=b:  x = L' \ b */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    X [i] = B [i] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_ltsolve (Numeric, X, Pattern) ;

    }
    else if (STRING_MATCH (sys, "UQ'x=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve UQ'x=b:  x = Q (U \ b) */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    W [i] = B [i] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_usolve (Numeric, W, Pattern) ;
	for (i = 0 ; i < n ; i++)
	{
	    X [Cperm [i]] = W [i] ;
	}

    }
    else if (STRING_MATCH (sys, "Ux=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve Ux=b:  x = U \ b */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    X [i] = B [i] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_usolve (Numeric, X, Pattern) ;

    }
    else if (STRING_MATCH (sys, "QU'x=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve QU'x=b:  x = U' \ Q'b */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    X [i] = B [Cperm [i]] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_utsolve (Numeric, X, Pattern) ;

    }
    else if (STRING_MATCH (sys, "U'x=b"))
    {

	/* ------------------------------------------------------------------ */
	/* Solve U'x=b:  x = U' \ b */
	/* ------------------------------------------------------------------ */

	for (i = 0 ; i < n ; i++)
	{
	    X [i] = B [i] ;
	}
	Info [UMFPACK_SOLVE_FLOPS] = UMF_utsolve (Numeric, X, Pattern) ;

    }
    else
    {
	return (UMFPACK_ERROR_invalid_system) ;
    }

    return (UMFPACK_OK) ;
}


/* ========================================================================== */
/* === iterative_step ======================================================= */
/* ========================================================================== */

/* Perform one step of iterative refinement, for Ax=b or A'x=b */

PRIVATE Int iterative_step	/* return TRUE if iterative refinement done */
(
    double omega [3],
    Int step,			/* which step of iterative refinement to do */
    const double B [ ],
    double X [ ],
    const double W [ ],
    const double Y [ ],
    const double Z [ ],
    double S [ ],
    Int n,
    double Info [UMFPACK_INFO]
)
{
    double last_omega [3], bi, tau, nctau, d1, wd1, d2, wd2, xnorm, xi, yix, wi;
    Int i ;

    /* DBL_EPSILON is a standard ANSI C term defined in <float.h> */
    /* It is the smallest positive x such that 1.0+x != 1.0 */

    nctau = 1000 * n * DBL_EPSILON ;
    DEBUG0 (("nctau = %30.20e\n", nctau)) ;

    /* for approximate flop count, assume d1 > tau is always true */
    Info [UMFPACK_SOLVE_FLOPS] += 5*n ;

    /* ---------------------------------------------------------------------- */
    /* save the last iteration in case we need to reinstate it */
    /* ---------------------------------------------------------------------- */

    last_omega [0] = omega [0] ;
    last_omega [1] = omega [1] ;
    last_omega [2] = omega [2] ;

    /* ---------------------------------------------------------------------- */
    /* compute sparse backward errors: omega [1] and omega [2] */
    /* ---------------------------------------------------------------------- */

    /* xnorm = ||x|| maxnorm */
    xnorm = 0 ;
    for (i = 0 ; i < n ; i++)
    {
	xi = ABS (X [i]) ;
	xnorm = MAX (xnorm, xi) ;
    }

    omega [1] = 0 ;
    omega [2] = 0 ;
    for (i = 0 ; i < n ; i++)
    {
	bi = ABS (B [i]) ;
	yix = Y [i] * xnorm ;
	tau = (yix + bi) * nctau ;
	d1 = Z [i] + bi ;
	wi = ABS (W [i]) ;
	if (d1 > tau)
	{
	    wd1 = wi / d1 ;
	    omega [1] = MAX (omega [1], wd1) ;
	}
	else if (tau > 0)
	{
	    d2 = Z [i] + yix ;
	    wd2 = wi / d2 ;
	    omega [2] = MAX (omega [2], wd2) ;
	}
    }

    omega [0] = omega [1] + omega [2] ;
    Info [UMFPACK_OMEGA1] = omega [1] ;
    Info [UMFPACK_OMEGA2] = omega [2] ;

    /* ---------------------------------------------------------------------- */
    /* stop the iterations if the backward error is small */
    /* ---------------------------------------------------------------------- */

    Info [UMFPACK_IR_TAKEN] = step ;
    Info [UMFPACK_IR_ATTEMPTED] = step ;
    if (omega [0] < DBL_EPSILON)
    {
	DEBUG0 (("%% omega[0] too small - done.\n")) ;
	return (TRUE) ;
    }

    /* ---------------------------------------------------------------------- */
    /* stop if insufficient decrease in omega */
    /* ---------------------------------------------------------------------- */

    if (step > 0 && omega [0] > last_omega [0] / 2)
    {
	DEBUG0 (("%% stop refinement\n")) ;
	if (omega [0] > last_omega [0])
	{
	    /* last iteration better than this one, reinstate it */
	    DEBUG0 (("%% last iteration better\n")) ;
	    for (i = 0 ; i < n ; i++)
	    {
		X [i] = S [i] ;
	    }
	    Info [UMFPACK_OMEGA1] = last_omega [1] ;
	    Info [UMFPACK_OMEGA2] = last_omega [2] ;
	}
	Info [UMFPACK_IR_TAKEN] = step - 1 ;
	return (TRUE) ;
    }

    /* ---------------------------------------------------------------------- */
    /* save current solution in case we need to reinstate */
    /* ---------------------------------------------------------------------- */

    for (i = 0 ; i < n ; i++)
    {
	S [i] = X [i] ;
    }

    /* ---------------------------------------------------------------------- */
    /* iterative refinement continues */
    /* ---------------------------------------------------------------------- */

    return (FALSE) ;
}

