/* ==================================================== ======== ======= *
 *
 *  uctrl.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _uctrl_hh
#define	_uctrl_hh
//pragma ident	"@(#)uctrl.hh	ubit:b1.11.6"


// NOTE that UCond, UOn, UFlag, UScaleRange are NOT derived from UBrick!
// (==> they can't be UBox children)
class UCond {
public:
  virtual u_bool verifies(const UContext*, const class UCtrl*) const = 0;
  virtual void setParentModes(UCtrl *parent) const = 0;
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UOn : public UCond {
  friend class UCtrl;
  friend class UGroup;
public:
  u_modes bmodes, cmodes;
  u_state istate;
public:
  static UOn
    // High-level events: 
    // The object (typically a UButton or a UCheckbox) has been activated,
    // armed or disarmed. 
    // -- These events are discarded if the object is not enabled
    //    (see methoed UGroup::enable())
    // -- Sensitivity to these events depends on the object class.
    //    This can be changed by adding brick UMode::canArm
    //    (or by setting the CAN_ARM cmode)
    action, arm, disarm, idle,

    // mouse events:
    // -- mdrag and mrelease are always received by the object that
    //    got mpress.
    // -- mmove is received by the object where the mouse is currently
    //    located
    // -- mrelax is similar to mrelease but it is received by the object
    //    where the mouse was released (instead of the object that 
    //    initialiy got mpress)
    enter, leave, 
    mpress, mrelease, 
    mdrag, mmove, mrelax,

    // mouse clicks
    // -- in case of multiple clicks: m1click is called at the 1st click,
    //    m2click at the 2nd click (a "double click"), and so on
    // -- mclick is called in all cases
    mclick, m1click, m2click, m3click, m4click,

    // key events:
    // -- kpress and krelease are generated by raw key events
    // -- ktype is a higher-level event that compose ASCII characters
    //    it is not activated by special or function keys (eg. LEFT, F1, etc)
    ktype, kpress, krelease,

    // object selection (for checkboxes and objects with UMode::canSelect)
    select, unselect,

    // change has different meanings depending on object type:
    // -- for containers (UGroup,UBox,UWin...)
    //    'change' means that the object was selected OR unselected
    // -- for UProp, UItem, UStr...
    //    'change' means that value of this object has changed
    change,

    // detects 'change' conditions that occur (respectively) on UProp,
    // UItem or UStr children of a container
    propChange, itemChange, strChange, 

    // adding and removing children:
    // -- these events are received by the parent(s) of the objects
    //    that are being added or removed (or deleted)
    add, remove,

    // drag and drop:
    dragStart, dragDone, dropEnter, dropLeave, dropDone,

    // The following events are related to object' *views*
    // (and thus, may be produced several times for a single Box object
    //  if this object has several parents)
    viewPaint, viewResize, viewMove,
    repaint, //obsolete: equivalent to viewPaint

    // object traversal
    // -- called when events traverse an object (typically a
    //    parent or a transparent tool)
    //==> SEE ALSO:
    // UGroup::onChildrenEvents() which detects events occuring on children
    traverse,

    // the following events are only received by hard windows
    // (typically the main frame)
    message,			// X client message events
    anyEvent;			// any X Event


  // Interaction State
  enum ISTATE {
    ANY = -1, 
    IDLE = 0,			// must be 0 (index in style arrays)
    DISABLED, ENTERED, ARMED, ACTIONED, DROPPED, DRAGGED, HIGHLIGHTED,
    ACTION_COUNT			// (used in Style Arrays)
    // !! BEWARE must be coherent with typedef u_state in udefs.hh !!
  };

  // NB: modes can be added in order to make the UGroup & UBox parents
  // sensitive to the action that is specified as the first argument
  UOn(u_state istate, u_modes bmodes, u_modes cmodes);
  virtual u_bool verifies(const UContext*, const class UCtrl*) const;
  virtual void setParentModes(UCtrl *parent) const;
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UFlag : public UCond {
protected:
  const char *flagname;
public:
  UFlag(const char* = null);  // !!CAUTION: NEVER delete flags using 'delete'

  const char* getName() const {return flagname;}
  virtual u_bool verifies(const UContext*, const class UCtrl*) const;
  virtual void setParentModes(UCtrl *parent) const;
};
				//NOTE that UFlag is NOT a UBrick!
UFlag& uflag(const char*);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UScaleRange : public UCond {
protected:
  int scale_min, scale_max;
public:
  UScaleRange(int, int);  // !!CAUTION: NEVER delete by using 'delete'
  virtual u_bool verifies(const UContext*, const class UCtrl*) const;
  virtual void setParentModes(UCtrl *parent) const;
};
				//NOTE that UScaleRange is NOT a UBrick !
UScaleRange& uscaleRange(int scale_min, int scale_max);

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UMode: public UBrick {
  friend class UCtrl;
protected:
  //bmodes is inherited from UBrick
  u_modes cmodes, bmask, cmask;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}
  static UMode 
    selected, unselected, canSelect, canSelectChildren,
    canArm,                    //OBSOLETE: canFocus
    enterHighlight, enterHighborder, 
    actionHighlight, actionHighborder, actionWatch,
    canDrag, canDrop,		// for Drag and Drop
    ignoreEvents;

  enum BMODES {
    // this object is constant (and can't be deleted) 
    UCONST     = 1<<0,
    // this object can't be deleted (but is not constant)
    NO_DEL     = 1<<1,
    // this object can't be deleted implicitely by its parents
    NO_AUTODEL = 1<<2,

    // objects can't be autodeleted if one of these conditions is true
    CANT_AUTODEL = UCONST | NO_DEL | NO_AUTODEL,

    // this item is part of a text selection
    IN_TEXTSEL = 1<<3,

    // ** This object is sensitive to the following callback conditions:
    CHANGE_CB        = 1<<4,
    CHILD_CHANGE_CB  = 1<<5,
    ADD_CB           = 1<<6,
    REMOVE_CB        = 1<<7,
    VIEW_PAINT_CB    = 1<<8,
    VIEW_CHANGE_CB   = 1<<9,

    // -- Key: UOn::kpress, UOn::krelease and UOn::ktype
    KEY_CB     = 1<<10,

    // -- Mouse: UOn::mpress, UOn::mrelease
    MOUSE_CB         = 1<<12,
    // -- Mouse: UOn::mclick, UOn::mclick2
    MOUSE_CLICK_CB   = 1<<13,
    // -- Enter, Leave and Cursor: UOn::enter and UOn::leave
    MOUSE_CROSS_CB   = 1<<14,
    // -- Drag & Move: UOn::mdrag and UOn::mmove
    MOUSE_DRAG_CB    = 1<<15,
    MOUSE_MOVE_CB    = 1<<16,  //!no comma after last item!

    // -- UOn::message, UOn::anyEvent
    MESSAGE_CB       = 1<<19,
    ANY_EVENT_CB     = 1<<20,

    // this object ignores events (its parent will get them)
    IGNORE_EVENTS     = 1<<21,

    // this object flags events for further processing
    // (such as modifing events by a callback function or propagating
    // them to ancestors or other objects)
    FLAG_EVENTS       = 1<<22 //!no comma after last item!

    //!! BEWARE total size must be coherent with typedef u_modes in udefs.hh !!
  };

  enum CMODES {
    //--- Types

    GROUP             = 1<<0,
    BOX               = 1<<1,
    SOFTWIN  	      = 1<<2, //NB: SOFTWIN => FLOATING!
    HARDWIN           = 1<<3,
    SUB_HARDWIN       = 1<<4, //for UIncrust...
    FLOATING          = 1<<5,
    SOFTWIN_LIST      = 1<<6,

    //--- Group, Box, Win attributes

    // CAN_SHOW means 'showable' not 'shown'. 
    // Refer to isShowable() and isShown() methods below for details.
    CAN_SHOW          = 1<<8,
    // this object is in the selected mode
    SELECTED          = 1<<9,
    // this object is modal when shown
    MODAL             = 1<<10,
    //ENABLED (ISTATE::DISABLED used instead)

    // ** This object can ...
    // -- be selected (in the same way as a checkbox)
    CAN_SELECT        = 1<<11, 
    //CAN_SELECT_CHILDREN = 1<<15, not used

    // -- can select or edit included text
    CAN_SELECT_TEXT   = 1<<13,
    CAN_EDIT_TEXT     = 1<<14,

    // -- can be dragged or dropped
    CAN_DRAG          = 1<<15,
    CAN_DROP          = 1<<16,

    // -- can be armed (meaning it will be repaint on Uon::arm conditions
    // and that corresponding object's callbacks will be called)
    CAN_ARM           = 1<<17,
    CAN_ACTION        = 1<<18,

    // ** Higligting:
    // -- on UOn::action conditions
    ACTION_HIGHLIGHT  = 1<<19,
    ACTION_HIGHBORDER = 1<<20,
    //ACTION_WATCH      = 1<<25,  not used!

    // -- on UOn::enter and UOn::leave conditions
    ENTER_HIGHLIGHT   = 1<<21,
    ENTER_HIGHBORDER  = 1<<22,

    //--- Box attributes

    // this object has a specific renderer (= a UViewStyle object)
    HAS_RENDERER      = 1<<23,
    // this object has a specific cursor
    HAS_CURSOR        = 1<<24,

    //--- Win attributes

    WIN_MAPPED        = 1<<25,
    WIN_AUTOPOS       = 1<<26,

    // autoclose behavior is dynamically inherited if this field is not set
    HAS_AUTOCLOSE_BHV = 1<<27,
    // redefines autoclose behavior if HAS_AUTOCLOSE_BHV is set
    // -- this object and its descendants auto close menus when clicked
    //    if this field is set and don't otherwise
    AUTOCLOSE_BHV     = 1<<28,
    // auto opening mode for windows ?
    AUTOOPEN_BHV      = 1<<29 //!no comma after last item!

    //!! BEWARE total size must be coherent with typedef u_modes in udefs.hh !!
  };

  UMode(u_modes bmodes, u_modes cmodes, u_bool on_off);
  UMode(u_modes bmodes = 0);
  virtual ~UMode() {clean();}

  //package_private: ====[Ubit Intrinsics]=============================
  //virtual void update();
  virtual void addingTo(ULink *selflink, UGroup *parent);
  //NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);
  virtual void setParentModes(UCtrl *parent) const;
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UCtrl: public UBrick {	// abstract class
  friend class UMode;
  friend class UOn;
  friend class UAppli;
protected:
  u_modes cmodes;
  u_state istate;
public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UCtrl();

  u_bool isSelected() const {return (cmodes & UMode::SELECTED) != 0;}
  u_bool isEnabled() const  {return istate != UOn::DISABLED;}

  // an object is transparent when its bgcolor is UBgcolor::none
  // (this property is now related to UViews)
  //u_bool isTransparent() const {return (cmodes & UMode::TRANSPARENT) != 0;}

  // can the object be selected interactively ?
  u_bool isSelectable() const{
    return (cmodes & UMode::CAN_SELECT) != 0;
  }

  // can the object be shown ?
  // if true, means that this object will be shown if its parents can also
  // be shown. This fct. is equivalent to isVisible() in Java (hoping this 
  // name will be slightly less unclear... :-)
  u_bool isShowable() const {
    return (cmodes & UMode::CAN_SHOW) != 0;
  }

  // is the object currently shown ?
  // true if the object is 'showable' and all its parents are also 'showable'.
  // Note that "shown" objects are not necessarily "visible" on the screen as
  // they can be hidden or clipped by other objects (such as scroll panes, etc.)
  virtual u_bool isShown() const;

  // a browse group defines a common identified for a group of objects
  // that can be broswed
  //EX: u_bool isBrowseGroup() const {return (cmodes & UMode::CAN_BROWSE_CHILDREN)!=0;}

  // is ONE of these bmodes OR ONE of these Cmodes verified ?
  u_bool isDef(u_modes some_Bmodes, u_modes some_Cmodes) const {
    return ((cmodes & some_Cmodes) != 0 || (bmodes & some_Bmodes) != 0);
  }
  u_bool isDef(UMode &m) const {
    return ((cmodes & m.cmodes) != 0 || (bmodes & m.bmodes) != 0);
  }
  // are ALL these bmodes AND ALL these Cmodes verified ?
  u_bool isAllDef(u_modes all_Bmodes, u_modes all_Cmodes) const {
    return ((cmodes & all_Cmodes) == all_Cmodes
	    && (bmodes & all_Bmodes) == all_Bmodes);
  }

  void setState(u_state istate);
  void setCmodes(u_modes cmodes, u_bool on_off);
  //inherited: void setBmodes(u_modes bmodes, u_bool on_off);

  u_state getState()  const {return istate;}
  u_modes getCmodes() const {return cmodes;}
  // inherited: u_modes getBmodes() const {return bmodes;}

  virtual void enable(u_bool state);
  virtual void select(u_bool state);

  // Updates the graphics
  virtual void update() = 0;
  virtual void update(UUpdate) = 0;  // Updates according to mode

  //package_private: ====[Ubit Intrinsics]=============================

  // fires callbacks
  // inherited: virtual void fire(UEvent*, const UOn&);

protected:
  static const int STANDARD;
  static const int BROWSING;
  static const int TYPING;

  virtual void updateView(UEvent&, UView*, const UUpdate&) = 0;

  virtual void enterBehavior(UAppli*, UEvent&, int bstyle);
  virtual void leaveBehavior(UAppli*, UEvent&);
  virtual void armBehavior(UAppli*, UEvent&, int bstyle);
  virtual u_bool disarmBehavior(UAppli*, UEvent&, int bstyle);
  virtual u_bool actionBehavior(UAppli*, UEvent&);
  virtual void keyPressBehavior(UAppli*, UEvent&);
  virtual void keyReleaseBehavior(UAppli*, UEvent&);
  virtual void changeCursorBehavior(UAppli*, UEvent&);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

#include <uprop.hh>

class UEdit : public UProp {
protected:
  // the UStr that contains the caret (if any)
  UStr *caret_str;
  // the position of the caret in the caret string
  int caret_pos;
  // true if the "Return" key can break the line
  u_bool enable_Return;

  // callbacks
  class UCall *call1, *call2, *call3;
  void kpressed(UEvent*);
  void mpressed(UEvent*);
  void mreleased(UEvent*);

public:
  static  const UClass  uclass;
  virtual const UClass* getClass() const {return &uclass;}

  UEdit();
  virtual ~UEdit() {clean();}

  // the "Return" key can break the line if argument is true
  UEdit& enableReturn(u_bool yes = true);

  // returns the UStr that contains the caret (if any)
  const UStr *getCaretStr() const {return caret_str;}

  // returns the position of the caret in the UStr that contains the caret
  // NOTE: this is NOT the position of the caret in the UBox parent
  //       if it contains several UStr
  int getCaretPosInCS() const {return caret_pos;}

  // sets the new caret string and the caret position if THIS string
  void setCaretPosInCS(UStr *caret_str, int caret_pos_in_str, 
		       u_bool update_views = true);

  // moves the position of the caret in the string that contains the caret
  // NOTES:
  // -- this is the position in the string that contains the caret
  //    (not the global position of the caret in the parent UBox)
  // -- position < 0 means : 
  //    the caret goes to the previous UStr in the parent Box
  // -- position > length of the caret string means:
  //    the caret goes to the next UStr in the parent Box
  void moveCaretPosInCS(int caret_pos_in_str, 
			u_bool update_views = true);

  //==== Ubit Intrinsics

  virtual void update();
  virtual void putProp(UContext*, UCtrl*);

  // strpos = pos in newstr / -1 means end of string
  void drawCaret(const UStr*, UContext*, const URegion&r, 
		 class UFlowCell*, class UWinGraph&) const;
  virtual void addingTo(ULink *selflink, UGroup *parent);
  //NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);
};

UEdit &uedit();

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UTextsel {
public:
  u_bool beingSelected;  
  UGroup *inObj;
  // reordered in the left right order
  ULink *fromLink, *toLink, *pressLink, *oldLink;
  int   fromPos, toPos, pressPos, oldPos;
  class UColor    *color;
  class UBgcolor  *bgcolor;
  class UFontDesc *fontdesc;

  UTextsel();

  virtual void reset(u_bool update);
  virtual void start(UAppli*, UEvent&);
  virtual void extend(UAppli*, UEvent&);
  virtual void complete(UAppli*, UEvent&);
  virtual void update(ULink *new_link2, int new_pos2);
  virtual void paint(int _from_pos, int _to_pos);
  virtual void copyTo(UStr&) const;
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
#endif

