// Copyright (C) 2002 Zbigniew Leyk (zbigniew.leyk@anu.edu.au)
//                and David E. Stewart (david.stewart@anu.edu.au)
//                and Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "mx_low_level.h"

namespace Torch {

/* __ip__ -- inner product */
real mxIp__(real * dp1, real * dp2, int len)
{
#ifdef VUNROLL
  int len4;
  real sum1, sum2, sum3;
#endif
  real sum;

  sum = 0.0;
#ifdef VUNROLL
  sum1 = sum2 = sum3 = 0.0;

  len4 = len / 4;
  len = len % 4;

  for (int i = 0; i < len4; i++)
  {
    sum += dp1[4 * i] * dp2[4 * i];
    sum1 += dp1[4 * i + 1] * dp2[4 * i + 1];
    sum2 += dp1[4 * i + 2] * dp2[4 * i + 2];
    sum3 += dp1[4 * i + 3] * dp2[4 * i + 3];
  }
  sum += sum1 + sum2 + sum3;
  dp1 += 4 * len4;
  dp2 += 4 * len4;
#endif

  for (int i = 0; i < len; i++)
    sum += dp1[i] * dp2[i];

  return sum;
}

/* __mltadd__ -- scalar multiply and add c.f. v_mltadd() */
void mxRealMulAdd__(real * dp1, real * dp2, real s, int len)
{
#ifdef VUNROLL
  int len4;

  len4 = len / 4;
  len = len % 4;
  for (int i = 0; i < len4; i++)
  {
    dp1[4 * i] += s * dp2[4 * i];
    dp1[4 * i + 1] += s * dp2[4 * i + 1];
    dp1[4 * i + 2] += s * dp2[4 * i + 2];
    dp1[4 * i + 3] += s * dp2[4 * i + 3];
  }
  dp1 += 4 * len4;
  dp2 += 4 * len4;
#endif

  for (int i = 0; i < len; i++)
    dp1[i] += s * dp2[i];
}

/* __smlt__ scalar multiply array c.f. sv_mlt() */
void mxRealMul__(real * dp, real s, real * out, int len)
{
  for (int i = 0; i < len; i++)
    out[i] = s * dp[i];
}

/* __add__ -- add arrays c.f. v_add() */
void mxAdd__(real * dp1, real * dp2, real * out, int len)
{
  for (int i = 0; i < len; i++)
    out[i] = dp1[i] + dp2[i];
}

/* __sub__ -- subtract arrays c.f. v_sub() */
void mxSub__(real * dp1, real * dp2, real * out, int len)
{
  for (int i = 0; i < len; i++)
    out[i] = dp1[i] - dp2[i];
}

/* __zero__ -- zeros an array of floating point numbers */
void mxZero__(real * dp, int len)
{
  /* else, need to zero the array entry by entry */
  for (int i = 0; i < len; i++)
    dp[i] = 0.0;
}

}

