// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "WeightedSumMachine.h"
#include "random.h"

namespace Torch {

WeightedSumMachine::WeightedSumMachine(Trainer **trainers_, int n_trainers_, List** trainers_measurers_, real *weights_)
{
  // Boaf
  trainers = trainers_;
  trainers_measurers = trainers_measurers_;
  n_trainers = n_trainers_;

  n_outputs = trainers[0]->machine->n_outputs;
  addToList(&outputs, n_outputs, (real *)xalloc(sizeof(real)*n_outputs));

  if(weights_)
  {
    weights = weights_;
    weights_is_allocated = false;
  }
  else
  {
    weights = (real *)xalloc(n_trainers*sizeof(real));
    for(int i = 0; i < n_trainers; i++)
      weights[i] = 0;

    weights_is_allocated = true;
  }

  n_trainers_trained = 0;
}

void WeightedSumMachine::reset()
{
  for (int i=0;i<n_trainers;i++)
    trainers[i]->machine->reset();
  n_trainers_trained = 0;
}

void WeightedSumMachine::forward(List *inputs)
{
  real* pout = (real*)outputs->ptr;
  for(int j = 0; j < n_outputs; j++)
    *pout++ = 0;
  
  for(int i = 0; i < n_trainers_trained; i++)
  {
    trainers[i]->machine->forward(inputs);
    real z = weights[i];
    pout = (real*)outputs->ptr;

    List *out = trainers[i]->machine->outputs;
    while(out)
    {
      real *ptr_out = (real *)out->ptr;
      for(int j = 0; j < out->n; j++)
        *pout++ += z * *ptr_out++;

      out = out->next;
    }
  }
}

void WeightedSumMachine::loadFILE(FILE *file)
{
  xfread(&n_trainers_trained, sizeof(int), 1, file);
  xfread(weights, sizeof(real), n_trainers, file);
  
  for (int i = 0; i < n_trainers; i++)
    trainers[i]->loadFILE(file);
}

void WeightedSumMachine::saveFILE(FILE *file)
{
  xfwrite(&n_trainers_trained, sizeof(int), 1, file);
  xfwrite(weights, sizeof(real), n_trainers, file);

  for (int i = 0; i < n_trainers; i++)
    trainers[i]->saveFILE(file);
}

WeightedSumMachine::~WeightedSumMachine()
{
  freeList(&outputs, true);
  if(weights_is_allocated)
    free(weights);
}

}

