// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "SVM.h"

namespace Torch {

SVM::SVM(Kernel *kernel_)
{
  addROption("C", &C_cst, 100, "trade off margin/classification error");

  kernel = kernel_;
  data = kernel->data;
  n_inputs = data->n_inputs;
  n_outputs = 1;

  support_vectors = NULL;
  real_index = NULL;
  n_support_vectors = 0;

  addToList(&outputs, 1, xalloc(sizeof(real)));
}

void SVM::init()
{
  QCMachine::init();
  reset();
}

void SVM::forward(List *inputs)
{
  real sum = 0;
  for(int it = 0; it < n_support_vectors; it++)
  {
    int t = support_vectors[it];
    sum += y[t]*alpha[t]*kernel->realEval(real_index[it], inputs);
  }

  sum += b;

  ((real *)outputs->ptr)[0] = sum;
}

bool SVM::bCompute()
{
  real sum = 0;
  int n_ = 0;
  for(int it = 0; it < n_support_vectors; it++)
  {
    int t = support_vectors[it];
    if( (alpha[t] > Cdown[t]+eps_bornes) && (alpha[t] < Cup[t]-eps_bornes) )
    {
      sum += y[t]*grad[t];
      n_++;
    }
  }
  
  if(n_)
  {
    b = -sum/(real)n_;
    return(true);
  }
  else
    return(false);
}

void SVM::loadFILE(FILE *file)
{
  xfread(alpha, sizeof(real), l, file);
  xfread(grad, sizeof(real), l, file);
  xfread(y, sizeof(real), l, file);
  checkSupportVectors();
}

void SVM::saveFILE(FILE *file)
{
  xfwrite(alpha, sizeof(real), l, file);
  xfwrite(grad, sizeof(real), l, file);
  xfwrite(y, sizeof(real), l, file);
}

SVM::~SVM()
{
  freeList(&outputs, true);
}

}

