// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "CmdLine.h"

namespace Torch {

#define CMD_INT    0
#define CMD_SWITCH 1
#define CMD_STRING 2
#define CMD_REAL   3

const char *cmd_type_str[4] = {"<int>", "", "<string>", "<real>"};

CmdLine::CmdLine()
{
  n_cmd_options = 0;
  n_cmd_params = 0;
  cmd_options = NULL;
  text_info = NULL;
}

void CmdLine::info(const char *text)
{
  if(text_info)
  {
    free(text_info);
    text_info = NULL;
  }

  text_info = (char *)xalloc(strlen(text)+1);

  strcpy(text_info, text);
}

void CmdLine::addCmdOption(const char *name, void *ptr, int type, const char *help, int status)
{
  cmd_options = (CmdOption *)xrealloc((void *)cmd_options, (n_cmd_options+1)*sizeof(CmdOption));

  CmdOption *optr = cmd_options+n_cmd_options;
  
  optr->name = (char *)xalloc(strlen(name)+1);
  optr->help = (char *)xalloc(strlen(help)+1);
  strcpy(optr->name, name);
  strcpy(optr->help, help);
  optr->ptr = ptr;
  optr->type = type;
  optr->status = status;
  
  if(status == CMD_PARAMS)
    n_cmd_params++;
  
  n_cmd_options++;  
}


void CmdLine::addICmdOption(const char *name, int *ptr, int initvalue, const char *help)
{
  *ptr = initvalue;
  addCmdOption(name, (void *)ptr, CMD_INT, help, CMD_OPTION);
}

void CmdLine::addBCmdOption(const char *name, bool *ptr, bool initvalue, const char *help)
{
  *ptr = initvalue;
  addCmdOption(name, (void *)ptr, CMD_SWITCH, help, CMD_OPTION);
}

void CmdLine::addRCmdOption(const char *name, real *ptr, real initvalue, const char *help)
{
  *ptr = initvalue;
  addCmdOption(name, (void *)ptr, CMD_REAL, help, CMD_OPTION);
}

void CmdLine::addSCmdOption(const char *name, char **ptr, const char *initvalue, const char *help)
{
  *ptr = (char *)xalloc(strlen(initvalue)+1);
  strcpy(*ptr, initvalue);
  addCmdOption(name, (void *)ptr, CMD_STRING, help, CMD_OPTION);
}

void CmdLine::addICmdArg(const char *name, int *ptr, const char *help)
{
  addCmdOption(name, (void *)ptr, CMD_INT, help, CMD_PARAMS);
}

void CmdLine::addBCmdArg(const char *name, bool *ptr, const char *help)
{
  addCmdOption(name, (void *)ptr, CMD_SWITCH, help, CMD_PARAMS);
}

void CmdLine::addRCmdArg(const char *name, real *ptr, const char *help)
{
  addCmdOption(name, (void *)ptr, CMD_REAL, help, CMD_PARAMS);
}

void CmdLine::addSCmdArg(const char *name, char **ptr, const char *help)
{
  *ptr = NULL;
  addCmdOption(name, (void *)ptr, CMD_STRING, help, CMD_PARAMS);
}

void CmdLine::addText(const char *text)
{
  addCmdOption(text, NULL, 0, "", CMD_TEXT);
}

void CmdLine::printCmdOption(CmdOption *ptro)
{
  char **ptr_s;
  switch(ptro->type)
  {
    case CMD_INT:
      print(" [%d]", *((int *)ptro->ptr));
      break;
    case CMD_REAL:
      print(" [%g]", *((real *)ptro->ptr));
      break;
    case CMD_STRING:
      ptr_s = (char **)ptro->ptr;
      print(" [%s]", *ptr_s);
      break;
  }
}

void CmdLine::setCmdOption(int argc, char **argv, int *current, CmdOption *ptro)
{
  int current_ = *current;
  int *ptr_i;
  bool *ptr_b;
  real *ptr_r;
  char **ptr_s;
  
  if(ptro->status == CMD_OPTION)
  {
    if( (current_ >= argc-n_cmd_params) && (ptro->type != CMD_SWITCH) )
      error("CmdLine: cannot correctly set option <%s>", ptro->name);
  }
  else
  {
    if( (current_ >= argc) && (ptro->type != CMD_SWITCH) )
      error("CmdLine: cannot correctly argument <%s>", ptro->name);
  }

  switch(ptro->type)
  {
    case CMD_INT:
      ptr_i  = (int *)ptro->ptr;
      *ptr_i = atoi(argv[current_++]);
      break;
    case CMD_SWITCH:
      ptr_b  = (bool *)ptro->ptr;
      *ptr_b = !(*ptr_b);
      break;
    case CMD_REAL:
      ptr_r  = (real *)ptro->ptr;
      *ptr_r = (real)atof(argv[current_++]);
      break;
    case CMD_STRING:
      ptr_s  = (char **)ptro->ptr;
      free(*ptr_s);
      *ptr_s = (char *)xalloc(strlen(argv[current_])+1);
      strcpy(*ptr_s, argv[current_++]);
      break;
    default:
      error("CmdLine: wrong format");
  }

  *current = current_;
}

void CmdLine::read(int argc, char **argv)
{
  if(argc-1 < n_cmd_params)
    help(argv[0]);
  
  if(argc >= 2)
  {
    if( ! (strcmp(argv[1], "-h") && strcmp(argv[1], "-help") && strcmp(argv[1], "--help")) )
      help(argv[0]);
  }

  int current = 1;
  while(current < argc-n_cmd_params)
  {
    // Look for the option
    int the_opt = -1;
    for(int j = 0; j < n_cmd_options; j++)
    {
      if(cmd_options[j].status == CMD_OPTION)
      {
        if(strcmp(cmd_options[j].name, argv[current]) == 0)
        {
          the_opt = j;
          break;
        }
      }
    }

    if(the_opt == -1)
      error("CmdLine: parse error near <%s>", argv[current]);
    
    // Okay, on a trouve l'option
    current++;

    // Traite l'option
    setCmdOption(argc, argv, &current, &cmd_options[the_opt]); 
  }

  // Les parametres...
  for(int j = 0; j < n_cmd_options; j++)
  {
    if(cmd_options[j].status == CMD_PARAMS)
      setCmdOption(argc, argv, &current, &cmd_options[j]); 
  }  
}

void CmdLine::help(char *name)
{
  if(text_info)
    print("%s\n", text_info);

  print(" usage: %s", name);
  if(n_cmd_options-n_cmd_params > 0)
    print(" [options]");

  for(int i = 0; i < n_cmd_options; i++)
  {
    if(cmd_options[i].status == CMD_PARAMS)
      print(" <%s>", cmd_options[i].name);
  }
  print("\n");

  // Cherche la longueur max du param
  int long_max = 0;
  int z = 0;
  for(int i = 0; i < n_cmd_options; i++)
  {
    switch(cmd_options[i].status)
    {
      case CMD_OPTION:
        z = strlen(cmd_options[i].name)+strlen(cmd_type_str[cmd_options[i].type])+1;
        break;
      case CMD_PARAMS:
        z = strlen(cmd_options[i].name)+2;
        break;
    }

    if(long_max < z)
      long_max = z;
  }

  for(int i = 0; i < n_cmd_options; i++)
  {
    switch(cmd_options[i].status)
    {
      case CMD_PARAMS:
        z = strlen(cmd_options[i].name)+2;
        print("  ");
        print("<%s>", cmd_options[i].name);
        break;        
      case CMD_TEXT:
        z = -1;
        print("%s", cmd_options[i].name);
        break;
      case CMD_OPTION:
        z = strlen(cmd_options[i].name)+strlen(cmd_type_str[cmd_options[i].type])+1;
        print("  ");
        print("%s", cmd_options[i].name);
        print(" %s", cmd_type_str[cmd_options[i].type]);
        break;
    }

    if(z >= 0)
    {
      for(int i = 0; i < long_max+1-z; i++)
        print(" ");
    }

    if(cmd_options[i].status != CMD_TEXT)
      print("-> %s", cmd_options[i].help);
    
    switch(cmd_options[i].status)
    {
      case CMD_OPTION:
        printCmdOption(&cmd_options[i]);
        break;
      case CMD_PARAMS:
        print(" (%s)", cmd_type_str[cmd_options[i].type]);
        break;
    }
    print("\n");
  }

  exit(0);
}

CmdLine::~CmdLine()
{
  for(int i = 0; i < n_cmd_options; i++)
  {
    free(cmd_options[i].name);
    if(cmd_options[i].type == CMD_STRING)
    {
      char **ptr = (char **)cmd_options[i].ptr;
      free(*ptr);
    }
    free(cmd_options[i].help);
  }
  free(cmd_options);
  free(text_info);
}

}

