// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                and Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef MLP_INC
#define MLP_INC

#include "ConnectedMachine.h"
#include "Linear.h"
#include "SumMachine.h"
#include "Softmax.h"
#include "LogSoftmax.h"
#include "Sigmoid.h"
#include "Tanh.h"
#include "SparseLinear.h"

namespace Torch {

/** This class is a simple interface to the #ConnectedMachine# class that
    ca be used to build the well-known Multi Layer Perceptron type of
    neural networks. It contains a layer of #Linear# followed by a layer
    of #Tanh#, followed by a layer of #Linear# and optionally
    \begin{itemize}
      \item a layer of softmax
      \item or a layer of sigmoid
      \item or a layer of log-softmax
      \item or a layer of tanh
    \end{itemize}
    Optionally, it also contains a direct connection from the inputs
    to the linear layer, and if you want, you can choose sparse inputs.

    Options:
    \begin{tabular}{lcll}
      "inputs to outputs"   & bool   & connections from inputs to outputs        & [false]\\
      "weight decay"        & real   & the weight decay                          & [0]\\
      "softmax outputs"     & bool   & softmax outputs                           & [false]\\
      "sigmoid outputs"     & bool   & sigmoid outputs                           & [false]\\
      "log-softmax outputs" & bool   & log-softmax outputs                       & [false]\\
      "tanh outputs"        & bool   & tanh outputs                              & [false]\\
      "sparse inputs"       & bool   & sparse inputs (to use with SparseDataSet) & [false]
    \end{tabular}

   @author Samy Bengio (bengio@idiap.ch)
   @author Ronan Collobert (collober@iro.umontreal.ca)
*/
class MLP : public ConnectedMachine
{
  public:
    /// the first #Linear# layer
    Linear *hidden_layer;
    /// the first #Linear# layer for sparse mode
    SparseLinear *sparse_hidden_layer;
    /// the first #Tanh# layer
    Tanh *hidden_tanh_layer;
    /// the second #Linear# layer
    Linear *outputs_layer;
    /// the second #Linear# layer for sparse mode, if there is no hidden units
    SparseLinear *sparse_outputs_layer;
    /// the optional second #Softmax# layer
    Softmax *outputs_softmax_layer;
    /// the optional second #Sigmoid# layer
    Sigmoid *outputs_sigmoid_layer;
    /// the optional second #Softmax# layer
    LogSoftmax *outputs_log_softmax_layer;
    /// the optional second #Tanh# layer
    Tanh *outputs_tanh_layer;
    /// the number of hidden units
    int n_hidden;
    /// the number of inputs
    int n_inputs;
    /// the number of outputs
    int n_outputs;

    /// if this is false, add a #Tanh# layer

    /** Flags (in order of priority if several are true)
        to know what will be the output.
    */
    //@{
    bool is_softmax_outputs;
    bool is_sigmoid_outputs;
    bool is_log_softmax_outputs;
    bool is_tanh_outputs;
    //@}

    /// To know if the inputs are sparse
    bool is_sparse_inputs;

    /// if this is true, add a direct connection from inputs to #Linear#
    bool inputs_to_outputs;
    /// the eventual weight_decay
    real weight_decay;
    /// the direct #Linear# layer
    Linear* add_layer;
    /// if #inputs_to_outputs# is true, we also need a #SumMachine#
    SumMachine* sum_layer;

    MLP(int n_inputs_, int n_hidden, int n_outputs_);
    virtual void init();
    virtual ~MLP();
};


}

#endif
