/* -*- mode: c; c-file-style: "gnu" -*-
 * src/authorisers/dotrealm.c -- A PAM-based .realm Authoriser for Thy.
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy-Auth.
 *
 * Thy-Auth is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Thy-Auth is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file dotrealm.c
 * This file implements the neccessary functions to handle Thy
 * 0.5-style .realm files with Thy-Auth. See thy-auth-dotrealm(7) for
 * more information.
 */

#include "system.h"

#if HAVE_ARGP_H
#include <argp.h>
#endif

#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "compat/compat.h"
#include "methods/pam.h"
#include "misc.h"

/** @defgroup ta_dotrealm thy-auth-dotrealm
 * @{
 */

static char *dotrealm_service = NULL;

/** Program version.
 * Used by the argp suite.
 */
const char *argp_program_version = "dotrealm (thy-auth " VERSION ")";

/** Address to send bugreports to.
 * Used by the argp suite.
 */
const char *argp_program_bug_address = "<algernon@bonehunter.rulez.org>";

/** Supported command-line options.
 * Used by the argp suite.
 */
static struct argp_option dotrealm_argp_options[] = {
  {"service", 's', "NAME", 0, "Name of the PAM service", 1},
  {0, 0, 0, 0, NULL, 0}
};

static error_t _dotrealm_opt_parse (int key, char *arg,
				    struct argp_state *state);

/** Final definition of the argp parser.
 */
static struct argp dotrealm_argp =
  {dotrealm_argp_options, _dotrealm_opt_parse, 0,
   "dotrealm -- .realm based Authoriser for Thy", NULL, NULL, NULL};


/** Parse one option.
 * See the argp docs for details.
 */
static error_t
_dotrealm_opt_parse (int key, char *arg, struct argp_state *state)
{
  switch (key)
    {
    case 's':
      dotrealm_service = strdup (arg);
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

void
_thy_auth_do_one_auth (int argc, char **argv)
{
  char *user, *pw, *realm;
  int i;

  if (argc < 3)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  /* Get stuff */
  user = argv[0];
  pw = argv[1];
  realm = argv[2];

  /* Authenticate */
  i = thy_auth_pam_auth (user, pw, realm, dotrealm_service);

  /* Response */
  thy_auth_respond ((i == 0) ? "1\n" : "0\n", 2);
}

void
_thy_auth_do_one_check (int argc, char **argv)
{
  char *tmp, *realm;
  off_t offs = 0;
  ssize_t r;
  int fd, len;
  size_t slen;
  struct stat st;

  if (argc < 1)
    {
      thy_auth_respond ("0\n", 2);
      return;
    }

  stat (argv[0], &st);
  fd = open (argv[0], O_RDONLY);
  realm = (char *)bhc_calloc (1, (size_t)(st.st_size + 1));
  do
    {
      r = read (fd, &realm[offs], (size_t)(st.st_size - offs));
      if (r == -1)
	break;
      offs += r;
    } while (offs < st.st_size);
  realm[st.st_size] = '\0';
  tmp = strchr (realm, '\n');
  if (tmp)
    tmp[0] = 0;
  close (fd);

  /* In our simplistic method, if a .realm is found, authorisation is
     always needed. */
  thy_auth_respond ("1\n", 2);

  /* Write out the length & the string itself. */
  slen = (size_t)(tmp - realm);
  len = asprintf (&tmp, "%05u\n%s\n", slen, realm);
  thy_auth_respond (tmp, len);

  free (tmp);
  free (realm);
}

/** thy-auth-dotrealm main core.
 */
int
main (int argc, char **argv)
{
  signal (SIGPIPE, SIG_IGN);

  argp_parse (&dotrealm_argp, argc, argv, 0, 0, NULL);

  for (;;)
    thy_auth_do_one ();

  return 0;
}

/** @} */
