#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "node.h"
#include "mem.h"
#include "support.h"
#include "path.h"

/**/
struct GRAPH
{
	struct NODE *nodehash[0x10000];
	struct NODE *first;
	struct HEAP *heap;
	int num_nodes;
};

/* */
static unsigned int string_hash(const char *str)
{
	unsigned int h = 0;
	for (; *str; str++)
		h = 31*h + *str;
	return h;
}

/**/
struct GRAPH *node_create_graph(struct HEAP *heap)
{
	/* allocate graph structure */
	/*struct GRAPH *graph = (struct GRAPH*)mem_allocate(heap, sizeof(struct GRAPH));*/
	/* TODO: modify the memory manager to manage this aswell */
	struct GRAPH *graph = (struct GRAPH*)malloc(sizeof(struct GRAPH));
	if(!graph)
		return (struct GRAPH *)0x0;

	/* init */
	graph->heap = heap;
	graph->num_nodes = 0;
	graph->first = (struct NODE*)0x0;
	memset(graph->nodehash, 0, sizeof(struct NODE*)*0x10000);
	return graph; 
}


/* creates a node */
int node_create(struct NODE **nodeptr, struct GRAPH *graph, const char *filename, const char *tool)
{
	struct NODE *node;
	int sn;
	
	/* zero out the return pointer */
	*nodeptr = (struct NODE *)0x0;
	
	/* */
	if(!path_isnice(filename))
		return NODECREATE_NOTNICE;
	
	/* */
	if(node_find(graph, filename))
		return NODECREATE_EXISTS;
	
	/* allocate and set pointers */
	node = (struct NODE *)mem_allocate(graph->heap, sizeof(struct NODE));
	node->graph = graph;
	node->id = graph->num_nodes++;
	node->timestamp = file_timestamp(filename);
	node->firstdep = (struct DEPENDENCY*)0x0;
	
	/* set filename */
	sn = strlen(filename)+1;
	node->filename = (char *)mem_allocate(graph->heap, sn);
	memcpy(node->filename, filename, sn);
	node->hashid = string_hash(filename);

	/* set tool */
	if(tool)
	{
		sn = strlen(tool)+1;
		node->tool = (char *)mem_allocate(graph->heap, sn);
		memcpy(node->tool, tool, sn);
	}
	else
		node->tool = (char*)0x0;
	
	/* add to hashed list */
	node->hashnext = graph->nodehash[node->hashid&0xffff];
	graph->nodehash[node->hashid&0xffff] = node;

	/* add to list */
	node->next = graph->first;
	graph->first = node;
	
	/* zero out flags */
	node->dirty = 0;
	node->depchecked = 0;
	node->cached = 0;
	node->workstatus = NODESTATUS_UNDONE;
	
	/* return new node */
	*nodeptr = node;
	return NODECREATE_OK;
}
/* finds a node based apun the filename */
struct NODE *node_find(struct GRAPH *graph, const char *filename)
{
	unsigned int hashid = string_hash(filename);
	struct NODE *node = graph->nodehash[hashid&0xffff];
	for(;node;node = node->hashnext)
	{
		if(node->hashid == hashid)
		{
			if(strcmp(node->filename, filename) == 0)
				return node;
		}
	}
	return (struct NODE *)0x0;
}

/* this will return the existing node or create a new one */
static struct NODE *node_get(struct GRAPH *graph, const char *filename)
{
	struct NODE *node = node_find(graph, filename);
	if(!node)
	{
		if(node_create(&node, graph, filename, 0) == NODECREATE_OK)
			return node;
	}
	return node;
}

/* helper for the node_add_dependency function */
/* searches for the node in the tree to make sure */
/* that it doesn't exist */
static int is_node_in_tree(struct NODE *stack, struct NODE *needle)
{
	struct DEPENDENCY *dep = stack->firstdep;
	while(dep)
	{
		if(dep->node == needle)
			return 1;
		
		if(is_node_in_tree(dep->node, needle))
			return 1;
	
		dep = dep->next;
	}
	
	return 0;
}

/* adds a dependency to a node */
struct NODE *node_add_dependency(struct NODE *node, const char *filename)
{
	struct NODE *depnode;
	struct DEPENDENCY *dep;
	
	/* get node (can't fail) */
	depnode = node_get(node->graph, filename);
	
	/* make sure that the node doesn't try to depend on it self */
	if(depnode == node)
	{
		printf("error: this file depends on it self\n  %s\n", node->filename);
		return (struct NODE*)0x0;
	}
	
	/* create dependency */
	dep = (struct DEPENDENCY *)mem_allocate(node->graph->heap, sizeof(struct DEPENDENCY));
	dep->node = depnode;
	
	/* add the dependency to the node */
	dep->next = node->firstdep;
	node->firstdep = dep;

	/* return the dependency */
	return depnode;
}

int node_walk_depth = 0;
struct NODEWALK
{
	int depth;
	int flags;
	int (*callback)(struct NODE*, void *u);
	void *userdata;
	unsigned char *mark;
};

/* functions to handle with bit array access */
static unsigned char *bitarray_allocate(int size)
{ return (unsigned char *)malloc((size+7)/8); }

static void bitarray_zeroall(unsigned char *a, int size)
{ memset(a, 0, (size+7)/8); }

static void bitarray_free(unsigned char *a)
{ free(a); }

static int bitarray_value(unsigned char *a, int id)
{ return a[id>>3]&(1<<(id&0x7)); }

static void bitarray_set(unsigned char *a, int id)
{ a[id>>3] |= (1<<(id&0x7)); }

static void bitarray_clear(unsigned char *a, int id)
{ a[id>>3] &= ~(1<<(id&0x7)); }

static int node_walk_r(
	struct NODEWALK *walk,
	struct NODE *node)
{
	/* we should detect changes here before we run */
	struct DEPENDENCY *dep;
	int result = 0;
	int needrebuild = 0;
	
	/* check and set mark */
	if(bitarray_value(walk->mark, node->id))
		return 0; 
	bitarray_set(walk->mark, node->id);
	
	if((walk->flags)&NODEWALK_UNDONE)
	{
		if(node->workstatus != NODESTATUS_UNDONE)
			return 0;
	}

	walk->depth++;

	if((walk->flags)&NODEWALK_TOPDOWN)
		result = walk->callback(node, walk->userdata);

	/* update timestamp */
	/* node->timestamp = file_timestamp(node->filename); */
	
	/* build all dependencies */
	for(dep = node->firstdep; dep; dep = dep->next)
	{
		result = node_walk_r(walk, dep->node);
		if(node->timestamp < dep->node->timestamp)
			needrebuild = 1;
		if(result)
			break;
	}
	
	/* unmark the node so we can walk this tree again if needed */
	if(!(walk->flags&NODEWALK_QUICK))
		bitarray_clear(walk->mark, node->id);
	
	/* return if we have an error */
	if(result)
	{
		walk->depth--;
		return result;
	}

	/* check if we need to rebuild this node */
	if(!((walk->flags)&NODEWALK_FORCE) && !node->dirty)
	{
		walk->depth--;
		return 0;
	}
	
	/* build */
	if((walk->flags)&NODEWALK_BOTTOMUP)
		result = walk->callback(node, walk->userdata);
	
	walk->depth--;
	return result;
}

int node_walk(
	struct NODE *node,
	int flags,
	int (*callback)(struct NODE*, void *u),
	void *u)
{
	struct NODEWALK walk;
	int result;
	
	/* set walk parameters */
	walk.depth = 0;
	walk.flags = flags;
	walk.callback = callback;
	walk.userdata = u;

	/* allocate and clear mark and sweep array */
	walk.mark = bitarray_allocate(node->graph->num_nodes);
	bitarray_zeroall(walk.mark, node->graph->num_nodes);

	/* do the walk */
	result = node_walk_r(&walk, node);

	/* free the array and return */
	bitarray_free(walk.mark);
	return result;
}

/* dumps all nodes to the stdout */
void node_debug_dump(struct GRAPH *graph)
{
	struct NODE *node = graph->first;
	struct DEPENDENCY *dep;
	const char *tool;
	for(;node;node = node->next)
	{
		static const char d[] = " D";
		tool = "***";
		if(node->tool)
			tool = node->tool;
		printf("%08x %-15s %c   %s\n", node->timestamp, tool, d[node->dirty], node->filename);
		for(dep = node->firstdep; dep; dep = dep->next)
			printf("%08x%-15s %c      %s\n", dep->node->timestamp, "", d[dep->node->dirty], dep->node->filename);
	}
}
