package Boot::ZIPL;

#   $Header$

#   Copyright (c) 2001-2002 International Business Machines

#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
 
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
 
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#   Donghwa John Kim <johkim@us.ibm.com>

=head1 NAME

Boot::ZIPL - ZIPL bootloader configuration module.

=head1 SYNOPSIS

  my $bootloader = new Boot::ZIPL(%bootvars);

  if($bootloader->footprint_loader()) {
      $bootloader->install_config();
  }
  
  if($bootloader->footprint_config() && $bootloader->footprint_loader()) {
      $boot->install_loader();
  }

  my @fileschanged = $bootloader->files();

=cut

use strict;
use Carp;
use vars qw($VERSION);
use Boot;
use Util::Cmd qw(:all);
use Util::Log qw(:print);

$VERSION = sprintf("%d", q$Revision: 664 $ =~ /(\d+)/);

push @Boot::boottypes, qw(Boot::ZIPL);

sub new {
    my $class = shift;
    my %this = (
                root => "",
                boot_rootdev => "",
		boot_bootdev => "",
                boot_extras => "",
                @_,                  ### Overwrite default value.
                filesmod => [],
               );

    $this{config_file} = $this{root} . "/etc/zipl.conf";
    $this{parm_file} = $this{root} . "/boot/parmfile"; 
    $this{bootloader_exe} = which("zipl");

    if($this{boot_bootdev}) {
	warn "BOOTDEV option under [BOOT] section ignored.\n";
	$this{boot_bootdev} = ""; ### zipl does not have an option to specify the boot device. 
    }

    verbose("ZIPL executable set to: $this{bootloader_exe}.");
    bless \%this, $class;
}

=head1 METHODS

The following methods exist in this module:

=over 4

=item files()

The files() method is merely an accessor method for the all files
touched by the instance during its run.

=cut

sub files {
    my $this = shift;
    return @{$this->{filesmod}};
}

=item footprint_config()

This method returns "TRUE" if the ZIPL configuration file (i.e. "/etc/zipl.conf") exists. 

=cut

sub footprint_config {
    my $this = shift;
    return -e $this->{config_file};
}

=item footprint_loader()

This method returns "TRUE" if the ZIPL executable exists. 

=cut

sub footprint_loader {
    my $this = shift;
    return $this->{bootloader_exe};
}

=item install_config()

This method reads the System Configurator config file and creates zipl.conf file.   

=cut

sub install_config {
    my $this = shift;

    if(!$this->{boot_defaultboot}) 
    {
	croak("Error: DEFAULTBOOT must be specified.\n");
    }

    open(OUT,">$this->{config_file}") or croak("Couldn\'t open $this->{config_file} for writing");    
    print OUT <<ZIPLCONF;
##################################################
# This file is generated by System Configurator. #
##################################################
    
[defaultboot]
default=$this->{boot_defaultboot}
ZIPLCONF

    # Ok, for each kernel image specified, create a boot section accordingly, plus, 
    # create a parm file that holds information about root device and append lines.
    foreach my $key (sort keys %$this) {
        if ($key =~ /^(kernel\d+)_path/) {
            $this->setup_config($1,\*OUT);
        }
    }

    close(OUT);
    push @{$this->{filesmod}}, "$this->{config_file}";
    1;
}

=item setup_config()

An "internal" method.
This method add an entry to zipl.conf file and creates parmfile for each 
kernel image specified.

=cut

sub setup_config {
    my ($this, $kernel, $outfh) = @_;
    my $label = $this->{$kernel . "_label"};
    my $parmfile = $this->{parm_file} . "-" . $label;
    my $initrd = $this->{$kernel . "_initrd"};
    my $append = $this->{$kernel . "_append"};
    my $rootdev;

    print $outfh "\n";
    print $outfh "[". $label . "]\n";
    print $outfh "target=" . $this->{root} . "/boot\n";
    print $outfh "image=" . $this->{root} . $this->{$kernel . "_path"} . "\n";
    print $outfh "parmfile=$parmfile\n";
    if ($initrd) {
	print $outfh "ramdisk=" . $this->{root} . "$initrd\n";
    }

    # Create parmfile for boot options.
    if ($this->{boot_rootdev}) {
	$rootdev = $this->{boot_rootdev};
    } elsif ($this->{$kernel . "_rootdev"}) {
	$rootdev = $this->{$kernel . "_rootdev"};
    } else {
	croak("Cannot derive the root device from configuration.");
    }

    open(PARMFILE, "> $parmfile") or croak("Couldn\'t open $parmfile for writing");
    print PARMFILE "root=" . $rootdev . "\n";    
    if ($append) {
	print PARMFILE "$append\n";
    }
    close(PARMFILE);

    push @{$this->{filesmod}}, "$parmfile";
    
    1;
}

=item install_loader()

This method invokes the ZIPL executable.
ZIPL writes the boot image onto the device specified as the boot partition.  

=cut

sub install_loader {
    my $this = shift;

    my $chroot = ($this->{root}) ? "-r $this->{root}" : "";

    my $output = qx/$this->{bootloader_exe} $chroot 2>&1/;
    
    my $exitval = $? >> 8;
    
    if ($exitval) {
        croak("Error: Cannot execute $this->{bootloader_exe}.\n$output\n");
    }
    1;
}

=back

=head1 AUTHOR

Donghwa John Kim <donghwajohnkim@yahoo.com, johkim@us.ibm.com>

=head1 SEE ALSO

L<Boot>, L<perl>

=cut

1;




















