############################################################################
##      Copyright (C) 2005 Subredu Manuel  <diablo@iasi.roedu.net>.        #
##                                                                         #
## This program is free software; you can redistribute it and/or modify    #
## it under the terms of the GNU General Public License v2 as published by #
## the Free Software Foundation.                                           #
##                                                                         #
## This program is distributed in the hope that it will be useful,         #
## but WITHOUT ANY WARRANTY; without even the implied warranty of          #
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
## GNU General Public License for more details.                            #
##                                                                         #
## You should have received a copy of the GNU General Public License       #
## along with this program; if not, write to the Free Software             #
## Foundation, Inc., 59 Temple Place - Suite 330, Boston,                  #
## MA 02111-1307,USA.                                                      #
############################################################################

package RoPkg::Simba::Mirror;

use strict;
use warnings;

use Scalar::Util qw(blessed);

use RoPkg::DBObject;
use RoPkg::Exceptions;

use vars qw($table $VERSION);

$table = 'Mirrors';
$VERSION='0.1.3';
use base qw(RoPkg::DBObject);

my $pf = {
  id                 => q{-},
  Name               => q{-},
  LogoURL            => q{-},
  Description        => q{-},
  HomeSite           => q{-},
  LocalDir           => q{-},
  Size               => q{-},
  SyncMethod         => q{-},
  CommandID          => q{-},
  LastErrorCode      => q{-},
  Active             => q{-},
  InProgress         => q{-},
  SyncSource         => q{-},
  SyncSourceUser     => q{-},
  SyncSourcePass     => q{-},
  SyncSourceModule   => q{-},
  Contact            => q{-},
  LastUpdated        => q{-},
  LastUpdateDuration => q{-},
  LastUpdateSpeed    => q{-},
  LastUpdateFilesNo  => q{-},
  LastUpdateBytes    => q{-},
  LocalURL           => q{-},
  StdOut             => q{-},
  StdErr             => q{-},
};

sub new {
  my ($class, %opt) = @_;
  my $self;

  $opt{methods} = $pf;

  $self = $class->SUPER::new(%opt);
  $self->{_sql_table} = $table;

  return $self;
}

sub table {
  return $table;
}

##################################
### Database functions - BEGIN ###
##################################

sub Add {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }

  return $self->SQL_Insert();
}

sub Delete {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(id));
    
  return $self->SQL_Delete(qw(id));
}

sub Update {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }
  $self->chkp(qw(id));
    
  return $self->SQL_Update(qw(id));
}

sub Load {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called from outside class');
  }

  return $self->SQL_Select(qw(Name)) if ( defined($self->{Name}));
  return $self->SQL_Select(qw(id))   if ( defined($self->{id}));
  
  Param::Missing->throw('nor id not Name are defined');

  #syntactic sugar
  return 0;
}

##################################
###  Database functions - END  ###
##################################

#################################
### General functions - BEGIN ###
#################################

# We use this to find all the methods (GET/SET) of this object.
# This method is usefull in plugins, when such information is needed

sub GetMethods {
  my ($self) = @_;

  if ( !blessed($self) ) {
    OutsideClass->throw('Called outside class instance');
  }

  return $self->methods();
}

#################################
### General functions -  END  ###
#################################

1;


__END__

=head1 NAME

RoPkg::Simba::Mirror - a mirror class

=head1 VERSION

0.1.3

=head1 DESCRIPTION

RoPkg::Simba::Mirror is a class used to hold all the information
a mirror has. Also, the RoPkg::Simba::Mirror can be used to 
add/del/update a mirror to/from the database. Simba is derivated
from I<RoPkg::DBObject>.

=head1 SYNOPSIS

 !#/usr/bin/perl
 
 use RoPkg::DB;
 use RoPkg::Simba::Mirror;
 
 sub main {
   my $dbp = new RoPkg::DB();
   $dbp->Add('dbi:mysql:database=mysql;host=localhost',
             'root',
             '',
             'local');

   my $m = new RoPkg::Simba::Mirror(dbo => $dbp, dbo_method => 'db_local');
   $m->Name('debian');
   $m->Load();
 }
 
 main();

=head1 SUBROUTINES/METHODS

All methods raise OutsideClass exception when called outside
class instance. Besides this, each method, may raise other
exceptions. Check each method section to find out more.

=head2 new()

The class constructor. At this moment, it just calls
RoPkg::DBObject->new() . Please read the RoPkg::DBObject
manual page for more information about the new() parameters.

=head2 table()

Returns the name of the mirrors database table.

=head2 Add()

Adds the mirror to the database. This method is a
wrapper for RoPkg::DBObject::SQL_Insert . On success
0 is returned. On error, DBI exception is raised.

=head2 Delete()

Deletes the current mirror from the database. Before
calling this method, you should set the B<id> of the
mirror . If you don't set the B<id> Param::Missing 
exception is raised. On database operation success,
0 is returned. On database error, DBI exception is
raised.

=head2 Update()

Update the current mirror object with the database. Before
calling this method, you should set the B<id> of the
mirror . If you don't set the B<id> Param::Missing 
exception is raised. On database operation success,
0 is returned. On database error, DBI exception is
raised.

=head2 Load()

Load the mirror information from the database, into
the current object. Before calling this method
you should have set B<id> or B<Name>. If id or
Name are not set, then Param::Missing is raised.
On database operation success 0 is returned. On
database error, DBI exception is raised.

=head2 GetMethods()

Returns a array with the current object methods
names. In scalar context returns the number
of methods.

The following methods are get/set methods for
all fields of a mirror.

=over 24

=item *) id

=item *) Name

=item *) Description

=item *) HomeSite

=item *) LocalDir

=item *) Size

=item *) SyncMethod

=item *) CommandID

=item *) LastErrorCode

=item *) Active

=item *) InProgress

=item *) SyncSource

=item *) SyncSourceUser

=item *) SyncSourcePass

=item *) SyncSourceModule

=item *) Contact

=item *) LastUpdated

=item *) LastUpdateDuration

=item *) LastUpdateSpeed

=item *) LastUpdateFilesNo

=item *) LastUpdateBytes

=item *) LocalURL

=item *) StdOut

=item *) StdErr

=back

=head1 DIAGNOSTICS

Unpack the source, and use 'make test' command

=head1 CONFIGURATION AND ENVIRONMENT

This module does not use any configuration files
or environment variables

=head1 DEPENDENCIES

RoPkg::DBObject and RoPkg::Exceptions

=head1 INCOMPATIBILITIES

None known to the author

=head1 BUGS AND LIMITATIONS

None known to the author

=head1 PERL CRITIC

This module is perl critic level 2 compliant (with 1 exception)

=head1 SEE ALSO

L<RoPkg::Simba> L<RoPkg::Simba::Mirrors> L<RoPkg::DBObject> L<RoPkg::Object>

=head1 AUTHOR

Subredu Manuel <diablo@iasi.roedu.net>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2005 Subredu Manuel.  All Rights Reserved.
This module is free software; you can redistribute it 
and/or modify it under the same terms as Perl itself.
The LICENSE file contains the full text of the license.

=cut
