/*
 * Copyright (C) 2006 Sjoerd Simons <sjoerd@luon.net>
 *
 * This file is part of Ruby/GStreamer.
 *
 * Ruby/GStreamer is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Ruby/GStreamer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Ruby/GStreamer; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "rbgst.h"

static VALUE crbGstFractionClass;
static VALUE crbGstFourccClass;

static VALUE crbGstRangeClass;
static VALUE crbGstDoubleRangeClass;
static VALUE crbGstFractionRangeClass;
static VALUE crbGstIntRangeClass;

static VALUE crbGstArrayClass;
static VALUE crbGstListClass;

/* GstFraction */
static VALUE
rbgst_fraction_new(VALUE self, VALUE numerator, VALUE denominator) {
  rb_iv_set(self, "@numerator", numerator);
  rb_iv_set(self, "@denominator", denominator);
  return self;
}

static VALUE
rbgst_fraction_inspect(VALUE self) {
  VALUE n,d;
  n = rb_iv_get(self, "@numerator");
  d = rb_iv_get(self, "@denominator");
  return rb_str_new_with_format("%d/%d", NUM2INT(n), NUM2INT(d)); 
}

static VALUE
rbgst_fraction_is_equal(VALUE self, VALUE other) {
  VALUE n0, n1, d0, d1;

  if (NIL_P(other)) return Qfalse;

  n0 = rb_iv_get(self, "@numerator");
  n1 = rb_iv_get(other, "@numerator");
  d0 = rb_iv_get(self, "@denominator");
  d1 = rb_iv_get(other, "@denominator");

  return (RTEST(rb_equal(n0, n1)) && RTEST(rb_equal(d0, d1))) ? Qtrue : Qfalse;
}

static VALUE
rbgst_fraction_g2rvalue(const GValue *from) {
  VALUE args[2];
  args[0] = INT2NUM(gst_value_get_fraction_numerator(from));
  args[1] = INT2NUM(gst_value_get_fraction_denominator(from));
  return rb_class_new_instance(2, args, crbGstFractionClass);
}

static void
rbgst_fraction_r2gvalue(VALUE from, GValue *to) {
  gst_value_set_fraction(to, 
                         NUM2INT(rb_iv_get(from, "@numerator")),
                         NUM2INT(rb_iv_get(from, "@denominator")));
}

/* GstFourcc */
static VALUE
rbgst_fourcc_new(VALUE self, VALUE fourcc) {
  VALUE i;

  if (TYPE(fourcc) == T_STRING) { 
    char *fourccstr;
    fourccstr = RVAL2CSTR(fourcc);
    if (strlen(fourccstr) != 4) {
      rb_raise(rb_eArgError, "fourcc string should be 4 characters long");
    }
    i = UINT2NUM(GST_STR_FOURCC(fourccstr));
  } else {
    i = fourcc;
  }
  if (rb_obj_is_kind_of(i, rb_cNumeric) == Qfalse) {
    rb_raise(rb_eTypeError, "Not a String or Numeric"); 
  }
  rb_iv_set(self, "fourcc", i);
  return self;
}

static VALUE
rbgst_fourcc_to_i(VALUE self) {
  return rb_iv_get(self, "fourcc"); 
}

static VALUE
rbgst_fourcc_inspect(VALUE self) {
  return 
    rb_str_new_with_format("%"GST_FOURCC_FORMAT, 
      GST_FOURCC_ARGS(NUM2UINT(rb_iv_get(self, "fourcc"))));
}

static VALUE
rbgst_fourcc_is_equal(VALUE self, VALUE other) {
  if (NIL_P(other)) return Qfalse;

  return  rb_equal(rb_iv_get(self, "fourcc"), rb_iv_get(other, "fourcc")); 
}


static VALUE
rbgst_fourcc_g2rvalue(const GValue *from) {
  VALUE fourcc;
  
  fourcc = UINT2NUM(gst_value_get_fourcc(from));
  return rb_class_new_instance(1, &fourcc, crbGstFourccClass);
}

static void
rbgst_fourcc_r2gvalue(VALUE from, GValue *to) {
  gst_value_set_fourcc(to, NUM2UINT(rb_iv_get(from, "fourcc")));
}

/* GstRange */
static VALUE
rbgst_range_inspect(VALUE self) {
  VALUE result = rb_str_new2("<");
  result = rb_str_concat(result, rb_inspect(rb_iv_get(self, "@minimum")));
  result = rb_str_cat2(result, ",");
  result = rb_str_concat(result, rb_inspect(rb_iv_get(self, "@maximum")));
  result = rb_str_cat2(result, ">");

  return result;
}

static VALUE
rbgst_range_is_equal(VALUE self, VALUE other) {
  VALUE min0, min1, max0, max1;
  if (NIL_P(other)) return Qfalse;

  min0 = rb_iv_get(self, "@minimum");
  min1 = rb_iv_get(other, "@minimum");
  max0 = rb_iv_get(self, "@maximum");
  max1 = rb_iv_get(other, "@maximum");

  return (RTEST(rb_equal(min0, min1)) && 
          RTEST(rb_equal(max0, max1))) 
        ? Qtrue : Qfalse;
}

/* GstDoubleRange */
static VALUE
rbgst_double_range_new(VALUE self, VALUE minimum, VALUE maximum) {
  Check_Type(minimum, T_FLOAT);
  Check_Type(maximum, T_FLOAT);
  rb_iv_set(self, "@minimum", minimum);
  rb_iv_set(self, "@maximum", maximum);
  return self;
}

static VALUE
rbgst_double_range_g2rvalue(const GValue *from) {
  VALUE args[2];
  args[0] = rb_float_new(gst_value_get_double_range_min(from));
  args[1] = rb_float_new(gst_value_get_double_range_max(from)); 
  return rb_class_new_instance(2, args, crbGstDoubleRangeClass);
}

static void
rbgst_double_range_r2gvalue(VALUE from, GValue *to) {
  gst_value_set_double_range(to, 
                          NUM2DBL(rb_iv_get(from, "@minimum")),
                          NUM2DBL(rb_iv_get(from, "@maximum")));
}

/* GstFractionRange */
static VALUE
rbgst_fraction_range_new(VALUE self, VALUE minimum, VALUE maximum) {
  if (!rb_obj_is_kind_of(minimum, crbGstFractionClass)) {
    rb_raise(rb_eTypeError, "Minimum should be a Gst::Fraction");
  }
  if (!rb_obj_is_kind_of(maximum, crbGstFractionClass)) {
    rb_raise(rb_eTypeError, "Maximum should be a Gst::Fraction");
  }

  rb_iv_set(self, "@minimum", minimum);
  rb_iv_set(self, "@maximum", maximum);
  return self;
}

static VALUE
rbgst_fraction_range_g2rvalue(const GValue *from) {
  VALUE args[2];
  args[0] = GVAL2RVAL(gst_value_get_fraction_range_min(from));
  args[1] = GVAL2RVAL(gst_value_get_fraction_range_max(from));
  return rb_class_new_instance(2, args, crbGstFractionRangeClass);
}

static void
rbgst_fraction_range_r2gvalue(VALUE from, GValue *to) {
  VALUE min = rb_iv_get(from, "@minimum");
  VALUE max = rb_iv_get(from, "@maximum");
  gst_value_set_fraction_range_full(to, 
                                   NUM2INT(rb_iv_get(min, "@numerator")),
                                   NUM2INT(rb_iv_get(min, "@denominator")),
                                   NUM2INT(rb_iv_get(max, "@numerator")),
                                   NUM2INT(rb_iv_get(max, "@denominator")));
}

/* GstIntRange */
static VALUE
rbgst_int_range_new(VALUE self, VALUE minimum, VALUE maximum) {
  if (!rb_obj_is_kind_of(minimum, rb_cNumeric)) {
    rb_raise(rb_eTypeError, "Minimum should be a Integer");
  }
  if (!rb_obj_is_kind_of(maximum, rb_cNumeric)) {
    rb_raise(rb_eTypeError, "Maximum should be a Integer");
  }

  rb_iv_set(self, "@minimum", minimum);
  rb_iv_set(self, "@maximum", maximum);
  return self;
}

static VALUE
rbgst_int_range_g2rvalue(const GValue *from) {
  VALUE args[2];
  args[0] = INT2NUM(gst_value_get_int_range_min(from));
  args[1] = INT2NUM(gst_value_get_int_range_max(from));
  return rb_class_new_instance(2, args, crbGstIntRangeClass);
}

static void
rbgst_int_range_r2gvalue(VALUE from, GValue *to) {
  gst_value_set_int_range(to, 
                          NUM2INT(rb_iv_get(from, "@minimum")),
                          NUM2INT(rb_iv_get(from, "@maximum")));
}

/* RbGstValueArray */
static VALUE
rbgst_value_array_g2rvalue(const GValue *from) {
  int size = gst_value_array_get_size(from);
  int x;
  VALUE ret = rb_class_new_instance(0, NULL, crbGstArrayClass);
  
  for (x = 0 ; x < size ; x++) {
    rb_ary_store(ret, x, GVAL2RVAL(gst_value_array_get_value(from, x)));
  }
  return ret;
}

static void
rbgst_value_array_r2gvalue(VALUE from, GValue *to) {
  int size = RARRAY(from)->len;
  int x;

  for (x = 0 ; x < size ; x++) {
    VALUE v = rb_ary_entry(from, x);
    GValue *gv = g_new0(GValue, 1);

    g_value_init(gv, RVAL2GTYPE(v));
    rbgobj_rvalue_to_gvalue(v, gv);
    gst_value_array_append_value(to, gv);
  }
}

/* RbGstValueList */
static VALUE
rbgst_value_list_g2rvalue(const GValue *from) {
  int size = gst_value_list_get_size(from);
  int x;
  VALUE ret = rb_class_new_instance(0, NULL, crbGstListClass);
  
  for (x = 0 ; x < size ; x++) {
    rb_ary_store(ret, x, GVAL2RVAL(gst_value_list_get_value(from, x)));
  }
  return ret;
}

static void
rbgst_value_list_r2gvalue(VALUE from, GValue *to) {
  int size = RARRAY(from)->len;
  int x;

  for (x = 0 ; x < size ; x++) {
    VALUE v = rb_ary_entry(from, x);
    GValue *gv = g_new0(GValue, 1);

    g_value_init(gv, RVAL2GTYPE(v));
    rbgobj_rvalue_to_gvalue(v, gv);
    gst_value_list_append_value(to, gv);
  }
}

void
Init_gst_values (void)
{
  /* GstFraction */
  crbGstFractionClass = rb_define_class_under(mGst, "Fraction", rb_cObject); 
  rb_funcall(crbGstFractionClass, rb_intern("module_eval"), 1, 
             rb_str_new2("attr_reader :numerator, :denominator"));
  rb_define_method(crbGstFractionClass, "initialize", rbgst_fraction_new, 2);
  rb_define_method(crbGstFractionClass, "inspect", rbgst_fraction_inspect, 0);
  rb_define_method(crbGstFractionClass, "==", rbgst_fraction_is_equal, 1);

  rbgobj_register_g2r_func(GST_TYPE_FRACTION, rbgst_fraction_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_FRACTION, rbgst_fraction_r2gvalue);
  rbgobj_register_class(crbGstFractionClass, GST_TYPE_FRACTION, TRUE, TRUE);

  /* GstFourcc */
  crbGstFourccClass = rb_define_class_under(mGst, "Fourcc", rb_cObject); 
  rb_define_method(crbGstFourccClass, "initialize", rbgst_fourcc_new, 1);
  rb_define_method(crbGstFourccClass, "to_i", rbgst_fourcc_to_i, 0);
  rb_define_method(crbGstFourccClass, "inspect", rbgst_fourcc_inspect, 0);
  rb_define_method(crbGstFourccClass, "==", rbgst_fourcc_is_equal, 1);

  rbgobj_register_g2r_func(GST_TYPE_FOURCC, rbgst_fourcc_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_FOURCC, rbgst_fourcc_r2gvalue);
  rbgobj_register_class(crbGstFourccClass, GST_TYPE_FOURCC, TRUE, TRUE);


  /* GstRange (Doesn't exist in the C api, but gives some nice ruby sugar) */
  crbGstRangeClass = rb_define_class_under(mGst, "Range", rb_cObject); 
  rb_funcall(crbGstRangeClass, rb_intern("module_eval"), 1, 
             rb_str_new2("attr_reader :minimum, :maximum"));
  rb_define_method(crbGstRangeClass, "inspect", rbgst_range_inspect, 0);
  rb_define_method(crbGstRangeClass, "==", rbgst_range_is_equal, 1);


  /* GstDoubleRange */
  crbGstDoubleRangeClass = 
    rb_define_class_under(mGst, "DoubleRange", crbGstRangeClass); 
  rb_define_method(crbGstDoubleRangeClass, "initialize", 
                   rbgst_double_range_new, 2);

  rbgobj_register_g2r_func(GST_TYPE_DOUBLE_RANGE, rbgst_double_range_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_DOUBLE_RANGE, rbgst_double_range_r2gvalue);
  rbgobj_register_class(crbGstDoubleRangeClass, GST_TYPE_DOUBLE_RANGE, TRUE, TRUE);

  /* GstFractionRange */
  crbGstFractionRangeClass = 
    rb_define_class_under(mGst, "FractionRange", crbGstRangeClass); 
  rb_define_method(crbGstFractionRangeClass, "initialize", 
                   rbgst_fraction_range_new, 2);

  rbgobj_register_g2r_func(GST_TYPE_FRACTION_RANGE, rbgst_fraction_range_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_FRACTION_RANGE, rbgst_fraction_range_r2gvalue);
  rbgobj_register_class(crbGstFractionRangeClass, GST_TYPE_FRACTION_RANGE, TRUE, TRUE);

  /* GstIntRange */
  crbGstIntRangeClass = 
    rb_define_class_under(mGst, "IntRange", crbGstRangeClass); 
  rb_define_method(crbGstIntRangeClass, "initialize", 
                   rbgst_int_range_new, 2);

  rbgobj_register_g2r_func(GST_TYPE_INT_RANGE, rbgst_int_range_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_INT_RANGE, rbgst_int_range_r2gvalue);
  rbgobj_register_class(crbGstIntRangeClass, GST_TYPE_INT_RANGE, TRUE, TRUE);

  /* GstArray */
  crbGstArrayClass = rb_define_class_under(mGst, "Array", rb_cArray); 
  rbgobj_register_g2r_func(GST_TYPE_ARRAY, rbgst_value_array_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_ARRAY, rbgst_value_array_r2gvalue);
  rbgobj_register_class(crbGstArrayClass, GST_TYPE_ARRAY, TRUE, TRUE);

  /* GstList */
  crbGstListClass = rb_define_class_under(mGst, "List", rb_cArray); 
  rbgobj_register_g2r_func(GST_TYPE_LIST, rbgst_value_list_g2rvalue);
  rbgobj_register_r2g_func(GST_TYPE_LIST, rbgst_value_list_r2gvalue);
  rbgobj_register_class(crbGstListClass, GST_TYPE_LIST, TRUE, TRUE);

}
