/* filein.c io for QuickList */

/*  Quicklist
 *  www.quicklist.org for more information. 
 *
 *  Author  1999 Robert Lissner
 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "includes.h"
#include "globals.h"

#define QL "  (QuickList)"

 /* reads a file from disk, returns 0 if OK */
void any_open ();
static void define_last_field ();
gint16  read_ql_file (); 
gint16  read_comma_file (); 
gint16  read_tab_file ();   
static void ok_clicked (GtkWidget *w, GtkFileSelection *fs);
static void report_error (gint16 (err_number));
/* End of prototypes */


/* ________________________________
  |                                |
  |        actually_open           |
  |________________________________|
 */
/* This callback is when someone clicks OK on a file selection dialog box */
void actually_open (char* filename) {
  gint16 temp;
  gint16 i;
  char* linebufp;
  char temp_path_name [1024]; /* just to hold while we clear out */

  /* Be careful of the order of the next instructions */
  strcpy (temp_path_name, filename);
  
  /* save path name for future file selections, without file name */
  if (strlen (temp_path_name) < 126) {
    strcpy (last_path_name, temp_path_name);
    linebufp = (char *) strrchr (last_path_name, '/');
    if (linebufp)
      *++linebufp  = '\0';
  }

  /* Check for this file already open */
  for (i = 0; i < MAX_FILES; i++)
    if (back [i])
      if (!strcmp (back [i]->file_path, temp_path_name)) {
	level2_error ((char*) _("  This file is already open  "),
		      (char*) _("Go back"));
	return;
      }

  /* pull out the name of the file itself, without directories attached */ 
  front = (whole_file*) calloc (1, sizeof (whole_file));
  file_ct++;
  destroy_dialog (); 

 /* Add this window to table of open windows so I can list */
  for (i=0; i < MAX_FILES; i++)
    if (!back [i]) {
      back [i] = front;
      break;
    }

  front->activate_row = front->activate_col = -5;  /* 0,0 is valid, so can't */
  front->file_path = g_strdup (temp_path_name);
  linebufp = (char *) strrchr (front->file_path, '/');
  if (linebufp)
    front->file_name = g_strdup (linebufp +1);
  else /* happens with command line file name */
    front->file_name = g_strdup (front->file_path);

  temp = 1; /* default error */
  if ((f = fopen (front->file_path, "r"))) {
    if (file_type == 1) /* .qlf */
      temp = read_ql_file ();
    else if (file_type == 3) /* .csv */
      temp = read_comma_file (); 
    else
      temp = read_tab_file ();
    fclose (f);
  }
  else { /* this happens if command line file name is no good */
    get_rid_of_front ();
    maybe_create_initial_db ();
    return;
  }
  if (temp) {
    get_rid_of_front();
    report_error (temp);
    return;
  } 

  front->changed = FALSE;
  dim_all_menus ();
  gtk_widget_show_all (front->list_win);
  connect_signals ( );
} /* end of actual open */

/* ________________________________
  |                                |
  |          add1field             |
  |________________________________|
  Add 1 column to sheet.  This is necessary for reading unspecified input
  files such as comma, html and tab */
void add1field () {
  gint fieldx = ++front->last_field;
  define_last_field ();
  gtk_sheet_add_column (GTK_SHEET (front->sheet), 1);  
  gtk_sheet_set_column_width (GTK_SHEET (front->sheet), fieldx, 80);
  gtk_sheet_column_button_add_label( GTK_SHEET (front->sheet), fieldx,
				       front->fields[fieldx].name);
  gtk_sheet_set_column_title( GTK_SHEET (front->sheet), 
				fieldx, front->fields[fieldx].name);
} /* end of add1field */


/* ________________________________
  |                                |
  |       any_open                 |
  |________________________________|
 */
void any_open (){
  if (check_if_changed())
    return;
  destroy_dialog ();
  if (dialog_mode == 'O')
    dialog1_win = gtk_file_selection_new ((char*)  _("Open a list"));
  else 
    dialog1_win = gtk_file_selection_new ((char*)  _("Import a file"));

  gtk_window_position (GTK_WINDOW (dialog1_win), GTK_WIN_POS_CENTER); 
  if (strlen (last_path_name) > 1)
    gtk_file_selection_set_filename( GTK_FILE_SELECTION (dialog1_win),
				     last_path_name); 
  /* Connect the ok_button to file_ok_sel function */
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION
				  (dialog1_win)->ok_button),
		      "clicked", 
		      (GtkSignalFunc) ok_clicked, dialog1_win);
  
  /* Connect the cancel_button to destroy the widget */
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION
				  (dialog1_win)->cancel_button),
		      "clicked",
		      (GtkSignalFunc) cancel_level1, NULL);
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) cancel_level1, NULL);
  if (dialog_mode == 'I')
    file_type_menu ();
  gtk_widget_show_all (dialog1_win);  
} /* end of any_open callback */


/* |--------------------------------|
   |   build_basic_list_mode        |
   |--------------------------------|
    win inits, but don't assemble list mode here */
void build_basic_list_mode () {
  GtkSheet *sheet;
  gchar *temp_file_name;
  gint colx = 0;
  gint fieldx = 0;
  GtkWidget  *list_vbox;      /* BOX that fills the list window */
  GtkWidget  *list_mb;  /* widget inside menu_box */
  GtkWidget  *scrolled_window;  

  front->list_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  temp_file_name = g_strconcat (front->file_name, QL, NULL);
  gtk_window_set_title (GTK_WINDOW (front->list_win), temp_file_name);
  g_free (temp_file_name);


  gtk_window_position (GTK_WINDOW (front->list_win), GTK_WIN_POS_NONE);
  gtk_window_set_policy (GTK_WINDOW (front->list_win), TRUE, TRUE, FALSE);
  if (!front->width || !front->height
      || !front->x || !front->y ) { /* fix if zero */
    front->width = 600;
    front->height = 400;
    front->x = 100;
    front->y = 100; 
  }
  set_window_size_loc (front->list_win);


  list_vbox = gtk_vbox_new (FALSE, 3);
  list_mb = build_list_mb ();
  gtk_box_pack_start (GTK_BOX (list_vbox), list_mb ,
		      FALSE, TRUE, 0);
  
  /* make sheet and add to scroll_win */
  front->sheet = gtk_sheet_new (1, front->last_field + 1,
				temp_file_name);
  sheet = (GTK_SHEET (front->sheet));
  gtk_sheet_row_button_add_label (sheet, 0, " "); 
  GTK_SHEET_SET_FLAGS (sheet, 
		       /* GTK_SHEET_ADJUST_TEXT + */
		       GTK_SHEET_AUTO_SCROLL);
  GTK_SHEET_UNSET_FLAGS (sheet, GTK_SHEET_AUTORESIZE);
 
  gtk_sheet_set_row_titles_width (sheet, 20);
  front->last_row = 0;
  gtk_container_border_width (GTK_CONTAINER (sheet), 4);
  
  /* set sensitivity for all column and row buttons */
  gtk_sheet_columns_set_sensitivity (sheet, TRUE);
  gtk_sheet_show_row_titles (sheet);
  gtk_sheet_show_column_titles (sheet);
  gtk_sheet_rows_set_sensitivity  (sheet, TRUE);
  for ( fieldx = 0; fieldx <= front->last_field; fieldx++ ) {
    colx = front->fields [fieldx].sheet_column;
    gtk_sheet_set_column_width (sheet, colx, 
				front->fields [fieldx].width*8);
    gtk_sheet_column_button_add_label(sheet, colx, 
				      front->fields[fieldx].name);
    gtk_sheet_column_set_justification(sheet, colx,
				       front->fields [fieldx].justification);
  }
  gtk_sheet_show_column_titles (sheet); 
 
  /* create a new scrolled window. */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  
  gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 
				  3);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW 
				  (scrolled_window),
				  GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
				  
 
  gtk_container_add (GTK_CONTAINER (scrolled_window), front->sheet);
  gtk_box_pack_start (GTK_BOX (list_vbox), scrolled_window, 
		      TRUE, TRUE, 0); 
  gtk_container_add (GTK_CONTAINER (front->list_win), list_vbox);
  front->display_mode = 'L'; 
 
} /* end of build_basic_list_mode */


 /* ________________________________
  |                                |
  |        define_last_field       |
  |________________________________|
 setup a new text field as a standard text field.  */
static void define_last_field () {
  gint temp = front->last_field;
  g_snprintf(front->fields [temp].name, MAX_FIELD_NAME, "Column %u",
	   temp + 1);
  front->fields [temp].sheet_column = temp;
  front->fields [temp].type = FIELD_TYPE_TEXT;
  front->fields [temp].formatting = 0;
  front->fields [temp].decimal_places = 0;
  front->fields [temp].justification = GTK_JUSTIFY_LEFT;
  front->fields [temp].width = 10;
  front->col_to_field [temp] = temp;
}


/* ________________________________
  |                                |
  |        fileio_type_change      |
  |________________________________|
  This is the callback for the file type menu button in the file
  selection dialog box.  Change the suffix that is displayed */
void fileio_type_change (GtkWidget *w, gint data) {
  char temp_file [1024];
  file_type = data; 
  if (dialog_mode != 'E') /* change suffix for output only */
    return;
  strcpy (temp_file, gtk_file_selection_get_filename
	  (GTK_FILE_SELECTION (dialog1_win)));
  set_suffix (temp_file, file_type);
  gtk_file_selection_set_filename( GTK_FILE_SELECTION (dialog1_win),
				     temp_file); 
}

void file_import (GtkWidget *w, gpointer *data) {
  dialog_mode = 'I';
  file_type = 3; /* csv */
  any_open ();
}


/* ________________________________
  |                                |
  |          file_type_menu        |
  |________________________________|
  Build the menu button for the file selection box, allowing choice
  of file types.  Works for input and output */
void file_type_menu () {
  GtkWidget *type_button,
            *menu,
            *menu_item,
            *name_label,
            *type_hbox;

  menu = gtk_menu_new ();
 
  if (dialog_mode == 'E') {
    menu_item = gtk_menu_item_new_with_label ("HTML");
    gtk_widget_show (menu_item);
    gtk_signal_connect (GTK_OBJECT (menu_item),
			"select",
			GTK_SIGNAL_FUNC (fileio_type_change),(gpointer) 2);
    gtk_menu_append (GTK_MENU(menu), menu_item);
  } 

  menu_item = gtk_menu_item_new_with_label ("Comma delimited");
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),
		      "select",
		      GTK_SIGNAL_FUNC (fileio_type_change),(gpointer) 3);
  gtk_menu_append (GTK_MENU(menu), menu_item);

  menu_item = gtk_menu_item_new_with_label ("Tab delimited");
  gtk_widget_show (menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item),
		      "select",
		      GTK_SIGNAL_FUNC (fileio_type_change),(gpointer) 4);
   gtk_menu_append (GTK_MENU(menu), menu_item); 
  
  type_button = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (type_button), menu);
  type_hbox = gtk_hbox_new (FALSE, 5);

  /* Pack Widgets into boxes */
  name_label = gtk_label_new (_("Type of file"));

  /* keep to right so still can see file name */
  gtk_box_pack_end (GTK_BOX (type_hbox), type_button, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (type_hbox), name_label, FALSE, FALSE, 0);

  /* Add file type box to the top_box */
  gtk_container_add (GTK_CONTAINER (GTK_FILE_SELECTION
				    (dialog1_win)->main_vbox), type_hbox);
}


/* ________________________________
  |                                |
  |        ok_clicked              |
  |________________________________|
  This callback is when someone clicks OK on a file selection dialog box */
static void ok_clicked (GtkWidget *w, GtkFileSelection *fs) {
  actually_open (gtk_file_selection_get_filename
	  (GTK_FILE_SELECTION (fs))); 
}

/* ________________________________
  |                                |
  |        open_file               |
  |________________________________|
 */
/* This is a callback that handles open file from various places */
void open_file (GtkWidget *w, gpointer *data){
  dialog_mode = 'O';
  file_type = 1; /* qlf */
  any_open ();
}


/* |--------------------------------|
   |    read_ql_file                |
   |--------------------------------|
*/  
/* The file selection dialog box has returned a file name.  Get it,
   open it, load it into a clist, open the window.  Build all window
   window components, but do not assemble them.  That is done by
   build_file_win ()
   Returns 0 if OK, negative if not OK */
gint16 read_ql_file () {
  char linebuf [4096];
  char tokbuf [256];
  char* tokbufp;
  char* linebufp; 
  char* prevbufp;
  gint16 fieldx = 0;
  gint16 sortx = 0;
  gint16 filterx = 0;
  gint16 subx = 0;
  gint rowx;
  gboolean got_data;
  gint16 reportx = 0;
  gint16 colx = 0; /* indexes into array */
  gint16 thislen, i, count;

  char singlechar;
  gint16 temp_field, temp_type;

  while (fgets (linebuf, sizeof (linebuf), f)) {
    linebufp = linebuf;
    if (sscanf (linebuf, "%s", tokbuf) != 1)
      return (2);
  
   /* process QUICKLIST header record */
    if (strcmp (tokbuf, "QUICKLIST") == 0) {
      if (sscanf (linebuf, "%*s%*s%*s%hu%hu%hu%hu%hu%hu%hu%hu", 
		  &front->last_field,
		  &front->sort_ct, 
		  &front->filter_ct, 
		  &front->report_ct,
		  &front->width, 
		  &front->height,
		  &front->x,
		  &front->y) < 6)
	return (3);
      front->last_field--;
      if (    front->last_field < 0 
	      || front->last_field  >= MAX_FIELDS
	      || front->sort_ct   > MAX_SORTS
	      || front->filter_ct > MAX_FILTERS
	      || front->report_ct > MAX_REPORTS)
	return (4);
     }
    /* process FIELD input header record */
    else if (strcmp (tokbuf, "FIELD") == 0) {
      if (fieldx > front->last_field)
	return (5);
      if (sscanf (linebuf, "%*s%hu%hu%hu%hu%hu%hu%hu%hu%40[^\n\\]", 
		  &front->fields[fieldx].type,
		  &front->fields[fieldx].formatting, 
		  &front->fields[fieldx].decimal_places,
		  &front->fields[fieldx].justification,
		  &front->fields[fieldx].sheet_column,
		  &front->fields[fieldx].width,
		  &front->fields[fieldx].unused1,
		  &front->fields[fieldx].unused2,
		  tokbuf) != 9)
	return (6);
      tokbufp = tokbuf;
      while (*tokbufp == ' ') /* remove leading spaces */
	tokbufp++;
      if (front->fields[fieldx].type < FIELD_TYPE_TEXT
	  || front->fields[fieldx].type > FIELD_TYPE_TIME
	  || front->fields[fieldx].formatting > 12 
	  || front->fields[fieldx].decimal_places > 6
	  || front->fields[fieldx].justification > GTK_JUSTIFY_CENTER 
	  || front->fields[fieldx].sheet_column > front->last_field 
	  || front->fields[fieldx].width > 80)
	return (7);

      /* temporary routine because of bug in sheet
      if (front->fields[fieldx].justification == GTK_JUSTIFY_CENTER) 
      front->fields[fieldx].justification = GTK_JUSTIFY_LEFT; */

      thislen = strlen (tokbufp);
      if (thislen < 1 || thislen > MAX_FIELD_NAME)
	return (8);
      strcpy (front->fields[fieldx].name, tokbufp);
      fieldx++;
    }
    /* convert the SORT record to memory */

    else if (strcmp (tokbuf, "SORT") == 0) {
      if (sortx >= front->sort_ct)
	return (9);
      count = sscanf (linebuf, 
		  "%*s%hu%hu%40[^\n\\]%c%hu%u%hu%u%hu%u%hu%u%hu%u%hu%u",
		      &front->sorts[sortx].unused1,
		      &front->sorts[sortx].unused2,
		      tokbuf,
		      &singlechar,
		      &front->sorts[sortx].line[0].field,
		      &front->sorts[sortx].line[0].ascending,
		      &front->sorts[sortx].line[1].field,
		      &front->sorts[sortx].line[1].ascending,
		      &front->sorts[sortx].line[2].field,
		      &front->sorts[sortx].line[2].ascending,
		      &front->sorts[sortx].line[3].field,
		      &front->sorts[sortx].line[3].ascending,
		      &front->sorts[sortx].line[4].field,
		      &front->sorts[sortx].line[4].ascending,
		      &front->sorts[sortx].line[5].field,
		      &front->sorts[sortx].line[5].ascending);
      if (count < 6)
	return (10);
      front->sorts[sortx].line_ct = (count - 4) / 2;
      /* number of sort lines read */
      tokbufp = tokbuf;
      while (*tokbufp == ' ') /* remove leading spaces */
	tokbufp++;
      thislen = strlen (tokbufp);
      if (thislen < 1 
	  || thislen > MAX_FIELD_NAME
	  || singlechar != '\\')
	return (11);
      strcpy (front->sorts[sortx].name, tokbufp);
      for (i = 0; i < MAX_SORT_NESTING; i++)
	if (front->sorts[sortx].line[i].field >= front->last_field + 1)
	  return (12);      
      sortx++;
    }
    /* now store a FILTER record, the most complicated record */
    else if (strcmp (tokbuf, "FILTER") == 0) {    
      if (filterx >= front->filter_ct)
	return (13);
      if (sscanf (linebuf, "%*s%d%d%40[^\n\\]%c%hn",
		  &front->filters[filterx].by_and,
		  &front->filters[filterx].use_nocase,
		  tokbuf,
		  &singlechar,
		  &count) != 4) {
	return (14);
      }
      tokbufp = tokbuf;
      while (*tokbufp == ' ') /* remove leading spaces */
	tokbufp++;
      thislen = strlen (tokbufp);
      if (thislen < 1 
	  || thislen > MAX_FIELD_NAME
	  || singlechar != '\\')
	return (15);
      strcpy (front->filters[filterx].name, tokbufp);
      subx = 0;

      /* now loop through the lines of the filter */
     linebufp = linebuf + count;
     while (TRUE) {
       if (sscanf(linebufp, "%hu%hu%hn", &temp_field,
		  &temp_type, &count) < 2)
	 return (16);
       linebufp += count;
       tokbuf [0] = '\0';

       while (*linebufp == ' ')
	 linebufp++;
       if (*linebufp != '\n' && *linebufp != '\\' && *linebufp != '\0') {
	 sscanf(linebufp, "%40[^\\\n]%hn", tokbuf, &count);
	 linebufp += count;
       }
       
       if (temp_field >  front->last_field
	   || temp_type < 0
	   || temp_type > 7) /* read and verify field and type */
	  return (17);	  
       front->filters[filterx].line[subx].field = temp_field;
       front->filters[filterx].line[subx].type = temp_type;
 
       tokbufp = tokbuf;
       if (strlen (tokbufp) > MAX_FILTER_COMPARE)
	 return (19);
       strcpy (front->filters[filterx].line[subx].compare,
	       tokbufp);
       subx++;
     
     /* now look for \ before next rule, or \n to end it */
     sscanf (linebufp, "%c", &singlechar);
     if (singlechar == '\n' || !singlechar)
       break; /* no more data */
     if (singlechar != '\\')
       return (20);
     linebufp++;
     if (subx >= MAX_FILTER_NESTING)
       return (21);
     
     } /* End of loop that scans filter lines */
     
     /* Break comes here */
     front->filters[filterx].line_ct = subx;
     filterx++;
    }
    
    /* store a report column */
    else if (strcmp (tokbuf, "COLUMN") == 0) {
      if (colx >= front->last_field + 1)
	return (22);
      if (sscanf (linebuf, 
		  "%*s%hu%hu%hu%hu%hu%hu",
		  &front->reports[reportx].column[colx].field,
		  &front->reports[reportx].column[colx].width,
		  &front->reports[reportx].column[colx].group,
		  &front->reports[reportx].column[colx].total,
		  &front->reports[reportx].column[colx].unused1,
		  &front->reports[reportx].column[colx].unused2) < 6)
	return (23);
      if (front->reports[reportx].column[colx].field >= 
	  front->last_field +1)
	return (24);      
      colx++;     
    }
    /* store a report record */
    else if (strcmp (tokbuf, "REPORT") == 0) {
      if (reportx >= front->report_ct
	  || colx == 0) /* never got any column records */
	return (25);
      if (sscanf (linebuf, 
		  "%*s%hd%hd%hd%hd%40[^\n\\]%c",
		  &front->reports[reportx].sort,
		  &front->reports[reportx].filter,
		  &front->reports[reportx].width,
		  &front->reports[reportx].height,
		  tokbuf,
		  &singlechar) < 6)
	return (26);
      tokbufp = tokbuf;
      while (*tokbufp == ' ') /* remove leading spaces */
	tokbufp++;
      thislen = strlen (tokbufp);
       if (thislen < 1 || thislen > MAX_FIELD_NAME)
	return (27);
      strcpy (front->reports[reportx].name, tokbufp);
      front->reports[reportx].last_column = colx - 1; 
      reportx++;
      colx = 0; /* reset the report column indicator */
    }

    else if (strcmp (tokbuf, "DATA") == 0) 
      break;
    else return (45);
  }
  
  /* Really ought to check file parameters about right here */
  if (colx != 0
      || fieldx  != front->last_field +1
      || sortx   != front->sort_ct
      || filterx != front->filter_ct
      || reportx != front->report_ct)
    return (28);

  /* fields are all OK so go build the basic window and sheet */
  reset_col_to_field ();
  build_basic_list_mode ();
  
  /* The file is already open, now read it into the clist 
     Make some effort to not load entirely blank records */
  while (fgets (linebuf, sizeof (linebuf), f)) {
    fieldx = 0;
    got_data = FALSE;
    linebufp = linebuf;   
    prevbufp = linebuf;
    rowx = front->last_row;
    singlechar = ' ';

    while (singlechar != '\n') {
      while (*linebufp != '\n' && *linebufp != '\\')
	linebufp++;
      
      singlechar = *linebufp;
      
      if (linebufp != prevbufp) { /* ie, not a zero length field */
	if (fieldx >  front->last_field) 
	  return (30);
	
        *linebufp  = '\0'; /* to terminate string move */
	got_data = TRUE;
	gtk_sheet_set_cell_text (GTK_SHEET (front->sheet), rowx,
				   front->fields [fieldx].sheet_column,
				   prevbufp);
      }
      
      prevbufp = ++linebufp;
      fieldx++;
    }
  
    if (got_data)
      add1row ();
  }

  return (0);
} /* end of read_ql_file */

  
/* |--------------------------------|
   |    read_comma_file             |
   |--------------------------------|
*/  
/* The file selection dialog box has returned a file name.  It is supposed
   to be a comma delimited file */
gint16 read_comma_file () {
  char linebuf [256];
  char* linebufp; 
  char* beginp;
  char* endp;
  gint16 fieldx = 0;
  gint rowx;
  int c;
  gboolean got_data;

  /* Make Column 1 in field [0] */
  define_last_field ();
  build_basic_list_mode ();
  linebufp = beginp = linebuf;
  rowx = fieldx = 0;
  endp = linebufp + 253;;
  got_data = FALSE;

  while ((c = getc (f)) != EOF) {
    if (c >= ' ' && c != ',') {
      *linebufp++ = c;
      if (linebufp >= endp) /* line is too big */
	return (45);
    } else if (c == '\n' || c == '\r' || c == ',') {
      if (linebufp != beginp) {  /* ie, there was data */
	got_data = TRUE;
	*linebufp = '\0';
	while (fieldx >  front->last_field) {
	  if (fieldx >= MAX_FIELDS - 1)
	    return (30);
	  else 
	    add1field ();
	}
	gtk_sheet_set_cell_text  (GTK_SHEET (front->sheet), rowx,
				  fieldx, linebuf);
	linebufp = beginp;
      } /* end of storing data */
      
      fieldx++;
      if (c != ',') {  /* process end of line or return */
	if (got_data) {
	  rowx++;
	  add1row ();
	}
	fieldx = 0;
	got_data = FALSE;
      }
    }
  }
  return (0);
} /* end of read_comma_file */


/* |--------------------------------|
   |    read_tab_file               |
   |--------------------------------|
   Read tab delimited files */

gint16 read_tab_file () {
  char linebuf [256];
  char* linebufp; 
  char* beginp;
  char* endp;
  gint16 fieldx = 0;
  gint rowx;
  int c;
  gboolean got_data;

  /* Make Column 1 in field [0] */
  define_last_field ();
  build_basic_list_mode ();
  linebufp = beginp = linebuf;
  rowx = fieldx = 0;
  endp = linebufp + 253;;
  got_data = FALSE;

  while ((c = getc (f)) != EOF) {
    if (c >= ' ') {
      *linebufp++ = c;
      if (linebufp >= endp) /* line is too big */
	return (45);
    }
    else if (c == '\n' || c == '\r' || c == '\t') {
      if (linebufp != beginp) {  /* ie, there was data */
	got_data = TRUE;
	*linebufp = '\0';
	while (fieldx >  front->last_field) {
	  if (fieldx >= MAX_FIELDS - 1)
	    return (30);
	  else 
	    add1field ();
	}
	gtk_sheet_set_cell_text  (GTK_SHEET (front->sheet), rowx,
				  fieldx, linebuf);
	linebufp = beginp;
      } /* end of storing data */
      
      fieldx++;
      if (c != '\t') {  /* process end of line or return */
	if (got_data) {
	  rowx++;
	  add1row ();
	}
	fieldx = 0;
	got_data = FALSE;
      }
    }
  }
  return (0);
} /* end of read_tab_file */
 
/* ________________________________
  |                                |
  |        report_error            |
  |________________________________|
 */
static void report_error (gint16 (err_number)) {
  char strbuf [90];
  g_snprintf(strbuf, sizeof (strbuf), 
     "   Error number %u.  ", err_number);
  strcat (strbuf, 
	  "This file is apparently not the type that you selected.  ");    
  level1_error (strbuf, (char*) "Go back");
}

/* ________________________________
  |                                |
  |       set_suffix               |
  |________________________________|
 Routine puts .qlf or .csv or whatever on end of filename.  
 Be careful: this function
 is counting on there being room at the end of the string for four or 5
 more bytes  */
void set_suffix (char* file_path, gint local_type) {
  size_t x;
  char* linebufp;

 /* Strip off any suffix that's there now */
  x = strlen (file_path);    
  linebufp = (char *) strrchr (file_path, '.');
  if (linebufp  && linebufp - file_path > x - 6)
    *linebufp  = '\0';

  if (local_type == 2)
    strcat (file_path, ".html");
  else if (local_type == 3)
    strcat (file_path, ".csv");
  else if (local_type == 4)
    strcat (file_path, ".tsv");
  else
    strcat (file_path, ".qlf");
}

