

/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: CashFlows.i,v 1.6 2002/01/16 14:51:30 nando Exp $

#ifndef quantlib_cash_flows_i
#define quantlib_cash_flows_i

%include Date.i
%include Types.i
%include Calendars.i
%include DayCounters.i
%include Indexes.i
%include TermStructures.i
%include Vectors.i

%{
    using QuantLib::Handle;
    using QuantLib::CashFlow;
    using QuantLib::CashFlows::SimpleCashFlow;
    using QuantLib::CashFlows::FixedRateCoupon;
    using QuantLib::CashFlows::FloatingRateCoupon;
    typedef Handle<CashFlow> CashFlowHandle;
    typedef Handle<SimpleCashFlow> SimpleCashFlowHandle;
    typedef Handle<FixedRateCoupon> FixedRateCouponHandle;
    typedef Handle<FloatingRateCoupon> FloatingRateCouponHandle;
%}

%name(CashFlow) class CashFlowHandle {
  private:
    // abstract class - inhibit construction
    CashFlowHandle();
  public:
    // constructor redefined below
    ~CashFlowHandle();
};

// replicate interface
%addmethods CashFlowHandle {
    void crash() {}
    double amount() {
        return (*self)->amount();
    }
    Date date() {
        return (*self)->date();
    }
}


// implementations

// fake inheritance between handles
%name(SimpleCashFlow) class SimpleCashFlowHandle : public CashFlowHandle {
  public:
    // constructor redefined below
    ~SimpleCashFlowHandle();
};

%addmethods SimpleCashFlowHandle {
    void crash() {}
    SimpleCashFlowHandle(double amount, Date date) {
        return new SimpleCashFlowHandle(
            new SimpleCashFlow(amount,date));
    }
}


%name(FixedRateCoupon) class FixedRateCouponHandle : public CashFlowHandle {
  public:
    // constructor redefined below
    ~FixedRateCouponHandle();
};

%addmethods FixedRateCouponHandle {
    void crash() {}
    FixedRateCouponHandle(double nominal, Rate rate,
      Calendar calendar, RollingConvention convention,
      DayCounter dayCounter, Date startDate, Date endDate,
      Date refPeriodStart, Date refPeriodEnd) {
        return new FixedRateCouponHandle(
            new FixedRateCoupon(nominal, rate, calendar, convention,
                dayCounter, startDate, endDate, refPeriodStart,
                refPeriodEnd));
    }
}


%name(FloatingRateCoupon)
class FloatingRateCouponHandle : public CashFlowHandle {
  public:
    // constructor redefined below
    ~FloatingRateCouponHandle();
};

%addmethods FloatingRateCouponHandle {
    void crash() {}
    FloatingRateCouponHandle(double nominal, XiborHandle index,
      TermStructureRelinkableHandle termStructure,
      Date startDate, Date endDate, int fixingDays, Spread spread,
      Date refPeriodStart, Date refPeriodEnd) {
        return new FloatingRateCouponHandle(
            new FloatingRateCoupon(nominal, index,
                termStructure, startDate, endDate, fixingDays, spread,
                refPeriodStart, refPeriodEnd));
    }
}


// convert array of cash flows to std::vector<Handle<CashFlow> >

%{
typedef std::vector<CashFlowHandle> CashFlowHandleVector;
%}

%name(CashFlowVector) class CashFlowHandleVector {
    %pragma(ruby) include = "Enumerable";
  public:
    ~CashFlowHandleVector();
};

%typemap(ruby,in) VALUE {
    $target = $source;
};

%addmethods CashFlowHandleVector {
    void crash() {}
    CashFlowHandleVector(VALUE v) {
    	if (rb_obj_is_kind_of(v,rb_cArray)) {
            int size = RARRAY(v)->len;
            CashFlowHandleVector* temp = new CashFlowHandleVector(size);
            for (int i=0; i<size; i++) {
                VALUE o = RARRAY(v)->ptr[i];
                CashFlowHandle* h;
                Get_CashFlowHandle(o,h);
                (*temp)[i] = *h;
            }
            return temp;
        } else {
            rb_raise(rb_eTypeError,
                "wrong argument type (expected array)");
        }
    }
    int __len__() {
        return self->size();
    }
    CashFlowHandle __getitem__(int i) {
        if (i>=0 && i<self->size()) {
            return (*self)[i];
        } else if (i<0 && -i<=self->size()) {
            return (*self)[self->size()+i];
        } else {
            throw IndexError("CashFlowVector index out of range");
        }
        QL_DUMMY_RETURN(CashFlowHandle())
    }
    void __setitem__(int i, const CashFlowHandle& x) {
        if (i>=0 && i<self->size()) {
            (*self)[i] = x;
        } else if (i<0 && -i<=self->size()) {
            (*self)[self->size()+i] = x;
        } else {
            throw IndexError("CashFlowVector index out of range");
        }
    }
    void each() {
        for (int i=0; i<self->size(); i++) {
            CashFlowHandle* h = new CashFlowHandle((*self)[i]);
            rb_yield(Wrap_CashFlowHandle(cCashFlow,h));
        }
    }
};





// cash flow vector builders

%{
using QuantLib::CashFlows::FixedRateCouponVector;
using QuantLib::CashFlows::FloatingRateCouponVector;
%}

class FixedRateCouponVector : public CashFlowHandleVector {
  public:
    // constructor redefined below
    ~FixedRateCouponVector();
};

%addmethods FixedRateCouponVector {
    void crash() {}
    FixedRateCouponVector(DoubleVector nominals, DoubleVector couponRates,
      Date startDate, Date endDate, int frequency, Calendar calendar,
      RollingConvention convention, bool isAdjusted, DayCounter dayCount) {
        return new FixedRateCouponVector(nominals, couponRates,
            startDate, endDate, frequency, calendar, convention,
            isAdjusted, dayCount, dayCount);
    }
}

class FloatingRateCouponVector : public CashFlowHandleVector {
  public:
    // constructor redefined below
    ~FloatingRateCouponVector();
};

%addmethods FloatingRateCouponVector {
    void crash() {}
    FloatingRateCouponVector(DoubleVector nominals,
      Date startDate, Date endDate,
      int frequency, Calendar calendar,
      RollingConvention convention,
      TermStructureRelinkableHandle termStructure,
      XiborHandle index, int indexFixingDays, DoubleVector spreads) {
        return new FloatingRateCouponVector(nominals,
            startDate, endDate, frequency, calendar,
            convention, termStructure, index,
            indexFixingDays, spreads);
    }
}


#endif
