/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#ifndef GIT_H
#define GIT_H

#include "exceptionmanager.h"
#include "common.h"
#include "lanes.h"
#include "myprocess.h"

class Cache;
class Annotate;
class MyProcess;
class QTextCodec;
class DataLoader;
class QRegExp;

class FileHistory {
public:
	FileHistory() {
		revs.setAutoDelete(true);
		reset("");
		revs.resize(QGit::MAX_DICT_SIZE);
	}
	void reset(SCRef name) {
		revs.clear();
		revOrder.clear();
		firstFreeLane = 0;
		lns.clear();
		fileName = name;
	}
	QString fileName;
	RevMap revs;
	Lanes lns;
	StrVect revOrder;
	uint firstFreeLane;
};

class Git : public QObject {
Q_OBJECT
public:
	Git(QWidget* par = 0, const char* name = 0);
	const QString getBaseDir(bool* c, SCRef wd, bool* ok = NULL, QString* gd = NULL);
	bool init(SCRef workDir, bool askForRange, QStringList* filterList, bool* quit);
	bool stop(bool saveCache);
	void setLane(SCRef sha, FileHistory* fh);
	Annotate* startAnnotate(FileHistory* fh, QObject* guiObj);
	const FileAnnotation* lookupAnnotation(Annotate* ann, SCRef fileName, SCRef sha);
	void cancelAnnotate(Annotate* ann);
	void startFileHistory(FileHistory* fh);
	void cancelDataLoading(const FileHistory* fh);
	void cancelProcess(MyProcess* p);
	bool isCommittingMerge(); // check for existence of MERGE_HEAD
	bool isStGITStack();
	bool isTagName(SCRef tag);
	bool isPatchName(SCRef patchName);
	bool isUnapplied(SCRef sha);
	bool isSameFiles(SCRef tree1Sha, SCRef tree2Sha);
	bool isNothingToCommit() const { return nothingToCommit; }
	bool isUnknownFiles() const { return unknownFiles; }
	bool isTextHighlighter() const { return isTextHighlighterFound; }
	const Rev* revLookup(SCRef sha, const FileHistory* fh = NULL);
	MyProcess* getDiff(SCRef sha, QObject* receiver, SCRef diffToSha, bool combined);
	MyProcess* getFile(SCRef file, SCRef revSha, QObject* receiver, QString* runOutput);
	MyProcess* getHighlightedFile(SCRef file, SCRef revSha, QObject* rcv, QString* ro);
	bool saveFile(SCRef fileName, SCRef sha, SCRef path);
	void getFileFilter(SCRef path, QMap<QString, bool>& shaMap);
	bool getPatchFilter(SCRef exp, bool isRegExp, QMap<QString, bool>& shaMap);
	const RevFile* getFiles(SCRef sha, SCRef sha2 = "", bool all = false, SCRef path = "");
	void getTree(SCRef ts, SList nm, SList sha, SList type, bool wd, SCRef treePath);
	const QString getLocalDate(SCRef gitDate);
	const QString getDesc(SCRef sha, QRegExp& shortLogRE, QRegExp& longLogRE);
	const QString getDefCommitMsg();
	const QString getLaneParent(SCRef fromSHA, uint laneNum);
	const QStringList getChilds(SCRef parent);
	const QStringList getNearTags(bool goDown, SCRef sha);
	const QStringList getDescendantBranches(SCRef sha);
	const QString getShortLog(SCRef sha);
	const QStringList getTagNames(bool onlyLoaded = false);
	const QStringList getBranchNames();
	const QString getTagMsg(SCRef sha);
	const QString getRefSha(SCRef refName);
	const QString getRevInfo(SCRef sha, FileHistory* fh);
	void getWorkDirFiles(const QChar& status, SList files, SList dirs);
	QTextCodec* getTextCodec(bool* isGitArchive);
	bool formatPatch(SCList shaList, SCRef dirPath, SCRef remoteDir = "");
	bool updateIndex(SCList selFiles);
	bool commitFiles(SCList files, SCRef msg);
	bool makeTag(SCRef sha, SCRef tag, SCRef msg);
	bool deleteTag(SCRef sha);
	bool applyPatchFile(SCRef patchPath, bool commit, bool fold, bool sign);
	bool resetCommits(int parentDepth);
	bool stgCommit(SCList selFiles, SCRef msg, SCRef patchName, bool fold);
	bool stgPush(SCRef sha);
	bool stgPop(SCRef sha);
	bool writeToFile(SCRef fileName, SCRef data);
	bool readFromFile(SCRef fileName, QString& data);
	void setTextCodec(QTextCodec* tc);
	void addExtraFileInfo(QString* rowName, SCRef sha, SCRef diffToSha, bool allMergeFiles);
	void removeExtraFileInfo(QString* rowName);
	void formatPatchFileHeader(QString* rowName, SCRef sha, SCRef dts, bool cmb, bool all);
	int findFileIndex(const RevFile& rf, SCRef name);
	const QString getPatchName(SCRef sha) const { return patchNames[sha];	}
	const QString filePath(const RevFile& rf, uint i) const {

		return dirNamesVec[rf.dirs[i]] + fileNamesVec[rf.names[i]];
	}
	void incRunningProcesses() { runningProcesses++; }
	void decRunningProcesses() { runningProcesses--; }

	enum { // used as self-documenting boolean parameters
		optFalse,
		optSaveCache,
		optGoDown,
		optOnlyLoaded,
		optSign,
		optFold,
		optOnlyInIndex
	};

signals:
	void newRevsAdded(const FileHistory*, const QValueVector<QString>&);
	void loadCompleted(const FileHistory*, const QString&);
	void cancelLoading(const FileHistory*);
	void cancelAllProcesses();
	void annotateReady(Annotate*, const QString&, bool, const QString&);

public slots:
	void on_procDataReady(const QString&);
	void on_eof() { filesLoadingPending = filesLoadingCurSha = ""; }

private slots:
	void loadFileNames();
	void on_runAsScript_eof();
	void on_getHighlightedFile_eof();
	void on_newDataReady(const FileHistory*);
	void on_loaded(const FileHistory*, ulong,int,bool,const QString&,const QString&);
private:
	friend class Annotate;
	friend class MainImpl;
	friend class DataLoader;
	friend class ConsoleImpl;

	typedef QMap<QString, QString> StrMap; // used to map SHA and patches file names
	typedef QMap<QString, int> StrSet;     // for fast QString set lookup

	bool run(SCRef cmd, QString* out = NULL, QObject* rcv = NULL, SCRef buf = "");
	MyProcess* runAsync(SCRef cmd, QObject* rcv, SCRef buf = "");
	MyProcess* runAsScript(SCRef cmd, QObject* rcv = NULL, SCRef buf = "");
	const QString getArgs(QString* range, bool askForRange, bool* quit);
	bool getRefs();
	bool getStGITPatches();
	bool lookUpPatchesSHA(SCRef pn, SList files, SList filesSHA, QString& sha);
	void dirWalker(SCRef dirPath, SList files, SList filesSHA, SCRef nameFilter = "");
	void clearRevs();
	bool startRevList(SCRef args, FileHistory* fh = NULL);
	bool startUnappliedList();
	bool startParseProc(SCRef initCmd, FileHistory* fh);
	void addChunk(FileHistory* fh, SCRef parseBuffer);
	void parseDiffFormat(RevFile& rf, SCRef buf);
	void parseDiffFormatLine(RevFile& rf, SCRef line, int parNum);
	void getDiffIndex();
	const QString getFileSha(SCRef file, SCRef revSha);
	const Rev* fakeWorkDirRev(SCRef parent, SCRef log, SCRef longLog, int idx);
	void copyDiffIndex(FileHistory* fh);
	const RevFile* getAllMergeFiles(const Rev* r);
	bool isParentOf(SCRef par, SCRef child);
	bool isTreeModified(SCRef sha);
	void indexTree();
	void annotateExited(Annotate* ann);
	void updateDescMap(const Rev* r, uint i, QMap<QPair<uint, uint>,bool>& dm,
	                   QMap<uint, QValueVector<int> >& dv);
	void mergeNearTags(bool down, Rev* p, const Rev* r, const QMap<QPair<uint, uint>, bool>&dm);
	void mergeBranches(Rev* p, const Rev* r);
	void updateRefs(Rev& c, FileHistory* fh);
	void updateLanes(Rev& c, Lanes& lns);
	void removeFiles(SCList selFiles, SCRef workDir, SCRef ext);
	void restoreFiles(SCList selFiles, SCRef workDir, SCRef ext);
	bool mkPatchFromIndex(SCRef msg, SCRef patchFile);
	const QStringList getOthersFiles();
	const QStringList getOtherFiles(SCList selFiles, bool onlyInIndex);
	const QString colorMatch(SCRef txt, QRegExp& regExp);
	void appendFileName(RevFile& rf, SCRef name);
	void populateFileDict();
	const QStringList noSpaceSepHack(SCRef cmd);
	void removeDeleted(SCList selFiles);
	void setStatus(RevFile& rf, SCRef stInfo, int parNum);

	EM_DECLARE(exGitStopped);

	QWidget* par;
	Cache* cache;
	QString workDir; // workDir is always without trailing '/'
	QString gitDir;
	QStringList tags;
	QStringList loadedTagNames;
	QStringList loadedBranchNames;
	QStringList tagsSHA;
	StrMap tagsObj;
	StrMap refsStillToFind; // to quick check if a rev is a ref during loading
	StrMap refsStillToFindMasterCopy; // to restore original content for next search
	QStringList heads;
	QStringList headsSHA;
	QStringList refs;
	QStringList refsSHA;
	QStringList unAppliedSHA;
	QStringList appliedSHA;
	StrMap patchNames;
	QString currentBranchSHA;
	QString filesLoadingPending;
	QString filesLoadingCurSha;
	QString curRange;
	bool cacheNeedsUpdate;
	bool errorReportingEnabled;
	bool isMergeHead;
	bool isStGIT;
	bool isGIT;
	bool isTextHighlighterFound;
	bool loadingUnAppliedPatches;
	bool unknownFiles;
	bool nothingToCommit;
	int runningProcesses;
	QString firstNonStGitPatch;
	RevMap revs;
	RevFileMap revsFiles;
	Lanes lns;
	int patchesStillToFind;
	StrVect fileNamesVec;
	StrVect dirNamesVec;
	StrSet fileNames; // quick look-up file name
	StrSet dirNames; // quick look-up dir name
	StrVect revOrder;
	uint firstFreeLane;
};

#endif
